/*
 * Copyright (C) 2000-2002 Chris Ross and various contributors
 * Copyright (C) 1999-2000 Chris Ross
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * o Redistributions of source code must retain the above copyright notice, this
 *   list of conditions and the following disclaimer.
 * o Redistributions in binary form must reproduce the above copyright notice,
 *   this list of conditions and the following disclaimer in the documentation
 *   and/or other materials provided with the distribution.
 * o Neither the name of the ferite software nor the names of its contributors may
 *   be used to endorse or promote products derived from this software without
 *   specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include "ferite.h"

/* this is so that each registered class gets a unique id. this allows for native
 * code to check whether they are being passed one of their objects or someone elses. */
long ferite_internal_class_counter = 1000;

/**
 * !group Classes
 * !description Class are what objects are made from. There are a number of functions in this group
 *              that allows for creation and manipulation of both objects and classes.
 */

/**
 * !function ferite_register_inherited_class
 * !declaration FeriteClass *ferite_register_inherited_class( FeriteScript *script, FeriteNamespace *ns, char *name, char *parent )
 * !brief Registers, creates and returns a class object
 * !param FeriteScript *script The current script being run
 * !param FeriteNamespace *ns     The namespace in which the class should be registered
 * !param char *name   The name of the class eg. Socket
 * !param char *parent The name of the parent class, this is the class that will be inherited from, if NULL the class will inherit from the base class 'Obj'.
 * !description This is the only way to create a class natively.
 * !return The newly created class
 */
FeriteClass *ferite_register_inherited_class( FeriteScript *script, FeriteNamespace *ns, char *name, char *parent )
{
    FeriteClass *ptr = NULL, *klass = NULL;

    FE_ENTER_FUNCTION;

    if( (ferite_namespace_element_exists( script, ns, name )) != NULL )
    {
        ferite_warning( script, "Class %s already exists can't register\n", name );
        FE_LEAVE_FUNCTION(NULL);
    }
    if( parent == NULL && strcmp( name, "Obj" ) )
    {
        parent = "Obj";
    }
    if( parent != NULL )
    {
        ptr = ferite_find_class( script, ns, parent );
        if( ptr == NULL )
        {
            ferite_warning( script, "Parent class %s does not exist. Not inheriting from it for %s.\n", parent, name );
        }
    }
    klass = fmalloc( sizeof( FeriteClass ) );
    klass->name = fstrdup( name );
    /* a cunning hack to automatically inhherit varaibles */
    klass->variables = ( ptr == NULL ?
                         ferite_variable_hash_alloc( script, FE_CLASS_VARIABLE_HASH_SIZE ) :
                         ferite_duplicate_variable_hash( script, ptr->variables ) );
    klass->functions = ferite_create_hash( script, FE_CLASS_FUNCTION_HASH_SIZE );
    klass->id = ++ferite_internal_class_counter;
    klass->parent = ptr;
    klass->next = NULL;
    klass->container = ns;
    ferite_register_ns_class( script, ns, klass );

    FE_LEAVE_FUNCTION( klass );
}

/**
 * !function ferite_register_class_function
 * !declaration int ferite_register_class_function( FeriteScript *script, FeriteClass *klass, FeriteFunction *f )
 * !brief Register a function within a class
 * !param FeriteScript *script The current script
 * !param FeriteClass *klass The class to place the function in
 * !param FeriteFunction *f      The function structure
 * !param int is_static Boolean, 0 = not static, 1 = static. Allows the specifcation of whether or not the function has static access within the class
 * !description It must be noted that the function being passed must not have the super, and self variables within their signitue, this
 *              function will handle that automatically
 * !return 1 if function was registered correctly, 0 otherwise
 */
int ferite_register_class_function( FeriteScript *script, FeriteClass *klass, FeriteFunction *f, int is_static )
{
    FeriteFunction *function = NULL;
    
    FE_ENTER_FUNCTION;
    if( klass != NULL )
    {
        if( (function = ferite_hash_get( script, klass->functions, f->name )) != NULL )
        {
            f->next = function->next;
            function->next = f;
        }
        else
          ferite_hash_add( script, klass->functions, f->name, f );

        f->is_static = is_static;
        if( !f->is_static )
          f->klass = klass;
        if( f->type == FNC_IS_EXTRL && !(f->is_static) )
        {
            /* these are sekret vars :) */
            f->signature[f->arg_count] = fmalloc( sizeof( FeriteParameterRecord ) );
            f->signature[f->arg_count]->variable = ferite_create_object_variable( script, "super", FE_STATIC );
            f->signature[f->arg_count++]->has_default_value = 0;
            f->signature[f->arg_count] = fmalloc( sizeof( FeriteParameterRecord ) );
            f->signature[f->arg_count]->variable = ferite_create_object_variable( script, "self", FE_STATIC );
            f->signature[f->arg_count++]->has_default_value = 0;
        }
        FE_LEAVE_FUNCTION( 1 );
    }
    FE_LEAVE_FUNCTION( 0 );
}

/**
 * !function ferite_register_class_variable
 * !declaration int ferite_register_class_variable( FeriteScript *script, FeriteClass *klass, FeriteVariable *variable, int is_static )
 * !brief Register a variable within a class
 * !param FeriteScript *script    The current script
 * !param FeriteClass *klass  The class to attach the variable to
 * !param FeriteVariable *variable  The variable to register
 * !param int is_static Boolean, 0 = not static, 1 = static. Allows the specifcation of whether or not the variable has static access within the class
 * !return 1 on success, 0 otherwise
 */
int ferite_register_class_variable( FeriteScript *script, FeriteClass *klass, FeriteVariable *variable, int is_static )
{
    FE_ENTER_FUNCTION;

    if( klass != NULL )
    {
        if( variable != NULL )
        {
            ferite_add_variable_to_hash( script, klass->variables, variable );
            if( is_static )
              MARK_VARIABLE_AS_STATIC( variable );
        }
        else
          ferite_error( script, 0, "Can't register a NULL variable in class %s", klass->name );
    }
    else
      ferite_error( script, 0, "Can't register a variable in a non existant class" );

    FE_LEAVE_FUNCTION( 1 );
}

/**
 * !function ferite_delete_class
 * !declaration void ferite_delete_class( FeriteScript *script, FeriteClass *klass )
 * !brief Clean up and free the memory a class takes up
 * !param FeriteScript *script The current script
 * !param FeriteClass *klass The class to be deleted
 */
void ferite_delete_class( FeriteScript *script, FeriteClass *klass )
{
    FE_ENTER_FUNCTION;

    if( klass != NULL )
    {
        FUD(("Deleting Class: %s\n", klass->name ));
        ffree( klass->name );
        ferite_delete_variable_hash( script, klass->variables );
        ferite_delete_function_hash( script, klass->functions );
        ffree( klass );
    }

    FE_LEAVE_FUNCTION( NOWT );
}

/**
 * !function ferite_find_class
 * !declaration FeriteClass *ferite_find_class( FeriteScript *script, FeriteNamespace *ns, char *name )
 * !brief Find a class within a namespace
 * !param FeriteScript *script The current script
 * !param FeriteNamespace *ns     The top level namespace to look in
 * !param char *name   The name of the class to find.
 * !return The class on success or NULL otherwise
 */
FeriteClass *ferite_find_class( FeriteScript *script, FeriteNamespace *ns, char *name )
{
    FeriteClass *ptr = NULL;
    FeriteNamespaceBucket *nsb = NULL;

    FE_ENTER_FUNCTION;
    FUD(("Trying to find class %s\n", name));
    nsb = ferite_find_namespace( script, ns, name, FENS_CLS );
    if( nsb != NULL )
    {
        ptr = nsb->data;
        FE_LEAVE_FUNCTION( ptr );
    }
    nsb = ferite_find_namespace( script, script->mainns, name, FENS_CLS );
    if( nsb != NULL )
    {
        ptr = nsb->data;
        FE_LEAVE_FUNCTION( ptr );
    }
    FE_LEAVE_FUNCTION( NULL );
}

/**
 * !function ferite_find_class_id
 * !declaration long ferite_find_class_id( FeriteScript *script, FeriteNamespace *ns, char *name )
 * !brief Find a class within a namespace, and return it's unique id
 * !param FeriteScript *script The current script
 * !param FeriteNamespace *ns     The top level namespace to look in
 * !param char *name   The name of the class to find.
 * !return The id or 0 otherwise
 */
long ferite_find_class_id( FeriteScript *script, FeriteNamespace *ns, char *name )
{
    FeriteClass *ptr = NULL;

    FE_ENTER_FUNCTION;

    FUD(("Trying to find class %s\n", name));
    ptr = ferite_find_class( script, ns, name );
    if( ptr )
    {
        FE_LEAVE_FUNCTION( ptr->id );
    }
    FE_LEAVE_FUNCTION( 0 );
}

/**
 * !function ferite_delete_class_object
 * !declaration void ferite_delete_class_object( FeriteObject *object )
 * !brief Dispose of an object, the only part of ferite that should be calling this is the garbage collector.
 * !param object The object to be nuked.
 * !description
 * If you want to get rid of an object the accepted method is as follows:<nl/>
 * <nl/>
 *    object->refcount = 0;<nl/>
 *    ferite_check_gc();<nl/>
 * <nl/>
 * This will dispose of the object properly. The object destructor will get called if the objects' class is native.
 */
void ferite_delete_class_object( FeriteScript *script, FeriteObject *object, int do_destructor )
{
    FeriteFunction *func = NULL;
    FeriteVariable *retv;
    FeriteVariable *(*fptr)( FeriteScript *s, FeriteFunction *fn, FeriteVariable **plist );
    FeriteVariable **params;
    FeriteClass *klass;

    FE_ENTER_FUNCTION;
    if( object != NULL )
    {
        FUD(( "ferite_delete_class_object: trying to delete %s[%s]\n", object->name, object->klass->name ));
        if( do_destructor && object->klass != NULL && object->klass->functions != NULL )
        {
            for( klass = object->klass; func == NULL && klass != NULL; klass = klass->parent )
              func = ferite_hash_get( script, klass->functions, "Destructor" );

            if( func != NULL )
            {
                params = fmalloc( sizeof( FeriteVariable * ) * 3 );
                params[0] = ferite_variable_alloc( script );
                params[0]->name = fstrdup("DestrcutorVariable");
                params[0]->type = F_VAR_OBJ;
                VAO(params[0]) = object;
                params[1] = params[0];
                params[2] = NULL;

                /* we have the destructor */
                if( func->type == FNC_IS_EXTRL )
                {
                    FUD(( "OPS: Calling destructor (%s) in class %s\n", func->name, object->klass->name));
                    fptr = func->fncPtr;
                    retv = (fptr)( script, func, params );
                }
                else
                  retv = ferite_script_function_execute( script, func, params );

                ffree( params[0]->name );
                ffree( params[0] );
                ffree( params );
                ferite_variable_destroy( script, retv );
            }
        }

        FUD(( "Deleting class %s's variable hash %p\n", object->name, object->variables ));
        if( object->variables != NULL )
          ferite_delete_variable_hash( script, object->variables );

        if( object->name != NULL )
          ffree( object->name );

       /* Object cache */
        if( script && script->objects->stack_ptr < script->objects->size-1 )
          ferite_stack_push( script->objects, object );
        else
          ffree( object );
    }
    else
      ferite_error( script, 0, "Error: trying to delete null object\n" );
    FE_LEAVE_FUNCTION( NOWT );
}

/**
 * !function ferite_object_get_var
 * !declaration FeriteVariable *ferite_object_get_var( FeriteScript *script, FeriteObject *object, char *name )
 * !brief Get a member variable from an object
 * !param FeriteScript *script The script
 * !param FeriteObject *object The object to get the variable from
 * !param char *name The name of the member to get
 * !return The variable on success, NULL otherwise
 */
FeriteVariable *ferite_object_get_var( FeriteScript *script, FeriteObject *object, char *name )
{
    FeriteVariable *ptr = NULL;
    FE_ENTER_FUNCTION;
    if( object != NULL )
    {
        ptr = ferite_hash_get( script, object->variables, name );
	if( ptr != NULL && FE_VAR_IS_STATIC(ptr) )
	  ptr = NULL;
    }
    FE_LEAVE_FUNCTION( ptr );
}

/**
 * !function ferite_class_get_var
 * !declaration FeriteVariable *ferite_class_get_var( FeriteScript *script, FeriteClass *klass, char *name )
 * !brief Get a member variable from an class
 * !param FeriteScript *script The script
 * !param FeriteClass *klass The class to get the variable from
 * !param char *name The name of the member to get
 * !return The variable on success, NULL otherwise
 */
FeriteVariable *ferite_class_get_var( FeriteScript *script, FeriteClass *klass, char *name )
{
    FeriteVariable *ptr = NULL;
    FE_ENTER_FUNCTION;
    if( klass != NULL )
    {
	ptr = ferite_hash_get( script, klass->variables, name );
	if( !FE_VAR_IS_STATIC(ptr) )
	  ptr = NULL;
    }
    FE_LEAVE_FUNCTION( ptr );
}

/**
 * !function ferite_object_get_function
 * !declaration FeriteFunction *ferite_object_get_function( FeriteScript *script, FeriteObject *object, char *name )
 * !brief Get a function from an object
 * !param FeriteScript *script The script
 * !param FeriteObject *object The object to get the function from
 * !param char *name The name of the function
 * !return The function on success, NULL otherwise
 */
FeriteFunction *ferite_object_get_function( FeriteScript *script, FeriteObject *object, char *name )
{
    FeriteFunction *ptr = NULL;
    FeriteClass *cls = NULL;
    
    FE_ENTER_FUNCTION;
    if( object != NULL )
    {
	cls = object->klass;
	while( cls != NULL )
	{
	    ptr = ferite_hash_get( script, cls->functions, name );
	    if( ptr != NULL && !ptr->is_static )
	    {
		FE_LEAVE_FUNCTION( ptr );
	    }
	    cls = cls->parent;
	}
    }
    FE_LEAVE_FUNCTION( ptr );
}

/**
 * !function ferite_class_get_function
 * !declaration FeriteFunction *ferite_class_get_function( FeriteScript *script, FeriteClass *cls, char *name )
 * !brief Get a function from a class
 * !param FeriteScript *script The script
 * !param FeriteClass *cls The class to get the function from
 * !param char *name The name of the function
 * !return The function on success, NULL otherwise
 */
FeriteFunction *ferite_class_get_function(FeriteScript *script, FeriteClass *cls, char *name)
{
    FeriteFunction *ptr = NULL;

    FE_ENTER_FUNCTION;
    while(cls != NULL)
    {
        ptr = ferite_hash_get(script, cls->functions, name);
        if( ptr != NULL && ptr->is_static )
        {
            FE_LEAVE_FUNCTION( ptr );
        }
        cls = cls->parent;
    }
    FE_LEAVE_FUNCTION( NULL );
}

/**
 * !function ferite_class_has_function
 * !declaration int ferite_class_has_function(FeriteScript *script, FeriteClass *cls, char *name)
 * !brief Check to see if a class has the named function
 * !param FeriteScript *script The script
 * !param FeriteClass *cls The class to check in
 * !param char *name The name of the function
 * !return FE_TRUE on success, FE_FALSE otherwise
 */
int ferite_class_has_function(FeriteScript *script, FeriteClass *cls, char *name)
{
    FeriteFunction *ptr = ferite_class_get_function(script,cls,name);

    FE_ENTER_FUNCTION;
    if( ptr == NULL )
    {
        FE_LEAVE_FUNCTION( FE_FALSE );
    }
    FE_LEAVE_FUNCTION( FE_TRUE );
}

/**
 * !function ferite_find_function_in_object
 * !declaration FeriteFunction *ferite_find_function_in_object( FeriteScript *script, FeriteObject *obj, char *function )
 * !brief Find a function in an object, go up the inheirtance tree if necessary
 * !param FeriteScript *script   The current script
 * !param FeriteObject *obj      The object to search
 * !param char *function The name of the function to find
 * !return The function on success, NULL otherwise
 */
FeriteFunction *ferite_find_function_in_object( FeriteScript *script, FeriteObject *obj, char *function )
{
    FeriteFunction *ptr = NULL;
    FeriteClass *cp = NULL;
    int offset = 0;

    FE_ENTER_FUNCTION;

    FUD(("Looking for function %s\n", function));
    for( offset = strlen( function ) - 1; offset >= 0; offset-- )
    {
        if( function[offset] == '.' )
        {
            offset++;
            break;
        }
    }
    if( offset < 0 )
      offset = 0;
    if( function[offset] == '.' )
      offset++;

    FUD(("Searching for function %s in %s, offset: %d\n", function+offset, obj->name, offset ));
    ptr = ferite_hash_get( script, obj->klass->functions, function+offset );
    if( ptr == NULL )
    {
      /* we need to go up the class tree */
        if( obj->klass != NULL )
        {
            for( cp = obj->klass->parent; cp != NULL; cp = cp->parent )
            {
                ptr = ferite_hash_get( script, cp->functions, function+offset );
                if( ptr != NULL )
                  break;
            }
        }
    }
    FE_LEAVE_FUNCTION( ptr );
}

/**
 * !function ferite_object_has_var
 * !declaration int ferite_object_has_var(FeriteScript* script, FeriteObject* obj, char *id)
 * !brief Check to see whether an object has a certain variable
 * !param FeriteScript *script The script
 * !param FeriteObject *obj The object to check
 * !param char *id The name of the variable
 * !return If the variable exists FE_TRUE is returned, otherwise FE_FALSE
 */
int ferite_object_has_var(FeriteScript* script, FeriteObject* obj, char *id)
{
    FeriteVariable *ptr = NULL;

    FE_ENTER_FUNCTION;
    ptr = ferite_object_get_var( script, obj, id );
    if( ptr != NULL )
    {
        FE_LEAVE_FUNCTION( FE_TRUE );
    }
    FE_LEAVE_FUNCTION( FE_FALSE );
}

/**
 * !function ferite_object_set_var
 * !declaration void ferite_object_set_var( FeriteScript* script, FeriteObject* obj, char *id, FeriteVariable *d_var )
 * !brief Set the value of an objects variable
 * !param FeriteScript *script The script
 * !param FeriteObject *obj The object whose member is to be set
 * !param char *name The name of the variable
 * !param FeriteVariable *d_var The data used
 * !description If the variable exists, the value is updated, otherwise a new variable is added to the
 *              objects variables
 */
void ferite_object_set_var( FeriteScript* script, FeriteObject* obj, char *id, FeriteVariable *d_var )
{
    FeriteVariable *ptr = NULL;

    FE_ENTER_FUNCTION;
    UNMARK_VARIABLE_AS_DISPOSABLE( d_var );
    ptr = ferite_hash_get( script, obj->variables, id );
    if( ptr != NULL )
    {
        ferite_hash_update( script, obj->variables, id, (void*)d_var );
        ferite_variable_destroy( script, ptr );
    }
    else
      ferite_hash_add( script, obj->variables, id, d_var );
    FE_LEAVE_FUNCTION(NOWT);
}

/**
 * !function ferite_object_call_super
 * !declaration void ferite_object_call_super( FeriteScript *script, FeriteObject *object, FeriteVariable **params )
 * !brief Call the objects parent class's constructor usefull for writing native classes
 * !param FeriteScript *script The script
 * !param FeriteObject *object The object on which to call the super class's constructor
 * !param FeriteVariable **params The parameters to pass to the function call
 */
FeriteVariable *ferite_object_call_super( FeriteScript *script, FeriteObject *object, FeriteVariable **params )
{
    FeriteFunction *function = NULL;
    FeriteClass *old_parent = NULL;
    FeriteVariable *rval = NULL;
    
    FE_ENTER_FUNCTION;
    if( object->klass->parent == NULL )
    {
        /* Our fake variable */
        rval = ferite_create_object_variable( script, "ferite_call_super", FE_STATIC );
        MARK_VARIABLE_AS_DISPOSABLE( rval );
        
        /* return no parental constructor to call */
        FE_LEAVE_FUNCTION( rval );
    }

    function = ferite_find_parent_constructor( script, object->klass->parent );

    if( function != NULL )
    {
        /* this basically makes the constructor being called play like it's own class */
        old_parent = object->klass;
        object->klass = old_parent->parent;
        
        rval = ferite_call_function( script, function, params );

        /* swap it back */
        object->klass = old_parent;
    }
    else
    {
        /* No super constructor, lets just return a null object */
        rval = ferite_create_object_variable( script, "ferite_call_super", FE_STATIC );
        MARK_VARIABLE_AS_DISPOSABLE( rval );
    }
    
    FE_LEAVE_FUNCTION( rval );
}

/**
 * !function ferite_object_is_sublass
 * !declaration int ferite_object_is_sublass( FeriteObject *obj, char *name )
 * !brief See if the object is from a subclass of the named class
 * !param FeriteObject *obj The object to check
 * !param char *name The name of the class
 * !return FE_TRUE if the object is, FE_FALSE otherwise
 */
int ferite_object_is_sublass( FeriteObject *obj, char *name )
{
    FeriteClass *ptr;

    FE_ENTER_FUNCTION;
    if( obj != NULL )
    {
        for( ptr = obj->klass; ptr != NULL; ptr = ptr->parent )
        {
            if( strcmp( ptr->name, name ) == 0 )
            {
                FE_LEAVE_FUNCTION( FE_TRUE );
            }
        }
    }
    FE_LEAVE_FUNCTION( FE_FALSE );
}

/**
 * !function ferite_find_parent_constructor
 * !declaration FeriteFunction *ferite_find_parent_constructor( FeriteScript *script, FeriteClass *klass )
 * !brief Get a pointer to the first constructor in a class tree
 * !param FeriteScript *script The script
 * !param FeriteClass *klass The class to start with
 * !return A pointer to the function, or NULL otherwise
 */
FeriteFunction *ferite_find_parent_constructor( FeriteScript *script, FeriteClass *klass )
{
    FeriteClass *ptr = NULL;
    FeriteFunction *func = NULL;

    FE_ENTER_FUNCTION;
    for( ptr = klass; ptr != NULL; ptr = ptr->parent )
    {
        if( (func = ferite_hash_get( script, ptr->functions, ptr->name )) != NULL )
        {
            FE_LEAVE_FUNCTION( func );
        }
    }
    FE_LEAVE_FUNCTION( NULL );
}

/**
 * !function ferite_object_add_self_variable_to_params
 * !declaration FeriteVariable **ferite_object_add_self_variable_to_params( FeriteScript *script, FeriteVariable **param, FeriteVariable *ptr )
 * !brief Add the important variables to the parameter list for an object based function call
 * !param FeriteScript *script The script
 * !param FeriteVariable **params The parameter list to add the variables onto
 * !param FeriteObject *ptr The object from which a function is being called
 * !return The parameter list
 */
FeriteVariable **ferite_object_add_self_variable_to_params( FeriteScript *script, FeriteVariable **param, FeriteObject *ptr )
{
    FeriteVariable *result = NULL, *result2 = NULL;
    FeriteVariable **params = param;

    FE_ENTER_FUNCTION;
    result = ferite_create_object_variable( script, "super", FE_STATIC );
    MARK_VARIABLE_AS_DISPOSABLE( result );
    result2 = ferite_create_object_variable( script, "self", FE_STATIC );
    MARK_VARIABLE_AS_DISPOSABLE( result2 );
   
    VAO(result) = ptr;
    VAO(result2) = ptr;
   
    ptr->refcount += 2;
   
    params = ferite_add_to_parameter_list( params, result );
    params = ferite_add_to_parameter_list( params, result2 );
    FE_LEAVE_FUNCTION( params );
}

/**
 * !function ferite_class_dup
 * !declaration FeriteClass *ferite_class_dup( FeriteScript *script, FeriteClass *klass, FeriteNamespace *container )
 * !brief Create a complete duplicate of a class
 * !param FeriteScript *script The script
 * !param FeriteClass *klass The class to duplicate
 * !param FeriteNamespace *container A pointer to the namespace that will hold the class
 * !return A pointer to a new FeriteClass structure
 */
FeriteClass *ferite_class_dup( FeriteScript *script, FeriteClass *klass, FeriteNamespace *container )
{
    FeriteClass *ptr = NULL;

    FE_ENTER_FUNCTION;
    if( klass != NULL )
    {
        ptr = fmalloc(sizeof(FeriteClass));
        ptr->name = fstrdup( klass->name );
        ptr->id = klass->id;
        /* FIXME: how to get the parent class done correctly */
        ptr->parent = NULL;
        ptr->variables = ferite_duplicate_variable_hash( script, klass->variables );
        ptr->functions = ferite_hash_dup( script, klass->functions, (void *(*)(FeriteScript *,void *,void*))ferite_function_dup, ptr );
        ptr->next = NULL;
        ptr->container = container;
    }
    FE_LEAVE_FUNCTION(ptr);
}

/**
 * !end
 */
