/*
 * Copyright (C) 2000-2002 Chris Ross and various contributors
 * Copyright (C) 1999-2000 Chris Ross
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * o Redistributions of source code must retain the above copyright notice, this
 *   list of conditions and the following disclaimer.
 * o Redistributions in binary form must reproduce the above copyright notice,
 *   this list of conditions and the following disclaimer in the documentation
 *   and/or other materials provided with the distribution.
 * o Neither the name of the ferite software nor the names of its contributors may
 *   be used to endorse or promote products derived from this software without
 *   specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include "ferite.h"
#include <pcre.h> /* perl compatible regualr expressions */

/**
 * !group Regular Expressions
 * !description Regular expressions play an important role within the ferite language. They can
 *              be used from native code using a very simple interface.
 */

/***************************************************************
 * The way some of this code operates (eg within the matches)  *
 * was semi borrowed from php, this is because the pcre module *
 * that is shiped with php is rather good.                     *
 *                                                             *
 * Those parts of the code borrowed were written by            *
 *   Andrei Zmievski <andrei@ispi.net> and are subject to the  *
 *   PHP 2.02 lisence - http://www.php.net/license/2_02.txt    *
 ***************************************************************/

/* memory functions so pcre uses ferite's memory management */
void *(*old_pcre_malloc)(size_t);
void  (*old_pcre_free)(void *);

void *ferite_regex_malloc( size_t size)
{
    void *ptr;
    FE_ENTER_FUNCTION;
    ptr = fmalloc( size );
    FE_LEAVE_FUNCTION( ptr );
}

void ferite_regex_free( void *ptr )
{
    FE_ENTER_FUNCTION;
    ffree( ptr );
    FE_LEAVE_FUNCTION( NOWT );
}

void ferite_init_regex()
{
    FE_ENTER_FUNCTION;
    if( ferite_use_mm_with_pcre == 1 )
    {
        old_pcre_malloc = pcre_malloc;
        old_pcre_free = pcre_free;
        pcre_malloc = ferite_regex_malloc;
        pcre_free = ferite_regex_free;
    }
    FUD(( "REGEX, Using: PCRE %s\n", pcre_version() ));
    FE_LEAVE_FUNCTION( NOWT );
}

void ferite_deinit_regex()
{
    FE_ENTER_FUNCTION;
    if( ferite_use_mm_with_pcre == 1 )
    {
        pcre_malloc = old_pcre_malloc;
        pcre_free = old_pcre_free;
    }
    FE_LEAVE_FUNCTION( NOWT );
}

/**
 * !function ferite_create_regex
 * !declaration FeriteRegex *ferite_create_regex()
 * !brief Create a regular expression structure used to store various information for them
 * !return A new FeriteRegex structure
 */
FeriteRegex *ferite_create_regex()
{
    FeriteRegex *ptr = NULL;

    FE_ENTER_FUNCTION;

    ptr = fmalloc( sizeof( FeriteRegex ) );
    ptr->pattern = NULL;
    ptr->type = F_RGX_MATCH;
    ptr->pcre_options = 0;
    ptr->fergx_options = 0;
    ptr->compiled_re = NULL;
    ptr->compile_buf = NULL;
    ptr->swap_buf = NULL;
    ptr->extra_info = NULL;

    FE_LEAVE_FUNCTION( ptr );
}

/**
 * !function ferite_delete_regex
 * !declaration void ferite_delete_regex( FeriteRegex *rgx )
 * !brief Delete a strucute created using ferite_create_regex()
 * !param FeriteRegex *rgx The regex to delete
 */
void ferite_delete_regex( FeriteRegex *rgx )
{
    FE_ENTER_FUNCTION;

    if( rgx != NULL )
    {
        if( rgx->pattern != NULL )
          ffree( rgx->pattern );
        if( rgx->compiled_re != NULL )
        {
            if( !ferite_use_mm_with_pcre )
              pcre_free( rgx->compiled_re );
            else
              ffree( rgx->compiled_re );
        }
        if( rgx->compile_buf != NULL )
          ffree( rgx->compile_buf );
        if( rgx->swap_buf != NULL )
          ffree( rgx->swap_buf );
        ffree( rgx );
    }
    FE_LEAVE_FUNCTION( NOWT );
}

/**
 * !function ferite_regex_dup
 * !declaration FeriteRegex *ferite_regex_dup( FeriteRegex *rgx )
 * !brief Duplicate a regular expression structure
 * !param FeriteRegex *rgx The regex to duplicate
 * !return An exact copy of the regexular expression
 */
FeriteRegex *ferite_regex_dup( FeriteRegex *rgx )
{
    FeriteRegex *ptr = NULL;

    FE_ENTER_FUNCTION;
    if( rgx != NULL )
    {
        ptr = fmalloc(sizeof(FeriteRegex));
        ptr->pattern = fstrdup( rgx->pattern );
        ptr->type = rgx->type;
        ptr->pcre_options = rgx->pcre_options;
        ptr->fergx_options = rgx->fergx_options;
        ptr->compile_buf = fstrdup( rgx->compile_buf );
        ptr->swap_buf = fstrdup( rgx->swap_buf );
        ptr->extra_info = NULL;
        ptr->compiled_re = NULL;
    }
    FE_LEAVE_FUNCTION( ptr );
}

/**
 * !function ferite_generate_regex
 * !declaration FeriteRegex *ferite_generate_regex( FeriteScript *script, char *pattern, char *swapbuffer, char type, char *flags )
 * !brief A convinience function to create and popular a regex structure
 * !param FeriteScript *script The script
 * !param char *pattern The regular expression
 * !param char *swapbuffer The buffer that is evaluated on a match
 * !param char type The type of the regular expression. s for a swap, m for a match, c for an assignment.
 * !param char *flags The flags the effect the behavior of the regular expression.
 * !return A regular expression structure setup and populated correctly
 */
FeriteRegex *ferite_generate_regex( FeriteScript *script, char *pattern, char *swapbuffer, char type, char *flags )
{
    FeriteRegex *ptr = NULL;
    int i = 0;

    FE_ENTER_FUNCTION;

    ptr = ferite_create_regex();
    ptr->pattern = fstrdup( pattern );
    ptr->swap_buf = fstrdup( swapbuffer );
    ptr->type = F_RGX_MATCH;

   /* go from start to end picking up modifiers (eg s and m) */
    FUD(( "REGEX prefix\n" ));
    switch( type )
    {
      case 's':
        FUD(( "REGEX - recognised SWAP\n" ));
        ptr->type = F_RGX_SWAP;
        break;
      case 'm':
        FUD(( " - recognised MATCH\n" ));
        ptr->type = F_RGX_MATCH;
        break;
      case 'c':
        ptr->type = F_RGX_ASSIGN;
        break;
      default:
        ferite_warning( script, "Regex Modifier %c - UNKOWN, ignoring\n", type );
    }
   /* go from end to beginning to get the options */
    FUD(( "REGEX postfix's\n" ));
    for( i = 0; i < strlen( flags ); i++ )
    {
        switch( flags[i] )
        {
            /* pcre supported perl'isms */
          case 'x': FUD(( "REGEX: `-> Extended Line\n" ));    ptr->pcre_options |= PCRE_EXTENDED; break;
          case 's': FUD(( "REGEX: `-> Single Line Mode\n" )); ptr->pcre_options |= PCRE_DOTALL; break;
          case 'm': FUD(( "REGEX: `-> Multi Line Mode\n" ));  ptr->pcre_options |= PCRE_MULTILINE; break;
          case 'i': FUD(( "REGEX: `-> Case Insensitive\n" )); ptr->pcre_options |= PCRE_CASELESS; break;
            /* pcre + ferite internal'isms */
          case 'g': FUD(( "REGEX: `-> Global Matching\n" ));  ptr->fergx_options |= F_RGX_GLOBAL; break;
          case 'o': FUD(( "REGEX: `-> Cache Compile\n" ));    ptr->fergx_options |= F_RGX_COMPILE; break;
          case 'e':
            if( ptr->type != F_RGX_SWAP )
              ferite_warning( script, "Script Evaluator can only be used on a swap, not a match\n" );
            else
            {
                FUD(( "REGEX: `-> Evaluate Swap Portion\n" ));
                ptr->fergx_options |= F_RGX_EVAL_SWP;
            }
            break;
            /* pcre's specific stuff */
          case 'A': FUD(( "REGEX: `-> Anchored\n" ));         ptr->pcre_options |= PCRE_ANCHORED; break;
          case 'D': FUD(( "REGEX: `-> Dollar @ ABS End\n" )); ptr->pcre_options |= PCRE_DOLLAR_ENDONLY; break;
          default:
            ferite_warning( script, "Regex Option %c - UNKNOWN, ignoring\n", flags[i] );
        }
    }

   /* setup compile buffers and pre-compile the regex */
    ptr->compile_buf = fstrdup( ptr->pattern );
    if( ptr->fergx_options & F_RGX_COMPILE )
    {
        FUD(( "REGEX: Compiling RGX: \"%s\"\n", ptr->compile_buf ));
        ptr->compiled_re = ferite_compile_regex( script, ptr->compile_buf, ptr->pcre_options );
    }
    FE_LEAVE_FUNCTION( ptr );
}

/**
 * !function ferite_compile_regex
 * !declaration void *ferite_compile_regex( FeriteScript *script, char *pattern, int options )
 * !brief Compile a regular expression to make things faster
 * !param FeriteScript *script The script
 * !param char *pattern The pattern to compile
 * !param int options The options that effect how it is compiled
 * !return A pointer to the in memory compiled regular expression
 */
void *ferite_compile_regex( FeriteScript *script, char *pattern, int options )
{
    void       *ptr;
    const char *error;
    int         erroroffset;

    FE_ENTER_FUNCTION;
    ptr = pcre_compile( pattern, options, &error, &erroroffset, NULL );

    if( ptr == NULL )
    {
        ferite_warning( script, "Regex Compilation failed: %s at offset %d\n", error, erroroffset );
        return NULL;
    }

    FE_LEAVE_FUNCTION( ptr );
}

#define islabelstart( c )   ( c == '_' || (c >= 'a' && c <= 'z') || (c >= 'A' && c <= 'Z') )
#define islabelchar( c )   ( c == '_' || (c >= 'a' && c <= 'z') || (c >= 'A' && c <= 'Z') || (c >= '0' && c <= '9') )

/**
 * !function ferite_regex_swap_vars
 * !declaration char *ferite_regex_swap_vars( char *rgxBuf, FeriteScript *script, FeriteExecuteRec *er )
 * !brief Swap tokens with a dollar '$' symbol infront with the variables value
 * !param char *rgxBuf The buffer to evaluate
 * !param FeriteScript *script The script where to get the global variables
 * !param FeriteExecuteRec *er The recod where to get local variables
 * !description This function will first check the locally scoped variables for values then
 *              will look at global variables. If the variable doesn't exist then it will be
 *              replaced by an empty string.
 * !return A null terminate string with the variables evaluated.
 */
char *ferite_regex_swap_vars( char *rgxBuf, FeriteScript *script, FeriteExecuteRec *er )
{
    FeriteVariable   *repVar = NULL;
    char             *oldRgx, buf[1024];
    char             *newRgx = fstrdup("");
    int               i, start, len = strlen(rgxBuf), newLength = 0;
    FeriteNamespaceBucket *nsb;

   /* fprintf( stderr, "--> start\n" ); */
    for( i = 0; i < len; i++ )
    {
        memset( buf, '\0', 1024 );
        oldRgx = newRgx;
        if( rgxBuf[i] == '$' )
        {
            if( islabelstart( rgxBuf[i+1] ) )
            {
                start = i++;
                while( i < len && islabelchar( rgxBuf[i] ) )
                  i++;
                strncpy( buf, rgxBuf+start, i - start );
                /*   fprintf( stderr, "  ---> Found Variable `%s' (%s)\n", buf, buf+1 ); */
                repVar = ferite_variable_from_array( er->variable_list, buf+1 );
                if( repVar == NULL )
                {
                    nsb = ferite_namespace_element_exists( script, script->mainns, buf+1 );
                    if( nsb != NULL )
                    {
                        repVar = nsb->data;
                    }
                }
                if( repVar != NULL && repVar->type == F_VAR_STR )
                {
         /*      fprintf( stderr, "        value: `%s'\n", VAS(repVar) ); */
                    newLength = strlen(oldRgx) + FE_STRLEN(repVar) + 1;
                    newRgx = fmalloc( newLength );
                    memset( newRgx, '\0', newLength );
                    strcpy( newRgx, oldRgx );
                    strcat( newRgx, FE_STR2PTR(repVar) );
                    ffree( oldRgx );
         /*      fprintf( stderr, "        newRgx `%s', allocated %d, actual %d\n", newRgx, newLength, strlen(newRgx) ); */
                }
                else
                {
         /*      fprintf( stderr, "        value: `%s'\n", "no value" );      */
                }
                i--;
            }
        }
        else
        {
            start = i;
            while( i < len && rgxBuf[i] != '$' )
              i++;
            /*   strncpy( buf, rgxBuf+start, i - start );
             fprintf( stderr, "  ---> Found Data      `%s'\n", buf );*/
            newLength = strlen(oldRgx) + (i-start) + 1;
            newRgx = fmalloc( newLength );
            memset( newRgx, '\0', newLength );
            strcpy( newRgx, oldRgx );
            strncat( newRgx, rgxBuf+start, i - start );
            ffree( oldRgx );
            /*   fprintf( stderr, "        newRgx `%s', allocated %d, actual %d\n", newRgx, newLength, strlen(newRgx) );*/
            i--;
        }
    }
    return newRgx;
}

/**
 * !function ferite_execute_regex
 * !declaration FeriteVariable *ferite_execute_regex( FeriteRegex *rgx, FeriteVariable *target, FeriteScript *script, FeriteExecuteRec *er )
 * !brief Execute the regular expression
 * !param FeriteRegex *rgx The regular expression to execute
 * !param FeriteVariable *target The string on which to apply the regular expression
 * !param FeriteScript *script The script in which the regular expression executes.
 * !param FeriteExecuteRec *er The execution record of the local function, this is needed for variable substitution.
 * !return The result of the execution, either an array or a string
 */
FeriteVariable *ferite_execute_regex( FeriteRegex *rgx, FeriteVariable *target, FeriteScript *script, FeriteExecuteRec *er )
{
    FeriteVariable *ptr = NULL;
    char *oldRgx = NULL, *oldSwap = NULL;

    FE_ENTER_FUNCTION;
    if( rgx == NULL )
      ferite_error( script, 0, "Trying to execute non-existant regex" );
    if( target == NULL )
      ferite_error( script, 0, "Trying to execute regex on non-existant variable" );
    if( target->type != F_VAR_STR )
    {
        ferite_warning( script, "Trying to execute on non-string value, returning false\n" );
        FE_LEAVE_FUNCTION( ferite_create_number_long_variable( script, "regex-exec-return", 0, FE_STATIC ) );
    }

    oldRgx = rgx->compile_buf;
    oldSwap = rgx->swap_buf;

    /* as Mr. Friedl says in Mastering Regular Expressions, this does the 'Doublequotish' behavior (ie variable substiution */
    rgx->compile_buf = ferite_regex_swap_vars( rgx->compile_buf, script, er );
    if( rgx->type == F_RGX_SWAP && rgx->swap_buf != NULL )
      rgx->swap_buf = ferite_regex_swap_vars( rgx->swap_buf, script, er );
    if( rgx->compiled_re != NULL && strcmp( rgx->compile_buf, oldRgx ) )
    {
        if( !ferite_use_mm_with_pcre )
        {
            pcre_free( rgx->compiled_re );
            rgx->compiled_re = NULL;
        }
        else
        {
            ffree( rgx->compiled_re );
        }
    }  
    
    switch( rgx->type )
    {
      case F_RGX_MATCH:
        ptr = ferite_execute_match_regex( rgx, target, ( rgx->fergx_options & F_RGX_GLOBAL ? 1 : 0 ), FE_FALSE, script->mainns, script, er );
        break;
      case F_RGX_SWAP:
        ptr = ferite_execute_swap_regex( rgx, target, ( rgx->fergx_options & F_RGX_GLOBAL ? 1 : 0 ), script->mainns, script );
        break;
      case F_RGX_ASSIGN:
        ptr = ferite_execute_match_regex( rgx, target, ( rgx->fergx_options & F_RGX_GLOBAL ? 1 : 0 ), FE_TRUE, script->mainns, script, er );
        break;
      default:
        ferite_warning( script, "Unknown regex type %d, returning false\n", rgx->type );
        ptr = ferite_create_number_long_variable( script, "regex-exec-return", 0, FE_STATIC );
    }
    
    ffree( rgx->compile_buf );
    if( rgx->type == F_RGX_SWAP && rgx->swap_buf != NULL )
    {
        ffree( rgx->swap_buf );
    }
    rgx->compile_buf = oldRgx;
    rgx->swap_buf = oldSwap;
    FE_LEAVE_FUNCTION( ptr );
}

FeriteVariable *ferite_execute_match_regex( FeriteRegex *rgx, FeriteVariable *target, int global, int should_assign, FeriteNamespace *ns, FeriteScript *script, FeriteExecuteRec *er )
{
    FeriteVariable *retv = NULL; /* rturn value */
    FeriteVariable *ptr = NULL;  /* so we can fudge with the arrays */
    FeriteNamespaceBucket *nsb = NULL;
    int captured_str_cnt = 0; /* number of strings that will be captured in ()'s */
    int *offsets;         /* array of subpattern offsets */
    int size_offsets;     /* size of the array */
    int start_offset;     /* damn nation, this is the new start :) */
    int current_var = 0;  /* the global variable we need to update */
    int current_match = 0;
    int count = 0;        /* count the number of subexpressions */
    int i, len;
    char buf[10], *match = NULL, buf2[1024];
    const char **stringlist /* the subparts () values */;
    FeriteVariable **assignment_array = NULL;
    int g_notempty = 0;

    FE_ENTER_FUNCTION;

    /* if the regex is not compiled already -> compile it */
    if( rgx->compiled_re == NULL )
    {
        rgx->compiled_re = ferite_compile_regex( script, rgx->compile_buf, rgx->pcre_options );
        if( rgx->compiled_re == NULL )
        {
            retv = ferite_create_number_long_variable( script, "regex-exec-return", 0, FE_STATIC );
            FE_LEAVE_FUNCTION( retv );
        }
    }

    /* get the number of subparts */
    captured_str_cnt = pcre_info( rgx->compiled_re, NULL, NULL ) + 1;
    /* create an offset array */
    size_offsets = captured_str_cnt * 3;
    offsets = (int *)fmalloc(size_offsets * sizeof(int));

    if( should_assign == FE_TRUE )
    {
        assignment_array = fcalloc( sizeof(FeriteVariable*) * captured_str_cnt, sizeof(FeriteVariable*) );
        current_var = 0;
        start_offset = 0;
        len = strlen(rgx->swap_buf) - 1;
        for( i = 0; i <= len; i++ )
        {
            if( rgx->swap_buf[i] == ',' || len == i )
            {
                current_var++;
                if( i - start_offset > 0 )
                {
                    memset( buf2, '\0', 1024 );
                    if( len == i ) i++;
                    strncpy( buf2, rgx->swap_buf+start_offset, i - start_offset);

                    /* lets find the variable */
                    assignment_array[current_var] = ferite_variable_from_array( er->variable_list, buf2 );
                    if( assignment_array[current_var] == NULL )
                    {
                        nsb = ferite_find_namespace( script, script->mainns, buf2, FENS_VAR );
                        if( nsb != NULL )
                          assignment_array[current_var] = nsb->data;
                        if( assignment_array[current_var] == NULL )
                          ferite_warning( script, "Unable to find variable '%s' for use in regular expression.\n", buf2 );
                    }
                    else
                    {
                        if( assignment_array[current_var]->type != F_VAR_STR )
                        {
                            assignment_array[current_var] = NULL;
                            ferite_warning( script, "Variable '%s' must be a string to be assigned to.\n", buf2 );
                        }
                    }
                }
                start_offset = i+1;
            }
        }
    }

    /* we need to setup the global variable hash so that r1->r99 holds the number of subparts */
    for( i = 1; i <= (captured_str_cnt > script->last_regex_count ? captured_str_cnt : script->last_regex_count); i++ )
    {
        memset( buf, '\0', 10 );
        sprintf( buf, "r%d", i );
        nsb = ferite_namespace_element_exists( script, ns, buf );
        ptr = ( nsb == NULL ? NULL : nsb->data );
        if( ptr == NULL )
        {
            ptr = ferite_create_string_variable( script, buf, NULL, FE_ALLOC );
            MARK_VARIABLE_AS_FINALSET( ptr ); /* we make these global variables read only */
            ferite_register_ns_variable( script, ns, ptr );
        }
    }
    script->last_regex_count = captured_str_cnt;

    /* setup the return value */
    if( global )
      retv = ferite_create_uarray_variable( script, "regex-exec-return", 32, FE_STATIC );
    else
      retv = ferite_create_string_variable( script, "regex-exec-return", NULL, FE_STATIC );

    start_offset = 0;
    current_var = 1;
    current_match = 1;

    do
    {
      /* execute regex */
        FUD(( "REGEX: Executing against \"%s\"\n", FE_STR2PTR(target) ));
        count = pcre_exec( rgx->compiled_re, rgx->extra_info, FE_STR2PTR(target), FE_STRLEN( target ),
                           start_offset, g_notempty, offsets, size_offsets );

      /* check for too many substrings */
        if( count == 0 )
        {
            ferite_warning( script, "A match was found but too many substrings found.\n" );
            count = size_offsets / 3;
        }

        FUD(( "REGEX: match count = %d\n", count ));
      /* we matched something */
        if( count >= 0 )
        {
            match = FE_STR2PTR(target) + offsets[0];

     /* get the list of substrings */
            if( pcre_get_substring_list( FE_STR2PTR(target), offsets, count, &stringlist) < 0 )
            {
                ffree(offsets);
                ferite_warning( script, "Unable to obtain captured strings in regular expression.\n");
                ferite_variable_destroy( script, retv );
                retv = ferite_create_number_long_variable( script, "regex-exec-return", 0, FE_STATIC );
                FE_LEAVE_FUNCTION( retv );
            }

            if( global )
            {
                FUD(( "REGEX: Setting index %d to \"%s\"\n", current_match, (char *)stringlist[0] ));
                memset( buf, '\0', 10 );
                sprintf( buf, "hash-%d", current_match );
                ptr = ferite_create_string_variable_from_ptr( script, buf, (char *)stringlist[0], strlen((char *)stringlist[0]), FE_CHARSET_DEFAULT, FE_ALLOC );
                FUD(("REGEX: Current Match: %d\n", current_match ));
                ferite_uarray_add( script, VAUA(retv), ptr, NULL, FE_ARRAY_ADD_AT_END );
                current_match++;
            }
            /* need to setup the r%% variables */
            /* 0 = total matched string 1++ is subparts */
            for( i = 1; i < count; i++ )
            {
                memset( buf, '\0', 10 );
                sprintf( buf, "r%d", i /*current_var*/ );
                nsb = ferite_namespace_element_exists( script, ns, buf );
                ptr = nsb->data;
                ferite_str_set(VAS(ptr), (char *)stringlist[i], strlen((char *)stringlist[i]), FE_CHARSET_DEFAULT );
                FUD(( "Setting %s to %s\n", ptr->name, FE_STR2PTR(ptr) ));
                current_var++;

                if( assignment_array != NULL && assignment_array[i] != NULL )
                {
                    ferite_str_set( VAS(assignment_array[i]), (char *)stringlist[i], strlen((char *)stringlist[i]), FE_CHARSET_DEFAULT );
                }
            }
            if( ferite_use_mm_with_pcre )
            {
                ffree( stringlist );
            }
            else
              pcre_free( stringlist );
        }
        else /* we didn't match something */
        {
            match = "\0";
            if (g_notempty != 0 && start_offset < FE_STRLEN(target) )
            {
                offsets[0] = start_offset;
                offsets[1] = start_offset + 1;
            }
            else
              break;
        }

        g_notempty = ( offsets[1] == offsets[0] ) ? PCRE_NOTEMPTY : 0;

      /* Advance to the position right after the last full match */
        start_offset = offsets[1];

    }
    while( global );

    ffree( offsets );
    if( !global )
    {
        ferite_str_set(VAS(retv), match, strlen(match), FE_CHARSET_DEFAULT );
    }
    if( assignment_array != NULL )
    {
        ffree( assignment_array );
    }
    FE_LEAVE_FUNCTION( retv );
}

FeriteVariable *ferite_execute_swap_regex( FeriteRegex *rgx, FeriteVariable *target, int global, FeriteNamespace *ns, FeriteScript *script )
{
    FeriteVariable *retv = NULL; /* rturn value */
    FeriteVariable *ptr = NULL;  /* so we can fudge with the arrays */
    FeriteVariable *ePtr = NULL, *ePtrRV = NULL;
    FeriteNamespaceBucket *nsb;
    int captured_str_cnt = 0; /* number of strings that will be captured in ()'s */
    int *offsets;         /* array of subpattern offsets */
    int size_offsets;     /* size of the array */
    int start_offset;     /* damn nation, this is the new start :) */
    int current_var = 0;  /* the global variable we need to update */
    int current_match = 0;
    int count = 0;        /* count the number of subexpressions */
    int i, swaps = 0, walkpos = 0, target_backtick;
    int endOfLastMatch = 0;
    char buf[10], *match, *replace_buffer, *tmpbuf = NULL;
    char *newstr = NULL;
    const char **stringlist /* the subparts () values */;
    int g_notempty = 0;
    int repbuf_len = 0;

    FE_ENTER_FUNCTION;

   /* if the regex is not compiled already -> compile it */
    if( rgx->compiled_re == NULL )
    {
        rgx->compiled_re = ferite_compile_regex( script, rgx->compile_buf, rgx->pcre_options );
        if( rgx->compiled_re == NULL )
        {
            retv = ferite_create_number_long_variable( script, "regex-exec-return", 0, FE_STATIC );
            FE_LEAVE_FUNCTION( retv );
        }
    }

    /* get the number of subparts */
    captured_str_cnt = pcre_info( rgx->compiled_re, NULL, NULL ) + 1;
    /* create an offset array */
    size_offsets = captured_str_cnt * 3;
    offsets = (int *)fmalloc(size_offsets * sizeof(int));

    /* we need to setup the global variable hash so that r1->r99 holds the number of subparts */
    for( i = 1; i <= (captured_str_cnt > script->last_regex_count ? captured_str_cnt : script->last_regex_count); i++ )
    {
        memset( buf, '\0', 10 );
        sprintf( buf, "r%d", i );
        nsb = ferite_namespace_element_exists( script, ns, buf );
        ptr = ( nsb == NULL ? NULL : nsb->data );
        if( ptr == NULL )
        {
            ptr = ferite_create_string_variable( script, buf, NULL, FE_ALLOC );
            MARK_VARIABLE_AS_FINALSET( ptr ); /* we make these global variables read only */
            ferite_register_ns_variable( script, ns, ptr );
        }
    }
    script->last_regex_count = captured_str_cnt;

    start_offset = 0;
    current_var = 1;
    current_match = 0;

    newstr = fmalloc( 1 );
    *newstr = '\0';

    do
    {
        /* execute regex */
        FUD(( "REGEX: Executing against \"%s\"\n", FE_STR2PTR(target) ));
        count = pcre_exec( rgx->compiled_re, rgx->extra_info, FE_STR2PTR(target), FE_STRLEN(target),
                           start_offset, g_notempty, offsets, size_offsets );

        /* check for too many substrings */
        if( count == 0 )
        {
            ferite_warning( script, "A match was found but too many substrings found.\n" );
            count = size_offsets / 3;
        }

        FUD(( "REGEX: match count = %d\n", count ));
        /* we matched something */
        if( count >= 0 )
        {
            match = FE_STR2PTR(target) + offsets[0];

            /* get the list of substrings */
            if( pcre_get_substring_list( FE_STR2PTR(target), offsets, count, &stringlist) < 0 )
            {
                ffree(offsets);
                ferite_warning( script, "Unable to obtain captured strings in regular expression.\n");
                retv = ferite_create_number_long_variable( script, "regex-exec-return", 0, FE_STATIC );
                FE_LEAVE_FUNCTION( retv );
            }

     /* need to setup the r%% variables we do this before doing the string stuff as
      * it allows us to use it in the eval operator */
     /* 0 = total matched string 1++ is subparts */
            for( i = 1; i < count; i++ )
            {
                memset( buf, '\0', 10 );
                sprintf( buf, "r%d", i /*current_var*/ );
                nsb = ferite_namespace_element_exists( script, ns, buf );
                ptr = nsb->data;
                ferite_str_set( VAS(ptr), (char *)stringlist[i], strlen((char *)stringlist[i]), FE_CHARSET_DEFAULT );
                FUD(( "Setting %s to %s\n", ptr->name, FE_STR2PTR(ptr) ));
                current_var++;
            }

     /* build the replace buffer */
            if( rgx->swap_buf != NULL )
            {
                repbuf_len = strlen(rgx->swap_buf) + (strlen(rgx->compile_buf) * 10) + 1;
                replace_buffer = fcalloc(repbuf_len, sizeof(char));
                for( i = 0, walkpos = 0; i < strlen( rgx->swap_buf ) && walkpos < repbuf_len; i++ )
                {
         /* check to see if we have a backtick */
                    if( rgx->swap_buf[i] == '\\' )
                    {
          /* account for escaped backticks */
                        if( rgx->swap_buf[i+1] == '\\' )
                        {
                            replace_buffer[walkpos++] = rgx->swap_buf[i];
                            i++;
                        }
                        else
                        {
           /* get the required backtick */
                            if( i < (strlen(rgx->swap_buf) - 1) )
                            {
                                target_backtick = rgx->swap_buf[i+1] - '0';
                                if( i < (strlen(rgx->swap_buf) - 2) )
                                  target_backtick = (rgx->swap_buf[i+2] <= '9' && rgx->swap_buf[i+2] >= '0' ? (target_backtick * 10) + rgx->swap_buf[i+2] - '0' : target_backtick);
                                if( target_backtick <= count )
                                {
                                    memcpy( replace_buffer + walkpos, stringlist[target_backtick], strlen( stringlist[target_backtick] ) );
                                    walkpos += strlen(stringlist[target_backtick]);
                                    if( target_backtick > 9 )
                                      i += 2;
                                    else
                                      i++;
                                }
                            }
                            else
                            {
                                break;
                            }
                        }
                    }
                    else
                    {
                        replace_buffer[walkpos++] = rgx->swap_buf[i];
                    }
                }
                replace_buffer[walkpos] = '\0'; /* terminate the string */
                FUD(("RGX: Walkpos %d, repbuf_len %d, strlen, %d, content: ``%s''\n", walkpos, repbuf_len, strlen( replace_buffer ), replace_buffer ));
            }
            else
            {
                replace_buffer = fstrdup("");
            }

     /*   printf( "Swap at index %d len %d\n", offsets[0], strlen(stringlist[0]) ); */

     /* ok this is what we do:
      *   setup the rx variables
      *   call eval
      *   use it's return as a string */
            if( rgx->fergx_options & F_RGX_EVAL_SWP )
            {
                FUD(( ">>>> Swap evaluation <<<< '%s' \n", replace_buffer ));
                ePtr = ferite_create_string_variable_from_ptr( script, "regex-eval", replace_buffer, strlen(replace_buffer), FE_CHARSET_DEFAULT, FE_STATIC );

      /* we dont need this any more */
                ffree( replace_buffer );
                ePtrRV = ferite_op_eval( script, ePtr );
                FUD(( "eval returned: %s, %s\n", ePtrRV->name, ferite_variable_id_to_str( script, ePtrRV->type ) ));
                if( ePtrRV->type != F_VAR_STR )
                {
                    ferite_warning( script, "Regular Expression Eval does not return a string - using \"\"\n" );
                    replace_buffer = fstrdup("");
                }
                else
                {
                    replace_buffer = fstrdup(FE_STR2PTR(ePtrRV));
                }
      /* clean up afterwards */
                ferite_variable_destroy( script, ePtr );
                ferite_variable_destroy( script, ePtrRV );
            }

     /* grow the buffer */
            FUD(( "New Str Length: %d\n", strlen( newstr ) + (offsets[0] - endOfLastMatch) + strlen( replace_buffer ) + 10 ));
            tmpbuf = fcalloc( strlen( newstr ) + (offsets[0] - endOfLastMatch) + strlen( replace_buffer ) + 10, sizeof(char) );
            strcpy( tmpbuf, newstr );
            ffree( newstr );

            newstr = tmpbuf;

     /* ok what we do here is find out the offset within the original string and then replace that with
      * the swap buffer */
            strncat( newstr, FE_STR2PTR(target) + endOfLastMatch, offsets[0] - endOfLastMatch );
            strcat( newstr, replace_buffer );

            FUD(( "Real Length: %d\n", strlen(newstr) ));
     /* new offset */
            endOfLastMatch = offsets[0] + strlen(stringlist[0]);
     /*   printf( "newstr: %s %d\n", newstr, endOfLastMatch );*/

            swaps++;

            ffree( replace_buffer );

            FUD(( "RGX: Captured String count: %d\n", captured_str_cnt ));
            for( i = 0; i < captured_str_cnt; i++ )
            {
                FUD(( "RGX: STRLIST: [%d] %s\n", i, stringlist[i] ));
            }
            if( ferite_use_mm_with_pcre )
            {
                ffree( stringlist );
            }
            else
              pcre_free( stringlist );
        }
        else /* we didn't match something */
        {
            match = "\0";
            if (g_notempty != 0 && start_offset < FE_STRLEN(target) )
            {
                offsets[0] = start_offset;
                offsets[1] = start_offset + 1;
            }
            else
              break;
        }

        g_notempty = ( offsets[1] == offsets[0] ) ? PCRE_NOTEMPTY : 0;

      /* Advance to the position right after the last full match */
        start_offset = offsets[1];

    }
    while( global );

   /* cleaning up */
    if( endOfLastMatch < FE_STRLEN(target) )
    {
        tmpbuf = fcalloc( strlen( newstr ) + FE_STRLEN(target) + endOfLastMatch + 2, sizeof(char) );
        strcpy( tmpbuf, newstr );
        ffree( newstr );
        newstr = tmpbuf;
        strcat( newstr, FE_STR2PTR(target) + endOfLastMatch );
    }
    ferite_str_set( VAS(target), newstr, strlen(newstr), FE_CHARSET_DEFAULT );
    ffree( newstr );
    ffree( offsets );

    retv = ferite_create_number_long_variable( script, "regex-swap-exec-return", swaps, FE_STATIC );
    MARK_VARIABLE_AS_DISPOSABLE( retv );
    FE_LEAVE_FUNCTION( retv );
}

/**
 * !end
 */
