/*
 * Copyright (C) 1999-2004 Chris Ross
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * o Redistributions of source code must retain the above copyright notice, this
 *   list of conditions and the following disclaimer.
 * o Redistributions in binary form must reproduce the above copyright notice,
 *   this list of conditions and the following disclaimer in the documentation
 *   and/or other materials provided with the distribution.
 * o Neither the name of the ferite software nor the names of its contributors may
 *   be used to endorse or promote products derived from this software without
 *   specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include <stdio.h>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/stat.h>
#ifndef WIN32
# include <unistd.h>
#endif
#include <string.h>

#ifndef APHEX_AS_COMPONENT
#include "../../../config.h"
#endif

#include "aphex.h"

/***************************************************
 * THREAD
 ***************************************************/
AphexThread *aphex_thread_create()
{
    AphexThread *thread = malloc(sizeof(AphexThread));
    return thread;
}

void aphex_thread_destroy( AphexThread *thread )
{
    if( thread != NULL )
    {
        if( thread->running )
          aphex_thread_stop( thread );
        free( thread );
    }
}

int aphex_thread_start( AphexThread *thread, void *(*start_routine)(void *), void *arg, int detach )
{
    int retval = 0;

    if( thread != NULL && start_routine != NULL )
    {
#ifdef USE_PTHREAD
		thread->running = 1;
        retval = pthread_create( &(thread->thread), NULL, start_routine, arg );
        if( detach )
          pthread_detach( thread->thread );
#endif
        if( retval != 0 )
        {
            printf( "aphex: ERROR: Not enough system resources to create thread\n" );
            return -1;
        }
    }
    return 0;
}

void aphex_thread_stop( AphexThread *thread )
{
    if( thread != NULL )
    {
        thread->running = 0;
#ifdef USE_PTHREAD
        pthread_cancel( thread->thread );
#endif
    }
}

#ifdef WIN32
void usleep( int useconds )
{
    if (useconds > 500)
      Sleep ((useconds+500)/1000);
    else if (useconds > 0)
      Sleep (1);
    else
      Sleep (0);
}
#endif

void aphex_thread_sleep( AphexThread *thread, int msecs )
{
    usleep( msecs * 1000 );
}

void aphex_thread_join( AphexThread *thread )
{
    if( thread != NULL )
    {
#ifdef USE_PTHREAD
		pthread_join( thread->thread, NULL );
#endif
        thread->running = 0;
    }
}

void aphex_thread_exit( AphexThread *thread, void *rval )
{
    if( thread != NULL )
    {
#ifdef USE_PTHREAD
		pthread_exit( rval );
#endif
        thread->running = 0;
    }
}

/***************************************************
 * MUTEX
 ***************************************************/
AphexMutex *aphex_mutex_create()
{
    AphexMutex *mutex = malloc(sizeof(AphexMutex));
#ifdef USE_PTHREAD
    pthread_mutex_init( &(mutex->mutex), NULL );
# if defined(USING_FAKE_RECURSIVE_MUTEX)
    mutex->count = -1;
    memset( &(mutex->owner), '\0', sizeof(pthread_t) );
# endif
#endif
    mutex->recursive = 0;
    return mutex;
}

AphexMutex *aphex_mutex_recursive_create()
{
    AphexMutex *mutex = malloc(sizeof(AphexMutex));

#ifdef USE_PTHREAD
#if defined(USING_FAKE_RECURSIVE_MUTEX)
    mutex->count = 0;
    mutex->owner = NULL;
    pthread_cond_init( &mutex->cond, NULL );
    pthread_mutex_init( &mutex->mutex, NULL );
#else
    pthread_mutexattr_init( &mutex->attr );
    pthread_mutexattr_settype( &mutex->attr, PTHREAD_MUTEX_RECURSIVE);
    pthread_mutex_init( &mutex->mutex, &mutex->attr );
    pthread_mutexattr_destroy( &mutex->attr );
#endif
    mutex->recursive = 1;
#endif

    return mutex;
}

void aphex_mutex_destroy( AphexMutex *mutex )
{
    if( mutex != NULL )
    {
#ifdef USE_PTHREAD
#if defined(USING_FAKE_RECURSIVE_MUTEX)
        pthread_cond_destroy( &mutex->cond );
#else
#endif
        pthread_mutex_destroy( &mutex->mutex );
        free( mutex );
#endif
    }
}

int aphex_mutex_lock( AphexMutex *mutex )
{
#ifdef USE_PTHREAD
    pthread_t self = pthread_self();

    if( mutex != NULL )
    {
        if( pthread_mutex_lock( &mutex->mutex ) == -1 )
          return -1;

#if defined(USING_FAKE_RECURSIVE_MUTEX)
        if( mutex->recursive == 1 )
        {
            while (1)
            {
                if( mutex->owner == self )
                {
                    mutex->count++;
                    break;
                }
                else if( mutex->owner == NULL )
                {
                    mutex->owner = self;
                    mutex->count = 1;
                    break;
                }
                else
                {
                    if( pthread_cond_wait( &mutex->cond, &mutex->mutex ) == -1 )
                      return -1;
                }
            }
            pthread_mutex_unlock( &mutex->mutex );
        }
#endif
    }
#endif
    return 0;
}

int aphex_mutex_unlock( AphexMutex *mutex )
{
    if( mutex != NULL )
    {
#ifdef USE_PTHREAD
#if defined(USING_FAKE_RECURSIVE_MUTEX)
        if( mutex->recursive == 1 )
        {
            if( pthread_mutex_lock (&mutex->mutex) == -1 )
              return -1;
            
            mutex->count--;
            if( mutex->count == 0 )
            {
                mutex->owner = NULL;
                pthread_cond_signal( &mutex->cond );
            }
        }
#endif
        pthread_mutex_unlock( &mutex->mutex );
#endif
    }
    return 0;
}

/***************************************************
 * EVENT
 ***************************************************/
AphexEvent *aphex_event_create()
{
    AphexEvent *event = malloc(sizeof(AphexEvent));
#ifdef USING_PTHREAD
    if(pthread_cond_init( &(event->cond), NULL) != 0){
        free(event);
	return NULL;
    }
    if(pthread_mutex_init( &(event->mutex), NULL ) != 0){
        free(event);
	return NULL;
    }    
#endif
    return event;
}

void aphex_event_destroy( AphexEvent *event )
{
    if( event != NULL )
    {
#ifdef USING_PTHREAD
         pthread_cond_destroy( &event->cond );
         pthread_mutex_destroy( &event->mutex );
#endif
         free(event);
    }
}

int aphex_event_signal( AphexEvent *event )
{
#ifdef USING_PTHREAD
    pthread_mutex_lock(&(event->mutex));
    pthread_cond_signal(&(event->cond));
    pthread_mutex_unlock(&(event->mutex));
#endif
    return 0;
}

int aphex_event_wait( AphexEvent *event )
{
#ifdef USING_PTHREAD
    pthread_mutex_lock(&(event->mutex));
    pthread_cond_wait(&(event->cond), &(event->mutex));
    pthread_mutex_unlock(&(event->mutex));
#endif
    return 0;
}

int aphex_event_timedwait( AphexEvent *event , int seconds)
{
    int t_ret = 0;

#ifdef USING_PTHREAD
    struct timespec   ts;
    struct timeval    tp;

    gettimeofday(&tp, NULL);
    ts.tv_sec  = tp.tv_sec;
    ts.tv_nsec = tp.tv_usec * 1000;
    ts.tv_sec += seconds;

    pthread_mutex_lock(&(event->mutex));
    t_ret = pthread_cond_timedwait(&(event->cond), &(event->mutex), &ts);
    pthread_mutex_unlock(&(event->mutex));
#endif
    
    if (t_ret != 0)
        return 1;
    else
        return 0;
}

