/*
 * Copyright (C) 2000-2004 Chris Ross and various contributors
 * Copyright (C) 1999-2000 Chris Ross
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * o Redistributions of source code must retain the above copyright notice, this
 *   list of conditions and the following disclaimer.
 * o Redistributions in binary form must reproduce the above copyright notice,
 *   this list of conditions and the following disclaimer in the documentation
 *   and/or other materials provided with the distribution.
 * o Neither the name of the ferite software nor the names of its contributors may
 *   be used to endorse or promote products derived from this software without
 *   specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include "ferite.h"

#ifdef WIN32
# include "snprintf.h" /* This is so that we have somethings */
#endif

/**
 * !group Error System
 * !description There are a number of things a lot of languages lack and that is error handling.
 *              The functions in this group allow for the raising of exceptions during runtime, 
 *              and compile time.
 */

/**
 * !function ferite_raise_script_error
 * !declaration void ferite_raise_script_error( FeriteScript *script, int err, char *fmt, ... )
 * !brief Raise an exception within the ferite engine.
 * !param FeriteScript *script The running script
 * !param int err    The error code
 * !param char *fmt    The format of the error string
 * !description Use the same formating codes as printf with this function
 */
void ferite_raise_script_error( FeriteScript *script, int err, char *fmt, ... )
{
    FeriteNamespaceBucket *nsb;
    FeriteVariable *gerr, *newError;
    FeriteVariable *errstr, *erno;
    char *msg;
    va_list ap;

    FE_ENTER_FUNCTION;
    msg = fmalloc( 4096 );
    va_start( ap, fmt );
    vsprintf( msg, fmt, ap );
    FUD(("ERROR RAISED: %s %d\n", msg, err ));

    nsb = ferite_namespace_element_exists( script, script->mainns, "err" );
    FE_ASSERT( nsb && nsb->type == FENS_VAR );
    gerr = nsb->data;
    script->error_state = FE_ERROR_THROWN;

    if( VAO(gerr) == NULL )
    {
        nsb = ferite_namespace_element_exists( script, script->mainns, "Error" );
        if( nsb == NULL )
        {
            FE_LEAVE_FUNCTION( NOWT );
            exit(1);
        }
        newError = ferite_new_object( script, nsb->data, NULL );
        VAO(gerr) = VAO(newError);
        VAO(gerr)->refcount++;
        ferite_variable_destroy( script, newError );
    }

    errstr = ferite_object_get_var( script, VAO(gerr), "str" );
    ferite_str_set( VAS(errstr), msg, strlen(msg), FE_CHARSET_DEFAULT );
    ffree( msg );

    erno = ferite_object_get_var( script, VAO(gerr), "num" );
    VAI(erno) = err;
    FE_LEAVE_FUNCTION( NOWT );
}

/**
 * !function ferite_verror
 * !declaration void ferite_verror( FeriteScript *script, char *errormsg, va_list *ap )
 * !brief Raise an error
 * !param FeriteScript *script The script
 * !param int err The error number associated with the error
 * !param char *errormsg The error with formating codes in it
 * !param va_list *ap The list of arguments
 */
void ferite_verror( FeriteScript *script, int err, char *errormsg, va_list *ap )
{
    char msg[1024];

    FE_ENTER_FUNCTION;

    if( script == NULL )
    {
        vprintf(errormsg, *ap );
        FE_LEAVE_FUNCTION( NOWT );
    }
    if( script->error == NULL )
      script->error = ferite_buffer_new( 0 );

    ferite_buffer_add_str( script->error, "Error: " );

    /* if( ferite_is_executing( script ) )
        ferite_buffer_printf( script->error, "[%s:%d] ", script->current_op_file, script->current_op_line ); */
    ferite_buffer_vprintf( script->error, errormsg, ap );
    if( script->error_state != FE_ERROR_THROWN )
	{
		if( ferite_is_executing( script ) )
		{
			vsnprintf( msg, 1024, errormsg, *ap );
			ferite_raise_script_error( script, err, msg );
		}
		script->error_state = FE_ERROR_THROWN;
	}

    FE_LEAVE_FUNCTION( NOWT );
}
/**
 * !function ferite_error
 * !declaration void ferite_error( FeriteScript *script, char *errormsg, ... )
 * !brief Throw an error within the engine
 * !param FeriteScript *script   The scipt we are using
 * !param int err The error number associated with the error
 * !param char *errormsg The error information
 * !param . .. The formating values
 */
void ferite_error( FeriteScript *script, int err, char *errormsg, ... )
{
    va_list ap;

    FE_ENTER_FUNCTION;
    va_start( ap, errormsg );
    ferite_verror( script, err, errormsg, &ap);
    va_end( ap );
    FE_LEAVE_FUNCTION( NOWT );
}

/**
 * !function ferite_vwarning
 * !declaration void ferite_vwarning( FeriteScript *script, char *errormsg, ... )
 * !brief Display a warning message. This does not cause an exception
 * !param FeriteScript *script   The current script
 * !param char *errormsg The warning to be displayed
 * !param va_list *ap The formatting values
 */
void ferite_vwarning( FeriteScript *script, char *errormsg, va_list *ap )
{

    FE_ENTER_FUNCTION;

    if( script == NULL )
    {
        printf("ferite_warning(): script was called with NULL, this shouldn't happen\n");
        vprintf(errormsg, *ap );
#ifdef DEBUG
        printf("ferite_warning(): sleeping for gdb interruption for 10 seconds\n");
        sleep(10);
#endif
        FE_LEAVE_FUNCTION( NOWT );
    }

    if( script->warning == NULL )
      script->warning = ferite_buffer_new( 0 );

    ferite_buffer_add_str( script->warning, "Warning: " );
    if( ferite_is_executing( script ) )
      ferite_buffer_printf( script->warning, "[%s:%d] ", script->current_op_file, script->current_op_line );
    ferite_buffer_vprintf( script->warning, errormsg, ap );

    FE_LEAVE_FUNCTION( NOWT );
}

/**
 * !function ferite_warning
 * !declaration void ferite_warning( FeriteScript *script, char *errormsg, ... )
 * !brief Display a warning message. This does not cause an exception
 * !param FeriteScript *script   The current script
 * !param char *errormsg The warning to be displayed
 * !param . .. The formatting values
 */
void ferite_warning( FeriteScript *script, char *errormsg, ... )
{
    va_list ap;

    FE_ENTER_FUNCTION;
    va_start( ap, errormsg );

    ferite_vwarning( script, errormsg, &ap );

    va_end( ap );
    FE_LEAVE_FUNCTION(NOWT);
}

/**
 * !function ferite_get_error_log
 * !declaration char *ferite_get_error_log( FeriteScript *script )
 * !brief Get a null terminated string containing the error and warning logs on a script
 * !param FeriteScript *script The script to get the errror logs from
 * !return A null terminated string, you will need to ffree the string when done to prevent memory leak
 */
char *ferite_get_error_log( FeriteScript *script )
{
    int err_size = 0, warn_size = 0;
    char *msg, *err_ptr, *warn_ptr;

    FE_ENTER_FUNCTION;
    if( script->error )
      err_ptr = ferite_buffer_get( script->error, &err_size );
    else
      err_ptr = fstrdup("");
    if( script->warning )
      warn_ptr = ferite_buffer_get( script->warning, &warn_size );
    else
      warn_ptr = fstrdup("");
    msg = fmalloc( err_size + warn_size + 1 );
    strcpy( msg, warn_ptr );
    strcat( msg, err_ptr );
    ffree( err_ptr );
    ffree( warn_ptr );
    FE_LEAVE_FUNCTION( msg );
}

/**
 * !function ferite_get_error_string
 * !declaration char *ferite_get_error_string( FeriteScript *script )
 * !brief Get a null terminated string containing the error log
 * !param FeriteScript *script The script whose errors are required
 * !return  A null terminated string, you will need to ffree the string when done to prevent memory leak
 */
char *ferite_get_error_string( FeriteScript *script )
{
    char *msg;
    FE_ENTER_FUNCTION;
    if( script->error )
      msg = ferite_buffer_get( script->error, NULL );
    else
      msg = fstrdup("");
    FE_LEAVE_FUNCTION( msg );
}

/**
 * !function ferite_get_warning_string
 * !declaration char *ferite_get_warning_string( FeriteScript *script )
 * !brief Get a null terminated string containing the warning log
 * !param FeriteScript *script The script whose warnings are required
 * !return  A null terminated string, you will need to ffree the string when done to prevent memory leak
 */
char *ferite_get_warning_string( FeriteScript *script )
{
    char *msg;
    FE_ENTER_FUNCTION;
    if( script->warning )
      msg = ferite_buffer_get( script->warning, NULL );
    else
      msg = fstrdup("");
    FE_LEAVE_FUNCTION( msg );
}

/**
 * !function ferite_reset_warnings
 * !declaration void ferite_reset_warnings( FeriteScript *script )
 * !brief Reset any warnings on the script
 * !param FeriteScript *script The script to check
 */
void ferite_reset_warnings( FeriteScript *script )
{
    FE_ENTER_FUNCTION;
    if( script->warning != NULL )
    {
	ferite_buffer_delete( script->warning );
	script->warning = NULL;
    }
    FE_LEAVE_FUNCTION(NOWT);
}

/**
 * !function ferite_reset_errors
 * !declaration void ferite_reset_errors( FeriteScript *script )
 * !brief Reset any errors on the script
 * !param FeriteScript *script The script to check
 */
void ferite_reset_errors( FeriteScript *script )
{
    FE_ENTER_FUNCTION;
    if( script->error != NULL )
    {
		ferite_buffer_delete( script->error );
		script->error = NULL;
		script->error_state = 0;
    }
    FE_LEAVE_FUNCTION(NOWT);
}

/**
 * !function ferite_has_compile_error
 * !declaration int ferite_has_compile_error( FeriteScript *script )
 * !brief Check to see if the script has had a compilation error
 * !param FeriteScript *script The script to check
 * !return FE_TRUE if a compilation error occured, FE_FALSE otherwise
 */
int ferite_has_compile_error( FeriteScript *script )
{
    FE_ENTER_FUNCTION;
    FE_LEAVE_FUNCTION( ( script->error == NULL ) ? FE_FALSE : FE_TRUE );
}

/**
 * !function ferite_has_warnings
 * !declaration int ferite_has_warnings( FeriteScript *script )
 * !brief Check to see if the script has warnings
 * !param FeriteScript *script The script to check
 * !return FE_TRUE if there are warnings, FE_FALSE otherwise
 */
int ferite_has_warnings( FeriteScript *script )
{
    FE_ENTER_FUNCTION;
    FE_LEAVE_FUNCTION( (script->warning == NULL) ? FE_FALSE : FE_TRUE );
}

/**
 * !function ferite_has_runtime_error
 * !declaration int ferite_has_runtime_error( FeriteScript *script )
 * !brief Check to see if the script has had a runtime error
 * !param FeriteScript *script The script to check
 * !return FE_TRUE if a compilation error occured, FE_FALSE otherwise
 */
int ferite_has_runtime_error( FeriteScript *script )
{
    FE_ENTER_FUNCTION;
    FE_LEAVE_FUNCTION( ( script->error_state == FE_ERROR_THROWN ) ? FE_TRUE : FE_FALSE );
}

/**
 * !function ferite_init_error_system
 * !declaration void ferite_init_error_system( FeriteScript *script, FeriteNamespace *ns )
 * !brief Setup the special error handling class on a script
 * !param FeriteScript *script The current script
 * !param FeriteNamespace *ns     The namespace in which the class should be registered
 */
void ferite_init_error_system( FeriteScript *script, FeriteNamespace *ns )
{
    FeriteClass *ferite_error_class = NULL;
    FeriteVariable *var, *errobj = NULL;
    FeriteNamespaceBucket *nsb = NULL;
    
    FE_ENTER_FUNCTION;
    ferite_error_class = ferite_register_class( script, ns, "Error" );
    ferite_register_class_variable( script, ferite_error_class, ferite_create_number_long_variable( script, "num", 0, FE_STATIC ), 0 );
    ferite_register_class_variable( script, ferite_error_class, ferite_create_string_variable( script, "str", NULL, FE_STATIC ), 0 );
    
    nsb = ferite_find_namespace( script, script->mainns, "err", FENS_VAR );
    if( nsb != NULL )
    {
	errobj = ferite_build_object( script, ferite_error_class );
	var = nsb->data;
	VAO(var) = VAO(errobj);
	VAO(errobj) = NULL;
	ferite_variable_destroy( script, errobj );
    }
    
    FE_LEAVE_FUNCTION(NOWT);
}

/**
 * !function ferite_set_error
 * !declaration void ferite_set_error( FeriteScript *script, int num, char *fmt, ... )
 * !brief Same as ferite_error except this wont raise an exception at runtime
 */
void ferite_set_error( FeriteScript *script, int num, char *fmt, ... )
{
    FeriteNamespaceBucket *nsb;
    FeriteVariable *gerr, *newError;
    FeriteVariable *errstr, *erno;
    va_list ap;
    char *buf;

    FE_ENTER_FUNCTION;

    if( !script->is_being_deleted )
    {
        buf = fmalloc( 4096 );
        va_start( ap, fmt );
        vsprintf( buf, fmt, ap );

        nsb = ferite_namespace_element_exists( script, script->mainns, "err" );
        FE_ASSERT( nsb && nsb->type == FENS_VAR );
        gerr = nsb->data;

        if( VAO(gerr) == NULL )
        {
            nsb = ferite_namespace_element_exists( script, script->mainns, "Error" );
            newError = ferite_new_object( script, nsb->data, NULL );
            VAO(gerr) = VAO(newError);
            VAO(gerr)->refcount++;
            ferite_variable_destroy( script, newError );
        }

        errstr = ferite_object_get_var( script, VAO(gerr), "str" );
        ferite_str_set( VAS(errstr), buf, strlen(buf), FE_CHARSET_DEFAULT );

        erno = ferite_object_get_var( script, VAO(gerr), "num" );
        VAI(erno) = num;

        ffree( buf );
        va_end( ap );
    }
    FE_LEAVE_FUNCTION( NOWT );
}
/* This function will never return, set ferite_assert_debug to generate
 * a segfault to get a backtrace */
int ferite_assert_debug = 0;
void ferite_assert( char *fmt, ... )
{
    char *p = NULL;
    va_list ap;
    va_start( ap, fmt );
    ferite_vwarning( NULL, fmt, &ap );
    va_end( ap );
    if( ferite_assert_debug )
      *p = '\0';
    exit( -1 );
}

/**
 * !end
 */
