/*
 * Copyright (C) 1999-2004 Chris Ross
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * o Redistributions of source code must retain the above copyright notice, this
 *   list of conditions and the following disclaimer.
 * o Redistributions in binary form must reproduce the above copyright notice,
 *   this list of conditions and the following disclaimer in the documentation
 *   and/or other materials provided with the distribution.
 * o Neither the name of the ferite software nor the names of its contributors may
 *   be used to endorse or promote products derived from this software without
 *   specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

/*
 * a heap allocator based on jedi mind powers. and the algorithm described
 * in perl's malloc.c
 */

#include "ferite.h"
#include "aphex.h"
#include <math.h>

/* various constants that can be tweaked. BLOCK_COUNT must be bigger then NBUCKETS! */
#define NBUCKETS        30               /* number of chains to have */
#define BLOCK_COUNT     (NBUCKETS + 2)   /* number of chunks to allocate each time */
#define MAGIC           42               /* the answer to life, the universe, everything and whether a pointer
                                          * is correct */

/* macors to keep real world stats correct */
#define rmalloc( size )         malloc( size ); real_stats.malloc_c++;
#define rcalloc( size, block )  calloc( size, block ); real_stats.calloc_c++;
#define rrealloc( ptr, size )   realloc( ptr, size ); real_stats.realloc_c++;
#define rfree( ptr )            free( ptr ); real_stats.free_c++;

/* these get the header and the body from a pointer */
#define PTR_GET_HEADER( ptr )   (FeriteMemoryChunkHeader*)((char *)ptr - sizeof(FeriteMemoryChunkHeader))
#define PTR_GET_BODY( ptr )     (void*)((char *)ptr + sizeof(FeriteMemoryChunkHeader))

/* threading stuff */
#ifdef THREAD_SAFE
AphexMutex *ferite_jedi_memory_lock = NULL;
# define LOCK_MEMORY     aphex_mutex_lock( ferite_jedi_memory_lock )
# define UNLOCK_MEMORY   aphex_mutex_unlock( ferite_jedi_memory_lock )
#else
# define LOCK_MEMORY
# define UNLOCK_MEMORY
#endif

typedef union ferite_memory_chunk_header FeriteMemoryChunkHeader;
union ferite_memory_chunk_header
{
    FeriteMemoryChunkHeader *next;
    struct
    {
        unsigned char index; /* bucket for index, < NBUCKETS */
        unsigned char magic; /* magic -> so we can check that the pointer is allocated and
                              * we haven't gone painfully wrong somewhere */
    }
    assigned_info;
    double alignment; /* force dword alignment */
};
extern int ferite_pow_lookup[];
FeriteMemoryChunkHeader *free_chunks[NBUCKETS];
FeriteMemoryChunkHeader *big_chunks = NULL;

typedef struct
{
    long malloc_c;
    long calloc_c;
    long realloc_c;
    long free_c;
}
FeriteMemoryStats;

FeriteMemoryStats       real_stats; /* how many times we hit the OS */
FeriteMemoryStats       vrtl_stats; /* how many times we hit the local heap */

void ferite_jedi_memory_init(void)
{
    int i = 0;

    FE_ENTER_FUNCTION;
    real_stats.malloc_c = 0;
    real_stats.calloc_c = 0;
    real_stats.realloc_c = 0;
    real_stats.free_c = 0;

    vrtl_stats.malloc_c = 0;
    vrtl_stats.calloc_c = 0;
    vrtl_stats.realloc_c = 0;
    vrtl_stats.free_c = 0;

    if( !ferite_hide_mem_use )
    {
#ifdef FERITE_MEM_DEBUG
        printf( "Jedi Memory System Active\n" );
#endif
    }

    for( i = 0; i < NBUCKETS; i++ )
        free_chunks[i] = NULL;

#ifdef THREAD_SAFE
    ferite_jedi_memory_lock = aphex_mutex_create();
#endif
    FE_LEAVE_FUNCTION(NOWT);
}

void ferite_jedi_memory_deinit(void)
{
    void *ptr = NULL;

    FE_ENTER_FUNCTION;
    while( big_chunks != NULL )
    {
        ptr = big_chunks->next;
        rfree( big_chunks );
        big_chunks = ptr;
    }
    if( !ferite_hide_mem_use ) /* 2/3's of all statistics are made up. unfortunatly not here */
    {
        printf( "Ferite Memory Usage Statistics (jedi)\n" );
        printf( " |- Virtual.. %ld mallocs, %ld callocs, %ld reallocs, %ld frees",
                vrtl_stats.malloc_c, vrtl_stats.calloc_c, vrtl_stats.realloc_c, vrtl_stats.free_c );
        printf( " [%ld block%s still allocated]\n", (vrtl_stats.malloc_c + vrtl_stats.calloc_c) - vrtl_stats.free_c,
                (((vrtl_stats.malloc_c + vrtl_stats.calloc_c) - vrtl_stats.free_c) == 1 ? "" : "s"));
        printf( " `- Real..... %ld mallocs, %ld callocs, %ld reallocs, %ld frees",
                real_stats.malloc_c, real_stats.calloc_c, real_stats.realloc_c, real_stats.free_c );
        printf( " [%ld block%s still allocated]\n", (real_stats.malloc_c + real_stats.calloc_c) - real_stats.free_c,
                (((real_stats.malloc_c + real_stats.calloc_c) - real_stats.free_c) == 1 ? "" : "s"));
    }
#ifdef THREAD_SAFE
    aphex_mutex_destroy( ferite_jedi_memory_lock );
#endif
    FE_LEAVE_FUNCTION(NOWT);
}

void ferite_jedi_dump_memory( int bucket )
{
#ifdef DEBUG
    int i = 0;
#endif
    FeriteMemoryChunkHeader *hdr;

   /* dump a free bucket chain... */
    FUD(( "================================ Table Dump ==================================\n" ));
    hdr = free_chunks[bucket];
    while( hdr != NULL ) /* keep looping until we hit the end... */
    {
        FUD(( " [%d]> hdr=%p, hdr->next=%p, %ld, sizeof(Header): %d, pow[%d]=%ld\n", ++i, hdr, hdr->next,
              (long)((void*)hdr->next) - (long)((void*)hdr), sizeof(FeriteMemoryChunkHeader), bucket, ferite_pow_lookup[bucket] ));
        hdr = hdr->next;
    }
}

void *ferite_jedi_malloc( size_t size, char *file, int line )
{
    char *return_ptr = NULL;
    int target_bucket = 3;
    FeriteMemoryChunkHeader *ptr = NULL;

  /* get the correct bucket size -> we should check the wastage and have an odd's and
   * sods chain we allocate random chunks from -> but thats for another day... */
    while( size > (size_t)ferite_pow_lookup[target_bucket] )
      target_bucket++;

    FUD(( "Target bucket for data of %d is %d(%ld)\n", size, target_bucket, ferite_pow_lookup[target_bucket] ));

    LOCK_MEMORY;
   /* check to see if we have memory :), if not go a eat some :) */
    if( free_chunks[target_bucket] == NULL )
      ferite_jedi_morecore( target_bucket );

   /* oooh we are up the creek, so to say */
    if( (ptr = free_chunks[target_bucket]) == NULL )
    {
#ifdef FERITE_MEM_DEBUG
        fprintf( stderr, "JEDI: Out of memory. Oh dear. Oh dear. Go out and buy some more :)\n" );
#endif
        UNLOCK_MEMORY;
        return NULL;
    }

   /* rebuild the chain */
    FUD(( "free_chunks[target_bucket]: %p\n", free_chunks[target_bucket] ));
    FUD(( "new free_chunks:            %p\n", ptr->next ));
    free_chunks[target_bucket] = ptr->next;

   /* setup the information for the wild goose chase :) */
    ptr->assigned_info.index = target_bucket;
    ptr->assigned_info.magic = MAGIC;
   /* get the memory chunk */
    return_ptr = PTR_GET_BODY(ptr);
    FUD(( "returning: %p, %d\n", return_ptr, (int)((void *)return_ptr) - (int)((void *)ptr) ));
    vrtl_stats.malloc_c++;
    UNLOCK_MEMORY;
    return return_ptr;
}

void *ferite_jedi_calloc( size_t size, size_t blk_size, char *file, int line )
{ /* surely the easist calloc *ever*? ;) */

    void *ptr = NULL;

    size *= blk_size;
    ptr = ferite_jedi_malloc( size, __FILE__, __LINE__ );
    vrtl_stats.malloc_c--;
    vrtl_stats.calloc_c++;
    memset( ptr, 0, size );
    return ptr;
}

void *ferite_jedi_realloc( void *ptr, size_t size )
{
    FeriteMemoryChunkHeader *hdr = NULL;
    long old_size = 0, old_index = 0;
    void *new_ptr = NULL;
    int target_bucket = 3;

    if( ptr != NULL )
    { /* we want to copy the old memory to the new memory */
        hdr = PTR_GET_HEADER( ptr );
        old_size = ferite_pow_lookup[hdr->assigned_info.index];
        old_index = hdr->assigned_info.index;

      /* we now have two options, either keep the same buffer, or copy to a
       * bigger one */
        while( size > (size_t)ferite_pow_lookup[target_bucket] )
          target_bucket++;

        if( target_bucket == old_index ) /* we already have enough space */
          return ptr;

      /* this is an evil slow method of realloc - but necessary */
        new_ptr = ferite_jedi_malloc( size, __FILE__, __LINE__ );
        memcpy( new_ptr, ptr, old_size );
      /* now we move the older ptr onto it's old block */
        LOCK_MEMORY;
        hdr->next = free_chunks[old_index];
        free_chunks[old_index] = hdr;
        vrtl_stats.malloc_c--;
        vrtl_stats.realloc_c++;
        UNLOCK_MEMORY;
    }
    return new_ptr;
}

void ferite_jedi_free( void *ptr, char *file, int line )
{
    FeriteMemoryChunkHeader *hdr = NULL;
    int bucket = 0;

    LOCK_MEMORY;
    if( ptr != NULL )
    {
        hdr = PTR_GET_HEADER(ptr);
        FUD(( "freeing %p (hdr=%p,diff=%ld)\n", ptr, hdr, (long)((void*)ptr) - (long)((void*)hdr) ));
        if( hdr->assigned_info.magic == MAGIC )
        {
            bucket = hdr->assigned_info.index;
     /* relink the chain */
            FUD(( "Setting next as %p\n", free_chunks[bucket] ));
            hdr->next = free_chunks[bucket];
            FUD(( "Setting new header as %p\n", hdr ));
            free_chunks[bucket] = hdr;
            vrtl_stats.free_c++;
        }
        else /* fubar'd pointer -> we must know about these */
        {
	    fprintf( stderr, "JEDI: expecting %d for magic, but got %d (addy %p) (culprit %s, line %d)\n", MAGIC, hdr->assigned_info.magic, ptr, file, line );
            fprintf( stderr, "JEDI: MEM DATA: `%s'\n", (char *)ptr );
	}
    }
#ifdef FERITE_MEM_DEBUG /* only if we are debugging shouyld we hear these */
    else
      fprintf( stderr, "JEDI: trying to free a NULL pointer in %s on line %d\n", file, line );
#endif
    UNLOCK_MEMORY;
}
//int jedi_count = 0;
void ferite_jedi_morecore( int bucket )
{
    int i = 0;
    void *new_block = NULL;
    int chunk_size = ferite_pow_lookup[bucket] + sizeof(FeriteMemoryChunkHeader);
    FeriteMemoryChunkHeader *hdr;
    int actual_block_count = BLOCK_COUNT - bucket;
    //printf("%d\n",jedi_count++);
    FUD(( "in more core -> allocating for %d\n", bucket ));

   /* check to see if we have actually run out of space on bucket list */
    if( free_chunks[bucket] )
      return;

   /* this is where the buckets will be */
    new_block = rmalloc( (chunk_size * actual_block_count) +       /* blocks of memory */
                         sizeof(FeriteMemoryChunkHeader) ); /* initial header   */
    ((FeriteMemoryChunkHeader*)new_block)->next = big_chunks; /* hook up  header */
    big_chunks = new_block; /* this now becomes the header */

   /* the memory on this chunk needs to be setup as follows:
    *       /---------------------\
    *   [header][memory         ][header][memory         ]
    *   |<-a ->||<-p_l[bucket]->||<-a ->||<-p_l[bucket]->|  */
    hdr = (FeriteMemoryChunkHeader*)(((char*)new_block) + sizeof(FeriteMemoryChunkHeader)); /* get the top of our memory */
    FUD(( "================================  Morecore %ld ================================\n", ferite_pow_lookup[bucket] ));
    for( i = 0; i < actual_block_count-1; i++ ) /* go through the block linking up the blocks */
    {
        hdr->next =  (FeriteMemoryChunkHeader*)(((char*)hdr) + ferite_pow_lookup[bucket] + sizeof(FeriteMemoryChunkHeader) );
        FUD(( " [%d]> hdr=%p, hdr->next=%p, %ld, sizeof(Header): %d, pow[%d]=%ld\n", i, hdr, hdr->next,
              (long)((void*)hdr->next) - (long)((void*)hdr), sizeof(FeriteMemoryChunkHeader), bucket, ferite_pow_lookup[bucket] ));
        hdr = hdr->next;
    }
    hdr->next = NULL; /* stop the chain -> so we know when to allocate more */
    FUD(( " [%d]> hdr=%p, hdr->next=%p, %ld, sizeof(Header): %d, pow[%d]=%ld\n", i, hdr, hdr->next,
          (long)((void*)hdr->next) - (long)((void*)hdr), sizeof(FeriteMemoryChunkHeader), bucket, ferite_pow_lookup[bucket] ));

   /* link the memory in */
    free_chunks[bucket] =  (FeriteMemoryChunkHeader*)(((char*)new_block)+sizeof(FeriteMemoryChunkHeader));
   /* dump the table so that we can check that is correlates with the above table */
#ifdef FERITE_MEM_DEBUG
    ferite_jedi_dump_memory( bucket );
#endif
    FUD(( "morecore: free_chunks[bucket]: %p\n", free_chunks[bucket] ));
}
