%{/* -*- mode: C; mode: fold -*- */
/*
 * Copyright (C) 2000-2004 Chris Ross and various contributors
 * Copyright (C) 1999-2000 Chris Ross
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * o Redistributions of source code must retain the above copyright notice, this
 *   list of conditions and the following disclaimer.
 * o Redistributions in binary form must reproduce the above copyright notice,
 *   this list of conditions and the following disclaimer in the documentation
 *   and/or other materials provided with the distribution.
 * o Neither the name of the ferite software nor the names of its contributors may
 *   be used to endorse or promote products derived from this software without
 *   specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

/* -*- mode: C; mode: fold -*-  
 *  This is the parser for the ferite programming language
 *  Written by Chris Ross (chris@darkrock.co.uk)
 */

#include "ferite.h"
#include "ferite/fcompile.h"
#include "ferite/foop.h" 
#include <string.h>
#include <setjmp.h>

#ifdef WIN32
# include "snprintf.h" /* This is so that we have somethings */
#endif

#define YYERROR_VERBOSE

extern jmp_buf  ferite_compiler_jmpback;
extern int      ferite_scanner_lineno;
extern char    *ferite_scanner_file;
extern int      ferite_compile_error;
extern char    *ferite_last_token_alloc;
extern FeriteCompileRecord *ferite_current_compile;

extern FeriteStack *ferite_compiled_arrays_stack;
extern FeriteStack *ferite_argcount_stack;

char           *ferite_last_function;
int             ferite_last_type;
FeriteVariable *ferite_temp_variable;
char            ferite_fetypebuf[1024];
char            ferite_fenamebuf[1024];
int             ferite_var_is_global = FE_FALSE;
int             ferite_var_is_params = FE_FALSE;
int             ferite_var_is_final = FE_FALSE;
int             ferite_var_is_static = FE_FALSE;
int             ferite_var_is_atomic = FE_FALSE;
int             ferite_var_is_native = FE_FALSE;
int             ferite_var_pass_type = FE_BY_VALUE;
int             ferite_item_state = FE_ITEM_IS_PRIVATE;
int             ferite_class_state = 0;
int             ferite_current_arg_count = 0;
int             ferite_var_array_count = 0;
int             ferite_namespace_naturaln = 0;

void feperror( char *message );
int  feplex();
void ferite_scanner_go_native();

#define FECT    ferite_last_token_alloc = NULL
   
   /***
  * Native Code Support
  ***/
   extern char ferite_current_native_block_file[4096];
   extern int  ferite_current_native_block_line;
   
/* #define NEW_FNC_CODE */
   
%}

%union {
  char   *text;
  int     inumber;
  double  fnumber;
}

%left     ','
%left     T_ASSIGN T_PLUS_ASSIGN T_MINUS_ASSIGN T_MULT_ASSIGN T_DIVIDE_ASSIGN T_LEFT_SHIFT_ASSIGN T_RIGHT_SHIFT_ASSIGN T_BINARY_AND_ASSIGN T_BINARY_OR_ASSIGN T_BINARY_XOR_ASSIGN
%left     T_LOGICAL_OR
%left     T_LOGICAL_AND T_RARROW
%left     T_BINARY_OR
%left     T_BINARY_AND
%left     T_BINARY_XOR
%left     T_EQUALS T_LESSTHAN_EQUALS T_GRTRTHAN_EQUALS T_LESSTHAN T_GRTRTHAN T_NOTEQUAL
%left     T_ISA T_INSTANCEOF
%left     T_LEFT_SHIFT T_RIGHT_SHIFT
%left     T_PLUS T_MINUS
%left     T_MULT T_DIVIDE T_MODULUS
%right    T_INCR T_DECR T_NOT_OPERATOR
%nonassoc T_UMINUS T_UPLUS
%token    T_OBJOP
%nonassoc T_NEW
%token    T_IF
%left     T_ELSE
%left     T_EVAL T_INCLUDE

%token T_FUNCTION T_IF T_FOR T_WHILE T_DO T_SWITCH T_IFERR T_FIX
%token T_CASE T_DEFAULT T_USES T_GLOBAL T_RETURN T_FINAL T_NAMESPACE
%token T_CLASS T_EXTENDS T_BREAK T_CONTINUE T_STATIC T_MULTIPLE_ARGS
%token T_MODIFIES T_RENAME T_ALIAS T_NATIVE
%token T_MODULE_INIT T_MODULE_DEINIT T_MODULE_HEADER T_MODULE_FLAGS
%token T_SLICE T_RARROW T_ATOMIC T_ERROR T_MODULE_REGISTER T_MODULE_UNREGISTER
%token T_FOREACH T_ARGS T_PRIVATE T_PROTECTED T_PUBLIC T_ABSTRACT
%token T_CLOSURE T_USING T_DELIVER T_RECIPIENT T_PROTOCOL T_IMPLEMENTS
%token T_SELF T_SUPER T_CONSTRUCTOR T_DESTRUCTOR
%token T_VAR_NUMBER T_VAR_STRING T_VAR_VOID T_VAR_OBJECT T_VAR_ARRAY

%token <text>    T_QSTRING T_VARTYPE T_LABEL T_OBJPROPERTY T_NATIVE_CODE_BLOCK T_DOT_LABEL
%token <inumber> T_NNUMBER T_TRUE T_FALSE
%token <fnumber> T_RNUMBER

%pure_parser
%expect 7

%%

script:                    
  script_block_list                 { 
                                        ferite_do_function_header( "!__start__", FE_FALSE, FE_FALSE, FE_FALSE, FE_ITEM_IS_PUBLIC, FE_FALSE ); 
                                        ferite_var_array_count = 0 
                                    }
  variable_block
  statement_block                   {
                                        ferite_temp_variable = ferite_create_void_variable( NULL, "forced-function-return", FE_STATIC );
                                        MARK_VARIABLE_AS_COMPILED( ferite_temp_variable );
                                        ferite_do_push( ferite_temp_variable );
                                        ferite_do_function_footer();
                                    }
  ;

script_block_list:
  /* empty */
| non_empty_script_block_list
  ;

non_empty_script_block_list:
  script_block
| non_empty_script_block_list script_block
  ;

script_block:
  uses_list
| global_decl
| protocol_def
| class_def
| namespace_def
| builder_item
| block_item_modifier_list function 
| namespace_delete_rename
  ;

builder_item:
  T_MODULE_INIT                     { ferite_scanner_go_native(); } 
  T_NATIVE_CODE_BLOCK               { ferite_do_create_builder_variable( "'module-init", $3 ); ffree( $3 ); }
| T_MODULE_DEINIT                   { ferite_scanner_go_native(); } 
  T_NATIVE_CODE_BLOCK               { ferite_do_create_builder_variable( "'module-deinit", $3 ); ffree( $3 ); }
| T_MODULE_REGISTER                 { ferite_scanner_go_native(); } 
  T_NATIVE_CODE_BLOCK               { ferite_do_create_builder_variable( "'module-register", $3 ); ffree( $3 ); }
| T_MODULE_UNREGISTER               { ferite_scanner_go_native(); } 
  T_NATIVE_CODE_BLOCK               { ferite_do_create_builder_variable( "'module-unregister", $3 ); ffree( $3 ); }
| T_MODULE_HEADER                   { ferite_scanner_go_native(); } 
  T_NATIVE_CODE_BLOCK               { ferite_do_create_builder_variable( "'module-header", $3 ); ffree( $3 ); }
| T_MODULE_FLAGS                    { ferite_scanner_go_native(); } 
  T_NATIVE_CODE_BLOCK               { ferite_do_create_builder_variable( "'module-flags", $3 ); ffree( $3 ); }
  ;

uses_list:
  T_USES module_list ';'
  ;

module_item:
  T_QSTRING                         { ferite_do_uses( $1 ); FECT }
  ;

module_list:
  module_item
| module_list ',' module_item
  ;

global_decl:
  T_GLOBAL                          { FUD(( "going into global block\n" )); ferite_var_is_global = FE_TRUE; }
    '{' namespace_variable_block '}' 
                                    { ferite_var_is_global = FE_FALSE; FUD(( "leaving global memory block\n" )); }
  ;

var_type:
  T_VAR_NUMBER  { ferite_last_type = F_VAR_NUM; }
| T_VAR_STRING  { ferite_last_type = F_VAR_STR; }
| T_VAR_VOID    { ferite_last_type = F_VAR_VOID; }
| T_VAR_OBJECT  { ferite_last_type = F_VAR_OBJ; }
| T_VAR_ARRAY   { ferite_last_type = F_VAR_UARRAY; }
  ;

param_var_type:
  var_type
| T_CLASS       { ferite_last_type = F_VAR_CLASS; }
| T_NAMESPACE   { ferite_last_type = F_VAR_NS; }
  ;

function_name:
   T_LABEL                          { ferite_last_function = $1; };
|  T_CONSTRUCTOR                    { ferite_last_function = fstrdup( "constructor" ); }
|  T_DESTRUCTOR                     { ferite_last_function = fstrdup( "destructor" ); }
   ;

function_header_prefix:
   T_FUNCTION function_name         {
                                        if( ferite_var_is_final )
                                          ferite_warning( ferite_current_compile->script, "Keyword 'final' can not be applied to functions ('%s' on line %d in %s)\n", ferite_last_function, ferite_scanner_lineno, ferite_scanner_file );
                                        ferite_do_function_header( ferite_last_function, ferite_var_is_static, ferite_var_is_native, ferite_var_is_atomic, ferite_item_state, ferite_var_is_final );
                                        ffree( ferite_last_function ); FECT;
                                        ferite_var_is_params = FE_TRUE;
                                        ferite_var_is_static = FE_FALSE;
                                        ferite_var_is_atomic = FE_FALSE;
                                        ferite_var_array_count = FE_FALSE;
                                        ferite_var_is_final = FE_FALSE; 
                                        ferite_item_state = FE_ITEM_IS_PRIVATE;
                                    }
   ;


function_parameter_declaration:
   '(' decl_parameter_list ')'      { 
                                        ferite_var_is_params = FE_FALSE;
                                        ferite_do_function_start();
                                        FUD(("PARSER: END FUNCTION PARAMETER DECLARATION\n"));
                                        FUD(("PARSER: BEGIN FUNCTION BODY\n")); 
                                    }
   ;

function_body:
   '{' 
      variable_block 
      statement_block '}'           { 
                                        FUD(("PARSER: END FUNCTION BODY\n"));
                                        ferite_temp_variable = ferite_create_void_variable( NULL, "forced-function-return", FE_STATIC );
                                        MARK_VARIABLE_AS_COMPILED( ferite_temp_variable );
                                        ferite_do_push( ferite_temp_variable );
                                        FUD(("PARSER: push(voidptr)        \n"));
                                        ferite_do_function_footer();
                                    }
|  T_NATIVE_CODE_BLOCK              { 
                                        ferite_do_function_native_block( $1,
                                        ferite_current_native_block_file, 
                                        ferite_current_native_block_line ); 
                                    }
|  statement                        {
    
                                        FUD(("PARSER: END FUNCTION BODY\n"));
                                        ferite_temp_variable = ferite_create_void_variable( NULL, "forced-function-return", FE_STATIC );
                                        MARK_VARIABLE_AS_COMPILED( ferite_temp_variable );
                                        ferite_do_push( ferite_temp_variable );
                                        FUD(("PARSER: push(voidptr)        \n"));
                                        ferite_do_function_footer();    
                                    }
   ;

function:
   function_header_prefix
   function_parameter_declaration   { 
                                        if( ferite_var_is_native == FE_TRUE )
                                        {
                                            ferite_scanner_go_native(); 
                                            ferite_var_is_native = FE_FALSE;
                                        }
                                    }
   function_body
   ;

parameter_byref:
   /* No by-ref char */
|  T_BINARY_AND                     { ferite_var_pass_type = FE_BY_REF; }
   ;
   
parameter_define_item:
   param_var_type 
   optional_variable_type_hint
   parameter_byref T_LABEL          {  
                                        FUD(("PARSER: Parameter Declared         %s (%d)\n", $4, ferite_last_type)); 
                                        ferite_do_add_variable_to_paramlist( $4, ferite_last_type, ferite_var_pass_type );
                                        ffree( $4 ); FECT;
                                        ferite_var_pass_type = FE_BY_VALUE;
                                    }
   ;

parameter_multiple_args:
   T_MULTIPLE_ARGS                  {
                                        FUD(("PARSER: Parameter Declared - Multiple Args\n"));
                                        ferite_do_add_variable_to_paramlist( "...", F_VAR_VOID, FE_BY_VALUE );
                                    }
   ;

decl_parameter_list:
   /* none */
|  non_empty_decl_parameter_list
|  non_empty_decl_parameter_list ',' parameter_multiple_args
|  parameter_multiple_args
   ;
   
non_empty_decl_parameter_list:
   parameter_define_item
|  non_empty_decl_parameter_list ',' parameter_define_item
   ;

namespace_header:
  T_NAMESPACE T_LABEL               { ferite_do_namespace_header( $2 ); ffree( $2 ); FECT; }
| T_NAMESPACE T_MODIFIES T_LABEL    { ferite_do_namespace_extends( $3 ); ffree( $3 );  FECT; }
  ;

namespace_block_list:
  /* empty */
| non_empty_namespace_block_list
  ;

non_empty_namespace_block_list:
  namespace_block
| non_empty_namespace_block_list namespace_block
  ;

namespace_delete_rename:
  T_RENAME T_LABEL T_LABEL ';'      {
                                        FUD(("Renaming class item\n"));
                                        ferite_do_namespace_item_rename( $2, $3 );
                                        ffree( $2 ); FECT; ffree( $3 ); FECT;
                                    }
  ;

namespace_block:
  block_item_modifier_list namespace_multiple_variable_item ';'
| block_item_modifier_list function
| protocol_def
| class_def
| namespace_def
| namespace_delete_rename
  ;

namespace_def:
  namespace_header
  '{'  namespace_block_list '}'     { ferite_do_namespace_footer(); }
  ;

class_block_list:
  /* empty */
| non_empty_class_block_list
  ;

non_empty_class_block_list:
  class_block
| non_empty_class_block_list class_block
  ;

class_block:
  { ferite_item_state = -1; }
  class_block_real
  ;

class_block_real:
  block_item_modifier_list namespace_multiple_variable_item ';'
| block_item_modifier_list function
| T_RENAME T_LABEL T_LABEL ';'      {
                                        FUD(("Renaming class item\n"));
                                        ferite_do_class_item_rename( $2, $3 );
                                        ffree( $2 ); FECT; ffree( $3 ); FECT;
                                    }
| T_ALIAS T_LABEL T_LABEL ';'       {
                                        FUD(("Aliasing class item\n"));
                                        ferite_do_class_function_alias( $2, $3 );
                                        ffree( $2 ); FECT; ffree( $3 ); FECT;
                                    }
  ;

class_header:
  T_LABEL                           {   
                                        FUD(("CLASS START\n" ));
                                        ferite_do_class_header( $1, NULL, ferite_class_state );
                                        ferite_class_state = 0;
                                        ffree( $1 ); FECT;
                                    }
| T_LABEL T_EXTENDS                 {
                                        ferite_scanner_want_dot_label();
                                    }
  T_DOT_LABEL                       {   
                                        FUD(("CLASS START\n" ));
                                        ferite_scanner_stop_dot_label();
                                        ferite_do_class_header( $1, $4, ferite_class_state );
                                        ferite_class_state = 0;
                                        ffree( $1 ); FECT;
                                        ffree( $4 ); FECT;
                                    }
  ;

class_modifies_header:
  T_MODIFIES                        {
                                        ferite_scanner_want_dot_label();
                                    }
  T_DOT_LABEL                       {
                                        FUD(("CLASS START\n" ));
                                        ferite_scanner_stop_dot_label();
                                        ferite_do_class_header( NULL, $3, ferite_class_state );
                                        ferite_class_state = 0;
                                        ffree( $3 ); FECT;
                                    }
  ;

implements_item:
                                    {
                                        ferite_scanner_want_dot_label();
                                    }
    T_DOT_LABEL                     {
                                        ferite_scanner_stop_dot_label();
                                        ferite_do_class_implements( $2 );
                                        ffree( $2 ); FECT;
                                    }
	;
	
implements_list:
	implements_item
|	implements_list ',' implements_item
	;

class_implements:
    /* empty - no implements */
|   T_IMPLEMENTS implements_list
    ;

standard_class:
   class_header
   class_implements
   '{' class_block_list '}'         {
                                        FUD(("CLASS END\n" ));
                                        ferite_do_class_footer();
                                    }
   ;

modifies_class:
   class_modifies_header 
   '{' protocol_block_list '}'      {
                                        ferite_do_class_footer();
                                    }
   ;

class_def:
  class_modifier_list
  T_CLASS
  class_body
  ;
  
class_body:
  standard_class
| modifies_class
  ;

protocol_header:
    T_PROTOCOL T_LABEL {
                            ferite_do_class_header( $2, NULL, FE_ITEM_IS_PROTOCOL );
                            ferite_class_state = 0;
                            ffree( $2 ); FECT;
                        }
    ;
	
protocol_item:
    block_item_modifier_list function
    ;
	
non_empty_protocol_block_list:
    protocol_item
|   non_empty_protocol_block_list protocol_item
    ;
	
protocol_block_list_real:
    /* empty */
|	non_empty_protocol_block_list
    ;

protocol_block_list:
    { ferite_class_state = -1; }
    protocol_block_list_real;
	
protocol_def:
    protocol_header
    '{' protocol_block_list '}'     {
                                        ferite_do_class_footer();
                                    }
    ;
	
   
variable_define_item:
   T_LABEL                          {  
                                        FUD(("PARSER: Variable Declared         %s(%d)\n", $1, ferite_last_type)); 
                                        if( ferite_var_is_native )
                                          ferite_warning( ferite_current_compile->script, "Keyword 'native' can not be applied to variables ('%s' on line %d in %s)\n", $1, ferite_scanner_lineno, ferite_scanner_file );
                                        ferite_do_add_variable( $1, ferite_last_type, ferite_var_is_global, ferite_var_is_final, ferite_var_is_static, ferite_var_is_atomic, ferite_item_state, FE_FALSE );
                                        
                                        if( ferite_last_type == F_VAR_VOID )
                                        {
                                            ferite_do_variable_push( $1 );
                                            ferite_do_void_reset();
                                        }
                                        
                                        ffree( $1 ); FECT;
                                    }
|  T_LABEL T_ASSIGN                 {  
                                        FUD(("PARSER: Variable Declared         %s(%d)\n", $1, ferite_last_type)); 
                                        if( ferite_var_is_native )
                                          ferite_warning( ferite_current_compile->script, "Keyword 'native' can not be applied to variables ('%s' on line %d in %s)\n", $1, ferite_scanner_lineno, ferite_scanner_file );
                                        ferite_do_add_variable( $1, ferite_last_type, FE_FALSE, ferite_var_is_final, ferite_var_is_static, ferite_var_is_atomic, ferite_item_state, FE_FALSE );
                                        
                                        if( ferite_last_type == F_VAR_VOID )
                                        {
                                            ferite_do_variable_push( $1 );
                                            ferite_do_void_reset();
                                        }
                                        
                                        ferite_do_variable_push( $1 );
                                        ffree( $1 ); FECT; 
                                    }
   expression                       {  
                                        ferite_do_binary_op( FERITE_OPCODE_assign );            
                                        FUD(("PARSER: op->assign       pop,pop\n")); 
                                        ferite_do_pop();
                                    } 
   ;

namespace_natural_number:
   T_MINUS T_NNUMBER                {   ferite_namespace_naturaln = 0 - $2; }
|  T_PLUS  T_NNUMBER                {   ferite_namespace_naturaln = $2; }
|  T_NNUMBER                        {   ferite_namespace_naturaln = $1; }
   ;

namespace_variable_define_item:
   T_LABEL                          {  
                                        FUD(("PARSER: Variable Declared         %s(%d)\n", $1, ferite_last_type)); 
                                        if( ferite_var_is_native )
                                          ferite_warning( ferite_current_compile->script, "Keyword 'native' can not be applied to variables ('%s' on line %d in %s)\n", $1, ferite_scanner_lineno, ferite_scanner_file );
                                        ferite_do_add_variable( $1, ferite_last_type, ferite_var_is_global, ferite_var_is_final, ferite_var_is_static, ferite_var_is_atomic, ferite_item_state, FE_FALSE );
                                        ffree( $1 ); FECT;
                                    }
|  T_LABEL T_ASSIGN namespace_natural_number   { 
                                        if( ferite_var_is_native )
                                          ferite_warning( ferite_current_compile->script, "Keyword 'native' can not be applied to variables ('%s' on line %d in %s)\n", $1, ferite_scanner_lineno, ferite_scanner_file );
                                        if( ferite_last_type == F_VAR_NUM )
                                        {
                                            FUD(("PARSER: Variable Declared         %s(%d)=%d\n", $1, ferite_last_type, ferite_namespace_naturaln )); 
                                            ferite_temp_variable = ferite_create_number_long_variable( NULL, $1, ferite_namespace_naturaln, FE_ALLOC );
                                            if( ferite_var_is_final )
                                              MARK_VARIABLE_AS_FINALSET( ferite_temp_variable );
                                            ferite_do_add_variable_with_value( $1, ferite_temp_variable, ferite_var_is_global, ferite_var_is_static, ferite_var_is_atomic, ferite_item_state );
                                            ffree( $1 ); FECT;
                                        } 
                                        else 
                                        {
                                            ferite_error( ferite_current_compile->script, 0, "Can't initialise variable '%s' of type '%s' with the number %d.\n", $1, ferite_variable_id_to_str( NULL,ferite_last_type), ferite_namespace_naturaln );
                                            ffree( $1 ); FECT;
                                            feperror( NULL );
                                        }
                                    }
|  T_LABEL T_ASSIGN T_RNUMBER       { 
                                        if( ferite_var_is_native )
                                          ferite_warning( ferite_current_compile->script, "Keyword 'native' can not be applied to variables ('%s' on line %d in %s)\n", $1, ferite_scanner_lineno, ferite_scanner_file );
                                        if( ferite_last_type == F_VAR_NUM )
                                        {
                                            FUD(("PARSER: Variable Declared         %s(%d)=%f\n", $1, ferite_last_type, $3)); 
                                            ferite_temp_variable = ferite_create_number_double_variable( NULL, $1, $3, FE_ALLOC );
                                            if( ferite_var_is_final )
                                              MARK_VARIABLE_AS_FINALSET( ferite_temp_variable );
                                            ferite_do_add_variable_with_value( $1, ferite_temp_variable, ferite_var_is_global, ferite_var_is_static, ferite_var_is_atomic, ferite_item_state );
                                            ffree( $1 ); FECT;
                                        } 
                                        else 
                                        {
                                            ferite_error( ferite_current_compile->script, 0, "Can't initialise variable '%s' of type '%s' with the number %f.\n", $1, ferite_variable_id_to_str( NULL,ferite_last_type), $3 );
                                            ffree( $1 ); FECT;
                                            feperror( NULL );
                                        }
                                    }
|  T_LABEL T_ASSIGN T_QSTRING       { 
                                        if( ferite_var_is_native )
                                          ferite_warning( ferite_current_compile->script, "Keyword 'native' can not be applied to variables ('%s' on line %d in %s)\n", $1, ferite_scanner_lineno, ferite_scanner_file );
                                        if( ferite_last_type == F_VAR_STR )
                                        {
                                            FUD(("PARSER: Variable Declared         %s(%d)='%s'\n", $1, ferite_last_type, $3)); 
                                            ferite_temp_variable = ferite_create_string_variable_from_ptr( NULL, $1, $3, strlen($3), FE_CHARSET_DEFAULT, FE_ALLOC );
                                            if( ferite_var_is_final )
                                              MARK_VARIABLE_AS_FINALSET( ferite_temp_variable );
                                            ferite_do_add_variable_with_value( $1, ferite_temp_variable, ferite_var_is_global, ferite_var_is_static, ferite_var_is_atomic, ferite_item_state );
                                            ffree( $1 ); ffree( $3 ); FECT;
                                        } 
                                        else 
                                        {
                                            ferite_error( ferite_current_compile->script, 0, "Can't initialise variable '%s' of type '%s' with the string '%s'.\n", $1, ferite_variable_id_to_str( NULL,ferite_last_type), $3 );
                                            ffree( $1 ); 
                                            ffree( $3 );
                                            FECT;
                                            feperror( NULL );
                                        }
                                    }
;

variable_define_list:
   variable_define_item
|  variable_define_list ',' variable_define_item
   ;

namespace_variable_define_list:
   namespace_variable_define_item
|  namespace_variable_define_list ',' namespace_variable_define_item
   ;

class_modifier:
  T_FINAL                          { 
                                      if( ferite_class_state > 0 )
                                      {
                                          ferite_error( ferite_current_compile->script, 0, "The class already has state - you can't make it final\n" );
                                          feperror( NULL );
                                      }
                                      else
                                        ferite_class_state = FE_ITEM_IS_FINAL; 
                                    }
|  T_ABSTRACT                       { 
                                      if( ferite_class_state > 0 )
                                      {
                                          ferite_error( ferite_current_compile->script, 0, "The class already has state - you can't make it abstract\n" );
                                          feperror( NULL );
                                      }
                                      else
                                        ferite_class_state = FE_ITEM_IS_ABSTRACT; 
                                    }
   ;

class_modifier_list:
   /* Empty */
|  class_modifier
   ;

item_modifier:
   T_FINAL                          { 
                                      if( ferite_var_is_final )
                                      {
                                          ferite_error( ferite_current_compile->script, 0, "Item is already final - you can't make it final again\n" );
                                          feperror( NULL );
                                      }
                                      else
                                        ferite_var_is_final = FE_TRUE; 
                                    }
|  T_STATIC                         { 
                                      if( ferite_var_is_static )
                                      {
                                          ferite_error( ferite_current_compile->script, 0, "Item is already static - you can't make it static again\n" );
                                          feperror( NULL );
                                      }
                                      else
                                        ferite_var_is_static = FE_TRUE;  
                                    }
|  T_ATOMIC                         { 
                                      if( ferite_var_is_atomic )
                                      {
                                          ferite_error( ferite_current_compile->script, 0, "Item is already atomic - you can't make it atomic again\n" );
                                          feperror( NULL );
                                      }
                                      else
                                        ferite_var_is_atomic = FE_TRUE;  
                                    }
|  T_NATIVE                         { 
                                      if( ferite_var_is_native )
                                      {
                                          ferite_error( ferite_current_compile->script, 0, "Item is already native - you can't make it native again\n" );
                                          feperror( NULL );
                                      }
                                      else
                                        ferite_var_is_native = FE_TRUE; 
                                    }
|  T_PRIVATE                        { 
                                      if( ferite_item_state > -1 )
                                      {
                                          ferite_error( ferite_current_compile->script, 0, "Item is already %s - you can't make it private\n", ferite_state_to_str( ferite_item_state ) );
                                          feperror( NULL );
                                      }
                                      else
                                        ferite_item_state = FE_ITEM_IS_PRIVATE; 
                                    }
|  T_PROTECTED                      { 
                                      if( ferite_item_state > -1 )
                                      {
                                          ferite_error( ferite_current_compile->script, 0, "Item is already %s - you can't make it protected\n", ferite_state_to_str( ferite_item_state ) );
                                          feperror( NULL );
                                      }
                                      else
                                        ferite_item_state = FE_ITEM_IS_PROTECTED; 
                                    }
|  T_PUBLIC                         { 
                                      if( ferite_item_state > -1 )
                                      {
                                          ferite_error( ferite_current_compile->script, 0, "Item is already %s - you can't make it public\n", ferite_state_to_str( ferite_item_state ) );
                                          feperror( NULL );
                                      }
                                      else
                                        ferite_item_state = FE_ITEM_IS_PUBLIC; 
                                    }
   ;

block_item_modifier_list:
   /* nowt */
|  non_empty_block_item_modifier_list
   ;

non_empty_block_item_modifier_list:
   item_modifier
|  non_empty_block_item_modifier_list item_modifier
   ;

variable_block:
   non_empty_variable_block
|  /* empty */
   ;

non_empty_variable_block: 
   multiple_variable_item ';'
|  non_empty_variable_block multiple_variable_item ';'
   ;

final_modifier:
   T_FINAL                          { 
                                      if( ferite_var_is_final )
                                      {
                                          ferite_error( ferite_current_compile->script, 0, "Item is already final - you can't make it final again\n" );
                                          feperror( NULL );
                                      }
                                      else
                                        ferite_var_is_final = FE_TRUE; 
                                    }
|  /* nothing */
   ;

optional_variable_type_hint_t:
   T_LABEL    { ffree( $1 ); FECT; }
|  T_QSTRING  { ffree( $1 ); FECT; }
|  T_RNUMBER  { }
|  T_NNUMBER  { }
   ;

optional_variable_type_hint:
   /* empty */
|  T_LESSTHAN 
   optional_variable_type_hint_t 
   T_GRTRTHAN    
   ;

multiple_variable_item:
   final_modifier var_type         {
                                        FUD(("PARSER: Setting current type to %d\n", ferite_last_type ));
                                    }
   optional_variable_type_hint
   variable_define_list             { ferite_var_is_final = FE_FALSE; ferite_var_is_static = FE_FALSE; ferite_var_is_atomic = FE_FALSE; ferite_var_is_native = FE_FALSE; }
   ;

namespace_multiple_variable_item:
   var_type                        {
                                        FUD(("PARSER: Setting current type to %d\n", ferite_last_type ));
                                    }
   optional_variable_type_hint
   namespace_variable_define_list   { ferite_var_is_final = FE_FALSE; ferite_var_is_static = FE_FALSE; ferite_var_is_atomic = FE_FALSE; ferite_var_is_native = FE_FALSE; }
   ;
   
namespace_variable_block:
   /* empty */
|  non_empty_namespace_variable_block
   ;
   
non_empty_namespace_variable_block:
   block_item_modifier_list namespace_multiple_variable_item ';'
|  non_empty_namespace_variable_block 
   block_item_modifier_list namespace_multiple_variable_item ';' 
   ;

nested_block:
   '{'              { ferite_do_begin_block(); }
      variable_block
      statement_block
   '}'              { ferite_do_end_block(); }
   ;

block:
   statement
|  nested_block
   ;

case_block:
   statement_block
   ;

statement_block:
   /* empty */
|  non_empty_statement_block
   ;

non_empty_statement_block_item:
   statement
|  nested_block
   ;

non_empty_statement_block:
   non_empty_statement_block_item
|  non_empty_statement_block non_empty_statement_block_item
   ;
                  
statement:
   expression ';'                   { ferite_do_pop(); /* clear up the stupid stack ie wham = 2 + 3 leaves a variable on the stack*/ }
|  select_statement
|  itterative_statement
|  return_statement ';'
|  T_BREAK ';'                      { ferite_do_break(); }
|  T_CONTINUE ';'                   { ferite_do_continue(); }
|  ';'
   ;
   
return_statement:
   T_RETURN                         {  
                                        ferite_temp_variable = ferite_create_void_variable( NULL, "compiled-in-return-value", FE_STATIC );
                                        MARK_VARIABLE_AS_COMPILED( ferite_temp_variable );
                                        ferite_do_push( ferite_temp_variable );
                                        FUD(("PARSER: push(voidptr)        \n"));
                                        ferite_do_exit();
                                    }
| T_RETURN expression               {  
                                        ferite_do_exit();
                                    }
  ;

select_statement:
   T_IF '(' expression ')'          { ferite_do_if_statement(); } 
   block else_block
|  T_IFERR                          { ferite_do_iferr_block(); } 
   block                            { ferite_do_before_fix_block(); } 
   T_FIX block fix_else_block
|  switch_statement
   ;

else_block:
   /* empty */                      { ferite_do_after_then_statement(); }
|  T_ELSE                           { ferite_do_after_then_before_else_statement(); }
     block {  ferite_do_after_else_statement(); }
   ;

switch_statement:
   T_SWITCH                         {
                                        ferite_do_pre_switch();  
                                    }
   '(' expression ')' 
   '{'
       switch_case_list   
   '}'                              {
                                        ferite_do_post_switch();
                                    }
   ;

switch_case_list:
   switch_cases_list
   default_item
   ;

switch_cases_list:
   /* empty */
|  non_empty_switch_cases_list
   ;

non_empty_switch_cases_list:
   switch_case_item
|  non_empty_switch_cases_list switch_case_item
   ;


switch_case_item:
   T_CASE               
   expression                       {
                                        ferite_do_case_block_start();
                                    }
   ':' case_block                   {
                                        ferite_do_case_block_end();
                                    }
   ;

default_item:
   /* nuttin */
|  T_DEFAULT ':'                    {
                                        ferite_do_default_block_start();
                                    }
   statement_block
   ;

fix_else_block:
   /* empty */                      { ferite_do_after_fix_block(); }
|  T_ELSE                           { ferite_do_after_fix_before_else_block(); } 
   block                            { ferite_do_after_fix_else_statement(); }
   ;

for_expression:
   /* nothing */                    {   
                                        ferite_temp_variable = ferite_create_number_long_variable( NULL, "forconst", 1, FE_STATIC );
                                        MARK_VARIABLE_AS_COMPILED( ferite_temp_variable );
                                        ferite_do_push( ferite_temp_variable );
                                        FUD(("PARSER: for truth value\n")); 
                                    }
|  expression                       
   ;

for_init_expression:
   for_expression                   { ferite_do_pop(); }
|  multiple_variable_item
   ;

foreach_start:
   T_FOREACH                        {
                                        ferite_error( ferite_current_compile->script, 0, "'foreach' has been removed from ferite for more elegant methods.\n" );
                                        ferite_error( ferite_current_compile->script, 0, "Please console the documentation for the Array module.\n" );
                                        feperror( NULL );
                                    }
   ;
   
itterative_statement:
   T_FOR '('                        {
	                                ferite_do_begin_block();
                                        FUD(("PARSER: [FOR] Begin Initial Expression\n"));
                                    }
      for_init_expression           {  
                                        FUD(("PARSER: [FOR] End Initial Expression\n")); 
                                        ferite_do_for_loop_start();
                                    }
         ';'                        { FUD(("PARSER: [FOR] Begin Test Expression\n")); }
      for_expression                { FUD(("PARSER: [FOR] End Test Expression\n")); }
         ';'                        {  
                                        FUD(("PARSER: [FOR] Begin Itterator Expression\n"));
                                        ferite_do_for_loop_itterate();
                                    } 
      for_expression                {  
                                        FUD(("PARSER: [FOR] End Itterator Expression\n"));
                                        ferite_do_pop();
                                        ferite_do_for_loop_block();
                                    }
         ')'
   block                            {  
                                        FUD(("PARSER: [FOR] Jump back to test expression\n")); 
                                        ferite_do_for_loop_end();
					ferite_do_end_block();
                                    }

|  T_WHILE                          { ferite_do_while_begin(); }
         '(' expression ')'         { ferite_do_while_after_expr(); }
         block                      { ferite_do_while_end(); }
|  T_DO                             { ferite_do_do_start(); } 
     block
   T_WHILE '(' expression ')' ';'   { ferite_do_do_end(); }

|  foreach_start
   ;
   
expression:
   cvar
|  cval
|  expression_without_var
   ;
   
cval:
   T_QSTRING                        {  
                                        ferite_temp_variable = ferite_create_string_variable_from_ptr( NULL, "strconst", $1, 0, FE_CHARSET_DEFAULT, FE_STATIC );
                                        MARK_VARIABLE_AS_COMPILED( ferite_temp_variable );
                                        ferite_do_push( ferite_temp_variable );
                                        FUD(("PARSER: push(str)        \"%s\"\n", $1));
                                        ffree( $1 ); FECT;
                                    }
|  T_TRUE                           {
                                        ferite_temp_variable = ferite_create_number_long_variable( NULL, "true", FE_TRUE, FE_STATIC );
                                        MARK_VARIABLE_AS_COMPILED( ferite_temp_variable );
                                        ferite_do_push( ferite_temp_variable );
                                        FUD(("PARSER: push(str)        \"true\"\n"));
                                    }
|  T_FALSE                          {
                                        ferite_temp_variable = ferite_create_number_long_variable( NULL, "false", FE_FALSE, FE_STATIC );
                                        MARK_VARIABLE_AS_COMPILED( ferite_temp_variable );
                                        ferite_do_push( ferite_temp_variable );
                                        FUD(("PARSER: push(str)        \"false\"\n"));
                                    }
|  T_NNUMBER                        {  
                                        ferite_temp_variable = ferite_create_number_long_variable( NULL, "longconst", $1, FE_STATIC );
                                        MARK_VARIABLE_AS_COMPILED( ferite_temp_variable );
                                        ferite_do_push( ferite_temp_variable );
                                        FUD(("PARSER: push(int)        %d\n", $1)); 
                                    } 
|  T_RNUMBER                        {  
                                        ferite_temp_variable = ferite_create_number_double_variable( NULL, "doubleconst", $1, FE_STATIC );
                                        MARK_VARIABLE_AS_COMPILED( ferite_temp_variable );
                                        ferite_do_push( ferite_temp_variable );
                                        FUD(("PARSER: push(float)      %f\n", $1)); 
                                    }
   ;
   
cvar:
  T_LABEL                           {  
                                        ferite_do_variable_push( $1 );
                                        FUD(("PARSER: push(cvar)       %s\n", $1)); 
                                        ffree( $1 ); FECT;
                                    }
|  T_SELF                           {   ferite_do_variable_push( "self" ); }
|  T_SUPER                          {   ferite_do_variable_push( "super" ); }
|  cvar '[' expression T_SLICE expression ']'  
                                    {  
                                        FUD(("PARSER: op->array_slice\n"));
                                        ferite_do_many_op( FERITE_OPCODE_array_slice,3);
                                    }
|  cvar '[' T_SLICE                 {
                                        FUD(("PARSER: op->array_slice\n"));
                                        ferite_temp_variable = ferite_create_number_long_variable( NULL, "slice_first",0, FE_STATIC );
                                        MARK_VARIABLE_AS_COMPILED( ferite_temp_variable );
                                        ferite_do_push( ferite_temp_variable ); 
                                    } 
  expression ']'                    { ferite_do_many_op(FERITE_OPCODE_array_slice,3); }
| cvar '[' expression T_SLICE ']'   {
                                        ferite_temp_variable = ferite_create_void_variable( NULL, "sliceconst", FE_STATIC );
                                        MARK_VARIABLE_AS_PLACEHOLDER( ferite_temp_variable );
                                        MARK_VARIABLE_AS_COMPILED( ferite_temp_variable );
                                        ferite_do_push( ferite_temp_variable );
                                        ferite_do_many_op(FERITE_OPCODE_array_slice,3);
                                    }
| cvar '[' T_SLICE ']'              {
                                        ferite_temp_variable = ferite_create_number_long_variable( NULL, "slice_first",0, FE_STATIC );
                                        MARK_VARIABLE_AS_COMPILED( ferite_temp_variable );
                                        ferite_do_push( ferite_temp_variable );
                                        ferite_temp_variable = ferite_create_void_variable( NULL, "sliceconst", FE_STATIC );
                                        MARK_VARIABLE_AS_PLACEHOLDER( ferite_temp_variable );
                                        MARK_VARIABLE_AS_COMPILED( ferite_temp_variable );
                                        ferite_do_push( ferite_temp_variable );
                                        ferite_do_many_op(FERITE_OPCODE_array_slice,3);
                                    } 
| cvar '[' expression ']'           {
                                        FUD(("PARSER: op->array_index"));
                                        ferite_do_binary_op(FERITE_OPCODE_array_index);
                                    }            
|  cvar '[' ']'                     {   /* that dash cunning end of array index type thingy.... */
                                        /* it wasnt cunning enough because it broke with the introduction of negative indices ;) - ntropy*/
                                        ferite_temp_variable = ferite_create_void_variable( NULL, "void-placeholder", FE_STATIC );
                                        MARK_VARIABLE_AS_PLACEHOLDER( ferite_temp_variable );
                                        MARK_VARIABLE_AS_COMPILED( ferite_temp_variable );
                                        ferite_do_push( ferite_temp_variable );
                                        FUD(("PARSER: push(int)        %d\n", FE_ARRAY_ADD_AT_END )); 
                                        FUD(("PARSER: op->array_index\n"));
                                        ferite_do_binary_op( FERITE_OPCODE_array_index );
                                    }
|  cvar object_list
|                                   { ferite_do_self_push(); } 
   object_list
   ;

object_list:
   object_property
|  object_list object_property
   ;

object_no_call_list: 
   object_property_no_call   
|  object_no_call_list object_property_no_call      
   ;

nvar:
   T_LABEL                          {  
                                        ferite_do_variable_push( $1 );
                                        FUD(("PARSER: push(cvar)       %s\n", $1)); 
                                        ffree( $1 ); FECT;
                                    }
|  nvar object_no_call_list 
   ;

object_property_no_call:
   T_OBJOP T_OBJPROPERTY            {  
                                        ferite_do_variable_pushattr( $2 );
                                        FUD(("PARSER: push(object property)       %s\n", $2));
                                        ffree( $2 ); FECT;
                                    }
   ;
   
object_property:
   object_property_no_call
|  T_OBJOP T_OBJPROPERTY
  '(' parameter_list ')'
  optional_function_with_clause     {  
                                        int *value = ferite_stack_pop( ferite_argcount_stack );
                                        FUD(("PARSER: op->c_obj_func       %s\n", $2 ));
                                        ferite_do_object_function_call( $2, ferite_current_arg_count );
                                        ffree( $2 ); FECT;
                                        if( value != NULL )
                                        {
                                            ferite_current_arg_count = *value;
                                            ffree( value );
                                        }
                                    }
   ;

lambda_argument:
   T_LABEL                          { ferite_do_add_variable_to_paramlist( $1, F_VAR_VOID, FE_BY_VALUE ); ffree( $1 ); FECT; }
   ;
   
non_empty_lambda_arguments:
   lambda_argument
|  non_empty_lambda_arguments ',' lambda_argument
   ;

lambda_arguments:
   /* Empty List */
|  '(' non_empty_lambda_arguments ')'
   ;
   
lambda_body:
                                    { ferite_do_closure_start(); }
   lambda_arguments                 { ferite_do_closure_end_of_args(); }
   nested_block                     { ferite_do_closure_end(); }
   ;

lambda_expression:
   T_CLOSURE lambda_body
   ;

expression_without_var:
   expression T_PLUS expression     {  
                                         ferite_do_binary_op( FERITE_OPCODE_add );
                                         FUD(("PARSER: op->add          pop,pop\n")); 
                                    }
|  expression T_MINUS expression    {   
                                         ferite_do_binary_op( FERITE_OPCODE_minus );
                                         FUD(("PARSER: op->minus        pop,pop\n")); 
                                    } 
|  expression T_MULT expression     {  
                                         ferite_do_binary_op( FERITE_OPCODE_mult );
                                         FUD(("PARSER: op->mult         pop,pop\n")); 
                                    }
|  expression T_DIVIDE expression   {  
                                         ferite_do_binary_op( FERITE_OPCODE_divide );
                                         FUD(("PARSER: op->divide       pop,pop\n")); 
                                    }
|  expression T_MODULUS expression  {  
                                         ferite_do_binary_op( FERITE_OPCODE_modulus );
                                         FUD(("PARSER: op->modulus      pop,pop\n")); 
                                    }
|  expression T_EQUALS expression   {  
                                         ferite_do_binary_op( FERITE_OPCODE_equals );
                                         FUD(("PARSER: op->equals       pop,pop\n")); 
                                    }
|  expression T_LESSTHAN_EQUALS expression  
                                    {  
                                         ferite_do_binary_op( FERITE_OPCODE_less_than_equals );
                                         FUD(("PARSER: op->less_than_eq pop,pop\n")); 
                                    }
|  expression T_GRTRTHAN_EQUALS expression
                                    {  
                                         ferite_do_binary_op( FERITE_OPCODE_greater_than_equals );
                                         FUD(("PARSER: op->grtr_then_eq pop,pop\n")); 
                                    }
|  expression T_LESSTHAN expression {  
                                         ferite_do_binary_op( FERITE_OPCODE_less_than );
                                         FUD(("PARSER: op->less_than    pop,pop\n")); 
                                    }
|  expression T_GRTRTHAN expression {  
                                         ferite_do_binary_op( FERITE_OPCODE_greater_than );
                                         FUD(("PARSER: op->grtr_then    pop,pop\n")); 
                                    }
|  expression T_NOTEQUAL expression {  
                                         ferite_do_binary_op( FERITE_OPCODE_notequals );
                                         FUD(("PARSER: op->not_equal    pop,pop\n")); 
                                    }
|  expression T_LEFT_SHIFT expression               
                                    {  
                                         ferite_do_binary_op( FERITE_OPCODE_left_shift );
                                         FUD(("PARSER: op->left_shift    pop,pop\n")); 
                                    }
|  expression T_RIGHT_SHIFT expression  
                                    {  
                                         ferite_do_binary_op( FERITE_OPCODE_right_shift );
                                         FUD(("PARSER: op->right_shift    pop,pop\n")); 
                                    }
|  expression T_BINARY_OR expression                
                                    {  
                                         ferite_do_binary_op( FERITE_OPCODE_binary_or );
                                         FUD(("PARSER: op->binary_or    pop,pop\n")); 
                                    }
|  expression T_BINARY_AND expression           
                                    {  
                                         ferite_do_binary_op( FERITE_OPCODE_binary_and );
                                         FUD(("PARSER: op->binary_and   pop,pop\n")); 
                                    }
|  expression T_BINARY_XOR expression  
                                    {  
                                         ferite_do_binary_op( FERITE_OPCODE_binary_xor );
                                         FUD(("PARSER: op->binary_xor    pop,pop\n")); 
                                    }
|  cvar T_ASSIGN expression         {  
                                         ferite_do_binary_op( FERITE_OPCODE_assign );
                                         FUD(("PARSER: op->assign       pop,pop\n")); 
                                    } 
|  cvar T_PLUS_ASSIGN expression    {  
                                         ferite_do_binary_op( FERITE_OPCODE_add_assign );
                                         FUD(("PARSER: op->plus_assign  pop,pop\n")); 
                                    }
|  cvar T_MINUS_ASSIGN expression   {  
                                         ferite_do_binary_op( FERITE_OPCODE_minus_assign );
                                         FUD(("PARSER: op->minus_assign pop,pop\n")); 
                                    }
|  cvar T_MULT_ASSIGN expression    {  
                                         ferite_do_binary_op( FERITE_OPCODE_mult_assign );
                                         FUD(("PARSER: op->mult_assign  pop,pop\n")); 
                                    }
|  cvar T_DIVIDE_ASSIGN expression  {  
                                         ferite_do_binary_op( FERITE_OPCODE_divide_assign );
                                         FUD(("PARSER: op->divie_assign pop,pop\n")); 
                                    }
|  cvar T_LEFT_SHIFT_ASSIGN expression
                                    {  
                                         ferite_do_binary_op( FERITE_OPCODE_left_shift_assign );
                                         FUD(("PARSER: op->left_shift_assign pop,pop\n")); 
                                    }
|  cvar T_RIGHT_SHIFT_ASSIGN expression 
                                    {  
                                         ferite_do_binary_op( FERITE_OPCODE_right_shift_assign );
                                         FUD(("PARSER: op->right_shift_assign pop,pop\n")); 
                                    }
|  cvar T_BINARY_AND_ASSIGN expression    
                                    {  
                                         ferite_do_binary_op( FERITE_OPCODE_binary_and_assign );
                                         FUD(("PARSER: op->binary_and__assign pop,pop\n")); 
                                    }
|  cvar T_BINARY_OR_ASSIGN expression   
                                    {  
                                         ferite_do_binary_op( FERITE_OPCODE_binary_or_assign );
                                         FUD(("PARSER: op->binary_or_assign pop,pop\n")); 
                                    }
|  cvar T_BINARY_XOR_ASSIGN expression         
                                    {  
                                         ferite_do_binary_op( FERITE_OPCODE_binary_xor_assign );
                                         FUD(("PARSER: op->binary_xor_assign pop,pop\n")); 
                                    }
|  T_NEW nvar                       
   '(' parameter_list ')'           {  
                                         int *value = ferite_stack_pop( ferite_argcount_stack );
                                         ferite_do_new_object( ferite_current_arg_count );
                                         if( value != NULL )
                                         {
                                             ferite_current_arg_count = *value;
                                             ffree( value );
                                         }
                                         FUD(("PARSER: new object\n" ));
                                    }
|  T_PLUS expression %prec T_UPLUS  {  
                                         FUD(("PARSER: unary plus\n"));
                                         ferite_do_unary_op( FERITE_OPCODE_positive_var );
                                    }
|  T_MINUS expression %prec T_UMINUS                
                                    {  
                                         FUD(("PARSER: unary minus\n"));
                                         ferite_do_unary_op( FERITE_OPCODE_negative_var );
                                    }
|  cvar T_INCR                      {  
                                         FUD(("PARSER: right increment\n"));
                                         ferite_do_unary_op( FERITE_OPCODE_right_incr );
                                    }
|  T_INCR cvar                      {  
                                         FUD(("PARSER: left increment\n"));
                                         ferite_do_unary_op( FERITE_OPCODE_left_incr );
                                    }
|  cvar T_DECR                      {  
                                         FUD(("PARSER: right decrement\n"));
                                         ferite_do_unary_op( FERITE_OPCODE_right_decr );
                                    }
|  T_DECR cvar                      {  
                                         FUD(("PARSER: left decrement\n"));
                                         ferite_do_unary_op( FERITE_OPCODE_left_decr );
                                    }
|  T_NOT_OPERATOR expression        {  
                                         ferite_do_unary_op( FERITE_OPCODE_not_op );
                                         FUD(("PARSER: op->not_op           pop\n" ));
                                    }
|  T_EVAL '(' expression ')'        {  
                                         ferite_do_unary_op( FERITE_OPCODE_eval );
                                         FUD(("PARSER: op->eval\n"));
                                    }
|  T_INCLUDE '(' expression ')'     {  
                                         ferite_do_unary_op( FERITE_OPCODE_include );
                                         FUD(("PARSER: op->include\n"));
                                    }
|  expression T_ISA var_type        {
                                         ferite_temp_variable = ferite_create_string_variable_from_ptr( NULL, "strconst", ferite_variable_id_to_str( NULL,ferite_last_type), 0, FE_CHARSET_DEFAULT, FE_STATIC );
                                         MARK_VARIABLE_AS_COMPILED( ferite_temp_variable );
                                         ferite_do_push( ferite_temp_variable );
                                         ferite_do_binary_op( FERITE_OPCODE_isa );
                                         FUD(("PARSER: op->include\n"));   
                                    }
|  expression T_INSTANCEOF nvar     {
                                         ferite_do_binary_op( FERITE_OPCODE_instanceof );
                                         FUD(("PARSER: op->include\n"));   
                                    }
|  T_ARGS '(' ')'                   { ferite_do_get_args(); }
|  T_RECIPIENT '(' ')'              { ferite_do_get_yield_block(); }
|  question_colon_expr
|  array_in_script
|  function_call
|  logical_operators
|  lambda_expression
|  '(' expression ')'
|  '(' expression ')' object_list
   ;
   
question_colon_expr:
  '(' expression '?'                { ferite_do_if_statement(); }
      expression ':'                { ferite_do_after_then_before_else_statement(); }
      expression ')'                { ferite_do_after_else_statement(); }
  ;
  
optional_function_with_clause:
   /* Empty */
|  T_USING lambda_body               { ferite_do_yield_block( FE_FALSE ); }
|  T_USING T_RECIPIENT '(' ')'       { ferite_do_yield_block( FE_TRUE ); }               
   ;
   
function_call:
   T_LABEL                          
   '(' parameter_list ')'           
   optional_function_with_clause    { 
                                        int *value = ferite_stack_pop( ferite_argcount_stack );
                                        FUD(("PARSER: op->c_func       %s\n", $1 )); 
                                        ferite_do_function_call( $1, ferite_current_arg_count );
                                        ffree( $1 ); FECT;
                                        if( value != NULL )
                                        {
                                            ferite_current_arg_count = *value;
                                            ffree( value );
                                        }
                                    }
|  T_SUPER                          { ferite_do_super_push(); }
   '(' parameter_list ')'
   optional_function_with_clause    {
                                        int *value = ferite_stack_pop( ferite_argcount_stack );
                                        ferite_do_object_function_call( "constructor", ferite_current_arg_count );
                                        if( value != NULL )
                                        {
                                            ferite_current_arg_count = *value;
                                            ffree( value );
                                        }
                                    }
|  T_SELF                           { ferite_do_self_push(); }
   '(' parameter_list ')'
   optional_function_with_clause    {
                                        int *value = ferite_stack_pop( ferite_argcount_stack );
                                        ferite_do_object_function_call( "constructor", ferite_current_arg_count );
                                        if( value != NULL )
                                        {
                                            ferite_current_arg_count = *value;
                                            ffree( value );
                                        }
                                    }
|  T_DELIVER   
   '(' parameter_list ')'           { 
                                        int *value = ferite_stack_pop( ferite_argcount_stack );
                                        FUD(("PARSER: op->yield\n")); 
                                        ferite_do_yield_call( ferite_current_arg_count );
                                        if( value != NULL )
                                        {
                                            ferite_current_arg_count = *value;
                                            ffree( value );
                                        }
			            }
   ;

parameter_list:
   /* empty */                      { 
                                        int *value = fmalloc(sizeof(int));
                                        *value = ferite_current_arg_count;
                                        ferite_stack_push( ferite_argcount_stack, value );
                                        ferite_current_arg_count = 0;
                                    } 
|                                   { 
                                        int *value = fmalloc(sizeof(int));
                                        *value = ferite_current_arg_count;
                                        ferite_stack_push( ferite_argcount_stack, value );
                                        ferite_current_arg_count = 0;
                                    } 
   non_empty_function_call_parameter_list
   ;

non_empty_function_call_parameter_list:
   function_call_parameter
|  non_empty_function_call_parameter_list ',' function_call_parameter
   ;

function_call_parameter:
   { ferite_current_arg_count++ } expression
   ; 

array_in_script:
  '['                               {
                                        char *buf = fmalloc(256);
                                        ferite_var_array_count++;
                                        snprintf( buf, 256, "!__array%d__", ferite_var_array_count );
                                        ferite_do_add_variable( buf, F_VAR_UARRAY, FE_FALSE, FE_FALSE, FE_FALSE, FE_FALSE, FE_ITEM_IS_PUBLIC, FE_FALSE );                                        
                                        ferite_stack_push( ferite_compiled_arrays_stack, buf );
                                        FUD(("PARSER: push             array_start\n"));
                                        
                                        ferite_do_variable_push( buf );
                                        ferite_do_unary_op( FERITE_OPCODE_array_clear );
                                        ferite_do_pop();
                                    }
  array_elements_list ']'           {
                                        char *buf = ferite_stack_pop( ferite_compiled_arrays_stack );
                                        ferite_do_variable_push( buf );
                                        ffree( buf );
                                    }
  ;

array_elements_list:
  /* empty */
| non_empty_array_elements_list
  ;

non_empty_array_elements_list:
  array_element
| non_empty_array_elements_list ',' array_element
  ;
  
array_element:
   expression
   array_element_post               {
                                        ferite_do_variable_push( ferite_stack_top( ferite_compiled_arrays_stack ) );
                                        ferite_do_swap_top();
                                        ferite_do_binary_op( FERITE_OPCODE_array_index );
                                        ferite_do_swap_top();
                                        ferite_do_binary_op( FERITE_OPCODE_assign ); 
                                        ferite_do_pop();
                                    }
array_element_post:
   /* empty */                      {  
                                        ferite_temp_variable = ferite_create_void_variable( NULL, "void-placeholder", FE_STATIC );
                                        MARK_VARIABLE_AS_PLACEHOLDER( ferite_temp_variable );
                                        MARK_VARIABLE_AS_COMPILED( ferite_temp_variable );
                                        ferite_do_push( ferite_temp_variable );
                                    }
|  T_RARROW expression              {   ferite_do_swap_top(); }
   ;

logical_operators:
   expression T_LOGICAL_AND         { ferite_do_if_statement(); }    
   expression                       { 
                                        ferite_do_after_then_before_else_statement();
                                        /* we have to push a value onto the stack, otherwise we end up with a smashed stack */
                                        ferite_temp_variable = ferite_create_number_long_variable( NULL, "true", FE_FALSE, FE_STATIC );
                                        MARK_VARIABLE_AS_COMPILED( ferite_temp_variable );
                                        ferite_do_push( ferite_temp_variable );                             
                                        ferite_do_after_else_statement();           
                                   }
|  expression T_LOGICAL_OR         { ferite_do_not_if_statement(); }
   expression                      { 
                                        ferite_do_after_then_before_else_statement();
                                        /* we have to push a value onto the stack, otherwise we end up with a smashed stack */
                                        ferite_temp_variable = ferite_create_number_long_variable( NULL, "true", FE_TRUE, FE_STATIC );
                                        MARK_VARIABLE_AS_COMPILED( ferite_temp_variable );
                                        ferite_do_push( ferite_temp_variable );                             
                                        ferite_do_after_else_statement();           
                                   }
   ;

%%

int ferite_parse()
{  
   int parser_return = 0;
   
   FE_ENTER_FUNCTION;
   parser_return = fepparse();
   FE_LEAVE_FUNCTION( parser_return );
}

