/******************************************************************** 
   Copyright (C) 2000 Bassoukos Tassos <abas@aix.meng.auth.gr>
   
   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License
   as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later
   version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
   
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*********************************************************************/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdlib.h>
#ifdef HAVE_STRINGS_H
#include <strings.h>
#endif
#include <gnome.h>

#include "tasks.h"
#include "protocol.h"
#include "server.h"
#include "guiprefs.h"
#include "news.h"
#include "guiutils.h"
#include "privs.h"
#include "pixmap.h"

#define NEWS "News"

typedef struct {
  Connection *c;
  char *news;
  char *posted_news;
  HLTransaction *trans;
  Task *news_task;
  NotebookPage *notebook_page;
  GtkWidget *news_widget;
  GtkWidget *post_frame;
  GtkWidget *post_text_widget;
} NewsData;


static void news_do_refresh(GtkButton *b,gpointer p);

static void news_post_send_transaction(Connection *c,gpointer p){
  HLTransaction *t=(HLTransaction *)p;
  NewsData *n=(NewsData *)t->data;
  HLObject *o;
  char *m;

  transaction_add_object(t,o=create_string(HLO_MESSAGE,n->posted_news));
  string_unix_to_net(o);
  server_transaction_reply_set(n->c,t,server_reply_default,strdup(_("Could not post news:")));
  transaction_send(t,n->c);
  m=n->posted_news;
  n->posted_news=NULL;
  free(m);
}

static void news_post_abort_send(Task *task){
  HLTransaction *t=(HLTransaction *)task->user_data;
  transaction_set_task(t,NULL,0.0,0.1);
}

static void news_post_start_transaction(NewsData *n){
  HLTransaction *t;
  Task *task;
  if(n->posted_news==NULL){
    printf(_("news_post_start_transaction: nothing to do!\n"));
    return;
  }
  t=transaction_new(HLCT_POSTNEWS,(gpointer)n,FALSE);
  task=task_new(S_NAME(n->c));
  task->user_data=t;
  task->user_cancel=news_post_abort_send;
  transaction_set_task(t,task,0.0,1.0);
  task_add(task);
  server_transaction_start(n->c,news_post_send_transaction,t);
}

static void news_post_cancel(GtkButton *b,gpointer p){
  NewsData *n=(NewsData *)p;
  if(n->post_frame!=NULL)
    gtk_widget_destroy(n->post_frame);
  n->post_frame=NULL;
  n->post_text_widget=NULL;
}

static void news_post_send(GtkButton *b,gpointer p){
  NewsData *n=(NewsData *)p;
  if(n->post_frame==NULL || n->posted_news!=NULL){
    news_post_cancel(b,p);
    return;
  }
  n->posted_news=gtk_editable_get_chars(GTK_EDITABLE(n->post_text_widget),0,-1);
  news_post_cancel(b,p);
  news_post_start_transaction(n);
}

static void news_post_gui(GtkButton *b,gpointer p){
  NewsData *n=(NewsData *)p;
  GtkWidget *d,*text,*sl;
  if(n==NULL || n->post_frame!=NULL)
    return;
  d=gnome_dialog_new(_("Post a Message"),GNOME_STOCK_BUTTON_CANCEL,GNOME_STOCK_BUTTON_OK,NULL);
  text=gtk_text_new(NULL,NULL);
  gtk_text_set_editable(GTK_TEXT(text),TRUE);
  guiprefs_widget_set_font(text,&news_font);
  sl=gtk_scrolled_window_new(NULL,NULL);
  gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(sl),
				 GTK_POLICY_NEVER,
				 GTK_POLICY_ALWAYS);
  gtk_container_add(GTK_CONTAINER(sl),text);
  gtk_box_pack_start(GTK_BOX(GNOME_DIALOG(d)->vbox),sl,TRUE,TRUE,0);
  gnome_dialog_button_connect(GNOME_DIALOG(d),0,
			      GTK_SIGNAL_FUNC(news_post_cancel),
			      (gpointer)n);
  gnome_dialog_button_connect(GNOME_DIALOG(d),1,
			      GTK_SIGNAL_FUNC(news_post_send),
			      (gpointer)n);
  if(n->c->gui->main_window!=NULL)
    gnome_dialog_set_parent(GNOME_DIALOG(d),GTK_WINDOW(n->c->gui->main_window));
  gtk_signal_connect(GTK_OBJECT(d),"destroy",
		     GTK_SIGNAL_FUNC(news_post_cancel),(gpointer)n);
  guiprefs_add_window(GTK_WINDOW(d),"Main/News/post_dialog_size");
  gtk_widget_show_all(d);
  n->post_text_widget=text;
  n->post_frame=d;
}

static void news_gui_destroy(NewsData *n){
  if(n->post_frame!=NULL)
    gtk_widget_destroy(n->post_frame);
  if(n->notebook_page!=NULL)
    gutils_nbpage_destroy(n->notebook_page);
  n->post_frame=NULL;
  n->notebook_page=NULL;
  n->news_widget=NULL;
  n->post_text_widget=NULL;
}

static void close_news(GtkButton *b,gpointer p){
  NewsData *n=(NewsData *)p;
  news_gui_destroy(n);
}

static void destroy_news(Connection *c,gpointer data){
  NewsData *n=(NewsData *)data;
  news_gui_destroy(n);
  if(n->news_task!=NULL){
    n->news_task->user_cancel=NULL;
    n->news_task->user_data=NULL;
    task_remove_when_idle(n->news_task);
  }
  if(n->news!=NULL)
    free(n->news);
  if(n->posted_news!=NULL)
    free(n->news);
  free(n);
  hooks_set_data(c,NEWS,NULL);
}

static void news_abort_user(Task *p){
  NewsData *n;

  if(p==NULL || p->user_data==NULL)
    return;
  n=(NewsData *)p->user_data;
  n->news_task=NULL;
  if(n->trans!=NULL)
    n->trans->task=NULL;
  p->user_data=NULL;
  p->user_cancel=NULL;
}

static GnomeUIInfo news_toolbar[] = {
  GNOMEUIINFO_ITEM_STOCK("Close",N_("Close news"),
			 close_news,
			 HL_STOCK_PIXMAP_CLOSE_PAGE),
  GNOMEUIINFO_ITEM_STOCK("Refresh",N_("Reload news"),
			 news_do_refresh,
			 HL_STOCK_PIXMAP_REFRESH),
  GNOMEUIINFO_SEPARATOR,
  GNOMEUIINFO_ITEM_STOCK(N_("Post"),N_("Post a Message"),
			 news_post_gui,
			 HL_STOCK_PIXMAP_NEWS_POST),
  GNOMEUIINFO_END
};

static void update_news_gui(NewsData *n){
  if(n->news_widget==NULL)
    return;
  gtk_text_freeze(GTK_TEXT(n->news_widget));
  gtk_editable_delete_text(GTK_EDITABLE(n->news_widget),0,-1);
  if(n->news!=NULL)
    gtk_text_insert(GTK_TEXT(n->news_widget),NULL,NULL,NULL,n->news,-1);
  gtk_text_thaw(GTK_TEXT(n->news_widget));
}

static void create_news_gui(NewsData *n){
  GtkWidget *text,*sw;
  if(n->notebook_page!=NULL)
    return;

  n->notebook_page=gutils_nbpage_new("Main",news_toolbar,n->c,_("News"),"news",n);
  text=gtk_text_new(NULL,NULL);
  sw=gtk_scrolled_window_new(NULL,NULL);
  gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(sw),
				 GTK_POLICY_NEVER,
				 GTK_POLICY_ALWAYS);
  gtk_container_add(GTK_CONTAINER(sw),text);
  gtk_text_set_editable(GTK_TEXT(text),FALSE);
  gtk_text_set_word_wrap(GTK_TEXT(text),TRUE);
  gutils_nbpage_set_main(n->notebook_page,sw);
  n->news_widget=text;
}

static gboolean news_show_news(gpointer p){
  NewsData *n=(NewsData *)p;
  create_news_gui(n);
  update_news_gui(n);
  return FALSE;
}

static gboolean news_remove_task(gpointer p){
  NewsData *n=(NewsData *)p;
  if(n->news_task!=NULL)
    task_remove(n->news_task);
  n->news_task=NULL;
  return FALSE;
}

static void news_handle_reply(Connection *c,HLTransaction *t,HLTransaction *r,gpointer p){
  NewsData *n=(NewsData *)p;
  HLObject *o;

  n->trans=NULL;
  transaction_destroy(t);
  transaction_set_task(r,n->news_task,0.1,1.0);
  transaction_read_objects(c,r);
  o=transaction_find_object(r,HLO_MESSAGE);
  gtk_idle_add(news_remove_task,n);
  if(o==NULL){
    printf(_("News without news!\n"));
  } else {
    char *s=n->news;;
    string_net_to_unix(o);
    n->news=strdup(o->data.string);
    if(s!=NULL)
      free(s);
    gtk_idle_add(news_show_news,n);
  }
  transaction_destroy(r);
}

static void news_send_request(Connection *c,gpointer p){
  NewsData *n=(NewsData *)p;
  if(n->trans==NULL)
    return;
  server_transaction_reply_set(c,n->trans,news_handle_reply,n);
  transaction_send(n->trans,c);
}

static void init_news(Connection *c){
  NewsData *n=(NewsData *)malloc(sizeof(NewsData));
  
  n->c=c;
  n->news=NULL;
  n->posted_news=NULL;
  n->news_task=NULL;
  n->notebook_page=NULL;
  n->news_widget=NULL;
  n->post_frame=NULL;
  n->post_text_widget=NULL;
  n->trans=NULL;
  hooks_create(c,NEWS,n,NULL,destroy_news);
}

static void send_news_refresh(NewsData *n){
  HLTransaction *t;
		       
  if(n->trans!=NULL)
    return;
  n->trans=t=transaction_new(HLCT_GETNEWS,NULL,FALSE);
  n->news_task=task_new(S_NAME(n->c));
  n->news_task->user_data=(gpointer)n;
  n->news_task->user_cancel=news_abort_user;

  sprintf(n->news_task->text,_("Retrieving news..."));
  task_update(n->news_task,TRUE,0.0);
  transaction_set_task(t,n->news_task,0.0,0.1);
  task_add(n->news_task);
  server_transaction_start(n->c,news_send_request,n);
}

static void news_do_refresh(GtkButton *b,gpointer p){
  NewsData *n=(NewsData *)p;
  send_news_refresh(n);
}

void want_news(Connection *c){
  NewsData *n=(NewsData *)hooks_get_data(c,NEWS);
  if(n==NULL){
    init_news(c);
    n=(NewsData *)hooks_get_data(c,NEWS);
    send_news_refresh(n);
  }
  if(n->notebook_page==NULL){
    create_news_gui(n);
    update_news_gui(n);
  }
  gutils_nbpage_to_front(n->notebook_page);
}

static gboolean handle_server_new_post_idle(gpointer p){
  update_news_gui(p);
  return FALSE;
}
    
void handle_server_new_post(Connection *c,HLTransaction *t,gpointer data){
  HLObject *o;
  char *s;
  NewsData *n=(NewsData *)hooks_get_data(c,NEWS);

  transaction_read_objects(c,t);
  o=transaction_find_object(t,HLO_MESSAGE);
  if(o==NULL || n==NULL){
    printf(_("New News Post without text....\n"));
  } else {
    string_net_to_unix(o);
    s=g_strconcat(o->data.string,n->news,NULL);
    free(n->news);
    n->news=s;
    gtk_idle_add(handle_server_new_post_idle,n);
  }
  transaction_destroy(t);
}

void check_news(Connection *c){
  if(auto_open_news==TRUE)
    want_news(c);
}
