/******************************************************************** 
   Copyright (C) 2000 Bassoukos Tassos <abas@aix.meng.auth.gr>
   
   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License
   as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later
   version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
   
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*********************************************************************/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#ifdef HAVE_MALLOC_H
#include <malloc.h>
#endif
#include <gnome.h>
#include <gtk/gtk.h>
#include <gtk/gtkprogressbar.h>
#include <glib.h>

#include "tasks.h"
#include "threads.h"
#include "guiprefs.h"
#include "main.h"
#include "server.h"

static GtkWidget *task_list = NULL;
static pthread_mutex_t task_mutex=PTHREAD_MUTEX_INITIALIZER;
static unsigned int TaskCount = 0;
static GSList *Tasks=NULL;

/* Whether update_tasks runs or not */
static gboolean update = FALSE;

static void task_clicked_abort(GtkButton *but,gpointer data);

Task *task_new_sname(const Connection *c)
{
	return task_new(S_NAME(c));
}

/* Initializes a Task structure and creates the associated gtk widgets */
Task *task_new(const gchar *description){
  Task *t;
  GtkWidget *hbox,*vbox,*button;
  t=calloc(sizeof(Task),1);
  t->list_pos=-1;
  t->is_running=FALSE;
  t->user_cancel=NULL;
  t->percdone=0.0;
  t->is_changed=0;
  t->widget=vbox=gtk_vbox_new(FALSE,0);
  hbox=gtk_hbox_new(FALSE,0);
  gtk_box_pack_start(GTK_BOX(vbox),hbox,TRUE,TRUE,0);
  button=gtk_button_new_with_label(_("Abort"));
  gtk_box_pack_start(GTK_BOX(hbox),button,FALSE,FALSE,1);
  gtk_signal_connect(GTK_OBJECT(button),
		     "clicked",
		     GTK_SIGNAL_FUNC(task_clicked_abort),
		     t);
  t->description=gtk_label_new(description);
  gtk_label_set_justify(GTK_LABEL(t->description),GTK_JUSTIFY_LEFT);
  gtk_box_pack_start(GTK_BOX(hbox),t->description,TRUE,TRUE,0);
  t->progressbar=gtk_progress_bar_new();
  gtk_widget_set_usize(t->progressbar,300,0);
  gtk_box_pack_end(GTK_BOX(hbox),t->progressbar,FALSE,FALSE,0);
  t->text[0]=0;
  t->label=gtk_label_new("---");
  gtk_label_set_justify(GTK_LABEL(t->label),GTK_JUSTIFY_LEFT);
  gtk_box_pack_end(GTK_BOX(vbox),t->label,FALSE,FALSE,0);
  return t;
}

static void order_task(Task *t, gpointer ignored)
{
  if(t->list_pos==-1)
    {
      gtk_box_pack_start(GTK_BOX(task_list),t->widget,FALSE,FALSE,2);
      gtk_widget_show_all(t->widget);
    }
  if(t->list_pos!=TaskCount)
    gtk_box_reorder_child(GTK_BOX(task_list),t->widget,(t->list_pos = TaskCount));
}

static void reorder_tasks(){
  
  tasks_count_set_title();
  set_cancel_task_widget_state(Tasks ? TRUE : FALSE);
  if(!Tasks)
    return;
  
  pthread_mutex_lock(&task_mutex);
  g_slist_foreach(Tasks, (GFunc) order_task, NULL);
  pthread_mutex_unlock(&task_mutex);
}

static void check_task(Task *t, gboolean *change)
{
  if(!t->is_changed)
    return;
  if(t->is_changed & TASK_CHANGED_POS)
    {
      gtk_progress_bar_update(GTK_PROGRESS_BAR(t->progressbar),t->percdone);
      if(t->is_running==FALSE && t->percdone>0.0)
	{
	  *change=TRUE;
	  t->is_running=TRUE;
	}
    }
  if(t->is_changed & TASK_CHANGED_TEXT)
    gtk_label_set_text(GTK_LABEL(t->label),t->text);
  t->is_changed=0;
}

static gboolean update_tasks(int refresh){
  gboolean need_reorder=FALSE;
  static int wait = 0;
  
  if(!Tasks) /* If there's nothing to do we stop checking after 10 secs */
    return update =(++wait*prefs_task_refresh_msec>10000 ? FALSE : TRUE);
  
  pthread_mutex_lock(&task_mutex);
  g_slist_foreach(Tasks, (GFunc)check_task, &need_reorder);
  pthread_mutex_unlock(&task_mutex);
  
  wait = 0;
  
  if(need_reorder)
    reorder_tasks();
  
  if(refresh != prefs_task_refresh_msec)
    {
      g_timeout_add(prefs_task_refresh_msec, (GSourceFunc)update_tasks, (gpointer)prefs_task_refresh_msec);
      return FALSE;
    }
  return TRUE;
}

struct tasks_count {int active,count;};
static void count_active_tasks(Task *t, struct tasks_count *tasks){
  tasks->count++;
  if(t->is_running || t->percdone>0.0)
    tasks->active++;
}
static void count_connections(Connection *c, gpointer count){(*(int *)count)++;}
void tasks_count_set_title()
{
  char buf1[256],buf2[256];
  
  struct tasks_count tasks={0,0};
  int connections_active=0;
  
  g_slist_foreach(Tasks, (GFunc)count_active_tasks, &tasks);
  forall_servers(count_connections,&connections_active);
  if(tasks.count==0 && connections_active==0){
    strcpy(buf2,_("idle"));
  } else {
    if(tasks.count==tasks.active){
      if(tasks.count==0) {
	sprintf(buf1,_("no"));
      } else {
	sprintf(buf1,_("%d active"),tasks.active);
      }
    } else {
      if(tasks.active==0){
	sprintf(buf1,_("%d inactive"),tasks.count);
      } else {
	sprintf(buf1,_("%d active of %d"),tasks.active,tasks.count);
      }
    }
    sprintf(buf2,_("%s %s, %d %s"),
	    buf1,(tasks.count!=1)?_("tasks"):_("task"),
	    connections_active,(connections_active!=1)?_("connections"):_("connection"));
  }
  sprintf(buf1,_("%s (%s)"),APPLICATION_NAME,buf2);
  gtk_window_set_title(GTK_WINDOW(main_app),buf1);
}

void task_add(Task *t)
{
  /* FIXME queuing */
  pthread_mutex_lock(&task_mutex);
  Tasks = g_slist_append(Tasks, t);
  TaskCount++;
  pthread_mutex_unlock(&task_mutex);
  if(!update)
    {
      update = TRUE;
      g_timeout_add(prefs_task_refresh_msec, (GSourceFunc)update_tasks, (gpointer)prefs_task_refresh_msec);
    }
  reorder_tasks();
  tasks_count_set_title();
}

static void dec_pos(Task *t, int pos)
{
  if(t->list_pos > pos)
    t->list_pos--;
}

static void nt_task_remove(Task *t){
  
  if(t->widget)
    gtk_widget_destroy(t->widget);
  
  Tasks = g_slist_remove(Tasks, t);
  TaskCount--;
  
  g_slist_foreach(Tasks, (GFunc)dec_pos, (gpointer)t->list_pos);    
  free(t);
  tasks_count_set_title();
}

void task_remove(Task *t){
  pthread_mutex_lock(&task_mutex);
  nt_task_remove(t);
  pthread_mutex_unlock(&task_mutex);
  reorder_tasks();
}

static gboolean task_do_deferred_remove(gpointer data){
  task_remove((Task *)data);
  return FALSE;
}

void task_remove_when_idle(Task *t){
  gtk_idle_add(task_do_deferred_remove,t);
}

static void nt_user_cancel_task(Task *t){
  if(t->user_cancel)
    t->user_cancel(t);
  nt_task_remove(t);
}

void task_newtext(Task *t, gfloat pos, const char *fmt, ...)
{
  va_list ap;
  va_start(ap, fmt);
  g_vsnprintf(t->text, sizeof(t->text)-1, fmt, ap);
  task_update(t, TRUE, pos); 
  va_end(ap);
}

void task_update(Task *t,gboolean new_text,gfloat pos){
  if(pos>=0.0 && pos<=1.0 && t->percdone!=pos){
    t->percdone=pos;
    t->is_changed|=TASK_CHANGED_POS;
  }
  if(new_text)
    t->is_changed|=TASK_CHANGED_TEXT;
}

void remove_all_tasks(void){
  pthread_mutex_lock(&task_mutex);
  while(Tasks)
    nt_user_cancel_task(Tasks->data);
  pthread_mutex_unlock(&task_mutex);
  reorder_tasks();
}

static void task_clicked_abort(GtkButton *but,gpointer data){
  pthread_mutex_lock(&task_mutex);
  nt_user_cancel_task((Task *)data);
  pthread_mutex_unlock(&task_mutex);
  reorder_tasks();
}

GtkWidget *new_task_widget(){
  return task_list=gtk_vbox_new(FALSE,0);
}
