#   ----------------------------------------------------------------------------
#   "THE BEER-WARE LICENSE" (Revision 42):
#   Daniel Kratzert <dkratzert@gmx.de> wrote this file.  As long as you retain
#   this notice you can do whatever you want with this stuff. If we meet some day,
#   and you think this stuff is worth it, you can buy me a beer in return.
#   ----------------------------------------------------------------------------
import os
import unittest
from pathlib import Path

from PyQt5.QtCore import Qt
from PyQt5.QtGui import QColor

from finalcif.appwindow import AppWindow
from finalcif.gui.custom_classes import COL_CIF, COL_DATA, COL_EDIT
from finalcif.tools.misc import unify_line_endings


class TestMainTableFieldBehavior(unittest.TestCase):

    def setUp(self) -> None:
        self.testcif = Path('tests/examples/1979688.cif').absolute()
        self.myapp = AppWindow(self.testcif, unit_test=True)
        self.myapp.running_inside_unit_test = True
        self.myapp.hide()  # For full screen view
        self.myapp.settings.empty_deleted_list()

    def tearDown(self) -> None:
        self.myapp.cif.finalcif_file.unlink(missing_ok=True)
        self.myapp.close()

    def key_row(self, key: str) -> int:
        return self.myapp.ui.cif_main_table.row_from_key(key)

    def cell_widget(self, row: int, col: int) -> str:
        return str(self.myapp.ui.cif_main_table.cellWidget(row, col).__class__)

    def cell_text(self, key: str, col: int) -> str:
        return unify_line_endings(self.myapp.ui.cif_main_table.getTextFromKey(key, col))

    def equipment_click(self, field: str):
        self.myapp.ui.EquipmentTemplatesStackedWidget.setCurrentIndex(0)
        item = self.myapp.ui.EquipmentTemplatesListWidget.findItems(field, Qt.MatchStartsWith)[0]
        self.myapp.ui.EquipmentTemplatesListWidget.setCurrentItem(item)
        self.myapp.equipment.load_selected_equipment()

    def get_background_color(self, key: str, col: int) -> QColor:
        return self.myapp.ui.cif_main_table.itemFromKey(key, col).background().color()

    ######

    def test_rowcounts(self):
        self.assertEqual(131, self.myapp.ui.cif_main_table.rowCount())

    def test_delete_row(self):
        self.myapp.ui.cif_main_table.delete_row(self.key_row('_audit_update_record'))
        self.assertEqual(130, self.myapp.ui.cif_main_table.rowCount())

    def test_delete_and_reappear(self):
        self.myapp.ui.cif_main_table.delete_row(
            self.myapp.ui.cif_main_table.row_from_key('_atom_sites_solution_primary'))
        # cline count stays the same:
        self.assertEqual(131, self.myapp.ui.cif_main_table.rowCount())
        self.assertEqual('?', self.cell_text('_atom_sites_solution_primary', COL_CIF))
        # method comes from solution program now:
        self.assertEqual('direct', self.cell_text('_atom_sites_solution_primary', COL_DATA))
        # This is an essential key, it reappears after reload:
        self.assertEqual(0, self.key_row('_atom_sites_solution_primary'))

    def test_get_text_by_key(self):
        self.assertEqual('geom', self.cell_text('_atom_sites_solution_hydrogens', COL_CIF))
        self.assertEqual('', self.cell_text('_atom_sites_solution_hydrogens', COL_DATA))
        self.assertEqual('', self.cell_text('_atom_sites_solution_hydrogens', COL_EDIT))

    def test_Crystallographer_in_equipment_list(self):
        self.assertEqual('Crystallographer Details', self.myapp.ui.EquipmentTemplatesListWidget.item(1).text())

    def test_load_equipment(self):
        self.myapp.equipment.import_equipment_from_file('test-data/Crystallographer_Details.cif')
        # make sure contact author is selected
        self.equipment_click('Crystallographer Details')
        # It is important here, that the first column has 'dkratzert@gmx.de' in it:
        self.assertEqual('?',
                         self.myapp.ui.cif_main_table.getTextFromKey('_audit_contact_author_email', COL_CIF))
        self.assertEqual('dkratzert@gmx.de',
                         self.myapp.ui.cif_main_table.getTextFromKey('_audit_contact_author_email', COL_DATA))
        self.assertEqual('dkratzert@gmx.de',
                         self.myapp.ui.cif_main_table.getTextFromKey('_audit_contact_author_email', COL_EDIT))

    def test_field_types(self):
        self.assertEqual("<class 'NoneType'>",
                         str(self.myapp.ui.cif_main_table.itemFromKey('_atom_sites_solution_hydrogens',
                                                                      COL_CIF).__class__))
        self.assertEqual("<class 'finalcif.gui.plaintextedit.MyQPlainTextEdit'>",
                         str(self.myapp.ui.cif_main_table.widget_from_key('_atom_sites_solution_hydrogens',
                                                                          COL_CIF).__class__))

    def test_combobox_field(self):
        self.assertEqual("<class 'finalcif.gui.combobox.MyComboBox'>",
                         str(self.myapp.ui.cif_main_table.widget_from_key('_atom_sites_solution_hydrogens',
                                                                          COL_EDIT).__class__))

    def test_plaintextedit_field(self):
        self.assertEqual("<class 'finalcif.gui.plaintextedit.MyQPlainTextEdit'>",
                         str(self.myapp.ui.cif_main_table.widget_from_key('_audit_contact_author_address',
                                                                          COL_CIF).__class__))
        self.assertEqual("<class 'finalcif.gui.plaintextedit.MyQPlainTextEdit'>",
                         str(self.myapp.ui.cif_main_table.widget_from_key('_audit_contact_author_address',
                                                                          COL_DATA).__class__))
        self.assertEqual("<class 'finalcif.gui.plaintextedit.MyQPlainTextEdit'>",
                         str(self.myapp.ui.cif_main_table.widget_from_key('_audit_contact_author_address',
                                                                          COL_EDIT).__class__))
