#include "backend.h"

#include <openssl/pem.h>

// some important outputs for debugging
#define DEBUG 1

// further outputs (entering and leaving procedures ...)
#define ERRORDEBUG 0

// when debugging is easier without this thread use this one
//#define SKIPTIMEOUTTHREAD

// constructor of tentry. (tentry is used for timeouting rules, the queue of "pending packets", ...
tentry::tentry(char *buffer, packet *ptr, int act)
{
	buf=new char[16];

	if (!buffer)
	{
		cerr << __LINE__ << ": buffer is null" << endl;
	}
	else if (!ptr)
	{
		cerr << __LINE__ << ": ptr is null" << endl;
	}
	else
	{
		if (buffer[11]||buffer[12]||buffer[13]||buffer[14]) // timeout is specified at buffer[11 - 14]
			timeout=time(0)+(((((((unsigned char)buffer[11])<<8)+(unsigned char)buffer[12])<<8)+(unsigned char)buffer[13])<<8)+(unsigned char)buffer[14];
		else
			timeout=0;
		memcpy(buf, buffer, 16);
		pack=new packet();
		action=act;
		memcpy(pack, ptr, sizeof(class packet));
		if (ptr->programname)
		{
			pack->programname=strdup(ptr->programname);
		}
		else
			pack->programname=0;
		if ((ptr->len>0) && (ptr->data))
		{
			pack->data=new char[ptr->len];
			memcpy(pack->data, ptr->data, ptr->len);
		}
		else
			pack->data=0;
	}
}

tentry::~tentry()
{
	if (ERRORDEBUG)
		cout << "!tentry" << endl;
	delete[] buf;
	delete(pack);
}

packet::~packet()
{
	if (ERRORDEBUG)
		cout << "!packet" << endl;
	if (data)
		delete[] data;
	if (programname)
		delete(programname);
}

// log a message to the system logger
// used on important errors
void system_log(char *message)
{
	openlog("fireflier", 0, LOG_DAEMON);
	syslog(LOG_ERR, "%s\n", message);
	closelog();
}

void free_ssl_auth()
{
	SSL_free(ssl_auth);
	ssl_auth=0;
}

bool file_readable(string filename)
{
    struct stat st;
	return (stat(filename.c_str(), &st)!=-1);
}

char *int_to_charptr(u_int32_t number, char *temp, int maxlen)
{
	int i;
	snprintf(temp, maxlen, "%u", number);
    temp[maxlen-1]=0;
    return temp;
}

char *int_to_ipaddr(u_int32_t number, char *temp, int maxlen)
{
	int i;
	struct in_addr addr;
    addr.s_addr=number;
	strncpy(temp, inet_ntoa(addr), maxlen);
    temp[maxlen-1]=0;
    return temp;
}


// send current packet information to client
void sendpacket()
{
	char *buffer;
	int count;
	if (!ssl_auth) return;
	if (!currentPacket) return;
	buffer=new char[sizeof(struct packet)-4+3+((currentPacket->programname)?strlen(currentPacket->programname)+1:1)];
	if (!buffer)
	{
		cerr << "LINE " << __LINE__ << ": COULD NOT ALLOCATE MEMORY!!!" << endl;
		system_log("Out of memory");
		return;
	}

	buffer[0]=0;
	buffer[3]=(currentPacket->packet_id>>24)&255;  // packet_id
	buffer[4]=(currentPacket->packet_id>>16)&255;
	buffer[5]=(currentPacket->packet_id>>8)&255;
	buffer[6]=(currentPacket->packet_id)&255;

	buffer[7]=(currentPacket->arrived>>24)&255;  // arrived
	buffer[8]=(currentPacket->arrived>>16)&255;
	buffer[9]=(currentPacket->arrived>>8)&255;
	buffer[10]=(currentPacket->arrived)&255;

	buffer[11]=(currentPacket->len>>8)&255; // len
	buffer[12]=(currentPacket->len)&255;

	buffer[13]=(currentPacket->ip_src>>24)&255; // source ip
	buffer[14]=(currentPacket->ip_src>>16)&255;
	buffer[15]=(currentPacket->ip_src>>8)&255;
	buffer[16]=(currentPacket->ip_src)&255;

	buffer[17]=(currentPacket->ip_dst>>24)&255; // destination ip
	buffer[18]=(currentPacket->ip_dst>>16)&255;
	buffer[19]=(currentPacket->ip_dst>>8)&255;
	buffer[20]=(currentPacket->ip_dst)&255;

	buffer[21]=currentPacket->protocol; // protocol

	buffer[22]=(currentPacket->port_src>>8)&255; // source port
	buffer[23]=(currentPacket->port_src)&255;

	buffer[24]=(currentPacket->port_dst>>8)&255; // destination port
	buffer[25]=(currentPacket->port_dst)&255;

	buffer[26]=(currentPacket->tcp_flags>>8)&255; // tcp flags
	buffer[27]=(currentPacket->tcp_flags)&255;

	buffer[28]=(currentPacket->icmp_type); // icmp_type

	buffer[29]=currentPacket->hook; // firewall chain

	buffer[30]=currentPacket->mac_addrlen;  // mac address length

	memcpy(buffer+31, currentPacket->mac_addr, 8); // mac address

	memcpy(buffer+39, currentPacket->interface_in, IFNAMSIZ); // incoming interface

	memcpy(buffer+39+IFNAMSIZ, currentPacket->interface_out, IFNAMSIZ); // outgoing interface

	if (currentPacket->programname) // program using localport/localip of this packet
	{
		buffer[39+2*IFNAMSIZ]=strlen(currentPacket->programname)+1;
		memcpy(buffer+39+2*IFNAMSIZ+1, currentPacket->programname, strlen(currentPacket->programname)+1);
		buffer[1]=(39+2*IFNAMSIZ+strlen(currentPacket->programname)+2)>>8;
		buffer[2]=(39+2*IFNAMSIZ+strlen(currentPacket->programname)+2)&255;
	}
	else
	{
		buffer[39+2*IFNAMSIZ]=0;
		buffer[1]=(39+2*IFNAMSIZ+1)>>8;
		buffer[2]=(39+2*IFNAMSIZ+1)&255;
	}

	// send the data
	if ((count=SSL_write(ssl_auth, buffer, 40+2*IFNAMSIZ+((currentPacket->programname)?strlen(currentPacket->programname)+1:0)))<=-1)
	{
		// on error disconnect client
		//		clientConnected=0;
		free_ssl_auth();
		cerr << "LINE " << __LINE__ << ": SEND FAILED!!!" << endl;
	}
	else if (DEBUG)
		cout << "sent " << count << "bytes" << endl;
	delete[] buffer;
}

// send source of current packet
void sendsource()
{
	char *buffer;

	if (!ssl_auth) return;
	if (!currentPacket) return;
	if (currentPacket->len<=0) return;
	if (!currentPacket->data) return;

	buffer=new char[currentPacket->len+1]; // allocate buffer
	if (!buffer)
	{
		cerr << "LINE " << __LINE__ << ": COULD NOT ALLOCATE MEMORY!!!" << endl;
		system_log("Out of memory");
		return;
	}
	buffer[0]=ff::CMD_SOURCE; // copy message to buffer
	memcpy(buffer+1, currentPacket->data, currentPacket->len);
	if (DEBUG)
		cout << "Sending " << currentPacket->len+1 << " source bytes" << endl;

	// send the data
	if (SSL_write(ssl_auth, buffer, currentPacket->len+1)<=0)
	{
		// on error disconnect client
		free_ssl_auth();
		//		clientConnected=0;
		if (DEBUG)
			cerr << "LINE " << __LINE__ << ": SEND FAILED!!!" << endl;
	}
	delete[] buffer;
}

// not used at the moment. usefull for debugging ssl problems
void SSL_print_error(int errorcode)
{
	switch (errorcode) {
	case SSL_ERROR_NONE:
		cout << "SSL Operation successful" << endl;
		break;
	case SSL_ERROR_ZERO_RETURN:
		cout << "SSL socket shutdown cleanly" << endl;
		break;
	case SSL_ERROR_WANT_READ:
		cout << "SSL want read" << endl;
		break;
	case SSL_ERROR_WANT_WRITE:
		cout << "SSL want write" << endl;
		break;
	case SSL_ERROR_WANT_CONNECT:
		cout << "SSL want connect" << endl;
		break;
	case SSL_ERROR_WANT_X509_LOOKUP:
		cout << "SSL want x509 lookup" << endl;
		break;
	case SSL_ERROR_SYSCALL:
		cout << "SSL io error" << endl;
		break;
	case SSL_ERROR_SSL:
		cout << "SSL protocol error" << endl;
		break;
	default:
		cout << "SSL Unknown error" << endl;
	}
}

// send the current userspace rules to the client in human readable form
void senduserspacerules()
{
	char *buffer;
	int count;
	tentry *p;
	int len=0;
	char temp[50];
	if (ERRORDEBUG)
		cout << __LINE__ << "+senduserspacerules" << endl;

	if (!ssl_auth) return;
	if (!userspacequeue) return;

	// determine size needed (not exact size, but "big enough" size)
	userspacequeue->rewind();
	while ((p=((tentry*)userspacequeue->next_element())))
	{
		len+=200; // just some value which is big enough
	}

	// allocate size
	buffer=new char[len+3];
	buffer[0]=ff::CMD_URULES;

	strcpy(buffer+3, "");
	userspacequeue->rewind();
	while ((p=(tentry*)userspacequeue->next_element())) // for each userspace rule: add to sendbuffer
	{
		strcat(buffer+3, p->pack->programname);
		if (p->action==ff::ACTION_ACCEPT)
			strcat(buffer+3, " ACCEPT");
		else
			strcat(buffer+3, " DROP");
		if (p->timeout) // timeout of packet
		{
			strcpy(temp, ctime((long*)&(p->timeout)));
			temp[strlen(temp)-1]=0;
			strcat(buffer+3, " Timeout: ");
			strcat(buffer+3, temp);
		}
		strcat(buffer+3, " filters: "); // filters on this rule
		if (p->buf[ff::SB_IP_SRC]) // source ip
		{
			sprintf(temp, "SRCIP=%d.%d.%d.%d", ((unsigned char*)&p->pack->ip_src)[0], ((unsigned char*)&p->pack->ip_src)[1], ((unsigned char*)&p->pack->ip_src)[2], ((unsigned char*)&p->pack->ip_src)[3]);
			strcat(buffer+3, temp);
		}
		if (p->buf[ff::SB_IP_DST]) // destination ip
		{
			sprintf(temp, " DSTIP=%d.%d.%d.%d", ((unsigned char*)&p->pack->ip_dst)[0], ((unsigned char*)&p->pack->ip_dst)[1], ((unsigned char*)&p->pack->ip_dst)[2], ((unsigned char*)&p->pack->ip_dst)[3]);
			strcat(buffer+3, temp);
		}
		if (p->buf[ff::SB_PORT_SRC]) // source port
		{
			sprintf(temp, " SRCPORT=%d", p->pack->port_src);
			strcat(buffer+3, temp);
		}
		if (p->buf[ff::SB_PORT_DST]) // destination port
		{
			sprintf(temp, " DSTPORT=%d", p->pack->port_dst);
			strcat(buffer+3, temp);
		}
		if (p->buf[ff::SB_INTERFACE_IN]) // incoming interface
		{
			sprintf(temp, " Interface(incoming)=%s", p->pack->interface_in);
			strcat(buffer+3, temp);
		}
		if (p->buf[ff::SB_INTERFACE_OUT]) // outgoing interface
		{
			sprintf(temp, " Interface(outgoing)=%s", p->pack->interface_out);
			strcat(buffer+3, temp);
		}
		if (p->buf[ff::SB_PROTOCOL]) // protocol
		{
			sprintf(temp, " Protocol=%d", p->pack->protocol);
			strcat(buffer+3, temp);
		}
		strcat(buffer+3, "\n");
	}

	if (DEBUG)
		cout << "Sending " << len+3 << " userspace rules bytes" << endl;

	len=strlen(buffer+3)+3;
	buffer[1]=(len-3)>>8; // length of sendbuffer, (buffer[0]=msgtype)
	buffer[2]=(len-3)&255;

	// send the data
	if ((count=SSL_write(ssl_auth, buffer, len))<=0)
	{
		// on error disconnect the client
		free_ssl_auth();
		//		clientConnected=0;
		if (DEBUG)
			cerr << "LINE " << __LINE__ << ": SEND FAILED!!!" << endl;
	}
	if (DEBUG)
		cout << "Sent " << count << " userspace rules bytes" << endl;
	delete[] buffer;
	if (ERRORDEBUG)
		cout << __LINE__ << "-senduserspacerules" << endl;
}

// send iptables rules to client
void sendrules()
{
	char *buffer;
	int count;
	int len;
	char *rules;

	if (ERRORDEBUG)
		cout << __LINE__ << "+sendrules" << endl;

	// get all rules of the three main chains
	rules=getChains();

	if (!ssl_auth) return;
	if (!rules) return;

	len=strlen(rules); // ensuring 0 byte in getChains
	if (len<=0) return;

	buffer=new char[len+3];
	if (!buffer)
	{
		cerr << "LINE " << __LINE__ << ": COULD NOT ALLOCATE MEMORY!!!" << endl;
		return;
	}
	buffer[0]=ff::CMD_RULES;
	buffer[1]=len>>8; // size of message
	buffer[2]=len&255;

	memcpy(buffer+3, rules, len);
	// send the data
	if (DEBUG)
		cout << "Sending " << len+3 << " rules bytes" << endl;
	if ((count=SSL_write(ssl_auth, buffer, len+3))<=0)
	{
		// on error disconnect client
		//		clientConnected=0;
		free_ssl_auth();
		if (DEBUG)
			cerr << "LINE " << __LINE__ << ": SEND FAILED!!!" << endl;
	}
	if (DEBUG)
		cout << "Sent " << count << " rules bytes" << endl;
	delete[] buffer;
	delete[] rules;
	if (ERRORDEBUG)
		cout << __LINE__ << "-sendrules" << endl;
}

// ipq error function
static void die(struct ipq_handle *h)
{
	if (rules_active)
	{
		deleteRule(ff::CHAIN_INPUT, 0, 0, 0, 0, 0, 0, 0, 0, ff::ACTION_QUEUE);
		deleteRule(ff::CHAIN_OUTPUT, 0, 0, 0, 0, 0, 0, 0, 0, ff::ACTION_QUEUE);
		deleteRule(ff::CHAIN_FORWARD, 0, 0, 0, 0, 0, 0, 0, 0, ff::ACTION_QUEUE);
		rules_active=0;
	}
	ipq_perror("abort");
	ipq_destroy_handle(h);
	system_log("ipq_error. server dies");
	exit(1);
}

// advance to next packet. usually after accepting or denying a packet
void nextpacket()
{
	if (currentPacket)
	{
		delete(currentPacket);
	}
	pthread_mutex_lock(&lock_mutex);
	currentPacket=(packet*)packetqueue->get();
	pthread_mutex_unlock(&lock_mutex);
	if (ERRORDEBUG)
		cout << "nextpacket_done" << endl;
}

// a routine for logging purposes
int my_ipq_set_verdict(ipq_handle *hand, unsigned long packet_id, int action, int bla, void *ptr)
{
	//    cout << ((action==ff::ACTION_DENY)?"DROP: ":"ACCEPT: ") << packet_id << endl;
	return ipq_set_verdict(h, packet_id, action, bla, NULL);
}

// after we have created a rule to accept or deny packets in iptables,
// this procedure deletes all packets, which would be covered by that rule
void filterRules(unsigned int chainid, unsigned int protocol, unsigned long int srcip, unsigned long int dstip, int srcport, int dstport, char *interface_in, char *interface_out, int conntrack, int action)
{
	class queue *newqueue=new queue();
	int filter;
	packet *temp;
	int status;

	if (!currentPacket) return;

	if (ERRORDEBUG)
		cout << "+filter" << endl;
	pthread_mutex_lock(&lock_mutex);
	while ((temp=(packet*)packetqueue->get())) // walk through packetqueue
	{
		filter=1;
		status=0;

		if (chainid!=temp->hook) // check filters
			filter=0;
		if (srcip && (srcip!=temp->ip_src))
			filter=0;
		if (dstip && (dstip!=temp->ip_dst))
			filter=0;
		if (srcport && (srcport!=temp->port_src))
			filter=0;
		if (dstport && (dstport!=temp->port_dst))
			filter=0;
		if (interface_in && (strcmp(interface_in, (char*)temp->interface_in)))
			filter=0;
		if (interface_out && (strcmp(interface_out, (char*)temp->interface_out)))
			filter=0;
		if ((protocol || srcport || dstport) && (protocol!=temp->protocol))
			filter=0;

		if (filter) // if packet matches current rule accept/deny it
		{
			status = my_ipq_set_verdict(h, temp->packet_id, (action==ff::ACTION_ACCEPT)?NF_ACCEPT:NF_DROP, 0, NULL);
			delete(temp);
		}
		else // otherwise reinsert it to the new queue
			newqueue->add(temp);

		if (status<0)
		{
			cerr << "Internal ip_queue error: " << __LINE__ << endl;
			ipq_perror(ipq_errstr());
			die(h);
		}
	}
	delete (packetqueue);
	packetqueue=newqueue; // reassign packetqueue to the new packetqueue
	pthread_mutex_unlock(&lock_mutex);
	if (ERRORDEBUG)
		cout << "-filter" << endl;
}

// input a rule for userspace filtering
// buffer tells by which elements of pack there has to be filtered
tentry *inputUserspaceRule(char *buffer, packet *pack, int action)
{
	tentry *p;
	if (ERRORDEBUG)
		cout << "+-inputuserspacerule" << endl;
	if (!buffer || !pack) return 0;

	if (!pack->programname)
		return 0;
	p=new tentry(buffer, pack, action);
	if (!p)
	{
		cerr << "NEW returned NULL" << endl;
		system_log("Out of memory");
		return 0;
	}
	if (buffer[11]||buffer[12]||buffer[13]||buffer[14]) // timeout is specified at buffer[11 - 14]
		p->timeout=time(0)+(((((((unsigned char)buffer[11])<<8)+(unsigned char)buffer[12])<<8)+(unsigned char)buffer[13])<<8)+(unsigned char)buffer[14];
	else
		p->timeout=0;

	userspacequeue->add(p);
	// on every new rule save rules (if the server should crash or be terminated by SIGKILL)
	// the new rules remain
	saveUserspaceRulesXml(Config_save_file);

	return p;
}

// delete a userspace rule by rule definition.
// used by timeoutThread
void deleteUserspaceRule(char *buffer, packet *pack, int action)
{
	tentry *p;
	int filter=1;

	if (!buffer || !pack) return;

	userspacequeue->rewind();
	while ((p=(tentry*)userspacequeue->next_element())) // look at every rule
	{
		filter=1;
		if (action!=p->action) // some filters
			filter=0;
		if (memcmp(buffer, p->buf, 16)!=0)
			filter=0;
		if (memcmp(pack, p->pack, sizeof(class packet)-8)!=0) // dont compare pointers to programname and packet source
			filter=0;
		if ((pack->programname && p->pack->programname) && (strcmp(p->pack->programname, pack->programname)))
			filter=0;
		if (pack->programname && !p->pack->programname)
			filter=0;
		if (!pack->programname && p->pack->programname)
			filter=0;

		if (filter) // rule matches --> delete
		{
			if (DEBUG)
				cout << "Deleting userspace rule" << endl;
			userspacequeue->del(p);
			return;
		}
	}
}

// after a new userspace rule is added, this procedure filters out all
// rules from the "pending packets", which match to this rule
// new packets wont be added to pending packets any more
void filterUserspaceRules(tentry *p)
{
	class queue *newqueue=new queue();
	int filter;
	packet *temp;
	int status;

	if (!p)
	{
		cerr << "Fatal error: p is null in filterUserspaceRules - continuing" << endl;
        return;
	}
	if (!p->buf)
	{
		cerr << "Fatal error: p->buf is null in filterUserspaceRules - continuing" << endl;
        return;
	}

	if (!currentPacket) return;

	if (ERRORDEBUG)
		cout << "+userfilter" << endl;
	pthread_mutex_lock(&lock_mutex);
	while ((temp=(packet*)packetqueue->get())) // for each "pending" packet
	{
		filter=1;
		status=0;

		if (p->buf[ff::SB_IP_SRC] && (p->pack->ip_src!=temp->ip_src)) // some filters
			filter=0;
		if (p->buf[ff::SB_IP_DST] && (p->pack->ip_dst!=temp->ip_dst))
			filter=0;
		if (p->buf[ff::SB_PORT_SRC] && (p->pack->port_src!=temp->port_src))
			filter=0;
		if (p->buf[ff::SB_PORT_DST] && (p->pack->port_dst!=temp->port_dst))
			filter=0;
		if (p->buf[ff::SB_INTERFACE_IN] && (strcmp((char*)p->pack->interface_in, (char*)temp->interface_in)))
			filter=0;
		if (p->buf[ff::SB_INTERFACE_OUT] && (strcmp((char*)p->pack->interface_out, (char*)temp->interface_out)))
			filter=0;
		if ((p->buf[ff::SB_PROTOCOL] || p->buf[ff::SB_PORT_SRC] || p->buf[ff::SB_PORT_DST]) && (p->pack->protocol!=temp->protocol))
			filter=0;
		if (!temp->programname)
			filter=0;
		else if (strcmp(temp->programname, p->pack->programname))
			filter=0;

		if (filter) // packet matches? --> tell ipq to drop/accept and delete packet
		{
			status = my_ipq_set_verdict(h, temp->packet_id, p->action, 0, NULL);
			delete(temp);
		}
		else // otherwise reinsert to new queue
			newqueue->add(temp);

		if (status<0)
		{
			cerr << "Internal ip_queue error: " << __LINE__ << endl;
			ipq_perror(ipq_errstr());
			die(h);
		}
	}
	delete (packetqueue);
	packetqueue=newqueue;
	pthread_mutex_unlock(&lock_mutex);
	if (ERRORDEBUG)
		cout << "-userfilter" << endl;
}

// handle a network command
void handle_cmd(int cmd, char *buffer)
{
	int status=0;
	//    char *rules;
	int action=ff::ACTION_ACCEPT;

	if (ERRORDEBUG)
		cout << "+handle_cmd" << endl;

	switch (cmd)
	{
	case ff::CMD_SOURCE: // get packet source
		if (ERRORDEBUG)
			cout << __LINE__ << "cmd_source" << endl;
		sendsource();
		break;
	case ff::CMD_RULES: // get iptables rules
		if (ERRORDEBUG)
			cout << __LINE__ << "cmd_rules" << endl;
		sendrules();
		break;
	case ff::CMD_URULES: // get userspace rules
		if (ERRORDEBUG)
			cout << __LINE__ << "cmd_urules" << endl;
		senduserspacerules();
		break;
	case ff::CMD_DROP: // deny a packet/create a rule denying a packet
		action=ff::ACTION_DROP;
		// NO break
	case ff::CMD_ACCEPT: // accept a packet/create a rule accepting a packet
		if (!buffer) return;
		if ((buffer[0]==ff::RC_THIS_PACKET) && (currentPacket)) // current packet
		{
			status = my_ipq_set_verdict(h, currentPacket->packet_id, action, 0, NULL);
			nextpacket();
			sendpacket();
		}
		else if (buffer[0]==ff::RC_QUEUED_PACKETS) // flush "pending" packets
		{
			while (currentPacket)
			{
				status = my_ipq_set_verdict(h, currentPacket->packet_id, action, 0, NULL);
				nextpacket();
			}
		}
		else if (buffer[0]==ff::RC_CREATE_RULE) // create a rule
		{
			if ((buffer[ff::SB_IP_SRC]||buffer[ff::SB_IP_DST]||buffer[ff::SB_PORT_SRC]||buffer[ff::SB_PORT_DST]||buffer[ff::SB_INTERFACE_IN]||buffer[ff::SB_INTERFACE_OUT]||
				 buffer[ff::SB_PROTOCOL]||buffer[ff::SB_PROGRAMNAME])&&currentPacket)
			{
				unsigned int timeout=0;

				if (buffer[ff::SB_PROGRAMNAME])
				{
					tentry *p=inputUserspaceRule(buffer, currentPacket, action);
					filterUserspaceRules(p);
				}
				else
				{
					// input rule to iptables
					inputRule(
							  currentPacket->hook,
							  (buffer[ff::SB_PROTOCOL]||buffer[ff::SB_PORT_SRC]||buffer[ff::SB_PORT_DST])?currentPacket->protocol:0,
							  (buffer[ff::SB_IP_SRC])?currentPacket->ip_src:0,
							  (buffer[ff::SB_IP_DST])?currentPacket->ip_dst:0,
							  (buffer[ff::SB_PORT_SRC])?currentPacket->port_src:0,
							  (buffer[ff::SB_PORT_DST])?currentPacket->port_dst:0,
							  (buffer[ff::SB_INTERFACE_IN])?(char*)currentPacket->interface_in:0,
							  (buffer[ff::SB_INTERFACE_OUT])?(char*)currentPacket->interface_out:0,
							  buffer[ff::SB_CONNTRACK], // connection tracking
							  action);
					// filter pending packets
					filterRules(
								currentPacket->hook,
								(buffer[ff::SB_PROTOCOL]||buffer[ff::SB_PORT_SRC]||buffer[ff::SB_PORT_DST])?currentPacket->protocol:0,
								(buffer[ff::SB_IP_SRC])?currentPacket->ip_src:0,
								(buffer[ff::SB_IP_DST])?currentPacket->ip_dst:0,
								(buffer[ff::SB_PORT_SRC])?currentPacket->port_src:0,
								(buffer[ff::SB_PORT_DST])?currentPacket->port_dst:0,
								(buffer[ff::SB_INTERFACE_IN])?(char*)currentPacket->interface_in:0,
								(buffer[ff::SB_INTERFACE_OUT])?(char*)currentPacket->interface_out:0,
								(buffer[ff::SB_CONNTRACK]&&(action!=ff::CMD_DROP)), // connection tracking
								action);
				}
				// timeout is stored in buffer[11 - 14]
				timeout=(((((((unsigned char)buffer[11])<<8)+(unsigned char)buffer[12])<<8)+(unsigned char)buffer[13])<<8)+(unsigned char)buffer[14];
				if (timeout>0) // timeouting rule ?
				{
					if (DEBUG)
						cout << "Rule has timeout: " << timeout << endl;
					tentry *t=new tentry(buffer, currentPacket, action);
					timeoutqueue->add(t); // put into queue for timeoutthread
				}
				// accept or deny current packet
				status = my_ipq_set_verdict(h, currentPacket->packet_id, (action==ff::ACTION_ACCEPT)?NF_ACCEPT:NF_DROP, 0, NULL);
				nextpacket();
			}
			sendpacket();
		}
		break;
	default:
		status=0;
		break;
	}
	if (status<0)
	{
		cerr << "Internal ip_queue error: " << __LINE__ << endl;
		ipq_perror(ipq_errstr());
		die(h);
	}
	if (ERRORDEBUG)
		cout << "-handle_cmd" << endl;
}

// delete userspace rule by rulenumber. used by interface
void deleteUserspaceRule(int rulenum)
{
	int i=0;
	tentry *p=0;

	if (!userspacequeue)
		return;
	if (rulenum<0)
		return;

	userspacequeue->rewind();
	while ((i<=rulenum) && (p=(tentry*)userspacequeue->next_element())) // advance to rule
		i++;

	if (p)
	{
		userspacequeue->del(p);
	}
}

// thread receiving data from client
void *rcvThread(void *parm)
{
	SSL *clientssl=(SSL*)parm;
	short msgtype;
	int result, len;
	char buffer[20];
	char username[50];
	char password[50];
	int temp;
	// is there an authenticated client? otherwise no commands except auth will be accepted
	int clientAuthenticated=0;
	// is there a client connected
	int clientConnected=1;

	if (DEBUG)
		cout << "new receive thread started" << endl;
	while (clientConnected)
	{
		msgtype=-1;
		if (ERRORDEBUG)
			cout << __LINE__ << endl;

		if (DEBUG)
			cout << "awaiting message" << endl;
		result=SSL_read(clientssl, &msgtype, 2); // first read message type
		if (clientAuthenticated && !ssl_auth) // error in one of the sends - close receiving thread too
		{
			clientAuthenticated=0;
			clientConnected=0;
		}
		if ((result<=0) || (msgtype<=0)) //invalid?
		{
			// on error disconnect client
			if (DEBUG)
				cout << __LINE__ << "(" << result << ")"<< ": Read failed" << endl;

			clientConnected=0;
		}
		else
		{
			if (ERRORDEBUG)
				cout << __LINE__ << endl;
			switch (msgtype)
			{
			case ff::CMD_DELETE_RULE: // delete a iptables rule by number
				if (!clientAuthenticated) break;
				unsigned char chainid;
				unsigned short rulenum;
				if (ERRORDEBUG)
					cout << __LINE__ << endl;
				result=SSL_read(clientssl, &chainid, 1);
				if (result<=0)
				{
					clientConnected=0;
					break;
				}
				result=SSL_read(clientssl, &rulenum, 2);
				if (result<=0)
				{
					clientConnected=0;
					break;
				}
				deleteRule(chainid, rulenum);
				sendrules();
				break;
			case ff::CMD_AUTH: // authenticate
				pthread_mutex_lock(&lock_mutex);

				// read username length
				result=SSL_read(clientssl, &temp, 1);
				if (result<=0)
				{
					pthread_mutex_unlock(&lock_mutex);
					clientConnected=0;
					break;
				}
				len=temp;
				result=SSL_read(clientssl, &temp, 1);
				if (result<=0)
				{
					clientConnected=0;
					pthread_mutex_unlock(&lock_mutex);
					break;
				}
				len+=temp<<8;

				temp=0;
				if (len<50) // length valid? --> read username
				{
					result=SSL_read(clientssl, username, len);
					if (result<=0)
					{
						clientConnected=0;
						pthread_mutex_unlock(&lock_mutex);
						break;
					}
					username[len]=0;

					// read password length
					result=SSL_read(clientssl, &temp, 1);
					if (result<=0)
					{
						clientConnected=0;
						pthread_mutex_unlock(&lock_mutex);
						break;
					}
					len=temp;
					result=SSL_read(clientssl, &temp, 1);
					if (result<=0)
					{
						clientConnected=0;
						pthread_mutex_unlock(&lock_mutex);
						break;
					}
					len+=temp<<8;
					temp=0;
					if (len<50) // length valid? --> read password
					{
						result=SSL_read(clientssl, password, len);
						if (result<=0)
						{
							clientConnected=0;
							pthread_mutex_unlock(&lock_mutex);
							break;
						}
						password[len]=0;
						if (auth(username, password)) // try to authenticate using username and password
						{
							clientAuthenticated=1;
							temp=1;
						}
					}
				}
				if (ssl_auth) // there is already an authenticated client
				{
					pthread_mutex_unlock(&lock_mutex);
					buffer[0]=8; // tell the client to log off
					buffer[1]=2;
					SSL_write(clientssl, buffer, 2);
					clientConnected=0;
					clientAuthenticated = 0;
					break;
				}
				if (temp!=1) // authentication error
				{
					buffer[0]=8;
					buffer[1]=0;
					SSL_write(clientssl, buffer, 2);
				}
				else // authentication successfull
				{
					buffer[0]=8;
					buffer[1]=1;
					ssl_auth=clientssl;
					SSL_write(clientssl, buffer, 2);
					sendpacket();
				}
				pthread_mutex_unlock(&lock_mutex);
				break;
			case ff::CMD_DELETE_URULE: // delete userspace rule by rulenumber
				if (!clientAuthenticated) break;
				if (ERRORDEBUG)
					cout << __LINE__ << endl;
				// get chain
				result=SSL_read(clientssl, &chainid, 1);
				if (result<=0)
				{
					clientConnected=0;
					break;
				}
				// get rulenumber
				result=SSL_read(clientssl, &rulenum, 2);
				if (result<=0)
				{
					clientConnected=0;
					break;
				}
				deleteUserspaceRule(rulenum);
				senduserspacerules();
				break;
			case ff::CMD_ACCEPT: // allow or deny packets/create rules allowing or denying packets
			case ff::CMD_DROP:
				if (!clientAuthenticated) break;
				if (ERRORDEBUG)
					cout << __LINE__ << endl;
				result=SSL_read(clientssl, buffer, 15);
				if (result==-1)
				{
					clientConnected=0;
					break;
				}
				if (DEBUG)
					for (int j=0;j<14;j++)
						cout << (int)buffer[j] << " ";
				if (DEBUG)
					cout << endl;
				handle_cmd(msgtype, buffer);
				break;
			default:
				if (!clientAuthenticated) break;
				if (ERRORDEBUG)
					cout << __LINE__ << endl;
				if (DEBUG)
					cout << "CMD received: " << msgtype << endl;
				handle_cmd(msgtype, NULL);
				break;
			}
		}
		if (ERRORDEBUG)
			cout << __LINE__ << endl;
	}
	if (!clientAuthenticated)
	{
		int fd=SSL_get_fd(clientssl);
		SSL_shutdown(clientssl);
		SSL_free(clientssl);
		close(fd);
		if (DEBUG)
			cout << __LINE__ << ": Closing clientssl connection" << endl;
	}
	if (clientAuthenticated && ssl_auth)
	{
		int fd=SSL_get_fd(ssl_auth);
		SSL_shutdown(ssl_auth);
		free_ssl_auth();
		close(fd);
		if (DEBUG)
			cout << __LINE__ << ": Closing ssl_auth connection" << endl;
	}
	return 0;
}

// listen on a specific port for clients
int openServerSocket(int port)
{
	int sock;
	struct sockaddr_in sockAddrIn;

	sock = socket(AF_INET, SOCK_STREAM, 0);
	sockAddrIn.sin_family = AF_INET;
	sockAddrIn.sin_port = htons(port);
	sockAddrIn.sin_addr.s_addr = htonl(INADDR_ANY); //inet_addr(SERVERADR);
	memset(&(sockAddrIn.sin_zero), '\0', 8);
	if (bind(sock, (struct sockaddr *)&sockAddrIn, sizeof(struct sockaddr)) == -1) // bind error?
	{
		char msg[80];
		sprintf(msg, "Could not bind to port %d", port);
		perror("bind");
		system_log(msg);
		if (Config_create_queue_rules) // delete previously inserted iptables rules before exiting
		{
			deleteRule(ff::CHAIN_INPUT, 0, 0, 0, 0, 0, 0, 0, 0, ff::ACTION_QUEUE);
			deleteRule(ff::CHAIN_OUTPUT, 0, 0, 0, 0, 0, 0, 0, 0, ff::ACTION_QUEUE);
			deleteRule(ff::CHAIN_FORWARD, 0, 0, 0, 0, 0, 0, 0, 0, ff::ACTION_QUEUE);
			rules_active=0;
		}

		return -1;
	}
	if (listen(sock, 5) == -1) // listen error?
	{
		char msg[80];
		sprintf(msg, "Could not listen on port %d", port);
		system_log(msg);
		perror("listen");
		if (Config_create_queue_rules) // delete previously inserted iptables rules before exiting
		{
			deleteRule(ff::CHAIN_INPUT, 0, 0, 0, 0, 0, 0, 0, 0, ff::ACTION_QUEUE);
			deleteRule(ff::CHAIN_OUTPUT, 0, 0, 0, 0, 0, 0, 0, 0, ff::ACTION_QUEUE);
			deleteRule(ff::CHAIN_FORWARD, 0, 0, 0, 0, 0, 0, 0, 0, ff::ACTION_QUEUE);
			rules_active=0;
		}

		return -1;
	}
	return sock;
}

// thread accepting clients
void *serverThread(void *parm)
{
	int socket=-1;
	unsigned int sinSize;
	struct sockaddr_in sockAddrIn;
	pthread_t rcvThr;
	SSL *ssl;
	int clientSocket;

	socket=openServerSocket(Config_port); // listen on port Config_port
	while (socket!=-1)
	{
		if (DEBUG)
			cout << "Opened socket ... Listening ..." << endl;

		sinSize = sizeof(struct sockaddr_in);
		clientSocket = accept(socket, (struct sockaddr *)&sockAddrIn, &sinSize);
		if (clientSocket != -1) // client connected?
		{
			if (DEBUG)
				cout << "Connection!!!" << endl;

			// initialize SSL
			ssl=SSL_new(ssl_ctx);
			if (!ssl)
			{
				cerr << "Error creating SSL object" << endl;
				//                SSL_get_error(ssl, 0);
			}

			if (!SSL_set_fd(ssl, clientSocket))
			{
				cerr << "Error initializing SSL-socket" << endl;
				SSL_print_error(SSL_get_error(ssl, 0));
			}
			else
			{
				int result=SSL_accept(ssl);
				if (result==1)
				{
					if (DEBUG)
						cout << "SSL accept done" << endl;
					// SSL init finished
					//					clientConnected=1;
					//					clientAuthenticated=0;
					pthread_create(&rcvThr, NULL, &rcvThread, (void*)ssl);
				}
				else
				{
					SSL_print_error(SSL_get_error(ssl, result));
					//					clientConnected=0;
				}
			}
		}
	}
	return 0;
}

// create a new packet from ipq packet
packet* createPacket(ipq_packet_msg_t *msg)
{
	struct iphdr *iph;
	struct tcphdr *tcph;
	struct udphdr *udph;
	struct icmphdr *icmph;
	struct packet *p;
	iph=(struct iphdr*)msg->payload;

	if (ERRORDEBUG)
		cout << "+createpacket" << endl;

	if (!msg) return 0;

	p=new packet();
	if (!p)
	{
		cerr << "LINE " << __LINE__ << ": COULD NOT ALLOCATE MEMORY" << endl;
		system_log("Out of memory");
		return 0;
	}

	p->port_src=0;
	p->port_dst=0;

	p->packet_id=msg->packet_id;
	p->arrived=msg->timestamp_sec;
	if (!p->arrived) // workaround for unspecified time in ipq codes
		p->arrived=time(NULL);
	memcpy(p->interface_in, msg->indev_name, IFNAMSIZ);
	memcpy(p->interface_out, msg->outdev_name, IFNAMSIZ);
	p->len=(short)msg->data_len;
	p->ip_src=iph->saddr;
	p->ip_dst=iph->daddr;
	p->protocol=iph->protocol;

	if (iph->protocol==ff::PROT_TCP) // copy ports and flags for tcp
	{
		tcph=(struct tcphdr*)((char*)iph+sizeof(struct iphdr));
		p->port_src=(tcph->source>>8)+((tcph->source&255)<<8);
		p->port_dst=(tcph->dest>>8)+((tcph->dest&255)<<8);
		p->tcp_flags=(int)tcph->fin+(((int)tcph->syn)<<1)+(((int)tcph->ack)<<2)+(((int)tcph->rst)<<3);   //*((short*)(((char*)(&tcph->ack_seq))+4));
	}
	else if (iph->protocol==ff::PROT_UDP) // copy ports for udp
	{
		udph=(struct udphdr*)((char*)iph+sizeof(struct iphdr));
		p->port_src=(udph->source>>8)+((udph->source&255)<<8);
		p->port_dst=(udph->dest>>8)+((udph->dest&255)<<8);
	}
	else if (iph->protocol==ff::PROT_ICMP) // copy icmp type for icmp
	{
		icmph=(struct icmphdr*)((char*)iph+sizeof(struct iphdr));
		p->icmp_type=icmph->type;
	}

	p->mac_addrlen=msg->hw_addrlen;
	memcpy(p->mac_addr, msg->hw_addr, 8);

	p->data=new char[p->len];
	if (!p->data)
	{
		cerr << "LINE " << __LINE__ << ": COULD NOT ALLOCATE MEMORY" << endl;
		system_log("Out of memory");
		delete(p);
		return 0;
	}

	memcpy(p->data, msg->payload, p->len);
	p->programname=0;
	if (msg->hook==ff::CHAIN_INPUT) // get programname for tcp and udp if chain is INPUT or OUTPUT
	{
		if ((p->protocol==ff::PROT_TCP) || (p->protocol==ff::PROT_UDP))
			p->programname=get_program_name(p->protocol, p->ip_dst, p->port_dst);
	}
	if (msg->hook==ff::CHAIN_OUTPUT)
	{
		if ((p->protocol==ff::PROT_TCP) || (p->protocol==ff::PROT_UDP))
			p->programname=get_program_name(p->protocol, p->ip_src, p->port_src);
	}

	p->hook=msg->hook;

	if (ERRORDEBUG)
		cout << "-createpacket" << endl;
	return(p);
}

// thread which periodically checks for rules which are timeouting
void *timeoutThread(void *parm)
{
	unsigned long tim;
	tentry *t;//, *tprev;

	// for debug purposes only
#ifdef SKIPTIMEOUTTHREAD
	cout << "Skipping Timoutthread" << endl;
	while(1)
		sleep(1);
#endif

	while (1)
	{
		tim=time(NULL);
		pthread_mutex_lock(&lock_mutex);
		if (timeoutqueue)
		{
			timeoutqueue->rewind();
			while ((t=(tentry*)timeoutqueue->next_element()))
			{
				if (t->timeout<tim) // timeout reached?
				{
					if (DEBUG)
						cout << "Deleting rule !!!" << endl;
					if (t->buf[10]) // userspace rule
						deleteUserspaceRule(t->buf, t->pack, t->action);
					else // iptables rule
						deleteRule(
								   t->pack->hook,
								   (t->buf[8]||t->buf[3]||t->buf[4])?t->pack->protocol:0,
								   (t->buf[1])?t->pack->ip_src:0,
								   (t->buf[2])?t->pack->ip_dst:0,
								   (t->buf[3])?t->pack->port_src:0,
								   (t->buf[4])?t->pack->port_dst:0,
								   (t->buf[5])?(char*)t->pack->interface_in:0,
								   (t->buf[6])?(char*)t->pack->interface_out:0,
								   t->buf[9], // connection tracking
								   t->action);

					timeoutqueue->del(t);
				}

			}
		}
		pthread_mutex_unlock(&lock_mutex);
		sleep(60); // check all minutes for expired rules
	}
}

// verdict because of userspace rules?
int userspaceverdict(packet *p)
{
	tentry *q;
	int filter;

	if (!p) return 0;

	if (!p->programname)
		return 0;

	if (ERRORDEBUG)
		cout << "+userspacefilter" << endl;

	pthread_mutex_lock(&lock_mutex);
	userspacequeue->rewind();
	while ((q=(tentry*)userspacequeue->next_element())) // check every userspace rule
	{
		filter=1;

		// some filtering
		if (strcmp(p->programname, q->pack->programname))
		{
			filter=0;
		}
		if (q->buf[1] && (p->ip_src!=q->pack->ip_src))
		{
			filter=0;
		}
		if (q->buf[2] && (q->pack->ip_dst!=p->ip_dst))
		{
			filter=0;
		}
		if (q->buf[3] && (q->pack->port_src!=p->port_src))
		{
			filter=0;
		}
		if (q->buf[4] && (q->pack->port_dst!=p->port_dst))
		{
			filter=0;
		}
		if (q->buf[5] && (strcmp((char*)q->pack->interface_in, (char*)p->interface_in)))
		{
			filter=0;
		}
		if (q->buf[6] && (strcmp((char*)q->pack->interface_out, (char*)p->interface_out)))
		{
			filter=0;
		}
		if ((q->buf[8] || q->buf[3] || q->buf[4]) && (q->pack->protocol!=p->protocol))
		{
			filter=0;
		}

		if (filter) // p matches rule?
		{
			if (ERRORDEBUG)
				cout << "-userspacefilter" << endl;

			pthread_mutex_unlock(&lock_mutex);
			return (q->action);
		}
	}
	pthread_mutex_unlock(&lock_mutex);
	if (ERRORDEBUG)
		cout << "-userspacefilter" << endl;
	return 0;
}

// this thread gets packets from the kernel and enqueues them into packetqueue = "pending" packets
void *kernelThread(void *parm)
{
	int status;
	unsigned long clientip;
	unsigned char buf[BUFSIZE];
	int c1, c2, c3, c4;

	// get ip of accepted client
	if (sscanf(Config_client_ip, "%d.%d.%d.%d", &c1, &c2, &c3, &c4)==4)
	{
		clientip=((unsigned long)c4<<24)+((unsigned long)c3<<16)+((unsigned long)c2<<8)+((unsigned long)c1); // ip address as number
	}
	else
	{
		cerr << "Invalid client_ip" << endl;
		system_log("Invalid client ip in config file");
		return 0;
	}

	// set queue to maximum QUEUE_LENGTH packets
	// if queue is full packets are dropped, add returns false --> refuse packet
	packetqueue->setLength(QUEUE_LENGTH);

	h = ipq_create_handle(0, PF_INET);
	if (!h)
	{
		die(h);
	}

	status = ipq_set_mode(h, IPQ_COPY_PACKET, BUFSIZE);
	if (status < 0)
	{
		cerr << "Internal ip_queue error: Maybe you forgot to load ip_queue module ? (cf README)" << endl;
		die(h);
	}

	do // forever
	{
		status = ipq_read(h, buf, BUFSIZE, 0);
		if (status < 0)
		{
			cerr << "Internal ip_queue error: LINE" << __LINE__ << ":" << status << endl;
			//            die(h);
		}

		switch (ipq_message_type(buf))
		{
		case NLMSG_ERROR:
			fprintf(stderr, "Received error message %d\n",
					ipq_get_msgerr(buf));
			if (ipq_get_msgerr(buf)==1)
				fprintf(stderr, "Maybe you are not root ?\n");

			ipq_perror(ipq_errstr());
			break;

		case IPQM_PACKET: {

			ipq_packet_msg_t *m = ipq_get_packet(buf); // receive packet
			struct packet *pack;

			if (ERRORDEBUG)
				cout << __LINE__ << "Recv" << endl;
			//	    dump(m);
			pack=createPacket(m); // create packet out of ipq data
			if (ERRORDEBUG)
				cout << __LINE__ << endl;

			int verdict;
			// is it a fireflier packet ?
			if (((pack->ip_dst==clientip) && (pack->port_src==Config_port)) ||
				((pack->ip_src==clientip) && (pack->port_dst==Config_port)))
				verdict=ff::ACTION_ACCEPT;
			else // is it matched by userspace rules?
				verdict=userspaceverdict(pack);

			if ((verdict==ff::ACTION_ACCEPT) || (verdict==ff::ACTION_DROP))
			{
				status = my_ipq_set_verdict(h, m->packet_id,
											(verdict==ff::ACTION_ACCEPT)?NF_ACCEPT:NF_DROP, 0, NULL);
				delete(pack);
				if (ERRORDEBUG)
					cout << "userspace filtered" << endl;
			}
			else // no verdict so far --> get packet to user
			{
				pthread_mutex_lock(&lock_mutex);
				if (packetqueue->isFull()) // queue full?-->drop oldest packet
				{
					packet *temp=(packet*)packetqueue->get();
					if (temp)
					{
						status = my_ipq_set_verdict(h, temp->packet_id,
													NF_DROP, 0, NULL);
						delete(temp);
					}
				}
				if (!packetqueue->add(pack)) // queue should not be full now
				{
					if (ERRORDEBUG)
						cout << "queue full --> drop" << endl;
					status = my_ipq_set_verdict(h, m->packet_id,
												NF_DROP, 0, NULL);
				}
				pthread_mutex_unlock(&lock_mutex);
				if (ERRORDEBUG)
					cout << __LINE__ << endl;

				if ((!currentPacket)) // first packet in queue --> immediately send to user
				{
					if (ERRORDEBUG)
						cout << __LINE__ << endl;
					nextpacket();
					if (ERRORDEBUG)
						cout << __LINE__ << endl;

					if (currentPacket)
						sendpacket();
				}
			}

			if (status < 0)
			{
				cerr << "Internal ip_queue error: " << __LINE__ << endl;
				//                    die(h);
			}
			if (ERRORDEBUG)
				cout << __LINE__ << endl;
			break;
		}

		default:
			fprintf(stderr, "Unknown message type!\n");
			break;
		}
	} while (1);

	ipq_destroy_handle(h);
	return 0;
}

// save rules of userspace
// obsolete use saveUserspaceRulesXml(filename) now
/*void saveuserspacerules()
{
	FILE *out;
	tentry *p;

	userspacequeue->rewind();
	char *filename=Config_save_file;

	umask(0077);
	out=fopen(filename, "wt");
	if (!out)
	{
		cerr << "Error while opening userspace file(" << filename << ") for writing" << endl;
		system_log("Error while opening userspace rules file for writing");
		return;
	}
	while ((p=(tentry*)userspacequeue->next_element()))
	{
		if (!p->timeout)
		{
			if (out)
			{
				fprintf(out, "%s\n", p->pack->programname);
				fprintf(out, "%u\n", (p->buf[1])?p->pack->ip_src:0);
				fprintf(out, "%u\n", (p->buf[2])?p->pack->ip_dst:0);
				fprintf(out, "%d\n", (p->buf[3])?p->pack->port_src:0);
				fprintf(out, "%d\n", (p->buf[4])?p->pack->port_dst:0);
				fprintf(out, "%s\n", (p->buf[5])?(char*)p->pack->interface_in:(char*)"none");
				fprintf(out, "%s\n", (p->buf[6])?(char*)p->pack->interface_out:(char*)"none");
				fprintf(out, "%d\n", (p->buf[8]||p->buf[3]||p->buf[4])?p->pack->protocol:0);
				fprintf(out, "%d\n", (p->action==ff::ACTION_ACCEPT)?1:0);
			}
		}
	}
	if (out)
		fclose(out);
}*/

// SIGTERM or SIGINT received --> save rules, kill timeouting rules and quit
void finalize(int sig)
{
	tentry *p;

	if (DEBUG)
		cout << "Finalizing" << endl;

	pthread_mutex_lock(&lock_mutex);
	if (timeoutqueue)
	{
		timeoutqueue->rewind();
		while ((p=(tentry*)timeoutqueue->next_element())) // delete every timeouting rule of iptables
		{
			deleteRule(
					   p->pack->hook,
					   (p->buf[ff::SB_PROTOCOL]||p->buf[ff::SB_PORT_SRC]||p->buf[ff::SB_PORT_DST])?p->pack->protocol:0,
					   (p->buf[ff::SB_IP_SRC])?p->pack->ip_src:0,
					   (p->buf[ff::SB_IP_DST])?p->pack->ip_dst:0,
					   (p->buf[ff::SB_PORT_SRC])?p->pack->port_src:0,
					   (p->buf[ff::SB_PORT_DST])?p->pack->port_dst:0,
					   (p->buf[ff::SB_INTERFACE_IN])?(char*)p->pack->interface_in:0,
					   (p->buf[ff::SB_INTERFACE_OUT])?(char*)p->pack->interface_out:0,
					   p->buf[ff::SB_CONNTRACK], // connection tracking
					   p->action);
			timeoutqueue->del(p);
		}
		timeoutqueue=0;

		// save userspace
		saveUserspaceRulesXml(Config_save_file);

		userspacequeue->rewind();
		while ((p=(tentry*)userspacequeue->next_element()))
			userspacequeue->del(p);
		userspacequeue=0;

		if (Config_create_queue_rules) // delete rules for fireflier of iptables
		{
			deleteRule(ff::CHAIN_INPUT, 0, 0, 0, 0, 0, 0, 0, 0, ff::ACTION_QUEUE);
			deleteRule(ff::CHAIN_OUTPUT, 0, 0, 0, 0, 0, 0, 0, 0, ff::ACTION_QUEUE);
			deleteRule(ff::CHAIN_FORWARD, 0, 0, 0, 0, 0, 0, 0, 0, ff::ACTION_QUEUE);
			rules_active=0;
		}
	}
    if (daemonize && strcmp(Config_pid_file, "none"))
        unlink(Config_pid_file);
	exit(0); // so no other threads access now deleted variables
	pthread_mutex_unlock(&lock_mutex);
}

// load rules from userspace file
void loaduserspacerulesold(string filename)
{
	//    tentry *t;
	FILE *in;
	char *buffer;
	packet *p;
	int action;
	int prot;

	in=fopen(filename.c_str(), "rt");
	if (!in)
	{
		cerr << "Error loading userspace rules file(" << filename << ")" << endl;
		system_log("Error while opening userspace rule file for reading");
		return;
	}

	while (!feof(in))
	{
		buffer=new char[20];
		p=new packet();
		char temp[100];
		for (int i=0;i<20;i++)
			buffer[i]=0;

		p->data=0;
		p->len=0;
		fscanf(in, "%s\n", temp);
		p->programname=new char[strlen(temp)+1];
		strcpy(p->programname, temp);
		fscanf(in, "%u\n", &p->ip_src);
		if (p->ip_src)
			buffer[1]=1;
		fscanf(in, "%u\n", &p->ip_dst);
		if (p->ip_dst)
			buffer[2]=1;
		fscanf(in, "%d\n", &p->port_src);
		if (p->port_src)
			buffer[3]=1;
		fscanf(in, "%d\n", &p->port_dst);
		if (p->port_dst)
			buffer[4]=1;
		fscanf(in, "%s\n", p->interface_in);
		if (strcmp((char*)p->interface_in, "none"))
			buffer[5]=1;
		else
			strcpy((char*)p->interface_in, "");

		buffer[11]=0; // timeout
		buffer[12]=0;
		buffer[13]=0;
		buffer[14]=0;

		fscanf(in, "%s\n", p->interface_out);
		if (strcmp((char*)p->interface_out, "none"))
			buffer[6]=1;
		else
			strcpy((char*)p->interface_out, "");
		fscanf(in, "%d\n", &prot);
		p->protocol=prot;

		if (p->protocol)
			buffer[8]=1;
		int result=fscanf(in, "%d\n", &action);
		if (result==1) // if the last element was retrieved successfully, anything else should have too
		{
			userspacequeue->add(new tentry(buffer, p, (action==1)?ff::ACTION_ACCEPT:ff::ACTION_DROP));
		}
		delete[] buffer;
		//        delete(p->programname);
		//        delete(p);
	}
	fclose(in);
}

int saveUserspaceRulesXml(string filename)
{
	tentry *elem;
    int numRules=0;

	if (!userspacequeue)
        return -1;

	if ((filename.length()>3) && (filename.substr(filename.length()-3, 3)=="dat"))
	{
		filename=filename.replace(filename.length()-3, 3, "xml");
	}

	userspacequeue->rewind();
	xmlDocPtr xmlDoc=0;

	try {
		xmlDoc = xmlNewDoc((const xmlChar *)"1.0");
		if (!xmlDoc)
            throw 1;

		xmlNodePtr xmlRoot=xmlNewDocNode(xmlDoc, NULL, (const xmlChar*)"rules", NULL);
		if (!xmlRoot)
            throw 1;
		xmlDocSetRootElement(xmlDoc, xmlRoot);

		while (elem=(tentry*)userspacequeue->next_element())
		{
			xmlNodePtr xmlRule;
			xmlNodePtr xmlRuleDetail;
			char temp[16];
			xmlRule=xmlNewChild(xmlRoot, NULL, (const xmlChar*)"rule", NULL);
			if (!xmlRule)
				throw 1;
			xmlNewProp(xmlRule, (const xmlChar*)"action", ((elem->action==ff::ACTION_ACCEPT)?(const xmlChar*)"accept":(const xmlChar*)"drop"));

			xmlRuleDetail=xmlNewChild(xmlRule, NULL, (const xmlChar*)"ProgramName", NULL);
			if (!xmlRuleDetail)
				throw 1;
			xmlNewProp(xmlRuleDetail, (const xmlChar*)"value", (const xmlChar*)elem->pack->programname);

			xmlRule=xmlNewChild(xmlRule, NULL, (const xmlChar*)"Filter", NULL);
			if (!xmlRule)
				throw 1;

			if (elem->buf[1] && elem->pack->ip_src!=0)
			{
				xmlRuleDetail=xmlNewChild(xmlRule, NULL, (const xmlChar*)"SourceIP", (const xmlChar*)int_to_ipaddr(elem->pack->ip_src, temp, 16));
				if (!xmlRuleDetail)
					throw 1;
				xmlNewProp(xmlRuleDetail, (const xmlChar*)"version", (const xmlChar*)int_to_charptr(4, temp, 16));
			}

			if (elem->buf[2] && elem->pack->ip_dst!=0)
			{
				xmlRuleDetail=xmlNewChild(xmlRule, NULL, (const xmlChar*)"DestinationIP", (const xmlChar*)int_to_ipaddr(elem->pack->ip_dst, temp, 16));
				if (!xmlRuleDetail)
					throw 1;
				xmlNewProp(xmlRuleDetail, (const xmlChar*)"version", (const xmlChar*)int_to_charptr(4, temp, 16));
			}

			if (elem->buf[3] && elem->pack->port_src!=0)
			{
				xmlRuleDetail=xmlNewChild(xmlRule, NULL, (const xmlChar*)"SourcePort", (const xmlChar*)int_to_charptr(elem->pack->port_src, temp, 16));
				if (!xmlRuleDetail)
					throw 1;
			}
			if (elem->buf[4] && elem->pack->port_dst!=0)
			{
				xmlRuleDetail=xmlNewChild(xmlRule, NULL, (const xmlChar*)"DestinationPort", (const xmlChar*)int_to_charptr(elem->pack->port_dst, temp, 16));
				if (!xmlRuleDetail)
					throw 1;
			}

			if (elem->buf[5] && elem->pack->interface_in && strcmp((char*)elem->pack->interface_in, ""))
			{
				xmlRuleDetail=xmlNewChild(xmlRule, NULL, (const xmlChar*)"incomingInterface", (const xmlChar*)elem->pack->interface_in);
				if (!xmlRuleDetail)
					throw 1;
			}
			if (elem->buf[6] && elem->pack->interface_out && strcmp((char*)elem->pack->interface_out, ""))
			{
				xmlRuleDetail=xmlNewChild(xmlRule, NULL, (const xmlChar*)"outgoingInterface", (const xmlChar*)elem->pack->interface_out);
				if (!xmlRuleDetail)
					throw 1;
			}
			if ((elem->buf[8] || elem->buf[3] || elem->buf[4]) && elem->pack->protocol!=0)
			{
				xmlRuleDetail=xmlNewChild(xmlRule, NULL, (const xmlChar*)"Protocol", (const xmlChar*)int_to_charptr(elem->pack->protocol, temp, 16));
				if (!xmlRuleDetail)
					throw 1;
			}
            numRules++;
		}
	}
	catch (...)
	{
		printf("Error creating config xml struture. save failed\n");
        if (xmlDoc)
			xmlFreeDoc(xmlDoc);
        return -1;
	}

	if (xmlSaveFormatFile (filename.c_str(), xmlDoc, 1)==-1)
	{
		printf("Error writing the config xml struture to disk. disk full ? permissions correct ?\n");
		xmlFreeDoc(xmlDoc);
        return -1;
	}
	xmlFreeDoc(xmlDoc);
    return numRules;
}

int loadUserspaceRulesXml(string filename)
{
	xmlDocPtr xmlDoc=0;
    int numRules=0;
	try {
		xmlDoc = xmlParseFile(filename.c_str());
		if (!xmlDoc)
			throw 1;

		xmlNodePtr xmlRoot = xmlDocGetRootElement(xmlDoc);
		if (!xmlRoot)
			throw 1;
		if (xmlStrcmp(xmlRoot->name, (const xmlChar *) "rules"))
			throw 1;

		xmlNodePtr rule = xmlRoot->xmlChildrenNode;
		while ( rule && xmlIsBlankNode ( rule ) )
		{
			rule = rule -> next;
		}
		while (rule)
		{
			if (xmlStrcmp(rule->name, (const xmlChar *) "rule"))
			{
                cerr << rule->name << endl;
				throw 1;
			}

			xmlNodePtr ruleData = rule->xmlChildrenNode;
			if (!ruleData)
				throw 1;

            int action=-1;
			char *temp=(char*)xmlGetProp(rule, (xmlChar*)"action");
			if (temp && !strcmp(temp, "accept"))
			{
				action=ff::ACTION_ACCEPT;
			}
			else
			{
				action=ff::ACTION_DROP;
			}
            xmlFree(temp);


			char *buffer=new char[20];
			packet *p=new packet();
            memset(buffer, 0, 20);
			p->data=0;
			p->len=0;

			while (ruleData)
			{
				if (xmlIsBlankNode ( ruleData )); // do nothing
				else if (xmlStrcmp(ruleData->name, (const xmlChar *) "ProgramName")==0)
				{
					p->programname=(char*)xmlGetProp(ruleData, (xmlChar*)"value");
				}
				else if (xmlStrcmp(ruleData->name, (const xmlChar *) "Filter")==0)
				{
					xmlNodePtr filterData = ruleData->xmlChildrenNode;
					while (filterData)
					{
                        int value;
						if (xmlStrcmp(filterData->name, (const xmlChar *) "SourcePort")==0)
						{
							value=atoi((char*)xmlNodeListGetString(xmlDoc, filterData->xmlChildrenNode, 1));
                            buffer[3]=1;
							if (value)
								p->port_src=value;
							else
                                if (DEBUG)
									cerr << "Error 0 in srcport " << endl;
						}
						if (xmlStrcmp(filterData->name, (const xmlChar *) "DestinationPort")==0)
						{
							value=atoi((char*)xmlNodeListGetString(xmlDoc, filterData->xmlChildrenNode, 1));
                            buffer[4]=1;
							if (value)
								p->port_dst=value;
							else
                                if (DEBUG)
									cerr << "Error 0 in dstport " << endl;
						}
						if (xmlStrcmp(filterData->name, (const xmlChar *) "Protocol")==0)
						{
							value=atoi((char*)xmlNodeListGetString(xmlDoc, filterData->xmlChildrenNode, 1));
                            buffer[8]=1;
							if (value)
								p->protocol=value;
							else
                                if (DEBUG)
									cerr << "Error 0 in protocol " << endl;
						}
						if (xmlStrcmp(filterData->name, (const xmlChar *) "incomingInterface")==0)
						{
							strncpy((char*)p->interface_in, (char*)xmlNodeListGetString(xmlDoc, filterData->xmlChildrenNode, 1), IFNAMSIZ);
                            buffer[5]=1;
						}
						if (xmlStrcmp(filterData->name, (const xmlChar *) "outgoingInterface")==0)
						{
							strncpy((char*)p->interface_out, (char*)xmlNodeListGetString(xmlDoc, filterData->xmlChildrenNode, 1), IFNAMSIZ);
                            buffer[6]=1;
						}
						if (xmlStrcmp(filterData->name, (const xmlChar *) "SourceIP")==0)
						{
							char *ipsrc=(char*)xmlNodeListGetString(xmlDoc, filterData->xmlChildrenNode, 1);
							in_addr adr;
                            buffer[1]=1;
							if (!ipsrc)
								throw 1;
							if (!inet_aton(ipsrc, &adr))
								throw 1;
                            p->ip_src=adr.s_addr;
						}
						if (xmlStrcmp(filterData->name, (const xmlChar *) "DestinationIP")==0)
						{
							char *ipdst=(char*)xmlNodeListGetString(xmlDoc, filterData->xmlChildrenNode, 1);
                            in_addr adr;
                            buffer[2]=1;
							if (!ipdst)
								throw 1;
							if (!inet_aton(ipdst, &adr))
								throw 1;
                            p->ip_dst=adr.s_addr;
						}


						filterData=filterData->next;
					}
				}
				else
					throw 1;
                ruleData=ruleData->next;
 			}

/*			cout << "parsed rule: " << endl;
			cout << "   action: " << action << endl;
			cout << "   name: " << p->programname << endl;
			cout << "   srcport: " << p->port_src << endl;
			cout << "   dstport: " << p->port_dst << endl;
			cout << "   srcip: " << p->ip_src << endl;
			cout << "   dstip: " << p->ip_dst << endl;
			cout << "   protocol: " << p->protocol << endl;
			cout << "   interface_in: " << p->interface_in << endl;
			cout << "   interface_out: " << p->interface_out << endl;
			*/
            if (action!=-1)
				userspacequeue->add(new tentry(buffer, p, action));

			numRules++;

            rule=rule->next;
			while ( rule && xmlIsBlankNode ( rule ) )
				rule = rule -> next;
		}
        xmlFreeDoc(xmlDoc);
	}
	catch (...)
	{
		cerr << "Error loading userspace rules xml file(" << filename << ")" << endl;
        if (xmlDoc)
			xmlFreeDoc(xmlDoc);
        return -1;
	}
    return numRules;
}

int loadUserspaceRules(string filename)
{
    string oldfilename;
	int temp=-1;
	if (filename=="")
	{
		if (DEBUG)
			cerr << "No userspace rule file specified" << endl;
		return -1;
	}

    tentry *t;
	while ((t=(tentry*)userspacequeue->get())!=0)
        delete(t);

	if ((filename.length()>3) && (filename.substr(filename.length()-3, 3)=="dat"))
	{
		if (file_readable(filename))
		{
            system_log("Importing old userspace rules file ... ");
            if (DEBUG)
				cerr << "Importing old userspace rules file ... " << endl;
			loaduserspacerulesold(filename);
			temp=userspacequeue->getNumElements();
            oldfilename=filename;
			filename=filename.replace(filename.length()-3, 3, "xml");
			if (saveUserspaceRulesXml(filename)!=temp)
			{
				cerr << "Error while saving userspace xml rules" << endl;
                system_log("Error while saving userspace xml rules");
			}
		}
		filename=filename.replace(filename.length()-3, 3, "xml");
	}

	while ((t=(tentry*)userspacequeue->get())!=0)
        delete(t);

	if ((loadUserspaceRulesXml(filename)==temp) && (temp!=-1))
	{
        unlink(oldfilename.c_str());
	}
	else if (temp!=-1)
	{
        system_log("Error loading userspace xml rules");
		cerr << "Error loading userspace xml rules" << endl;
	}
	return 1;
}


// get various tokens from configfile. Does not handle files with spaces
// in it correctly at the moment
int nexttoken(char *line, int *pos, int maxpos,int *tokenstart, int *tokenend)
{
	//    int i;

	while ((*pos<maxpos) && (line[*pos]<=32) && (line[*pos]!='#'))
		*pos=*pos+1;
	if ((*pos>=maxpos) || (line[*pos]=='#'))
	{
		*tokenstart=-1;
		return 1;
	}
	*tokenstart=*pos;
	if (line[*tokenstart]=='=')
	{
		*tokenend=*tokenstart+1;
		*pos=*pos+1;
		return 1;
	}
	*pos=*pos+1;
	while ((*pos<maxpos) && (line[*pos]>32) && (line[*pos]!='#') && (line[*pos]!='='))
		*pos=*pos+1;

	if ((*pos>=maxpos))
		return 0;
	*tokenend=*pos;
	if (*tokenstart>=*tokenend) // should not happen
		return 0;

	return 1;
}

// load configuration file
void loadconfig(char *filename)
{
	FILE *in;
	char line[256];
	char temp[256];
	//    int len;
	int pos;
	int linenr;
	int keystart, keyend, valuestart, valueend;
	if ((in=fopen(filename, "rt"))==0)
	{
		cerr << "Error opening configfile (" << filename << "). Using defaults" << endl;
		system_log("Config file not found, using defaults");
		return;
	}

	linenr=1;
	while (!feof(in))
	{
		pos=0;
		fgets(line, 256, in);
		line[255]=0; // if line was too long
		if (!nexttoken(line, &pos, 256, &keystart, &keyend))
			cerr << "Config syntax error in line: " << linenr << endl;
		else if (keystart>=0)
		{
			if (!nexttoken(line, &pos, 256, &valuestart, &valueend) || (valuestart<0))
				cerr << "Config syntax error in line: " << linenr << endl;
			else
			{
				if (!nexttoken(line, &pos, 256, &valuestart, &valueend) || (valuestart<0))
					cerr << "Config syntax error in line: " << linenr << endl;
				else
				{
					// file to save userspace rules to?
					if (strncasecmp(line+keystart, "save_file", (keyend-keystart<9)?(keyend-keystart):9)==0)
					{
						Config_save_file=new char[valueend-valuestart+1];
						strncpy(Config_save_file, line+valuestart, valueend-valuestart);
						Config_save_file[valueend-valuestart]=0;
					} // port to listen at ?
					else if (strncasecmp(line+keystart, "pid_file", (keyend-keystart<8)?(keyend-keystart):8)==0)
					{
						Config_pid_file=new char[valueend-valuestart+1];
						strncpy(Config_pid_file, line+valuestart, valueend-valuestart);
						Config_pid_file[valueend-valuestart]=0;
					} // port to listen at ?
					else if (strncasecmp(line+keystart, "port", (keyend-keystart<4)?(keyend-keystart):4)==0)
					{
						strncpy(temp, line+valuestart, valueend-valuestart);
						temp[valueend-valuestart]=0;
						if (atoi(temp)!=0)
							Config_port=atoi(temp);
						else
							cerr << "Config syntax error in line: " << linenr << " - port invalid" << endl;
					} // should we create the iptables queue rules ?
					else if (strncasecmp(line+keystart, "create_queue_rules", (keyend-keystart<18)?(keyend-keystart):18)==0)
					{
						strncpy(temp, line+valuestart, valueend-valuestart);
						temp[valueend-valuestart]=0;
						Config_create_queue_rules=atoi(temp);
					} // ssl certificate and key
					else if (strncasecmp(line+keystart, "ssl_file", (keyend-keystart<8)?(keyend-keystart):8)==0)
					{
						Config_ssl_file=new char[valueend-valuestart+1];
						strncpy(Config_ssl_file, line+valuestart, valueend-valuestart);
						Config_ssl_file[valueend-valuestart]=0;
					} // password for ssl key
					else if (strncasecmp(line+keystart, "ssl_password", (keyend-keystart<12)?(keyend-keystart):12)==0)
					{
						Config_ssl_password=new char[valueend-valuestart+1];
						strncpy(Config_ssl_password, line+valuestart, valueend-valuestart);
						Config_ssl_password[valueend-valuestart]=0;
					} // ip of client allowed to access the firewall
					else if (strncasecmp(line+keystart, "client_ip", (keyend-keystart<9)?(keyend-keystart):9)==0)
					{
						Config_client_ip=new char[valueend-valuestart+1];
						strncpy(Config_client_ip, line+valuestart, valueend-valuestart);
						Config_client_ip[valueend-valuestart]=0;
					}
				}
			}
		}
		linenr++;
	}
	fclose(in);
}

// password callback for ssl key usage
int pem_passwd_cb(char *buf, int size, int rwflag, void *password)
{
	strncpy(buf, Config_ssl_password, size);
	buf[size - 1] = '\0';
	return(strlen(buf));
}

// SIGPIPE received --> log and ignore
void pipesignal(int sig)
{
    // we keep this as a shutdown because of that reason is more harmful
    // than having a SIGPIPE in our program
    cerr << "not good. we received SIGPIPE. Please fix me. Continueing" << endl;
}

int main(int args, char **argv)
{
	pthread_t kernelthr;
	pthread_t timeoutthr;
	char *Configfile="/etc/fireflier/fireflier.conf";
	int option;
	int usage=0;
	void *thrparms[2];
    bool want_daemonize=false;

	ssl_auth=0;
	rules_active=0;

	while ((option=getopt(args, argv, "f:hd"))!=-1) // using this the first time, hope it works this way :)
	{
		switch (option)
		{
		case 'f':
			Configfile=argv[optind-1]; // overriding default config file
			break;
		case 'h':
			usage=1;
			break;
		case 'd':
            want_daemonize=true;
            break;
		default:
			usage=1;
			break;
		}
	}

	if (usage)
	{
		cout << "fireflier [-h] [-d] [-f] configfile" << endl;
		cout << "  -h ... this help" << endl;
		cout << "  -f ... configfile (/etc/fireflier/fireflier.conf)" << endl;
		cout << "  -d ... daemonize" << endl;
		exit(1);
	}

	if (getuid()!=0)
	{
		cerr << "You must run this program with root privileges." << endl;
        exit(1);
	}

	packetqueue=new queue();
	userspacequeue=new queue();
	timeoutqueue=new queue();


	loadconfig(Configfile);

	if (DEBUG)
	{
		cout << "Config: " << endl;
		cout << "  save_file: " << Config_save_file << endl;
		cout << "  ssl_file: " << Config_ssl_file << endl;
		cout << "  port: " << Config_port << endl;
		cout << "  client_ip: " << Config_client_ip << endl;
	}

	SSL_load_error_strings();
	SSLeay_add_ssl_algorithms();

	SSL_library_init();
	ssl_ctx=SSL_CTX_new(SSLv23_server_method());

	SSL_CTX_set_default_passwd_cb(ssl_ctx,pem_passwd_cb);

	if (SSL_CTX_use_certificate_file(ssl_ctx, Config_ssl_file, SSL_FILETYPE_PEM) <= 0) {
		fprintf(stderr, "Error in use_certificate_file\n");
		system_log("Error loading ssl certificate");
		//      ERR_print_errors_fp(stderr);
		exit(3);
	}
	if (SSL_CTX_use_PrivateKey_file(ssl_ctx, Config_ssl_file, SSL_FILETYPE_PEM) <= 0) {
		fprintf(stderr, "Error in user_privatekey_file\n");
		system_log("Error loading ssl certificate");
		//      ERR_print_errors_fp(stderr);
		exit(4);
	}

	if (!SSL_CTX_check_private_key(ssl_ctx)) {
		fprintf(stderr,"Private key does not match the certificate public key\n");
		system_log("Error loading ssl certificate");
		exit(5);
	}

	if (!initrules())
	{
		system_log("Error while finding iptables binary in (/sbin, /usr/sbin, /usr/local/sbin)");
		fprintf(stderr,"Error while finding iptables binary in (/sbin, /usr/sbin, /usr/local/sbin)\n");
		exit(6);
	}

    if (want_daemonize)
	{
        if (daemon(0, 0)==-1)
		{
            system_log("Error in daemon(0,0) call - exiting");
            exit(1);
		}
		daemonize=true;
		if (strcmp(Config_pid_file, "none"))
		{
			// first lets safe the pid
			FILE *f=fopen(Config_pid_file, "wt");
			if (f)
			{
				// first lets safe the pid
				fprintf(f, "%d", getpid());
				fclose(f);
			}
			else
			{
				system_log("Error while saving the pid of the child process");
				fprintf(stderr,"Error while saving the pid of the child process\n");
			}
		}
	}
	signal(SIGTERM, &finalize);
	signal(SIGINT, &finalize);
	signal(SIGPIPE, &pipesignal);

	loadUserspaceRules(Config_save_file); // load and convert userspace rules file

	if (Config_create_queue_rules) // create rules to pass packets from iptables to userspace
	{
		inputRule(ff::CHAIN_INPUT, 0, 0, 0, 0, 0, 0, 0, 0, ff::ACTION_QUEUE);
		inputRule(ff::CHAIN_OUTPUT, 0, 0, 0, 0, 0, 0, 0, 0, ff::ACTION_QUEUE);
		inputRule(ff::CHAIN_FORWARD, 0, 0, 0, 0, 0, 0, 0, 0, ff::ACTION_QUEUE);
		rules_active=1;
	}


	pthread_create(&kernelthr, NULL, kernelThread, NULL);
	pthread_create(&timeoutthr, NULL, timeoutThread, NULL);

	serverThread(NULL);
	return 0;
}
