/*---[ menus.c ]------------------------------------------------------
 * Copyright (C) 2000-2002 Tomas Junnonen (majix@sci.fi)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Menu and toolbar related functions
 *--------------------------------------------------------------------*/

#include "menus.h"
#include "firestarter.h"
#include "wizard.h"
#include "savelog.h"
#include "preferences.h"
#include "modrules.h"
#include "gui.h"
#include "hitview.h"
#include "ruleview.h"
#include "util.h"

#include "xpm/firestarter-pixbufs.h"

enum {
	TOOLBAR_WIZARD,
	TOOLBAR_SEP1,
	TOOLBAR_CLEAR,
	TOOLBAR_RELOAD,
	TOOLBAR_SAVE,
	TOOLBAR_SEP2,
	TOOLBAR_START,
	TOOLBAR_STOP,
};

static GtkAccelGroup *main_accel_group;

static void goto_homepage (void);
static void goto_manual (void);
static gchar *menu_translate(const char* path, gpointer data);
static void menu_set_checked (GtkItemFactory *factory, const gchar *path, gboolean active);
static GtkWidget *create_hits_popup (void);
static GtkWidget *create_rules_popup (void);
static void copy_selected (void);

GnomeUIInfo toolbar_data[] = {

	GNOMEUIINFO_SEPARATOR,

	{ GNOME_APP_UI_ITEM, 
	  N_("Reload"), N_("Reload Hit List"), 
	  hitview_reload, NULL, NULL,
	  GNOME_APP_PIXMAP_STOCK, GTK_STOCK_REVERT_TO_SAVED,
	  0, 0, NULL },
	
	{ GNOME_APP_UI_ITEM, 
	  N_("Save"), N_("Save Hit List to File"), 
	  create_savelog_filesel, NULL, NULL,
	  GNOME_APP_PIXMAP_STOCK, GTK_STOCK_SAVE_AS,
	  0, 0, NULL },

	GNOMEUIINFO_SEPARATOR,

	GNOMEUIINFO_END
};

static GtkItemFactory *main_menu_factory = NULL;
static int main_menu_entries_qty = 45;
static GtkItemFactoryEntry main_menu_entries[] =
{
	/*   */ {N_("/_Firewall"), NULL, NULL, 0, "<Branch>"},
	/* w */ {N_("/_Firewall/Run _Wizard"), "<control>W", run_wizard, 0, "<ImageItem>", icon_wizard},
	/*   */ {N_("/_Firewall/---"), NULL, NULL, 0, "<Separator>"},
	/* s */ {N_("/_Firewall/_Start Firewall"), "<control>S", start_firewall, 0, "<ImageItem>", icon_start_small},
	/* p */ {N_("/_Firewall/Sto_p Firewall"), "<control>P", stop_firewall, 0, "<ImageItem>", icon_stop_small},
	/* h */ {N_("/_Firewall/_Halt Network Traffic"), NULL, halt_firewall, 0, "<StockItem>", GTK_STOCK_DIALOG_WARNING},
	/*   */ {N_("/_Firewall/---"), NULL, NULL, 0, "<Separator>"},
	/* q */ {N_("/_Firewall/_Quit"), NULL, exit_firestarter, 0, "<StockItem>", GTK_STOCK_QUIT},

	/*   */ {N_("/_Edit"), NULL, NULL, 0, "<Branch>"},
	/* c */ {N_("/_Edit/_Copy"), NULL, copy_selected, 0, "<StockItem>", GTK_STOCK_COPY},
	/*   */ {N_("/_Edit/---"), NULL, NULL, 0, "<Separator>"},
	/* p */ {N_("/_Edit/_Preferences..."), NULL, preferences_cb, 0, "<StockItem>", GTK_STOCK_PREFERENCES},

	/*   */ {N_("/_View"), NULL, NULL, 0, "<Branch>"},
	/* c */ {N_("/_View/_Clear Hit List"), "<control>L", hitview_clear, 0, "<StockItem>", GTK_STOCK_CLEAR},
	/* l */ {N_("/_View/_Reload Hit List"), NULL, hitview_reload, 0, "<StockItem>", GTK_STOCK_REVERT_TO_SAVED},
	/*   */ {N_("/_View/---"), NULL, NULL, 0, "<Separator>"},
	/*   */ {N_("/_View/Hit columns"), NULL, 0, 0, "<Branch>"},
	/*   */ {N_("/_View/Hit columns/Time"), "<control>1", hitview_toggle_column_visibility, 0, "<CheckItem>"},
	/*   */ {N_("/_View/Hit columns/In"), "<control>2", hitview_toggle_column_visibility, 1, "<CheckItem>"},
	/*   */ {N_("/_View/Hit columns/Out"), "<control>3", hitview_toggle_column_visibility, 2, "<CheckItem>"},
	/*   */ {N_("/_View/Hit columns/Port"), "<control>4", hitview_toggle_column_visibility, 3, "<CheckItem>"},
	/*   */ {N_("/_View/Hit columns/Source"), "<control>5", hitview_toggle_column_visibility, 4, "<CheckItem>"},
	/*   */ {N_("/_View/Hit columns/Destination"), "<control>6", hitview_toggle_column_visibility, 5, "<CheckItem>"},
	/*   */ {N_("/_View/Hit columns/Length"), "<control>7", hitview_toggle_column_visibility, 6, "<CheckItem>"},
	/*   */ {N_("/_View/Hit columns/ToS"), "<control>8", hitview_toggle_column_visibility, 7, "<CheckItem>"},
	/*   */ {N_("/_View/Hit columns/Protocol"), "<control>9", hitview_toggle_column_visibility, 8, "<CheckItem>"},
	/*   */ {N_("/_View/Hit columns/Service"), "<control>0", hitview_toggle_column_visibility, 9, "<CheckItem>"},

	/*   */ {N_("/_Hits"), NULL, NULL, 0, "<Branch>"},
	/* b */ {N_("/_Hits/_Block Host"), NULL, block_host, 0, NULL},
	/* t */ {N_("/_Hits/_Trust Host"), NULL, trust_host, 0, NULL},
	/* o */ {N_("/_Hits/_Open Port"), NULL, open_port, 0, NULL},
	/* h */ {N_("/_Hits/Open Port for This _Host Only"), NULL, open_port_for_machine, 0, NULL},
	/* s */ {N_("/_Hits/Block and _Stop Logging Port"), NULL, block_port, 0, NULL},
	/* l */ {N_("/_Hits/_Lookup Hostname"), "<control>R", lookup_selected_hit, 0, "<StockItem>", GTK_STOCK_INDEX},
	/*   */ {N_("/_Hits/---"), NULL, NULL, 0, "<Separator>"},
	/* f */ {N_("/_Hits/Save Hit List to _File"), "F12", create_savelog_filesel, 0, "<StockItem>", GTK_STOCK_SAVE},


	/*   */ {N_("/_Rules"), NULL, NULL, 0, "<Branch>"},
	/* n */ {N_("/_Rules/_New Rule"), NULL, create_new_rule, 0, "<StockItem>", GTK_STOCK_ADD},
	/* r */ {N_("/_Rules/_Remove Rule"), "Delete", remove_rule, 0, "<StockItem>", GTK_STOCK_REMOVE},
	/* e */ {N_("/_Rules/_Edit Rule"), "<control>E", edit_selected_rule, 0, "<StockItem>", GTK_STOCK_FIND_AND_REPLACE},

	/*   */ {N_("/_Help"), NULL, NULL, 0, "<Branch>"},
	/* m */ {N_("/_Help/Online Users` _Manual"), NULL, goto_manual, 0, "<StockItem>", GTK_STOCK_HELP},
	/* h */ {N_("/_Help/Firestarter _Homepage"), NULL, goto_homepage, 0, "<StockItem>", GTK_STOCK_HOME},
	/*   */ {N_("/_Help/---"), NULL, NULL, 0, "<Separator>"},
	/* a */ {N_("/_Help/_About..."), NULL, show_about, 0, "<StockItem>", GTK_STOCK_DIALOG_QUESTION}
};

static GtkItemFactory *hits_popup_factory = NULL;
static GtkWidget *hits_popup_menu = NULL;
static int hits_popup_entries_qty = 7;

static GtkItemFactoryEntry hits_popup_entries[] =
{
	{N_("/_Block Host"), NULL, block_host, 0, NULL},
	{N_("/_Trust Host"), NULL, trust_host, 0, NULL},
	{N_("/_Open Port"), NULL, open_port, 0, NULL},
	{N_("/Open Port for This _Host Only"), NULL, open_port_for_machine, 0, NULL},
	{N_("/Block and _Stop Logging Port"), NULL, block_port, 0, NULL},
	{   "/---", NULL, NULL, 0, "<Separator>"},
	{N_("/_Lookup Hostname"), "<control>R", lookup_selected_hit, 0, NULL},
};

static GtkItemFactory *rules_popup_factory = NULL;
static GtkWidget *rules_popup_menu = NULL;
static int rules_popup_entries_qty = 3;

static GtkItemFactoryEntry rules_popup_entries[] =
{
	{N_("/New rule"), NULL, create_new_rule, 0, NULL},
	{N_("/Remove rule"), "Delete", remove_rule, 0, NULL},
	{N_("/Edit rule"), "<control>E", edit_selected_rule, 0, NULL}
};

static GtkWidget *
create_icon (const guint8 rgba_data [])
{
	GtkWidget *image;
	GdkPixbuf *pixbuf;
	
	pixbuf = gdk_pixbuf_new_from_inline (-1, rgba_data, FALSE, NULL);

	if (pixbuf == NULL)
		return NULL;

	image = gtk_image_new_from_pixbuf (pixbuf);
	g_object_unref (pixbuf);

	return image;
}

/* [ install_menus_and_toolbar ]
 * Install the menus and the toolbar in *window
 */
void
install_menus_and_toolbar (GtkWidget *window)
{
	GtkWidget *menubar;
	GtkWidget *toolbar;

	main_accel_group = gtk_accel_group_new ();
	gtk_window_add_accel_group (GTK_WINDOW (window), main_accel_group);
	g_object_unref (main_accel_group);

	main_menu_factory = gtk_item_factory_new (GTK_TYPE_MENU_BAR, "<main>", main_accel_group);
	gtk_item_factory_set_translate_func (main_menu_factory, menu_translate, NULL, NULL);
	gtk_item_factory_create_items (main_menu_factory, main_menu_entries_qty, main_menu_entries, NULL);
	menubar = gtk_item_factory_get_widget (main_menu_factory, "<main>");

	gnome_app_set_menus (GNOME_APP (window), GTK_MENU_BAR (menubar));

	/* Make the check marks match the column visibility */
	menu_set_checked (main_menu_factory, "/View/Hit columns/Time",
		preferences_get_bool (PREFS_HITVIEW_TIME_COL));

	menu_set_checked (main_menu_factory, "/View/Hit columns/In",
		preferences_get_bool (PREFS_HITVIEW_IN_COL));

	menu_set_checked (main_menu_factory, "/View/Hit columns/Out",
		preferences_get_bool (PREFS_HITVIEW_OUT_COL));

	menu_set_checked (main_menu_factory, "/View/Hit columns/Port",
		preferences_get_bool (PREFS_HITVIEW_PORT_COL));
	
	menu_set_checked (main_menu_factory, "/View/Hit columns/Source",
		preferences_get_bool (PREFS_HITVIEW_SOURCE_COL));

	menu_set_checked (main_menu_factory, "/View/Hit columns/Destination",
		preferences_get_bool (PREFS_HITVIEW_DESTINATION_COL));

	menu_set_checked (main_menu_factory, "/View/Hit columns/Length",
		preferences_get_bool (PREFS_HITVIEW_LENGTH_COL));

	menu_set_checked (main_menu_factory, "/View/Hit columns/ToS",
		preferences_get_bool (PREFS_HITVIEW_TOS_COL));

	menu_set_checked (main_menu_factory, "/View/Hit columns/Protocol",
		preferences_get_bool (PREFS_HITVIEW_PROTOCOL_COL));

	menu_set_checked (main_menu_factory, "/View/Hit columns/Service",
		preferences_get_bool (PREFS_HITVIEW_SERVICE_COL));


	toolbar = gtk_toolbar_new ();
	gnome_app_fill_toolbar_with_data (GTK_TOOLBAR (toolbar), 
	                                  toolbar_data, 
	                                  (GtkAccelGroup*) NULL,
	                                  window);

	gtk_toolbar_insert_element (GTK_TOOLBAR (toolbar),
				    GTK_TOOLBAR_CHILD_BUTTON,
				    NULL, 
				    _("Wizard"), 
				    _("Run Wizard"),
				    NULL, 
				    create_icon (icon_wizard),
				    GTK_SIGNAL_FUNC (run_wizard), 
				    window,
				    TOOLBAR_WIZARD);

	gtk_toolbar_insert_element (GTK_TOOLBAR (toolbar),
				    GTK_TOOLBAR_CHILD_BUTTON,
				    NULL, 
				    _("Clear"), 
				    _("Clear Hit List"),
				    NULL, 
				    create_icon (icon_clear),
				    GTK_SIGNAL_FUNC (hitview_clear), 
				    window,
				    TOOLBAR_CLEAR);

	gtk_toolbar_insert_element (GTK_TOOLBAR (toolbar),
				    GTK_TOOLBAR_CHILD_BUTTON,
				    NULL, 
				    _("Start"), 
				    _("Start Firewall"),
				    NULL, 
				    create_icon (icon_start),
				    GTK_SIGNAL_FUNC (start_firewall), 
				    window,
				    TOOLBAR_START);

	gtk_toolbar_insert_element (GTK_TOOLBAR (toolbar),
				    GTK_TOOLBAR_CHILD_BUTTON,
				    NULL, 
				    _("Stop"), 
				    _("Stop Firewall"),
				    NULL, 
				    create_icon (icon_stop),
				    GTK_SIGNAL_FUNC (stop_firewall), 
				    window,
				    TOOLBAR_STOP);

	gnome_app_set_toolbar (GNOME_APP (window), GTK_TOOLBAR (toolbar));

	appbar = gnome_appbar_new (FALSE, TRUE, GNOME_PREFERENCES_USER);
	gnome_app_set_statusbar (GNOME_APP (window), appbar);
	
	hits_popup_menu = create_hits_popup ();
	rules_popup_menu = create_rules_popup ();
}

/* [ hitview_popup_menu ]
 * Pops up a menu and sets the sensitivity of the menu items
 */
void
hitview_popup_menu (GdkEventButton *bevent)
{
	gtk_menu_popup (GTK_MENU (hits_popup_menu),
	                          NULL, NULL, NULL, NULL,
	                          bevent->button, bevent->time);
}

/* [ ruleview_popup_menu ]
 * Pops up a menu and sets the sensitivity of the menu items
 */
void
ruleview_popup_menu (GdkEventButton *bevent)
{
	gtk_menu_popup (GTK_MENU (rules_popup_menu),
	                          NULL, NULL, NULL, NULL,
	                          bevent->button, bevent->time);
}

static void
goto_homepage (void)
{
	gnome_url_show ("http://firestarter.sourceforge.net", NULL);
}

static void
goto_manual (void)
{
	gnome_url_show ("http://firestarter.sourceforge.net/manual", NULL);
}

static gchar *
menu_translate(const char* path, gpointer data)
{
	return gettext (path);
}

static GtkWidget *
menu_create_items (GtkItemFactoryEntry  *entries,
                   guint                 entries_qty,
                   const char           *path,
                   GtkItemFactory      **factory,
                   gpointer              data)
{
	*factory = gtk_item_factory_new (GTK_TYPE_MENU, path, NULL);
	gtk_item_factory_set_translate_func (*factory, menu_translate, NULL, NULL);
	gtk_item_factory_create_items (*factory, entries_qty, entries, data);

	return gtk_item_factory_get_widget (*factory, path);
}

/* [ menu_item_set_checked ]
 * Set the checked state of an CheckMenuItem
 */
static void
menu_set_checked (GtkItemFactory *ifactory,
                  const gchar *path,
		  gboolean active)
{
	GtkCheckMenuItem *w;

	g_return_if_fail (ifactory != NULL);

	w = GTK_CHECK_MENU_ITEM (gtk_item_factory_get_item (ifactory, path));
	if (w == NULL)
		g_error ("Couldn't find menu item \"%s\"", path);
	else if (!!w->active != !!active)
		gtk_check_menu_item_set_active (w, active);
}

/* [ create_hits_popup ]
 * Create the popup menu for the hitview
 */
static GtkWidget *
create_hits_popup (void)
{
	GtkWidget *menu;

	menu = menu_create_items (hits_popup_entries,
	                          hits_popup_entries_qty,
	                          "<HitView>",
	                          &hits_popup_factory,
	                          NULL);

	return menu;
}

/* [ create_rules_popup ]
 * Create the popup menu for the ruleview
 */
static GtkWidget *
create_rules_popup (void)
{
	GtkWidget *menu;

	menu = menu_create_items (rules_popup_entries,
	                          rules_popup_entries_qty,
	                          "<RuleView>",
	                          &rules_popup_factory,
	                          NULL);

	return menu;
}

/* [ copy_selected ]
 * Copy the selection in the active view to the clipboard
 */
static void
copy_selected (void)
{
	FirestarterView v;

	v = gui_get_active_view ();

	if (v == HITVIEW_PAGE)
		copy_selected_hit ();
	else if (v == RULEVIEW_PAGE)
		copy_selected_rule ();
}

