/*---[ modrules.c ]---------------------------------------------------
 * Copyright (C) 2000-2002 Tomas Junnonen (majix@sci.fi)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Rule creation and manipulation 
 *--------------------------------------------------------------------*/

#include <sys/stat.h>
#include <sys/types.h>

#include "globals.h"
#include "firestarter.h"
#include "modrules.h"
#include "util.h"
#include "ruleview.h"
#include "hitview.h"

static void save_rule_group (gint rulegroupnum);
static void write_rule (FILE *f, Rule *r);

/* [ remove_rule ]
 * Removes the selected rule from the view and the disk file
 */
void
remove_rule (void)
{
	GtkTreeSelection *selection;
	GtkTreeIter iter;
	gboolean has_selected;
	Rule *r;

	r = ruleview_get_selected_rule ();
	if (r == NULL)
		return;

	/* Get the position (iter) of the rule */
	selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (Firestarter.ruleview));
	has_selected = gtk_tree_selection_get_selected (selection,
	                                 NULL,
	                                 &iter);

	/* Check that the root items aren't removed */
	if (is_root_item (iter)) {
		g_free (r);
		return;
	}

	if (has_selected)
		ruleview_remove_rule (iter);	

	save_rule_group (r->type);
	g_free (r);
}

/* [ edit_selected_rule ]
 * Edit the currently selected rule
 */
void
edit_selected_rule (void)
{
	Rule *r;
	GtkWidget *dialog;

	r = ruleview_get_selected_rule ();
	if (r == NULL)
		return;

	print_rule (r);

	if (r->type == RULETYPE_GROUP) {
		g_free (r);
		return;
	}

	dialog = create_rule_dialog (r, _("Edit rule"));
	/* Note: this rule is free'd in rule_edit_response_cb */
	gtk_widget_show_all (dialog);
}

/* [ create_new_rule ]
 * Create a new rule in the same group as the selected rule
 */
void
create_new_rule (void)
{
	Rule *r;
	Rule *new_rule;
	GtkWidget *dialog;
	GtkTreeSelection *selection;
	gchar *title = "";

	r = ruleview_get_selected_rule ();
	if (r == NULL)
		return;

	new_rule = g_new (Rule, 1);

	if (r->type == RULETYPE_GROUP) {
	/* Here I compare with the port string because it is unique and not
	   translated for the special root menu entries */
		if (strcmp (r->port, "1") == 0) {
			new_rule->type = RULETYPE_TRUSTED_HOST;
		}
		if (strcmp (r->port, "2") == 0) {
			new_rule->type = RULETYPE_BLOCKED_HOST;
		}
		if (strcmp (r->port, "3") == 0) {
			Forward *f = g_new (Forward, 1);

			new_rule->type = RULETYPE_FORWARD;
			f->int_host = "";
			f->int_port = "";
			new_rule->extension = f;			
		}
		if (strcmp (r->port, "4") == 0) {
			new_rule->type = RULETYPE_OPEN_PORT;
		}
		if (strcmp (r->port, "5") == 0) {
			new_rule->type = RULETYPE_STEALTHED_PORT;
		}
		if (strcmp (r->port, "6") == 0) {
			new_rule->type = RULETYPE_BLOCKED_PORT;
		}
	} else
		new_rule->type = r->type;

	switch (new_rule->type) {
	  case RULETYPE_TRUSTED_HOST: title = g_strdup (_("Add trust host rule")); break;
	  case RULETYPE_BLOCKED_HOST: title = g_strdup (_("Add block host rule")); break;
	  case RULETYPE_FORWARD: title = g_strdup (_("Add forwarding rule")); break;
	  case RULETYPE_OPEN_PORT: title = g_strdup (_("Add open port rule")); break;
	  case RULETYPE_STEALTHED_PORT: title = g_strdup (_("Add stealth port rule")); break;
	  case RULETYPE_BLOCKED_PORT: title = g_strdup (_("Add block port rule")); break;
	}

	new_rule->host = "";
	new_rule->port = "";
	new_rule->active = 1;

	selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (Firestarter.ruleview));
	gtk_tree_selection_unselect_all (selection);

	dialog = create_rule_dialog (new_rule, title);
	gtk_widget_show_all (dialog);

	g_free (r);
}

/* [ save_rule_group ]
 * Write the current state of one rulegroup to disk
 */
static void
save_rule_group (gint rulegroupnum)
{
	GtkTreeIter *parent;
	GtkTreeIter child;
	GtkTreeModel *model;
	Rule *r;
	gchar *path;
	FILE *f;
	FirewallStatus state;

	parent = get_rulegroup_iter (rulegroupnum);
	model = gtk_tree_view_get_model (GTK_TREE_VIEW (Firestarter.ruleview));

	/* Overwrite existing rule file with current ruleview state */
	path = (gchar *)get_file_path (rulegroupnum);
	f = fopen (path, "w");

	state = get_current_status ();
	
	/* Get the first child of the rulegroup */
	if (!gtk_tree_model_iter_children (model, &child, parent)) {
		printf ("No rules remaining\n");
		fclose (f);
		if (state == STATUS_RUNNING || state == STATUS_HIT)
			start_firewall ();
		return;
	}

	/* Write all the rules in the group to the file */
	do {
		r = get_rule (model, child);
		write_rule (f, r);		

		g_free (r);
	} while (gtk_tree_model_iter_next (model, &child));

	fclose (f);
	if (state == STATUS_RUNNING || state == STATUS_HIT)
		start_firewall ();
}

/* [ save_rule_group ]
 * Write the rule data to a file
 */
static void
write_rule (FILE *f, Rule *r)
{
	switch (r->type) {
	  case RULETYPE_TRUSTED_HOST:
	  case RULETYPE_BLOCKED_HOST:  
		fprintf (f, "%s\n", r->host);
		break;

	  case RULETYPE_FORWARD: {
		Forward *ext;

		ext = (Forward *)r->extension;
		fprintf (f, "%s %s %s\n", r->port, ext->int_host, ext->int_port);
	  	break;
	  }

	  case RULETYPE_OPEN_PORT:
	  case RULETYPE_BLOCKED_PORT:
		fprintf (f, "%s\n", r->port);
		break;

	  case RULETYPE_STEALTHED_PORT:
		fprintf (f, "%s %s\n", r->port, r->host);
		break;
	  default: g_assert (FALSE);
	}
}

/* [ append_rule ]
 * Append a single rule to end of the correct rule file and the ruleview
 */
void
append_rule (Rule *r)
{
	gchar *path;
	FILE *f;
	FirewallStatus state;

	path = (gchar *)get_file_path (r->type);
	f = fopen (path, "a");
	if (!f) {
		perror(g_strconcat ("Error reading file ", path, NULL));
		return;
	}

	ruleview_insert_rule (r);

	write_rule (f, r);
	fclose (f);

	state = get_current_status ();
	if (state == STATUS_RUNNING || state == STATUS_HIT)
		start_firewall ();
}

/* [ trust_host ]
 * Allow all traffic from selected host
 */
void
trust_host (void)
{
	Hit *h;
	Rule *r = g_new (Rule, 1);

	h = hitview_get_selected_hit ();
	
	if (h) {
		r->type = RULETYPE_TRUSTED_HOST;
		r->host = g_strdup(h->source);
		r->port = "";
		r->active = 1;

		append_rule (r);	
	}

	free_hit (h);
	g_free (r);
}

/* [ block_host ]
 * Deny all traffic from selected host
 */
void
block_host (void)
{
	Hit *h;
	Rule *r = g_new (Rule, 1);

	h = hitview_get_selected_hit ();

	if (h) {
		r->type = RULETYPE_BLOCKED_HOST;
		r->host = g_strdup(h->source);
		r->port = "";
		r->active = 1;

		append_rule (r);
	}

	free_hit (h);
	g_free (r);
}

/* [ open_port_for_machine ]
 * Open a specific service to a specific machine
 */
void
open_port_for_machine (void)
{
	Hit *h;
	Rule *r = g_new (Rule, 1);

	h = hitview_get_selected_hit ();

	if (h) {
		r->type = RULETYPE_STEALTHED_PORT;
		r->host = g_strdup(h->source);
		r->port = g_strdup(h->port);
		r->active = 1;

		append_rule (r);
	}

	free_hit (h);
	g_free (r);
}

/* [ open_port ]
 * Open a specific service to everyone
 */
void
open_port (void)
{
	Hit *h;
	Rule *r = g_new (Rule, 1);

	h = hitview_get_selected_hit ();

	if (h) {
		r->type = RULETYPE_OPEN_PORT;
		r->host = "";
		r->port = g_strdup(h->port);
		r->active = 1;

		append_rule (r);
	}

	free_hit (h);
	g_free (r);
}

/* [ block_port ]
 * Block a specific service from everyone explicitly
 */
void
block_port (void)
{
	Hit *h;
	Rule *r = g_new (Rule, 1);

	h = hitview_get_selected_hit ();

	if (h) {
		r->type = RULETYPE_BLOCKED_PORT;
		r->host = "";
		r->port = g_strdup(h->port);
		r->active = 1;

		append_rule (r);
	}

	free_hit (h);
	g_free (r);
}
