/*---[ netfilter-script.c ]-------------------------------------------
 * Copyright (C) 2000-2003 Tomas Junnonen (majix@sci.fi)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Functions to write the netfilter shell scripts
 *--------------------------------------------------------------------*/

#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <stdio.h>   
#include <errno.h>

#include "wizard.h"
#include "netfilter-script.h"
#include "util.h"
#include "preferences.h"

/* [ write_netfilter_script ]
 * Creates the netfilter shell script
 */
void
write_netfilter_script (void)
{
	gboolean nat = preferences_get_bool (PREFS_FW_NAT);
	gchar *scriptpath = FIRESTARTER_RULES_DIR "/firestarter/firewall.sh";
	FILE *script = fopen (scriptpath, "w");

        if (script == NULL) {
                /* Use perror to get sane error messages */
                perror(scriptpath);
                g_printerr("Script not written!");

		return;
	}
	
	chmod (scriptpath, 00700);

	fprintf (script, "#!/bin/sh\n");
	fprintf (script, "# Generated by Firestarter " VERSION ", NETFILTER in use\n\n");
  
    fprintf (script, "# --------( Initial Setup - Variables (required) )--------\n\n");

	fprintf (script, "# Type of Service (TOS) parameters\n");
	fprintf (script, "# 8: Maximum Throughput - Minimum Delay\n");
	fprintf (script, "# 4: Minimize Delay - Maximize Reliability\n");
	fprintf (script, "# 16: No Delay - Moderate Throughput - High Reliability\n\n");
	
	if (preferences_get_bool (PREFS_FW_TOS_OPT_TROUGHPUT))
		fprintf (script, "TOSOPT=8\n\n");
	if (preferences_get_bool (PREFS_FW_TOS_OPT_RELIABILITY))
		fprintf (script, "TOSOPT=4\n\n");
	if (preferences_get_bool (PREFS_FW_TOS_OPT_DELAY))
		fprintf (script, "TOSOPT=16\n\n");

	fprintf (script, "# Default Packet Rejection Type\n");
	fprintf (script, "# ( do NOT change this here - set it in the GUI instead )\n\n");

    fprintf (script, "# --------( Initial Setup - System Utilities Configuration )--------\n\n");

	/* If the system binaries can't be found, try to locate them */
	if (access("/sbin/iptables", R_OK) == 0)
		fprintf (script, "IPT=/sbin/iptables\n");
	else
		fprintf (script, "IPT=`which iptables`\n");
	if (access("/sbin/ifconfig", R_OK) == 0)
		fprintf (script, "IFC=/sbin/ifconfig\n");
	else
		fprintf (script, "IFC=`which ifconfig`\n");
	if (access("/sbin/modprobe", R_OK) == 0)
		fprintf (script, "MPB=/sbin/modprobe\n");
	else
		fprintf (script, "MPB=`which modprobe`\n");
	if (access("/sbin/lsmod", R_OK) == 0)
		fprintf (script, "LSM=/sbin/lsmod\n");
	else
		fprintf (script, "LSM=`which lsmod`\n");
	if (access("/sbin/rmmod", R_OK) == 0)
		fprintf (script, "RMM=/sbin/rmmod\n\n");
	else
		fprintf (script, "RMM=`which rmmod`\n\n");
			
    fprintf (script, "# --------( Initial Setup - Network Information (required) )--------\n\n");

	/* The external network interface */
	fprintf (script, "IF=%s\n", preferences_get_string (PREFS_FW_EXT_IF));
	/* IP address of external interface */
	fprintf (script, "IP=`/sbin/ifconfig $IF | grep inet | cut -d : -f 2 | cut -d \\  -f 1`\n");
	/* Netmask/net of external interface */
	fprintf (script, "MASK=`/sbin/ifconfig $IF | grep Mas | cut -d : -f 4`\n");
	fprintf (script, "NET=$IP/$MASK\n\n");

	/* The internal network interface */
	if (nat) {
		fprintf (script, "INIF=%s\n", preferences_get_string (PREFS_FW_INT_IF));
		/* IP address of internal interface */
		fprintf (script, "INIP=`/sbin/ifconfig $INIF | grep inet | cut -d : -f 2 | cut -d \\  -f 1`\n");
		/* Netmask/net of internal interface */
		fprintf (script, "INMASK=`/sbin/ifconfig $INIF | grep Mas | cut -d : -f 4`\n");
		fprintf (script, "INNET=$INIP/$INMASK\n\n");
	}

	fprintf (script, "if [ \"$MASK\" = \"\" ]; then\n");
	fprintf (script, "\techo \"External network device $IF is not ready. Aborting..\"\n\texit 2\nfi\n\n");

	if (nat) {
		fprintf (script, "if [ \"$INMASK\" = \"\" ]; then\n");
		fprintf (script, "\techo \"Internal network device $INIF is not ready. Aborting..\"\n\texit 3\nfi\n\n");
	}

    fprintf (script, "# --------( Initial Setup - Firewall Modules Check )--------\n\n");

	fprintf (script, "# Some distributions still load ipchains\n");
	fprintf (script, "$LSM | grep ipchains -q -s && $RMM ipchains\n\n");

	/* Hack for autoloading of netfilter modules (such as the reject chain or ToS) - must be done before
	chains are flushed.*/

    fprintf (script, "# --------( Initial Setup - Firewall Modules Autoloader )--------\n\n");

	fprintf (script, "if ! ( $LSM | /bin/grep ip_conntrack > /dev/null ); then\n");
	fprintf (script, "$MPB ip_conntrack\nfi\n");
	fprintf (script, "if ! ( $LSM | /bin/grep ip_conntrack_ftp > /dev/null ); then\n");
	fprintf (script, "$MPB ip_conntrack_ftp\nfi\n");
	fprintf (script, "if ! ( $LSM | /bin/grep ip_conntrack_irc > /dev/null ); then\n");
	fprintf (script, "$MPB ip_conntrack_irc\nfi\n");
	fprintf (script, "if ! ( $LSM | /bin/grep ipt_REJECT > /dev/null ); then\n");
	fprintf (script, "$MPB ipt_REJECT\nfi\n");
	fprintf (script, "if ! ( $LSM | /bin/grep ipt_REDIRECT > /dev/null ); then\n");
	fprintf (script, "$MPB ipt_REDIRECT\nfi\n");
	fprintf (script, "if ! ( $LSM | /bin/grep ipt_TOS > /dev/null ); then\n");
	fprintf (script, "$MPB ipt_TOS\nfi\n");
	fprintf (script, "if ! ( $LSM | /bin/grep ipt_MASQUERADE > /dev/null ); then\n");
	fprintf (script, "$MPB ipt_MASQUERADE\nfi\n");
	fprintf (script, "if ! ( $LSM | /bin/grep ipt_LOG > /dev/null ); then\n");
	fprintf (script, "$MPB ipt_LOG\nfi\n");
	fprintf (script, "if ! ( $LSM | /bin/grep iptable_mangle > /dev/null ); then\n");
	fprintf (script, "$MPB iptable_mangle\nfi\n");
	fprintf (script, "if ! ( $LSM | /bin/grep iptable_nat > /dev/null ); then\n");
	fprintf (script, "$MPB iptable_nat\nfi\n\n");
	fprintf (script, "if ! ( $LSM | /bin/grep ipt_ipv4optsstrip > /dev/null ); then\n");
	fprintf (script, "$MPB iptable_nat 2> /dev/null\nfi\n\n");

	if (preferences_get_bool (PREFS_FW_EXT_PPP)) {
		fprintf (script, "# --------( Initial Setup - Additional Firewall Modules (PPP) )--------\n\n");

		fprintf (script, "if ! ( $LSM | /bin/grep bsd_comp > /dev/null ); then\n");
		fprintf (script, "$MPB bsd_comp 2> /dev/null\nfi\n");
		fprintf (script, "if ! ( $LSM | /bin/grep _deflate > /dev/null ); then\n");
		fprintf (script, "$MPB ppp_deflate 2> /dev/null\nfi\n\n");
	}

	/* Flushing must be done before we load the external files */

   fprintf (script, "# --------( Chain Configuration - Flush Existing Chains )--------\n\n");

	fprintf (script, "# Purge standard chains (INPUT, OUTPUT, FORWARD).\n\n");
	fprintf (script, "$IPT -F\n$IPT -X\n$IPT -Z\n\n");

	fprintf (script, "# Purge extended chains (MANGLE & NAT) if they exist.\n\n");
	fprintf (script, "if ( $LSM | /bin/grep iptable_mangle > /dev/null ); then\n");
	fprintf (script, "$IPT -t mangle -F\n$IPT -t mangle -X\n$IPT -t mangle -Z\nfi\n");
	fprintf (script, "if ( $LSM | /bin/grep iptable_nat > /dev/null ); then\n");
	fprintf (script, "$IPT -t nat -F\n$IPT -t nat -X\n$IPT -t nat -Z\nfi\n\n");

	fprintf (script, "# Remove Firestarter lock\n");
	fprintf (script, "if [ -e /var/lock/subsys ]; then\n"
	"  rm -f /var/lock/subsys/firestarter\nelse\n"
	"  rm -f /var/lock/firestarter\n"
	"fi\n\n");

   fprintf (script, "# --------( Chain Configuration - Configure Default Policy )--------\n\n");
	fprintf (script, "# Configure standard chains (INPUT, OUTPUT, FORWARD).\n\n");
	fprintf (script, "$IPT -P INPUT DROP\n");
	fprintf (script, "$IPT -P OUTPUT DROP\n");
	fprintf (script, "$IPT -P FORWARD DROP\n\n");

	fprintf (script, "# Configure extended chains (MANGLE & NAT) if required.\n\n");
	fprintf (script, "if ( $LSM | /bin/grep iptable_mangle > /dev/null ); then\n");
	fprintf (script, "$IPT -t mangle -P INPUT ACCEPT\n");
	fprintf (script, "$IPT -t mangle -P OUTPUT ACCEPT\n");
	fprintf (script, "$IPT -t mangle -P PREROUTING ACCEPT\n");
	fprintf (script, "$IPT -t mangle -P POSTROUTING ACCEPT\nfi\n");
	fprintf (script, "if ( $LSM | /bin/grep iptable_nat > /dev/null ); then\n");
	fprintf (script, "$IPT -t nat -P OUTPUT ACCEPT\n");
	fprintf (script, "$IPT -t nat -P PREROUTING ACCEPT\n");
	fprintf (script, "$IPT -t nat -P POSTROUTING ACCEPT\nfi\n\n");

   fprintf (script, "# --------( Chain Configuration - Create Default Result Chains )--------\n\n");

	if (preferences_get_bool (PREFS_FW_DENY_PACKETS)) {
		fprintf (script, "# Create a new log and drop (LD) convenience chain.\n");
		fprintf (script, "$IPT -N LD 2> /dev/null\n$IPT -F LD\n$IPT -A LD -j LOG --log-level=info\n$IPT -A LD -j DROP\n\n");
		fprintf (script, "STOP=LD\n\n");
	} else {
		fprintf (script, "# Create a new log and reject (LR) convenience chain.\n");
		fprintf (script, "$IPT -N LR 2> /dev/null\n$IPT -F LR\n$IPT -A LR -j LOG --log-level=info\n$IPT -A LR -j REJECT\n\n");
		fprintf (script, "STOP=LR\n\n");
	}

	/* use the LD / LR chains as the overall result - these just mean we can process local packets
	safer as well as using some of the newer netfilter functionality to cut down on code later
	kudos to gShield and the netfilter mailing list for helping me nail these down -fnk 06/2k1*/

   fprintf (script, "# --------( Chain Configuration - Create Default Traffic Chains )--------\n\n");

	fprintf (script, "# Create a new 'stateful module check' (STATE) convenience chain.\n");
	fprintf (script, "$IPT -N STATE 2> /dev/null\n$IPT -F STATE\n$IPT -I STATE -m state --state NEW -i ! lo -j $STOP\n$IPT -A STATE -m state --state ESTABLISHED,RELATED -j ACCEPT\n$IPT -A STATE -j $STOP\n\n");
	fprintf (script, "# Create a new 'sanity (check, mark and fwd) check' (SANITY) convenience chain.\n");
	fprintf (script, "$IPT -N SANITY 2> /dev/null\n$IPT -F SANITY\n$IPT -I SANITY -p tcp --tcp-flags SYN,ACK SYN,ACK -m state --state NEW -j REJECT --reject-with tcp-reset\n$IPT -A SANITY -j $STOP\n\n");

	fprintf (script, "# --------( Initial Setup - Nameservers )--------\n\n");

	fprintf (script, "# Allow responses from the nameservers\n");
	fprintf (script, "while read keyword value garbage\n\tdo\n");
	fprintf (script, "\t\tif [ \"$keyword\" = \"nameserver\" ] ; then\n");
	fprintf (script, "\t\t\t$IPT -A INPUT -p tcp ! --syn -s $value -d 0/0 -j ACCEPT\n");
	fprintf (script, "\t\t\t$IPT -A INPUT -p udp -s $value -d 0/0 -j ACCEPT\n");
	fprintf (script, "\t\tfi\n\tdone < /etc/resolv.conf\n\n");

	fprintf (script, "# --------( User Defined Pre Script )--------\n\n");
	fprintf (script, "sh "FIRESTARTER_RULES_DIR"/firestarter/user-pre\n\n");

	/* Pipe in the modrules files */
	fprintf (script, "# --------( Initial Setup - External Lists )--------\n\n");

	fprintf (script, "# Trusted hosts\n");
	fprintf (script, "while read host garbage\n\tdo\n");
	fprintf (script, "\t\t$IPT -A INPUT -s $host -d 0/0 -j ACCEPT\n");
	fprintf (script, "\tdone < %s\n\n", get_file_path (RULETYPE_TRUSTED_HOST));

	fprintf (script, "# Blocked hosts\n");
	fprintf (script, "while read host garbage\n\tdo\n");
	fprintf (script, "\t\t$IPT -A INPUT -s $host -d 0/0 -j DROP\n");
	fprintf (script, "\tdone < %s\n\n", get_file_path (RULETYPE_BLOCKED_HOST));

	if (is_capable_of_nat ()) {
		fprintf (script, "# Forwarded ports\n");
		fprintf (script, "while read port int_host int_port garbage\n\tdo\n");
		fprintf (script, "\t\t$IPT -A FORWARD -p tcp -d $int_host --dport $int_port -j ACCEPT\n");
		fprintf (script, "\t\t$IPT -A FORWARD -p udp -d $int_host --dport $int_port -j ACCEPT\n");
		fprintf (script, "\t\t$IPT -A PREROUTING -t nat -p tcp -d $NET --dport $port -j DNAT --to $int_host:$int_port\n");
		fprintf (script, "\t\t$IPT -A PREROUTING -t nat -p udp -d $NET --dport $port -j DNAT --to $int_host:$int_port\n");
		fprintf (script, "\tdone < %s\n\n", get_file_path (RULETYPE_FORWARD));
	}

	fprintf (script, "# Open ports\n");
	fprintf (script, "while read port garbage\n\tdo\n");
	fprintf (script, "\t\t$IPT -A INPUT -p tcp -s 0/0 -d $NET --dport $port -j ACCEPT\n");
	fprintf (script, "\t\t$IPT -A INPUT -p udp -s 0/0 -d $NET --dport $port -j ACCEPT\n");
	fprintf (script, "\tdone < %s\n\n", get_file_path (RULETYPE_OPEN_PORT));
	
	fprintf (script, "# Stealthed ports (Ports open to specific hosts)\n");
	fprintf (script, "while read port host garbage\n\tdo\n");
	fprintf (script, "\t\t$IPT -A INPUT -p tcp -s $host -d $NET --dport $port -j ACCEPT\n");
	fprintf (script, "\t\t$IPT -A INPUT -p udp -s $host -d $NET --dport $port -j ACCEPT\n");
	fprintf (script, "\tdone < %s\n\n", get_file_path (RULETYPE_STEALTHED_PORT));

	fprintf (script, "# Blocked ports (explicit, no logging)\n");
	fprintf (script, "while read port garbage\n\tdo\n");
	fprintf (script, "\t\t$IPT -A INPUT -p tcp -s 0/0 -d 0/0 --dport $port -j DROP\n");
	fprintf (script, "\t\t$IPT -A INPUT -p udp -s 0/0 -d 0/0 --dport $port -j DROP\n");
	fprintf (script, "\tdone < %s\n\n", get_file_path (RULETYPE_BLOCKED_PORT));	

   fprintf (script, "# --------( Sysctl Tuning - Recommended Parameters )--------\n\n");
   
	fprintf (script, "# Turn off IP forwarding by default\n");
	fprintf (script, "# (this will be enabled if you require masquerading)\n\n");
	fprintf (script, "if [ -e /proc/sys/net/ipv4/ip_forward ]; then\n"
	"  echo 0 > /proc/sys/net/ipv4/ip_forward\nfi\n\n");
	
	fprintf (script, "# Do not log 'odd' IP addresses (excludes 0.0.0.0 & 255.255.255.255)\n\n");
	fprintf (script, "if [ -e /proc/sys/net/ipv4/conf/all/log_martians ]; then\n"
	"  echo 0 > /proc/sys/net/ipv4/conf/all/log_martians\nfi\n\n");	

   fprintf (script, "# --------( Sysctl Tuning - TCP Parameters )--------\n\n");
   
	fprintf (script, "# Turn off TCP Timestamping in kernel\n");
	fprintf (script, "if [ -e /proc/sys/net/ipv4/tcp_timestamps ]; then\n"
	"  echo 0 > /proc/sys/net/ipv4/tcp_timestamps\nfi\n\n");    

	fprintf (script, "# Set TCP Re-Ordering value in kernel to '5'\n");
	fprintf (script, "if [ -e /proc/sys/net/ipv4/tcp_reordering ]; then\n"
	"  echo 5 > /proc/sys/net/ipv4/tcp_reordering\nfi\n\n");
 
	fprintf (script, "# Turn off TCP ACK in kernel\n");
	fprintf (script, "if [ -e /proc/sys/net/ipv4/tcp_sack ]; then\n"
	"  echo 0 > /proc/sys/net/ipv4/tcp_sack\nfi\n\n");

	fprintf (script, "#Turn off TCP Window Scaling in kernel\n");
	fprintf (script, "if [ -e /proc/sys/net/ipv4/tcp_window_scaling ]; then\n"
	"  echo 0 > /proc/sys/net/ipv4/tcp_window_scaling\nfi\n\n");

	fprintf (script, "#Set Keepalive timeout to 1800 seconds\n");
	fprintf (script, "if [ -e /proc/sys/net/ipv4/tcp_keepalive_time ]; then\n"
	"  echo 1800 > /proc/sys/net/ipv4/tcp_keepalive_time\nfi\n\n");

	fprintf (script, "#Set FIN timeout to 30 seconds\n");
	fprintf (script, "if [ -e /proc/sys/net/ipv4/tcp_fin_timeout ]; then\n"
	"  echo 30 > /proc/sys/net/ipv4/tcp_fin_timeout\nfi\n\n");

	fprintf (script, "# Set TCP retry count to 3\n");
	fprintf (script, "if [ -e /proc/sys/net/ipv4/tcp_retries1 ]; then\n"
	"  echo 3 > /proc/sys/net/ipv4/tcp_retries1\nfi\n\n");
    
/* note: ECN is now actually an RFC - this is just a stopgap measure until certain
 OS'es get their act together */
 
	fprintf (script, "#Turn off ECN notification in kernel\n");
	fprintf (script, "if [ -e /proc/sys/net/ipv4/tcp_ecn ]; then\n"
	"  echo 0 > /proc/sys/net/ipv4/tcp_ecn\nfi\n\n");
	
   fprintf (script, "# --------( Sysctl Tuning - SYN Parameters )--------\n\n");
   
	fprintf (script, "# Turn on SYN cookies protection in kernel\n");
	fprintf (script, "if [ -e /proc/sys/net/ipv4/tcp_syncookies ]; then\n"
	"  echo 1 > /proc/sys/net/ipv4/tcp_syncookies\nfi\n\n");
	
	fprintf (script, "# Set SYN ACK retry attempts to '3'\n");
	fprintf (script, "if [ -e /proc/sys/net/ipv4/tcp_synack_retries ]; then\n"
	"  echo 3 > /proc/sys/net/ipv4/tcp_synack_retries\nfi\n\n");

	fprintf (script, "# Set SYN backlog buffer to '64'\n");
	fprintf (script, "if [ -e /proc/sys/net/ipv4/tcp_max_syn_backlog ]; then\n"
	"  echo 64 > /proc/sys/net/ipv4/tcp_max_syn_backlog\nfi\n\n");
	
	fprintf (script, "# Set SYN retry attempts to '6'\n");
	fprintf (script, "if [ -e /proc/sys/net/ipv4/tcp_syn_retries ]; then\n"
	"  echo 6 > /proc/sys/net/ipv4/tcp_syn_retries\nfi\n\n");
	
   fprintf (script, "# --------( Sysctl Tuning - Routing / Redirection Parameters )--------\n\n");

/* under 2.4 - source route verification only has 0 (off) and 1 (RFC compliant) */

	fprintf (script, "# Turn on source address verification in kernel\n");
	fprintf (script, "if [ -e /proc/sys/net/ipv4/conf/all/rp_filter ]; then\n"
	"  for f in /proc/sys/net/ipv4/conf/*/rp_filter\n  do\n   echo 1 > $f\n  done\nfi\n\n");
	
	fprintf (script, "# Turn off source routes in kernel\n");
	fprintf (script, "if [ -e /proc/sys/net/ipv4/conf/all/accept_source_route ]; then\n"
	"  for f in /proc/sys/net/ipv4/conf/*/accept_source_route\n  do\n   echo 0 > $f\n  done\nfi\n\n");

	fprintf (script, "# Do not respond to 'redirected' packets\n");
	fprintf (script, "if [ -e /proc/sys/net/ipv4/secure_redirects ]; then\n"
	"  echo 0 > /proc/sys/net/ipv4/secure_redirects\nfi\n\n");
	
	fprintf (script, "# Do not reply to 'redirected' packets if requested\n");
	fprintf (script, "if [ -e /proc/sys/net/ipv4/send_redirects ]; then\n"
	"  echo 0 > /proc/sys/net/ipv4/send_redirects\nfi\n\n");
    
	fprintf (script, "# Do not reply to 'proxyarp' packets\n");
	fprintf (script, "if [ -e /proc/sys/net/ipv4/proxy_arp ]; then\n"
	"  echo 0 > /proc/sys/net/ipv4/proxy_arp\nfi\n\n");
	
	fprintf (script, "# Set FIB model to be RFC1812 Compliant\n");
	fprintf (script, "# (certain policy based routers may break with this - if you find\n");
	fprintf (script, "#  that you can't access certain hosts on your network - please set\n");
	fprintf (script, "#  this option to '0' - which is the default)\n\n");
	fprintf (script, "if [ -e /proc/sys/net/ipv4/ip_fib_model ]; then\n"
	"  echo 2 > /proc/sys/net/ipv4/ip_fib_model\nfi\n\n");

   fprintf (script, "# --------( Sysctl Tuning - ICMP/IGMP Parameters )--------\n\n");
   
	fprintf (script, "# ICMP Dead Error Messages protection\n");
	fprintf (script, "if [ -e /proc/sys/net/ipv4/icmp_ignore_bogus_error_responses ]; then\n"
	"  echo 1 > /proc/sys/net/ipv4/icmp_ignore_bogus_error_responses\nfi\n\n");

	fprintf (script, "# ICMP Broadcasting protection\n");
	fprintf (script, "if [ -e /proc/sys/net/ipv4/icmp_echo_ignore_broadcasts ]; then\n"
	"  echo 1 > /proc/sys/net/ipv4/icmp_echo_ignore_broadcasts\nfi\n\n");
	
	fprintf (script, "# IGMP Membership 'overflow' protection\n");
	fprintf (script, "# (if you are planning on running your box as a router - you should either\n");
	fprintf (script, "#  set this option to a number greater than 5, or disable this protection\n");
	fprintf (script, "#  altogether by commenting out this option)\n\n");
	fprintf (script, "if [ -e /proc/sys/net/ipv4/igmp_max_memberships ]; then\n"
	"  echo 1 > /proc/sys/net/ipv4/igmp_max_memberships\nfi\n\n");

   fprintf (script, "# --------( Sysctl Tuning - Miscellanous Parameters )--------\n\n");
   
   	fprintf (script, "# Set TTL to '64' hops\n");
   	fprintf (script, "# (If you are running a masqueraded network, or use policy-based\n");
   	fprintf (script, "#  routing - you may want to increase this value depending on the load\n");
   	fprintf (script, "#  on your link.)\n\n");
	fprintf (script, "if [ -e /proc/sys/net/ipv4/conf/all/ip_default_ttl ]; then\n"
	"  for f in /proc/sys/net/ipv4/conf/*/ip_default_ttl\n  do\n   echo 64 > $f\n  done\nfi\n\n");

  	fprintf (script, "# Always defragment incoming packets\n");
   	fprintf (script, "# (Some cable modems [ Optus @home ] will suffer intermittent connection\n");
   	fprintf (script, "#  droputs with this setting. If you experience problems, set this to '0')\n\n");	
	fprintf (script, "if [ -e /proc/sys/net/ipv4/ip_always_defrag ]; then\n"
	"  echo 1 > /proc/sys/net/ipv4/ip_always_defrag\nfi\n\n");
	
  	fprintf (script, "# Keep packet fragments in memory for 8 seconds\n");
   	fprintf (script, "# (Note - this option has no affect if you turn packet defragmentation\n");
   	fprintf (script, "#  (above) off!)\n\n");	
	fprintf (script, "if [ -e /proc/sys/net/ipv4/ipfrag_time ]; then\n"
	"  echo 8 > /proc/sys/net/ipv4/ipfrag_time\nfi\n\n");

  	fprintf (script, "# Do not reply to Address Mask Notification Warnings\n");
   	fprintf (script, "# (If you are using your machine as a DMZ router or a PPP dialin server\n");
   	fprintf (script, "#  that relies on proxy_arp requests to provide addresses to it's clients\n");
   	fprintf (script, "#  you may wish to disable this option by setting the value to '1'\n\n");
	fprintf (script, "if [ -e /proc/sys/net/ipv4/ip_addrmask_agent ]; then\n"
	"  echo 0 > /proc/sys/net/ipv4/ip_addrmask_agent\nfi\n\n");


	if (preferences_get_bool (PREFS_FW_EXT_PPP)) {
		fprintf (script, "# Turn on dynamic TCP/IP address hacking\n");
		fprintf (script, "# (Some broken PPPoE clients require this option to be enabled)\n");
		fprintf (script, "if [ -e /proc/sys/net/ipv4/ip_dynaddr ]; then\n"
		"  echo 1 > /proc/sys/net/ipv4/ip_dynaddr\nfi\n\n\n");
	} else {
		fprintf (script, "# Turn off dynamic TCP/IP address hacking\n");
		fprintf (script, "# (Some broken PPPoE clients have issues when this is disabled\n");
		fprintf (script, "#  If you experience problems with DSL or Cable providers, set this to '1')\n\n");		
		fprintf (script, "if [ -e /proc/sys/net/ipv4/ip_dynaddr ]; then\n"
		"  echo 0 > /proc/sys/net/ipv4/ip_dynaddr\nfi\n\n\n");        
	}
    
   fprintf (script, "# --------( Sysctl Tuning - IPTables Specific Parameters )--------\n\n");
   
	fprintf (script, "# Doubling current limit for ip_conntrack\n");
	fprintf (script, "if [ -e /proc/sys/net/ipv4/ip_conntrack_max ]; then\n"	
	"  echo 16384 > /proc/sys/net/ipv4/ip_conntrack_max\nfi\n\n");


   fprintf (script, "# --------( Rules Configuration - Specific Rule - Loopback Interfaces )--------\n\n");

	fprintf (script, "# Allow all traffic on the loopback interface\n");
	fprintf (script, "$IPT -t filter -A INPUT -i lo -s 0/0 -d 0/0 -j ACCEPT\n");
	fprintf (script, "$IPT -t filter -A OUTPUT -o lo -s 0/0 -d 0/0 -j ACCEPT\n\n");

/* TOS RULES */
if (preferences_get_bool (PREFS_FW_FILTER_TOS)) {

   fprintf (script, "# --------( Rules Configuration - Type of Service (ToS) - Ruleset Filtered by GUI )--------\n\n");

	if (preferences_get_bool (PREFS_FW_TOS_CLIENT)) {
		fprintf (script, "# ToS: Client Applications\n");
		fprintf (script, "$IPT -t mangle -A OUTPUT -p tcp -j TOS --dport 20:21 --set-tos $TOSOPT\n");
		fprintf (script, "$IPT -t mangle -A OUTPUT -p tcp -j TOS --dport 22 --set-tos $TOSOPT\n");
		fprintf (script, "$IPT -t mangle -A OUTPUT -p tcp -j TOS --dport 68 --set-tos $TOSOPT\n");
		fprintf (script, "$IPT -t mangle -A OUTPUT -p tcp -j TOS --dport 80 --set-tos $TOSOPT\n");
		fprintf (script, "$IPT -t mangle -A OUTPUT -p tcp -j TOS --dport 443 --set-tos $TOSOPT\n\n");
	} else if (preferences_get_bool (PREFS_FW_TOS_SERVER)) {
		fprintf (script, "# ToS: Server Applications\n");
		fprintf (script, "$IPT -t mangle -A OUTPUT -p tcp -j TOS --dport 20:21 --set-tos $TOSOPT\n");
		fprintf (script, "$IPT -t mangle -A OUTPUT -p tcp -j TOS --dport 22 --set-tos $TOSOPT\n");
		fprintf (script, "$IPT -t mangle -A OUTPUT -p tcp -j TOS --dport 25 --set-tos $TOSOPT\n");
		fprintf (script, "$IPT -t mangle -A OUTPUT -p tcp -j TOS --dport 53 --set-tos $TOSOPT\n");
		fprintf (script, "$IPT -t mangle -A OUTPUT -p tcp -j TOS --dport 67 --set-tos $TOSOPT\n");
		fprintf (script, "$IPT -t mangle -A OUTPUT -p tcp -j TOS --dport 80 --set-tos $TOSOPT\n");
		fprintf (script, "$IPT -t mangle -A OUTPUT -p tcp -j TOS --dport 110 --set-tos $TOSOPT\n");
		fprintf (script, "$IPT -t mangle -A OUTPUT -p tcp -j TOS --dport 143 --set-tos $TOSOPT\n");
		fprintf (script, "$IPT -t mangle -A OUTPUT -p tcp -j TOS --dport 443 --set-tos $TOSOPT\n");
		fprintf (script, "$IPT -t mangle -A OUTPUT -p tcp -j TOS --dport 1812 --set-tos $TOSOPT\n");
		fprintf (script, "$IPT -t mangle -A OUTPUT -p tcp -j TOS --dport 1813 --set-tos $TOSOPT\n");
		fprintf (script, "$IPT -t mangle -A OUTPUT -p tcp -j TOS --dport 2401 --set-tos $TOSOPT\n");
		fprintf (script, "$IPT -t mangle -A OUTPUT -p tcp -j TOS --dport 8080 --set-tos $TOSOPT\n\n");
	} else if (preferences_get_bool (PREFS_FW_TOS_X)) {
		fprintf (script, "# ToS: The X Window System\n");
		fprintf (script, "$IPT -t mangle -A OUTPUT -p tcp -j TOS --dport 22 --set-tos 0x10\n");
		fprintf (script, "$IPT -t mangle -A OUTPUT -p tcp -j TOS --dport 6000:6015 --set-tos 0x08\n\n");
	}		
}

/* ICMP RULES */

if (preferences_get_bool (PREFS_FW_FILTER_ICMP)) {

        fprintf (script, "\n# --------( Rules Configuration - ICMP - Ruleset Filtered by GUI )--------\n\n");
        
	if (preferences_get_bool (PREFS_FW_ICMP_ECHO)) {
		fprintf (script, "# ICMP: Ping Requests\n");
		fprintf (script, "$IPT -t filter -A INPUT -p icmp -s 0/0 -d $NET --icmp-type echo-request -j $STOP\n");
	} else {
		fprintf (script, "# ICMP: Ping Requests\n");
		fprintf (script, "$IPT -t filter -A INPUT -p icmp -s 0/0 -d $NET --icmp-type echo-request -m limit --limit 1/s -j ACCEPT\n");
		fprintf (script, "$IPT -t filter -A INPUT -p icmp -s 0/0 -d $NET --icmp-type echo-reply -m limit --limit 1/s -j ACCEPT\n");
	}

	if (preferences_get_bool (PREFS_FW_ICMP_TRACEROUTE)) {
		fprintf (script, "# ICMP: Traceroute Requests\n");
		fprintf (script, "$IPT -t filter -A INPUT -p udp -s 0/0 -d $NET --dport 33434 -j $STOP\n");
	} else {
		fprintf (script, "# ICMP: Traceroute Requests\n");
		fprintf (script, "$IPT -t filter -A INPUT -p udp -s 0/0 -d $NET --dport 33434 -j ACCEPT\n");
	}

	if (preferences_get_bool (PREFS_FW_ICMP_MSTRACEROUTE)) {
		fprintf (script, "# ICMP: MS Traceroute Requests\n");
		fprintf (script, "$IPT -t filter -A INPUT -p icmp -s 0/0 -d $NET --icmp-type destination-unreachable -j $STOP\n");
	} else {
		fprintf (script, "# ICMP: MS Traceroute Requests\n");
		fprintf (script, "$IPT -t filter -A INPUT -p icmp -s 0/0 -d $NET --icmp-type destination-unreachable -j ACCEPT\n");
	}

	if (preferences_get_bool (PREFS_FW_ICMP_UNREACHABLE)) {
		fprintf (script, "# ICMP: Unreachable Requests\n");
		fprintf (script, "$IPT -t filter -A INPUT -p icmp -s 0/0 -d $NET --icmp-type host-unreachable -j $STOP\n");
	} else {
		fprintf (script, "# ICMP: Unreachable Requests\n");
		fprintf (script, "$IPT -t filter -A INPUT -p icmp -s 0/0 -d $NET --icmp-type host-unreachable -j ACCEPT\n");
	}

	if (preferences_get_bool (PREFS_FW_ICMP_TIMESTAMPING)) {
		fprintf (script, "# ICMP: Timestamping Requests\n");
		fprintf (script, "$IPT -t filter -A INPUT -p icmp -s 0/0 -d $NET --icmp-type timestamp-request -j $STOP\n");
		fprintf (script, "$IPT -t filter -A INPUT -p icmp -s 0/0 -d $NET --icmp-type timestamp-reply -j $STOP\n");
	} else {
		fprintf (script, "# ICMP: Timestamping Requests\n");
		fprintf (script, "$IPT -t filter -A INPUT -p icmp -s 0/0 -d $NET --icmp-type timestamp-request -j ACCEPT\n");
		fprintf (script, "$IPT -t filter -A INPUT -p icmp -s 0/0 -d $NET --icmp-type timestamp-reply -j ACCEPT\n");
	}

	if (preferences_get_bool (PREFS_FW_ICMP_MASKING)) {
		fprintf (script, "# ICMP: Address Masking\n");
		fprintf (script, "$IPT -t filter -A INPUT -p icmp -s 0/0 -d $NET --icmp-type address-mask-request -j $STOP\n");
		fprintf (script, "$IPT -t filter -A INPUT -p icmp -s 0/0 -d $NET --icmp-type address-mask-reply -j $STOP\n");
	} else {
		fprintf (script, "# ICMP: Address Masking\n");
		fprintf (script, "$IPT -t filter -A INPUT -p icmp -s 0/0 -d $NET --icmp-type address-mask-request -j ACCEPT\n");
		fprintf (script, "$IPT -t filter -A INPUT -p icmp -s 0/0 -d $NET --icmp-type address-mask-reply -j ACCEPT\n");
	}

	if (preferences_get_bool (PREFS_FW_ICMP_REDIRECTION)) {
		fprintf (script, "# ICMP: Redirection Requests\n");
		fprintf (script, "$IPT -t filter -A INPUT -p icmp -s 0/0 -d $NET --icmp-type redirect -j $STOP\n");
	} else {
		fprintf (script, "# ICMP: Redirection Requests\n");
		fprintf (script, "$IPT -t filter -A INPUT -p icmp -s 0/0 -d $NET --icmp-type redirect -m limit --limit 2/s -j ACCEPT\n");
	}

	if (preferences_get_bool (PREFS_FW_ICMP_SOURCE_QUENCHES)) {
		fprintf (script, "# ICMP: Source Quench Requests\n");
		fprintf (script, "$IPT -t filter -A INPUT -p icmp -s 0/0 -d $NET --icmp-type source-quench -j $STOP\n");
	} else {
		fprintf (script, "# ICMP: Source Quench Requests\n");
		fprintf (script, "$IPT -t filter -A INPUT -p icmp -s 0/0 -d $NET --icmp-type source-quench -m limit --limit 2/s -j ACCEPT\n\n");
	}
} else {
    fprintf (script, "\n# --------( Rules Configuration - ICMP - Default Ruleset )--------\n\n");
	fprintf (script, "# Allowing all ICMP\n");
	fprintf (script, "$IPT -t filter -A INPUT -p icmp -s 0/0 -d $NET -m limit --limit 10/s -j ACCEPT\n\n");
}

/* NAT RULES */
	if (nat) {

		fprintf (script, "\n# --------( Rules Configuration - Masquerading )--------\n\n");
       
		fprintf (script, "# --------( Rules Configuration - Masquerading - Firewall Modules Autoloader )--------\n\n");

		fprintf (script, "if ! ( $LSM | /bin/grep ip_nat_ftp > /dev/null ); then\n");
		fprintf (script, "$MPB ip_nat_ftp\nfi\n\n");

		fprintf (script, "if ! ( $LSM | /bin/grep ip_nat_irc > /dev/null ); then\n");
		fprintf (script, "$MPB ip_nat_irc\nfi\n\n");
       
		fprintf (script, "# --------( Rules Configuration - Masquerading - Sysctl Modifications )--------\n\n");
   
		fprintf (script, "#Turn on IP forwarding\n");
		fprintf (script, "if [ -e /proc/sys/net/ipv4/ip_forward ]\n then\n"
	         "  echo 1 > /proc/sys/net/ipv4/ip_forward\nfi\n\n");

		fprintf (script, "# --------( Rules Configuration - Masquerading - Default Ruleset )--------\n\n");
      		
		fprintf (script, "#TCPMSS Fix - Needed for *many* broken PPPO{A/E} clients\n");
		fprintf (script, "$IPT -I FORWARD -p tcp --tcp-flags SYN,RST SYN -j TCPMSS --clamp-mss-to-pmtu\n\n");

		fprintf (script, "if ( $LSM | /bin/grep ipt_ipv4optsstrip > /dev/null ); then\n");
		fprintf (script, "#IPv4OPTIONS Fix - Strip IP options from a forwarded packet\n");
		fprintf (script, "$IPT -t mangle -A PREROUTING -j IPV4OPTSSTRIP\nfi\n\n");

		fprintf (script, "# --------( Rules Configuration - Forwarded Traffic - Block Traffic w/ Invalid Flags )--------\n\n");
		fprintf (script, "$IPT -t filter -A INPUT -m state --state INVALID -j $STOP\n\n");

		fprintf (script, "# --------( Rules Configuration - Forwarded Traffic - Block Traffic w/ Excessive Fragmented Packets )--------\n\n");
		fprintf (script, "$IPT -t filter -A INPUT -f -m limit --limit 10/minute -j $STOP\n\n");

		fprintf (script, "#Forward Int/Ext & Ext/Int Traffic before Masquerading\n");
		fprintf (script, "$IPT -t filter -A FORWARD -d 0/0 -s $INNET -o $IF -j ACCEPT\n");
		fprintf (script, "$IPT -t filter -A FORWARD -d $INNET -j ACCEPT\n");

		fprintf (script, "#Masquerade outgoing traffic\n");
		fprintf (script, "$IPT -t nat -A POSTROUTING -o $IF -j MASQUERADE\n\n");

		fprintf (script, "#Don't masq external interface traffic\n");
		fprintf (script, "$IPT -t nat -A POSTROUTING -s $NET -d 0/0 -j ACCEPT\n\n");

		fprintf (script, "#Allow traffic from internal network going anywhere\n");
		fprintf (script, "$IPT -t filter -A INPUT -s $INNET -d 0/0 -j ACCEPT\n");
		fprintf (script, "$IPT -t filter -A OUTPUT -s $INNET -d 0/0 -j ACCEPT\n");
		fprintf (script, "$IPT -t filter -A OUTPUT -p icmp -s $INNET -d 0/0 -j ACCEPT\n\n");
	}

   fprintf (script, "\n# --------( Rules Configuration - Inbound Traffic - Block nonroutable IP Addresses )--------\n\n");

	fprintf (script, "$IPT -N NR 2> /dev/null\n$IPT -F NR\n");
	fprintf (script, "while read block garbage\n\tdo\n");
	fprintf (script, "\t\t$IPT -A NR -s $block -d $NET -i $IF -j $STOP\n");
	fprintf (script, "\tdone < %s\n\n", FIRESTARTER_RULES_DIR "/firestarter/non-routables");
	fprintf (script, "$IPT -t filter -A INPUT -s ! $NET -i $IF -j NR\n");

   fprintf (script, "\n# --------( Rules Configuration - Inbound Traffic - Block known Trojan Ports )--------\n\n");
   
	fprintf (script, "#Block Back Orifice\n");
	fprintf (script, "$IPT -t filter -A INPUT -p tcp -s 0/0 -d $NET --dport 31337 -m limit --limit 2/minute -j $STOP\n");
	fprintf (script, "$IPT -t filter -A INPUT -p udp -s 0/0 -d $NET --dport 31337 -m limit --limit 2/minute -j $STOP\n\n");
	fprintf (script, "$IPT -t filter -A OUTPUT -p tcp -s $NET -d 0/0 --dport 31337 -m limit --limit 2/minute -j $STOP\n");
	fprintf (script, "$IPT -t filter -A OUTPUT -p udp -s $NET -d 0/0 --dport 31337 -m limit --limit 2/minute -j $STOP\n\n");
	fprintf (script, "#Block Trinity v3\n");
	fprintf (script, "$IPT -t filter -A INPUT -p tcp -s 0/0 -d $NET --dport 33270 -m limit --limit 2/minute -j $STOP\n");
	fprintf (script, "$IPT -t filter -A INPUT -p udp -s 0/0 -d $NET --dport 33270 -m limit --limit 2/minute -j $STOP\n\n");
	fprintf (script, "$IPT -t filter -A OUTPUT -p tcp -s $NET -d 0/0 --dport 33270 -m limit --limit 2/minute -j $STOP\n");
	fprintf (script, "$IPT -t filter -A OUTPUT -p udp -s $NET -d 0/0 --dport 33270 -m limit --limit 2/minute -j $STOP\n\n");
	fprintf (script, "#Block Subseven (1.7/1.9)\n");
	fprintf (script, "$IPT -t filter -A INPUT -p tcp -s 0/0 -d $NET --dport 1234 -m limit --limit 2/minute -j $STOP\n");
	fprintf (script, "$IPT -t filter -A INPUT -p tcp -s 0/0 -d $NET --dport 6711 -m limit --limit 2/minute -j $STOP\n\n");
	fprintf (script, "$IPT -t filter -A OUTPUT -p tcp -s $NET -d 0/0 --dport 1234 -m limit --limit 2/minute -j $STOP\n");
	fprintf (script, "$IPT -t filter -A OUTPUT -p tcp -s $NET -d 0/0 --dport 6711 -m limit --limit 2/minute -j $STOP\n\n");
	fprintf (script, "#Block Stacheldraht\n");
	fprintf (script, "$IPT -t filter -A INPUT -p tcp -s 0/0 -d $NET --dport 16660 --syn -m limit --limit 2/minute -j $STOP\n");
	fprintf (script, "$IPT -t filter -A INPUT -p tcp -s 0/0 -d $NET --dport 60001 --syn -m limit --limit 2/minute -j $STOP\n\n");
	fprintf (script, "$IPT -t filter -A OUTPUT -p tcp -s $NET -d 0/0 --dport 16660 --syn -m limit --limit 2/minute -j $STOP\n");
	fprintf (script, "$IPT -t filter -A OUTPUT -p tcp -s $NET -d 0/0 --dport 60001 --syn -m limit --limit 2/minute -j $STOP\n\n");
	fprintf (script, "#Block NetBus\n");
	fprintf (script, "$IPT -t filter -A INPUT -p tcp -s 0/0 -d $NET --dport 12345:12346 -m limit --limit 2/minute -j $STOP\n");
	fprintf (script, "$IPT -t filter -A INPUT -p udp -s 0/0 -d $NET --dport 12345:12346 -m limit --limit 2/minute -j $STOP\n\n");
	fprintf (script, "$IPT -t filter -A OUTPUT -p tcp -s $NET -d 0/0 --dport 12345:12346 -m limit --limit 2/minute -j $STOP\n");
	fprintf (script, "$IPT -t filter -A OUTPUT -p udp -s $NET -d 0/0 --dport 12345:12346 -m limit --limit 2/minute -j $STOP\n\n");
	fprintf (script, "#Block MS-RPC (dce)\n");
	fprintf (script, "$IPT -t filter -A INPUT -p tcp -s 0/0 -d $NET --dport 135 -m limit --limit 2/minute -j $STOP\n");
	fprintf (script, "$IPT -t filter -A INPUT -p udp -s 0/0 -d $NET --dport 135 -m limit --limit 2/minute -j $STOP\n\n");
	fprintf (script, "$IPT -t filter -A OUTPUT -p tcp -s $NET -d 0/0 --dport 135 -m limit --limit 2/minute -j $STOP\n");
	fprintf (script, "$IPT -t filter -A OUTPUT -p udp -s $NET -d 0/0 --dport 135 -m limit --limit 2/minute -j $STOP\n\n");
	fprintf (script, "#Block Trin00\n");
	fprintf (script, "$IPT -t filter -A INPUT -p tcp -s 0/0 -d $NET --dport 1524 -m limit --limit 2/minute -j $STOP\n");
	fprintf (script, "$IPT -t filter -A INPUT -p tcp -s 0/0 -d $NET --dport 27665 -m limit --limit 2/minute -j $STOP\n");
	fprintf (script, "$IPT -t filter -A INPUT -p udp -s 0/0 -d $NET --dport 27444 -m limit --limit 2/minute -j $STOP\n");
	fprintf (script, "$IPT -t filter -A INPUT -p udp -s 0/0 -d $NET --dport 31335 -m limit --limit 2/minute -j $STOP\n\n");
	fprintf (script, "$IPT -t filter -A OUTPUT -p tcp -s $NET -d 0/0 --dport 1524 -m limit --limit 2/minute -j $STOP\n");
	fprintf (script, "$IPT -t filter -A OUTPUT -p tcp -s $NET -d 0/0 --dport 27665 -m limit --limit 2/minute -j $STOP\n");
	fprintf (script, "$IPT -t filter -A OUTPUT -p udp -s $NET -d 0/0 --dport 27444 -m limit --limit 2/minute -j $STOP\n");
	fprintf (script, "$IPT -t filter -A OUTPUT -p udp -s $NET -d 0/0 --dport 31335 -m limit --limit 2/minute -j $STOP\n\n");

   fprintf (script, "\n# --------( Rules Configuration - Inbound Traffic - Block Multicast Traffic )--------\n\n");
   fprintf (script, "# (some cable/DSL providers require their clients to accept multicast transmissions\n");
   fprintf (script, "#  you should remove the following four rules if you are affected by multicasting\n");

	fprintf (script, "$IPT -t filter -A INPUT -s 224.0.0.0/8 -d 0/0 -j $STOP\n");
	fprintf (script, "$IPT -t filter -A INPUT -s 0/0 -d 224.0.0.0/8 -j $STOP\n");
	fprintf (script, "$IPT -t filter -A OUTPUT -s 224.0.0.0/8 -d 0/0 -j $STOP\n");
	fprintf (script, "$IPT -t filter -A OUTPUT -s 0/0 -d 224.0.0.0/8 -j $STOP\n\n");

   fprintf (script, "\n# --------( Rules Configuration - Inbound Traffic - Block Traffic w/ Stuffed Routing )--------\n\n");
   fprintf (script, "# (early versions of PUMP - (the DHCP client application included in RH / Mandrake) require\n");
   fprintf (script, "#  inbound packets to be accepted from a source address of 255.255.255.255.  If you have issues\n");
   fprintf (script, "#  with DHCP clients on your local LAN - either update PUMP, or remove the first rule below)\n");

	fprintf (script, "$IPT -t filter -A INPUT -s 255.255.255.255 -j $STOP\n");
	fprintf (script, "$IPT -t filter -A INPUT -d 0.0.0.0 -j $STOP\n");
	fprintf (script, "$IPT -t filter -A OUTPUT -s 255.255.255.255 -j $STOP\n");
	fprintf (script, "$IPT -t filter -A OUTPUT -d 0.0.0.0 -j $STOP\n\n");
	

    fprintf (script, "\n# --------( Rules Configuration - Inbound Traffic - Block Traffic w/ Invalid Flags )--------\n\n");
	fprintf (script, "$IPT -t filter -A INPUT -m state --state INVALID -j DROP\n\n");

    fprintf (script, "\n# --------( Rules Configuration - Inbound Traffic - Block Traffic w/ Excessive Fragmented Packets )--------\n\n");
	fprintf (script, "$IPT -t filter -A INPUT -f -m limit --limit 10/minute -j $STOP\n\n");

/* RULES BASED ON WIZARD SELECTIONS */

    fprintf (script, "\n# --------( Rules Configuration - Inbound Traffic - Ruleset Filtered by GUI )--------\n\n");

	/* Allow DHCP */
	if (preferences_get_bool (PREFS_FW_EXT_DHCP)) {
		fprintf (script, "#DHCP\n");
		fprintf (script, "$IPT -t filter -A INPUT -p tcp -s 0/0 -d 0/0 --dport 67:68 -i $IF -j ACCEPT\n\n");
		fprintf (script, "$IPT -t filter -A INPUT -p udp -s 0/0 -d 0/0 --dport 67:68 -i $IF -j ACCEPT\n\n");
	}

if (preferences_get_bool (PREFS_FW_SERVICES_ENABLE)) {

	if (preferences_get_bool (PREFS_FW_SERVICES_FTP)) {
		fprintf (script, "#FTP\n");
		fprintf (script, "$IPT -t filter -A INPUT -p tcp -s 0/0 -d $NET --dport 20  ! --syn -j ACCEPT\n");
		fprintf (script, "$IPT -t filter -A INPUT -p tcp -s 0/0 -d $NET --dport 21 -j ACCEPT\n\n");
	}
	if (preferences_get_bool (PREFS_FW_SERVICES_SSH)) {
		fprintf (script, "#SSH\n");		
		fprintf (script, "$IPT -t filter -A INPUT -p tcp -s 0/0 -d $NET --dport 22 -j ACCEPT\n\n");
	}
	if (preferences_get_bool (PREFS_FW_SERVICES_TELNET)) {
		fprintf (script, "#Telnet\n");
		fprintf (script, "$IPT -t filter -A INPUT -p tcp -s 0/0 -d $NET --dport 23 -j ACCEPT\n\n");
	}
	if (preferences_get_bool (PREFS_FW_SERVICES_SMTP)) {
		fprintf (script, "#SMTP\n");
		fprintf (script, "$IPT -t filter -A INPUT -p tcp -s 0/0 -d $NET --dport 25 -j ACCEPT\n\n");
	}
	if (preferences_get_bool (PREFS_FW_SERVICES_DNS)) {
		fprintf (script, "#DNS\n");
		fprintf (script, "$IPT -t filter -A INPUT -p tcp -s 0/0 -d $NET --dport 53 -j ACCEPT\n");
		fprintf (script, "$IPT -t filter -A INPUT -p udp -s 0/0 -d $NET --dport 53 -j ACCEPT\n\n");
	}
	if (preferences_get_bool (PREFS_FW_SERVICES_FINGER)) {
		fprintf (script, "#Finger\n");
		fprintf (script, "$IPT -t filter -A INPUT -p tcp -s 0/0 -d $NET --dport 79 -j ACCEPT\n\n");
	}
	if (preferences_get_bool (PREFS_FW_SERVICES_WWW)) {
		fprintf (script, "#HTTP\n");
		fprintf (script, "$IPT -t filter -A INPUT -p tcp -s 0/0 -d $NET --dport 80 -j ACCEPT\n\n");
	}
	if (preferences_get_bool (PREFS_FW_SERVICES_SSLWEB)) {
		fprintf (script, "#SSL HTTP\n");
		fprintf (script, "$IPT -t filter -A INPUT -p tcp -s 0/0 -d $NET --dport 443 -j ACCEPT\n\n");
	}
	if (preferences_get_bool (PREFS_FW_SERVICES_POP)) {
		fprintf (script, "#POP\n");
		fprintf (script, "$IPT -t filter -A INPUT -p tcp -s 0/0 -d $NET --dport 110 -j ACCEPT\n\n");
	}
	if (preferences_get_bool (PREFS_FW_SERVICES_IMAP)) {
		fprintf (script, "#IMAP\n");
		fprintf (script, "$IPT -t filter -A INPUT -p tcp -s 0/0 -d $NET --dport 143 -j ACCEPT\n\n");
	}
	if (preferences_get_bool (PREFS_FW_SERVICES_IDENT)) {
		fprintf (script, "#IDENT\n");
		fprintf (script, "$IPT -t filter -A INPUT -p tcp -s 0/0 -d $NET --dport 113 -j ACCEPT\n");
		fprintf (script, "$IPT -t filter -A INPUT -p udp -s 0/0 -d $NET --dport 113 -j ACCEPT\n\n");
	}
	if (preferences_get_bool (PREFS_FW_SERVICES_NNTP)) {
		fprintf (script, "#NNTP\n");
		fprintf (script, "$IPT -t filter -A INPUT -p tcp -s 0/0 -d $NET --dport 119 -j ACCEPT\n\n");
	}
	if (preferences_get_bool (PREFS_FW_SERVICES_NTP)) {
		fprintf (script, "#NTP\n");		
		fprintf (script, "$IPT -t filter -A INPUT -p tcp -s 0/0 -d $NET --dport 123 -j ACCEPT\n");
		fprintf (script, "$IPT -t filter -A INPUT -p udp -s 0/0 -d $NET --dport 123 -j ACCEPT\n\n");
	}
	if (preferences_get_bool (PREFS_FW_SERVICES_SAMBA)) {
		fprintf (script, "#SAMBA\n");
		fprintf (script, "$IPT -t filter -A INPUT -p tcp -s 0/0 -d 0/0 --dport 137:139 -i $IF -j ACCEPT\n");
		fprintf (script, "$IPT -t filter -A INPUT -p udp -s 0/0 -d 0/0 --dport 137:139 -i $IF -j ACCEPT\n");
		fprintf (script, "$IPT -t filter -A INPUT -p tcp -s 0/0 -d 0/0 --dport 445 -i $IF -j ACCEPT\n");
		fprintf (script, "$IPT -t filter -A INPUT -p udp -s 0/0 -d 0/0 --dport 445 -i $IF -j ACCEPT\n\n");
	}
	if (preferences_get_bool (PREFS_FW_SERVICES_IPSEC)) {
		fprintf (script, "#IPSec / KLIPS\n");
		fprintf (script, "$IPT -t filter -A INPUT -p udp -s 0/0 -d 0/0 --dport 500 -j ACCEPT\n");
		fprintf (script, "$IPT -t filter -A INPUT -p tcp -s 0/0 -d 0/0 --dport 500 -j ACCEPT\n");
		fprintf (script, "$IPT -t filter -A INPUT -p 50 -s 0/0 -d 0/0 -j ACCEPT\n\n");
		fprintf (script, "$IPT -t filter -A INPUT -p 51 -s 0/0 -d 0/0 -j ACCEPT\n\n");
	}
	if (preferences_get_bool (PREFS_FW_SERVICES_ROUTED)) {
		fprintf (script, "#Routed\n");
		fprintf (script, "$IPT -t filter -A INPUT -p udp -s 0/0 -d 0/0 --dport 520 -j ACCEPT\n\n");
	}
	if (preferences_get_bool (PREFS_FW_SERVICES_NFS)) {
		fprintf (script, "#NFS\n");
		fprintf (script, "$IPT -t filter -A INPUT -p tcp -s 0/0 -d 0/0 --dport 2049 -i $IF -j ACCEPT\n");
		fprintf (script, "$IPT -t filter -A INPUT -p udp -s 0/0 -d 0/0 --dport 2049 -i $IF -j ACCEPT\n\n");
	}
	if (preferences_get_bool (PREFS_FW_SERVICES_X)) {
		fprintf (script, "#Xwindows\n");
		fprintf (script, "$IPT -t filter -A INPUT -p tcp -s 0/0 -d 0/0 --dport 6000:6015 -i $IF -j ACCEPT\n");
		fprintf (script, "$IPT -t filter -A INPUT -p udp -s 0/0 -d 0/0 --dport 6000:6015 -i $IF -j ACCEPT\n\n");
	}
	if (preferences_get_bool (PREFS_FW_SERVICES_DHCP)) {
		fprintf (script, "#DHCP\n");
		fprintf (script, "$IPT -t filter -A INPUT -p tcp -s 0/0 -d 0/0 --dport 67:68 -j ACCEPT\n");
		fprintf (script, "$IPT -t filter -A INPUT -p udp -s 0/0 -d 0/0 --dport 67:68 -j ACCEPT\n\n");
	}
	if (preferences_get_bool (PREFS_FW_SERVICES_UPNP)) {
		fprintf (script, "#uPNP\n");
		fprintf (script, "$IPT -t filter -A INPUT -p tcp -s 0/0 -d 0/0 --dport 5000 -i $IF -j ACCEPT\n");
		fprintf (script, "$IPT -t filter -A INPUT -p udp -s 0/0 -d 0/0 --dport 5000 -i $IF -j ACCEPT\n\n");
	}
	if (preferences_get_bool (PREFS_FW_SERVICES_PPTP)) {
		fprintf (script, "#PPTP\n");
		fprintf (script, "$IPT -t filter -A INPUT -p 47 -j ACCEPT \n");
		fprintf (script, "$IPT -t filter -A INPUT -p tcp -s 0/0 -d $NET --dport 1723 -j ACCEPT \n");
	}
}

   fprintf (script, "\n# --------( Rules Configuration - Inbound Traffic - Highport Connection Fixes )--------\n\n");
	fprintf (script, "$IPT -A INPUT -p tcp ! --syn -m state --state NEW -j $STOP\n\n");

	if (preferences_get_bool (PREFS_FW_SERVICES_SSH)) {
		fprintf (script, "#SSH fix\n");
		fprintf (script, "$IPT  -A INPUT -p tcp --sport 22 --dport 513:65535 ! --syn -m state --state RELATED -j ACCEPT\n\n");
	}
	if (preferences_get_bool (PREFS_FW_SERVICES_FTP)) {
		fprintf (script, "#FTP Data fix\n");
		fprintf (script, "$IPT  -A INPUT -p tcp --sport 20 --dport 1023:65535 ! --syn -m state --state RELATED -j ACCEPT\n\n");
	}

   fprintf (script, "\n# --------( Rules Configuration - Inbound Traffic - Highport Connections )--------\n\n");
	fprintf (script, "$IPT  -A INPUT -p tcp -s 0/0 -d $NET --dport 1024:65535 -j STATE\n");
	fprintf (script, "$IPT  -A INPUT -p udp -s 0/0 -d $NET --dport 1023:65535 -j ACCEPT\n\n");

   fprintf (script, "\n# --------( Rules Configuration - Outbound Traffic - Highport Connection Fixes )--------\n\n");
	fprintf (script, "$IPT -A OUTPUT -p tcp ! --syn -m state --state NEW -j DROP\n\n");

   fprintf (script, "\n# --------( Rules Configuration - Outbound Traffic - Block Traffic w/ Invalid Flags )--------\n\n");
	fprintf (script, "$IPT -A OUTPUT -m state --state INVALID -j DROP\n");

   fprintf (script, "\n# --------( Rules Configuration - Outbound Traffic - TTL Mangling )--------\n\n");
	fprintf (script, "$IPT -A OUTPUT -m ttl --ttl 64\n\n");

   fprintf (script, "\n# --------( Rules Configuration - Outbound Traffic - Default Ruleset )--------\n\n");
	fprintf (script, "$IPT -A OUTPUT -p icmp -s $NET -d 0/0 -o $IF -j ACCEPT\n");
	fprintf (script, "$IPT -A OUTPUT -j ACCEPT\n\n");

/* REQUIRED RULES */

   fprintf (script, "\n# --------( Catch all Rules (required) )--------\n\n");
	fprintf (script, "# Deny everything not let through earlier\n");
	fprintf (script, "$IPT -A INPUT -j $STOP\n\n");

	fprintf (script, "# --------( User Defined Post Script )--------\n\n");
	fprintf (script, "sh "FIRESTARTER_RULES_DIR"/firestarter/user-post\n\n");

	fprintf (script, "# Create Firestarter lock file\n");
	fprintf (script, "if [ -e /var/lock/subsys ]; then\n"
	"  touch /var/lock/subsys/firestarter\nelse\n"
	"  touch /var/lock/firestarter\n"
	"fi\n\n");

	fclose (script);

	g_print (_("Firewall script saved as %s\n"), scriptpath);
}
