/*---[ hitview.c ]-----------------------------------------------------
 * Copyright (C) 2002 Tomas Junnonen (majix@sci.fi)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * The Hits page and related functions
 *--------------------------------------------------------------------*/

#include <config.h>
#include <gnome.h>

#include "firestarter.h"
#include "globals.h"
#include "ruleview.h"
#include "util.h"
#include "modrules.h"
#include "menus.h"
#include "service.h"
#include "parse.h"
#include "preferences.h"

GtkTreeIter *rulegroups[NUM_RULETYPES];
GtkTreeStore *rulestore;

GtkTreeIter *
get_rulegroup_iter (RuleType type)
{
	g_assert (type < NUM_RULETYPES);
	
	return rulegroups[type];
}

gboolean
is_root_item (GtkTreeIter iter)
{
	return 	(gtk_tree_store_iter_depth (rulestore, &iter) == 0);

}

void
ruleview_remove_rule (GtkTreeIter iter)
{
	gtk_tree_store_remove (rulestore, &iter);	
}

/* [ ruleview_insert_rule ]
 * Append a rule to the ruletree
 */
GtkTreeIter *
ruleview_insert_rule (Rule *r)
{
	GtkTreeIter iter;
	gchar *data; /* Visible information about a rule */
	gchar *sname;

	if (r->type == RULETYPE_GROUP) {
		gtk_tree_store_append (rulestore, &iter, NULL);
	} else {
		/* Normal rules are always children of one of the rule groups */
		gtk_tree_store_append (rulestore, &iter, rulegroups[r->type]);
	}

	switch (r->type) {
		case RULETYPE_GROUP:
		case RULETYPE_TRUSTED_HOST: 
		case RULETYPE_BLOCKED_HOST:
			data = r->host;
			break;

		case RULETYPE_OPEN_PORT: 
		case RULETYPE_BLOCKED_PORT:
			sname = get_service_name_any_protocol(atoi (r->port));
		
			data = g_strconcat (
				r->port,
				" (",
				sname,
				")",
				NULL);

			break;

		case RULETYPE_STEALTHED_PORT:
			sname = get_service_name_any_protocol(atoi (r->port));

			data = g_strdup_printf (_("%s (%s) is open to %s"), r->port, sname, r->host);

			break;


		case RULETYPE_FORWARD: {
			Forward *f;
			
			f = (Forward *)r->extension;

			data = g_strdup_printf (_("From port %s to %s, port %s"), r->port, f->int_host, f->int_port);

			gtk_tree_store_set (rulestore, &iter,
		        	            RULECOL_EXTRA1, f->int_host,
					    RULECOL_EXTRA2, f->int_port,
					    -1);

			break;
		}
		default: data = r->host; break;
	}
	
	gtk_tree_store_set (rulestore, &iter,
	                    RULECOL_ACTIVE, r->active,
			    RULECOL_DATA, data,
	                    RULECOL_TYPE, r->type,
			    RULECOL_HOST, r->host,
			    RULECOL_PORT, r->port,
			    -1);

	return gtk_tree_iter_copy(&iter);
}

/* [ create_ruletree_model ]
 * Creates the tree for editing of rules
 */
static GtkTreeModel *
create_ruletree_model (void)
{
	Rule groups[] =
	{
		{ RULETYPE_GROUP, _("Trusted Hosts"),  "1", FALSE, NULL },
		{ RULETYPE_GROUP, _("Blocked Hosts"),  "2", FALSE, NULL },
		{ RULETYPE_GROUP, _("Forwarded Ports"),"3", FALSE, NULL },
		{ RULETYPE_GROUP, _("Open Ports"),     "4", FALSE, NULL },
		{ RULETYPE_GROUP, _("Stealthed Ports"),"5", FALSE, NULL },
		{ RULETYPE_GROUP, _("Blocked Ports"),  "6", FALSE, NULL },
	};

	/* create list store */
	rulestore = gtk_tree_store_new (7,
		G_TYPE_BOOLEAN, // Active
		G_TYPE_STRING,  // Visible Data
		G_TYPE_INT,     // Type
		G_TYPE_STRING,  // Host
		G_TYPE_STRING,  // Port
		G_TYPE_STRING,  // Extra data 1
		G_TYPE_STRING   // Extra data 2
		);

	rulegroups[RULETYPE_TRUSTED_HOST]   = ruleview_insert_rule (&groups[0]);
	rulegroups[RULETYPE_BLOCKED_HOST]   = ruleview_insert_rule (&groups[1]);
	if (is_capable_of_nat ())
		rulegroups[RULETYPE_FORWARD]= ruleview_insert_rule (&groups[2]);
	rulegroups[RULETYPE_OPEN_PORT]      = ruleview_insert_rule (&groups[3]);
	rulegroups[RULETYPE_STEALTHED_PORT] = ruleview_insert_rule (&groups[4]);
	rulegroups[RULETYPE_BLOCKED_PORT]   = ruleview_insert_rule (&groups[5]);

	/* Don't parse rule files at first run */
	if (!preferences_get_bool (PREFS_FIRST_RUN))
		parse_all_rules ();

	return GTK_TREE_MODEL (rulestore);
}

/* [ create_text_column ]
 * Convinience funtion for creating a text column for a treeview
 */
static GtkTreeViewColumn *
create_text_column (RuleColumn cnum, gchar *title)
{
	GtkTreeViewColumn *column;
	GtkCellRenderer *renderer;

	renderer = gtk_cell_renderer_text_new ();
	column = gtk_tree_view_column_new_with_attributes (title,
		renderer, "text", cnum, NULL);

	return column;
}

/* [ add_ruleview_columns ]
 * Add the columns to the rules TreeView
 */
static void
add_ruleview_columns (GtkTreeView *treeview)
{
	GtkTreeViewColumn *column;
	GtkCellRenderer *renderer;
	
	renderer = gtk_cell_renderer_toggle_new ();

	/* Column for visible rule data */
	column = create_text_column (RULECOL_DATA, _("Rule"));
	gtk_tree_view_column_set_min_width (column, 250);
	gtk_tree_view_append_column (treeview, column);

	/* column for active toggle */
	column = gtk_tree_view_column_new_with_attributes (_("Active"),
		renderer, NULL);
	gtk_tree_view_column_set_alignment (column, 0.0);
	gtk_tree_view_append_column (treeview, column);
	gtk_tree_view_column_set_visible(column, FALSE);

	/* Columns for rule reconstruction, not necessarily visible */
	column = create_text_column (RULECOL_TYPE, _("Type"));
	gtk_tree_view_append_column (treeview, column);
	gtk_tree_view_column_set_visible(column, FALSE);

	column = create_text_column (RULECOL_HOST, _("Host"));
	gtk_tree_view_append_column (treeview, column);
	gtk_tree_view_column_set_visible(column, FALSE);

	column = create_text_column (RULECOL_PORT, _("Port"));
	gtk_tree_view_append_column (treeview, column);
	gtk_tree_view_column_set_visible(column, FALSE);

	gtk_tree_view_set_headers_visible (treeview, FALSE);

}

/* [ get_rule ]
 * Retrieve the specific rule iter points to
 */
Rule *
get_rule (GtkTreeModel *model,
          GtkTreeIter iter)
{
	Rule *r = g_new (Rule, 1);

	gtk_tree_model_get (model, &iter,
	                    RULECOL_TYPE,     &r->type,
	                    RULECOL_HOST,     &r->host,
			    RULECOL_PORT,     &r->port,
			    RULECOL_ACTIVE,   &r->active,
	                    -1);

	if (r->type == RULETYPE_FORWARD) {
		Forward *f = g_new (Forward, 1);

		gtk_tree_model_get (model, &iter,
		                    RULECOL_EXTRA1, &f->int_host,
		                    RULECOL_EXTRA2, &f->int_port,
		                    -1);

		r->extension = f;
	}

	return r;
}

static void
activate_dialog_cb (GtkWidget *dialog)
{
	gtk_dialog_response (GTK_DIALOG (dialog), GTK_RESPONSE_ACCEPT);
}

/* [ rule_edit_response_cb ]
 * Insert new rule when editing complete and remove selected (old) rule
 */
static void
rule_edit_response_cb (GtkDialog *dialog,
                       gint response_id,
                       Rule *r)
{
	GtkEntry *entry;
	const gchar *hostname = NULL;
	const gchar *port = NULL;
	const gchar *int_host = NULL;
	const gchar *int_port = NULL;
	
	if (response_id == GTK_RESPONSE_HELP) {
		gnome_url_show ("http://firestarter.sourceforge.net/manual/rules.php#creating", NULL);
		return;
	}

	if (response_id == GTK_RESPONSE_ACCEPT) {
		entry = (GtkEntry *)g_object_get_data (G_OBJECT (dialog), "entry1");

		if (r->type == RULETYPE_TRUSTED_HOST ||
		    r->type == RULETYPE_BLOCKED_HOST) {
			hostname = gtk_entry_get_text (entry);
			r->host = g_strdup (hostname);
		}
		if (r->type == RULETYPE_FORWARD ||
		    r->type == RULETYPE_OPEN_PORT ||
		    r->type == RULETYPE_STEALTHED_PORT ||
		    r->type == RULETYPE_BLOCKED_PORT) {
			port = gtk_entry_get_text (entry);
			r->port = g_strdup (port);
		}
		if (r->type == RULETYPE_STEALTHED_PORT) {
			entry = (GtkEntry *)g_object_get_data (G_OBJECT (dialog), "entry2");

			hostname = gtk_entry_get_text (entry);
			r->host = g_strdup (hostname);
		}
		if (r->type == RULETYPE_FORWARD) {
			Forward *f;

			entry = (GtkEntry *)g_object_get_data (G_OBJECT (dialog), "entry2");
		
			f = (Forward *)r->extension;

			int_host = gtk_entry_get_text (entry);
			f->int_host = g_strdup (int_host);

			entry = (GtkEntry *)g_object_get_data (G_OBJECT (dialog), "entry3");

			int_port = gtk_entry_get_text (entry);
			f->int_port = g_strdup (int_port);
		}

			if (hostname != NULL && !is_a_valid_host (hostname)) {
				error_dialog (_("Please specify a valid hostname or IP address."));
				return;
			}

			if (port != NULL && !is_a_valid_port (port)) {
				error_dialog (_("Please specify a valid port number."));
				return;
			}


			if (int_host != NULL && !is_a_valid_host (int_host)) {
				error_dialog (_("Please specify a valid internal hostname or IP address."));
				return;
			}


			if (int_port != NULL && !is_a_valid_port (int_port)) {
				error_dialog (_("Please specify a valid port number for the internal host."));
				return;
			}

		/* TODO: In place edit of rule? Now we restart the fw twice */
		remove_rule ();
		append_rule (r);
	}

	gtk_widget_destroy (GTK_WIDGET (dialog));
	g_free (r);
}

/* [ create_rule_dialog ]
 * Create a dialog for editing a rule
 */
GtkWidget *
create_rule_dialog (Rule *r, gchar *title)
{
	GtkWidget *dialog;
	GtkWidget *label;
	GtkWidget *entry;
	GtkWidget *vbox;
	GtkWidget *hbox;
	GtkWidget *catbox;
	GtkWidget *rowbox;
	GtkSizeGroup *sizegroup;
	gchar *label1 = NULL;
	gchar *label2 = NULL;
	gchar *text1 = NULL;
	gchar *text2 = NULL;
	gchar *text3 = NULL;
	gchar *data1 = NULL;
	gchar *data2 = NULL;
	gchar *data3 = NULL;
	
	sizegroup = gtk_size_group_new (3); // FIXME: the mode enum is missing?

	dialog = gtk_dialog_new_with_buttons (title,
	                                      GTK_WINDOW (Firestarter.window),
	                                      GTK_DIALOG_DESTROY_WITH_PARENT | GTK_DIALOG_NO_SEPARATOR,
	                                      GTK_STOCK_HELP,
					      GTK_RESPONSE_HELP,
	                                      GTK_STOCK_CANCEL,
	                                      GTK_RESPONSE_REJECT,
	                                      GTK_STOCK_OK,
	                                      GTK_RESPONSE_ACCEPT,
	                                      NULL);

	gtk_container_set_border_width (GTK_CONTAINER (dialog), 8);	

	g_signal_connect (G_OBJECT (dialog), "response", 
	                  G_CALLBACK (rule_edit_response_cb), r);

	switch (r->type) {
	  case RULETYPE_TRUSTED_HOST:
	  	label1 = g_strdup (_("Trust Host"));
	  	text1 = g_strdup (_("IP or hostname:"));
		data1 = r->host;
		break;
	  case RULETYPE_BLOCKED_HOST:
	  	label1 = g_strdup (_("Block Host"));
	  	text1 = g_strdup (_("IP or hostname:"));
		data1 = r->host;
		break;
	  case RULETYPE_FORWARD: {
		Forward *f = (Forward *)r->extension;
  
		label1 = g_strdup (_("Forward From"));
		text1 = g_strdup (_("Firewall port:"));
		data1 = r->port;
		label2 = g_strdup (_("Forward To"));
	 	text2 = g_strdup (_("IP or hostname:"));
		data2 = f->int_host;
	 	text3 = g_strdup (_("Port:"));
		data3 = f->int_port;
		break;
	  }
	  case RULETYPE_OPEN_PORT:
	  	label1 = g_strdup (_("Open Port"));
	  	text1 = g_strdup (_("Port:"));
		data1 = r->port;
		break;
	  case RULETYPE_STEALTHED_PORT:
	  	label1 = g_strdup (_("Stealth Port"));
	  	text1 = g_strdup (_("Port:"));
		data1 = r->port;
		label2 = g_strdup (_("For Machine"));
	  	text2 = g_strdup (_("IP or hostname:"));
		data2 = r->host;
		break;
	  case RULETYPE_BLOCKED_PORT: 
	  	label1 = g_strdup (_("Block Port"));
	  	text1 = g_strdup (_("Port:"));
		data1 = r->port;
		break;
	}

	/* HIG test */

	/* Control Group 1 */
	vbox = gtk_vbox_new (FALSE, 14);
	gtk_box_pack_start  (GTK_BOX (GTK_DIALOG (dialog)->vbox), vbox, FALSE, FALSE, 8);
	catbox = gtk_vbox_new (FALSE, 6);
	gtk_box_pack_start (GTK_BOX (vbox), catbox, FALSE, FALSE, 0);

	label = gtk_label_new (NULL);
	gtk_label_set_markup (GTK_LABEL (label), g_strconcat (
		"<span weight=\"bold\">", label1, "</span>", NULL));
	gtk_misc_set_alignment (GTK_MISC (label), 0, 0.0);
	gtk_box_pack_start (GTK_BOX (catbox), label, FALSE, FALSE, 0);

	hbox = gtk_hbox_new (FALSE, 0);
	gtk_box_pack_start (GTK_BOX (catbox), hbox, FALSE, FALSE, 0);

	label = gtk_label_new ("    ");
	gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 0);

	vbox = gtk_vbox_new (FALSE, 6);
	gtk_box_pack_start (GTK_BOX (hbox), vbox, FALSE, FALSE, 0);

	rowbox = gtk_hbox_new (FALSE, 6);
	gtk_box_pack_start (GTK_BOX (vbox), rowbox, FALSE, FALSE, 0);

	label = gtk_label_new (text1);
	gtk_misc_set_alignment (GTK_MISC (label), 0, 0.0);
	gtk_size_group_add_widget (sizegroup, label);
	gtk_box_pack_start (GTK_BOX (rowbox), label, FALSE, FALSE, 0);

	entry = gtk_entry_new ();
	gtk_entry_set_text (GTK_ENTRY (entry), data1);
	gtk_widget_set_size_request (entry, 150, -1);
	g_signal_connect_swapped (G_OBJECT (entry), "activate",
	                        G_CALLBACK (activate_dialog_cb), G_OBJECT (dialog));
	gtk_box_pack_start (GTK_BOX (rowbox), entry, FALSE, FALSE, 0);

	g_object_set_data (G_OBJECT (dialog), "entry1", entry);

	if (label2 != NULL) {
		/* Control Group 2 */
		vbox = gtk_vbox_new (FALSE, 18);
		gtk_box_pack_start  (GTK_BOX (GTK_DIALOG (dialog)->vbox), vbox, FALSE, FALSE, 8);
		catbox = gtk_vbox_new (FALSE, 6);
		gtk_box_pack_start (GTK_BOX (vbox), catbox, FALSE, FALSE, 0);

		label = gtk_label_new (NULL);
		gtk_label_set_markup (GTK_LABEL (label), g_strconcat (
			"<span weight=\"bold\">", label2, "</span>", NULL));
		gtk_misc_set_alignment (GTK_MISC (label), 0, 0.0);
		gtk_box_pack_start (GTK_BOX (catbox), label, FALSE, FALSE, 0);

		hbox = gtk_hbox_new (FALSE, 0);
		gtk_box_pack_start (GTK_BOX (catbox), hbox, FALSE, FALSE, 0);

		label = gtk_label_new ("    ");
		gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 0);

		vbox = gtk_vbox_new (FALSE, 6);
		gtk_box_pack_start (GTK_BOX (hbox), vbox, FALSE, FALSE, 0);

		rowbox = gtk_hbox_new (FALSE, 6);
		gtk_box_pack_start (GTK_BOX (vbox), rowbox, FALSE, FALSE, 0);

		label = gtk_label_new (text2);
		gtk_misc_set_alignment (GTK_MISC (label), 0, 0.0);
		gtk_size_group_add_widget (sizegroup, label);
		gtk_box_pack_start (GTK_BOX (rowbox), label, FALSE, FALSE, 0);

		entry = gtk_entry_new ();
		gtk_entry_set_text (GTK_ENTRY (entry), data2);
		gtk_widget_set_size_request (entry, 150, -1);
		g_signal_connect_swapped (G_OBJECT (entry), "activate",
	                        G_CALLBACK (activate_dialog_cb), G_OBJECT (dialog));
		gtk_box_pack_start (GTK_BOX (rowbox), entry, FALSE, FALSE, 0);

		g_object_set_data (G_OBJECT (dialog), "entry2", entry);

		/* Extra entry for forwarding */
		if (text3 != NULL) {
			hbox = gtk_hbox_new (FALSE, 0);
			gtk_box_pack_start (GTK_BOX (catbox), hbox, FALSE, FALSE, 0);

			label = gtk_label_new ("    ");
			gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 0);

			vbox = gtk_vbox_new (FALSE, 6);
			gtk_box_pack_start (GTK_BOX (hbox), vbox, FALSE, FALSE, 0);

			rowbox = gtk_hbox_new (FALSE, 6);
			gtk_box_pack_start (GTK_BOX (vbox), rowbox, FALSE, FALSE, 0);

			label = gtk_label_new (text3);
			gtk_misc_set_alignment (GTK_MISC (label), 0, 0.0);
			gtk_size_group_add_widget (sizegroup, label);
			gtk_box_pack_start (GTK_BOX (rowbox), label, FALSE, FALSE, 0);

			entry = gtk_entry_new ();
			gtk_entry_set_text (GTK_ENTRY (entry), data3);
			gtk_widget_set_size_request (entry, 150, -1);
			g_signal_connect_swapped (G_OBJECT (entry), "activate",
		                        G_CALLBACK (activate_dialog_cb), G_OBJECT (dialog));
			gtk_box_pack_start (GTK_BOX (rowbox), entry, FALSE, FALSE, 0);

			g_object_set_data (G_OBJECT (dialog), "entry3", entry);
		}
	}

	return dialog;
}

/* [ rule_selected_cb ]
 * Callback for selecting a rule in the rules view
 * Default actions: if rule is user created, edit it, else
 * give option to add a new rule to the group in question
 */
static void 
rule_selected_cb (GtkTreeView *treeview,
                  GtkTreePath *path,
                  GtkTreeViewColumn *arg2,
                  gpointer data)
{
	Rule *r;

	r = ruleview_get_selected_rule ();

	if (r->type == RULETYPE_GROUP)
		create_new_rule ();
	else
		edit_selected_rule ();
}

/* [ ruleview_get_selected_rule ]
 * Get the rule that is currently selected in the ruleview
 */
Rule *
ruleview_get_selected_rule (void)
{
	GtkTreeSelection *selection;
	GtkTreeModel *model;
	GtkTreeIter iter;
	Rule *r = NULL;
	gboolean has_selected;

	selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (Firestarter.ruleview));

	has_selected = gtk_tree_selection_get_selected (selection,
	                                                NULL,
	                                                &iter);
	if (has_selected) {
		model = gtk_tree_view_get_model (GTK_TREE_VIEW (Firestarter.ruleview));
		r = get_rule (model, iter);
	}

	return r;
}

/* [ ruleview_button_press_cb ]
 * Pop up an menu when right clicking the ruleview
 */
static gboolean
ruleview_button_press_cb (GtkWidget* widget, GdkEventButton* bevent)
{
	gboolean retval = FALSE;

	switch (bevent->button) {
		case 3: ruleview_popup_menu (bevent);
			retval = TRUE;
			break;

	}

	return retval;
}

/* [ copy_selected_rule ]
 * Copy the selected rule to the clipboard
 */
void
copy_selected_rule (void)
{
	Rule *r;
	gchar *text;
	GtkClipboard *cb;

	r = ruleview_get_selected_rule ();

	if (r == NULL)
		return;

	cb = gtk_clipboard_get (GDK_SELECTION_CLIPBOARD);

	switch (r->type) {
	  case RULETYPE_TRUSTED_HOST: text = g_strconcat ("Trust ", r->host, NULL); break;
	  case RULETYPE_BLOCKED_HOST: text = g_strconcat ("Block ", r->host, NULL); break;
	  case RULETYPE_FORWARD: {
		Forward *f;

		f = (Forward *)r->extension;
		text = g_strconcat ("Forward port ", r->port,
			            " to ", f->int_host, " port ", f->int_port, NULL);
		g_free (f);
		break;
	  }
	  case RULETYPE_OPEN_PORT: text = g_strconcat ("Open port ", r->port, NULL); break;
	  case RULETYPE_STEALTHED_PORT: text = g_strconcat ("Open port ", r->port,
	                                                    " for ", r->host, NULL); break;
	  case RULETYPE_BLOCKED_PORT: text = g_strconcat ("Block port ", r->port, NULL); break;
	  default: return;

	}

	gtk_clipboard_set_text (cb, text, strlen (text));
	g_free (text);
	g_free (r);
}

GtkWidget *
create_ruleview_page (void)
{
	GtkWidget *rulevbox;
	GtkTreeModel *rulemodel;
	GtkWidget *scrolledwin;

	rulevbox = gtk_vbox_new (FALSE, GNOME_PAD_SMALL);
	rulemodel = create_ruletree_model ();
	Firestarter.ruleview = gtk_tree_view_new_with_model (rulemodel);

	scrolledwin = gtk_scrolled_window_new (NULL, NULL);
	gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scrolledwin),
	                                GTK_POLICY_NEVER,
	                                GTK_POLICY_ALWAYS);

	gtk_box_pack_start (GTK_BOX (rulevbox), scrolledwin, TRUE, TRUE, 0);
	
	gtk_container_add (GTK_CONTAINER (scrolledwin), Firestarter.ruleview);
	add_ruleview_columns (GTK_TREE_VIEW (Firestarter.ruleview));
	gtk_tree_view_set_rules_hint (GTK_TREE_VIEW (Firestarter.ruleview), FALSE);

	g_signal_connect (G_OBJECT (Firestarter.ruleview), "row-activated",
	                  G_CALLBACK (rule_selected_cb), NULL);
	g_signal_connect (G_OBJECT (Firestarter.ruleview), "button_press_event",
	                  G_CALLBACK (ruleview_button_press_cb), NULL);
	
	g_object_unref (G_OBJECT (rulemodel));

	return rulevbox;
}
