#include "tray.h"
#include "eggtrayicon.h"
#include "globals.h"
#include "xpm/firestarter-pixbufs.h"
#include "firestarter.h"
#include "gui.h"
#include "util.h"
#include "hitview.h"

static EggTrayIcon *tray_icon;
static GtkWidget *tray_icon_image;
static GtkTooltips *tray_icon_tooltip;

static gboolean tray_clicked (GtkWidget *event_box, GdkEventButton *event, gpointer data);
static gboolean tray_menu (GtkWidget *event_box, GdkEventButton *event, gpointer data);


/* [ tray_destroyed ]
 * Catch the destroy signal and restart (work around for the panel crashing)
 */
static gboolean
tray_destroyed (GtkWidget *widget, GdkEvent *event, gpointer user_data)
{
	tray_init ();
	return TRUE;
}

/* [ tray_init ]
 * Create the tray application
 */
void tray_init (void)
{
	GtkWidget *eventbox;
	GdkPixbuf *pixbuf;

	tray_icon = egg_tray_icon_new ("Firestarter");
 	pixbuf = gdk_pixbuf_new_from_inline (-1, tray_stopped, FALSE, NULL);
	tray_icon_image = gtk_image_new_from_pixbuf (pixbuf);

	eventbox = gtk_event_box_new ();
	gtk_widget_show (eventbox);
	gtk_container_add (GTK_CONTAINER (eventbox), tray_icon_image);
	gtk_container_add (GTK_CONTAINER (tray_icon), eventbox);

	g_signal_connect (G_OBJECT (eventbox), "button_press_event",
	                  G_CALLBACK (tray_clicked), NULL );

	gtk_widget_show_all (GTK_WIDGET (tray_icon));

	tray_icon_tooltip = gtk_tooltips_new ();

	g_signal_connect (G_OBJECT (tray_icon), "destroy",
	                  G_CALLBACK (tray_destroyed), NULL);
}

/* [ animation_timeout ]
 * Timeout function used to change the tray icon as part of an animation
 */
static int
animation_timeout (gpointer image)
{
	GdkPixbuf *pixbuf;
	pixbuf = gdk_pixbuf_new_from_inline (-1, image, FALSE, NULL);
	gtk_image_set_from_pixbuf (GTK_IMAGE (tray_icon_image), pixbuf);

	return FALSE;
}

/* [ tray_update ]
 * Set the tray icon status to state
 */
void tray_update (FirewallStatus state)
{
	GdkPixbuf *pixbuf = NULL;
	gchar *tooltip = NULL;

	if (state == STATUS_HIT) {
		const Hit *h = get_last_hit ();
		gchar *ip = g_strdup (h->source);

		gtk_timeout_add (0, animation_timeout, (gpointer)tray_hit1);
		gtk_timeout_add (100, animation_timeout, (gpointer)tray_hit2);
		gtk_timeout_add (200, animation_timeout, (gpointer)tray_hit3);
		gtk_timeout_add (300, animation_timeout, (gpointer)tray_hit4);
		gtk_timeout_add (400, animation_timeout, (gpointer)tray_hit5);
		gtk_timeout_add (500, animation_timeout, (gpointer)tray_hit6);
		gtk_timeout_add (700, animation_timeout, (gpointer)tray_hit5);
		gtk_timeout_add (800, animation_timeout, (gpointer)tray_hit4);
		gtk_timeout_add (900, animation_timeout, (gpointer)tray_hit3);
		gtk_timeout_add (1000, animation_timeout, (gpointer)tray_hit2);
		gtk_timeout_add (1100, animation_timeout, (gpointer)tray_hit1);
		
		tooltip = g_strdup_printf ("Hit from %s detected", lookup_ip (ip));
		g_free (ip);

	} else if (state == STATUS_STOPPED) {
 		pixbuf = gdk_pixbuf_new_from_inline (-1, tray_stopped, FALSE, NULL);
		tooltip = g_strdup (_("Firewall stopped"));
	} else if (state == STATUS_RUNNING) {
	 	pixbuf = gdk_pixbuf_new_from_inline (-1, tray_running, FALSE, NULL);
		tooltip = g_strdup (_("Firewall running"));
	} else if (state == STATUS_HALTED) {
	 	pixbuf = gdk_pixbuf_new_from_inline (-1, tray_halted, FALSE, NULL);
		tooltip = g_strdup (_("Firewall halted"));
	}

	if (state != STATUS_HIT) {
		gtk_image_set_from_pixbuf (GTK_IMAGE (tray_icon_image), pixbuf);
		gtk_widget_show (tray_icon_image);
	}

	gtk_tooltips_set_tip (tray_icon_tooltip, GTK_WIDGET (tray_icon), tooltip, NULL);
	g_free (tooltip);
}

/* [ tray_clicked ]
 * Callback for when the system tray icon is clicked
 */
static gboolean
tray_clicked (GtkWidget *event_box, GdkEventButton *event, gpointer data)
{
	/* Clear state */
	if (event->button == 1 && get_current_status () == STATUS_HIT) {
		update_status (STATUS_RUNNING);
	}

	/* Clear state and toggle visibility */
	if ((event->button == 1 && event->type == GDK_2BUTTON_PRESS) || event->button == 2) {
		if (get_current_status () == STATUS_HIT)
			update_status (STATUS_RUNNING);
		gui_toggle_visibility ();
		return TRUE;

	/* Pop up tray context menu */
	} else if (event->button == 3) {
		return tray_menu (event_box, event, data);
	}

	return FALSE;
}

/* [ tray_menu ]
 * Callback for popping up the menu
 */
static gboolean
tray_menu (GtkWidget *widget, GdkEventButton *event, gpointer data)
{
	GtkWidget *status_menu;
	GtkWidget *item, *image;
	GdkPixbuf *pixbuf;

	status_menu = gtk_menu_new();

	item = gtk_image_menu_item_new_with_label (_("Start firewall"));
 	pixbuf = gdk_pixbuf_new_from_inline (-1, icon_start_small, FALSE, NULL);
	image = gtk_image_new_from_pixbuf (pixbuf);
	gtk_container_add (GTK_CONTAINER (status_menu), item);
	gtk_image_menu_item_set_image (GTK_IMAGE_MENU_ITEM (item), image);
	g_signal_connect (G_OBJECT (item), "activate",
	                  G_CALLBACK (start_firewall),
	                  NULL);

	item = gtk_image_menu_item_new_with_label (_("Stop firewall"));
 	pixbuf = gdk_pixbuf_new_from_inline (-1, icon_stop_small, FALSE, NULL);
	image = gtk_image_new_from_pixbuf (pixbuf);
	gtk_container_add (GTK_CONTAINER (status_menu), item);
	gtk_image_menu_item_set_image (GTK_IMAGE_MENU_ITEM (item), image);
	g_signal_connect (G_OBJECT (item), "activate",
	                  G_CALLBACK (stop_firewall),
	                  NULL);

	item = gtk_menu_item_new ();
	gtk_widget_set_sensitive (item, FALSE);
	gtk_container_add (GTK_CONTAINER (status_menu), item);

	item = gtk_image_menu_item_new_with_label (_("Exit"));
	gtk_container_add (GTK_CONTAINER (status_menu), item);
	g_signal_connect (G_OBJECT (item), "activate",
	                  G_CALLBACK (exit_firestarter),
	                  NULL);


	gtk_widget_show_all (status_menu);

	gtk_menu_popup (GTK_MENU(status_menu), NULL, NULL,
	                NULL, NULL,
	                event->button, event->time);

	return TRUE;
}
