/*---[ wizard-choices.c ]---------------------------------------------
 * Copyright (C) 2000 Tomas Junnonen (majix@sci.fi)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Functions dealing with saving/restoring the users wizard choices
 *--------------------------------------------------------------------*/

#include "wizard-choices.h"
#include "wizard.h"
#include "util.h"
#include "preferences.h"

static void
update_toggle_from_conf (GtkWidget *button, const gchar *gconf_key)
{
	gboolean state;

	g_return_if_fail (GTK_IS_TOGGLE_BUTTON (button));

	state = preferences_get_bool (gconf_key);
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (button), state);
}

static void
update_entry_from_conf (GtkWidget *entry, const gchar *gconf_key)
{
	gchar *data;
	
	g_return_if_fail (GTK_IS_ENTRY (entry));

	data = preferences_get_string (gconf_key);
	gtk_entry_set_text (GTK_ENTRY (entry), data);
	
	g_free (data);
}

void
load_choices (Wizard *data)
{
	/* External device */
	update_entry_from_conf (data->extdevice, PREFS_FW_EXT_IF);

	update_toggle_from_conf (data->pppcheck, PREFS_FW_EXT_PPP);
	update_toggle_from_conf (data->dhcpcheck, PREFS_FW_EXT_DHCP);

	/* NAT */
	update_toggle_from_conf (data->masq, PREFS_FW_NAT);

	update_entry_from_conf (data->intdevice, PREFS_FW_INT_IF);

	/* Services */
	update_toggle_from_conf (data->services, PREFS_FW_SERVICES_ENABLE);

	update_toggle_from_conf (data->ftp, PREFS_FW_SERVICES_FTP);
	update_toggle_from_conf (data->ssh, PREFS_FW_SERVICES_SSH);
	update_toggle_from_conf (data->telnet, PREFS_FW_SERVICES_TELNET);
	update_toggle_from_conf (data->smtp, PREFS_FW_SERVICES_SMTP);
	update_toggle_from_conf (data->dns, PREFS_FW_SERVICES_DNS);
	update_toggle_from_conf (data->finger, PREFS_FW_SERVICES_FINGER);
	update_toggle_from_conf (data->www, PREFS_FW_SERVICES_WWW);
	update_toggle_from_conf (data->sslweb, PREFS_FW_SERVICES_SSLWEB);
	update_toggle_from_conf (data->pop, PREFS_FW_SERVICES_POP);
	update_toggle_from_conf (data->imap, PREFS_FW_SERVICES_IMAP);
	update_toggle_from_conf (data->ident, PREFS_FW_SERVICES_IDENT);
	update_toggle_from_conf (data->nntp, PREFS_FW_SERVICES_NNTP);
	update_toggle_from_conf (data->ntp, PREFS_FW_SERVICES_NTP);
	update_toggle_from_conf (data->samba, PREFS_FW_SERVICES_SAMBA);
	update_toggle_from_conf (data->ipsec, PREFS_FW_SERVICES_IPSEC);
	update_toggle_from_conf (data->routed, PREFS_FW_SERVICES_ROUTED);
	update_toggle_from_conf (data->nfs, PREFS_FW_SERVICES_NFS);
	update_toggle_from_conf (data->x, PREFS_FW_SERVICES_X);
	update_toggle_from_conf (data->dhcp, PREFS_FW_SERVICES_DHCP);
	update_toggle_from_conf (data->upnp, PREFS_FW_SERVICES_UPNP);
	update_toggle_from_conf (data->pptp, PREFS_FW_SERVICES_PPTP);

	/* TOS Filtering */
	update_toggle_from_conf (data->tos, PREFS_FW_FILTER_TOS);

	update_toggle_from_conf (data->tos_client, PREFS_FW_TOS_CLIENT);
	update_toggle_from_conf (data->tos_server, PREFS_FW_TOS_SERVER);
	update_toggle_from_conf (data->tos_X, PREFS_FW_TOS_X);

	update_toggle_from_conf (data->tos_option_throughput, PREFS_FW_TOS_OPT_TROUGHPUT);
	update_toggle_from_conf (data->tos_option_reliability, PREFS_FW_TOS_OPT_RELIABILITY);
	update_toggle_from_conf (data->tos_option_delay, PREFS_FW_TOS_OPT_DELAY);

	/* ICMP filtering */
	update_toggle_from_conf (data->icmp, PREFS_FW_FILTER_ICMP);

	update_toggle_from_conf (data->icmp_echo, PREFS_FW_ICMP_ECHO);
	update_toggle_from_conf (data->icmp_traceroute, PREFS_FW_ICMP_TRACEROUTE);
	update_toggle_from_conf (data->icmp_tracert, PREFS_FW_ICMP_MSTRACEROUTE);
	update_toggle_from_conf (data->icmp_unreach, PREFS_FW_ICMP_UNREACHABLE);
	update_toggle_from_conf (data->icmp_timestamp, PREFS_FW_ICMP_TIMESTAMPING);
	update_toggle_from_conf (data->icmp_masking, PREFS_FW_ICMP_MASKING);
	update_toggle_from_conf (data->icmp_redir, PREFS_FW_ICMP_REDIRECTION);
	update_toggle_from_conf (data->icmp_quench, PREFS_FW_ICMP_SOURCE_QUENCHES);
}


static void
update_conf_from_toggle (GtkWidget *button, const gchar *gconf_key)
{
	g_return_if_fail (GTK_IS_TOGGLE_BUTTON (button));

	preferences_set_bool (gconf_key,
		gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (button)));
}

static void
update_conf_from_entry (GtkWidget *entry, const gchar *gconf_key)
{
	const gchar *data;
	
	g_return_if_fail (GTK_IS_ENTRY (entry));

	data = gtk_entry_get_text (GTK_ENTRY (entry));
	preferences_set_string (gconf_key, data);
}

void
save_choices (Wizard *data)
{
	/* External device */
	update_conf_from_entry (data->extdevice, PREFS_FW_EXT_IF);

	update_conf_from_toggle (data->pppcheck, PREFS_FW_EXT_PPP);
	update_conf_from_toggle (data->dhcpcheck, PREFS_FW_EXT_DHCP);

	/* NAT */
	update_conf_from_toggle (data->masq, PREFS_FW_NAT);

	update_conf_from_entry (data->intdevice, PREFS_FW_INT_IF);

	/* Services */
	update_conf_from_toggle (data->services, PREFS_FW_SERVICES_ENABLE);

	update_conf_from_toggle (data->ftp, PREFS_FW_SERVICES_FTP);
	update_conf_from_toggle (data->ssh, PREFS_FW_SERVICES_SSH);
	update_conf_from_toggle (data->telnet, PREFS_FW_SERVICES_TELNET);
	update_conf_from_toggle (data->smtp, PREFS_FW_SERVICES_SMTP);
	update_conf_from_toggle (data->dns, PREFS_FW_SERVICES_DNS);
	update_conf_from_toggle (data->finger, PREFS_FW_SERVICES_FINGER);
	update_conf_from_toggle (data->www, PREFS_FW_SERVICES_WWW);
	update_conf_from_toggle (data->sslweb, PREFS_FW_SERVICES_SSLWEB);
	update_conf_from_toggle (data->pop, PREFS_FW_SERVICES_POP);
	update_conf_from_toggle (data->imap, PREFS_FW_SERVICES_IMAP);
	update_conf_from_toggle (data->ident, PREFS_FW_SERVICES_IDENT);
	update_conf_from_toggle (data->nntp, PREFS_FW_SERVICES_NNTP);
	update_conf_from_toggle (data->ntp, PREFS_FW_SERVICES_NTP);
	update_conf_from_toggle (data->samba, PREFS_FW_SERVICES_SAMBA);
	update_conf_from_toggle (data->ipsec, PREFS_FW_SERVICES_IPSEC);
	update_conf_from_toggle (data->routed, PREFS_FW_SERVICES_ROUTED);
	update_conf_from_toggle (data->nfs, PREFS_FW_SERVICES_NFS);
	update_conf_from_toggle (data->x, PREFS_FW_SERVICES_X);
	update_conf_from_toggle (data->dhcp, PREFS_FW_SERVICES_DHCP);
	update_conf_from_toggle (data->upnp, PREFS_FW_SERVICES_UPNP);
	update_conf_from_toggle (data->pptp, PREFS_FW_SERVICES_PPTP);

	/* TOS Filtering */
	update_conf_from_toggle (data->tos, PREFS_FW_FILTER_TOS);

	update_conf_from_toggle (data->tos_client, PREFS_FW_TOS_CLIENT);
	update_conf_from_toggle (data->tos_server, PREFS_FW_TOS_SERVER);
	update_conf_from_toggle (data->tos_X, PREFS_FW_TOS_X);

	update_conf_from_toggle (data->tos_option_throughput, PREFS_FW_TOS_OPT_TROUGHPUT);
	update_conf_from_toggle (data->tos_option_reliability, PREFS_FW_TOS_OPT_RELIABILITY);
	update_conf_from_toggle (data->tos_option_delay, PREFS_FW_TOS_OPT_DELAY);

	/* ICMP filtering */
	update_conf_from_toggle (data->icmp, PREFS_FW_FILTER_ICMP);

	update_conf_from_toggle (data->icmp_echo, PREFS_FW_ICMP_ECHO);
	update_conf_from_toggle (data->icmp_traceroute, PREFS_FW_ICMP_TRACEROUTE);
	update_conf_from_toggle (data->icmp_tracert, PREFS_FW_ICMP_MSTRACEROUTE);
	update_conf_from_toggle (data->icmp_unreach, PREFS_FW_ICMP_UNREACHABLE);
	update_conf_from_toggle (data->icmp_timestamp, PREFS_FW_ICMP_TIMESTAMPING);
	update_conf_from_toggle (data->icmp_masking, PREFS_FW_ICMP_MASKING);
	update_conf_from_toggle (data->icmp_redir, PREFS_FW_ICMP_REDIRECTION);
	update_conf_from_toggle (data->icmp_quench, PREFS_FW_ICMP_SOURCE_QUENCHES);
}
