/* -*- Mode: C; indent-tabs-mode: t; c-basic-offset: 4; tab-width: 4 -*- */
/*
 * fische-3.0
 * Copyright (C) Marcel Ebmer 2009 <marcel@26elf.at>
 * 
 * fische-3.0 is free software: you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 * 
 * fische-3.0 is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License along
 * with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <iostream> 
#include <fstream>
#include <cstring>
#include <cstdlib>
#include <stdint.h>
#include <malloc.h>
#include "blurengine.h"
#include "vectorfield.h"
#include "sdlscreen.h"
extern "C"
{
	#include "traditional.h"
}

using namespace std;

BlurEngine::BlurEngine(SdlScreen* screen, VectorField* vfield, bool mt)
{
	sdlscreen = screen;
	vectorfield = vfield;
	x0 = (sdlscreen->width() - vectorfield->width()) / 2;
	y0 = (sdlscreen->height() - vectorfield->height()) / 2;
	xRes = vectorfield->width();
	yRes = vectorfield->height();
	buffer = malloc(xRes * yRes * 4);
	if(!buffer)
	{
		cerr << "ERROR: unable to allocate memory for graphics buffer." << endl;
		exit(EXIT_FAILURE);
	}

	ifstream cpuinfo("/proc/cpuinfo");
	string str;
	cpus = 0;
	if(mt)
	{
		while(cpuinfo >> str)
		{
			if(str.find("processor") != string::npos) cpus ++;
		}
		cpuinfo.close();
		if(cpus == 1) cout << "* detected one CPU - using single threaded blur algorithm" << endl;
		else cout << "* detected " << cpus << " CPUs - using multi threaded blur algorithm" << endl;
	}
	else cpus = 1;
}

BlurEngine::~BlurEngine()
{
	if(buffer) free(buffer);
	buffer = NULL;
}

void BlurEngine::blur()
{
	emptyBuffer();
	sdlscreen->lock();

	if(cpus == 1) blurTrad();
	else blurTrad_MT();

	writeBuffer();
	sdlscreen->unlock();
}

void BlurEngine::blurTrad()
{
	static param_t p;
	p.pixels = sdlscreen->pixels();
	p.xRes = xRes;
	p.y0 = y0;
	p.x0 = x0;
	p.yS = 0;
	p.yE = yRes;
	p.pitch = sdlscreen->width();
	p.vectors = vectorfield->get();
	p.buffer = buffer;
	traditional(&p);
}

void BlurEngine::blurTrad_MT()
{
	int n = cpus;
	if(n > 8) n = 8;
	if(n % 2 != 0) n--;
	
	pthread_t t[n];
	param_t p[n];
	for(int i = 0; i < n; i ++)
	{
		p[i].pixels = sdlscreen->pixels();
		p[i].xRes = xRes;
		p[i].y0 = y0;
		p[i].x0 = x0;
		p[i].pitch = sdlscreen->width();
		p[i].vectors = vectorfield->get();
		p[i].buffer = buffer;
		p[i].yS = (i * yRes ) / n;
		p[i].yE = ((i + 1) * yRes ) / n;
		pthread_create(&t[i], NULL, traditional, &p[i]);
	}
	for(int i = 0; i < n; i ++)
	{
		pthread_join(t[i], NULL);
	}
}

void BlurEngine::emptyBuffer()
{
	bzero(buffer, xRes * yRes * 4);
}

void BlurEngine::writeBuffer()
{
	uint32_t* pixels = (uint32_t*)sdlscreen->pixels();
	uint32_t* src = (uint32_t*)buffer;
	int pitch = sdlscreen->width();
	for(int y = 0; y < yRes; y ++)
	{
		memcpy(pixels + (y + y0) * pitch + x0, src + y * xRes, xRes * 4);
	}
}
