/** \file common.c
	
Various functions, mostly string utilities, that are used by most
parts of fish.
*/
#include <stdlib.h>
#include <termios.h>
#include <wchar.h>
#include <string.h>
#include <stdio.h>
#include <dirent.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <wctype.h>
#include <errno.h>
#include <limits.h>
#include <stdarg.h>		
#include <signal.h>		

#include "config.h"
#include "util.h"
#include "wutil.h"
#include "common.h"

/**
   Maximum number of bytes in a utf-8 character
*/
#define MAX_UTF8_BYTES 6

#define STR2WCS_MSG "fish: Invalid multibyte sequence \'"
#define STR2WCS_MSG2 "fish: Invalid UTF-8 sequence \'"

/**
   The maximum number of minor errors to report. Further errors will be omitted.
*/
#define ERROR_MAX_COUNT 1

struct termios shell_modes;      /* so we can change the modes */

static int error_count=0;

wchar_t **list_to_char_arr( array_list_t *l )
{
	wchar_t ** res = malloc( sizeof(wchar_t *)*(al_get_count( l )+1) );
	int i;
	if( res == 0 )
	{
		fwprintf( stderr, L"fish: Out of memory in list_to_char_arr()\n" );
		exit(1);
	}
	for( i=0; i<al_get_count( l ); i++ )
		res[i] = (wchar_t *)al_get(l,i);
	res[i]='\0';
	return res;	
}


int fgetws2( wchar_t **b, int *len, FILE *f )
{
	int i=0;
	wint_t c;
	
	wchar_t *buff = *b;

	/*
	  This is a kludge: We block SIGCHLD while reading, since I can't
	  get getwc to perform reliably when signals are flying. Even when
	  watching for EINTR errors, bytes are lost. 
	*/
	sigset_t chldset; 
	sigemptyset( &chldset );
	sigaddset( &chldset, SIGCHLD );
	sigprocmask(SIG_BLOCK, &chldset, 0);

	while( 1 )
	{
		/* Reallocate the buffer if necessary */
		if( i+1 >= *len )
		{
			int new_len = maxi( 128, (*len)*2);
			buff = realloc( buff, sizeof(wchar_t)*new_len );
			if( buff == 0 )
			{
				sigprocmask(SIG_UNBLOCK, &chldset, 0);
				return -1;
			}
			else
			{
				*len = new_len;
				*b = buff;
			}			
		}
		
		c = getwc( f );
		//fwprintf( stderr, L"b\n" );
		
		switch( c )
		{
			/* End of line */ 
			case WEOF:
			case L'\n':
			case L'\0':
				buff[i]=L'\0';
				sigprocmask(SIG_UNBLOCK, &chldset, 0);
				return i;				
				/* Ignore carriage returns */
			case L'\r':
				break;
				
			default:
				buff[i++]=c;
				break;
		}		
	}
}


/**
   Wrapper for wcsfilecmp
*/
static int completion_cmp( const void *a, const void *b )
{
	wchar_t *c= *((wchar_t **)a);
	wchar_t *d= *((wchar_t **)b);
	return wcsfilecmp( c, d );
}

void sort_list( array_list_t *comp )
{
	
	qsort( comp->arr, 
		   al_get_count( comp ),
		   sizeof( void*),
		   &completion_cmp );
}

/*
static int validate_utf8( const char *in )
{
	while( *in )
	{
		if( *in & 0x80 )
		{
			int i, len = -1;
			if( ((*in)&0xE0) == 0xC0 )
			{
				len=2;
			}
			else if( ((*in)&0xf0) == 0xE0 )
			{
				len=3;
			}
			else if( ((*in)&0xf8) == 0xf0 )
			{
				len=4;
			}
			if( len == -1 )
				return 0;
			in++;			
			for( i=1; i<len; i++, in++ )
			{
				if( ((*in)&0xC0) != 0x80 )
					return 0;
			}
		}
		else
			in++;		
	}
	return 1;
}
*/

wchar_t *str2wcs( const char *in )
{
	wchar_t *res;

/*	if( !validate_utf8( in ) )
	{
		write( 2,
			   STR2WCS_MSG2,
			   strlen(STR2WCS_MSG2) );
		write( 2,
			   in,
			   strlen(in ));
		write( 2, 
			   "\'\n",
			   2 );
		
		return 0;
	}
*/	
	res = malloc( sizeof(wchar_t)*(strlen(in)+1) );
	
	if( !res )
	{
		fwprintf( stderr, L"fish: Out of memory in function str2wcs\n" );
		exit(1);
	}
	
	if( (size_t)-1 == mbstowcs( res, in, sizeof(wchar_t)*(strlen(in)) +1) )
	{
		error_count++;
		if( error_count <= ERROR_MAX_COUNT )
		{
					
			write( 2,
				   STR2WCS_MSG,
				   strlen(STR2WCS_MSG) );
			write( 2,
				   in,
				   strlen(in ));
			write( 2, 
				   "\'\n",
				   2 );
		}
		
		free(res);
		return 0;
	}	
	
	return res;
	
}

void error_reset()
{
	error_count=0;
}


char *wcs2str( const wchar_t *in )
{
	char *res = malloc( MAX_UTF8_BYTES*wcslen(in)+1 );
	if( res == 0 )
	{
		fwprintf( stderr, L"fish: Out of memory in wcs2str()\n" );
		exit(1);
	}
	wcstombs( res, in, MAX_UTF8_BYTES*wcslen(in)+1 );
	res = realloc( res, strlen( res )+1 );
	return res;
}

char **wcsv2strv( const wchar_t **in )
{
	int count =0;
	int i;

	while( in[count] != 0 )
		count++;
	char **res = malloc( sizeof( char *)*(count+1));
	if( res == 0 )
	{
		fwprintf( stderr, L"fish: Out of memory in wcsv2strv()\n" );
		exit(1);
	}

	for( i=0; i<count; i++ )
	{
		res[i]=wcs2str(in[i]);
	}
	res[count]=0;
	return res;

}

wchar_t *wcsdupcat( const wchar_t *a, const wchar_t *b )
{
	return wcsdupcat2( a, b, 0 );
}

wchar_t *wcsdupcat2( const wchar_t *a, ... )
{
	int len=wcslen(a);
	int pos;
	va_list va, va2;
	wchar_t *arg;

	va_start( va, a );
	va_copy( va2, va );
	while( (arg=va_arg(va, wchar_t *) )!= 0 ) 
	{
		len += wcslen( arg );
	}
	va_end( va );

	wchar_t *res = malloc( sizeof(wchar_t)*(len +1 ));
	if( res == 0 )
		return 0;

	wcscpy( res, a );
	pos = wcslen(a);
	while( (arg=va_arg(va2, wchar_t *) )!= 0 ) 
	{
		wcscpy( res+pos, arg );
		pos += wcslen(arg);
	}
	va_end( va2 );
	return res;	

}


wchar_t **strv2wcsv( const char **in )
{
	int count =0;
	int i;

	while( in[count] != 0 )
		count++;
	wchar_t **res = malloc( sizeof( wchar_t *)*(count+1));
	if( res == 0 )
	{
		fwprintf( stderr, L"fish: Out of memory in strv2wcsv()\n" );
		exit(1);
	}

	for( i=0; i<count; i++ )
	{
		res[i]=str2wcs(in[i]);
	}
	res[count]=0;
	return res;

}



wchar_t *wcsndup( const wchar_t *in, int c )
{
	wchar_t *res = malloc( sizeof(wchar_t)*(c+1) );
	if( res == 0 )
	{
		fwprintf( stderr, L"fish: Out of memory allocating %d bytes in wcsndup()\n", sizeof(wchar_t)*(c+1) );
		exit(1);
	}
	wcsncpy( res, in, c );
	res[c] = L'\0';	
	return res;	
}

/**
   Converts from wide char to digit in the specified base. If d is not
   a valid digit in the specified base, return -1.
*/
static long digit( wchar_t d, int base )
{
	long res=-1;
	if( (d <= L'9') && (d >= L'0') )
	{
		res = d - L'0';
	}
	else if( (d <= L'z') && (d >= L'a') )
	{
		res = d - L'a';		
	}
	else if( (d <= L'Z') && (d >= L'A') )
	{
		res = d - L'A';		
	}
	if( res >= base )
	{
		res = -1;
	}
	
	return res;
}


long wcstol(const wchar_t *nptr, 
			wchar_t **endptr,
			int base)
{
	long long res=0;
	int is_set=0;
	if( base > 36 )
	{
		errno = EINVAL;
		return 0;
	}

	while( 1 )
	{
		long nxt = digit( *nptr, base );
		if( nxt < 0 )
		{
			if( endptr != 0 )
				*endptr = (wchar_t *)nptr;
			if( !is_set )
			{
				errno = EINVAL;
			}
			return res;			
		}
		nptr++;
		res = (res*base)+nxt;
		is_set = 1;
		if( res > LONG_MAX )
		{
			errno = ERANGE;
			return LONG_MAX;
		}
		if( res < LONG_MIN )
		{
			errno = ERANGE;
			return LONG_MIN;
		}
	}
}

/*$OpenBSD: strlcat.c,v 1.11 2003/06/17 21:56:24 millert Exp $*/

/*
 * Copyright (c) 1998 Todd C. Miller <Todd.Miller@courtesan.com>
 *
 * Permission to use, copy, modify, and distribute this software for any
 * purpose with or without fee is hereby granted, provided that the above
 * copyright notice and this permission notice appear in all copies.
 *
 * THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
 * WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
 * ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
 * OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */

/**
   Appends src to string dst of size siz (unlike wcsncat, siz is the
   full size of dst, not space left).  At most siz-1 characters will be
   copied.  Always NUL terminates (unless siz <= wcslen(dst)).  Returns
   wcslen(src) + MIN(siz, wcslen(initial dst)).  If retval >= siz,
   truncation occurred.

   This is the OpenBSD strlcat function, modified for wide characters,
   and renamed to reflect this change.

*/
size_t
wcslcat(wchar_t *dst, const wchar_t *src, size_t siz)
{
	
	register wchar_t *d = dst;
	register const wchar_t *s = src;
	register size_t n = siz;	
	size_t dlen;

	/* Find the end of dst and adjust bytes left but don't go past end */
	while (n-- != 0 && *d != '\0')
		d++;
	
	dlen = d - dst;
	n = siz - dlen;	

	if (n == 0)
		return(dlen + wcslen(s));

	while (*s != '\0') 
	{
		if (n != 1) 
		{
			*d++ = *s;
			n--;
		}
		s++;
	}
	*d = '\0';

	return(dlen + (s - src));
	/* count does not include NUL */
}

/*$OpenBSD: strlcpy.c,v 1.8 2003/06/17 21:56:24 millert Exp $*/

/*
 * Copyright (c) 1998 Todd C. Miller <Todd.Miller@courtesan.com>
 *
 * Permission to use, copy, modify, and distribute this software for any
 * purpose with or without fee is hereby granted, provided that the above
 * copyright notice and this permission notice appear in all copies.
 *
 * THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
 * WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
 * ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
 * OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */

/**
   Copy src to string dst of size siz.  At most siz-1 characters will
   be copied.  Always NUL terminates (unless siz == 0).  Returns
   wcslen(src); if retval >= siz, truncation occurred.

   This is the OpenBSD strlcpy function, modified for wide characters,
   and renamed to reflect this change. 
*/
size_t
wcslcpy(wchar_t *dst, const wchar_t *src, size_t siz)
{
	register wchar_t *d = dst;
	register const wchar_t *s = src;
	register size_t n = siz;
	
	/* Copy as many bytes as will fit */
	if (n != 0 && --n != 0) 
	{		
		do 
		{			
			if ((*d++ = *s++) == 0)
				break;
		}
		while (--n != 0);
	}

	/* Not enough room in dst, add NUL and traverse rest of src */
	if (n == 0) 
	{		
		if (siz != 0)
			*d = '\0';
		/* NUL-terminate dst */
		while (*s++)
			;
	}
	return(s - src - 1);
	/* count does not include NUL */
}

wchar_t *wcsdup( const wchar_t *in )
{
	wchar_t *out = malloc( sizeof( wchar_t)*(wcslen(in)+1));
	if( out == 0 )
	{
		fwprintf( stderr, L"fish: Out of memory in wcsdup\n" );
		exit(1);
	}
	wcscpy( out, in );
	return out;
	
}

int wcscasecmp( const wchar_t *a, const wchar_t *b )
{
	if( *a == 0 )
	{
		return (*b==0)?0:-1;
	}
	else if( *b == 0 )
	{
		return 1;
	}
	int diff = towlower(*a)-towlower(*b);
	if( diff != 0 )
		return diff;
	else
		return wcscasecmp( a+1,b+1);
}

int wcsvarname( wchar_t *str )
{
	while( *str )
	{
		if( (!iswalnum(*str)) && (*str != L'_' ) )
		{
			return 0;
		}
		str++;
	}
	return 1;
	
	
}

#if !HAVE_WCWIDTH
/**
   Return the number of columns used by a character. 

   In locales without a native wcwidth, Unicode is probably so broken
   that it isn't worth trying to implement a real wcwidth. This
   wcwidth assumes any printing character takes up one column.
*/
int wcwidth( wchar_t c )
{
	if( c < 32 )
		return 0;
	if ( c == 127 )
		return 0;
	return 1;
}
#endif

/** 
	The glibc version of wcswidth seems to hang on some strings. fish uses this replacement.
*/
int my_wcswidth( const wchar_t *c )
{
	int res=0;
	while( *c )
	{
		int w = wcwidth( *c++ );
		if( w < 0 )
			w = 1;
		if( w > 2 )
			w=1;
		
		res += w;		
	}
	return res;
}

wchar_t *quote_end( const wchar_t *in )
{
	int level=1;
	const wchar_t *q_char = L"\'\"";
	int offset = (*in != L'\"');

	in++;
	
	while(1)
	{
/*		fwprintf( stderr, L"Check %c\n", *tok->buff );*/
		switch( *in )
		{
			case L'\\':
				in++;
				if( *in == L'\0' )
				{
					return 0;
				}
				break;
			case L'\"':
			case L'\'':
				if( q_char[(level+offset) % 2] == *in )
				{
					level--;
				}
				else
				{
					level++;
				}
				
				break;
		}
		if( (*in == L'\0') ||(level==0))
			break;

		in++;
	}
	return level?0:(wchar_t *)in;
		
}


