/** \file complete.c
	Functions related to tab-completion.

	These functions are used for storing and retrieving tab-completion data, as well as for performing tab-completion.
*/
#include <stdlib.h>
#include <stdio.h>
#include <limits.h>
#include <string.h>
#include <wchar.h>
#include <wctype.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <dirent.h>
#include <errno.h>
#include <termios.h>
#include <ctype.h>
#include <pwd.h>
#include <signal.h>

#include "config.h"
#include "util.h"
#include "tokenizer.h"
#include "wildcard.h"
#include "proc.h"
#include "parser.h"
#include "function.h"
#include "complete.h"
#include "builtin.h"
#include "env.h"
#include "exec.h"
#include "expand.h"
#include "common.h"
#include "reader.h"
#include "history.h"

#include "wutil.h"


/*
  Completion description strings for files

  There are a few other completion description strings defined in expand.c
*/

/**
   Description for ~USER completion
*/
#define COMPLETE_USER_DESC COMPLETE_SEP_STR L"User home"

/**
   Description for short variables. The value is concatenated to this description
*/
#define COMPLETE_VAR_DESC_VAL COMPLETE_SEP_STR L"Variable: "

/**
   Description for generic executable
*/
#define COMPLETE_EXEC_DESC COMPLETE_SEP_STR L"Executable"
/**
   Description for link to executable
*/
#define COMPLETE_EXEC_LINK_DESC COMPLETE_SEP_STR L"Executable link"
/**
   Description for executable ending with .sh
*/
#define COMPLETE_EXEC_SHELL_DESC COMPLETE_SEP_STR L"Shell script"
/**
   Description for executable ending with .py
*/
#define COMPLETE_EXEC_PYTHON_DESC COMPLETE_SEP_STR L"Python script"
/**
   Description for executable ending with .pl
*/
#define COMPLETE_EXEC_PERL_DESC COMPLETE_SEP_STR L"Perl script"
/**
   Description for executable ending with .php
*/
#define COMPLETE_EXEC_PHP_DESC COMPLETE_SEP_STR L"PHP script"
/**
   Description for executable ending with .rb
*/
#define COMPLETE_EXEC_RUBY_DESC COMPLETE_SEP_STR L"Ruby script"
/**
   Description for executable ending with .tcl
*/
#define COMPLETE_EXEC_TCL_DESC COMPLETE_SEP_STR L"Tcl/Tk script"

/**
   Description for regular file
*/
#define COMPLETE_FILE_DESC COMPLETE_SEP_STR L"File"
/**
   Description for character device
*/
#define COMPLETE_CHAR_DESC COMPLETE_SEP_STR L"Character device"
/**
   Description for block device
*/
#define COMPLETE_BLOCK_DESC COMPLETE_SEP_STR L"Block device"
/**
   Description for fifo buffer
*/
#define COMPLETE_FIFO_DESC COMPLETE_SEP_STR L"Fifo"
/**
   Description for symlink
*/
#define COMPLETE_SYMLINK_DESC COMPLETE_SEP_STR L"Symbolic link"
/**
   Description for Rotten symlink
*/
#define COMPLETE_ROTTEN_SYMLINK_DESC COMPLETE_SEP_STR L"Rotten symbolic link"
/**
   Description for socket
*/
#define COMPLETE_SOCKET_DESC COMPLETE_SEP_STR L"Socket"
/**
   Description for directory
*/
#define COMPLETE_DIRECTORY_DESC COMPLETE_SEP_STR L"Directory"

/**
   Description for function
*/
#define COMPLETE_FUNCTION_DESC COMPLETE_SEP_STR L"Function"
/**
   Description for builtin command
*/
#define COMPLETE_BUILTIN_DESC COMPLETE_SEP_STR L"Builtin"

/** 
	The command to run to get a description from a file suffix
*/
#define SUFFIX_CMD_STR L"mimedb 2>/dev/null -fd "

/**
   Struct describing a completion option entry
*/
typedef struct complete_entry_opt
{
	/** Short style option */
	wchar_t short_opt;
	/** Long style option */
	wchar_t *long_opt;
	/** Arguments to the option */
	wchar_t *comp;
	/** Description of the completion */
	wchar_t *desc;
	/** Must be one of the values SHARED, NO_FILES, NO_COMMON, EXCLUSIVE. */
	int result_mode;
	/** Next option in the linked list */
	struct complete_entry_opt *next;
}
	complete_entry_opt;

/**
   Struct describing a command completion
*/
typedef struct complete_entry
{
	/** True if command is a path */
	int cmd_type;
	
	/** Command string */
	wchar_t *cmd;
	/** String containing all short option characters */
	wchar_t *short_opt_str;
	/** Linked list of all options */
	complete_entry_opt *first_option;
	/** Next command completion in the linked list */
	struct complete_entry *next;
	/** True if old style long options are used */
	int old_mode;	
	/** True if no other options than the ones supplied are possible */
	int authorative;	
}
	complete_entry;

/** First node in the linked list of all completion entries */
static complete_entry *first_entry=0;

/** Hashtable containing all descriptions that describe an executable */
static hash_table_t executable_desc_hash, suffix_hash;


void complete_init()
{
	hash_init( &suffix_hash, &hash_wcs_func, &hash_wcs_cmp );

	hash_init( &executable_desc_hash, &hash_wcs_func, &hash_wcs_cmp );
	hash_put(  &executable_desc_hash, COMPLETE_EXEC_DESC, (void *)1 );
	hash_put(  &executable_desc_hash, COMPLETE_EXEC_LINK_DESC, (void *)1 );
	hash_put(  &executable_desc_hash, COMPLETE_EXEC_SHELL_DESC, (void *)1 );
	hash_put(  &executable_desc_hash, COMPLETE_EXEC_PYTHON_DESC, (void *)1 );
	hash_put(  &executable_desc_hash, COMPLETE_EXEC_PERL_DESC, (void *)1 );
	hash_put(  &executable_desc_hash, COMPLETE_EXEC_PHP_DESC, (void *)1 );
	hash_put(  &executable_desc_hash, COMPLETE_EXEC_RUBY_DESC, (void *)1 );
	hash_put(  &executable_desc_hash, COMPLETE_EXEC_TCL_DESC, (void *)1 );	
}

/**
   Recursively free all complete_entry_opt structs and their contents
*/
static void complete_free_opt_recursive( complete_entry_opt *o )
{
	if( o->next != 0 )
		complete_free_opt_recursive( o->next );
	if( wcslen( o->long_opt ) )
		free( o->long_opt );
	if( wcslen( o->comp ) )
		free( o->comp );
	if( wcslen( o->desc ) )
		free( o->desc );

	free(o);
	return;
}

/**
   Free a complete_entry and its contents
*/
static void complete_free_entry( complete_entry *c )
{
	free( c->cmd );
	free( c->short_opt_str );
	complete_free_opt_recursive( c->first_option );
	free( c );	
}

/**
   Free hash key and hash value
*/
static void clear_hash_entry( const void *key, const void *data )
{
	free( (void *)key );
	free( (void *)data );
}


void complete_destroy()
{
	complete_entry *i=first_entry, *prev;
	while( i )
	{
		prev = i;
		i=i->next;
		complete_free_entry( prev );
	}	
	hash_destroy( &executable_desc_hash );

	hash_foreach( &suffix_hash, &clear_hash_entry );
	hash_destroy( &suffix_hash );
}

/**
   Search for an exactly matching completion entry
*/
static complete_entry *complete_find_exact_entry( const wchar_t *cmd, 
												  const int cmd_type )
{
	complete_entry *i;
	for( i=first_entry; i; i=i->next )
	{
		if( ( wcscmp(cmd, i->cmd)==0) && ( cmd_type == i->cmd_type ))
		{
			return i;
		}
	}		
	return 0;
	
}

void complete_add( const wchar_t *cmd, 
				   int cmd_type, 
				   wchar_t short_opt,
				   const wchar_t *long_opt,
				   int old_mode, 
				   int result_mode, 
				   int authorative,
				   const wchar_t *comp,
				   const wchar_t *desc )
{
	complete_entry *c = 
		complete_find_exact_entry( cmd, cmd_type );
	complete_entry_opt *opt;

	if( c == 0 )
	{
		c = malloc( sizeof(complete_entry) );
		c->next = first_entry;
		first_entry = c;
		
		c->first_option = 0;
		
		c->cmd = wcsdup( cmd );
		c->cmd_type = cmd_type;
		c->short_opt_str = wcsdup(L"");
		c->old_mode = 0;
		
	}

/*		wprintf( L"Add completion to option (short %lc, long %ls)\n", short_opt, long_opt );*/
	opt = malloc( sizeof( complete_entry_opt ) );
	opt->next = c->first_option;
	c->first_option = opt;
	c->authorative = authorative;
	if( short_opt != L'\0' )
	{
		int len = 1 + ((result_mode & NO_COMMON) != 0);
		c->short_opt_str = 
			realloc( c->short_opt_str, 
					 sizeof(wchar_t)*(wcslen( c->short_opt_str ) + 1 + len) );
		wcsncat( c->short_opt_str,
				 &short_opt, 1 );
		if( len == 2 )
			wcscat( c->short_opt_str, L":" );		  
	}
									
	opt->short_opt = short_opt;
	if( wcslen( long_opt ) > 0 )
		c->old_mode = old_mode;
	opt->result_mode = result_mode;
	
	if( wcslen( comp ) )
	{
		opt->comp = wcsdup(comp);
	}
	else
	{
		opt->comp = L"";
	}

	if( wcslen( long_opt ) )
	{
		opt->long_opt = wcsdup( long_opt );
	}
	else
	{		
		opt->long_opt = L"";
	}
	
	if( wcslen( desc ) )
	{
		opt->desc = wcsdupcat( COMPLETE_SEP_STR, desc );
	}
	else
	{
		opt->desc = L"" ;
	}
}

void complete_remove( const wchar_t *cmd, 
					  int cmd_type, 
					  wchar_t short_opt,
					  const wchar_t *long_opt )
{
	complete_entry *e, *eprev=0, *enext=0;
	for( e = first_entry; e; e=enext )
	{
		enext=e->next;
		
		if( (cmd_type == e->cmd_type ) &&
			( wcscmp( cmd, e->cmd) == 0 ) )
		{
			complete_entry_opt *o, *oprev=0, *onext=0;

			if(( short_opt == 0 ) && (long_opt == 0 ) )
			{
				complete_free_entry( e );
				e=0;
			}
			else
			{
				
				for( o= e->first_option; o; o=onext )
				{
					onext=o->next;
					
					if( ( short_opt==o->short_opt ) ||
						( wcscmp( long_opt, o->long_opt ) == 0 ) )
					{
						wchar_t *pos;
						fwprintf( stderr,
								  L"remove option -%lc --%ls\n",
								  o->short_opt?o->short_opt:L' ', 
								  o->long_opt );
					
						if( o->short_opt )
						{
							pos = wcschr( e->short_opt_str, 
										  o->short_opt );
							if( pos )
							{
								wchar_t *pos2 = pos+1;
								while( *pos2 == L':' )
									pos2++;
							
								memmove( pos,
										 pos2,
										 sizeof(wchar_t)*wcslen(pos2) );
							}
						}
					
						if( oprev == 0 )
						{
							e->first_option = o->next;
						}
						else
						{
							oprev->next = o->next;
						}
						if( o->long_opt != L"" )
							free( o->long_opt );
						if( o->desc != L"" )
							free( o->desc );
						if( o->comp != L"" )
							free( o->comp );
						free( o );
					}
					else
						oprev = o;
				}
			
			}
		}			
		
		if( e && (e->first_option == 0) )
		{
			fwprintf( stderr, L"remove entry\n" );
			if( eprev == 0 )
			{
				first_entry = e->next;
			}
			else
			{
				eprev = e->next;
			}
			
			free( e->cmd );
			free( e->short_opt_str );
			free( e );
		}

		if( e )
			eprev = e;
	}
}

/**
   Check if the two file suffixes are identical. The first string may
   contain a trailing tilde (\~), which will be ignored.
*/
static int suffix_compare( wchar_t *str, 
						   wchar_t *suffix )
{
	int strlen = wcslen( str );
	int sufflen = wcslen( suffix );	

	if( wcsncmp( str, suffix, sufflen )==0 )
	{
		if( strlen == sufflen )
			return 1;
		if(( sufflen == strlen-1 ) && (str[strlen-1]==L'~') )
			return 1;
	}
	return 0;
}

/**
   Find the full path and commandname from a command string.  the
   result of pathp must be freed by the caller, the result of cmdp
   must not be freed by the caller.
*/
static void parse_cmd_string( const wchar_t *str, 
							  wchar_t **pathp,
							  wchar_t **cmdp )
{
    wchar_t *cmd, *path;

	/* Get the path of the command */
	path = get_filename( str );
	if( path == 0 )
		path = wcsdup(L"");
	
	/* Make sure the path is not included in the command */
	cmd = wcsrchr( str, L'/' );
	if( cmd != 0 )
		cmd++;
	else
		cmd = (wchar_t *)str;

	*pathp=path;
	*cmdp=cmd;	
}

int complete_is_valid_option( const wchar_t *str, 
							  const wchar_t *opt, 
							  array_list_t *errors )
{

	complete_entry *i;
	complete_entry_opt *o;
	wchar_t *cmd, *path;
	int found_match = 0;
	int authorative = 1;
	
	int opt_found=0;

	hash_table_t gnu_match_hash;
	
	int is_gnu_opt=0;
	int is_old_opt=0;
	int is_short_opt=0;
	int is_gnu_exact=0;
	int gnu_opt_len=0;
	
	char *short_validated;
	/*
	  Check some generic things like -- and - options.
	*/
	switch( wcslen(opt ) )
	{
		
		case 0:
			return 1;
			
				
		case 1:
			return  opt[0] == L'-';
			
		case 2:
			if( wcscmp( L"--", opt ) == 0 )
				return 1;
	}
	
	if( opt[0] != L'-' )
	{
		if( errors )
			al_push( errors, wcsdup(L"Option does not begin with a '-'") );
		
		return 0;
	}
	
	short_validated = malloc( wcslen( opt ) );
	memset( short_validated, 0, wcslen( opt ) );

	hash_init( &gnu_match_hash, 
			   &hash_wcs_func,
			   &hash_wcs_cmp );

	is_gnu_opt = opt[1]==L'-';
	if( is_gnu_opt )
	{
		wchar_t *opt_end = wcschr(opt, L'=' );
		if( opt_end )
			gnu_opt_len = (opt_end-opt)-2;
		else
			gnu_opt_len = wcslen(opt)-2;
//		fwprintf( stderr, L"Length %d optend %d\n", gnu_opt_len, opt_end );
		
	}
		
	parse_cmd_string( str, &path, &cmd );

	for( i=first_entry; i; i=i->next )
	{
		wchar_t *match = i->cmd_type?path:cmd;
		const wchar_t *a;		
		
		if( !wildcard_match( match, i->cmd, 0 ) )
			continue;
		found_match = 1;

		if( !i->authorative )
		{
			authorative = 0;
			break;
		}
		
		
		if( is_gnu_opt )
		{
			if( i->old_mode )
				continue;
			
			for( o = i->first_option; o; o=o->next )
			{
				//fwprintf( stderr, L"Compare \'%ls\' against \'%ls\'\n", &opt[2], o->long_opt );
				if( wcsncmp( &opt[2], o->long_opt, gnu_opt_len )==0)
				{
					//fwprintf( stderr, L"Found gnu match %ls\n", o->long_opt );
					hash_put( &gnu_match_hash, o->long_opt, L"" );
					if( (wcsncmp( &opt[2], 
								  o->long_opt,
								  wcslen( o->long_opt) )==0) )
						is_gnu_exact=1;
				}		
			}
		}
		else
		{
			if( i->old_mode )
			{
				/* Check for old style options */
				for( o = i->first_option; o; o=o->next )
				{
					if( wcscmp( &opt[1], o->long_opt )==0)
					{
//						fwprintf( stderr, L"Found old match %ls\n", o->long_opt );
						opt_found = 1;
						is_old_opt = 1;
						break;
					}
				}
			}
			
			if( is_old_opt )
				break;
			
			
			for( a = &opt[1]; *a; a++ )
			{
				
				wchar_t *str_pos = wcschr(i->short_opt_str, *a);
			
				if  (str_pos )
				{
					if( *(str_pos +1)==L':' )
					{
						/*
						  This is a short option with an embedded argument, 
						  call complete_is_valid_argument on the argument.
						*/
						wchar_t nopt[3];
						nopt[0]=L'-';
						nopt[1]=opt[1];
						nopt[2]=L'\0';
						
						//fwprintf( stderr, L"Pos %d, shortopt %lc has argument\n", a-opt, *a );
						short_validated[a-opt] =
							complete_is_valid_argument( str, nopt, &opt[2]);
					}
					else
					{
						//	fwprintf( stderr, L"Pos %d, shortopt %lc is ok\n", a-opt, *a );
						short_validated[a-opt]=1;
					}
					
				}			
			}
			
		}
		
		
	}
	free( path );

	if( authorative )
	{
		
		if( !is_gnu_opt && !is_old_opt )
			is_short_opt = 1;
	

		if( is_short_opt )
		{
			int j;
		
			opt_found=1;
			for( j=1; j<wcslen(opt); j++)
			{
				if ( !short_validated[j])
				{
					//			fwprintf( stderr, L"Pos %d, shortopt %lc is not ok\n", j, opt[j] );
					if( errors )
					{
						wchar_t str[2];
						str[0] = opt[j];
						str[1]=0;
						al_push( errors,
								 wcsdupcat2(L"Unknown option \'", str, L"\'", 0) );
					}
				
					opt_found = 0;
					break;
				}
			
			}
		}	

		if( is_gnu_opt )
		{
//		fwprintf( stderr, L"Found %d matches\n", hash_get_count( &gnu_match_hash ) );	
			opt_found = is_gnu_exact || (hash_get_count( &gnu_match_hash )==1);
			if( errors && !opt_found )
			{
				if( hash_get_count( &gnu_match_hash )==0)
				{
					al_push( errors,
							 wcsdupcat2(L"Unknown option \'", opt, L"\'", 0) );
				}		
				else
				{
					al_push( errors,
							 wcsdupcat2(L"Multiple matches for option \'", opt, L"\'", 0) );
				}
			}		
		}
	}
	
	hash_destroy( &gnu_match_hash );
	free( short_validated );

	return (authorative && found_match)?opt_found:1;
}

int complete_is_valid_argument( const wchar_t *str, 
								const wchar_t *opt, 
								const wchar_t *arg )
{
	return 1;
}

/**
   Use the mimedb command to look up a description for a given suffix
*/
static wchar_t *complete_get_desc_suffix( const wchar_t *suff_orig )
{
	
	int len = wcslen(suff_orig );
	if( len == 0 )
		return COMPLETE_FILE_DESC;
	
	wchar_t *suff;
	wchar_t *pos;
	
	suff = wcsdup(suff_orig);
	
	for( pos=suff; *pos; pos++ )
	{		
		if( wcschr( L"?;#~@&", *pos ) )
		{
			*pos=0;
			break;
		}
	}
		
	suff = expand_escape( suff, 0 );
	
	wchar_t *desc = (wchar_t *)hash_get( &suffix_hash, suff );
	
	if( !desc )
	{

		wchar_t *cmd = wcsdupcat( SUFFIX_CMD_STR, suff );

		if( cmd )
		{
			
			array_list_t l;
			al_init( &l );
			
			exec_subshell( cmd, &l );
			free(cmd);
			
			if( al_get_count( &l )>0 )
			{
				wchar_t *ln = (wchar_t *)al_get(&l, 0 );
				if( wcscmp( ln, L"unknown" ) != 0 )
				{
					desc = wcsdupcat( COMPLETE_SEP_STR, ln);
					/*
					  I have decided I prefer to have the description
					  begin in uppercase and the whole universe will just
					  have to accept it.
					*/
					desc[1]=towupper(desc[1]);
				}			
			}
			
			al_foreach( &l, (void (*)(const void *))&free );
			al_destroy( &l );
		}
		
		if( !desc )
		{
			desc = wcsdup(COMPLETE_FILE_DESC);
		}
		
		hash_put( &suffix_hash, suff!=suff_orig?suff:wcsdup(suff), desc );		
	}
	else
	{
		free( suff );
	}
	
	return desc;	
}


wchar_t *complete_get_desc( const wchar_t *filename )
{
	struct stat buf;			
	wchar_t *desc = COMPLETE_FILE_DESC;
	
	if( lwstat( filename, &buf )==0)
	{

		if( waccess( filename, X_OK ) == 0 )
		{
			desc = COMPLETE_EXEC_DESC;
			wchar_t *suffix = wcsrchr( filename, L'.' );
			if( suffix )
			{
				if( suffix_compare( suffix, L".sh" ) )
				{
					desc = COMPLETE_EXEC_SHELL_DESC;
				}
				else if( suffix_compare( suffix, L".csh" ) )
				{
					desc = COMPLETE_EXEC_SHELL_DESC;
				}
				else if( suffix_compare( suffix, L".fish" ) )
				{
					desc = COMPLETE_EXEC_SHELL_DESC;
				}
				else if( suffix_compare( suffix, L".py" ) )
				{
					desc = COMPLETE_EXEC_PYTHON_DESC;
				}
				else if( suffix_compare( suffix, L".pl" ) )
				{
					desc = COMPLETE_EXEC_PERL_DESC;
				}
				else if( suffix_compare( suffix, L".php" ) )
				{
					desc = COMPLETE_EXEC_PHP_DESC;
				}
				else if( suffix_compare( suffix, L".rb" ) )
				{
					desc = COMPLETE_EXEC_RUBY_DESC;
				}
				else if( suffix_compare( suffix, L".tcl" ) )
				{
					desc = COMPLETE_EXEC_TCL_DESC;
				}
			}		   
		}		

		if( S_ISCHR(buf.st_mode) )
			desc= COMPLETE_CHAR_DESC;
		else if( S_ISBLK(buf.st_mode) )
			desc = COMPLETE_BLOCK_DESC;
		else if( S_ISFIFO(buf.st_mode) )
			desc = COMPLETE_FIFO_DESC;
		else if( S_ISLNK(buf.st_mode))
		{
			struct stat buf2;
			desc = COMPLETE_SYMLINK_DESC;

			if( waccess( filename, X_OK ) == 0 )
				desc = COMPLETE_EXEC_LINK_DESC;

			if( wstat( filename, &buf2 ) == 0 )
			{
				if( S_ISDIR(buf2.st_mode) )
				{ 
					desc = L"/" COMPLETE_SYMLINK_DESC;
				}
			}			
			else
			{
				switch( errno )
				{
					case ENOENT:
						desc = COMPLETE_ROTTEN_SYMLINK_DESC;
						break;
						
					case EACCES:
						break;

					default:
						wperror( L"stat" );
						break;
				}
			}
		}
		else if( S_ISSOCK(buf.st_mode))
			desc= COMPLETE_SOCKET_DESC;
		else if( S_ISDIR(buf.st_mode) )
			desc= L"/" COMPLETE_DIRECTORY_DESC;
	}
/*	else 
	{

		switch( errno )
		{
			case EACCES:
				break;
				
			default:
				fprintf( stderr, L"The following error happened on file %ls\n", filename );
				wperror( L"lstat" );
				break;
		}
	}
*/
	if( desc == COMPLETE_FILE_DESC )
	{
		wchar_t *suffix = wcsrchr( filename, L'.' );
		if( suffix != 0 )
		{
			if( !wcsrchr( suffix, L'/' ) )
			{
				desc = complete_get_desc_suffix( suffix );
			}
		}
	}

	return desc;	
}

/**
   Copy any strings in possible_comp which have the specified prefix
   to the list comp_out. The prefix may contain wildcards. If there is
   no embedded description in str, add the description desc.

   \param comp_out the destination list
   \param wc_escaped the prefix, possibly containing wildcards. The wildcard should not have been unescaped, i.e. '*' should be used for any string, not the ANY_STRING character.
   \param desc the default description, used for completions with no embedded description
   \param possible_comp the list of possible completions to iterate over
*/
static void copy_strings_with_prefix( array_list_t *comp_out,
									  const wchar_t *wc_escaped,
									  const wchar_t *desc,
									  const wchar_t *(*desc_func)(const wchar_t *),
									  array_list_t *possible_comp )
{
	int i;
	wchar_t *wc;

	wc = expand_one( wcsdup(wc_escaped), EXPAND_SKIP_SUBSHELL | EXPAND_SKIP_WILDCARDS);
	if(!wc)
		return;
	
	if( wc[0] == L'~' )
	{
		wc=expand_tilde(wc);
	}

//	int str_len = wcslen( str );
	for( i=0; i<al_get_count( possible_comp ); i++ )
	{
		wchar_t *next_str = (wchar_t *)al_get( possible_comp, i );
//		fwprintf( stderr, L"try wc %ls against %ls\n", wc, next_str );
		
		wildcard_complete( next_str, wc, desc, desc_func, comp_out );
	}
	free( wc );
	
}

/**
   If sufficiently few entries are listed in the list \c comp, substitute
   the description with the whatis information for the executable.
*/
static void complete_cmd_desc( const wchar_t *cmd, array_list_t *comp )
{
	int i;
	const wchar_t *cmd_start=wcschr(cmd, L'/') != 0?wcsrchr(cmd, L'/')+1:cmd;
	wchar_t *whatis_path = env_get( L"__fish_whatis_path" );
	
	/* If there are too many executables, the lookup will take too much time and the descriptions won't fit on screen anyway, so we skip them */
	if( al_get_count( comp ) > mini((reader_get_height()*3),4) )
		return;
	/* If we do not know the location of the whatis database, we skip this */
	if( !whatis_path || !wcslen(whatis_path))
		return;

	for( i=0; i<al_get_count(comp); i++ )
	{
		wchar_t *old_element = (wchar_t *)al_get( comp, i );
		if( wildcard_has( cmd_start, 0 ) )
		{
			continue;
		}
		
		wchar_t *whole_cmd = wcsdupcat( cmd_start, old_element );
		wchar_t *cmd_end = wcschr( whole_cmd, COMPLETE_SEP );
		if( cmd_end != 0 )
			*cmd_end = 0;
		
		wchar_t expr[256];
//		fwprintf( stderr, L"cmd %ls\n", whole_cmd );
	
		swprintf( expr, 256, L"grep '^%ls.*([18nol])' '%ls' |cut -d \\) -f 2- -s", whole_cmd, whatis_path );
		
		array_list_t l;
		al_init( &l );
//		fwprintf( stderr, L"subshell: \'%ls\'\n", expr );
		exec_subshell( expr, &l );
		if( al_get_count( &l ) > 0 )
		{
			wchar_t * new_desc = (wchar_t *)al_get( &l, 0 );
			/* 
			   If the new description is too wide (wider than a third
			   of the screen) and we have a lot of results (more than
			   one pagefull), the new description probably won't fit
			   on screen, so we skip it.
			*/
//			if( ( wcslen( new_desc ) < (reader_get_width()/3)) || al_get_count( comp) < (reader_get_height()-1))
//			{
				/*
				  Construct a new completion, with only the description changed
				*/
				wchar_t * new_element = malloc( sizeof(wchar_t)*(wcslen(old_element) - wcslen( L"Executable" ) + wcslen( new_desc ) +1 ) );
				wcslcpy( new_element, old_element, wcschr( old_element, COMPLETE_SEP ) - old_element +1 );
				wcscat( new_element, COMPLETE_SEP_STR );
				wcscat( new_element, &new_desc[4] );
				free( old_element );
				al_set( comp, i, new_element );
//			}
//			fwprintf( stderr, L"set desc\n" );
		}
		al_foreach( &l, (void (*)(const void *))&free );
		al_destroy( &l );
		free( whole_cmd );		
	}
}

/**
   Complete the specified command name
*/
static void complete_cmd( const wchar_t *cmd, 
						  array_list_t *comp )
{
	if( wcschr( cmd, L'/') != 0 )
	{
		int i;
		array_list_t tmp;
		al_init( &tmp );
		
		expand_file_completion( wcsdup(cmd), &tmp );
			
		for( i=0; i<al_get_count(&tmp); i++ )
		{
			wchar_t *nxt = (wchar_t *)al_get( &tmp, i );
			wchar_t *desc = wcsrchr( nxt, COMPLETE_SEP );
			
//			fwprintf( stderr, L"DESC: %ls\n", desc );	

			int is_valid =  (wcsstr( nxt, L"/" COMPLETE_SEP_STR )!=0) ||
				(desc && hash_get( &executable_desc_hash, desc ) );
			if( is_valid )
			{

				
				al_push( comp, nxt );
			}
			else
			{
				free(nxt);
			}
		}
		complete_cmd_desc( cmd, comp );
		al_destroy( &tmp );
	}
	else
	{
		wchar_t *path = env_get(L"PATH");
		wchar_t *path_cpy = wcsdup( path );
		wchar_t *nxt_path;
		wchar_t *state;
		
		array_list_t possible_comp;
		
		for( nxt_path = wcstok( path_cpy, ARRAY_SEP_STR, &state );
			 nxt_path != 0;
			 nxt_path = wcstok( 0, ARRAY_SEP_STR, &state) )
		{
			int i;
			array_list_t tmp;
			wchar_t *nxt_completion= 
				wcsdupcat2( nxt_path, 
							(nxt_path[wcslen(nxt_path)-1]==L'/'?L"":L"/"),
							cmd,
							0 );
			if( ! nxt_completion )
				continue;
			
			

			al_init( &tmp );
			
			expand_file_completion( nxt_completion, &tmp );
			
			for( i=0; i<al_get_count(&tmp); i++ )
			{
				wchar_t *nxt = (wchar_t *)al_get( &tmp, i );
				wchar_t *desc = wcsrchr( nxt, COMPLETE_SEP );
				int is_valid = (desc && hash_get( &executable_desc_hash, desc ));
				if( is_valid )
				{
					al_push( comp, nxt );
				}
				else
				{
					free(nxt);
				}
			}
			
			al_destroy( &tmp );

		}
		free( path_cpy );


		complete_cmd_desc( cmd, comp );
		
		/*
		  These return the original strings - don't free them 
		*/

		al_init( &possible_comp );
		function_get_names( &possible_comp ); 
		copy_strings_with_prefix( comp, cmd, COMPLETE_FUNCTION_DESC, &function_get_desc, &possible_comp );
		al_truncate( &possible_comp, 0 );		

		builtin_get_names( &possible_comp );
		copy_strings_with_prefix( comp, cmd, COMPLETE_BUILTIN_DESC, &builtin_get_desc, &possible_comp );
		al_destroy( &possible_comp );

	}
	

	{
		wchar_t *cdpath = env_get(L"CDPATH");
		
		wchar_t *cdpath_cpy = wcsdup( cdpath?cdpath:L"." );
		wchar_t *nxt_path;
		wchar_t *state;
//		fwprintf( stderr, L"cdpath %ls\n", cdpath_cpy );
		/*
		  Tab complete implicit cd for directories in CDPATH
		*/
		for( nxt_path = wcstok( cdpath_cpy, ARRAY_SEP_STR, &state );
			 nxt_path != 0;
			 nxt_path = wcstok( 0, ARRAY_SEP_STR, &state) )
		{
			int i;
			array_list_t tmp;
			wchar_t *nxt_completion= 
				wcsdupcat2( nxt_path, 
							(nxt_path[wcslen(nxt_path)-1]==L'/'?L"":L"/"),
							cmd,
							0 );
			if( ! nxt_completion )
				continue;			

			al_init( &tmp );
			
			expand_file_completion( nxt_completion, &tmp );
			
			for( i=0; i<al_get_count(&tmp); i++ )
			{
				wchar_t *nxt = (wchar_t *)al_get( &tmp, i );
				
				wchar_t *desc = wcsrchr( nxt, COMPLETE_SEP );
				int is_valid = (desc && (wcscmp(desc, COMPLETE_DIRECTORY_DESC)==0));
				if( is_valid )
				{
					al_push( comp, nxt );
				}
				else
				{
					free(nxt);
				}
			}
			
			al_destroy( &tmp );

		}


		free( cdpath_cpy );
	}

}

/**
   Evaluate the argument list (as supplied by complete -a) and insert any return matching completions.

   \param str The string to complete.
   \param args The list of option arguments to be evaluated.
   \param desc Description of the completion
   \param comp_out The list into which the results will be inserted
*/
static void complete_from_args( wchar_t *str,
								wchar_t *args,
								wchar_t *desc,
								array_list_t *comp_out )
{
	array_list_t possible_comp;
	al_init( &possible_comp );
	
	eval_args( args, &possible_comp );

	copy_strings_with_prefix( comp_out, str, desc, 0, &possible_comp );		
	
	al_foreach( &possible_comp, (void (*)(const void *))&free );
	al_destroy( &possible_comp );	
}

/**
   Match against an old style long option
*/
static int param_match_old( complete_entry_opt *e, 
							wchar_t *optstr )
{
	return  (optstr[0] == L'-') && (wcscmp( e->long_opt, &optstr[1] ) == 0);
}

/** 
	Match a parameter 
*/ 
static int param_match( complete_entry_opt *e, 
						wchar_t *optstr,
						int old_mode )
{
	if( e->short_opt != L'\0' && 
		e->short_opt == optstr[1] )
		return 1;

	if( !old_mode && (wcsncmp( L"--", optstr, 2 ) == 0 ))
	{
		if( wcscmp( e->long_opt, &optstr[2] ) == 0 )
		{
			return 1;
		}
	}
	
	return 0;
}

/** 
	Test if a string is an option with an argument, like --color=auto or -I/usr/include
*/ 
static wchar_t *param_match2( complete_entry_opt *e, 
							  wchar_t *optstr,
							  int old_mode )
{	
	if( e->short_opt != L'\0' && e->short_opt == optstr[1] )
		return &optstr[2];
	if( !old_mode && (wcsncmp( L"--", optstr, 2 ) == 0) )
	{
		int len = wcslen( e->long_opt );
		
		if( wcsncmp( e->long_opt, &optstr[2],len ) == 0 )
		{
			if( optstr[len+2] == L'=' )
				return &optstr[len+3];
		}
	}
	return 0;	
}

/**
   Tests whether a short option is a viable completion
*/
static int short_ok( wchar_t *arg,
					 wchar_t nextopt, 
					 wchar_t *allopt )
{
	wchar_t *ptr;
	
	if( arg[0] != L'-')
		return arg[0] == L'\0';
	if( arg[1] == L'-' )
		return 0;

	if( wcschr( arg, nextopt ) != 0 )
		return 0;
	
	for( ptr = arg+1; *ptr; ptr++ )
	{
		wchar_t *tmp = wcschr( allopt, *ptr );
		/* Unknown option */
		if( tmp == 0 )
		{
			/*fwprintf( stderr, L"Unknown option %lc", *ptr );*/
			
			return 0;
		}
		
		if( *(tmp+1) == L':' )
		{
/*			fwprintf( stderr, L"Woot %ls", allopt );*/
			return 0;
		}
		
	}	
	
	return 1;
}

/**
   Find completion for the argument str of command cmd_orig with
   previous option popt. Insert results into comp_out. Return 0 if file
   completion should be disabled, 1 otherwise.
*/
static int complete_param( wchar_t *cmd_orig, 
						   wchar_t *popt, 
						   wchar_t *str,
						   array_list_t *comp_out )
{
	complete_entry *i;
	complete_entry_opt *o;
	
	array_list_t matches;
	wchar_t *cmd, *path;
	parse_cmd_string( cmd_orig, &path, &cmd );

	al_init( &matches );

	int use_common=1, use_files=1;

	for( i=first_entry; i; i=i->next )
	{
		wchar_t *match = i->cmd_type?path:cmd;

		if( ( (!wildcard_match( match, i->cmd, 0 ) ) ) )
			continue;
		
/*		wprintf( L"Found matching command %ls\n", i->cmd );		*/
		
		use_common=1;
		if( str[0] == L'-' )
		{
			/* Check if we are entering a combined option and argument
			 * (like --color=auto or -I/usr/include) */
			for( o = i->first_option; o; o=o->next )
			{
				wchar_t *arg;
				if( (arg=param_match2( o, str, i->old_mode ))!=0 )
				{
/*					wprintf( L"Use option with desc %ls\n", o->desc );		*/
					use_common &= ((o->result_mode & NO_COMMON )==0);
					use_files &= ((o->result_mode & NO_FILES )==0);
					complete_from_args( arg, o->comp, o->desc, comp_out );
				}
			}			
		}
		else if( popt[0] == L'-' )
		{
			/* Check if the previous option has any specified
			 * arguments to match against */
			int found_old = 0;

			/* 
			   If we are using old style long options, check for them
			   first
			*/
			if( i->old_mode )
			{
				for( o = i->first_option; o; o=o->next )
				{		
					if( param_match_old( o, popt ) )
					{
						found_old = 1;
						use_common &= ((o->result_mode & NO_COMMON )==0);
						use_files &= ((o->result_mode & NO_FILES )==0);
						complete_from_args( str, o->comp, o->desc, comp_out );
					}
				}
			}
			
			/* 
			   No old style option matched, or we are not using old
			   style options. We check if any short (or gnu style
			   options do.
			*/
			if( !found_old )
			{
				for( o = i->first_option; o; o=o->next )
				{		
					if( param_match( o, popt, i->old_mode ) )
					{
						use_common &= ((o->result_mode & NO_COMMON )==0);
						use_files &= ((o->result_mode & NO_FILES )==0);
						complete_from_args( str, o->comp, o->desc, comp_out );
					}
				}
			}
		}
		
		if( use_common )
		{
			
			for( o = i->first_option; o; o=o->next )
			{
				/*
				  If this entry is for the base command, 
				  check if any of the arguments match
				*/

				if( (o->short_opt == L'\0' ) && (o->long_opt[0]==L'\0'))
				{
					use_files &= ((o->result_mode & NO_FILES )==0);
					complete_from_args( str, o->comp, o->desc, comp_out );
				}
				
				if( wcslen(str) > 0 )
				{
					/*
					  Check if the short style option matches
					*/
					if( o->short_opt != L'\0' && 
						short_ok( str, o->short_opt, i->short_opt_str ) )
					{
						wchar_t *next_opt = 
							malloc( sizeof(wchar_t)*(2 + wcslen(o->desc)));
						next_opt[0]=o->short_opt;
						next_opt[1]=L'\0';
						wcscat( next_opt, o->desc );
						al_push( comp_out, next_opt );
					}
					
					/*
					  Check if the long style option matches
					*/
					if( o->long_opt[0] != L'\0' )
					{											
						string_buffer_t whole_opt;
						sb_init( &whole_opt );
						sb_append2( &whole_opt, i->old_mode?L"-":L"--", o->long_opt, 0 );
						
						if( wcsncmp( str, (wchar_t *)whole_opt.buff, wcslen(str) )==0)
						{
							/*
							  If this option requires an argument, we add an '=' to the end of the completion
							*/
							if( (o->result_mode & NO_COMMON ) && (!i->old_mode))
							{
								sb_append( &whole_opt, L"=" );
							}
							
							/* 
							   A longopt which is longer than 10
							   characters should be self-describing,
							   so we skip the description to save
							   screen real estate
							*/
							if( wcslen( o->long_opt ) < 10 )
							{
								sb_append( &whole_opt, o->desc ); 
							}

							al_push( comp_out, 
									 wcsdup(&((wchar_t *)whole_opt.buff)[wcslen(str)]) );						
					
							
						}
						sb_destroy( &whole_opt );
						
					}
				}
			}			
		}		
	}
	free( path );
	return use_files;
}

/**
   Perform file completion on the specified string
*/
static void complete_param_file( wchar_t *str, array_list_t *comp_out )
{
	wchar_t *comp_str;
	if( (wcsncmp( str, L"--", 2 )) == 0 && (comp_str = wcschr(str, L'=' ) ) )
	{
		comp_str++;
	}
	else
		comp_str = str;
	
	expand_file_completion( wcsdup(comp_str), comp_out );
}

/**
   Complete the specified string as an environment variable
*/
static int complete_variable( const wchar_t *var, array_list_t *comp )
{
	int i;
	int varlen = wcslen( var );
	int res = 0;
	
	array_list_t names;
	al_init( &names );
	env_get_names( &names, 0 );
/*	wprintf( L"Search string %ls\n", var );*/
	
/*	wprintf( L"Got %d variables\n", al_get_count( &names ) );*/
	for( i=0; i<al_get_count( &names ); i++ )
	{
		wchar_t *name = (wchar_t *)al_get( &names, i );
		int namelen = wcslen( name );
		if( varlen > namelen )
			continue;

/*		wprintf( L"Try %ls\n", name );*/

		if( wcsncmp( var, name, varlen) == 0 )
		{
			wchar_t *value = expand_escape_variable( env_get( name ));
			wchar_t *blarg;
			/*
			  What should the description of the variable be?

			  If the variable doesn't have a value, or of the value is
			  really long, we just describe it as 'Variable', but if
			  the value is 1..16 characters long, we describe it as
			  'Variable: VALUE'.
			*/
/*			if( wcslen(value) < 1 || wcslen(value) > 16 ) 
			{
				blarg = wcsdupcat( &name[varlen], COMPLETE_VAR_DESC );
			}
			else
			{*/
				blarg = wcsdupcat2( &name[varlen], COMPLETE_VAR_DESC_VAL, value, 0 );				
//			}
			
			if( blarg )
			{
				res =1;
				al_push( comp, blarg );
			}
			free( value );
			 
		}
	}

	al_destroy( &names );
	return res;
}

/**
   Search the specified string for the \$ sign, try to complete as an environment variable
*/
static int try_complete_variable( const wchar_t *cmd, array_list_t *comp )
{
	int len = wcslen( cmd );
	int i;
	
	for( i=len-1; i>=0; i-- )
	{
		if( cmd[i] == L'$' )
		{
/*			wprintf( L"Var prefix \'%ls\'\n", &cmd[i+1] );*/
			return complete_variable( &cmd[i+1], comp );
		}
		if( !isalnum(cmd[i]) && cmd[i]!=L'_' )
		{
			return 0;
		}
		
	}
	return 0;
	
	
}

/**
   Try to complete the specified string as a username. This is used by ~USER type expansion.
*/

static int try_complete_user( const wchar_t *cmd, array_list_t *comp )
{
	const wchar_t *first_char=0;
	const wchar_t *p;
	int mode = 0;
	int res = 0;
	
	
	for( p=cmd; *p; p++ )
	{
		switch( mode )
		{
			/*Between parameters*/
			case 0:
				switch( *p )
				{
					case L'\"':
						mode=2;
						p++;
						first_char = p;
						break;
					case L' ':
					case L'\t':
					case L'\n':
					case L'\r':
						break;
					default:
						mode=1;
						first_char = p;
				}
				break;
				/*Inside non-quoted parameter*/
			case 1:
				switch( *p )
				{
					case L' ':
					case L'\t':
					case L'\n':
					case L'\r':
						mode = 0;
						break;					
				}
				break;
			case 2:
				switch( *p )
				{
					case L'\"':
						if( *(p-1) != L'\\' )
							mode =0;
						break;
				}
				break;
		}
	}
	
	if( mode != 0 )
	{
		if( *first_char ==L'~' )
		{
			const wchar_t *user_name = first_char+1;
			wchar_t *name_end = wcschr( user_name, L'~' );
			if( name_end == 0 )
			{
				struct passwd *pw;
				int name_len = wcslen( user_name );

/*				wprintf( L"Complete name \'%ls\'\n", user_name );*/

				setpwent();

				while((pw=getpwent()) != 0)
				{
/*					wprintf( L"Try %ls\n", pw->pw_name );*/
					wchar_t *pw_name = str2wcs( pw->pw_name );
					if( pw_name )
					{
						if( wcsncmp( user_name, pw_name, name_len )==0 )
						{
							wchar_t *blarg = wcsdupcat2( &pw_name[name_len], L"/", COMPLETE_USER_DESC, 0 );
							if( blarg != 0 )
							{
								al_push( comp, blarg );
								res=1;
							}
						}
						free( pw_name );
					}
				}
			
				endpwent();
				
			}
			
		}
		
		
	}
		
	return res;
}

void complete( const wchar_t *cmd,
			   array_list_t *comp )
{
	if( try_complete_variable( cmd, comp ))
	{
		return;
	}
		
	if( try_complete_user( cmd, comp ))
	{
		return;
	}
		
	if( parser_check_command( cmd, wcslen( cmd ) ) )
	{
		int on_command = env_get( PARSER_CURSOR_ON_COMMAND) != 0;
		
		if( on_command )
		{
			/* Complete command filename */
			complete_cmd( env_get( PARSER_END_STR), 
						  comp );
		}
		else
		{
			/*
			  Complete parameter. Parameter expansion should be
			  performed against both the globbed and the unglobbed
			  version of the string, so we create a list containing
			  all possible versions of the string that is to be
			  expanded. This is potentially very slow.
			*/
			
			int cmd_ok = 1;
			int do_file = 1;

			wchar_t *end_str = env_get(PARSER_END_STR );

			/* 
			   If the command is an function, we use the
			   completions of the first command in the function
			   and hope for the best...
			*/
			if( function_exists( env_get( PARSER_COMMAND) ) )
			{
				tokenizer tok2;
				tok_init( &tok2, function_get_definition( env_get( PARSER_COMMAND ) ), 0 );
				wchar_t *new_cmd=0; 
				
				switch( tok_last_type( &tok2 ) )
				{
					case TOK_STRING:
						new_cmd = expand_one( wcsdup(tok_last( &tok2 )), 
											  EXPAND_SKIP_SUBSHELL | EXPAND_SKIP_VARIABLES );
						break;
					default:
						cmd_ok = 0;
						break;
				}
				
				tok_destroy( &tok2 );
				
				if( cmd_ok )
				{
					do_file &= complete_param( new_cmd, 
											   env_get( PARSER_PREV_STR ),
											   end_str,
											   comp );	
				}
				if( new_cmd != 0 )
					free(new_cmd);			
			}
			//				fwprintf( stderr, L"complete_param with end_str %ls\n", end_str );
			do_file &= complete_param( env_get( PARSER_COMMAND ), env_get( PARSER_PREV_STR ), end_str, comp );

			if( do_file )
				complete_param_file( env_get( PARSER_END_STR ), comp );
		}
		env_pop();

	}
}

void complete_print( string_buffer_t *out )
{
	complete_entry *e;
	for( e = first_entry; e; e=e->next )
	{
		complete_entry_opt *o;		
		for( o= e->first_option; o; o=o->next )
		{
			wchar_t *modestr[] = 
				{
					L"",
					L" --no-files",
					L" --require-parameter",
					L" --exclusive"
				}
			;
	
			wchar_t sopt[2];
			sopt[0]= (o->short_opt==L'\0' ? L'\0' :o->short_opt);
			sopt[1]=L'\0';
			 
			sb_append2( out,
						L"complete",
						e->cmd_type?L" --path ":L" --command ",
						L"\"",
						e->cmd,
						L"\"",
						o->short_opt==L'\0' ? L"": L" -s ",
						sopt,
						wcslen(o->long_opt)==0?L"":(e->old_mode?L" --old-option \"":L" --long-option \""),
						o->long_opt,
						wcslen(o->long_opt)==0?L"":L"\"",
						modestr[o->result_mode],
						L" --description \"",
						*o->desc?o->desc+1:o->desc,
						L"\"",
						wcslen(o->comp)>0?L" --args \"":L"",
						o->comp,
						wcslen(o->comp)>0?L"\"":L"",
						L"\n",
						0);

		}
	}	
}
