/** \file exec.c
	Functions for executing a program.

	Some of the code in this file is based on code from the Glibc manual.
*/

#include <stdlib.h>
#include <stdio.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <termios.h>
#include <unistd.h>
#include <fcntl.h>
#include <errno.h>
#include <wchar.h>
#include <string.h>
#include <limits.h>
#include <signal.h>
#include <sys/wait.h>
#include <assert.h>
#include <dirent.h>

#include "config.h"
#include "util.h"
#include "common.h"
#include "wutil.h"
#include "proc.h"
#include "exec.h"
#include "parser.h"
#include "builtin.h"
#include "function.h"
#include "env.h"
#include "wildcard.h"

/**
   Prototype for the getpgid library function.
*/
pid_t getpgid(pid_t pid);


/**
  This function is executed by the child process created by a call to
  fork().  This function begins by updating FDs as specified by the io
  parameter.  If the command requested for this process is not a
  builtin, execv is then called with the appropriate parameters. If
  the command is a builtin, the contents of out and err are printed.
*/
static void launch_process (process_t *p, 
							pid_t pgid,
							io_data_t *io,
							int foreground,
							wchar_t *out,
							wchar_t *err )
{
	pid_t pid;
	if( shell_is_interactive )
    {
		/* 
		   Put the process into the process group and give the process group
		   the terminal, if appropriate.
		   This has to be done both by the shell and in the individual
		   child processes because of potential race conditions.  
		*/
		pid = getpid ();
		if (pgid == 0)
			pgid = pid;

		/* Wait till shell puts os in our own group */
		while( getpgrp() != pgid )
			sleep(0);

		/* Wait till shell gives us stdin */
		if (foreground)
		{
			while( tcgetpgrp( 0 ) != pgid )
				sleep(0);
		}
		
		/* Set the handling for job control signals back to the default.  */
		signal (SIGINT, SIG_DFL);
		signal (SIGQUIT, SIG_DFL);
		signal (SIGTSTP, SIG_DFL);
		signal (SIGTTIN, SIG_DFL);
		signal (SIGTTOU, SIG_DFL);
		signal (SIGCHLD, SIG_DFL);
    }
	
	/* Set the standard input/output channels of the new process.  */

	for( ; io; io=io->next )
	{
		int tmp;
		
		if( close(io->fd) == -1 )
		{
			wperror( L"close" );
			exit(1);
		}
		
		switch( io->io_mode )
		{
			case IO_CLOSE:
				break;
			case IO_FILE:
				if( (tmp=wopen( io->filename, io->flags, 0666 ))==-1 )
				{
					fwprintf( stderr, 
							  L"fish: An error occurred while redirecting file descriptor %d, "
							  L"the process %ls will not be run\n", 
							  io->fd,
							  p->actual_cmd );
					
					wperror( L"open" );
					exit(1);
				}				
				else if( tmp != io->fd)
				{
					if(dup2( tmp, io->fd ) == -1 )
					{
						wperror( L"dup2" );
						exit(1);
					}
					if( close( tmp ) == -1 )
					{
						wperror( L"close" );
						exit(1);
					}
				}				
				break;
			case IO_FD:
				if( dup2( io->old_fd, io->fd ) == -1 )
				{
					wperror( L"dup2" );
					exit(1);					
				}
				break;
				
			case IO_BUFFER:
				
/*				fwprintf( stderr, 
						  L"Buffer fd %d in process %ls (%d) (using fd %d)\n", 
						  io->fd, 
						  p->actual_cmd,
						  p->pid,
						  io->pipe_fd[io->fd] );
*/			
			case IO_PIPE:
/*				fwprintf( stderr, L"Pipe fd %d in process %ls (%d) (Through fd %d)\n", 
						  io->fd, 
						  p->actual_cmd,
						  p->pid,
						  io->pipe_fd[io->fd] );
*/
				if( dup2( io->pipe_fd[io->fd], io->fd ) == -1 )
				{
					wperror( L"dup2" );
					exit(1);
				}
				close( io->pipe_fd[io->fd] );
				
/*				
				if( close( io[i].pipe_fd[ io->fd ] ) == -1 )
				{
					wperror( L"close" );
					exit(1);
				}
*/			
/*				if( close( io[i].pipe_fd[1] ) == -1 )
				{
					wperror( L"close" );
					exit(1);
				}
*/
		
/*		    fprintf( stderr, "fd %d points to fd %d\n", i, io[i].fd[i>0?1:0] );*/
				break;
		}
	}
	
	if( p->builtin )
	{
		if( out != 0 )
			fwprintf( stdout, L"%ls", out );
				
		if( err != 0 )
			fwprintf( stderr, L"%ls", err );
/*		fflush( stdout);
		fflush(stderr);
		fsync(1);
		fsync(2);*/
		exit(0);
	}
	else
	{
		/* 
		   Make sure the shell gets a decent chance to set up groups,
		   etc, correctly before calling exec, since the new program
		   may do all sorts of strangeness. 

		*/

		/*  Exec the new process. Make sure we exit. */
		execve (wcs2str(p->actual_cmd), wcsv2strv( (const wchar_t **) p->argv), env_export_arr() );
		wperror( L"execve" );
		exit(1);
	}
}

/**
   Specialized reader for reading from subshells. Since a subshell can
   fork, and a child of the subshell may continue to use the fd of the
   subshell perpetually, we need to stop waiting for new data to
   become available whenever the subshell finishes. Therefore a
   specialized reader is required.
*/

static int read_subshell( int fd, subshell *src, array_list_t *l )
{
	int i=0;
	int c=EOF;

	char *buff=0;
	int buff_size=0;

	sigset_t sigset; 
	sigemptyset( &sigset );
	sigaddset( &sigset, SIGCHLD );
	
	while( 1 )
	{
		/* Reallocate the buffer if necessary */
		if( i+1 >= buff_size )
		{
			int new_len = maxi( 128, (buff_size)*2);
			
			char *tmp = realloc( buff, sizeof(wchar_t)*new_len );
				
			if( tmp == 0 )
			{
				wperror( L"realloc" );
				
				return -1;
			}
			else
			{
				buff = tmp;
				buff_size = new_len;
			}			
		}

		while( 1 )
		{
//			fwprintf( stderr, L"a\n" );

			fd_set rfds;		
			int retval;
			
			/* Watch fd to see when it has input. */
			FD_ZERO(&rfds);
			FD_SET(fd, &rfds);
			
			if( src->completed )
			{
				struct timeval tv;			
				tv.tv_sec = 0;
				tv.tv_usec = 0;
				retval = select(fd+1, &rfds, NULL, NULL, &tv);				
			}
			else
			{
				retval = select(fd+1, &rfds, NULL, NULL, 0);
			}
			
			/* Don’t rely on the value of tv now! */
			if (retval == -1)
			{
				if( errno == EINTR )
				{
//					wprintf(L"Interrupted, try again\n");
					if( src->completed )
					{
						c = EOF;
						break;
					}
					else
						continue;
				}
				wperror(L"select()");
				c = EOF;
			}
			else if (retval)
			{
				/*
				  Data is available now.
				*/
				/* FD_ISSET(0, &rfds) will be true. */
				while(1)
				{
					char chr=0;
					
					sigprocmask(SIG_BLOCK, &sigset, 0);
					int res=read( fd, &chr, 1 );
					sigprocmask(SIG_UNBLOCK, &sigset, 0);
					
					if( res == 0 )
					{
						c = EOF;
						break;
					}
					else if( res == -1 )
					{
						if( errno == EINTR )
						{
							/* If a character was read, we use it even if we were interrupted */
							if( chr )
								c = chr;
							else
								continue;
						}
						else
							c = EOF;
					}
					else
						c = chr;
					break;	
				}
			}
			else
			{
				/* 
				   retval == 0, which means we called it with a
				   timeout. We only use a timeout if the subshell has
				   finished, and 0 means no data is available. This
				   means we have reached the end of the stream.
				 */
				c=EOF;
			}
			
			break;
		}
	
		switch( c )
		{
			/* End of line */ 
			case EOF:
			case '\n':
			case '\0':
			{
				wchar_t *wbuff;
				
				buff[i]=L'\0';
				if( strlen( buff ) > 0 )
				{
					wbuff =  str2wcs( buff );
					if( wbuff )
						al_push( l, wbuff );				
				}
				
//				fwprintf( stderr, L"Add line %ls\n", wbuff );
								
				if( c == EOF )
				{
					free( buff );
					return 0;
				}
				i=0;
			}
				/* Ignore carriage returns */
			case '\r':
				break;
				
			default:
				buff[i++]=c;
				break;
		}		
	}	
}

void ggg( wchar_t *s, io_data_t *d )
{
	if( s )
		fwprintf( stderr, L"%ls: ", s );

	if( d )
	{
		wchar_t *g[]=
			{
				L"file",
				L"pipe",
				L"fd",
				L"buffer",
				L"close"
			}
		;
		
		fwprintf( stderr, L"[fd: %d, type:%ls] ", d->fd, g[d->io_mode] );
		ggg( 0, d->next );
	}
	else
	{
		fwprintf( stderr, L"\n" );
	}	
}


/**
   Check if the IO redirection chains contains redirections for the specified file descriptor
*/

static int has_fd( io_data_t *d, int fd )
{
	if( d==0)
		return 0;
	if( d->fd == fd )
		return 1;
	return has_fd(d->next, fd);
}



void exec( job_t *j )
{
	process_t *p;
	pid_t pid;
	int mypipe[2];
	sigset_t chldset; 
	sigemptyset( &chldset );
	sigaddset( &chldset, SIGCHLD );
	int skip_fork;
	
	/* This call is used so the global environment variable array is regenerated, if needed, before the fork. That way, we avoid a lot of duplicate work where EVERY child would need to generate it */
	env_export_arr();

	io_data_t pipe_read, pipe_write, *buffer_redirect=0;

	pipe_read.fd=0;
	pipe_write.fd=1;
	pipe_read.io_mode=IO_PIPE;
	pipe_read.pipe_fd[0] = -1;
	pipe_write.io_mode=IO_PIPE;
	pipe_read.next=0;
	pipe_write.next=0;

//	ggg( j->command, j->io );

	//fwprintf( stderr, L"Run command %ls\n", j->command );
		
	if( current_block->out_buffer && !has_fd( j->io, 1) )
	{
		buffer_redirect=malloc(sizeof(io_data_t));
		
		buffer_redirect->io_mode=IO_BUFFER;
		buffer_redirect->next=0;
		buffer_redirect->out_buffer=current_block->out_buffer;
		buffer_redirect->fd=1;
		
		if( pipe( buffer_redirect->pipe_fd ) == -1 )
		{
			wperror (L"pipe");
			exit(1);
		}
		
//		fwprintf( stderr, L"We have an output buffer for job %ls\n", j->command );		
	}
	
/*
	if true;
	    read foo; read bar; read baz;
	end <foo.txt
*/

	j->io = io_add( j->io, &pipe_write );
	
	for (p = j->first_process; p; p = p->next)
	{
		mypipe[1]=-1;
		skip_fork=0;
		
		/*
		  Set up fd:s that will be used in the pipe 
		*/
		
		if( p == j->first_process->next )
		{
			j->io = io_add( j->io, &pipe_read );
		}
		
		if (p->next)
		{
/*			fwprintf( stderr, L"Make pipe from %ls to %ls\n", p->actual_cmd, p->next->actual_cmd );*/
			if (pipe( mypipe ) == -1)
			{
				wperror (L"pipe");
				exit (1);
			}
			memcpy( pipe_write.pipe_fd, mypipe, sizeof(int)*2);
		}
		else
		{
			/*
			  This is the last element of the pipeline.
			*/

			/*
			  Remove the io redirection for pipe output.
			*/
			j->io = io_remove( j->io, &pipe_write );
			
			/*
			  If the output should be redirected to a temporary
			  buffer, add the redirection.
			*/
			if( buffer_redirect )
			{
//				fwprintf( stderr, L"Add output buffer %d to job %ls\n", buffer_redirect, j->command );
				j->io = io_add( j->io, buffer_redirect );
			}			
		}
		
		if( p->builtin ) 
		{
			FILE *builtin_stdin=0;
			if( p == j->first_process )
			{
				builtin_stdin = stdin;
			}
			else
			{
				builtin_stdin = fdopen( pipe_read.pipe_fd[0], "r");
			}
			builtin_push_io( builtin_stdin );
			
			/* 
			   Since this may be the foreground job, and since a
			   builtin may execute another foreground job, we need to
			   pretend to suspend this job while running the builtin.
			*/
			int fg = j->fg;
			j->fg = 0;

			builtin_out_redirect = has_fd( j->io, 1 );
			builtin_err_redirect = has_fd( j->io, 2 );
			

			builtin_exec( p->argv, p->builtin );			
			
			j->fg = fg;
			skip_fork =
				( sb_out->used == 0) &&
				( sb_err->used == 0) &&
				( p->next == 0 );

			if( builtin_stdin != stdin )
				fclose( builtin_stdin );

		}

		/*Temporary signal block for SIGCHLD */
		sigprocmask(SIG_BLOCK, &chldset, 0);
		
		if( skip_fork )
		{
			p->completed=1;
		}
		else
		{
//			fwprintf( stderr, 
//					  L"fork on %ls\n", j->command );
			pid = fork ();
			if (pid == 0)
			{
				/*
				  This is the child process. 
				*/
				p->pid = getpid();
				launch_process (p, 
								j->pgid, 
								j->io, 
								j->fg, 
								sb_out?(wchar_t *)sb_out->buff:0, sb_err?(wchar_t *)sb_err->buff:0 );
			}
			else if (pid < 0)
			{
				/* The fork failed. */
				wperror (L"fork");
				exit (1);
			}
			else
			{
				/* This is the parent process.  */
				p->pid = pid;
				/* 
				   If we do not output to stdout, builtin IO will not appear. 
				   I have no idea why...
				*/
				fwprintf( stdout, L"" );

				if(shell_is_interactive)
				{
					if (!j->pgid)
						j->pgid = p->pid;
					if( setpgid (p->pid, j->pgid) )
					{						
						if( getpgid( p->pid) != j->pgid )
						{
							fwprintf( stderr, 
									  L"fish: Could not send process %d from group %d to group %d\n",
									  p->pid, 
									  getpgid( p->pid),
									  j->pgid );
							wperror( L"setpgid" );
						}
					}
				}
				else
				{
					j->pgid = getpid();
				}
			}
		}

		if( p->builtin )
			builtin_pop_io();			
		
		sigprocmask(SIG_UNBLOCK, &chldset, 0);
		
		/* 
		   Close the pipe the current process uses to read from the previous process_t 
		*/
		if( pipe_read.pipe_fd[0] >= 0 )
			close( pipe_read.pipe_fd[0] );
		/* 
		   Set up the pipe the next process uses to read from the current process_t 
		*/
		if( p->next )
			pipe_read.pipe_fd[0] = mypipe[0];
		
		/* 
		   If there is a next process, close the output end of the
		   pipe (the child subprocess already has a copy of the pipe).
		*/
		if( p->next )
		{
			if( close(mypipe[1]) != 0 )
			{
				wperror( L"close" );
			}
		}		
	}
	
	if( buffer_redirect )
	{
		close( buffer_redirect->pipe_fd[1] );
	}

	j->io = io_remove( j->io, &pipe_read );
	/* This may be needed for builtins which may do strange thing with pipes. Can't hurt? */
	j->io = io_remove( j->io, &pipe_write );
//	assert( !job_is_stopped(j));
	j->constructed = 1;
//	ggg( j->command, j->io );

	job_continue (j, 0);
}

int exec_subshell( wchar_t *cmd, 
					array_list_t *l )
{
	pid_t child_pid;
	int fd[2];
	sigset_t chldset; 
	sigemptyset( &chldset );
	sigaddset( &chldset, SIGCHLD );

	subshell *s;
	s = ss_create();
	
	if( pipe(fd) == -1 )
	{
		wperror( L"pipe" );
		fwprintf( stderr, L"Subshell error\n" );
		return 0;		
	}

	sigprocmask(SIG_BLOCK, &chldset, 0);
	
	if( (child_pid = fork()) == 0 )
	{	
		/* I am the child*/
		first_job = 0;
		first_subshell=0;

		/* Make sure output goes to parent, not to a string_buffer_t. */
		current_block=0;
		
		shell_is_interactive = 0;
		first_job=0;
		
		signal (SIGINT, SIG_DFL);
		signal (SIGQUIT, SIG_DFL);
		signal (SIGTSTP, SIG_DFL);
		signal (SIGTTIN, SIG_DFL);
		signal (SIGTTOU, SIG_DFL);
		signal (SIGCHLD, SIG_DFL);
		signal (SIGIO, SIG_DFL );
		
		if( close( 1 ) == -1 )
		{
			wperror( L"close" );
			exit(1);			
		}
		
		if( dup(fd[1]) == -1 )
		{
			wperror( L"dup" );
			exit(1);
		}
		
		if( eval( cmd, 0, TOP ) )
		{
			fwprintf( stderr, L"fish: Subshell failed\n" );
		}
		
		exit(0);
	}
	else 
	{
		/*I am the parent*/
		s->pid = child_pid;
		sigprocmask(SIG_UNBLOCK, &chldset, 0);
		
		/*Fork failed*/
		if( child_pid < 0 )
		{
			ss_free( s );			
			wperror( L"fork" );
			if( close(fd[0]) == -1 )
			{
				wperror( L"close" );
			}
			if( close(fd[1]) == -1 )
			{
				wperror( L"close" );
			}
			return 0;
		}
		
//		fwprintf( stderr, L"forked\n" );
		if( close(fd[1]) == -1 )
		{
			wperror( L"close" );
			return 0;		
		}

		if( l!= 0 )
			read_subshell( fd[0], s, l );

		if( close(fd[0]) == -1 )
		{
			wperror( L"close" );
			return 0;		
		}
		
	}
	return 1;	
}

