/** \file function.c
  Functions for storing and retrieving function information.
*/
#include <stdlib.h>
#include <stdio.h>
#include <wchar.h>
#include <unistd.h>
#include <termios.h>
#include <signal.h>


#include "config.h"
#include "util.h"
#include "function.h"
#include "proc.h"
#include "parser.h"
#include "common.h"

/**
   Table containing all functions
*/
static hash_table_t function;

/**
   Struct describing an function
*/
typedef struct
{
	/** Function definition */
	wchar_t *cmd;
	wchar_t *desc;	
}
function_data;

/**
   Free all contents of an entry to the function hash table
*/
static void clear_function_entry( const void *key, 
							   const void *data )
{
	function_data *d = (function_data *)data;
	free( (void *)key);
	free( (void *)d->cmd );
	free( (void *)d->desc );
	free( (void *)d );
}

void function_init()
{
	hash_init( &function,
			   &hash_wcs_func,
			   &hash_wcs_cmp );
}

void function_destroy()
{
	hash_foreach( &function, &clear_function_entry );
	hash_destroy( &function );
}


void function_add( const wchar_t *name, 
				   const wchar_t *val,
				   const wchar_t *desc )
{
	if( function_exists( name ) )
		function_remove( name );
	
	function_data *d = malloc( sizeof( function_data ) );
	d->cmd = wcsdup( val );
	d->desc = desc?wcsdup( desc ):0;

	hash_put( &function, wcsdup(name), d );
}

int function_exists( const wchar_t *cmd )
{
	return (hash_get(&function, cmd) != 0 );
}

void function_remove( const wchar_t *name )
{
	void *key;
	function_data *d;
	if( !hash_contains( &function,
						name ) )
		return;
	hash_remove( &function,
				 name,
				 (const void **) &key,
				 (const void **)&d );
	
	free( key );
	free( d->cmd );
	free( d->desc );
	free( d );
}

	
const wchar_t *function_get_definition( const wchar_t *argv )
{
	function_data *data = 
		(function_data *)hash_get( &function, argv );
	if( data == 0 )
		return 0;
	return data->cmd;
}
	
const wchar_t *function_get_desc( const wchar_t *argv )
{
	function_data *data = 
		(function_data *)hash_get( &function, argv );
	if( data == 0 )
		return 0;
	
	return data->desc?data->desc:data->cmd;
}

void function_set_desc( const wchar_t *name, const wchar_t *desc )
{
	function_data *data = 
		(function_data *)hash_get( &function, name );
	if( data == 0 )
		return;

	data->desc =wcsdup(desc);
}

void function_get_names( array_list_t *list )
{
	hash_get_keys( &function, list );
}
