/** \file highlight.h
	Prototypes for functions for syntax highlighting
*/

/**
   Constants for various character classifications. Each character of a command string can be classified as one of the following types.
*/
enum
{
	HIGHLIGHT_NORMAL,
	HIGHLIGHT_COMMAND,
	HIGHLIGHT_SUBSHELL,
	HIGHLIGHT_REDIRECTION, 
	HIGHLIGHT_END, 
	HIGHLIGHT_ERROR,
	HIGHLIGHT_PARAM,
	HIGHLIGHT_COMMENT,
	HIGHLIGHT_MATCH,
	HIGHLIGHT_SEARCH_MATCH,
	HIGHLIGHT_PAGER_PREFIX,
	HIGHLIGHT_PAGER_COMPLETION,
	HIGHLIGHT_PAGER_DESCRIPTION,
	HIGHLIGHT_PAGER_PROGRESS
}
;

/**
   Constants for various colors as used by the set_color function. 
*/
enum
{
	FISH_COLOR_BLACK,
	FISH_COLOR_RED,
	FISH_COLOR_GREEN,
	FISH_COLOR_YELLOW,
	FISH_COLOR_BLUE,
	FISH_COLOR_MAGENTA,
	FISH_COLOR_CYAN,
	FISH_COLOR_WHITE,
	/** The default fg color of the terminal */
	FISH_COLOR_NORMAL
}
;
 
/**
   Perform syntax highlighting for the shell commands in buff. The result is
   stored in the color array as a color_code from the HIGHLIGHT_ enum
   for each character in buff.

   \param buff The buffer on which to perform syntax highlighting
   \param color The array in wchich to store the color codes. The first 8 bits are used for fg color, the next 8 bits for bg color. 
   \param pos the cursor position. Used for quote matching, etc.
   \param error a list in which a description of each error will be inserted. May be 0, in whcich case no error descriptions will be generated.
*/
void highlight_shell( wchar_t * buff, int *color, int pos, array_list_t *error );

/**
   Perform syntax highlighting for the text in buff. Matching quotes and paranthesis are highlighted. The result is
   stored in the color array as a color_code from the HIGHLIGHT_ enum
   for each character in buff.

   \param buff The buffer on which to perform syntax highlighting
   \param color The array in wchich to store the color codes. The first 8 bits are used for fg color, the next 8 bits for bg color. 
   \param pos the cursor position. Used for quote matching, etc.
   \param error a list in which a description of each error will be inserted. May be 0, in whcich case no error descriptions will be generated.
*/
void highlight_universal( wchar_t * buff, int *color, int pos, array_list_t *error );

/**
   Translate from HIGHLIGHT_* to FISH_COLOR_* according to environment
   variables. Defaults to FISH_COLOR_NORMAL.

   Example: 

   If the environment variable FISH_FISH_COLOR_ERROR is set to 'red', a
   call to highlight_get_color( HIGHLIGHT_ERROR) will return
   FISH_COLOR_RED.
*/
int highlight_get_color( int highlight );
