#
# Command completions for fish. 
#

#
# Don't need completions in non-interactive mode
#

if test -z $fish_interactive
   exit
end

#
# fish needs to know the location of the whatis database for command
# completion descriptions. We cant use apropos to find the
# descriptions, since that is a full order of magnitude
# slower. Unfortunatly, the location of the whatis database varies
# between platforms and distributions. This loop searches through some
# common places, but this isn't fool proof
#

for i in /var/cache/man/{whatis,windex} /usr{,/local}{,/share}/man/{whatis,windex} 
	if test -f $i
		set -g __fish_whatis_path $i
		break
	end
end

#
# Various options can take a list of all possible filesystems as an
# argument, we define it once
#

set filesystems "
	adfs
	affs
	autofs
	coda
	coherent
	cramfs
	devpts
	efs
	ext
	ext2
	ext3
	hfs
	hpfs
	iso9660
	jfs
	minix
	msdos
	ncpfs
	nfs
	ntfs
	proc
	qnx4
	ramfs
	reiserfs
	romfs
	smbfs
	sysv
	tmpfs
	udf
	ufs
	umsdos
	vfat
	xenix
	xfs
	xiafs
"

#
# Completions for the shell, it's builtin commands and utilities
#

complete -c fish -s c -l "command" -d "Run fish with this command" 
complete -c fish -s h -l help -d "Display help and exit"
complete -c fish -l version -d "Display version and exit"
complete -c fish -s i -l interactive -d "Run in interactive mode"

complete -c functions -s e -l erase -d "Erase function" -x -a "(function|cut -d ' ' -f 1)"
complete -c functions -s h -l help -d "Display help and exit"
complete -c functions -s d -l description -d "Set function description" -x

complete -c function -s d -l description -d "Set function description" -x

#
# This one is quite complex...
#
# It searches the CDPATH for any directories
#

complete -x -c cd -a "
(
	
	set wd $PWD

	#Check if CDPATH is set
	if test -z $CDPATH[1]
		set -g CDPATH .
	end

	if echo $end_str|grep '^/' >/dev/null
	   	# This is an absolute search path
		eval printf '\%s\\tDirectory\\n' $end_str\*/	   
	else
		# This is a relative search path
		# Iterate over every directory in CDPATH and check for possible completions
		for i in $CDPATH
			cd $wd
			cd $i
			eval printf '"%s\tDirectory in "'$i'"\n"' $end_str\*/
		end
	end
)
"
complete -c cd -s h -l help -d "Display help and exit"

complete -c complete -s c -l command -d "Command to add completion to" -r
complete -c complete -s p -l path -d "Path to add completion to"
complete -c complete -s s -l short-option -d "Posix-style option to complete"
complete -c complete -s l -l long-option -d "GNU-style option to complete"
complete -c complete -s o -l old-option -d "Old style long option to complete"
complete -c complete -s f -l no-files -d "Do not use file completion"
complete -c complete -s r -l require-parameter -d "Require parameter"
complete -c complete -s x -l exclusive -d "Require parameter and do not use file completion"
complete -c complete -s a -l arguments -d "A list of possible arguments"
complete -c complete -s d -l description -d "Description of this completions"
complete -c complete -s u -l unauthorative -d "Option list is not complete"
complete -c complete -s e -l erase -d "Remove completion"
complete -c complete -s h -l help -d "Display help and exit"

complete -c read -s x -l export -d "Export variable to subprocess"
complete -c read -s h -l help -d "Display help and exit"
complete -c read -s p -l prompt -d "Set prompt command" -x

complete -c set -s e -l erase -d "Erase variable"
complete -c set -s x -l export -d "Export variable to subprocess"
complete -c set -s g -l global -d "Make variable scope global"
complete -c set -s l -l local -d "Make variable scope local"
complete -c set -s h -l help -d "Display help and exit"
complete -c set -x -a "(set|sed -e 's/ /\tVariable: /')"

complete -c mimedb -s t -l input-file-data -d "Input is a file, use name and contents to determine mimetype"
complete -c mimedb -s f -l input-filename -d "Input is a file, use name to determine mimetype"
complete -c mimedb -s i -l input-mime -d "Input is a mimetype"
complete -c mimedb -s m -l output-mime -d "Output mimetype"
complete -c mimedb -s d -l output-description -d "Output description of mimetype"
complete -c mimedb -s a -l output-action -d "Output default action for mimetype"
complete -c mimedb -s l -l launch -d "Launch default action for each file"
complete -c mimedb -s h -l help -d "Display help and exit"
complete -c mimedb -s v -l version -d "Display version and exit"

complete -c set_color -x -d "Color" -a "black red green brown blue magenta cyan white"

complete -c tokenize -s e -l with-empty -d "Allow empty tokens"
complete -c tokenize -s n -l no-empty -d "Ignore empty tokens"
complete -c tokenize -s d -l delimiter -d "Delimiters for tokenization"


complete -c while -s h -l help -d "Display help and exit"
complete -c for  -s h -l help -d "Display help and exit"
complete -c if   -s h -l help -d "Display help and exit"
complete -c eval -s h -l help -d "Display help and exit"
complete -c command -s h -l help -d "Display help and exit"
complete -c builtin -s h -l help -d "Display help and exit"
complete -c bg   -s h -l help -d "Display help and exit"
complete -c fg   -s h -l help -d "Display help and exit"
complete -c exit -s h -l help -d "Display help and exit"
complete -c jobs -s h -l help -d "Display help and exit"


complete -c vared -x -a "(set|sed -e 's/ /\tVariable: /')"
complete -c vared -s h -l help -d "Display help and exit"


#
# Completions for the help command
#

for i in (builtin -n)
  complete -c help -x -a $i -d 'Help for the '$i' builtin' 
end

for i in syntax todo bugs history; 
  complete -c help -x -a $i -d 'Help section on '$i 
end

complete -c help -x -a completion -d "Help on how tab-completion works"
complete -c help -x -a job-control -d "Help on how job control works"
complete -c help -x -a difference -d "Summary on how fish differs from other shells"

complete -c help -x -a prompt -d "Help on how to set the prompt"
complete -c help -x -a title -d "Help on how to set the titlebar message"
complete -c help -x -a killring -d "Help on how to copy and paste"
complete -c help -x -a editor -d "Help on editor shortcuts"
complete -c help -x -a expand -d "Help on parameter expantion (Globbing)"
complete -c help -x -a globbing -d "Help on parameter expantion (Globbing)"
complete -c help -x -a variables -d "Help on environment variables"
complete -c help -x -a color -d "Help on setting syntax highlighting colors"
complete -c help -x -a prompt -d "Help on changing the prompt"
complete -c help -x -a title -d "Help on changing the titlebar messages"
complete -c help -x -a builtin-overview -d "A short summary of all builtin commands"
complete -c help -x -a changes -d "The changelog"


#
# Completions for external commands
#


# Completions for the binary calculator

complete -c bc -s i -l interactive -d "Force interactive mode"
complete -c bc -s l -l math-lib -d "Define math library"
complete -c bc -s w -l warn -d "Give warnings for extensions to POSIX bc"
complete -c bc -s s -l standard -d "Process exactly POSIX bc"
complete -c bc -s q -l quiet -d "Do not print the GNU welcome"
complete -c bc -s v -l version -d "Print version and exit"
complete -c bc -s h -l help -d "Print help and exit"


# Bittorrent commands

for i in btdownloadcurses.py btdownloadheadless.py;
	complete -c $i -l max_uploads -x -d "Maximum uploads at once"
	complete -c $i -l keepalive_interval -x -d "Number of seconds between keepalives"
	complete -c $i -l download_slice_size -x -d "Bytes per request"
	complete -c $i -l request_backlog -x -d "Requests per pipe"
	complete -c $i -l max_message_length -x -d "Maximum length prefix encoding"
	complete -c $i -l ip -d "IP to report to the tracker" -x -a "
	(
		/sbin/ifconfig |grep 'inet addr'|cut -d : -f 2|cut -d ' ' -f 1
	)
	"
	complete -c $i -l minport -d "Minimum port to listen to"
	complete -c $i -l maxport -d "Maximum port to listen to"
	complete -c $i -l responsefile -r -d "File for server response"
	complete -c $i -l url -x -d "URL to get file from"
	complete -c $i -l saveas -r -d "Local file target"
	complete -c $i -l timeout -x -d "Time to close inactive socket"
	complete -c $i -l timeout_check_interval -x -d "Time between checking timeouts"
	complete -c $i -l max_slice_length -x -d "Maximum outgoing slice length"
	complete -c $i -l max_rate_period -x -d "Maximum time to guess rate"
	complete -c $i -l bind -x -d "IP to bind to locally" -a "
	(
		/sbin/ifconfig |grep 'inet addr'|cut -d : -f 2|cut -d ' ' -f 1
	)
	"
	complete -c $i -l upload_rate_fudge -x -d "time equivalent of writing to kernel-level TCP buffer"
	complete -c $i -l display_interval -x -d "Time between screen updates"
	complete -c $i -l rerequest_interval -x -d "Time to wait between requesting more peers"
	complete -c $i -l min_peers -x -d "Minimum number of peers to not do requesting"
	complete -c $i -l http_timeout -x -d "Number of seconds before assuming http timeout"
	complete -c $i -l max_initiate -x -d "Number of peers at which to stop initiating new connections"
	complete -c $i -l max_allow_in -x -d "Maximum number of connections to allow"
	complete -c $i -l check_hashes -x -d "Whether to check hashes on disk"
	complete -c $i -l max_upload_rate -x -d "Maximum kB/s to upload at"
	complete -c $i -l snub_time -x -d "Seconds to wait for data to come in before assuming choking"
	complete -c $i -l spew -x -d "Whether to display diagnostic info"
	complete -c $i -l rarest_first_cutoff -x -d "Number of downloads at which to switch from random to rarest first"
	complete -c $i -l min_uploads -x -d "Number of uploads to fill out to with optimistic unchokes"
	complete -c $i -l report_hash_failiures -x -d "Whether to inform the user that hash failures occur"
end;

# Completions for the bzip2 family of commands

complete -c bunzip2 -x -a "$end_str*.bz $end_str*.bz2 $end_str*.tbz $end_str*.tbz2" -d "Compressed file"
complete -c bunzip2 -s f -l force -d "Overwrite"
complete -c bunzip2 -s k -l keep -d "Do not overwrite"
complete -c bunzip2 -s s -l small -d "Reduce memory usage"
complete -c bunzip2 -s v -l verbose -d "Print compression ratios"
complete -c bunzip2 -s L -l license -d "Print license"
complete -c bunzip2 -s V -l version -d "Print version"

complete -c bzip2 -s c -l stdout -d "Compress to stdout"
complete -c bzip2 -s d -l decompress -x -a "$end_str*.bz $end_str*.bz2 $end_str.tbz $end_str*.tbz2" -d "Compressed file"
complete -c bzip2 -s z -l compress -d "Compress file"
complete -c bzip2 -s t -l test -d "Check integrity"
complete -c bzip2 -s f -l force -d "Overwrite"
complete -c bzip2 -s k -l keep -d "Do not overwrite"
complete -c bzip2 -s s -l small -d "Reduce memory usage"
complete -c bzip2 -s q -l quiet -d "Supress errors"
complete -c bzip2 -s v -l verbose -d "Print compression ratios"
complete -c bzip2 -s L -l license -d "Print license"
complete -c bzip2 -s V -l version -d "Print version"
complete -c bzip2 -s 1 -l fast -d "Small block size"
complete -c bzip2 -s 9 -l best -d "Large block size"

complete -c bzcat -x -a "$end_str*.bz $end_str*.bz2 $end_str.tbz $end_str*.tbz2" -d "Compressed file"
complete -c bzcat -s s -l small -d "Reduce memory usage"

complete -c bzip2recover -x -a "$end_str*.bz $end_str*.bz2 $end_str.tbz $end_str*.tbz2" -d "Compressed file"


complete -c cat -s A -l show-all -d "Escape all non-printing characters"
complete -c cat -s b -l number-nonblank -d "Number nonblank lines"
complete -c cat -s e -d "Escape non-printing characters except tab"
complete -c cat -s E -l show-ends -d "Display $ at end of line"
complete -c cat -s n -l number -d "Number all lines"
complete -c cat -s s -l squeeze-blank -d "Never more than single blank line"
complete -c cat -s t -d "Escape non-printing characters except newline"
complete -c cat -s T -l show-tabs -d "Escape tab"
complete -c cat -s v -d "Escape non-printing except newline and tab"
complete -c cat -l help -d "Display help and exit"
complete -c cat -l version -d "Display version and exit"


complete -c chown -s c -l changes -d "Output diagnostic for changed files"
complete -c chown -l dereference -d "Dereferense symbolic links"
complete -c chown -s h -l no-dereference -d "Do not dereference symbolic links"
complete -c chown -l from -d "Change from owner/group"
complete -c chown -s f -l silent -d "Supress errors"
complete -c chown -l reference -d "Use same owner/group as file" -r
complete -c chown -s R -l recursive -d "Operate recursively"
complete -c chown -s v -l verbose -d "Output diagnostic for every file"
complete -c chown -s h -l help -d "Display help and exit"
complete -c chown -l version -d "Display version and exit"
complete -c chown -d "Username" -a "(cat /etc/passwd|cut -d : -f 1):"
complete -c chown -d "Username" -a "(echo $end_str|grep -o '.*:')(cat /etc/group |cut -d : -f 1)"

complete -c chgrp -s c -l changes -d "Output diagnostic for changed files"
complete -c chgrp -l dereference -d "Dereferense symbolic links"
complete -c chgrp -s h -l no-dereference -d "Do not dereference symbolic links"
complete -c chgrp -l from -d "Change from owner/group"
complete -c chgrp -s f -l silent -d "Supress errors"
complete -c chgrp -l reference -d "Use same owner/group as file" -r
complete -c chgrp -s R -l recursive -d "Operate recursively"
complete -c chgrp -s v -l verbose -d "Output diagnostic for every file"
complete -c chgrp -s h -l help -d "Display help and exit"
complete -c chgrp -l version -d "Display version and exit"
complete -c chgrp -d Group -a "(cat /etc/group|cut -d : -f 1):"

complete -c cut -s b -l bytes -x -d "Ouput byte range" 
complete -c cut -s c -l characters -x -d "Output character range" 
complete -c cut -s d -l delimiter -x -d "Select field delimiter" 
complete -c cut -s d -l fields -x -d "Select fields" 
complete -c cut -s n -d "Dont split mutibyte characters" 
complete -c cut -s s -l only-delimited -d "Do not print lines without delimiter" 
complete -c cut -l output-delimiter -d "Select output delimiter" 
complete -c cut -l help -d "Display help and exit" 
complete -c cut -l version -d "Display version and exit" 

#
# I don't use CVS, so these completions are probably not all that good.
#

complete -c cvs -x -a "add" -d "Add a new file/directory to the repository"
complete -c cvs -x -a "admin" -d "Administration front end for rcs"
complete -c cvs -x -a "annotate" -d "Show last revision where each line was modified"
complete -c cvs -x -a "checkout" -d "Checkout sources for editing"
complete -c cvs -x -a "commit" -d "Check files into the repository"
complete -c cvs -x -a "diff" -d "Show differences between revisions"
complete -c cvs -x -a "edit" -d "Get ready to edit a watched file"
complete -c cvs -x -a "editors" -d "See who is editing a watched file"
complete -c cvs -x -a "export" -d "Export sources from CVS, similar to checkout"
complete -c cvs -x -a "history" -d "Show repository access history"
complete -c cvs -x -a "import" -d "Import sources into CVS, using vendor branches"
complete -c cvs -x -a "init" -d "Create a CVS repository if it doesnt exist"
complete -c cvs -x -a "kserver" -d "Kerberos server mode"
complete -c cvs -x -a "log" -d "Print out history information for files"
complete -c cvs -x -a "login" -d "Prompt for password for authenticating server"
complete -c cvs -x -a "logout" -d "Removes entry in .cvspass for remote repository"
complete -c cvs -x -a "pserver" -d "Password server mode"
complete -c cvs -x -a "rannotate" -d "Show last revision where each line of module was modified"
complete -c cvs -x -a "rdiff" -d "Create 'patch' format diffs between releases"
complete -c cvs -x -a "release" -d "Indicate that a Module is no longer in use"
complete -c cvs -x -a "remove" -d "Remove an entry from the repository"
complete -c cvs -x -a "rlog" -d "Print out history information for a module"
complete -c cvs -x -a "rtag" -d "Add a symbolic tag to a module"
complete -c cvs -x -a "server" -d "Server mode"
complete -c cvs -x -a "status" -d "Display status information on checked out files"
complete -c cvs -x -a "tag" -d "Add a symbolic tag to checked out version of files"
complete -c cvs -x -a "unedit" -d "Undo an edit command"
complete -c cvs -x -a "update" -d "Bring work tree in sync with repository"
complete -c cvs -x -a "version" -d "Show current CVS version(s)"
complete -c cvs -x -a "watch" -d "Set watches"
complete -c cvs -x -a "watchers" -d "See who is watching a file"

complete -c cvs -x -s H -d "Displays usage information for command"
complete -c cvs -x -s Q -d "Cause CVS to be really quiet"
complete -c cvs -x -s q -d "Cause CVS to be somewhat quiet"
complete -c cvs -x -s r -d "Make checked-out files read-only"
complete -c cvs -x -s w -d "Make checked-out files read-write (default)"
complete -c cvs -x -s n -d "Do not execute anything that will change the disk"
complete -c cvs -x -s t -d "Show trace of program execution -- try with -n"
complete -c cvs -x -s v -d "CVS version and copyright"
complete -c cvs -x -s T -r -d "Use 'tmpdir' for temporary files"
complete -c cvs -x -s e -r -d "Use 'editor' for editing log information"
complete -c cvs -x -s d -r -d "Overrides $CVSROOT as the root of the CVS tree"
complete -c cvs -x -s f -d "Do not use the ~/.cvsrc file"
complete -c cvs -x -s z -d "Compression level for net traffic" -x -a "1 2 3 4 5 6 7 8 9"
complete -c cvs -x -s x -d "Encrypt all net traffic"
complete -c cvs -x -s a -d "Authenticate all net traffic"
complete -c cvs -x -s s -d "Set CVS user variable" -x


complete -c date -s d -l date -d "Display date described by string" -x
complete -c date -s f -l file -d "Display date for each line in file" -r
complete -c date -s I -l iso-8601 -d "Output in ISO 8601 format" -a "date hours minutes seconds"
complete -c date -s s -l set -d "Set time" -x
complete -c date -s R -l rfc-2822 -d "Output RFC-2822 compliant date string"
complete -c date -s r -l reference -d "display the last modification time of file" -r
complete -c date -s u -l utc -d "print or set Coordinated Universal Time"
complete -c date -l universal -d "print or set Coordinated Universal Time"
complete -c date -s h -l help -d "Display help and exit"
complete -c date -s v -l version -d "Display version and exit"

complete -c df -s a -l all -d "Include empty filesystems"
complete -c df -s B -l block-size -r -d "Block size"
complete -c df -s h -l human-readable -d "Human readable sizes"
complete -c df -s H -l si -d "Human readable sizes, powers of 1000"
complete -c df -s i -l inodes -d "List inode information"
complete -c df -s k -d "Use 1KB block size"
complete -c df -s l -l local -d "List only local filesystems"
complete -c df -l no-sync -d "Do not sync before getting usage info"
complete -c df -s P -l portability -d "Use Posix format"
complete -c df  -l sync -d "Sync before getting usage info"
complete -c df -s t -l type -r -d "Filesystem type" -a $filesystems
complete -c df -s T -l print-type -d "Print filesystem type"
complete -c df -s x -l exclude-type -d "Excluded filesystem type" -r -a $filesystems
complete -c df -l help -d "Display help and exit"
complete -c df -l version -d "Display version and exit"

complete -c du -s a -l all -d "Write size for all files"
complete -c du -l apparent-size -d "Print file size, not disk usage"
complete -c du -s B -l block-size -d "Block size"
complete -c du -s b -l bytes -d "Use 1B block size"
complete -c du -s c -l total -d "Produce grand total"
complete -c du -s D -l dereference-args -d "Dereference file symlinks"
complete -c du -s h -l human-readable -d "Human readable sizes"
complete -c du -s H -l si -d "Human readable sizes, powers of 1000"
complete -c du -s k -d "Use iKB block size"
complete -c du -s l -l count-links -d "Count hard links multiple times"
complete -c du -s L -l dereference -d "Dereference all symlinks"
complete -c du -s S -l separate-dirs -d "Do not include subdirectory size"
complete -c du -s s -l summarize -d "Display only a total for each argument"
complete -c du -s x -l one-file-system -d "Skip other filesystems"
complete -c du -s X -l exclude-from -r -d "Exclude files thet match pattern in file"
complete -c du -l exclude -r -d "Exclude files that match pattern"
complete -c du -l max-depth -r -d "Recursion limit"
complete -c du -l help -d "Display help and exit"
complete -c du -l version -d "Display version and exit"


complete -c echo -s n -d "No newline"
complete -c echo -s e -d "Use backslashe-escaped characters"
complete -c echo -s E -d "Do not use backslash escaped characters"
complete -c echo -l help -d "Display help and exit"
complete -c echo -l version -d "Display version and exit"

#
# The gcc completion list is incomplete. There are just so many of them...
#

complete -c gcc -s x -d "Language" -a "
	c
	c-header
	cpp-output
	c++
	c++-cpp-output
	objective-c
	objc-cpp-output
	assembler
	assembler-with-cpp
	ada
	f77
	f77-cpp-input
	ratfor
	java
	treelang
	none
"
complete -c gcc -o pass-exit-codes  -d "Pass program exit codes"
complete -c gcc -s c -d "Stop after assembler"
complete -c gcc -s S -d "Stop after compile"
complete -c gcc -s E -d "Stop after preprocesswor"
complete -c gcc -s o -r -d "Output file"
complete -c gcc -s v -d "Print commands to stderr"
complete -c gcc -o \#\#\# -d "Print quoted commands to stderr, do not run"
complete -c gcc -o pipe -d "Use pipes"
complete -c gcc -o ansi  -d "Use ansi mode"
complete -c gcc -o std  -d "Standard mode" -a "
	c89\tISO\ C90 
	iso9899:1990\tISO\ C90  
	iso9899:199409\tISO\ C90\ as\ modified\ in\ amendment\ 1
	c99\tISO\ C99 
	c9x \tISO\ C99 
	iso9899:1999\tISO\ C99  
	iso9899:199x\tISO\ C99  
	gnu89\tISO\ C90\ plus\ GNU\ extentions 
	gnu99\tISO\ C99\ plus\ GNU\ extentions  
	gnu9x\tISO\ C99\ plus\ GNU\ extentions  
	c++98\tISO\ C++98
	gnu++98\tISO\ C++98\ plus\ GNU\ extentions
"
complete -c gcc -o aux-info -r -d "Write prototypes to file"
complete -c gcc -o fno-asm  -d "Do not recognize asm, inline or typeof keywords"
complete -c gcc -o fno-builtin  -d "Do not use builtin functions"
complete -c gcc -o fhosted -d "Assert hosted environment"
complete -c gcc -o ffreestanding  -d "Assert freestanding environment"
complete -c gcc -o fms-extensions  -d "Use Microsoft extensions"
complete -c gcc -o trigraphs  -d "Use ANSI trigraphs"
complete -c gcc -o no-integrated-cpp -d "Do not use integrated preprocessor"
complete -c gcc -o funsigned-char -d "char is unsigned"
complete -c gcc -o fsigned-char -d "char is signed"
complete -c gcc -o funsigned-bitfields -d "bifield is unsigned"
complete -c gcc -o fsigned-bitfields -d "bifield is signed"
complete -c gcc -o fno-unsigned-bitfields -d "All bifields are signed"
complete -c gcc -o fno-signed-bitfields -d "All bifield are signed"
complete -c gcc -o fwritable-strings -d "String constants are not const"
complete -c gcc -o fabi-version -d "C++ ABI version" -r -a "1 0"
complete -c gcc -o fno-access-control -d "Turn off access checking"
complete -c gcc -o fcheck-new -d "Check pointer returned by new"
complete -c gcc -o fconserve-space -d "Put globals in the common segment"
complete -c gcc -o fno-const-strings -d "String constants are not const"
complete -c gcc -o fdollars-in-identifiers -d "Accept $ in identifiers"
complete -c gcc -o fno-dollars-in-identifiers -d "Reject $ in identifiers"
complete -c gcc -o fno-elide-constructors -d "Do not omit unneeded temporarys"
complete -c gcc -o fno-enforce-eh-specs -d "Allow exception violations"
complete -c gcc -o ffor-scope -d "Do not extend for-loop scope"
complete -c gcc -o fno-for-scope -d "Extend for-loop scope"
complete -c gcc -o fno-gnu-keywords -d "Do not recognize typeof as keyword"
complete -c gcc -o fno-implicit-templates -d "Do not emit code for implicit templates"
complete -c gcc -o fno-implicit-inline-templates -d "Do not emit code for implicit inline templates"
complete -c gcc -o fno-implement-inlines -d "Do not emit out-of-line code for inline functions"
complete -c gcc -o fms-extensions -d "Disable warnings about MFC"
complete -c gcc -o fno-nonansi-builtins -d "Disable some built-in functions"
complete -c gcc -o fno-operator-names -d "Disable operator keywords"
complete -c gcc -o fno-optional-diags -d "Disable optional diagnostics"
complete -c gcc -o fpermissive -d "Downgrade some errors to warnings"
complete -c gcc -o frepo -d "Enable automatic template instantiation at link time"
complete -c gcc -o fno-rtti -d "Disable generation of C++ runtime type information"
#complete -c gcc -o fstats -d "Emit front-end usage statistics"

for i in 1 2 3 4 5 6 7 8 9 10 11 12 13 14 15 16 17;
	complete -c gcc -o ftemplate-depth-1 -d Set\ maximum\ template\ depth\ to\ $i;
end;

complete -c gcc -o fno-threadsafe-statistics -d "Do not emit code for thread-safe initialization of local statics"
complete -c gcc -o fuse-cxa-atexit -d "Use __cxa_atexit for destructors"
complete -c gcc -o fvisibility-inlines-hidden -d "Hides inline methods from export table"
complete -c gcc -o fno-weak -d "Do not use weak symbol support"
# gcc completion listing is incomplete. gcc
#complete -c gcc -o -d ""


complete -c grep -s A -l after-context -d "Print NUM lines of trailing context"
complete -c grep -s a -l text -d "Process binary file as text"
complete -c grep -s B -l before-context -d "Print NUM lines of leading context"
complete -c grep -s C -l context -d "Print NUM lines of context"
complete -c grep -s b -l byte-offset -d "Print byte offset of matches"
complete -c grep -l binary-files -d "Assume data type for binary files" -x -a "binary text"
complete -c grep -l colour -x -a "never always auto"
complete -c grep -l color -x -a "never always auto"
complete -c grep -s c -l count -d "Only print number of matches"
complete -c grep -s D -l devices -x -a "read skip" -d "Action for devices"
complete -c grep -s d -l directories -x -a "read skip recurse" -d "Action for directories"
complete -c grep -s E -l extended-regexp -d "Pattern is extended regexp"
complete -xc grep -s e -l regexp -d "Pattern is a regexp"
complete -c grep -s F -l fixed -d "Pattern is a fixed string"
complete -c grep -s f -l file -r -d "Use pattern from file"
complete -c grep -s G -l basic-regexp -d "Pattern is basic regex"
complete -c grep -s H -l with-filename -d "Print filename"
complete -c grep -s h -l no-filename -d "Supress printing filename"
complete -c grep -l help -d "Display help and exit"
complete -c grep -s I -d "Skip binary files"
complete -c grep -s i -l ignore-case -d "Ignore case"
complete -c grep -s L -l files-without-match -d "Print first non-matching file"
complete -c grep -s l -l files-with-match -d "Print first matching file"
complete -c grep -s m -l max-count -d "Stop reading after NUM matches"
complete -c grep -l mmap -d "Use the mmap system call to read input"
complete -c grep -s n -l line-number -d "Print linenumber"
complete -c grep -s o -l only-matching -d "Show only matching part"
complete -c grep -l label -d "Rename stdin"
complete -c grep -l line-buffered -d "Use line buffering"
complete -c grep -s q -l quiet -d "Do not write anything"
complete -c grep -l silent -d "Do not write anything"
complete -c grep -s R -l recursive -d "Read files under each directory"
complete -c grep -s r -d "Read files under each directory"
complete -c grep -l include -d "Recurse, search file matching PATTERN"
complete -c grep -l exclude -d "Recurse, skip file matching PATTERN"
complete -c grep -s s -l no-messages -d "Suppress error messages"
complete -c grep -s U -l binary -d "Treat files as binary"
complete -c grep -s u -l unix-byte-offsets -d "Report Unix-style byte offsets"
complete -c grep -s V -l version -d "Print the version number"
complete -c grep -s v -l invert-match -d "Invert the sense of matching"
complete -c grep -s w -l word-regexp -d "Only whole matching words"
complete -c grep -s x -l line-regexp -d "Only whole matching lines"
complete -c grep -s y -d "Synonym for -i"
complete -c grep -s Z -l null -d "Output a zero byte after filename"


complete -c gprof -s A -l annoted-source -d "Print annotated source" 
complete -c gprof -s b -l brief -d "Do not print explanations"
complete -c gprof -s C -l exec-counts -d "Print tally" 
complete -c gprof -s i -l file-info -d "Display summary"
complete -c gprof -s I -l directory-path -d "Search directories for source"
complete -c gprof -s J -l no-annotated-source -d "No annotated source" 
complete -c gprof -s L -l print-path -d "Print full path of source"
complete -c gprof -s p -l flat-profile -d "Print flat profile" 
complete -c gprof -s P -l no-flat-profile -d "No flat profile" 
complete -c gprof -s q -l graph -d "Print call graph"
complete -c gprof -s Q -l no-graph -d "No call graph"
complete -c gprof -s y -l separate-files -d "Annotate to file"
complete -c gprof -s Z -l no-exec-counts -d "No tally"
complete -c gprof -l function-ordering -d "Suggest function ordering"
complete -rc gprof -l file-ordering -d "Suggest file ordering"
complete -c gprof -s T -l traditional -d "Traditional mode"
complete -xc gprof -s w -l width -d "Set width of output"
complete -c gprof -s x -l all-lines -d "Anotate every line"
complete -xc gprof -l demangle -d "Set demangling style"
complete -c gprof -l no-demangle -d "Turn of demangling"
complete -xc gprof -s a -l no-static -d "Supress static functions"
complete -xc gprof -s c -l static-call-graph -d ""
complete -xc gprof -s D -l ignore-non-functions -d "Ignore symbols not known to be functions"
complete -xc gprof -s k -r -d "Delete arcs from callgraph"
complete -xc gprof -s l -l line -d "Line by line profiling"
complete -xc gprof -s m -l min-count -r -d "Supress output when executed less than specified times"
complete -xc gprof -s n -l time -d "Only propagate times for matching symbols"
complete -xc gprof -s N -l no-time -d "Do not propagate times for matching symbols"
complete -xc gprof -s z -l display-unused-functions -d "Mention unused functions in flat profile"
complete -xc gprof -s d -l debug -d "Specify debugging options"
complete -xc gprof -s h -l help -d "Print help and exit"
complete -xc gprof -s v -l version -d "Print version and exit"
complete -xc gprof -s O -l file-format -a "auto bsd 4.4bsd magic prof" -d "Profile data format"
complete -xc gprof -s s -l sum -d "Print summary"


#Completion for the gzip family of commands

complete -c gunzip -s c -l stdout -d "Compress to stdout"
complete -c gunzip -x -a "$end_str*.gz $end_str*.tgz" -d "Compressed file"
complete -c gunzip -s f -l force -d "Overwrite"
complete -c gunzip -s h -l help -d "Display help"
complete -c gunzip -s l -l list -d "List compression information"
complete -c gunzip -s L -l license -d "Print license"
complete -c gunzip -s n -l no-name -d "Do not save/restore filename"
complete -c gunzip -s N -l name -d "Save/restore filename"
complete -c gunzip -s q -l quiet -d "Supress warnings"
complete -c gunzip -s r -l recursive -d "Recurse directories"
complete -c gunzip -s S -l suffix -r -d "Suffix"
complete -c gunzip -s t -l test -d "Check integrity"
complete -c gunzip -s v -l verbose -d "Display compression ratios"
complete -c gunzip -s V -l version -d "Display version"

complete -c gzip -s c -l stdout -d "Compress to stdout"
complete -c gzip -s d -l decompress -x -a "$end_str*.gz $end_str*.tgz" -d "Compressed file"
complete -c gzip -s f -l force -d "Overwrite"
complete -c gzip -s h -l help -d "Display help"
complete -c gzip -s l -l list -d "List compression information"
complete -c gzip -s L -l license -d "Print license"
complete -c gzip -s n -l no-name -d "Do not save/restore filename"
complete -c gzip -s N -l name -d "Save/restore filename"
complete -c gzip -s q -l quiet -d "Supress warnings"
complete -c gzip -s r -l recursive -d "Recurse directories"
complete -c gzip -s S -l suffix -r -d "Suffix"
complete -c gzip -s t -l test -d "Check integrity"
complete -c gzip -s v -l verbose -d "Display compression ratios"
complete -c gzip -s V -l version -d "Display version"
complete -c gzip -s 1 -l fast -d "Use fast setting"
complete -c gzip -s 9 -l best -d "Use high compression setting"

complete -c zcat -x -a "$end_str*.gz $end_str*.tgz" -d "Compressed file"
complete -c zcat -s f -l force -d "Overwrite"
complete -c zcat -s h -l help -d "Display help"
complete -c zcat -s L -l license -d "Print license"
complete -c zcat -s V -l version -d "Display version"

complete -c gunzip -x -a "$end_str*.gz $end_str*.tgz" -d "Compressed file"

complete -c id -s g -l group -d "Print effective group id"
complete -c id -s G -l groups -d "Print all group ids"
complete -c id -s n -l name -d "Print name, not number"
complete -c id -s r -l real -d "Print real ID, not effective"
complete -c id -s u -l user -d "Print effective user ID"
complete -c id -l help -d "Display help and exit"
complete -c id -l version -d "Display version and exit"


complete -x -c ifconfig -a down -d "Stop interface"
complete -x -c ifconfig -a up -d "Start interface"
complete -x -c ifconfig -a "
	{,-}arp
	{,-}promisc
	{,-}allmulti
	metric
	mtu
	dstaddr
	netmask
	add
	del
	tunnel
	irq
	io_addr
	mem_start
	media
	{,-}broadcast
	{,-}pointopoint
	hw
	multicast
	address
	txqueuelen
" 
complete -x -c ifconfig -a "(cat /proc/net/dev ^/dev/null |cut -s -d : -f 1|grep -o '[^ ]*$')" -d "Network interface"

complete -c less -s \? -l help -d "Display help and exit"
complete -c less -s a -l search-skip-screen -d "Search after end of screen"
complete -c less -s b -l buffers -r -d "Buffer space"
complete -c less -s B -l auto-buffers -d "Disable automtic buffer allocation"
complete -c less -s c -l clear-screen -d "Repaint from top"
complete -c less -s C -l CLEAR-SCREEN -d "Clear and repaint from top"
complete -c less -s d -l dumb -d "Supress error for lacking terminal capability"
complete -c less -s e -l quit-at-eof -d "Exit on second EOF"
complete -c less -s E -l QUIT-AT-EOF -d "Exit on EOF"
complete -c less -s f -l force -d "Open non-regular files"
complete -c less -s F -l quit-if-one-screen -d "Quit if file shorter than one screen"
complete -c less -s g -l hilite-search -d "Hilight one search target"
complete -c less -s G -l HILITE-SEARCH -d "No search highlighting"
complete -c less -s h -l max-back-scroll -d "Maximum backward scroll" -r -a "1 2 3 4 5 6 7 8 9 10 11 12 13 14 15 16 17 18 19"
complete -c less -s i -l ignore-case -d "Search ignores lowercase case"
complete -c less -s I -l IGNORE-CASE -d "Search ignores all case"
complete -c less -s j -l jump-target -d "Target line" -r -a "0 1 2 3 4 5 6 7 8 9 10 11 12 13 14 15 16 17 18 19 -1 -2 -3 -4 -5 -6 -7 -8 -9 -10 -11 -12 -13 -14 -15 -16 -17 -18 -19"
complete -c less -s J -l status-column -d "Display status column"
complete -c less -s k -l lesskey-file -d "Specify key bindings file" -r
complete -c less -s L -l no-lessopen -d "Ignore $LESSOPEN"
complete -c less -s m -l long-prompt -d "Prompt with percentage"
complete -c less -s M -l LONG-PROMPT -d "Verbose prompt"
complete -c less -s n -l line-numbers -d "Display line number"
complete -c less -s N -l LINE-NUMBERS -d "Display line number for each line"
complete -c less -s o -l log-file -d "Log input to file" -r
complete -c less -s O -l LOG-FILE -d "Log to file, overwrite" -r
complete -c less -s p -l pattern -d "Start at first occorance of pattern" -r
complete -c less -s P -l prompt -d "Prompt string" -r
complete -c less -s q -l quiet -d "Silent mode"
complete -c less -l silent -d "Silent mode"
complete -c less -s Q -l QUIET -d "Completly silent mode"
complete -c less -l SILENT -d "Completly silent mode"
complete -c less -s r -l raw-control-chars -d "Display control chars"
complete -c less -s R -l RAW-CONTROL-CHARS -d "Display control chars, guess screen appearance"
complete -c less -s s -l squeeze-blank-lines -d "Miltiple blank lines sqeezed"
complete -c less -s S -l chop-long-lines -d "Do not fold long lines"
complete -c less -s t -l tag -d "Edit tag" -r
complete -c less -s T -l tag-file -d "Set tag file" -r
complete -c less -s u -l underline-special -d "Allow backspace and carriage return"
complete -c less -s U -l UNDERLINE-SPECIAL -d "Allow backspace, tab and carriage return"
complete -c less -s V -l version -d "Display version and exit"
complete -c less -s w -l hilite-unread -d "Flash first unread line on new page"
complete -c less -s W -l HILITE-UNREAD -d "Flash first unread line on any movement"
complete -c less -s x -l tabs -d "set tab stops" -r -a "1 2 3 4 5 6 7 8 9 10 11 12 13 14 15 16"
complete -c less -s X -l no-init -d "No termcap init"
complete -c less -l no-keypad -d "No keypad init"
complete -c less -s y -l max-forw-scroll -d "Maximum forward scroll" -r -a "1 2 3 4 5 6 7 8 9 10 11 12 13 14 15 16 17 18 19"
complete -c less -s z -l window -d "Max scroll window" -r -d "1 2 3 4 5 6 7 8 9 10 11 12 13 14 15 16 17 18 19"
complete -c less -s \" -l quotes -d "Set quote char" -r
complete -c less -s \~ -l tilde -d "Lines after EOF are blank"
complete -c less -s \# -l shift -d "Scroll on arrows" -a "0 1 2 3 4 5 6 7 8 9 10 11 12 13 14 15 16 17 18 19"

complete -c ls -s a -l all -d "Show hidden"
complete -c ls -s A -l almost-all -d "Show hidden except . and .."
complete -c ls -l author -d "Print author"
complete -c ls -s b -l escape -d "Octal escapes for non graphic"
complete -c ls -l blocksize -x -d "Use SIZE-byte blocks"
complete -c ls -s B -l ignore-backups -d "Ignore files ending with ~"
complete -c ls -s c -d "Show and sort by ctime"
complete -c ls -s C -d "List by columns"
complete -c ls -l color -f -a "never always auto" -d "Show colors"
complete -c ls -s d -l directory -d "List directories"
complete -c ls -s D -l dired -d "Generate dired output"
complete -c ls -s f -d "Do not sort"
complete -c ls -s F -l classify -d "Append filetype indicator"
complete -c ls -l format -x -a "across commas horizontal long single-column verbose vertical" -d "List format"
complete -c ls -l full-time -d "Long forma, full-iso time"
complete -c ls -s g -d "Long format without owner"
complete -c ls -s G -l no-group -d "Do not print group information"
complete -c ls -s h -l human-readable -d "Human readable size"
complete -c ls -l si -d "Human readable size, base 1000"
complete -c ls -s H -l dereference-command-line -d "Follow symlinks"
complete -c ls -l dereference-command-line-symlink-to-dir #-d "Follow directory symlinks from command line"
complete -c ls -l indicator-style -x -a "none classify file-type" -d "Append indicator to entry"
complete -c ls -s i -l inode -d "Print index number of files"
complete -c ls -s I -l ignore -r -d "Skip entries matching pattern"
complete -c ls -s k -d "like --block-size=1K"
complete -c ls -s l -d "Long format"
complete -c ls -s L -l dereference -d "Follow symlinks"
complete -c ls -s m -d "Comma separated format"
complete -c ls -s n -l numeric-uid-gid -d "Long format, numeric IDs"
complete -c ls -s N -l literal -d "Print raw entry names"
complete -c ls -s o -d "Long format without groups"
complete -c ls -s p -l file-type -d "Append indicator"
complete -c ls -s q -l hide-control-chars -d "? fo non graphic"
complete -c ls -l show-control-chars -d "Non graphic as-is"
complete -c ls -s Q -l quote-name -d "Enclose entry in quotes"
complete -c ls -l quoting-style -x -a "literal locale shell shell-always c escape" -d "Select quoting style"
complete -c ls -s r -l reverse -d "Reverse sort order"
complete -c ls -s R -l recursive -d "List subdirectory recursively"
complete -c ls -s s -l size -d "Print size of files"
complete -c ls -s S -d "Sort by size"
complete -c ls -l sort -x -a "extension none size time version status time atime access use" -d "Sort criteria"
complete -c ls -l time -x -a "atime access use ctime status" -d "Time as WORD"
complete -c ls -l time-style -x -a "full-iso long-iso iso locale" -d "Select time style" 
complete -c ls -s t -d "Sort by modification time"
complete -c ls -s T -l tabsize -a "1 2 3 4 5 6 7 8 9 10 11 12" -d "Assume tab stops at each COLS"
complete -c ls -s u -d "Show access time"
complete -c ls -s U -d "Do not sort"
complete -c ls -s v -d "Sort by version"
complete -c ls -s w -l width -x -d "Assume screen width"
complete -c ls -s x -d "List entries by lines"
complete -c ls -s X -d "sort by extension"
complete -c ls -s 1 -d "List one file per line"
complete -c ls -l help -d "Display help and exit"
complete -c ls -l version -d "Output version and exit"


#Completions for make
complete -x -c make -a "(grep -h '^.*:' Makefile|cut -d : -f 1|grep -v '\$' ^/dev/null)" -d "Target"
complete -r -c make -s f -d "Use file as makefile" -r
complete -x -c make -s C -x -a "$end_str*/" -d "Change directory"
complete -c make -s d -d "Debug"
complete -c make -s e -d "Environment before makefile"
complete -c make -s i -d "Ignore errors"
complete -x -c make -s I -d "Search directory for Makefile" -a "$end_str*/"
complete -x -c make -s j -d "Number of jobs" 
complete -c make -s k -d "Continue after an error"
complete -c make -s l -d "Start when load drops"
complete -c make -s n -d "Do not execute commands"
complete -c make -s o -r -d "Ignore specified file"
complete -c make -s p -d "Print database"
complete -c make -s q -d "Question mode"
complete -c make -s r -d "Eliminate implicit rules"
complete -c make -s s -d "Silent operation"
complete -c make -s S -d "Cancel the effect of -k"
complete -c make -s t -d "Touch files, dont run commands"
complete -c make -s v -d "Print version"
complete -c make -s w -d "Print working directory"
complete -c make -s W -r -d "Pretend file is modified"


complete -c makedepend -s D -r -d "Define"
complete -c makedepend -s I -r -d "Include directory"
complete -c makedepend -s Y -r -d "Replace include directories"
complete -c makedepend -s a -d "Append dependencies to makefile"
complete -c makedepend -s f -r -d "Specify makefile"
complete -c makedepend -o include -r -d "Prepend file to input"
complete -c makedepend -s o -r -d "Object file suffix"
complete -c makedepend -s p -r -d "Object file prefix"
complete -c makedepend -s s -r -d "Starting string delimiter"
complete -c makedepend -s w -r -d "Line width"
complete -c makedepend -s v -d "Verbose mode"
complete -c makedepend -s m -d "Warn about multiple inclusion"

#
# Completions for man and whatis
#
# Ouch. The sed expression takes a whatis data entry and converts it to a tab completion entry. 
# I am for from a sed wizard...
#

set __FISH_MAN_CMD "
(
	if test $end_str; 
		apropos $end_str| grep -v ': nothing appropriate$'| grep \^$end_str | sed -e 's/\([^ ]*\).*(\([^)]*\)) *- */\1\t\2: /'
	end
)
"
complete -xc whatis -a $__FISH_MAN_CMD
complete -xc man -a $__FISH_MAN_CMD
set -e __FISH_MAN_CMD

complete -xc man -a 1 -d "Program section"
complete -xc man -a 2 -d "Syscall section"
complete -xc man -a 3 -d "Library section"
complete -xc man -a 4 -d "Device section"
complete -xc man -a 5 -d "File format section"
complete -xc man -a 6 -d "Games section"
complete -xc man -a 7 -d "Misc section"
complete -xc man -a 8 -d "Admin section"
complete -xc man -a 9 -d "Kernel section"
complete -xc man -a tcl -d "Tcl section"
complete -xc man -a n -d "New section"
complete -xc man -a l -d "Local section"
complete -xc man -a p 
complete -xc man -a o -d "Old section"
complete -rc man -s C -d "Configuration file"
complete -xc man -s M -a "$end_str*/" -d "Manpath"
complete -rc man -s P -d "Pager"
complete -xc man -s S -d "Manual sections"
complete -c man -s a -d "Display all matches"
complete -c man -s c -d "Always reformat"
complete -c man -s d -d "Debug"
complete -c man -s D -d "Debug and run"
complete -c man -s f -d "Whatis" 
complete -c man -s F -l preformat -d "Format only"
complete -c man -s h -d "Display help and exit"
complete -c man -s k -d "Apropos"
complete -c man -s K -d "Search in all man pages" 
complete -xc man -s m -d "Set system"
complete -xc man -s p -d "Preprocessors"
complete -c man -s t -d "Format for printing"
complete -c man -s w -l path -d "Only print locations" 
complete -c man -s W  -d "Only print locations" 


# Completions for mount
complete -x -c mount -a "(cat /etc/fstab|sed -e 's/^\([^ \t]*\)[ \t]*\([^ \t]*\).*/\1\n\2/'|grep '^/')" -d "Mount point"
complete -c mount -s V -d "Display version and exit"
complete -c mount -s h -d "Display help and exit"
complete -c mount -s v -d "Verbose mode"
complete -c mount -s a -d "Mount filesystems in fstab"
complete -c mount -s F -d "Fork process for each mount"
complete -c mount -s f -d "Fake mounting"
complete -c mount -s l -d "Add label to output"
complete -c mount -s n -d "Do not write mtab"
complete -c mount -s s -d "Tolerate sloppy mount options"
complete -c mount -s r -d "Read only"
complete -c mount -s w -d "Read/Write mode"
complete -x -c mount -s L -d "Mount partition with specified label"
complete -x -c mount -s U -d "Mount partition with specified UID"
complete -c mount -s O -x -d "Exclude filesystems" 
complete -c mount -l bind -f -d "Remount a subtree to a second position"
complete -c mount -l move -f -d "Move a subtree to a new position"
complete -c mount -x -s t -d "Filesystem" -a $filesystems


complete -c mount -x -s o -d "Mount option" -a "
	async\tUse\ asynchronous\ I/O
	atime\tUpdate\ time\ on\ each\ access
	auto\tMounted\ with\ -a
	defaults\tUse\ default\ options
	dev\tInterpret\ character/block\ special\ devices
	exec\tPermit\ executables 
	_netdev\tFilesystem\ uses\network
	noatime\tDo\ not\ update\ time\ on\ each\ access
	noauto\tNot\ mounted\ by\ -a
	nodev\tDo\ not\ interpret\ character/block\ special\ devices
	noexec\tDo\ not\ permit\ executables
	nosuid\tIgnore\ suid\ bits
	nouser\tOnly\ root\ may\ mount
	remount\tRemount\ read-only\ filesystem
	ro\tMount\ read-only
	rw\tMount\ read-write
	suid\tAllow\ suid\ bits
	sync\tUse\ synchronous\ I/O
	dirsync\tUse\ synchronous\ directory\ operations
	user\tAny\ user\ may\ mount
	users\tAny\ user\ may\ mount\ and\ unmount
"

# Completions for mplayer (Incomplete, there are too many options and
# I am too lazy. Please send in suggested additions)

#
# List of two letter language codes for dvd audio, etc. Very
# incomplete. Please send in additions.
#

set mplayer_lang "de dk en fi fr jp no hu pl se"

complete -c mplayer -o autoq -d "Dynamically change postprocessing" -x
complete -c mplayer -o autosync -x -d "A/V sync speed" 
complete -c mplayer -o framedrop -d "Skip frames to maintain A/V sync"
complete -c mplayer -s h -o help -d "Show help and exit"
complete -c mplayer -o hardframedrop -d "Skip frames to maintain A/V sync"
complete -c mplayer -o loop -x -d "Loop playback" -r -a "0 1 2 3 4 5 6 7 8 9"
complete -c mplayer -o shuffle -d "Play in random order"

complete -c mplayer -s h -o help -d "Display help and exit"
complete -c mplayer -o fs -d "Full screen"
complete -c mplayer -o playlist -d "Set playlist"-r
complete -c mplayer -o alang -d "Audio language" -x -a $mplayer_lang
complete -c mplayer -o audiofile -d "Play audio from file" -r
complete -c mplayer -o cdrom-device -d "Set default CD-ROM drive"
complete -c mplayer -o channels -d "Set number of audio channels" -a "2 4 6"
complete -c mplayer -o chapter -d "Set start chapter" -x
complete -c mplayer -o dvd-device -d "Set default DVD-ROM drive"
complete -c mplayer -o dvdangle -d "Set dvd viewing angle" -a "1 2 3 4 5 6 7 8"
complete -c mplayer -o forceidx -d "Force rebuilding index"
complete -c mplayer -o fps -d "Override framerate" -x -a "24 25 30"
complete -c mplayer -o idx -d "Rebuild index if unavailable"
complete -c mplayer -o loadidx -d "Load index from file" -r
complete -c mplayer -o ni -d "Force non-interleaved AVI parser"
complete -c mplayer -o saveidx -d "Rebuild index and save to file" -r
complete -c mplayer -o ss -d "Seek to give time position" -x
complete -c mplayer -o tv -d "TV capture mode"
complete -c mplayer -o slang -d "Subtitle language" -x -a $mplayer_lang
complete -c mplayer -o sub -d "Subtitle file" -r
complete -c mplayer -o unicode -d "Handle subtitlefile as unicode"
complete -c mplayer -o utf8 -d "Handle subtitlefile as utf8"

complete -c mplayer -o vo -x -d "Video output" -a "
(
	mplayer -vo help|grep \t.\*\t'\|^ *[a-zA-Z0-9][a-zA-Z0-9]*  '|sed -e 's/[\t ]*\([a-zA-Z0-9]*\)[\t ]*\(.*\)/\1\t\2/'
)
"

complete -c mplayer -o ao -x -d "Audio output" -a "
(
	mplayer -ao help|grep \t.\*\t'\|^ *[a-zA-Z0-9][a-zA-Z0-9]*  '|sed -e 's/[\t ]*\([a-zA-Z0-9]*\)[\t ]*\(.*\)/\1\t\2/'
)
"

complete -c mplayer -o afm -x -d "Audio output" -a "
(
	mplayer -afm help|grep \t.\*\t'\|^ *[a-zA-Z0-9][a-zA-Z0-9]*  '|sed -e 's/[\t ]*\([a-zA-Z0-9]*\)[\t ]*\(.*\)/\1\t\2/'
)
"

complete -c mplayer -o vfm -x -d "Audio output" -a "
(
	mplayer -vfm help|grep \t.\*\t'\|^ *[a-zA-Z0-9][a-zA-Z0-9]*  '|sed -e 's/[\t ]*\([a-zA-Z0-9]*\)[\t ]*\(.*\)/\1\t\2/'
)
"

set -e mplayer_lang

complete -c mv -s b -l backup -d "Make backup of each existing destination file"
complete -c mv -s f -l force -d "Do not prompt before overwriting"
complete -c mv -s i -l interactive -d "Prompt before overwrite"
complete -c mv -l reply -a "yes no query" -d "Answer for overwrite questions"
complete -c mv -l strip-trailing-slashes -d "Remove trailing slashes from source"
complete -c mv -s S -l suffix -r -d "Backup suffix"
complete -c mv -l target-directory -d "Target directory" -a "$end_str*/"
complete -c mv -s u -l update -d "Do not overwrite newer files"
complete -c mv -s v -l vervose -d "Explain what is done"
complete -c mv -l help -d "Print help and exit"
complete -c mv -l version -d "Print version and exit"


complete -c perl -s 0 -d "Specify record separator"
complete -c perl -s a -d "Feed input to split"
complete -c perl -s c -d "Check syntax"
complete -c perl -s d -d "Debugger"
complete -c perl -s D -x -d "Debug option" 
complete -c perl -s e -x -d "Execute command"
complete -c perl -s F -d "Set regexp used to split input"
complete -c perl -s i -d "Edit files in-place"
complete -c perl -s I -d "Include path"
complete -c perl -s l -d "Line ending processing"
complete -c perl -s n -d "Loop script"
complete -c perl -s p -d "Loop script, print $_"
complete -c perl -s P -d "Invoke CPP"
complete -c perl -s s -d "Define custom switches"
complete -c perl -s S -d "Search $PATH for script"
complete -c perl -s T -d "Taint checking"
complete -c perl -s U -d "Unsafe mode"
complete -c perl -s v -d "Display version"
complete -c perl -s x -d "Extract script"


complete -c pine -s f -d "Open folder" -a "(cd ~/.Mail; ls -d *)"
complete -c pine -s F -d "Open file" -r
complete -c pine -s h -d "Display help and exit"
complete -c pine -s i -d "Start in folder index"
complete -c pine -s I -d "Initial set of keystrokes" -x
complete -c pine -s k -d "Use function keys for commands"
complete -c pine -s l -d "Expand collections in FOLDER LIST display"
complete -c pine -s n -d "Start with specified current message number" -x
complete -c pine -s o -d "Open folder read-only"
complete -c pine -s p -d "Set configuration file" -r
complete -c pine -s P -d "Set global configuration file"
complete -c pine -s r -d "Restricted mode"
complete -c pine -s z -d "Enable suspention support"
complete -c pine -o conf -d "Produce a sample global configuration file"
complete -c pine -o pinerc -d "Produce sample configuration file" -r
complete -c pine -o sort -d "Set mail sort order" -a "
	arrival
	subject 
	from 
	date 
	size 
	orderedsubj 
	reverse
"

complete -c pine -o option  -d "Config option" -x


complete -c ps -s A -d "Select all"
complete -c ps -s N -d "Invert selection"
complete -c ps -s a -d "Select all processes except session leaders and terminal-less"
complete -c ps -s d -d "Select all processes except session leaders"
complete -c ps -s e -d "Select all"
complete -c ps -l deselect  -d "Deselect all processes that do not fulfill conditions"

complete -c ps -s C -d "Select by command" -r
complete -c ps -s G -l Group -d "Select by group" -x -a "
(
	cat /etc/group |cut -d : -f 1
)
"

complete -c ps -s U -l User -d "Select by user" -x -a "
(
	cat /etc/passwd|cut -d : -f 1
)
"

complete -c ps -s u -l user -d "Select by user" -x -a "
(
	cat /etc/passwd|cut -d : -f 1
)
"

complete -c ps -s g -l group -d "Select by group/session" -x -a "
(
	cat /etc/group |cut -d : -f 1
)
"

complete -c ps -s p -l pid -d "Select by PID" -x -a "
(
	cd /proc; command ls -d {1,2,3,4,5,6,7,8,9}*
)
"

complete -c ps -l ppid -d "Select by parent PID" -x -a "
(
	cd /proc; command ls -d {1,2,3,4,5,6,7,8,9}*
)
"

complete -c ps -s s -l sid -d "Select by session ID" -x -a "
(
	cd /proc; command ls -d {1,2,3,4,5,6,7,8,9}*
)
"

complete -c ps -s t -l tty -d "Select by tty" -r

complete -c ps -s F -d "Extra full format"
complete -c ps -s O -d "User defined format" -x
complete -c ps -s M -d "Add column for security data"
complete -c ps -s f -d "Full format"
complete -c ps -s j -d "Jobs format"
complete -c ps -s l -d "Long format"
complete -c ps -s o -l format -d "User defined format" -x
complete -c ps -s y -d "Do not show flags"
complete -c ps -s Z -l context -d "Display security context format"

complete -c ps -s H -l forest -d "Show hierarchy"
complete -c ps -s n -d "Set namelist file" -r
complete -c ps -s w -d "Wide output"
complete -c ps -s L -d "Show threads"
complete -c ps -s T -d "Show threads"

complete -c ps -s V -l version -d "Display vesrion and exit"
complete -c ps -l help  -d "Display help and exit"
complete -c ps -l info  -d "Display debugging info"


complete -c python -s c -x -d "Execute argument as command"
complete -c python -s d -d "Debug on"
complete -c python -s E -d "Ignore environment variables"
complete -c python -s h -d "Display help and exit"
complete -c python -s i -d "Interactive mode after executing commands"
complete -c python -s O -d "Enable optimizations"
complete -c python -s Q -x -a "old new" -d "Division control"
complete -c python -s S -d "Disable import of site module"
complete -c python -s t -d "Warn on mixed tabs and spaces"
complete -c python -s u -d "Unbuffered inputg and output"
complete -c python -s v -d "Verbose mode"
complete -c python -s V -d "Display version and exit"
complete -c python -s W -x -d "Warning control" -a "ignore default all module once error"
complete -c python -s x -d "Skip first line of input"

#Completions for rm
complete -c rm -s d -l directory -d "Unlink directory (Only by superuser)"
complete -c rm -s f -l force -d "Never prompt before removal"
complete -c rm -s i -l interactive -d "Prompt before removal"
complete -c rm -s r -l recursive -d "Remove content of subdirectories"
complete -c rm -s R -d "Remove content of subdirectories"
complete -c rm -s v -l verbose -d "Explain what is done"
complete -c rm -s h -l help -d "Display help and exit"
complete -c rm -l version -d "Display version and exit"


#Completions for rmdir
complete -x -c rmdir -d "Directory" -a "$end_str*/" 
complete -c rmdir -l ignore-fail-on-non-empty -d "Ignore errors from non-empty directories"
complete -c rmdir -s p -l parents -d "Remove each component of path"
complete -c rmdir -s v -l verbose -d "Explain what is done"
complete -c rmdir -l help -d "Display help and exit"
complete -c rmdir -l version -d "Display version and exit"

complete -c ruby -s 0 -d "Specify record separator"
complete -c ruby -s a -d "Feed input to split"
complete -c ruby -s c -d "Check syntax"
complete -c ruby -s K -d "Kanji code-set"
complete -c ruby -s d -l debug -d "Debugger"
complete -c ruby -s e -x -d "Execute command"
complete -c ruby -s h -l help -d "Display help"
complete -c ruby -s F -d "Set regexp used to split input"
complete -c ruby -s i -d "Edit files in-place"
complete -c ruby -s I -d "Include path"
complete -c ruby -s l -d "Line ending processing"
complete -c ruby -s n -d "Loop script"
complete -c ruby -s p -d "Loop script, print $_"
complete -c ruby -s r -r -d "Require file"
complete -c ruby -s s -d "Define custom switches"
complete -c ruby -s S -d "Search $PATH for script"
complete -c ruby -s T -d "Taint checking"
complete -c ruby -s v -l verbose -d "Verbose mode"
complete -c ruby -s w -d "Verbose mode without message"
complete -c ruby -l version -d "Display version"
complete -c ruby -s x -d "Extract script"
complete -c ruby -s X -a "$end_str*/" -d "Directory"
complete -c ruby -s y -l yydebug -d "Compiler debug mode"

set __FISH_SCREEN_CMD "
(
	screen -list |grep \^\t.\*\(.\*\)|sed -e 's/\t\(.*\)\t(\(.*\))/\1\tScreen: \2/'
)
"
complete -c screen -x
complete -c screen -s a -d "Include all capabilitys"
complete -c screen -s A -d "Adapt window size"
complete -c screen -s c -r -d "Specify init file"
complete -c screen -s d -d "Detach screen" -a $__FISH_SCREEN_CMD
complete -c screen -s D -d "Detach screen" -a $__FISH_SCREEN_CMD 
complete -c screen -s r -d "Reattach session" -a $__FISH_SCREEN_CMD
complete -c screen -s R -d "Reattach/create session" 
complete -c screen -o RR -d "Reattach/create any session" 
complete -c screen -s e -x -d "Escape character"
complete -c screen -s f -d "Flow control on"
complete -c screen -o fn -d "Flow control off"
complete -c screen -o fa -d "Flow control automatic"
complete -c screen -s h -x -d "History length"
complete -c screen -s i -d "Interrupt display on C-c"
complete -c screen -s l -d "Login on"
complete -c screen -o ln -d "Login off"
complete -c screen -o ls -d "List sessions"
complete -c screen -o list -d "List sessions"
complete -c screen -s L -d "Log on"
complete -c screen -s m -d "Ignore $STY"
complete -c screen -s O -d "Optimal output"
complete -c screen -s p -d "Preselect window"
complete -c screen -s q -d "Quiet mode"
complete -c screen -s s -r -d "Set shell"
complete -c screen -s S -x -d "Session name"
complete -c screen -s t -x -d "Session title"
complete -c screen -s U -d "UTF-8 mode"
complete -c screen -s v -d "Display version"
complete -c screen -o wipe -d "Wipe dead sessions"
complete -c screen -s x -d "Multi attach"
complete -c screen -s X -r -d "Send command"
set -e __FISH_SCREEN_CMD

complete -c sed -s n -l quiet -d "Silent mode"
complete -c sed -l silent -d "Silent mode"
complete -c sed -s e -l expression -x -d "Evaluate expression"
complete -c sed -s f -l file -r -d "Evalute file"
complete -c sed -s i -l in-place -d "Edit files in place"
complete -c sed -s l -l line-length -x -d "Specify line-length"
complete -c sed -l posix -d "Disable all GNU extensions"
complete -c sed -s r -l regexp-extended -d "Use extended regexp"
complete -c sed -s s -l separate -d "Consider files as separate"
complete -c sed -s u -l unbuffered -d "Use minimal IO buffers"
complete -c sed -l help -d "Display help and exit"
complete -c sed -s V -l version -d "Display version and exit"


complete -c sort -s b -l ignore-leading-blanks -d "Ignore leading blanks"
complete -c sort -s d -l dictionary-order -d "Consider only blanks and alphanumerics"
complete -c sort -s f -l ignore-case -d "Ignore case"
complete -c sort -s g -l general-numeric-sort -d "Compare general numeric value"
complete -c sort -s i -l ignore-nonprinting -d "Consider only printable"
complete -c sort -s M -l month-sort -d "Compare month names"
complete -c sort -s n -l numeric-sort -d "Compare string numerical value"
complete -c sort -s r -l reverse -d "Reverse results"
complete -c sort -s c -l check -d "Only check if sorted"
complete -c sort -s k -l key -d "Define key"
complete -c sort -s m -l merge -d "Merge sorted files"
complete -c sort -s o -l output -f -d "Write to file"
complete -c sort -s s -l stable -d "Stabilize sort"
complete -c sort -s S -l buffer-size -r -d "Set main memory buffer"
complete -c sort -s t -l field-separator -d "Field separator"
complete -c sort -s T -l temporary-directory -r -d "Set temporary directory"
complete -c sort -s u -l unique -d "Output only first of equal lines"
complete -c sort -s z -l zero-terminated -d "Lines end with 0 byte"
complete -c sort -l help -d "Display help and exit"
complete -c sort -l version -d "Display version and exit"


#Completions for ssh family of commands

#
# Common completions for all ssh commands 
#

for i in ssh scp
	complete -c $i -s 1 -d "Protocall version 1 only"
	complete -c $i -s 2 -d "Protocall version 2 only"
	complete -c $i -s 4 -d "IPv4 addresses only"
	complete -c $i -s 6 -d "IPv6 addresses only"
	complete -c $i -s C -d "Compress all data"
	complete -xc $i -s c -d "Encryption algorithm" -a "blowfish 3des des"
	complete -r -c $i -s F -d "Configuration file"
	complete -r -c $i -s i -d "Identity file"
	complete -x -c $i -s o -x -d "Options" -a "
		AddressFamily
		BatchMode
		BindAddress
		ChallengeResponseAuthentication
		CheckHostIP
		Cipher
		Ciphers
		Compression
		CompressionLevel
		ConnectionAttempts
		ConnectTimeout
		ControlMaster
		ControlPath
		GlobalKnownHostsFile
		GSSAPIAuthentication
		GSSAPIDelegateCredentials
		Host
		HostbasedAuthentication
		HostKeyAlgorithms
		HostKeyAlias
		HostName
		IdentityFile
		IdentitiesOnly
		LogLevel
		MACs
		NoHostAuthenticationForLocalhost
		NumberOfPasswordPrompts
		PasswordAuthentication
		Port
		PreferredAuthentications
		Protocol
		ProxyCommand
		PubkeyAuthentication
		RhostsRSAAuthentication
		RSAAuthentication
		SendEnv
		ServerAliveInterval
		ServerAliveCountMax
		SmartcardDevice
		StrictHostKeyChecking
		TCPKeepAlive
		UsePrivilegedPort
		User
		UserKnownHostsFile
		VerifyHostKeyDNS
	"
	complete -c $i -s v -d "Verbose mode"
end;

#
# ssh specific completions
#

complete -x -c ssh -d Hostname -a "

(
	#Find a suitable hostname from the knownhosts files
	cat ~/.ssh/known_hosts{,2} ^/dev/null|cut -d ' ' -f 1| cut -d , -f 1
) 

(
	#Prepend any username specified in the completion to the hostname
	echo $end_str|grep -o '.*@'
)(
	cat ~/.ssh/known_hosts{,2} ^/dev/null|cut -d ' ' -f 1| cut -d , -f 1
) 

(
	#Find a suitable username in passwd
	cat /etc/passwd|cut -d : -f 1
)@
"
complete -c ssh -s a -d "Disables forwarding of the authentication agent"
complete -c ssh -s A -d "Enables forwarding of the authentication agent"
complete -x -c ssh -s b -d "Interface to transmit from" -a "
(
	cat /proc/net/arp ^/dev/null| grep -v '^IP'|cut -d ' ' -f 1 ^/dev/null
)
"

complete -x -c ssh -s e -d "Escape character" -a "^ none"
complete -c ssh -s f -d "Go to background"
complete -c ssh -s g -d "Allow remote host to connect to local forwarded ports"
complete -c ssh -s I -d "Smartcard device"
complete -c ssh -s k -d "Disable forwarding of Kerberos tickets"
complete -c ssh -s l -x -a "(cat /etc/passwd|cut -d : -f 1)" -d "User"
complete -c ssh -s m -d "MAC algorithm"
complete -c ssh -s n -d "Prevent reading from stdin"
complete -c ssh -s N -d "Do not execute remote command" 
complete -c ssh -s p -x -d "Port"
complete -c ssh -s q -d "Quiet mode"
complete -c ssh -s s -d "Subsystem"
complete -c ssh -s t -d "Force pseudo-tty allocation"
complete -c ssh -s T -d "Disable pseudo-tty allocation"
complete -c ssh -s x -d "Disable X11 forwarding"
complete -c ssh -s X -d "Enable X11 forwarding"
complete -c ssh -s L -d "Locally forwarded ports"
complete -c ssh -s R -d "Remotely forwarded ports"
complete -c ssh -s D -d "Dynamic port forwarding"

#
# scp specific completions
#

complete -x -c scp -d Hostname -a "

(
	#Find a suitable hostname from the knownhosts files
	cat ~/.ssh/known_hosts{,2} ^/dev/null|cut -d ' ' -f 1| cut -d , -f 1
): 

(
	#Prepend any username specified in the completion to the hostname
	echo $end_str|grep -o '.*@'
)(
	cat ~/.ssh/known_hosts{,2} ^/dev/null|cut -d ' ' -f 1| cut -d , -f 1
): 

(
	#Find a suitable username in passwd
	cat /etc/passwd|cut -d : -f 1
)@
"
complete -c scp -s B -d "Batch mode"
complete -c scp -s l -x -d "Bandwidth limit"
complete -c scp -s P -x -d "Port"
complete -c scp -s p -d "Preserves modification times, access times, and modes from the original file"
complete -c scp -s q -d "Disable progress meter"
complete -c scp -s r -d "Recursively copy"
complete -c scp -s S -d "Encyption program"


#
# Completions for su command
#

complete -x -c su -d "Username" -a "(cat /etc/passwd|cut -d : -f 1)" 
complete -c su -s l -l login -d "Make login shell"
complete -r -c su -s c -l command -d "Pass command to shell" 
complete -c su -s f -l fast -d "Pass -f to the shell"
complete -c su -s m -l preserve_environment -d "Preserve environment"
complete -c su -s p -d "Preserve environment"
complete -x -c su -s s -l shell -a "(cat /etc/shells)"
complete -c su -l help -d "Display help and exit"
complete -c su -l version -d "Display version and exit"


complete -c tar -s A -l catenate -d "Append archive to archive"
complete -c tar -l concatenate -d "Append archive to archive"
complete -c tar -s c -l create -d "Create archive"
complete -c tar -s d -l diff -d "Compare archive and filesystem"
complete -c tar -l compare -d "Compare archive and filesystem"
complete -c tar -l delete -d "Delete from archive"
complete -c tar -s r -l append -d "Append files to archive"
complete -c tar -s t -l list -d "List archive"
complete -c tar -s u -l update -d "Append new files"
complete -c tar -s x -l extract -d "Extract from archive"
complete -c tar -l get -d "Extract from archive"
complete -c tar -l atime-preserve -d "Keep access time"
complete -c tar -s b -l block-size -d "Block size"
complete -c tar -s B -l read-full-blocks -d "Reblock while reading"
complete -c tar -s C -l directory -r -d "Change directory"
complete -c tar -l checkpoint -d "Print directory names"
complete -c tar -s f -l file -r -d "Archive file"
complete -c tar -l force-local -d "Archive is local"
complete -c tar -s F -l info-script -d "Run script at end of tape"
complete -c tar -s G -l incremental -d "Use old incremental GNU format"
complete -c tar -s g -l listed-incremental -d "Use new incremental GNU format"
complete -c tar -s h -l dereference -d "Derefrerence symlinks"
complete -c tar -s i -l ignore-zeros -d "Ignore zero bloch in archive"
complete -c tar -s j -l bzip -d "Filter through bzip2"
complete -c tar -l ignore-failed-read -d "Dont exit on unreadable files"
complete -c tar -s k -l keep-old-files -d "Do not overwrite"
complete -c tar -s K -l starting-file -r -d "Starting file in archive"
complete -c tar -s l -l one-file-system -d "Stay in local filesystem"
complete -c tar -s L -l tape-length -r -d "Tape length"
complete -c tar -s m -l modification-time -d "Dont extract modification time"
complete -c tar -s M -l multi-volume -d "Multi volume archive"
complete -c tar -s N -l after-date -r -d "Only store newer files"
complete -c tar -s o -l old-archive -d "Use V7 format"
complete -c tar -l portability -d "Use V7 format"
complete -c tar -s O -l to-stdout -d "Extract to stdout"
complete -c tar -s p -l same-permissions -d "Extract all permissions"
complete -c tar -l preserve-permissions -d "Extract all permissions"
complete -c tar -s P -l absolute-paths -d "Dont strip leading /"
complete -c tar -l preserve -d "Preserve all permissions and do not sort file arguments"
complete -c tar -s R -l record-number -d "Show record number"
complete -c tar -l remove-files -d "Remove files after adding to archive"
complete -c tar -s s -l same-order -d "Do not sort file arguments"
complete -c tar -l preserve-order -d "Do not sort file arguments"
complete -c tar -l same-owner -d "Preserve file ownership"
complete -c tar -s S -l sparse -d "Handle sparse files"
complete -c tar -s T -l files-from -r -d "Extract file from file"
complete -c tar -l null -d "-T has null-terminated names"
complete -c tar -l totals -d "Print total bytes written"
complete -c tar -s v -l verbose -d "Vorbose mode"
complete -c tar -s V -l label -r -d "Set volume name"
complete -c tar -l version -d "Display version and exit"
complete -c tar -s w -l interactive -d "Ask for confirmation"
complete -c tar -l confirmation -d "Ask for confirmation"
complete -c tar -s W -l vevrify -d "Verify archive"
complete -c tar -l exclude -r -d "Exclude file"
complete -c tar -s X -l exclude-from -r -d "Exclude files listed in specified file"
complete -c tar -s Z -l compress -d "Filter through compress"
complete -c tar -l uncompress -d "Filter through compress"
complete -c tar -s z -l gzip -d "Filter through gzip"
complete -c tar -l gunzip -d "Filter through gzip"
complete -c tar -l use-compress-program -r -d "Filter through specified program"


complete -c test -l help  -d "Display help and exit"
complete -c test -l version  -d "Display version and exit"
complete -c test -a ! -d "Negate expression"
complete -c test -s a -d "Logical and"
complete -c test -s o -d "Logical or"
complete -c test -s n -d "String length is non-zero"
complete -c test -s z -d "String length is zero"
complete -c test -a = -d "Strings are equal"
complete -c test -a !=  -d "Strings are not equal"
complete -c test -o eq -d "Integers are equal"
complete -c test -o ge -d "Left integer larger than or equal to right integer"
complete -c test -o gt -d "Left integer larger than right integer"
complete -c test -o le -d "Left integer less than or equal to right integer"
complete -c test -o lt -d "Left integer less than right integer"
complete -c test -o ne -d "Left integer not equal to right integer"
complete -c test -o ef -d "Left file equal to right file"
complete -c test -o nt -d "Left file newer than right file"
complete -c test -o ot -d "Left file older than right file"
complete -c test -s b -d "File is block device"
complete -c test -s c -d "File is character device"
complete -c test -s d -d "File is directory"
complete -c test -s e -d "File exists"
complete -c test -s f -d "File is regular"
complete -c test -s g -d "File is set-group-ID"
complete -c test -s h -d "File is symlink"
complete -c test -s G -d "File owned by effective group ID"
complete -c test -s k -d "File has sticky bit set"
complete -c test -s L -d "File is symlink"
complete -c test -s O -d "File owned by effective user ID"
complete -c test -s p -d "File is named pipe"
complete -c test -s r -d "File is readable"
complete -c test -s s -d "File size is non-zero"
complete -c test -s S -d "File is socket"
complete -c test -s t -d "FD is terminal"
complete -c test -s u -d "File set-user-ID bit is set"
complete -c test -s w -d "File is writable"
complete -c test -s x -d "File is executable"


complete -c top -s b -d "Batch mode"
complete -c top -s c -d "Toggle command line/program name"
complete -c top -s d -d "Update interval" -x
complete -c top -s h -d "Display help and exit"
complete -c top -s i -d "Toggle idle processes"
complete -c top -s n -d "Maximium iterations" -x
complete -c top -s u -d "Monitor effective UID" -x -a "(cat /etc/passwd|cut -d : -f 1)"
complete -c top -s U -d "Monitor user" -x -a "(cat /etc/passwd|cut -d : -f 1)"
complete -c top -s p -d "Monitor PID" -x -a "(cd /proc; command ls -d {1,2,3,4,5,6,7,8,9}*)"
complete -c top -s s -d "Secure mode"
complete -c top -s S -d "Cumulative mode"
complete -c top -s v -d "Display version and exit"


complete -c touch -s a -d "Change access time"
complete -c touch -s B -l backward -x -d "Set date back"
complete -c touch -s c -l no-create -d "Do not create file"
complete -c touch -s d -l date -x -d "Set date"
complete -c touch -s f -l forward -x -d "Set date forward"
complete -c touch -s m -d "Change modification time"
complete -c touch -s r -l reference -d "Use this files times"
complete -c touch -s t -d "Set date"
complete -c touch -l time -x -d "Set time"
complete -c touch -l help -d "Display help and exit"
complete -c touch -l version -d "Display version nd exit"


complete -c uname -s a -l all -d "Print all"
complete -c uname -s s -l kernel-name -d "Print kernel name"
complete -c uname -s n -l nodename -d "Print network node hostname"
complete -c uname -s r -l kernel-release -d "Print kernel release"
complete -c uname -s v -l kernel-version -d "Print kernel version"
complete -c uname -s m -l machine -d "Print machine name"
complete -c uname -s p -l processor -d "Print procssor"
complete -c uname -s i -l hardware-platform -d "Print harware platform"
complete -c uname -s o -l operating-system -d "Print operating system"
complete -c uname -l help -d "Print help"
complete -c uname -l verion -d "Print version"


complete -c uniq -s c -l count -d "Print number of occurences"
complete -c uniq -s d -l repeated -d "Only print duplicates"
complete -c uniq -s D -l all-repeated -d "Remove non-suplicate lines" -f -a "
	none\t'Remove none-duplicate lines' 
	prepend\t'Remove non-duplicate lines and print an empty line before each non-duplicate' 
	separate\t'Remove non-duplicate lines and print an empty line between each non-duplicate' 
"
complete -c uniq -s f -l skip-fields -d "Avoid comparing first N fields" -r
complete -c uniq -s i -l ignore-case -d "Case insensitive"
complete -c uniq -s s -l skip-chars -d "Avoid comparing first N characters" -r
complete -c uniq -s u -l unique -d "Only print unique lines"
complete -c uniq -s w -l check-chars -d "Compare only N characters" -r
complete -c uniq -l help -d "Display help and exit"
complete -c uniq -l version -d "Display version and exit"


complete -xc valgrind -l skin -d "Skin" -a "
	memcheck\tHeavyweight\ memory\ checker
	cachegrind\tCache-miss\ profiler
	addrcheck\tLightweight\ memory\ checker
	helgrind\tData-race\ detector
	massif\tHeap\ profiler
"
complete -c valgrind -l help -d "Print help"
complete -c valgrind -l help-debug -d "Print help and debug options"
complete -c valgrind -l version -d "Print version"
complete -c valgrind -s q -l quiet -d "Run silently"
complete -c valgrind -s v -l verbose -d "Verbose mode"
complete -xc valgrind -l trace-children -d "Valgrind-ise children" -a "yes no"
complete -xc valgrind -l track-fds -d "Track file descriptors" -a "yes no"
complete -xc valgrind -l logfile-fd -d "Log to file descriptor" -a "0 1 2 3 4 5 6 7 8 9"
complete -rc valgrind -l logfile -d "Log to file"
complete -xc valgrind -l logsocket -d "Log to socket"
complete -c valgrind -l demangle -xd "Demangle C++ names" -a "yes no"
complete -xc valgrind -l num-callers -d "Callers in stack trace" 
complete -xc valgrind -l error-limit -d "Stop showing errors if too many" -a "yes no" 
complete -xc valgrind -l show-below-main -d "Continue trace below main()" -a "yes no"
complete -rc valgrind -l supressions -d "Supress errors from file"
complete -c valgrind -l gen-supressions -d "Print suppressions for detected errors"
complete -xc valgrind -l db-attach -d "Start debugger on error" -a "yes no"
complete -rc valgrind -l db-command -d "Debugger command"
complete -xc valgrind -l leak-check -d "Check for memory leaks" -a "yes no"
complete -xc valgrind -l input-fd -d "File descriptor for input" -a "0 1 2 3 4 5 6 7 8 9"
complete -xc valgrind -l show-reacable -d "Show reachable leaked memory" -a "yes no"


complete -c w -s h -d "Dont print header"
complete -c w -s u -d "Ignore username for time calculations"
complete -c w -s s -d "Short format"
complete -c w -s f -d "Toggle printing of remote hostname"
complete -c w -s V -d "Print version and exit"
complete -c w -a "(cat /etc/passwd|cut -d : -f 1)" -d "Username"

complete -c wc -s c -l bytes -d "Print byte counts"
complete -c wc -s m -l chars -d "Print character counts"
complete -c wc -s l -l lines -d "Print newline counts"
complete -c wc -s L -l max-line-length -d "Print length of longest line"
complete -c wc -s w -l words -d "Print word counts"
complete -c wc -l help -d "Print help and exit"
complete -c wc -l version -d "Print version and exit"


complete -c who -s a -l all -d "Same as -b -d --login -p -r -t -T -u"
complete -c who -s b -l boot -d "Print time of last boot"
complete -c who -s d -l dead -d "Print dead processes"
complete -c who -s H -l heading -d "Print line of headings"
complete -c who -s i -l idle -d "Print idle time"
complete -c who -s l -l login -d "Print login process"
complete -c who -l lookup -d "Canonicalize hostnames via DNS"
complete -c who -s m -d "Print hostname and user for stdin"
complete -c who -s p -l process -d "Print active processes spawned by init"
complete -c who -s q -l count -d "Print all login names and number of users logged on"
complete -c who -s r -l runlevel -d "Print current runlevel"
complete -c who -s s -l short -d "Print name, line, and time"
complete -c who -s t -l time -d "Print last system clock change"
complete -c who -s T -l mesg -d "Print users message status as +, - or ?"
complete -c who -s w -l writable -d "Print users message status as +, - or ?"
complete -c who -l message -d "Print users message status as +, - or ?"
complete -c who -s u -l users -d "List users logged in"
complete -c who -l help -d "Print help and exit"
complete -c who -l version -d "Print version and exit"


complete -c xprop -o help -d "Display help and exit"
complete -c xprop -o grammar -d "Display grammar and exit"
complete -c xprop -o id -x -d "Select window by id"
complete -c xprop -o name -d "Select window by name"
complete -c xprop -o font -x -d "Display font properties"
complete -c xprop -o root -d "Select root window"
complete -c xprop -o display -d "Specify X server"
complete -c xprop -o len -x -d "Maximum display length"
complete -c xprop -o notype -d "Do not show property type"
complete -c xprop -o fs -r -d "Set format file"
complete -c xprop -o frame -d "Select window manager frame"
complete -c xprop -o remove -d "Remove property" -a "
(
	xprop -root -notype|cut -d ' ' -f 1|cut -d \t -f 1
)
" 

complete -c xprop -o set -d "Set property" -a "
(
	xprop -root -notype|cut -d ' ' -f 1|cut -d \t -f 1
)
" 

complete -c xprop -o spy -d "Examine property updates forever"
complete -c xprop -o f -d "Set format"
complete -c xprop -d Property -a "
(
	xprop -root -notype|cut -d ' ' -f 1|cut -d \t -f 1
)
" 


complete -c xsel -s a -l append -d "Append input to selection"
complete -c xsel -s f -l follow -d "Append to selection as input grows"
complete -c xsel -s i -l input -d "Read into selection"
complete -c xsel -s o -l output -d "Write selection"
complete -c xsel -s c -l clear -d "Clear selection"
complete -c xsel -s d -l delete -d "Delete selection"
complete -c xsel -s p -l primary -d "Use primary selection"
complete -c xsel -s s -l secondary -d "Use secondary selection"
complete -c xsel -s b -l clipboard -d "Use clipboard selection"
complete -c xsel -s k -l keep -d "Make current selections persistent after program exit"
complete -c xsel -s x -l exchange -d "Exchange primary and secondary selections"
complete -c xsel -l display -x -d "X server display"
complete -c xsel -s t -l selectionTimeout -d "Timeout for retrieving selection"
complete -c xsel -s l -l logfile -f -d "Error log"
complete -c xsel -s n -l nodetach -d "Do not detach from the controlling terminal"
complete -c xsel -s h -l help -d "Display help and exit"
complete -c xsel -s v -l verbose -d "Print informative messages"
complete -c xsel -l version -d "Display version and exit"

#Completions for SysV startup scripts

complete -x -p "/etc/init.d/*" -a "start" -d "Start service"
complete -x -p "/etc/init.d/*" -a "stop" -d "Stop service"
complete -x -p "/etc/init.d/*" -a "status" -d "Print service status"
complete -x -p "/etc/init.d/*" -a "restart" -d "Stop and start service"
complete -x -p "/etc/init.d/*" -a "reload" -d "Reload service configuration"


#
# Clean up variables that aren't needed any more
#

set -e filesystems
