/** \file input.h

Functions for reading a character of input from stdin, using the
inputrc information for key bindings.

*/


/** 
	Hopefully, the biggest value that a wchar_t can have. UCS4 is a
	31-bit character set, we use the upper half for special key
	sequences. On systems where wchar_t is not a 31 (or less) bit character set
	in a 32 (or more) bit type this will fail horribly. 
*/
#define WCHAR_END 0x80000000

/**
   Key codes for inputrc-style keyboard functions that are passed on
   to the caller of input_read()
*/
enum
{
	R_BEGINING_OF_LINE = 1 + WCHAR_END, 
	R_END_OF_LINE,
	R_FORWARD_CHAR,
	R_BACKWARD_CHAR,
	R_FORWARD_WORD,
	R_BACKWARD_WORD,
	R_HISTORY_SEARCH_BACKWARD,
	R_HISTORY_SEARCH_FORWARD,
	R_DELETE_CHAR,
	R_BACKWARD_DELETE_CHAR,
	R_KILL_LINE,
	R_YANK,
	R_YANK_POP,
	R_EXPLAIN,
	R_COMPLETE,
	R_BEGINNING_OF_HISTORY,
	R_END_OF_HISTORY,
	R_DELETE_LINE,
	R_BACKWARD_KILL_LINE,
	R_KILL_WHOLE_LINE,
	R_KILL_WORD,
	R_BACKWARD_KILL_WORD,
	R_DUMP_FUNCTIONS,
	R_CLEAR_SCREEN,
	R_EXIT
}
;

/**
   Initialize the terminal by calling setupterm, and set up arrays
   used by readch to detect escape sequences for special keys.

   Before calling input_init, terminfo is not initialized and MUST not be used
*/
int input_init();

/**
   free up memory used by terminal functions.
*/
void input_destroy();

/**
   Read a character from fd 0. Try to convert some escape sequences
   into character constants, but do not permanently block the escape
   character.

   This is performed in the same way vim does it, i.e. if an escape
   character is read, wait for more input for a short time (a few
   milliseconds). If more input is avaialable, it is assumed to be an
   escape sequence for a special character (such as an arrow key), and
   readch attempts to parse it. If no more input follows after the
   escape key, it is assumed to be an actual escape key press, and is
   returned as such.
*/
wint_t input_readch();

/**
   Push a character onto the stack of unread characters that readch2
   will return before actually reading from fd 0.
*/
void input_unreadch( wint_t ch );

/**
   Add a key mapping from the specified sequence to the specified
   shell command.

   \param s the sequence
   \param d a description of the sequence
   \param c a shell commond that will be evaluated whenever the key sequence occurs
*/
void add_mapping_command( wchar_t *s, wchar_t *d, wchar_t *c );

/**
   Add a key mapping from the specified sequence to the specified
   input function.

   \param s the sequence
   \param d a description of the sequence
   \param c am input function that will be run whenever the key sequence occurs
*/
void add_mapping_function( wchar_t *s, wchar_t * d, wint_t c );

