/*
Copyright (C) 2005 Axel Liljencrantz

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*/


/** \file main.c
	The main loop of <tt>fish</tt>.
*/

#include "config.h"

#include <stdlib.h>
#include <stdio.h>
#include <wchar.h>
#include <string.h>
#include <unistd.h>
#include <errno.h>
#include <unistd.h>
#include <termios.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>

#ifdef HAVE_GETOPT_H
#include <getopt.h>
#endif

#include <locale.h>
#include <dirent.h>

#include "util.h"
#include "common.h"
#include "reader.h"
#include "builtin.h"
#include "function.h"
#include "complete.h"
#include "wutil.h"
#include "env.h"
#include "sanity.h"
#include "parser.h"
#include "expand.h"

/**
   Parse init files
*/
static int read_init()
{
	char cwd[4096];
	wchar_t *wcwd;
	
	if( !getcwd( cwd, 4096 ) )
	{
		wperror( L"getcwd" );		
		return 0;
	}

	env_set( L"__fish_help_dir", PREFIX L"/share/doc/fish", 0);	
 	
	eval( L"cd /etc 2>/dev/null; . fish 2>/dev/null", 0, TOP );
	eval( L"cd 2>/dev/null;. .fish 2>/dev/null", 0, TOP );
	
	if( chdir( cwd ) == -1 )
	{
//		fwprintf( stderr, L"Invalid directory: %s\n", cwd );
//		wperror( L"chdir" );
//		return 0;
	}	
	wcwd = str2wcs( cwd );
	if( wcwd )
	{
		env_set( L"PWD", wcwd, ENV_EXPORT );
		free( wcwd );
	}
	env_remove( L"OLDPWD", 0 );

	return 1;
}

/**
   Calls a bunch of init functions, parses the init files and then
   parses commands from stdin or files, depending on arguments
*/

int main( int argc, char **argv )
{
	int res=1;
	int interactive = 1;
	int login=0;
	int my_optind;
	
	char *cmd=0;	

	setlocale(LC_ALL,"");

	while( 1 )
	{
#ifdef __GLIBC__
		static struct option
			long_options[] =
			{
				{
					"command", required_argument, 0, 'c' 
				}
				,
				{
					"interactive", no_argument, 0, 'i' 
				}
				,
				{
					"help", no_argument, 0, 'h' 
				}
				,
				{
					"version", no_argument, 0, 'v' 
				}
				,
				{ 
					0, 0, 0, 0 
				}
			}
		;
		
		int opt_index = 0;
		
		int opt = getopt_long( argc,
							   argv, 
							   "hivc:", 
							   long_options, 
							   &opt_index );
		
#else	
		int opt = getopt( argc,
						  argv, 
						  "hivc:" );
#endif
		if( opt == -1 )
			break;
		
		switch( opt )
		{
			case 0:
				break;
				
			case 'c':		
				cmd = optarg;				
				interactive = 0;
				break;

			case 'h':
				cmd = "help";
				//interactive=0;
				
				break;

			case 'i':
				interactive = 1;
				break;				

			case 'v':
				fwprintf( stderr, 
						  L"%s, version %s\n", 
						  PACKAGE_NAME,
						  PACKAGE_VERSION );
				exit( 0 );				

			case '?':
				return 1;
				
		}		
	}

	my_optind = optind;
	
	login |= strcmp( argv[0], "-") == 0;

	interactive &= (cmd == 0);
	interactive &= (my_optind == argc);
	interactive &= isatty(STDIN_FILENO);	
	
	env_init();
	builtin_init();
	function_init();
	complete_init();
	reader_init();	
	parser_init();

	if( interactive )
		env_set( L"fish_interactive", L"1", ENV_GLOBAL );
	else
		env_remove( L"fish_interactive", 0 );

	if( login )
		env_set( L"fish_login", L"1", ENV_GLOBAL );
	else
		env_remove( L"fish_login", 0 );

	reader_push_current_filename( L"(internal)" );


	if( read_init() )
	{
		if( cmd != 0 )
		{
			wchar_t *cmd_wcs = str2wcs( cmd );
			res = eval( cmd_wcs, 0, TOP );
			free(cmd_wcs);			
		}
		else
		{
			if( my_optind == argc )
			{
				reader_push_current_filename( L"(stdin)" );
				res = reader_read();				
				reader_pop_current_filename();
			}
			else
			{
				char **ptr; 
				char *file = *(argv+1);
				int i; 
				string_buffer_t sb;
				
				if( close( 0 ) )
				{
					wperror(L"close");
					return 1;
				}
				if( open(file, O_RDONLY) == -1 )
				{
					wperror( L"open" );
					return 1;
				}

				sb_init( &sb );
				
				if( *(argv+2))
				{
					for( i=1,ptr = argv+2; *ptr; i++, ptr++ )
					{
						if( i != 1 )
							sb_append( &sb, ARRAY_SEP_STR );
						wchar_t *val = str2wcs( *ptr );
						sb_append( &sb, val );
						free( val );
					}
				
					env_set( L"argv", (wchar_t *)sb.buff, 0 );
					sb_destroy( &sb );
				}
				
				reader_push_current_filename( str2wcs( file ) );
				res = reader_read();
				free(reader_pop_current_filename());


			}
		}
	}

	if( function_exists(L"fish_on_exit"))
	{
		eval( L"fish_on_exit", 0, TOP );
	}
	
	reader_pop_current_filename();	
	
	env_destroy();
	builtin_destroy();
	function_destroy();
	complete_destroy();
	reader_destroy();
	parser_destroy();

	return res;	
}
