/** \file reader.c
	
Functions for reading data from stdin and passing to the
parser. If stdin is a keyboard, it supplies a killring, history,
syntax highlighting, tab-completion and various other interactive features.

Internally the interactive mode functions rely in the functions of the
input library to read individual characters of input.

*/

#include "config.h"

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <termios.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/ioctl.h>
#include <sys/time.h>
#include <unistd.h>
#include <wctype.h>

#if HAVE_NCURSES_H
#include <ncurses.h>
#else
#include <curses.h>
#endif


#if HAVE_TERMIO_H
#include <termio.h>
#endif

#include <term.h>
#include <signal.h>
#include <fcntl.h>
#include <dirent.h>
#include <time.h>
#include <wchar.h>

#include "util.h"
#include "wutil.h"
#include "highlight.h"
#include "reader.h"
#include "proc.h"
#include "parser.h"
#include "complete.h"
#include "history.h"
#include "common.h"
#include "sanity.h"
#include "env.h"
#include "exec.h"
#include "expand.h"
#include "tokenizer.h"
#include "kill.h"
#include "input.h"
#include "function.h"

/**
   Color code for set_color. Does not update the color.
*/

#define FISH_COLOR_IGNORE -1

/**
   Color code for set_color. Sets the default color.
*/
#define FISH_COLOR_RESET -2


/** 
	Maximum length of prefix string when printing completion
	list. Longer prefixes will be ellipsized.
*/
#define PREFIX_MAX_LEN 8

/**
   The default prompt for reading shell commands. This is used by read_i.
*/
#define DEFAULT_PROMPT L"whoami; echo @; hostname|cut -d . -f 1; echo \" \"; pwd; printf '> ';"

/**
   The default title for the reader. This is used by reader_readline.
*/
#define DEFAULT_TITLE L"echo $_ \" \"; pwd"

/**
   A struct describing the state of the interactive reader. These
   states can be stacked, in case reader_readline is called from
   input_read().
*/
typedef struct reader_data
{
	/**
	   Buffer containing the current commandline
	*/
	wchar_t *buff;

	/**
	   The output string, may be different than buff if buff can't fit on one line.
	*/
	wchar_t *output;
	
	/**
	   The number of characters used by the prompt
	*/
	int prompt_width;	
	
	/**
	   Buffer containing the current search item
	*/
	wchar_t *search_buff;
	/**
	   Current size of the buffers
	*/
	size_t buff_sz;
	/**
	   Length of the command in buff. (Not the length of buff itself)
	*/
	size_t buff_len;
	/** 
		The current position of the cursor in buff.
	*/
	size_t buff_pos;
	/** 
		The current position of the cursor in output buffer.
	*/
	size_t output_pos;
	/**
	   Name of the current application 
	*/
	wchar_t *name;

	/** The prompt text */
	wchar_t *prompt;
	
	/**
	   Color is the syntax highlighting for buff.  The format is that
	   color[i] is the classification (according to the enum in
	   highlight.h) of buff[i].
	*/
	int *color;
	/**
	   New color buffer, used for syntax highlighting.
	*/
	int *new_color;

	/**
	   Color for the actual output string.
	*/
	int *output_color;
	
	/**
	   Should the prompt command be reexecuted on the next repaint
	*/
	int exec_prompt;
	
	/**
	   Function for tab completion
	*/
	void (*complete_func)( const wchar_t *, 
						   array_list_t * );
	/**
	   Function for syntax highlighting
	*/
	void (*highlight_func)( wchar_t *, 
							int *, 
							int, 
							array_list_t * );
	/**
	   Function for testing if the string can be returned
	*/
	int (*test_func)( wchar_t * );
	
	/** 
		When this is true, the reader will exit 
	*/
	int end_loop;

	/**
	   Pointer to previous reader_data
	*/
	struct reader_data *next;
}
	reader_data_t;

/**
   The current interactive reading context
*/
static reader_data_t *data=0;


/** 
	Flag for ending non-interactive shell
*/
static int end_loop = 0;


/**
   This struct should be continually updated by signals as the term resizes, and as such always contain the correct current size.
*/
static struct winsize termsize;

/**
   The list containing names of files that are being parsed
*/
static array_list_t current_filename;

/**
   These status buffers are used to check if any output has occurred
   other than from fish's main loop, in which case we need to redraw.
*/
static struct stat prev_buff_1, prev_buff_2, post_buff_1, post_buff_2;



/**
   List containing strings which make up the prompt
*/
static array_list_t prompt_list;

/**
   Stores the previous termios mode so we can reset the modes when
   we execute programs and when the shell exits.
*/	
static struct termios saved_modes;


/**
   Store the pid of the parent process, so the exit function knows whether it should reset the terminal or not.
*/
static pid_t original_pid;

/**
   Interrupted flag. Set to 1 when the user presses \^C. 
*/
static int interupted;

/*
  Prototypes for a bunch of functions 
*/

static void reader_save_status();
static void reader_check_status();
static void reader_super_highlight_me_plenty( wchar_t * buff, int *color, int pos, array_list_t *error );



/**
   Test if there is space between the time fields of struct stat to use
   for sub second information.
*/
static int room_for_usec(struct stat *st)
{
	int res = ((&(st->st_atime) + 2) == &(st->st_mtime) && 
			   (&(st->st_atime) + 4) == &(st->st_ctime));
	return res;
}

/**
   string_buffer used for the reader_readline function
*/
static string_buffer_t *readline_buffer=0;

int reader_get_width()
{
	return termsize.ws_col;
}


int reader_get_height()
{
	return termsize.ws_row;
}	


wchar_t *reader_current_filename()
{
	return (wchar_t *)al_peek( &current_filename );
}


void reader_push_current_filename( wchar_t *fn )
{
	al_push( &current_filename, fn );
}


wchar_t *reader_pop_current_filename()
{
	return (wchar_t *)al_pop( &current_filename );
}


/**
   Make sure buffers are large enough to hold current data plus one extra character.
*/
static int check_size()
{
	if( data->buff_sz < data->buff_len + 2 )
	{
		data->buff_sz = maxi( 128, data->buff_len*2 );

		data->buff = realloc( data->buff, sizeof(wchar_t)*data->buff_sz);
		data->search_buff = realloc( data->search_buff, sizeof(wchar_t)*data->buff_sz);
		data->output = realloc( data->output, sizeof(wchar_t)*data->buff_sz);

		data->color = realloc( data->color, sizeof(int)*data->buff_sz);
		data->new_color = realloc( data->new_color, sizeof(int)*data->buff_sz);
		data->output_color = realloc( data->output_color, sizeof(int)*data->buff_sz);
		
		if( data->buff==0 || data->search_buff==0 || data->color==0 || data->new_color == 0 )
		{
			wperror( L"realloc" );
			fwprintf( stderr, L"Out of memory while allocating read buffer\n" );
			sanity_lose();
			return 0;
		}		
	}	
	return 1;
}


static int force_repaint()
{
	int max_width = reader_get_width() - data->prompt_width;
	int pref_width = my_wcswidth( data->buff ) + (data->buff_pos==data->buff_len);
	return pref_width >= max_width;	
}


/**
   Calculate what part of the buffer should be visible

   \return returns 1 screen needs repainting, 0 otherwise
*/
static int calc_output()
{
	int max_width = reader_get_width() - data->prompt_width;
	int pref_width = my_wcswidth( data->buff ) + (data->buff_pos==data->buff_len);
	if( pref_width <= max_width )
	{
		wcscpy( data->output, data->buff );
		memcpy( data->output_color, data->color, sizeof(int) * data->buff_len );
		data->output_pos=data->buff_pos;
		
		return 1;
	}
	else
	{
		int offset = data->buff_pos;
		int offset_end = data->buff_pos;		
		int w = 0;
		wchar_t *pos=data->output;
		*pos=0;		

		
		w = (data->buff_pos==data->buff_len)?1:wcwidth( data->buff[offset] );		
		while( 1 )
		{
			int inc=0;
			int ellipsis_width;
			
			ellipsis_width = wcwidth(L'\u2026')*((offset?1:0)+(offset_end<data->buff_len?1:0));
			
			if( offset > 0 && (ellipsis_width + w + wcwidth( data->buff[offset-1] ) <= max_width ) )
			{
				inc=1;				
				offset--;
				w+= wcwidth( data->buff[offset]);				
			}

			ellipsis_width = wcwidth(L'\u2026')*((offset?1:0)+(offset_end<data->buff_len?1:0));
			
			if( offset_end < data->buff_len && (ellipsis_width + w + wcwidth( data->buff[offset_end+1] ) <= max_width ) )
			{
				inc = 1;				
				offset_end++;
				w+= wcwidth( data->buff[offset_end]);				
			}
			
			if( !inc )
				break;
			
		}

		data->output_pos = data->buff_pos - offset + (offset?1:0);		

		if( offset )
		{
			wcscpy( data->output, L"\u2026");
		}
		
		wcsncat( data->output, 
				 data->buff+offset,
				 offset_end-offset );
		
		if( offset_end<data->buff_len )
			wcscat( data->output, L"\u2026");
		
		*data->output_color=HIGHLIGHT_NORMAL;		

		memcpy( data->output_color+(offset?1:0),
				data->color+offset, 
				sizeof(int) * (data->buff_len-offset) );		
		return 1;				
	}	
}


/**
   Compare two completions, ignoring their description.
*/
static int fldcmp( wchar_t *a, wchar_t *b )
{
	while( 1 )
	{
		if( *a != *b )
			return *a-*b;
		if( ( (*a == COMPLETE_SEP) || (*a == L'\0') ) && 
			( (*b == COMPLETE_SEP) || (*b == L'\0') ) )
			return 0;
		a++;
		b++;
	}
	
}

/**
   Remove any duplicate completions in the list.
*/
static void remove_duplicates( array_list_t *l )
{
	int in, out;
	wchar_t *prev;
	if( al_get_count( l ) == 0 )
		return;
	
	prev = (wchar_t *)al_get( l, 0 );
	for( in=1, out=1; in < al_get_count( l ); in++ )
	{
		wchar_t *curr = (wchar_t *)al_get( l, in );
		if( fldcmp( prev, curr )==0 )
		{
			free( curr );
		}
		else
		{
			al_set( l, out++, curr );
			prev = curr;
		}		
	}
	al_truncate( l, out );
}


/**
   Write a wide character to FD 1
*/
static int writech( wint_t ch )
{
	static mbstate_t out_state;
	char buff[MB_CUR_MAX];
	size_t bytes = wcrtomb( buff, ch, &out_state );
	int err;
	
	while( (err =write( 1, buff, bytes ) ) )
	{
		if( err >= 0 )
			break;

		if( errno == EINTR )
			continue;

		wperror( L"write" );
		return 1;
	}
	
	return 0;
}

/**
   Write a wide character string to FD 1.   
*/
static void writestr( const wchar_t *str )
{
	while( *str != 0 )
		writech( *(str++) );
}


/**
   Write a wide character string to FD 1. If the string is wider than
   the specified maximum, truncate and ellipsize it.
*/
static void writestr_ellipsis( const wchar_t *str, int max_width )
{
	int written=0;
	int tot = my_wcswidth(str);

	if( tot <= max_width ) 
	{
		writestr( str );
		return;
	}

	while( *str != 0 )
	{
		int w = wcwidth( *str );
		if( written+w+wcwidth( L'\u2026' )>max_width )
			break;
		written+=w;
		writech( *(str++) );
	}

	written += wcwidth( L'\u2026' );	
	writech( L'\u2026' );

	while( written < max_width )
	{
		written++;
		writestr( L" " );
	}
}

/**
   Escape and write a string to fd 1
*/
static int write_escaped_str( const wchar_t *str, int max_len )
{

	wchar_t *out = expand_escape( wcsdup(str), 1 );
	int i;
	int len = my_wcswidth( out );
	int written=0;
	
	if( max_len && (max_len < len))
	{
		for( i=0; (written+wcwidth(out[i]))<=(max_len-3); i++ )
		{
			writech( out[i] );
			written += wcwidth( out[i] );
		}
		for( i=0; i<mini( 3, max_len ); i++ )
		{
			writech( L'.' );
			written++;
		}
	}
	else
	{
		written = len;
		writestr( out );
	}
	
	free( out );
	return written;
}

/**
   We cant use putchar as the output function for putp, since we
   ourselves use FD based IO, which should not be mixed to liberally
   with stream IO. So we provide this putchar workalike.
*/

static int writeb( tputs_arg_t b )
{
	write( 1, &b, 1 );
	return 0;
}

/**
   Write a char * narrow string to FD 1, needed for the terminfo
   strings.
*/
static int writembs( char *str )
{
#ifdef TPUTS_KLUDGE
	write( 1, str, strlen(str));
#else
	tputs(str,1,&writeb);
#endif
	return 0;
}

/**
   parm_ich seems to often be undefined, so we use this
   workalike. Writes the specified number of spaces.
*/
static int writespace( int c )
{
	write( 1, "        ", mini(c,8) );
	if( c>8)
	{
		writespace( c-8);
	}
	return 0;	
}


/**
   Sets the fg and bg color. May be called as often as you like, since
   if the new color is the same as the previous, nothing will be
   written. Negative values for set_color will also be ignored. Since
   the terminfo string this function emits can potentially cause the
   screen to flicker, the function takes care to write as little as
   possible.

   Possible values for color are any form the FISH_COLOR_* enum,
   FISH_COLOR_IGNORE and FISH_COLOR_RESET. FISH_COLOR_IGNORE will
   leave the color unchanged, and FISH_COLOR_RESET will perform an
   exit_attribute_mode, even if set_color thinks it is already in
   FISH_COLOR_NORMAL mode.

   In order to set the color to normal, three terminfo strings may
   have to be written.

   - First a string to set the color, such as set_a_foreground. This
   is needed because otherwise the previous strings colors might be
   removed as well.

   - After that we write the exit_attribute_mode string to reset all
   color attributes.

   - Lastly we may need to write set_a_background or set_a_foreground
   to set the other half of the color pair to what it should be.

   \param c Foreground color.
   \param c2 Background color.
*/


static void set_color( int c, int c2 )
{
	static int last_color = FISH_COLOR_NORMAL, last_color2=FISH_COLOR_NORMAL;
	int bg_set=0, last_bg_set=0;	
	char *fg = 0, *bg=0;
	
	if( (set_a_foreground != 0) && (strlen( set_a_foreground) != 0 ) )
	{
		fg = set_a_foreground;
		bg = set_a_background;
	}
	else if( (set_foreground != 0) && (strlen( set_foreground) != 0 ) )
	{
		fg = set_foreground;
		bg = set_background;
	}

	if( (c == FISH_COLOR_RESET) || (c2 == FISH_COLOR_RESET))
	{
		c = c2 = FISH_COLOR_NORMAL;
		if( fg )
			writembs( tparm( set_a_foreground, 0 ) );		
		writembs( exit_attribute_mode );	
		return;
	}	

	if( last_color2 != FISH_COLOR_NORMAL && 
		last_color2 != FISH_COLOR_RESET && 
		last_color2 != FISH_COLOR_IGNORE )
	{
		/*
		  Background was set
		*/
		last_bg_set=1;		
	}
	
	if( c2 != FISH_COLOR_NORMAL && 
		c2 != FISH_COLOR_RESET && 
		c2 != FISH_COLOR_IGNORE )
	{
		/*
		  Background is set
		*/
		bg_set=1;		
		c = (c2==FISH_COLOR_WHITE)?FISH_COLOR_BLACK:FISH_COLOR_WHITE;
	}
	
	if( (enter_bold_mode != 0) && (strlen(enter_bold_mode) > 0))
	{		
		if(bg_set && !last_bg_set) 
		{
			/*
			  Background color changed and is set, so we enter bold mode to make reading easier
			*/
			writembs( enter_bold_mode );
		}
		if(!bg_set && last_bg_set) 
		{
			/*
			  Background color changed and is no longer set, so we exit bold mode
			*/
			writembs( exit_attribute_mode );
			/*
			  We don't know if exit_attribute_mode resets colors, so
			  we set it to something known.
			*/
			if( fg )
			{
				writembs( tparm( fg, 0 ) );		
				last_color=0;	
			}			
		}
	}

	if( last_color != c )
	{
		if( c==FISH_COLOR_NORMAL )
		{
			if( fg )
				writembs( tparm( fg, 0 ) );		
			writembs( exit_attribute_mode );	
			
			last_color2 = FISH_COLOR_NORMAL;
		}
		else if( ( c >= 0) && ( c < FISH_COLOR_NORMAL ) )
		{
			if( fg )
			{
				writembs( tparm( fg, c ) );
			}
		}
	}

	last_color = c;
	
	if( last_color2 != c2 )
	{
		if( c2 == FISH_COLOR_NORMAL )
		{
			if( bg )
			{
				writembs( tparm( bg, 0 ) );
			}
			
			writembs(exit_attribute_mode);
			if(( last_color != FISH_COLOR_NORMAL ) && fg )
			{
				writembs(tparm( fg, last_color ));
			}

			last_color2 = c2;
		}
		else if ((c2 >= 0 ) &&(c2 < FISH_COLOR_NORMAL))
		{
			if( bg )
			{
				writembs( tparm( bg, c2 ) );
			}
			last_color2 = c2;
		}
	}
}

/**
   Translate a highlighting code as returned by the highlight function
   into a color code which is then passed on to set_color.
*/
static void set_color_translated( int c )
{
	set_color( highlight_get_color( c & 0xff ), 
			   highlight_get_color( (c>>8)&0xff ) );	
}

int reader_interupted()
{
	int res=interupted;
	if( res )
		interupted=0;
	return res;
}

void reader_write_title()
{
	char *n = ttyname( STDIN_FILENO );
	wchar_t *title;
	array_list_t l;

	/*
	  This is a pretty lame heuristic for detecting terminals that do
	  not support setting the title.
	*/
	if( strstr( n, "tty" ) || strstr( n, "/vc/") )
		return;

	
	title = function_exists( L"fish_title" )?L"fish_title":DEFAULT_TITLE;

	if( wcslen( title ) ==0 )
		return;
	
	al_init( &l );
	
	if( exec_subshell( title, &l ) )
	{
		int i;
		writestr( L"\e]2;" );
		for( i=0; i<al_get_count( &l ); i++ )
		{
			writestr( (wchar_t *)al_get( &l, i ) );			
		}
		writestr( L"\7" );
	}		
	al_foreach( &l, (void (*)(const void *))&free );
	al_destroy( &l );
	set_color( FISH_COLOR_RESET, FISH_COLOR_RESET );
}

/**
   Write the prompt to screen. If data->exec_prompt is set, the prompt
   command is first evaluated, and the title will be reexecuted as
   well.
*/
static void write_prompt()
{
	int i;
	set_color( FISH_COLOR_NORMAL, FISH_COLOR_NORMAL );

	/*
	  Check if we need to reexecute the prompt command
	*/
	if( data->exec_prompt )
	{	
		
		al_foreach( &prompt_list, (void (*)(const void *))&free );
		al_truncate( &prompt_list, 0 );	
		
		if( data->prompt )
		{
			if( !exec_subshell( data->prompt, &prompt_list ) )
			{
				/* If executing the prompt fails, make sure we at least don't print any junk */
				al_foreach( &prompt_list, (void (*)(const void *))&free );
				al_destroy( &prompt_list );	
				al_init( &prompt_list );
			}
		}
		data->prompt_width=0;
		for( i=0; i<al_get_count( &prompt_list ); i++ )
		{
			wchar_t *next = (wchar_t *)al_get( &prompt_list, i );
			if( *next == L'\e' )
				continue;
			data->prompt_width += my_wcswidth( next );
		}
		
		data->exec_prompt = 0;
		reader_write_title();
	}

	/*
	  Write out the prompt strings
	*/
	
	for( i=0; i<al_get_count( &prompt_list); i++ )
	{
		writestr( (wchar_t *)al_get( &prompt_list, i ) );			
	}
	set_color( FISH_COLOR_RESET, FISH_COLOR_RESET );

}

/**
   Write the whole command line (but not the prompt) to the screen. Do
   not set the cursor correctly afterwards.
*/
static void write_cmdline()
{
	int i;

	for( i=0; data->output[i]; i++ )
	{
		set_color_translated( data->output_color[i] );		
		writech( data->output[i] );
	}
}

/**
   perm_left_cursor and parm_right_cursor don't seem to be defined as
   often as cursor_left and cursor_right, so we use this workalike.
*/
static void move_cursor( int steps )
{
	int i;
	
	if( steps < 0 ){
		for( i=0; i>steps; i--)
		{
			writembs(cursor_left);
		}
	}
	else
		for( i=0; i<steps; i++)
			writembs(cursor_right);
}


void reader_init()
{
	al_init( &current_filename);
}


void reader_destroy()
{
	al_destroy( &current_filename);
	if( readline_buffer )
	{
		sb_destroy( readline_buffer );
		free( readline_buffer );
		readline_buffer=0;
	}	
}


void do_exit()
{
	if( shell_is_interactive )
		data->end_loop=1;
	else
		end_loop=1;
}

void repaint()
{
	calc_output();	
	set_color( FISH_COLOR_RESET, FISH_COLOR_RESET );
	writech('\r');
	writembs(clr_eol);				
	write_prompt();
	write_cmdline();
	
/*
  fwprintf( stderr, L"Width of \'%ls\' (length is %d): ", 
  &data->buff[data->buff_pos], 
  wcslen(&data->buff[data->buff_pos]));
  fwprintf( stderr, L"%d\n", my_wcswidth(&data->buff[data->buff_pos]));
*/
	
//	if( data->output_pos != data->buff_len )
	{
		//int len = wcslen(&data->buff[data->output_pos]);
		//fwprintf( stderr, L"\'%ls\' len %d\n", &data->buff[data->output_pos], len );
		int steps = my_wcswidth( &data->output[data->output_pos]);
//		if( steps < 0 )
//			fwprintf( stderr, L"woot %d\n", steps );
		if( steps )
			move_cursor( -steps );
	}
	
	set_color( FISH_COLOR_NORMAL, -1 );
	reader_save_status();
}

/**
   Make sure color values are correct, and repaint if they are not.
*/
static void check_colors()
{
	reader_super_highlight_me_plenty( data->buff, data->new_color, data->buff_pos, 0 );
	if( memcmp( data->new_color, data->color, sizeof(int)*data->buff_len )!=0 ) 	
	{
		memcpy( data->color, data->new_color,  sizeof(int)*data->buff_len );
		
		repaint();
	}
}

/**
   Stat stdout and stderr and save result.

   This should be done before calling a function that may cause output.
*/

static void reader_save_status()
{

#if (defined(__FreeBSD__) ||defined(__NetBSD__)) 
	/*
	  This futimes call tries to trick the system into using st_mtime
	  as a tampering flag. This of course only works on systems where
	  futimes is defined, but it should make the status saving stuff
	  failsafe.
	*/
	struct timeval t=
		{
			time(0)-1,
			0
		}
	;
	
	if( futimes( 1, &t ) || futimes( 2, &t ) )
	{
		wperror( L"futimes" );
	}
#endif

	fstat( 1, &prev_buff_1 );
	fstat( 2, &prev_buff_2 );
}

/**
   Stat stdout and stderr and compare result to previous result in
   reader_save_status. Repaint if modification time has changed.

   Unfortunately, for some reason this call seems to give a lot of
   false positives, at least under Linux.
*/

static void reader_check_status()
{
	fflush( stdout );
	fflush( stderr );
	
	fstat( 1, &post_buff_1 );
	fstat( 2, &post_buff_2 );

	int changed = ( prev_buff_1.st_mtime != post_buff_1.st_mtime ) || 
		( prev_buff_2.st_mtime != post_buff_2.st_mtime );
	
	if (room_for_usec( &post_buff_1))
	{
		changed = changed || ( (&prev_buff_1.st_mtime)[1] != (&post_buff_1.st_mtime)[1] ) ||
			( (&prev_buff_2.st_mtime)[1] != (&post_buff_2.st_mtime)[1] );
	}
	
	if( changed )
	{
		repaint();
		set_color( FISH_COLOR_RESET, FISH_COLOR_RESET );
	}
}

/**
   Remove the previous character in the character buffer and on the
   screen using syntax highlighting, etc.
*/
static void remove_backward()
{
	int wdt;
	
	if( data->buff_pos <= 0 )
		return;

	if( data->buff_pos < data->buff_len )
	{
		memmove( &data->buff[data->buff_pos-1], 
				 &data->buff[data->buff_pos],
				 sizeof(wchar_t)*(data->buff_len-data->buff_pos+1) );
		
		memmove( &data->color[data->buff_pos-1], 
				 &data->color[data->buff_pos],
				 sizeof(wchar_t)*(data->buff_len-data->buff_pos+1) );
	}
	data->buff_pos--;
	data->buff_len--;

	wdt=wcwidth(data->buff[data->buff_pos]);	
	move_cursor(-wdt);
	data->buff[data->buff_len]='\0';
//	wcscpy(data->search_buff,data->buff);

	history_reset();
	
	reader_super_highlight_me_plenty( data->buff, 
									  data->new_color, 
									  data->buff_pos,
									  0 );
	if( (!force_repaint()) && ( memcmp( data->new_color, 
									data->color, 
									sizeof(int)*data->buff_len )==0 ) &&
		( delete_character != 0) && (wdt==1) )
	{
		/*
		  Only do this if delete mode functions, and only for a column
		  wide characters, since terminfo seems to break for other
		  characters. This last check should be removed when terminfo
		  is fixed.
		*/
		if( enter_delete_mode != 0 )
			writembs(enter_delete_mode);
		writembs(delete_character);
		if( exit_delete_mode != 0 )
			writembs(exit_delete_mode);
	}
	else
	{
		memcpy( data->color, 
				data->new_color,
				sizeof(int) * data->buff_len );
				
		repaint();
	}
}

/**
   Remove the current character in the character buffer and on the
   screen using syntax highlighting, etc.
*/
static void remove_forward()
{
	if( data->buff_pos >= data->buff_len )
		return;
	
	move_cursor(wcwidth(data->buff[data->buff_pos]));
	data->buff_pos++;
	
	remove_backward();
}

/**
   Insert the character into the command line buffer and print it to
   the screen using syntax highlighting, etc.
*/
static int insert_char( int c )
{
	
	if( !check_size() )
		return 0;
	
	/* Insert space for extra character at the right position */
	if( data->buff_pos < data->buff_len )
	{
		memmove( &data->buff[data->buff_pos+1], 
				 &data->buff[data->buff_pos], 
				 sizeof(wchar_t)*(data->buff_len-data->buff_pos) );
		
		memmove( &data->color[data->buff_pos+1], 
				 &data->color[data->buff_pos], 
				 sizeof(int)*(data->buff_len-data->buff_pos) );
	}
	/* Set character */
	data->buff[data->buff_pos]=c;

    /* Update lengths, etc */
	data->buff_pos++;
	data->buff_len++;
	data->buff[data->buff_len]='\0';
	
	/* Syntax highlight */

	reader_super_highlight_me_plenty( data->buff, 
									  data->new_color,
									  data->buff_pos-1,
									  0 );
	data->color[data->buff_pos-1] = data->new_color[data->buff_pos-1];

	/* Check if the coloring has changed */
	if( (!force_repaint()) && ( memcmp( data->new_color, 
									data->color, 
									sizeof(int)*data->buff_len )==0 ) &&
		( insert_character || 
		  ( data->buff_pos == data->buff_len ) || 
		  enter_insert_mode) )
	{
		/* 
		   Colors look ok, so we set the right color and insert a
		   character
		*/
		set_color_translated( data->color[data->buff_pos-1] );
		if( data->buff_pos < data->buff_len )
		{
			if( enter_insert_mode != 0 )
				writembs(enter_insert_mode);
			else
				writembs(insert_character);
			writech(c);
			if( insert_padding != 0 )
				writembs(insert_padding);
			if( exit_insert_mode != 0 )
				writembs(exit_insert_mode);
		}
		else
			writech(c);
		set_color( FISH_COLOR_NORMAL, -1 );
	}
	else
	{
		/* Nope, colors are off, so we repaint the entire command line */
		memcpy( data->color, data->new_color, sizeof(int) * data->buff_len );
			
		repaint();
	}
//	wcscpy(data->search_buff,data->buff);
	return 1;
}

/**
   Insert the characters of the string into the command line buffer
   and print them to the screen using syntax highlighting, etc.
*/
static int insert_str(wchar_t *str)
{
	while( (*str)!=0 )
		if(!insert_char( *str++ ))
			return 0;
	return 1;
}

/**
   Calculate the length of the common prefix substring of two strings.
*/
static int comp_len( wchar_t *a, wchar_t *b )
{
	int i;
	for( i=0; 
		 a[i] != '\0' && b[i] != '\0' && a[i]==b[i]; 
		 i++ )
		;
	return i;
}

static wchar_t get_quote( wchar_t *cmd, int l )
{
	int i=0;
	wchar_t res=0;

//	fwprintf( stderr, L"Woot %ls\n", cmd );
	
	while( 1 )
	{		
		if( !cmd[i] )
			break;
	
		if( cmd[i] == L'\'' || cmd[i] == L'\"' )
		{
			wchar_t *end = quote_end( &cmd[i] );
			//fwprintf( stderr, L"Jump %d\n",  end-cmd );
			if(( end == 0 ) || (!*end) || (end-cmd > l))
			{
				res = cmd[i];
				break;
			}
//			fwprintf( stderr, L"Jump from %d to %d\n", i, end-cmd+1 );
			i = end-cmd+1;
		}
		else
			i++;
		
	}
	return res;	
}

/**
   Calculates information on the parameter at the specified index.
  
   \param cmd The command to be analyzed
   \param pos An index in the string which is inside the parameter
   \param quote If not 0, store the type of quote this parameter has, can be either ', " or \\0, meaning the string is not quoted.
   \param offset If not 0, get_param will store a pointer to the beginning of the parameter.
   \param string If not o, get_parm will store a copy of the parameter string as returned by the tokenizer.
   \param type If not 0, get_param will store the token type as returned by tok_last.
*/
static void get_param( wchar_t *cmd, 
					   int pos,
					   wchar_t *quote,
					   wchar_t **offset, 
					   wchar_t **string, 
					   int *type )
{
	int prev_pos=0;
	wchar_t last_quote = '\0';
	int unfinished;
	
	tokenizer tok;
	tok_init( &tok, cmd, TOK_ACCEPT_UNFINISHED );
	
	for( ; tok_has_next( &tok ); tok_next( &tok ) )
	{
		if( tok_get_pos( &tok ) > pos )
			break;

		if( tok_last_type( &tok ) == TOK_STRING )
			last_quote = get_quote( tok_last( &tok ), 
									pos - tok_get_pos( &tok ) );
		
		if( type != 0 )
			*type = tok_last_type( &tok );
		if( string != 0 )
			wcscpy( *string, tok_last( &tok ) );
		prev_pos = tok_get_pos( &tok );
	}

	tok_destroy( &tok );
	
	wchar_t c = cmd[pos];
	cmd[pos]=0;
	int cmdlen = wcslen( cmd );
	unfinished = (cmdlen==0);	
	if( !unfinished )
	{
		unfinished = (quote != 0);
		
		if( !unfinished )
		{
			if( wcschr( L" \t\n\r", cmd[cmdlen-1] ) != 0 )
			{
				if( ( cmdlen == 1) || (cmd[cmdlen-2] != L'\\') )
				{
					unfinished=1;
				}
			}
		}
	}

	if( quote )
		*quote = last_quote;
	
	if( offset != 0 )
	{
		if( !unfinished )
		{
			while( (cmd[prev_pos] != 0) && (wcschr( L";|",cmd[prev_pos])!= 0) )
				prev_pos++;
			
			*offset = cmd + prev_pos;
		}
		else
		{
			*offset = cmd + pos;
		}
	}
	cmd[pos]=c;
}

/**
   Insert the string at the current cursor position. The function
   checks if the string is quoted or not and correctly escapes the
   string.

   \param val the string to insert
   \param is_complete Whether this completion is the whole string or
   just the common prefix of several completions. If the former, end by
   printing a space (and an end quote if the parameter is quoted).
*/
static void completion_insert( wchar_t *val, int is_complete )
{
	wchar_t *replaced;

	wchar_t quote;
	
	get_param( data->buff, 
			   data->buff_pos,
			   &quote, 
			   0, 0, 0 );

	if( quote == L'\0' )
	{
		replaced = expand_escape( wcsdup(val), 1 );
	}
	else
	{
		int unescapable=0;
		
		wchar_t *pin, *pout;

		replaced = pout = 
			malloc( sizeof(wchar_t)*(wcslen(val) + 1) );
		
		for( pin=val; *pin; pin++ )
		{
			switch( *pin )
			{
				case L'\n':
				case L'\t':
				case L'\b':
				case L'\r':
					unescapable=1;					
					break;				
				default:
					*pout++ = *pin;
					break;
			}
		}
		if( unescapable )
		{
			free( replaced );
			wchar_t *tmp = expand_escape( wcsdup(val), 1 );
			replaced = wcsdupcat( L" ", tmp );
			free( tmp);
			replaced[0]=quote;			
		}
		else
			*pout = 0;
	}

	if( insert_str( replaced ) )
	{
		
		if( is_complete ) /* Print trailing space since this is the only completion */
		{
			
			if( (quote) &&
				(data->buff[data->buff_pos] != quote ) ) /* This is a quoted parameter, first print a quote */
			{
				insert_char( quote );
			}
			insert_char( L' ' );
		}	
	}
	
	free(replaced);	
}


/**
   Print the list of completions
   
   \param l The list of completions to print
   \param cols number of columns to print in
   \param width An array specifying the width of each column
   \param row_start The first row to print
   \param row_stop the row after the last row to print
   \param prefix The string to print before each completion
   \param is_quoted Whether to print the completions are in a quoted environment
*/

static void completion_print( int cols,
							  int *width, 
							  int row_start,
							  int row_stop,
							  wchar_t *prefix,
							  int is_quoted,
							  array_list_t *l)
{
	
	int rows = (al_get_count( l )-1)/cols+1;
	int i, j;
	int prefix_width= my_wcswidth(prefix);

	for( i = row_start; i<row_stop; i++ )
	{
		for( j = 0; j < cols; j++ )
		{
			wchar_t *el, *el_end;
			
			if( al_get_count( l ) <= j*rows + i )
				continue;
			
			el = (wchar_t *)al_get( l, j*rows + i );
			el_end= wcschr( el, COMPLETE_SEP );
			
			set_color( highlight_get_color(HIGHLIGHT_PAGER_PREFIX),FISH_COLOR_NORMAL );
			
			writestr( prefix );

			set_color( highlight_get_color(HIGHLIGHT_PAGER_COMPLETION),FISH_COLOR_IGNORE );
			
			if( el_end == 0 )
			{
				/* We do not have a description for this completion */
				int written = 0;
				int max_written = width[j] - prefix_width - 2;
				
				if( is_quoted )
				{
					for( i=0; i<max_written; i++ )
					{
						if( !el[i] )
							break;
						writech( el[i] );
						written+= wcwidth( el[i] );
					}					
				}				
				else
				{
					written = write_escaped_str( el, max_written );
				}

				set_color( highlight_get_color( HIGHLIGHT_PAGER_DESCRIPTION ), 
						   FISH_COLOR_IGNORE );
				
				writespace( width[j]-
							written-
							prefix_width );
			}
			else
			{
				int whole_desc_width = my_wcswidth(el_end+1);
				int whole_comp_width;

				/*
				  Temporarily drop the description so that wcswidth et
				  al only calculate the width of the completion.
				*/
				*el_end = L'\0';

				/*
				  Calculate preferred completion width
				*/
				if( is_quoted )
				{
					whole_comp_width = my_wcswidth(el);
				}
				else
				{
					wchar_t *tmp = expand_escape( wcsdup(el), 1 );
					whole_comp_width = my_wcswidth( tmp );
					free(tmp);
				}

				/*
				  Calculate how wide this entry 'wants' to be
				*/
				int pref_width = whole_desc_width + 4 + prefix_width + 2 - 
					(j==cols-1?2:0) + whole_comp_width;

				int comp_width, desc_width;

				if( pref_width <= width[j] )
				{
					/*
					  The entry fits, we give it as much space as it wants
					*/
					comp_width = whole_comp_width;
					desc_width = whole_desc_width;
				}
				else
				{
					/*
					  The completion and description won't fit on the
					  allocated space. Give a maximum of 2/3 of the
					  space to the completion, and whatever is left to
					  the description.
					*/
					int sum = width[j] - prefix_width - 4 - 2 + (j==cols-1?2:0);					
					
					comp_width = maxi( mini( whole_comp_width, 
											 2*sum/3 ), 
									   sum - whole_desc_width );
					desc_width = sum-comp_width;					
				}
				
				/* First we must print the completion. */
				if( is_quoted )
				{
					writestr_ellipsis( el, comp_width);
				}				
				else
				{
					write_escaped_str( el, comp_width );
				}
				
				/* Put the description back */
				*el_end = COMPLETE_SEP;

				/* And print it */
				set_color( highlight_get_color(HIGHLIGHT_PAGER_DESCRIPTION), 
						   FISH_COLOR_IGNORE );
				writespace( maxi( 2, 
								  width[j] 
								  - comp_width
								  - desc_width
								  - 4 
								  - prefix_width
								  + (j==cols-1?2:0) ) );
				/* Print description */
				writestr(L"(");
				writestr_ellipsis( el_end+1, desc_width);
				writestr(L")");
					
				if( j != cols-1)
					writestr( L"  " );
				
			}
		}
		writech( L'\n' );
	}
}

/**
   Calculates how long the specified string would be when printed on the command line. 
   
   \param str The string to be printed.
   \param is_quoted Whether the string would be printed quoted or unquoted
   \param pref_width the preferred width for this item
   \param min_width the minimum width for this item
*/
static void printed_length( wchar_t *str, 
							int is_quoted,
							int *pref_width,
							int *min_width )
{
	if( is_quoted )
	{
		wchar_t *sep = wcschr(str,COMPLETE_SEP);
		if( sep )
		{
			*sep=0;
			int cw = my_wcswidth( str );
			int dw = my_wcswidth(sep+1);

			if( reader_get_width() > 80 )
				dw = mini( dw, reader_get_width()/3 );
			

			*pref_width = cw+dw+4;
			
			if( dw > reader_get_width()/3 )
			{
				dw = reader_get_width()/3;
			}

			*min_width=cw+dw+4;

			*sep= COMPLETE_SEP;
			return;
		}
		else
		{
			*pref_width=*min_width= my_wcswidth( str );
			return;
		}
		
	}
	else
	{
		int comp_len=0, desc_len=0;		
		int has_description = 0;
		while( *str != 0 )
		{
			switch( *str )
			{
				case L'\n':
				case L'\b':
				case L'\r':
				case L'\e':
				case L'\t':
				case L'\\':
				case L'&':
				case L'$':
				case L' ':
				case L'#':
				case L'^':
				case L'<':
				case L'>':
				case L'@':
				case L'(':
				case L')':
				case L'{':
				case L'}':
				case L'?':
				case L'*':
				case L'|':
				case L';':
				case L':':
					if( has_description )
						desc_len++;
					else
						comp_len+=2;
					break;

				case COMPLETE_SEP:
					has_description = 1;
					break;

				default:
					if( has_description )
						desc_len+= wcwidth(*str);
					else
						comp_len+= wcwidth(*str);
					break;
			}
			str++;			
		}
		if( has_description )
		{
			if( reader_get_width() > 80 )
				desc_len = mini( desc_len, reader_get_width()/3 );
			
			*pref_width = comp_len+ desc_len+4;;
			
			comp_len = mini( comp_len, maxi(0,reader_get_width()/3 - 2));
			desc_len = mini( desc_len, maxi(0,reader_get_width()/5 - 4));
			
			*min_width = comp_len+ desc_len+4;
			return;			
		}
		else
		{
			*pref_width=*min_width= comp_len;			
			return;
		}
		
	}
}


/**
   Try to print the list of completions l with the prefix prefix using
   cols as the number of columns. Return 1 if the completion list was
   printed, 0 if the terminal is to narrow for the specified number of
   columns. Always succeeds if cols is 1.

   If all the elements do not fit on the screen at once, make the list
   scrollable using the up, down and space keys to move. The list will
   exit when any other key is pressed.
*/

static int completion_try_print( int cols,
								 wchar_t *prefix,
								 int is_quoted,
								 array_list_t *l )
{
	/*
	  The calculated preferred width of each column
	*/
	int pref_width[32];
	/*
	  The calculated minimum width of each column
	*/
	int min_width[32];
	/*
	  If the list can be printed with this width, width will contain the width of each column
	*/
	int *width=pref_width;
	/*
	  Set to one if the list should be printed at this width
	*/
	int print=0;
	
	int i, j;
	
	int rows = (al_get_count( l )-1)/cols+1;

	int pref_tot_width=0, min_tot_width = 0;
	int prefix_width = my_wcswidth( prefix );

	/*
	  Skip completions on tiny terminals
	*/

	if( reader_get_width() < 16 )
		return 1;
	
	memset( pref_width, 0, sizeof(pref_width) );
	memset( min_width, 0, sizeof(min_width) );

	/* Calculated how wide the list would be */
	for( j = 0; j < cols; j++ )
	{
		for( i = 0; i<rows; i++ )
		{
			int pref,min;
			wchar_t *el;
			if( al_get_count( l ) <= j*rows + i )
				continue;
				
			el = (wchar_t *)al_get( l, j*rows + i );
			printed_length( el, is_quoted, &pref, &min );
			
			pref += prefix_width;
			min += prefix_width;
			if( j != cols-1 )
			{
				pref += 2;
				min += 2;
			}
			min_width[j] = maxi( min_width[j], 
								 min );
			pref_width[j] = maxi( pref_width[j], 
								  pref );
		}
		min_tot_width += min_width[j];
		pref_tot_width += pref_width[j];
	}
	/*
	  Force fit if one column
	*/
	if( cols == 1) 
	{
		if( pref_tot_width > reader_get_width() )
		{
			pref_width[0] = termsize.ws_col;
		}
		width = pref_width;
		print=1;
	}
	else if( pref_tot_width <= termsize.ws_col )
	{
		/* Terminal is wide enough. Print the list! */	
		width = pref_width;
		print=1;
	}
	else
	{
		int next_rows = (al_get_count( l )-1)/(cols-1)+1;
/*		fwprintf( stderr,
  L"cols %d, min_tot %d, term %d, rows=%d, nextrows %d, termrows %d, diff %d\n",
  cols,
  min_tot_width, termsize.ws_col,
  rows, next_rows, reader_get_height(),
  pref_tot_width-reader_get_width() );
*/
		if( min_tot_width < termsize.ws_col && 
			( ( (rows < reader_get_height()) && (next_rows >= reader_get_height() ) ) ||
			  ( pref_tot_width-reader_get_width()< 4 && cols < 3 ) ) )
		{
			/*
			  Terminal almost wide enough, or squeezing makes the whole list fit on-screen
			*/
			int tot_width = min_tot_width;
			width = min_width;
			
			while( tot_width < termsize.ws_col )
			{
				for( i=0; (i<cols) && ( tot_width < termsize.ws_col ); i++ )
				{
					if( width[i] < pref_width[i] )
					{
						width[i]++;
						tot_width++;
					}
				}	
			}		
			print=1;
		}
	}

//	return cols==1;	

	if( print )
	{
		
		if( rows < termsize.ws_row )
		{
			/* List fits on screen. Print it and leave */
			completion_print( cols, width, 0, rows, prefix, is_quoted, l);			
		}
		else
		{
			int npos, pos = 0;
			int do_loop = 1;
			completion_print( cols, 
							  width, 
							  0,
							  termsize.ws_row-1, 
							  prefix,
							  is_quoted,
							  l);			
			/* 
			   List does not fit on screen. Print one screenfull and
			   leave a scrollable interface
			*/
			while(do_loop)
			{
				wchar_t msg[10];
				int percent = 100*pos/(rows-termsize.ws_row+1);
				set_color( FISH_COLOR_BLACK, 
						   highlight_get_color(HIGHLIGHT_PAGER_PROGRESS) );
				swprintf( msg, 12,
						  L" %ls(%d%%) \r", 
						  percent==100?L"":(percent >=10?L" ": L"  "), 
						  percent );
				writestr(msg);
				set_color( FISH_COLOR_NORMAL, FISH_COLOR_NORMAL );
				int c = input_readch();
				
				switch( c )
				{
					case R_HISTORY_SEARCH_BACKWARD:
					{
						if( pos > 0 )
						{
							pos--;
							writembs(tparm( cursor_address, 0, 0));
							writembs(scroll_reverse);
							completion_print( cols,
											  width,
											  pos,
											  pos+1,
											  prefix, 
											  is_quoted,
											  l );
							writembs( tparm( cursor_address, 
											 termsize.ws_row-1, 0) );
							writembs(clr_eol );
											
						}
						
						break;
					}
					
					case R_HISTORY_SEARCH_FORWARD:
					{
						if( pos <= (rows - termsize.ws_row ) )
						{
							pos++;
							completion_print( cols,
											  width,
											  pos+termsize.ws_row-2,
											  pos+termsize.ws_row-1,
											  prefix, 
											  is_quoted,
											  l );
						}
						break;
					}
					
					case R_COMPLETE:
					case R_END_OF_HISTORY:
					case ' ':
					{
						
						npos = mini( rows - termsize.ws_row+1,
									 pos + termsize.ws_row-1 );
						if( npos != pos )
						{
							pos = npos;
							completion_print( cols,
											  width,
											  pos,
											  pos+termsize.ws_row-1,
											  prefix, 
											  is_quoted,
											  l );
						}
						else
						{
							writembs( flash_screen );
						}
						
						break;
					}
					
					case R_BEGINNING_OF_HISTORY:
					{
						npos = maxi( 0, 
									 pos - termsize.ws_row+1 );
						
						if( npos != pos )
						{
							pos = npos;
							completion_print( cols,
											  width,
											  pos,
											  pos+termsize.ws_row-1,
											  prefix, 
											  is_quoted,
											  l );
						}
						else
						{
							writembs( flash_screen );
						}						
						break;
					}
					
					default:
					{
						input_unreadch( c );
						do_loop = 0;
						break;
					}
					
				}
				
			}
			writembs(clr_eol);				
		}
		return 1;

	}
	return 0;
}
/**
   Substitute tabs and newlines with whitespace in description
*/

static void mangle_descriptions( array_list_t *l )
{
	int i;
	for( i=0; i<al_get_count( l ); i++ )
	{
		wchar_t *next = (wchar_t *)al_get(l, i);
		while( *next != COMPLETE_SEP && *next != 0 )
			next++;
		
		while( *next != 0 )
		{
			switch( *next )
			{
				case L'\t':
				case L'\n':
				case L'\r':
					*next=L' ';
					break;
			}			
			next++;			
		}
	}
	
}

/**
   Handle the completion list comp. This means the following:

   - If the list is empty, flash the terminal.
   - If the list contains one element, write the whole element, and if
   the element does not end on a '/', '@', ':', or a '=', also write a trailing
   space.
   - If the list contains multiple elements with a common prefix, write
   the prefix.
   - If the list contains multiple elements without 
   a common prefix, call completion_try_print to show a list of the 
   available completions.
*/


static int handle_completions( array_list_t *comp )
{
	int i;
	
	if( al_get_count( comp ) == 0 )
	{
		if( flash_screen != 0 )
			writembs( flash_screen );
		return 0;
	}
	else if( al_get_count( comp ) == 1 )
	{
		wchar_t *comp_str = wcsdup((wchar_t *)al_get( comp, 0 ));
		wchar_t *woot = wcschr( comp_str, COMPLETE_SEP );
		if( woot != 0 )
			*woot = L'\0';
		completion_insert( comp_str, 
						   ( wcslen(comp_str) == 0 ) || 
						   ( wcschr( L"/=@:", 
									 comp_str[wcslen(comp_str)-1] ) == 0 ) );
		free( comp_str );
		return 1;
	}
	else 
	{
		wchar_t *base = wcsdup( (wchar_t *)al_get( comp, 0 ) );
		int len = wcslen( base );
		for( i=1; i<al_get_count( comp ); i++ )
		{
			int new_len = comp_len( base, (wchar_t *)al_get( comp, i ) );
			len = new_len < len ? new_len: len;
		}
		if( len > 0 )
		{
			base[len]=L'\0';
			wchar_t *woot = wcschr( base, COMPLETE_SEP );
			if( woot != 0 )
				*woot = L'\0';
			completion_insert(base, 0);
		}
		else
		{
			/* 
			   There is no common prefix in the completions, and show_list
			   is true, so we print the list 
			*/
			int len;
			wchar_t * prefix;
			wchar_t * prefix_start;
			get_param( data->buff, 
					   data->buff_pos, 
					   0,
					   &prefix_start,
					   0,
					   0 );

			mangle_descriptions( comp );
			
			len = &data->buff[data->buff_pos]-prefix_start;

			if( len <= PREFIX_MAX_LEN )
			{
				prefix = malloc( sizeof(wchar_t)*(len+1) );
				wcsncpy( prefix, prefix_start, len );
				prefix[len]=L'\0';
			}
			else
			{
				prefix = wcsdupcat( L"\u2026", 
									prefix_start + (len - PREFIX_MAX_LEN+1) );			
			}

			{
				int is_quoted;
			
				wchar_t quote;
				get_param( data->buff, data->buff_pos, &quote, 0, 0, 0 );
				is_quoted = (quote != L'\0');
			
				writech(L'\n');

				/*
				  Try to print a list of completions. First try with five
				  columns, then four, etc. completion_try_print always
				  succeeds with one column.
				*/ 
				for( i = 6; i>0; i-- )
				{
					if( completion_try_print( i, prefix, is_quoted, comp ) )
						break;
				}
			
			}
		
			free( prefix );
			
			repaint();
			
		}
		
		free( base );
		return len;
	}
}

/**
   Respond to a winch signal by checking the terminal size
*/
static void handle_winch( int sig )
{
	wchar_t tmp[64];
	if (ioctl(1,TIOCGWINSZ,&termsize)!=0) 
	{
		return;
	}	
	swprintf(tmp, 64, L"%d", termsize.ws_row );
	env_set( L"LINES", tmp, 0 );
	swprintf(tmp, 64, L"%d", termsize.ws_col );
	env_set( L"COLUMNS", tmp, 0 );
}

/**
   Interactive mode ^C handler. Respond to int signal by setting
   interrupted-flag and stopping all loops and conditionals.
*/
static void handle_int( int sig )
{
	interupted=1;
	
	block_t *c = current_block;
	while( c )
	{
		c->skip=1;
		c=c->outer;
	}
	
}

/**
   Reset the terminal. This function is placed in the list of
   functions to call when exiting by using the atexit function. It
   checks whether it is the original parent process that is exiting
   and not a subshell, and if it is the parent, it restores the
   terminal.
*/
static void exit_func()
{
	if( getpid() == original_pid )
		tcsetattr(0, TCSANOW, &saved_modes);
}

/**
   Sets appropriate signal handlers.
*/
static void set_signal_handlers()
{
	struct sigaction act;
	sigemptyset( & act.sa_mask );
	act.sa_flags=0;		
	act.sa_handler=SIG_DFL;

	/*
	  First reset everything
	*/
	sigaction( SIGINT, &act, 0);
	sigaction( SIGQUIT, &act, 0);
	sigaction( SIGTSTP, &act, 0);
	sigaction( SIGTTIN, &act, 0);
	sigaction( SIGTTOU, &act, 0);
	sigaction( SIGCHLD, &act, 0);
	
	if( shell_is_interactive )
	{
		
		/* 
		   Interactive mode. Ignore interactive signals.  We are a
		   shell, we know whats best for the user. ;-)
		*/
	
		act.sa_handler=SIG_IGN;

		sigaction( SIGINT, &act, 0);
		sigaction( SIGQUIT, &act, 0);
		sigaction( SIGTSTP, &act, 0);
		sigaction( SIGTTIN, &act, 0);
		sigaction( SIGTTOU, &act, 0);
		
		act.sa_handler = &handle_int; 
		act.sa_flags = 0;
		if( sigaction( SIGINT, &act, 0) )
		{
			wperror( L"sigaction" );
			exit(1);
		}
		
		act.sa_sigaction = &job_handle_signal; 
		act.sa_flags = SA_SIGINFO;
		if( sigaction( SIGCHLD, &act, 0) )
		{
			wperror( L"sigaction" );
			exit(1);
		}

		act.sa_flags = 0;
		act.sa_handler= &handle_winch;
		if( sigaction( SIGWINCH, &act, 0 ) )
		{
			wperror( L"sigaction" );
			exit(1);
		}
	}
	else
	{
		/*
		  Non-interactive. Ignore interrupt, check exit status of
		  processes to determine result instead.
		*/
		act.sa_handler=SIG_IGN;
		
		sigaction( SIGINT, &act, 0);
		sigaction( SIGQUIT, &act, 0);
		
		act.sa_handler=SIG_DFL;
	}
}


/**
   Initialize data for interactive use
*/
static void reader_interactive_init()
{
	/* See if we are running interactively.  */
	pid_t shell_pgid;

	input_init();
	kill_init();	
	shell_pgid = getpgrp ();
	
	/* Loop until we are in the foreground.  */
	while (tcgetpgrp( 0 ) != shell_pgid)
	{
		kill (- shell_pgid, SIGTTIN);
	}
	
	/* Put ourselves in our own process group.  */
	shell_pgid = getpid ();
	if( getpgrp() != shell_pgid )
	{
		if (setpgid (shell_pgid, shell_pgid) < 0)
		{
			wperror( L"setpgid" );
			fwprintf (stderr,
					  L"Couldn't put the shell in its own process group");
			exit (1);
		}
	}
	
	
	/* Grab control of the terminal.  */
	if( tcsetpgrp (STDIN_FILENO, shell_pgid) )
	{
		wperror( L"tcsetpgrp" );
		exit(1);
	}
	
	
	al_init( &prompt_list );
	history_init();
	
	
	handle_winch( 0 );                /* Set handler for window change events */	
	tcgetattr(0,&shell_modes);        /* get the current terminal modes */
	memcpy( &saved_modes, 
			&shell_modes, 
			sizeof(saved_modes));     /* save a copy so we can reset the terminal later */
	
	shell_modes.c_lflag &= ~ICANON;   /* turn off canonical mode */
	shell_modes.c_lflag &= ~ECHO;     /* turn off echo mode */
    shell_modes.c_cc[VMIN]=1;
    shell_modes.c_cc[VTIME]=0;
	
    if( tcsetattr(0,TCSANOW,&shell_modes))      /* set the new modes */
    {
        wperror(L"tcsetattr");
        exit(1);
    }
	
	/* We need to know the parents pid so we'll know if we are a subshell */
	original_pid = getpid();                    
	
	if( atexit( &exit_func ) )
		fwprintf( stderr, L"Cannot set exit function\n" );

	env_set( L"_", L"fish", ENV_GLOBAL );
}

/**
   Destroy data for interactive use
*/
static void reader_interactive_destroy()
{
	kill_destroy();	
	al_foreach( &prompt_list, (void (*)(const void *))&free );
	al_destroy( &prompt_list );
	history_destroy();

	writestr( L"\n" );
	set_color( FISH_COLOR_RESET, FISH_COLOR_RESET );
	input_destroy();
}


void reader_sanity_check()
{
	if( shell_is_interactive)
	{
		if(!( data->buff_pos <= data->buff_len ))
			sanity_lose();
		if(!( data->buff_pos >=0 ))
			sanity_lose();
		if(!( data->buff_len >=0 ))
			sanity_lose();
		if(!( data->buff_len == wcslen( data->buff ) ))
			sanity_lose();
	}
}

/**
   Set the specified string from the history as the current buffer. Do
   not modify prefix_width.
*/
static void handle_history( const wchar_t *new_str )
{
	data->buff_len = wcslen( new_str );
	check_size();
	wcscpy( data->buff, new_str );
	data->buff_pos=wcslen(data->buff);
	reader_super_highlight_me_plenty( data->buff, data->color, data->buff_pos, 0 );	
		
	repaint();		
}

static wchar_t *reader_get_current_token()
{
	
	
}

static void reader_replace_current_token()
{
	
}




/**
   Move buffer position one word or erase one word. This function updates both
   the internal buffer an the screen. It is used by M-left, M-right
   and ^W to do block movement or block erase.

   \param dir Direction to move/erase. 0 means move left, 1 means move right.
   \param erase Whether to erase the characters along the way or only move past them.
   
*/
static void move_word( int dir, int erase )
{
	int end_buff_pos=data->buff_pos;
	int mode=0;
	int step = dir?1:-1;
	
	while( mode < 3 )
	{
		if( end_buff_pos < data->buff_len )
		{
			switch( mode )
			{
				case 0:
					if( !iswspace(data->buff[end_buff_pos] ) )
						mode++;
					break;
					
				case 1:
					if( iswspace(data->buff[end_buff_pos] ) )
						mode++;
					break;
					
				case 2:
					if( !iswspace(data->buff[end_buff_pos] ) )
					{
						mode++;
						if( !dir )
							end_buff_pos-=step;
					}
					break;
			}
		}
		
		if( mode==3)
			break;

		if( !dir )
		{
			if( end_buff_pos == 0 )
				break;
		}
		else
		{
			if( end_buff_pos == data->buff_len )
				break;
		}
		end_buff_pos+=step;
	}

	if( erase )
	{
		int remove_count = abs(data->buff_pos - end_buff_pos);
		int first_char = mini( data->buff_pos, end_buff_pos );	
		wchar_t *woot = wcsndup( data->buff + first_char, remove_count);
//		fwprintf( stderr, L"Remove from %d to %d\n", first_char, first_char+remove_count );
		
		kill_add( woot );
		free( woot );
		memmove( data->buff + first_char, data->buff + first_char+remove_count, sizeof(wchar_t)*(data->buff_len-first_char-remove_count) );
		data->buff_len -= remove_count;
		data->buff_pos = first_char;
		data->buff[data->buff_len]=0;
		
		reader_super_highlight_me_plenty( data->buff, data->color, data->buff_pos, 0 );	
				
		repaint();		
	}
	else
	{
/*		move_cursor(end_buff_pos-data->buff_pos);
  data->buff_pos = end_buff_pos;
*/
		if( end_buff_pos < data->buff_pos )
		{
			while( data->buff_pos != end_buff_pos )
			{
				data->buff_pos--;
				move_cursor( -wcwidth(data->buff[data->buff_pos]));
			}				
		}
		else
		{
			while( data->buff_pos != end_buff_pos )
			{
				move_cursor( wcwidth(data->buff[data->buff_pos]));
				data->buff_pos++;
				check_colors();
				
			}
		}
				
		repaint();		
//		check_colors();
	}
	history_reset();
}


wchar_t *reader_get_buffer()
{
	return data->buff;
}

void reader_set_buffer( wchar_t *b, int p )
{
	int l = wcslen( b );
	data->buff_len = l;
	check_size();
	wcscpy( data->buff, b );
	
	if( p>=0 )
	{
		data->buff_pos=p;
	}
	else
	{
		data->buff_pos=l;
//		fwprintf( stderr, L"Pos %d\n", l );
	}
	
	reader_super_highlight_me_plenty( data->buff, 
									  data->color, 
									  data->buff_pos, 
									  0 );
}


int reader_get_cursor_pos()
{
	return data->buff_pos;
}


void reader_run_command( wchar_t *cmd )
{
	struct termios old_modes;
	tcgetattr(0,&old_modes);        /* get the current terminal modes */

	set_color(FISH_COLOR_NORMAL, FISH_COLOR_NORMAL);


	while( 1 )
	{
		if(	tcsetattr(0,TCSANOW,&saved_modes) )
		{
			if( errno != EINTR )
			{
				fwprintf( stderr, L"Could not set terminal mode for new job\n" );
				wperror( L"tcsetattr" );
				break;
			}
		}
		else
			break;
	}					
	
	wchar_t *ft = tok_first( cmd );
	if( ft != 0 )
		env_set( L"_", ft, ENV_GLOBAL );
	free(ft);
				   
	reader_write_title();
					
	if( eval( cmd, 0, TOP ) == 0 )
	{
		job_do_notification();
	}

	
	while( 1 )
	{
		if(	tcsetattr(0,TCSANOW,&shell_modes) )
		{
			if( errno != EINTR )
			{
				fwprintf( stderr, L"Could not set terminal mode for shell\n" );
				wperror( L"tcsetattr" );
				break;
			}
		}
		else
			break;
	}

	env_set( L"_", L"fish", ENV_GLOBAL );
	handle_winch( 0 );

    if( tcsetattr(0,TCSANOW,&old_modes))      /* return to previous mode */
    {
        wperror(L"tcsetattr");
        exit(1);
    }
	
#ifdef HAVE__PROC_SELF_STAT
	proc_update_jiffies();
#endif

}


/**
   Test if the given shell command contains errors. Uses parser_test for testing.
*/

  static int shell_test( wchar_t *b )
  {
	  return !wcslen(b);	  
  }

/**
   Test if the given string contains error. Since this is the error
   detection for general purpose, there are no invalid strings, so
   this function always returns false.
*/
static int default_test( wchar_t *b )
{
	return 0;
}

void reader_push( wchar_t *name )
{
	reader_data_t *n = calloc( 1, sizeof( reader_data_t ) );
	n->name = wcsdup( name );
	n->next = data;
	data=n;
	check_size();
	data->buff[0]=data->search_buff[0]=0;
	data->exec_prompt=1;
	
	if( data->next == 0 )
	{
		reader_interactive_init();
	}
	reader_set_highlight_function( &highlight_universal );
	reader_set_test_function( &default_test );	
	reader_set_prompt( L"" );
	history_set_mode( name );
	
}

void reader_pop()
{
	reader_data_t *n = data;

	if( data == 0 )
	{
		fwprintf( stderr, L"Pop null reader block\n" );
		sanity_lose();
		return;
	}
	
	data=data->next;
	
	free(n->name );
	free( n->prompt );
	free( n->buff );
	free( n->color );
	free( n->new_color );
	free( n->search_buff );
	free( n->output );
	free( n->output_color );
	free(n);
	
	if( data == 0 )
	{
		reader_interactive_destroy();
	}
	else
	{
		history_set_mode( data->name );
		data->exec_prompt=1;
	}
}

void reader_set_prompt( wchar_t *new_prompt )
{
 	free( data->prompt );
	data->prompt=wcsdup(new_prompt);
}

void reader_set_complete_function( void (*f)( const wchar_t *, 
											  array_list_t * ) )
{
	data->complete_func = f;
}

void reader_set_highlight_function( void (*f)( wchar_t *, 
											   int *,
											   int, 
											   array_list_t * ) )
{
	data->highlight_func = f;
}

void reader_set_test_function( int (*f)( wchar_t * ) )
{
	data->test_func = f;
}

/**
   Call specified external highlighting function and then do search highlighting.
*/
static void reader_super_highlight_me_plenty( wchar_t * buff, int *color, int pos, array_list_t *error )
{
	data->highlight_func( buff, color, pos, error );
	if( wcslen(data->search_buff) )
	{
		wchar_t * match = wcsstr( buff, data->search_buff );
		if( match )
		{
			int start = match-buff;
			int count = wcslen(data->search_buff );
			int i;
//			fwprintf( stderr, L"WEE color from %d to %d\n", start, start+count );
			
			for( i=0; i<count; i++ )
			{
				/* 
				   Do not overwrite previous highlighting color
				*/
				if( color[start+i]>>8 == 0 )
				{
					color[start+i] |= HIGHLIGHT_SEARCH_MATCH<<8;
				}
			}
		}
	}
}


int exit_status()
{
	if( shell_is_interactive )
		return first_job == 0 && data->end_loop;
	else
		return end_loop;
}

/**
   Read interactively. Read input from stdin while providing editing facilities.
*/
static int read_i()
{	
	int prev_end_loop=0;
	

	reader_push(L"fish");
	reader_set_complete_function( &complete );
	reader_set_highlight_function( &highlight_shell );
	reader_set_test_function( &shell_test );

	data->prompt_width=60;

	while( (!data->end_loop) && (!sanity_check()) )
	{
		wchar_t *prompt;
		wchar_t *tmp;

		if( function_exists( L"fish_prompt" ) )
			reader_set_prompt( L"fish_prompt" );
		else
			reader_set_prompt( DEFAULT_PROMPT );
			
		/*
		  Put buff in temporary string and clear buff, so
		  that we can handle a call to reader_set_buffer
		  during evaluation.
		*/
				
		tmp = wcsdup( reader_readline() );
		
		data->buff_pos=data->buff_len=0;		
		data->buff[data->buff_len]=L'\0';
		if( function_exists(L"fish_on_exec"))
		{
			eval( L"fish_on_exec", 0, TOP );
		}
		reader_run_command( tmp );
		free( tmp );
		if( function_exists(L"fish_on_return"))
		{
			eval( L"fish_on_return", 0, TOP );
		}
	
		if( data->end_loop)
		{
			if( !prev_end_loop && first_job != 0 )
			{
				writestr(L"There are stopped jobs\n");
				write_prompt();				
				data->end_loop = 0;
				prev_end_loop=1;
			}
		}	
		else
		{
			prev_end_loop=0;		
		}					
		
		error_reset();
	}
	reader_pop();
	return 0;
}


wchar_t *reader_readline()
{
	
	wchar_t c;	
	int i;
	int last_char=0, yank=0;
	wchar_t *yank_str;
	array_list_t comp;
	int comp_empty=1;		
	int finished=0;
	struct termios old_modes;
	
	check_size();
	data->search_buff[0]=data->buff[data->buff_len]='\0';
	
	
	al_init( &comp );
	
	data->exec_prompt=1;
	
	reader_super_highlight_me_plenty( data->buff, data->color, data->buff_pos, 0 );	
	repaint();

	tcgetattr(0,&old_modes);        /* get the current terminal modes */
    if( tcsetattr(0,TCSANOW,&shell_modes))      /* set the new modes */
    {
        wperror(L"tcsetattr");
        exit(1);
    }
	
	while( !finished && !data->end_loop)
	{

		/*
		  Save the terminal status so we know if we have to redraw
		*/
		
		/* 
		   Sometimes strange input sequences seem to generate a zero
		   byte. I believe these simply mean a character was pressed
		   but it should be ignored. (Example: Trying to add a tilde
		   (~) to digit)
		*/

		reader_save_status();
		while( (c=input_readch()) == 0 )
			;

		reader_check_status();

		if( (last_char == R_COMPLETE) && (c != R_COMPLETE) && (!comp_empty) )
		{
			al_foreach( &comp, (void (*)(const void *))&free );
			al_truncate( &comp, 0 );
			comp_empty = 1;
		}

		if( (c != R_HISTORY_SEARCH_BACKWARD) &&
			(c != R_HISTORY_SEARCH_FORWARD) )
		{
			data->search_buff[0]=0;
		}
		
		if( last_char != R_YANK && last_char != R_YANK_POP )
			yank=0;
		switch (c)
		{
			/* go to beginning of line*/

			case R_BEGINING_OF_LINE:
			{
				data->buff_pos = 0;				
				
				repaint();
				break;
			}
			
				/* go to EOL*/
			case R_END_OF_LINE:
			{
				data->buff_pos = data->buff_len;				
				
				repaint();				
				break;
			}
			
			case R_EXPLAIN:
			{
				int i;
					
				array_list_t l;
				al_init( &l );
				reader_super_highlight_me_plenty( data->buff, data->new_color, data->buff_pos, &l );
				if( al_get_count( &l ) > 0 )
				{
					set_color(FISH_COLOR_NORMAL, FISH_COLOR_NORMAL);
					writestr( L"\n" );
					for( i=0; i<al_get_count( &l ); i++ )
					{
						wchar_t *str = (wchar_t *)al_get( &l, i );
						writestr( str );
						writestr( L"\n" );
					}

					repaint();
					al_foreach( &l, (void (*)(const void *))&free );
				}
					
				al_destroy( &l );
					
				break;
			}
				
			/* complete */
			case R_COMPLETE:
			{
				
//					fwprintf( stderr, L"aaa\n" );
				if( !data->complete_func )
					break;

				if( !comp_empty && last_char == R_COMPLETE )
					break;

				if( comp_empty )
				{						
					wchar_t *buffcpy = wcsdup( data->buff );
					wchar_t *begin, *end;

					while( 1 )
					{
						int bc, ec;
						
						if( expand_locate_subshell( buffcpy, 
													&begin, 
													&end,
													1 ) <= 0)
						{
							begin=buffcpy;
							end = buffcpy + wcslen(data->buff);
							break;
						}
						bc = begin-buffcpy;
						ec = end-buffcpy;
						if(( bc < data->buff_pos ) && (ec >= data->buff_pos) )
						{
							end--;
							begin++;
								
							//fwprintf( stderr, L"Subshell!\n" );
							break;
						}
						*begin=0;
					}
						
					int len = data->buff_pos-(begin-buffcpy); 
					wcsncpy( buffcpy, 
							 data->buff+(begin-buffcpy), 
							 len );
					buffcpy[len]=0;
					
					//fwprintf( stderr, L"String is %ls\n", buffcpy );
					
					reader_save_status();
					data->complete_func( buffcpy, &comp );					
					reader_check_status();
						
					sort_list( &comp );
					remove_duplicates( &comp );
						
					free( buffcpy );
				}
				if( (comp_empty =
					 handle_completions( &comp ) ) )
				{
					al_foreach( &comp, (void (*)(const void *))&free );
					al_truncate( &comp, 0 );
				}
				
				break;
			}		
				/* kill*/
			case R_KILL_LINE:
			{
				kill_add( &data->buff[data->buff_pos] );
				data->buff_len = data->buff_pos;
				data->buff[data->buff_len]=L'\0';
				
				
				repaint();
//				wcscpy(data->search_buff,data->buff);
				break;
			}
			
			case R_BACKWARD_KILL_LINE:
			{
				wchar_t prev = data->buff[data->buff_pos];
				data->buff[data->buff_pos]=0;
				kill_add( data->buff );
				data->buff[data->buff_pos]=prev;
				data->buff_len = wcslen(data->buff +data->buff_pos);
				memmove( data->buff, data->buff +data->buff_pos, sizeof(wchar_t)*data->buff_len );
				data->buff[data->buff_len]=L'\0';
				data->buff_pos=0;
				reader_super_highlight_me_plenty( data->buff, data->color, data->buff_pos, 0 );	
				
				repaint();
//				wcscpy(data->search_buff,data->buff);
				break;
			}
			
			case R_KILL_WHOLE_LINE:
			{
				kill_add( data->buff );
				data->buff_len = data->buff_pos = 0;
				data->buff[data->buff_len]=L'\0';
				reader_super_highlight_me_plenty( data->buff, data->color, data->buff_pos, 0 );	
				
				repaint();
//				wcscpy(data->search_buff,data->buff);
				break;
			}
			
			/* yank*/
			case R_YANK:
				yank_str = kill_yank();
				insert_str( yank_str );
				yank = wcslen( yank_str );
//				wcscpy(data->search_buff,data->buff);
				break;
				
				/* rotate killring*/	
			case R_YANK_POP:
				if( yank )
				{
					for( i=0; i<yank; i++ )
						remove_backward();
						
					yank_str = kill_yank_rotate();
					insert_str(yank_str);
					yank = wcslen(yank_str);
				}					
				break;

				/* Escape, do nothing*/
			case L'\e':
				break;

				/* delete backward*/
			case R_BACKWARD_DELETE_CHAR:	
				remove_backward();
				break;
					
				/* delete forward*/
			case R_DELETE_CHAR:
				remove_forward();
				break;

				/* exit, but only if line is empty */
			case R_EXIT:

				if( data->buff_len == 0 )
				{
					writestr( L"\n" );
					data->end_loop=1;
				}
				break;						

				/* Newline, evaluate*/
			case L'\n':
			{
				data->buff[data->buff_len]=L'\0';

				if( !data->test_func( data->buff ) )
				{
					
					if( wcslen( data->buff ) )
					{
//						wcscpy(data->search_buff,L"");
						history_add( data->buff );
					}
					history_reset();
					finished=1;
					data->buff_pos=data->buff_len;
					check_colors();
					writestr( L"\n" );
				}
				else
					repaint();
								
				break;
			}
			
			/* History up*/
			case R_HISTORY_SEARCH_BACKWARD:
//					fwprintf( stderr, L"Search history for \'%ls\' %d long\n", data->search_buff, wcslen(data->search_buff) );
				if( (last_char != R_HISTORY_SEARCH_BACKWARD) &&
					(last_char != R_HISTORY_SEARCH_FORWARD) )
				{
					wcscpy(data->search_buff, data->buff );
					data->search_buff[data->buff_pos]=0;					
				}
				
				handle_history(history_prev_match(data->search_buff));
				break;
				
				/* History down*/
			case R_HISTORY_SEARCH_FORWARD:				
				if( (last_char != R_HISTORY_SEARCH_BACKWARD) &&
					(last_char != R_HISTORY_SEARCH_FORWARD) )
				{
					wcscpy(data->search_buff, data->buff );
					data->search_buff[data->buff_pos]=0;
				}

				handle_history(history_next_match(data->search_buff));
				break;
					
				/* Move left*/
			case R_BACKWARD_CHAR:
				if( data->buff_pos > 0 )
				{
					data->buff_pos--;
					if( !force_repaint() )
					{
						move_cursor( -wcwidth(data->buff[data->buff_pos]));
						check_colors();
					}
					else
					{
						repaint();
					}
				}				
				break;
				
				/* Move right*/
			case R_FORWARD_CHAR:
				if( data->buff_pos < data->buff_len )
				{
					if( !force_repaint() )
					{
						move_cursor( wcwidth(data->buff[data->buff_pos]));
						data->buff_pos++;
						check_colors();
					}
					else
					{
						data->buff_pos++;
						
						repaint();
					}					
				}
				break;
					
			case R_DELETE_LINE:
				data->buff[0]=0;
				data->buff_len=0;
				data->buff_pos=0;
				
				repaint();
				
				/* kill one word left */
			case R_BACKWARD_KILL_WORD:
				move_word(0,1);
				break;
					
				/* kill one word right */
			case R_KILL_WORD:
				move_word(1,1);
				break;
					
				/* move one word left*/
			case R_BACKWARD_WORD:
				move_word(0,0);
				break;
					
				/* move one word right*/
			case R_FORWARD_WORD:
				move_word( 1,0);
				break;
					
			case R_CLEAR_SCREEN:
			{								
				writembs( clear_screen );
				
				repaint();
				break;
			}

			case R_BEGINNING_OF_HISTORY:
			{
				history_first();
				break;
			}
			
			case R_END_OF_HISTORY:
			{
				history_reset();
				
				break;
			}

			/* Other, if a normal character, we add it to the command */
			default:
			{
				if( (c< WCHAR_END) && (c>31) && (c != 127) )
				{
					history_reset();
					insert_char( c );
				}
				break;
			}
			
		}
		last_char = c;
	}
	
	al_destroy( &comp );
    if( tcsetattr(0,TCSANOW,&old_modes))      /* return to previous mode */
    {
        wperror(L"tcsetattr");
        exit(1);
    }
	
	set_color( FISH_COLOR_RESET, FISH_COLOR_RESET );
	
	return data->buff;
}

/**
   Read non-interactively.  Read input from stdin without displaying
   the prompt, using syntax highlighting. This is used for reading
   scripts and init files.
*/
static int read_ni()
{
	FILE *in_stream;
	wchar_t *buff=0;
	buffer_t acc;
	
	int des = dup( 0 );
	int res=0;
	
	if (des == -1)
	{
		wperror( L"dup" );
		return 1;
	}

	b_init( &acc );
	
	in_stream = fdopen( des, "r" );
	if( in_stream != 0 )
	{
		wchar_t *str;
		
		while(!feof( in_stream ))
		{
			char buff[4096];
			int c = fread(buff, 1, 4096, in_stream);
			
			b_append( &acc, buff, c );

		}
		b_append( &acc, "\0", 1 );
		str = str2wcs( acc.buff );
		b_destroy( &acc );
		
//		fwprintf( stderr, L"Woot is %d chars\n", wcslen( acc.buff ) );
		
		if( !parser_test( str, 1 ) )
		{
			//fwprintf( stderr, L"We parse it\n" );
			eval( str, 0, TOP );
		}
		else
		{
			res = 1;
		}

		if(	fclose( in_stream ))
		{
			wperror( L"fclose" );
			res = 1;			
		}
		
		free( str );
		
		return 0;
	}
	else
	{
		wperror( L"fdopen" );
		free( buff );
		res=1;
		
	}		
	error_reset();	
	return res;	
}


int reader_read()
{
	int res;	
	/* 
	   If reader_read is called recursively through the '.' builtin,
	   we need to preserve shell_is_interactive, so we save the
	   original state. We also update the signal handlers.
	*/
	int shell_was_interactive = shell_is_interactive;
	shell_is_interactive = isatty(STDIN_FILENO);
	set_signal_handlers();
	
	if( shell_is_interactive )
		res = read_i();
	else 
		res = read_ni();
	
	/* 
	   If the exit command was called in a script, only exit the
	   script, not the program
	*/
	end_loop = 0;	

	shell_is_interactive = shell_was_interactive;
	set_signal_handlers();
	return res;
}
