/** \file reader.h 

    Prototypes for functions for reading data from stdin and passing
	to the parser. If stdin is a keyboard, it supplies a killring,
	history, syntax highlighting, tab-completion and various other
	features.
*/

/**
  Read commands from fd 0 until encountering EOF
*/
int reader_read();

/**
  Tell the shell that it should exit after the currently running command finishes.
*/
void do_exit();

/**
   Check that the reader is in a sane state
*/
void reader_sanity_check();

/**
   Initialize the reader
*/
void reader_init();

/**
   Destroy and free resources used by the reader
*/
void reader_destroy();

/**
   Returns the filename of the file currently read
*/
wchar_t *reader_current_filename();

/**
   Push a new filename on the stack of read files
   
   \param fn The fileanme to push
*/
void reader_push_current_filename( wchar_t *fn );
/**
   Pop the current filename from the stack of read files
 */
wchar_t *reader_pop_current_filename();

/** 
	Returns the width of the terminal window, so that not all
	functions that use these values continually have to keep track of
	it.
*/
int reader_get_width();
/**
   Returns the height of the terminal window, so that not all
   functions that use these values continually have to keep track of
   it.
*/
int reader_get_height();

/**
   Write the title to the titlebar. This function is called just
   before a new application starts executing and just after it
   finishes.
*/
void reader_write_title();

/**
   Repaint the entire commandline. This means reset and clear the
   commandline, write the prompt, perform syntax highlighting, write
   the commandline and move the cursor.
*/
void repaint();

/**
   Run the specified command with the correct terminal modes, and while taking care to perform job notification, set the title, etc.
*/
void reader_run_command( wchar_t *buff );

/**
   Get the string of character currently entered into the command buffer
*/
wchar_t *reader_get_buffer();

/**
   Set the string of characters in the command buffer, as well as the cursor position.

   \param b the new buffer value
   \param p the cursor position. If \c p is less than zero, the cursor is placed on the last character.
*/
void reader_set_buffer( wchar_t *b, int p );

/**
   Get the current cursor position in the command line
*/
int reader_get_cursor_pos();

/**
   Return the value of the interupted flag, which is set by the sigint handler, and clear it if it was set.
*/
int reader_interupted();

/**
   Read one line of input. Before calling this function, reader_push() must have been called in order to set up a valid reader environment.
*/
wchar_t *reader_readline();

/**
   Push a new reader environment. 
*/
void reader_push( wchar_t *name );

/**
   Return to previous reader environment
*/
void reader_pop();

/**
   Specify function to use for finding possible tab completions. The function must take these arguments: 

   - The command to be completed as a null terminated array of wchar_t
   - An array_list_t in which completions will be inserted.
*/
void reader_set_complete_function( void (*f)( const wchar_t *, array_list_t * ) );

/**
   Specify function for syntax highlighting. The function must take these arguments:

   - The command to be highlighted as a null terminated array of wchar_t
   - The color code of each character as an array of ints
   - The cursor position
   - An array_list_t used for storing error messages
*/
void reader_set_highlight_function( void (*f)( wchar_t *, int *, int, array_list_t * ) );

/**
   Specify function for testing if the command buffer contains syntax errors that must be corrected before returning. 
*/
void reader_set_test_function( int (*f)( wchar_t * ) );

/**
   Specify string of shell commands to be run in order to generate the prompt
*/
void reader_set_prompt( wchar_t *prompt );

/**
   Returns true if the shell is exiting, 0 otherwise. 
*/
int exit_status();

