/** \file util.c
	Generic utilities library.

	Contains datastructures such as hash tables, automatically growing array lists, priority queues, etc.
*/

#include "config.h"

#include <stdio.h>
#include <stdlib.h>
#include <wchar.h>
#include <math.h>
#include <sys/time.h>
#include <stdarg.h>		
#include <string.h>		
#include <ctype.h>		
#include <wctype.h>		
#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <dirent.h>


#include "util.h"
#include "wutil.h"

/** 
	Minimum allocated size for data structures. Used to avoid excessive
	memory allocations for lists, hash tables, etc, which are nearly
	empty. 
*/
#define MIN_SIZE 16

float minf( float a,
			float b )
{
	return a<b?a:b;
}


float maxf( float a,
			float b )
{
	return a>b?a:b;
}

int mini( int a,
		  int b )
{
	return a<b?a:b;
}


int maxi( int a, 
		  int b )
{
	return a>b?a:b;
}


/* Queue functions */


void q_init( queue_t *q )
{
	q->start = (void **)malloc( sizeof(void*)*1 );
	q->stop = &q->start[1];
	q->put_pos = q->get_pos = q->start;
}

void q_destroy( queue_t *q )
{
	free( q->start );
}

/*
  static q_print( queue_t *q )
  {
  int i;
  int size = (q->stop-q->start);

  printf( "Storlek: %d\n", size );
  for( i=0; i< size; i++ )
  {
  printf( " %c%c %d: %d\n",
  &q->start[i]==q->get_pos?'g':' ', 
  &q->start[i]==q->put_pos?'p':' ', 
  i,
  q->start[i] );
  }
  }

*/

/**
   Reallocate the queue_t
*/
static int q_realloc( queue_t *q )
{
	void **old_start = q->start;
	void **old_stop = q->stop;
	int diff;
	int new_size;

	new_size = 2*(q->stop-q->start);
	
	q->start=(void**)realloc( q->start, sizeof(void*)*new_size );
	if( q->start == 0 )
	{
		q->start = old_start;
		return 0;
	}
	
	diff = q->start - old_start;
	q->get_pos += diff;
	q->put_pos += diff;
	q->stop = &q->start[new_size];
	memcpy( old_stop + diff, q->start, sizeof(void*)*(q->get_pos-q->start));

	return 1;
}

int q_put( queue_t *q, void *e )
{
	*q->put_pos = e;
	
	if( ++q->put_pos == q->stop )
		q->put_pos = q->start;
	if( q->put_pos == q->get_pos )
		return q_realloc( q );
	return 1;
}

void *q_get( queue_t *q)
{
	void *e = *q->get_pos;
	if( ++q->get_pos == q->stop )
		q->get_pos = q->start;
	return e;
}

void *q_peek( queue_t *q )
{
	return *q->get_pos;
}

int q_empty( queue_t *q )
{
	return q->put_pos == q->get_pos;
}

/* Stack functions */




/* Hash table functions */

void hash_init( hash_table_t *h, 
				int (*hash_func)(const void *key),
				int (*compare_func)(const void *key1, const void *key2) )
{
	int i;
	
	h->arr = malloc( sizeof(hash_struct_t)*31 );
	h->size = 31;
	for( i=0; i< 31; i++ )
		h->arr[i].key = 0;
	h->count=0;
	h->hash_func = hash_func;
	h->compare_func = compare_func;
}

void hash_destroy( hash_table_t *h )
{
	free( h->arr );
}

/**
   Search for the specified hash key in the table
   \return index in the table, or to the first free index if the key is not in the table
*/
static int hash_search( hash_table_t *h,
						const void *key )
{
	int hv = h->hash_func( key );
	int pos = abs(hv) % h->size;
	while(1)
	{
		if( (h->arr[pos].key == 0 ) || 
			( h->compare_func( h->arr[pos].key, key ) ) )
		{
			return pos;
		}
		pos++;
		pos %= h->size;
	}
}

/**
   Reallocate the hash array. This is quite expensive, as every single entry has to be rehashed and moved.
*/
static int hash_realloc( hash_table_t *h, 
						 int sz )
{

	/* Avoid reallocating when using pathetically small tables */
	if( ( sz < h->size ) && (h->size < MIN_SIZE))
		return 1;
	sz = maxi( sz, MIN_SIZE );

	hash_struct_t *old_arr = h->arr;
	int old_size = h->size;
	
	int i;

	h->arr = malloc( sizeof( hash_struct_t) * sz );
	if( h->arr == 0 )
	{
		h->arr = old_arr;
		return 0;
	}
	
	memset( h->arr, 
			0,
			sizeof( hash_struct_t) * sz );
	h->size = sz;
	
	for( i=0; i<old_size; i++ )
	{
		if( old_arr[i].key != 0 )
		{
			int pos = hash_search( h, old_arr[i].key );
			h->arr[pos].key = old_arr[i].key;
			h->arr[pos].data = old_arr[i].data;
		}
	}
	free( old_arr );

	return 1;
}


int hash_put( hash_table_t *h, 
			  const void *key,
			  const void *data )
{
	int pos;
	
	if( (float)(h->count+1)/h->size > 0.75f )
	{
		if( !hash_realloc( h, (h->size+1) * 2 -1 ) )
		{
			return 0;
		}
	}

	pos = hash_search( h, key );

	if( h->arr[pos].key == 0 )
	{		
		h->count++;
	}
	
	h->arr[pos].key = key;
	h->arr[pos].data = data;
	return 1;
}

const void *hash_get( hash_table_t *h, 
					  const void *key )
{
	int pos = hash_search( h, key );
	if( h->arr[pos].key == 0 )
		return 0;
	else
		return h->arr[pos].data;
}

const void *hash_get_key( hash_table_t *h, 
						  const void *key )
{
	int pos = hash_search( h, key );
	if( h->arr[pos].key == 0 )
		return 0;
	else
		return h->arr[pos].key;
}

int hash_get_count( hash_table_t *h)
{
	return h->count;
}

void hash_remove( hash_table_t *h, 
				  const void *key,
				  const void **old_key, 
				  const void **old_val )
{
	int pos = hash_search( h, key );
	int next_pos;
	
	if( h->arr[pos].key == 0 )
	{

		if( old_key != 0 )
			*old_key = 0;
		if( old_val != 0 )
			*old_val = 0;
		return;
	}

	h->count--;
	
	if( old_key != 0 )
		*old_key = h->arr[pos].key;
	if( old_val != 0 )
		*old_val = h->arr[pos].data;

	h->arr[pos].key = 0;
	
	next_pos = pos+1;
	next_pos %= h->size;
	
	while( h->arr[next_pos].key != 0 )
	{
		
		int hv = h->hash_func( h->arr[next_pos].key );
		int ideal_pos = abs( hv ) % h->size;
		int dist_old = (next_pos - ideal_pos + h->size)%h->size;
		int dist_new = (pos - ideal_pos + h->size)%h->size;
		if ( dist_new < dist_old )
		{
			h->arr[pos].key = h->arr[next_pos].key;
			h->arr[pos].data = h->arr[next_pos].data;
			h->arr[next_pos].key = 0;
			pos = next_pos;
		}
		next_pos++;
		
		next_pos %= h->size;
		
	}

	if( (float)(h->count+1)/h->size < 0.3f )
	{
		hash_realloc( h, (h->size+1) / 2 -1 );
	}

	return;
}

int hash_contains( hash_table_t *h, 
				   const void *key )
{
	int pos = hash_search( h, key );
	return h->arr[pos].key != 0;
}

/**
   Push hash value into array_list_t
*/
static void hash_put_data( const void *key, 
						   const void *data,
						   void *al )
{
	al_push( (array_list_t *)al,
			 data );
}


void hash_get_data( hash_table_t *h, 
					array_list_t *arr )
{
	hash_foreach2( h, &hash_put_data, arr );
}

/**
   Push hash key into array_list_t
*/
static void hash_put_key( const void *key, const void *data, void *al )
{
	al_push( (array_list_t *)al, key );
}


void hash_get_keys( hash_table_t *h, 
					array_list_t *arr )
{
	hash_foreach2( h, &hash_put_key, arr );
}

void hash_foreach( hash_table_t *h, 
				   void (*func)(const void *, const void *) )
{
	int i;
	for( i=0; i<h->size; i++ )
	{
		if( h->arr[i].key != 0 )
		{
			func( h->arr[i].key, h->arr[i].data );
		}
	}	
}

void hash_foreach2( hash_table_t *h, 
					void (*func)( const void *, const void *, void * ), 
					void *aux )
{
	int i;
	for( i=0; i<h->size; i++ )
	{
		if( h->arr[i].key != 0 )
		{
			func( h->arr[i].key, h->arr[i].data, aux );
		}
	}	
}


int hash_str_cmp( const void *a, const void *b )
{
	return strcmp((char *)a,(char *)b) == 0;
}

/**
   Helper function for hash_wcs_func
*/
static uint rotl1( uint in )
{
	return (in<<1|in>>31);
}

/**
   Helper function for hash_wcs_func
*/
static uint rotl5( uint in )
{
	return (in<<5|in>>27);
}

/**
   Helper function for hash_wcs_func
*/
static uint rotl30( uint in )
{
	return (in<<30|in>>2);
}

#define WORD_COUNT 16

int hash_str_func( const void *data )
{
	const char *in = (const char *)data;
	uint a,b,c,d,e;
	int t;
	uint k0 = 0x5a827999u;	
	uint k1 = 0x6ed9eba1u;
	
	uint w[2*WORD_COUNT];	
	char *wc=(char *)w;
	
	/*
	  Same constants as used by sha1
	*/
	a=0x67452301u;
	b=0xefcdab89u;
	c=0x98badcfeu;
	d=0x10325476u;
	e=0xc3d2e1f0u;
	
	if( data == 0 )
		return 0;
	
	while( *in )
	{
		int i;

		/*
		  Read WORD_COUNT words of data into w
		*/
		for( i=0; i<(sizeof(uint)*WORD_COUNT); i++ )
		{
			if( *in)
			{
				wc[i]=*in++;
			}
			else
			{
				memset( &wc[i], 0, sizeof(char)*(sizeof(uint)*WORD_COUNT-i) );
				break;
			}
		}
		
		/*
		  And fill up the rest by rotating the previous content
		*/
		for( i=WORD_COUNT; i<(2*WORD_COUNT); i++ )
		{
			w[i]=rotl1( w[i-1]^w[i-WORD_COUNT] );
		}

		/*
		  Only 2*WORD_COUNT laps, not 80 like in sha1. Only two types
		  of laps, not 4 like in sha1
		*/
		for( t=0; t<WORD_COUNT; t++ )
		{
			uint temp;
			temp = (rotl5(a)+(b^c^d)+e+w[t]+k0);
			e=d;
			d=c;
			c=rotl30(b);
			b=a;
			a=temp;
		}
		for( t=WORD_COUNT; t<(2*WORD_COUNT); t++ )
		{
			uint temp;
			temp = (rotl5(a)+((b&c)|(b&d)|(c&d))+e+w[t]+k1);
			e=d;
			d=c;
			c=rotl30(b);
			b=a;
			a=temp;
		}
	}

	/*
	  Implode from 160 to 32 bit hash and return
	*/
	return a^b^c^d^e;
}

int hash_wcs_func( const void *data )
{
	const wchar_t *in = (const wchar_t *)data;
	uint a,b,c,d,e;
	int t;
	uint k0=0x5a827999u;	
	uint k1 =0x6ed9eba1u;
	
	uint w[2*WORD_COUNT];	
	
	/*
	  Same constants used by sha1
	*/
	a=0x67452301u;
	b=0xefcdab89u;
	c=0x98badcfeu;
	d=0x10325476u;
	e=0xc3d2e1f0u;
	
	if( data == 0 )
		return 0;
	
	while( *in )
	{
		int i;

		/*
		  Read WORD_COUNT words of data into w
		*/
		for( i=0; i<WORD_COUNT; i++ )
		{
			if( *in)
			{
				w[i]=*in++;
			}
			else
			{
				memset( &w[i], 0, sizeof(uint)*(WORD_COUNT-i) );
				break;
			}
		}
		
		/*
		  And fill up the rest by rotating the previous content
		*/
		for( i=WORD_COUNT; i<(2*WORD_COUNT); i++ )
		{
			w[i]=rotl1( w[i-1]^w[i-WORD_COUNT] );
		}

		/*
		  Only 2*WORD_COUNT laps, not 80 like in sha1. Only two types
		  of laps, not 4 like in sha1
		*/
		for( t=0; t<WORD_COUNT; t++ )
		{
			uint temp;
			temp = (rotl5(a)+(b^c^d)+e+w[t]+k0);
			e=d;
			d=c;
			c=rotl30(b);
			b=a;
			a=temp;
		}
		for( t=WORD_COUNT; t<(2*WORD_COUNT); t++ )
		{
			uint temp;
			temp = (rotl5(a)+((b&c)|(b&d)|(c&d))+e+w[t]+k1);
			e=d;
			d=c;
			c=rotl30(b);
			b=a;
			a=temp;
		}
	}

	/*
	  Implode from 160 to 32 bit hash and return
	*/
	return a^b^c^d^e;
}

int hash_wcs_cmp( const void *a, const void *b )
{
	return wcscmp((wchar_t *)a,(wchar_t *)b) == 0;
}

void pq_init( priority_queue_t *q,
			  int (*compare)(void *e1, void *e2) )
{
	q->arr=0;
	q->size=0;
	q->count=0;
	q->compare = compare;
}

/**
   Check that the priority queue is in a valid state
*/
/*
  static void pq_check( priority_queue_t *q, int i )
  {
  int l,r;
  if( q->count <= i )
  return;
	
  l=i*2+1;
  r=i*2+2;
	
	
  if( (q->count > l) && (q->compare(q->arr[i], q->arr[l]) < 0) )
  {
  printf( "ERROR: Place %d less than %d\n", i, l );
  }
  if( (q->count > r) && (q->compare(q->arr[i], q->arr[r]) < 0) )
  {
  printf( "ERROR: Place %d less than %d\n", i, r );
  }
  pq_check( q, l );
  pq_check( q, r );
  }
*/

int pq_put( priority_queue_t *q,
			void *e )
{
	int i;
	
	if( q->size == q->count )
	{
		void **old_arr = q->arr;
		int old_size = q->size;
		q->size = maxi( 4, 2*q->size );
		q->arr = (void **)realloc( q->arr, sizeof(void*)*q->size );
		if( q->arr == 0 )
		{
			q->arr = old_arr;
			q->size = old_size;
			return 0;
		}
	}
	
	i = q->count;
	while( (i>0) && (q->compare( q->arr[(i-1)/2], e )<0 ) )
	{
		q->arr[i] = q->arr[(i-1)/2];
		i = (i-1)/2;
	}
	q->arr[i]=e;

	q->count++;

	return 1;
	
}

/**
   Make a valid head
*/
static void pq_heapify( priority_queue_t *q, int i )
{
	int l, r, largest;
	l = 2*(i)+1;
	r = 2*(i)+2;
	if( (l < q->count) && (q->compare(q->arr[l],q->arr[i])>0) )
	{
		largest = l;
	}
	else
	{
		largest = i;
	}
	if( (r < q->count) && (q->compare( q->arr[r],q->arr[largest])>0) )
	{
		largest = r;
	}

	if( largest != i )
	{
		void *tmp = q->arr[largest];
		q->arr[largest]=q->arr[i];
		q->arr[i]=tmp;
		pq_heapify( q, largest );
	}
}

void *pq_get( priority_queue_t *q )
{
	void *result = q->arr[0];
	q->arr[0] = q->arr[--q->count];
	pq_heapify( q, 0 );

/*	pq_check(q, 0 );	*/
/*	pq_print( q ); */

	return result;
}

void *pq_peek( priority_queue_t *q )
{
	return q->arr[0];
}

int pq_empty( priority_queue_t *q )
{
	return q->count == 0;
}

int pq_get_count( priority_queue_t *q )
{
	return q->count;
}

void pq_destroy(  priority_queue_t *q )
{
	free( q->arr );
}


void al_init( array_list_t *l )
{
	memset( l, 0, sizeof( array_list_t ) );
}

void al_destroy( array_list_t *l )
{
	free( l->arr );
}

int al_push( array_list_t *l, const void *o )
{
	if( l->pos >= l->size )
	{
		int new_size = l->pos == 0 ? MIN_SIZE : 2 * l->pos;
		void *tmp = realloc( l->arr, sizeof( void *)*new_size );
		if( tmp == 0 )
			return 0;
		l->arr = tmp;
	}
	l->arr[l->pos++] = o;
	return 1;
}

int al_push_all( array_list_t *a, array_list_t *b )
{
	int k;
	for( k=0; k<al_get_count( b ); k++ )
	{
		if( !al_push( a, al_get( b, k ) ) )
			return 0;
	}
	return 1;
}


int al_set( array_list_t *l, int pos, const void *o )
{
	if( pos < 0 )
		return 0;
	if( pos < l->pos )
	{
		l->arr[pos] = o;
		return 1;
	}
	l->pos = pos;
	return al_push( l, o );
}

const void *al_get( array_list_t *l, int pos )
{
	if( pos < 0 )
		return 0;
	if( pos >= l->pos )
		return 0;
	return l->arr[pos];
}

void al_truncate( array_list_t *l, int new_sz )
{
	l->pos = new_sz;
}

const void *al_pop( array_list_t *l )
{
	const void *e = l->arr[--l->pos];
	if( (l->pos*3 < l->size) && (l->size < MIN_SIZE) )
	{
		const void ** old_arr = l->arr;
		int old_size = l->size;
		l->size = l->size/2;
		l->arr = realloc( l->arr, sizeof(void*)*l->size );
		if( l->arr == 0 )
		{
			l->arr = old_arr;
			l->size = old_size;
		}
	}
	return e;
}

const void *al_peek( array_list_t *l )
{
	
	return l->pos>0?l->arr[l->pos-1]:0;
}

int al_empty( array_list_t *l )
{
	return l->pos == 0;
}

int al_get_count( array_list_t *l )

{
	return l->pos;
}

void al_foreach( array_list_t *l, void (*func)( const void * ))
{
	int i;
	for( i=0; i<l->pos; i++ )
		func( l->arr[i] );
}

void al_foreach2( array_list_t *l, void (*func)( const void *, void *), void *aux)
{
	int i;
	for( i=0; i<l->pos; i++ )
		func( l->arr[i], aux );
}

int wcsfilecmp( const wchar_t *a, const wchar_t *b )
{
	if( *a==0 )
	{
		if( *b==0)
			return 0;
		return -1;
	}
	if( *b==0 )
	{
		return 1;
	}
	int secondary_diff=0;
	if( iswdigit( *a ) && iswdigit( *b ) )
	{
		wchar_t *aend, *bend;
		long al = wcstol( a, &aend, 10 );
		long bl = wcstol( b, &bend, 10 );
		int diff = al - bl;
		if( diff )
			return diff>0?2:-2;			

		secondary_diff = (aend-a) - (bend-b);
		
		a=aend-1;
		b=bend-1;
	}
	else
	{
		int diff = towlower(*a) - towlower(*b);
		if( diff != 0 )
			return (diff>0)?2:-2;			
		
		secondary_diff = *a-*b;
	}

	int res = wcsfilecmp( a+1, b+1 );
	switch( abs(res) )
	{
		case 2:
			return res;
		default:
			if( secondary_diff )
				return secondary_diff>0?1:-1;			
	}
	return 0;

}

void sb_init( string_buffer_t * b)
{
	memset( b, 0, sizeof(string_buffer_t) );
}

void sb_append( string_buffer_t *b, const wchar_t * s)
{
//	fwprintf( stderr, L"Append string \'%ls\'\n", s );
	
	b_append( b, s, sizeof(wchar_t)*(wcslen(s)+1) );
	b->used -= sizeof(wchar_t);	
}

void sb_append_substring( string_buffer_t *b, const wchar_t *s, size_t l )
{
    wchar_t tmp=0;
    b_append( b, s, sizeof(wchar_t)*l );
    b_append( b, &tmp, sizeof(wchar_t) );
    b->used -= sizeof(wchar_t);	
}


void sb_append_char( string_buffer_t *b, wchar_t c )
{
	wchar_t buff[2]=
		{
			c, 0 
		}
	;
	sb_append( b, buff );
	
}

void sb_append_int( string_buffer_t *b, int d )
{
	wchar_t buff[32];
	swprintf( buff, 32, L"%d", d );
	sb_append( b, buff );	
}

void sb_append2( string_buffer_t *b, ... )
{
	va_list va;
	wchar_t *arg;

	va_start( va, b );
	while( (arg=va_arg(va, wchar_t *) )!= 0 ) 
	{
		sb_append( b, arg );
	}
	va_end( va );
}


void sb_destroy( string_buffer_t * b )
{
	free( b->buff );
}

void sb_clear( string_buffer_t * b )
{
	free( b->buff );
	sb_init( b );
}


void b_init( buffer_t *b)
{
	memset( b,0,sizeof(buffer_t));
}



void b_destroy( buffer_t *b )
{
	free( b->buff );
}


void b_append( buffer_t *b, const void *d, ssize_t len )
{
	if( len<=0 )
		return;
	
	if( !b )
	{
		fwprintf( stderr, L"Copy to null buffer\n" );
		return;
	}
		
	if( !d )
	{
		fwprintf( stderr, L"Copy from null pointer\n" );
		return;
	}

	if( len < 0 )
	{
		fwprintf( stderr, L"Negative number of characters to be copied\n" );
		return;		
	}
	

	if( b->length <= (b->used + len) )
	{
		size_t l = maxi( b->length*2, 
						 maxi( b->used+len+128,256));

		void *d = realloc( b->buff, l );
		if( !d )
		{
			fwprintf( stderr, L"fish: Out of memory\n" );
			exit(1);
		}
		b->buff=d;
		b->length = l;
	}
	memcpy( ((char*)b->buff)+b->used, 
			d,
			len );

//	fwprintf( stderr, L"Copy %s, new value %s\n", d, b->buff );
	b->used+=len;
}
