/** \file wildcard.c
	My own globbing implementation. Needed to implement this to
	support tab-expansion of globbed parameters.

*/

#include "config.h"

#include <stdlib.h>
#include <stdio.h>
#include <limits.h>
#include <wchar.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <dirent.h>
#include <errno.h>

#include "util.h"
#include "wutil.h"
#include "complete.h"
#include "common.h"
#include "wildcard.h"
#include "complete.h"
#include "reader.h"

int wildcard_has( const wchar_t *str, int internal )
{
	wchar_t prev=0;
	if( internal )
	{
		for( ; *str; str++ )
		{
			if( ( (*str == ANY_CHAR ) || (*str == ANY_STRING) ) && (prev != L'\\') )
				return 1;
			prev = *str;
		}
	}
	else
	{
//		fwprintf( stderr, L"Test %ls\n", str );
		for( ; *str; str++ )
		{
			if( ( (*str == L'*' ) || (*str == L'?' ) ) && (prev != L'\\') )
				return 1;
			prev = *str;
		}
//		fwprintf( stderr, L"No wildcards\n" );
	}
	
	return 0;
}

/**
   Check whether the string str matches the wildcard string wc.
  
   \param str String to be matched.
   \param wc The wildcard.
   \param is_first Whether files beginning with dots should not be matched against wildcards.
   \param wc_unescaped Whether the unescaped special character ANY_CHAR abd ANY_STRING should be used instead of '?' and '*' for wildcard matching
*/


static int wildcard_match2( const wchar_t *str, 
							const wchar_t *wc, 
							int is_first,
							int wc_unescaped)
{
	
	if( *str == 0 && *wc==0 )
		return 1;
	
	if( *wc == (wc_unescaped?ANY_STRING:L'*') )
	{		
		/* Ignore hidden file */
		if( is_first && *str == L'.' )
			return 0;
		
		/* Try all submatches */
		do
		{
			if( wildcard_match2( str, wc+1, 0, wc_unescaped ) )
				return 1;
		}
		while( *(str++) != 0 );
		return 0;
	}
	if( *wc == (wc_unescaped?ANY_CHAR:L'?') )
		return wildcard_match2( str+1, wc+1, 0, wc_unescaped );
	
	if( *wc == *str )
		return wildcard_match2( str+1, wc+1, 0, wc_unescaped );

	return 0;
}

/**
   Matches the string against the wildcard, and if the wildcard is a
   possible completion of the string, the remainder of the string is
   inserted into the array_list_t.
*/
static void wildcard_complete_internal( const wchar_t *orig, 
										const wchar_t *str, 
										const wchar_t *wc, 
										int is_first,
										const wchar_t *desc,
										const wchar_t *(*desc_func)(const wchar_t *),
										array_list_t *out )
{
	if( *wc == 0 && 
		( ( *str != L'.') || (!is_first)) )
	{
//		fwprintf( stderr, L"Match!! %ls \n", str );

		wchar_t *new;
			
		if( wcschr( str, PROG_COMPLETE_SEP ) )
		{
			/*
			  This completion has an embedded description, du not use the generic description
			*/
			wchar_t *sep;
				
			new = wcsdup( str );
			sep = wcschr(new, PROG_COMPLETE_SEP );
			*sep = COMPLETE_SEP;
		}
		else if( desc_func )
		{
			new = wcsdupcat2( str, COMPLETE_SEP_STR, desc_func( orig ), 0);			
		}
		else
		{
			
			/*
			  Append generic description to item
			*/
			new = wcsdupcat( str, desc );
		}

		if( new )
			al_push( out, new );

// 		al_push( out, wcsdupcat( str, desc ) );
		return;
	}
	
	
	if( *wc == ANY_STRING )
	{		
		/* Ignore hidden file */
		if( is_first && str[0] == L'.' )
			return;
		
		/* Try all submatches */
		do
		{
			wildcard_complete_internal( orig, str, wc+1, 0, desc, desc_func, out );
		}
		while( *str++ != 0 );
		return;
	}
	
	if( *wc == ANY_CHAR )
	{
		wildcard_complete_internal( orig, str+1, wc+1, 0, desc, desc_func, out );
		return;	
	}	
	
	if( *wc == *str )
		wildcard_complete_internal( orig, str+1, wc+1, 0, desc, desc_func, out );
	
}

void wildcard_complete( const wchar_t *str,
						const wchar_t *wc,
						const wchar_t *desc,						
						const wchar_t *(*desc_func)(const wchar_t *),
						array_list_t *out )
{
	wildcard_complete_internal( str, str, wc, 1, desc, desc_func, out );	
}


int wildcard_match( const wchar_t *str, const wchar_t *wc, int wc_unescaped )
{
	return wildcard_match2( str, wc, 1, wc_unescaped );	
}

/**
   Creates a path from the specified directory and filename. 
*/
static wchar_t *make_path( const wchar_t *base_dir, const wchar_t *name )
{
	
	wchar_t *long_name;
	int base_len = wcslen( base_dir );
	if( !(long_name= malloc( sizeof(wchar_t)*(base_len+wcslen(name)+1) )))
	{
		
		return 0;
	}					
	wcscpy( long_name, base_dir );
	wcscpy(&long_name[base_len], name );
	return long_name;
}


int wildcard_expand( const wchar_t *wc, 
					 const wchar_t *base_dir,
					 int accept_incomplete,
					 array_list_t *out )
{
//	fwprintf( stderr, L"WILDCARD_EXPAND %ls in %ls\n", wc, base_dir );

	if( accept_incomplete )
	{	
		/* Avoid excessive number of returned matches for wc ending with a * */
		int len = wcslen(wc);
		if( len && (wc[len-1]==ANY_STRING) )
		{
			wchar_t * foo = wcsdup( wc );
			foo[len-1]=0;
			int res = wildcard_expand( foo, base_dir, accept_incomplete, out );
			free( foo );
			return res;			
		}
	}
	
	struct dirent *next;
	wchar_t *wc_end = wcschr(wc,L'/');
	DIR *dir;
	
	int res = 0;
	int base_len = wcslen( base_dir );

	const wchar_t *dir_string = base_dir[0]==L'\0'?L".":base_dir;
//	if( accept_incomplete )
//		wprintf( L"Glob %ls in '%ls'\n", wc, base_dir );//[0]==L'\0'?L".":base_dir );
	
	if( !(dir = wopendir( dir_string )))
	{
//		if( errno != EACCES && errno != ENOENT )
//			wperror( L"opendir" );
		return 0;
	}
	
	/*
	  Is this segment of the wildcard the last?
	*/
	if( wc_end == 0 )
	{
		/*
		  Wildcard segment is the last segment

		  Insert all matching files/directories
		*/
		if( wc[0]=='\0' )
		{
			if( accept_incomplete )
			{
				while( (next=readdir(dir))!=0 )
				{
					if( next->d_name[0] != '.' )
					{
						wchar_t *name = str2wcs(next->d_name);
						if( name == 0 )
						{
/*							fwprintf( stderr, L"%s\n", next->d_name );*/
/*							closedir( dir );*/
/*							return -1;							*/
							continue;
						}
						wchar_t *long_name = make_path( base_dir, name );
						if( long_name == 0 )
						{
							wperror( L"malloc" );
							closedir( dir );
							free(name);
							return 0;						
						}
						wchar_t *desc = complete_get_desc( long_name );
						wchar_t *tot = wcsdupcat( name, desc );
						al_push( out, tot );
						free(name);

						free( long_name );
					}					
				}
			}
			else
			{								
				res = 1;
				al_push( out, wcsdup( base_dir ) );
			}							
		}
		else
		{
			while( (next=readdir(dir))!=0 )
			{
				wchar_t *name = str2wcs(next->d_name);
				if( name == 0 )
				{
/*					fwprintf( stderr, L"Sequence: %s\n", next->d_name );*/
/*					closedir( dir );*/
/*					return -1;*/
					continue;
				}
				
/*				wprintf( L"Filen heter %s\n\n\n", next->d_name );*/
/*				wprintf( L"Match %ls (%s) against %ls\n\n\n", name, "tjo", wc );*/
				if( accept_incomplete )
				{
/*					wprintf( L"match %ls to %ls\n", name, wc );*/
					
					wchar_t *long_name = make_path( base_dir, name );
					if( long_name == 0 )
					{
						wperror( L"malloc" );
						closedir( dir );
						free(name);
						return 0;						
					}
//					fwprintf( stderr, L"Test %ls\n", name );
					wildcard_complete( name,
									   wc,
									   complete_get_desc( long_name ),
									   0,
									   out );			
					free( long_name );
					
				}
				else
				{
					if( wildcard_match2( name, wc, 1, 1 ) )
					{
						wchar_t *long_name = make_path( base_dir, name );
						if( long_name == 0 )
						{
							wperror( L"malloc" );
							closedir( dir );
							free(name);
							return 0;						
						}
						
						al_push( out, long_name );
						res = 1;
					}
				}
				free( name );
			}
		}
	}
	else
	{
		/*
		  Wilcard segment is not the last segment.
		  Recursively call wildcard_expand for all matching subdirectories.
		*/
		wchar_t *wc_str;
		wchar_t *new_dir;
 		static size_t ln=1024;
        char * narrow_dir_string = wcs2str( dir_string );
		
		if( narrow_dir_string )
		{
			ln = pathconf( narrow_dir_string, _PC_NAME_MAX ); /* Find out how long the filename can be in a worst case scenario */
			if( ln < 0 )
				ln = 1024;		
			free( narrow_dir_string );
		}
		new_dir= malloc( sizeof(wchar_t)*(base_len+ln+2)  );

		wc_str = wcsndup(wc, wc_end-wc);
		if( (!new_dir) || (!wc_str) )
		{
			if( new_dir ) 
				free( new_dir );
			if( wc_str ) 
				free( wc_str );
			wperror( L"malloc" );			
			closedir( dir );
			return 0;			
		}
		wcscpy( new_dir, base_dir );
		
		while( (next=readdir(dir))!=0 )
		{
			wchar_t *name = str2wcs(next->d_name);
			if( name == 0 )
			{
				continue;
			}			
			if( wildcard_match2( name, wc_str, 1, 1 ) )
			{
				int new_len;
				struct stat buf;			
				wcscpy(&new_dir[base_len], name );
				free(name);
				char *dir_str = wcs2str( new_dir );
				int stat_res;
				
				if( !dir_str )
				{
					continue;					
				}
				
				stat_res= stat( dir_str, &buf );
				free( dir_str );
				
				if( stat_res )
				{
					continue;
				}

				if( buf.st_mode & S_IFDIR )
				{
					new_len = wcslen( new_dir );
					new_dir[new_len] = L'/';
					new_dir[new_len+1] = L'\0';
					//fwprintf( stderr, L"Call wildcard_expand for subdir %ls\n", new_dir );
					switch( wildcard_expand( wc_end + 1, new_dir, accept_incomplete, out ) )
					{
						case 0:
							break;
						case 1:
							res = 1;
							break;
					}
				}								
			}
			else
			{
				free(name);
			}			
		}
		free( wc_str );
		free( new_dir );
	}
	closedir( dir );
	return res;
}

