/*
 *										
 *		File		: callbacks.c					
 *										
 *		Title		: Flink email monitoring applet for GNOME 2	
 *										
 *		Description	: Callbacks sourcefile (originally by glade)	
 *										
 *		Authors		: Marcus Leyman, 2002 <marcus@leyman.nu>	
 *										
 *		License		: GPL						
 *										
 *		Date		: 27 May 2002					
 *										
 */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <gnome.h>
#include <gconf/gconf-client.h>

#include "callbacks.h"
#include "interface.h"
#include "support.h"
#include "configuration.h"

/* Globals */
Account			*new_account;
FlinkConfig		*new_config;
gint			current_id;

/********************************************************
 * Function:	flink_timeout_reached			*
 *							*
 * Description:	Runs mailcheck functions with supplied  *
 *		data					*
 *							*
 * Arguments:	gpointer data, name of account		*
 *							*
 * Returns:	TRUE					*
 ********************************************************/
gboolean
flink_timeout_reached	(gpointer data)
{
  flink_mail_check_account(NULL,NULL, data);
  return TRUE;
}

/********************************************************
 * Function:	flink_timeout_changed			*
 *							*
 * Description:	When poll interval is changed		*
 *							*
 * Arguments:	GConfClient *client,			*
 *		guint *cnxn_id, notifyer connection id	*
 *		GConfEntry *entry, [name]_interval	*
 *		gpointer user_data, timeout id		*
 *							*
 * Returns:	nothing					*
 ********************************************************/
void
flink_timeout_changed 	(GConfClient *client, 
			guint cnxn_id,
			GConfEntry *entry,
			gpointer user_data)
{
  GConfClient 	*new_client;
  guint			ref;
  gchar			**name_a;
  gchar			**name_b;
  gint			timeout;
	
  gconf_client_notify_remove(client, cnxn_id);
	
  gtk_timeout_remove((guint) user_data);
	
  name_a = g_strsplit((gchar *) entry->key, "/apps/flink/accounts/", 0);
  name_b = g_strsplit(name_a[1], "_interval", 0);
	
  gconf_client_set_string(client, "/apps/flink/debug", name_b[0], NULL);
	
  timeout = gconf_client_get_int(client, entry->key, NULL);

  if (timeout != 0 ) {
    ref = gtk_timeout_add(((guint) timeout * 60000),
			  flink_timeout_reached,
			  (gpointer) name_b[0]);
  }
	
  gconf_client_notify_add(client,
			  entry->key,
			  flink_timeout_changed,
			  (gpointer) ref,
			  NULL,
			  NULL);
}

/********************************************************
 * Function:	flink_prefs_tree_selection_changed_cb	*
 *							*
 * Description:	When new item is selected in acc-list	*
 *							*
 * Arguments:	GtkTreeSelection *selection,		*
 *		gpointer data, 				*
 *							*
 * Returns:	nothing					*
 ********************************************************/
void
flink_prefs_tree_selection_changed_cb 	(GtkTreeSelection *selection, 
					 gpointer data) 
{
  GtkTreeIter iter;
  GtkTreeModel *model;

  if (gtk_tree_selection_get_selected (selection, &model, &iter))
    gtk_tree_model_get (model, &iter, 1, &current_id, -1);
}

/********************************************************
 * Function:	flink_gerror_handler_prefs		*
 *							*
 * Description:	handles errors				*
 *							*
 * Arguments:	GtkWindow *window, parent window	*
 *		GError *err, pointer to error		*
 *							*
 * Returns:	nothing					*
 ********************************************************/
void
flink_gerror_handler_prefs	(GtkWindow 	*window, 
				 GError 	*err) 
{
  GtkWidget *dialog;

  dialog = gtk_message_dialog_new (GTK_WINDOW (window),
				   GTK_DIALOG_DESTROY_WITH_PARENT,
				   GTK_MESSAGE_ERROR,
				   GTK_BUTTONS_CLOSE,
				   _("There was an configuration error: %s"),
				   err->message);

  g_signal_connect (G_OBJECT (dialog), "response",
		    G_CALLBACK (gtk_widget_destroy),
		    NULL);

  gtk_window_set_resizable (GTK_WINDOW (dialog), FALSE);

  gtk_widget_show (dialog);

  g_error_free (err);
}

/********************************************************
 * Function:	on_prefs_acc_new_button_clicked		*
 *							*
 * Description:	Creates new account guide		*
 *							*
 * Arguments:	GtkButton *button,			*
 *		gpointer user_data,			*
 *							*
 * Returns:	nothing					*
 ********************************************************/
void
on_prefs_acc_new_button_clicked        (GtkButton       *button,
                                        gpointer         user_data)
{
  GtkWidget       *flink_new_dialog;
        
  new_account = flink_config_create_template_account();
        
  flink_new_dialog = create_configure_account_dialog();
        
  gtk_window_set_title(GTK_WINDOW(lookup_widget(GTK_WIDGET(flink_new_dialog), "configure_account_dialog")),
		       "Create New Account");
        
  gtk_widget_show(GTK_WIDGET(lookup_widget(GTK_WIDGET(flink_new_dialog), "type_hseparator")));
  gtk_widget_show(GTK_WIDGET(lookup_widget(GTK_WIDGET(flink_new_dialog), "pop_radiobutton")));
  gtk_widget_show(GTK_WIDGET(lookup_widget(GTK_WIDGET(flink_new_dialog), "imap_radiobutton")));
  gtk_widget_show(GTK_WIDGET(lookup_widget(GTK_WIDGET(flink_new_dialog), "mbox_radiobutton")));
       
  gtk_widget_hide(GTK_WIDGET(lookup_widget(GTK_WIDGET(flink_new_dialog), "label22")));
  gtk_widget_hide(GTK_WIDGET(lookup_widget(GTK_WIDGET(flink_new_dialog), "cfg_account_label_type")));
        
  /* this is butt ugly, but we have to make sure the right things get shown */
  gtk_toggle_button_toggled(GTK_TOGGLE_BUTTON(lookup_widget(GTK_WIDGET(flink_new_dialog), "pop_radiobutton")));

  gtk_widget_show(flink_new_dialog);
        
  /*
    GtkWidget	*flink_guide;
	
    new_account = flink_config_create_template_account();
	
    flink_guide = create_flink_guide_window();
    gtk_widget_show(flink_guide);
  */
	
}

/********************************************************
 * Function:	on_prefs_okbutton_clicked		*
 *							*
 * Description:	destroys preferences dialog		*
 *							*
 * Arguments:	GtkButton *button,			*
 *		gpointer user_data, the dialog		*
 *							*
 * Returns:	nothing					*
 ********************************************************/
void
on_prefs_okbutton_clicked	(GtkButton       *button,
				gpointer         user_data)
{
  GtkWidget *dialog = (GtkWidget *) user_data;
  gtk_widget_destroy(dialog);
}

/********************************************************
 * Function:	create_dialog_with_text			*
 *							*
 * Description:	creates new messagedialog with supplied *
 *		text					*
 *							*
 * Arguments:	GtkWidget *widget, parent window	*
 *		gchar *text, message to display		*
 *							*
 * Returns:	nothing					*
 ********************************************************/
void 
create_dialog_with_text	(GtkWidget	*widget,
			gchar		*text) 
{
  GtkWidget	*dialog;
  dialog = gtk_message_dialog_new (GTK_WINDOW(lookup_widget(GTK_WIDGET(widget), 
							    "configure_account_dialog")),
				   GTK_DIALOG_DESTROY_WITH_PARENT,
				   GTK_MESSAGE_ERROR,
				   GTK_BUTTONS_CLOSE,
				   text);

  g_signal_connect (G_OBJECT (dialog), "response",
		    G_CALLBACK (gtk_widget_destroy),
		    NULL);	

  gtk_window_set_resizable (GTK_WINDOW (dialog), FALSE);	

  gtk_widget_show (dialog);
}

/********************************************************
 * Function:	on_prefs_acc_delete_button_clicked	*
 *							*
 * Description:	deletes account from current_path	*
 *							*
 * Arguments:	GtkButton *button, pressed button	*
 *		gpointer user_data,			*
 *							*
 * Returns:	nothing					*
 ********************************************************/
void
on_prefs_acc_delete_button_clicked 	(GtkButton       *button,
					 gpointer         user_data)
{
  GtkTreeSelection	*selection;
  GtkWidget		*configure;
  Account 		*acc;
  gchar			*path;

  if (current_id) {
    acc = flink_config_get_account_from_path((gchar *)
					     g_strdup_printf("/apps/flink/accounts/%d", current_id));
    flink_config_delete_account(acc);
  } else {
    create_dialog_with_text(GTK_WIDGET(button),
			    _("Mark an account for deletion before clicking this button."));
  }		
}

/********************************************************
 * Function:	on_prefs_acc_properties_button_clicked	*
 *							*
 * Description:	opens up properties window		*
 *							*
 * Arguments:	GtkButton *button, Properties button	*
 *		gpointer user_data,			*
 *							*
 * Returns:	nothing					*
 ********************************************************/
void
on_prefs_acc_properties_button_clicked 	(GtkButton       *button,
					 gpointer         user_data)
{
  GtkTreeSelection	*selection;
  GtkWidget		*configure;
  Account 		*acc;
  gchar			*path;

  if (current_id) {
    acc = flink_config_get_account_from_path((gchar *)
					     g_strdup_printf("/apps/flink/accounts/%d", current_id));
    configure = create_configure_account_dialog ();
		
    switch(acc->type) {
    case 1:
      gtk_label_set_text(GTK_LABEL(lookup_widget(configure, "cfg_account_label_type")), "POP3");
      break;
    case 2: 
      gtk_label_set_text(GTK_LABEL(lookup_widget(configure, "cfg_account_label_type")), "IMAPv4");
      break;
    case 3:
      gtk_label_set_text(GTK_LABEL(lookup_widget(configure, "cfg_account_label_type")), "mbox");
      break;
    default:
      break;
    }
		
    //	gtk_label_set_text(GTK_LABEL(g_strdup_printf("%d", lookup_widget(configure, "label33")), acc->id));
	
    gtk_spin_button_set_value(GTK_SPIN_BUTTON(lookup_widget(configure, "id_spinbutton")),
			      (gdouble) acc->id);
    gtk_spin_button_set_value(GTK_SPIN_BUTTON(lookup_widget(configure, "spinbutton2")),
			      (gdouble) acc->interval);
    gtk_spin_button_set_value(GTK_SPIN_BUTTON(lookup_widget(configure, "spinbutton3")),
			      (gdouble) acc->port);

    gtk_widget_show(lookup_widget(configure, "label36"));
    gtk_widget_show(lookup_widget(configure, "entry14"));
    gtk_entry_set_text(GTK_ENTRY(lookup_widget(configure, "entry14")), acc->title);
    gtk_widget_show(configure);
    if (acc->type != 3) {
      gtk_widget_show(lookup_widget(configure, "label24"));
      gtk_widget_show(lookup_widget(configure, "entry9"));
      gtk_entry_set_text(GTK_ENTRY(lookup_widget(configure, "entry9")), acc->hostname);
      gtk_widget_show(lookup_widget(configure, "label25"));
      gtk_widget_show(lookup_widget(configure, "entry10"));
      gtk_entry_set_text(GTK_ENTRY(lookup_widget(configure, "entry10")), acc->username);
      gtk_widget_show(lookup_widget(configure, "label26"));
      gtk_widget_show(lookup_widget(configure, "entry11"));
      gtk_entry_set_text(GTK_ENTRY(lookup_widget(configure, "entry11")), acc->password);
      /* This is for configuring a port to connect to, stupid really so we won't show it */
      gtk_widget_hide(lookup_widget(configure, "label32"));
      gtk_widget_hide(lookup_widget(configure, "spinbutton3"));
      gtk_spin_button_set_value(GTK_SPIN_BUTTON(lookup_widget(configure, "entry14")), 
				(gdouble) acc->port);
    }
    if (acc->type == 2) {
      gtk_widget_show(lookup_widget(configure, "label27"));
      gtk_widget_show(lookup_widget(configure, "entry12"));
      gtk_entry_set_text(GTK_ENTRY(lookup_widget(configure, "entry12")), acc->folder);
    }
    if (acc->type == 3) {
      gtk_widget_show(lookup_widget(configure, "label28"));
      gtk_widget_show(lookup_widget(configure, "entry13"));
      gtk_entry_set_text(GTK_ENTRY(lookup_widget(configure, "entry13")), acc->path);
    }
			

  } else {
    create_dialog_with_text(GTK_WIDGET(button),
			    _("Mark an account for configuration before clicking this button."));
  }		
}

/********************************************************
 * Function:	on_prefs_helpbutton_clicked		*
 *							*
 * Description:	opens up help for preferences		*
 *							*
 * Arguments:	GtkButton *button,			*
 *		gpointer user_data,			*
 *							*
 * Returns:	nothing					*
 ********************************************************/
void
on_prefs_helpbutton_clicked	(GtkButton       *button,
				gpointer         user_data)
{

}

/********************************************************
 * Function:	on_prefs_general_beep_check_toggled	*
 *							*
 * Description:	changes gconf key for system beep	*
 *							*
 * Arguments:	GtkToggleButton *togglebutton, 		*
 *		gpointer user_data			*
 *							*
 * Returns:	nothing					*
 ********************************************************/
void
on_prefs_general_beep_check_toggled 	(GtkToggleButton *togglebutton,
					 gpointer         user_data)
{
  GConfClient	*client;
  client = gconf_client_get_default();
	
  gconf_client_set_bool(client, 
			"/apps/flink/general/beep", 
			gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(togglebutton)), 
			NULL);
}

/********************************************************
 * Function:	on_prefs_general_sound_check_toggled	*
 *							*
 * Description:	changes gconf key for sound output	*
 *							*
 * Arguments:	GtkToggleButton *togglebutton,		*
 *		gpointer user_data,			*
 *							*
 * Returns:	nothing					*
 ********************************************************/
void
on_prefs_general_sound_check_toggled 	(GtkToggleButton *togglebutton,
					 gpointer         user_data)
{
  GConfClient	*client;
  client = gconf_client_get_default();

  gconf_client_set_bool(client, 
			"/apps/flink/general/sound", 
			gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(togglebutton)),
			NULL);
}

/********************************************************
 * Function:	on_druid1_cancel			*
 *							*
 * Description:	destroys account guide			*
 *							*
 * Arguments:	GnomeDruid *gnomedruid,			*
 *		gpointer user_data,			*
 *							*
 * Returns:	nothing					*
 ********************************************************/
void
on_druid1_cancel	(GnomeDruid      *gnomedruid,
			 gpointer         user_data)
{
  gtk_widget_destroy(GTK_WIDGET(lookup_widget(GTK_WIDGET(gnomedruid), "flink_guide_window")));
}

/********************************************************
 * Function:	on_druid1_help				*
 *							*
 * Description:	Shows help for account guide		*
 *							*
 * Arguments:	GnomeDruid *gnomedruid,			*
 *		gpointer user_data,			*
 *							*
 * Returns:	nothing					*
 ********************************************************/
void
on_druid1_help	(GnomeDruid      *gnomedruid,
		gpointer         user_data)
{

}

/********************************************************
 * Function:	on_druidpagefinish1_finish		*
 *							*
 * Description:	tells to add account and destroys guide	*
 *							*
 * Arguments:	GnomeDruidPage *gnomedruidpage,		*
 *		GtkWidget *widget,			*
 *		gpointer user_data,			*
 *							*
 * Returns:	nothing					*
 ********************************************************/
void
on_druidpagefinish1_finish	(GnomeDruidPage	*gnomedruidpage,
				GtkWidget       *widget,
				gpointer         user_data)
{
								
  flink_config_add_new_account(new_account);
  gtk_widget_destroy(GTK_WIDGET(lookup_widget(GTK_WIDGET(widget), "flink_guide_window")));
}

/********************************************************
 * Function:	on_treeview_update			*
 *							*
 * Description:	updates account list on name_list change*
 *							*
 * Arguments:	GConfClient *client,			*
 *		guint cnxn_id,				*
 *		GConfEntry *entry,			*
 *		gpointer user_data,			*
 *							*
 * Returns:	nothing					*
 ********************************************************/
void
on_treeview_update	(GConfClient 	*client, 
			guint 		cnxn_id,
			GConfEntry 	*entry,
			gpointer 	user_data) 
{
				
  GtkTreeIter		iter;
  guint 		current;
  gint			accounts;
  FlinkTree 		*tree = (FlinkTree *) user_data;

  if(!tree)
    return;

  /* FIXME: proper error handling */
	
  accounts = gconf_client_get_int(client, "/apps/flink/accounts/accounts", NULL);
	
  /* clear liststore */
	
  gtk_list_store_clear(tree->model);
	
  /* add objects */
	
  for (current = 1; current <= accounts; current++) {
    gchar	*title;

    (gchar *)title = gconf_client_get_string(client,
					     g_strdup_printf("/apps/flink/accounts/%d_title",
							     current), NULL);
    gtk_list_store_insert (tree->model, &iter, 0);
    gtk_list_store_set (tree->model, &iter, 0, title, 1, current, -1);
  }
}

/********************************************************
 * Function:	on_prefs_treeview_show			*
 *							*
 * Description:	creates treeview/accountlist		*
 *							*
 * Arguments:	GtkWidget *widget,			*
 *		gpointer user_data,			*
 *							*
 * Returns:	nothing					*
 ********************************************************/
void
on_prefs_treeview_show 	(GtkWidget       *widget,
			gpointer         user_data)
{
  FlinkTree		*tree;
  GConfClient		*client;
  GtkTreeIter 		iter;
  guint			current;
  gint			accounts;
	
	
  tree = g_new0(FlinkTree, 1);
	
  client = gconf_client_get_default ();
	
  /* FIXME: proper error handling */

  accounts = gconf_client_get_int(client, "/apps/flink/accounts/accounts", NULL);
	
  /*create listwidget*/
	
  tree->model = gtk_list_store_new (2, G_TYPE_STRING, G_TYPE_INT);
  gtk_tree_view_set_model(GTK_TREE_VIEW(lookup_widget(GTK_WIDGET(widget), "prefs_treeview")), 
			  GTK_TREE_MODEL(tree->model));
  g_object_unref (tree->model);
	
  tree->cell = gtk_cell_renderer_text_new ();
	
  tree->column1 = gtk_tree_view_column_new_with_attributes ("Title", tree->cell,
							    "text", 0, NULL);

  tree->column2 = gtk_tree_view_column_new_with_attributes ("ID", tree->cell,
							    "text", 1, NULL);


  gtk_tree_view_append_column (GTK_TREE_VIEW (lookup_widget(GTK_WIDGET(widget), 
							    "prefs_treeview")),
			       tree->column1);
  gtk_tree_view_append_column (GTK_TREE_VIEW (lookup_widget(GTK_WIDGET(widget),
							    "prefs_treeview")),
			       tree->column2);

  gtk_tree_view_column_set_visible (GTK_TREE_VIEW_COLUMN(tree->column2), FALSE);
		
  tree->selection = gtk_tree_view_get_selection (GTK_TREE_VIEW 
						 (lookup_widget(GTK_WIDGET(widget), 
								"prefs_treeview")));

  g_signal_connect (G_OBJECT (tree->selection), "changed", 
		    G_CALLBACK (flink_prefs_tree_selection_changed_cb),
		    NULL);
		
  /* add objects */
				
  for (current = 1; current <= accounts; current++) {
    gchar	*title;

    (gchar *)title = gconf_client_get_string(client,
					     g_strdup_printf("/apps/flink/accounts/%d_title",
							     current), NULL);
    gtk_list_store_insert (tree->model, &iter, 0);
    gtk_list_store_set (tree->model, &iter, 0, title, 1, current, -1);
  }
  if(widget)
    gconf_client_notify_add(client, "/apps/flink/accounts", 
			    on_treeview_update, (gpointer) tree,
			    NULL, NULL);
}

/********************************************************
 * Function:	on_configure_help_button_clicked	*
 *							*
 * Description:	Opens help for configuration dialog	*
 *							*
 * Arguments:	GtkButton *button, pressed button	*	
 *		gpointer user_data,			*
 *							*
 * Returns:	nothing					*
 ********************************************************/
void
on_configure_help_button_clicked (GtkButton       *button,
				 gpointer         user_data)
{

}

/********************************************************
 * Function:	on_configure_accont_button_clicked 	*
 *							*
 * Description:	Updates account and destroys config dia	*
 *							*
 * Arguments:	GtkButton *button,			*
 *		gpointer user_data,			*
 *							*
 * Returns:	nothing					*
 ********************************************************/
void
on_configure_accont_button_clicked (GtkButton       *button,
				    gpointer         user_data)
{
  GConfClient	*client;
  Account	*acc;
	
  client = gconf_client_get_default();
	
  acc = g_new0(Account, 1);
		
  (const gchar *)acc->title		= gtk_entry_get_text(GTK_ENTRY(lookup_widget(GTK_WIDGET(button),
										     "entry14")));
  (const gchar *)acc->hostname 	= gtk_entry_get_text(GTK_ENTRY(lookup_widget(GTK_WIDGET(button), 
									     "entry9")));
  (const gchar *)acc->username	= gtk_entry_get_text(GTK_ENTRY(lookup_widget(GTK_WIDGET(button), 
									     "entry10")));
  (const gchar *)acc->password	= gtk_entry_get_text(GTK_ENTRY(lookup_widget(GTK_WIDGET(button), 
									     "entry11")));
  (const gchar *)acc->path	= gtk_entry_get_text(GTK_ENTRY(lookup_widget(GTK_WIDGET(button),
									     "entry13")));
  (const gchar *)acc->folder	= gtk_entry_get_text(GTK_ENTRY(lookup_widget(GTK_WIDGET(button), 
									     "entry12")));
									
  acc->id       = gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(lookup_widget(GTK_WIDGET(button),
										 "id_spinbutton")));
  acc->interval	= gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(lookup_widget(GTK_WIDGET(button), 
										 "spinbutton2")));
  acc->port 	= gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(lookup_widget(GTK_WIDGET(button), 
										 "spinbutton3")));
  if (acc->id > 0) {
    flink_config_modify_account(acc);
  } else {
    if(gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(lookup_widget(GTK_WIDGET(button), 
								    "pop_radiobutton")))) {
      acc->type = 1;
    }
    if(gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(lookup_widget(GTK_WIDGET(button), 
								    "imap_radiobutton")))) {
      acc->type = 2;
    }
    if(gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(lookup_widget(GTK_WIDGET(button), 
								    "mbox_radiobutton")))) {
      acc->type = 3;
    }
                        
    acc->id = gconf_client_get_int(client, "/apps/flink/accounts/accounts", NULL) + 1;
    flink_config_add_new_account(acc);
  }

  gtk_widget_destroy(GTK_WIDGET(lookup_widget(GTK_WIDGET(button), "configure_account_dialog")));
}

/********************************************************
 * Function:	on_pop3_radio_clicked			*
 *							*
 * Description:	Modifies guide to create pop account	*
 *							*
 * Arguments:	GtkButton *button,			*
 *		gpointer user_data,			*
 *							*
 * Returns:	nothing					*
 ********************************************************/
void
on_pop3_radio_clicked 	(GtkButton       *button,
			gpointer         user_data)
{
  new_account->type = 1;
  gtk_spin_button_set_value(GTK_SPIN_BUTTON(lookup_widget(GTK_WIDGET(button), "port_spinbutton")),
			    110);
  gtk_widget_hide(GTK_WIDGET(lookup_widget(GTK_WIDGET(button), "folder_label")));
  gtk_widget_hide(GTK_WIDGET(lookup_widget(GTK_WIDGET(button), "folder_entry")));
  gtk_widget_hide(GTK_WIDGET(lookup_widget(GTK_WIDGET(button), "path_label")));
  gtk_widget_hide(GTK_WIDGET(lookup_widget(GTK_WIDGET(button), "path_entry")));

  gtk_widget_show(GTK_WIDGET(lookup_widget(GTK_WIDGET(button), "hostname_label")));
  gtk_widget_show(GTK_WIDGET(lookup_widget(GTK_WIDGET(button), "hostname_entry")));
  gtk_widget_show(GTK_WIDGET(lookup_widget(GTK_WIDGET(button), "username_label")));
  gtk_widget_show(GTK_WIDGET(lookup_widget(GTK_WIDGET(button), "username_entry")));
  gtk_widget_show(GTK_WIDGET(lookup_widget(GTK_WIDGET(button), "password_label")));
  gtk_widget_show(GTK_WIDGET(lookup_widget(GTK_WIDGET(button), "password_entry")));
  gtk_widget_show(GTK_WIDGET(lookup_widget(GTK_WIDGET(button), "port_label")));
  gtk_widget_show(GTK_WIDGET(lookup_widget(GTK_WIDGET(button), "port_spinbutton")));	
}

/********************************************************
 * Function:	on_imap_radio_clicked			*
 *							*
 * Description:	Modifies guide to create imap account	*
 *							*
 * Arguments:	GtkButton *button,			*
 *		gpointer user_data,			*
 *							*
 * Returns:	nothing					*
 ********************************************************/
void
on_imap_radio_clicked 	(GtkButton       *button,
			gpointer         user_data)
{
  new_account->type = 2;
  gtk_spin_button_set_value(GTK_SPIN_BUTTON(lookup_widget(GTK_WIDGET(button), "port_spinbutton")), 143);
  gtk_widget_show(GTK_WIDGET(lookup_widget(GTK_WIDGET(button), "folder_label")));
  gtk_widget_show(GTK_WIDGET(lookup_widget(GTK_WIDGET(button), "folder_entry")));
  gtk_widget_hide(GTK_WIDGET(lookup_widget(GTK_WIDGET(button), "path_label")));
  gtk_widget_hide(GTK_WIDGET(lookup_widget(GTK_WIDGET(button), "path_entry")));

  gtk_widget_show(GTK_WIDGET(lookup_widget(GTK_WIDGET(button), "hostname_label")));
  gtk_widget_show(GTK_WIDGET(lookup_widget(GTK_WIDGET(button), "hostname_entry")));
  gtk_widget_show(GTK_WIDGET(lookup_widget(GTK_WIDGET(button), "username_label")));
  gtk_widget_show(GTK_WIDGET(lookup_widget(GTK_WIDGET(button), "username_entry")));
  gtk_widget_show(GTK_WIDGET(lookup_widget(GTK_WIDGET(button), "password_label")));
  gtk_widget_show(GTK_WIDGET(lookup_widget(GTK_WIDGET(button), "password_entry")));
  gtk_widget_show(GTK_WIDGET(lookup_widget(GTK_WIDGET(button), "port_label")));
  gtk_widget_show(GTK_WIDGET(lookup_widget(GTK_WIDGET(button), "port_spinbutton")));
}

/********************************************************
 * Function:	on_mbox_radio_clicked			*
 *							*
 * Description:	Modifies guide to create mbox account	*
 *							*	
 * Arguments:	GtkButton *button,			*
 *		gpointer user_data,			*
 *							*
 * Returns:	nothing					*
 ********************************************************/
void
on_mbox_radio_clicked	(GtkButton       *button,
			gpointer         user_data)
{
  new_account->type = 3;
  gtk_widget_hide(GTK_WIDGET(lookup_widget(GTK_WIDGET(button), "folder_label")));
  gtk_widget_hide(GTK_WIDGET(lookup_widget(GTK_WIDGET(button), "folder_entry")));
  gtk_widget_show(GTK_WIDGET(lookup_widget(GTK_WIDGET(button), "path_label")));
  gtk_widget_show(GTK_WIDGET(lookup_widget(GTK_WIDGET(button), "path_entry")));

  gtk_widget_hide(GTK_WIDGET(lookup_widget(GTK_WIDGET(button), "hostname_label")));
  gtk_widget_hide(GTK_WIDGET(lookup_widget(GTK_WIDGET(button), "hostname_entry")));
  gtk_widget_hide(GTK_WIDGET(lookup_widget(GTK_WIDGET(button), "username_label")));
  gtk_widget_hide(GTK_WIDGET(lookup_widget(GTK_WIDGET(button), "username_entry")));
  gtk_widget_hide(GTK_WIDGET(lookup_widget(GTK_WIDGET(button), "password_label")));
  gtk_widget_hide(GTK_WIDGET(lookup_widget(GTK_WIDGET(button), "password_entry")));
  gtk_widget_hide(GTK_WIDGET(lookup_widget(GTK_WIDGET(button), "port_label")));
  gtk_widget_hide(GTK_WIDGET(lookup_widget(GTK_WIDGET(button), "port_spinbutton")));
}

/********************************************************
 * Function:	on_druidpagestandard1_next		*
 *							*
 * Description:	Stores down entered guide data to acc	*
 *							*
 * Arguments:	GnomeDruidPage *gnomedruidpage,		*
 *		GtkWidget *widget,			*
 *		gpointer user_data,			*
 *							*
 * Returns:	TRUE if name value is invalid		*
 ********************************************************/
gboolean
on_druidpagestandard1_next	(GnomeDruidPage  	*gnomedruidpage,
				GtkWidget       	*widget,
				gpointer         	user_data)
{
  GConfClient   *client;
  gboolean	retval = FALSE;
  gchar		*key;
	
  client = gconf_client_get_default();
	
  (const gchar *)new_account->id 			= gconf_client_get_int(client,
									       "/apps/flink/accounts/accounts", 
									       NULL) +1;

  (const gchar *)new_account->title		= gtk_entry_get_text(GTK_ENTRY(lookup_widget(GTK_WIDGET(widget),
											     "title_entry")));
  (const gchar *)new_account->hostname 		= gtk_entry_get_text(GTK_ENTRY(lookup_widget(GTK_WIDGET(widget), 
										     "hostname_entry")));
  (const gchar *)new_account->username		= gtk_entry_get_text(GTK_ENTRY(lookup_widget(GTK_WIDGET(widget), 
										     "username_entry")));
  (const gchar *)new_account->password		= gtk_entry_get_text(GTK_ENTRY(lookup_widget(GTK_WIDGET(widget), 
										     "password_entry")));
  (const gchar *)new_account->path		= gtk_entry_get_text(GTK_ENTRY(lookup_widget(GTK_WIDGET(widget), 
											     "path_entry")));
  (const gchar *)new_account->folder		= gtk_entry_get_text(GTK_ENTRY(lookup_widget(GTK_WIDGET(widget), 
											     "folder_entry")));
									
  new_account->interval = gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(lookup_widget(GTK_WIDGET(widget), 
											 "interval_spinbutton")));
							
  new_account->port = gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(lookup_widget(GTK_WIDGET(widget), 
										     "port_spinbutton")));
		
  return retval;
}

/********************************************************
 * Function:	on_general_prefs_sndbrws_filesel_cancel	*
 *							*
 * Description:	destroys fileselection for sound	*
 *							*
 * Arguments:	GtkWidget *w,				*
 *		gpointer data, fileselector		*
 *							*
 * Returns:	nothing					*
 ********************************************************/
void
on_general_prefs_sndbrws_filesel_cancel	(GtkWidget 	*w, 
					gpointer 	data)
{
  gtk_widget_destroy(GTK_WIDGET(data));
}

/********************************************************
 * Function:	on_general_prefs_sndbrws_filesel_ok	*
 *							*
 * Description:	stores filename in gconf and closes	*
 *							*
 * Arguments:	GtkWidget *w,				*
 *		gpointer data, fileselector		*
 *							*
 * Returns:	nothing					*
 ********************************************************/
void
on_general_prefs_sndbrws_filesel_ok	(GtkWidget 	*w,
					gpointer 	data)
{
  GConfClient	*client;
  gchar		*filename;

  client = gconf_client_get_default();
	
  (G_CONST_RETURN) filename = gtk_file_selection_get_filename(GTK_FILE_SELECTION(data));
	
  gconf_client_set_string(client, "/apps/flink/general/soundfile", filename, NULL);
	
  gtk_widget_destroy(GTK_WIDGET(data));
	
}

/********************************************************
 * Function:	on_prefs_general_soundfile_changed	*
 *							*
 * Description:	changes cfg-dia entry for soundfile	*
 *							*
 * Arguments:	GConfClient *client, gconf client	*
 *		guint cnxn_id, notify connection id 	*
 *		GConfEntry *entry, general/soundfile	*
 *		gpointer user_data, soundfile entry	*
 *							*
 * Returns:	nothing					*
 ********************************************************/
void
on_prefs_general_soundfile_changed (GConfClient	*client,
				    guint	cnxn_id,
				    GConfEntry	*entry,
				    gpointer	user_data)
{
  GtkWidget	*widget = (GtkWidget *) user_data;
  gchar		*soundfile;

  soundfile = gconf_client_get_string(client,
				      "/apps/flink/general/soundfile",
				      NULL);
  gtk_entry_set_text(GTK_ENTRY(lookup_widget(GTK_WIDGET(widget), "general_prefs_sound_entry")),
		     soundfile);
  gconf_client_notify_remove(client, cnxn_id);
}

/********************************************************
 * Function:	on_prefs_general_sndbrws_button_clicked	*
 *							*
 * Description:	creates fileselector for soundfile	*
 *							*
 * Arguments:	GtkButton *button,			*
 *		gpointer user_data,			*
 *							*
 * Returns:	nothing					*
 ********************************************************/
void
on_prefs_general_sndbrws_button_clicked	(GtkButton       *button,
					gpointer         user_data)
{
  GConfClient	*client;
  GtkWidget	*filesel;

  client = gconf_client_get_default();
	
  filesel = create_prefs_fileselect();
	
  gtk_widget_show(filesel);

  g_signal_connect(GTK_OBJECT(lookup_widget(GTK_WIDGET(filesel), "ok_button1")),
		   "clicked",
		   GTK_SIGNAL_FUNC(on_general_prefs_sndbrws_filesel_ok),
		   (gpointer) filesel);
  g_signal_connect(GTK_OBJECT(lookup_widget(GTK_WIDGET(filesel), "cancel_button1")),
		   "clicked",
		   GTK_SIGNAL_FUNC(on_general_prefs_sndbrws_filesel_cancel),
		   (gpointer) filesel);
	
  gconf_client_notify_add(client,
			  "/apps/flink/general/soundfile",
			  on_prefs_general_soundfile_changed,
			  (gpointer) button,
			  NULL,
			  NULL);
}

/********************************************************
 * Function:	on_general_prefs_sound_preview_button_clicked*
 *							*
 * Description:	Plays sound in soundfile entry		*
 *							*
 * Arguments:	GtkButton *button,			*
 *		gpointer user_data,			*
 *							*
 * Returns:	nothing					*
 ********************************************************/
void
on_general_prefs_sound_preview_button_clicked	(GtkButton       *button,
						 gpointer         user_data)
{
  gnome_sound_play(gtk_entry_get_text(GTK_ENTRY(lookup_widget(GTK_WIDGET(button), 
							      "general_prefs_sound_entry"))));
}

/********************************************************
 * Function:	on_general_prefs_autorun_program_check_toggled*
 *							*
 * Description:	Changes gconf according to autorun	*
 *		toggle in prefs dialog			*
 *							*
 * Arguments:	GtkToggleButton *togglebutton,		*
 *		gpointer user_data,			*
 *							*
 * Returns:	nothing					*
 ********************************************************/
void
on_general_prefs_autorun_program_check_toggled	(GtkToggleButton *togglebutton,
						 gpointer         user_data)
{
  GConfClient	*client;

  client = gconf_client_get_default();

  gconf_client_set_bool(client,
			"/apps/flink/general/program_autorun",
			gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(togglebutton)),
			NULL);
}

/********************************************************
 * Function:	on_general_prefs_program_filesel_cancel	*
 *							*
 * Description:	Destroys program fileselector		*
 *							*
 * Arguments:	GtkWidget *w,				*
 *		gpointer data, fileselector		*
 *							*
 * Returns:	nothing					*
 ********************************************************/
void
on_general_prefs_program_filesel_cancel (GtkWidget *w,
					gpointer data)
{
  gtk_widget_destroy(GTK_WIDGET(data));
}

/********************************************************
 * Function:	on_general_prefs_program_filesel_ok	*
 *							*
 * Description:	Sets programfile from program filesel	*
 *							*
 * Arguments:	GtkWidget *w,				*
 *		gpointer data, fileselector		*
 *							*
 * Returns:	nothing					*
 ********************************************************/
void
on_general_prefs_program_filesel_ok	(GtkWidget *w,
					gpointer data)
{
  GConfClient     *client;
  gchar           *filename;

  client = gconf_client_get_default();

  (G_CONST_RETURN) filename = gtk_file_selection_get_filename(GTK_FILE_SELECTION(data));
	
  gconf_client_set_string(client, "/apps/flink/general/programfile", filename, NULL);

  gtk_widget_destroy(GTK_WIDGET(data));
}

/********************************************************
 * Function:	on_prefs_general_programfile_changed	*
 *							*
 * Description:	Changes program entry according to gconf*
 *							*
 * Arguments:	GConfClient *client, gconfclient	*
 *		guint cnxn_id, notifyer connection id	*
 *		GConfEntry *entry, general/programfile	*
 *		gpointer user_data, programfile entry	*
 *							*
 * Returns:	nothing					*
 ********************************************************/
void
on_prefs_general_programfile_changed 	(GConfClient	*client,
					 guint		cnxn_id,
					 GConfEntry	*entry,
					 gpointer	user_data)
{
  GtkWidget	*widget = (GtkWidget *) user_data;
  gchar		*programfile;

  programfile = gconf_client_get_string(client,
					"/apps/flink/general/programfile",
					NULL);
  gtk_entry_set_text(GTK_ENTRY(lookup_widget(GTK_WIDGET(widget), "general_prefs_program_entry")),
		     programfile);
  gconf_client_notify_remove(client, cnxn_id);
}

/********************************************************
 * Function:	on_general_prefs_program_browse_clicked	*
 *							*
 * Description:	opens fileselector for programfile	*
 *							*
 * Arguments:	GtkButton *button, pressed button	*
 *		gpointer data,				*
 *							*
 * Returns:	nothing					*
 ********************************************************/
void
on_general_prefs_program_browse_clicked		(GtkButton       *button,
						 gpointer         user_data)
{
  GConfClient	*client;
  GtkWidget       *filesel;

  client = gconf_client_get_default();

  filesel = create_prefs_fileselect();
  gtk_widget_show(filesel);

  g_signal_connect(GTK_OBJECT(lookup_widget(GTK_WIDGET(filesel), "ok_button1")),
		   "clicked",
		   GTK_SIGNAL_FUNC(on_general_prefs_program_filesel_ok),
		   (gpointer) filesel);
  g_signal_connect(GTK_OBJECT(lookup_widget(GTK_WIDGET(filesel), "cancel_button1")),
		   "clicked",
		   GTK_SIGNAL_FUNC(on_general_prefs_program_filesel_cancel),
		   (gpointer) filesel);					

  gconf_client_notify_add(client,
			  "/apps/flink/general/programfile",
			  on_prefs_general_programfile_changed,
			  (gpointer) button,
			  NULL,
			  NULL);

}

/********************************************************
 * Function:	on_prefs_dialog_show			*
 *							*
 * Description:	changes general settings in prefs dialog*
 *		according to gconf database		*
 *							*
 * Arguments:	GtkWidget *widget,			*
 *		gpointer user_data,			*
 *							*
 * Returns:	nothing					*
 ********************************************************/
void
on_prefs_dialog_show	(GtkWidget       *widget,
			gpointer         user_data)
{
  GConfClient	*client;

  client = gconf_client_get_default();

  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(lookup_widget(GTK_WIDGET(widget), "prefs_general_beep_check")),
			       gconf_client_get_bool(client, "/apps/flink/general/beep", NULL));
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(lookup_widget(GTK_WIDGET(widget), "prefs_general_sound_check")),
			       gconf_client_get_bool(client, "/apps/flink/general/sound", NULL));
  gtk_entry_set_text(GTK_ENTRY(lookup_widget(GTK_WIDGET(widget), "general_prefs_sound_entry")),
		     gconf_client_get_string(client, "/apps/flink/general/soundfile", NULL));
	
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(lookup_widget(GTK_WIDGET(widget), 
							       "general_prefs_program_in_status_check")),
			       gconf_client_get_bool(client, "/apps/flink/general/program_in_menu", NULL));
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(lookup_widget(GTK_WIDGET(widget), 
							       "general_prefs_autorun_program_check")),
			       gconf_client_get_bool(client, "/apps/flink/general/program_autorun", NULL));
  gtk_entry_set_text(GTK_ENTRY(lookup_widget(GTK_WIDGET(widget), "general_prefs_program_entry")),
		     gconf_client_get_string(client, "/apps/flink/general/programfile", NULL));
}

/********************************************************
 * Function:	on_flink_preferences_ok			*
 *							*
 * Description:	realises and destroys prefs dialog	*
 *							*
 * Arguments:	GtkButton *button,			*
 *		gpointer user_data,			*
 *							*
 * Returns:	nothing					*
 ********************************************************/
void
on_flink_preferences_ok	(GtkButton       *button,
			gpointer         user_data)
{
  GConfClient	*client;

  client = gconf_client_get_default();

  gconf_client_set_string(client, 
			  "/apps/flink/general/soundfile",
			  (G_CONST_RETURN gchar*) gtk_entry_get_text(GTK_ENTRY(lookup_widget(GTK_WIDGET(button),
											     "general_prefs_sound_entry"))),
			  NULL);
  gconf_client_set_string(client,
			  "/apps/flink/general/programfile",
			  (G_CONST_RETURN gchar*) gtk_entry_get_text(GTK_ENTRY(lookup_widget(GTK_WIDGET(button),
											     "general_prefs_program_entry"))),
			  NULL);

  gtk_widget_destroy(GTK_WIDGET(lookup_widget(GTK_WIDGET(button), "prefs_dialog")));
}

/********************************************************
 * Function:	on_pop_radiobutton_toggled		*
 *							*
 * Description:	alters newaccount dialog to reflect 	*
 *              the correct account type                *
 *							*
 * Arguments:	GtkToggleButton *togglebutton,		*
 *		gpointer user_data,			*
 *							*
 * Returns:	nothing					*
 ********************************************************/
void
on_pop_radiobutton_toggled             (GtkToggleButton *togglebutton,
                                        gpointer         user_data)
{
        
  /* server hostname */
  gtk_widget_show(GTK_WIDGET(lookup_widget(GTK_WIDGET(togglebutton),
					   "label24")));
  gtk_widget_show(GTK_WIDGET(lookup_widget(GTK_WIDGET(togglebutton),
					   "entry9")));
  /* Server port, unnecessary, lets hide it instead */
  gtk_widget_hide(GTK_WIDGET(lookup_widget(GTK_WIDGET(togglebutton),
					   "label32")));
  gtk_widget_hide(GTK_WIDGET(lookup_widget(GTK_WIDGET(togglebutton),
					   "spinbutton3")));
  gtk_spin_button_set_value(GTK_SPIN_BUTTON(lookup_widget(GTK_WIDGET(togglebutton),
							  "spinbutton3")), 110);
  /* username */
  gtk_widget_show(GTK_WIDGET(lookup_widget(GTK_WIDGET(togglebutton),
					   "label25")));
  gtk_widget_show(GTK_WIDGET(lookup_widget(GTK_WIDGET(togglebutton),
					   "entry10")));
  /* password */
  gtk_widget_show(GTK_WIDGET(lookup_widget(GTK_WIDGET(togglebutton),
					   "label26")));
  gtk_widget_show(GTK_WIDGET(lookup_widget(GTK_WIDGET(togglebutton),
					   "entry11")));
  /* folder */
  gtk_widget_hide(GTK_WIDGET(lookup_widget(GTK_WIDGET(togglebutton),
					   "label27")));
  gtk_widget_hide(GTK_WIDGET(lookup_widget(GTK_WIDGET(togglebutton),
					   "entry12")));
  /* path */
  gtk_widget_hide(GTK_WIDGET(lookup_widget(GTK_WIDGET(togglebutton),
					   "label28")));
  gtk_widget_hide(GTK_WIDGET(lookup_widget(GTK_WIDGET(togglebutton),
					   "entry13")));

}


void
on_imap_radiobutton_toggled            (GtkToggleButton *togglebutton,
                                        gpointer         user_data)
{
  /* server hostname */
  gtk_widget_show(GTK_WIDGET(lookup_widget(GTK_WIDGET(togglebutton),
					   "label24")));
  gtk_widget_show(GTK_WIDGET(lookup_widget(GTK_WIDGET(togglebutton),
					   "entry9")));
  /* Server port */
  gtk_widget_hide(GTK_WIDGET(lookup_widget(GTK_WIDGET(togglebutton),
					   "label32")));
  gtk_widget_hide(GTK_WIDGET(lookup_widget(GTK_WIDGET(togglebutton),
					   "spinbutton3")));
  gtk_spin_button_set_value(GTK_SPIN_BUTTON(lookup_widget(GTK_WIDGET(togglebutton),
							  "spinbutton3")), 143);
  /* username */
  gtk_widget_show(GTK_WIDGET(lookup_widget(GTK_WIDGET(togglebutton),
					   "label25")));
  gtk_widget_show(GTK_WIDGET(lookup_widget(GTK_WIDGET(togglebutton),
					   "entry10")));
  /* password */
  gtk_widget_show(GTK_WIDGET(lookup_widget(GTK_WIDGET(togglebutton),
					   "label26")));
  gtk_widget_show(GTK_WIDGET(lookup_widget(GTK_WIDGET(togglebutton),
					   "entry11")));
  /* folder */
  gtk_widget_show(GTK_WIDGET(lookup_widget(GTK_WIDGET(togglebutton),
					   "label27")));
  gtk_widget_show(GTK_WIDGET(lookup_widget(GTK_WIDGET(togglebutton),
					   "entry12")));
  /* path */
  gtk_widget_hide(GTK_WIDGET(lookup_widget(GTK_WIDGET(togglebutton),
					   "label28")));
  gtk_widget_hide(GTK_WIDGET(lookup_widget(GTK_WIDGET(togglebutton),
					   "entry13")));

}


void
on_mbox_radiobutton_toggled            (GtkToggleButton *togglebutton,
                                        gpointer         user_data)
{
  /* server hostname */
  gtk_widget_hide(GTK_WIDGET(lookup_widget(GTK_WIDGET(togglebutton),
					   "label24")));
  gtk_widget_hide(GTK_WIDGET(lookup_widget(GTK_WIDGET(togglebutton),
					   "entry9")));
  /* Server port */
  gtk_widget_hide(GTK_WIDGET(lookup_widget(GTK_WIDGET(togglebutton),
					   "label32")));
  gtk_widget_hide(GTK_WIDGET(lookup_widget(GTK_WIDGET(togglebutton),
					   "spinbutton3")));
  /* username */
  gtk_widget_hide(GTK_WIDGET(lookup_widget(GTK_WIDGET(togglebutton),
					   "label25")));
  gtk_widget_hide(GTK_WIDGET(lookup_widget(GTK_WIDGET(togglebutton),
					   "entry10")));
  /* password */
  gtk_widget_hide(GTK_WIDGET(lookup_widget(GTK_WIDGET(togglebutton),
					   "label26")));
  gtk_widget_hide(GTK_WIDGET(lookup_widget(GTK_WIDGET(togglebutton),
					   "entry11")));
  /* folder */
  gtk_widget_hide(GTK_WIDGET(lookup_widget(GTK_WIDGET(togglebutton),
					   "label27")));
  gtk_widget_hide(GTK_WIDGET(lookup_widget(GTK_WIDGET(togglebutton),
					   "entry12")));
  /* path */
  gtk_widget_show(GTK_WIDGET(lookup_widget(GTK_WIDGET(togglebutton),
					   "label28")));
  gtk_widget_show(GTK_WIDGET(lookup_widget(GTK_WIDGET(togglebutton),
					   "entry13")));

}


void
on_theme_treeview_show                 (GtkWidget       *widget,
                                        gpointer         user_data)
{
  GConfClient		*client;
  GtkListStore		*model;
  GtkTreeViewColumn	*column1;
  GtkCellRenderer	*text_cell;
  GtkTreeSelection	*selection;
  GtkTreeIter		*iter;

  model = gtk_list_store_new (1, G_TYPE_STRING);
  gtk_tree_view_set_model(GTK_TREE_VIEW(widget), GTK_TREE_MODEL(model));
  g_object_unref(model);

  text_cell = gtk_cell_renderer_text_new();
  column1 = gtk_tree_view_column_new_with_attributes("Title", text_cell, "text", 0, NULL);
  gtk_tree_view_append_column (GTK_TREE_VIEW(widget), column1);
  selection = gtk_tree_view_get_selection (GTK_TREE_VIEW(widget));

  g_signal_connect (G_OBJECT(selection), "changed",
		    NULL,
		    NULL);
}

