## Module: wxgui

##    Fonty Python Copyright  (C)  2006 Donn.C.Ingle
##
##    Contact: donn.ingle@gmail.com - I hope this email lasts.
##
##    This program is free software; you can redistribute it and/or modify
##    it under the terms of the GNU General Public License as published by
##    the Free Software Foundation; either version 2 of the License, or
##     ( at your option )  any later version.
##
##    This program is distributed in the hope that it will be useful,
##    but WITHOUT ANY WARRANTY; without even the implied warranty of
##    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
##    GNU General Public License for more details.
##
##    You should have received a copy of the GNU General Public License
##    along with this program; if not, write to the Free Software
##    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
##

import sys, os
import pathcontrol
import strings
import fpsys # Global objects
import fontybugs
from pog import *
import fpversion

## Now, bring in all those big modules
try: import wx
except: sys.exit  ( strings.wxError ( ) ) 

## PIL
import Image, ImageFont, ImageDraw 

import wx.lib.scrolledpanel
import wx.lib.statbmp
from wx.lib.splitter import MultiSplitterWindow

## Fetch my own pusub stuff
from pubsub import *
ps = CPubsub ( )

## Fetch the dialogue classes *About and Settings* if the gui is in max mode.
if fpsys.config.max: import dialogues

## This class is a bitmap of a TTF font - it detects a click and also displays itself with
## other information below it.
class Fitmap ( wx.lib.statbmp.GenStaticBitmap ):
    """A Fitmap is a bitmap of a font, with events and a label and a bunch of stuff."""
    def __init__ ( self,parent, pos ,fitem ) :
        self.name = fitem.name
        
        self.fitem = fitem
        self.parent = parent
        
        ## I prevent font's own height due to mad ttfs.
        ## I based this on my preference. Some ttfs had these crazy heights
        ## and would shoot off the page. So, I nixed that.
        self.height = fpsys.config.points * 1.55
        
        self.bitmap, self.goodfont = self.__wxTtfBitmap ( self.fitem.paf, fpsys.config.points, " " + fpsys.config.text + " " ) 
        self.whitebrush = wx.Brush ( 'WHITE',wx.SOLID ) 
        
        sz =  ( self.bitmap.GetWidth (  ) , self.height ) 
        self.gsb = wx.lib.statbmp.GenStaticBitmap.__init__ ( self,parent,-1,self.bitmap,pos,sz )
        
        self.width = self.bitmap.GetWidth (  ) 
        
        self.Bind ( wx.EVT_LEFT_UP,self.__onClick ) 
        self.Bind ( wx.EVT_PAINT,  self.__onPaint ) 
        
    def __onPaint ( self, event ) :
        dc = wx.PaintDC ( self ) 
        dc.SetBackground ( self.whitebrush ) 
        dc.Clear (  ) 

        if self._bitmap:
            if self.fitem.name == "EMPTY":
                dc.SetTextForeground ( "black" ) 
                dc.SetFont ( wx.Font ( 20, family=wx.SWISS, style=wx.NORMAL, weight=wx.NORMAL, underline=False )  ) 
                dc.DrawText ( "Choose a Folder or a Pog,", 1, 1 )
                dc.SetFont ( wx.Font ( 14, family=wx.SWISS, style=wx.NORMAL, weight=wx.NORMAL, underline=False )  ) 
                dc.DrawText ( "or clear the filter box below.", 1, 30 )
            else:
                ## Draw the actual font bitmap with a tick/cross over it  ( If ticked ) 
                if self.fitem.ticked:
                    dc.DrawBitmap ( self._bitmap, 10, 0, True ) 
                    dc.DrawBitmap ( self.parent.parent.TICKMAP , 20, 5, True ) 
                else:
                    dc.DrawBitmap ( self._bitmap, 0, 0, True ) 
    
                #Now draw the text showing the font name
                if self.goodfont:
                    w = wx.NORMAL
                    if self.fitem.inactive: fcol = "gray"
                    else: fcol = "black"
                    txt = self.name
                else:
                    fcol = "red"
                    w = wx.BOLD
                    txt = "Bad font:%s" % self.name
                #tw , th = dc.GetTextExtent ( txt ) 
                th = 18 #Seems faster to guess the height... gulp!
                ## Prep the text to show "font family name (font file name)" -- suggested by Chris Mohler.
                ## Added Style info 3 Dec 2006:
                txt = "%s - %s - [%s]" % ( self.fitem.family, self.fitem.style, txt )
                self.__fatFont ( dc, 4, self.height - th, fcol, "white", 10, w, txt )
                
                ## Special message
                if self.fitem.inactive:
                    self.__fatFont ( dc, 2, self.height - th - th, "gray", "white", 10, wx.NORMAL, self.fitem.msg )
                    
                ## Now a dividing line
                dc.SetPen ( wx.Pen ( "black",1 )  ) 
                dc.SetBrush ( wx.TRANSPARENT_BRUSH ) 
                dc.DrawLine ( 0,self.height-1,self.width,self.height-1 ) 
                
    ## Draw a string with a backdrop highlight. Makes it "fat"
    ## I suppose I should draw one string, then try blitting it around rather than DrawTexting it. Later ...
    def __fatFont ( self, dc, x, y, fcol, bcol, points, weight, txt ):
                dc.SetFont ( wx.Font ( points, family=wx.SWISS, style=wx.NORMAL, weight=weight, underline=False )  ) 
                #Draw the back-drop text
                dc.SetTextForeground ( bcol )
                dc.DrawText ( txt, x-1, y-1 )
                dc.DrawText ( txt, x-1, y+1 )
                dc.DrawText ( txt, x+1, y-1 )
                dc.DrawText ( txt, x+1, y+1 )
                
                dc.DrawText ( txt, x, y-1 )
                dc.DrawText ( txt, x-1, y )
                dc.DrawText ( txt, x+1, y )
                dc.DrawText ( txt, x, y+1 )
                #Final text
                dc.SetTextForeground ( fcol )
                dc.DrawText ( txt, x, y )
                
    def __onClick ( self, event ) :
        if self.goodfont and fpsys.state.cantick and not self.fitem.inactive:
            self.Refresh (  )  #forces a redraw.
            self.fitem.ticked = not ( self.fitem.ticked ) 
            ## Inc or dec a counter depending on the tickedness of this item
            if self.fitem.ticked: fpsys.state.numticks += 1
            if not self.fitem.ticked: fpsys.state.numticks -= 1
            ps.pub ( toggle_main_button )
            
    def __wxTtfBitmap ( self, paf, points, text ) :
        """Make a font and draw it using PIL. Then convert it to a wx bitmap.
        If there is a faster/better way to do this - please let me know!
        """
        if self.fitem.name == "EMPTY":
            wximage = wx.EmptyImage (100,100 ) 
            goodfont = True
        else:
            goodfont = True #If False, then the ttf could not be drawn.
            ly = int ( self.height )  #We control the height.
            try:
                font = ImageFont.truetype ( paf, points ) 
                ## Family and Style :- Added Nov 2006
                self.fitem.family = font.getname ( ) [ 0 ]
                self.fitem.style =  font.getname ( ) [ 1 ]
                lx,discard = font.getsize ( text ) 
                lx = int ( lx )  #Gotta int it because PIL prints a warning.
                pilimage = Image.new ( "RGB",  ( lx,ly ) ,  ( 255,255,255 )  ) 
                draw = ImageDraw.Draw ( pilimage ) 
                if self.fitem.inactive:
                    col =  ( 230, 230, 230 ) 
                else:
                    col = 0
                draw.text (  ( 0,0 ) , text, font=font, fill=col ) 
            except IOError: #This one happens a lot. :(
                lx = 200 #Random width -- a wild guess.
                pilimage = Image.new ( "RGB",  ( lx,ly ) ,  ( 255,255,255 )  ) 
                goodfont = False
            except:
                ps.pub ( show_error_and_abort, "Unhandled PIL error.%s" % sys.exc_info ( ) [ 0 ] ) 
                sys.exit ( )
            try:    
                if lx == 0: #Bad pilimage all-round.
                    wximage = wx.EmptyImage ( 1,ly ) #This makes a black buffer (0,0,0) by default.
                    wximage.InitAlpha (  ) 
                    goodfont = False
                else:
                    wximage = wx.EmptyImage ( lx,ly ) 
                    wximage.SetData ( pilimage.convert ( 'RGB' ) .tostring (  )  ) #pilimage can be bad here. Bad TTF from draw.text
                    wximage.InitAlpha (  ) 
            except:
                ps.pub ( show_error_and_abort, "Error creating a wximage of %s" %  paf ) 
        return  ( wximage.ConvertToBitmap (  ) ,goodfont ) 


## This is the main font control, the child of FontViewPanel.
class FontView  ( wx.lib.scrolledpanel.ScrolledPanel ) :
    """Draw a list of fitmaps from a font list object (derived from BasicFontList)"""
    def __init__  ( self, parent ):
        wx.lib.scrolledpanel.ScrolledPanel.__init__ ( self, parent, -1 ) 
        self.fitmaps = [ ]
        self.parent = parent
        
    def DrawFitmaps ( self, viewobject ) :
        """Draw X number of fitmaps down the control"""
        ## Ensure we destroy all old fitmaps -- and I mean it.
        for f in self.fitmaps:
           f.Destroy (  )  #Ah, nailed ya! You bastard! I fart in your general direction!

        self.fitmaps = [ ]
        
        ####
        ## It's NB to notice that the fitems being put into self.fitmaps are
        ## the SAME items that are in the viewobject.
        ## Hence, when one is flagged as ticked, it's not just the one
        ## in here (self.fitmaps), but because they are ONE OBJECT,
        ## you can rely on the the fitem within viewobject having a ticked True/False
        ## attribute outside of this class.
        
        i = 0
        h = 0
        self.mySizer = wx.BoxSizer ( wx.VERTICAL ) 
        ## If our viewobject has NO FONTS inside it (i.e. it's an EmptyView object )
        ## then setup a fake FontItem so we can have a dud Fitmap to show.
        if len ( viewobject ) == 0:
            empty_fitem = FontItem ( "EMPTY", ticked = False )
            fm = Fitmap ( self, ( 0, 0 ), empty_fitem )
            self.mySizer.Add ( fm, 0, wx.GROW )
        else:
            for fitem in viewobject:
                fm = Fitmap ( self, ( 0, i * h ) , fitem ) ##Actually draws the fitmap too.
                h = fm.height + 10
                self.fitmaps.append ( fm ) 
                self.mySizer.Add ( fm, 0, wx.GROW ) 
                i += 1
                #per =  ( 100.00/L )  * i
                #print "%3d" % per

        self.SetSizer ( self.mySizer ) 
        self.SetAutoLayout ( True ) 
        self.SetupScrolling ( rate_x=5, rate_y=4 ) 


## I have tried to make this a stand-alone object. It's a Panel that holds the 
## FontView control, the controls above it and the buttons below it.
## I would like to shift it out to another file, but that's for another day.
class FontViewPanel ( wx.Panel ):
    """Standalone visual control to select TTF fonts."""
    def __init__ ( self, parent ):
        wx.Panel.__init__ ( self, parent, id = -1 )
        
        self.pageindex = 1 # I start here
        self.total_number_of_pages = 0
        
        self.filter = ""
        
        self.TICKMAP = None
        self.TICK = wx.Bitmap ( fpsys.mythingsdir + "tick.png", type=wx.BITMAP_TYPE_PNG )
        self.CROSS = wx.Bitmap ( fpsys.mythingsdir + "cross.png", type=wx.BITMAP_TYPE_PNG )
        
        ## Main Label on top
        sizerMainLabel = wx.BoxSizer ( wx.HORIZONTAL ) 
        self.textMainInfo = wx.StaticText  ( self, -1, "Information display" ) 
        self.textMainInfo.SetFont ( wx.Font ( 10, wx.FONTFAMILY_DEFAULT, wx.NORMAL, wx.FONTWEIGHT_BOLD ) )
        sizerMainLabel.Add  ( self.textMainInfo,1,wx.ALIGN_CENTER_VERTICAL | wx.ALIGN_LEFT ) 
        
        ## Page choice and Filter controls
        sizerOtherControls = wx.BoxSizer ( wx.HORIZONTAL )

        self.textFilter = wx.StaticText  ( self, -1, "Filter:" )
        self.inputFilter = wx.TextCtrl ( self, -1, "" )
        self.inputFilter.Bind ( wx.EVT_CHAR, self.__evtChar ) #catch the enter char
        self.choicePage = wx.Choice  ( self, -1, choices = ["busy"] ) 
        self.choicePage.Bind  ( wx.EVT_CHOICE, self.__onPagechoiceClick ) #Bind choice event
        
        sizerOtherControls.Add ( self.textFilter, 0, wx.ALIGN_LEFT | wx.ALIGN_CENTER_VERTICAL )
        sizerOtherControls.Add ( self.inputFilter, 1, wx.ALIGN_LEFT | wx.EXPAND )
        sizerOtherControls.Add ( ( 4,-1 ), 0, wx.EXPAND )
        sizerOtherControls.Add  ( self.choicePage, 0 ,wx.EXPAND | wx.ALIGN_RIGHT )  #Added it to the sizer
        
        ## The FONT panell:
        self.panelFontView = FontView  ( self ) 
        
        buttonsSizer = wx.BoxSizer  ( wx.HORIZONTAL ) 
        self.buttPrev = wx.Button ( self, wx.ID_BACKWARD ) 

        self.buttMain   = wx.Button ( self, label="loading", id = 3 ) 
        self.buttNext = wx.Button ( self, wx.ID_FORWARD ) 
        self.buttPrev.Enable ( False )  #Starts out disabled
        
        buttonsSizer.Add ( self.buttPrev,0,wx.EXPAND ) 
        buttonsSizer.Add (  ( 10,1 ) ,0,wx.EXPAND ) 
        buttonsSizer.Add ( self.buttMain,1,wx.EXPAND ) 
        buttonsSizer.Add (  ( 10,1 ) ,0,wx.EXPAND ) 
        buttonsSizer.Add ( self.buttNext,0,wx.EXPAND ) 

        ## Now the sizer to hold all the fontview controls
        self.sizerFontView = wx.BoxSizer ( wx.VERTICAL ) 
        self.sizerFontView.Add ( sizerMainLabel, 0, wx.EXPAND | wx.TOP | wx.BOTTOM, border = 5 )  #The Main label
        self.sizerFontView.Add ( self.panelFontView, 1, wx.EXPAND )  #The font view
        self.sizerFontView.Add ( sizerOtherControls, 0, wx.EXPAND | wx.TOP | wx.BOTTOM, border = 3 ) #Choice and Filter
        self.sizerFontView.Add  ( buttonsSizer,0,wx.EXPAND )  #The buttons       
        
        self.SetSizer ( self.sizerFontView )
        
        self.buttPrev.Bind ( wx.EVT_LEFT_UP,self.__navClick ) 
        self.buttNext.Bind ( wx.EVT_LEFT_UP,self.__navClick ) 
        self.buttMain.Bind ( wx.EVT_LEFT_UP,self.__onMainClick ) 
    
        ps.sub ( toggle_main_button, self.ToggleMainButton ) 
        
    ## Catch the ENTER key in the filter text input control
    def __evtChar ( self, e ):
        if e.GetKeyCode ( ) == 13:
            ## Time to filter and redraw my list
            self.filter = self.inputFilter.GetValue ( )
            ## Now command a change of the view.
            ## First, return user to page 1:
            self.pageindex = 1
            self.__drawTheFonts ( )
            self.buttMain.SetFocus  (  )  #a GTK bug demands this move. Restore the ESC key func.
        if e.GetKeyCode ( ) == 27:
	    self.buttMain.SetFocus ( )
	e.Skip ( )
        
    def __drawTheFonts ( self ):
        """Figure out what list of fonts to draw, divide them into pages and draw them."""
        self.total_number_of_pages = 1 # A default
        if len ( fpsys.state.viewobject ) > 0:
            ## STEP 1 : Handle the Filter:
            filteredList = fpsys.state.viewobject
            if self.filter is not "":
            ## Okay, we have some kind of filter.
            ## This idea was suggested by  user Chris Mohler.
            ##
            ## I tried to make this more efficient by remembering the last filter and the last viewobject that
            ## used it - to try minimize the number of times the ENTIRE list is filtered, but it caused such
            ## problems for my poor brain. It's not a very good brain anyway. So I stopped trying. H.A.N.D
            ##
            ## Have NO IDEA what this will do with Unicode and so forth...
                import re
                test = re.compile ( self.filter, re.IGNORECASE )
                filteredList = [ f for f in fpsys.state.viewobject if test.search ( f.name + f.family + f.style ) ] 

            ## STEP 2 : Figure out how many pages we have to display
            current_page = self.pageindex - 1
            num_in_one_page = fpsys.config.numinpage
            total_num_fonts = len ( filteredList )
            #print "total_num_fonts=", total_num_fonts
            gross = total_num_fonts / float ( num_in_one_page )
            
            if gross <= 1:
                ## There are less than num_in_one_page fonts to be viewed at all.
                self.total_number_of_pages = 1
            else:
                ## Okay, we have at least 1 page, perhaps more.
                whole_number_of_pages = int ( gross )
                remainder = whole_number_of_pages % num_in_one_page
                if remainder > 0: whole_number_of_pages += 1
                self.total_number_of_pages = whole_number_of_pages
            
            
            start = current_page * num_in_one_page #leaf thru the pages to the one we are on now.
            fin = start + num_in_one_page
            if fin > len ( filteredList ): fin = len ( filteredList ) #Make sure we don't overshoot.
            
            sublist = filteredList [ start:fin ] #Extract a single page of fonts to display
           
            self.choicePage.Clear ( ) #Empty the choice control, then refill it:
            [ self.choicePage.Append ( str ( n ) )  for n in range ( 1, self.total_number_of_pages +1 ) ] 
            self.choicePage.SetSelection ( self.pageindex-1 )
        else: #It's an empty.
            sublist = fpsys.state.viewobject #There should be only one item in an Empty type object.

        if self.total_number_of_pages == 1: 
            self.choicePage.Enable ( False ) #I tried to hide/show the choice, but it would not redraw properly.
        else:
            self.choicePage.Enable ( True )
            
        self.panelFontView.DrawFitmaps ( sublist )
        self.__buttonState  (  )
        
    ## Main BUTTON click -- the main "do something" button.
    def __onMainClick ( self, e ):
        xPos, yPos = self.panelFontView.GetViewStart ( ) #Saved by Robin Dunn, once again ! ! !
        wx.BeginBusyCursor ( )
        ps.pub ( main_button_click )
        wx.EndBusyCursor ( )
        self.panelFontView.Scroll ( xPos, yPos )
         
    def __onPagechoiceClick ( self,event ) :
        wx.BeginBusyCursor ( )
        if self.pageindex != int ( event.GetString (  )  ) : #Only redraw if actually onto another page.
            self.pageindex =  int ( event.GetString (  )  ) 
            self.__drawTheFonts (  ) 
        wx.EndBusyCursor ( )
    def __navClick ( self,event ) :
        wx.BeginBusyCursor ( )
        if event.GetId (  )  == wx.ID_FORWARD: 
            self.pageindex += 1
        else: #wx.ID_BACKWARD
            self.pageindex -= 1
        if self.pageindex > self.total_number_of_pages:
            self.pageindex = self.total_number_of_pages
        if self.pageindex == 0:
            self.pageindex = 1
         
        self.buttMain.SetFocus  (  )  #a GTK bug demands this move.
        self.__drawTheFonts  (  ) 
        wx.EndBusyCursor ( )
    def __buttonState ( self ) :
        #state of buttons
        n = True
        p = True
        if self.pageindex == self.total_number_of_pages: 
            n = False
        if self.pageindex == 1:
            p = False
        self.buttNext.Enable ( n )         
        self.buttPrev.Enable ( p )   
    def ToggleMainButton ( self, args = None):
        if fpsys.state.numticks > 0: self.buttMain.Enable ( True )
        else: self.buttMain.Enable ( False )
            
    def MainFontViewUpdate ( self ):
        """Vital routine - the heart if the app. 
        This decides what to do based on what has been selected.
        It draws the controls and the fonts as appropriate. It also sets flags in fpsys.state."""
        
        ## Get shorter vars to use.
        V= fpsys.state.viewobject
        T = fpsys.state.targetobject
            
        Vpatt = fpsys.state.viewpattern # View Pattern
        Tpatt = fpsys.state.targetpattern # Target pattern
    
        Patt = Vpatt + Tpatt # Patt = Pattern
        
        #print "Vpatt", Vpatt
        #print "Tpatt", Tpatt

        lab = ""
        status = ""
            
        if Vpatt == "E": #NOTE : TESTING VPATT, not PATT - ergo: this covers E, EN, EP
            ## Empty "E" - when the chosen Folder or Pog has NO FONTS IN IT.
            if Tpatt == "P":
                lab = "Active target:%s" % T.name
            else:
                status = "Please choose a Pog or a Font folder on the left"
            btext = "Close"
            fpsys.state.cantick = False
            fpsys.state.action = "CLOSE" # We will test this in mainframe::OnMainClick
            
        elif Patt == "FN":
            #View a Folder, no target
            lab = V.path
            fpsys.state.cantick = False
            btext = "Close"
            fpsys.state.action = "CLOSE" # We will test this in mainframe::OnMainClick
        elif Patt == "PN": #A single Pog in the VIEW
            #View a pog, no target
            if V.isInstalled ( ):
                ## Cannot remove fonts from an installed pog
                lab = "Viewing %s" % V.name
                btext = "Close"
                fpsys.state.action = "CLOSE"
                fpsys.state.cantick = False
                status = "You cannot change an installed Pog"
            else:
                lab = "Viewing %s" % V.name
                fpsys.state.cantick = True
                btext = "Remove fonts from %s" % V.name
                self.TICKMAP = self.CROSS
                fpsys.state.action = "REMOVE" # We will test this in mainframe::OnMainClick
        elif Patt == "FP":
            #Folder to Pog
            if T.isInstalled ( ):
                ## We cannot put stuff into an installed pog
                lab = "Viewing %s" % T.name
                btext = "Close"
                fpsys.state.action = "CLOSE"
                fpsys.state.cantick = False
                status = "You cannot change an installed Pog"
            else:
                lab = "%s to %s" % ( V.path, T.name )
                btext = "Put fonts into %s" % T.name
                self.TICKMAP = self.TICK
                fpsys.state.cantick = True
                fpsys.state.action = "APPEND" # We will test this in mainframe::OnMainClick
        elif Patt == "PP":
            #Pog to Pog
            if T.isInstalled ( ):
                ## We cannot put fonts into an installed pog
                lab = "%s is installed" % T.name
                btext = "Close"
                fpsys.state.action = "CLOSE"
                fpsys.state.cantick = False
                status = "You cannot change an installed Pog"
            else: #Not installed.
                if fpsys.state.samepogs: #Are the two pogs the same?
                    ## The validate routines determined the samepogs value.
                    lab = "Same pogs"
                    fpsys.state.cantick = True
                    btext = "Close"
                    fpsys.state.action = "CLOSE"
                    status = "Your pogs are the same"
                else: # Normal pog to pog
                    lab = "%s to %s" % ( V.name, T.name )
                    btext = "Put fonts into %s" % T.name
                    self.TICKMAP = self.TICK
                    fpsys.state.cantick = True     
                    fpsys.state.action = "APPEND" # We will test this in mainframe::OnMainClick
        else:
            sys.exit ( "%s and trouble" % Patt )
            
        self.buttMain.SetLabel ( btext )
        self.textMainInfo.SetLabel ( lab )
        self.textMainInfo.Show ( )
        if status is not "":
            ps.pub ( print_to_status_bar, status )
        self.ToggleMainButton ( )
        
        fpsys.markInactive ( )
        self.__drawTheFonts ( )

    def ResetToPageOne ( self ):
        self.pageindex = 1 # I start here
        
## Far right-hand side control. Chooses target pogs. Houses control buttons.
class TargetPogChooser ( wx.Panel ):
    """Standalone object to display pognames and allow user to choose one."""
    def __init__ ( self, parent ):
        wx.Panel.__init__ ( self, parent, id = -1 )    
        
        self.icon = wx.StaticBitmap ( self, -1, wx.Bitmap ( fpsys.mythingsdir + 'icon_target_16x16.png', wx.BITMAP_TYPE_PNG ) )
        self.textInfo = wx.StaticText  ( self, -1, "Target Pogs", style = wx.ALIGN_LEFT ) #CENTER_HORIZONTAL ) 
        s = None
        if fpsys.state.targetpattern == "P":
            s = fpsys.state.targetobject.name
        self.pogTargetlist = PogChooser ( self, select = s ) 
        
        ## Subscriptions:
        ps.sub ( pog_selected, self.OnPogTargetClick )
        ps.sub ( toggle_buttons, self.ToggleButtons )
        ps.sub ( select_no_target_pog, self.SelectNoTargetPog )
        
        ## The "no pog" button
        self.idnone = wx.NewId ( )
        self.buttNoPog = wx.Button ( self, label = "Select Nothing", id = self.idnone )
        self.buttNoPog.SetToolTipString ( "Deselects any chosen pogs." )
        ## The buttons under the pog list
        ## Giving them all id numbers so my single handler can tell them apart.
        self.idnew = wx.NewId ( )
        self.idinstall = wx.NewId ( )
        self.iduninstall = wx.NewId ( )
        self.iddelete = wx.NewId ( )
        self.idpurge = wx.NewId ( )
        
        self.buttNew = wx.Button  ( self, label = "New Pog", id = self.idnew  ) 
        self.buttInstall = wx.Button  ( self, label = "Install Pog", id = self.idinstall  ) 
        self.buttUninstall = wx.Button  ( self, label = "uninstall Pog", id = self.iduninstall  ) 
        self.buttDelete = wx.Button  ( self, label = "Delete Pog" , id = self.iddelete ) 
        self.buttPurge = wx.Button ( self, label = "Purge Pog", id = self.idpurge ) 
        
        self.sizer = wx.BoxSizer  ( wx.VERTICAL )         
        self.iconandtext = wx.BoxSizer ( wx.HORIZONTAL )
        self.iconandtext.Add  ( self.icon, 0, wx.TOP | wx.BOTTOM, border = 4 )
        self.iconandtext.Add  ( self.textInfo, 1, wx.EXPAND | wx.TOP | wx.BOTTOM | wx.LEFT, border = 4)
        self.sizer.Add  ( self.iconandtext, 0, wx.EXPAND )
        self.sizer.Add  ( self.buttNoPog, 0, wx.EXPAND )      
        self.sizer.Add  ( self.pogTargetlist, 1, wx.EXPAND )
        self.sizer.Add  ( self.buttInstall, 0, wx.EXPAND ) 
        self.sizer.Add  ( self.buttUninstall, 0, wx.EXPAND ) 
        self.sizer.Add  ( self.buttNew, 0, wx.EXPAND ) 
        self.sizer.Add  ( self.buttDelete, 0, wx.EXPAND ) 
        self.sizer.Add  ( self.buttPurge, 0, wx.EXPAND )
        self.SetSizer ( self.sizer )
        
        ## Bind the events:
        self.buttNoPog.Bind ( wx.EVT_LEFT_UP, self.__multiClick )
        self.buttNew.Bind ( wx.EVT_LEFT_UP, self.__multiClick )
        self.buttInstall.Bind ( wx.EVT_LEFT_UP, self.__multiClick )
        self.buttUninstall.Bind ( wx.EVT_LEFT_UP, self.__multiClick )
        self.buttDelete.Bind ( wx.EVT_LEFT_UP, self.__multiClick )
        self.buttPurge.Bind ( wx.EVT_LEFT_UP, self.__multiClick )
        
        self.__toggleButtons ( )
    ## Catch all the button clicks on the control.
    def __multiClick ( self, e ):
        ## NEW
        if e.GetId ( ) == self.idnew: 
            ## New Pog button pressed
            dlg = wx.TextEntryDialog(
                    self, 'Enter a name for the new pog',
                    'New Pog', 'FontyPython')
            dlg.SetValue ( "" )
            if dlg.ShowModal ( ) == wx.ID_OK:
                ## Is it unique?
                nam = dlg.GetValue ( )
                if fpsys.isPog ( nam ):
                    ## Nope, it's there already
                    ps.pub ( show_message, "%s already exists." % nam )
                else:
                    ## We have a winner.
                    ## Make a pog object and then write it,
                    ipog = Pog ( nam )
                    try:
                        ipog.write ( )
                    except fontybugs.PogWriteError, e:
                        ps.pub ( show_error_and_abort, str ( e ) )
                    del ipog
                    ## Now put it into the list
                    self.pogTargetlist.AddItem ( nam )
                    ps.pub ( add_item_to_notebook, nam )
                    ps.pub ( update_font_view )
            dlg.Destroy()
            return
            
        ## DELETE
        if e.GetId ( ) == self.iddelete:
            ## Selected Pog to be deleted
            pogname = fpsys.state.targetobject.name
            print "%s is to be deleted" % pogname
            dlg = wx.MessageDialog(self, "Remove %s, are you sure?" % pogname,
                                   'Are you sure?',
                                   wx.YES_NO | wx.ICON_INFORMATION
                                   )
            if dlg.ShowModal ( ) == wx.ID_YES:
                ## Now kill the file on disk:
                try:
                    fpsys.state.targetobject.delete ( )
                except fontybugs.PogCannotDelete, e:
                    ps.pub ( show_error, str ( e  ) )
                    return
                ## This object was also our target object (it was selected - duh!)
                ## So, we must now select no pog.
                self.__selectNoPog ( )
                ## Remove from the list:
                self.pogTargetlist.RemoveItem ( pogname )
                ps.pub ( remove_item_from_notebook, pogname )
                ## What if it was ALSO our view object?
                if fpsys.state.viewobject.label ( ) == pogname:
                    ## It was! We must declare it Empty.
                    fpsys.SetViewPogToEmpty ( )
                ## Now re-draw things
                ps.pub ( update_font_view )
            dlg.Destroy ( )
            return 
        
        ## NO POG pressed
        if e.GetId ( ) == self.idnone:
            ## Select No Pog button pressed
            if fpsys.state.targetobject is None: return #Already done.
            self.__selectNoPog ( )
            ps.pub ( update_font_view )
            return #No need to tell mainframe about this.
        ## PURGE
        if e.GetId ( ) == self.idpurge:
            if not fpsys.state.targetobject.isInstalled ( ):
                pogname = fpsys.state.targetobject.name
                dlg = wx.MessageDialog(self, """Do you want to purge %s?
                Purging means all the fonts in the pog that
                are not pointing to actual TTF files will be 
                removed from this pog.""" % pogname,
                                       'Purge font?',
                                       wx.YES_NO | wx.ICON_INFORMATION
                                       )
                if dlg.ShowModal ( ) == wx.ID_YES:
                    ## pog.purge ( ) Raises
                    ##          PogEmpty
                    ##          PogInstalled
                    try:
                        fpsys.state.targetobject.purge ( )
                    except ( fontybugs.PogEmpty, fontybugs.PogInstalled ),e:
                        ps.pub ( show_error, str ( e  ) )
                        ps.pub ( print_to_status_bar, "%s has not been purged." % pogname )
                        return 
                ## Update GUI
                ps.pub ( print_to_status_bar, "%s has been purged." % pogname )
                ps.pub ( update_font_view )
                
                
        ## The next two get passed onto mainframe.
        if e.GetId ( ) == self.idinstall:
            ## Install button
            ps.pub ( install_pog )
        if e.GetId ( ) == self.iduninstall:
            ps.pub ( uninstall_pog )
        
    def OnPogTargetClick ( self, args ):
        ## id of the originating instance is sent in args [1]
        ## We must be sure it came from pogTargetlist ...
        if args [ 1 ] != self.pogTargetlist.GetId ( ): return
        ## Made it so a second click on a target pog will unselect it.
        if args [ 2 ]: #pognochange = True, so let's deselect this pog
            self.__selectNoPog ( )
            ps.pub ( update_font_view )
            return
        try:
            fpsys.validateTargetPog ( args [ 0 ] )
        except fontybugs.PogInvalid, e:
            ps.pub ( show_error_and_abort, str ( e ) )
            return
        if fpsys.state.samepogs: #forbid same pogs selection
           # ps.pub ( print_to_status_bar, "Pog already selected in the View" )
            ps.pub ( select_no_view_pog )
        ps.pub ( update_font_view )
        self.__toggleButtons ( )
        
    def ToggleButtons ( self, args ):
        ## Shadow the __toggleButtons func. 
        self.__toggleButtons ( )
    def __toggleButtons ( self ):
        ## If this is a no target pog situation, hide 'em all.
        if fpsys.state.targetobject is None:
            self.buttDelete.Enable ( False )
            self.buttInstall.Enable ( False )
            self.buttUninstall.Enable ( False )
            self.buttPurge.Enable ( False )
            return 
        installed = fpsys.state.targetobject.isInstalled ( )
        self.buttDelete.Enable ( not ( installed ) ) # DELETE button is inverse of installed status
        self.buttInstall.Enable ( not ( installed ) ) # INSTALL button is inverse 
        self.buttUninstall.Enable ( installed ) # UNINSTALL = True if pog is installed.
        self.buttPurge.Enable ( not ( installed ) )
        
    def SelectNoTargetPog ( self, args ):
        ## Shadow
        self.__selectNoPog ( )
    def __selectNoPog ( self ):
        """Public method : for access from main"""
        wx.BeginBusyCursor ( )
        self.pogTargetlist.ClearSelection ( ) #Select nothing.
        fpsys.SetTargetPogToNone ( )  # Tell fpsys that we have no pog target selected        
        self.__toggleButtons ( ) # Disable the buttons on the bottom right.        
        wx.EndBusyCursor ( )
## Basic list control for pogs - instanced by TargetPogChooser and NoteBook
class PogChooser ( wx.ListCtrl ) :
    """See TargetPogChooser"""
    def __init__ ( self, parent, select = None ) :
        self.indexselected = 0
        self.lastindexselected = -1
        
        il = wx.ImageList ( 16,16,True ) 
        png = wx.Bitmap ( fpsys.mythingsdir + "/pog16x16.png",wx.BITMAP_TYPE_PNG ) 
        il.Add ( png ) 
        png = wx.Bitmap ( fpsys.mythingsdir + "/pog16x16.installed.png",wx.BITMAP_TYPE_PNG ) 
        il.Add ( png )
        
        wx.ListCtrl.__init__ ( self,parent,-1, style=wx.LC_LIST | wx.LC_AUTOARRANGE | wx.LC_SORT_ASCENDING ) 
        self.AssignImageList  ( il, wx.IMAGE_LIST_SMALL ) 

        self.__PC = pathcontrol.PathControl ( )
        
        self.__fillPogList  (  ) 
        
        if select:
            i = self.FindItem ( -1, select )
            self.Select ( i, True)
        else:
            self.Select ( 0, False )

        self.ClearBackground  (  ) 
        self.items = None
        
        ## These two work together:
        self.Bind  ( wx.EVT_LIST_ITEM_SELECTED, self.__onSelect ) 
        
        self.SetCursor ( wx.StockCursor ( wx.CURSOR_HAND ) )
        
        ## A subscribe line here, will register TWICE since this PogChooser object is instanced
        ## twice by the app...
        ps.sub ( change_pog_icon, self.ChangeIcon )
        
    def __onSelect ( self, e ):
        wx.BeginBusyCursor ( )
        self.indexselected = e.m_itemIndex
        pognochange = False
        if self.indexselected == self.lastindexselected:
            ## We have dclicked on the same Pog as last time - ergo do nothing.
            pognochange = True
            #wx.EndBusyCursor ( )
            #return
        self.lastindexselected = self.indexselected # Record this for next time around
        ##Raise a SELECT
        t = self.GetItemText ( self.indexselected )
        ps.pub ( pog_selected, t, self.GetId ( ), pognochange )
        wx.EndBusyCursor ( )

    def __del__ ( self ) :
        del self.items
        
    ## This is among the very FIRST things we do.
    ## Fill the list with pogs.
    ## This will CULL any bad pogs (i.e. those with malformed files)
    ## Thus the PogInvalid error should not happen any more after a run.
    def __fillPogList  ( self ) :
        pl = self.__PC.getPogNames ( )
        pl.sort ( reverse = True ) # Half-assed attempt at sorting the item lists.
        for p in pl :
            ipog = Pog ( p )
            try: #catch pogs that are not properly formed
                if ipog.isInstalled ( ): i = 1
                else: i = 0
                #self.InsertImageStringItem ( 0, p ,0 ) 
                li = wx.ListItem  (  ) 
                li.SetImage  ( i ) 
                li.SetText  ( p ) 
                self.InsertItem  ( li ) 
            except fontybugs.PogInvalid, eInst:
                print "\"%s\" skipped" % p
            except:
                sys.exit ( sys.exc_info ( ) [ 0 ] )
                
    def AddItem ( self, pogname ):
        li = wx.ListItem ( )
        li.SetImage ( 0 )
        li.SetText ( pogname )
        self.InsertItem ( li )
        
    def RemoveItem ( self, pogname ):
        i = self.FindItem ( -1, pogname )
        self.DeleteItem ( i )

    def ClearSelection ( self ):
        self.Select ( self.indexselected, False )
        self.lastindexselected = -1
    
    def ChangeIcon ( self, args ):
        T = fpsys.state.targetobject
        pn = T.name
        index = self.FindItem ( 0, pn ) 
        if T.isInstalled ( ): n = 1
        else: n = 0
        self.SetItemImage ( index, n ) ## Found in wxWidgets documentation!
        
    
    ## Need to reset the lastindexselected so that a click on the 
    ## same item as last time will register. This is used in the Notebook
    ## when the page changes back to page 1, we want the user to
    ## be able to re-click the item that was clicked last time.
    def ClearLastIndex ( self ):
        self.lastindexselected = -1

## Used in mainframe - part of the left, middle, right split
class Splitter ( wx.SplitterWindow ) :
    def __init__ ( self, parent ) :
        wx.SplitterWindow.__init__ ( self, parent, -1, style = wx.SP_LIVE_UPDATE | wx.SP_3D ) 


## A short class as I started to abstract the Directory Control. 
## Eventually I want a fully standalone version.
class DirControl ( wx.GenericDirCtrl ) :
    def __init__ ( self, parent ):
        if fpsys.state.viewpattern == "F": startdir = fpsys.state.viewobject.path
        else: 
            ##Let's get it from the config object
            lastdir = fpsys.config.lastdir
            if os.path.exists ( lastdir ):
                startdir = lastdir
            else:
                startdir = os.environ [ 'HOME' ]
        wx.GenericDirCtrl.__init__ ( self, parent, -1, dir = startdir, style=wx.DIRCTRL_DIR_ONLY )  #size= ( 200,225 )
        self.tree = self.GetTreeCtrl (  ) 
        self.tree.SetCursor ( wx.StockCursor ( wx.CURSOR_HAND ) )

## Used in the left part of the splitter in mainframe.
## Has two tabs - Folders and Pogs
## THIS IS THE VIEW or SOURCE of fonts.
class NoteBook ( wx.Notebook ) :
    def __init__ ( self, parent ):
        wx.Notebook.__init__ ( self, parent )
        self.imlist = wx.ImageList ( 16, 16 )
        
        pan1 = wx.Panel ( self )
        self.dircontrol = DirControl ( pan1 ) 
        
        box = wx.BoxSizer ( wx.HORIZONTAL ) 
        box.Add ( self.dircontrol,1, wx.EXPAND ) 
        pan1.SetSizer ( box ) 
        box.Layout (  ) 

        self.pogindexselected = 0
        
        pan2 = wx.Panel ( self ) 
        
        page = 0
        s = None
        if fpsys.state.viewpattern  == "P": 
            s = fpsys.state.viewobject.name
            if s == "EMPTY": s= None #Very first run, the view will be an EMPTY object.
            page = 1
        self.listctrl = PogChooser ( pan2, select = s )

        ps.sub ( pog_selected, self.OnViewPogClick )
        ps.sub ( select_no_view_pog, self.SelectNoView )
        
        self.tree = self.dircontrol.GetTreeCtrl ( )
        #self.tree.Bind ( wx.EVT_LEFT_DCLICK, self.__onDirCtrlDClick ) #Old system - double click.
        #self.tree.Bind ( wx.EVT_LIST_ITEM_SELECTED, self.__onDirCtrlDClick ) #Did not fire.
        self.tree.Bind ( wx.EVT_LEFT_UP, self.__onDirCtrlClick )
        
        box2 = wx.BoxSizer ( wx.HORIZONTAL ) 
        box2.Add ( self.listctrl,1,wx.EXPAND ) 
        pan2.SetSizer ( box2 ) 
        box2.Layout (  ) 
        
        self.AddPage ( pan1, "Folders" ) 
        self.AddPage ( pan2, "Pogs" ) 
        
        source_pog_icon = self.imlist.Add ( wx.Bitmap ( fpsys.mythingsdir + "/icon_source_pog_16x16.png",wx.BITMAP_TYPE_PNG ) )

        target_pog_icon = self.imlist.Add ( wx.Bitmap ( fpsys.mythingsdir + "/icon_source_folder_16x16.png",wx.BITMAP_TYPE_PNG ) )
        self.AssignImageList ( self.imlist )
        self.SetPageImage ( 1, source_pog_icon )
        self.SetPageImage ( 0, target_pog_icon )
        
        self.SetSelection ( page )
    
        self.Bind ( wx.EVT_NOTEBOOK_PAGE_CHANGED, self.__onPageChanged ) # Bind page changed event
        
    def __onPageChanged ( self, e ):
        self.listctrl.ClearLastIndex  ( )
        e.Skip ( )
        
    def __onDirCtrlClick ( self, e ):
        wx.BeginBusyCursor ( ) #Thanks to Suzuki Alex on the wxpython list!
        p = self.dircontrol.GetPath ( )
        try:
            fpsys.validateViewFolder ( p )
            fpsys.config.lastdir = p
        except fontybugs.FolderHasNoFonts, e:
            pass # update_font_view handles this with a std message.
        ps.pub ( update_font_view )
        wx.EndBusyCursor ( )
        
    def OnViewPogClick ( self, args ):
        ## id of the originating instance is sent in args[1]
        ## We must be sure it came from listctrl ...
        if args [ 1 ] != self.listctrl.GetId ( ): return
        ## Check pognochange, it means this is the same pog as last time.
        if args [ 2 ]: return 
        try:
            fpsys.validateViewPog ( args [ 0 ] )
        except fontybugs.PogInvalid,e :
            ps.pub ( show_error_and_abort, str ( e ) )
            return
        if fpsys.state.samepogs: #forbid same pogs selection
            #ps.pub ( print_to_status_bar, "Pog already selected in the View" )
            ps.pub ( select_no_target_pog )
        else:
            ps.pub ( reset_to_page_one )
        ps.pub ( update_font_view )
    
    ## Public Interface:
    def AddItem ( self, pogname ):
        self.listctrl.AddItem ( pogname )
    def RemoveItem ( self, pogname ):
        self.listctrl.RemoveItem ( pogname )
        
    def SelectNoView ( self, args ):
        ## Purpose: To select no viewobject and clear view pog list selections
        ## Called when a TARGET item is clicked AND samepogs it True
        wx.BeginBusyCursor ( )
        self.listctrl.ClearSelection ( )
        fpsys.SetViewPogToEmpty ( )
        wx.EndBusyCursor ( )
        
## Status bar
class StatusBar ( wx.StatusBar ):
    def __init__(self, parent ):
        wx.StatusBar.__init__(self, parent, -1)
        self.SetFieldsCount ( 1 )
        self.SetStatusText ( "Welcome to Fonty Python %s" % fpversion.version, 0 )
    def Report ( self, msg ):
        self.SetStatusText ( msg, 0 )
        
## The main brain of the app.
class MainFrame ( wx.Frame ) :
    def __init__ ( self,parent,title ) :
        ## Draw the frame
        wx.Frame.__init__ ( self,parent,-1,title,fpsys.config.pos,fpsys.config.size ) 
        self.SetSizeHintsSz (  ( 300,400 )  )  #After hours of hell, this is all that seems to work.
        
        ## Try to show an icon
        try:
            image = wx.Image ( fpsys.mythingsdir + 'fplogo.png', wx.BITMAP_TYPE_PNG ) 
            image = image.ConvertToBitmap (  ) 
            icon = wx.EmptyIcon (  ) 
            icon.CopyFromBitmap ( image ) 
            self.SetIcon ( icon ) 
        except:
            pass

        max = fpsys.config.max # Do we draw min or max interface?
        
        PARENT = self
        
        ## STATUS BAR
        self.sb = StatusBar ( self )
        self.SetStatusBar ( self.sb )
        
        if max:
            ## Prepare the menu bar
            menuBar = wx.MenuBar ( )
    
            ## 1st menu from left
            menu1 = wx.Menu ( )
            menu1.Append ( 101, "&Settings\tCtrl+S", "This the text in the Statusbar" )
            menu1.AppendSeparator ( )
            exit = menu1.Append ( 102, "&Close", "Close this frame" )
            ## Add menu to the menu bar
            menuBar.Append ( menu1, "&File" )
    
            ## 2nd menu from left
            menu2 = wx.Menu ( ) 
            menu2.Append ( 201, "H&elp\tF1" )
            menu2.Append ( 202, "&About" )
            ## Append 2nd menu
            menuBar.Append ( menu2, "&Help" )
    
            ## Tell the frame the news
            self.SetMenuBar ( menuBar )
    
            ## Setup the ESC key trap
            accel = wx.AcceleratorTable ( [ ( wx.ACCEL_NORMAL, wx.WXK_ESCAPE, exit.GetId ( ) ) ] )
            self.SetAcceleratorTable ( accel )
            
            self.Bind ( wx.EVT_MENU, self.__onHandleESC, exit )
            self.Bind ( wx.EVT_MENU, self.__menuSettings, id = 101 )
            self.Bind ( wx.EVT_MENU, self.__menuAbout, id = 202 )
            self.Bind ( wx.EVT_MENU, self.__menuHelp, id = 201 )
        else:
            ## For min : special ESC key setup code :: thanks Robin Dunn.
            ## Not 100% effective. It does not fire when focus is on FontViewPanel ... :(
            ID_ESC = 1001
            self.accel = wx.AcceleratorTable ( [ ( wx.ACCEL_NORMAL, wx.WXK_ESCAPE, ID_ESC ) ] )
            self.SetAcceleratorTable ( self.accel )
            self.Bind ( wx.EVT_MENU, self.__onHandleESC, id=ID_ESC )
            
        ## Create a splitter 
        if max: 
            self.splitter = Splitter  ( self ) 
        
            ## The notebook : has a bunch of callbacks it needs to make.
            self.panelNotebook = wx.Panel  ( self.splitter ) 
            self.nb = NoteBook  ( self.panelNotebook )
            
            self.sizerNotebook = wx.BoxSizer ( wx.HORIZONTAL ) 
            self.sizerNotebook.Add ( self.nb,1,wx.EXPAND ) 
            
            self.panelNotebook.SetSizer  ( self.sizerNotebook ) 
            self.sizerNotebook.Layout  (  ) 
            
            PARENT = self.splitter
        
        self.panelMain = wx.Panel ( PARENT  )
        
        ## Font View Panel Control:
        self.panelFontView = FontViewPanel ( self.panelMain )
        
        self.sizerFontView  = wx.BoxSizer ( wx.VERTICAL ) 
        self.sizerFontView.Add ( self.panelFontView, 1, wx.EXPAND )
 
        if max:
            ## THE FAR RIGHT HAND SIDE
            ## The TargetPogChooser
            self.panelTargetPogChooser = TargetPogChooser ( self.panelMain )

            self.sizerSpace = wx.BoxSizer ( wx.VERTICAL )
            self.sizerRight = wx.BoxSizer ( wx.HORIZONTAL ) 
            self.sizerSpace.Add ( self.panelTargetPogChooser, 1, wx.EXPAND | wx.LEFT, border = 3 ) ## couched the panelTargetPogChooser in this to allow space around edges
            
            self.sizerRight.Add ( self.sizerFontView, 1, wx.EXPAND ) 
            self.sizerRight.Add ( self.sizerSpace, 0, wx.EXPAND ) 
            
            self.panelMain.SetSizer ( self.sizerRight ) 
        else:
            self.panelMain.SetSizer ( self.sizerFontView )
            
        self.panelMain.Layout  (  ) 

        if max:
            self.splitter.SetMinimumPaneSize  ( 64 ) 
            self.splitter.SplitVertically  ( self.panelNotebook, self.panelMain ,180 ) 
    
    
        self.SetCursor ( wx.StockCursor ( wx.CURSOR_ARROW ) )
        
        ## Now to subscribe to have my various def called from other places:
        ps.sub ( update_font_view, self.UpdateFontViewPanel )
        ps.sub ( show_error, self.ErrorBox )
        ps.sub ( show_error_and_abort, self.ErrorAbort )
        ps.sub ( show_message, self.MessageBox )
        ps.sub ( reset_to_page_one, self.ResetToPageOne )
        ps.sub ( add_item_to_notebook, self.NotebookAddItem )
        ps.sub ( remove_item_from_notebook, self.NotebookRemoveItem )
        ps.sub ( print_to_status_bar, self.StatusbarPrint )
        ps.sub ( install_pog, self.InstallPog )
        ps.sub ( uninstall_pog, self.UninstallPog )
        ps.sub ( main_button_click, self.OnMainClick )

        ## call the big one - the big chief, the big cheese:
        self.UpdateFontViewPanel ( ) #Go and fill in the font view and surrounding controls
        
    ## PUBLIC API:
    def StatusbarPrint ( self, args ):
        self.sb.Report ( args [ 0 ] )
    def NotebookAddItem ( self,args ):
        self.nb.AddItem ( args [ 0 ] )
    def NotebookRemoveItem ( self, args ):
        self.nb.RemoveItem ( args [ 0 ] )
    def ResetToPageOne ( self, args ):
        """Calls ResetToPageOne in the FontViewPanel"""
        self.panelFontView.ResetToPageOne ( )
        
    def MessageBox ( self, args ):
        dlg = wx.MessageDialog ( self, args [ 0 ] , "Warning", wx.OK | wx.ICON_INFORMATION )
        dlg.ShowModal ( )
        dlg.Destroy ( )
    def ErrorBox ( self, args ):
        dlg = wx.MessageDialog ( self, args [ 0 ], "Error", wx.OK | wx.ICON_ERROR )
        dlg.ShowModal ( )
        dlg.Destroy ( )
    def ErrorAbort ( self, args ):
        self.ErrorBox ( args ) #Pass it along to be displayed
        self.__endApp ( )
        
    def UpdateFontViewPanel ( self, args = None ):
        self.panelFontView.MainFontViewUpdate (  )
        self.Refresh ( )
    
    def InstallPog ( self, args ):
        ## pog.install ( ) Raises:
        ##          PogEmpty
        ##          PogAllFontsFailedToInstall
        ##          PogSomeFontsDidNotInstall
        try:
            fpsys.state.targetobject.install ( )
        except ( fontybugs.PogSomeFontsDidNotInstall), e:
            ## Show a warning, but continue.
            self.ErrorBox ( [ str ( e ), ] ) #send it as a list
        except ( fontybugs.PogEmpty, fontybugs.PogAllFontsFailedToInstall ), e:
            ## Either Pog is empty, or
            ## not a single font in this pog actually installed.
            ## It has already been flagged as NOT INSTALLED
            self.ErrorBox ( [ str ( e ), ] )
            return # Make no changes to the display or icon - yet... badpog icon ?
        ## Update GUI
        self.sb.Report ( "%s has been installed." % fpsys.state.targetobject.name )
        ps.pub ( change_pog_icon )
        ps.pub ( toggle_buttons ) #Update the greying of the buttons
        self.UpdateFontViewPanel ( )

    def UninstallPog ( self, args ):
        ## pog.uninstall ( ) Raises:
        ##          PogEmpty
        ##          PogLinksRemain
        ##          PogNotInstalled        
        try:
            fpsys.state.targetobject.uninstall ( )
        except (  fontybugs.PogEmpty, 
                        fontybugs.PogNotInstalled, #This one is prevented by the buttons greying out in the gui
                        fontybugs.PogLinksRemain
                      ), e:
            self.ErrorBox ( [ str ( e ), ] )
            return
        ## Update GUI
        self.sb.Report ( "%s has been uninstalled." % fpsys.state.targetobject.name )
        ps.pub ( change_pog_icon )
        ps.pub ( toggle_buttons ) #Update the greying of the buttons
        self.UpdateFontViewPanel ( )

            
    def OnMainClick ( self, args ) :
        """Closes app, or Removes fonts, or appends fonts. Depends on situation in fpsys.state"""
        ## Let's determine what kind of thing to do:
        if fpsys.state.action == "CLOSE":
            ## Exit the app
            print "Done. No changes have been made."
            self.__endApp (  ) 
        
        if fpsys.state.action == "REMOVE":
            ## We have a pog in viewobject and we must remove the selected fonts from it.
            vo = fpsys.state.viewobject
            victims = [ ]
            dowrite = False
            for fi in vo:
                if fi.ticked:
                    victims.append ( fi ) #Put it into another list
                    dowrite = True
            for fi in victims:
                vo.remove ( fi ) #Now remove it from the vo
            del victims
            
            if dowrite:
                fpsys.flushTicks ( )
                bug = False
                try:
                    vo.write ( )      
                except ( fontybugs.PogWriteError ), e:
                    bug = True
                    self.errorBox ( [ str ( e ), ] )
                ## Now, let's redraw the vo
                self.UpdateFontViewPanel ( )
                if not bug:
                    self.sb.Report ( "Selected fonts have been removed." )
                else:
                    self.sb.Report ( "There was an error writing the pog to disk. Nothing has been done" )
        
        ## APPEND - Copy ttf to a pog.
        if fpsys.state.action == "APPEND":
            ## We must append the fonts to the Pog
            vo = fpsys.state.viewobject
            to = fpsys.state.targetobject
            print "Copying fonts from %s to %s" %  ( vo.label ( ), to.label ( ) ) 
            dowrite = False
            for fi in vo:
                if fi.ticked:
                    to.append ( fi ) 
                    dowrite = True
            if dowrite: 
                fpsys.flushTicks ( ) #Ensure we have no more ticks after a succ xfer.
                bug = False
                try:
                    to.write ( )      
                except ( fontybugs.PogWriteError ), e:
                    bug = True
                    self.errorBox ( [ str ( e ), ] )
                self.UpdateFontViewPanel ( )
                if not bug:
                    self.sb.Report ( "Selected fonts are now in %s." % to.label ( ) )           
                else:
                    self.sb.Report ( "There was an error writing the pog to disk. Nothing has been done" )
                    
                    
    def __onHandleESC ( self, e ) :
        print strings.done
        self.__endApp (  ) 

    def __endApp ( self ) :
        fpsys.config.pos = self.GetPositionTuple (  ) 
        fpsys.config.size = self.GetClientSizeTuple (  )
        
        self.Destroy (  ) 
   
    def __menuSettings ( self, e ):
        dlg = dialogues.DialogSettings ( self )
        val = dlg.ShowModal ( )
        if val == wx.ID_OK:
            fpsys.config.numinpage = int ( dlg.inputPageLen.GetValue ( ) )
            fpsys.config.points = int ( dlg.inputPointSize.GetValue ( ) )
            txt = dlg.inputSampleString.GetValue ( )
            if len ( txt ) > 0: fpsys.config.text =  txt
            ## Now to refresh things:
            self.UpdateFontViewPanel ( )
        dlg.Destroy ( )

    def __menuAbout ( self, e ):
        dlg =dialogues.DialogAbout ( self )
        val = dlg.ShowModal ( )
        dlg.Destroy ( )
        
    def __menuHelp ( self, e ):
        dlg = dialogues.DialogHelp ( self, size=(800, 600) )
        val = dlg.ShowModal ( )
        dlg.Destroy ( )

        
## Start the main frame and then show it.
class App ( wx.App ) :
    def OnInit ( self ) :
        frame = MainFrame ( None, "Fonty Python: bring out your fonts!" ) 
        #frame = TempFrame ( None, "Fonty Python: bring out your fonts!" ) 
        self.SetTopWindow ( frame ) 
        frame.Show ( True ) 
        return True
        
## app
app = App ( 0 ) 

## start 
app.MainLoop ( ) 
