/**************************************************************************

   fotoxx      digital photo edit program

   Copyright 2007, 2008, 2009, 2010  Michael Cornelison
   source URL:  kornelix.squarespace.com
   contact: kornelix@yahoo.de
   
   This program is free software: you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program. If not, see http://www.gnu.org/licenses/.

***************************************************************************/

#include <tiffio.h>
#include "zfuncs.h"

#define fversion "fotoxx v.10.3.1  2010.05.17"
#define flicense "Free software - GNU General Public License v.3"
#define fhomepage "http://kornelix.squarespace.com/fotoxx"
#define ftranslators "Translators:"                                        \
   "\n Stanislas Zeller, Miguel Bouzada, Jie Luo, Eugenio Baldi,"          \
   "\n Justa, Peter Landgren, Alexander Krasnopolsky"
#define fcredits "Programs used: libtiff, ufraw, exiftool"
#define fcontact "Bug reports: kornelix@yahoo.de"

//  GTK definitions

#define PXB GdkPixbuf
#define textwin GTK_TEXT_WINDOW_TEXT                                       //  GDK window of GTK text view
#define nodither GDK_RGB_DITHER_NONE
#define ALWAYS GTK_POLICY_ALWAYS
#define NEVER GTK_POLICY_NEVER
#define colorspace GDK_COLORSPACE_RGB
#define lineattributes GDK_LINE_SOLID, GDK_CAP_BUTT, GDK_JOIN_MITER

//  EXIF keys for embedded image data

#define exif_tags_key_old "UserComment"                                    //  tags                obsolete    v.10.0
#define exif_tags_key "Keywords"                                           //  tags                new         v.10.0
#define exif_rating_key "Rating"                                           //  "star" rating       new         v.10.0
#define exif_date_key "DateTimeOriginal"                                   //  date/time
#define exif_width_key "ImageWidth"                                        //  width, pixels
#define exif_height_key "ImageHeight"                                      //  height, pixels
#define exif_orientation_key "Orientation"                                 //  orientation
#define exif_log_key "EditStatus"                                          //  edit log            new         v.10.2
#define exif_user_comment_key "UserComment"                                //  user comments       new         v.10.2
#define exif_focal_length_key "FocalLengthIn35mmFormat"                    //  focal length        new         v.10.3

//  fotoxx definitions

#define mwinww 800                                                         //  main window default size
#define mwinhh 600
#define mega (1024 * 1024)
#define track(name) printf("track: %s \n",#name);                          //  debug path tracker
#define ftrash "Desktop/fotoxx-trash"                                      //  trash folder location
#define maxedits 100                                                       //  max. edits, undo/redo depth     v.10.2
#define def_jpeg_quality "90"                                              //  default jpeg quality (high)
#define max_threads 4                                                      //  max. working threads

#define max_images 100000                                                  //  max. images (m_tags_index)
#define maxtag1 50                                                         //  max tag cc for one tag          v.9.7
#define maxtag2 1000                                                       //  max tag cc for one image file
#define maxtag3 50000                                                      //  max tag cc for all image files
#define maxtag4 200                                                        //  max tag cc for search tags
#define maxtag5 200                                                        //  max tag cc for recent tags
#define maxntags 2000                                                      //  max tag count for all images
#define maxtagF 500                                                        //  max image search /path*/file*

//  must be > maxfcc (filespecs) and maxtag2                               //  v.10.0
#define tags_index_bcc maxtag2+20                                          //  tags index file buffer cc

#define bmpixel(rgb,px,py) ((uint16 *) rgb->bmp+((py)*(rgb->ww)+(px))*3)   //  return RGB pixel[3] at (px,py)
#define brightness(pix) (0.25*(pix)[0]+0.65*(pix)[1]+0.10*(pix)[2])        //  pixel brightness, 0-64K
#define redness(pix) (25 * pix[0] / (brightness(pix)+1))                   //  pixel redness, 0-100%

#define pixed_undomaxmem (100 * mega)                                      //  pixel edit max. memory alloc.
#define pixed_undomaxpix (mega)                                            //  pixel edit max. pixel blocks

namespace  image_navi {                                                    //  zfuncs: image_gallery() etc.
   extern int     xwinW, xwinH;                                            //  image gallery window size
   extern int     thumbsize;                                               //  thumbnail image size
}

namespace  zfuncs {
   extern char    zlanguage[8];                                            //  current language lc_RC  v.8.5
   extern cchar   *F1_help_topic;                                          //  F1 context help topic
   extern char    zicondir[200];
}

GtkWidget      *mWin, *drWin, *mVbox;                                      //  main and drawing window
GtkWidget      *mMbar, *mTbar, *STbar;                                     //  menu bar, tool bar, status bar
GdkGC          *gdkgc = 0;                                                 //  GDK graphics context
GdkColor       black, white, red, green;
GdkColormap    *colormap = 0;
uint           maxcolor = 0xffff;
GdkCursor      *arrowcursor = 0;
GdkCursor      *dragcursor = 0;
GdkCursor      *drawcursor = 0;
GdkCursor      *busycursor = 0;
GdkCursor      *currentcursor = 0;

char        command[maxfcc*2+200];                                         //  command, parameters, 2 filespecs   v.10.0
int         ccc = maxfcc*2 + 200;                                          //  command line size

double      pi = 3.141592654;
char        PIDstring[12];                                                 //  process PID as string
pthread_t   tid_fmain = 0;                                                 //  thread ID for main()
int         Nwt = 0;                                                       //  working threads to use
int         max_wt = max_threads;                                          //  max. worker threads to use
int         wtnx[4] = { 0, 1, 2, 3 };                                      //  internal thread IDs

int         Fexiftool = 0;                                                 //  exiftool program available
int         Fexifwarned = 0;                                               //  missing exiftool was warned
int         Fufraw = 0;                                                    //  ufraw program is available
int         Fxdgopen = 0;                                                  //  xdg-open program available
int         Fshutdown = 0;                                                 //  app shutdown underway
int         Fdebug = 0;                                                    //  debug flag
int         Fkillfunc = 0;                                                 //  signal, kill running function      v.9.1
int         Wrepaint = 0;                                                  //  request window paint
int         Wpainted = 0;                                                  //  window was repainted
int         Fmodified = 0;                                                 //  image was edited/modified
int         Fsaved = 0;                                                    //  image saved since last mod
int         Fpreview = 0;                                                  //  use window image for edits
int         Fblowup = 0;                                                   //  zoom small images to window size
int         Fshowarea = 0;                                                 //  show select area outline
char        FareaRGB = 'G';                                                //  select area outlint color
int         Fgrid = 0;                                                     //  gridlines on / off 
int         gridx1 = 0, gridx2 = 0, gridnx = 0;                            //  X gridlines, start, end, count
int         gridy1 = 0, gridy2 = 0, gridny = 0;                            //  Y gridlines
int         Fautolens = 0;                                                 //  lens parameter search underway
int         Fsearchlist = 0;                                               //  file list via search tags
int         Fimageturned = 0;                                              //  image was turned when loaded
int         Fslideshow = 0;                                                //  slide show mode is active
int         SS_interval = 3;                                               //  slide show interval
int         SS_timer = 0;                                                  //  slide show timer
int         SBupdate = 0;                                                  //  request to update status bar
int         progress_goal = 0, progress_done = 0;                          //  status bar progress tracking
int         wthreads_busy = 0;                                             //  active thread count

cchar       *zhelpcontext = 0;                                             //  F1 help context
char        *image_file = 0;                                               //  current image file
double      file_MB;                                                       //  disk file size, MB
int         file_bpc;                                                      //  disk file bits/color 8/16
char        file_type[8];                                                  //  "jpeg" "tiff" "png" "other"
char        *recentfiles[40];                                              //  40 most recent image files
int         Nrecentfiles = 40;                                             //  recent file list size
char        *tags_index_file = 0;                                          //  tags index file (new)        v.10.0
char        *asstags_file = 0;                                             //  tags index file (old)
char        *select_area_dirk = 0;                                         //  directory for saved select area files
char        *topdirk = 0;                                                  //  top-level image directory
char        *topmenu;                                                      //  latest top-menu selection
char        jpeg_quality[8] = def_jpeg_quality;                            //  jpeg file save quality (1/compression)

//  fotoxx RGB pixmaps                                                     //  v.6.5

typedef struct  {                                                          //  RGB pixmap
   char     wmi[8];
   int      ww, hh, bpc;                                                   //  width, height, bits per color
   void     *bmp;                                                          //  uint8*/uint16* (bpc=8/16)
}  RGB;

RGB         *Frgb8 = 0;                                                    //  input file 1 pixmap, RGB-8
RGB         *Frgb16 = 0;                                                   //  input file 1 pixmap, RGB-16
RGB         *Grgb16 = 0;                                                   //  input file 2 pixmap, RGB-16
RGB         *E1rgb16 = 0;                                                  //  edit pixmap, base image
RGB         *E3rgb16 = 0;                                                  //  edit pixmap, edited image
RGB         *ERrgb16 = 0;                                                  //  edit pixmap, undo/redo image
RGB         *E9rgb16 = 0;                                                  //  scratch image for some functions
RGB         *Drgb8 = 0;                                                    //  drawing window pixmap
RGB         *A1rgb16 = 0, *A2rgb16 = 0;                                    //  align image pixmaps (HDR, pano)

int         Fww, Fhh, Gww, Ghh;                                            //  input image dimensions
int         E1ww, E1hh, E3ww, E3hh;                                        //  edit image dimensions
int         ERww, ERhh;                                                    //  undo/redo image dimensions
int         E9ww, E9hh;                                                    //  scratch image dimensions
int         Dww = mwinww, Dhh = mwinhh;                                    //  drawing window size (defaults)
int         A1ww, A1hh, A2ww, A2hh;                                        //  alignment image dimensions

double      Fzoom = 0;                                                     //  image zoom scale (0 = fit window)
int         zoomx = 0, zoomy = 0;                                          //  req. zoom center of window
double      Mscale = 1;                                                    //  scale factor, image to window
int         Iww, Ihh;                                                      //  current image size at 1x
int         iww, ihh;                                                      //  area in drawing window at Mscale
int         dww, dhh;                                                      //  drawing window image, iww * Mscale
int         Iorgx, Iorgy;                                                  //  drawing window origin in image
int         Dorgx, Dorgy;                                                  //  image origin in drawing window

mutex       pixmaps_lock;                                                  //  lock for accessing RGB pixmaps
int         menu_lock = 0;                                                 //  lock for some menu functions

cchar       *edit_function = 0;                                            //  current edit function
pvlist      *editlog = 0;                                                  //  log of image edits done   v.10.2
char        *undo_files = 0;                                               //  undo/redo stack, image files
int         Pundo = 0;                                                     //  undo/redo stack position
int         Pumax = 0;                                                     //  undo/redo stack depth

int         Mbutton = 0;                                                   //  mouse button, 1/3 = left/right
int         LMclick = 0, RMclick = 0;                                      //  mouse left, right click
int         Mxclick, Myclick;                                              //  mouse click position
int         Mxposn, Myposn;                                                //  mouse move position
int         Mxdown, Mydown, Mxdrag, Mydrag;                                //  mouse drag vector
int         Mdrag = 0;                                                     //  mouse drag underway
int         Mcapture = 0;                                                  //  mouse captured by edit function

int         KBcapture = 0;                                                 //  KB key captured by edit function
int         KBkey = 0;                                                     //  active keyboard key

int         Ntoplines = 0, Nptoplines = 0;                                 //  lines overlayed on image in window
int         toplinex1[4], topliney1[4], toplinex2[4], topliney2[4];
int         ptoplinex1[4], ptopliney1[4], ptoplinex2[4], ptopliney2[4];

int         Ftoparc = 0, ptoparc = 0;                                      //  arc (circle/ellipse) on top
int         toparcx,toparcy,toparcw,toparch;                               //    of image in window
int         ptoparcx,ptoparcy,ptoparcw,ptoparch;

zdialog     *zdedittags = null;                                            //  edit tags zdialog
zdialog     *zdmasstags = null;                                            //  mass add tags dialog
zdialog     *zdsearchtags = null;                                          //  search tags dialog
zdialog     *zdexifview = null;                                            //  dialog to view EXIF data
zdialog     *zdrename = null;                                              //  rename file zdialog
zdialog     *zdRGB = null;                                                 //  show RGB dialog
zdialog     *zdsela = null;                                                //  select area dialog
zdialog     *zdedit = null;                                                //  image edit dialog
zdialog     *zdburn = null;                                                //  burn CD/DVD dialog

//  pano and HDR control data

int      curr_lens = 0;                                                    //  current lens, 0-3
int      lens_cc = 19;                                                     //  lens name cc limit
char     *lens4_name[4];                                                   //  names for 4 lenses
double   lens4_mm[4], lens4_bow[4];                                        //  characteristics for 4 lenses
double   lens_mm, lens_bow;                                                //  current lens characteristics
double   pano_prealign_size = 500;                                         //  pre-align image height
double   pano_image_increase = 1.6;                                        //  image size increase per stage
double   pano_blend_decrease = 0.8;                                        //  blend width decrease per stage
double   pano_min_alignwidth = 0.10;                                       //  min. align area, 10% width
double   pano_max_alignwidth = 0.20;                                       //  max. align area, 20%
double   pano_ycurveF = 1.41;                                              //  image curve, y-adjust factor

int      fullSize, alignSize;                                              //  full and align image sizes
int      xshrink, yshrink;                                                 //  image shrinkage (pano, HDF)
int      pxL, pxH, pyL, pyH;                                               //  image overlap area
int      pxM, pxmL, pxmH;                                                  //  align area width mid/low/high
int      pyM, pymL, pymH;                                                  //  align area height
int      Nalign = 0;                                                       //  alignment active, cycle count
int      aligntype = 0;                                                    //  align type (HDR, HDF, pano)
int      alignWidth, alignHeight;                                          //  pano/HDR/HDF alignment area
int      overlapixs = 0;                                                   //  overlapped pixels count
int      pixsamp = 5000;                                                   //  pixel sample size
int      showRedpix = 0;                                                   //  flag, highlight alignment pixels
char     *redpixels = 0;                                                   //  flags edge pixels for image align

double   Bratios1[3][256], Bratios2[3][256];                               //  brightness ratios/color/brightness
double   R12match[65536], G12match[65536], B12match[65536];                //  image1/2 color matching factors 
double   R21match[65536], G21match[65536], B21match[65536];                //  image2/1 color matching factors
double   Radjust, Gadjust, Badjust;                                        //  RGB manual adjustmants
double   xoff, yoff, toff;                                                 //  align data: x, y, theta
double   warpxu, warpyu, warpxl, warpyl;                                   //  pano image2 warp: upper/lower corners
double   xoffB, yoffB, toffB;                                              //  align data: current best values
double   warpxuB, warpyuB, warpxlB, warpylB;                               //  pano image2 warp: current best values
double   matchlev, matchB;                                                 //  image alignment match level

//  GTK functions

int   main(int argc, char * argv[]);                                       //  main program
int   gtkinitfunc(void *data);                                             //  GTK initz. function
int   gtimefunc(void *arg);                                                //  periodic function
int   delete_event();                                                      //  window delete event function
void  destroy_event();                                                     //  window destroy event function
int   mwpaint();                                                           //  window repaint
void  mwpaint2();                                                          //  window repaint (thread callable)
void  update_statusbar();                                                  //  update main window status bar
void  mouse_event(GtkWidget *, GdkEventButton *, void *);                  //  mouse event function
int   KBpress(GtkWidget *, GdkEventKey *, void *);                         //  KB key press event function
int   KBrelease(GtkWidget *, GdkEventKey *, void *);                       //  KB key release event
void  paint_toplines(int arg);                                             //  paint lines on image
void  paint_toparc(int arg);                                               //  paint arc on image
void  paint_gridlines();                                                   //  paint grid lines on image
void  draw_line(int x1, int y1, int x2, int y2);                           //  draw line, image space
void  erase_line(int x1, int y1, int x2, int y2);                          //  erase line
void  topmenufunc(GtkWidget *, cchar *menu);                               //  menu function

typedef void CBfunc();                                                     //  callback function type
CBfunc   *mouseCBfunc = 0;                                                 //  current mouse handler function
CBfunc   *KBkeyCBfunc = 0;                                                 //  current KB handler function

//  file functions

void  m_gallery(GtkWidget *, cchar *);                                     //  show image gallery window
void  m_gallery2(char *);                                                  //  converts function types
void  m_open(GtkWidget *, cchar *);                                        //  open image file (menu)
void  m_open_drag(int x, int y, char *file);                               //  open drag-drop file
void  m_recent(GtkWidget *, cchar *);                                      //  open recently accessed file
void  add_recent_file(cchar *file);                                        //  add file to recent file list
void  f_open(cchar *file);                                                 //  open file helper function
void  m_prev(GtkWidget *, cchar *);                                        //  open previous file
void  m_next(GtkWidget *, cchar *);                                        //  open next file
void  m_save(GtkWidget *, cchar *);                                        //  save modified image to same file
void  m_saveas(GtkWidget *, cchar *);                                      //  save modified image to another file
void  m_print(GtkWidget *, cchar *);                                       //  print image file
void  m_trash(GtkWidget *, cchar *);                                       //  move image to trash
void  m_rename(GtkWidget *, cchar *);                                      //  rename file
void  rename_dialog();                                                     //  rename file dialog
void  m_massrename(GtkWidget *, cchar *);                                  //  rename many files, base + sequence
void  m_quit(GtkWidget *, cchar *);                                        //  exit application

//  tools functions

void  m_zoom(GtkWidget *, cchar *);                                        //  zoom image +/-
void  m_montest(GtkWidget *, cchar *);                                     //  check monitor
void  m_histogram(GtkWidget *, cchar *);                                   //  start brightness histogram
void  histogram_paint();                                                   //  update brightness histogram
void  histogram_destroy();                                                 //  remove histogram window
void  m_clone(GtkWidget *, cchar *);                                       //  start another fotoxx instance
void  m_slideshow(GtkWidget *, cchar *);                                   //  slideshow mode
void  m_showRGB(GtkWidget *, cchar *);                                     //  show RGB values at mouse click
void  m_gridlines(GtkWidget *, cchar *);                                   //  set up grid lines
void  m_parms(GtkWidget *, cchar *);                                       //  edit parameters
void  m_lang(GtkWidget *, cchar *);                                        //  change language
void  m_launcher(GtkWidget *, cchar *);                                    //  make desktop icon/launcher
void  m_conv_raw(GtkWidget *, cchar *);                                    //  convert RAW files to tiff
void  m_burn(GtkWidget *, cchar *);                                        //  burn images to CD/DVD
void  burn_insert_file(cchar *);                                           //  called from image gallery window

//  EXIF functions

void  m_exif_view(GtkWidget *, cchar *);                                   //  view EXIF data in popup window
void  m_exif_edit(GtkWidget *, cchar *);                                   //  add or change EXIF data
void  m_exif_delete(GtkWidget *, cchar *);                                 //  delete EXIF data
char  ** exif_get(cchar *file, cchar **keys, int nkeys);                   //  get EXIF data for given key(s)
int   exif_set(cchar *file, cchar **keys, cchar **text, int nkeys);        //  set EXIF data for given key(s)
int   exif_copy(cchar *f1, cchar *f2, cchar **k, cchar **t, int nk);       //  copy EXIF data + opt. updates

//  tags functions

void  m_edittags(GtkWidget *, cchar *);                                    //  edit tags dialog
void  load_filetags(cchar *file);                                          //  load tags from an image file
void  update_filetags(cchar *file);                                        //  write updated tags to image file
void  load_asstags();                                                      //  load all assigned tags
void  update_asstags(cchar *file, int del = 0);                            //  update tags index file
void  m_masstags(GtkWidget *, cchar *);                                    //  add tags to many files at once
void  m_searchtags(GtkWidget *, cchar *);                                  //  search images for matching tags
void  exif_tagdate(cchar *exifdate, char *tagdate);                        //  convert exif date to tag date format
void  tag_exifdate(cchar *tagdate, char *exifdate);                        //  convert tag date to exif date format
void  m_tags_convert(GtkWidget *, cchar *);                                //  convert tags to new standard
void  m_tags_index(GtkWidget *, cchar *);                                  //  generate tags index file

//  select area functions

void  m_select(GtkWidget *, cchar *);                                      //  select area for subsequent edits
void  m_select_invert(GtkWidget *, cchar *);                               //  invert area
void  m_select_enable(GtkWidget *, cchar *);                               //  enable area
void  m_select_disable(GtkWidget *, cchar *);                              //  disable area
void  m_select_delete(GtkWidget *, cchar *);                               //  delete area
void  select_show(int flag);                                               //  show or hide area
void  m_select_copy(GtkWidget *, cchar *);                                 //  copy and save area
void  m_select_paste(GtkWidget *, cchar *);                                //  paste saved area into image
void  m_select_read_file(GtkWidget *, cchar *);                            //  read area from a file
void  m_select_save_file(GtkWidget *, cchar *);                            //  save area to a file

//  edit functions

void  m_whitebal(GtkWidget *, cchar *);                                    //  adjust white balance
void  m_flatten(GtkWidget *, cchar *);                                     //  flatten brightness distribution
void  m_tune(GtkWidget *, cchar *);                                        //  brightness / color adjustments
void  m_brightramp(GtkWidget *, cchar *);                                  //  ramp brightness across image
void  m_xbright(GtkWidget *, cchar *);                                     //  expand brightness range, clip 
void  m_tonemap(GtkWidget *, cchar *);                                     //  tone mapping
void  m_redeye(GtkWidget *, cchar *);                                      //  red-eye removal
void  m_blur(GtkWidget *, cchar *);                                        //  blur image
void  m_sharpen(GtkWidget *, cchar *);                                     //  sharpen image
void  m_denoise(GtkWidget *, cchar *);                                     //  image noise reduction
void  m_resize(GtkWidget *, cchar *);                                      //  resize image
void  m_trim(GtkWidget *, cchar *);                                        //  trim image
void  m_rotate(GtkWidget *, cchar *);                                      //  rotate image
void  m_flip(GtkWidget *, cchar *);                                        //  flip horizontally or vertically
void  m_unbend(GtkWidget *, cchar *);                                      //  fix perspective problems
void  m_WarpA(GtkWidget *, cchar *);                                       //  warp image area
void  m_WarpI(GtkWidget *, cchar *);                                       //  warp image globally, curvy transform
void  m_WarpF(GtkWidget *, cchar *);                                       //  warp image globally, affine transform
void  m_colordep(GtkWidget *, cchar *);                                    //  set color depth 1-16 bits/color
void  m_draw(GtkWidget *, cchar *);                                        //  make simulated drawing
void  m_emboss(GtkWidget *, cchar *);                                      //  make simulated embossing
void  m_tiles(GtkWidget *, cchar *);                                       //  make simulated tiles (pixelate)
void  m_painting(GtkWidget *, cchar *);                                    //  make simulated painting
void  m_pixedit(GtkWidget *, cchar *);                                     //  edit individual pixels
void  m_HDR(GtkWidget *, cchar *);                                         //  make HDR image
void  m_HDF(GtkWidget *, cchar *);                                         //  make HDF image
void  m_pano(GtkWidget *, cchar *);                                        //  make panorama image

//  pano and HDR common functions

int      sigdiff(double d1, double d2, double signf);                      //  test for significant difference
void     getAlignArea();                                                   //  get image overlap area
void     getBrightRatios();                                                //  get color brightness ratios
void     setColorfixFactors(int state);                                    //  set color matching factors
void     flagEdgePixels();                                                 //  flag high-contrast pixels
double   matchImages();                                                    //  match images in overlap region
double   matchPixels(uint16 *pix1, uint16 *pix2);                          //  match two pixels
int      vpixel(RGB *rgb, double px, double py, uint16 *vpix);             //  get virtual pixel at (px,py)

//  edit support functions

typedef void * threadfunc(void *);                                         //  edit thread function

int   edit_setup(cchar *func, int preview, int delsa);                     //  start new edit transaction
void  edit_cancel();                                                       //  cancel edit
void  edit_done();                                                         //  commit edit, add undo stack
void  edit_undo();                                                         //  undo edit, revert
void  edit_redo();                                                         //  redo the edit after undo
void  edit_reset();                                                        //  reset to initial status
void  edit_zapredo();                                                      //  clear image redo copy
void  edit_fullsize();                                                     //  convert to full-size pixmaps

void  start_thread(threadfunc func, void *arg);                            //  start edit thread
void  signal_thread();                                                     //  signal thread, work is pending
void  wait_thread_idle();                                                  //  wait for work complete
void  wrapup_thread(int command);                                          //  wait for exit or command exit
int   thread_working();                                                    //  0/1 >> thread is idle/working
void  thread_idle_loop();                                                  //  thread: wait for work or exit command
void  exit_thread();                                                       //  thread: exit unconditionally

void  start_wt(threadfunc func, void *arg);                                //  start working thread (per processor core)
void  exit_wt();                                                           //  exit working thread
void  wait_wts();                                                          //  wait for working threads

//  toolbar functions not also in file menu

void  m_undo(GtkWidget *, cchar *);                                        //  undo one edit
void  m_redo(GtkWidget *, cchar *);                                        //  redo one edit
void  save_undo();                                                         //  undo/redo save function
void  load_undo();                                                         //  undo/redo read function

//  other support functions

void  m_help(GtkWidget *, cchar *);                                        //  various help functions
int   load_fotoxx_state();                                                 //  load state from prior session
int   save_fotoxx_state();                                                 //  save state for next session
int   mod_keep();                                                          //  query keep/discard edited image
int   menulock(int lock);                                                  //  lock/unlock menu for some funcs
void  turn_image(int angle);                                               //  turn image (upright)
void  brhood_calc(int radius, char method);                                //  compute neighborhood brightness    v.9.6
inline float get_brhood(int px, int py);                                   //  get pixel neighborhood brightness
void  free_resources();                                                    //  free all allocated resources
void  set_cursor(GdkCursor *cursor);                                       //  set cursor for main/drawing window

RGB * f_load(cchar *filespec, int bpc);                                    //  load a file into an RGB pixmap
int   f_save(cchar *outfile, cchar *format);                               //  save an RGB pixmap to a file
RGB * TIFFread(cchar *filespec);                                           //  read TIFF file, native bpc
int   TIFFwrite(RGB *rgb, cchar *filespec);                                //  write TIFF file, bpc from rgb
void  tiffwarninghandler(cchar *mod, cchar *fmt, va_list);                 //  intercepts TIFF lib warnings
RGB * PXBread(cchar *filespec);                                            //  read using pixbuf library, bpc = 8
int   PXBwrite(RGB *rgb, cchar *filespec);                                 //  write using pixbuf library, bpc = 8

//  RGB pixmap conversion functions

RGB * RGB_make(int ww, int hh, int bpc);                                   //  initialize RGB pixmap
void  RGB_free(RGB *rgb);                                                  //  free RGB pixmap
RGB * RGB_copy(RGB *rgb);                                                  //  copy RGB pixmap
RGB * RGB_copy_area(RGB *rgb, int orgx, int orgy, int ww, int hh);         //  copy section of RGB pixmap
RGB * RGB_convbpc(RGB *rgb);                                               //  convert from 8/16 to 16/8 bpc
RGB * RGB_rescale(RGB *rgb, int ww, int hh);                               //  rescale RGB pixmap (ww/hh)
RGB * RGB_rotate(RGB *rgb, double angle);                                  //  rotate RGB pixmap

//  translatable strings used in multiple dialogs

cchar  *Baddall;
cchar  *Bamount;
cchar  *Bapply;
cchar  *Bblendwidth;
cchar  *Bblue;
cchar  *Bbrightness;
cchar  *Bcancel;
cchar  *Bclear;
cchar  *Bcolor;
cchar  *Bdarker;
cchar  *Bdelete;
cchar  *Bdone;
cchar  *Bedit;
cchar  *Bexiftoolmissing;
cchar  *Bfetch;
cchar  *Bfinish;
cchar  *Bgreen;
cchar  *Bheight;
cchar  *Bhide;
cchar  *Binsert;
cchar  *Binvert;
cchar  *Blighter;
cchar  *Blimit;
cchar  *BOK;
cchar  *Bopenrawfile;
cchar  *Bpause;
cchar  *Bpercent;
cchar  *Bpresets;
cchar  *Bproceed;
cchar  *Bradius;
cchar  *Brange;
cchar  *Bred;
cchar  *Bredo;
cchar  *Breduce;
cchar  *Bresume;
cchar  *Bsave;
cchar  *Bsavetoedit;
cchar  *Bsearch;
cchar  *Bselect;
cchar  *Bshow;
cchar  *Bstart;
cchar  *Bsuspend;
cchar  *Bthresh;
cchar  *Bundo;
cchar  *Bundoall;
cchar  *Bundolast;
cchar  *Bwidth;


/**************************************************************************
      main program and GTK/GDK functions
***************************************************************************/

int main(int argc, char *argv[])
{
   using namespace zfuncs;

   char           lang[8] = "";
   GError         *gerror = 0;
   GdkPixbuf      *pixbuf;
   GdkDisplay     *display;
   char           iconpath[200];
   
   printf(fversion "\n");                                                  //  print fotoxx version
   if (argc > 1 && strEqu(argv[1],"-v")) return 0;

   gtk_init(&argc,&argv);                                                  //  initz. GTK
   zlockInit();

   initz_appfiles("fotoxx",null);                                          //  get app directories

   load_fotoxx_state();                                                    //  restore data from last session

   for (int ii = 1; ii < argc; ii++)                                       //  command line options
   {
      if (strEqu(argv[ii],"-d"))                                           //  -d (debug flag)
            Fdebug = 1;
      else if (strEqu(argv[ii],"-l") && argc > ii+1)                       //  -l language code
            strncpy0(lang,argv[++ii],7);
      else  image_file = strdupz(argv[ii],0,"initfile");                   //  initial file or directory
   }

   ZTXinit(lang);                                                          //  setup translations

   mWin = gtk_window_new(GTK_WINDOW_TOPLEVEL);                             //  create main window
   gtk_window_set_title(GTK_WINDOW(mWin),fversion);
   gtk_window_set_position(GTK_WINDOW(mWin),GTK_WIN_POS_CENTER);
   gtk_window_set_default_size(GTK_WINDOW(mWin),Dww,Dhh);

   mVbox = gtk_vbox_new(0,0);                                              //  add vert. packing box
   gtk_container_add(GTK_CONTAINER(mWin),mVbox);

   mMbar = create_menubar(mVbox);                                          //  menus / sub-menus

   GtkWidget *mFile = add_menubar_item(mMbar,ZTX("File"),topmenufunc);
      add_submenu_item(mFile,       ZTX("Image Gallery"),         m_gallery);
      add_submenu_item(mFile,       ZTX("Open Image File"),       m_open);
      add_submenu_item(mFile,       ZTX("Open Recent File"),      m_recent);
      add_submenu_item(mFile,       ZTX("Save to Same File"),     m_save);
      add_submenu_item(mFile,       ZTX("Save to New File"),      m_saveas);
      add_submenu_item(mFile,       ZTX("Print Image File"),      m_print);
      add_submenu_item(mFile,       ZTX("Trash Image File"),      m_trash);
      add_submenu_item(mFile,       ZTX("Rename Image File"),     m_rename);
      add_submenu_item(mFile,       ZTX("Mass Rename Files"),     m_massrename);
      add_submenu_item(mFile,       ZTX("Quit fotoxx"),           m_quit);

   GtkWidget *mTools = add_menubar_item(mMbar,ZTX("Tools"),topmenufunc);
      add_submenu_item(mTools,      ZTX("Check Monitor"),         m_montest);
      add_submenu_item(mTools,      ZTX("Brightness Graph"),      m_histogram);
      add_submenu_item(mTools,      ZTX("Clone fotoxx"),          m_clone);
      add_submenu_item(mTools,      ZTX("Slide Show"),            m_slideshow);
      add_submenu_item(mTools,      ZTX("Show RGB"),              m_showRGB);
      add_submenu_item(mTools,      ZTX("Grid Lines"),            m_gridlines);
      add_submenu_item(mTools,      ZTX("Lens Parameters"),       m_parms);
      add_submenu_item(mTools,      ZTX("Change Language"),       m_lang);
      add_submenu_item(mTools,      ZTX("Create Launcher"),       m_launcher);
      add_submenu_item(mTools,      ZTX("Convert RAW files"),     m_conv_raw);
      add_submenu_item(mTools,      ZTX("Burn Images to CD/DVD"), m_burn);

   GtkWidget *mExif = add_menubar_item(mMbar,"EXIF",topmenufunc);
      add_submenu_item(mExif,       ZTX("Basic EXIF data"),       m_exif_view);
      add_submenu_item(mExif,       ZTX("All EXIF data"),         m_exif_view);
      add_submenu_item(mExif,       ZTX("Edit EXIF data"),        m_exif_edit);
      add_submenu_item(mExif,       ZTX("Delete EXIF data"),      m_exif_delete);

   GtkWidget *mTags = add_menubar_item(mMbar,ZTX("Tags"),topmenufunc);
      add_submenu_item(mTags,       ZTX("Edit Tags"),             m_edittags);
      add_submenu_item(mTags,       ZTX("Mass Add Tags"),         m_masstags);
      add_submenu_item(mTags,       ZTX("Search Tags"),           m_searchtags);
      add_submenu_item(mTags,       ZTX("Index Tags"),            m_tags_index);
      add_submenu_item(mTags,       ZTX("Convert Tags !!!"),      m_tags_convert);

   GtkWidget *mArea = add_menubar_item(mMbar,ZTX("Area"),topmenufunc);
      add_submenu_item(mArea,       ZTX("Select"),                m_select);
      add_submenu_item(mArea,       ZTX("Invert"),                m_select_invert);
      add_submenu_item(mArea,       ZTX("Enable"),                m_select_enable);
      add_submenu_item(mArea,       ZTX("Disable"),               m_select_disable);
      add_submenu_item(mArea,       ZTX("Delete"),                m_select_delete);
      add_submenu_item(mArea,       ZTX("Copy"),                  m_select_copy);
      add_submenu_item(mArea,       ZTX("Paste"),                 m_select_paste);
      add_submenu_item(mArea,       ZTX("Read File"),             m_select_read_file);
      add_submenu_item(mArea,       ZTX("Save File"),             m_select_save_file);

   GtkWidget *mRetouch = add_menubar_item(mMbar,ZTX("Retouch"),topmenufunc);
      add_submenu_item(mRetouch,    ZTX("White Balance"),         m_whitebal);
      add_submenu_item(mRetouch,    ZTX("Flatten Brightness"),    m_flatten);
      add_submenu_item(mRetouch,    ZTX("Brightness/Color"),      m_tune);
      add_submenu_item(mRetouch,    ZTX("Brightness Ramp"),       m_brightramp);
      add_submenu_item(mRetouch,    ZTX("Expand Brightness"),     m_xbright);
      add_submenu_item(mRetouch,    ZTX("Tone Mapping"),          m_tonemap);
      add_submenu_item(mRetouch,    ZTX("Red Eyes"),              m_redeye);
      add_submenu_item(mRetouch,    ZTX("Blur Image"),            m_blur);
      add_submenu_item(mRetouch,    ZTX("Sharpen Image"),         m_sharpen);
      add_submenu_item(mRetouch,    ZTX("Reduce Noise"),          m_denoise);

   GtkWidget *mTransf = add_menubar_item(mMbar,ZTX("Transform"),topmenufunc);
      add_submenu_item(mTransf,     ZTX("Trim Image"),            m_trim);
      add_submenu_item(mTransf,     ZTX("Resize Image"),          m_resize);
      add_submenu_item(mTransf,     ZTX("Rotate Image"),          m_rotate);
      add_submenu_item(mTransf,     ZTX("Flip Image"),            m_flip);
      add_submenu_item(mTransf,     ZTX("Unbend Image"),          m_unbend);
      add_submenu_item(mTransf,     ZTX("Warp Area"),             m_WarpA);
      add_submenu_item(mTransf,     ZTX("Warp Image (curvy)"),    m_WarpI);
      add_submenu_item(mTransf,     ZTX("Warp Image (affine)"),   m_WarpF);

   GtkWidget *mArt = add_menubar_item(mMbar,ZTX("Art"),topmenufunc);
      add_submenu_item(mArt,        ZTX("Color Depth"),           m_colordep);
      add_submenu_item(mArt,        ZTX("Simulate Drawing"),      m_draw);
      add_submenu_item(mArt,        ZTX("Simulate Embossing"),    m_emboss);
      add_submenu_item(mArt,        ZTX("Simulate Tiles"),        m_tiles);
      add_submenu_item(mArt,        ZTX("Simulate Painting"),     m_painting);
      add_submenu_item(mArt,        ZTX("Edit Pixels"),           m_pixedit);

   GtkWidget *mComb = add_menubar_item(mMbar,ZTX("Combine"),topmenufunc);
      add_submenu_item(mComb,       ZTX("HDR"),                   m_HDR);
      add_submenu_item(mComb,       ZTX("HDF"),                   m_HDF);
      add_submenu_item(mComb,       ZTX("Panorama"),              m_pano);

   GtkWidget *mHelp = add_menubar_item(mMbar,ZTX("Help"),topmenufunc);
      add_submenu_item(mHelp,       ZTX("About"),                 m_help);
      add_submenu_item(mHelp,       ZTX("User Guide"),            m_help);
      add_submenu_item(mHelp,       "README",                     m_help);
      add_submenu_item(mHelp,       ZTX("Change Log"),            m_help);
      add_submenu_item(mHelp,       ZTX("Translate"),             m_help);
      add_submenu_item(mHelp,       ZTX("Home Page"),             m_help);

   mTbar = create_toolbar(mVbox);                                          //  toolbar buttons
      add_toolbar_button(mTbar,  ZTX("Gallery"),   ZTX("Image Gallery"),         "gallery.png", m_gallery);
      add_toolbar_button(mTbar,  ZTX("Open"),      ZTX("Open Image File"),       "open.png",    m_open);
      add_toolbar_button(mTbar,  ZTX("Prev"),      ZTX("Open Previous File"),    "prev.png",    m_prev);
      add_toolbar_button(mTbar,  ZTX("Next"),      ZTX("Open Next File"),        "next.png",    m_next);
      add_toolbar_button(mTbar,  ZTX("Save"),      ZTX("Save to Same File"),     "save.png",    m_save);
      add_toolbar_button(mTbar,  ZTX("Save As"),   ZTX("Save to New File"),      "save.png",    m_saveas);
      add_toolbar_button(mTbar,  ZTX("Undo"),      ZTX("Undo One Edit"),         "undo.png",    m_undo);
      add_toolbar_button(mTbar,  ZTX("Redo"),      ZTX("Redo One Edit"),         "redo.png",    m_redo);
      add_toolbar_button(mTbar,  "Zoom+",          ZTX("Zoom-in (bigger)"),      "zoom+.png",   m_zoom);
      add_toolbar_button(mTbar,  "Zoom-",          ZTX("Zoom-out (smaller)"),    "zoom-.png",   m_zoom);
      add_toolbar_button(mTbar,  ZTX("Trash"),     ZTX("Move Image to Trash"),   "trash.png",   m_trash);
      add_toolbar_button(mTbar,  ZTX("Quit"),      ZTX("Quit fotoxx"),           "quit.png",    m_quit);

   drWin = gtk_drawing_area_new();                                         //  add drawing window
   gtk_box_pack_start(GTK_BOX(mVbox),drWin,1,1,0);

   STbar = create_stbar(mVbox);                                            //  add status bar

   G_SIGNAL(mWin,"delete_event",delete_event,0)                            //  connect signals to windows
   G_SIGNAL(mWin,"destroy",destroy_event,0)
   G_SIGNAL(drWin,"expose-event",mwpaint,0)

   gtk_widget_add_events(drWin,GDK_BUTTON_PRESS_MASK);                     //  connect mouse events
   gtk_widget_add_events(drWin,GDK_BUTTON_RELEASE_MASK);
   gtk_widget_add_events(drWin,GDK_BUTTON_MOTION_MASK);                    //  all buttons         v.6.8
   gtk_widget_add_events(drWin,GDK_POINTER_MOTION_MASK);                   //  pointer motion      v.8.3
   G_SIGNAL(drWin,"button-press-event",mouse_event,0)
   G_SIGNAL(drWin,"button-release-event",mouse_event,0)
   G_SIGNAL(drWin,"motion-notify-event",mouse_event,0)

   G_SIGNAL(mWin,"key-press-event",KBpress,0)                           	//  connect KB events
   G_SIGNAL(mWin,"key-release-event",KBrelease,0)
   
   drag_drop_connect(drWin,m_open_drag);                                   //  connect drag-drop event   v.7.3

   gtk_widget_show_all(mWin);                                              //  show all widgets

   gdkgc = gdk_gc_new(drWin->window);                                      //  initz. graphics context

   black.red = black.green = black.blue = 0;                               //  set up colors
   white.red = white.green = white.blue = maxcolor;
   red.red = maxcolor;  red.green = red.blue = 0;
   green.green = maxcolor; green.red = green.blue = 0;

   colormap = gtk_widget_get_colormap(drWin);
   gdk_rgb_find_color(colormap,&black);
   gdk_rgb_find_color(colormap,&white);
   gdk_rgb_find_color(colormap,&red);
   gdk_rgb_find_color(colormap,&green);

   gdk_gc_set_foreground(gdkgc,&black);
   gdk_gc_set_background(gdkgc,&white);
   gdk_gc_set_line_attributes(gdkgc,1,lineattributes);

   arrowcursor = gdk_cursor_new(GDK_TOP_LEFT_ARROW);                       //  cursor for selection
   dragcursor = gdk_cursor_new(GDK_CROSSHAIR);                             //  cursor for dragging
   drawcursor = gdk_cursor_new(GDK_PENCIL);                                //  cursor for drawing lines
   busycursor = gdk_cursor_new(GDK_WATCH);                                 //  cursor for function busy

   display = gdk_display_get_default();                                    //  special function busy cursor  v.10.1
   *iconpath = 0;
   strncatv(iconpath,199,zicondir,"/","busy.png",null);
   pixbuf = gdk_pixbuf_new_from_file(iconpath,&gerror);
   if (pixbuf && display)
      busycursor = gdk_cursor_new_from_pixbuf(display,pixbuf,0,0);

   gtk_init_add((GtkFunction) gtkinitfunc,0);                              //  set initz. call from gtk_main()
   gtk_main();                                                             //  start processing window events
   return 0;
}


/**************************************************************************/

//  initial function called from gtk_main() at startup

int gtkinitfunc(void *data)
{
   int            err, err1, err2, flag, npid;
   char           procfile[20];
   cchar          *ppc, *ppc2;
   struct stat    statb;

   Baddall = ZTX("Add All");
   Bamount = ZTX("Amount");
   Bapply = ZTX("Apply");
   Bblendwidth = ZTX("Blend Width");
   Bblue = ZTX("Blue");
   Bbrightness = ZTX("Brightness");
   Bcancel = ZTX("Cancel");
   Bclear = ZTX("Clear");
   Bcolor = ZTX("Color");
   Bdarker = ZTX("Darker Areas");
   Bdelete = ZTX("Delete");
   Bdone = ZTX("Done");
   Bedit = ZTX("Edit");
   Bexiftoolmissing = ZTX("exiftool missing, please install \n"
                          " package libimage-exiftool-perl" );
   Bfetch = ZTX("Fetch");
   Bfinish = ZTX("Finish");
   Bgreen = ZTX("Green");
   Bheight = ZTX("Height");
   Bhide = ZTX("Hide");
   Binsert = ZTX("Insert");
   Binvert = ZTX("Invert");
   Blighter = ZTX("Lighter Areas");
   Blimit = ZTX("limit");
   BOK = ZTX("OK");
   Bopenrawfile = ZTX("Open RAW File");
   Bpause = ZTX("Pause");
   Bpercent = ZTX("Percent");
   Bpresets = ZTX("Presets");
   Bproceed = ZTX("Proceed");
   Bradius = ZTX("Radius");
   Brange = ZTX("range");
   Bredo = ZTX("Redo");
   Breduce = ZTX("Reduce");
   Bred = ZTX("Red");
   Bresume = ZTX("Resume");
   Bsave = ZTX("Save");
   Bsavetoedit = ZTX("Unknown file type, save as tiff/jpeg/png to edit");
   Bsearch = ZTX("Search");
   Bselect = ZTX("Select");
   Bshow = ZTX("Show");
   Bstart = ZTX("Start");
   Bsuspend = ZTX("Suspend");
   Bthresh = ZTX("Threshold");
   Bundoall = ZTX("Undo All");
   Bundolast = ZTX("Undo Last");
   Bundo = ZTX("Undo");
   Bwidth = ZTX("Width");

   tid_fmain = pthread_self();                                             //  get main() thread ID
   snprintf(PIDstring,11,"%06d",getpid());                                 //  get fotoxx process PID pppppp

   Nwt = sysconf(_SC_NPROCESSORS_ONLN);                                    //  get SMP CPU count   v.7.1
   if (! Nwt) Nwt = 1;
   if (Nwt > max_wt) Nwt = max_wt;                                         //  compile time limit
   printf("using %d threads \n",Nwt);

   err = system("echo -n \"exiftool \"; exiftool -ver");                   //  check for exiftool  v.6.9.1
   if (! err) Fexiftool = 1;
   err = system("xdg-open --version");                                     //  check for xdg-open
   if (! err) Fxdgopen = 1;
   err = system("ufraw --version");                                        //  check for ufraw
   if (! err) Fufraw = 1;

   asstags_file = zmalloc(200,"asstagsfile");                              //  tags index file (old)
   *asstags_file = 0;                                                      //  bugfix v.9.9
   strncatv(asstags_file,199,get_zuserdir(),"/assigned_tags",null);        //  home/user/.fotoxx/assigned_tags

   tags_index_file = zmalloc(200,"tagsindexfile");                         //  tags index file (new)    v.10.0
   *tags_index_file = 0;
   strncatv(tags_index_file,199,get_zuserdir(),"/tags_index",null);        //  home/user/.fotoxx/tags_index

   undo_files = zmalloc(200,"undofiles");                                  //  look for orphaned undo files
   *undo_files = 0;
   strncatv(undo_files,199,get_zuserdir(),"/*_undo_*",null);               //  home/user/.fotoxx/*_undo_*
   flag = 1;
   while ((ppc = SearchWild(undo_files,flag)))
   {
      ppc2 = strstr(ppc,".fotoxx/");
      if (! ppc2) continue;
      npid = atoi(ppc2+8);                                                 //  pid of file owner
      snprintf(procfile,19,"/proc/%d",npid);
      err = stat(procfile,&statb);
      if (! err) continue;                                                 //  pid is active, keep file
      printf("orphaned undo file deleted: %s \n",ppc);
      snprintf(command,ccc,"rm -f %s",ppc);                                //  delete orphaned files     v.8.0
      err = system(command);
   }

   *undo_files = 0;                                                        //  setup undo filespec template
   strncatv(undo_files,199,get_zuserdir(),"/",PIDstring,"_undo_nn",null);  //  home/user/.fotoxx/pppppp_undo_nn
   
   editlog = pvlist_create(maxedits);                                      //  history log of image edits done
   for (int ii = 0; ii < maxedits; ii++)                                   //  pre-load all pvlist entries   v.10.2
      pvlist_append(editlog,"nothing");
   
   select_area_dirk = zmalloc(200,"selectareadirk");                       //  directory for saved areas    v.9.9
   *select_area_dirk = 0;
   strncatv(select_area_dirk,199,get_zuserdir(),"/saved_areas/",null);     //  home/user/.fotoxx/saved_areas/
   err = stat(select_area_dirk,&statb);
   if (err) mkdir(select_area_dirk,0750);

   mutex_init(&pixmaps_lock,0);                                            //  setup lock for edit pixmaps

   TIFFSetWarningHandler(tiffwarninghandler);                              //  intercept TIFF warning messages
   
   g_timeout_add(50,gtimefunc,0);                                          //  start periodic function (50 ms)

   //  determine if tags conversion is needed                              //  v.10.0   

   err1 = stat(asstags_file,&statb);                                       //  is old tags index present?
   printf("%s",asstags_file);
   if (! err1) printf("  - found \n");
   else printf("  - not found \n");

   err2 = stat(tags_index_file,&statb);                                    //  is new tags index present?
   printf("%s",tags_index_file);
   if (! err2) printf("  - found \n");
   else printf("  - not found \n");

   if (err2 && ! err1) {                                                   //  have old file and no new file
      showz_userguide("convert_tags");
      zmessageACK(GTK_WINDOW(mWin),"WARNING: Convert tags to new standard \n"
                                   "before any edits! (menu Tags -> Convert Tags)");
   }

   if (image_file) {
      char * pp = canonicalize_file_name(image_file);                      //  add cwd if needed
      zfree(image_file);
      image_file = null;
      if (pp) image_file = strdupz(pp,0,"imagefile");                      //  change pp to zmalloc
      if (pp) free(pp);
   }
      
   if (image_file) {
      err = stat(image_file,&statb);
      if (! err && S_ISREG(statb.st_mode)) f_open(image_file);             //  open initial file
   }

   return 0;
}


/**************************************************************************/

//  Periodic function - runs every few milliseconds.
//  Avoid any thread usage of gtk/gdk functions.

int gtimefunc(void *arg)
{
   double      secs;
   static int  fbusy = 0;

   if (Wrepaint) {                                                         //  update drawing window
      Wrepaint = 0;
      mwpaint();
   }

   if (SBupdate || Nalign || progress_goal) {                              //  status bar update
      SBupdate = 0;
      update_statusbar();
   }
   
   if (Fslideshow) {
      secs = get_seconds();                                                //  show next slide
      if (secs > SS_timer) {
         SS_timer = secs + SS_interval;
         m_next(0,0);  
      }
   }
   
   if (! fbusy && thread_working()) {
      if (currentcursor == null) {                                   
         set_cursor(busycursor);                                           //  set function busy cursor
         fbusy = 1;
      }
   }
   
   if (fbusy && ! thread_working()) {
      fbusy = 0;
      set_cursor(0);                                                       //  restore normal cursor
   }

   return 1;
}


/**************************************************************************/

//  functions for main window event signals

int delete_event()                                                         //  main window closed
{
   if (mod_keep()) return 1;                                               //  allow user bailout
   Fshutdown++;                                                            //  shutdown in progress
   save_fotoxx_state();                                                    //  save state for next session
   free_resources();                                                       //  delete undo files
   return 0;
}

void destroy_event()                                                       //  main window destroyed
{
   Fshutdown++;
   printf("main window destroyed \n");
   exit(1);                                                                //  instead of gtk_main_quit();
   return;
}


/**************************************************************************/

//  cause (modified) output image to get repainted immediately
//  this function may be called from threads

void mwpaint2()
{
   Wrepaint++;
   SBupdate++;                                                             //  moved for fvwm window manager   v.9.7
   return;
}


/**************************************************************************/

//  paint window when created, exposed, resized, or modified (edited)      //  overhauled  v.7.4

int mwpaint()
{
   GdkRectangle   wrect;
   int            incrx, incry;                                            //  mouse drag
   static int     pincrx = 0, pincry = 0;                                  //  prior mouse drag
   static int     pdww = 0, pdhh = 0;                                      //  prior window size
   static double  pscale = 1;                                              //  prior scale
   double         wscale, hscale;
   RGB            *rgbtemp1, *rgbtemp2;
   
   if (Fshutdown) return 1;                                                //  shutdown underway
   if (! Frgb8) {
      gdk_window_clear(drWin->window);
      return 1;                                                            //  no image
   }

   Dww = drWin->allocation.width;                                          //  (new) drawing window size
   Dhh = drWin->allocation.height;
   if (Dww < 20 || Dhh < 20) return 1;

   if (Dww != pdww || Dhh != pdhh) {                                       //  new window size       v.9.7
      SBupdate++;                                                          //  trigger status bar update
      pdww = Dww; 
      pdhh = Dhh;
   }

   if (mutex_trylock(&pixmaps_lock) != 0) {                                //  lock pixmaps
      Wrepaint++;                                                          //  cannot, come back later
      return 1;
   }
   
   if (E3rgb16) {                                                          //  get image size
      Iww = E3ww;
      Ihh = E3hh;                                                          //  edit in progress
   }
   else {
      Iww = Fww;                                                           //  no edit
      Ihh = Fhh;
   }
   
   if (Fzoom == 0) {                                                       //  scale to fit window
      wscale = 1.0 * Dww / Iww;
      hscale = 1.0 * Dhh / Ihh;
      if (wscale < hscale) Mscale = wscale;                                //  use greatest ww/hh ratio
      else  Mscale = hscale;
      if (Iww < Dww && Ihh < Dhh && ! Fblowup) Mscale = 1.0;               //  small image 1x unless Fblowup
      zoomx = zoomy = 0;
   }
   else Mscale = Fzoom;                                                    //  scale to Fzoom level

   if (Mscale > pscale) {                                                  //  zoom increased
      Iorgx += iww * 0.5 * (1.0 - pscale / Mscale);                        //  keep current image center  v.7.5
      Iorgy += ihh * 0.5 * (1.0 - pscale / Mscale);
   }
   pscale = Mscale;
   
   iww = Dww / Mscale;                                                     //  image space fitting in window
   if (iww > Iww) iww = Iww;
   ihh = Dhh / Mscale;
   if (ihh > Ihh) ihh = Ihh;

   if (zoomx || zoomy) {                                                   //  req. zoom center       v.7.5
      Iorgx = zoomx - 0.5 * iww;                                           //  corresp. image origin
      Iorgy = zoomy - 0.5 * ihh;
      zoomx = zoomy = 0;
   }

   if ((Mxdrag || Mydrag) && ! Mcapture) {                                 //  scroll via mouse drag
      incrx = (Mxdrag - Mxdown) * 1.3 * Iww / iww;                         //  scale   v.7.5
      incry = (Mydrag - Mydown) * 1.3 * Ihh / ihh;
      if (pincrx > 0 && incrx < 0) incrx = 0;                              //  stop bounce at extremes
      if (pincrx < 0 && incrx > 0) incrx = 0;
      pincrx = incrx;
      if (pincry > 0 && incry < 0) incry = 0;
      if (pincry < 0 && incry > 0) incry = 0;
      pincry = incry;
      Iorgx += incrx;                                                      //  new image origin after scroll
      Iorgy += incry;
      Mxdown = Mxdrag + incrx;                                             //  new drag origin
      Mydown = Mydrag + incry;
      Mxdrag = Mydrag = 0;
   }

   if (iww == Iww) {                                                       //  scaled image <= window width
      Iorgx = 0;                                                           //  center image in window
      Dorgx = 0.5 * (Dww - Iww * Mscale);
   }
   else Dorgx = 0;                                                         //  image > window, use entire window

   if (ihh == Ihh) {                                                       //  same for image height
      Iorgy = 0;
      Dorgy = 0.5 * (Dhh - Ihh * Mscale);
   }
   else Dorgy = 0;
   
   if (Iorgx + iww > Iww) Iorgx = Iww - iww;                               //  set limits
   if (Iorgy + ihh > Ihh) Iorgy = Ihh - ihh;
   if (Iorgx < 0) Iorgx = 0;
   if (Iorgy < 0) Iorgy = 0;

   if (E3rgb16) {                                                          //  edit in progress
      rgbtemp1 = RGB_copy_area(E3rgb16,Iorgx,Iorgy,iww,ihh);               //  copy RGB-16
      rgbtemp2 = RGB_convbpc(rgbtemp1);                                    //  convert to RGB-8
      RGB_free(rgbtemp1);
   }
   else rgbtemp2 = RGB_copy_area(Frgb8,Iorgx,Iorgy,iww,ihh);               //  no edit, copy RGB-8

   dww = iww * Mscale;                                                     //  scale to window
   dhh = ihh * Mscale;
   RGB_free(Drgb8);
   Drgb8 = RGB_rescale(rgbtemp2,dww,dhh);
   RGB_free(rgbtemp2);

   wrect.x = wrect.y = 0;                                                  //  stop flicker
   wrect.width = Dww;
   wrect.height = Dhh;
   gdk_window_begin_paint_rect(drWin->window,&wrect);

   gdk_window_clear(drWin->window);                                        //  clear window 

   gdk_draw_rgb_image(drWin->window, gdkgc, Dorgx, Dorgy, dww, dhh,        //  draw scaled image to window
                      nodither, (uint8 *) Drgb8->bmp, dww*3);

   if (Ntoplines) paint_toplines(1);                                       //  draw line overlays        v.8.3
   if (Ftoparc) paint_toparc(1);                                           //  draw arc overlay
   if (Fgrid) paint_gridlines();                                           //  draw grid lines           v.10.3.1
   if (Fshowarea && ! Fpreview) select_show(1);                            //  draw select area outline  v.9.7

   gdk_window_end_paint(drWin->window);                                    //  release all window updates

   mutex_unlock(&pixmaps_lock);                                            //  unlock pixmaps
   Wpainted++;                                                             //  notify edit function of repaint
   histogram_paint();                                                      //  update brightness histogram if exists
   return 1;
}


/**************************************************************************/

//  update status bar with image data and status
//  called periodically from timer function

void update_statusbar()
{
   static char    text1[200], text2[100];
   int            ww, hh, bpc, scale;
   int            percent_done;
   
   if (! image_file) return;
   
   *text1 = *text2 = 0;
   
   if (E3rgb16) {                                                          //  v.8.5
      ww = E3ww;
      hh = E3hh;
      bpc = 16;
   }
   else if (Frgb16) {
      ww = Fww;
      hh = Fhh;
      bpc = 16;
   }
   else {
      ww = Fww;
      hh = Fhh;
      bpc = file_bpc;
   }
   
   snprintf(text1,199,"%dx%dx%d",ww,hh,bpc);                               //  2345x1234x16 (preview) 1.56MB 45%
   if (Fpreview) strcat(text1," (preview)");                               //       ... (turned)
   sprintf(text2," %.2fMB",file_MB);
   if (Fmodified || Pundo > Fsaved) sprintf(text2," ?? MB");               //  file size TBD
   strcat(text1,text2);
   scale = int(Mscale * 100 + 0.5);
   sprintf(text2," %d%c",scale,'%');
   strcat(text1,text2);
   if (Fimageturned) strcat(text1," (turned)");
   
   if (Pundo)                                                              //  edit undo stack depth 
   {
      snprintf(text2,99,"  edits: %d",Pundo);
      strcat(text1,text2);
   }

   if (Nalign && aligntype == 1)                                           //  HDR alignment data
   {
      snprintf(text2,99,"  align: %d  offsets: %+.1f %+.1f %+.4f  match: %.5f",
                     Nalign,xoffB,yoffB,toffB,matchB);
      strcat(text1,text2);
   }
   
   if (Nalign && aligntype == 2)                                           //  HDF alignment data
   {
      snprintf(text2,99,"  align: %d  offsets: %+.1f %+.1f %+.4f  match: %.5f",
                     Nalign,xoffB,yoffB,toffB,matchB);
      strcat(text1,text2);
   }

   if (Nalign && aligntype == 3)                                           //  pano alignment data
   {
      snprintf(text2,99,"  align: %d  offsets: %+.1f %+.1f %+.4f %+.1f %+.1f %+.1f %+.1f  match: %.5f",
                     Nalign,xoffB,yoffB,toffB,warpxuB,warpyuB,warpxlB,warpylB,matchB);
      strcat(text1,text2);
   }

   if (Fautolens)                                                          //  lens parameters search status
   {
      snprintf(text2,99,"  lens: %.1f %.2f",lens_mm,lens_bow);
      strcat(text1,text2);
   }
   
   if (progress_goal)                                                      //  v.9.6
   {
      percent_done = 100 * (1.0 * progress_done / progress_goal);
      snprintf(text2,99,"  done: %d%c",percent_done,'%');
      strcat(text1,text2);
   }

   stbar_message(STbar,text1);
   return;
}


/**************************************************************************/

//  mouse event function - capture buttons and drag movements

void mouse_event(GtkWidget *, GdkEventButton *event, void *)
{
   void mouse_convert(int &xpos, int &ypos);

   static int     bdtime = 0, butime = 0, mbusy = 0;
   int            button, time, type;

   type = event->type;
   button = event->button;                                                 //  button, 1/3 = left/right
   time = event->time;
   Mxposn = int(event->x);                                                 //  mouse position in window
   Myposn = int(event->y);

   mouse_convert(Mxposn,Myposn);                                           //  convert to image space  v.8.4

   if (type == GDK_MOTION_NOTIFY) {
      if (mbusy) return;                                                   //  discard excess motion events
      mbusy++;
      zmainloop();
      mbusy = 0;
   }

   if (type == GDK_BUTTON_PRESS) {                                         //  button down
      bdtime = time;                                                       //  time of button down
      Mxdown = Mxposn;                                                     //  position at button down time
      Mydown = Myposn;
      if (button) {
         Mdrag++;                                                          //  possible drag start
         Mbutton = button;
      }
      Mxdrag = Mydrag = 0;
   }

   if (type == GDK_BUTTON_RELEASE) {                                       //  button up
      Mxclick = Myclick  = 0;                                              //  reset click status
      butime = time;                                                       //  time of button up
      if (butime - bdtime < 400)                                           //  less than 0.4 secs down
         if (Mxposn == Mxdown && Myposn == Mydown) {                       //       and not moving          v.8.6.1
            if (Mbutton == 1) LMclick++;                                   //  left mouse click
            if (Mbutton == 3) RMclick++;                                   //  right mouse click
            Mxclick = Mxdown;                                              //  click = button down position
            Myclick = Mydown;
         }
      Mxdown = Mydown = Mxdrag = Mydrag = Mdrag = Mbutton = 0;             //  forget buttons and drag
   }
   
   if (type == GDK_MOTION_NOTIFY && Mdrag) {                               //  drag underway
      Mxdrag = Mxposn;
      Mydrag = Myposn;
   }
   
   if (mouseCBfunc) {                                                      //  pass to handler function
      (* mouseCBfunc)();
      return;
   }

   if (LMclick && ! Mcapture) {                                            //  left click = zoom request
      LMclick = 0;
      zoomx = Mxclick;                                                     //  zoom center = mouse   v.7.5
      zoomy = Myclick;
      m_zoom(null, (char *) "+");
   }

   if (RMclick && ! Mcapture) {                                            //  right click = reset zoom
      RMclick = 0;
      zoomx = zoomy = 0;                                                   //  v.7.5
      m_zoom(null, (char *) "-");
   }

   if ((Mxdrag || Mydrag) && ! Mcapture) mwpaint();                        //  drag = scroll
   return;
}


//  convert mouse position from window space to image space

void mouse_convert(int &xpos, int &ypos)
{
   xpos = int((xpos - Dorgx) / Mscale + Iorgx + 0.5);
   ypos = int((ypos - Dorgy) / Mscale + Iorgy + 0.5);

   if (xpos < 0) xpos = 0;                                                 //  if outside image put at edge
   if (ypos < 0) ypos = 0;                                                 //                   v.8.4

   if (E3rgb16) { 
      if (xpos >= E3ww) xpos = E3ww-1;
      if (ypos >= E3hh) ypos = E3hh-1;
   }
   else {
      if (xpos >= Fww) xpos = Fww-1;
      if (ypos >= Fhh) ypos = Fhh-1;
   }
   
   return;
}


/**************************************************************************/

//  keyboard event function - some toolbar buttons have KB equivalents
//  GDK key symbols: /usr/include/gtk-2.0/gdk/gdkkeysyms.h

int   KBcontrolkey = 0;
int   KBzmalloclog = 0;

int KBpress(GtkWidget *win, GdkEventKey *event, void *)                    //  prevent propagation of key-press
{                                                                          //    events to toolbar buttons
   KBkey = event->keyval;
   if (KBkey == 65507) KBcontrolkey = 1;                                   //  Ctrl key is pressed    v.8.3
   return 1;
}

int KBrelease(GtkWidget *win, GdkEventKey *event, void *)
{
   KBkey = event->keyval;

   if (KBkeyCBfunc) {                                                      //  pass to handler function
      (* KBkeyCBfunc)();                                                   //  v.6.5
      KBkey = 0;
      return 1;
   }

   if (KBcapture) return 1;                                                //  let function handle it

   if (KBkey == 65507) KBcontrolkey = 0;                                   //  Ctrk key released   v.8.3
   
   if (KBcontrolkey) {
      if (KBkey == GDK_s) m_save(0,0);                                     //  Ctrl-* shortcuts   v.8.3
      if (KBkey == GDK_S) m_saveas(0,0);  
      if (KBkey == GDK_q) m_quit(0,0);  
      if (KBkey == GDK_Q) m_quit(0,0);  
   }

   if (KBkey == GDK_G) m_gallery(0,0);                                     //  key G  >>  image gallery   v.8.4.2
   if (KBkey == GDK_g) m_gallery(0,0);  

   if (KBkey == GDK_Left) m_prev(0,0);                                     //  arrow keys  >>  prev/next image
   if (KBkey == GDK_Right) m_next(0,0);  

   if (KBkey == GDK_plus) m_zoom(null, (char *) "+");                      //  +/- keys  >>  zoom in/out
   if (KBkey == GDK_equal) m_zoom(null, (char *) "+");                     //  = key: same as +
   if (KBkey == GDK_minus) m_zoom(null, (char *) "-");
   if (KBkey == GDK_KP_Add) m_zoom(null, (char *) "+");                    //  keypad +
   if (KBkey == GDK_KP_Subtract) m_zoom(null, (char *) "-");               //  keypad -
   
   if (KBkey == GDK_Z) m_zoom(null, (char *) "Z");                         //  Z key: zoom to 100%
   if (KBkey == GDK_z) m_zoom(null, (char *) "Z");

   if (KBkey == GDK_Escape) {                                              //  escape                 v.7.0
      if (Fslideshow) m_slideshow(0,0);                                    //  exit slideshow mode
      Fslideshow = 0;                                                      //  v.8.6
   }

   if (KBkey == GDK_Delete) m_trash(0,0);                                  //  delete  >>  trash

   if (KBkey == GDK_R) turn_image(+90);                                    //  keys L, R  >>  rotate
   if (KBkey == GDK_r) turn_image(+90);
   if (KBkey == GDK_L) turn_image(-90);
   if (KBkey == GDK_l) turn_image(-90);

   if (KBkey == GDK_F1) showz_userguide(zhelpcontext);                     //  F1 = help request   v.9.0
   
   if (KBcontrolkey && KBkey == GDK_m) {                                   //  zmalloc() activity log   v.3.5
      KBzmalloclog = 1 - KBzmalloclog;                                     //  Ctrl+M = toggle on/off
      if (KBzmalloclog) zmalloc_log(999999);
      else zmalloc_log(0);
   }
   
   if (KBkey == GDK_T || KBkey == GDK_t) m_trim(0,0);                      //  Key T = Trim function        v.10.3.1
   if (KBkey == GDK_B || KBkey == GDK_b) m_tune(0,0);                      //  Key B = brightness/color     v.10.3.1
   
   KBkey = 0;                                                              //  v.7.0
   return 1;
}


/**************************************************************************/

//  paint a grid of horizontal and vertical lines

void paint_gridlines()                                                     //  v.10.3.1
{
   int      px, py;
   int      startx, endx, stepx, endxx;
   int      starty, endy, stepy, endyy;
   
   if (! Frgb8) return;                                                    //  no image
   
   startx = gridx1;
   if (gridx2) endx = gridx2;
   else endx = dww;
   stepx = (endx - startx) / (gridnx + 1);
   endxx = endx - stepx / 2;                                               //  prevent last line almost at edge

   starty = gridy1;
   if (gridy2) endy = gridy2;
   else endy = dhh;
   stepy = (endy - starty) / (gridny + 1);
   endyy = endy - stepy / 2;

   gdk_gc_set_function(gdkgc,GDK_INVERT);                                  //  invert pixels

   if (gridnx) 
      for (px = Dorgx+startx+stepx; px < Dorgx+endxx; px += stepx)
         gdk_draw_line(drWin->window,gdkgc,px,Dorgy+starty,px,Dorgy+endy);

   if (gridny)
      for (py = Dorgy+starty+stepy; py < Dorgy+endyy; py += stepy)
         gdk_draw_line(drWin->window,gdkgc,Dorgx+startx,py,Dorgx+endx,py);

   gdk_gc_set_function(gdkgc,GDK_COPY);
   
   return;
}


/**************************************************************************/

//  refresh overlay lines on top of image
//  arg = 1:   paint lines only (because window repainted)
//        2:   erase lines and forget them
//        3:   erase old lines, paint new lines, save new in old

void paint_toplines(int arg)                                               //  v.8.3
{
   int      ii;

   if (arg == 2 || arg == 3)                                               //  erase old lines
      for (ii = 0; ii < Nptoplines; ii++)
         erase_line(ptoplinex1[ii],ptopliney1[ii],ptoplinex2[ii],ptopliney2[ii]);
   
   if (arg == 1 || arg == 3)                                               //  draw new lines
      for (ii = 0; ii < Ntoplines; ii++)
         draw_line(toplinex1[ii],topliney1[ii],toplinex2[ii],topliney2[ii]);

   if (arg == 2) {
      Nptoplines = Ntoplines = 0;                                          //  forget lines
      return;
   }

   for (ii = 0; ii < Ntoplines; ii++)                                      //  save for future erase
   {
      ptoplinex1[ii] = toplinex1[ii];
      ptopliney1[ii] = topliney1[ii];
      ptoplinex2[ii] = toplinex2[ii];
      ptopliney2[ii] = topliney2[ii];
   }

   Nptoplines = Ntoplines;

   return;
}


/**************************************************************************/

//  refresh overlay arc (circle/ellipse) on top of image
//  arg = 1:   paint arc only (because window repainted)
//        2:   erase arc and forget it
//        3:   erase old arc, paint new arc, save new in old

void paint_toparc(int arg)                                                 //  v.8.3
{
   int      arcx, arcy, arcw, arch;

   if (ptoparc && (arg == 2 || arg == 3)) {                                //  erase old arc
      arcx = int((ptoparcx-Iorgx) * Mscale + Dorgx + 0.5);                 //  image to window space
      arcy = int((ptoparcy-Iorgy) * Mscale + Dorgy + 0.5);
      arcw = int(ptoparcw * Mscale);
      arch = int(ptoparch * Mscale);

      gdk_gc_set_function(gdkgc,GDK_INVERT);                               //  invert pixels
      gdk_draw_arc(drWin->window,gdkgc,0,arcx,arcy,arcw,arch,0,64*360);    //  draw arc
      gdk_gc_set_function(gdkgc,GDK_COPY);
   }
   
   if (Ftoparc && (arg == 1 || arg == 3)) {                                //  draw new arc
      arcx = int((toparcx-Iorgx) * Mscale + Dorgx + 0.5);                  //  image to window space
      arcy = int((toparcy-Iorgy) * Mscale + Dorgy + 0.5);
      arcw = int(toparcw * Mscale);
      arch = int(toparch * Mscale);

      gdk_gc_set_function(gdkgc,GDK_INVERT);                               //  invert pixels
      gdk_draw_arc(drWin->window,gdkgc,0,arcx,arcy,arcw,arch,0,64*360);    //  draw arc
      gdk_gc_set_function(gdkgc,GDK_COPY);
   }

   if (arg == 2) {
      Ftoparc = ptoparc = 0;                                               //  forget arcs
      return;
   }
   
   ptoparc = Ftoparc;                                                      //  save for future erase
   ptoparcx = toparcx;
   ptoparcy = toparcy;
   ptoparcw = toparcw;
   ptoparch = toparch;

   return;
}


/**************************************************************************/

//  draw red/green line. coordinates are in image space.                   //  overhauled   v.8.4.2

void draw_line(int ix1, int iy1, int ix2, int iy2)
{
   void draw_pixel(double pxm, double pym);

   double      x1, y1, x2, y2;   
   double      pxm, pym, slope;
   
   x1 = Mscale * (ix1-Iorgx);                                              //  image to window space
   y1 = Mscale * (iy1-Iorgy);
   x2 = Mscale * (ix2-Iorgx);
   y2 = Mscale * (iy2-Iorgy);
   
   if (abs(y2 - y1) > abs(x2 - x1)) {
      slope = 1.0 * (x2 - x1) / (y2 - y1);
      if (y2 > y1) {
         for (pym = y1; pym <= y2; pym++) {
            pxm = round(x1 + slope * (pym - y1));
            draw_pixel(pxm,pym);
         }
      }
      else {
         for (pym = y1; pym >= y2; pym--) {
            pxm = round(x1 + slope * (pym - y1));
            draw_pixel(pxm,pym);
         }
      }
   }
   else {
      slope = 1.0 * (y2 - y1) / (x2 - x1);
      if (x2 > x1) {
         for (pxm = x1; pxm <= x2; pxm++) {
            pym = round(y1 + slope * (pxm - x1));
            draw_pixel(pxm,pym);
         }
      }
      else {
         for (pxm = x1; pxm >= x2; pxm--) {
            pym = round(y1 + slope * (pxm - x1));
            draw_pixel(pxm,pym);
         }
      }
   }

   gdk_gc_set_foreground(gdkgc,&black);
   return;
}

void draw_pixel(double px, double py)
{
   int            pxn, pyn;
   static int     flip = 0;
   
   pxn = int(px);
   pyn = int(py);
   
   if (pxn < 0 || pxn > dww-1) return;
   if (pyn < 0 || pyn > dhh-1) return;
   
   if (++flip > 2) flip = -3;
   if (flip < 0) gdk_gc_set_foreground(gdkgc,&red);
   else gdk_gc_set_foreground(gdkgc,&green);
   gdk_draw_point(drWin->window, gdkgc, pxn + Dorgx, pyn + Dorgy);
   
   return;
}


//  erase line. refresh line path from Drgb8 pixels.

void erase_line(int ix1, int iy1, int ix2, int iy2)
{
   void erase_pixel(double pxm, double pym);

   double      x1, y1, x2, y2;   
   double      pxm, pym, slope;
   
   if (! Drgb8) zappcrash("Drgb8 = 0");                                    //  v.10.3

   x1 = Mscale * (ix1-Iorgx);
   y1 = Mscale * (iy1-Iorgy);
   x2 = Mscale * (ix2-Iorgx);
   y2 = Mscale * (iy2-Iorgy);
   
   if (abs(y2 - y1) > abs(x2 - x1)) {
      slope = 1.0 * (x2 - x1) / (y2 - y1);
      if (y2 > y1) {
         for (pym = y1; pym <= y2; pym++) {
            pxm = x1 + slope * (pym - y1);
            erase_pixel(pxm,pym);
         }
      }
      else {
         for (pym = y1; pym >= y2; pym--) {
            pxm = x1 + slope * (pym - y1);
            erase_pixel(pxm,pym);
         }
      }
   }
   else {
      slope = 1.0 * (y2 - y1) / (x2 - x1);
      if (x2 > x1) {
         for (pxm = x1; pxm <= x2; pxm++) {
            pym = y1 + slope * (pxm - x1);
            erase_pixel(pxm,pym);
         }
      }
      else {
         for (pxm = x1; pxm >= x2; pxm--) {
            pym = y1 + slope * (pxm - x1);
            erase_pixel(pxm,pym);
         }
      }
   }

   return;
}

void erase_pixel(double px, double py)
{
   int            pxn, pyn;
   
   pxn = int(px);
   pyn = int(py);
   
   if (pxn < 0 || pxn > dww-1) return;
   if (pyn < 0 || pyn > dhh-1) return;

   uint8 *pixel = (uint8 *) Drgb8->bmp + (pyn * dww + pxn) * 3;
   gdk_draw_rgb_image(drWin->window, gdkgc, pxn + Dorgx, pyn + Dorgy, 
                                 1, 1, nodither, pixel, dww * 3);         
   return;
}


/**************************************************************************
   spline curve setup and edit functions                                   //  consolidated    v.9.5

   Initialize no. of curves:  Nspc
   Initialize vert/horz flag: vert[]
   Initialize anchor points:  nap[], apx[][], apy[][]
   Generate data for curve N:  curve_generate(N)
   Set up drawing functions:  curve_init(frame, func)
   Nsp curves will now be shown and edited inside frame.
   Callback func(N) will be called when curve N is edited.
   Get data for curve N: yval = curve_yval(N, xval)

***/

namespace  splinecurve                                                     //  up to 10 spline curves
{
   void     curve_init(GtkWidget *frame, void func(int));                  //  initialize spline curves
   int      curve_adjust(void *, GdkEventButton *event);                   //  curve editing function
   int      curve_draw();                                                  //  curve drawing function
   int      curve_generate(int spc);                                       //  generate data from anchor points
   inline 
   double   curve_yval(int spc, double xval);                              //  get curve y-value
   void     (*spcfunc)(int spc);                                           //  user callback function

   GtkWidget   *drawarea = 0;
   int         Nspc = 0;                                                   //  no. active spline curves
   int         vert[10];                                                   //  curve is vert. (1) or horz. (0)
   int         nap[10];                                                    //  no. anchor points per curve
   double      apx[10][50], apy[10][50];                                   //  up to 50 anchor points per curve
   double      yval[10][1000];                                             //  y-values for x = 0 to 1 by 0.001
};


//  initialize for spline curve editing
//  initial anchor points are pre-loaded into splinecurve variables above

void splinecurve::curve_init(GtkWidget *frame, void func(int))
{
   using namespace splinecurve;

   if (func) spcfunc = func;                                               //  set user callback function
   
   if (frame) {
      drawarea = gtk_drawing_area_new();                                   //  connect mouse edit function
      gtk_container_add(GTK_CONTAINER(frame),drawarea);                    //    to drawing area
      gtk_widget_add_events(drawarea,GDK_BUTTON_PRESS_MASK);
      gtk_widget_add_events(drawarea,GDK_BUTTON_RELEASE_MASK);
      gtk_widget_add_events(drawarea,GDK_BUTTON1_MOTION_MASK); 
      G_SIGNAL(drawarea,"motion-notify-event",curve_adjust,0)
      G_SIGNAL(drawarea,"button-press-event",curve_adjust,0)
      G_SIGNAL(drawarea,"expose-event",curve_draw,0)
   }
   
   return;
}


//  modify anchor points in curve using mouse

int splinecurve::curve_adjust(void *, GdkEventButton *event)
{
   using namespace splinecurve;

   int            ww, hh, kk;
   int            mx, my, button, evtype;
   static int     spc, ap, Fdrag = 0;                                      //  drag continuation logic   v.9.7
   int            minspc, minap;
   double         mxval, myval, cxval, cyval;
   double         dist2, mindist2 = 0;
   
   mx = int(event->x);                                                     //  mouse position in drawing area
   my = int(event->y);
   evtype = event->type;
   button = event->button;
   ww = drawarea->allocation.width;                                        //  drawing area size
   hh = drawarea->allocation.height;
   
   if (evtype == GDK_BUTTON_RELEASE) {
      Fdrag = 0;
      return 0;
   }
   
   if (mx < 0) mx = 0;                                                     //  limit edge excursions
   if (mx > ww) mx = ww;
   if (my < 0) my = 0;
   if (my > hh) my = hh;
   
   if (evtype == GDK_BUTTON_PRESS) Fdrag = 0;                              //  left or right click
   
   if (Fdrag)                                                              //  continuation of drag
   {
      if (vert[spc]) {
         mxval = 1.0 * my / hh;                                            //  mouse position in curve space
         myval = 1.0 * mx / ww;
      }
      else {
         mxval = 1.0 * mx / ww;
         myval = 1.0 * (hh - my) / hh;
      }

      if (ap < nap[spc]-1 && apx[spc][ap+1] - mxval < 0.05) return 0;      //  disallow < 0.05 from next
      if (ap > 0 && mxval - apx[spc][ap-1] < 0.05) return 0;               //    or prior anchor point 
   }
   
   else                                                                    //  mouse click or new drag begin
   {
      minspc = minap = -1;                                                 //  find closest curve/anchor point
      mindist2 = 999999;

      for (spc = 0; spc < Nspc; spc++)                                     //  loop curves
      {
         if (vert[spc]) {
            mxval = 1.0 * my / hh;                                         //  mouse position in curve space
            myval = 1.0 * mx / ww;
         }
         else {
            mxval = 1.0 * mx / ww;
            myval = 1.0 * (hh - my) / hh;
         }

         for (ap = 0; ap < nap[spc]; ap++)                                 //  loop anchor points
         {
            cxval = apx[spc][ap];
            cyval = apy[spc][ap];
            dist2 = (mxval-cxval)*(mxval-cxval) 
                  + (myval-cyval)*(myval-cyval);
            if (dist2 < mindist2) {
               mindist2 = dist2;                                           //  remember closest anchor point
               minspc = spc;
               minap = ap;
            }
         }
      }

      if (minspc < 0) return 0;                                            //  impossible
      spc = minspc;
      ap = minap;
   }
   
   if (evtype == GDK_BUTTON_PRESS && button == 3)                          //  right click, remove anchor point
   {
      if (sqrt(mindist2) > 0.05) return 0;                                 //  not close enough
      if (nap[spc] < 3) return 0;                                          //  < 2 anchor points would remain
      for (kk = ap; kk < nap[spc]-1; kk++) {
         apx[spc][kk] = apx[spc][kk+1];
         apy[spc][kk] = apy[spc][kk+1];
      }
      nap[spc]--;
      curve_generate(spc);                                                 //  regenerate data for modified curve
      curve_draw();                                                        //  regen and redraw all curves
      spcfunc(spc);                                                        //  call user function
      return 0;
   }

   if (! Fdrag)                                                            //  new drag or left click
   {
      if (vert[spc]) {
         mxval = 1.0 * my / hh;                                            //  mouse position in curve space
         myval = 1.0 * mx / ww;
      }
      else {
         mxval = 1.0 * mx / ww;
         myval = 1.0 * (hh - my) / hh;
      }

      if (sqrt(mindist2) < 0.05)                                           //  existing point close enough,
      {                                                                    //    move this anchor point to mouse
         if (ap < nap[spc]-1 && apx[spc][ap+1] - mxval < 0.05) return 0;   //  disallow < 0.05 from next
         if (ap > 0 && mxval - apx[spc][ap-1] < 0.05) return 0;            //    or prior anchor point 
      }
   
      else                                                                 //  none close, add new anchor point
      {
         minspc = -1;                                                      //  find closest curve to mouse
         mindist2 = 999999;

         for (spc = 0; spc < Nspc; spc++)                                  //  loop curves
         {
            if (vert[spc]) {
               mxval = 1.0 * my / hh;                                      //  mouse position in curve space
               myval = 1.0 * mx / ww;
            }
            else {
               mxval = 1.0 * mx / ww;
               myval = 1.0 * (hh - my) / hh;
            }

            cyval = curve_yval(spc,mxval);
            dist2 = fabs(myval - cyval);
            if (dist2 < mindist2) {
               mindist2 = dist2;                                           //  remember closest curve
               minspc = spc;
            }
         }

         if (minspc < 0) return 0;                                         //  impossible
         if (mindist2 > 0.05) return 0;                                    //  not close enough to any curve
         spc = minspc;

         if (nap[spc] > 49) {
            zmessageACK(GTK_WINDOW(mWin),ZTX("Exceed 50 anchor points"));
            return 0;
         }

         if (vert[spc]) {
            mxval = 1.0 * my / hh;                                         //  mouse position in curve space
            myval = 1.0 * mx / ww;
         }
         else {
            mxval = 1.0 * mx / ww;
            myval = 1.0 * (hh - my) / hh;
         }

         for (ap = 0; ap < nap[spc]; ap++)                                 //  find anchor point with next higher x
            if (mxval <= apx[spc][ap]) break;                              //    (ap may come out 0 or nap)

         if (ap < nap[spc] && apx[spc][ap] - mxval < 0.05) return 0;       //  disallow < 0.05 from next
         if (ap > 0 && mxval - apx[spc][ap-1] < 0.05) return 0;            //    or prior anchor point

         for (kk = nap[spc]; kk > ap; kk--) {                              //  make hole for new point
            apx[spc][kk] = apx[spc][kk-1];
            apy[spc][kk] = apy[spc][kk-1];
         }

         nap[spc]++;                                                       //  up point count
      }
   }

   apx[spc][ap] = mxval;                                                   //  new or moved anchor point
   apy[spc][ap] = myval;                                                   //    at mouse position

   curve_generate(spc);                                                    //  regenerate data for modified curve
   curve_draw();                                                           //  regen and redraw all curves
   spcfunc(spc);                                                           //  call user function
   
   if (evtype == GDK_MOTION_NOTIFY) Fdrag = 1;                             //  remember drag is underway
   return 0;
}


//  draw all curves based on current anchor points
//  (refresh all curves in drawing area)

int splinecurve::curve_draw()
{
   using namespace splinecurve;

   int         ww, hh, px, py, qx, qy, spc, ap;
   double      xval, yval;

   ww = drawarea->allocation.width;                                        //  drawing area size
   hh = drawarea->allocation.height;
   if (ww < 50 || hh < 50) return 0;

   gdk_window_clear(drawarea->window);                                     //  clear window
   
   for (spc = 0; spc < Nspc; spc++)
   {
      if (vert[spc])                                                       //  vert. curve
      {
         for (py = 0; py < hh; py++)                                       //  generate all points for curve
         {
            xval = 1.0 * py / hh;
            if (xval < apx[spc][0]) continue;
            if (xval > apx[spc][nap[spc]-1]) continue;
            yval = curve_yval(spc,xval);
            px = ww * yval + 0.49;                                         //  almost round - erratic FP in Intel CPUs
            gdk_draw_point(drawarea->window,gdkgc,px,py);                  //    causes "bumps" in a flat curve
         }
         
         for (ap = 0; ap < nap[spc]; ap++)                                 //  draw boxes at anchor points
         {
            xval = apx[spc][ap];
            yval = apy[spc][ap];
            px = ww * yval;
            py = hh * xval;
            for (qx = -2; qx < 3; qx++)
            for (qy = -2; qy < 3; qy++) {
               if (px+qx < 0 || px+qx >= ww) continue;
               if (py+qy < 0 || py+qy >= hh) continue;
               gdk_draw_point(drawarea->window,gdkgc,px+qx,py+qy);
            }
         }
      }
      else                                                                 //  horz. curve
      {
         for (px = 0; px < ww; px++)                                       //  generate all points for curve
         {
            xval = 1.0 * px / ww;
            if (xval < apx[spc][0]) continue;
            if (xval > apx[spc][nap[spc]-1]) continue;
            yval = curve_yval(spc,xval);
            py = hh - hh * yval + 0.49;                                    //  almost round - erratic FP in Intel CPUs
            gdk_draw_point(drawarea->window,gdkgc,px,py);                  //    causes "bumps" in a flat curve
         }
         
         for (ap = 0; ap < nap[spc]; ap++)                                 //  draw boxes at anchor points
         {
            xval = apx[spc][ap];
            yval = apy[spc][ap];
            px = ww * xval;
            py = hh - hh * yval;
            for (qx = -2; qx < 3; qx++)
            for (qy = -2; qy < 3; qy++) {
               if (px+qx < 0 || px+qx >= ww) continue;
               if (py+qy < 0 || py+qy >= hh) continue;
               gdk_draw_point(drawarea->window,gdkgc,px+qx,py+qy);
            }
         }
      }
   }

   return 0;
}


//  generate all curve data points when anchor points are modified

int splinecurve::curve_generate(int spc)
{
   using namespace splinecurve;

   int      kk, kklo, kkhi;
   double   xval, yvalx;

   spline1(nap[spc],apx[spc],apy[spc]);                                    //  compute curve fitting anchor points

   kklo = 1000 * apx[spc][0] - 30;                                         //  xval range = anchor point range
   if (kklo < 0) kklo = 0;                                                 //    + 0.03 extra below/above      v.9.5
   kkhi = 1000 * apx[spc][nap[spc]-1] + 30;
   if (kkhi > 1000) kkhi = 1000;

   for (kk = 0; kk < 1000; kk++)                                           //  generate all points for curve
   {
      xval = 0.001 * kk;
      if (kk < kklo || kk > kkhi) yvalx = 0;
      else yvalx = spline2(xval);
      if (yvalx < 0) yvalx = 0;                                            //  yval < 0 not allowed, > 1 OK    v.9.5
      yval[spc][kk] = yvalx;
   }

   return 0;
}


//  retrieve curve data using interpolation of saved table of values

inline double splinecurve::curve_yval(int spc, double xval)
{
   using namespace splinecurve;

   int      ii;
   double   x1, x2, y1, y2, y3;
   
   if (xval <= 0) return yval[spc][0];
   if (xval >= 0.999) return yval[spc][999];
   
   x2 = 1000.0 * xval;
   ii = int(x2);
   x1 = ii;
   y1 = yval[spc][ii];
   y2 = yval[spc][ii+1];
   y3 = y1 + (y2 - y1) * (x2 - x1);
   return y3;
}


/**************************************************************************
      begin menu functions
***************************************************************************/

//  process top-level menu entry

void topmenufunc(GtkWidget *, cchar *menu)
{
   topmenu = (char *) menu;                                                //  remember top-level menu in
   return;                                                                 //    case this is needed somewhere
}


/**************************************************************************
      file menu functions
***************************************************************************/

//  display image gallery (thumbnails) in a separate window

void m_gallery(GtkWidget *, cchar *)
{
   if (image_file) image_gallery(image_file,"paint1",0,m_gallery2);        //  show image gallery window
   else {
      char *pp = get_current_dir_name();                                   //  initz. gallery file list and show
      if (pp) {
         image_gallery(pp,"init",0,m_gallery2);
         image_gallery(0,"paint1");
         free(pp);
         Fsearchlist = 0;
      }
   }

   return;
}


//  gallery clicked thumbnail will call this function

void  m_gallery2(char *file)
{
   if (zdburn) {                                                           //  add file for CD burn  v.7.2
      burn_insert_file(file);
      return;
   }
   
   if (strEqu(file,"F1")) {                                                //  v.9.0
      showz_userguide("navigation");
      return;
   }

   f_open((cchar *) file);
   return;
}


/**************************************************************************/

//  open file menu function

void m_open(GtkWidget *, cchar *)
{
   Fsearchlist = 0;
   f_open(null);
   return;
}


/**************************************************************************/

//  open drag-drop file

void  m_open_drag(int x, int y, char *file)                                //  v.7.3
{
   Fsearchlist = 0;                                                        //  v.8.1
   f_open(file);
   zfree(file);
   return;
}


/**************************************************************************/

//  open an image file from the list of recent image files

char     recent_selection[300];

void  m_recent(GtkWidget *, cchar *)                                       //  v.8.1
{
   int  recent_dialog_event(zdialog *zd, cchar *event);

   zdialog     *zd;
   int         ii, zstat;
   
   if (! recentfiles[0]) return;

   zd = zdialog_new(ZTX("Open Recent File"),mWin,Bcancel,null);
   zdialog_add_widget(zd,"combo","combo","dialog",0);

   for (ii = 0; ii < Nrecentfiles && recentfiles[ii]; ii++)                //  stuff files into combo box list
      zdialog_cb_app(zd,"combo",recentfiles[ii]);
   zdialog_stuff(zd,"combo",recentfiles[0]);

   *recent_selection = 0;
   zdialog_run(zd,recent_dialog_event);                                    //  run dialog
   zstat = zdialog_wait(zd);
   zdialog_free(zd);

   if (zstat == 1 && *recent_selection == '/') 
      f_open(recent_selection);                                            //  open selected file
   return;
}

int  recent_dialog_event(zdialog *zd, cchar *event)                        //  dialog event function  v.8.2
{
   if (strNeq(event,"combo")) return 0;
   zdialog_fetch(zd,"combo",recent_selection,299);                         //  get user selection
   zdialog_send_response(zd,1);                                            //  complete dialog with OK status
   return 1;
}


/**************************************************************************/

//  add a file to the list of recent files                                 //  v.8.4

void  add_recent_file(cchar *file)
{
   int      ii;

   for (ii = 0; ii < Nrecentfiles-1 && recentfiles[ii]; ii++)              //  find file in recent list   v.8.1
      if (strEqu(file,recentfiles[ii])) break;                             //    (or find last entry in list)
   if (recentfiles[ii]) zfree(recentfiles[ii]);                            //  free this slot in list
   for (; ii > 0; ii--) recentfiles[ii] = recentfiles[ii-1];               //  move list down to fill hole
   recentfiles[0] = strdupz(file,0,"recentfile");                          //  current file >> first in list
   return;
}


/**************************************************************************/

//  open a file and initialize RGB bitmap

void f_open(cchar *filespec)
{
   int         cc, fposn, fcount;
   char        *pp, wtitle[200], fname[100], fdirk[100];
   RGB         *temp8;
   char        **orientation;
   cchar       *orientationkey[1] = { exif_orientation_key };
   
   zmondirk("close",0,0);                                                  //  reset monitored directory

   if (! menulock(1)) return;                                              //  lock menu
   if (mod_keep()) goto openfail;
   
   zfuncs::F1_help_topic = "open_image_file";                              //  v.9.0
   
   if (filespec) filespec = strdupz(filespec,0,"imagefile");
   else filespec = zgetfile(ZTX("Open Image File"),image_file,"open");
   if (! filespec) goto openfail;
   if (strlen(filespec) >= maxfcc) {                                       //  disallow humongous files     v.10.0
      zmessageACK(GTK_WINDOW(mWin),"filespec > %d bytes",maxfcc);
      goto openfail;
   }
   
   temp8 = f_load(filespec,8);                                             //  load image as RGB-8 pixmap
   if (! temp8) goto openfail;

   free_resources();                                                       //  free resources for old image file

   mutex_lock(&pixmaps_lock);                                              //  lock pixmaps

   image_file = (char *) filespec;                                         //  setup new image file
   Frgb8 = temp8;
   Fww = Frgb8->ww;
   Fhh = Frgb8->hh;
   
   pp = (char *) strrchr(image_file,'/');                                  //  get image file name
   strncpy0(fname,pp+1,99);
   cc = pp - image_file;
   if (cc < 99) strncpy0(fdirk,image_file,cc+2);                           //  get dirk/path/ if short enough
   else {
      strncpy(fdirk,image_file,96);                                        //  or use /dirk/path...
      strcpy(fdirk+95,"...");
   }

   Fzoom = 0;                                                              //  zoom level = fit window
   zoomx = zoomy = 0;                                                      //  no zoom center      v.7.5

   mutex_unlock(&pixmaps_lock);                                            //  unlock pixmaps

   if (zdedittags) m_edittags(0,0);                                        //  update active tags dialog
   if (zdexifview) m_exif_view(0,0);                                       //  update active EXIF view window  v.10.2
   if (zdrename) rename_dialog();                                          //  update active rename dialog

   Fimageturned = 0;
   orientation = exif_get(image_file,orientationkey,1);                    //  upright turned image
   if (*orientation) {
      if (strstr(*orientation,"90")) turn_image(90);
      if (strstr(*orientation,"270")) turn_image(270);
      zfree(*orientation);                                                 //  memory leak v.10.2
   }

   pp = image_gallery(image_file,"find",0);                                //  file in current image gallery ?
   if (! pp) {
      image_gallery(image_file,"init");                                    //  no, reset gallery file list
      image_gallery(0,"paint2");                                           //  refresh gallery window if active
      Fsearchlist = 0;
   }
   else zfree(pp);

   image_position(image_file,fposn,fcount);                                //  position and count in gallery list
   add_recent_file(image_file);                                            //  first in recent files list

   snprintf(wtitle,199,"%s  %d/%d  %s",fname,fposn,fcount,fdirk);          //  set window title
   gtk_window_set_title(GTK_WINDOW(mWin),wtitle);
   gtk_window_present(GTK_WINDOW(mWin));                                   //  bring main window to front

   mwpaint2();                                                             //  refresh main window

   if (! Fsearchlist) {                                                    //  start monitoring this directory
      pp = (char *) strrchr(image_file,'/');                               //    if not from search tags
      *pp = 0;                                                             //                v.6.4
      zmondirk("open",image_file,0);
      *pp = '/';
   }
   
openfail:
   menulock(0);
   mwpaint2();
   return;
}


/**************************************************************************/

//  open previous or next file in same gallery as last file opened

void m_prev(GtkWidget *, cchar *)
{
   if (! image_file) return;

   int mods = 0;
   while (zmondirk("event",0,0) > 0) mods++;                               //  detect directory mods  v.6.4
   if (mods) image_gallery(image_file,"init");                             //  refresh gallery file list

   char *pp = image_gallery(image_file,"prev");
   if (! pp) return;
   if (image_file_type(pp) == 2) f_open(pp);
   zfree(pp);
   return;
}

void m_next(GtkWidget *, cchar *)
{
   if (! image_file) return;

   int mods = 0;
   while (zmondirk("event",0,0) > 0) mods++;                               //  v.6.4
   if (mods) image_gallery(image_file,"init");

   char *pp = image_gallery(image_file,"next");
   if (! pp) return;
   if (image_file_type(pp) == 2) f_open(pp);
   zfree(pp);
   return;
}


/**************************************************************************/

//  save (modified) image to same file - no confirmation of overwrite.

void m_save(GtkWidget *, cchar *)                                          //  v.8.3
{   
   char        *outfile, *pext;
   cchar       *format;
   
   if (! image_file) return;
   
   format = "jpeg";                                                        //  use jpeg unless png or tiff
   if (strEqu(file_type,"png")) format = "png";                            //  png file
   if (strEqu(file_type,"tiff")) format = "tiff-8";                        //  8 or 16 bits/color tiff
   if (file_bpc == 16) format = "tiff-16";

   strcpy(jpeg_quality,def_jpeg_quality);                                  //  default jpeg save quality

   outfile = strdupz(image_file,8,"imagefile");                            //  use input file name

   pext = (char *) strrchr(outfile,'/');                                   //  force compatible file extension
   if (pext) pext = (char *) strrchr(pext,'.');                            //    if not already
   if (! pext) pext = outfile + strlen(outfile);
   if (strEqu(format,"jpeg") && ! strcmpv(pext,".jpg",".JPG",".jpeg",".JPEG",null))
      strcpy(pext,".jpeg");
   if (strEqu(format,"png") && ! strcmpv(pext,".png",".PNG",null))
      strcpy(pext,".png");
   if (strnEqu(format,"tiff",4) && ! strcmpv(pext,".tif",".TIF",".tiff",".TIFF",null))
      strcpy(pext,".tiff");

   f_save(outfile,format);
   zfree(outfile);
   return;
}


/**************************************************************************/

//  save (modified) image to new file, confirm if overwrite existing file.

GtkWidget   *saveas_fchooser;

void m_saveas(GtkWidget *, cchar *)
{
   void  saveas_radiobutt(void *, int button);

   GtkWidget      *fdialog, *hbox;
   GtkWidget      *label1, *tiff8, *tiff16, *jpeg, *png, *jqlab, *jqval;
   char           *outfile = 0, *outfile2 = 0, *pext;
   cchar          *format;
   int            ii, err, yn, status;
   struct stat    fstat;

   if (! image_file) return;

   fdialog = gtk_dialog_new_with_buttons(ZTX("Save File"),                 //  build file save dialog
                           GTK_WINDOW(mWin), GTK_DIALOG_MODAL,
                           GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL, 
                           GTK_STOCK_SAVE, GTK_RESPONSE_ACCEPT, null);
   gtk_window_set_default_size(GTK_WINDOW(fdialog),600,500);

   saveas_fchooser = gtk_file_chooser_widget_new(GTK_FILE_CHOOSER_ACTION_SAVE);
   gtk_container_add(GTK_CONTAINER(GTK_DIALOG(fdialog)->vbox),saveas_fchooser);
   gtk_file_chooser_set_filename(GTK_FILE_CHOOSER(saveas_fchooser),image_file);
   
   hbox = gtk_hbox_new(0,0);
   gtk_container_add(GTK_CONTAINER(GTK_DIALOG(fdialog)->vbox),hbox);
   gtk_box_set_child_packing(GTK_BOX(GTK_DIALOG(fdialog)->vbox),hbox,0,0,10,GTK_PACK_END);

   label1 = gtk_label_new("file type");                                    //  add file type options
   tiff8 = gtk_radio_button_new_with_label(null,"tiff-8");
   tiff16 = gtk_radio_button_new_with_label_from_widget(GTK_RADIO_BUTTON(tiff8),"tiff-16");
   png = gtk_radio_button_new_with_label_from_widget(GTK_RADIO_BUTTON(tiff8),"png");
   jpeg = gtk_radio_button_new_with_label_from_widget(GTK_RADIO_BUTTON(tiff8),"jpeg");
   jqlab = gtk_label_new(ZTX("jpeg quality"));
   jqval = gtk_entry_new();

   gtk_entry_set_width_chars(GTK_ENTRY(jqval),3);
   gtk_box_pack_start(GTK_BOX(hbox),label1,0,0,5);                         //  file type  (o) tiff8  (o) tiff16 ...
   gtk_box_pack_start(GTK_BOX(hbox),tiff8,0,0,5);
   gtk_box_pack_start(GTK_BOX(hbox),tiff16,0,0,5);
   gtk_box_pack_start(GTK_BOX(hbox),png,0,0,5);
   gtk_box_pack_start(GTK_BOX(hbox),jpeg,0,0,10);
   gtk_box_pack_start(GTK_BOX(hbox),jqlab,0,0,5);                          //  ... jpeg quality [___]
   gtk_box_pack_start(GTK_BOX(hbox),jqval,0,0,5);
   
   G_SIGNAL(tiff8,"pressed",saveas_radiobutt,0);                           //  connect file type radio buttons
   G_SIGNAL(tiff16,"pressed",saveas_radiobutt,1);                          //    to handler function
   G_SIGNAL(png,"pressed",saveas_radiobutt,2);
   G_SIGNAL(jpeg,"pressed",saveas_radiobutt,3);

   gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(jpeg),1);                //  set default file type = jpeg
   gtk_entry_set_text(GTK_ENTRY(jqval),def_jpeg_quality);                  //  default jpeg save quality
   
   if (strEqu(file_type,"png"))                                            //  default matches file type
      gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(png),1);              //    if png or tiff
   if (strEqu(file_type,"tiff"))
      gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(tiff8),1);
   if (file_bpc == 16)
      gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(tiff16),1);

dialog_run:

   gtk_widget_show_all(fdialog);                                           //  run dialog
   status = gtk_dialog_run(GTK_DIALOG(fdialog));
   if (status != GTK_RESPONSE_ACCEPT) {                                    //  user cancelled
      gtk_widget_destroy(fdialog);
      return;
   }

   outfile2 = gtk_file_chooser_get_filename(GTK_FILE_CHOOSER(saveas_fchooser));
   if (! outfile2) goto dialog_run;
   outfile = strdupz(outfile2,8,"imagefile");
   g_free(outfile2);

   format = "jpeg";
   if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(png)))
      format = "png";
   if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(tiff8)))
      format = "tiff-8";
   if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(tiff16)))
      format = "tiff-16";

   if (strEqu(format,"jpeg")) {                                            //  get save quality
      ii = atoi(gtk_entry_get_text(GTK_ENTRY(jqval)));
      if (ii < 1 || ii > 100) {
         zmessageACK(GTK_WINDOW(mWin),ZTX("jpeg quality must be 1-100"));
         goto dialog_run;
      }
      sprintf(jpeg_quality,"%d",ii);
   }

   gtk_widget_destroy(fdialog);                                            //  kill dialog

   pext = strrchr(outfile,'/');                                            //  force compatible file extension
   if (pext) pext = strrchr(pext,'.');                                     //    if not already
   if (! pext) pext = outfile + strlen(outfile);
   if (strEqu(format,"jpeg") && ! strcmpv(pext,".jpg",".JPG",".jpeg",".JPEG",null))
      strcpy(pext,".jpeg");
   if (strEqu(format,"png") && ! strcmpv(pext,".png",".PNG",null))
      strcpy(pext,".png");
   if (strnEqu(format,"tiff",4) && ! strcmpv(pext,".tif",".TIF",".tiff",".TIFF",null))
      strcpy(pext,".tiff");

   err = stat(outfile,&fstat);                                             //  check if file exists
   if (! err) {                                                            //  confirm overwrite
      yn = zmessageYN(GTK_WINDOW(mWin),ZTX("Overwrite file? \n %s"),outfile);
      if (! yn) {
         zfree(outfile);
         return;
      }
   }
   
   f_save(outfile,format);
   zfree(outfile);
   return;
}


//  set dialog file type from user selection of file type radio button

void saveas_radiobutt(void *, int button)                                  //  v.9.4
{
   cchar    *filetypes[4] = { ".tiff", ".tiff", ".png", ".jpeg" };
   cchar    *filespec1;
   char     *filespec2, *pp;

   filespec1 = gtk_file_chooser_get_filename(GTK_FILE_CHOOSER(saveas_fchooser));
   if (! filespec1) return;
   filespec2 = strdupz(filespec1,6,"filespec2");
   pp = strrchr(filespec2,'.');
   if (! pp) pp = filespec2 + strlen(filespec2);
   strcpy(pp,filetypes[button]);
   gtk_file_chooser_set_current_name(GTK_FILE_CHOOSER(saveas_fchooser),filespec2);        //  v.9.5
   gtk_widget_show_all(saveas_fchooser);
   zfree(filespec2);
   return;
}


/**************************************************************************/

//  print current image file
//  do not use f_save() (not valid from here) and simplify.                //  v.10.2

void m_print(GtkWidget *, cchar *)                                         //  use GTK print   v.9.4
{
   int      err;
   char     *printfile;

   if (! image_file) return;                                               //  no image file
   
   printfile = strdupz(get_zuserdir(),20,"printfile");                     //  make temp print file:
   strcat(printfile,"/printfile.jpeg");                                    //    ~/.fotoxx/printfile.jpeg

   set_cursor(busycursor);                                                 //  set function busy cursor  v.8.4
   
   if (Frgb16) err = PXBwrite(Frgb16,printfile);                           //  write image to print file
   else  err = PXBwrite(Frgb8,printfile);

   set_cursor(0);                                                          //  restore normal cursor

   if (err) {
      zfree(printfile);                                                    //  v.10.3
      return;
   }
   
   print_imagefile(printfile);                                             //  GTK print utility in zfuncs.cpp
   zfree(printfile);
   return;
}


/**************************************************************************/

//  Delete image file - move image_file to trash.
//  Trash has no standard location, so use a trash folder on the desktop.
//  User must delete or move to the official trash bin.

void m_trash(GtkWidget *, cchar *)
{
   int            err, yn;
   char           trashdir[200];
   struct stat    trstat;

   if (! image_file) return;                                               //  nothing to trash
   
   err = stat(image_file,&trstat);                                         //  get file status
   if (err) {
      zmessLogACK(GTK_WINDOW(mWin),strerror(errno));
      return;
   }

   if (! (trstat.st_mode & S_IWUSR)) {                                     //  check permission
      yn = zmessageYN(GTK_WINDOW(mWin),ZTX("Move read-only file to trash?"));
      if (! yn) return;
      trstat.st_mode |= S_IWUSR;
      chmod(image_file,trstat.st_mode);
   }
   
   snprintf(trashdir,199,"%s/%s",getenv("HOME"),ftrash);                   //  fotoxx trash filespec
   
   trstat.st_mode = 0;
   err = stat(trashdir,&trstat);
   if (! S_ISDIR(trstat.st_mode)) {
      snprintf(command,ccc,"mkdir -m 0750 \"%s\"",trashdir);
      err = system(command);
      if (err) {
         zmessLogACK(GTK_WINDOW(mWin),ZTX("Cannot create trash folder: %s"),wstrerror(err));
         return;
      }
   }

   snprintf(command,ccc,"cp \"%s\" \"%s\" ",image_file,trashdir);          //  copy image file to trash
   err = system(command);
   if (err) {
      zmessLogACK(GTK_WINDOW(mWin),ZTX("error: %s"),wstrerror(err));
      return;
   }

   snprintf(command,ccc,"rm \"%s\"",image_file);                           //  delete image file
   err = system(command);
   if (err) {
      zmessLogACK(GTK_WINDOW(mWin),ZTX("error: %s"),wstrerror(err));
      return;
   }
   
   update_asstags(image_file,1);                                           //  delete in tags index file
   if (! Fsearchlist) image_gallery(image_file,"init");                    //  reset image gallery file list
   image_gallery(0,"paint2");                                              //  refresh gallery window if active
   m_next(0,0);                                                            //  step to next file if there

   return;
}


/**************************************************************************/

//  rename menu function

char     rename_old[100] = "";
char     rename_new[100] = "";

void  m_rename(GtkWidget *, cchar *)
{
   rename_dialog();                                                        //  activate rename dialog
   return;
}
   

//  activate rename dialog, stuff data from current file

void rename_dialog()
{
   int rename_dialog_event(zdialog *zd, cchar *event);
   
   char     *pdir, *pfile, *pext;

   if (! image_file) return;

   if (! zdrename)                                                         //  restart dialog
   {
      zdrename = zdialog_new(ZTX("Rename Image File"),mWin,Bcancel,null);
      zdialog_add_widget(zdrename,"hbox","hb1","dialog",0,"space=10");
      zdialog_add_widget(zdrename,"vbox","vb1","hb1",0,"homog|space=5");
      zdialog_add_widget(zdrename,"vbox","vb2","hb1",0,"homog|expand");

      zdialog_add_widget(zdrename,"button","Bold","vb1",ZTX("old name"));
      zdialog_add_widget(zdrename,"button","Bnew","vb1",ZTX("rename to"));
      zdialog_add_widget(zdrename,"button","Bprev","vb1",ZTX("previous"));

      zdialog_add_widget(zdrename,"hbox","hb21","vb2",0);                  //  [ old name ] [ oldname  ]
      zdialog_add_widget(zdrename,"hbox","hb22","vb2",0);                  //  [ new name ] [ newname  ] [+1]
      zdialog_add_widget(zdrename,"hbox","hb23","vb2",0);                  //  [ previous ] [ prevname ]

      zdialog_add_widget(zdrename,"label","Lold","hb21");
      zdialog_add_widget(zdrename,"entry","Enew","hb22",0,"expand|scc=30");
      zdialog_add_widget(zdrename,"button","B+1","hb22"," +1 ","space=5");
      zdialog_add_widget(zdrename,"label","Lprev","hb23");

      zdialog_run(zdrename,rename_dialog_event);                           //  run dialog
   }

   parsefile(image_file,&pdir,&pfile,&pext);
   strncpy0(rename_old,pfile,99);
   strncpy0(rename_new,pfile,99);
   zdialog_stuff(zdrename,"Lold",rename_old);                              //  current file name
   zdialog_stuff(zdrename,"Enew",rename_new);                              //  entered file name

   return;
}


//  dialog event and completion callback function

int rename_dialog_event(zdialog *zd, cchar *event)
{
   char           *pp, *pdir, *pfile, *pext, *pnew;
   int            nseq, digits, ccp, ccn, ccx, err;
   struct stat    statb;
   
   if (zd->zstat) {                                                        //  complete
      zdialog_free(zd);                                                    //  kill dialog
      zdrename = 0;
      return 0;
   }
   
   if (strEqu(event,"Bold"))                                               //  reset to current file name
      zdialog_stuff(zd,"Enew",rename_old);

   if (strEqu(event,"Bprev")) {                                            //  previous name >> new name
      zdialog_fetch(zd,"Lprev",rename_new,99);
      zdialog_stuff(zd,"Enew",rename_new);
   }

   if (strEqu(event,"B+1"))                                                //  increment sequence number
   {
      zdialog_fetch(zd,"Enew",rename_new,94);                              //  get entered filename
      pp = rename_new + strlen(rename_new);
      digits = 0;
      while (pp[-1] >= '0' && pp[-1] <= '9') {
         pp--;                                                             //  look for NNN in filenameNNN
         digits++;
      }
      nseq = 1 + atoi(pp);                                                 //  NNN + 1
      if (nseq > 9999) nseq = 0;
      if (digits < 2) digits = 2;                                          //  keep digit count if enough
      if (nseq > 99 && digits < 3) digits = 3;                             //  use leading zeros
      if (nseq > 999 && digits < 4) digits = 4;
      snprintf(pp,digits+1,"%0*d",digits,nseq);
      zdialog_stuff(zd,"Enew",rename_new);
   }

   if (strEqu(event,"Bnew")) 
   {
      parsefile(image_file,&pdir,&pfile,&pext);                            //  existing /directories/file.ext

      zdialog_fetch(zd,"Enew",rename_new,94);                              //  new file name from user

      ccp = strlen(pdir);                                                  //  length of /directories/
      ccn = strlen(rename_new);                                            //  length of file
      if (pext) ccx = strlen(pext);                                        //  length of .ext
      else ccx = 0;

      pnew = zmalloc(ccp + ccn + ccx + 1,"newfile");                       //  put it all together
      strncpy(pnew,image_file,ccp);                                        //   /directories/file.ext
      strcpy(pnew+ccp,rename_new);
      if (ccx) strcpy(pnew+ccp+ccn,pext);
      
      err = stat(pnew,&statb);                                             //  check if new name exists
      if (! err) {
         zmessageACK(GTK_WINDOW(mWin),ZTX("The target file already exists"));
         zfree(pnew);
         return 0;
      }
      
      snprintf(command,ccc,"cp -p \"%s\" \"%s\"",image_file,pnew);         //  copy to new file   -p v.10.3
      err = system(command);
      if (err) {
         zmessageACK(GTK_WINDOW(mWin),ZTX("Rename failed \n %s"),wstrerror(err));
         zfree(pnew);
         return 0;
      }

      zdialog_stuff(zd,"Lprev",rename_new);                                //  set previous name in dialog

      load_filetags(pnew);                                                 //  update tags index file
      update_asstags(pnew);
      zfree(pnew);

      pnew = strdupz(image_file,0,"imagefile");                            //  save file name to be deleted
      update_asstags(pnew,1);                                              //  delete in tags index file   v.9.7

      m_next(0,0);                                                         //  move to image_file + 1
      snprintf(command,ccc,"rm \"%s\"",pnew);                              //  delete old file
      err = system(command);
      zfree(pnew);

      if (! Fsearchlist) image_gallery(image_file,"init");                 //  update image gallery file list
      image_gallery(0,"paint2");                                           //  refresh gallery window if active
      gtk_window_present(GTK_WINDOW(mWin));                                //  bring main to foreground
   }
   
   if (strEqu(event,"F1")) showz_userguide("rename");                      //  F1 help          v.9.0

   return 0;
}


/**************************************************************************/

//  menu function - rename many files at once

char     **massrename_filelist = 0;
int      massrename_filecount = 0;

void m_massrename(GtkWidget *, cchar *)                                    //  new v.9.7
{
   int   massrename_dialog_event(zdialog *zd, cchar *event);

   zdialog     *zdmassrename;   
   cchar       *Brename = ZTX("rename files");

   if (mod_keep()) return;

   zdmassrename = zdialog_new(ZTX("Mass Rename"),mWin,Brename,Bcancel,null);

   zdialog_add_widget(zdmassrename,"hbox","hb1","dialog",0,"space=5");
   zdialog_add_widget(zdmassrename,"button","files","hb1",ZTX("select files"),"space=10");
   zdialog_add_widget(zdmassrename,"label","labcount","hb1","0 files selected","space=10");
   zdialog_add_widget(zdmassrename,"hbox","hb2","dialog","space=5");
   zdialog_add_widget(zdmassrename,"label","lab2","hb2",ZTX("new base name"),"space=10");
   zdialog_add_widget(zdmassrename,"entry","basename","hb2");
   zdialog_add_widget(zdmassrename,"hbox","hb3","dialog",0,"space=5");
   zdialog_add_widget(zdmassrename,"label","lab31","hb3",ZTX("starting sequence"),"space=10");
   zdialog_add_widget(zdmassrename,"entry","sequence","hb3","100","scc=5");
   zdialog_add_widget(zdmassrename,"label","lab32","hb3",ZTX("increment"),"space=10");
   zdialog_add_widget(zdmassrename,"entry","increment","hb3","01","scc=3");

   massrename_filelist = 0;
   massrename_filecount = 0;
   
   zdialog_run(zdmassrename,massrename_dialog_event);
   zdialog_wait(zdmassrename);                                             //  wait for dialog completion
   return;
}


//  dialog event and completion callback function

int massrename_dialog_event(zdialog *zd, cchar *event)
{
   char           **flist = massrename_filelist, countmess[32];
   cchar          *selectmess = ZTX("select files to rename");
   int            ii, err, cc, ccp, ccn, ccx;
   int            sequence, increment, adder;
   char           basename[100], filename[120], *oldfile, *newfile;
   char           *pdir, *pfile, *pext;
   cchar          *errmess = ZTX("base name / sequence / increment not reasonable");
   struct stat    statb;
   
   if (strEqu(event,"F1")) showz_userguide("mass_rename");                 //  F1 help

   if (strEqu(event,"files"))                                              //  select images to rename
   {
      if (flist) {                                                         //  free prior list
         for (ii = 0; flist[ii]; ii++) 
            zfree(flist[ii]);
         zfree(flist);
      }

      flist = zgetfiles(selectmess,image_file);                            //  get file list from user
      massrename_filelist = flist;

      if (flist)                                                           //  count files in list
         for (ii = 0; flist[ii]; ii++);
      else ii = 0;
      massrename_filecount = ii;

      sprintf(countmess,"%d files selected",massrename_filecount);
      zdialog_stuff(zd,"labcount",countmess);
   }
   
   if (! zd->zstat) return 0;
   if (zd->zstat != 1) goto cleanup;                                       //  dialog completed
   if (! massrename_filecount) goto cleanup;

   zdialog_fetch(zd,"basename",basename,99);
   zdialog_fetch(zd,"sequence",sequence);
   zdialog_fetch(zd,"increment",increment);

   if (strlen(basename) < 2 || sequence < 1 || increment < 1) {
      zmessageACK(GTK_WINDOW(mWin),errmess);
      return 0;
   }
   
   write_popup_text("open","Renaming files",500,200,mWin);                 //  status monitor popup window    v.10.3

   for (ii = 0; flist[ii]; ii++)
   {
      oldfile = flist[ii];
      parsefile(oldfile,&pdir,&pfile,&pext);
      ccp = strlen(pdir);
      if (pext) ccx = strlen(pext);
      else ccx = 0;

      adder = sequence + ii * increment;
      snprintf(filename,119,"%s-%d",basename,adder);
      ccn = strlen(filename);

      newfile = zmalloc(ccp + ccn + ccx + 1,"newfile");                    //  construct /path/filename.ext
      strcpy(newfile,pdir);
      strcpy(newfile+ccp,filename);
      if (ccx) strcpy(newfile+ccp+ccn,pext);

      err = stat(newfile,&statb);
      if (! err) {
         snprintf(command,ccc,"%s %s",ZTX("new file already exists:"),newfile);
         write_popup_text("write",command);
         zfree(newfile);
         break;
      }

      cc = snprintf(command,ccc,"cp -p \"%s\" \"%s\"",oldfile,newfile);    //  copy to new file   -p v.10.3
      if (cc >= maxfcc*2) {
         snprintf(command,ccc,"%s %s",ZTX("filespec too long:"),oldfile);
         write_popup_text("write",command);
         zfree(newfile);
         break;
      }

      write_popup_text("write",command);

      err = system(command);
      if (err) {
         snprintf(command,ccc,"%s %s",ZTX("Rename failed:"),wstrerror(err));
         zfree(newfile);
         break;
      }

      load_filetags(newfile);                                              //  update tags index file
      update_asstags(newfile);
      zfree(newfile);

      snprintf(command,ccc,"rm \"%s\"",oldfile);                           //  delete old file
      err = system(command);
      update_asstags(oldfile,1);                                           //  remove from tags index
   }

   write_popup_text("write","COMPLETED");

cleanup:
   zdialog_free(zd);
   if (massrename_filecount) {
      for (ii = 0; flist[ii]; ii++) 
         zfree(flist[ii]);
      zfree(flist);
   }
   return 0;
}


/**************************************************************************/

//  forced quit - can cause running function to crash

void m_quit(GtkWidget *, cchar *)
{
   if (image_file) update_filetags(image_file);                            //  commit tag changes, if any
   if (mod_keep()) return;                                                 //  keep or discard pending changes
   printf("quit \n");
   Fshutdown++;
   save_fotoxx_state();                                                    //  save state for next session
   free_resources();                                                       //  delete temp files
   gtk_main_quit();                                                        //  gone forever
   return;
}


/**************************************************************************
      tools menu functions
***************************************************************************/

//  set new image zoom level or magnification

void m_zoom(GtkWidget *, cchar *menu)
{
   int      ii, iww, ihh, Dww, Dhh;
   char     zoom;
   double   scalew, scaleh, fitscale;
   double   scales[9] = { 0.125, 0.176, 0.25, 0.354, 0.5, 0.71, 1.0, 1.41, 2.0 };
   
   if (strnEqu(menu,"Zoom",4)) zoom = menu[4];                             //  get + or -
   else  zoom = *menu;
   
   Dww = drWin->allocation.width;                                          //  drawing window size
   Dhh = drWin->allocation.height;
   
   if (E3rgb16) {                                                          //  bugfix  v.8.1
      iww = E3ww;
      ihh = E3hh;
   }
   else  {
      iww = Fww;
      ihh = Fhh;
   }

   if (iww > Dww || ihh > Dhh) {                                           //  get window fit scale
      scalew = 1.0 * Dww / iww;
      scaleh = 1.0 * Dhh / ihh;
      if (scalew < scaleh) fitscale = scalew;
      else fitscale = scaleh;
   }
   else fitscale = 1.0;                                                    //  if image < window use 100%
   
   if (zoom == '+') {                                                      //  zoom bigger
      if (! Fzoom) Fzoom = fitscale / 1.2;
      Fzoom = Fzoom * sqrt(2.0);                                           //  new scale: 41% bigger
      for (ii = 0; ii < 9; ii++)
         if (Fzoom < 1.01 * scales[ii]) break;                             //  next higher scale in table
      if (ii == 9) ii = 8;
      Fzoom = scales[ii];
      if (Fzoom < fitscale) Fzoom = 0;                                     //  image < window
   }

   if (zoom == '-') Fzoom = 0;                                             //  zoom to fit window

   if (zoom == 'Z') {
      if (Fzoom != 0) Fzoom = 0;                                           //  toggle 100% and fit window
      else  Fzoom = 1;
   }
   
   if (! Fzoom) zoomx = zoomy = 0;                                         //  no req. zoom center    v.7.5
   
   mwpaint2();                                                             //  refresh window
   return;
}


/**************************************************************************/

//  monitor test function

void m_montest(GtkWidget *, cchar *)
{
   uint8       *pixel;
   int         red, green, blue;
   int         row, col, row1, row2;
   int         ww = 800, hh = 500;
   
   if (mod_keep()) return;
   if (! menulock(1)) return;
   
   mutex_lock(&pixmaps_lock);

   RGB_free(Frgb8);
   Frgb8 = RGB_make(ww,hh,8);
   Fww = ww;
   Fhh = hh;
   file_bpc = 8;
   file_MB = 0;

   for (red = 0; red <= 1; red++)
   for (green = 0; green <= 1; green++)
   for (blue = 0; blue <= 1; blue++)
   {
      row1 = 4 * red + 2 * green + blue;                                   //  row 0 to 7
      row1 = row1 * hh / 8;                                                //  stripe, 1/8 of image
      row2 = row1 + hh / 8;
      
      for (row = row1; row < row2; row++)
      for (col = 0; col < ww; col++)
      {
         pixel = (uint8 *) Frgb8->bmp + (row * ww + col) * 3;
         pixel[0] = red * 256 * col / ww;
         pixel[1] = green * 256 * col / ww;
         pixel[2] = blue * 256 * col / ww;
      }
   }

   Fzoom = 0;                                                              //  scale to window
   gtk_window_set_title(GTK_WINDOW(mWin),"monitor check");
   mutex_unlock(&pixmaps_lock);
   mwpaint2();                                                             //  repaint window
   menulock(0);
   return;
}


/**************************************************************************/

//  create or update brightness histogram graph

GtkWidget      *winhisto = 0;                                              //  brightness histogram window
GtkWidget      *winhistoH = 0;                                             //  histogram drawing area
GtkWidget      *winhistoB = 0;                                             //  brightness band underneath


void m_histogram(GtkWidget *, cchar *)                                     //  menu function
{
   if (! Drgb8) return;

   if (winhisto) {
      histogram_paint();
      return;
   }

   winhisto = gtk_window_new(GTK_WINDOW_TOPLEVEL);
   gtk_window_set_title(GTK_WINDOW(winhisto),ZTX("Brightness Distribution"));
   gtk_window_set_transient_for(GTK_WINDOW(winhisto),GTK_WINDOW(mWin));
   gtk_window_set_default_size(GTK_WINDOW(winhisto),300,200);
   gtk_window_set_position(GTK_WINDOW(winhisto),GTK_WIN_POS_MOUSE);
   GtkWidget * vbox = gtk_vbox_new(0,3);
   gtk_container_add(GTK_CONTAINER(winhisto),vbox);

   winhistoH = gtk_drawing_area_new();                                     //  histogram drawing area
   gtk_box_pack_start(GTK_BOX(vbox),winhistoH,1,1,3);

   winhistoB = gtk_drawing_area_new();                                     //  brightness scale under histogram
   gtk_box_pack_start(GTK_BOX(vbox),winhistoB,0,0,0);                      //  (progresses from black to white)
   gtk_widget_set_size_request(winhistoB,300,12);                          //                   v.9.3

   G_SIGNAL(winhisto,"destroy",histogram_destroy,0)
   G_SIGNAL(winhistoH,"expose-event",histogram_paint,0)
   G_SIGNAL(winhistoB,"expose-event",histogram_paint,0)

   gtk_widget_show_all(winhisto);
   
   return;
}


void histogram_paint()                                                     //  paint graph window
{
   GdkGC          *gdkgc = 0;                                              //  GDK graphics context
   GdkColor       color;
   GdkColormap    *colormap = 0;

   int         brdist[40], bin, nbins = 40;                                //  increased                 v.9.3
   int         dist_maxbin = 0;
   int         winww, winhh;
   int         px, py, ww, hh, orgx, orgy;
   uint8       *pixel;
   double      bright;
   
   if (! winhisto) return;
   if (! Drgb8) return;

   gdkgc = gdk_gc_new(winhistoH->window);                                  //  use separate graphics context  v.8.7
                                                                           //  (compensate new GDK bug)
   for (bin = 0; bin < nbins; bin++)                                       //  clear brightness distribution
      brdist[bin] = 0;

   mutex_lock(&pixmaps_lock);

   for (py = 0; py < dhh; py++)                                            //  compute brightness distribution
   for (px = 0; px < dww; px++)                                            //    for image in visible window
   {                                                                       //  Dww/hh -> dww/hh   bugfix v.7.4.2
      pixel = (uint8 *) Drgb8->bmp + (py * dww + px) * 3;
      bright = brightness(pixel);                                          //  0 to 255
      brdist[int(bright / 256 * nbins)]++;                                 //  0 to nbins
   }

   for (bin = 0; bin < nbins; bin++)                                       //  find max. bin
      if (brdist[bin] > dist_maxbin) dist_maxbin = brdist[bin];

   mutex_unlock(&pixmaps_lock);

   gdk_window_clear(winhistoH->window);

   winww = winhistoH->allocation.width;                                    //  drawing window size
   winhh = winhistoH->allocation.height;
   ww = winww / nbins;                                                     //  bin width
   bin = -1;

   for (px = 0; px < winww; px++)                                          //  draw each bin
   {
      if (px * nbins / winww > bin) {
         bin++;
         hh = int(0.9 * winhh * brdist[bin] / dist_maxbin);
         orgx = px;
         orgy = winhh - hh;
         gdk_draw_rectangle(winhistoH->window,gdkgc,1,orgx,orgy,ww+1,hh);
      }
   }

   colormap = gtk_widget_get_colormap(winhistoH);
   hh = winhistoB->allocation.height;

   for (px = 0; px < winww; px++)                                          //  draw brightness scale underneath
   {                                                                       //                      v.9.3
      color.red = color.green = color.blue = 65536 * px / winww;
      gdk_rgb_find_color(colormap,&color);
      gdk_gc_set_foreground(gdkgc,&color);
      gdk_draw_line(winhistoB->window,gdkgc,px,0,px,hh-1);
   }

   gdk_gc_set_foreground(gdkgc,&black);
   return;
}


void histogram_destroy()                                                   //  delete window
{
   if (winhisto) gtk_widget_destroy(winhisto);
   winhisto = 0;
   return;
}


/**************************************************************************/

//  start a new parallel instance of fotoxx

void m_clone(GtkWidget *, cchar *)
{
   int      ignore;

   snprintf(command,ccc,"fotoxx -l %s",zfuncs::zlanguage);                 //  keep language   v.8.5 
   if (image_file) strncatv(command,ccc," \"",image_file,"\"",null);
   strcat(command," &");
   ignore = system(command);
   return;
}


/**************************************************************************/

//  enter or leave slideshow mode

void m_slideshow(GtkWidget *, cchar *)
{
   static int     ww, hh;
   zdialog        *zd;
   int            zstat, secs;
   
   if (! Fslideshow)
   {
      gtk_window_get_size(GTK_WINDOW(mWin),&ww,&hh);
      gtk_widget_hide_all(GTK_WIDGET(mMbar));                              //  enter slide show mode
      gtk_widget_hide_all(GTK_WIDGET(mTbar));                              //  (full screen, no extras)
      gtk_widget_hide_all(GTK_WIDGET(STbar));
      gtk_window_fullscreen(GTK_WINDOW(mWin));

      zd = zdialog_new(ZTX("Time Interval"),mWin,Bapply,Bcancel,null);
      zdialog_add_widget(zd,"hbox","hb1","dialog",0,"space=10");
      zdialog_add_widget(zd,"label","lab1","hb1",ZTX("seconds"));
      zdialog_add_widget(zd,"entry","secs","hb1",0,"scc=5");
      zdialog_stuff(zd,"secs",SS_interval);
      zdialog_run(zd);
      zstat = zdialog_wait(zd);
      zdialog_fetch(zd,"secs",secs);
      zdialog_free(zd);
      SS_interval = secs;                                                  //  interval between slides
      if (zstat != 1) secs = 9999;                                         //  cancel, use huge interval
      SS_timer = get_seconds() + secs + 1;                                 //  set timer for next slide
      Fslideshow = 1;
   }

   else
   {
      gtk_window_unfullscreen(GTK_WINDOW(mWin));                           //  leave slide show mode
      gtk_window_resize(GTK_WINDOW(mWin),ww,hh);
      gtk_widget_show_all(GTK_WIDGET(mMbar));
      gtk_widget_show_all(GTK_WIDGET(mTbar));
      gtk_widget_show_all(GTK_WIDGET(STbar));
      Fslideshow = 0;
   }

   Fzoom = 0;                                                              //  fit image to window
   Fblowup = Fslideshow;                                                   //  blow-up small images if SS mode
   mwpaint2(); 
   return;
}


/**************************************************************************/

//  show RGB values for pixel at mouse click

void m_showRGB(GtkWidget *, cchar *)                                       //  menu function
{
   int   RGB_dialog_event(zdialog *zd, cchar *event);
   void  RGB_mousefunc();
   
   cchar  *rgbmess = ZTX("click on window to show RGB");
   cchar  *format = "Pixel: 0 0   RGB: 0.0  0.0  0.0";

   if (! Frgb8) return;                                                    //  no image

   mouseCBfunc = RGB_mousefunc;                                            //  connect mouse function
   Mcapture = 1;                                                           //  capture mouse clicks

   if (zdRGB) zdialog_free(zdRGB);

   zdRGB = zdialog_new(ZTX("Show RGB"),mWin,Bcancel,null);
   zdialog_add_widget(zdRGB,"label","lab1","dialog",rgbmess,"space=5");
   zdialog_add_widget(zdRGB,"label","labrgb","dialog",format,"space=5");

   zdialog_run(zdRGB,RGB_dialog_event);                                    //  run dialog
   return;
}


//  dialog event function

int RGB_dialog_event(zdialog *zd, cchar *event)
{
   zdialog_free(zdRGB);                                                    //  kill dialog
   zdRGB = null;
   mouseCBfunc = 0;                                                        //  disconnect mouse
   Mcapture = 0;
   return 0;
}


//  mouse function

void RGB_mousefunc()                                                       //  mouse function
{
   int         px, py;
   double      red, green, blue;
   double      fbright, fred;
   char        text[60];
   uint8       *ppix8;
   uint16      *ppix16;
   
   if (LMclick)                                                            //  left mouse click
   {
      LMclick = 0;
      px = Mxclick;                                                        //  click position
      py = Myclick;
      
      if (E3rgb16) {                                                       //  use current image being edited
         if (px < 0 || px > E3ww-1 ||                                      //  outside image area     v.6.4
             py < 0 || py > E3hh-1) return;
         ppix16 = bmpixel(E3rgb16,px,py);                                  //  bugfix: * Mscale removed  v.6.7
         red = ppix16[0] / 256.0;
         green = ppix16[1] / 256.0;
         blue = ppix16[2] / 256.0;
         fbright = brightness(ppix16) / 256.0;
         fred = redness(ppix16);
      }

      else if (Frgb16) {                                                   //  use edited image
         if (px < 0 || px > Fww-1 || 
             py < 0 || py > Fhh-1) return;
         ppix16 = bmpixel(Frgb16,px,py);
         red = ppix16[0] / 256.0;
         green = ppix16[1] / 256.0;
         blue = ppix16[2] / 256.0;
         fbright = brightness(ppix16) / 256.0;
         fred = redness(ppix16);
      }

      else  {                                                              //  use 8 bpc image
         if (px < 0 || px > Fww-1 || 
             py < 0 || py > Fhh-1) return;
         ppix8 = (uint8 *) Frgb8->bmp + (py * Fww + px) * 3;
         red = ppix8[0];
         green = ppix8[1];
         blue = ppix8[2];
         fbright = brightness(ppix8);
         fred = redness(ppix8);
      }
      
      snprintf(text,59,"Pixel: %d %d  RGB: %6.3f %6.3f %6.3f",             //  show pixel and RGB colors
                                  px, py, red, green, blue);
      zdialog_stuff(zdRGB,"labrgb",text);
   }
   
   return;
}


/**************************************************************************/

//  toggle grid lines on / off                                             //  v.10.3

void m_gridlines(GtkWidget *, cchar *)
{
   int gridlines_dialog_event(zdialog *zd, cchar *event);

   zdialog     *zd;
   
   zd = zdialog_new(ZTX("Grid Lines"),mWin,Bdone,null);
   zdialog_add_widget(zd,"hbox","hb1","dialog",0,"space=5");
   zdialog_add_widget(zd,"label","lab1x","hb1","x count","space=5");
   zdialog_add_widget(zd,"spin","countx","hb1","0|30|1|0");
   zdialog_add_widget(zd,"label","space","hb1",0,"space=5");
   zdialog_add_widget(zd,"label","lab1y","hb1","y count","space=5");
   zdialog_add_widget(zd,"spin","county","hb1","0|30|1|0");
   zdialog_add_widget(zd,"hbox","hb2","dialog",0,"space=5");
   zdialog_add_widget(zd,"radio","gon","hb2","grid on","space=5");
   zdialog_add_widget(zd,"radio","goff","hb2","grid off","space=5");

   zdialog_stuff(zd,"countx",gridnx);
   zdialog_stuff(zd,"county",gridny);

   if (Fgrid) zdialog_stuff(zd,"gon",1);
   else zdialog_stuff(zd,"goff",1);
   
   zdialog_run(zd,gridlines_dialog_event);
   zdialog_wait(zd);
   zdialog_free(zd);
   return;
}


//  dialog event function

int gridlines_dialog_event(zdialog *zd, cchar *event)
{   
   if (strEqu(event,"gon")) {
      if (! Fgrid) {
         Fgrid = 1;
         mwpaint2();
      }
   }
   
   if (strEqu(event,"goff")) {
      if (Fgrid) {
         Fgrid = 0;
         mwpaint2();
      }
   }
   
   if (strEqu(event,"countx")) {
      zdialog_fetch(zd,"countx",gridnx);
      if (Fgrid) mwpaint2();
   }

   if (strEqu(event,"county")) {
      zdialog_fetch(zd,"county",gridny);
      if (Fgrid) mwpaint2();
   }

   return 0;
}


/**************************************************************************/

//  choose or set lens parameters for panoramas

void  m_parms(GtkWidget *, cchar *)
{
   zdialog     *zd;
   int         ii, zstat, radb;
   char        text[20];
   
   zd = zdialog_new(ZTX("Lens Parameters"),mWin,Bapply,Bcancel,null);
   zdialog_add_widget(zd,"hbox","hb1","dialog",0,"space=5");
   zdialog_add_widget(zd,"vbox","vb1","hb1",0,"space=5|homog");            //        Lens    mm    bow
   zdialog_add_widget(zd,"vbox","vb2","hb1",0,"space=5|homog");            //   (o)  name1   30    0.33
   zdialog_add_widget(zd,"vbox","vb3","hb1",0,"space=5|homog");            //   (x)  name2   40    0.22
   zdialog_add_widget(zd,"vbox","vb4","hb1",0,"space=5|homog");            //   (o)  name3   45    0.28
   zdialog_add_widget(zd,"label","space","vb1");                           //   (o)  name4   50    0.44
   zdialog_add_widget(zd,"radio","radb0","vb1",0);                         //
   zdialog_add_widget(zd,"radio","radb1","vb1",0);                         //                   [apply]
   zdialog_add_widget(zd,"radio","radb2","vb1",0);
   zdialog_add_widget(zd,"radio","radb3","vb1",0);
   zdialog_add_widget(zd,"label","lname","vb2",ZTX("lens name"));          //  fix translation   v.8.4.1
   zdialog_add_widget(zd,"entry","name0","vb2","scc=10");
   zdialog_add_widget(zd,"entry","name1","vb2","scc=10");
   zdialog_add_widget(zd,"entry","name2","vb2","scc=10");
   zdialog_add_widget(zd,"entry","name3","vb2","scc=10");
   zdialog_add_widget(zd,"label","lmm","vb3",ZTX("lens mm"));
   zdialog_add_widget(zd,"entry","mm0","vb3","0","scc=5");
   zdialog_add_widget(zd,"entry","mm1","vb3","0","scc=5");
   zdialog_add_widget(zd,"entry","mm2","vb3","0","scc=5");
   zdialog_add_widget(zd,"entry","mm3","vb3","0","scc=5");
   zdialog_add_widget(zd,"label","lbow","vb4",ZTX("lens bow"));
   zdialog_add_widget(zd,"entry","bow0","vb4","0.0","scc=6");
   zdialog_add_widget(zd,"entry","bow1","vb4","0.0","scc=6");
   zdialog_add_widget(zd,"entry","bow2","vb4","0.0","scc=6");
   zdialog_add_widget(zd,"entry","bow3","vb4","0.0","scc=6");
   
   for (ii = 0; ii < 4; ii++)                                              //  stuff lens data into dialog
   {
      snprintf(text,20,"name%d",ii);
      zdialog_stuff(zd,text,lens4_name[ii]);
      snprintf(text,20,"mm%d",ii);
      zdialog_stuff(zd,text,lens4_mm[ii]);
      snprintf(text,20,"bow%d",ii);
      zdialog_stuff(zd,text,lens4_bow[ii]);
   }

   snprintf(text,20,"radb%d",curr_lens);                                   //  current lens = selected
   zdialog_stuff(zd,text,1);

   zdialog_run(zd);                                                        //  run dialog, get inputs
   zstat = zdialog_wait(zd);

   if (zstat != 1) {
      zdialog_free(zd);                                                    //  canceled
      return;
   }
   
   for (ii = 0; ii < 4; ii++)                                              //  fetch lens data (revisions)
   {
      snprintf(text,20,"name%d",ii);
      zdialog_fetch(zd,text,lens4_name[ii],lens_cc);
      repl_1str(lens4_name[ii],lens4_name[ii]," ","_");                    //  replace blank with _
      snprintf(text,20,"mm%d",ii);
      zdialog_fetch(zd,text,lens4_mm[ii]);
      snprintf(text,20,"bow%d",ii);
      zdialog_fetch(zd,text,lens4_bow[ii]);
      snprintf(text,20,"radb%d",ii);                                       //  detect which is selected
      zdialog_fetch(zd,text,radb);
      if (radb) curr_lens = ii;
   }
   
   zdialog_free(zd);
   return;
}


/**************************************************************************/

//  set GUI language

void  m_lang(GtkWidget *, cchar *)                                         //  overhauled   v.10.1
{
   zdialog     *zd;
   int         ii, cc, err, val, zstat;
   char        lang[10], *pp;

   cchar  *langs[10] = { "en English", "de German", "es Spanish",          //  english first
                         "fr French", "gl Galacian", "it Italian", 
                         "sv Swedish", "zh_CN Chinese", "ru_RU Russian",
                          null  };
   
   cchar  *title = ZTX("Available Translations");

   zd = zdialog_new(ZTX("Set Language"),mWin,Bapply,Bcancel,null);
   zdialog_add_widget(zd,"label","title","dialog",title,"space=5");
   zdialog_add_widget(zd,"vbox","vb1","dialog");

   for (ii = 0; langs[ii]; ii++)                                           //  make radio button per language
      zdialog_add_widget(zd,"radio",langs[ii],"vb1",langs[ii]);

   cc = strlen(zfuncs::zlanguage);                                         //  current language
   for (ii = 0; langs[ii]; ii++)                                           //  match on lc_RC
      if (strnEqu(zfuncs::zlanguage,langs[ii],cc)) break;
   if (! langs[ii]) 
      for (ii = 0; langs[ii]; ii++)                                        //  failed, match on lc alone
         if (strnEqu(zfuncs::zlanguage,langs[ii],2)) break;
   if (! langs[ii]) ii = 0;                                                //  failed, default english
   zdialog_stuff(zd,langs[ii],1);

   zdialog_run(zd);                                                        //  run dialog
   zstat = zdialog_wait(zd);

   for (ii = 0; langs[ii]; ii++) {                                         //  get active radio button
      zdialog_fetch(zd,langs[ii],val);
      if (val) break;
   }

   zdialog_free(zd);                                                       //  kill dialog

   if (zstat != 1) return;                                                 //  user cancel
   if (! val) return;                                                      //  no selection
   
   strncpy0(lang,langs[ii],8);
   pp = strchr(lang,' ');                                                  //  isolate lc_RC part
   *pp = 0;

   sprintf(command,"fotoxx -l %s &",lang);                                 //  start new fotoxx with language
   err = system(command);

   m_quit(0,0);                                                            //  exit
   return;
}


/**************************************************************************/

//  create desktop icon / launcher

void  m_launcher(GtkWidget *, cchar *)                                     //  v.7.0
{
   zmake_launcher("Graphics","Image Editor");
   return;
}


/**************************************************************************/

//  convert multiple RAW files to tiff

char     **raw_files;
int      raw_filecount;
mutex    raw_mutex;
char     *raw_outfile = 0;

void  m_conv_raw(GtkWidget *, cchar *)                                     //  multi-threaded      v.10.1.1
{
   void * conv_raw_wthread(void *arg);

   char     *pp, *dirk = 0, *outfile1 = 0;
   int      ii;

   if (mod_keep()) return;
   if (! menulock(1)) return;

   if (! Fufraw) {
      zmessageACK(GTK_WINDOW(mWin),ZTX("Package ufraw required for this function"));
      goto rawdone;
   }

   if (image_file) {                                                       //  use this directory
      dirk = strdupz(image_file,0,"rawdirk");
      pp = strrchr(dirk,'/');
      *pp = 0;
   }
   else {
      pp = getcwd(null,0);                                                 //  use curr. directory
      dirk = strdupz(pp,0,"rawdirk");
      free(pp);
   }

   zfuncs::F1_help_topic = "convert_RAW";                                  //  v.9.0

   raw_files = zgetfiles(ZTX("Select RAW files to convert"),dirk);
   if (! raw_files) goto rawdone;
   
   for (ii = 0; raw_files[ii]; ii++);                                      //  count selected files
   raw_filecount = ii;

   set_cursor(busycursor);                                                 //  set function busy cursor

   mutex_init(&raw_mutex,0);

   for (ii = 0; ii < Nwt && ii < raw_filecount; ii++)                      //  start worker threads
      start_wt(conv_raw_wthread,&wtnx[ii]);

   write_popup_text("open","Converting RAW files",500,200,mWin);           //  status monitor popup window
   
   while (true)
   {
      zsleep(0.1);
      mutex_lock(&raw_mutex);

      if (! raw_outfile && ! wthreads_busy) {                              //  finished
         mutex_unlock(&raw_mutex);
         break;
      }
      
      if (! raw_outfile) {                                                 //  wait for more output
         mutex_unlock(&raw_mutex);
         continue;
      }

      write_popup_text("write",raw_outfile);                               //  write output to monitor window

      if (*raw_outfile == '*') {                                           //  an error message
         zfree(raw_outfile);
         raw_outfile = 0;
         mutex_unlock(&raw_mutex);
         continue;
      }
      
      menulock(0);                                                         //  an output file
      f_open(raw_outfile);                                                 //  display file
      menulock(1);
      mwpaint2();
      if (! outfile1) outfile1 = strdupz(raw_outfile,0,"raw_out1");        //  remember first output

      zfree(raw_outfile);
      raw_outfile = 0;

      mutex_unlock(&raw_mutex);
   }

   write_popup_text("write","COMPLETED");

   if (outfile1) {
      image_gallery(outfile1,"init");                                      //  update image gallery file list
      image_gallery(0,"paint2");                                           //  refresh gallery window if active
      Fsearchlist = 0;
      zfree(outfile1);
   }

   set_cursor(0);                                                          //  restore normal cursor

rawdone:

   if (dirk) zfree(dirk);                                                  //  free memory

   if (raw_files) {
      for (ii = 0; raw_files[ii]; ii++) 
         zfree(raw_files[ii]);
      zfree(raw_files);
   }

   menulock(0);                                                            //  unlock menus
   return;
}


//  convert RAW working threads

void * conv_raw_wthread(void *arg)
{
   int      index = *((int *) (arg));
   char     *rawfile, *outfile, *pp;
   int      ii, err, ccc = maxfcc*2;
   char     command[maxfcc*2];

   for (ii = index; ii < raw_filecount; ii += Nwt)
   {
      rawfile = raw_files[ii];
      outfile = strdupz(rawfile,5,"raw_out");
      pp = strrchr(rawfile,'.');
      pp = outfile + (pp - rawfile);
      strcpy(pp,".tiff");
      
      //  try new ufraw command format first, then old command if error

      snprintf(command,ccc,"ufraw-batch --out-type=tiff --out-depth=16"
                  " --overwrite --output=\"%s\" \"%s\" ",outfile, rawfile);
      err = system(command);

      if (err) {
         snprintf(command,ccc,"ufraw-batch --out-type=tiff16"
                  " --overwrite --output=\"%s\" \"%s\" ",outfile, rawfile);
         err = system(command);
      }
      
      mutex_lock(&raw_mutex);

      if (err) {
         zfree(outfile);
         raw_outfile = zmalloc(1000,"raw_errmess");
         snprintf(raw_outfile,999,"*** %s %s",wstrerror(err),rawfile);
      }
      else 
         raw_outfile = outfile;

      mutex_unlock(&raw_mutex);
   }

   exit_wt();                                                              //  exit thread
   return 0;                                                               //  not executed, stop gcc warning
}


/**************************************************************************/

//  burn images to CD/DVD                                                  //  v.7.2

int  burn_showthumb();

GtkWidget      *burn_drawarea = 0;
GtkWidget      *burn_files = 0;
cchar          *burn_font = "Monospace 8";
int            burn_fontheight = 14;
int            burn_cursorpos = 0;

void  m_burn(GtkWidget *, cchar *)
{
   int  burn_dialog_event(zdialog *zd, cchar *event);
   int  burn_mouseclick(GtkWidget *, GdkEventButton *event, void *);

   PangoLanguage           *plang;
   PangoFontDescription    *pfontdesc;
   PangoContext            *pcontext;
   PangoFont               *pfont;
   PangoFontMetrics        *pmetrics;
   int                     fontascent, fontdescent;
   GdkCursor               *cursor;
   GdkWindow               *gdkwin;

   int               line, nlines, cc1, cc2, err;
   char              *imagefile = 0;
   char              *bcommand;
   GtkTextBuffer     *textBuff;
   GtkTextIter       iter1, iter2;

   if (! menulock(1)) return;                                              //  lock menus

   m_gallery(0,0);                                                         //  activate image gallery window

   zdburn = zdialog_new(ZTX("Burn Images to CD/DVD"),0,ZTX("Burn"),Bcancel,null);
   zdialog_add_widget(zdburn,"hbox","hb1","dialog",0,"expand|space=5");
   zdialog_add_widget(zdburn,"frame","fr11","hb1",0,"expand");
   zdialog_add_widget(zdburn,"scrwin","scrwin","fr11",0,"expand");
   zdialog_add_widget(zdburn,"edit","files","scrwin");
   zdialog_add_widget(zdburn,"vbox","vb12","hb1");
   zdialog_add_widget(zdburn,"frame","fr12","vb12");
   zdialog_add_widget(zdburn,"hbox","hb2","dialog",0,"space=5");
   zdialog_add_widget(zdburn,"button","delete","hb2",Bdelete,"space=8");
   zdialog_add_widget(zdburn,"button","insert","hb2",Binsert,"space=8");
   zdialog_add_widget(zdburn,"button","addall","hb2",Baddall,"space=30");

   GtkWidget *frame = zdialog_widget(zdburn,"fr12");                       //  drawing area for thumbnail image
   burn_drawarea = gtk_drawing_area_new();
   gtk_widget_set_size_request(burn_drawarea,128,128);
   gtk_container_add(GTK_CONTAINER(frame),burn_drawarea);
   
   burn_files = zdialog_widget(zdburn,"files");                            //  activate mouse-clicks for
   gtk_widget_add_events(burn_files,GDK_BUTTON_PRESS_MASK);                //    file list widget
   G_SIGNAL(burn_files,"button-press-event",burn_mouseclick,0)

   pfontdesc = pango_font_description_from_string(burn_font);              //  set default font for files window
   gtk_widget_modify_font(burn_files,pfontdesc);

   plang = pango_language_get_default();                                   //  get font metrics (what a mess)
   pcontext = gtk_widget_get_pango_context(burn_files);                    //  bugfix  v.9.7
   pfont = pango_context_load_font(pcontext,pfontdesc);
   pmetrics = pango_font_get_metrics(pfont,plang);
   fontascent = pango_font_metrics_get_ascent(pmetrics) / PANGO_SCALE;
   fontdescent = pango_font_metrics_get_descent(pmetrics) / PANGO_SCALE;
   burn_fontheight = fontascent + fontdescent;                             //  effective line height

   cursor = gdk_cursor_new(GDK_TOP_LEFT_ARROW);                            //  arrow cursor for file list widget
   gdkwin = gtk_text_view_get_window(GTK_TEXT_VIEW(burn_files),textwin);
   gdk_window_set_cursor(gdkwin,cursor);
   burn_cursorpos = 0;

   zdialog_resize(zdburn,400,0);                                           //  start dialog
   zdialog_run(zdburn,burn_dialog_event);
   zdialog_wait(zdburn);

   if (zdburn->zstat != 1) {                                               //  cancelled
      zdialog_free(zdburn);                                                //  kill dialog
      zdburn = null;
      menulock(0);
      return;
   }

   textBuff = gtk_text_view_get_buffer(GTK_TEXT_VIEW(burn_files));
   nlines = gtk_text_buffer_get_line_count(textBuff);
   cc1 = gtk_text_buffer_get_char_count(textBuff);
   cc1 = cc1 + 5 * nlines + 20;
   bcommand = zmalloc(cc1,"brasero-command");
   strcpy(bcommand,"brasero");
   cc2 = strlen(bcommand);

   for (line = 0; line < nlines; line++)
   {
      gtk_text_buffer_get_iter_at_line(textBuff,&iter1,line);              //  iter at line start
      iter2 = iter1;
      gtk_text_iter_forward_to_line_end(&iter2);
      imagefile = gtk_text_buffer_get_text(textBuff,&iter1,&iter2,0);      //  get imagefile at line
      if (imagefile && *imagefile == '/') {
         strcpy(bcommand+cc2," \"");
         cc2 += 2;
         strcpy(bcommand+cc2,imagefile);
         cc2 += strlen(imagefile);
         strcpy(bcommand+cc2,"\"");
         cc2 += 1;
         free(imagefile);
      }
   }
   
   zdialog_free(zdburn);                                                   //  kill dialog
   zdburn = null;
   
   strcat(bcommand," &");                                                  //  do command in background  v.8.4
   err = system(bcommand);                                                 //  start brasero
   zfree(bcommand);
   menulock(0);
   return;
}


//  burn dialog event function

int burn_dialog_event(zdialog *zd, cchar *event)
{
   GtkTextBuffer  *textBuff;
   GtkTextIter    iter1, iter2;
   static char    *imagefile = 0;
   cchar          *xfile;
   int            line;

   if (strEqu(event,"delete"))                                             //  delete file at cursor position
   {
      if (imagefile) free(imagefile);
      imagefile = 0;
      textBuff = gtk_text_view_get_buffer(GTK_TEXT_VIEW(burn_files));
      line = burn_cursorpos;
      gtk_text_buffer_get_iter_at_line(textBuff,&iter1,line);              //  iter at line start
      iter2 = iter1;
      gtk_text_iter_forward_to_line_end(&iter2);                           //  iter at line end

      imagefile = gtk_text_buffer_get_text(textBuff,&iter1,&iter2,0);      //  save selected file
      if (*imagefile != '/') {
         free(imagefile);
         imagefile = 0;
         return 0;
      }

      gtk_text_buffer_delete(textBuff,&iter1,&iter2);                      //  delete file text
      gtk_text_buffer_get_iter_at_line(textBuff,&iter2,line+1);
      gtk_text_buffer_delete(textBuff,&iter1,&iter2);                      //  delete empty line (\n)

      burn_showthumb();                                                    //  thumbnail = next file
   }

   if (strEqu(event,"insert"))                                             //  insert last deleted file
   {
      if (! imagefile) return 0;                                           //    at current cursor position
      burn_insert_file(imagefile);
   }

   if (strEqu(event,"addall"))                                             //  insert all files in image gallery
   {
      if (imagefile) free(imagefile);
      imagefile = 0;

      textBuff = gtk_text_view_get_buffer(GTK_TEXT_VIEW(burn_files));
      xfile = "first";

      while (true)
      {
         imagefile = image_gallery(imagefile,xfile,1,0);                   //  next file
         if (! imagefile) break;
         xfile = "next";
         gtk_text_buffer_get_iter_at_line(textBuff,&iter1,burn_cursorpos);
         gtk_text_buffer_insert(textBuff,&iter1,"\n",1);                   //  insert new blank line
         gtk_text_buffer_get_iter_at_line(textBuff,&iter1,burn_cursorpos);
         gtk_text_buffer_insert(textBuff,&iter1,imagefile,-1);             //  insert image file
         burn_cursorpos++;                                                 //  advance cursor position
      }
   }

   if (strEqu(event,"F1")) showz_userguide("burn");                        //  F1 help          v.9.0

   return 0;
}


//  called from image gallery window when a thumbnail is clicked
//  add image file to list at current cursor position, set thumbnail = file

void burn_insert_file(cchar *imagefile)
{
   GtkTextIter    iter;
   GtkTextBuffer  *textBuff;

   if (*imagefile == '/') {
      textBuff = gtk_text_view_get_buffer(GTK_TEXT_VIEW(burn_files));
      gtk_text_buffer_get_iter_at_line(textBuff,&iter,burn_cursorpos);
      gtk_text_buffer_insert(textBuff,&iter,"\n",1);                       //  insert new blank line
      gtk_text_buffer_get_iter_at_line(textBuff,&iter,burn_cursorpos);
      gtk_text_buffer_insert(textBuff,&iter,imagefile,-1);                 //  insert image file
      burn_showthumb();                                                    //  update thumbnail
      burn_cursorpos++;                                                    //  advance cursor position
   }

   return;
}


//  process mouse click in files window: 
//  set new cursor position and set thumbnail = clicked file

int burn_mouseclick(GtkWidget *, GdkEventButton *event, void *)
{
   int            mpx, mpy;
   GtkWidget      *scrollwin;
   GtkAdjustment  *scrolladj;
   double         scrollpos;

   if (event->type != GDK_BUTTON_PRESS) return 0;
   mpx = int(event->x);                                                    //  mouse position
   mpy = int(event->y);
   scrollwin = zdialog_widget(zdburn,"scrwin");                            //  window scroll position
   scrolladj = gtk_scrolled_window_get_vadjustment(GTK_SCROLLED_WINDOW(scrollwin));
   scrollpos = gtk_adjustment_get_value(scrolladj);
   burn_cursorpos = (mpy + scrollpos) / burn_fontheight;                   //  line selected
   burn_showthumb();                                                       //  show thumbnail image
   return 0;
}


//  show thumbnail for file at current cursor position

int burn_showthumb()
{
   int            line;
   char           *imagefile;
   GtkTextBuffer  *textBuff;
   GtkTextIter    iter1, iter2;
   GdkPixbuf      *thumbnail = 0;

   gdk_window_clear(burn_drawarea->window);

   line = burn_cursorpos;
   textBuff = gtk_text_view_get_buffer(GTK_TEXT_VIEW(burn_files));
   gtk_text_buffer_get_iter_at_line(textBuff,&iter1,line);                 //  iter at line start
   iter2 = iter1;
   gtk_text_iter_forward_to_line_end(&iter2);                              //  iter at line end

   imagefile = gtk_text_buffer_get_text(textBuff,&iter1,&iter2,0);         //  get selected file
   if (*imagefile != '/') {
      free(imagefile);
      return 0;
   }

   thumbnail = image_thumbnail(imagefile,128);                             //  get thumbnail
   free(imagefile);

   if (thumbnail) {
      gdk_draw_pixbuf(burn_drawarea->window,0,thumbnail,0,0,0,0,-1,-1,nodither,0,0);
      g_object_unref(thumbnail);
   }
   return 0;
}


/**************************************************************************
      Image EXIF functions
***************************************************************************/

//  menu function and popup dialog to show EXIF data
//  window is updated when navigating to another image

void m_exif_view(GtkWidget *, cchar *menu)                                 //  overhauled    v.10.2
{
   int   exif_view_dialog_event(zdialog *zd, cchar *event);

   static int     basic = 1;
   char           *buff;
   int            err, contx = 0;
   GtkWidget      *widget;

   if (! Fexiftool) {                                                      //  exiftool is required
      zmessageACK(GTK_WINDOW(mWin),Bexiftoolmissing);
      return;
   }
   
   if (! image_file) return;

   if (menu) {                                                             //  remember basic or full request
      basic = 0;
      if (strEqu(menu,ZTX("Basic EXIF data"))) basic = 1;
   }

   if (! zdexifview)                                                       //  popup dialog if not already
   {
      zdexifview = zdialog_new(ZTX("EXIF data"),mWin,Bcancel,null);
      zdialog_add_widget(zdexifview,"scrwin","scroll","dialog",0,"expand");
      zdialog_add_widget(zdexifview,"edit","exifdata","scroll",0,"expand");
      zdialog_resize(zdexifview,500,400);
      zdialog_run(zdexifview,exif_view_dialog_event);
   }

   if (basic)                                                              //  set up exiftool command
      snprintf(command,ccc,"exiftool -common -%s -%s -%s -%s -%s \"%s\" ",
         exif_tags_key, exif_rating_key, exif_log_key, 
         exif_user_comment_key, exif_focal_length_key, image_file);
   else 
      snprintf(command,ccc,"exiftool -e \"%s\" ",image_file);
      
   widget = zdialog_widget(zdexifview,"exifdata");                         //  widget for output
   wclear(widget);

   while ((buff = command_output(contx,command)))                          //  run command, output into window
      wprintf(widget,"%s\n",buff);

   err = command_status(contx);                                            //  free resources

   return;
}


//  dialog event and completion callback function

int exif_view_dialog_event(zdialog *zd, cchar *event)                      //  kill dialog
{
   if (! zd->zstat) return 0;
   zdialog_free(zdexifview);
   zdexifview = null;
   return 0;
}


/**************************************************************************/

//  edit EXIF data - add or change specified EXIF key

void m_exif_edit(GtkWidget *, cchar *menu)                                 //  new v.10.2
{
   int   exif_edit_dialog_event(zdialog *zd, cchar *event);
   
   if (! Fexiftool) {                                                      //  exiftool is required
      zmessageACK(GTK_WINDOW(mWin),Bexiftoolmissing);
      return;
   }
   
   if (! image_file) return;

   zdialog *zd = zdialog_new(ZTX("Edit EXIF data"),mWin,Bfetch,Bsave,Bcancel,null);
   zdialog_add_widget(zd,"vbox","hb1","dialog");
   zdialog_add_widget(zd,"hbox","hbkey","dialog",0,"space=5");
   zdialog_add_widget(zd,"hbox","hbdata","dialog",0,"space=5");
   zdialog_add_widget(zd,"label","labkey","hbkey","key name");
   zdialog_add_widget(zd,"entry","keyname","hbkey",0,"scc=20");
   zdialog_add_widget(zd,"label","labdata","hbdata","key value");
   zdialog_add_widget(zd,"entry","keydata","hbdata",0,"scc=30");

   zdialog_run(zd,exif_edit_dialog_event);
   return;
}


//  dialog event and completion callback function

int  exif_edit_dialog_event(zdialog *zd, cchar *event)
{
   char        keyname[40], keydata[100];
   cchar       *pp1[1], *pp2[1];
   char        **pp3;
   int         err;
   
   if (! zd->zstat) return 0;
   
   zdialog_fetch(zd,"keyname",keyname,40);
   zdialog_fetch(zd,"keydata",keydata,100);

   if (zd->zstat == 1)                                                     //  fetch
   {
      zd->zstat = 0;
      pp1[0] = keyname;
      pp3 = exif_get(image_file,pp1,1);
      if (pp3[0]) {
         strncpy0(keydata,pp3[0],99);
         zfree(pp3[0]);
         zdialog_stuff(zd,"keydata",keydata);
      }
   }

   else if (zd->zstat == 2)                                                //  save
   {
      zd->zstat = 0;
      pp1[0] = keyname;
      pp2[0] = keydata;
      err = exif_set(image_file,pp1,pp2,1);
      if (err) zmessageACK(0,"error: %s",strerror(err));
      if (zdexifview) m_exif_view(0,0);
   }

   else zdialog_free(zd);                                                  //  other
   
   return 1;
}


/**************************************************************************/

//  delete EXIF data, specific key or all data

void m_exif_delete(GtkWidget *, cchar *menu)                               //  new v.10.2
{
   int   exif_delete_dialog_event(zdialog *zd, cchar *event);
   
   if (! Fexiftool) {                                                      //  exiftool is required
      zmessageACK(GTK_WINDOW(mWin),Bexiftoolmissing);
      return;
   }
   
   if (! image_file) return;

   zdialog *zd = zdialog_new(ZTX("Delete EXIF data"),mWin,Bapply,Bcancel,null);
   zdialog_add_widget(zd,"hbox","hb1","dialog",0,"space=5");
   zdialog_add_widget(zd,"radio","kall","hb1",ZTX("All"),"space=5");
   zdialog_add_widget(zd,"radio","key1","hb1",ZTX("One Key:"));
   zdialog_add_widget(zd,"entry","keyname","hb1",0,"scc=20");
   zdialog_stuff(zd,"key1",1);

   zdialog_run(zd,exif_delete_dialog_event);
   return;
}


//  dialog event and completion callback function

int  exif_delete_dialog_event(zdialog *zd, cchar *event)
{
   int         kall, key1, err;
   char        keyname[40];
   
   if (! zd->zstat) return 0;

   if (zd->zstat != 1) {                                                   //  canceled
      zdialog_free(zd);
      return 1;
   }
   
   zd->zstat = 0;                                                          //  dialog remains active

   zdialog_fetch(zd,"kall",kall);
   zdialog_fetch(zd,"key1",key1);
   zdialog_fetch(zd,"keyname",keyname,40);
                                                                           //  bugfix - quotes around file  v.10.3
   if (kall)
      snprintf(command,ccc,"exiftool -m -q -overwrite_original -all=  \"%s\"",image_file);
   else if (key1)
      snprintf(command,ccc,"exiftool -m -q -overwrite_original -%s=  \"%s\"",keyname,image_file);
   else return 1;
   
   err = system(command);
   if (err) zmessageACK(0,"%s",wstrerror(err));
   
   if (zdexifview) m_exif_view(0,0);

   return 1;
}


/**************************************************************************/

//  get EXIF metadata for given image file and EXIF key(s)
//  returns array of pointers to corresponding key values
//  if a key is missing, corresponding pointer is null
//  returned strings belong to caller, are subject for zfree()
//  up to 9 keynames may be requested per call
//  EXIF command: 
//       exiftool -keyname1 -keyname2 ... "file"
//  command output: 
//       keyname1: keyvalue1
//       keyname2: keyvalue2
//       ...

char ** exif_get(cchar *file, cchar **keys, int nkeys)
{
   char           *pp;
   static char    *rettext[10];
   int            contx = 0, err, ii;
   uint           cc;
   
   if (nkeys < 1 || nkeys > 9) appcrash("exif_get nkeys: %d",nkeys);

   strcpy(command,"exiftool -m -q -S -fast");
   
   for (ii = 0; ii < nkeys; ii++)
   {
      rettext[ii] = null;
      strncatv(command,ccc," -",keys[ii],null);                            //  "-exif:" replaced with "-"  v.10.0
   }

   strncatv(command,ccc," \"",file,"\"",null);

   while (true)
   {
      pp = command_output(contx,command);
      if (! pp) break;

      for (ii = 0; ii < nkeys; ii++)
      {
         cc = strlen(keys[ii]);
         if (strncasecmp(pp,keys[ii],cc) == 0)                             //  ignore case       bugfix v.10.2
            if (strlen(pp) > cc+2) 
               rettext[ii] = strdupz(pp+cc+2,0,"exif_data");               //  check not empty   bugfix v.7.3
      }

      zfree(pp);
   }
   
   err = command_status(contx);
   if (err) printf("exif_get failed \n");                                  //  v.6.9

   return rettext;
}


/**************************************************************************/

//  create or change EXIF metadata for given image file and key(s)
//  up to 9 keys may be processed
//  EXIF command: 
//    exiftool -overwrite_original -keyname="keyvalue" ... "file"

int exif_set(cchar *file, cchar **keys, cchar **text, int nkeys)
{
   int      ii, err;
   
   if (nkeys < 1 || nkeys > 9) zappcrash("exif_set nkeys: %d",nkeys);
   
   for (ii = 0; ii < nkeys; ii++)
      if (! text[ii]) text[ii] = "";                                       //  if null pointer use empty string
   
   strcpy(command,"exiftool -m -q -overwrite_original");
   
   for (ii = 0; ii < nkeys; ii++)
      strncatv(command,ccc," -",keys[ii],"=\"",text[ii],"\"",null);        //  "-exif:" replaced with "-"  v.10.0
   strncatv(command,ccc," \"",file,"\"",null);
   
   err = system(command);
   if (err) printf(" exif_set: %s \n",wstrerror(err));
   return err;
}


/**************************************************************************/

//  copy EXIF data from original image file to new (edited) image file
//  if nkeys > 0, up to 9 keys may be replaced with new values
//  EXIF command:
//    exiftool -tagsfromfile "file1" -keyname="keyvalue" ... "file2"

int exif_copy(cchar *file1, cchar *file2, cchar **keys, cchar **text, int nkeys)
{
   int      ii, err;
   
   strcpy(command,"exiftool -m -q -overwrite_original -tagsfromfile");
   strncatv(command,ccc," \"",file1,"\"",null);

   for (ii = 0; ii < nkeys; ii++)
      if (text[ii])                                                        //  v.10.2
         strncatv(command,ccc," -",keys[ii],"=\"",text[ii],"\"",null);     //  "-exif:" replaced with "-"  v.10.0
   strncatv(command,ccc," \"",file2,"\"",null);

   err = system(command);
   if (err) printf(" exif_copy: %s \n",wstrerror(err));
   return err;
}


/**************************************************************************
      Image Tags functions
***************************************************************************/

void  edittags_fixwidget(zdialog *, cchar * widgetname);                   //  fix tags widget for selecting with mouse
void  edittags_mouse(GtkTextView *, GdkEventButton *, cchar *);            //  select tag via mouse click
void  masstags_fixwidget(zdialog *, cchar * widgetname);                   //  fix tags widget for selecting with mouse
void  masstags_mouse(GtkTextView *, GdkEventButton *, cchar *);            //  select tag via mouse click
void  searchtags_fixwidget(zdialog *, cchar * widgetname);                 //  fix tags widget for selecting with mouse
void  searchtags_mouse(GtkTextView *, GdkEventButton *, cchar *);          //  select tag via mouse click

int   get_mouse_tag(GtkTextView *, int px, int py, cchar *);               //  get tag selected by mouse
int   add_unique_tag(cchar *tag, char *taglist, int maxcc);                //  add tag if unique and enough space
void  add_new_filetag();                                                   //  add tags_atag to tags_filetags
void  add_new_recentag();                                                  //  add tags_atag to tags_recentags
void  add_new_masstag();                                                   //  add tags_atag to tags_masstags
void  add_new_searchtag();                                                 //  add tags_atag to tags_masstags
int   delete_atag(char *taglist);                                          //  remove tags_atag from a tag list

char     tags_imdate[12] = "";                                             //  image date, yyyymmdd
char     tags_primdate[12] = "";                                           //  previous image date read or set
char     tags_stars = '0';                                                 //  image rating in stars, '0' to '5'
int      tags_changed = 0;                                                 //  tags have been changed
char     tags_searchfile[maxtagF] = "";                                    //  image search /path*/file*

char     tags_atag[maxtag1] = "";                                          //  one tag
char     tags_filetags[maxtag2] = "";                                      //  tags for one file
char     tags_asstags[maxtag3] = "";                                       //  all assigned tags
char     tags_recentags[maxtag5] = "";                                     //  recently added tags
char     tags_masstags[maxtag2] = "";                                      //  mass add tags 
char     tags_searchtags[maxtag4] = "";                                    //  image search tags


/**************************************************************************/

//  edit tags menu function

void m_edittags(GtkWidget *, cchar *)
{
   int   edittags_dialog_event(zdialog *zd, cchar *event);

   char     *ppv, pstarsN[12];

   if (! Fexiftool) {                                                      //  exiftool is required
      zmessageACK(GTK_WINDOW(mWin),Bexiftoolmissing);
      return;
   }
   
   if (zdmasstags) return;                                                 //  mass add tags dialog active
   if (! image_file) return;

   if (! zdedittags)                                                       //  (re) start tag edit dialog 
   {
      load_asstags();                                                      //  get all assigned tags

      zdedittags = zdialog_new(ZTX("Edit Tags"),mWin,Bdone,Bcancel,null);

      zdialog_add_widget(zdedittags,"hbox","hb1","dialog",0,"space=5");
      zdialog_add_widget(zdedittags,"label","labfile","hb1",ZTX("file:"),"space=10");
      zdialog_add_widget(zdedittags,"label","file","hb1");

      zdialog_add_widget(zdedittags,"hbox","hb2","dialog",0,"space=5");
      zdialog_add_widget(zdedittags,"label","lab21","hb2",ZTX("image date (yyyymmdd)"),"space=10");
      zdialog_add_widget(zdedittags,"entry","imdate","hb2",0,"scc=12");
      zdialog_add_widget(zdedittags,"button","primdate","hb2",ZTX("use last"),"space=10");

      zdialog_add_widget(zdedittags,"hbox","hb3","dialog",0,"space=5");
      zdialog_add_widget(zdedittags,"label","labstars","hb3",ZTX("image stars"),"space=10");
      zdialog_add_widget(zdedittags,"vbox","vb3","hb3");
      zdialog_add_widget(zdedittags,"hbox","hb31","vb3",0,"homog");
      zdialog_add_widget(zdedittags,"hbox","hb32","vb3",0,"homog");
      zdialog_add_widget(zdedittags,"label","lab30","hb31","0");
      zdialog_add_widget(zdedittags,"label","lab31","hb31","1");
      zdialog_add_widget(zdedittags,"label","lab32","hb31","2");
      zdialog_add_widget(zdedittags,"label","lab33","hb31","3");
      zdialog_add_widget(zdedittags,"label","lab34","hb31","4");
      zdialog_add_widget(zdedittags,"label","lab35","hb31","5");
      zdialog_add_widget(zdedittags,"radio","pstars0","hb32",0);
      zdialog_add_widget(zdedittags,"radio","pstars1","hb32",0);
      zdialog_add_widget(zdedittags,"radio","pstars2","hb32",0);
      zdialog_add_widget(zdedittags,"radio","pstars3","hb32",0);
      zdialog_add_widget(zdedittags,"radio","pstars4","hb32",0);
      zdialog_add_widget(zdedittags,"radio","pstars5","hb32",0);

      zdialog_add_widget(zdedittags,"hbox","hb4","dialog",0,"space=5");
      zdialog_add_widget(zdedittags,"label","lab4","hb4",ZTX("current tags"),"space=10");
      zdialog_add_widget(zdedittags,"frame","frame4","hb4",0,"expand");
      zdialog_add_widget(zdedittags,"edit","filetags","frame4",0,"expand");

      zdialog_add_widget(zdedittags,"hbox","hb5","dialog",0,"space=5");
      zdialog_add_widget(zdedittags,"label","recent","hb5",ZTX("recently added"),"space=10");
      zdialog_add_widget(zdedittags,"frame","frame5","hb5",0,"expand");
      zdialog_add_widget(zdedittags,"edit","recentags","frame5",0,"expand");

      zdialog_add_widget(zdedittags,"hbox","hb6","dialog",0,"space=5");
      zdialog_add_widget(zdedittags,"button","addtag","hb6",ZTX("create tag"),"space=10");
      zdialog_add_widget(zdedittags,"entry","atag","hb6",0);

      zdialog_add_widget(zdedittags,"hbox","hb7","dialog",0,"space=5");
      zdialog_add_widget(zdedittags,"hbox","hb8","dialog");
      zdialog_add_widget(zdedittags,"label","labasstags","hb8",ZTX("assigned tags"),"space=10");
      zdialog_add_widget(zdedittags,"frame","frame8","dialog",0,"space=3|expand");
      zdialog_add_widget(zdedittags,"edit","asstags","frame8",0,"expand");

      zdialog_resize(zdedittags,400,300);                                  //  run dialog
      zdialog_run(zdedittags,edittags_dialog_event);
      
      edittags_fixwidget(zdedittags,"filetags");                           //  setup for mouse tag selection
      edittags_fixwidget(zdedittags,"asstags");
      edittags_fixwidget(zdedittags,"recentags");
   }

   load_filetags(image_file);                                              //  get file tags from EXIF data

   ppv = (char *) strrchr(image_file,'/');
   zdialog_stuff(zdedittags,"file",ppv+1);                                 //  stuff dialog file name

   zdialog_stuff(zdedittags,"imdate",tags_imdate);                         //  stuff dialog data
   sprintf(pstarsN,"pstars%c",tags_stars);
   zdialog_stuff(zdedittags,pstarsN,1);
   zdialog_stuff(zdedittags,"filetags",tags_filetags);
   zdialog_stuff(zdedittags,"asstags",tags_asstags);
   zdialog_stuff(zdedittags,"recentags",tags_recentags);

   tags_changed = 0;
   return;
}


//  setup tag display widget for tag selection using mouse clicks

void edittags_fixwidget(zdialog *zd, cchar * widgetname)
{
   GtkWidget         *widget;
   GdkWindow         *gdkwin;

   widget = zdialog_widget(zd,widgetname);                                 //  make widget wrap text
   gtk_text_view_set_wrap_mode(GTK_TEXT_VIEW(widget),GTK_WRAP_WORD);
   gtk_text_view_set_editable(GTK_TEXT_VIEW(widget),0);                    //  disable widget editing

   gdkwin = gtk_text_view_get_window(GTK_TEXT_VIEW(widget),textwin);       //  cursor for tag selection
   gdk_window_set_cursor(gdkwin,arrowcursor);

   gtk_widget_add_events(widget,GDK_BUTTON_PRESS_MASK);                    //  connect mouse-click event
   G_SIGNAL(widget,"button-press-event",edittags_mouse,widgetname)
}


//  edit tags mouse-click event function
//  get clicked tag and add to or remove from tags_filetags

void edittags_mouse(GtkTextView *widget, GdkEventButton *event, cchar *widgetname)
{
   int            mpx, mpy, cc, found;

   if (event->type != GDK_BUTTON_PRESS) return;
   mpx = int(event->x);                                                    //  mouse click position
   mpy = int(event->y);
   
   cc = get_mouse_tag(widget,mpx,mpy,widgetname);                          //  tags_atag = clicked tag in list
   if (! cc) return;

   if (strEqu(widgetname,"filetags")) {
      found = delete_atag(tags_filetags);
      if (found) tags_changed++;
      zdialog_stuff(zdedittags,"filetags",tags_filetags);                  //  update dialog widgets
   }
   
   if (strEqu(widgetname,"asstags")) {
      add_new_filetag();                                                   //  add assigned tag to file tags
      add_new_recentag();
      zdialog_stuff(zdedittags,"filetags",tags_filetags);
   }

   if (strEqu(widgetname,"recentags")) {
      add_new_filetag();                                                   //  add recent tag to file tags
      zdialog_stuff(zdedittags,"filetags",tags_filetags);
   }

   return;
}


//  dialog event and completion callback function

int edittags_dialog_event(zdialog *zd, cchar *event)
{
   int         err;

   if (zd->zstat) {   
      if (zd->zstat == 1) update_filetags(image_file);                     //  done, update file EXIF tags
      zdialog_free(zdedittags);                                            //  kill dialog
      zdedittags = null;
      return 0;
   }
   
   if (strEqu(event,"F1")) showz_userguide("edit_tags");                   //  F1 help          v.9.0

   if (strEqu(event,"imdate")) {                                           //  image date revised
      err = zdialog_fetch(zd,"imdate",tags_imdate,11);
      if (err) return 1;
      if (strlen(tags_imdate) == 4) strcat(tags_imdate,"0101");            //  yyyy >> yyyy0101       v.8.8
      if (strlen(tags_imdate) == 6) strcat(tags_imdate,"01");              //  yyyymm >> yyyymm01
      tags_changed++;
   }
   
   if (strEqu(event,"primdate")) {                                         //  repeat last date used  v.8.1
      if (*tags_primdate) {
         zdialog_stuff(zd,"imdate",tags_primdate);
         strcpy(tags_imdate,tags_primdate);
         tags_changed++;
      }
   }

   if (strnEqu(event,"pstars",6)) {                                        //  stars revised
      tags_stars = event[6];                                               //  '0' to '5'
      tags_changed++;
   }

   if (strEqu(event,"addtag")) {
      err = zdialog_fetch(zd,"atag",tags_atag,maxtag1);                    //  add new tag to file
      if (err) return 1;                                                   //  reject too big tag
      add_new_filetag();
      add_new_recentag();
      zdialog_stuff(zd,"filetags",tags_filetags);                          //  update dialog widgets
      zdialog_stuff(zd,"asstags",tags_asstags);
      zdialog_stuff(zd,"atag","");
   }

   return 0;
}


//  convert mouse click position in a tag list into the selected tag
//  cc of selected tag is returned, or zero if failed to find
//  selected tag is returned in tags_atag

int get_mouse_tag(GtkTextView *widget, int mpx, int mpy, cchar *widgetname)
{
   GtkTextIter    iter;
   int            tbx, tby, offset, cc;
   char           *ptext, *pp1, *pp2;

   gtk_text_view_window_to_buffer_coords(widget,GTK_TEXT_WINDOW_TEXT,mpx,mpy,&tbx,&tby);
   gtk_text_view_get_iter_at_location(widget,&iter,tbx,tby);
   offset = gtk_text_iter_get_offset(&iter);                               //  graphic position in widget text

   ptext = 0;   
   if (strEqu(widgetname,"filetags")) ptext = tags_filetags;               //  get corresponding text
   if (strEqu(widgetname,"asstags")) ptext = tags_asstags;
   if (strEqu(widgetname,"recentags")) ptext = tags_recentags;
   if (strEqu(widgetname,"masstags")) ptext = tags_masstags;               //  get corresponding text
   if (! ptext) return 0;

   pp1 = ptext + utf8_position(ptext,offset);                              //  graphic position to byte position
   if (! *pp1 || *pp1 == ';') return 0;                                    //  reject ambiguity
   while (pp1 > ptext && *pp1 != ';') pp1--;                               //  find preceeding delimiter
   if (*pp1 == ';') pp1++;
   if (*pp1 == ' ') pp1++;

   pp2 = strchr(pp1,';');                                                  //  find following delimiter
   if (pp2) cc = pp2 - pp1;
   else cc = strlen(pp1);
   if (cc >= maxtag1) return 0;                                            //  reject tag too big
   strncpy0(tags_atag,pp1,cc+1);                                           //  tags_atag = selected tag
   return cc;
}


//  add input tag to output tag list if not already there and enough room
//  returns:   0 = added OK     1 = not unique (case ignored)
//             2 = overflow     3 = bad utf8 characters
//                              4 = imbedded delimiter (;)

int add_unique_tag(cchar *tag, char *taglist, int maxcc)
{
   char     *pp1, *pp2, temptag1[maxtag1], temptag2[maxtag1];
   int      atcc, cc1, cc2;

   strncpy0(temptag1,tag,maxtag1);                                         //  remove leading and trailing blanks
   atcc = strTrim2(temptag2,temptag1);
   if (! atcc) return 0;

   if (utf8_check(temptag2)) {                                             //  check for valid utf8 encoding
      printf("bad utf8 characters: %s \n",temptag2);
      return 3;
   }
   
   if (strchr(temptag2,';')) return 4;                                     //  reject imbedded delimiter
   
   pp1 = taglist;
   cc1 = strlen(temptag2);

   while (true)                                                            //  check if already in tag list
   {
      while (*pp1 == ' ' || *pp1 == ';') pp1++;
      if (! *pp1) break;
      pp2 = pp1 + 1;
      while (*pp2 && *pp2 != ';') pp2++;
      cc2 = pp2 - pp1;
      if (cc2 == cc1 && strncaseEqu(temptag2,pp1,cc1)) return 1;
      pp1 = pp2;
   }
   
   cc2 = strlen(taglist);                                                  //  append to tag list if space enough
   if (cc1 + cc2 + 2 >= maxcc) return 2;
   strcpy(taglist + cc2,temptag2);
   strcpy(taglist + cc2 + cc1,"; ");                                       //  add semicolon delimiter + space
   return 0;
}


/**************************************************************************/

//  image file EXIF data >> tags_imdate, tags_stars, tags_filetags in memory

void load_filetags(cchar *file)
{
   int         ii, jj, cc;
   cchar       *pp;
   cchar       *exifkeys[3] = { exif_date_key, exif_tags_key, exif_rating_key };
   char        **ppv, *imagedate, *imagetags, *imagestars;

   *tags_filetags = *tags_imdate = 0;
   tags_stars = '0';
   
   ppv = exif_get(file,exifkeys,3);                                        //  stars rating added        v.10.0
   imagedate = ppv[0];
   imagetags = ppv[1];
   imagestars = ppv[2];

   if (imagedate) {
      exif_tagdate(imagedate,tags_imdate);                                 //  EXIF date/time >> yyyymmdd  v.8.8
      strcpy(tags_primdate,tags_imdate);
      zfree(imagedate);
   }

   if (imagetags)
   {
      for (ii = 1; ; ii++)
      {
         pp = strField(imagetags,';',ii);                                  //  assume colon delimited tags
         if (! pp) break;
         if (*pp == ' ') continue;
         cc = strlen(pp);
         if (cc >= maxtag1) continue;                                      //  reject tags too big
         for (jj = 0; jj < cc; jj++)
            if (pp[jj] > 0 && pp[jj] < ' ') break;                         //  reject tags with control characters
         if (jj < cc) continue;
         strcpy(tags_atag,pp);                                             //  add to file tags if unique
         add_new_filetag();
      }

      zfree(imagetags);
   }

   if (imagestars) {                                                       //  v.10.0
      tags_stars = *imagestars;
      if (tags_stars < '0' || tags_stars > '5') tags_stars = '0';
      zfree(imagestars);
   }

   tags_changed = 0;                                                       //  v.9.7   
   return;
}


/**************************************************************************/

//  tags_imdate, tags_stars, tags_filetags in memory >> image file EXIF data

void update_filetags(cchar *file)
{
   cchar       *exifkeys[3] = { exif_date_key, exif_tags_key, exif_rating_key };
   cchar       *exifdata[3];
   char        imagedate[24], sstars[4];

   if (! tags_changed) return;

   *imagedate = 0;

   if (*tags_imdate) {
      tag_exifdate(tags_imdate,imagedate);                                 //  yyyymmdd >> EXIF date/time  v.8.8
      strcpy(tags_primdate,tags_imdate);
   }
   
   sstars[0] = tags_stars;                                                 //  string for stars rating
   sstars[1] = 0;

   exifdata[0] = imagedate;                                                //  update file EXIF data
   exifdata[1] = tags_filetags;
   exifdata[2] = sstars;

   exif_set(file,exifkeys,exifdata,3);                                     //  update EXIF data
   
   update_asstags(file);                                                   //  update tags index file
   tags_changed = 0;

   if (zdexifview) m_exif_view(0,0);                                       //  live update   v.10.2

   return;
}


/**************************************************************************/

//  add new tag to file tags, if not already and enough space.

void add_new_filetag()
{
   int err = add_unique_tag(tags_atag,tags_filetags,maxtag2);
   if (err == 2) 
      zmessageACK(GTK_WINDOW(mWin),ZTX("tags exceed %d characters"),maxtag2);
   if (err == 0) tags_changed++;                                           //  mark if file tags changed
   return;
}


//  add new tag to mass tags, if not already and enough space.

void add_new_masstag()
{
   int err = add_unique_tag(tags_atag,tags_masstags,maxtag2);
   if (err == 2) 
      zmessageACK(GTK_WINDOW(mWin),ZTX("tags exceed %d characters"),maxtag2);
   return;
}


//  add new tag to search tags, if not already and enough space.

void add_new_searchtag()
{
   int err = add_unique_tag(tags_atag,tags_searchtags,maxtag4);
   if (err == 2) 
      zmessageACK(GTK_WINDOW(mWin),ZTX("tags exceed %d characters"),maxtag4);
   return;
}


/**************************************************************************/

//  add new tag to recent tags, if not already.
//  remove oldest to make space if needed.

void add_new_recentag()
{
   int         err;
   char        *ppv, temp_recentags[maxtag5];

   err = add_unique_tag(tags_atag,tags_recentags,maxtag5);                 //  add tag to recent tags

   while (err == 2)
   {
      strncpy0(temp_recentags,tags_recentags,maxtag5);                     //  remove oldest to make room
      ppv = temp_recentags;
      while (*ppv && (*ppv == ' ' || *ppv == ';')) ppv++;
      while (*ppv && *ppv != ';') ppv++;
      while (*ppv && *ppv == ' ') ppv++;
      strcpy(tags_recentags,ppv);
      err = add_unique_tag(tags_atag,tags_recentags,maxtag5);
   }

   zdialog_stuff(zdedittags,"recentags",tags_recentags);                   //  update dialog
   return;
}


/**************************************************************************/

//  remove tag_atag from input taglist, if present
//  returns 0 if not present, 1 if found and deleted

int delete_atag(char *taglist)
{
   int         ii, ftcc, atcc, found = 0;
   char        *temptags;
   cchar       *pp;
   
   temptags = strdupz(taglist,0,"temptags");
   
   *taglist = 0;
   ftcc = 0;
   
   for (ii = 1; ; ii++)
   {
      pp = strField(temptags,';',ii);
      if (! pp) {
         zfree(temptags);                                                  //  bugfix, leak     v.10.3
         return found;
      }
      if (*pp == ' ') continue;
      
      if (strcaseEqu(pp,tags_atag)) {
         found = 1;
         continue;
      }

      atcc = strlen(pp);
      strcpy(taglist + ftcc, pp);
      ftcc += atcc;
      strcpy(taglist + ftcc,"; ");
      ftcc += 2;
   }
}


/**************************************************************************/

//  load tags index file >> tags_asstags in memory
//  create list of all assigned tags with no duplicates

void load_asstags()
{
   FILE        *fid;
   int         ntags = 0, ntcc, atcc, ii, err;
   char        *ppv, tagsbuff[tags_index_bcc];
   cchar       *pp1;
   char        *tags[maxntags];
   
   ntcc = 0;
   *tags_asstags = 0;

   fid = fopen(tags_index_file,"r");
   if (! fid) return;                                                      //  no tags
   
   while (true)                                                            //  read tags index file
   {
      ppv = fgets_trim(tagsbuff,tags_index_bcc,fid);
      if (! ppv) break;
      if (strnNeq(tagsbuff,"tags: ",6)) continue;

      for (ii = 1; ; ii++)                                                 //  add all file tags to assigned tags
      {                                                                    //    unless already present
         pp1 = strField(tagsbuff+6,';',ii);
         if (! pp1) break;
         if (*pp1 == ' ') continue;
         err = add_unique_tag(pp1,tags_asstags,maxtag3);
         if (err == 2) goto overflow;
      }
   }

   err = fclose(fid);
   if (err) goto tagsfileerr;
   
   for (ii = 1; ; ii++)                                                    //  build sort list
   {
      pp1 = strField(tags_asstags,';',ii);
      if (! pp1) break;
      if (*pp1 == ' ') continue;                                           //  trailing ' ' after last ';'
      tags[ntags] = strdupz(pp1,0,"asstag");
      ntags++;
      if (ntags == maxntags) goto toomanytags;
   }
   
   HeapSort(tags,ntags);                                                   //  sort alphabetically
   
   ntcc = 0;
   *tags_asstags = 0;

   for (ii = 0; ii < ntags; ii++)                                          //  build sorted assigned tags list
   {
      atcc = strlen(tags[ii]);
      if (ntcc + atcc + 1 > maxtag3) goto overflow;
      strcpy(tags_asstags + ntcc,tags[ii]);
      ntcc += atcc;
      strcpy(tags_asstags + ntcc,"; ");
      ntcc += 2;
      zfree(tags[ii]);
   }

   tags_asstags[ntcc] = 0;
   return;

overflow:
   zmessageACK(GTK_WINDOW(mWin),ZTX("Total tags exceed %d characters"),maxtag3);
   return;

toomanytags:
   zmessageACK(GTK_WINDOW(mWin),ZTX("Too many tags: %d"),maxntags);
   return;

tagsfileerr:
   zmessLogACK(GTK_WINDOW(mWin),ZTX("tags index file error: %s"),strerror(errno));
   return;
}


/**************************************************************************/

//  update tags_asstags in memory from tags_filetags
//  update tags index file (add or replace changed file and its tags)

void update_asstags(cchar *file, int del)
{
   char           *ppv, temp_tags_index_file[200];
   char           imagedate[12], filedate[16];
   char           filebuff[tags_index_bcc], datebuff[tags_index_bcc];
   char           tagsbuff[tags_index_bcc], starsbuff[tags_index_bcc];
   cchar          *pp1;
   int            ii, ntcc, err;
   FILE           *fidr, *fidw;
   struct stat    statb;
   struct tm      bdt;

   ntcc = strlen(tags_asstags);
   
   if (! del)                                                              //  unless deleted
   {
      for (ii = 1; ; ii++)                                                 //  add file tags to assigned tags
      {                                                                    //    unless already present
         pp1 = strField(tags_filetags,';',ii);
         if (! pp1) break;
         if (*pp1 == ' ') continue;
         
         err = add_unique_tag(pp1,tags_asstags,maxtag3);
         if (err == 2) {
            zmessageACK(GTK_WINDOW(mWin),ZTX("Total tags exceed %d characters"),maxtag3);
            break;
         }
      }
   }

   strcpy(temp_tags_index_file,tags_index_file);                           //  temp tag file
   strcat(temp_tags_index_file,"_temp");

   fidr = fopen(tags_index_file,"r");                                      //  read tag file
   
   fidw = fopen(temp_tags_index_file,"w");                                 //  write temp tag file
   if (! fidw) goto tagserror;
   
   if (fidr)
   {   
      while (true)                                                         //  copy tags index file to temp
      {                                                                    //    file, omitting this image file
         ppv = fgets_trim(filebuff,tags_index_bcc,fidr);
         if (! ppv) break;
         if (strnNeq(filebuff,"file: ",6)) continue;

         if (strEqu(filebuff+6,file)) continue;                            //  if my file, skip copy

         ppv = fgets_trim(datebuff,tags_index_bcc,fidr);
         if (! ppv) break;
         if (strnNeq(datebuff,"date: ",6)) continue;

         ppv = fgets_trim(tagsbuff,tags_index_bcc,fidr);
         if (! ppv) break;
         if (strnNeq(tagsbuff,"tags: ",6)) continue;

         ppv = fgets_trim(starsbuff,tags_index_bcc,fidr);
         if (! ppv) break;
         if (strnNeq(starsbuff,"stars: ",7)) continue;

         fprintf(fidw,"%s\n",filebuff);
         fprintf(fidw,"%s\n",datebuff);                                    //  copy to temp file
         fprintf(fidw,"%s\n",tagsbuff);
         fprintf(fidw,"%s\n",starsbuff);
         fprintf(fidw,"\n");
      }
   }
   
   if (! del)                                                              //  append revised file data to temp file 
   {
      err = fprintf(fidw,"file: %s\n",file);                               //  output filespec
      if (err <= 0) goto tagserror;

      if (*tags_imdate) {                                                  //  image date
         strncpy(imagedate,tags_imdate,4);
         strncpy(imagedate+5,tags_imdate+4,2);                             //  yyyymmdd >> yyyy:mm:dd
         strncpy(imagedate+8,tags_imdate+6,2);
         imagedate[4] = imagedate[7] = ':';
         imagedate[10] = 0;
      }
      else strcpy(imagedate,"null");                                       //  signifies missing image date

      err = stat(file,&statb);
      gmtime_r(&statb.st_mtime,&bdt);
      sprintf(filedate,"%04d%02d%02d%02d%02d%02d",                         //  file date = yyyymmddhhmmss   v.8.4
               bdt.tm_year + 1900, bdt.tm_mon + 1, bdt.tm_mday,
               bdt.tm_hour, bdt.tm_min, bdt.tm_sec);

      err = fprintf(fidw,"date: %s  %s\n",imagedate,filedate);             //  output image date and file date

      if (*tags_filetags)                                                  //  output filetags
         err = fprintf(fidw,"tags: %s\n",tags_filetags);
      else  err = fprintf(fidw,"tags: null\n");                            //  "null" if none
      
      err = fprintf(fidw,"stars: %c\n",tags_stars);                        //  output stars rating

      err = fprintf(fidw,"\n");                                            //  EOL
   }

   if (fidr) {
      err = fclose(fidr);
      if (err) goto tagserror;
   }

   err = fclose(fidw);
   if (err) goto tagserror;
   
   err = rename(temp_tags_index_file,tags_index_file);                     //  replace tags file with temp file
   if (err) goto tagserror;

   return;
   
tagserror:
   zmessLogACK(GTK_WINDOW(mWin),ZTX("tags index file error: %s"),strerror(errno));
   return;
}


/**************************************************************************/

//  menu function - add tags to many files at once

char     **masstags_filelist = 0;
int      masstags_filecount = 0;

void m_masstags(GtkWidget *, cchar *)                                      //  new v.9.7
{
   int   masstags_dialog_event(zdialog *zd, cchar *event);
   
   cchar       *Baddtags = ZTX("add tags");
   cchar       *errormess = ZTX("%s \n tag limit exceeded");
   cchar       *ptag;
   char        **flist, *file;
   int         ii, jj, err;
   
   if (zdedittags) return;                                                 //  tags edit active
   if (zdmasstags) return;

   if (! Fexiftool) {                                                      //  exiftool is required
      zmessageACK(GTK_WINDOW(mWin),Bexiftoolmissing);
      return;
   }

   if (mod_keep()) return;

   load_asstags();                                                         //  get all assigned tags

   zdmasstags = zdialog_new(ZTX("Mass Add Tags"),mWin,Baddtags,Bcancel,null);

   zdialog_add_widget(zdmasstags,"hbox","hb1","dialog",0,"space=5");
   zdialog_add_widget(zdmasstags,"label","lab1","hb1",ZTX("tags to add"),"space=10");
   zdialog_add_widget(zdmasstags,"frame","frame1","hb1",0,"expand");
   zdialog_add_widget(zdmasstags,"edit","masstags","frame1",0,"expand");
   zdialog_add_widget(zdmasstags,"hbox","hb2","dialog",0,"space=5");
   zdialog_add_widget(zdmasstags,"button","addtag","hb2",ZTX("create tag"),"space=10");
   zdialog_add_widget(zdmasstags,"entry","atag","hb2",0);
   zdialog_add_widget(zdmasstags,"hbox","hb3","dialog",0,"space=5");
   zdialog_add_widget(zdmasstags,"button","files","hb3",ZTX("select files"),"space=10");
   zdialog_add_widget(zdmasstags,"label","labcount","hb3","0 files selected","space=10");
   zdialog_add_widget(zdmasstags,"hbox","hb4","dialog",0,"space=5");
   zdialog_add_widget(zdmasstags,"hbox","hb5","dialog");
   zdialog_add_widget(zdmasstags,"label","labasstags","hb5",ZTX("assigned tags"),"space=10");
   zdialog_add_widget(zdmasstags,"frame","frame5","dialog",0,"space=3|expand");
   zdialog_add_widget(zdmasstags,"edit","asstags","frame5",0,"expand");

   zdialog_resize(zdmasstags,400,200);                                     //  run dialog
   zdialog_run(zdmasstags,masstags_dialog_event);
   
   masstags_fixwidget(zdmasstags,"masstags");                              //  setup for mouse tag selection
   masstags_fixwidget(zdmasstags,"asstags");

   zdialog_stuff(zdmasstags,"asstags",tags_asstags);                       //  load tag list into dialog

   masstags_filelist = 0;
   masstags_filecount = 0;
   *tags_masstags = 0;

   zdialog_wait(zdmasstags);                                               //  wait for dialog completion

   flist = masstags_filelist;

   if (zdmasstags->zstat != 1) goto cleanup;
   if (! masstags_filecount) goto cleanup;

   progress_goal = masstags_filecount;
   progress_done = 0;

   for (ii = 0; flist[ii]; ii++)                                           //  loop all selected files
   {
      file = flist[ii];                                                    //  display image
      f_open(file);
      zmainloop();
      load_filetags(file);                                                 //  load current file tags

      for (jj = 1; ; jj++)                                                 //  add new tags unless already
      {
         ptag = strField(tags_masstags,';',jj);
         if (! ptag) break;
         if (*ptag == ' ') continue;
         err = add_unique_tag(ptag,tags_filetags,maxtag2);
         if (err == 2) {
            zmessageACK(GTK_WINDOW(mWin),errormess,file);
            break;
         }
         if (err == 0) tags_changed = 1;
      }

      update_filetags(file);                                               //  update image and tags index files
      progress_done++;
   }

cleanup:
   progress_goal = progress_done = 0;
   zdialog_free(zdmasstags);
   zdmasstags = 0;
   if (masstags_filecount) {
      for (ii = 0; flist[ii]; ii++) 
         zfree(flist[ii]);
      zfree(flist);
   }
   return;
}


//  masstags dialog event function

int masstags_dialog_event(zdialog *zd, cchar *event)
{
   int      ii, err;
   char     **flist = masstags_filelist, countmess[32];

   cchar    *selectmess = ZTX("select image files to add tags");
   
   if (strEqu(event,"F1")) showz_userguide("mass_tags");                   //  F1 help

   if (strEqu(event,"addtag")) {
      err = zdialog_fetch(zd,"atag",tags_atag,maxtag1);                    //  add new tag to list
      if (err) return 0;                                                   //  reject too big tag
      add_new_masstag();
      zdialog_stuff(zd,"masstags",tags_masstags);                          //  update dialog widgets
      zdialog_stuff(zd,"atag","");
   }

   if (strEqu(event,"files"))                                              //  select images to add tags
   {
      if (flist) {                                                         //  free prior list
         for (ii = 0; flist[ii]; ii++) 
            zfree(flist[ii]);
         zfree(flist);
      }

      flist = zgetfiles(selectmess,image_file);                            //  get file list from user
      masstags_filelist = flist;

      if (flist)                                                           //  count files in list
         for (ii = 0; flist[ii]; ii++);
      else ii = 0;
      masstags_filecount = ii;
      
      sprintf(countmess,"%d files selected",masstags_filecount);
      zdialog_stuff(zd,"labcount",countmess);
   }
   
   return 0;
}
   

//  setup tag display widget for tag selection using mouse clicks

void masstags_fixwidget(zdialog *zd, cchar * widgetname)
{
   GtkWidget         *widget;
   GdkWindow         *gdkwin;

   widget = zdialog_widget(zd,widgetname);                                 //  make widget wrap text
   gtk_text_view_set_wrap_mode(GTK_TEXT_VIEW(widget),GTK_WRAP_WORD);
   gtk_text_view_set_editable(GTK_TEXT_VIEW(widget),0);                    //  disable widget editing

   gdkwin = gtk_text_view_get_window(GTK_TEXT_VIEW(widget),textwin);       //  cursor for tag selection
   gdk_window_set_cursor(gdkwin,arrowcursor);

   gtk_widget_add_events(widget,GDK_BUTTON_PRESS_MASK);                    //  connect mouse-click event
   G_SIGNAL(widget,"button-press-event",masstags_mouse,widgetname)
}


//  masstags mouse-click event function
//  get clicked tag and add to or remove from tags_masstags

void masstags_mouse(GtkTextView *widget, GdkEventButton *event, cchar *widgetname)
{
   int            mpx, mpy, cc;

   if (event->type != GDK_BUTTON_PRESS) return;
   mpx = int(event->x);                                                    //  mouse click position
   mpy = int(event->y);

   cc = get_mouse_tag(widget,mpx,mpy,widgetname);                          //  tags_atag = clicked tag in list
   if (! cc) return;

   if (strEqu(widgetname,"masstags")) {
      delete_atag(tags_masstags);                                          //  remove tag from masstags
      zdialog_stuff(zdmasstags,"masstags",tags_masstags);                  //  update dialog widgets
   }
   
   if (strEqu(widgetname,"asstags")) {
      add_new_masstag();                                                   //  add assigned tag to masstags
      zdialog_stuff(zdmasstags,"masstags",tags_masstags);
   }

   return;
}


/**************************************************************************/

//  search image tags for matching images

char     searchDateFrom[12] = "";                                          //  image search date range
char     searchDateTo[12] = "";
int      searchStarsFrom = 0;                                              //  image search stars range
int      searchStarsTo = 0;

void m_searchtags(GtkWidget *, cchar *)
{
   int   searchtags_dialog_event(zdialog*, cchar *event);

   zdsearchtags = zdialog_new(ZTX("Search Tags"),mWin,Bsearch,Bcancel,null);
   
   zdialog_add_widget(zdsearchtags,"hbox","hb1","dialog",0,"space=2");
   zdialog_add_widget(zdsearchtags,"vbox","vb1","hb1",0,"space=5|homog");
   zdialog_add_widget(zdsearchtags,"vbox","vb2","hb1",0,"homog|expand");

   zdialog_add_widget(zdsearchtags,"label","labDR","vb1",ZTX("date range"));
   zdialog_add_widget(zdsearchtags,"label","labSR","vb1",ZTX("stars range"));
   zdialog_add_widget(zdsearchtags,"label","labF","vb1",ZTX("/path*/file*"));
   zdialog_add_widget(zdsearchtags,"label","labT","vb1",ZTX("search tags"));

   zdialog_add_widget(zdsearchtags,"hbox","hbDR","vb2",0,"space=1");          //  date range    yyyymmdd  yyyymmdd
   zdialog_add_widget(zdsearchtags,"entry","datefrom","hbDR",0,"scc=10");     //  stars range   4  5
   zdialog_add_widget(zdsearchtags,"entry","dateto","hbDR",0,"scc=10");       //  /path*/file*  /dname/dname*/fname*
   zdialog_add_widget(zdsearchtags,"label","labDF","hbDR"," (yyyymmdd)");     //  search tags   rosi  alaska

   zdialog_add_widget(zdsearchtags,"hbox","hbSR","vb2",0,"space=1|expand");   //  (o) match all  (o) match any tags
   zdialog_add_widget(zdsearchtags,"entry","starsfrom","hbSR",0,"scc=2");
   zdialog_add_widget(zdsearchtags,"entry","starsto","hbSR",0,"scc=2");       //  assigned tags list

   zdialog_add_widget(zdsearchtags,"hbox","hbF","vb2",0,"space=1|expand");
   zdialog_add_widget(zdsearchtags,"entry","searchfile","hbF",0,"expand");
   zdialog_add_widget(zdsearchtags,"button","fileclear","hbF",Bclear);

   zdialog_add_widget(zdsearchtags,"hbox","hbT","vb2",0,"space=1|expand");
   zdialog_add_widget(zdsearchtags,"entry","searchtags","hbT",0,"expand");
   zdialog_add_widget(zdsearchtags,"button","tagsclear","hbT",Bclear);

   zdialog_add_widget(zdsearchtags,"hbox","hbM","dialog",0,"space=5");
   zdialog_add_widget(zdsearchtags,"radio","rmall","hbM",ZTX("match all tags"),"space=5");
   zdialog_add_widget(zdsearchtags,"label","lspace","hbM","","space=10");
   zdialog_add_widget(zdsearchtags,"radio","rmany","hbM",ZTX("match any tag"));

   zdialog_add_widget(zdsearchtags,"hbox","hbsp","dialog",0,"space=1");

   zdialog_add_widget(zdsearchtags,"hbox","hbAT","dialog");
   zdialog_add_widget(zdsearchtags,"label","labasstags","hbAT",ZTX("assigned tags"));
   zdialog_add_widget(zdsearchtags,"frame","frameAT","dialog",0,"space=3|expand");
   zdialog_add_widget(zdsearchtags,"edit","asstags","frameAT",0,"expand");

   zdialog_resize(zdsearchtags,400,0);                                     //  start dialog
   zdialog_run(zdsearchtags,searchtags_dialog_event);

   searchtags_fixwidget(zdsearchtags,"asstags");                           //  setup tag selection via mouse
   
   zdialog_stuff(zdsearchtags,"datefrom",searchDateFrom);                  //  stuff previous date range
   zdialog_stuff(zdsearchtags,"dateto",searchDateTo);
   zdialog_stuff(zdsearchtags,"searchtags",tags_searchtags);               //  stuff previous search tags
   zdialog_stuff(zdsearchtags,"rmall",1);                                  //  default is match all tags
   zdialog_stuff(zdsearchtags,"rmany",0);
   load_asstags();                                                         //  stuff assigned tags
   zdialog_stuff(zdsearchtags,"asstags",tags_asstags);
   
   return;
}

//  setup tag display widget for tag selection using mouse clicks

void searchtags_fixwidget(zdialog *zd, cchar * widgetname)
{
   GtkWidget         *widget;
   GdkWindow         *gdkwin;

   widget = zdialog_widget(zd,widgetname);                                 //  make widget wrap text
   gtk_text_view_set_wrap_mode(GTK_TEXT_VIEW(widget),GTK_WRAP_WORD);
   gtk_text_view_set_editable(GTK_TEXT_VIEW(widget),0);                    //  disable widget editing

   gdkwin = gtk_text_view_get_window(GTK_TEXT_VIEW(widget),textwin);       //  cursor for tag selection
   gdk_window_set_cursor(gdkwin,arrowcursor);

   gtk_widget_add_events(widget,GDK_BUTTON_PRESS_MASK);                    //  connect mouse-click event
   G_SIGNAL(widget,"button-press-event",searchtags_mouse,widgetname)
}


//  search tags mouse-click event function
//  get clicked tag and add to or remove from tags_searchtags

void searchtags_mouse(GtkTextView *widget, GdkEventButton *event, cchar *widgetname)
{
   int            mpx, mpy, cc;
   
   if (event->type != GDK_BUTTON_PRESS) return;
   mpx = int(event->x);                                                    //  mouse click position
   mpy = int(event->y);

   cc = get_mouse_tag(widget,mpx,mpy,widgetname);                          //  tags_atag = clicked tag in list
   if (! cc) return;

   if (strEqu(widgetname,"asstags")) {
      add_new_searchtag();                                                 //  add assigned tag to search tags
      zdialog_stuff(zdsearchtags,"searchtags",tags_searchtags);
   }

   return;
}


//  dialog event and completion callback function

int searchtags_dialog_event(zdialog *zd, cchar *event)
{
   cchar          *pps, *ppf;
   char           resultsfile[tags_index_bcc];
   char           *ppv, *file, *tags;
   char           filebuff[tags_index_bcc], tagsbuff[tags_index_bcc];
   char           date1[12], date2[12], lcfile[tags_index_bcc];
   int            err, nfiles, iis, iif, stars, nmatch, nfail;
   int            date1cc, date2cc, Fmall, Fdates, Ffiles, Ftags, Fstars;
   FILE           *fidr, *fidw;
   struct stat    statbuf;

   if (strEqu(event,"fileclear"))
      zdialog_stuff(zd,"searchfile","");

   if (strEqu(event,"tagsclear")) {
      *tags_searchtags = 0;
      zdialog_stuff(zd,"searchtags","");
   }

   if (strEqu(event,"F1")) showz_userguide("search_tags");                 //  F1 help          v.9.0
   
   if (! zd->zstat) return 0;

   if (zd->zstat == 1)                                                     //  dialog complete
   {
      zdialog_fetch(zd,"datefrom",searchDateFrom,10);                      //  get search date range
      zdialog_fetch(zd,"dateto",searchDateTo,10);
      zdialog_fetch(zd,"starsfrom",searchStarsFrom);                       //  get search stars range
      zdialog_fetch(zd,"starsto",searchStarsTo);
      zdialog_fetch(zd,"searchfile",tags_searchfile,maxtagF);              //  get search /path*/file*   v.6.6
      zdialog_fetch(zd,"searchtags",tags_searchtags,maxtag4);              //  get search tags
      zdialog_fetch(zd,"rmall",Fmall);                                     //  get match all/any option
   }
   
   zdialog_free(zdsearchtags);                                             //  kill dialog
   zdsearchtags = null;
   if (zd->zstat != 1) return 0;                                           //  cancelled

   strcpy(date1,"0000");                                                   //  defaults for missing dates  v.6.6
   strcpy(date2,"9999");                                                   //  (year 0000 to year 9999)
   date1cc = date2cc = 4;
   Fdates = 0;

   if (*searchDateFrom) {                                                  //  date from is given
      Fdates++;
      strncpy(date1,searchDateFrom,4);                                     //  convert format
      strncpy(date1+5,searchDateFrom+4,2);                                 //    yyyymmdd >> yyyy:mm:dd
      strncpy(date1+8,searchDateFrom+6,2);
      date1[4] = date1[7] = ':';
      date1[10] = 0;                                                       //  bugfix  v.8.8
      date1cc = strlen(date1);
   }
   if (*searchDateTo) {                                                    //  date to is given
      Fdates++;
      strncpy(date2,searchDateTo,4);
      strncpy(date2+5,searchDateTo+4,2);
      strncpy(date2+8,searchDateTo+6,2);
      date2[4] = date2[7] = ':';
      date2[10] = 0;                                                       //  bugfix  v.8.8
      date2cc = strlen(date2);
   }
   
   Fstars = 0;
   if (searchStarsFrom || searchStarsTo) Fstars = 1;                       //  stars given
   
   Ffiles = 0;
   if (! blank_null(tags_searchfile)) Ffiles = 1;                          //  search file* given     v.6.6

   Ftags = 0;
   if (! blank_null(tags_searchtags)) Ftags = 1;                           //  search tags given

   if (! Ffiles && ! Ftags && ! Fdates && ! Fstars) {                      //  no search criteria was given,
      strcpy(tags_searchtags,"null");                                      //    find images with no tags   v.6.9.3
      Ftags = 1;
   }
   
   if (Ffiles) strToLower(tags_searchfile);                                //  v.6.6
   if (Ftags) strToLower(tags_searchtags);
   
   snprintf(resultsfile,199,"%s/search_results",get_zuserdir());
   fidw = fopen(resultsfile,"w");                                          //  search results output file
   if (! fidw) goto writerror;

   fidr = fopen(tags_index_file,"r");                                      //  read tags index file
   if (! fidr) goto noasstags;
   
   nfiles = 0;                                                             //  count matching files found

   while (true)
   {
      ppv = fgets_trim(filebuff,tags_index_bcc,fidr,1);                    //  next record
      if (! ppv) break;
      if (! strnEqu(ppv,"file: ",6)) continue;                             //  file: /dir.../filename.jpg

      file = ppv+6;
      err=stat(file,&statbuf);                                             //  check file exists
      if (err) continue;
      if (! S_ISREG(statbuf.st_mode)) continue;

      if (Ffiles)                                                          //  test for path*/file* match
      {
         strToLower(lcfile,file);
         nmatch = 0;

         for (iis = 1; ; iis++)
         {
            pps = strField(tags_searchfile,' ',iis);                       //  step thru search files
            if (! pps) break;
            if (MatchWild(pps,lcfile) == 0) { nmatch++; break; }
         }
         
         if (nmatch == 0) continue;
      }

      ppv = fgets_trim(tagsbuff,tags_index_bcc,fidr);                      //  next record
      if (! ppv) break;
      if (! strnEqu(ppv,"date: ",6)) continue;                             //  date: yyyy:mm:dd 

      if (Fdates) {      
         if (strncmp(ppv+6,date1,date1cc) < 0) continue;                   //  check search date range
         if (strncmp(ppv+6,date2,date2cc) > 0) continue;
      }

      ppv = fgets_trim(tagsbuff,tags_index_bcc,fidr);                      //  next record
      if (! ppv) break;
      if (! strnEqu(ppv,"tags: ",6)) continue;                             //  tags: xxxx xxxxx ...

      if (Ftags)
      {                                                                    //  tag search
         tags = ppv + 6;
         strToLower(tags);                                                 //  v.6.6

         nmatch = nfail = 0;

         for (iis = 1; ; iis++)
         {
            pps = strField(tags_searchtags,';',iis);                       //  step thru search tags
            if (! pps) break;
            if (*pps == ' ') continue;

            for (iif = 1; ; iif++)                                         //  step thru file tags
            {
               ppf = strField(tags,';',iif);
               if (! ppf) { nfail++; break; }                              //  count matches and fails
               if (*ppf == ' ') continue;
               if (MatchWild(pps,ppf) == 0) { nmatch++; break; }           //  wildcard match   v.6.6
            }
         }

         if (nmatch == 0) continue;                                        //  no match to any tag
         if (Fmall && nfail) continue;                                     //  no match to all tags
      }
      
      ppv = fgets_trim(tagsbuff,tags_index_bcc,fidr);                      //  next record
      if (! ppv) break;
      if (! strnEqu(ppv,"stars: ",7)) continue;                            //  stars: N

      if (Fstars)
      {                                                                    //  stars search
         stars = ppv[7] - '0';
         if (searchStarsFrom && stars < searchStarsFrom) continue;
         if (searchStarsTo && stars > searchStarsTo) continue;
      }

      fprintf(fidw,"%s\n",file);                                           //  write matching image file
      nfiles++;
   }

   fclose(fidr);
   
   err = fclose(fidw);
   if (err) goto writerror;
   
   if (! nfiles) {
      zmessageACK(GTK_WINDOW(mWin),ZTX("No matching images found"));
      return 0;
   }
   
   image_gallery(resultsfile,"initF",0,m_gallery2);                        //  generate gallery of matching files
   image_gallery(0,"paint1");                                              //  show new image gallery window
   Fsearchlist = 1;                                                        //  restricted updates     v.6.4

   return 0;

noasstags:
   zmessageACK(GTK_WINDOW(mWin),ZTX("No tags index file"));
   return 0;

writerror:
   zmessLogACK(GTK_WINDOW(mWin),ZTX("Search results file error %s"),strerror(errno));
   return 0;
}


/**************************************************************************/

//  convert between EXIF and fotoxx tag date formats                       //  v.8.8
//  EXIF date: yyyy:mm:dd hh:mm:ss[.ss]
//  tag date: yyyymmdd
//  

void exif_tagdate(cchar *exifdate, char *tagdate)
{
   time_t      tnow;
   struct tm   *snow;
   
   if (! exifdate || strlen(exifdate) < 10) {                              //  bad EXIF date, use current date
      tnow = time(0);
      snow = localtime(&tnow);
      snprintf(tagdate,8,"%4d%02d%02d",
               snow->tm_year+1900, snow->tm_mon+1, snow->tm_mday);
      tagdate[8] = 0;
      return;
   }

   strncpy(tagdate,exifdate,4);                                            //  convert 
   strncpy(tagdate+4,exifdate+5,2);
   strncpy(tagdate+6,exifdate+8,2);
   tagdate[8] = 0;
   return;
}

void tag_exifdate(cchar *tagdate, char *exifdate)
{
   int         cc;
   time_t      tnow;
   struct tm   *snow;
   
   if (! tagdate || strlen(tagdate) < 4) {
      tnow = time(0);
      snow = localtime(&tnow);
      snprintf(exifdate,20,"%4d:%02d:%02d %02d:%02d:%02d",
               snow->tm_year+1900, snow->tm_mon+1, snow->tm_mday,
               snow->tm_hour, snow->tm_min, snow->tm_sec);
      exifdate[19] = 0;
      return;
   }

   strncpy(exifdate,tagdate,4);
   strcpy(exifdate+4,":01:01 00:00:00");
   cc = strlen(tagdate);
   if (cc >= 6) strncpy(exifdate+5,tagdate+4,2);
   if (cc >= 8) strncpy(exifdate+8,tagdate+6,2);
   exifdate[19] = 0;
   return;
}


/**************************************************************************/

//  Convert EXIF tags data to new standard:
//     move all tags from EXIF:UserComment to IPTC:Keywords
//     tag delimiter is semicolon instead of blank
//     special tag "stars=N" is converted to EXIF:Rating
//     EXIF:UserComment is not erased, so this function can be repeated

void m_tags_convert(GtkWidget *, cchar *)                                  //  v.10.0
{
   int   tags_convert_dialog_event(zdialog *zd, cchar *event);

   zdialog        *zd;
   int            zstat, yn, contx, ii, cc, tcc, Nfiles = 0;
   char           *ppv, stbartext[200], *subdirk, sstars[4];
   char           *oldtags, *newtags, **olddata;
   cchar          *filespec1, *filespec2;
   cchar          *oldkeys[1] = { exif_tags_key_old };
   cchar          *newkeys[2] = { exif_tags_key, exif_rating_key };
   cchar          *newdata[2], *pp;

   if (! Fexiftool) {                                                      //  exiftool is required
      zmessageACK(GTK_WINDOW(mWin),Bexiftoolmissing);
      return;
   }

   yn = zmessageYN(GTK_WINDOW(mWin),ZTX("Convert tags to new standard now?\n"
                                        "Are your image files backed-up?"));
   if (! yn) return;

   zd = zdialog_new(ZTX("Convert tags to new standard"),mWin,Bproceed,Bcancel,null);
   zdialog_add_widget(zd,"hbox","hb1","dialog");
   zdialog_add_widget(zd,"label","lab1","hb1",ZTX("Top Image Directory:"),"space=5");
   zdialog_add_widget(zd,"hbox","hb2","dialog");
   zdialog_add_widget(zd,"entry","topdirk","hb2","??","scc=40|space=5");
   zdialog_add_widget(zd,"button","browse","hb2",ZTX("browse"),"space=5");
   
   if (topdirk) zdialog_stuff(zd,"topdirk",topdirk);

   zdialog_run(zd,tags_convert_dialog_event);
   zstat = zdialog_wait(zd);

   if (zstat != 1) {
      zdialog_free(zd);
      return;
   }

   ppv = zmalloc(maxfcc);   
   zdialog_fetch(zd,"topdirk",ppv,maxfcc-1);
   if (topdirk) zfree(topdirk);
   topdirk = strdupz(ppv,0,"topdirk");
   zfree(ppv);
   
   zdialog_free(zd);

   if (! menulock(1)) return;

   remove(tags_index_file);                                                //  tags index is now invalid

   set_cursor(busycursor);                                                 //  set function busy cursor

   snprintf(command,ccc,"find \"%s\" -type d",topdirk);                    //  find all image files
   contx = 0;

   while ((subdirk = command_output(contx,command)))                       //  find directories under top directory
   {
      pp = strrchr(subdirk,'/');
      if (pp && strEqu(pp,"/.thumbnails")) {                               //  ignore .thumbnails
         zfree(subdirk);
         continue;
      }

      image_gallery(subdirk,"init");                                       //  get all image files in directory
      filespec1 = image_gallery(subdirk,"first");

      while (filespec1)
      {
         if (image_file_type(filespec1) != 2) goto nextfile;

         olddata = exif_get(filespec1,oldkeys,1);                          //  get exif old tags data
         oldtags = olddata[0];
         if (! oldtags) {
            Nfiles++;                                                      //  none
            goto nextfile;
         }
         
         newtags = zmalloc(maxtag2,"newtags");
         *newtags = 0;
         tcc = 0;
         strcpy(sstars,"0");
         
         for (ii = 1; ; ii++)
         {
            pp = strField(oldtags,' ',ii);                                 //  blank delimited tags
            if (! pp) break;

            if (strnEqu(pp,"stars=",6)) {                                  //  special tag, save rating '0' to '5'
               sstars[0] = pp[6];
               continue;
            }
            
            if (strEqu(pp,"null")) continue;

            cc = strlen(pp);                                               //  check enough room
            if (cc + 2 + tcc >= maxtag2) continue;

            strcpy(newtags + tcc, pp);                                     //  copy tag to newtags list
            strcpy(newtags + tcc + cc, "; ");                              //  add semicolon delimiter and blank
            tcc += cc + 2;
         }
         
         newdata[0] = newtags;                                             //  add Keywords tag
         newdata[1] = sstars;                                              //  add Ratings tag
         exif_set(filespec1,newkeys,newdata,2);                            //  update EXIF data

         Nfiles++;                                                         //  count files processed
         zfree(oldtags);
         zfree(newtags);

      nextfile:
         snprintf(stbartext,199,"%5d %s",Nfiles,filespec1);                //  update status bar
         stbar_message(STbar,stbartext);

         filespec2 = image_gallery(filespec1,"next");                      //  next image file
         zfree((char *) filespec1);
         filespec1 = filespec2;
      }

      zfree(subdirk);
   }
   
   printf("processed %d image files \n",Nfiles);
   set_cursor(0);                                                          //  restore normal cursor
   menulock(0);
                                                                           //  create new tags index 
   zmessageACK(GTK_WINDOW(mWin),ZTX("new tags index will now be created"));
   m_tags_index(0,0);

   return;
}


//  dialog event function - file chooser for top image directory

int tags_convert_dialog_event(zdialog *zd, cchar *event)
{
   char     *pp;

   if (strEqu(event,"F1")) showz_userguide("convert_tags");                //  F1 help

   if (! strEqu(event,"browse")) return 0;
   pp = zgetfile(ZTX("Select top image directory"),topdirk,"folder");      //  get topmost directory
   if (! pp) return 0;
   zdialog_stuff(zd,"topdirk",pp);
   zfree(pp);
   return 0;
}


/**************************************************************************/

//  Rebuild tags index file.
//  Process all image files within given top-level directory.
//  Works incrementally and is very fast after the first run.              //  overhauled v.10.0

struct tags_rec {
   char        *file;                                                      //  image filespec
   char        imagedate[12], filedate[16];                                //  image date, file date
   char        *tags;                                                      //  image tags
   char        stars;                                                      //  image rating, '0' to '5'
   char        update;                                                     //  flag, update is needed
   char        fill[2];
};
tags_rec   *trec_old, *trec_new;


void m_tags_index(GtkWidget *widget, cchar *menu)                          //  menu function
{
   int  tags_index_dialog_event(zdialog *zd, cchar *event);
   int  tags_index_compare(cchar *rec1, cchar *rec2);

   zdialog        *zd;
   FILE           *fid;
   int            zstat, err, contx, fcount;
   char           tagsbuff[tags_index_bcc], stbartext[200];
   char           *subdirk, *pp, **ppv;
   char           *filespec1, *filespec2;
   char           *imagedate, *imagetags, *imagerating;
   cchar          *exifkeys[3] = { exif_date_key, exif_tags_key, exif_rating_key };
   cchar          *ppc, *mode;
   int            Nold = 0, Nnew = 0;
   int            comp, orec, nrec;
   struct tm      bdt;
   struct stat    statb;

   if (! menu) {                                                           //  called from tags_convert
      mode = "full";
      goto proceed;
   }

   if (! Fexiftool) {                                                      //  exiftool is required
      zmessageACK(GTK_WINDOW(mWin),Bexiftoolmissing);
      return;
   }
   
   zd = zdialog_new(ZTX("Rebuild Tags Index"),mWin,Bproceed,Bcancel,null);
   zdialog_add_widget(zd,"hbox","hb1","dialog",0,"space=5");
   zdialog_add_widget(zd,"label","lab1","hb1",ZTX("Top Image Directory:"));
   zdialog_add_widget(zd,"hbox","hb2","dialog",0,"space=1");
   zdialog_add_widget(zd,"entry","topdirk","hb2","??","scc=40|space=5");
   zdialog_add_widget(zd,"button","browse","hb2",ZTX("browse"),"space=5");
   zdialog_add_widget(zd,"hbox","hb3","dialog",0,"space=5");
   zdialog_add_widget(zd,"radio","rb1","hb3",ZTX("full rebuild"),"space=5");
   zdialog_add_widget(zd,"radio","rb2","hb3",ZTX("incremental"),"space=10");
   
   if (topdirk) zdialog_stuff(zd,"topdirk",topdirk);
   zdialog_stuff(zd,"rb1",0);
   zdialog_stuff(zd,"rb2",1);

   zdialog_run(zd,tags_index_dialog_event);
   zstat = zdialog_wait(zd);

   if (zstat != 1) {
      zdialog_free(zd);
      return;
   }

   pp = zmalloc(maxfcc);   
   zdialog_fetch(zd,"topdirk",pp,maxfcc-1);
   if (topdirk) zfree(topdirk);
   topdirk = strdupz(pp,0,"topdirk");
   zfree(pp);
   
   zdialog_fetch(zd,"rb1",zstat);
   if (zstat) mode = "full";
   else mode = "incr";

   zdialog_free(zd);

proceed:

   if (! menulock(1)) return;

   set_cursor(busycursor);                                                 //  set function busy cursor
   
   trec_old = (tags_rec *) zmalloc(max_images * sizeof(tags_rec),"trec_old");
   trec_new = (tags_rec *) zmalloc(max_images * sizeof(tags_rec),"trec_new");
   
//  read current tags index file and build "old list" of tags

   fid = fopen(tags_index_file,"r");                                       //  open tags index file
   if (fid) 
   {
      while (true)
      {
         trec_old[Nold].file = 0;
         trec_old[Nold].imagedate[0] = 0;
         trec_old[Nold].filedate[0] = 0;
         trec_old[Nold].tags = 0;
         trec_old[Nold].stars = '0';
         trec_old[Nold].update = '0';

         pp = fgets_trim(tagsbuff,tags_index_bcc,fid);                     //  file: /directory/.../filename.jpg
         if (! pp) break;
         if (! strnEqu(tagsbuff,"file: ",6)) continue;
         trec_old[Nold].file = strdupz(pp+6,0,"trec_old.file");

         pp = fgets_trim(tagsbuff,tags_index_bcc,fid);                     //  date: yyyy:mm:dd yyyymmddhhmmss
         if (! pp) break;
         if (! strnEqu(tagsbuff,"date: ",6)) continue;
         ppc = strField(tagsbuff,' ',2);                                   
         if (ppc) strncpy0(trec_old[Nold].imagedate,ppc,12);
         ppc = strField(tagsbuff,' ',3);
         if (ppc) strncpy0(trec_old[Nold].filedate,ppc,16);

         pp = fgets_trim(tagsbuff,tags_index_bcc,fid);                     //  tags: xxxxx; xxxxx; xxxxxxx; xxxx;
         if (! pp) break;
         if (! strnEqu(tagsbuff,"tags: ",6)) continue;
         trec_old[Nold].tags = strdupz(pp+6,0,"trec_old.tags");
         
         pp = fgets_trim(tagsbuff,tags_index_bcc,fid);                     //  stars: N
         if (! pp) break;
         if (! strnEqu(tagsbuff,"stars: ",7)) continue;
         trec_old[Nold].stars = *(pp+7);
   
         if (++Nold == max_images) 
            zappcrash("more than %d image files: %d",max_images);
      }
      
      fclose(fid);
   }

   printf("%d current tag records found \n",Nold);
   
//  find all image files and create "new list" with no tags

   snprintf(command,ccc,"find \"%s\" -type d",topdirk);
   contx = 0;

   while ((subdirk = command_output(contx,command)))                       //  find directories under top directory
   {
      pp = strrchr(subdirk,'/');
      if (pp && strEqu(pp,"/.thumbnails"))                                 //  /.../.thumbnails/ directory
      {
         if (strEqu(mode,"full")) {                                        //  if full rebuild, delete thumbnails
            snprintf(command,ccc,"rm -R \"%s\"",subdirk);
            err = system(command);
         }
         zfree(subdirk);
         continue;
      }

      image_gallery(subdirk,"init");                                       //  get all image files in directory
      filespec1 = image_gallery(subdirk,"first");

      while (filespec1)
      {
         if (image_file_type(filespec1) == 2)                              //  construct new tag record
         {
            err = stat(filespec1,&statb);
            if (err) continue;
            trec_new[Nnew].file = strdupz(filespec1,0,"trec_new.file");    //  filespec
            trec_new[Nnew].imagedate[0] = 0;                               //  image date = empty
            gmtime_r(&statb.st_mtime,&bdt);                                //  file date = yyyymmddhhmmss
            snprintf(trec_new[Nnew].filedate,16,"%04d%02d%02d%02d%02d%02d",
                     bdt.tm_year + 1900, bdt.tm_mon + 1, bdt.tm_mday,
                     bdt.tm_hour, bdt.tm_min, bdt.tm_sec);
            trec_new[Nnew].tags = 0;                                       //  tags = empty
            trec_new[Nnew].stars = '0';                                    //  stars = '0'
            if (++Nnew == max_images) 
               zappcrash("more than %d image files: %d",max_images);
         }
      
         filespec2 = image_gallery(filespec1,"next");                      //  next image file
         zfree(filespec1);
         filespec1 = filespec2;
      }

      zfree(subdirk);
   }
   
   printf("found %d image files \n",Nnew);

//  merge and compare lists
//  if filespecs match and have the same date, then old tags are OK

   HeapSort((char *) trec_old,sizeof(tags_rec),Nold,tags_index_compare);
   HeapSort((char *) trec_new,sizeof(tags_rec),Nnew,tags_index_compare);

   for (orec = nrec = 0; nrec < Nnew; )
   {
      trec_new[nrec].update = '1';                                         //  assume update is needed

      if (orec == Nold) comp = +1;
      else comp = strcmp(trec_old[orec].file, trec_new[nrec].file);        //  compare filespecs
      
      if (comp > 0) nrec++;
      else if (comp < 0) orec++;
      else                                                                 //  matching filespecs
      {
         if (strEqu(trec_new[nrec].filedate, trec_old[orec].filedate))     //  and matching file dates
         {                                                                 //  copy data from old to new
            strncpy0(trec_new[nrec].imagedate, trec_old[orec].imagedate,12);
            trec_new[nrec].tags = trec_old[orec].tags;
            trec_old[orec].tags = 0;
            trec_new[nrec].stars = trec_old[orec].stars;                       
            trec_new[nrec].update = '0';                                   //  update is not needed
         }
         nrec++;
         orec++;
      }
   }

   for (orec = 0; orec < Nold; orec++)                                     //  release old list memory
   {
      zfree(trec_old[orec].file);
      if (trec_old[orec].tags) zfree(trec_old[orec].tags);
   }

   zfree(trec_old);
   trec_old = 0;

//  process entries needing update in new list
//  get updated tags from image file EXIF data

   for (fcount = nrec = 0; nrec < Nnew; nrec++)
   {
      if (trec_new[nrec].update == '0') continue;

      ppv = exif_get(trec_new[nrec].file,exifkeys,3);                      //  get exif data
      imagedate = ppv[0];
      imagetags = ppv[1];
      imagerating = ppv[2];
      
      if (imagedate && strlen(imagedate)) {                                //  image date
         if (strlen(imagedate) > 9) imagedate[10] = 0;                     //  truncate to yyyy:mm:dd
         strcpy(trec_new[nrec].imagedate,imagedate);
      }
      else strcpy(trec_new[nrec].imagedate,"null");
      
      if (imagetags && strlen(imagetags))                                  //  image tags
         trec_new[nrec].tags = strdupz(imagetags,0,"trec_new.tags");
      else trec_new[nrec].tags = strdupz("null;",0,"trec_new.tags");

      if (imagerating && strlen(imagerating))                              //  image rating
         trec_new[nrec].stars = *imagerating;

      if (imagedate) zfree(imagedate);
      if (imagetags) zfree(imagetags);
      if (imagerating) zfree(imagerating);

      snprintf(stbartext,199,"%5d %s",++fcount,trec_new[nrec].file);       //  update status bar
      stbar_message(STbar,stbartext);
   }

   fid = fopen(tags_index_file,"w");                                       //  write new tags index file
   if (! fid) zappcrash("cannot write tags file");                         //    with merged data

   for (nrec = 0; nrec < Nnew; nrec++)
   {
      fprintf(fid,"file: %s""\n",trec_new[nrec].file);
      fprintf(fid,"date: %s  %s""\n", trec_new[nrec].imagedate, trec_new[nrec].filedate);
      fprintf(fid,"tags: %s""\n",trec_new[nrec].tags);
      fprintf(fid,"stars: %c""\n",trec_new[nrec].stars);
      fprintf(fid,"\n");
   }
   
   fclose(fid);
   
   for (fcount = nrec = 0; nrec < Nnew; nrec++)                            //  create missing thumbnails
   {
      pp = image_thumbfile(trec_new[nrec].file);                           //  find/update/create thumbnail
      if (pp) zfree(pp);
      snprintf(stbartext,199,"%5d %s",++fcount,trec_new[nrec].file);       //  update status bar
      stbar_message(STbar,stbartext);
   }

   for (nrec = 0; nrec < Nnew; nrec++)                                     //  release new list memory
   {
      zfree(trec_new[nrec].file);
      if (trec_new[nrec].tags) zfree(trec_new[nrec].tags);
   }
   zfree(trec_new);
   trec_new = 0;

   image_gallery(image_file,"init");                                       //  reset image gallery file list
   image_gallery(0,"paint2");                                              //  refresh gallery window if active
   Fsearchlist = 0;

   set_cursor(0);                                                          //  restore normal cursor
   menulock(0);
   
   printf("tags index completed \n");
   zmessageACK(GTK_WINDOW(mWin),ZTX("completed"));
   return;
}


//  dialog event function - file chooser for top image directory

int tags_index_dialog_event(zdialog *zd, cchar *event)
{
   char     *pp;
   
   if (strEqu(event,"F1")) showz_userguide("index_tags");                  //  F1 help

   if (! strEqu(event,"browse")) return 0;
   pp = zgetfile(ZTX("Select top image directory"),topdirk,"folder");      //  get topmost directory
   if (! pp) return 0;
   zdialog_stuff(zd,"topdirk",pp);
   zfree(pp);
   return 0;
}


//  sort compare function - compare tag record filespecs and return
//   <0 | 0 | >0   for   file1 < | == | > file2

int tags_index_compare(cchar *rec1, cchar *rec2)
{
   char * file1 = ((tags_rec *) rec1)->file;
   char * file2 = ((tags_rec *) rec2)->file;
   return strcmp(file1,file2);
}


/**************************************************************************
   Select an area within the current image.
   Subsequent edit functions are carried out within the area.
   Otherwise, edit functions apply to the entire image.
***************************************************************************/

int         sa_stat = 0;                                                   //  0/1/2/3 = none/edit/pause/complete
int         sa_mode = 0;                                                   //  1/2/3/4 = draw/follow/range/radius
int         sa_active = 0;                                                 //  select area is complete and enabled

uint16      *sa_pixseq = 0;                                                //  mark pixels by select sequence no.
int         sa_currseq = 0;                                                //  current select sequence no.
int         sa_Ncurrseq = 0;                                               //  current sequence pixel count

uint16      *sa_pixisin = 0;                                               //  mark pixels, 1=edge, 2+=inside (edge dist.)
int         sa_Npixel = 0;                                                 //  total select_area pixel count

int         sa_calced = 0;                                                 //  edge calculation done
int         sa_blend = 0;                                                  //  edge blend width

char        *sa_stackdirec = 0;                                            //  pixel search stack
int         *sa_stackii = 0;
int         sa_maxstack;
int         sa_Nstack;

int         sa_minx, sa_maxx;                                              //  enclosing rectangle for area
int         sa_miny, sa_maxy;

//  functions and data for area selection using the mouse

void     sa_draw_mousefunc();                                              //  line drawing functions
int      sa_nearpix(int mx, int my, int rad, int &npx, int &npy);
void     sa_drawline(int px1, int py1, int px2, int py2);
void     sa_followedge(int mx1, int my1, int &mx2, int &my2);
double   sa_getcontrast(int mx, int my);
void     sa_color_mousefunc();                                             //  color range functions
void     sa_select_pixels();
void     sa_unselect_pixels();
void     sa_radius_mousefunc();                                            //  radius function

uint16      sa_endpx[10000], sa_endpy[10000];                              //  last pixel drawn per seqence no.
int         sa_maxseq = 9999;
int         sa_thresh;                                                     //  mouse pixel distance threshold
int         sa_radius;
int         sa_mousex, sa_mousey;                                          //  mouse position in image
uint16      sa_targetRGB[3];                                               //  target pixel RGB
double      sa_targmatch;                                                  //  color range to match (0.001 to 1.0)

//  non-menu private functions

void  select_finish();                                                     //  finish - find inside pixels
void  select_edgecalc();                                                   //  calculate area pixel distances from edge
void  select_delete();                                                     //  delete area
void  select_drawpixel(int px, int py);                                    //  draw 1 pixel unconditionally


/**************************************************************************/

//  user select area dialog
//  line drawing and selection by color range are combined                 //  v.9.7

void m_select(GtkWidget *, cchar *)                                        //  menu function
{
   int   select_dialog_event(zdialog *, cchar *event);                     //  dialog event and completion funcs

   cchar  *title = ZTX("Select Area for Edits");
   cchar  *helptext = ZTX("Use F1 for context help");

   if (! image_file) return;                                               //  no image
   if (zdsela) return;                                                     //  already active
   
   if (Fpreview) edit_fullsize();                                          //  use full-size pixmaps
   if (Fimageturned) turn_image(-Fimageturned);                            //  use native orientation
   
   if (! Frgb16) {                                                         //  create Frgb16 if not already
      mutex_lock(&pixmaps_lock);
      Frgb16 = f_load(image_file,16);
      mutex_unlock(&pixmaps_lock);
      if (! Frgb16) return;
      SBupdate++;
   }

   zdsela = zdialog_new(title,mWin,null);
   zdialog_add_widget(zdsela,"label","labhelp","dialog",helptext,"space=5");
   zdialog_add_widget(zdsela,"hbox","hb1","dialog");
   zdialog_add_widget(zdsela,"vbox","vb1","hb1",0,"space=4|homog");
   zdialog_add_widget(zdsela,"vbox","vb2","hb1",0,"space=4|homog|expand");
   zdialog_add_widget(zdsela,"radio","rbdraw","vb1",ZTX("freehand draw"));
   zdialog_add_widget(zdsela,"radio","rbfollow","vb1",ZTX("follow edge"));
   zdialog_add_widget(zdsela,"radio","rbcolor","vb1",ZTX("color range"));
   zdialog_add_widget(zdsela,"radio","rbradius","vb1",ZTX("radius"));
   zdialog_add_widget(zdsela,"label","space1","vb2");
   zdialog_add_widget(zdsela,"label","space2","vb2");
   zdialog_add_widget(zdsela,"hbox","hbg","vb2",0,"expand");
   zdialog_add_widget(zdsela,"hscale","range","hbg","0|99.9|0.1|30","expand");
   zdialog_add_widget(zdsela,"hbox","hbr","vb2",0,"expand");
   zdialog_add_widget(zdsela,"spin","radius","hbr","0|999|1|0");
   zdialog_add_widget(zdsela,"label","space3","hbr",0,"expand");
   zdialog_add_widget(zdsela,"hbox","hb2","dialog",0,"space=5");
   zdialog_add_widget(zdsela,"label","labblend","hb2",Bblendwidth,"space=5");
   zdialog_add_widget(zdsela,"hscale","blendwidth","hb2","1|300|1|0","space=5|expand");
   zdialog_add_widget(zdsela,"hbox","hbb","dialog",0,"space=5");
   zdialog_add_widget(zdsela,"button","select","hbb",Bselect);             //  was "edit"   v.9.8
   zdialog_add_widget(zdsela,"button","pause","hbb",Bpause);
   zdialog_add_widget(zdsela,"button","show-hide","hbb",Bhide);
   zdialog_add_widget(zdsela,"button","color","hbb",Bcolor);
   zdialog_add_widget(zdsela,"button","finish","hbb",Bfinish);
   zdialog_add_widget(zdsela,"button","delete","hbb",Bdelete);

   zdialog_run(zdsela,select_dialog_event);                                //  run dialog - parallel

   select_show(1);                                                         //  show existing area
   sa_mode = 1;                                                            //  default mode = freehand draw
   sa_radius = 0;                                                          //  initial radius
   sa_targmatch = 0.40;                                                    //  initial color match range
   return;
}


//  dialog event and completion callback function

int select_dialog_event(zdialog *zd, cchar *event)
{
   static int  ignore1 = 0;
   int         ii, cc;
   
   if (zd->zstat)                                                          //  cancel dialog
   {
      mouseCBfunc = 0;                                                     //  disconnect mouse function
      Mcapture = 0;
      set_cursor(0);                                                       //  restore normal cursor
      paint_toparc(2);                                                     //  erase radius circle
      zdialog_free(zdsela);                                                //  kill dialog
      zdsela = null;
      return 0;
   }
   
   if (strEqu(event,"F1")) showz_userguide("area_overview");               //  F1 help    v.9.8 name corrected
   
   ii = strcmpv(event,"rbdraw","rbfollow","rbcolor","rbradius",null);
   if (ii) sa_mode = ii;

   if (strEqu(event,"radius"))                                             //  get radius of action
      zdialog_fetch(zd,"radius",sa_radius);

   if (strEqu(event,"range")) {                                            //  get color range
      zdialog_fetch(zdsela,"range",sa_targmatch);                          //  0.0 to 99.9
      sa_targmatch = 1.0 - 0.01 * sa_targmatch;                            //  target match level, 0.001 to 1.0
   }

   if (strEqu(event,"select")) {                                           //  new select or continue after pause
      sa_Npixel = sa_blend = sa_calced = sa_active = 0;
      sa_stat = 1;                                                         //  edit active
      select_show(1);
      zdialog_stuff(zdsela,"show-hide",Bhide);

      if (! sa_pixseq) {                                                   //  start new area
         cc = Fww * Fhh;
         sa_pixseq = (uint16 *) zmalloc(2*cc,"pixseq");
         memset(sa_pixseq,0,2*cc);
         sa_currseq = 0;
      }
   }

   if (strEqu(event,"show-hide")) {                                        //  toggle show/hide
      if (! Fshowarea) {
         select_show(1);
         zdialog_stuff(zdsela,"show-hide",Bhide);
      }
      else {
         select_show(0);
         zdialog_stuff(zdsela,"show-hide",Bshow);
      }
   }
   
   if (strEqu(event,"color")) {                                            //  switch colors    v.9.8
      if (FareaRGB == 'R') FareaRGB = 'G';
      else if (FareaRGB == 'G') FareaRGB = 'B';
      else FareaRGB = 'R';
      select_show(1);
      zdialog_stuff(zdsela,"show-hide",Bhide);
   }

   if (strEqu(event,"pause"))                                              //  pause edit, free mouse
      sa_stat = 2;

   if (strEqu(event,"finish")) {                                           //  finish area
      sa_stat = 2;
      select_finish();                                                     //  finish, sa_stat=3 if success
   }

   if (strEqu(event,"delete"))                                             //  delete area
      m_select_delete(0,0);
   
   if (strEqu(event,"blendwidth") && sa_active) {                          //  blend width changed
      if (ignore1) ignore1 = 0;                                            //  ignore once      v.9.4
      else if (sa_Npixel) {
         select_edgecalc();                                                //  edge calc. unless already
         if (sa_calced && zdedit) {
            zdialog_fetch(zd,"blendwidth",sa_blend);                       //  update sa_blend
            zdialog_send_event(zdedit,event);                              //  notify active edit dialog
         }
         else ignore1 = 1;                                                 //  edge calc failed or killed   v.9.4
      }                                                                    //  (gtk bug workaround, repeated event)
   }

   if (sa_stat == 1) {                                                     //  active edit mode
      if (sa_mode == 1) mouseCBfunc = sa_draw_mousefunc;                   //  connect mouse function
      if (sa_mode == 2) mouseCBfunc = sa_draw_mousefunc;
      if (sa_mode == 3) mouseCBfunc = sa_color_mousefunc;
      if (sa_mode == 4) mouseCBfunc = sa_radius_mousefunc;
      Mcapture++;
      set_cursor(drawcursor);                                              //  set draw cursor
      if (sa_mode != 4) paint_toparc(2);                                   //  erase radius circle
   }
   else {
      mouseCBfunc = 0;                                                     //  disconnect mouse function
      Mcapture = 0;
      set_cursor(0);                                                       //  normal cursor
      paint_toparc(2);                                                     //  erase radius circle
   }

   mwpaint2();                                                             //  update window
   return 0;
}


//  select area mouse function - draw lines

void sa_draw_mousefunc()
{
   int         mx1, my1, mx2, my2;
   int         npdist, npx, npy;
   int         ii, click, newseq, thresh;
   static int  drag = 0, mdx0, mdy0, mdx1, mdy1;

   sa_thresh = 4.0 / Mscale + 1;                                           //  mouse pixel distance threshold
   click = newseq = 0;
   
   if (LMclick || Mxdrag || Mydrag)                                        //  left mouse click or mouse drag
   {
      if (LMclick)                                                         //  left mouse click
      {
         LMclick = 0;
         mx1 = mx2 = Mxclick;                                              //  click position
         my1 = my2 = Myclick;
         newseq++;
         click++;
         drag = 0;
      }
      else                                                                 //  drag motion
      {
         if (Mxdown != mdx0 || Mydown != mdy0) {                           //  new drag initiated
            mdx0 = mdx1 = Mxdown;
            mdy0 = mdy1 = Mydown;
            newseq++;
         }
         mx1 = mdx1;                                                       //  drag start
         my1 = mdy1;
         mx2 = Mxdrag;                                                     //  drag position
         my2 = Mydrag;
         mdx1 = mx2;                                                       //  next drag start
         mdy1 = my2;
         drag++;
         click = 0;
      }
      
      if (Mbutton == 3)                                                    //  right mouse >> erase
      {
         while (true) {
            thresh = sa_thresh;
            npdist = sa_nearpix(mx2,my2,thresh,npx,npy);
            if (! npdist) break;
            ii = npy * Fww + npx;
            sa_pixseq[ii] = 0;
         }
         mwpaint2();
         return;
      }

      if (sa_currseq > sa_maxseq-2) {
         zmessageACK(GTK_WINDOW(mWin),ZTX("exceed %d edits"),sa_maxseq);   //  cannot continue
         return;
      }
      
      if (sa_currseq == 0 && newseq)                                       //  1st pixel(s) of 1st sequence
      {
         sa_currseq = 1;
         sa_drawline(mx1,my1,mx2,my2);                                     //  draw initial pixel or line
         sa_endpx[sa_currseq] = mx2;
         sa_endpy[sa_currseq] = my2;
         return;
      }
      
      if (click) {
         mx1 = sa_endpx[sa_currseq];                                       //  prior sequence end pixel
         my1 = sa_endpy[sa_currseq];                                       //  (before this click)
      }
      
      if (drag) {
         if (newseq) thresh = 2 * sa_thresh;                               //  new drag threshold
         else thresh = 5 * sa_thresh;                                      //  continuation drag threshold
         npx = sa_endpx[sa_currseq];                                       //  distance from prior end pixel
         npy = sa_endpy[sa_currseq];                                       //    (before this drag)
         if (abs(mx1-npx) < thresh && abs(my1-npy) < thresh) {
            mx1 = sa_endpx[sa_currseq];                                    //  if < threshold, connect this
            my1 = sa_endpy[sa_currseq];                                    //    drag to prior drag or click
         }
      }

      if (newseq || drag > 50) {
         sa_currseq++;                                                     //  next sequence no.
         drag = 1;                                                         //  drag length within sequence
      }
      
      if (sa_mode == 2) sa_followedge(mx1,my1,mx2,my2);
      else sa_drawline(mx1,my1,mx2,my2);                                   //  draw end pixel to mouse
      
      sa_endpx[sa_currseq] = mx2;                                          //  set end pixel for this sequence
      sa_endpy[sa_currseq] = my2;
   }

   else if (RMclick)                                                       //  right mouse click
   {
      RMclick = 0;
      if (! sa_currseq) return;
      for (int ii = 0; ii < Fww * Fhh; ii++)                               //  undo last draw action
         if (sa_pixseq[ii] == sa_currseq) sa_pixseq[ii] = 0;
      sa_currseq--;
      mwpaint2();
   }
   
   return;
}


//  Find the nearest pixel within a radius of a given pixel.
//  Returns distance to pixel, or zero if nothing found.
//  Returns 1 for adjacent or diagonally adjacent pixel.

int sa_nearpix(int mx, int my, int rad2, int &npx, int &npy)
{
   int      ii, rad, qx, qy, dx, dy;
   int      seq, mindist, dist;

   npx = npy = 0;
   mindist = (rad2+1) * (rad2+1);

   for (rad = 1; rad <= rad2; rad++)                                       //  seek neighbors within range
   {
      if (rad * rad > mindist) break;                                      //  can stop searching now

      for (qx = mx-rad; qx <= mx+rad; qx++)                                //  search within rad
      for (qy = my-rad; qy <= my+rad; qy++)
      {
         if (qx != mx-rad && qx != mx+rad &&                               //  exclude within rad-1
             qy != my-rad && qy != my+rad) continue;                       //  (already searched)
         if (qx < 0 || qx >= Fww) continue;
         if (qy < 0 || qy >= Fhh) continue;
         ii = qy * Fww + qx;
         seq = sa_pixseq[ii];
         if (! seq) continue;
         dx = (mx - qx) * (mx - qx);                                       //  found pixel
         dy = (my - qy) * (my - qy);
         dist = dx + dy;                                                   //  distance**2
         if (dist < mindist) {
            mindist = dist;
            npx = qx;                                                      //  save nearest pixel found
            npy = qy;
         }
      }
   }
   
   if (npx + npy) return sqrt(mindist) + 0.5;
   return 0;
}


//  draw a line between two given pixels
//  add all in-line pixels to sa_pixseq[]
  
void sa_drawline(int px1, int py1, int px2, int py2)
{
   void  select_drawline1(int px, int py);

   int      pxm, pym;
   double   slope;
   
   if (px1 == px2 && py1 == py2) {                                         //  only one pixel
      select_drawline1(px1,py1);
      return;
   }
   
   if (abs(py2 - py1) > abs(px2 - px1)) {
      slope = 1.0 * (px2 - px1) / (py2 - py1);
      if (py2 > py1) {
         for (pym = py1; pym <= py2; pym++) {
            pxm = round(px1 + slope * (pym - py1));
            select_drawline1(pxm,pym);
         }
      }
      else {
         for (pym = py1; pym >= py2; pym--) {
            pxm = round(px1 + slope * (pym - py1));
            select_drawline1(pxm,pym);
         }
      }
   }
   else {
      slope = 1.0 * (py2 - py1) / (px2 - px1);
      if (px2 > px1) {
         for (pxm = px1; pxm <= px2; pxm++) {
            pym = round(py1 + slope * (pxm - px1));
            select_drawline1(pxm,pym);
         }
      }
      else {
         for (pxm = px1; pxm >= px2; pxm--) {
            pym = round(py1 + slope * (pxm - px1));
            select_drawline1(pxm,pym);
         }
      }
   }

   return;
}


//  draw one pixel only if not already drawn

void select_drawline1(int px, int py)
{
   int ii = Fww * py + px;
   if (sa_pixseq[ii]) return;
   sa_pixseq[ii] = sa_currseq;
   select_drawpixel(px,py);
   return;
}


//  Find series of edge pixels from mx1/my1 to mx2/my2 and connect them together.
//  Return mx2/my2 = last edge pixel found (closest to input mx2/my2).

void sa_followedge(int mx1, int my1, int &mx2, int &my2)
{
   void   sa_findedge(int px, int py, int thresh, int &epx, int &epy);

   int         last = 0, epx, epy, pepx, pepy, thresh;
   double      dx, dy, epdist, f1dist;
   double      fx1, fy1, hyp, xdir, ydir;
   
   dx = mx2 - mx1;
   dy = my2 - my1;
   thresh = 0.01 * sqrt(dx*dx + dy*dy) + 0.5 * sa_thresh + 1;              //  more control with short segments

   if (mx1 == mx2 && my1 == my2) last = 1;                                 //  input line is 1 pixel
   
   pepx = mx1;
   pepy = my1;

   fx1 = mx1;
   fy1 = my1;
   
   while (true)
   {
      sa_findedge(round(fx1),round(fy1),thresh,epx,epy);                   //  find edge pixel near fx1/fy1
      
      if (pepx != epx || pepy != epy) {
         sa_drawline(pepx,pepy,epx,epy);                                   //  connect edge pixels
         pepx = epx;
         pepy = epy;
      }
      
      if (last) {
         mx2 = epx;                                                        //  return mx2/my2 =
         my2 = epy;                                                        //    last edge pixel found
         return;
      }
      
      dx = mx2 - epx;
      dy = my2 - epy;
      epdist = dx * dx + dy * dy;                                          //  edge pixel to mx2/my2
      dx = mx2 - fx1;
      dy = my2 - fy1;
      f1dist = dx * dx + dy * dy;                                          //  fx1/fy1 to mx2/my2

      if (epdist < f1dist) {
         fx1 = epx;                                                        //  if edge pixel closer, 
         fy1 = epy;                                                        //    move fx1/fy1 to edge pixel
         dx = mx2 - fx1;
         dy = my2 - fy1;
         f1dist = dx * dx + dy * dy;
         if (! f1dist) {
            last = 1;
            continue;
         }
      }
      
      hyp = sqrt(f1dist);
      xdir = dx / hyp;                                                     //  unit vector in direction to mx2/my2
      ydir = dy / hyp;
      xdir = 0.5 * thresh * xdir;                                          //  vector length = thresh/2
      ydir = 0.5 * thresh * ydir;
      fx1 = fx1 + xdir;                                                    //  move fx1/fy1 toward mx2/my2
      fy1 = fy1 + ydir;
      
      dx = mx2 - fx1;                                                      //  distance remaining to mx2/my2
      dy = my2 - fy1;
      f1dist = dx * dx + dy * dy;
      if (f1dist < thresh*thresh) last = 1;                                //  last iteration if < thresh
   }
}


//  Find highest contrast pixel within threshold distance of px/py.

void sa_findedge(int px, int py, int thresh, int &epx, int &epy)
{
   int      qx, qy;
   double   contrast, maxcontrast = 0;
   
   epx = px;
   epy = py;
   
   for (qx = px-thresh; qx <= px+thresh; qx++)
   for (qy = py-thresh; qy <= py+thresh; qy++)
   {
      contrast = sa_getcontrast(qx,qy);
      if (contrast > maxcontrast) {
         maxcontrast = contrast;
         epx = qx;
         epy = qy;
      }
   }

   return;
}


//  Find max. contrast between given pixel and its neighbors.

double sa_getcontrast(int px, int py)
{
   int         map[4][2] = { {1, 0}, {1, 1}, {0, 1}, {-1, 1} };
   int         ii, qx, qy;
   uint16      *pix1, *pix2;
   double      red, green, blue;
   double      contrast, maxcontrast = 0;
   
   if (px < 1 || px > Fww-2) return 0;                                     //  avoid edge pixels
   if (py < 1 || py > Fhh-2) return 0;
   
   for (ii = 0; ii < 4; ii++)                                              //  compare pixels around target
   {                                                                       //  e.g. (px-1,py) to (px+1,py)
      qx = map[ii][0];
      qy = map[ii][1];
      pix1 = bmpixel(Frgb16,px+qx,py+qy);
      pix2 = bmpixel(Frgb16,px-qx,py-qy);
      red = abs(pix1[0] - pix2[0]);                                        //  0 to 65536
      green = abs(pix1[1] - pix2[1]);
      blue = abs(pix1[2] - pix2[2]);
      contrast = (red + green + blue); 
      if (contrast > maxcontrast) maxcontrast = contrast;
   }
   
   return maxcontrast;
}


//  select area by color range - mouse function

void sa_color_mousefunc()
{
   int         cc = Fww * Fhh;
   static int  mxdown, mydown, drag = 0;

   if (sa_stackdirec) zfree(sa_stackdirec);                                //  allocate pixel search stack
   if (sa_stackii) zfree(sa_stackii);
   sa_stackdirec = zmalloc(cc,"stack.direc");
   sa_stackii = (int *) zmalloc(4*cc,"stack.ii");
   sa_maxstack = cc;
   sa_Nstack = 0;

   sa_mousex = sa_mousey = 0;

   if (LMclick) {                                                          //  get mouse position at click
      sa_mousex = Mxclick;
      sa_mousey = Myclick;
      LMclick = 0;
      sa_currseq++;                                                        //  new sequence number for undo
      drag = 1;
   }
      
   if (Mxdrag || Mydrag) {                                                 //  get mouse drag position
      sa_mousex = Mxdrag;
      sa_mousey = Mydrag;
      Mxdrag = Mydrag = 0;

      if (Mxdown != mxdown || Mydown != mydown) {                          //  detect if new drag started
         mxdown = Mxdown;
         mydown = Mydown;
         sa_currseq++;                                                     //  yes - new sequence number
         drag = 1;
      }
      else if (++drag > 50) {                                              //  limit work per sequence no.  v.8.7
         sa_currseq++;
         drag = 1;
      }
   }
   
   if (sa_mousex || sa_mousey) {
      sa_select_pixels();                                                  //  accumulate pixels
      mwpaint2();
   }

   if (RMclick) {
      RMclick = 0;
      if (sa_currseq) {                                                    //  remove selected pixels having
         sa_unselect_pixels();                                             //    current sequence number
         sa_currseq--;
      }
      mwpaint2();
   }

   return;
}


//  find all contiguous pixels within the specified color range

void sa_select_pixels()
{
   void  sa_select_pushstack(int px, int py, char direc);

   int      ii, kk, px, py, rx, ry;
   int      radius, radius2;
   uint16   *targpix;
   char     direc;
   
   if (! sa_pixseq) return;
   
   radius = 1 + 1.0 / Mscale;
   radius2 = radius * radius;
   
   for (rx = -radius; rx <= radius; rx++)                                  //  loop every pixel in radius of mouse
   for (ry = -radius; ry <= radius; ry++)                                  //  v.9.2
   {
      if (rx * rx + ry * ry > radius2) continue;                           //  outside radius
      px = sa_mousex + rx;
      py = sa_mousey + ry;
      if (px < 0 || px >= Fww) continue;                                   //  off the image edge
      if (py < 0 || py >= Fhh) continue;
      ii = Fww * py + px;
      if (sa_pixseq[ii]) continue;                                         //  already selected

      sa_pixseq[ii] = sa_currseq;                                          //  map pixel to current sequence
      sa_Ncurrseq = 1;                                                     //  current sequence pixel count

      targpix = bmpixel(Frgb16,px,py);                                     //  get color at mouse position
      sa_targetRGB[0] = targpix[0];                                        //    = target color
      sa_targetRGB[1] = targpix[1];
      sa_targetRGB[2] = targpix[2];
      
      sa_stackii[0] = ii;                                                  //  put 1st pixel into stack
      sa_stackdirec[0] = 'r';                                              //  direction = right
      sa_Nstack = 1;                                                       //  stack count

      while (sa_Nstack)
      {
         kk = sa_Nstack - 1;                                               //  get last pixel in stack
         ii = sa_stackii[kk];
         direc = sa_stackdirec[kk];
         
         py = ii / Fww;                                                    //  reconstruct px, py
         px = ii - Fww * py;

         if (direc == 'x') {                                               //  no neighbors left to check
            sa_Nstack--;
            continue;
         }
         
         if (direc == 'r') {                                               //  push next right pixel into stack
            sa_select_pushstack(px,py,'r');                                //    if color within range
            sa_stackdirec[kk] = 'l';                                       //  this pixel next direction to look
            continue;
         }

         if (direc == 'l') {                                               //  or next left pixel
            sa_select_pushstack(px,py,'l');
            sa_stackdirec[kk] = 'a';
            continue;
         }

         if (direc == 'a') {                                               //  or next ahead pixel
            sa_select_pushstack(px,py,'a');
            sa_stackdirec[kk] = 'x';
            continue;
         }
      }
   }

   return;
}      


//  push pixel into stack memory if its color is within range
//  and not already mapped to a prior sequence number

void sa_select_pushstack(int px, int py, char direc)
{
   int         ii, kk, ppx, ppy, npx, npy;
   uint16      *matchpix;
   double      match, ff = 1.0 / 65536.0;
   double      dred, dgreen, dblue;

   if (sa_Nstack > 1) {
      kk = sa_Nstack - 2;                                                  //  get prior pixel in stack
      ii = sa_stackii[kk];
      ppy = ii / Fww;
      ppx = ii - ppy * Fww;
   }
   else {
      ppx = px - 1;                                                        //  if only one, assume prior = left
      ppy = py;
   }
   
   if (direc == 'r') {                                                     //  get pixel in direction right
      npx = px + ppy - py;
      npy = py + px - ppx;
   }
   else if (direc == 'l') {                                                //  or left
      npx = px + py - ppy;
      npy = py + ppx - px;
   }
   else if (direc == 'a') {                                                //  or ahead
      npx = px + px - ppx;
      npy = py + py - ppy;
   }
   else npx = npy = -1;                                                    //  stop warning
   
   if (npx < 0 || npx >= Fww) return;                                      //  pixel off the edge
   if (npy < 0 || npy >= Fhh) return;
   
   ii = npy * Fww + npx;
   if (sa_pixseq[ii]) return;                                              //  pixel already mapped
   
   matchpix = bmpixel(Frgb16,npx,npy);                                     //  match pixel RGB colors
   dred =   ff * abs(sa_targetRGB[0] - matchpix[0]);                       //    with target pixel colors
   dgreen = ff * abs(sa_targetRGB[1] - matchpix[1]);
   dblue =  ff * abs(sa_targetRGB[2] - matchpix[2]);
   match = (1.0 - dred) * (1.0 - dgreen) * (1.0 - dblue);
   if (match < sa_targmatch) return;                                       //  inadequate match

   if (sa_Nstack == sa_maxstack) return;                                   //  stack is full

   sa_pixseq[ii] = sa_currseq;                                             //  map pixel to curr. sequence
   sa_Ncurrseq++;
   
   kk = sa_Nstack++;                                                       //  put pixel into stack
   sa_stackii[kk] = ii;
   sa_stackdirec[kk] = 'r';                                                //  direction = right

   return;
}


//  un-select all pixels mapped to current sequence number

void sa_unselect_pixels()
{
   if (! sa_currseq) return;
   
   for (int ii = 0; ii < Fww * Fhh; ii++)
   {
      if (sa_pixseq[ii] != sa_currseq) continue;
      sa_pixseq[ii] = 0;
   }
   
   sa_Ncurrseq = 0;
   return;
}


//  select or un-select all pixels within radius - mouse function

void  sa_radius_mousefunc()
{
   int      ii, radius, radius2;
   int      px, py, rx, ry;
   
   radius = sa_radius;                                                     //  pixel selection radius
   radius2 = radius * radius;

   if (! radius) {
      paint_toparc(2);                                                     //  no radius, no circle
      return;
   }

   toparcx = Mxposn - radius;                                              //  draw radius outline circle
   toparcy = Myposn - radius;
   toparcw = toparch = 2 * radius;
   Ftoparc = 1;
   paint_toparc(3);

   if (LMclick || RMclick) {                                               //  mouse click
      if (sa_Ncurrseq) {
         sa_currseq++;                                                     //  start new sequence number
         sa_Ncurrseq = 0;                                                  //    if prior one has pixels
      }
      LMclick = RMclick = 0;
   }

   if (Mbutton != 1 && Mbutton != 3) {                                     //  no button is pressed
      if (sa_Ncurrseq) {
         sa_currseq++;                                                     //  start new sequence number
         sa_Ncurrseq = 0;                                                  //    if prior one has pixels
      }
      return;
   }
   
   for (rx = -radius; rx <= radius; rx++)                                  //  loop every pixel in radius
   for (ry = -radius; ry <= radius; ry++)
   {
      if (rx * rx + ry * ry > radius2) continue;                           //  outside radius
      px = Mxposn + rx;
      py = Myposn + ry;
      if (px < 0 || px >= Fww) continue;                                   //  off the image edge
      if (py < 0 || py >= Fhh) continue;

      ii = Fww * py + px;
      
      if (Mbutton == 3)                                                    //  right mouse button
         sa_pixseq[ii] = 0;                                                //  remove pixel from select area

      if (Mbutton == 1)                                                    //  left mouse button
      {
         if (sa_pixseq[ii]) continue;                                      //  pixel already selected

         if (sa_Ncurrseq > 50) {                                           //  start new sequence no.
            sa_currseq++;                                                  //    after 50 steps
            sa_Ncurrseq = 0;
         }

         sa_pixseq[ii] = sa_currseq;                                       //  map pixel to current sequence
         sa_Ncurrseq++;
      }
   }

   mwpaint2();
}


/**************************************************************************
   menu functions common to both mouse-selected and color-selected areas
***************************************************************************/

//  invert a selected area

void m_select_invert(GtkWidget *, cchar *)                                 //  v.8.7
{
   int      ii, jj, kk;

   if (! sa_Npixel) {
      zmessageACK(GTK_WINDOW(mWin),ZTX("the area is not finished"));
      return;
   }

   for (ii = kk = 0; ii < Fww * Fhh; ii++)                                 //  reverse member pixels
   {
      jj = sa_pixisin[ii];                                                 //  0/1/2 = outside/edge/inside

      if (jj > 1)                                                          //  inside pixel >> outside
         sa_pixisin[ii] = sa_pixseq[ii] = 0;
      else if (jj == 1)                                                    //  edge >> edge
         kk++;                                                             //  count
      else  {                                                              //  outside pixel >> inside
         sa_pixisin[ii] = sa_pixseq[ii] = 2;
         kk++;                                                             //  count
      }
   }

   sa_Npixel = kk;                                                         //  new select area pixel count
   sa_calced = sa_blend = 0;                                               //  edge calculation missing
   return;
}


//  enable select area that was disabled

void m_select_enable(GtkWidget *, cchar *)
{
   if (! sa_Npixel) return;                                                //  v.9.7
   sa_active = 1;
   return;
}


//  disable select area

void m_select_disable(GtkWidget *, cchar *)
{
   sa_active = 0;                                                          //  v.9.7
   return;
}


//  clear selected image area, free memory

void m_select_delete(GtkWidget *, cchar *)                                 //  v.8.7
{
   select_delete();
   return;
}


/**************************************************************************
   select area non-menu functions
***************************************************************************/

//  finish select area - map pixels enclosed by edge pixels 
//    into  sa_pixisin[]   1=edge, 2=inside, [ii]=py*Fww+px
//  total count = sa_Npixel

namespace select_finish_names
{
   zdialog     *finzd = 0;
   int         finish_fails;
   int         pxmin, pxmax, pymin, pymax;
}

void select_finish()                                                       //  overhauled          v.9.7
{
   using namespace select_finish_names;

   void select_finish_mousefunc();
   int  select_finish_dialog_event(zdialog *, cchar *event);

   cchar  *fmess = ZTX("Search all areas for edge and inside pixels. \n"
                       "Click inside each enclosed area in sequence.");
   int         ii, kk, cc, px, py, npix;

   if (! sa_pixseq) return;                                                //  nothing selected
   sa_Npixel = sa_active = 0;                                              //  area disabled, unfinished
   sa_stat = 2;                                                            //  disable edits

   if (sa_pixisin) zfree(sa_pixisin);                                      //  allocate pixisin[]
   cc = Fww * Fhh * 2;
   sa_pixisin = (uint16 *) zmalloc(cc,"pixisin");
   memset(sa_pixisin,0,cc);                                                //  clear included pixel map

   pxmin = Fww;
   pxmax = 0;
   pymin = Fhh;
   pymax = 0;

   for (ii = 0; ii < Fww * Fhh; ii++)                                      //  get enclosing rectangle
   {                                                                       //    for selected area
      if (! sa_pixseq[ii]) continue;
      py = ii / Fww;
      px = ii - Fww * py;
      if (px > pxmax) pxmax = px;
      if (px < pxmin) pxmin = px;
      if (py > pymax) pymax = py;
      if (py < pymin) pymin = py;
   }
   
   sa_minx = pxmin;                                                        //  save enclosing rectangle     v.10.1
   sa_maxx = pxmax;
   sa_miny = pymin;
   sa_maxy = pymax;
   
   pxmin -= 10;                                                            //  add margins if possible      v.8.8
   if (pxmin < 0) pxmin = 0;                                               //  (enhance gap visibility if
   pxmax += 10;                                                            //    finish algorithm fails)
   if (pxmax > Fww-1) pxmax = Fww - 1;
   pymin -= 10;
   if (pymin < 0) pymin = 0;
   pymax += 10;
   if (pymax > Fhh-1) pymax = Fhh - 1;
   
   cc = (pxmax-pxmin) * (pymax-pymin);                                     //  allocate stack memory
   if (sa_stackdirec) zfree(sa_stackdirec);
   sa_stackdirec = zmalloc(cc,"stack.direc");
   if (sa_stackii) zfree(sa_stackii);
   sa_stackii = (int *) zmalloc(cc*4,"stack.ii");
   sa_maxstack = cc;

   for (ii = npix = 0; ii < Fww * Fhh; ii++)                               //  find pixels already included in area
   {
      if (! sa_pixseq[ii]) continue;
      sa_pixisin[ii] = 1;                                                  //  populate pixisin[]
      npix++;
   }

   if (npix < 20) return;                                                  //  ridiculous

   for (py = 0; py < Fhh; py++)                                            //  find edge pixels
   for (px = 0; px < Fww; px++)
   {
      ii = py * Fww + px;
      if (! sa_pixisin[ii]) continue;                                      //  outside of selected area

      if (! sa_pixisin[ii-1] || ! sa_pixisin[ii+1]) goto edgepixel;        //  check 8 neighbor pixels
      kk = ii - Fww;
      if (! sa_pixisin[kk] || ! sa_pixisin[kk-1] || ! sa_pixisin[kk+1]) goto edgepixel;
      kk = ii + Fww;
      if (! sa_pixisin[kk] || ! sa_pixisin[kk-1] || ! sa_pixisin[kk+1]) goto edgepixel;

      sa_pixisin[ii] = 2;                                                  //  interior pixel
      continue;
   edgepixel:
      sa_pixisin[ii] = 1;                                                  //  edge pixel
   }

   for (ii = 0; ii < Fww * Fhh; ii++)                                      //  eliminate interior pixels
      if (sa_pixisin[ii] == 2) sa_pixisin[ii] = 0;

   mouseCBfunc = select_finish_mousefunc;                                  //  connect mouse function
   Mcapture++;
   set_cursor(0);                                                          //  normal cursor
   finish_fails = 0;

   finzd = zdialog_new(ZTX("finish area"),mWin,Bdone,Bcancel,null);
   zdialog_add_widget(finzd,"label","fmess","dialog",fmess);
   zdialog_add_widget(finzd,"label","smess","dialog","status:");

   zdialog_run(finzd,select_finish_dialog_event);                          //  run dialog, parallel
   zdialog_wait(finzd);
   return;
}


//  dialog event and completion callback function

int  select_finish_dialog_event(zdialog *zd, cchar *event)
{
   using namespace select_finish_names;

   int         zstat, ii, kk, npix, px, py;
   
   mouseCBfunc = 0;                                                        //  disconnect mouse
   Mcapture = 0;

   zstat = zd->zstat;

   zdialog_free(finzd);                                                    //  kill dialog
   finzd = 0;

   if (zstat != 1 || finish_fails) {                                       //  user cancel or pixel search failure
      mwpaint2();
      return 0;
   }

   npix = 0;

   for (px = 0; px < Fww; px += 1)                                         //  image top & bottom edges
   for (py = 0; py < Fhh; py += Fhh-1)
   {
      ii = py * Fww + px;
      if (! sa_pixisin[ii]) continue;
      sa_pixisin[ii] = 2;                                                  //  if selected, force non-edge
      npix++;
   }

   for (px = 0; px < Fww; px += Fww-1)                                     //  image left and right edges
   for (py = 0; py < Fhh; py += 1)
   {
      ii = py * Fww + px;
      if (! sa_pixisin[ii]) continue;
      sa_pixisin[ii] = 2;                                                  //  if selected, force non-edge
      npix++;
   }

   for (py = 1; py < Fhh-1; py++)                                          //  check all other pixels
   for (px = 1; px < Fww-1; px++)
   {
      ii = py * Fww + px;
      if (! sa_pixisin[ii]) continue;                                      //  outside of selected area
      npix++;

      if (! sa_pixisin[ii-1] || ! sa_pixisin[ii+1]) goto edgepixel;        //  check 8 neighbor pixels
      kk = ii - Fww;
      if (! sa_pixisin[kk] || ! sa_pixisin[kk-1] || ! sa_pixisin[kk+1]) goto edgepixel;
      kk = ii + Fww;
      if (! sa_pixisin[kk] || ! sa_pixisin[kk-1] || ! sa_pixisin[kk+1]) goto edgepixel;

      sa_pixisin[ii] = 2;                                                  //  non-edge pixel
      continue;
   edgepixel:
      sa_pixisin[ii] = 1;                                                  //  edge pixel
   }

   sa_Npixel = npix;                                                       //  total pixel count
   sa_stat = 3;                                                            //  area is finished
   sa_active = 1;                                                          //  area is active by default
   sa_calced = sa_blend = 0;                                               //  edge calculation is missing

   mwpaint2();
   return 0;
}


//  mouse function - get user clicks and perform pixel searches

void select_finish_mousefunc()
{
   using namespace select_finish_names;

   void  select_finish_pushstack(int px, int py, char direc);
   
   int         seedpx, seedpy;
   int         px, py, ii, kk;
   char        direc;

   if (! LMclick) return;
   LMclick = 0;

   seedpx = Mxclick;
   seedpy = Myclick;
   
   zdialog_stuff(finzd,"smess",ZTX("searching"));

   ii = seedpy * Fww + seedpx;                                             //  seed pixel
   sa_stackii[0] = ii;                                                     //  put 1st pixel into stack
   sa_stackdirec[0] = 'r';                                                 //  direction = right
   sa_Nstack = 1;                                                          //  stack count

   while (sa_Nstack)                                                       //  find all pixels inside drawn area
   {
      kk = sa_Nstack - 1;                                                  //  get last pixel in stack
      ii = sa_stackii[kk];
      direc = sa_stackdirec[kk];
      
      py = ii / Fww;                                                       //  reconstruct px, py
      px = ii - Fww * py;

      if (px < pxmin || px > pxmax || py < pymin || py > pymax) {          //  ran off the edge, seed pixel
         zdialog_stuff(finzd,"smess",ZTX("area outline has a hole"));      //    was not inside an area or
         finish_fails++;                                                   //      area boundary has a hole
         return;      
      }

      if (direc == 'x') {                                                  //  no neighbors left to check
         sa_Nstack--;
         continue;
      }
      
      if (direc == 'r') {                                                  //  push next right pixel into stack
         select_finish_pushstack(px,py,'r');
         sa_stackdirec[kk] = 'l';                                          //  this pixel next direction to look
         continue;
      }

      if (direc == 'l') {                                                  //  or next left pixel
         select_finish_pushstack(px,py,'l');
         sa_stackdirec[kk] = 'a';
         continue;
      }

      if (direc == 'a') {                                                  //  or next ahead pixel
         select_finish_pushstack(px,py,'a');
         sa_stackdirec[kk] = 'x';
         continue;
      }
   }
   
   zdialog_stuff(finzd,"smess",ZTX("success"));
   return;
}


//  push pixel into stack memory if not already mapped

void select_finish_pushstack(int px, int py, char direc)
{
   int      ii, kk, ppx, ppy, npx, npy;

   if (sa_Nstack > 1) {
      kk = sa_Nstack - 2;                                                  //  get prior pixel in stack
      ii = sa_stackii[kk];
      ppy = ii / Fww;
      ppx = ii - ppy * Fww;
   }
   else {
      ppx = px - 1;                                                        //  if only one, assume prior = left
      ppy = py;
   }
   
   if (direc == 'a') {                                                     //  get pixel in direction ahead
      npx = px + ppy - py;                                                 //  leak >> straight line out
      npy = py + px - ppx;
   }
   else if (direc == 'l') {                                                //  or left
      npx = px + py - ppy;
      npy = py + ppx - px;
   }
   else if (direc == 'r') {                                                //  or right
      npx = px + px - ppx;
      npy = py + py - ppy;
   }
   else npx = npy = -1;                                                    //  stop warning
   
   if (npx < 0 || npx >= Fww) return;                                      //  pixel off the edge
   if (npy < 0 || npy >= Fhh) return;
   
   ii = npy * Fww + npx;
   if (sa_pixisin[ii]) return;                                             //  pixel already mapped
   if (sa_Nstack == sa_maxstack) return;                                   //  stack is full (impossible)
   
   sa_pixisin[ii] = 1;                                                     //  pixel is in area
   kk = sa_Nstack++;                                                       //  put pixel into stack
   sa_stackii[kk] = ii;
   sa_stackdirec[kk] = 'r';                                                //  direction = right
   select_drawpixel(npx,npy);                                              //  feedback to see holes     v.9.7

   return;
}


//  compute distance from all pixels in area to nearest edge

namespace select_edgecalc_names
{
   uint16      *sa_edgepx, *sa_edgepy, *sa_edgedist;
   int         sa_Nedge;
}

void select_edgecalc()                                                     //  overhauled   v.8.7
{
   using namespace select_edgecalc_names;
   
   int    edgecalc_dialog_event(zdialog*, cchar *event);
   void * edgecalc_wthread(void *);

   int         ii, nn, cc, px, py;
   zdialog     *zecdialog = 0;
   cchar       *zectext = ZTX("Edge calculation in progress");

   if (sa_calced) return;                                                  //  done already
   if (! sa_active) select_finish();                                       //  finish if needed
   if (! sa_active) return;                                                //  no finished area
   
   zecdialog = zdialog_new(ZTX("Area Edge Calc"),mWin,Bcancel,null);
   zdialog_add_widget(zecdialog,"label","lab1","dialog",zectext,"space=10");
   zdialog_run(zecdialog,edgecalc_dialog_event);

   cc = Fww * Fhh * 2;                                                     //  allocate memory for calculations
   sa_edgedist = (uint16 *) zmalloc(cc,"edge.dist");
   memset(sa_edgedist,0,cc);
   
   for (ii = nn = 0; ii < Fww * Fhh; ii++)                                 //  count edge pixels in select area
      if (sa_pixisin[ii] == 1) nn++;

   sa_edgepx = (uint16 *) zmalloc(nn*2,"edge.px");                         //  allocate memory
   sa_edgepy = (uint16 *) zmalloc(nn*2,"edge.py");

   for (ii = nn = 0; ii < Fww * Fhh; ii++)                                 //  build list of edge pixels
   {                                                                       //  v.9.6
      if (sa_pixisin[ii] != 1) continue;
      py = ii / Fww;
      px = ii - py * Fww;
      sa_edgepx[nn] = px;
      sa_edgepy[nn] = py;
      nn++;
   }

   sa_Nedge = nn;
   progress_goal = sa_Npixel;
   progress_done = 0;
   
   for (int ii = 0; ii < Nwt; ii++)                                        //  start worker threads to calculate
      start_wt(edgecalc_wthread,&wtnx[ii]);                                //    sa_pixisin[] edge distances
   wait_wts();                                                             //  wait for completion

   progress_goal = progress_done = 0;                                      //  v.9.6
   
   zdialog_free(zecdialog);                                                //  kill dialog
   sa_calced = 1;                                                          //  edge calculation available

   zfree(sa_edgedist);                                                     //  free memory
   zfree(sa_edgepx);
   zfree(sa_edgepy);

   if (Fkillfunc) {
      Fkillfunc = 0;
      sa_calced = 0;
   }

   return;
}


//  dialog event and completion callback function

int edgecalc_dialog_event(zdialog *zd, cchar *event)                       //  respond to user cancel
{
   Fkillfunc = 1;
   printf("edge calc killed \n");
   return 0;
}


void * edgecalc_wthread(void *arg)                                         //  worker thread function
{
   using namespace select_edgecalc_names;
   
   void  edgecalc_f1(int px, int py);
   void  edgecalc_f2(int px, int py);

   int      index = *((int *) (arg));
   int      ii, nn, px, py;
   int64    seed = 34567 * index;

   for (nn = 0; nn < Fww * Fhh * 2; nn++)                                  //  do random pixels    v.9.6
   {
      ii = drandz(&seed) * Fww * Fhh;
      if (sa_pixisin[ii] < 2) continue;                                    //  ignore outside and edge pixels
      if (sa_edgedist[ii]) continue;                                       //  already calculated
      py = ii / Fww;
      px = ii - py * Fww;
      edgecalc_f1(px,py);                                                  //  calculate edge distance
      if (Fkillfunc) exit_wt();
   }
   
   for (ii = index; ii < Fww * Fhh; ii += Nwt)                             //  do all pixels       v.9.6
   {
      if (sa_pixisin[ii] < 2) continue;
      if (sa_edgedist[ii]) continue;
      py = ii / Fww;
      px = ii - py * Fww;
      edgecalc_f2(px,py);
      if (Fkillfunc) exit_wt();
   }

   for (ii = index; ii < Fww * Fhh; ii += Nwt)                             //  copy data from sa_edgedist[]
   {                                                                       //    to sa_pixisin[]
      if (sa_pixisin[ii] < 2) continue;
      sa_pixisin[ii] = sa_edgedist[ii];
   }

   exit_wt();
   return 0;                                                               //  not executed, stop gcc warning
}


//  Find the nearest edge pixel for a given pixel.
//  For all pixels in a line from the given pixel to the edge pixel, 
//  the same edge pixel can be used to compute edge distance.

void edgecalc_f1(int px1, int py1)
{
   using namespace select_edgecalc_names;
   
   int      ii, px2, py2;
   uint     dist2, mindist2;
   int      epx, epy, pxm, pym, dx, dy, inc;
   double   slope;

   mindist2 = 2000000000;
   epx = epy = 0;

   for (ii = 0; ii < sa_Nedge; ii++)                                       //  loop all edge pixels   v.8.8
   {
      px2 = sa_edgepx[ii];
      py2 = sa_edgepy[ii];
      if (px2 < 4 || px2 > Fww-5) continue;                                //  omit pixels < 4 from image edge
      if (py2 < 4 || py2 > Fhh-5) continue;                                //                              v.9.2
      dx = px2 - px1;
      dy = py2 - py1;
      dist2 = dx*dx + dy*dy;                                               //  avoid sqrt()       v.8.8
      if (dist2 < mindist2) {
         mindist2 = dist2;                                                 //  remember minimum
         epx = px2;                                                        //  remember nearest edge pixel
         epy = py2;
      }
   }
   
   if (abs(epy - py1) > abs(epx - px1)) {                                  //  find all pixels along a line
      slope = 1.0 * (epx - px1) / (epy - py1);                             //    to the edge pixel
      if (epy > py1) inc = 1;
      else inc = -1;
      for (pym = py1; pym != epy; pym += inc) {
         pxm = px1 + slope * (pym - py1);
         ii = pym * Fww + pxm;
         if (sa_edgedist[ii]) continue;
         dx = epx - pxm;                                                   //  calculate distance to edge
         dy = epy - pym;
         dist2 = sqrt(dx*dx + dy*dy) + 0.5;
         sa_edgedist[ii] = dist2;                                          //  save
         progress_done++;
      }
   }

   else {
      slope = 1.0 * (epy - py1) / (epx - px1);
      if (epx > px1) inc = 1;
      else inc = -1;
      for (pxm = px1; pxm != epx; pxm += inc) {
         pym = py1 + slope * (pxm - px1);
         ii = pym * Fww + pxm;
         if (sa_edgedist[ii]) continue;
         dx = epx - pxm;
         dy = epy - pym;
         dist2 = sqrt(dx*dx + dy*dy) + 0.5;
         sa_edgedist[ii] = dist2;
         progress_done++;
      }
   }

   return;
}


void edgecalc_f2(int px1, int py1)                                         //  calculate 1 pixel
{
   using namespace select_edgecalc_names;
   
   int      ii, kk, px2, py2;
   uint     dist2, mindist2;
   int      epx, epy, dx, dy;

   mindist2 = 2000000000;
   epx = epy = 0;

   for (ii = 0; ii < Fww * Fhh; ii++)
   {
      if (sa_pixisin[ii] != 1) continue;                                   //  find all edge pixels
      py2 = ii / Fww;
      px2 = ii - py2 * Fww;
      if (px2 < 4 || px2 > Fww-5) continue;                                //  < pixels 4 from image edge   v.9.2
      if (py2 < 4 || py2 > Fhh-5) continue;
      dx = px2 - px1;                                                      //  calculate distance to edge pixel
      dy = py2 - py1;
      dist2 = dx*dx + dy*dy;                                               //  avoid sqrt()       v.8.8
      if (dist2 < mindist2) {
         mindist2 = dist2;                                                 //  remember minimum
         epx = px2;                                                        //  remember nearest edge pixel
         epy = py2;
      }
   }
   
   kk = py1 * Fww + px1;
   sa_edgedist[kk] = sqrt(mindist2) + 0.5;

   progress_done++;
   return;
}


//  show outline of select area
//  also called from mwpaint() if Fshowarea = 1

void select_show(int flag)
{
   int      px, py, ii, kk;
   
   Fshowarea = flag;
   if (! flag) return;
   if (! sa_pixseq) return;
   
   for (py = 1; py < Fhh-1; py++)                                          //  find pixels in area
   for (px = 1; px < Fww-1; px++)
   {
      ii = py * Fww + px;
      if (! sa_pixseq[ii]) continue;                                       //  outside of area

      if (! sa_pixseq[ii-1] || ! sa_pixseq[ii+1]) goto edgepixel;          //  check 8 neighbor pixels
      kk = ii - Fww;
      if (! sa_pixseq[kk] || ! sa_pixseq[kk-1] || ! sa_pixseq[kk+1]) goto edgepixel;
      kk = ii + Fww;
      if (! sa_pixseq[kk] || ! sa_pixseq[kk-1] || ! sa_pixseq[kk+1]) goto edgepixel;
      continue;

   edgepixel:
      select_drawpixel(px,py);
   }

   gdk_gc_set_foreground(gdkgc,&black);
   return;
}


//  callable version without user query

void select_delete()
{
   sa_stat = sa_Npixel = sa_blend = sa_calced = sa_active = 0;
   sa_currseq = sa_Ncurrseq = 0;
   if (sa_pixisin) zfree(sa_pixisin);
   if (sa_pixseq) zfree(sa_pixseq);
   if (sa_stackii) zfree(sa_stackii);
   if (sa_stackdirec) zfree(sa_stackdirec);
   sa_pixisin = 0;
   sa_pixseq = 0;
   sa_stackii = 0;
   sa_stackdirec = 0;
   mwpaint2();
   return;
}


//  draw one pixel using current FareaRGB color

void select_drawpixel(int px, int py)
{
   int            qx, qy;
   static int     pqx, pqy;

   if (FareaRGB == 'G') gdk_gc_set_foreground(gdkgc,&green);               //  v.9.8
   else if (FareaRGB == 'B') gdk_gc_set_foreground(gdkgc,&black);
   else gdk_gc_set_foreground(gdkgc,&red);

   qx = Mscale * (px-Iorgx) + 0.5;                                         //  image to window space
   qy = Mscale * (py-Iorgy) + 0.5;
   if (qx == pqx && qy == pqy) return;                                     //  avoid redundant points   v.9.7
   pqx = qx;
   pqy = qy;

   gdk_draw_point(drWin->window,gdkgc,qx+Dorgx,qy+Dorgy);                  //  draw pixel

   gdk_gc_set_foreground(gdkgc,&black);
   return;
}


/**************************************************************************
   select area copy and paste menu functions         v.8.7
***************************************************************************/

RGB      *sacp_rgb16 = 0;                                                  //  select area pixmap copy
int      sacp_ww, sacp_hh;                                                 //  dimensions
uint16   *sacp_dist = 0;                                                   //  pixel edge distances

RGB      *sacpR_rgb16 = 0;                                                 //  resized pixmap
int      sacpR_ww, sacpR_hh;                                               //  dimensions
double   sacp_resize;                                                      //  size, 1.0 = original size
int      sacp_orgx, sacp_orgy;                                             //  origin in target image
int      sacp_blend;                                                       //  edge blend with target image
int      sacp_susp = 0;                                                    //  dialog is suspended

void select_paste_pixmap();                                                //  copy saved area into image


//  copy selected area, save in memory

void m_select_copy(GtkWidget *, cchar *)                                   //  menu function
{
   int      ii, jj, cc, px, py, qx, qy, dist;
   int      pxmin, pxmax, pymin, pymax;
   uint16   *pix1, *pix2;

   if (! sa_active) select_finish();                                       //  finish area if not already
   if (! sa_active) return;
   select_edgecalc();                                                      //  do edge calc if not already
   
   pxmin = Fww;
   pxmax = 0;
   pymin = Fhh;
   pymax = 0;

   for (ii = 0; ii < Fww * Fhh; ii++)                                      //  find pixels in select area
   {                                                                       //  v.9.6
      if (! sa_pixisin[ii]) continue;
      py = ii / Fww;
      px = ii - py * Fww;
      if (px > pxmax) pxmax = px;                                          //  find enclosing rectangle
      if (px < pxmin) pxmin = px;
      if (py > pymax) pymax = py;
      if (py < pymin) pymin = py;
   }
   
   RGB_free(sacp_rgb16);
   sacp_ww = pxmax - pxmin + 11;                                           //  create new RGB pixmap for area
   sacp_hh = pymax - pymin + 11;                                           //  add 5-pixel margins    v.8.8
   sacp_rgb16 = RGB_make(sacp_ww,sacp_hh,16);
   
   for (py = 0; py < sacp_hh; py++)                                        //  clear all pixels to black
   for (px = 0; px < sacp_ww; px++)
   {
      pix2 = bmpixel(sacp_rgb16,px,py);
      pix2[0] = pix2[1] = pix2[2] = 0;
   }
   
   if (sacp_dist) zfree(sacp_dist);
   cc = 2 * sacp_ww * sacp_hh;                                             //  set all edge distances = 0
   sacp_dist = (uint16 *) zmalloc(cc+100,"sacp.dist");
   memset(sacp_dist,0,cc);
   
   for (ii = 0; ii < Fww * Fhh; ii++)                                      //  find pixels in select area
   {                                                                       //  v.9.6
      if (! sa_pixisin[ii]) continue;
      py = ii / Fww;
      px = ii - py * Fww;
      dist = sa_pixisin[ii];
      pix1 = bmpixel(Frgb16,px,py);                                        //  copy pixel into pixmap
      px = px - pxmin + 5;                                                 //  leave margin           v.8.8
      py = py - pymin + 5;
      pix2 = bmpixel(sacp_rgb16,px,py);
      pix2[0] = pix1[0];
      pix2[1] = pix1[1];
      pix2[2] = pix1[2];
      jj = py * sacp_ww + px;
      sacp_dist[jj] = dist + 1;                                            //  0/1/2+ = outside/edge/inside
   }

   for (ii = 0; ii < sacp_ww * sacp_hh; ii++)                              //  propagate edge pixels outward,
   {                                                                       //    prevent mixing black pixels
      if (sacp_dist[ii] != 1) continue;                                    //      when pixmap is rescaled
      py = ii / sacp_ww;                                                   //                         v.8.8
      px = ii - sacp_ww * py;
      pix1 = bmpixel(sacp_rgb16,px,py);
      for (qx = px-4; qx <= px+4; qx++)
      for (qy = py-4; qy <= py+4; qy++)
      {
         pix2 = bmpixel(sacp_rgb16,qx,qy);
         if (pix2[0] || pix2[1] || pix2[2]) continue;
         pix2[0] = pix1[0];
         pix2[1] = pix1[1];
         pix2[2] = pix1[2];
      }
   }  

   return;
}


//  paste selected area into current image
//  this is an edit function - select area image is copied into main image

void m_select_paste(GtkWidget *, cchar *)                                  //  menu function
{
   int   select_paste_dialog_event(zdialog *, cchar *event);
   void  select_paste_mousefunc();

   cchar  *dragmess = ZTX("position image\nwith mouse drag");
   
   if (! sacp_rgb16) return;                                               //  nothing to paste
   select_delete();                                                        //  delete select area if present   v.9.7
   if (! edit_setup("paste",0,0)) return;                                  //  setup edit
   
   sacp_resize = 1.0;                                                      //  size = 1x
   sacp_blend = 1;                                                         //  edge blend = 1
   sacp_susp = 0;                                                          //  not suspended

   sacpR_ww = sacp_ww;                                                     //  setup resized paste image
   sacpR_hh = sacp_hh;                                                     //  (initially 1x)
   sacpR_rgb16 = RGB_copy(sacp_rgb16);
   
   sacp_orgx = Iorgx + iww/2 - sacp_ww/2;                                  //  paste at center of visible window
   sacp_orgy = Iorgy + ihh/2 - sacp_hh/2;                                  //  v.9.7
   if (sacp_orgx < 0) sacp_orgx = 0;
   if (sacp_orgy < 0) sacp_orgy = 0;
   select_paste_pixmap();
   
   zdedit = zdialog_new(ZTX("Paste Image"),mWin,Bdone,Bcancel,null);
   zdialog_add_widget(zdedit,"hbox","hb0","dialog",0,"space=8");
   zdialog_add_widget(zdedit,"label","lab1","hb0",dragmess,"space=8");
   zdialog_add_widget(zdedit,"button","susp-resm","hb0",Bsuspend);
   zdialog_add_widget(zdedit,"hbox","hb1","dialog",0,"space=5");
   zdialog_add_widget(zdedit,"label","lab1","hb1","resize");
   zdialog_add_widget(zdedit,"button","+.1%","hb1","+.1%");
   zdialog_add_widget(zdedit,"button","+1%","hb1","+1%");
   zdialog_add_widget(zdedit,"button","+10%","hb1","+10%");
   zdialog_add_widget(zdedit,"button","-.1%","hb1","-.1%");
   zdialog_add_widget(zdedit,"button","-1%","hb1","-1%");
   zdialog_add_widget(zdedit,"button","-10%","hb1","-10%");
   zdialog_add_widget(zdedit,"hbox","hb2","dialog",0,"space=5");
   zdialog_add_widget(zdedit,"label","lab2","hb2","edge blend");
   zdialog_add_widget(zdedit,"hscale","blend","hb2","1|50|1|1","expand");

   zdialog_run(zdedit,select_paste_dialog_event);

   mouseCBfunc = select_paste_mousefunc;                                   //  connect mouse function
   Mcapture = 1;
   return;
}


//  Dialog event and completion callback function
//  Get dialog values and convert image. When done, commit edited image 
//  (with pasted pixels) and set up a new select area for the pasted 
//  pixels, allowing further editing of the area.

int select_paste_dialog_event(zdialog *zd, cchar *event)
{
   void  select_paste_mousefunc();

   int      cc, ii;
   int      px1, py1, px2, py2;
   uint16   *pix1;
   
   if (zd->zstat)                                                          //  dialog completed
   {
      mouseCBfunc = 0;                                                     //  disconnect mouse
      Mcapture = 0;

      if (zd->zstat != 1) {                                                //  cancel paste
         edit_cancel();                                                    //  cancel edit, restore image
         return 0;
      }

      edit_done();                                                         //  commit the edit (pasted image)
      
      if (sa_pixseq) zfree(sa_pixseq);                                     //  allocate sa_pixseq[]
      cc = 2 * Fww * Fhh;
      sa_pixseq = (uint16 *) zmalloc(cc,"pixseq");
      memset(sa_pixseq,0,cc);
      
      for (py1 = 0; py1 < sacpR_hh; py1++)                                 //  map non-transparent pixels
      for (px1 = 0; px1 < sacpR_ww; px1++)                                 //    into sa_pixseq[]
      {
         px2 = px1 / sacp_resize + 0.5;
         py2 = py1 / sacp_resize + 0.5;
         ii = sacp_ww * py2 + px2;                                         //  corresponding sacp_dist[]
         if (sacp_dist[ii] == 0) continue;                                 //  outside select area    v.8.8
         pix1 = bmpixel(sacpR_rgb16,px1,py1);
         px2 = px1 + sacp_orgx;
         py2 = py1 + sacp_orgy;
         if (px2 < 0 || px2 >= Fww) continue;                              //  parts may be beyond edges
         if (py2 < 0 || py2 >= Fhh) continue;
         ii = py2 * Fww + px2;
         sa_pixseq[ii] = 1;                                                //  result is equivalent to a
      }                                                                    //    1-click select-color area

      sa_stat = 1;                                                         //  active edit
      select_finish();                                                     //  finish the area

      RGB_free(sacpR_rgb16);                                               //  free memory
      return 0;
   }

   if (strEqu(event,"F1")) showz_userguide("copy_paste_area");             //  F1 help          v.9.0

   if (strEqu(event,"susp-resm")) {                                        //  toggle suspend/resume     v.8.8
      if (! sacp_susp) {
         mouseCBfunc = 0;                                                  //  disconnect mouse function
         Mcapture = 0;
         set_cursor(0);                                                    //  restore normal cursor
         zdialog_stuff(zdedit,"susp-resm",Bresume);
         sacp_susp = 1;
      }
      else {
         mouseCBfunc = select_paste_mousefunc;                             //  connect mouse function
         Mcapture = 1;
         zdialog_stuff(zdedit,"susp-resm",Bsuspend);
         sacp_susp = 0;
      }
   }

   if (strchr(event,'%')) {                                                //  resize by 1% or 10%       v.8.8
      if (strEqu(event,"+.1%")) sacp_resize = sacp_resize * 1.001;
      if (strEqu(event,"+1%")) sacp_resize = sacp_resize * 1.01;
      if (strEqu(event,"+10%")) sacp_resize = sacp_resize * 1.10;
      if (strEqu(event,"-.1%")) sacp_resize = sacp_resize * 0.999;
      if (strEqu(event,"-1%")) sacp_resize = sacp_resize * 0.99;
      if (strEqu(event,"-10%")) sacp_resize = sacp_resize * 0.90;

      RGB_free(sacpR_rgb16);
      sacpR_ww = sacp_resize * sacp_ww;                                    //  make new rescaled paste image
      sacpR_hh = sacp_resize * sacp_hh;
      sacpR_rgb16 = RGB_make(sacpR_ww,sacpR_hh,16);
      sacpR_rgb16 = RGB_rescale(sacp_rgb16,sacpR_ww,sacpR_hh);
      select_paste_pixmap();                                               //  copy onto image
   }

   if (strEqu(event,"blend")) {
      zdialog_fetch(zd,"blend",sacp_blend);                                //  blend, 0-50 pixels       v.8.8
      select_paste_pixmap();                                               //  copy onto image
   }

   return 0;
}


//  mouse function - follow mouse drags and move pasted area accordingly

void select_paste_mousefunc()
{
   int            mx1, my1, mx2, my2;
   static int     mdx0, mdy0, mdx1, mdy1;
   
   if (Mxposn < sacp_orgx || Mxposn > sacp_orgx + sacpR_ww ||              //  mouse outside select area
      Myposn < sacp_orgy || Myposn > sacp_orgy + sacpR_hh)
      set_cursor(0);                                                       //  set normal cursor
   else 
      set_cursor(dragcursor);                                              //  set drag cursor

   if (Mxdrag + Mydrag == 0) return;                                       //  no drag underway

   if (Mxdown != mdx0 || Mydown != mdy0) {                                 //  new drag initiated
      mdx0 = mdx1 = Mxdown;
      mdy0 = mdy1 = Mydown;
   }

   mx1 = mdx1;                                                             //  drag start
   my1 = mdy1;
   mx2 = Mxdrag;                                                           //  drag position
   my2 = Mydrag;
   mdx1 = mx2;                                                             //  next drag start
   mdy1 = my2;
   
   sacp_orgx += (mx2 - mx1);                                               //  move position of select area
   sacp_orgy += (my2 - my1);                                               //    by mouse drag amount
   select_paste_pixmap();                                                  //  re-copy area to new position

   return;      
}


//  copy select area into edit image, starting at sacp_orgx/y

void select_paste_pixmap()
{
   int      ii, px1, py1, px3, py3;
   uint16   *pix1, *pix3;
   double   f1, f2, dist;
   
   edit_reset();                                                           //  restore initial image   v.10.3

   mutex_lock(&pixmaps_lock);

   for (py1 = 4; py1 < sacpR_hh-5; py1++)                                  //  copy pixels less margins     v.8.8
   for (px1 = 4; px1 < sacpR_ww-5; px1++)
   {
      px3 = px1 / sacp_resize + 0.5;
      py3 = py1 / sacp_resize + 0.5;
      ii = sacp_ww * py3 + px3;                                            //  corresp. sacp_dist[]         v.8.8
      dist = sacp_dist[ii];
      if (dist == 0) continue;                                             //  pixel is outside select area
      pix1 = bmpixel(sacpR_rgb16,px1,py1);
      px3 = px1 + sacp_orgx;
      py3 = py1 + sacp_orgy;
      if (px3 < 0 || px3 >= E3ww) continue;                                //  parts may be beyond edges
      if (py3 < 0 || py3 >= E3hh) continue;
      pix3 = bmpixel(E3rgb16,px3,py3);                                     //  target image pixel
      f1 = (dist-1) * sacp_resize / sacp_blend;
      if (f1 > 1.0) f1 = 1.0;                                              //  blend area pixel and target pixel
      f2 = 1.0 - f1;                                                       //    v.8.8
      pix3[0] = f2 * pix3[0] + f1 * pix1[0];
      pix3[1] = f2 * pix3[1] + f1 * pix1[1];
      pix3[2] = f2 * pix3[2] + f1 * pix1[2];
   }
   
   Fmodified = 1;
   mutex_unlock(&pixmaps_lock);
   mwpaint2();
   return;
}


/**************************************************************************
   select area load from file and save to file functions           v.9.9
***************************************************************************/

//  Load a select area from a disk file and paste into current image.

char     sa_filename[40] = "area name";

void m_select_read_file(GtkWidget *, cchar *)
{
   char        safile[200], *pfile;
   char        *pp, buff[24], fotoxx[8];
   int         fid, ww, hh, cc, cc2;
   RGB         *temprgb;
   uint16      *tempdist;

   *safile = 0;                                                            //  get area name from user
   strncatv(safile,199,select_area_dirk,"/",sa_filename,null);
   pfile = zgetfile(ZTX("load select area from a file"),safile,"open");
   if (! pfile) return;
   
   temprgb = TIFFread(pfile);                                              //  TIFF file --> RGB
   if (! temprgb) return;

   strncpy0(safile,pfile,195);                                             //  replace .tiff with .dist
   pp = strrchr(safile,'/');
   if (pp) pp = strrchr(pp,'.');
   if (pp) strcpy(pp,".dist");
   else strcat(safile,".dist");
   
   fid = open(safile,O_RDONLY);                                            //  open .dist file
   if (! fid) {
      zmessageACK(GTK_WINDOW(mWin),ZTX("cannot read .dist file"));
      RGB_free(temprgb);
      return;
   }
   
   *fotoxx = 0;                                                            //  .dist file --> area ww/hh
   cc = read(fid,buff,20);
   sscanf(buff," %d %d %8s ",&ww, &hh, fotoxx);
   if (! strEqu(fotoxx,"fotoxx")) {
      zmessageACK(GTK_WINDOW(mWin),ZTX("cannot read .dist file"));
      RGB_free(temprgb);
      close(fid);
      return;
   }

   cc = 2 * ww * hh;                                                       //  allocate memory for edge distance data
   tempdist = (uint16 *) zmalloc(cc+100,"select-distfile");

   cc2 = read(fid,tempdist,cc);                                            //  .dist file --> edge distance data
   if (cc2 != cc) {
      zmessageACK(GTK_WINDOW(mWin),ZTX("cannot read .dist file"));
      RGB_free(temprgb);
      zfree(tempdist);
      close(fid);
      return;
   }

   RGB_free(sacp_rgb16);                                                   //  replace select area in memory
   sacp_rgb16 = temprgb;
   sacp_ww = ww;
   sacp_hh = hh;

   if (sacp_dist) zfree(sacp_dist);
   sacp_dist = tempdist;

   m_select_paste(0,0);                                                    //  paste it into current image
   return;
}


//  save a select area as a disk file

void m_select_save_file(GtkWidget *, cchar *)
{
   char        safile[200], *pfile, *pp, buff[24];
   int         fid, err, cc, cc2;

   if (! sa_active) select_finish();                                       //  finish select area if not already
   if (sa_active) {
      select_edgecalc();                                                   //  do edge calc if not already
      m_select_copy(0,0);                                                  //  copy select area to memory
   }

   if (! sacp_rgb16) return;                                               //  no active area, none in memory
   
   *safile = 0;                                                            //  get area name from user
   strncatv(safile,199,select_area_dirk,"/",sa_filename,null);
   pfile = zgetfile(ZTX("save select area as a file"),safile,"save");
   if (! pfile) return;

   strncpy0(safile,pfile,195);                                             //  insure extension is .tiff
   pp = strrchr(safile,'/');
   if (pp) pp = strrchr(pp,'.');
   if (pp) strcpy(pp,".tiff");
   else strcat(safile,".tiff");
   
   err = TIFFwrite(sacp_rgb16,safile);                                     //  select area RGB --> .tiff file
   if (err) return;
   
   pp = strrchr(safile,'.');
   strcpy(pp,".dist");
   fid = open(safile,O_WRONLY|O_CREAT|O_TRUNC,0640);                       //  edge distance data --> .dist file
   if (! fid) zappcrash("cannot open .dist file");

   snprintf(buff,24," %05d %05d fotoxx ",sacp_ww,sacp_hh);
   cc = write(fid,buff,20);
   if (cc != 20) zappcrash("cannot write .dist file");
   
   cc = 2 * sacp_ww * sacp_hh;
   cc2 = write(fid,sacp_dist,cc);
   if (cc2 != cc) zappcrash("cannot write .dist file");

   close(fid);
   return;
}


/**************************************************************************
      begin image edit functions
***************************************************************************/

//  adjust white balance

double   whitebal_red, whitebal_green, whitebal_blue;

void m_whitebal(GtkWidget *, cchar *)                                      //  v.8.6
{
   int    whitebal_dialog_event(zdialog* zd, cchar *event);
   void   whitebal_mousefunc();
   void * whitebal_thread(void *);

   cchar  *wbtitle = ZTX("Adjust White Balance");
   cchar  *wbhelp = ZTX("Click white or gray image location");

   if (! edit_setup("whitebal",1,2)) return;                               //  setup edit: preview

   zdedit = zdialog_new(wbtitle,mWin,Bdone,Bcancel,null);
   zdialog_add_widget(zdedit,"hbox","hb1","dialog",0,"space=10");
   zdialog_add_widget(zdedit,"label","labwbh","hb1",wbhelp,"space=5");
   zdialog_add_widget(zdedit,"hbox","hb2","dialog",0,"space=5");
   zdialog_add_widget(zdedit,"label","labpix","hb2","pixel:");
   zdialog_add_widget(zdedit,"label","pixel","hb2","0000 0000");
   zdialog_add_widget(zdedit,"label","labrgb","hb2","   RGB:");
   zdialog_add_widget(zdedit,"label","rgb","hb2","000 000 000");

   zdialog_run(zdedit,whitebal_dialog_event);                              //  run dialog - parallel

   whitebal_red = whitebal_green = whitebal_blue = 1.0;
   start_thread(whitebal_thread,0);                                        //  start working thread

   mouseCBfunc = whitebal_mousefunc;                                       //  connect mouse function
   Mcapture = 1;                                                           //  capture mouse clicks
   return;
}


//  dialog event and completion callback function

int whitebal_dialog_event(zdialog *zd, cchar *event)                       //  dialog event function     v.10.2
{
   if (zd->zstat)
   {
      mouseCBfunc = 0;                                                     //  disconnect mouse
      Mcapture = 0;
      if (zd->zstat == 1) edit_done();                                     //  done
      else edit_cancel();                                                  //  cancel or destroy
      return 0;
   }
   
   if (strEqu(event,"F1")) showz_userguide("white_balance");               //  F1 help          v.10.3
   if (strEqu(event,"undo")) edit_undo();                                  //  v.10.2
   if (strEqu(event,"redo")) edit_redo();                                  //  v.10.3
   if (strEqu(event,"blendwidth")) signal_thread();                        //  v.10.3
   return 1;
}


void whitebal_mousefunc()                                                  //  mouse function
{
   int         px, py, dx, dy;
   double      red, green, blue, rgbmean;
   char        work[40];
   uint16      *ppix16;
   
   if (! LMclick) return;
   
   LMclick = 0;
   px = Mxclick;                                                           //  mouse click position
   py = Myclick;
   
   if (px < 2) px = 2;                                                     //  pull back from edge
   if (px > E3ww-3) px = E3ww-3;
   if (py < 2) py = 2;
   if (py > E3hh-3) py = E3hh-3;
   
   red = green = blue = 0;

   for (dy = -2; dy <= 2; dy++)                                            //  5x5 block around mouse position
   for (dx = -2; dx <= 2; dx++)
   {
      ppix16 = bmpixel(E1rgb16,px+dx,py+dy);                               //  input image
      red += ppix16[0];
      green += ppix16[1];
      blue += ppix16[2];
   }
   
   red = red / 25.0;                                                       //  mean RGB levels 
   green = green / 25.0;
   blue = blue / 25.0;
   rgbmean = (red + green + blue) / 3.0;

   whitebal_red = rgbmean / red;
   whitebal_green = rgbmean / green;
   whitebal_blue = rgbmean / blue;

   signal_thread();                                                        //  trigger image update

   snprintf(work,40,"%d %d",px,py);
   zdialog_stuff(zdedit,"pixel",work);

   snprintf(work,40,"%7.3f %7.3f %7.3f",red/256,green/256,blue/256);
   zdialog_stuff(zdedit,"rgb",work);
   
   return;
}


//  Update image based on neutral pixel that was clicked

void * whitebal_thread(void *)
{
   void * whitebal_wthread(void *arg);

   while (true)
   {
      thread_idle_loop();                                                  //  wait for work or exit request
      
      for (int ii = 0; ii < Nwt; ii++)                                     //  start worker threads
         start_wt(whitebal_wthread,&wtnx[ii]);
      wait_wts();                                                          //  wait for completion

      Fmodified = 1;
      mwpaint2();                                                          //  update window
   }

   return 0;                                                               //  not executed, stop g++ warning
}


void * whitebal_wthread(void *arg)                                         //  worker thread function
{
   int         index = *((int *) arg);
   int         px, py, ii, dist = 0;
   uint16      *pix1, *pix3;
   double      red1, green1, blue1;
   double      red3, green3, blue3;
   double      brmax, dold, dnew;

   for (py = index; py < E1hh; py += Nwt)
   for (px = 0; px < E1ww; px++)
   {
      if (sa_active) {                                                     //  select area active
         ii = py * Fww + px;
         dist = sa_pixisin[ii];                                            //  distance from edge
         if (! dist) continue;                                             //  pixel outside area
      }

      pix1 = bmpixel(E1rgb16,px,py);                                       //  input pixel
      pix3 = bmpixel(E3rgb16,px,py);                                       //  output pixel
      
      red1 = pix1[0];
      green1 = pix1[1];
      blue1 = pix1[2];
      
      red3 = whitebal_red * red1;                                          //  change color ratios
      green3 = whitebal_green * green1;
      blue3 = whitebal_blue * blue1;

      if (sa_active && dist < sa_blend) {                                  //  select area is active,
         dnew = 1.0 * dist / sa_blend;                                     //    blend changes over sa_blend
         dold = 1.0 - dnew;
         red3 = dnew * red3 + dold * red1;
         green3 = dnew * green3 + dold * green1;
         blue3 = dnew * blue3 + dold * blue1;
      }
      
      brmax = red3;                                                        //  brmax = brightest color
      if (green3 > brmax) brmax = green3;
      if (blue3 > brmax) brmax = blue3;
      
      if (brmax > 65535) {                                                 //  if overflow, reduce
         brmax = 65535 / brmax;
         red3 = red3 * brmax;
         green3 = green3 * brmax;
         blue3 = blue3 * brmax;
      }

      pix3[0] = int(red3);
      pix3[1] = int(green3);
      pix3[2] = int(blue3);
   }
   
   exit_wt();
   return 0;                                                               //  not executed, stop gcc warning
}


/**************************************************************************/

//  flatten brightness distribution

double   flatten_value = 0;                                                //  flatten value, 0 - 100%
double   flatten_brdist[65536];


void m_flatten(GtkWidget *, cchar *)
{
   int    flatten_dialog_event(zdialog* zd, cchar *event);
   void * flatten_thread(void *);

   cchar  *title = ZTX("Flatten Brightness Distribution");

   if (! edit_setup("flatten",1,2)) return;                                //  setup edit: preview, select area OK

   zdedit = zdialog_new(title,mWin,Bdone,Bcancel,null);
   zdialog_add_widget(zdedit,"hbox","hb1","dialog",0,"space=15");
   zdialog_add_widget(zdedit,"label","labfd","hb1",ZTX("Flatten"),"space=5");
   zdialog_add_widget(zdedit,"hscale","flatten","hb1","0|100|1|0","expand");

   zdialog_resize(zdedit,300,0);
   zdialog_run(zdedit,flatten_dialog_event);                               //  run dialog - parallel
   
   flatten_value = 0;
   start_thread(flatten_thread,0);                                         //  start working thread
   return;
}


//  dialog event and completion callback function

int flatten_dialog_event(zdialog *zd, cchar *event)                        //  flatten dialog event function
{
   if (zd->zstat)
   {
      if (zd->zstat == 1) edit_done();                                     //  done
      else edit_cancel();                                                  //  cancel or destroy
      return 0;
   }
   
   if (strEqu(event,"F1")) showz_userguide("flatten");                     //  F1 help          v.9.0
   if (strEqu(event,"undo")) edit_undo();                                  //  v.10.2
   if (strEqu(event,"redo")) edit_redo();                                  //  v.10.3
   if (strEqu(event,"blendwidth")) signal_thread();                        //  v.10.3

   if (strEqu(event,"flatten")) {
      zdialog_fetch(zd,"flatten",flatten_value);                           //  get slider value
      signal_thread();                                                     //  trigger update thread
   }
   
   return 1;
}


//  thread function - use multiple working threads

void * flatten_thread(void *)
{
   void  * flatten_wthread(void *arg);

   int         px, py, ii;
   double      bright1;
   uint16      *pix1;

   while (true)
   {
      thread_idle_loop();                                                  //  wait for work or exit request

      for (ii = 0; ii < 65536; ii++)                                       //  clear brightness distribution data
         flatten_brdist[ii] = 0;

      if (sa_active)                                                       //  process selected area
      {
         for (ii = 0; ii < Fww * Fhh; ii++)                                //  v.9.6
         {
            if (! sa_pixisin[ii]) continue;
            py = ii / Fww;
            px = ii - py * Fww;
            pix1 = bmpixel(E1rgb16,px,py);
            bright1 = brightness(pix1);
            flatten_brdist[int(bright1)]++;
         }
         
         for (ii = 1; ii < 65536; ii++)                                    //  cumulative brightness distribution
            flatten_brdist[ii] += flatten_brdist[ii-1];                    //   0 ... sa_Npixel

         for (ii = 0; ii < 65536; ii++)
            flatten_brdist[ii] = flatten_brdist[ii]                        //  multiplier per brightness level
                               / sa_Npixel * 65536.0 / (ii + 1);
      }

      else                                                                 //  process whole image
      {
         for (py = 0; py < E1hh; py++)                                     //  compute brightness distribution
         for (px = 0; px < E1ww; px++)
         {
            pix1 = bmpixel(E1rgb16,px,py);
            bright1 = brightness(pix1);
            flatten_brdist[int(bright1)]++;
         }
         
         for (ii = 1; ii < 65536; ii++)                                    //  cumulative brightness distribution
            flatten_brdist[ii] += flatten_brdist[ii-1];                    //   0 ... (ww1 * hh1)

         for (ii = 0; ii < 65536; ii++)
            flatten_brdist[ii] = flatten_brdist[ii]                        //  multiplier per brightness level 
                               / (E1ww * E1hh) * 65536.0 / (ii + 1);
      }
      
      for (ii = 0; ii < Nwt; ii++)                                         //  start worker threads
         start_wt(flatten_wthread,&wtnx[ii]);
      wait_wts();                                                          //  wait for completion

      Fmodified = 1;
      mwpaint2();                                                          //  update window
   }

   return 0;                                                               //  not executed, stop g++ warning
}


void * flatten_wthread(void *arg)                                          //  worker thread function
{
   int         index = *((int *) (arg));
   int         px, py, ii, dist = 0;
   uint16      *pix1, *pix3;
   double      fold, fnew, dold, dnew, cmax;
   double      red1, green1, blue1, red3, green3, blue3;
   double      bright1, bright2;

   for (py = index; py < E1hh; py += Nwt)                                  //  flatten brightness distribution
   for (px = 0; px < E1ww; px++)
   {
      if (sa_active) {                                                     //  select area active
         ii = py * Fww + px;
         dist = sa_pixisin[ii];                                            //  distance from edge
         if (! dist) continue;                                             //  outside pixel
      }

      pix1 = bmpixel(E1rgb16,px,py);                                       //  input pixel
      pix3 = bmpixel(E3rgb16,px,py);                                       //  output pixel
         
      fnew = 0.01 * flatten_value;                                         //  0.0 - 1.0  how much to flatten
      fold = 1.0 - fnew;                                                   //  1.0 - 0.0  how much to retain

      red1 = pix1[0];
      green1 = pix1[1];
      blue1 = pix1[2];

      bright1 = brightness(pix1);                                          //  input brightness
      bright2 = flatten_brdist[int(bright1)];                              //  output brightness adjustment

      red3 = bright2 * red1;                                               //  flattened brightness
      green3 = bright2 * green1;
      blue3 = bright2 * blue1;

      red3 = fnew * red3 + fold * red1;                                    //  blend new and old brightness
      green3 = fnew * green3 + fold * green1;
      blue3 = fnew * blue3 + fold * blue1;

      if (sa_active && dist < sa_blend) {                                  //  select area is active,
         dnew = 1.0 * dist / sa_blend;                                     //    blend changes over sa_blend
         dold = 1.0 - dnew;
         red3 = dnew * red3 + dold * red1;
         green3 = dnew * green3 + dold * green1;
         blue3 = dnew * blue3 + dold * blue1;
      }

      cmax = red3;                                                         //  stop overflow, keep color balance
      if (green3 > cmax) cmax = green3;
      if (blue3 > cmax) cmax = blue3;
      if (cmax > 65535) {
         cmax = 65535 / cmax;
         red3 = red3 * cmax;
         green3 = green3 * cmax;
         blue3 = blue3 * cmax;
      }

      pix3[0] = int(red3 + 0.5);
      pix3[1] = int(green3 + 0.5);
      pix3[2] = int(blue3 + 0.5);
   }

   exit_wt();
   return 0;                                                               //  not executed, avoid gcc warning
}


/**************************************************************************/

//  brightness / color / contrast adjustment

void  tune_curve_update(int spc);                                          //  curve update callback function
int   tune_spc;                                                            //  current spline curve 1-6


void m_tune(GtkWidget *, cchar *)                                          //  menu function
{
   using namespace splinecurve;

   int   tune_dialog_event(zdialog *zd, cchar *event);
   void  *tune_thread(void *);

   cchar  *title = ZTX("Adjust Brightness and Color");

   if (! edit_setup("britecolor",1,2)) return;                             //  setup edit: preview
   
   for (int spc = 0; spc < 7; spc++)                                       //  setup 7 spline curves
   {                                                                       //  no. 0 is active curve
      vert[spc] = 0;                                                       //  all curves are horizontal
      nap[spc] = 3;                                                        //  1-6 are copied to 0 when active
      apx[spc][0] = 0.01;
      apy[spc][0] = 0.5;
      apx[spc][1] = 0.5;
      apy[spc][1] = 0.5;
      apx[spc][2] = 0.98;
      apy[spc][2] = 0.5;
      curve_generate(spc);
   }

   Nspc = 1;                                                               //  only one at a time is active
   tune_spc = 1;                                                           //  default curve = brightness

/***
       --------------------------------------------
      |                                            |
      |                                            |
      |           curve drawing area               |
      |                                            |
      |                                            |
       --------------------------------------------
       darker areas                   lighter areas

      [+++] [---] [+ -] [- +] [+-+] [-+-]

      (o) brightness
      (o) color intensity    [ reset 1 ]  [reset all]
      (o) color saturation   [ undo ] [ redo ] 
      (o) red                [ histogram ]
      (o) green
      (o) blue

***/

   zdedit = zdialog_new(title,mWin,Bdone,Bcancel,null);
   zdialog_add_widget(zdedit,"frame","fr1","dialog",0,"expand");
   zdialog_add_widget(zdedit,"hbox","hba","dialog");
   zdialog_add_widget(zdedit,"label","labda","hba",Bdarker,"space=5");
   zdialog_add_widget(zdedit,"label","space","hba",0,"expand");
   zdialog_add_widget(zdedit,"label","labba","hba",Blighter,"space=5");
   zdialog_add_widget(zdedit,"hbox","hbb","dialog",0,"space=5");
   zdialog_add_widget(zdedit,"button","b +++","hbb","+++");
   zdialog_add_widget(zdedit,"button","b ---","hbb","‒ ‒ ‒");
   zdialog_add_widget(zdedit,"button","b +-", "hbb"," + ‒ ");
   zdialog_add_widget(zdedit,"button","b -+", "hbb"," ‒ + ");
   zdialog_add_widget(zdedit,"button","b +-+","hbb","+ ‒ +");
   zdialog_add_widget(zdedit,"button","b -+-","hbb","‒ + ‒");
   zdialog_add_widget(zdedit,"hbox","hb2","dialog");
   zdialog_add_widget(zdedit,"vbox","vb21","hb2");
   zdialog_add_widget(zdedit,"vbox","vb22","hb2");
   zdialog_add_widget(zdedit,"radio","radbri","vb21",Bbrightness);
   zdialog_add_widget(zdedit,"radio","radcol","vb21",ZTX("color intensity"));
   zdialog_add_widget(zdedit,"radio","radsat","vb21",ZTX("color saturation"));
   zdialog_add_widget(zdedit,"radio","radR","vb21",Bred);
   zdialog_add_widget(zdedit,"radio","radG","vb21",Bgreen);
   zdialog_add_widget(zdedit,"radio","radB","vb21",Bblue);
   zdialog_add_widget(zdedit,"hbox","hbrs","vb22",0,"space=5");
   zdialog_add_widget(zdedit,"label","space","hbrs",0,"space=10");
   zdialog_add_widget(zdedit,"button","reset1","hbrs",ZTX(" reset 1 "));
   zdialog_add_widget(zdedit,"button","resetA","hbrs",ZTX("reset all"));
   zdialog_add_widget(zdedit,"hbox","hbhist","vb22",0,"space=6");
   zdialog_add_widget(zdedit,"label","space","hbhist",0,"space=10");
   zdialog_add_widget(zdedit,"button","histo","hbhist",ZTX("histogram"));
   
   GtkWidget *frame = zdialog_widget(zdedit,"fr1");                        //  setup for curve editing
   curve_init(frame,tune_curve_update);

   zdialog_stuff(zdedit,"radbri",1);                                       //  stuff default selection
   
   zdialog_resize(zdedit,0,380);
   zdialog_run(zdedit,tune_dialog_event);                                  //  run dialog - parallel
   start_thread(tune_thread,0);                                            //  start working thread
   return;
}


//  dialog event and completion callback function

int tune_dialog_event(zdialog *zd, cchar *event)
{
   using namespace splinecurve;

   int         Fupdate = 0;
   int         ii, jj;
   double      px, py;
   
   if (zd->zstat)
   {
      if (zd->zstat == 1) edit_done();                                     //  done
      else edit_cancel();                                                  //  cancel or destroy
      return 0;
   }

   if (strEqu(event,"F1")) showz_userguide("tune");                        //  F1 context help
   if (strEqu(event,"undo")) edit_undo();                                  //  v.10.2
   if (strEqu(event,"redo")) edit_redo();                                  //  v.10.3
   if (strEqu(event,"blendwidth")) signal_thread();                        //  v.10.3
   if (strEqu(event,"histo")) m_histogram(0,0);                            //  popup brightness histogram

   if (strnEqu(event,"rad",3)) {                                           //  new choice of curve
      ii = strcmpv(event,"radbri","radcol","radsat","radR","radG","radB",null);
      tune_spc = ii;

      nap[0] = nap[ii];                                                    //  copy active curve to curve 0
      for (jj = 0; jj < nap[0]; jj++) {
         apx[0][jj] = apx[ii][jj];
         apy[0][jj] = apy[ii][jj];
      }
      Fupdate++;
   }
   
   ii = tune_spc;                                                          //  current active curve

   if (strnEqu(event,"b ",2)) {                                            //  button to move entire curve
      for (jj = 0; jj < nap[ii]; jj++) {
         px = apx[0][jj];
         py = apy[0][jj];
         if (strEqu(event,"b +++")) py += 0.1;
         if (strEqu(event,"b ---")) py -= 0.1;
         if (strEqu(event,"b +-"))  py += 0.1 - 0.2 * px;
         if (strEqu(event,"b -+"))  py -= 0.1 - 0.2 * px;
         if (strEqu(event,"b +-+")) py -= 0.05 - 0.2 * fabs(px-0.5);
         if (strEqu(event,"b -+-")) py += 0.05 - 0.2 * fabs(px-0.5);
         if (py > 1) py = 1;
         if (py < 0) py = 0;
         apy[0][jj] = py;
      }
      Fupdate++;
   }
   
   if (strEqu(event,"reset1")) {                                           //  reset current curve
      nap[0] = 3;
      apx[0][0] = 0.01;                                                    //  3 anchor points, flatline
      apy[0][0] = 0.5;
      apx[0][1] = 0.5;
      apy[0][1] = 0.5;
      apx[0][2] = 0.99;
      apy[0][2] = 0.5;
      Fupdate++;
   }
   
   if (strEqu(event,"resetA")) 
   {
      for (jj = 0; jj < 7; jj++) {                                         //  reset all curves
         nap[jj] = 3;
         apx[jj][0] = 0.01;
         apy[jj][0] = 0.5;
         apx[jj][1] = 0.5;
         apy[jj][1] = 0.5;
         apx[jj][2] = 0.99;
         apy[jj][2] = 0.5;
         curve_generate(jj);                                               //  regenerate all
      }
      Fupdate++;
   }
   
   if (Fupdate)                                                            //  curve has changed
   {
      curve_generate(0);                                                   //  regenerate curve 0

      nap[ii] = nap[0];                                                    //  copy curve 0 to active curve
      for (jj = 0; jj < nap[0]; jj++) {
         apx[ii][jj] = apx[0][jj];
         apy[ii][jj] = apy[0][jj];
      }
      for (jj = 0; jj < 1000; jj++)
         yval[ii][jj] = yval[0][jj];

      curve_draw();                                                        //  draw curve
      signal_thread();                                                     //  trigger image update
   }
   
   return 1;
}


//  this function is called when curve 0 is edited using mouse

void  tune_curve_update(int)
{
   using namespace splinecurve;
   
   int   ii = tune_spc, jj;

   nap[ii] = nap[0];                                                       //  copy curve 0 to current curve
   for (jj = 0; jj < nap[0]; jj++) {
      apx[ii][jj] = apx[0][jj];
      apy[ii][jj] = apy[0][jj];
   }
   for (jj = 0; jj < 1000; jj++)
      yval[ii][jj] = yval[0][jj];
   
   signal_thread();                                                        //  trigger image update
   return;
}


//  Update image based on latest settings of all dialog controls.

void * tune_thread(void *)
{
   void * tune_wthread(void *arg);

   while (true)
   {
      thread_idle_loop();                                                  //  wait for work or exit request
      
      for (int ii = 0; ii < Nwt; ii++)                                     //  start worker threads
         start_wt(tune_wthread,&wtnx[ii]);
      wait_wts();                                                          //  wait for completion

      Fmodified = 1;
      mwpaint2();                                                          //  update window
   }

   return 0;                                                               //  not executed, stop g++ warning
}


void * tune_wthread(void *arg)                                             //  worker thread function
{
   void  tune1pix(int px, int py);

   int      px, py;
   int      index = *((int *) arg);

   for (py = index; py < E1hh; py += Nwt)
   for (px = 0; px < E1ww; px++)
      tune1pix(px,py);

   exit_wt();
   return 0;                                                               //  not executed, avoid gcc warning
}


void tune1pix(int px, int py)                                              //  process one pixel
{
   using namespace splinecurve;

   int         ii, dist = 0;
   uint16      *pix1, *pix3;
   double      red1, green1, blue1, red3, green3, blue3;
   double      xval, brmax, brout;

   if (sa_active) {                                                        //  select area active
      ii = py * Fww + px;
      dist = sa_pixisin[ii];                                               //  distance from edge
      if (! dist) return;                                                  //  outside pixel
   }

   pix1 = bmpixel(E1rgb16,px,py);                                          //  input pixel
   pix3 = bmpixel(E3rgb16,px,py);                                          //  output pixel
   
   red1 = red3 = pix1[0];
   green1 = green3 = pix1[1];
   blue1 = blue3 = pix1[2];

   brmax = red1;                                                           //  brmax = brightest color
   if (green1 > brmax) brmax = green1;
   if (blue1 > brmax) brmax = blue1;
   
   xval = brmax/65536.0;                                                   //  index into curve data, 0 to 0.999

/* ------------------------------------------------------------------------

      brightness curve values:
           0 = dark
         0.5 = normal, unchanged
         1.0 = 200% brightness, clipped
*/

   brout = curve_yval(1,xval);                                             //  brightness factor, 0 - 1
   
   if (brout < 0.49 || brout > 0.51)
   {
      brout = brout / 0.5;                                                 //  0 - 2.0
      if (brout * brmax > 65535.0) brout = 65535.0 / brmax;                //  reduce if necessary
      
      red3 = red3 * brout;                                                 //  apply to all colors
      green3 = green3 * brout;
      blue3 = blue3 * brout;
   }
      
/* ------------------------------------------------------------------------

      color intensity curve values:
           0 = no color (grey scale)  
         0.5 = normal, unchanged
         1.0 = highest color

      0.5 >> 0:  move all RGB values to their mean: (R+G+B)/3
      0.5 >> 1.0:  increase all RGB values by same factor
      
      In the 2nd case, the movement is greater for darker pixels
*/

   double   red50, green50, blue50, red100, green100, blue100;
   double   rgb0, max50, min50, color, bright, ramper;
   
   brout = curve_yval(2,xval);                                             //  brightness factor, 0 - 1
   
   if (brout < 0.49 || brout > 0.51)
   {
      red50 = red3;                                                        //  50%  color values (normal)
      green50 = green3;
      blue50 = blue3;
      
      rgb0 = (red50 + green50 + blue50) / 3;                               //  0%  color values (grey scale)
         
      max50 = min50 = red50;
      if (green50 > max50) max50 = green50;                                //  get max/min normal color values
      else if (green50 < min50) min50 = green50;
      if (blue50 > max50) max50 = blue50;
      else if (blue50 < min50) min50 = blue50;
      
      color = (max50 - min50) * 1.0 / (max50 + 1);                         //  gray .. color       0 .. 1
      color = sqrt(color);                                                 //  accelerated curve   0 .. 1
      bright = max50 / 65535.0;                                            //  dark .. bright      0 .. 1
      bright = sqrt(bright);                                               //  accelerated curve   0 .. 1
      ramper = 1 - color + bright * color;                                 //  1 - color * (1 - bright)
      ramper = 1.0 / ramper;                                               //  large if color high and bright low

      red100 = int(red50 * ramper);                                        //  100%  color values (max)
      green100 = int(green50 * ramper);
      blue100 = int(blue50 * ramper);
      
      if (brout < 0.50) 
      {
         red3 = rgb0 + (brout) * 2 * (red50 - rgb0);                       //  compute new color value
         green3 = rgb0 + (brout) * 2 * (green50 - rgb0);
         blue3 = rgb0 + (brout) * 2 * (blue50 - rgb0);
      }

      if (brout > 0.50)
      {
         red3 = red50 + (brout - 0.5) * 2 * (red100 - red50);
         green3 = green50 + (brout - 0.5) * 2 * (green100 - green50);
         blue3 = blue50 + (brout - 0.5) * 2 * (blue100 - blue50);
      }
   }

/* ------------------------------------------------------------------------

      color saturation curve values:
           0 = no color saturation (gray scale)
         0.5 = normal (initial unmodified RGB)
         1.0 = max. color saturation

      0.5 >> 0:  move all RGB values to their mean: (R+G+B)/3
      0.5 >> 1.0:  increase RGB spread until one color is 0 or 65535

      In both cases, the average of RGB is not changed.
*/

   double   rinc, ginc, binc, scale;
   double   spread, spread1, spread2;
   int      againlimit = 10;
   
   brout = curve_yval(3,xval);                                             //  saturation factor, 0 - 1
   
   if (brout < 0.49 || brout > 0.51)
   {
      spread = brout - 0.5;                                                //  -0.5  to  0  to  +0.5
      spread1 = 2 * spread + 1;                                            //     0  to  1  to   2
      spread2 = spread1 - 1.0;                                             //    -1  to  0  to   1

      red50 = red3;                                                        //  50%  color values (normal)
      green50 = green3;
      blue50 = blue3;
      
      rgb0 = (red50 + green50 + blue50 + 1) / 3;

      rinc = red50 - rgb0;
      ginc = green50 - rgb0;
      binc = blue50 - rgb0;
      
      scale = 1.0;

   again:
      rinc = scale * rinc;
      ginc = scale * ginc;
      binc = scale * binc;

      red100 = red50 + rinc;
      green100 = green50 + ginc;
      blue100 = blue50 + binc;
      
      if (--againlimit > 0)                                                //  prevent loops   v.8.5.2
      {
         if (red100 > 65535) { scale = (65535.0 - red50) / rinc;  goto again; }
         if (red100 < 0) { scale = -1.0 * red50 / rinc; goto again; }
         if (green100 > 65535) { scale = (65535.0 - green50) / ginc; goto again; }
         if (green100 < 0) { scale = -1.0 * green50 / ginc; goto again; }
         if (blue100 > 65535) { scale = (65535.0 - blue50) / binc; goto again; }
         if (blue100 < 0) { scale = -1.0 * blue50 / binc; goto again; }
      }
      
      if (spread < 0) {                                                    //  make mid-scale == original RGB
         red3 = int(rgb0 + spread1 * (red50 - rgb0));
         green3 = int(rgb0 + spread1 * (green50 - rgb0));
         blue3 = int(rgb0 + spread1 * (blue50 - rgb0));
      }
      else {
         red3 = int(red50 + spread2 * (red100 - red50));
         green3 = int(green50 + spread2 * (green100 - green50));
         blue3 = int(blue50 + spread2 * (blue100 - blue50));
      }
   }

/* ------------------------------------------------------------------------

      color balance curve values:
           0 = 0.5 * original color
         0.5 = unmodified
         1.0 = 1.5 * original color, clipped
*/

   brout = curve_yval(4,xval);
   if (brout < 0.49 || brout > 0.51) red3 = red3 * (brout + 0.5);
   brout = curve_yval(5,xval);
   if (brout < 0.49 || brout > 0.51) green3 = green3 * (brout + 0.5);
   brout = curve_yval(6,xval);
   if (brout < 0.49 || brout > 0.51) blue3 = blue3 * (brout + 0.5);

/* ------------------------------------------------------------------------
   
   if working within a select area, blend changes over distance from edge

*/
      
   double      dold, dnew;

   if (sa_active && dist < sa_blend) {
      dnew = 1.0 * dist / sa_blend;
      dold = 1.0 - dnew;
      red3 = dnew * red3 + dold * red1;
      green3 = dnew * green3 + dold * green1;
      blue3 = dnew * blue3 + dold * blue1;
   }

/* ------------------------------------------------------------------------
   
   prevent clipping and set output RGB values

*/

   if (red3 > 65535) red3 = 65535;
   if (green3 > 65535) green3 = 65535;
   if (blue3 > 65535) blue3 = 65535;

   pix3[0] = int(red3);
   pix3[1] = int(green3);
   pix3[2] = int(blue3);
   
   return;
}


/**************************************************************************/

//  ramp brightness across image, vertical and horizontal gradients        //  new  v.9.3

void m_brightramp(GtkWidget *, cchar *)
{
   using namespace splinecurve;
   
   int    brramp_dialog_event(zdialog* zd, cchar *event);
   void   brramp_curvedit(int spc);
   void * brramp_thread(void *);
   void   brramp_mousefunc();

   if (! edit_setup("briteramp",1,2)) return;                              //  setup edit: preview, select area OK

   zdedit = zdialog_new(ZTX("Ramp brightness across image"),mWin,Bdone,Bcancel,null);
   zdialog_add_widget(zdedit,"hbox","hb1","dialog",0,"space=5|expand");
   zdialog_add_widget(zdedit,"vbox","vb1","hb1");
   zdialog_add_widget(zdedit,"vbox","vb2","hb1",0,"expand");
   zdialog_add_widget(zdedit,"label","lmax","vb1","+","space=3");
   zdialog_add_widget(zdedit,"label","lspace","vb1",0,"expand");
   zdialog_add_widget(zdedit,"label","lmin","vb1","‒","space=3");
   zdialog_add_widget(zdedit,"label","lspace","vb1");
   zdialog_add_widget(zdedit,"frame","frame","vb2",0,"expand");
   zdialog_add_widget(zdedit,"hbox","hb2","vb2");
   zdialog_add_widget(zdedit,"label","lmin","hb2","‒","space=3");
   zdialog_add_widget(zdedit,"label","lspace","hb2",0,"expand");
   zdialog_add_widget(zdedit,"label","lmax","hb2","+","space=3");

   Nspc = 2;

   nap[0] = 4;
   vert[0] = 0;
   apx[0][0] = 0.01;
   apx[0][1] = 0.4;
   apx[0][2] = 0.6;
   apx[0][3] = 0.99;
   apy[0][0] = apy[0][1] = apy[0][2] = apy[0][3] = 0.5;

   nap[1] = 4;
   vert[1] = 1;
   apx[1][0] = 0.01;
   apx[1][1] = 0.4;
   apx[1][2] = 0.6;
   apx[1][3] = 0.99;
   apy[1][0] = apy[1][1] = apy[1][2] = apy[1][3] = 0.5;

   curve_generate(0);
   curve_generate(1);

   GtkWidget *frame = zdialog_widget(zdedit,"frame");
   curve_init(frame,brramp_curvedit);

   zdialog_resize(zdedit,260,280);
   zdialog_run(zdedit,brramp_dialog_event);                                //  run dialog, parallel

   start_thread(brramp_thread,0);                                          //  start working thread
   signal_thread();
   return;
}


//  dialog event and completion callback function

int brramp_dialog_event(zdialog *zd, cchar *event)
{
   if (zd->zstat)                                                          //  dialog complete
   {
      if (zd->zstat == 1) edit_done();
      else edit_cancel();
      return 1;
   }
   
   if (strEqu(event,"F1")) showz_userguide("brightness_ramp");             //  F1 context help 
   if (strEqu(event,"undo")) edit_undo();                                  //  v.10.2
   if (strEqu(event,"redo")) edit_redo();                                  //  v.10.3
   if (strEqu(event,"blendwidth")) signal_thread();                        //  v.10.3
   return 1;
}


//  this function is called when a curve is edited

void brramp_curvedit(int spc)
{
   signal_thread();
   return;
}


//  brramp thread function

void * brramp_thread(void *arg)
{
   void * brramp_wthread(void *);
   
   while (true)
   {
      thread_idle_loop();                                                  //  wait for work or exit request

      for (int ii = 0; ii < Nwt; ii++)                                     //  start worker threads
         start_wt(brramp_wthread,&wtnx[ii]);
      wait_wts();                                                          //  wait for completion

      mwpaint2();                                                          //  update window
      Fmodified = 1;                                                       //  image3 modified
   }

   return 0;                                                               //  not executed, stop g++ warning
}


void * brramp_wthread(void *arg)                                           //  worker thread function
{
   using namespace splinecurve;

   int         index = *((int *) arg);
   int         ii, dist = 0, px3, py3;
   int         red1, green1, blue1, maxrgb;
   int         red3, green3, blue3;
   uint16      *pix1, *pix3;
   double      dispx, dispy;
   double      hramp, vramp, tramp;
   double      spanw, spanh, dold, dnew;

   if (sa_active) {                                                        //  if select area active, ramp
      spanw = sa_maxx - sa_minx;                                           //    brightness over enclosing rectangle
      spanh = sa_maxy - sa_miny;
   }
   else {
      spanw = E3ww;                                                        //  else over entire image
      spanh = E3hh;
   }   

   for (py3 = index; py3 < E3hh; py3 += Nwt)                               //  loop output pixels
   for (px3 = 0; px3 < E3ww; px3++)
   {
      if (sa_active) {                                                     //  select area active        v.10.1
         ii = py3 * E3ww + px3;
         dist = sa_pixisin[ii];                                            //  distance from edge
         if (! dist) continue;                                             //  pixel outside area
         dispx = (px3 - sa_minx) / spanw;
         dispy = (py3 - sa_miny) / spanh;
      }
      else {
         dispx = px3 / spanw;                                              //  left > right = 0 to 1
         dispy = py3 / spanh;                                              //  top > bottom = 0 to 1
      }
      
      hramp = curve_yval(0,dispx) - 0.5;                                   //  -0.5 to +0.5
      vramp = curve_yval(1,dispy) - 0.5;                                   //  -0.5 to +0.5
      tramp = 1.0 + hramp + vramp;

      pix1 = bmpixel(E1rgb16,px3,py3);                                     //  input pixel
      pix3 = bmpixel(E3rgb16,px3,py3);                                     //  output pixel
      
      red1 = pix1[0];
      green1 = pix1[1];
      blue1 = pix1[2];

      maxrgb = red1;
      if (green1 > maxrgb) maxrgb = green1;
      if (blue1 > maxrgb) maxrgb = blue1;

      if (tramp * maxrgb > 65535) tramp = 65535.0 / maxrgb;

      red3 = tramp * red1;
      green3 = tramp * green1;
      blue3 = tramp * blue1;

      if (sa_active && dist < sa_blend) {                                  //  blend changes over blendwidth   v.10.1
         dnew = 1.0 * dist / sa_blend;
         dold = 1.0 - dnew;
         red3 = dnew * red3 + dold * red1;
         green3 = dnew * green3 + dold * green1;
         blue3 = dnew * blue3 + dold * blue1;
      }

      pix3[0] = red3;
      pix3[1] = green3;
      pix3[2] = blue3;
   }

   exit_wt();
   return 0;                                                               //  not executed, avoid gcc warning
}


/**************************************************************************/

//  Expand the brightness range and clip the magnitude of pixels
//    that overflow the max/min allowed values.

double   xbrightD, xbrightB;

void m_xbright(GtkWidget *, cchar *)                                       //  new v.10.1
{
   int      xbright_dialog_event(zdialog *zd, cchar *event);
   void *   xbright_thread(void *);
   
   cchar  *title = ZTX("Expand Brightness Range");
   cchar  *labP = ZTX("brightness to clip (percent)");
   cchar  *labD = ZTX("dark pixels");
   cchar  *labB = ZTX("bright pixels");

   if (! edit_setup("expand",1,2)) return;                                 //  setup: preview, select area OK

   zdedit = zdialog_new(title,mWin,Bdone,Bcancel,null);
   zdialog_add_widget(zdedit,"label","labP","dialog",labP,"space=5");      //  brightness to clip (percent)
   zdialog_add_widget(zdedit,"hbox","hb1","dialog",0,"space=5");           //  dark pixels [__]  bright pixels [__]
   zdialog_add_widget(zdedit,"label","labD","hb1",labD,"space=4");         //
   zdialog_add_widget(zdedit,"spin","clipD","hb1","0|49|0.1|0");           //  [done] [cancel]
   zdialog_add_widget(zdedit,"label","labS","hb1","","space=10");
   zdialog_add_widget(zdedit,"label","labB","hb1",labB,"space=4");
   zdialog_add_widget(zdedit,"spin","clipB","hb1","0|49|0.1|0");

   xbrightD = xbrightB = 0;                                                //  initial clip = 0
   
   zdialog_run(zdedit,xbright_dialog_event);                               //  run dialog, parallel

   start_thread(xbright_thread,0);                                         //  start working thread
   return;
}


//  dialog event and completion callback function

int xbright_dialog_event(zdialog *zd, cchar *event)
{
   if (zd->zstat)                                                          //  dialog complete
   {
      if (zd->zstat == 1) edit_done();
      else edit_cancel();
      return 1;
   }

   if (strEqu(event,"F1")) showz_userguide("expand_brightness");           //  F1 context help
   if (strEqu(event,"undo")) edit_undo();                                  //  v.10.2
   if (strEqu(event,"redo")) edit_redo();                                  //  v.10.3
   if (strEqu(event,"blendwidth")) signal_thread();                        //  v.10.3

   if (strEqu(event,"clipD")) {
      zdialog_fetch(zd,"clipD",xbrightD);
      signal_thread();
   }

   if (strEqu(event,"clipB")) {
      zdialog_fetch(zd,"clipB",xbrightB);
      signal_thread();
   }

   return 0;
}


//  thread function

void * xbright_thread(void *)
{
   int         ii, px, py, rgb, dist = 0;
   double      dark, bright, b1, b3, bf;
   double      pval1, pval3, f1, f2;
   uint16      *pix1, *pix3;
   
   while (true)
   {
      thread_idle_loop();                                                  //  wait for work or exit request

      dark = 0.01 * xbrightD * 65536;                                      //  clipping brightness levels
      bright = (1.0 - 0.01 * xbrightB) * 65536;

      for (py = 0; py < E3hh; py++)                                        //  loop all image pixels
      for (px = 0; px < E3ww; px++)
      {
         ii = py * E3ww + px;

         if (sa_active) {                                                  //  select area active     bugfix v.9.9
            dist = sa_pixisin[ii];                                         //  distance from edge
            if (! dist) continue;                                          //  pixel is outside area
         }

         pix1 = bmpixel(E1rgb16,px,py);                                    //  input pixel
         pix3 = bmpixel(E3rgb16,px,py);                                    //  output pixel

         b1 = brightness(pix1);

         if (b1 < dark)                                                    //  clip dark pixels
            b3 = 0;
         else if (b1 > bright)                                             //  clip bright pixels
            b3 = bright;
         else b3 = b1;
         
         if (b3 > dark) b3 = b3 - dark;                                    //  expand the rest
         b3 = b3 * (65535.0 / (bright - dark));

         bf = b3 / (b1 + 1);                                               //  brightness ratio
         
         for (rgb = 0; rgb < 3; rgb++)                                     //  loop 3 RGB colors
         {
            pval1 = pix1[rgb];
            pval3 = bf * pval1;                                            //  apply ratio
            if (pval3 > 65535) pval3 = 65535;

            if (sa_active && dist < sa_blend) {                            //  select area is active,
               f1 = 1.0 * dist / sa_blend;                                 //    blend changes over sa_blend
               f2 = 1.0 - f1;
               pval3 = int(f1 * pval3 + f2 * pval1);
            }

            pix3[rgb] = pval3;
         }
      }

      Fmodified = 1;
      mwpaint2();
   }

   return 0;                                                               //  not executed, stop g++ warning
}


/**************************************************************************

   Image Tone Mapping function
   enhance local contrast as opposed to overall contrast

   methodology:
   get brightness gradients for each pixel in 4 directions: SE SW NE NW
   amplify gradients using the edit curve (x-axis range 0-max. gradient)
   integrate 4 new brightness surfaces from the amplified gradients:
     - pixel brightness = prior pixel brightness + amplified gradient
     - use constraint factor to push brightness toward original value
   new pixel brightness = average from 4 calculated brightness surfaces

***************************************************************************/

float    *Tmap_brmap1;
float    *Tmap_brmap3[4];
int      Tmap_contrast99;
double   Tmap_constraint;

void m_tonemap(GtkWidget *, cchar *)                                       //  new v.9.8
{
   using namespace splinecurve;
   
   int      Tmap_dialog_event(zdialog *zd, cchar *event);
   void     Tmap_curvedit(int);
   void *   Tmap_thread(void *);

   int         ii, cc, px, py;
   uint16      *pix1;
   cchar       *title = ZTX("Tone Mapping");
   cchar       *Bcons = ZTX("Constrain");
   int         jj, sum, limit, condist[100];

   if (! edit_setup("tonemap",0,2)) return;                                //  setup: no preview, select area OK

   zdedit = zdialog_new(title,mWin,Bdone,Bcancel,null);                       //   _____________________________
   zdialog_add_widget(zdedit,"frame","frame","dialog",0,"expand");            //  |                             |
   zdialog_add_widget(zdedit,"hbox","hb1","dialog",0,"space=5");              //  |    curve drawing area       |
   zdialog_add_widget(zdedit,"label","labcL","hb1","low","space=4");          //  |                             |
   zdialog_add_widget(zdedit,"label","labcM","hb1","contrast","expand");      //  |_____________________________| 
   zdialog_add_widget(zdedit,"label","labcH","hb1","high","space=5");         //    low      contrast      high
   zdialog_add_widget(zdedit,"hbox","hb2","dialog",0,"space=5");              //
   zdialog_add_widget(zdedit,"label","labcon","hb2",Bcons,"space=5");         //    constrain ======[]======
   zdialog_add_widget(zdedit,"hscale","cons","hb2","0|1|0.01|0.4","expand");
   
   GtkWidget *frame = zdialog_widget(zdedit,"frame");                      //  set up curve edit
   curve_init(frame,Tmap_curvedit);

   Nspc = 1;
   vert[0] = 0;
   nap[0] = 3;                                                             //  initial curve anchor points
   apx[0][0] = 0.01;
   apy[0][0] = 0.2;
   apx[0][1] = 0.50;
   apy[0][1] = 0.4;
   apx[0][2] = 0.99;
   apy[0][2] = 0.0;

   curve_generate(0);                                                      //  generate curve data

   cc = Fww * Fhh * sizeof(float);                                         //  allocate brightness map memory
   Tmap_brmap1 = (float *) zmalloc(cc,"tmap.br1");
   for (ii = 0; ii < 4; ii++)
      Tmap_brmap3[ii] = (float *) zmalloc(cc,"tmap.br3");

   for (py = 0; py < Fhh; py++)                                            //  map initial image brightness
   for (px = 0; px < Fww; px++)
   {
      ii = py * Fww + px;
      pix1 = bmpixel(E1rgb16,px,py);
      Tmap_brmap1[ii] = brightness(pix1);
   }

   for (ii = 0; ii < 100; ii++) 
      condist[ii] = 0;

   for (py = 1; py < Fhh; py++)                                            //  map contrast distribution
   for (px = 1; px < Fww; px++)
   {
      ii = py * Fww + px;
      jj = 0.00152 * fabsf(Tmap_brmap1[ii] - Tmap_brmap1[ii-1]);           //  contrast, ranged 0 - 99
      condist[jj]++;
      jj = 0.00152 * fabsf(Tmap_brmap1[ii] - Tmap_brmap1[ii-Fww]);
      condist[jj]++;
   }
   
   sum = 0;
   limit = 0.99 * 2 * (Fww-1) * (Fhh-1);                                   //  find 99th percentile contrast

   for (ii = 0; ii < 100; ii++) {
      sum += condist[ii];
      if (sum > limit) break;
   }
   
   Tmap_contrast99 = 65535.0 * ii / 100.0;                                 //  0 to 65535

   zdialog_resize(zdedit,300,250);
   zdialog_run(zdedit,Tmap_dialog_event);                                  //  run dialog, parallel

   Tmap_constraint = 0.5;
   start_thread(Tmap_thread,0);                                            //  start working thread
   signal_thread();
   return;
}


//  dialog event and completion callback function

int Tmap_dialog_event(zdialog *zd, cchar *event)
{
   if (zd->zstat)                                                          //  dialog complete
   {
      if (zd->zstat == 1) edit_done();
      else edit_cancel();
      zfree(Tmap_brmap1);                                                  //  free memory
      for (int ii = 0; ii < 4; ii++)
      zfree(Tmap_brmap3[ii]);
      return 1;
   }

   if (strEqu(event,"F1")) showz_userguide("tone_mapping");                //  F1 context help
   if (strEqu(event,"undo")) edit_undo();                                  //  v.10.2
   if (strEqu(event,"redo")) edit_redo();                                  //  v.10.3
   if (strEqu(event,"blendwidth")) signal_thread();                        //  v.10.3

   if (strEqu(event,"cons")) {
      zdialog_fetch(zd,"cons",Tmap_constraint);
      signal_thread();
   }
   
   return 0;
}


//  this function is called when the curve is edited

void Tmap_curvedit(int)
{
   signal_thread();
   return;
}


//  thread function

void * Tmap_thread(void *)
{
   void * Tmap_wthread1(void *arg);
   void * Tmap_wthread2(void *arg);

   while (true)
   {
      thread_idle_loop();                                                  //  wait for work or exit request

      if (sa_active) progress_goal = sa_Npixel;                            //  setup progress tracking   v.10.2
      else  progress_goal = E3ww * E3hh;
      progress_goal = progress_goal * 2;                                   //  2 passes through all pixels
      progress_done = 0;

      for (int ii = 0; ii < 4; ii++)                                       //  start working threads 1 (must be 4)
         start_wt(Tmap_wthread1,&wtnx[ii]);
      wait_wts();                                                          //  wait for completion

      for (int ii = 0; ii < Nwt; ii++)                                     //  start working threads 2
         start_wt(Tmap_wthread2,&wtnx[ii]);
      wait_wts();                                                          //  wait for completion

      progress_goal = progress_done = 0;

      Fmodified = 1;
      mwpaint2(); 
   }

   return 0;                                                               //  not executed, stop g++ warning
}


//  working threads

void * Tmap_wthread1(void *arg)                                            //  hardened   v.9.9
{
   using namespace splinecurve;

   int         ii, bii, pii, dist = 0;
   int         px, px1, px2, pxinc;
   int         py, py1, py2, pyinc;
   float       b1, b3, xval, yval, grad;
   float       cons1, cons2, contrast99;

   bii = *((int *) arg);

   if (bii == 0) {                                                         //  direction SE
      px1 = 1; px2 = E3ww; pxinc = 1;
      py1 = 1; py2 = E3hh; pyinc = 1;
      pii = - 1 - E3ww;
   }
   
   else if (bii == 1) {                                                    //  direction SW
      px1 = E3ww-2; px2 = 0; pxinc = -1;
      py1 = 1; py2 = E3hh; pyinc = 1;
      pii = + 1 - E3ww;
   }
   
   else if (bii == 2) {                                                    //  direction NE
      px1 = 1; px2 = E3ww; pxinc = 1;
      py1 = E3hh-2; py2 = 0; pyinc = -1;
      pii = - 1 + E3ww;
   }
   
   else {   /* bii == 3 */                                                 //  direction NW
      px1 = E3ww-2; px2 = 0; pxinc = -1;
      py1 = E3hh-2; py2 = 0; pyinc = -1;
      pii = + 1 + E3ww;
   }

   contrast99 = Tmap_contrast99;                                           //  99th percentile contrast
   contrast99 = 1.0 / contrast99;                                          //  inverted

   cons1 = pow(Tmap_constraint,2);                                         //  get constraint
   cons2 = 1.0 - cons1;

   for (ii = 0; ii < E3ww * E3hh; ii++)                                    //  initial brightness map
      Tmap_brmap3[bii][ii] = Tmap_brmap1[ii];

   for (py = py1; py != py2; py += pyinc)                                  //  loop all image pixels
   for (px = px1; px != px2; px += pxinc)
   {
      ii = py * E3ww + px;

      if (sa_active) {                                                     //  select area active
         dist = sa_pixisin[ii];                                            //  distance from edge
         if (! dist) continue;                                             //  outside pixel
      }

      b1 = Tmap_brmap1[ii];                                                //  this pixel brightness
      grad = b1 - Tmap_brmap1[ii+pii];                                     //  - prior pixel --> gradient

      xval = fabsf(grad) * contrast99;                                     //  gradient scaled 0 to 1+
      yval = 1.0 + 5 * curve_yval(0,xval);                                 //  1 to 6         1.6x faster    v.10.1

      grad = grad * yval;                                                  //  magnified gradient

      b3 = Tmap_brmap3[bii][ii+pii] + grad;                                //  pixel brightness = prior + gradient
      b3 = cons1 * b1 + cons2 * b3;                                        //  constrain: push toward b1
      
      if (b3 > 65535) b3 = 65535;                                          //  constrain
      if (b3 < 10) b3 = 10;
      
      Tmap_brmap3[bii][ii] = b3;                                           //  new pixel brightness

      progress_done++;                                                     //  track progress         v.10.2
   }

   exit_wt();
   return 0;                                                               //  not executed, avoid gcc warning
}


void * Tmap_wthread2(void *arg)                                            //  speedup   v.10.2
{
   uint16      *pix1, *pix3;
   int         index, ii, px, py, dist = 0;
   int         rgb, pval1, pval3;
   float       b1, b3, bf, f1, f2;

   index = *((int *) arg);

   for (py = index; py < E3hh; py += Nwt)                                  //  loop image pixels
   for (px = 0; px < E3ww; px++)
   {
      ii = py * E3ww + px;

      if (sa_active) {                                                     //  select area active     bugfix v.9.9
         dist = sa_pixisin[ii];                                            //  distance from edge
         if (! dist) continue;                                             //  pixel is outside area
      }

      pix1 = bmpixel(E1rgb16,px,py);                                       //  input pixel
      pix3 = bmpixel(E3rgb16,px,py);                                       //  output pixel

      b1 = Tmap_brmap1[ii];                                                //  initial pixel brightness
      b3 = Tmap_brmap3[0][ii] + Tmap_brmap3[1][ii]                         //  new brightness = average of four
         + Tmap_brmap3[2][ii] + Tmap_brmap3[3][ii];                        //    calculated brightness surfaces
      bf = 0.25 * b3 / (b1 + 1);                                           //  brightness ratio
      
      for (rgb = 0; rgb < 3; rgb++)                                        //  loop 3 RGB colors
      {
         pval1 = pix1[rgb];
         pval3 = bf * pval1;                                               //  apply ratio
         if (pval3 > 65535) pval3 = 65535;

         if (sa_active && dist < sa_blend) {                               //  select area is active,
            f1 = 1.0 * dist / sa_blend;                                    //    blend changes over sa_blend
            f2 = 1.0 - f1;
            pval3 = int(f1 * pval3 + f2 * pval1);
         }

         pix3[rgb] = pval3;
      }

      progress_done++;                                                     //  track progress         v.10.2
   }

   exit_wt();
   return 0;                                                               //  not executed, stop g++ warning
}


/**************************************************************************/

//  red eye removal function

struct sredmem {                                                           //  red-eye struct in memory
   char        type, space[3];
   int         cx, cy, ww, hh, rad, clicks;
   double      thresh, tstep;
};
sredmem  redmem[100];                                                      //  store up to 100 red-eyes

int      Nredmem = 0, maxredmem = 100;


void m_redeye(GtkWidget *, cchar *)
{
   void     redeye_mousefunc();
   int      redeye_dialog_event(zdialog *zd, cchar *event);

   cchar    *redeye_message = ZTX(
               "Method 1:\n"
               "  Left-click on red-eye to darken.\n"
               "Method 2:\n"
               "  Drag down and right to enclose red-eye.\n"
               "  Left-click on red-eye to darken.\n"
               "Undo red-eye:\n"
               "  Right-click on red-eye.");

   if (! edit_setup("redeye",0,1)) return;                                 //  setup edit: no preview

   zdedit = zdialog_new(ZTX("Red Eye Reduction"),mWin,Bdone,Bcancel,null);
   zdialog_add_widget(zdedit,"label","lab1","dialog",redeye_message);
   zdialog_run(zdedit,redeye_dialog_event);                                //  run dialog, parallel mode

   Nredmem = 0;
   mouseCBfunc = redeye_mousefunc;                                         //  connect mouse function
   Mcapture = 1;                                                           //  capture mouse clicks
   return;
}


//  dialog event and completion callback function

int redeye_dialog_event(zdialog *zd, cchar *event)
{
   if (zd->zstat)
   {
      mouseCBfunc = 0;                                                     //  disconnect mouse
      Mcapture = 0;
      if (Nredmem > 0) Fmodified = 1;
      Ftoparc = ptoparc = 0;
      if (zd->zstat == 1) edit_done();
      else edit_cancel();
      return 0;
   }
   
   if (strEqu(event,"F1")) showz_userguide("red_eye");                     //  F1 help       v.9.0
   return 0;
}


//  mouse functions to define, darken, and undo red-eyes

int      redeye_createF(int px, int py);                                   //  create 1-click red-eye (type F)
int      redeye_createR(int px, int py, int ww, int hh);                   //  create robust red-eye (type R)
void     redeye_darken(int ii);                                            //  darken red-eye
void     redeye_distr(int ii);                                             //  build pixel redness distribution
int      redeye_find(int px, int py);                                      //  find red-eye at mouse position
void     redeye_remove(int ii);                                            //  remove red-eye at mouse position
int      redeye_radlim(int cx, int cy);                                    //  compute red-eye radius limit


void redeye_mousefunc()
{
   int         ii, px, py, ww, hh;

   if (Nredmem == maxredmem) {
      zmessageACK(GTK_WINDOW(mWin),"%d red-eye limit reached",maxredmem);  //  too many red-eyes
      return;
   }

   if (LMclick)                                                            //  left mouse click
   {
      LMclick = 0;

      px = Mxclick;                                                        //  click position
      py = Myclick;
      if (px < 0 || px > E3ww-1 || py < 0 || py > E3hh-1) return;          //  outside image area

      ii = redeye_find(px,py);                                             //  find existing red-eye
      if (ii < 0) ii = redeye_createF(px,py);                              //  or create new type F
      redeye_darken(ii);                                                   //  darken red-eye
   }
   
   if (RMclick)                                                            //  right mouse click
   {
      RMclick = 0;
      px = Mxclick;                                                        //  click position
      py = Myclick;
      ii = redeye_find(px,py);                                             //  find red-eye
      if (ii >= 0) redeye_remove(ii);                                      //  if found, remove
   }

   if (Mxdrag || Mydrag)                                                   //  mouse drag underway
   {
      px = Mxdown;                                                         //  initial position
      py = Mydown;
      ww = Mxdrag - Mxdown;                                                //  increment
      hh = Mydrag - Mydown;
      if (ww < 2 && hh < 2) return;
      if (ww < 2) ww = 2;
      if (hh < 2) hh = 2;
      if (px < 1) px = 1;                                                  //  keep within image area
      if (py < 1) py = 1;      
      if (px + ww > E3ww-1) ww = E3ww-1 - px;
      if (py + hh > E3hh-1) hh = E3hh-1 - py;
      ii = redeye_find(px,py);                                             //  find existing red-eye
      if (ii >= 0) redeye_remove(ii);                                      //  remove it
      ii = redeye_createR(px,py,ww,hh);                                    //  create new red-eye type R
   }

   mwpaint2();
   return;
}


//  create type F redeye (1-click automatic)

int redeye_createF(int cx, int cy)
{
   int         cx0, cy0, cx1, cy1, px, py, rad, radlim;
   int         loops, ii;
   int         Tnpix, Rnpix, R2npix;
   double      rd, rcx, rcy, redpart;
   double      Tsum, Rsum, R2sum, Tavg, Ravg, R2avg;
   double      sumx, sumy, sumr;
   uint16      *ppix;
   
   cx0 = cx;
   cy0 = cy;
   
   for (loops = 0; loops < 8; loops++)
   {
      cx1 = cx;
      cy1 = cy;

      radlim = redeye_radlim(cx,cy);                                       //  radius limit (image edge)
      Tsum = Tavg = Ravg = Tnpix = 0;

      for (rad = 0; rad < radlim-2; rad++)                                 //  find red-eye radius from (cx,cy)
      {
         Rsum = Rnpix = 0;
         R2sum = R2npix = 0;

         for (py = cy-rad-2; py <= cy+rad+2; py++)
         for (px = cx-rad-2; px <= cx+rad+2; px++)
         {
            rd = sqrt((px-cx)*(px-cx) + (py-cy)*(py-cy));
            ppix = bmpixel(E3rgb16,px,py);
            redpart = redness(ppix);

            if (rd <= rad + 0.5 && rd > rad - 0.5) {                       //  accum. redness at rad
               Rsum += redpart;
               Rnpix++;
            }
            else if (rd <= rad + 2.5 && rd > rad + 1.5) {                  //  accum. redness at rad+2
               R2sum += redpart;
               R2npix++;
            }
         }
         
         Tsum += Rsum;
         Tnpix += Rnpix;
         Tavg = Tsum / Tnpix;                                              //  avg. redness over 0-rad
         Ravg = Rsum / Rnpix;                                              //  avg. redness at rad
         R2avg = R2sum / R2npix;                                           //  avg. redness at rad+2
         if (R2avg > Ravg || Ravg > Tavg) continue;
         if ((Ravg - R2avg) < 0.2 * (Tavg - Ravg)) break;                  //  0.1 --> 0.2      v.8.6
      }
      
      sumx = sumy = sumr = 0;
      rad = int(1.2 * rad + 1);
      if (rad > radlim) rad = radlim;
      
      for (py = cy-rad; py <= cy+rad; py++)                                //  compute center of gravity for
      for (px = cx-rad; px <= cx+rad; px++)                                //   pixels within rad of (cx,cy)
      {
         rd = sqrt((px-cx)*(px-cx) + (py-cy)*(py-cy));
         if (rd > rad + 0.5) continue;
         ppix = bmpixel(E3rgb16,px,py);
         redpart = redness(ppix);                                          //  weight by redness    v.8.6
         sumx += redpart * (px - cx);
         sumy += redpart * (py - cy);
         sumr += redpart;
      }

      rcx = cx + 1.0 * sumx / sumr;                                        //  new center of red-eye
      rcy = cy + 1.0 * sumy / sumr;
      if (fabs(cx0 - rcx) > 0.6 * rad) break;                              //  give up if big movement
      if (fabs(cy0 - rcy) > 0.6 * rad) break;
      cx = int(rcx + 0.5);
      cy = int(rcy + 0.5);
      if (cx == cx1 && cy == cy1) break;                                   //  done if no change
   }

   radlim = redeye_radlim(cx,cy);
   if (rad > radlim) rad = radlim;

   ii = Nredmem++;                                                         //  add red-eye to memory
   redmem[ii].type = 'F';
   redmem[ii].cx = cx;
   redmem[ii].cy = cy;
   redmem[ii].rad = rad;
   redmem[ii].clicks = 0;
   redmem[ii].thresh = 0;
   return ii;
}


//  create type R red-eye (drag an ellipse over red-eye area)

int redeye_createR(int cx, int cy, int ww, int hh)
{
   int      rad, radlim;

   Ftoparc = 1;                                                            //  paint ellipse over image
   toparcx = cx - ww;                                                      //  v.8.3
   toparcy = cy - hh;
   toparcw = 2 * ww;
   toparch = 2 * hh;

   if (ww > hh) rad = ww;
   else rad = hh;
   radlim = redeye_radlim(cx,cy);
   if (rad > radlim) rad = radlim;

   int ii = Nredmem++;                                                     //  add red-eye to memory
   redmem[ii].type = 'R';
   redmem[ii].cx = cx;
   redmem[ii].cy = cy;
   redmem[ii].ww = 2 * ww;
   redmem[ii].hh = 2 * hh;
   redmem[ii].rad = rad;
   redmem[ii].clicks = 0;
   redmem[ii].thresh = 0;
   return ii;
}


//  darken a red-eye and increase click count

void redeye_darken(int ii)
{
   int         cx, cy, ww, hh, px, py, rad, clicks;
   double      rd, thresh, tstep;
   char        type;
   uint16      *ppix;

   type = redmem[ii].type;
   cx = redmem[ii].cx;
   cy = redmem[ii].cy;
   ww = redmem[ii].ww;
   hh = redmem[ii].hh;
   rad = redmem[ii].rad;
   thresh = redmem[ii].thresh;
   tstep = redmem[ii].tstep;
   clicks = redmem[ii].clicks++;
   
   if (thresh == 0)                                                        //  1st click 
   {
      redeye_distr(ii);                                                    //  get pixel redness distribution
      thresh = redmem[ii].thresh;                                          //  initial redness threshhold
      tstep = redmem[ii].tstep;                                            //  redness step size
      Ftoparc = 0;
   }

   tstep = (thresh - tstep) / thresh;                                      //  convert to reduction factor
   thresh = thresh * pow(tstep,clicks);                                    //  reduce threshhold by total clicks

   for (py = cy-rad; py <= cy+rad; py++)                                   //  darken pixels over threshhold
   for (px = cx-rad; px <= cx+rad; px++)
   {
      if (type == 'R') {
         if (px < cx - ww/2) continue;
         if (px > cx + ww/2) continue;
         if (py < cy - hh/2) continue;
         if (py > cy + hh/2) continue;
      }
      rd = sqrt((px-cx)*(px-cx) + (py-cy)*(py-cy));
      if (rd > rad + 0.5) continue;
      ppix = bmpixel(E3rgb16,px,py);                                       //  set redness = threshhold
      if (redness(ppix) > thresh)
         ppix[0] = int(thresh * (0.65 * ppix[1] + 0.10 * ppix[2] + 1) / (25 - 0.25 * thresh));
   }

   return;
}


//  Build a distribution of redness for a red-eye. Use this information 
//  to set initial threshhold and step size for stepwise darkening.

void redeye_distr(int ii)
{
   int         cx, cy, ww, hh, rad, px, py;
   int         bin, npix, dbins[20], bsum, blim;
   double      rd, maxred, minred, redpart, dbase, dstep;
   char        type;
   uint16      *ppix;
   
   type = redmem[ii].type;
   cx = redmem[ii].cx;
   cy = redmem[ii].cy;
   ww = redmem[ii].ww;
   hh = redmem[ii].hh;
   rad = redmem[ii].rad;
   
   maxred = 0;
   minred = 100;

   for (py = cy-rad; py <= cy+rad; py++)
   for (px = cx-rad; px <= cx+rad; px++)
   {
      if (type == 'R') {
         if (px < cx - ww/2) continue;
         if (px > cx + ww/2) continue;
         if (py < cy - hh/2) continue;
         if (py > cy + hh/2) continue;
      }
      rd = sqrt((px-cx)*(px-cx) + (py-cy)*(py-cy));
      if (rd > rad + 0.5) continue;
      ppix = bmpixel(E3rgb16,px,py);
      redpart = redness(ppix);
      if (redpart > maxred) maxred = redpart;
      if (redpart < minred) minred = redpart;
   }
   
   dbase = minred;
   dstep = (maxred - minred) / 19.99;

   for (bin = 0; bin < 20; bin++) dbins[bin] = 0;
   npix = 0;

   for (py = cy-rad; py <= cy+rad; py++)
   for (px = cx-rad; px <= cx+rad; px++)
   {
      if (type == 'R') {
         if (px < cx - ww/2) continue;
         if (px > cx + ww/2) continue;
         if (py < cy - hh/2) continue;
         if (py > cy + hh/2) continue;
      }
      rd = sqrt((px-cx)*(px-cx) + (py-cy)*(py-cy));
      if (rd > rad + 0.5) continue;
      ppix = bmpixel(E3rgb16,px,py);
      redpart = redness(ppix);
      bin = int((redpart - dbase) / dstep);
      ++dbins[bin];
      ++npix;
   }
   
   bsum = 0;
   blim = int(0.5 * npix);

   for (bin = 0; bin < 20; bin++)                                          //  find redness level for 50% of
   {                                                                       //    pixels within red-eye radius
      bsum += dbins[bin];
      if (bsum > blim) break;
   }

   redmem[ii].thresh = dbase + dstep * bin;                                //  initial redness threshhold
   redmem[ii].tstep = dstep;                                               //  redness step (5% of range)   v.6.9

   return;
}


//  find a red-eye (nearly) overlapping the mouse click position

int redeye_find(int cx, int cy)
{
   for (int ii = 0; ii < Nredmem; ii++)
   {
      if (cx > redmem[ii].cx - 2 * redmem[ii].rad && 
          cx < redmem[ii].cx + 2 * redmem[ii].rad &&
          cy > redmem[ii].cy - 2 * redmem[ii].rad && 
          cy < redmem[ii].cy + 2 * redmem[ii].rad) 
            return ii;                                                     //  found
   }
   return -1;                                                              //  not found
}


//  remove a red-eye from memory

void redeye_remove(int ii)
{
   int      cx, cy, rad, px, py;
   uint16   *pix1, *pix3;

   cx = redmem[ii].cx;
   cy = redmem[ii].cy;
   rad = redmem[ii].rad;

   for (px = cx-rad; px <= cx+rad; px++)
   for (py = cy-rad; py <= cy+rad; py++)
   {
      pix1 = bmpixel(E1rgb16,px,py);
      pix3 = bmpixel(E3rgb16,px,py);
      pix3[0] = pix1[0];
      pix3[1] = pix1[1];
      pix3[2] = pix1[2];
   }
   
   for (ii++; ii < Nredmem; ii++) 
      redmem[ii-1] = redmem[ii];
   Nredmem--;
   
   Ftoparc = 0;
   return;
}


//  compute red-eye radius limit: smaller of 100 and nearest image edge

int redeye_radlim(int cx, int cy)
{
   int radlim = 100;
   if (cx < 100) radlim = cx;
   if (E3ww-1 - cx < 100) radlim = E3ww-1 - cx;
   if (cy < 100) radlim = cy;
   if (E3hh-1 - cy < 100) radlim = E3hh-1 - cy;
   return radlim;
}


/**************************************************************************/

//  image blur function                                                    //  radius steps of 0.5   v.9.2

double      blur_radius;
double      blur_weight[101][101];                                         //  up to blur radius = 99   v.9.2
int         blur_Npixels, blur_pixdone;


void m_blur(GtkWidget *, cchar *)
{
   int    blur_dialog_event(zdialog *zd, cchar *event);
   void * blur_thread(void *);

   if (! edit_setup("blur",0,2)) return;                                   //  setup edit: no preview

   zdedit = zdialog_new(ZTX("Set Blur Radius"),mWin,Bdone,Bcancel,null);
   zdialog_add_widget(zdedit,"hbox","hb2","dialog",0,"space=10");
   zdialog_add_widget(zdedit,"label","labrad","hb2",Bradius,"space=5");
   zdialog_add_widget(zdedit,"spin","radius","hb2","0|99|0.5|0.5","space=5");
   zdialog_add_widget(zdedit,"button","apply","hb2",Bapply,"space=5");

   zdialog_run(zdedit,blur_dialog_event);                                  //  start dialog
   
   blur_radius = 0.5;
   start_thread(blur_thread,0);                                            //  start working thread
   return;
}


//  dialog event and completion callback function

int blur_dialog_event(zdialog * zd, cchar *event)
{
   if (zd->zstat)
   {
      if (zd->zstat == 1) edit_done();                                     //  done
      else edit_cancel();                                                  //  cancel or destroy
      return 1;
   }
   
   if (strEqu(event,"F1")) showz_userguide("blur");                        //  F1 context help
   if (strEqu(event,"undo")) edit_undo();                                  //  v.10.2
   if (strEqu(event,"redo")) edit_redo();                                  //  v.10.3
   if (strEqu(event,"blendwidth")) signal_thread();                        //  v.10.3

   if (strEqu(event,"apply")) {
      zdialog_fetch(zd,"radius",blur_radius);                              //  get blur radius
      if (blur_radius == 0) edit_reset();
      else signal_thread();                                                //  trigger working thread
   }

   return 1;
}


//  image blur thread function

void * blur_thread(void *)
{
   void * blur_wthread(void *arg);

   int         dx, dy;
   double      rad, rad2;
   double      m, d, w, sum;
  
   while (true)
   {
      thread_idle_loop();                                                  //  wait for work or exit request

      rad = blur_radius - 0.2;                                             //  v.9.2
      rad2 = rad * rad;

      for (dx = 0; dx <= rad+1; dx++)                                      //  clear weights array
      for (dy = 0; dy <= rad+1; dy++)
         blur_weight[dx][dy] = 0;

      for (dx = -rad-1; dx <= rad+1; dx++)                                 //  blur_weight[dx][dy] = no. of pixels
      for (dy = -rad-1; dy <= rad+1; dy++)                                 //    at distance (dx,dy) from center
         ++blur_weight[abs(dx)][abs(dy)];

      m = sqrt(rad2 + rad2);                                               //  corner pixel distance from center
      sum = 0;

      for (dx = 0; dx <= rad+1; dx++)                                      //  compute weight of pixel
      for (dy = 0; dy <= rad+1; dy++)                                      //    at distance dx, dy
      {
         d = sqrt(dx*dx + dy*dy);
         w = (m + 1.2 - d) / m;                                            //  v.9.2
         w = w * w;
         sum += blur_weight[dx][dy] * w;
         blur_weight[dx][dy] = w;
      }

      for (dx = 0; dx <= rad+1; dx++)                                      //  make weights add up to 1.0
      for (dy = 0; dy <= rad+1; dy++)
         blur_weight[dx][dy] = blur_weight[dx][dy] / sum;
      
      if (sa_active) progress_goal = sa_Npixel;
      else  progress_goal = E3ww * E3hh;
      progress_done = 0;

      for (int ii = 0; ii < Nwt; ii++)                                     //  start worker threads
         start_wt(blur_wthread,&wtnx[ii]);
      wait_wts();                                                          //  wait for completion

      progress_goal = progress_done = 0;                                   //  v.9.6
      Fmodified = 1;
      mwpaint2();                                                          //  update window
   }

   return 0;                                                               //  not executed, stop g++ warning
}


void * blur_wthread(void *arg)                                             //  worker thread function
{
   void  blur_pixel(int px, int py);

   int      index = *((int *) arg);
   int      px, py;
   
   for (py = index; py < E3hh-1; py += Nwt)                                //  loop all image pixels
   for (px = 0; px < E3ww-1; px++)
   {
      blur_pixel(px,py);
      if (Fkillfunc) break;
   }

   exit_wt();
   return 0;                                                               //  not executed, avoid gcc warning
}


void blur_pixel(int px, int py)
{
   int         ii, dist = 0;
   int         jj, dx, dy, adx, ady, rad;
   double      red, green, blue;
   double      weight1, weight2, f1, f2;
   uint16      *pix1, *pix3, *pixN;

   if (sa_active) {                                                        //  select area active
      ii = py * Fww + px;
      dist = sa_pixisin[ii];                                               //  distance from edge
      if (! dist) return;                                                  //  outside pixel
   }

   pix1 = bmpixel(E1rgb16,px,py);                                          //  source pixel
   pix3 = bmpixel(E3rgb16,px,py);                                          //  target pixel
   
   rad = blur_radius;
   red = green = blue = 0;
   weight2 = 0.0;
   
   if (sa_active)                                                          //  select area active
   {
      for (dy = -rad-1; dy <= rad+1; dy++)                                 //  loop neighbor pixels within radius
      for (dx = -rad-1; dx <= rad+1; dx++)
      {
         if (px+dx < 0 || px+dx > E3ww-1) continue;                        //  omit pixels off edge
         if (py+dy < 0 || py+dy > E3hh-1) continue;
         jj = (py+dy) * E3ww + (px+dx);
         if (! sa_pixisin[jj]) continue;                                   //  omit pixels outside area   v.6.3
         adx = abs(dx);
         ady = abs(dy);
         pixN = pix1 + (dy * E3ww + dx) * 3;
         weight1 = blur_weight[adx][ady];                                  //  weight at distance (dx,dy)
         weight2 += weight1;
         red += pixN[0] * weight1;                                         //  accumulate contributions
         green += pixN[1] * weight1;
         blue += pixN[2] * weight1;
      }
      
      red = red / weight2;                                                 //  weighted average
      green = green / weight2;
      blue = blue / weight2;

      if (dist < sa_blend) {                                               //  select area is active,
         f1 = 1.0 * dist / sa_blend;                                       //    blend changes over sa_blend
         f2 = 1.0 - f1;
         red = f1 * red + f2 * pix1[0];
         green = f1 * green + f2 * pix1[1];
         blue = f1 * blue + f2 * pix1[2];
      }

      pix3[0] = int(red);
      pix3[1] = int(green);
      pix3[2] = int(blue);
   }

   else
   {
      for (dy = -rad-1; dy <= rad+1; dy++)                                 //  loop neighbor pixels within radius
      for (dx = -rad-1; dx <= rad+1; dx++)
      {
         if (px+dx < 0 || px+dx > E3ww-1) continue;                        //  omit pixels off edge   v.6.3
         if (py+dy < 0 || py+dy > E3hh-1) continue;
         adx = abs(dx);
         ady = abs(dy);
         pixN = pix1 + (dy * E3ww + dx) * 3;
         weight1 = blur_weight[adx][ady];                                  //  weight at distance (dx,dy)
         weight2 += weight1;
         red += pixN[0] * weight1;                                         //  accumulate contributions
         green += pixN[1] * weight1;
         blue += pixN[2] * weight1;
      }

      red = red / weight2;                                                 //  weighted average   v.6.3
      green = green / weight2;
      blue = blue / weight2;

      pix3[0] = int(red);
      pix3[1] = int(green);
      pix3[2] = int(blue);
   }
   
   progress_done++;
   return;
}


/**************************************************************************/

//  image sharpening function

int      sharp_ED_cycles;
int      sharp_ED_reduce;
int      sharp_ED_thresh;
int      sharp_UM_radius;
int      sharp_UM_amount;
int      sharp_UM_thresh;
int      sharp_UM_Fcalc;
int      sharp_GR_amount;
int      sharp_GR_thresh;
char     sharp_function[4];


void m_sharpen(GtkWidget *, cchar *)
{
   int    sharp_dialog_event(zdialog *zd, cchar *event);
   void * sharp_thread(void *);

   if (! edit_setup("sharp",0,2)) return;                                  //  setup edit: no preview

   zdedit = zdialog_new(ZTX("Sharpen Image"),mWin,Bdone,Bcancel,null); 

   zdialog_add_widget(zdedit,"hbox","hb1","dialog",0,"space=5");
   zdialog_add_widget(zdedit,"vbox","vb11","hb1",0,"space=5");
   zdialog_add_widget(zdedit,"vbox","vb12","hb1",0,"homog|space=5");
   zdialog_add_widget(zdedit,"vbox","vb13","hb1",0,"homog|space=5");
   zdialog_add_widget(zdedit,"button","ED","vb11",ZTX("edge detection"),"space=5");
   zdialog_add_widget(zdedit,"label","lab11","vb12",ZTX("cycles"));
   zdialog_add_widget(zdedit,"label","lab12","vb12",ZTX("reduce"));
   zdialog_add_widget(zdedit,"label","lab13","vb12",Bthresh);
   zdialog_add_widget(zdedit,"spin","cyclesED","vb13","1|30|1|10");
   zdialog_add_widget(zdedit,"spin","reduceED","vb13","50|95|1|80");
   zdialog_add_widget(zdedit,"spin","threshED","vb13","1|99|1|1");

   zdialog_add_widget(zdedit,"hsep","sep2","dialog");
   zdialog_add_widget(zdedit,"hbox","hb2","dialog",0,"space=5");
   zdialog_add_widget(zdedit,"vbox","vb21","hb2",0,"space=5");
   zdialog_add_widget(zdedit,"vbox","vb22","hb2",0,"homog|space=5");
   zdialog_add_widget(zdedit,"vbox","vb23","hb2",0,"homog|space=5");
   zdialog_add_widget(zdedit,"button","UM","vb21",ZTX("unsharp mask"),"space=5");
   zdialog_add_widget(zdedit,"label","lab21","vb22",Bradius);
   zdialog_add_widget(zdedit,"label","lab22","vb22",Bamount);
   zdialog_add_widget(zdedit,"label","lab23","vb22",Bthresh);
   zdialog_add_widget(zdedit,"spin","radiusUM","vb23","1|20|1|2");
   zdialog_add_widget(zdedit,"spin","amountUM","vb23","0|200|1|100");
   zdialog_add_widget(zdedit,"spin","threshUM","vb23","0|100|1|0");

   zdialog_add_widget(zdedit,"hsep","sep3","dialog");
   zdialog_add_widget(zdedit,"hbox","hb3","dialog",0,"space=5");
   zdialog_add_widget(zdedit,"vbox","vb31","hb3",0,"space=5");
   zdialog_add_widget(zdedit,"vbox","vb32","hb3",0,"homog|space=5");
   zdialog_add_widget(zdedit,"vbox","vb33","hb3",0,"homog|space=5");
   zdialog_add_widget(zdedit,"button","GR","vb31",ZTX("brightness gradient"),"space=5");
   zdialog_add_widget(zdedit,"label","lab32","vb32",Bamount);
   zdialog_add_widget(zdedit,"label","lab33","vb32",Bthresh);
   zdialog_add_widget(zdedit,"spin","amountGR","vb33","0|400|1|100");
   zdialog_add_widget(zdedit,"spin","threshGR","vb33","0|100|1|0");

   zdialog_run(zdedit,sharp_dialog_event);                                 //  run dialog, parallel

   *sharp_function = 0;
   sharp_UM_Fcalc = 1;
   start_thread(sharp_thread,0);                                           //  start working thread
   return;
}


//  dialog event and completion callback function

int sharp_dialog_event(zdialog *zd, cchar *event)
{
   if (zd->zstat)
   {
      if (zd->zstat == 1) edit_done();
      else edit_cancel();
      return 0;
   }
   
   if (strEqu(event,"F1")) showz_userguide("sharpen");                     //  F1 help       v.9.0
   if (strEqu(event,"undo")) edit_undo();                                  //  v.10.2
   if (strEqu(event,"redo")) edit_redo();                                  //  v.10.3
   if (strEqu(event,"blendwidth")) signal_thread();                        //  v.10.3
   if (strEqu(event,"radiusUM")) sharp_UM_Fcalc = 1;                       //  must recalculate

   if (strcmpv(event,"ED","UM","GR",null))
   {
      edit_reset();                                                        //  restore original image

      zdialog_fetch(zd,"cyclesED",sharp_ED_cycles);                        //  get all input values
      zdialog_fetch(zd,"reduceED",sharp_ED_reduce);
      zdialog_fetch(zd,"threshED",sharp_ED_thresh);
      zdialog_fetch(zd,"radiusUM",sharp_UM_radius);
      zdialog_fetch(zd,"amountUM",sharp_UM_amount);
      zdialog_fetch(zd,"threshUM",sharp_UM_thresh);
      zdialog_fetch(zd,"amountGR",sharp_GR_amount);
      zdialog_fetch(zd,"threshGR",sharp_GR_thresh);

      strcpy(sharp_function,event);                                        //  pass to working thread
      signal_thread();
   }

   return 0;
}


//  sharpen image thread function

void * sharp_thread(void *)
{
   int    sharp_ED();
   int    sharp_UM();
   int    sharp_GR();
   
   while (true)
   {
      thread_idle_loop();                                                  //  wait for work or exit request
      
      if (strEqu(sharp_function,"ED")) sharp_ED();                         //  do requested function
      if (strEqu(sharp_function,"UM")) sharp_UM();
      if (strEqu(sharp_function,"GR")) sharp_GR();

      Fmodified = 1;
      mwpaint2(); 
   }

   return 0;                                                               //  not executed, stop g++ warning
}


//  image sharpen function by edge detection and compression

int sharp_ED()
{
   void  sharp_pixel_ED(int px, int py, int thresh);

   int      sharp_thresh1 = 100;                                           //  initial threshold
   double   sharp_thresh2 = 0.01 * sharp_ED_reduce;                        //  decline rate
   int      px, py, thresh, cycles;
   
   thresh = sharp_thresh1;
    
   if (sa_active) progress_goal = sa_Npixel;                               //  v.9.6
   else  progress_goal = E3ww * E3hh;
   progress_goal *= sharp_ED_cycles;
   progress_done = 0;

   for (cycles = 0; cycles < sharp_ED_cycles; cycles++)
   {
      if (cycles > 0) thresh = int(thresh * sharp_thresh2);

      for (py = 2; py < E3hh-2; py++)
      for (px = 2; px < E3ww-2; px++)                                      //  loop all pixels
      {
         sharp_pixel_ED(px,py,thresh);
         if (Fkillfunc) break;                                             //  v.9.1
      }

      if (Fkillfunc) break;
   }

   progress_goal = progress_done = 0;                                      //  v.9.6
   return 1;
}


void sharp_pixel_ED(int px, int py, int thresh)
{
   uint16   *pix1, *pix1u, *pix1d;
   uint16   *pix3, *pix3u, *pix3d, *pix3uu, *pix3dd;
   int      ii, dist = 0;
   int      dd, rgb, pthresh;
   int      dx[4] = { -1, 0, 1, 1 };                                       //  4 directions: NW N NE E
   int      dy[4] = { -1, -1, -1, 0 };
   int      pv2, pv2u, pv2d, pv2uu, pv2dd, pvdiff;
   double   f1, f2;
   
   if (sa_active) {                                                        //  select area active
      ii = py * Fww + px;
      dist = sa_pixisin[ii];                                               //  distance from edge
      if (! dist) return;                                                  //  outside pixel
   }

   pthresh = sharp_ED_thresh;                                              //  pthresh = larger
   if (thresh > pthresh) pthresh = thresh;

   pix1 = bmpixel(E1rgb16,px,py);                                          //  input pixel
   pix3 = bmpixel(E3rgb16,px,py);                                          //  output pixel

   for (dd = 0; dd < 4; dd++)                                              //  4 directions
   {
      pix3u = pix3 + (dy[dd] * E3ww + dx[dd]) * 3;                         //  upstream pixel
      pix3d = pix3 - (dy[dd] * E3ww - dx[dd]) * 3;                         //  downstream pixel

      for (rgb = 0; rgb < 3; rgb++)                                        //  loop 3 RGB colors
      {
         pv2 = pix3[rgb];
         pv2u = pix3u[rgb];                                                //  brightness difference
         pv2d = pix3d[rgb];                                                //    across target pixel

         pvdiff = pv2d - pv2u;
         if (pvdiff < 0) pvdiff = -pvdiff;
         if (pvdiff < 256 * pthresh) continue;                             //  brightness slope < threshold

         if (pv2u < pv2 && pv2 < pv2d)                                     //  slope up, monotone
         {
            pix3uu = pix3u + (dy[dd] * E3ww + dx[dd]) * 3;                 //  upstream of upstream pixel
            pix3dd = pix3d - (dy[dd] * E3ww - dx[dd]) * 3;                 //  downstream of downstream
            pv2uu = pix3uu[rgb];
            pv2dd = pix3dd[rgb];

            if (pv2uu >= pv2u) {                                           //  shift focus of changes to
               pix3u = pix3;                                               //    avoid up/down/up jaggies
               pv2u = pv2;
            }
            
            if (pv2dd <= pv2d) {
               pix3d = pix3;
               pv2d = pv2;
            }
               
            if (pv2u > 256) pv2u -= 256;
            if (pv2d < 65279) pv2d += 256;
         }
         
         else if (pv2u > pv2 && pv2 > pv2d)                                //  slope down, monotone
         {
            pix3uu = pix3u + (dy[dd] * E3ww + dx[dd]) * 3;
            pix3dd = pix3d - (dy[dd] * E3ww - dx[dd]) * 3;
            pv2uu = pix3uu[rgb];
            pv2dd = pix3dd[rgb];

            if (pv2uu <= pv2u) {
               pix3u = pix3;
               pv2u = pv2;
            }
            
            if (pv2dd >= pv2d) {
               pix3d = pix3;
               pv2d = pv2;
            }

            if (pv2d > 256) pv2d -= 256;
            if (pv2u < 65279) pv2u += 256;
         }

         else continue;                                                    //  slope too small

         if (sa_active && dist < sa_blend) {                               //  select area is active,
            f1 = 1.0 * dist / sa_blend;                                    //    blend changes over sa_blend
            f2 = 1.0 - f1;
            pix1u = pix1 + (dy[dd] * E1ww + dx[dd]) * 3;                   //  upstream input pixel      bugfix
            pix1d = pix1 - (dy[dd] * E1ww - dx[dd]) * 3;                   //  downstream input pixel     v.7.2.1
            pv2u = int(f1 * pv2u + f2 * pix1u[rgb]);
            pv2d = int(f1 * pv2d + f2 * pix1d[rgb]);
         }

         pix3u[rgb] = pv2u;                                                //  modified brightness values
         pix3d[rgb] = pv2d;                                                //    >> image3 pixel
      }
   }

   progress_done++;                                                        //  v.9.6
   return;
}


//  image sharpen function using unsharp mask

int sharp_UM()
{
   void * sharp_UM_wthread(void *arg);

   if (sharp_UM_Fcalc) {                                                   //  speedup   v.9.6 
      sharp_UM_Fcalc = 0;
      brhood_calc(sharp_UM_radius,'f');
   }   

   if (sa_active) progress_goal = sa_Npixel;                               //  v.9.6
   else  progress_goal = E3ww * E3hh;
   progress_done = 0;

   for (int ii = 0; ii < Nwt; ii++)                                        //  start worker threads
      start_wt(sharp_UM_wthread,&wtnx[ii]);
   wait_wts();                                                             //  wait for completion

   progress_goal = progress_done = 0;                                      //  v.9.6
   return 1;
}


void * sharp_UM_wthread(void *arg)                                         //  worker thread function   v.7.7
{
   void  sharp_pixel_UM(int px, int py);

   int      index = *((int *) arg);
   int      px, py;
   
   for (py = index; py < E3hh; py += Nwt)                                  //  loop all image3 pixels
   for (px = 0; px < E3ww; px++)
   {
      sharp_pixel_UM(px,py);
      if (Fkillfunc) break;                                                //  v.9.1
   }

   exit_wt();
   return 0;                                                               //  not executed, avoid gcc warning
}


void sharp_pixel_UM(int px, int py)                                        //  process one pixel
{                                                                          //  revised  v.9.6
   int         ii, dist = 0;
   double      amount, thresh, bright;
   double      mean, incr, ratio, f1, f2;
   int         rgb, cval1, cval3;
   uint16      *pix1, *pix3;
   
   if (sa_active) {                                                        //  select area active
      ii = py * Fww + px;
      dist = sa_pixisin[ii];                                               //  distance from edge
      if (! dist) return;                                                  //  outside pixel
   }

   amount = 0.01 * sharp_UM_amount;                                        //  0.0 to 2.0
   thresh = 100 * sharp_UM_thresh;                                         //  0 to 10K (64K max. possible)

   pix1 = bmpixel(E1rgb16,px,py);                                          //  input pixel
   pix3 = bmpixel(E3rgb16,px,py);                                          //  output pixel

   bright = brightness(pix1);
   if (bright < 100) return;                                               //  effectively black
   mean = get_brhood(px,py);
   incr = (bright - mean);
   if (fabs(incr) < thresh) return;                                        //  omit low-contrast pixels

   incr = incr * amount;                                                   //  0.0 to 2.0
   if (bright + incr > 65535) incr = 65535 - bright;
   ratio = (bright + incr) / bright;

   for (rgb = 0; rgb < 3; rgb++)                                           //  loop 3 RGB colors
   {
      cval1 = pix1[rgb];
      cval3 = ratio * cval1;
      if (cval3 < 0) cval3 = 0;
      if (cval3 > 65535) cval3 = 65535;
      
      if (sa_active && dist < sa_blend) {                                  //  select area is active,
         f1 = 1.0 * dist / sa_blend;                                       //    blend changes over sa_blend
         f2 = 1.0 - f1;
         cval3 = f1 * cval3 + f2 * cval1;
      }
      
      pix3[rgb] = cval3;
   }

   progress_done++;   
   return;
}


//  sharpen image by increasing brightness gradient                        //  new v.9.8

int sharp_GR()
{
   uint16      *pix1, *pix3;
   int         ii, px, py, dist = 0, rgb;   
   double      amount, thresh;
   double      b1, b1x, b1y, b3x, b3y, b3, bf;
   double      pval, f1, f2;

   amount = 1 + 0.01 * sharp_GR_amount;                                    //  1.0 - 5.0
   thresh = 655.35 * sharp_GR_thresh;                                      //  0 - 64K

   if (sa_active) progress_goal = sa_Npixel;
   else  progress_goal = E3ww * E3hh;
   progress_done = 0;

   for (py = 1; py < E1hh; py++)                                           //  loop all image pixels
   for (px = 1; px < E1ww; px++)
   {
      if (sa_active) {                                                     //  select area active
         ii = py * Fww + px;
         dist = sa_pixisin[ii];                                            //  distance from edge
         if (! dist) continue;                                             //  pixel is outside area
      }

      pix1 = bmpixel(E1rgb16,px,py);                                       //  input pixel
      pix3 = bmpixel(E3rgb16,px,py);                                       //  output pixel

      b1 = brightness(pix1);                                               //  pixel brightness, 0 - 64K
      b1x = b1 - brightness(pix1 - 3);                                     //  brightness gradient (x,y)
      b1y = b1 - brightness(pix1 - 3 * E1ww);
      
      b1x = b1x * amount;                                                  //  amplified gradient
      b1y = b1y * amount;

      b3x = brightness(pix1 - 3) + b1x;                                    //  + prior pixel brightness
      b3y = brightness(pix1 - 3 * E3ww) + b1y;                             //  = new brightness

      b3 = 0.5 * (b3x + b3y);
      if (b1 < thresh) b3 = b1;

      bf = b3 / b1;
      if (bf < 0) bf = 0;
      if (bf > 4) bf = 4;
      
      for (rgb = 0; rgb < 3; rgb++)                                        //  loop 3 RGB colors
      {
         pval = bf * pix3[rgb];                                            //  apply factor
         if (pval > 65535) pval = 65535;
         
         if (sa_active && dist < sa_blend) {                               //  select area is active,
            f1 = 1.0 * dist / sa_blend;                                    //    blend changes over sa_blend
            f2 = 1.0 - f1;
            pval = int(f1 * pval + f2 * pix1[rgb]);
         }
         
         pix3[rgb] = pval;
      }

      progress_done++;
   }

   progress_goal = progress_done = 0;
   return 1;
}


/**************************************************************************/

//  image noise reduction

int      denoise_method = 5;                                               //  default algorithm
int      denoise_radius = 4;


void m_denoise(GtkWidget *, cchar *)
{
   int    denoise_dialog_event(zdialog *zd, cchar *event);                 //  dialog event function
   void * denoise_thread(void *);

   cchar  *denoise_message = ZTX(" Press the reduce button to \n"
                                 " reduce noise in small steps. \n"
                                 " Use undo to start over.");

   if (! edit_setup("noise",0,2)) return;                                  //  setup edit: no preview

   zdedit = zdialog_new(ZTX("Noise Reduction"),mWin,Bdone,Bcancel,null);
   zdialog_add_widget(zdedit,"label","lab1","dialog",denoise_message,"space=5");
   zdialog_add_widget(zdedit,"hbox","hb1","dialog",0,"space=5");
   zdialog_add_widget(zdedit,"label","labalg","hb1",ZTX("algorithm"),"space=5");
   zdialog_add_widget(zdedit,"combo","method","hb1",0,"space=5|expand");
   zdialog_add_widget(zdedit,"hbox","hb2","dialog",0,"space=5");
   zdialog_add_widget(zdedit,"label","labrad","hb2",Bradius,"space=5");
   zdialog_add_widget(zdedit,"spin","radius","hb2","1|9|1|4","space=5");
   zdialog_add_widget(zdedit,"button","reduce","hb2",Breduce,"space=5");
   
   zdialog_cb_app(zdedit,"method",ZTX("flatten outliers by color (1)"));
   zdialog_cb_app(zdedit,"method",ZTX("flatten outliers by color (2)"));
   zdialog_cb_app(zdedit,"method",ZTX("set median brightness by color"));
   zdialog_cb_app(zdedit,"method",ZTX("top hat filter by color"));
   zdialog_stuff(zdedit,"method",ZTX("top hat filter by color"));          //  default

   zdialog_run(zdedit,denoise_dialog_event);                               //  run dialog
   start_thread(denoise_thread,0);                                         //  start working thread
   return;
}


//  dialog event and completion callback function

int denoise_dialog_event(zdialog * zd, cchar *event)
{
   char     method[40];

   if (zd->zstat)
   {
      if (zd->zstat == 1) edit_done();
      else edit_cancel();
      return 0;
   }

   if (strEqu(event,"F1")) showz_userguide("reduce_noise");                //  F1 context help
   if (strEqu(event,"undo")) edit_undo();                                  //  v.10.2
   if (strEqu(event,"redo")) edit_redo();                                  //  v.10.3
   if (strEqu(event,"blendwidth")) signal_thread();                        //  trigger update thread

   if (strEqu(event,"radius")) 
      zdialog_fetch(zd,"radius",denoise_radius);

   if (strEqu(event,"method")) 
   {
      zdialog_fetch(zd,"method",method,39);

      if (strEqu(method,"flatten outliers by color (1)")) {
         denoise_method = 1;
         denoise_radius = 1;
      }

      if (strEqu(method,"flatten outliers by color (2)")) {
         denoise_method = 2;
         denoise_radius = 3;
      }

      if (strEqu(method,"set median brightness by color")) {
         denoise_method = 4;
         denoise_radius = 2;
      }

      if (strEqu(method,"top hat filter by color")) {
         denoise_method = 5;
         denoise_radius = 4;
      }
      
      zdialog_stuff(zd,"radius",denoise_radius);
   }
   
   if (strEqu(event,"reduce")) signal_thread();                            //  trigger update thread

   return 1;
}


//  image noise reduction thread

void * denoise_thread(void *)
{
   void * denoise_wthread(void *arg);

   while (true)
   {
      thread_idle_loop();                                                  //  wait for work or exit request

      E9rgb16 = RGB_copy(E3rgb16);                                         //  image3 is reference source
                                                                           //  image9 will be modified
      if (sa_active) progress_goal = sa_Npixel;
      else  progress_goal = E3ww * E3hh;
      progress_done = 0;

      for (int ii = 0; ii < Nwt; ii++)                                     //  start worker threads
         start_wt(denoise_wthread,&wtnx[ii]);
      wait_wts();                                                          //  wait for completion

      progress_goal = progress_done = 0;                                   //  v.9.6

      mutex_lock(&pixmaps_lock);
      RGB_free(E3rgb16);                                                   //  image9 >> image3
      E3rgb16 = E9rgb16;
      E9rgb16 = 0;
      mutex_unlock(&pixmaps_lock);

      Fmodified = 1;
      mwpaint2();                                                          //  update window
   }

   return 0;                                                               //  not executed, stop g++ warning
}


void * denoise_wthread(void *arg)                                          //  worker thread function   v.7.7
{
   void  denoise_func1(uint16 *pix3, uint16 *pix9);
   void  denoise_func2(uint16 *pix3, uint16 *pix9);
   void  denoise_func4(uint16 *pix3, uint16 *pix9);
   void  denoise_func5(uint16 *pix3, uint16 *pix9);
   
   int         index = *((int *) arg);
   int         ii, px, py, rad, dist = 0;
   double      f1, f2;
   uint16      *pix1, *pix3, *pix9;

   rad = denoise_radius;

   for (py = index+rad; py < E3hh-rad; py += Nwt)                          //  loop all image3 pixels
   for (px = rad; px < E3ww-rad; px++)
   {
      if (sa_active) {                                                     //  select area active
         ii = py * Fww + px;
         dist = sa_pixisin[ii];                                            //  distance from edge
         if (! dist) continue;                                             //  outside pixel
      }

      pix3 = bmpixel(E3rgb16,px,py);                                       //  source pixel
      pix9 = bmpixel(E9rgb16,px,py);                                       //  target pixel

      if (denoise_method == 1) denoise_func1(pix3,pix9);
      if (denoise_method == 2) denoise_func2(pix3,pix9);
      if (denoise_method == 4) denoise_func4(pix3,pix9);
      if (denoise_method == 5) denoise_func5(pix3,pix9);

      if (sa_active && dist < sa_blend) {                                  //  select area is active,
         f1 = 1.0 * dist / sa_blend;                                       //    blend changes over sa_blend
         f2 = 1.0 - f1;
         pix1 = bmpixel(E1rgb16,px,py);                                    //  source pixel
         pix9[0] = int(f1 * pix9[0] + f2 * pix1[0]);
         pix9[1] = int(f1 * pix9[1] + f2 * pix1[1]);
         pix9[2] = int(f1 * pix9[2] + f2 * pix1[2]);
      }

      progress_done++;
      if (Fkillfunc) break;                                                //  v.9.1
   }

   exit_wt();
   return 0;                                                               //  not executed, avoid gcc warning
}


//  flatten outliers within radius, by color 
//  an outlier is the max or min value within a radius

void denoise_func1(uint16 *pix3, uint16 *pix9)
{
   int         dy, dx, rad;
   int         min0, min1, min2, max0, max1, max2;
   uint16      *pixN;

   min0 = min1 = min2 = 65535;
   max0 = max1 = max2 = 0;
   rad = denoise_radius;

   for (dy = -rad; dy <= rad; dy++)                                        //  loop surrounding pixels
   for (dx = -rad; dx <= rad; dx++)
   {
      if (dy == 0 && dx == 0) continue;                                    //  skip self

      pixN = pix3 + (dy * E3ww + dx) * 3;
      if (pixN[0] < min0) min0 = pixN[0];                                  //  find min and max per color
      if (pixN[0] > max0) max0 = pixN[0];
      if (pixN[1] < min1) min1 = pixN[1];
      if (pixN[1] > max1) max1 = pixN[1];
      if (pixN[2] < min2) min2 = pixN[2];
      if (pixN[2] > max2) max2 = pixN[2];
   }
   
   if (pix3[0] <= min0 && min0 < 65279) pix9[0] = min0 + 256;              //  if outlier, flatten a little
   if (pix3[0] >= max0 && max0 > 256) pix9[0] = max0 - 256;
   if (pix3[1] <= min1 && min1 < 65279) pix9[1] = min1 + 256;
   if (pix3[1] >= max1 && max1 > 256) pix9[1] = max1 - 256;
   if (pix3[2] <= min2 && min2 < 65279) pix9[2] = min2 + 256;
   if (pix3[2] >= max2 && max2 > 256) pix9[2] = max2 - 256;
   
   return;
}


//  flatten outliers
//  An outlier pixel has an RGB value outside one sigma of 
//  the mean for all pixels within a given radius of the pixel.

void denoise_func2(uint16 *pix3, uint16 *pix9)                             //  v.8.5
{
   int         rgb, dy, dx, rad, nn;
   double      nn1, val, sum, sum2, mean, variance, sigma;
   uint16      *pixN;

   rad = denoise_radius;
   nn = (rad * 2 + 1);
   nn = nn * nn - 1;
   nn1 = 1.0 / nn;

   for (rgb = 0; rgb < 3; rgb++)                                           //  loop RGB color
   {
      sum = sum2 = 0;

      for (dy = -rad; dy <= rad; dy++)                                     //  loop surrounding pixels
      for (dx = -rad; dx <= rad; dx++)
      {
         if (dy == 0 && dx == 0) continue;                                 //  skip self
         pixN = pix3 + (dy * E3ww + dx) * 3;
         val = pixN[rgb];
         sum += val;
         sum2 += val * val;
      }
      
      mean = nn1 * sum;
      variance = nn1 * (sum2 - 2.0 * mean * sum) + mean * mean;
      sigma = sqrt(variance);

      val = pix3[rgb];      
      if (val > mean + sigma) {                                            //  move value to mean +/- sigma
         val = mean + sigma;                                               //  v.8.6
         pix9[rgb] = val;
      }
      else if (val < mean - sigma) {
         val = mean - sigma;
         pix9[rgb] = val;
      }
   }
   
   return;
}


//  use median brightness for pixels within radius

void denoise_func4(uint16 *pix3, uint16 *pix9)
{
   int         dy, dx, rad;
   int         ns, rgb, bsortN[400];
   uint16      *pixN;

   rad = denoise_radius;

   for (rgb = 0; rgb < 3; rgb++)                                           //  loop all RGB colors
   {
      ns = 0;

      for (dy = -rad; dy <= rad; dy++)                                     //  loop surrounding pixels
      for (dx = -rad; dx <= rad; dx++)                                     //  get brightness values
      {
         pixN = pix3 + (dy * E3ww + dx) * 3;
         bsortN[ns] = pixN[rgb];
         ns++;
      }

      HeapSort(bsortN,ns);
      pix9[rgb] = bsortN[ns/2];                                            //  median brightness of ns pixels
   }

   return;
}


//  modified top hat filter: execute with increasing radius from 1 to limit
//  detect outlier by comparing with pixels in outer radius

void denoise_func5(uint16 *pix3, uint16 *pix9)
{
   int         dy, dx, rad;
   int         min0, min1, min2, max0, max1, max2;
   uint16      *pixN;

   for (rad = 1; rad <= denoise_radius; rad++)
   for (int loops = 0; loops < 2; loops++)
   {
      min0 = min1 = min2 = 65535;
      max0 = max1 = max2 = 0;

      for (dy = -rad; dy <= rad; dy++)                                     //  loop all pixels within rad
      for (dx = -rad; dx <= rad; dx++)
      {
         if (dx > -rad && dx < rad) continue;                              //  skip inner pixels
         if (dy > -rad && dy < rad) continue;

         pixN = pix3 + (dy * E3ww + dx) * 3;
         if (pixN[0] < min0) min0 = pixN[0];                               //  find min and max per color
         if (pixN[0] > max0) max0 = pixN[0];                               //    among outermost pixels
         if (pixN[1] < min1) min1 = pixN[1];
         if (pixN[1] > max1) max1 = pixN[1];
         if (pixN[2] < min2) min2 = pixN[2];
         if (pixN[2] > max2) max2 = pixN[2];
      }
      
      if (pix3[0] < min0 && pix9[0] < 65279) pix9[0] += 256;               //  if central pixel is outlier,
      if (pix3[0] > max0 && pix9[0] > 256) pix9[0] -= 256;                 //    moderate its values
      if (pix3[1] < min1 && pix9[1] < 65279) pix9[1] += 256;
      if (pix3[1] > max1 && pix9[1] > 256) pix9[1] -= 256;
      if (pix3[2] < min2 && pix9[2] < 65279) pix9[2] += 256;
      if (pix3[2] > max2 && pix9[2] > 256) pix9[2] -= 256;
   }

   return;
}


/**************************************************************************/

//  trim image - use mouse to select image region to retain

int      trimx1, trimy1, trimx2, trimy2;                                   //  trim rectangle
int      trimww, trimhh;
double   trimR;																				//  trim aspect ratio
int      trim_status = 0;


void m_trim(GtkWidget *, cchar *)
{
   void   trim_mousefunc();
   int    trim_dialog_event(zdialog *zd, cchar *event);
   void * trim_thread(void *);

   cchar  *trim_message = ZTX("Drag middle to move, "
                              "drag corners to resize.");
   char        text[40];
   
   if (! edit_setup("trim",1,0)) return;                                   //  setup edit: use preview   v.8.4.1

   zdedit = zdialog_new(ZTX("Trim Image"),mWin,ZTX("Trim"),Bcancel,null);
   zdialog_add_widget(zdedit,"label","lab1","dialog",trim_message,"space=10");
   zdialog_add_widget(zdedit,"hbox","hb1","dialog");
   zdialog_add_widget(zdedit,"label","space","hb1",0,"space=4");
   zdialog_add_widget(zdedit,"label","labwhr","hb1","2345 x 1234  (R=1.90)");
   zdialog_add_widget(zdedit,"check","lock","hb1",ZTX("Lock Ratio"),"space=10");
   zdialog_add_widget(zdedit,"hbox","hb2","dialog",0,"space=5");
   zdialog_add_widget(zdedit,"button","R1:1","hb2"," 1:1 ");
   zdialog_add_widget(zdedit,"button","R2:1","hb2"," 2:1 ");
   zdialog_add_widget(zdedit,"button","R3:2","hb2"," 3:2 ");
   zdialog_add_widget(zdedit,"button","R4:3","hb2"," 4:3 ");
   zdialog_add_widget(zdedit,"button","R16:9","hb2","16:9 ");
   zdialog_add_widget(zdedit,"button","gold","hb2","Gold ");
   zdialog_add_widget(zdedit,"button","invert","hb2",ZTX("invert"));

   zdialog_run(zdedit,trim_dialog_event);                                  //  run dialog, parallel

   mouseCBfunc = trim_mousefunc;                                           //  connect mouse function
   Mcapture++;
   
   trimx1 = int(0.2 * E3ww);                                               //  start with 20% trim margins
   trimy1 = int(0.2 * E3hh);
   trimx2 = int(0.8 * E3ww);
   trimy2 = int(0.8 * E3hh);

   trimww = (trimx2 - trimx1);
   trimhh = (trimy2 - trimy1);
   trimR = 1.0 * trimww / trimhh;
   
   snprintf(text,39,"%d x %d  (R=%.2f)", trimww * Fww / E3ww,              //  stuff dialog poop 
                                    trimhh * Fhh / E3hh, trimR);           //    (final size)    v.8.4.1
   zdialog_stuff(zdedit,"labwhr",text);

   trim_status = 0;
   start_thread(trim_thread,0);                                            //  start working thread
   signal_thread();
   return;
}


//  dialog event and completion callback function

int trim_dialog_event(zdialog *zd, cchar *event)
{
   double   ratio = 0;
   char     text[40];
   
   if (zd->zstat)                                                          //  dialog complete
   {
      mouseCBfunc = 0;                                                     //  disconnect mouse
      Mcapture = 0;
      
      gridx1 = gridx2 = gridy1 = gridy2 = 0;                               //  remove grid margins   v.10.3.1
      
      if (zd->zstat != 1) {
         edit_cancel();
         return 0;
      }
      
      trimx1 = trimx1 * Fww / E3ww;                                        //  scale from preview size
      trimy1 = trimy1 * Fhh / E3hh;                                        //    to final size    v.8.4.1
      trimx2 = trimx2 * Fww / E3ww;
      trimy2 = trimy2 * Fhh / E3hh;
      trimww = trimx2 - trimx1;
      trimhh = trimy2 - trimy1;

      trim_status = 1;                                                     //  trim full image
      Fmodified = 1;
      edit_done();
      return 0;
   }

   if (strEqu(event,"F1")) {
      showz_userguide("crop");                                             //  F1 help       v.9.0
      return 0;
   }
   
   if (strEqu(event,"R1:1")) ratio = 1.0;                                  //  preset ratios   v.10.1
   if (strEqu(event,"R2:1")) ratio = 2.0;
   if (strEqu(event,"R3:2")) ratio = 1.5;
   if (strEqu(event,"R4:3")) ratio = 1.333;
   if (strEqu(event,"R16:9")) ratio = 1.778;
   if (strEqu(event,"gold")) ratio = 1.618;
   
   if (ratio) zdialog_stuff(zd,"lock",1);                                  //  assume lock is wanted

   if (strEqu(event,"invert")) ratio = 1.0 / trimR;                        //  v.10.3.1
   
   if (ratio) 
   {
      trimR = ratio;
      
      if (trimx2 - trimx1 > trimy2 - trimy1)
         trimy2 = trimy1 + (trimx2 - trimx1) / trimR;                      //  adjust smaller dimension  v.10.3.1
      else 
         trimx2 = trimx1 + (trimy2 - trimy1) * trimR;

      if (trimy2 >= E3hh) {                                                //  if off the bottom edge,
         trimy2 = E3hh-1;                                                  //  adjust width
         trimx2 = trimx1 + (trimy2 - trimy1) * trimR;
      }

      if (trimx2 >= E3ww) {                                                //  if off the right edge,
         trimx2 = E3ww-1;                                                  //  adjust height
         trimy2 = trimy1 + (trimx2 - trimx1) / trimR;
      }

      trimww = trimx2 - trimx1;                                            //  new rectangle dimensions
      trimhh = trimy2 - trimy1;

      snprintf(text,39,"%d x %d  (R=%.2f)", trimww * Fww / E3ww,           //  stuff dialog poop 
                                    trimhh * Fhh / E3hh, trimR);           //    (final size)
      zdialog_stuff(zdedit,"labwhr",text);

      signal_thread();
   }

   return 0;
}


//  trim mouse function                                                    //  overhauled  v.8.4.1

void trim_mousefunc()
{
   int         mpx, mpy, xdrag, ydrag, rlock;
   int         corner, chop, moveall = 0;
   int         dx, dy, dd, d1, d2, d3, d4;
   char        text[40];
   double      drr;
   
   if (LMclick || Mxdrag || Mydrag)                                        //  mouse click or drag
   {
      if (LMclick) {
         mpx = Mxclick;                                                    //  click
         mpy = Myclick;
         xdrag = ydrag = 0;
         LMclick = 0;
      }
      else {
         mpx = Mxdrag;                                                     //  drag
         mpy = Mydrag;
         xdrag = Mxdrag - Mxdown;
         ydrag = Mydrag - Mydown;
         Mxdown = Mxdrag;                                                  //  reset drag origin
         Mydown = Mydrag;
      }
      
      if (Mxdrag || Mydrag) {
         moveall = 1;
         dd = 0.1 * (trimx2 - trimx1);                                     //  test if mouse is in the broad
         if (mpx < trimx1 + dd) moveall = 0;                               //    middle of the rectangle
         if (mpx > trimx2 - dd) moveall = 0;
         dd = 0.1 * (trimy2 - trimy1);
         if (mpy < trimy1 + dd) moveall = 0;
         if (mpy > trimy2 - dd) moveall = 0;
      }

      if (moveall) {                                                       //  yes, move the whole rectangle
         trimx1 += xdrag;
         trimx2 += xdrag;
         trimy1 += ydrag;
         trimy2 += ydrag;
         corner = 0;
      }

      else {                                                               //  no, find closest corner
         dx = mpx - trimx1;
         dy = mpy - trimy1;
         d1 = sqrt(dx*dx + dy*dy);
         
         dx = mpx - trimx2;
         dy = mpy - trimy1;
         d2 = sqrt(dx*dx + dy*dy);
         
         dx = mpx - trimx2;
         dy = mpy - trimy2;
         d3 = sqrt(dx*dx + dy*dy);
         
         dx = mpx - trimx1;
         dy = mpy - trimy2;
         d4 = sqrt(dx*dx + dy*dy);
         
         corner = 1;                                                       //  NW
         dd = d1;
         if (d2 < dd) { corner = 2; dd = d2; }                             //  NE
         if (d3 < dd) { corner = 3; dd = d3; }                             //  SE
         if (d4 < dd) { corner = 4; dd = d4; }                             //  SW
         
         if (corner == 1) { trimx1 = mpx; trimy1 = mpy; }                  //  move this corner to mouse
         if (corner == 2) { trimx2 = mpx; trimy1 = mpy; }
         if (corner == 3) { trimx2 = mpx; trimy2 = mpy; }
         if (corner == 4) { trimx1 = mpx; trimy2 = mpy; }
      }

      if (trimx1 > trimx2-10) trimx1 = trimx2-10;                          //  sanity limits
      if (trimy1 > trimy2-10) trimy1 = trimy2-10;

      zdialog_fetch(zdedit,"lock",rlock);                                  //  w/h ratio locked
      if (rlock && corner) {
         if (corner < 3)                                                   //  bugfix   v.10.3.1
            trimy2 = trimy1 + 1.0 * (trimx2 - trimx1) / trimR;
         else
            trimy1 = trimy2 - 1.0 * (trimx2 - trimx1) / trimR;
      }

      chop = 0;
      if (trimx1 < 0) { trimx1 = 0; chop = 1; }                            //  look for off the edge   v.10.1
      if (trimx2 > E3ww) { trimx2 = E3ww; chop = 2; }                      //  bugfix, after corner move    v.10.3.1
      if (trimy1 < 0) { trimy1 = 0; chop = 3; }
      if (trimy2 > E3hh) { trimy2 = E3hh; chop = 4; }

      if (rlock && chop) {                                                 //  keep ratio if off the edge   v.10.1
         if (chop < 3)
            trimy2 = trimy1 + 1.0 * (trimx2 - trimx1) / trimR;
         else
            trimx2 = trimx1 + 1.0 * (trimy2 - trimy1) * trimR;
      }

      if (trimx1 > trimx2-10) trimx1 = trimx2-10;                          //  sanity limits
      if (trimy1 > trimy2-10) trimy1 = trimy2-10;

      if (trimx1 < 0) trimx1 = 0;
      if (trimy1 < 0) trimy1 = 0;
      if (trimx2 > E3ww) trimx2 = E3ww;
      if (trimy2 > E3hh) trimy2 = E3hh;

      trimww = trimx2 - trimx1;                                            //  new rectangle dimensions
      trimhh = trimy2 - trimy1;

      drr = 1.0 * trimww / trimhh;                                         //  new w/h ratio
      if (! rlock) trimR = drr;

      snprintf(text,39,"%d x %d  (R=%.2f)", trimww * Fww / E3ww,           //  stuff dialog poop 
                                    trimhh * Fhh / E3hh, drr);             //    (final size)
      zdialog_stuff(zdedit,"labwhr",text);
      
      signal_thread();                                                     //  trigger update thread
   }

   return;
}


//  trim thread function

void * trim_thread(void *)
{
   int      px1, py1, px2, py2;
   uint16   *pix1, *pix3;
   
   while (true)
   {
      thread_idle_loop();                                                  //  wait for work or exit request
      
      if (trim_status == 0)                                                //  darken margins   v.8.4
      {
         for (py1 = 0; py1 < E3hh; py1++)                                  //  copy pixels E1 >> E3
         for (px1 = 0; px1 < E3ww; px1++)
         {
            pix1 = bmpixel(E1rgb16,px1,py1);
            pix3 = bmpixel(E3rgb16,px1,py1);

            if (px1 < trimx1 || px1 > trimx2 || py1 < trimy1 || py1 > trimy2)
            {
               pix3[0] = pix1[0] / 2;
               pix3[1] = pix1[1] / 2;
               pix3[2] = pix1[2] / 2;
            }

            else
            {
               pix3[0] = pix1[0];
               pix3[1] = pix1[1];
               pix3[2] = pix1[2];
            }
         }
         
         gridx1 = trimx1;                                                  //  set grid margins     v.10.3.1
         gridx2 = trimx2;
         gridy1 = trimy1;
         gridy2 = trimy2;

         mwpaint2();                                                       //  update window
      }
      
      if (trim_status == 1)                                                //  do the trim
      {
         mutex_lock(&pixmaps_lock);
         RGB_free(E3rgb16);
         E3rgb16 = RGB_make(trimww,trimhh,16);                             //  new pixmap with requested size
         E3ww = trimww;
         E3hh = trimhh;
         
         for (py1 = trimy1; py1 < trimy2; py1++)                           //  copy pixels
         for (px1 = trimx1; px1 < trimx2; px1++)
         {
            px2 = px1 - trimx1;
            py2 = py1 - trimy1;
            pix1 = bmpixel(E1rgb16,px1,py1);
            pix3 = bmpixel(E3rgb16,px2,py2);
            pix3[0] = pix1[0];
            pix3[1] = pix1[1];
            pix3[2] = pix1[2];
         }

         Fmodified = 1;
         mutex_unlock(&pixmaps_lock);
         mwpaint2();                                                       //  update window
      }
   }

   return 0;                                                               //  not executed, stop g++ warning
}


/**************************************************************************/

//  Resize (rescale) image
//
//  Output pixels are composites of input pixels, e.g. 2/3 size means 
//  that 3x3 input pixels are mapped into 2x2 output pixels, and an 
//  image size of 1000 x 600 becomes 667 x 400.


int      resize_ww0, resize_hh0, resize_ww1, resize_hh1;


void m_resize(GtkWidget *, cchar *)
{
   int      resize_dialog_event(zdialog *zd, cchar *event);
   void   * resize_thread(void *);

   cchar  *lockmess = ZTX("Lock aspect ratio");

   if (! edit_setup("resize",0,0)) return;                                 //  setup edit: no preview

   zdedit = zdialog_new(ZTX("Resize Image"),mWin,Bapply,Bcancel,null);
   zdialog_add_widget(zdedit,"hbox","hb1","dialog",0,"space=10");
   zdialog_add_widget(zdedit,"vbox","vb11","hb1",0,"homog|space=5");
   zdialog_add_widget(zdedit,"vbox","vb12","hb1",0,"homog|space=5");
   zdialog_add_widget(zdedit,"vbox","vb13","hb1",0,"homog|space=5");
   zdialog_add_widget(zdedit,"label","placeholder","vb11",0);              //             pixels       percent
   zdialog_add_widget(zdedit,"label","labw","vb11",Bwidth);                //    width    [______]     [______]
   zdialog_add_widget(zdedit,"label","labh","vb11",Bheight);               //    height   [______]     [______]
   zdialog_add_widget(zdedit,"label","labpix","vb12","pixels");            //
   zdialog_add_widget(zdedit,"spin","wpix","vb12","20|9999|1|0");          //    presets  [2/3] [1/2] [1/3] [1/4] 
   zdialog_add_widget(zdedit,"spin","hpix","vb12","20|9999|1|0");          //
   zdialog_add_widget(zdedit,"label","labpct","vb13",Bpercent);            //    [_] lock width/height ratio
   zdialog_add_widget(zdedit,"spin","wpct","vb13","1|500|0.1|100");        //
   zdialog_add_widget(zdedit,"spin","hpct","vb13","1|500|0.1|100");        //       [  done  ]  [ cancel ]  
   zdialog_add_widget(zdedit,"hbox","hb2","dialog",0,"space=5");
   zdialog_add_widget(zdedit,"label","preset","hb2",Bpresets,"space=5");
   zdialog_add_widget(zdedit,"button","b 3/4","hb2"," 3/4 ");
   zdialog_add_widget(zdedit,"button","b 2/3","hb2"," 2/3 ");
   zdialog_add_widget(zdedit,"button","b 1/2","hb2"," 1/2 ");
   zdialog_add_widget(zdedit,"button","b 1/3","hb2"," 1/3 ");
   zdialog_add_widget(zdedit,"button","b 1/4","hb2"," 1/4 ");
   zdialog_add_widget(zdedit,"check","lock","dialog",lockmess);

   resize_ww0 = Frgb16->ww;                                                //  original width, height
   resize_hh0 = Frgb16->hh;
   zdialog_stuff(zdedit,"wpix",resize_ww0);
   zdialog_stuff(zdedit,"hpix",resize_hh0);
   zdialog_stuff(zdedit,"lock",1);
   
   zdialog_run(zdedit,resize_dialog_event);                                //  run dialog - parallel
   start_thread(resize_thread,0);                                          //  start working thread
   return;
}


//  dialog event and completion callback function

int resize_dialog_event(zdialog *zd, cchar * event)
{
   int         lock;
   double      wpct1, hpct1;
   
   if (zd->zstat)                                                          //  dialog complete
   {
      if (zd->zstat == 1) {
         signal_thread();
         edit_done();
      }
      else edit_cancel();
      return 0;
   }

   if (strEqu(event,"F1")) showz_userguide("resize");                      //  F1 help       v.9.0

   zdialog_fetch(zd,"wpix",resize_ww1);                                    //  get all widget values
   zdialog_fetch(zd,"hpix",resize_hh1);
   zdialog_fetch(zd,"wpct",wpct1);
   zdialog_fetch(zd,"hpct",hpct1);
   zdialog_fetch(zd,"lock",lock);
   
   if (strEqu(event,"b 3/4")) {
      resize_ww1 = (3 * resize_ww0 + 3) / 4;
      resize_hh1 = (3 * resize_hh0 + 3) / 4;
   }
   
   if (strEqu(event,"b 2/3")) {
      resize_ww1 = (2 * resize_ww0 + 2) / 3;
      resize_hh1 = (2 * resize_hh0 + 2) / 3;
   }
   
   if (strEqu(event,"b 1/2")) {
      resize_ww1 = (resize_ww0 + 1) / 2;
      resize_hh1 = (resize_hh0 + 1) / 2;
   }
   
   if (strEqu(event,"b 1/3")) {
      resize_ww1 = (resize_ww0 + 2) / 3;
      resize_hh1 = (resize_hh0 + 2) / 3;
   }
   
   if (strEqu(event,"b 1/4")) {
      resize_ww1 = (resize_ww0 + 3) / 4;
      resize_hh1 = (resize_hh0 + 3) / 4;
   }

   if (strEqu(event,"wpct"))                                               //  width % - set pixel width
      resize_ww1 = int(wpct1 / 100.0 * resize_ww0 + 0.5);

   if (strEqu(event,"hpct"))                                               //  height % - set pixel height
      resize_hh1 = int(hpct1 / 100.0 * resize_hh0 + 0.5);
   
   if (lock && event[0] == 'w')                                            //  preserve width/height ratio
      resize_hh1 = int(resize_ww1 * (1.0 * resize_hh0 / resize_ww0) + 0.5);
   if (lock && event[0] == 'h') 
      resize_ww1 = int(resize_hh1 * (1.0 * resize_ww0 / resize_hh0) + 0.5);
   
   hpct1 = 100.0 * resize_hh1 / resize_hh0;                                //  set percents to match pixels
   wpct1 = 100.0 * resize_ww1 / resize_ww0;
   
   zdialog_stuff(zd,"wpix",resize_ww1);                                    //  index all widget values
   zdialog_stuff(zd,"hpix",resize_hh1);
   zdialog_stuff(zd,"wpct",wpct1);
   zdialog_stuff(zd,"hpct",hpct1);
   
   return 1;
}


//  resize image based on dialog controls.

void * resize_thread(void *)
{
   while (true)
   {
      thread_idle_loop();                                                  //  wait for work or exit request

      if (resize_ww1 != resize_ww0 || resize_hh1 != resize_hh0) 
      {                                                                    //  rescale to target size
         mutex_lock(&pixmaps_lock);
         RGB_free(E3rgb16);
         E3rgb16 = RGB_rescale(Frgb16,resize_ww1,resize_hh1);
         E3ww = resize_ww1;
         E3hh = resize_hh1;
         Fmodified = 1;
         mutex_unlock(&pixmaps_lock);
      }

      mwpaint2();                                                          //  update window
   }

   return 0;                                                               //  not executed, stop g++ warning
}


/**************************************************************************/

//  rotate image through any arbitrary angle

double      rotate_angle = 0;                                              //  E3 rotatation vs. F
double      rotate_delta = 0;
int         rotate_trim = 0;


void m_rotate(GtkWidget *, cchar *menu)                                    //  menu function
{
   int    rotate_dialog_event(zdialog *zd, cchar *event);
   void * rotate_thread(void *);
   void   rotate_mousefunc();
   
   cchar  *rotmess = ZTX("Use buttons or drag right edge with mouse");

   if (! edit_setup("rotate",1,0)) return;                                 //  setup edit: use preview   v.6.2

   zdedit = zdialog_new(ZTX("Rotate Image"),mWin,Bdone,Bcancel,null);
   zdialog_add_widget(zdedit,"label","labrot","dialog",ZTX(rotmess),"space=5");
   zdialog_add_widget(zdedit,"label","labdeg","dialog",ZTX("degrees"),"space=5");
   zdialog_add_widget(zdedit,"hbox","hb1","dialog",0,"homog|space=5");
   zdialog_add_widget(zdedit,"vbox","vb1","hb1",0,"space=5");
   zdialog_add_widget(zdedit,"vbox","vb2","hb1",0,"space=5");
   zdialog_add_widget(zdedit,"vbox","vb3","hb1",0,"space=5");
   zdialog_add_widget(zdedit,"vbox","vb4","hb1",0,"space=5");
   zdialog_add_widget(zdedit,"button"," +0.1  ","vb1"," + 0.1 ");          //  button name is increment to use
   zdialog_add_widget(zdedit,"button"," -0.1  ","vb1"," - 0.1 ");
   zdialog_add_widget(zdedit,"button"," +1.0  ","vb2"," + 1   ");
   zdialog_add_widget(zdedit,"button"," -1.0  ","vb2"," - 1   ");
   zdialog_add_widget(zdedit,"button"," +10.0 ","vb3"," + 10  ");
   zdialog_add_widget(zdedit,"button"," -10.0 ","vb3"," - 10  ");
   zdialog_add_widget(zdedit,"button"," +90.0 ","vb4"," + 90  ");
   zdialog_add_widget(zdedit,"button"," -90.0 ","vb4"," - 90  ");
   zdialog_add_widget(zdedit,"hbox","hb2","dialog",0,"space=10");
   zdialog_add_widget(zdedit,"button","trim","hb2",ZTX("Trim"),"space=10");

   zdialog_run(zdedit,rotate_dialog_event);                                //  run dialog - parallel

   mouseCBfunc = rotate_mousefunc;                                         //  connect mouse function
   Mcapture++;
   set_cursor(dragcursor);                                                 //  set drag cursor

   rotate_angle = rotate_delta = rotate_trim = 0;
   start_thread(rotate_thread,0);                                          //  start working thread
   return;
}


//  dialog event and completion callback function

int rotate_dialog_event(zdialog *zd, cchar * event)
{
   int         err, trim = 0;
   double      incr;
   char        text[20];
   
   if (zd->zstat)                                                          //  dialog complete
   {
      if (zd->zstat == 1) {
         rotate_delta = rotate_angle;                                      //  rotate main image   v.6.2
         rotate_angle = 0;
         edit_done();
      }
      else edit_cancel();

      rotate_angle = rotate_delta = rotate_trim = 0;
      mouseCBfunc = 0;                                                     //  disconnect mouse
      Mcapture = 0;
      set_cursor(0);                                                       //  restore normal cursor
      return 0;
   }
   
   if (strEqu(event,"F1")) showz_userguide("rotate");                      //  F1 context help
   if (strEqu(event,"undo")) edit_undo();                                  //  v.10.2
   if (strEqu(event,"redo")) edit_redo();                                  //  v.10.3

   if (strEqu(event,"trim")) {
      rotate_trim = 1 - rotate_trim;                                       //  toggle trim button   v.8.3
      if (rotate_trim) zdialog_stuff(zd,"trim",ZTX("Undo Trim"));
      else zdialog_stuff(zd,"trim",ZTX("Trim"));
      trim = 1;                                                            //  v.10.3
   }
   
   if (strpbrk(event,"+-")) {
      err = convSD(event,incr);                                            //  button name is increment to use
      if (err) return 0;
      rotate_delta += incr;
   }

   if (rotate_delta || trim) {
      trim = 0;
      zdialog_stuff(zd,"labdeg","computing");
      signal_thread();                                                     //  do rotate in thread
      wait_thread_idle();
      sprintf(text,ZTX("degrees: %.1f"),rotate_angle);                     //  update dialog angle display
      zdialog_stuff(zd,"labdeg",text);
   }

   return 1;
}


//  rotate mouse function - drag right edge of image up/down for rotation

void rotate_mousefunc()
{
   static int     mpx0 = 0, mpy0 = 0;
   static int     mpy1, mpy2, dist;

   if (! Mxdrag && ! Mydrag) return;                                       //  no drag underway
   if (Mxdrag < 0.8 * E3ww) return;                                        //  not right edge of image

   if (Mxdown != mpx0 || Mydown != mpy0) {
      mpx0 = Mxdown;                                                       //  new drag started
      mpy0 = mpy1 = Mydown;
   }
   
   mpy2 = Mydrag;
   dist = mpy2 - mpy1;                                                     //  drag distance
   mpy1 = mpy2;                                                            //  reset origin for next time
   if (! dist) return;

   rotate_delta = 30.0 * dist / E3ww;                                      //  convert to angle
   rotate_dialog_event(zdedit,"mouse");
   return;
}


//  rotate thread function

void * rotate_thread(void *)
{
   int         px3, py3, px9, py9;
   int         wwcut, hhcut, ww, hh;
   double      trim_angle, radians;
   uint16      *pix3, *pix9;
   
   while (true)
   {
      thread_idle_loop();                                                  //  wait for work or exit request

      mutex_lock(&pixmaps_lock);

      rotate_angle += rotate_delta;                                        //  accum. net rotation   
      rotate_delta = 0;                                                    //    from dialog widget
      
      if (rotate_angle >= 360) rotate_angle -=360;
      if (rotate_angle <= -360) rotate_angle +=360;
      if (fabs(rotate_angle) < 0.01) rotate_angle = 0;

      if (! rotate_angle) {
         RGB_free(E3rgb16);                                                //  E1 >> E3
         E3rgb16 = RGB_copy(E1rgb16);
         E3ww = E1ww;
         E3hh = E1hh;
         Fmodified = 0;
      }
      
      if (rotate_angle) {
         RGB_free(E3rgb16);
         E3rgb16 = RGB_rotate(E1rgb16,rotate_angle);                       //  E3 is rotated E1
         E3ww = E3rgb16->ww;
         E3hh = E3rgb16->hh;
         Fmodified = 1;
      }

      if (rotate_trim)
      {                                                                    //  auto trim      no reset v.8.3
         trim_angle = fabs(rotate_angle);
         while (trim_angle > 45) trim_angle -= 90;
         radians = fabs(trim_angle / 57.296);
         wwcut = int(E3rgb16->hh * sin(radians) + 1);                      //  amount to trim
         hhcut = int(E3rgb16->ww * sin(radians) + 1);
         ww = E3rgb16->ww - 2 * wwcut;
         hh = E3rgb16->hh - 2 * hhcut;
         if (ww > 0 && hh > 0) {
            E9rgb16 = RGB_make(ww,hh,16);
            
            for (py3 = hhcut; py3 < E3hh-hhcut; py3++)                     //  E9 = trimmed E3
            for (px3 = wwcut; px3 < E3ww-wwcut; px3++)
            {
               px9 = px3 - wwcut;
               py9 = py3 - hhcut;
               pix3 = bmpixel(E3rgb16,px3,py3);
               pix9 = bmpixel(E9rgb16,px9,py9);
               pix9[0] = pix3[0];
               pix9[1] = pix3[1];
               pix9[2] = pix3[2];
            }

            RGB_free(E3rgb16);                                             //  E3 = E9
            E3rgb16 = E9rgb16;
            E9rgb16 = 0;
            E3ww = ww;
            E3hh = hh;
         }
      }
      
      mutex_unlock(&pixmaps_lock);
      mwpaint2();                                                          //  update window
   }

   return 0;                                                               //  not executed, stop g++ warning
}


/**************************************************************************/

//  flip an image horizontally or vertically

void m_flip(GtkWidget *, cchar *)                                          //  v.8.8
{
   int flip_dialog_event(zdialog *zd, cchar *event);

   if (! edit_setup("flip",0,0)) return;                                   //  setup edit: no preview

   zdedit = zdialog_new(ZTX("Flip Image"),mWin,Bdone,Bcancel,null);
   zdialog_add_widget(zdedit,"hbox","hb1","dialog",0,"space=10");
   zdialog_add_widget(zdedit,"button","horz","hb1",ZTX("horizontal"),"space=5");
   zdialog_add_widget(zdedit,"button","vert","hb1",ZTX("vertical"),"space=5");

   zdialog_run(zdedit,flip_dialog_event);                                  //  run dialog - parallel
   return;
}


//  dialog event and completion callback function

int flip_dialog_event(zdialog *zd, cchar *event)
{
   int flip_horz();
   int flip_vert();

   if (zd->zstat)                                                          //  dialog complete
   {
      if (zd->zstat == 1) edit_done();
      else edit_cancel();
      return 0;
   }

   if (strEqu(event,"F1")) showz_userguide("flip_image");                  //  F1 help       v.9.0
   if (strEqu(event,"undo")) edit_undo();                                  //  v.10.2
   if (strEqu(event,"redo")) edit_redo();                                  //  v.10.3
   if (strEqu(event,"horz")) flip_horz();
   if (strEqu(event,"vert")) flip_vert();
   return 0;
}


int flip_horz()
{
   int      px, py;
   uint16   *pix3, *pix9;
   
   edit_zapredo();                                                         //  delete redo copy    v.10.3

   E9rgb16 = RGB_copy(E3rgb16);
   
   for (py = 0; py < E3hh; py++)
   for (px = 0; px < E3ww; px++)
   {
      pix3 = bmpixel(E3rgb16,px,py);                                       //  image9 = flipped image3
      pix9 = bmpixel(E9rgb16,E3ww-1-px,py);
      pix9[0] = pix3[0];
      pix9[1] = pix3[1];
      pix9[2] = pix3[2];
   }
   
   mutex_lock(&pixmaps_lock);
   RGB_free(E3rgb16);                                                      //  image9 >> image3
   E3rgb16 = E9rgb16;
   E9rgb16 = 0;
   mutex_unlock(&pixmaps_lock);

   Fmodified = 1;
   mwpaint2();
   return 0;
}


int flip_vert()
{
   int      px, py;
   uint16   *pix3, *pix9;
   
   edit_zapredo();                                                         //  delete redo copy    v.10.3

   E9rgb16 = RGB_copy(E3rgb16);
   
   for (py = 0; py < E3hh; py++)
   for (px = 0; px < E3ww; px++)
   {
      pix3 = bmpixel(E3rgb16,px,py);                                       //  image9 = flipped image3
      pix9 = bmpixel(E9rgb16,px,E3hh-1-py);
      pix9[0] = pix3[0];
      pix9[1] = pix3[1];
      pix9[2] = pix3[2];
   }
   
   mutex_lock(&pixmaps_lock);
   RGB_free(E3rgb16);                                                      //  image9 >> image3
   E3rgb16 = E9rgb16;
   E9rgb16 = 0;
   mutex_unlock(&pixmaps_lock);

   Fmodified = 1;
   mwpaint2();
   return 0;
}


/**************************************************************************/

//  unbend an image
//  straighten curvature added by pano or improve perspective

int      unbend_horz, unbend_vert;                                         //  unbend values from dialog
int      unbend_vert2, unbend_horz2;
double   unbend_x1, unbend_x2, unbend_y1, unbend_y2;                       //  unbend axes scaled 0 to 1
int      unbend_hx1, unbend_hy1, unbend_hx2, unbend_hy2;
int      unbend_vx1, unbend_vy1, unbend_vx2, unbend_vy2;


void m_unbend(GtkWidget *, cchar *)                                        //  overhaul for preview   v.6.2
{
   int    unbend_dialog_event(zdialog* zd, cchar *event);
   void * unbend_thread(void *);
   void   unbend_mousefunc();

   if (! edit_setup("unbend",1,0)) return;                                 //  setup edit: preview

   zdedit = zdialog_new(ZTX("Unbend Image"),mWin,Bdone,Bcancel,null);
   zdialog_add_widget(zdedit,"hbox","hb1","dialog",0,"space=10");
   zdialog_add_widget(zdedit,"vbox","vb1","hb1",0,"homog|space=10");
   zdialog_add_widget(zdedit,"vbox","vb2","hb1",0,"homog|space=10");
   zdialog_add_widget(zdedit,"spin","spvert","vb1","-30|30|1|0");
   zdialog_add_widget(zdedit,"spin","sphorz","vb1","-20|20|1|0");
   zdialog_add_widget(zdedit,"label","labvert","vb2",ZTX("vertical unbend"));
   zdialog_add_widget(zdedit,"label","labhorz","vb2",ZTX("horizontal unbend"));
   
   zdialog_resize(zdedit,260,0);
   zdialog_run(zdedit,unbend_dialog_event);                                //  run dialog, parallel

   unbend_x1 = unbend_x2 = unbend_y1 = unbend_y2 = 0.5;                    //  initial axes thru image middle
   unbend_horz = unbend_vert = 0;                                          //  v.6.3

   Mcapture = 1;
   mouseCBfunc = unbend_mousefunc;                                         //  connect mouse function
   
   start_thread(unbend_thread,0);                                          //  start working thread
   signal_thread();

   return;
}


//  dialog event and completion callback function

int unbend_dialog_event(zdialog *zd, cchar *event)
{
   if (zd->zstat)                                                          //  dialog complete
   {
      paint_toplines(2);                                                   //  erase axes-lines       v.8.4.3
      mouseCBfunc = 0;                                                     //  disconnect mouse
      Mcapture = 0;

      if (zd->zstat != 1) {
         edit_cancel();                                                    //  canceled
         return 0;
      }

      if (unbend_vert || unbend_horz) Fmodified = 1;                       //  image3 modified
      else Fmodified = 0;
      edit_done();                                                         //  commit changes to image3
      return 1;
   }
   
   if (strEqu(event,"F1")) showz_userguide("unbend");                      //  F1 context help
   if (strEqu(event,"undo")) edit_undo();                                  //  v.10.2
   if (strEqu(event,"redo")) edit_redo();                                  //  v.10.3

   if (strstr(event,"sp")) {
      zdialog_fetch(zd,"spvert",unbend_vert);                              //  get new unbend values
      zdialog_fetch(zd,"sphorz",unbend_horz);
      signal_thread();                                                     //  trigger thread
   }

   return 1;
}


//  unbend mouse function                                                  //  adjustable axes

void unbend_mousefunc()   
{
   cchar       *close;
   double      dist1, dist2;
   double      mpx = 0, mpy = 0;
   
   if (LMclick) {                                                          //  left mouse click   v.7.5
      LMclick = 0;
      mpx = Mxclick;
      mpy = Myclick;
   }
   
   if (Mxdrag || Mydrag) {                                                 //  mouse dragged
      mpx = Mxdrag;
      mpy = Mydrag;
   }
   
   if (! mpx && ! mpy) return;

   mpx = 1.0 * mpx / E3ww;                                                 //  scale mouse position 0 to 1
   mpy = 1.0 * mpy / E3hh;

   if (mpx < 0.2 || mpx > 0.8 ) {                                          //  check reasonable position
      if (mpy < 0.1 || mpy > 0.9) return;
   }
   else if (mpy < 0.2 || mpy > 0.8) {
      if (mpx < 0.1 || mpx > 0.9) return;
   }
   else return;

   close = "?";                                                            //  find closest axis end-point
   dist1 = 2;

   dist2 = mpx * mpx + (mpy-unbend_y1) * (mpy-unbend_y1);
   if (dist2 < dist1) {
      dist1 = dist2;
      close = "left";
   }

   dist2 = (1-mpx) * (1-mpx) + (mpy-unbend_y2) * (mpy-unbend_y2);
   if (dist2 < dist1) {
      dist1 = dist2;
      close = "right";
   }

   dist2 = (mpx-unbend_x1) * (mpx-unbend_x1) + mpy * mpy;
   if (dist2 < dist1) {
      dist1 = dist2;
      close = "top";
   }

   dist2 = (mpx-unbend_x2) * (mpx-unbend_x2) + (1-mpy) * (1-mpy);
   if (dist2 < dist1) {
      dist1 = dist2;
      close = "bottom";
   }
   
   if (strEqu(close,"left")) unbend_y1 = mpy;                              //  set new axis end-point
   if (strEqu(close,"right")) unbend_y2 = mpy;
   if (strEqu(close,"top")) unbend_x1 = mpx;
   if (strEqu(close,"bottom")) unbend_x2 = mpx;

   signal_thread();                                                        //  trigger thread 

   return ;
}


//  unbend thread function

void * unbend_thread(void *arg)
{
   void * unbend_wthread(void *);

   while (true)
   {
      thread_idle_loop();                                                  //  wait for work or exit request

      unbend_vert2 = int(unbend_vert * 0.01 * E3hh);                       //  convert % to pixels
      unbend_horz2 = int(unbend_horz * 0.005 * E3ww);

      unbend_hx1 = 0;                                                      //  scale axes to E3ww/hh
      unbend_hy1 = unbend_y1 * E3hh;
      unbend_hx2 = E3ww;
      unbend_hy2 = unbend_y2 * E3hh;

      unbend_vx1 = unbend_x1 * E3ww;
      unbend_vy1 = 0;
      unbend_vx2 = unbend_x2 * E3ww;
      unbend_vy2 = E3hh;

      if (Fpreview) {                                                      //  omit for final unbend   v.8.4.3
         Ntoplines = 2;
         toplinex1[0] = unbend_hx1;                                        //  lines on window
         topliney1[0] = unbend_hy1;
         toplinex2[0] = unbend_hx2;
         topliney2[0] = unbend_hy2;
         toplinex1[1] = unbend_vx1;
         topliney1[1] = unbend_vy1;
         toplinex2[1] = unbend_vx2;
         topliney2[1] = unbend_vy2;
      }
      
      for (int ii = 0; ii < Nwt; ii++)                                     //  start worker threads
         start_wt(unbend_wthread,&wtnx[ii]);
      wait_wts();                                                          //  wait for completion

      mwpaint2();                                                          //  update window
   }

   return 0;                                                               //  not executed, stop g++ warning
}

   
void * unbend_wthread(void *arg)                                           //  worker thread function   v.7.7
{
   int         index = *((int *) arg);
   int         vstat, px3, py3, cx3, cy3;
   double      px1, py1, dispx, dispx2, dispy;
   uint16      vpix[3], *pix3;

   for (py3 = index; py3 < E3hh; py3 += Nwt)                               //  step through F3 pixels
   for (px3 = 0; px3 < E3ww; px3++)
   {
      pix3 = bmpixel(E3rgb16,px3,py3);                                     //  output pixel

      cx3 = unbend_vx1 + (unbend_vx2 - unbend_vx1) * py3 / E3hh;           //  center of unbend
      cy3 = unbend_hy1 + (unbend_hy2 - unbend_hy1) * px3 / E3ww;
      dispx = 2.0 * (px3 - cx3) / E3ww;                                    //  -1.0 ..  0.0 .. +1.0 (roughly)
      dispy = 2.0 * (py3 - cy3) / E3hh;                                    //  -1.0 ..  0.0 .. +1.0
      dispx2 = dispx * dispx - 0.5;                                        //  +0.5 .. -0.5 .. +0.5  curved

      px1 = px3 + dispx * dispy * unbend_horz2;                            //  input virtual pixel, x
      py1 = py3 - dispy * dispx2 * unbend_vert2;                           //  input virtual pixel, y
      vstat = vpixel(E1rgb16,px1,py1,vpix);                                //  input virtual pixel

      if (vstat) {
         pix3[0] = vpix[0];                                                //  input pixel >> output pixel
         pix3[1] = vpix[1];
         pix3[2] = vpix[2];
      }
      else pix3[0] = pix3[1] = pix3[2] = 0;
   }
      
   exit_wt();
   return 0;                                                               //  not executed, avoid gcc warning
}


/**************************************************************************/

//  warp/distort area - select image area and pull with mouse

float       *WarpAx, *WarpAy;                                              //  memory of all displaced pixels
float       WarpAmem[4][100];                                              //  undo memory, last 100 warps
int         NWarpA;                                                        //  WarpA mem count

void  WarpA_warpfunc(float wdx, float wdy, float wdw, float wdh, int acc);


void m_WarpA(GtkWidget *, cchar *)
{
   int      WarpA_dialog_event(zdialog *zd, cchar *event);
   
   cchar  *WarpA_message = ZTX(
             " Select an area to warp using select area function. \n"      //  v.6.3
             " Press [start warp] and pull area with mouse. \n"
             " Make multiple mouse pulls until satisfied. \n"
             " When finished, select another area or press [done]."); 
   
   int         px, py, ii;

   if (! edit_setup("warparea",0,2)) return;                               //  setup edit: no preview

   zdedit = zdialog_new(ZTX("Warp Image in Selected Area"),mWin,Bdone,Bcancel,null);
   zdialog_add_widget(zdedit,"label","lab1","dialog",WarpA_message,"space=5");
   zdialog_add_widget(zdedit,"hbox","hb1","dialog",0,"space=10");
   zdialog_add_widget(zdedit,"button","swarp","hb1",ZTX("start warp"),"space=5");
   zdialog_add_widget(zdedit,"button","undlast","hb1",Bundolast,"space=5");
   zdialog_add_widget(zdedit,"button","undall","hb1",Bundoall,"space=5");

   zdialog_run(zdedit,WarpA_dialog_event);                                 //  run dialog

   WarpAx = (float *) zmalloc(E3ww * E3hh * sizeof(float),"warp.ax");      //  get memory for pixel displacements
   WarpAy = (float *) zmalloc(E3ww * E3hh * sizeof(float),"warp.ay");
   
   NWarpA = 0;                                                             //  no warp data

   for (py = 0; py < E3hh; py++)                                           //  no pixel displacements
   for (px = 0; px < E3ww; px++)
   {
      ii = py * E3ww + px;
      WarpAx[ii] = WarpAy[ii] = 0.0;
   }
   
   return;
}


//  dialog event and completion callback function

int WarpA_dialog_event(zdialog * zd, cchar *event)
{
   void     WarpA_mousefunc(void);

   int         px, py, ii;
   float       wdx, wdy, wdw, wdh;

   if (zd->zstat)                                                          //  dialog complete
   {
      if (NWarpA) Fmodified = 1;
      else Fmodified = 0;

      if (zd->zstat == 1) edit_done();
      else edit_cancel();

      mouseCBfunc = 0;                                                     //  disconnect mouse
      Mcapture = 0;
      set_cursor(0);                                                       //  restore normal cursor

      zfree(WarpAx);                                                       //  release undo memory
      zfree(WarpAy);
      return 0;
   }
   
   if (strEqu(event,"F1")) showz_userguide("warp_local");                  //  F1 help       v.9.0
   if (strEqu(event,"undo")) edit_undo();                                  //  v.10.2
   if (strEqu(event,"redo")) edit_redo();                                  //  v.10.3

   if (strEqu(event,"swarp"))                                              //  start warp
   {
      if (! sa_active) {                                                   //  no select area active
      zmessageACK(GTK_WINDOW(mWin),ZTX("Select area first"));
         return 0;
      }

      select_edgecalc();                                                   //  calculate area edge distances

      mouseCBfunc = WarpA_mousefunc;                                       //  connect mouse function
      Mcapture++;
      set_cursor(dragcursor);                                              //  set drag cursor
   }

   if (strEqu(event,"undlast")) {
      if (NWarpA) {                                                        //  undo most recent warp
         ii = --NWarpA;
         wdx = WarpAmem[0][ii];
         wdy = WarpAmem[1][ii];
         wdw = WarpAmem[2][ii];
         wdh = WarpAmem[3][ii];
         WarpA_warpfunc(wdx,wdy,-wdw,-wdh,0);                              //  unwarp image
         WarpA_warpfunc(wdx,wdy,-wdw,-wdh,1);                              //  unwarp memory
      }
   }

   if (strEqu(event,"undall"))                                             //  undo all warps
   {
      edit_reset();                                                        //  v.10.3

      for (py = 0; py < E3hh; py++)                                        //  reset pixel displacements
      for (px = 0; px < E3ww; px++)
      {
         ii = py * E3ww + px;
         WarpAx[ii] = WarpAy[ii] = 0.0;
      }

      NWarpA = 0;                                                          //  erase undo memory
      mwpaint2(); 
   }

   return 1;
}


//  warp mouse function

void  WarpA_mousefunc(void)
{
   static float   wdx, wdy, wdw, wdh;
   static int     ii, warped = 0;

   if (Mxdrag || Mydrag)                                                   //  mouse drag underway
   {
      wdx = Mxdown;                                                        //  drag origin, image coordinates
      wdy = Mydown;
      wdw = Mxdrag - Mxdown;                                               //  drag increment
      wdh = Mydrag - Mydown;
      WarpA_warpfunc(wdx,wdy,wdw,wdh,0);                                   //  warp image
      warped = 1;
      return;
   }
   
   else if (warped) 
   {
      warped = 0;
      WarpA_warpfunc(wdx,wdy,wdw,wdh,1);                                   //  drag done, add to warp memory

      if (NWarpA == 100)                                                   //  if full, throw away oldest
      {
         NWarpA = 99;
         for (ii = 0; ii < NWarpA; ii++)
         {
            WarpAmem[0][ii] = WarpAmem[0][ii+1];
            WarpAmem[1][ii] = WarpAmem[1][ii+1];
            WarpAmem[2][ii] = WarpAmem[2][ii+1];
            WarpAmem[3][ii] = WarpAmem[3][ii+1];
         }
      }

      ii = NWarpA;
      WarpAmem[0][ii] = wdx;                                               //  save warp for undo
      WarpAmem[1][ii] = wdy;
      WarpAmem[2][ii] = wdw;
      WarpAmem[3][ii] = wdh;
      NWarpA++;
   }
   
   return;
}


//  warp image and accumulate warp memory

void  WarpA_warpfunc(float wdx, float wdy, float wdw, float wdh, int acc)
{
   int            ii, jj, px, py, vstat;
   double         ddx, ddy, dpe, dpm, mag, dispx, dispy;
   uint16         vpix[3], *pix3;

   edit_zapredo();                                                         //  delete redo copy    v.10.3

   for (ii = 0; ii < Fww * Fhh; ii++)                                      //  find pixels in select area
   {                                                                       //  v.9.6
      if (! sa_pixisin[ii]) continue;
      py = ii / Fww;
      px = ii - py * Fww;
      dpe = sa_pixisin[ii];                                                //  distance from area edge

      ddx = (px - wdx);
      ddy = (py - wdy);
      dpm = sqrt(ddx*ddx + ddy*ddy);                                       //  distance from drag origin

      if (dpm < 1) mag = 1;
      else mag = dpe / (dpe + dpm);                                        //  magnification, 0...1
      mag = mag * mag;

      dispx = -wdw * mag;                                                  //  warp = drag * magnification
      dispy = -wdh * mag;
      
      jj = py * E3ww + px;

      if (acc) {                                                           //  mouse drag done,
         WarpAx[jj] += dispx;                                              //    accumulate warp memory
         WarpAy[jj] += dispy;
         continue;
      }

      dispx += WarpAx[jj];                                                 //  add this warp to prior
      dispy += WarpAy[jj];

      vstat = vpixel(E1rgb16,px+dispx,py+dispy,vpix);                      //  input virtual pixel
      if (vstat) {
         pix3 = bmpixel(E3rgb16,px,py);                                    //  output pixel
         pix3[0] = vpix[0];
         pix3[1] = vpix[1];
         pix3[2] = vpix[2];
      }
   }

   Fmodified = 1;                                                          //  v.10.2
   mwpaint2();                                                             //  update window
   return;
}


/**************************************************************************/

//  warp/distort whole image with a curvy transform
//  fix perspective problems (e.g. curved walls, leaning buildings)

float       *WarpIx, *WarpIy;                                              //  memory of all dragged pixels
float       WarpImem[4][100];                                              //  undo memory, last 100 drags
int         NWarpI;                                                        //  WarpImem count
int         WarpIdrag;
int         WarpIww, WarpIhh;

void  WarpI_warpfunc(float wdx, float wdy, float wdw, float wdh, int acc);


void m_WarpI(GtkWidget *, cchar *)                                         //  v.6.1
{
   int      WarpI_dialog_event(zdialog *zd, cchar *event);
   void     WarpI_mousefunc(void);

   cchar  *WarpI_message = ZTX(
             " Pull on an image edge using the mouse. \n"
             " Make multiple mouse pulls until satisfied. \n"
             " When finished, press [done]."); 
   
   int         px, py, ii;

   if (! edit_setup("warpcurve",1,0)) return;                              //  setup edit: use preview

   zdedit = zdialog_new(ZTX("Fix Image Perspective"),mWin,Bdone,Bcancel,null);
   zdialog_add_widget(zdedit,"label","lab1","dialog",WarpI_message,"space=5");
   zdialog_add_widget(zdedit,"hbox","hb1","dialog",0,"space=10");
   zdialog_add_widget(zdedit,"button","undlast","hb1",Bundolast,"space=5");
   zdialog_add_widget(zdedit,"button","undall","hb1",Bundoall,"space=5");

   zdialog_run(zdedit,WarpI_dialog_event);                                 //  run dialog

   NWarpI = WarpIdrag = 0;                                                 //  no drag data

   WarpIx = (float *) zmalloc(E3ww * E3hh * sizeof(float),"warp.ix");      //  get memory for pixel displacements
   WarpIy = (float *) zmalloc(E3ww * E3hh * sizeof(float),"warp.iy");
   
   for (py = 0; py < E3hh; py++)                                           //  no pixel displacements
   for (px = 0; px < E3ww; px++)
   {
      ii = py * E3ww + px;
      WarpIx[ii] = WarpIy[ii] = 0.0;
   }
   
   WarpIww = E3ww;                                                         //  preview dimensions
   WarpIhh = E3hh;
   
   mouseCBfunc = WarpI_mousefunc;                                          //  connect mouse function
   Mcapture++;
   set_cursor(dragcursor);                                                 //  set drag cursor

   return;
}


//  dialog event and completion callback function

int WarpI_dialog_event(zdialog * zd, cchar *event)
{
   int         px, py, ii;
   float       wdx, wdy, wdw, wdh;
   int         fpx, fpy, epx, epy, vstat;
   double      scale, dispx, dispy;
   uint16      vpix[3], *pix3;

   if (zd->zstat) goto complete;

   if (strEqu(event,"F1")) showz_userguide("warp_global");                 //  F1 help       v.9.0
   if (strEqu(event,"undo")) edit_undo();                                  //  v.10.2
   if (strEqu(event,"redo")) edit_redo();                                  //  v.10.3

   if (strEqu(event,"undlast")) 
   {
      if (NWarpI == 1) event = "undall";
      else if (NWarpI) {                                                   //  undo most recent drag
         ii = --NWarpI;
         wdx = WarpImem[0][ii];
         wdy = WarpImem[1][ii];
         wdw = WarpImem[2][ii];
         wdh = WarpImem[3][ii];
         WarpI_warpfunc(wdx,wdy,-wdw,-wdh,0);                              //  undrag image
         WarpI_warpfunc(wdx,wdy,-wdw,-wdh,1);                              //  undrag memory
      }
   }

   if (strEqu(event,"undall"))                                             //  undo all drags
   {
      NWarpI = 0;                                                          //  erase undo memory

      for (py = 0; py < E3hh; py++)                                        //  reset pixel displacements
      for (px = 0; px < E3ww; px++)
      {
         ii = py * E3ww + px;
         WarpIx[ii] = WarpIy[ii] = 0.0;
      }
      edit_reset();                                                        //  restore image 1  v.10.3
   }

   return 1;

complete:

   if (zd->zstat != 1) edit_cancel();
   else if (NWarpI == 0) edit_cancel();
   else 
   {
      edit_fullsize();                                                     //  get full-size E1/E3

      scale = 1.0 * (E3ww + E3hh) / (WarpIww + WarpIhh);

      for (fpy = 0; fpy < E3hh; fpy++)                                     //  scale net pixel displacements
      for (fpx = 0; fpx < E3ww; fpx++)                                     //    to full image size
      {
         epx = WarpIww * fpx / E3ww;
         epy = WarpIhh * fpy / E3hh;
         ii = epy * WarpIww + epx;
         dispx = WarpIx[ii] * scale;
         dispy = WarpIy[ii] * scale;

         vstat = vpixel(E1rgb16,fpx+dispx,fpy+dispy,vpix);                 //  input virtual pixel
         pix3 = bmpixel(E3rgb16,fpx,fpy);                                  //  output pixel
         if (vstat) {
            pix3[0] = vpix[0];
            pix3[1] = vpix[1];
            pix3[2] = vpix[2];
         }
         else pix3[0] = pix3[1] = pix3[2] = 0;
      }

      edit_done();
   }

   mouseCBfunc = 0;                                                        //  disconnect mouse
   Mcapture = 0;
   set_cursor(0);                                                          //  restore normal cursor

   zfree(WarpIx);                                                          //  release memory
   zfree(WarpIy);
   return 0;
}


//  WarpI mouse function

void  WarpI_mousefunc(void)
{
   static float   wdx, wdy, wdw, wdh;
   int            ii;

   if (Mxdrag || Mydrag)                                                   //  mouse drag underway
   {
      wdx = Mxdown;                                                        //  drag origin, window coordinates
      wdy = Mydown;
      wdw = Mxdrag - Mxdown;                                               //  drag increment
      wdh = Mydrag - Mydown;
      WarpI_warpfunc(wdx,wdy,wdw,wdh,0);                                   //  drag image
      WarpIdrag = 1;
      return;
   }
   
   else if (WarpIdrag) 
   {
      WarpIdrag = 0;
      WarpI_warpfunc(wdx,wdy,wdw,wdh,1);                                   //  drag done, add to memory

      if (NWarpI == 100)                                                   //  if full, throw away oldest
      {
         NWarpI = 99;
         for (ii = 0; ii < NWarpI; ii++)
         {
            WarpImem[0][ii] = WarpImem[0][ii+1];
            WarpImem[1][ii] = WarpImem[1][ii+1];
            WarpImem[2][ii] = WarpImem[2][ii+1];
            WarpImem[3][ii] = WarpImem[3][ii+1];
         }
      }

      ii = NWarpI;
      WarpImem[0][ii] = wdx;                                               //  save drag for undo
      WarpImem[1][ii] = wdy;
      WarpImem[2][ii] = wdw;
      WarpImem[3][ii] = wdh;
      NWarpI++;
   }
   
   return;
}


//  warp image and accumulate warp memory
//  mouse at (mx,my) is moved (mw,mh) pixels

void  WarpI_warpfunc(float mx, float my, float mw, float mh, int acc)
{
   int         ii, px, py, vstat;
   double      mag, dispx, dispy;
   double      d1, d2;
   uint16      vpix[3], *pix3;
   
   edit_zapredo();                                                         //  delete redo copy    v.10.3

   d1 = E3ww * E3ww + E3hh * E3hh;
   
   for (py = 0; py < E3hh; py++)                                           //  process all pixels
   for (px = 0; px < E3ww; px++)
   {
      d2 = (px-mx)*(px-mx) + (py-my)*(py-my);                              //  better algorithm   v.8.0
      mag = (1.0 - d2 / d1);
      mag = mag * mag;                                                     //  faster than pow(mag,16);
      mag = mag * mag;
      mag = mag * mag;

      dispx = -mw * mag;                                                   //  displacement = drag * mag
      dispy = -mh * mag;
      
      ii = py * E3ww + px;

      if (acc) {                                                           //  drag done, accumulate drag sum
         WarpIx[ii] += dispx;
         WarpIy[ii] += dispy;
         continue;
      }

      dispx += WarpIx[ii];                                                 //  add this drag to prior sum
      dispy += WarpIy[ii];

      vstat = vpixel(E1rgb16,px+dispx,py+dispy,vpix);                      //  input virtual pixel
      pix3 = bmpixel(E3rgb16,px,py);                                       //  output pixel
      if (vstat) {
         pix3[0] = vpix[0];
         pix3[1] = vpix[1];
         pix3[2] = vpix[2];
      }
      else pix3[0] = pix3[1] = pix3[2] = 0;
   }

   Fmodified = 1;                                                          //  v.6.3
   mwpaint2();                                                             //  update window
   return;
}


/**************************************************************************/

//  warp/distort whole image using affine transform 
//  (straight lines remain straight)

double      WarpF_old[3][2];                                               //  3 original image points 
double      WarpF_new[3][2];                                               //  corresponding warped points
double      WarpF_coeff[6];                                                //  transform coefficients
double      WarpF_Icoeff[6];                                               //  inverse transform coefficients
int         WarpF_ftf;                                                     //  first time flag

void  WarpF_warpfunc();                                                    //  image warp function
void  WarpF_affine(double po[3][2], double pn[3][2], double coeff[6]);     //  compute affine transform coefficients
void  WarpF_invert(double coeff[6], double Icoeff[6]);                     //  compute reverse transform coefficients


void m_WarpF(GtkWidget *, cchar *)                                         //  new  v.9.3
{
   int      WarpF_dialog_event(zdialog *zd, cchar *event);
   void     WarpF_mousefunc(void);

   cchar  *WarpF_message = ZTX(
             " Pull on an image corner using the mouse. \n"
             " Make multiple mouse pulls until satisfied. \n"
             " When finished, press [done]."); 
   
   if (! edit_setup("warpaff",1,0)) return;                                //  setup edit: use preview

   zdedit = zdialog_new(ZTX("Fix Image Perspective"),mWin,Bdone,Bcancel,null);
   zdialog_add_widget(zdedit,"label","lab1","dialog",WarpF_message,"space=5");

   zdialog_run(zdedit,WarpF_dialog_event);                                 //  run dialog, parallel

   WarpF_ftf = 1;                                                          //  1st warp flag

   mouseCBfunc = WarpF_mousefunc;                                          //  connect mouse function
   Mcapture++;
   set_cursor(dragcursor);                                                 //  set drag cursor

   return;
}


//  dialog event and completion callback function

int WarpF_dialog_event(zdialog *zd, cchar *event)
{
   double      scale;
   int         ww, hh;

   if (zd->zstat) goto complete;

   if (strEqu(event,"F1")) showz_userguide("warp_affine");                 //  F1 help 
   if (strEqu(event,"undo")) edit_undo();                                  //  v.10.2
   if (strEqu(event,"redo")) edit_redo();                                  //  v.10.3
   return 1;

complete:

   if (zd->zstat != 1) edit_cancel();
   
   else 
   {
      ww = E3ww;                                                           //  preview image dimensions
      hh = E3hh;

      edit_fullsize();                                                     //  get full-size images

      scale = 1.0 * (E3ww + E3hh) / (ww + hh);                             //  preview to full-size scale factor
      
      WarpF_old[0][0] = WarpF_old[0][0] * scale;                           //  re-scale new and old points
      WarpF_old[0][1] = WarpF_old[0][1] * scale;
      WarpF_old[1][0] = WarpF_old[1][0] * scale;
      WarpF_old[1][1] = WarpF_old[1][1] * scale;
      WarpF_old[2][0] = WarpF_old[2][0] * scale;
      WarpF_old[2][1] = WarpF_old[2][1] * scale;

      WarpF_new[0][0] = WarpF_new[0][0] * scale;
      WarpF_new[0][1] = WarpF_new[0][1] * scale;
      WarpF_new[1][0] = WarpF_new[1][0] * scale;
      WarpF_new[1][1] = WarpF_new[1][1] * scale;
      WarpF_new[2][0] = WarpF_new[2][0] * scale;
      WarpF_new[2][1] = WarpF_new[2][1] * scale;
      
      WarpF_warpfunc();                                                    //  warp full-size image
      edit_done();
   }

   mouseCBfunc = 0;                                                        //  disconnect mouse
   Mcapture = 0;
   set_cursor(0);                                                          //  restore normal cursor
   return 0;
}


//  WarpF mouse function

void  WarpF_mousefunc(void)
{
   int      mdx1, mdy1, mdx2, mdy2;
   double   x1o, y1o, x2o, y2o, x3o, y3o;
   double   x1n, y1n, x2n, y2n, x3n, y3n;
   double   a, b, c, d, e, f;
   
   if (Mxdrag + Mydrag == 0) return;

   mdx1 = Mxdown;                                                          //  mouse drag origin
   mdy1 = Mydown;
   mdx2 = Mxdrag;                                                          //  mouse drag position
   mdy2 = Mydrag;

   Mxdown = Mxdrag;                                                        //  reset origin for next time
   Mydown = Mydrag;
   
   x1n = mdx1;                                                             //  point 1 = drag origin
   y1n = mdy1;
   x2n = E3ww - x1n;                                                       //  point 2 = mirror of point1
   y2n = E3hh - y1n;
   x3n = E3ww * (y2n / E3hh);
   y3n = E3hh * (1.0 - (x2n / E3ww));

   if (WarpF_ftf)                                                          //  first warp
   {
      WarpF_ftf = 0;
      x1o = x1n;                                                           //  old = current positions
      y1o = y1n;
      x2o = x2n;
      y2o = y2n;
      x3o = x3n;
      y3o = y3n;
   }
   else
   {
      WarpF_invert(WarpF_coeff,WarpF_Icoeff);                              //  get inverse coefficients
      a = WarpF_Icoeff[0];
      b = WarpF_Icoeff[1];
      c = WarpF_Icoeff[2];
      d = WarpF_Icoeff[3];
      e = WarpF_Icoeff[4];
      f = WarpF_Icoeff[5];
      
      x1o = a * x1n + b * y1n + c;                                         //  compute old from current positions
      y1o = d * x1n + e * y1n + f;
      x2o = a * x2n + b * y2n + c;
      y2o = d * x2n + e * y2n + f;
      x3o = a * x3n + b * y3n + c;
      y3o = d * x3n + e * y3n + f;
   }
      
   WarpF_old[0][0] = x1o;                                                  //  set up 3 old points and corresponding
   WarpF_old[0][1] = y1o;                                                  //    new points for affine translation
   WarpF_old[1][0] = x2o;
   WarpF_old[1][1] = y2o;
   WarpF_old[2][0] = x3o;
   WarpF_old[2][1] = y3o;

   x1n = mdx2;                                                             //  point 1 new position = drag position
   y1n = mdy2;
   x2n = E3ww - x1n;                                                       //  point 2 new = mirror of point1 new
   y2n = E3hh - y1n;

   WarpF_new[0][0] = x1n;                                                  //  3 new points 
   WarpF_new[0][1] = y1n;
   WarpF_new[1][0] = x2n;
   WarpF_new[1][1] = y2n;
   WarpF_new[2][0] = x3n;
   WarpF_new[2][1] = y3n;
   
   WarpF_warpfunc();                                                       //  do the warp

   return;
}


//  warp image and accumulate warp memory

void  WarpF_warpfunc()
{
   double      a, b, c, d, e, f;
   int         px3, py3, vstat;
   double      px1, py1;
   uint16      vpix1[3], *pix3;
   
   edit_zapredo();                                                         //  delete redo copy    v.10.3

   WarpF_affine(WarpF_old, WarpF_new, WarpF_coeff);                        //  get coefficients for forward transform
   WarpF_invert(WarpF_coeff, WarpF_Icoeff);                                //  get coefficients for reverse transform
   
   a = WarpF_Icoeff[0];                                                    //  coefficients to map output pixels
   b = WarpF_Icoeff[1];                                                    //    to corresponding input pixels
   c = WarpF_Icoeff[2];
   d = WarpF_Icoeff[3];
   e = WarpF_Icoeff[4];
   f = WarpF_Icoeff[5];
   
   for (py3 = 0; py3 < E3hh; py3++)                                        //  loop all output pixels
   for (px3 = 0; px3 < E3ww; px3++)
   {
      px1 = a * px3 + b * py3 + c;                                         //  corresponding input pixel
      py1 = d * px3 + e * py3 + f;

      vstat = vpixel(E1rgb16,px1,py1,vpix1);                               //  input virtual pixel
      pix3 = bmpixel(E3rgb16,px3,py3);                                     //  output pixel

      if (vstat) {
         pix3[0] = vpix1[0];
         pix3[1] = vpix1[1];
         pix3[2] = vpix1[2];
      }
      else pix3[0] = pix3[1] = pix3[2] = 0;
   }

   Fmodified = 1;
   mwpaint2();                                                             //  update window
   return;
}


/**************************************************************************

   Compute affine transformation of an image (warp image).

   Given 3 new (warped) positions for 3 image points, derive the 
   coefficients of the translation function to warp the entire image.

   Inputs:
      pold[3][2]  (x,y) coordinates for 3 points in original image
      pnew[3][2]  (x,y) coordinates for same points in warped image
   
   Output: 
      coeff[6]  coefficients of translation function which can be used
                to convert all image points to their warped positions

   If coeff[6] = (a, b, c, d, e, f) then the following formula
   can be used to convert an image point to its warped position:

      Xnew = a * Xold + b * Yold + c
      Ynew = d * Xold + e * Yold + f

***************************************************************************/

void WarpF_affine(double pold[3][2], double pnew[3][2], double coeff[6])
{
   double   x11, y11, x12, y12, x13, y13;                                  //  original points
   double   x21, y21, x22, y22, x23, y23;                                  //  moved points
   double   a, b, c, d, e, f;                                              //  coefficients
   double   A1, A2, B1, B2, C1, C2;
   
   x11 = pold[0][0];
   y11 = pold[0][1];
   x12 = pold[1][0];
   y12 = pold[1][1];
   x13 = pold[2][0];
   y13 = pold[2][1];

   x21 = pnew[0][0];
   y21 = pnew[0][1];
   x22 = pnew[1][0];
   y22 = pnew[1][1];
   x23 = pnew[2][0];
   y23 = pnew[2][1];
   
   A1 = x11 - x12;
   A2 = x12 - x13;
   B1 = y11 - y12;
   B2 = y12 - y13;
   C1 = x21 - x22;
   C2 = x22 - x23;
   
   a = (B1 * C2 - B2 * C1) / (A2 * B1 - A1 * B2);
   b = (A1 * C2 - A2 * C1) / (A1 * B2 - A2 * B1);
   c = x23 - a * x13 - b * y13;
   
   C1 = y21 - y22;
   C2 = y22 - y23;
   
   d = (B1 * C2 - B2 * C1) / (A2 * B1 - A1 * B2);
   e = (A1 * C2 - A2 * C1) / (A1 * B2 - A2 * B1);
   f = y23 - d * x13 - e * y13;
   
   coeff[0] = a;
   coeff[1] = b;
   coeff[2] = c;
   coeff[3] = d;
   coeff[4] = e;
   coeff[5] = f;

   return;
}   


/**************************************************************************

   Invert affine transform

   Input:
      coeff[6]  coefficients of translation function to convert
                image points to their warped positions
   Output: 
      Icoeff[6]  coefficients of translation function to convert
                 warped image points to their original positions

   If Icoeff[6] = (a, b, c, d, e, f) then the following formula can be
      used to translate a warped image point to its original position:

      Xold = a * Xnew + b * Ynew + c
      Yold = d * Xnew + e * Ynew + f

***************************************************************************/

void WarpF_invert(double coeff[6], double Icoeff[6])
{
   double   a, b, c, d, e, f, Z;
   
   a = coeff[0];
   b = coeff[1];
   c = coeff[2];
   d = coeff[3];
   e = coeff[4];
   f = coeff[5];
   
   Z = 1.0 / (a * e - b * d);
   
   Icoeff[0] = e * Z;
   Icoeff[1] = - b * Z;
   Icoeff[2] = Z * (b * f - c * e);
   Icoeff[3] = - d * Z;
   Icoeff[4] = a * Z;
   Icoeff[5] = Z * (c * d - a * f);
   
   return;
}


/**************************************************************************/

//  image color-depth reduction

int      colordep_depth = 16;                                              //  bits per RGB color


void m_colordep(GtkWidget *, cchar *)
{
   int    colordep_dialog_event(zdialog *zd, cchar *event);
   void * colordep_thread(void *);

   cchar  *colmess = ZTX("Set color depth to 1-16 bits");
   
   if (! edit_setup("colordepth",1,2)) return;                             //  setup edit: preview

   zdedit = zdialog_new(ZTX("Set Color Depth"),mWin,Bdone,Bcancel,null);
   zdialog_add_widget(zdedit,"label","lab1","dialog",colmess,"space=5");
   zdialog_add_widget(zdedit,"hbox","hb1","dialog",0,"space=10");
   zdialog_add_widget(zdedit,"spin","colors","hb1","1|16|1|16","space=5");

   zdialog_run(zdedit,colordep_dialog_event);                              //  run dialog - parallel
   
   colordep_depth = 16;
   start_thread(colordep_thread,0);                                        //  start working thread
   return;
}


//  dialog event and completion callback function

int colordep_dialog_event(zdialog *zd, cchar *event)
{
   if (zd->zstat)                                                          //  dialog complete
   {
      if (zd->zstat == 1) edit_done();
      else edit_cancel();
      return 0;
   }

   if (strEqu(event,"F1")) showz_userguide("color_depth");                 //  F1 help       v.9.0
   if (strEqu(event,"undo")) edit_undo();                                  //  v.10.2
   if (strEqu(event,"redo")) edit_redo();                                  //  v.10.3
   if (strEqu(event,"blendwidth")) signal_thread();                        //  v.10.3

   if (strEqu(event,"colors")) {
      zdialog_fetch(zd,"colors",colordep_depth);
      signal_thread();
   }
   
   return 0;
}


//  image color depth thread function

void * colordep_thread(void *)
{
   int         ii, px, py, rgb, dist = 0;
   uint16      m1, m2, val1, val3;
   uint16      *pix1, *pix3;
   double      fmag, f1, f2;
   
   while (true)
   {
      thread_idle_loop();                                                  //  wait for work or exit request
   
      m1 = 0xFFFF << (16 - colordep_depth);                                //  5 > 1111100000000000
      m2 = 0x8000 >> colordep_depth;                                       //  5 > 0000010000000000
      
      fmag = 65535.0 / m1;                                                 //  full brightness range  v.7.0

      for (py = 0; py < E3hh; py++)
      for (px = 0; px < E3ww; px++)
      {
         if (sa_active) {                                                  //  select area active
            ii = py * Fww + px;
            dist = sa_pixisin[ii];                                         //  distance from edge
            if (! dist) continue;                                          //  outside pixel
         }

         pix1 = bmpixel(E1rgb16,px,py);                                    //  input pixel
         pix3 = bmpixel(E3rgb16,px,py);                                    //  output pixel
         
         for (rgb = 0; rgb < 3; rgb++)
         {
            val1 = pix1[rgb];
            if (val1 < m1) val3 = (val1 + m2) & m1;                        //  round   v.7.0
            else val3 = m1;
            val3 = uint(val3 * fmag);

            if (sa_active && dist < sa_blend) {                            //  select area is active,
               f2 = 1.0 * dist / sa_blend;                                 //    blend changes over sa_blend
               f1 = 1.0 - f2;
               val3 = int(f1 * val1 + f2 * val3);
            }

            pix3[rgb] = val3;
         }
      }

      Fmodified = 1;
      mwpaint2();                                                          //  update window
   }

   return 0;                                                               //  not executed, stop g++ warning
}


/**************************************************************************/

//  convert image to simulate a drawing

int      draw_contrast;
int      draw_threshold;
int      draw_pixcon;
int      draw_reverse;
double   draw_trfunc[256];
double   draw_pixcon3;
uint8    *draw_pixcon_map = 0;

void m_draw(GtkWidget *, cchar *)                                          //  v.6.7
{
   int    draw_dialog_event(zdialog* zd, cchar *event);
   void * draw_thread(void *);

   cchar       *title = ZTX("Simulate Drawing");
   uint16      *pix1, *pix2;
   int         ii, px, py, qx, qy;
   int         red, green, blue, con, maxcon;

   if (! edit_setup("draw",0,2)) return;                                   //  setup edit: no preview

   draw_pixcon_map = (uint8 *) zmalloc(E1ww*E1hh,"pixconmap");             //  set up pixel contrast map
   memset(draw_pixcon_map,0,E1ww*E1hh);

   for (py = 1; py < E1hh-1; py++)                                         //  scan image pixels
   for (px = 1; px < E1ww-1; px++)
   {
      pix1 = bmpixel(E1rgb16,px,py);                                       //  pixel at (px,py)
      red = pix1[0];                                                       //  pixel RGB levels
      green = pix1[1];
      blue = pix1[2];
      maxcon = 0;

      for (qy = py-1; qy < py+2; qy++)                                     //  loop 3x3 block of neighbor
      for (qx = px-1; qx < px+2; qx++)                                     //    pixels around pix1
      {
         pix2 = bmpixel(E1rgb16,qx,qy);                                    //  find max. contrast with
         con = abs(red-pix2[0]) + abs(green-pix2[1]) + abs(blue-pix2[2]);  //    neighbor pixel
         if (con > maxcon) maxcon = con;
      }

      ii = py * E1ww + px;
      draw_pixcon_map[ii] = (maxcon/3) >> 8;                               //  contrast for (px,py) 0-255
   }

   zdedit = zdialog_new(title,mWin,Bdone,Bcancel,null);
   zdialog_add_widget(zdedit,"hbox","hb1","dialog",0,"space=10");
   zdialog_add_widget(zdedit,"vbox","vb1","hb1",0,"homog");
   zdialog_add_widget(zdedit,"vbox","vb2","hb1",0,"homog|expand");
   zdialog_add_widget(zdedit,"label","lab1","vb1",ZTX("contrast"));
   zdialog_add_widget(zdedit,"label","lab2","vb1",Bthresh);
   zdialog_add_widget(zdedit,"label","lab3","vb1",ZTX("outlines"));
   zdialog_add_widget(zdedit,"hscale","contrast","vb2","0|100|1|0","expand");
   zdialog_add_widget(zdedit,"hscale","threshold","vb2","0|100|1|0","expand");
   zdialog_add_widget(zdedit,"hscale","pixcon","vb2","0|255|1|0","expand");
   zdialog_add_widget(zdedit,"hbox","hb4","dialog");
   zdialog_add_widget(zdedit,"radio","pencil","hb4",ZTX("pencil"),"space=10");
   zdialog_add_widget(zdedit,"radio","chalk","hb4",ZTX("chalk"),"space=10");

   zdialog_resize(zdedit,300,0);
   zdialog_run(zdedit,draw_dialog_event);                                  //  run dialog - parallel
   
   start_thread(draw_thread,0);                                            //  start working thread
   return;
}


//  dialog event and completion callback function

int draw_dialog_event(zdialog *zd, cchar *event)                           //  draw dialog event function
{
   if (zd->zstat)                                                          //  dialog complete
   {
      if (zd->zstat == 1) edit_done();
      else edit_cancel();
      zfree(draw_pixcon_map);
      return 0;
   }

   if (strEqu(event,"F1")) showz_userguide("drawing");                     //  F1 context help
   if (strEqu(event,"undo")) edit_undo();                                  //  v.10.2
   if (strEqu(event,"redo")) edit_redo();                                  //  v.10.3
   if (strEqu(event,"blendwidth")) signal_thread();                        //  v.10.3

   if (strcmpv(event,"contrast","threshold","pixcon","chalk",null)) 
   {
      zdialog_fetch(zd,"contrast",draw_contrast);                          //  get slider values
      zdialog_fetch(zd,"threshold",draw_threshold);
      zdialog_fetch(zd,"pixcon",draw_pixcon);
      zdialog_fetch(zd,"chalk",draw_reverse);
      signal_thread();                                                     //  trigger update thread
   }

   return 1;
}


//  thread function - use multiple working threads

void * draw_thread(void *)
{
   void  * draw_wthread(void *arg);

   int         ii;
   double      threshold, contrast, trf;
   
   while (true)
   {
      thread_idle_loop();                                                  //  wait for work or exit request

      threshold = 0.01 * draw_threshold;                                   //  range 0 to 1
      contrast = 0.01 * draw_contrast;                                     //  range 0 to 1
      
      for (ii = 0; ii < 256; ii++)                                         //  brightness transfer function
      {
         trf = 1.0 - 0.003906 * (256 - ii) * contrast;                     //  ramp-up from 0-1 to 1
         if (ii < 256 * threshold) trf = 0;                                //  0 if below threshold
         draw_trfunc[ii] = trf;
      }
      
      for (ii = 0; ii < Nwt; ii++)                                         //  start worker threads
         start_wt(draw_wthread,&wtnx[ii]);
      wait_wts();                                                          //  wait for completion

      Fmodified = 1;
      mwpaint2();                                                          //  update window
   }

   return 0;                                                               //  not executed, stop g++ warning
}


void * draw_wthread(void *arg)                                             //  worker thread function
{
   void  draw_1pix(int px, int py);

   int         index = *((int *) (arg));
   int         px, py;
   double      pixcon;

   pixcon = draw_pixcon / 255.0;                                           //  0-1 linear ramp
   draw_pixcon3 = 255 * pixcon * pixcon * pixcon;                          //  0-255 cubic ramp

   for (py = index+1; py < E1hh-1; py += Nwt)                              //  process all pixels
   for (px = 1; px < E1ww-1; px++)
      draw_1pix(px,py);

   exit_wt();
   return 0;                                                               //  not executed, avoid gcc warning
}


void draw_1pix(int px, int py)                                             //  process one pixel
{
   uint16      *pix1, *pix3;
   int         ii, dist = 0;
   int         bright1, bright2;
   int         red1, green1, blue1;
   int         red3, green3, blue3;
   double      dold, dnew;
   double      pixcon = draw_pixcon3;

   if (sa_active) {                                                        //  select area active
      ii = py * Fww + px;
      dist = sa_pixisin[ii];                                               //  distance from edge
      if (! dist) return;                                                  //  outside pixel
   }
   
   pix1 = bmpixel(E1rgb16,px,py);                                          //  input pixel
   pix3 = bmpixel(E3rgb16,px,py);                                          //  output pixel
      
   red1 = pix1[0];
   green1 = pix1[1];
   blue1 = pix1[2];
   
   bright1 = ((red1 + green1 + blue1) / 3) >> 8;                           //  old brightness  0-255
   bright2 = bright1 * draw_trfunc[bright1];                               //  new brightness  0-255
   
   ii = py * E1ww + px;
   if (draw_pixcon_map[ii] < pixcon) bright2 = 255;

   if (pixcon > 1 && bright2 > draw_threshold) bright2 = 255;              //  empirical !!!

   if (draw_reverse) bright2 = 255 - bright2;                              //  negate if "chalk"

   red3 = green3 = blue3 = bright2 << 8;                                   //  gray scale, new brightness

   if (sa_active && dist < sa_blend) {                                     //  select area is active,
      dnew = 1.0 * dist / sa_blend;
      dold = 1.0 - dnew;
      red3 = dnew * red3 + dold * red1;
      green3 = dnew * green3 + dold * green1;
      blue3 = dnew * blue3 + dold * blue1;
   }

   pix3[0] = red3;
   pix3[1] = green3;
   pix3[2] = blue3;
   
   return;
}


/**************************************************************************/

//  convert image to simulate an embossing

int      emboss_radius, emboss_color;
double   emboss_depth;
double   emboss_kernel[20][20];                                            //  support radius <= 9

void m_emboss(GtkWidget *, cchar *)                                        //  v.6.7
{
   int    emboss_dialog_event(zdialog* zd, cchar *event);
   void * emboss_thread(void *);

   cchar  *title = ZTX("Simulate Embossing");

   if (! edit_setup("emboss",0,2)) return;                                 //  setup edit: no preview

   zdedit = zdialog_new(title,mWin,Bdone,Bcancel,null);
   zdialog_add_widget(zdedit,"hbox","hb1","dialog",0,"space=10");
   zdialog_add_widget(zdedit,"label","lab1","hb1",Bradius,"space=5");
   zdialog_add_widget(zdedit,"spin","radius","hb1","0|9|1|0");
   zdialog_add_widget(zdedit,"label","lab2","hb1",ZTX("depth"),"space=5");
   zdialog_add_widget(zdedit,"spin","depth","hb1","0|99|1|0");
   zdialog_add_widget(zdedit,"check","color","hb1",ZTX("color"),"space=8");

   zdialog_run(zdedit,emboss_dialog_event);                                //  run dialog - parallel
   
   start_thread(emboss_thread,0);                                          //  start working thread
   return;
}


//  dialog event and completion callback function

int emboss_dialog_event(zdialog *zd, cchar *event)                         //  emboss dialog event function
{
   if (zd->zstat)                                                          //  dialog complete
   {
      if (zd->zstat == 1) edit_done();
      else edit_cancel();
      return 0;
   }

   if (strEqu(event,"F1")) showz_userguide("embossing");                   //  F1 context help
   if (strEqu(event,"undo")) edit_undo();                                  //  v.10.2
   if (strEqu(event,"redo")) edit_redo();                                  //  v.10.3
   if (strEqu(event,"blendwidth")) signal_thread();                        //  v.10.3

   if (strcmpv(event,"radius","depth","color",null))
   {
      zdialog_fetch(zd,"radius",emboss_radius);                            //  get user inputs
      zdialog_fetch(zd,"depth",emboss_depth);
      zdialog_fetch(zd,"color",emboss_color);
      signal_thread();                                                     //  trigger update thread
   }

   return 1;
}


//  thread function - use multiple working threads

void * emboss_thread(void *)
{
   void  * emboss_wthread(void *arg);

   int         ii, dx, dy, rad;
   double      depth, kern, coeff;
   
   while (true)
   {
      thread_idle_loop();                                                  //  wait for work or exit request
      
      rad = emboss_radius;
      depth = emboss_depth;

      coeff = 0.1 * depth / (rad * rad + 1);

      for (dy = -rad; dy <= rad; dy++)                                     //  build kernel with radius and depth
      for (dx = -rad; dx <= rad; dx++)
      {
         kern = coeff * (dx + dy);
         emboss_kernel[dx+rad][dy+rad] = kern;
      }
      
      emboss_kernel[rad][rad] = 1;                                         //  kernel center cell = 1

      for (ii = 0; ii < Nwt; ii++)                                         //  start worker threads
         start_wt(emboss_wthread,&wtnx[ii]);
      wait_wts();                                                          //  wait for comletion

      Fmodified = 1;
      mwpaint2();                                                          //  update window
   }

   return 0;                                                               //  not executed, stop g++ warning
}


void * emboss_wthread(void *arg)                                           //  worker thread function
{
   void  emboss_1pix(int px, int py);

   int         index = *((int *) (arg));
   int         px, py;

   for (py = index; py < E1hh; py += Nwt)                                  //  process all pixels
   for (px = 0; px < E1ww; px++)
      emboss_1pix(px,py);

   exit_wt();
   return 0;                                                               //  not executed, avoid gcc warning
}


void emboss_1pix(int px, int py)                                           //  process one pixel
{
   int         ii, dist = 0;
   int         bright1, bright3;
   int         rgb, dx, dy, rad;
   uint16      *pix1, *pix3, *pixN;
   double      sumpix, kern, dold, dnew;
   
   if (sa_active) {                                                        //  select area active
      ii = py * Fww + px;
      dist = sa_pixisin[ii];                                               //  distance from edge
      if (! dist) return;                                                  //  outside pixel
   }

   rad = emboss_radius;

   if (px < rad || py < rad) return;
   if (px > E3ww-rad-1 || py > E3hh-rad-1) return;
   
   pix1 = bmpixel(E1rgb16,px,py);                                          //  input pixel
   pix3 = bmpixel(E3rgb16,px,py);                                          //  output pixel
   
   if (emboss_color)                                                       //  keep color   v.6.9
   {
      for (rgb = 0; rgb < 3; rgb++)
      {      
         sumpix = 0;
         
         for (dy = -rad; dy <= rad; dy++)                                  //  loop surrounding block of pixels
         for (dx = -rad; dx <= rad; dx++)
         {
            pixN = pix1 + (dy * E1ww + dx) * 3;
            kern = emboss_kernel[dx+rad][dy+rad];
            sumpix += kern * pixN[rgb];
      
            bright1 = pix1[rgb];
            bright3 = sumpix;
            if (bright3 < 0) bright3 = 0;
            if (bright3 > 65535) bright3 = 65535;

            if (sa_active && dist < sa_blend) {                            //  select area is active,
               dnew = 1.0 * dist / sa_blend;                               //    blend changes over sa_blend
               dold = 1.0 - dnew;
               bright3 = dnew * bright3 + dold * bright1;
            }

            pix3[rgb] = bright3;
         }
      }
   }
   
   else                                                                    //  use gray scale
   {
      sumpix = 0;
         
      for (dy = -rad; dy <= rad; dy++)                                     //  loop surrounding block of pixels
      for (dx = -rad; dx <= rad; dx++)
      {
         pixN = pix1 + (dy * E1ww + dx) * 3;
         kern = emboss_kernel[dx+rad][dy+rad];
         sumpix += kern * (pixN[0] + pixN[1] + pixN[2]);
      }
      
      bright1 = 0.3333 * (pix1[0] + pix1[1] + pix1[2]);
      bright3 = 0.3333 * sumpix;
      if (bright3 < 0) bright3 = 0;
      if (bright3 > 65535) bright3 = 65535;
      
      if (sa_active && dist < sa_blend) {                                  //  select area is active,
         dnew = 1.0 * dist / sa_blend;                                     //    blend changes over sa_blend
         dold = 1.0 - dnew;
         bright3 = dnew * bright3 + dold * bright1;
      }

      pix3[0] = pix3[1] = pix3[2] = bright3;
   }

   return;
}


/**************************************************************************/

//  convert image to simulate square tiles

int         tile_size, tile_gap;
uint16      *tile_pixmap = 0;


void m_tiles(GtkWidget *, cchar *)                                         //  new  v.6.8
{
   int    tile_dialog_event(zdialog *zd, cchar *event);
   void * tile_thread(void *);

   if (! edit_setup("tiles",0,2)) return;                                  //  setup edit: no preview

   zdedit = zdialog_new(ZTX("Set Tile and Gap Size"),mWin,Bdone,Bcancel,null);
   zdialog_add_widget(zdedit,"hbox","hb1","dialog",0,"space=5");
   zdialog_add_widget(zdedit,"label","labt","hb1",ZTX("tile size"),"space=5");
   zdialog_add_widget(zdedit,"spin","size","hb1","1|99|1|5","space=5");
   zdialog_add_widget(zdedit,"button","apply","hb1",Bapply,"space=10");
   zdialog_add_widget(zdedit,"hbox","hb2","dialog",0,"space=5");
   zdialog_add_widget(zdedit,"label","labg","hb2",ZTX("tile gap"),"space=5");
   zdialog_add_widget(zdedit,"spin","gap","hb2","0|9|1|1","space=5");

   zdialog_run(zdedit,tile_dialog_event);                                  //  start dialog

   tile_size = 5;
   tile_gap = 1;

   tile_pixmap = (uint16 *) zmalloc(E1ww*E1hh*6,"tile.pixmap");            //  set up pixel color map
   memset(tile_pixmap,0,E1ww*E1hh*6);

   start_thread(tile_thread,0);                                            //  start working thread
   return;
}


//  dialog event and completion callback function

int tile_dialog_event(zdialog * zd, cchar *event)
{
   if (zd->zstat)                                                          //  dialog complete
   {
      if (zd->zstat == 1) edit_done();
      else edit_cancel();
      zfree(tile_pixmap);
      return 0;
   }

   if (strEqu(event,"F1")) showz_userguide("tiles");                       //  F1 help       v.9.0
   if (strEqu(event,"undo")) edit_undo();                                  //  v.10.2
   if (strEqu(event,"redo")) edit_redo();                                  //  v.10.3
   if (strEqu(event,"blendwidth")) signal_thread();                        //  v.10.3

   if (strNeq(event,"apply")) return 0;

   zdialog_fetch(zd,"size",tile_size);                                     //  get tile size
   zdialog_fetch(zd,"gap",tile_gap);                                       //  get tile gap 

   if (tile_size < 2) {
      edit_reset();                                                        //  restore original image
      return 0;
   }
   
   signal_thread();                                                        //  trigger working thread
   return 1;
}


//  image tiles thread function

void * tile_thread(void *)
{
   int         sg, gg;
   int         sumpix, red, green, blue;
   int         ii, jj, px, py, qx, qy, dist;
   double      dnew, dold;
   uint16      *pix1, *pix3;

   while (true)
   {
      thread_idle_loop();                                                  //  wait for work or exit request

      sg = tile_size + tile_gap;
      gg = tile_gap;

      for (py = 0; py < E1hh; py += sg)                                    //  initz. pixel color map for
      for (px = 0; px < E1ww; px += sg)                                    //    given pixel size
      {
         sumpix = red = green = blue = 0;

         for (qy = py + gg; qy < py + sg; qy++)                            //  get mean color for pixel block
         for (qx = px + gg; qx < px + sg; qx++)
         {
            if (qy > E1hh-1 || qx > E1ww-1) continue;

            pix1 = bmpixel(E1rgb16,qx,qy);
            red += pix1[0];
            green += pix1[1];
            blue += pix1[2];
            sumpix++;
         }

         if (sumpix) {
            red = (red / sumpix);
            green = (green / sumpix);
            blue = (blue / sumpix);
         }
         
         for (qy = py; qy < py + sg; qy++)                                 //  set color for pixels in block
         for (qx = px; qx < px + sg; qx++)
         {
            if (qy > E1hh-1 || qx > E1ww-1) continue;
            
            jj = (qy * E1ww + qx) * 3;

            if (qx-px < gg || qy-py < gg) {
               tile_pixmap[jj] = tile_pixmap[jj+1] = tile_pixmap[jj+2] = 0;
               continue;
            }

            tile_pixmap[jj] = red;
            tile_pixmap[jj+1] = green;
            tile_pixmap[jj+2] = blue;
         }
      }

      if (sa_active)                                                       //  process selected area
      {
         for (ii = 0; ii < Fww * Fhh; ii++)                                //  find pixels in select area
         {                                                                 //  v.9.6
            if (! sa_pixisin[ii]) continue;
            py = ii / Fww;
            px = ii - py * Fww;
            dist = sa_pixisin[ii];
            pix3 = bmpixel(E3rgb16,px,py);
            jj = (py * E3ww + px) * 3;

            if (dist >= sa_blend) {                                        //  blend changes over sa_blend   v.10.2
               pix3[0] = tile_pixmap[jj];
               pix3[1] = tile_pixmap[jj+1];                                //  apply block color to member pixels
               pix3[2] = tile_pixmap[jj+2];
            }
            else {
               dnew = 1.0 * dist / sa_blend;
               dold = 1.0 - dnew;
               pix1 = bmpixel(E1rgb16,px,py);
               pix3[0] = dnew * tile_pixmap[jj] + dold * pix1[0];
               pix3[1] = dnew * tile_pixmap[jj+1] + dold * pix1[1];
               pix3[2] = dnew * tile_pixmap[jj+2] + dold * pix1[2];
            }
         }
      }

      else                                                                 //  process entire image
      {
         for (py = 0; py < E3hh-1; py++)                                   //  loop all image pixels
         for (px = 0; px < E3ww-1; px++)
         {
            pix3 = bmpixel(E3rgb16,px,py);                                 //  target pixel
            jj = (py * E3ww + px) * 3;                                     //  color map for (px,py)
            pix3[0] = tile_pixmap[jj];
            pix3[1] = tile_pixmap[jj+1];
            pix3[2] = tile_pixmap[jj+2];
         }
      }

      Fmodified = 1;
      mwpaint2();                                                          //  update window
   }

   return 0;                                                               //  not executed, stop g++ warning
}


/**************************************************************************/

//  convert image to simulate a painting                                   //  v.7.0
//  processing a 10 megapixel image needs 140 MB of main memory            //  v.7.3  select area added

namespace paint_names 
{
   int         color_depth;
   int         group_area;
   double      color_match;
   int         borders;

   typedef struct  {
      int16       px, py;
      char        direc;
   }  spixstack;

   int         Nstack;
   spixstack   *pixstack;                                                  //  pixel group search memory
   int         *pixgroup;                                                  //  maps (px,py) to pixel group no.
   int         *groupcount;                                                //  count of pixels in each group

   int         group;
   char        direc;
   uint16      gcolor[3];
}


void m_painting(GtkWidget *, cchar *)
{
   using namespace paint_names;

   int    painting_dialog_event(zdialog *zd, cchar *event);
   void * painting_thread(void *);

   if (! edit_setup("paint",0,2)) return;                                  //  setup edit: no preview

   zdedit = zdialog_new(ZTX("Simulate Painting"),mWin,Bdone,Bcancel,null);

   zdialog_add_widget(zdedit,"hbox","hbcd","dialog",0,"space=1");
   zdialog_add_widget(zdedit,"label","lab1","hbcd",ZTX("color depth"),"space=5");
   zdialog_add_widget(zdedit,"spin","colordepth","hbcd","1|5|1|3","space=5");

   zdialog_add_widget(zdedit,"hbox","hbts","dialog",0,"space=1");
   zdialog_add_widget(zdedit,"label","labts","hbts",ZTX("target group area"),"space=5");
   zdialog_add_widget(zdedit,"spin","grouparea","hbts","0|999|1|100","space=5");

   zdialog_add_widget(zdedit,"hbox","hbcm","dialog",0,"space=1");
   zdialog_add_widget(zdedit,"label","labcm","hbcm",ZTX("req. color match"),"space=5");
   zdialog_add_widget(zdedit,"spin","colormatch","hbcm","0|99|1|50","space=5");

   zdialog_add_widget(zdedit,"hbox","hbbd","dialog",0,"space=1");
   zdialog_add_widget(zdedit,"label","labbd","hbbd",ZTX("borders"),"space=5");
   zdialog_add_widget(zdedit,"check","borders","hbbd",0,"space=2");
   zdialog_add_widget(zdedit,"button","apply","hbbd",Bapply,"space=10");

   zdialog_run(zdedit,painting_dialog_event);                              //  run dialog - parallel

   start_thread(painting_thread,0);                                        //  start working thread
   return;
}


//  dialog event and completion callback function

int painting_dialog_event(zdialog *zd, cchar *event)
{
   using namespace paint_names;

   if (zd->zstat)                                                          //  dialog complete
   {
      if (zd->zstat == 1) edit_done();
      else edit_cancel();
      return 0;
   }

   if (strEqu(event,"F1")) showz_userguide("painting");                    //  F1 context help
   if (strEqu(event,"undo")) edit_undo();                                  //  v.10.2
   if (strEqu(event,"redo")) edit_redo();                                  //  v.10.3
   if (strEqu(event,"blendwidth")) signal_thread();                        //  v.10.3

   if (strEqu(event,"apply")) {                                            //  apply user settings
      zdialog_fetch(zd,"colordepth",color_depth);                          //  color depth
      zdialog_fetch(zd,"grouparea",group_area);                            //  target group area (pixels)
      zdialog_fetch(zd,"colormatch",color_match);                          //  req. color match to combine groups
      zdialog_fetch(zd,"borders",borders);                                 //  borders wanted
      color_match = 0.01 * color_match;                                    //  scale 0 to 1
      signal_thread();
   }
   
   return 0;
}


//  painting thread function

void * painting_thread(void *)
{
   void  paint_colordepth();
   void  paint_pixgroups();
   void  paint_mergegroups();
   void  paint_paintborders();
   void  paint_blend();

   while (true)
   {
      thread_idle_loop();                                                  //  wait for work or exit request
      
      paint_colordepth();                                                  //  set new color depth
      paint_pixgroups();                                                   //  group pixel patches of a color
      paint_mergegroups();                                                 //  merge smaller into larger groups
      paint_paintborders();                                                //  add borders around groups
      paint_blend();                                                       //  blend edges of selected area

      Fmodified = 1;
      mwpaint2();                                                          //  update window
   }

   return 0;                                                               //  not executed, stop g++ warning
}


//  set the specified color depth, 1-5 bits/color

void paint_colordepth()
{
   using namespace paint_names;

   int            ii, px, py, rgb;
   double         fmag;
   uint16         m1, m2, val1, val3;
   uint16         *pix1, *pix3;

   m1 = 0xFFFF << (16 - color_depth);                                      //  5 > 1111100000000000
   m2 = 0x8000 >> color_depth;                                             //  5 > 0000010000000000

   fmag = 65535.0 / m1;                                                    //  full brightness range

   if (sa_active)                                                          //  process select area
   {
      for (ii = 0; ii < Fww * Fhh; ii++)
      {                                                                    //  v.9.6
         if (! sa_pixisin[ii]) continue;
         py = ii / Fww;
         px = ii - py * Fww;

         pix1 = bmpixel(E1rgb16,px,py);                                    //  input pixel
         pix3 = bmpixel(E3rgb16,px,py);                                    //  output pixel

         for (rgb = 0; rgb < 3; rgb++)
         {
            val1 = pix1[rgb];
            if (val1 < m1) val3 = (val1 + m2) & m1;
            else val3 = m1;
            val3 = uint(val3 * fmag);
            pix3[rgb] = val3;
         }
      }
   }
   
   else                                                                    //  process entire image
   {
      for (py = 0; py < E3hh; py++)                                        //  loop all pixels
      for (px = 0; px < E3ww; px++)
      {
         pix1 = bmpixel(E1rgb16,px,py);                                    //  input pixel
         pix3 = bmpixel(E3rgb16,px,py);                                    //  output pixel
         
         for (rgb = 0; rgb < 3; rgb++)
         {
            val1 = pix1[rgb];
            if (val1 < m1) val3 = (val1 + m2) & m1;                        //  round   v.7.0
            else val3 = m1;
            val3 = uint(val3 * fmag);
            pix3[rgb] = val3;
         }
      }
   }

   return;
}


//  find all groups of contiguous pixels with the same color

void paint_pixgroups()
{
   using namespace paint_names;

   void  paint_pushpix(int px, int py);

   int            cc1, cc2;
   int            kk, px, py;
   uint16         *pix3;
   
   cc1 = E3ww * E3hh;

   cc2 = cc1 * sizeof(int);
   pixgroup = (int *) zmalloc(cc2,"pixgroup");                             //  maps pixel to assigned group
   memset(pixgroup,0,cc2);
   
   if (sa_active) cc1 = sa_Npixel;

   cc2 = cc1 * sizeof(spixstack);
   pixstack = (spixstack *) zmalloc(cc2,"pixstack");                       //  memory stack for pixel search
   memset(pixstack,0,cc2);
   
   cc2 = cc1 * sizeof(int);
   groupcount = (int *) zmalloc(cc2,"groupcount");                         //  counts pixels per group
   memset(groupcount,0,cc2);
   
   group = 0;
   
   for (py = 0; py < E3hh; py++)                                           //  loop all pixels
   for (px = 0; px < E3ww; px++)
   {
      kk = py * E3ww + px;
      if (sa_active && ! sa_pixisin[kk]) continue;
      if (pixgroup[kk]) continue;                                          //  already assigned to group

      pixgroup[kk] = ++group;                                              //  assign next group
      ++groupcount[group];

      pix3 = bmpixel(E3rgb16,px,py);
      gcolor[0] = pix3[0];
      gcolor[1] = pix3[1];
      gcolor[2] = pix3[2];

      pixstack[0].px = px;                                                 //  put pixel into stack with
      pixstack[0].py = py;                                                 //    direction = right
      pixstack[0].direc = 'r';
      Nstack = 1;

      while (Nstack)
      {
         kk = Nstack - 1;                                                  //  get last pixel in stack
         px = pixstack[kk].px;
         py = pixstack[kk].py;
         direc = pixstack[kk].direc;
         
         if (direc == 'x') {
            Nstack--;
            continue;
         }
         
         if (direc == 'r') {                                               //  push next right pixel into stack
            paint_pushpix(px,py);                                          //   if no group assigned and if
            pixstack[kk].direc = 'l';                                      //    same color as group color
            continue;
         }

         if (direc == 'l') {                                               //  or next left pixel
            paint_pushpix(px,py);
            pixstack[kk].direc = 'a';
            continue;
         }

         if (direc == 'a') {                                               //  or next ahead pixel
            paint_pushpix(px,py);
            pixstack[kk].direc = 'x';
            continue;
         }
      }
   }
   
   return;
}      


//  push a pixel into the stack memory if it is not already assigned
//  to another group and it has the same color as the current group

void paint_pushpix(int px, int py)
{
   using namespace paint_names;

   int      kk, ppx, ppy, npx, npy;
   uint16   *pix3;

   if (Nstack > 1) {
      kk = Nstack - 2;                                                     //  get prior pixel in stack
      ppx = pixstack[kk].px;
      ppy = pixstack[kk].py;
   }
   else {
      ppx = px - 1;                                                        //  if only one, assume prior = left
      ppy = py;
   }
   
   if (direc == 'r') {                                                     //  get pixel in direction right
      npx = px + ppy - py;
      npy = py + px - ppx;
   }
   else if (direc == 'l') {                                                //  or left
      npx = px + py - ppy;
      npy = py + ppx - px;
   }
   else if (direc == 'a') {                                                //  or ahead
      npx = px + px - ppx;
      npy = py + py - ppy;
   }
   else npx = npy = -1;                                                    //  stop warning
   
   if (npx < 0 || npx >= E3ww) return;                                     //  pixel off the edge
   if (npy < 0 || npy >= E3hh) return;
   
   kk = npy * E3ww + npx;

   if (sa_active)
      if (! sa_pixisin[kk]) return;                                        //  pixel outside area

   if (pixgroup[kk]) return;                                               //  pixel already assigned

   pix3 = bmpixel(E3rgb16,npx,npy);
   if (pix3[0] != gcolor[0] || pix3[1] != gcolor[1]                        //  not same color as group
                            || pix3[2] != gcolor[2]) return;
   
   pixgroup[kk] = group;                                                   //  assign pixel to group
   ++groupcount[group];

   kk = Nstack++;                                                          //  put pixel into stack
   pixstack[kk].px = npx;
   pixstack[kk].py = npy;
   pixstack[kk].direc = 'r';                                               //  direction = right
   
   return;
}


//  merge small pixel groups into adjacent larger groups with best color match

void paint_mergegroups()
{
   using namespace paint_names;

   int         ii, jj, kk, px, py, npx, npy;
   int         nccc, mcount, group2;
   double      ff = 1.0 / 65536.0;
   double      fred, fgreen, fblue, match;
   int         nnpx[4] = {  0, -1, +1, 0 };
   int         nnpy[4] = { -1, 0,  0, +1 };
   uint16      *pix3, *pixN;

   typedef struct  {
      int         group;
      double      match;
      uint16      pixM[3];
   }  snewgroup;

   snewgroup      *newgroup;
   
   nccc = (group + 1) * sizeof(snewgroup);
   newgroup = (snewgroup *) zmalloc(nccc,"newgroup");
   
   if (sa_active)                                                          //  process select area
   {
      while (true)
      {
         memset(newgroup,0,nccc);

         for (ii = 0; ii < Fww * Fhh; ii++)                                //  find pixels in select area
         {                                                                 //  v.9.6
            if (! sa_pixisin[ii]) continue;
            py = ii / Fww;
            px = ii - py * Fww;

            kk = E3ww * py + px;                                           //  get assigned group
            group = pixgroup[kk];
            if (groupcount[group] >= group_area) continue;                 //  group count large enough

            pix3 = bmpixel(E3rgb16,px,py);

            for (jj = 0; jj < 4; jj++)                                     //  get 4 neighbor pixels
            {
               npx = px + nnpx[jj];
               npy = py + nnpy[jj];

               if (npx < 0 || npx >= E3ww) continue;                       //  off the edge
               if (npy < 0 || npy >= E3hh) continue;

               kk = E3ww * npy + npx;
               if (! sa_pixisin[kk]) continue;                             //  pixel outside area
               if (pixgroup[kk] == group) continue;                        //  already in same group
               
               pixN = bmpixel(E3rgb16,npx,npy);                            //  match color of group neighbor
               fred = ff * abs(pix3[0] - pixN[0]);                         //    to color of group
               fgreen = ff * abs(pix3[1] - pixN[1]);
               fblue = ff * abs(pix3[2] - pixN[2]);
               match = (1.0 - fred) * (1.0 - fgreen) * (1.0 - fblue);      //  color match, 0 to 1.0
               if (match < color_match) continue;

               if (match > newgroup[group].match) {
                  newgroup[group].match = match;                           //  remember best match
                  newgroup[group].group = pixgroup[kk];                    //  and corresp. group no.
                  newgroup[group].pixM[0] = pixN[0];                       //  and corresp. new color
                  newgroup[group].pixM[1] = pixN[1];
                  newgroup[group].pixM[2] = pixN[2];
               }
            }
         }

         mcount = 0;

         for (ii = 0; ii < Fww * Fhh; ii++)                                //  find pixels in select area
         {                                                                 //  v.9.6
            if (! sa_pixisin[ii]) continue;
            py = ii / Fww;
            px = ii - py * Fww;

            kk = E3ww * py + px;
            group = pixgroup[kk];                                          //  test for new group assignment
            group2 = newgroup[group].group;
            if (! group2) continue;
            
            if (groupcount[group] > groupcount[group2]) continue;          //  accept only bigger new group

            pixgroup[kk] = group2;                                         //  make new group assignment
            --groupcount[group];
            ++groupcount[group2];

            pix3 = bmpixel(E3rgb16,px,py);                                 //  make new color assignment
            pix3[0] = newgroup[group].pixM[0];
            pix3[1] = newgroup[group].pixM[1];
            pix3[2] = newgroup[group].pixM[2];

            mcount++;
         }
         
         if (mcount == 0) break;
      }
   }

   else                                                                    //  process entire image
   {
      while (true)
      {
         memset(newgroup,0,nccc);

         for (py = 0; py < E3hh; py++)                                     //  loop all pixels
         for (px = 0; px < E3ww; px++)
         {
            kk = E3ww * py + px;                                           //  get assigned group
            group = pixgroup[kk];
            if (groupcount[group] >= group_area) continue;                 //  group count large enough

            pix3 = bmpixel(E3rgb16,px,py);

            for (jj = 0; jj < 4; jj++)                                     //  get 4 neighbor pixels
            {
               npx = px + nnpx[jj];
               npy = py + nnpy[jj];

               if (npx < 0 || npx >= E3ww) continue;                       //  off the edge
               if (npy < 0 || npy >= E3hh) continue;
               
               kk = E3ww * npy + npx;
               if (pixgroup[kk] == group) continue;                        //  in same group

               pixN = bmpixel(E3rgb16,npx,npy);                            //  match color of group neighbor
               fred = ff * abs(pix3[0] - pixN[0]);                         //    to color of group
               fgreen = ff * abs(pix3[1] - pixN[1]);
               fblue = ff * abs(pix3[2] - pixN[2]);
               match = (1.0 - fred) * (1.0 - fgreen) * (1.0 - fblue);      //  color match, 0 to 1.0
               if (match < color_match) continue;

               if (match > newgroup[group].match) {
                  newgroup[group].match = match;                           //  remember best match
                  newgroup[group].group = pixgroup[kk];                    //  and corresp. group no.
                  newgroup[group].pixM[0] = pixN[0];                       //  and corresp. new color
                  newgroup[group].pixM[1] = pixN[1];
                  newgroup[group].pixM[2] = pixN[2];
               }
            }
         }

         mcount = 0;

         for (py = 0; py < E3hh; py++)                                     //  loop all pixels
         for (px = 0; px < E3ww; px++)
         {
            kk = E3ww * py + px;
            group = pixgroup[kk];                                          //  test for new group assignment
            group2 = newgroup[group].group;
            if (! group2) continue;
            
            if (groupcount[group] > groupcount[group2]) continue;          //  accept only bigger new group

            pixgroup[kk] = group2;                                         //  make new group assignment
            --groupcount[group];
            ++groupcount[group2];

            pix3 = bmpixel(E3rgb16,px,py);                                 //  make new color assignment
            pix3[0] = newgroup[group].pixM[0];
            pix3[1] = newgroup[group].pixM[1];
            pix3[2] = newgroup[group].pixM[2];

            mcount++;
         }
         
         if (mcount == 0) break;
      }
   }

   zfree(pixgroup);
   zfree(pixstack);
   zfree(groupcount);
   zfree(newgroup);

   return;
}


//  paint borders between the groups of contiguous pixels

void paint_paintborders()
{
   using namespace paint_names;

   int            ii, kk, px, py, cc;
   uint16         *pix3, *pixL, *pixA;
   
   if (! borders) return;
   
   cc = E3ww * E3hh;
   char * pixblack = zmalloc(cc,"pixblack");
   memset(pixblack,0,cc);

   if (sa_active)
   {
      for (ii = 0; ii < Fww * Fhh; ii++)                                   //  find pixels in select area
      {                                                                    //  v.9.6
         if (! sa_pixisin[ii]) continue;
         py = ii / Fww;
         px = ii - py * Fww;
         if (px < 1 || py < 1) continue;

         pix3 = bmpixel(E3rgb16,px,py);
         pixL = bmpixel(E3rgb16,px-1,py);
         pixA = bmpixel(E3rgb16,px,py-1);
         
         if (pix3[0] != pixL[0] || pix3[1] != pixL[1] || pix3[2] != pixL[2])
         {
            kk = ii - 1;
            if (pixblack[kk]) continue;
            kk += 1;
            pixblack[kk] = 1;
            continue;
         }

         if (pix3[0] != pixA[0] || pix3[1] != pixA[1] || pix3[2] != pixA[2])
         {
            kk = ii - E3ww;
            if (pixblack[kk]) continue;
            kk += E3ww;
            pixblack[kk] = 1;
         }
      }

      for (ii = 0; ii < Fww * Fhh; ii++)                                   //  find pixels in select area
      {                                                                    //  v.9.6
         if (! sa_pixisin[ii]) continue;
         py = ii / Fww;
         px = ii - py * Fww;
         if (px < 1 || py < 1) continue;

         if (! pixblack[ii]) continue;
         pix3 = bmpixel(E3rgb16,px,py);
         pix3[0] = pix3[1] = pix3[2] = 0;
      }
   }
         
   else
   {
      for (py = 1; py < E3hh; py++)                                        //  loop all pixels
      for (px = 1; px < E3ww; px++)                                        //  omit top and left
      {
         pix3 = bmpixel(E3rgb16,px,py);                                    //  output pixel
         pixL = bmpixel(E3rgb16,px-1,py);                                  //  pixel to left
         pixA = bmpixel(E3rgb16,px,py-1);                                  //  pixel above
         
         if (pix3[0] != pixL[0] || pix3[1] != pixL[1] || pix3[2] != pixL[2])
         {
            kk = E3ww * py + px-1;                                         //  have horiz. transition
            if (pixblack[kk]) continue;
            kk += 1;
            pixblack[kk] = 1;
            continue;
         }

         if (pix3[0] != pixA[0] || pix3[1] != pixA[1] || pix3[2] != pixA[2])
         {
            kk = E3ww * (py-1) + px;                                       //  have vertical transition
            if (pixblack[kk]) continue;
            kk += E3ww;
            pixblack[kk] = 1;
         }
      }

      for (py = 1; py < E3hh; py++)
      for (px = 1; px < E3ww; px++)
      {
         kk = E3ww * py + px;
         if (! pixblack[kk]) continue;
         pix3 = bmpixel(E3rgb16,px,py);
         pix3[0] = pix3[1] = pix3[2] = 0;
      }
   }
   
   zfree(pixblack);
   return;
}


//  blend edges of selected area

void paint_blend()
{
   int         ii, px, py, rgb, dist;
   uint16      *pix1, *pix3;
   double      f1, f2;
   
   if (sa_active && sa_blend > 0)
   {
      for (ii = 0; ii < Fww * Fhh; ii++)                                   //  find pixels in select area
      {                                                                    //  v.9.6
         dist = sa_pixisin[ii];
         if (! dist || dist >= sa_blend) continue;

         py = ii / Fww;
         px = ii - py * Fww;
         pix1 = bmpixel(E1rgb16,px,py);                                    //  input pixel
         pix3 = bmpixel(E3rgb16,px,py);                                    //  output pixel

         f2 = 1.0 * dist / sa_blend;                                       //  changes over distance sa_blend
         f1 = 1.0 - f2;

         for (rgb = 0; rgb < 3; rgb++)
            pix3[rgb] = int(f1 * pix1[rgb] + f2 * pix3[rgb]);
      }
   }

   return;
}


/**************************************************************************/

//  pixel edit function - edit individual pixels

void  pixed_mousefunc();
void  pixed_dopixels(int px, int py);
void  pixed_saveundo(int px, int py);
void  pixed_undo1();
void  pixed_freeundo();

int      pixed_RGB[3];
int      pixed_mode;
int      pixed_suspend;
int      pixed_radius;
double   pixed_kernel[200][200];                                           //  radius <= 99

int      pixed_undototpix = 0;                                             //  total undo pixel blocks
int      pixed_undototmem = 0;                                             //  total undo memory allocated
int      pixed_undoseq = 0;                                                //  undo sequence no.
char     pixed_undomemmessage[100];                                        //  translated undo memory message

typedef struct {                                                           //  pixel block before edit
   int         seq;                                                        //  undo sequence no.
   uint16      npix;                                                       //  no. pixels in this block
   uint16      px, py;                                                     //  center pixel (radius org.)
   uint16      radius;                                                     //  radius of pixel block
   uint16      pixel[][3];                                                 //  array of pixel[npix][3] 
}  pixed_savepix;

pixed_savepix   **pixed_undopixmem = 0;                                    //  array of *pixed_savepix


void m_pixedit(GtkWidget *, cchar *)
{
   int   pixed_dialog_event(zdialog* zd, cchar *event);

   char        undomemmessage[100];

   if (! edit_setup("pixedit",0,1)) return;                                //  setup edit: no preview

   strncpy0(pixed_undomemmessage,ZTX("Undo Memory %d%c"),99);              //  translate undo memory message

   zdedit = zdialog_new(ZTX("Edit Pixels"),mWin,Bdone,Bcancel,null);
   zdialog_add_widget(zdedit,"hbox","hbc","dialog",0,"space=5");
   zdialog_add_widget(zdedit,"label","labc","hbc",ZTX("color"),"space=8");
   zdialog_add_widget(zdedit,"colorbutt","color","hbc","100|100|100");
   zdialog_add_widget(zdedit,"label","space","hbc",0,"space=10");
   zdialog_add_widget(zdedit,"radio","radio1","hbc",ZTX("pick"),"space=3");
   zdialog_add_widget(zdedit,"radio","radio2","hbc",ZTX("paint"),"space=3");
   zdialog_add_widget(zdedit,"radio","radio3","hbc",ZTX("erase"),"space=3");
   zdialog_add_widget(zdedit,"hbox","hbbr","dialog",0,"space=5");
   zdialog_add_widget(zdedit,"vbox","vbbr1","hbbr",0,"homog|space=3");
   zdialog_add_widget(zdedit,"vbox","vbbr2","hbbr",0,"homog|space=3");
   zdialog_add_widget(zdedit,"label","space","hbbr",0,"space=10");
   zdialog_add_widget(zdedit,"vbox","vbbr3","hbbr",0,"homog|space=3");
   zdialog_add_widget(zdedit,"hbox","hbrad","vbbr1",0,"space=3");
   zdialog_add_widget(zdedit,"label","space","hbrad",0,"expand");
   zdialog_add_widget(zdedit,"label","labbr","hbrad",ZTX("paintbrush radius"));
   zdialog_add_widget(zdedit,"label","labtc","vbbr1",ZTX("transparency center"));
   zdialog_add_widget(zdedit,"label","labte","vbbr1",ZTX("transparency edge"));
   zdialog_add_widget(zdedit,"spin","radius","vbbr2","1|99|1|2");
   zdialog_add_widget(zdedit,"spin","trcent","vbbr2","0|99|1|60");
   zdialog_add_widget(zdedit,"spin","tredge","vbbr2","0|99|1|99");
   zdialog_add_widget(zdedit,"button","susp-resm","vbbr3",Bsuspend);
   zdialog_add_widget(zdedit,"button","undlast","vbbr3",Bundolast);
   zdialog_add_widget(zdedit,"button","undall","vbbr3",Bundoall);
   zdialog_add_widget(zdedit,"label","labmem","dialog");

   zdialog_run(zdedit,pixed_dialog_event);                                 //  run dialog - parallel

   zdialog_send_event(zdedit,"radius");                                    //  get kernel initialized

   snprintf(undomemmessage,99,pixed_undomemmessage,0,'%');                 //  stuff undo memory status
   zdialog_stuff(zdedit,"labmem",undomemmessage);
   
   pixed_RGB[0] = pixed_RGB[1] = pixed_RGB[2] = 100;                       //  initialize color
   
   pixed_mode = 1;                                                         //  mode = pick color
   pixed_suspend = 0;                                                      //  not suspended

   pixed_undopixmem = 0;                                                   //  no undo data
   pixed_undototpix = 0;
   pixed_undototmem = 0;
   pixed_undoseq = 0;

   mouseCBfunc = pixed_mousefunc;                                          //  connect mouse function
   Mcapture = 1;                                                           //  capture mouse clicks
   set_cursor(drawcursor);                                                 //  set draw cursor
   return;
}


//  dialog event and completion callback function

int pixed_dialog_event(zdialog *zd, cchar *event)                          //  pixedit dialog event function
{
   char        color[20];
   cchar       *pp;
   int         radius, dx, dy, brad;
   double      rad, kern, trcent, tredge;
   
   if (zd->zstat) goto complete;

   if (strEqu(event,"F1")) showz_userguide("edit_pixels");                 //  F1 context help
   if (strEqu(event,"undo")) edit_undo();                                  //  v.10.2
   if (strEqu(event,"redo")) edit_redo();                                  //  v.10.3

   zdialog_fetch(zd,"radio1",brad);                                        //  pick       v.6.8
   if (brad) pixed_mode = 1;
   zdialog_fetch(zd,"radio2",brad);                                        //  paint
   if (brad) pixed_mode = 2;
   zdialog_fetch(zd,"radio3",brad);                                        //  erase
   if (brad) pixed_mode = 3;
   
   if (strEqu(event,"color")) 
   {
      zdialog_fetch(zd,"color",color,19);                                  //  get color from color wheel
      pp = strField(color,"|",1);
      if (pp) pixed_RGB[0] = atoi(pp);
      pp = strField(color,"|",2);
      if (pp) pixed_RGB[1] = atoi(pp);
      pp = strField(color,"|",3);
      if (pp) pixed_RGB[2] = atoi(pp);
   }
   
   if (strstr("radius trcent tredge",event))                               //  get new brush attributes
   {
      zdialog_fetch(zd,"radius",radius);                                   //  radius
      zdialog_fetch(zd,"trcent",trcent);                                   //  center transparency       v.7.8
      zdialog_fetch(zd,"tredge",tredge);                                   //  edge transparency

      pixed_radius = radius;
      trcent = 0.01 * trcent;                                              //  scale 0 ... 1
      tredge = 0.01 * tredge;
      tredge = (1 - trcent) * (1 - tredge);
      tredge = 1 - tredge;
      trcent = sqrt(trcent);                                               //  speed up the curve
      tredge = sqrt(tredge);

      for (dy = -radius; dy <= radius; dy++)                               //  build kernel
      for (dx = -radius; dx <= radius; dx++)
      {
         rad = sqrt(dx*dx + dy*dy);
         kern = (radius - rad) / radius;                                   //  1 ... 0 
         kern = kern * (trcent - tredge) + tredge;                         //  trcent ... tredge
         if (rad > radius) kern = 1;
         if (kern < 0) kern = 0;
         if (kern > 1) kern = 1;
         pixed_kernel[dx+radius][dy+radius] = kern;
      }
   }
   
   if (strEqu(event,"undlast"))                                            //  undo last edit (click or drag)
      pixed_undo1();                                                       //  v.7.8

   if (strEqu(event,"undall")) {                                           //  undo all edits      v.7.8
      edit_reset();                                                        //  v.10.3
      pixed_freeundo();
   }

   if (strEqu(event,"susp-resm"))                                          //  toggle suspend / resume   v.7.8
   {
      if (pixed_suspend) {
         pixed_suspend = 0;
         mouseCBfunc = pixed_mousefunc;                                    //  connect mouse function
         Mcapture++;
         set_cursor(drawcursor);                                           //  set draw cursor
         zdialog_stuff(zd,"susp-resm",Bsuspend);
      }
      else  {
         pixed_suspend = 1;
         mouseCBfunc = 0;                                                  //  disconnect mouse function
         Mcapture = 0;
         set_cursor(0);                                                    //  restore normal cursor
         zdialog_stuff(zd,"susp-resm",Bresume);
      }
   }

   return 1;

complete:

   if (zd->zstat == 1) edit_done();                                        //  done
   else edit_cancel();                                                     //  cancel or destroy

   paint_toparc(2);                                                        //  remove brush outline      v.8.3
   Mcapture = 0;                                                           //  disconnect mouse
   mouseCBfunc = 0;
   set_cursor(0);                                                          //  restore normal cursor
   pixed_freeundo();                                                       //  free undo memory
   return 0;
}


//  pixel edit mouse function

void pixed_mousefunc()
{
   static int  pmxdown = 0, pmydown = 0;
   int         px, py;
   char        color[20];
   uint16      *ppix3;
   
   toparcx = Mxposn - pixed_radius;                                        //  define brush outline circle
   toparcy = Myposn - pixed_radius;                                        //  v.8.3
   toparcw = toparch = 2 * pixed_radius;
   if (pixed_mode == 1) Ftoparc = 0;
   else Ftoparc = 1;
   if (Ftoparc) paint_toparc(3);
   
   if (LMclick)                                                            //  left mouse click
   {
      LMclick = 0;
      px = Mxclick;
      py = Myclick;

      if (pixed_mode == 1)                                                 //  pick new color from image
      {
         ppix3 = bmpixel(E3rgb16,px,py);
         pixed_RGB[0] = ppix3[0] / 256;
         pixed_RGB[1] = ppix3[1] / 256;
         pixed_RGB[2] = ppix3[2] / 256;
         snprintf(color,19,"%d|%d|%d",pixed_RGB[0],pixed_RGB[1],pixed_RGB[2]);
         if (zdedit) zdialog_stuff(zdedit,"color",color);
      }
      else {                                                               //  paint or erase
         pixed_undoseq++;                                                  //  new undo seq. no.
         pixed_saveundo(px,py);                                            //  save for poss. undo
         pixed_dopixels(px,py);                                            //  do 1 block of pixels
      }
   }
   
   if (Mxdrag || Mydrag)                                                   //  drag in progress
   {
      px = Mxdrag;
      py = Mydrag;
      Mxdrag = Mydrag = 0;

      if (Mxdown != pmxdown || Mydown != pmydown) {                        //  new drag
         pixed_undoseq++;                                                  //  new undo seq. no.
         pmxdown = Mxdown;
         pmydown = Mydown;
      }
      pixed_saveundo(px,py);                                               //  save for poss. undo
      pixed_dopixels(px,py);                                               //  do 1 block of pixels
   }
   
   return;
}


//  paint or erase 1 block of pixels within radius of px, py

void pixed_dopixels(int px, int py)
{
   uint16      *ppix1, *ppix3;
   int         radius, dx, dy;
   int         red, green, blue;
   double      kern;

   edit_zapredo();                                                         //  delete redo copy    v.10.3

   radius = pixed_radius;

   red = 256 * pixed_RGB[0];
   green = 256 * pixed_RGB[1];
   blue = 256 * pixed_RGB[2];

   for (dy = -radius; dy <= radius; dy++)                                  //  loop surrounding block of pixels
   for (dx = -radius; dx <= radius; dx++)
   {
      if (px + dx < 0 || px + dx > E3ww-1) continue;                       //  v.7.5
      if (py + dy < 0 || py + dy > E3hh-1) continue;
      
      kern = pixed_kernel[dx+radius][dy+radius];
      ppix1 = bmpixel(E1rgb16,(px+dx),(py+dy));                            //  original image pixel
      ppix3 = bmpixel(E3rgb16,(px+dx),(py+dy));                            //  edited image pixel

      if (pixed_mode == 2)                                                 //  color pixels transparently
      {
         ppix3[0] = (1.0 - kern) * red   + kern * ppix3[0];
         ppix3[1] = (1.0 - kern) * green + kern * ppix3[1];
         ppix3[2] = (1.0 - kern) * blue  + kern * ppix3[2];
         Fmodified = 1;
      }

      if (pixed_mode == 3)                                                 //  restore org. pixels transparently
      {
         ppix3[0] = (1.0 - kern) * ppix1[0] + kern * ppix3[0];
         ppix3[1] = (1.0 - kern) * ppix1[1] + kern * ppix3[1];
         ppix3[2] = (1.0 - kern) * ppix1[2] + kern * ppix3[2];
      }
   }

   mwpaint2();
   return;
}


//  save 1 block of pixels for possible undo

void pixed_saveundo(int px, int py)
{
   int            npix, radius, dx, dy;
   uint16         *ppix3;
   pixed_savepix  *ppixsave1;
   char           undomemmessage[100];
   int            mempercent;
   static int     ppercent = 0;

   if (! pixed_undopixmem)                                                 //  first call
   {
      pixed_undopixmem = (pixed_savepix **) zmalloc(pixed_undomaxpix * sizeof(void *),"pixed.undomem");
      pixed_undototpix = 0;
      pixed_undototmem = 0;
   }
   
   if (pixed_undototmem > pixed_undomaxmem) 
   {
      zmessageACK(GTK_WINDOW(mWin),
                  ZTX("Undo memory limit has been reached (100 MB). \n"
                      "Save work with [done], then resume editing."));
      Mdrag = 0;                                                           //  stop mouse drag   v.8.3
      return;
   }

   radius = pixed_radius;
   npix = 0;

   for (dy = -radius; dy <= radius; dy++)                                  //  count pixels in block
   for (dx = -radius; dx <= radius; dx++)
   {
      if (px + dx < 0 || px + dx > E3ww-1) continue;
      if (py + dy < 0 || py + dy > E3hh-1) continue;
      npix++;
   }
   
   ppixsave1 = (pixed_savepix *) zmalloc(npix*6+12,"pixed.pixsave");       //  allocate memory for block
   pixed_undopixmem[pixed_undototpix] = ppixsave1;
   pixed_undototpix += 1;
   pixed_undototmem += npix * 6 + 12;
   
   ppixsave1->seq = pixed_undoseq;                                         //  save pixel block poop
   ppixsave1->npix = npix;
   ppixsave1->px = px;
   ppixsave1->py = py;
   ppixsave1->radius = radius;

   npix = 0;

   for (dy = -radius; dy <= radius; dy++)                                  //  save pixels in block
   for (dx = -radius; dx <= radius; dx++)
   {
      if (px + dx < 0 || px + dx > E3ww-1) continue;
      if (py + dy < 0 || py + dy > E3hh-1) continue;
      ppix3 = bmpixel(E3rgb16,(px+dx),(py+dy));                            //  edited image pixel
      ppixsave1->pixel[npix][0] = ppix3[0];
      ppixsave1->pixel[npix][1] = ppix3[1];
      ppixsave1->pixel[npix][2] = ppix3[2];
      npix++;
   }

   mempercent = int(100.0 * pixed_undototmem / pixed_undomaxmem);          //  update undo memory status
   if (mempercent != ppercent) {
      ppercent = mempercent;
      snprintf(undomemmessage,99,pixed_undomemmessage,mempercent,'%');
      zdialog_stuff(zdedit,"labmem",undomemmessage);
   }

   return;
}


//  undo last undo sequence number

void pixed_undo1()
{
   int            pindex, npix, radius, px, py, dx, dy;
   uint16         *ppix3;
   pixed_savepix  *ppixsave1;
   char           undomemmessage[100];
   int            mempercent;
   
   pindex = pixed_undototpix;
   
   while (pindex > 0)
   {
      --pindex;
      ppixsave1 = pixed_undopixmem[pindex];
      if (ppixsave1->seq != pixed_undoseq) break;
      px = ppixsave1->px;
      py = ppixsave1->py;
      radius = ppixsave1->radius;

      npix = 0;

      for (dy = -radius; dy <= radius; dy++)
      for (dx = -radius; dx <= radius; dx++)
      {
         if (px + dx < 0 || px + dx > E3ww-1) continue;
         if (py + dy < 0 || py + dy > E3hh-1) continue;
         ppix3 = bmpixel(E3rgb16,(px+dx),(py+dy));
         ppix3[0] = ppixsave1->pixel[npix][0];
         ppix3[1] = ppixsave1->pixel[npix][1];
         ppix3[2] = ppixsave1->pixel[npix][2];
         npix++;
      }

      npix = ppixsave1->npix;
      zfree(ppixsave1);
      pixed_undopixmem[pindex] = 0;
      pixed_undototmem -= (npix * 6 + 12);
      --pixed_undototpix;
   }
   
   if (pixed_undoseq > 0) --pixed_undoseq;

   mempercent = int(100.0 * pixed_undototmem / pixed_undomaxmem);          //  update undo memory status
   snprintf(undomemmessage,99,pixed_undomemmessage,mempercent,'%');
   zdialog_stuff(zdedit,"labmem",undomemmessage);

   mwpaint2();
   return;
}


//  free all undo memory

void pixed_freeundo()
{
   int            pindex;
   pixed_savepix  *ppixsave1;
   char           undomemmessage[100];

   pindex = pixed_undototpix;
   
   while (pindex > 0)
   {
      --pindex;
      ppixsave1 = pixed_undopixmem[pindex];
      zfree(ppixsave1);
   }
   
   if (pixed_undopixmem) zfree(pixed_undopixmem);
   pixed_undopixmem = 0;
   
   pixed_undoseq = 0;
   pixed_undototpix = 0;
   pixed_undototmem = 0;

   if (zdedit) {
      snprintf(undomemmessage,99,pixed_undomemmessage,0,'%');              //  undo memory = 0%
      zdialog_stuff(zdedit,"labmem",undomemmessage);
   }

   return;
}


/**************************************************************************

   Make an HDR (high dynamic range) image from several images of the same
   subject with different exposure levels. The composite image has better
   visibility of detail in both the brightest and darkest areas.

   New version allows up to 10 images to be combined.          v.9.0

***************************************************************************/

void * HDR_align2_thread(void *);                                          //  align 2 images
int    HDR_combine2();                                                     //  combine 2 images
int    HDR_brightness();                                                   //  compute pixel brightness levels
int    HDR_dialog();                                                       //  adjust image contribution curves
void * HDR_combine_thread(void *);                                         //  combine images per contribution curves

int      HDRNF = 0;                                                        //  no. image files, 2-10
char     *HDRfile[10];                                                     //  image filespecs
RGB      *HDRrgb[10];                                                      //  RGB pixmaps
double   HDRxoff[10], HDRyoff[10], HDRtoff[10];                            //  alignment offsets
RGB      *HDRrgb1, *HDRrgb2;                                               //  alignment pixmaps
float    *HDRbright = 0;                                                   //  maps brightness per pixel


void m_HDR(GtkWidget *, cchar *)                                           //  menu function
{
   char     **flist, *ftemp;
   int      ii, jj, px, py, stat;
   double   fbright[10], btemp;
   double   pixsum, fnorm = 3.0 / 65536.0;
   double   maxtoff, mintoff, midtoff;
   uint16   *pixel;
   RGB      *rgbtemp;

   if (! edit_setup("HDR",0,0)) return;                                    //  setup edit: no preview

   HDRNF = 0;
   flist = 0;
   HDRbright = 0;

   for (ii = 0; ii < 10; ii++)                                             //  clear file and pixmap lists
   {
      HDRfile[ii] = 0;
      HDRrgb[ii] = 0;
   }

   zfuncs::F1_help_topic = "HDR";                                          //  v.9.0

   flist = zgetfiles(ZTX("Select 2 to 10 files to combine"),image_file);   //  select images to combine
   if (! flist) goto cancel;                                               //  (current image pre-selected)

   for (ii = 0; flist[ii]; ii++);                                          //  count selected files

   if (ii < 2 || ii > 10) {
      zmessageACK(GTK_WINDOW(mWin),ZTX("Select between 2 and 10 files to combine"));
      goto cancel;
   }
   
   HDRNF = ii;
   for (ii = 0; ii < HDRNF; ii++)                                          //  set up file list
      HDRfile[ii] = flist[ii];

   for (ii = 0; ii < HDRNF; ii++) 
      if (strEqu(HDRfile[ii],image_file)) break;
   if (ii == HDRNF) {
      zmessageACK(GTK_WINDOW(mWin),ZTX("Current file must be included"));
      goto cancel;
   }

   for (ii = 0; ii < HDRNF; ii++)                                          //  set up pixmap list
   {
      HDRrgb[ii] = f_load(HDRfile[ii],16);
      if (! HDRrgb[ii]) goto cancel;
      if (HDRrgb[ii]->ww != Fww || HDRrgb[ii]->hh != Fhh) {
         zmessageACK(GTK_WINDOW(mWin),ZTX("Images are not all the same size"));
         goto cancel;
      }
   }

   for (ii = 0; ii < HDRNF; ii++)                                          //  compute image brightness levels
   {
      pixsum = 0;
      for (py = 0; py < Fhh; py++)
      for (px = 0; px < Fww; px++)
      {
         pixel = bmpixel(HDRrgb[ii],px,py);
         pixsum += fnorm * (pixel[0] + pixel[1] + pixel[2]);
      }
      fbright[ii] = pixsum / (Fww * Fhh);
   }
   
   for (ii = 0; ii < HDRNF; ii++)                                          //  sort file and pixmap lists
   for (jj = ii+1; jj < HDRNF; jj++)                                       //    by decreasing brightness
   {
      if (fbright[jj] > fbright[ii]) {                                     //  bubble sort
         btemp = fbright[jj];
         fbright[jj] = fbright[ii];
         fbright[ii] = btemp;
         ftemp = HDRfile[jj];
         HDRfile[jj] = HDRfile[ii];
         HDRfile[ii] = ftemp;
         rgbtemp = HDRrgb[jj];
         HDRrgb[jj] = HDRrgb[ii];
         HDRrgb[ii] = rgbtemp;
      }
   }
   
   for (ii = 1; ii < HDRNF; ii++)                                          //  align each image to prior image
   {
      HDRrgb1 = HDRrgb[ii-1];                                              //  2 images to align
      HDRrgb2 = HDRrgb[ii];
      start_thread(HDR_align2_thread,0);                                   //  start align thread
      wrapup_thread(0);                                                    //  wait for completion
      HDRxoff[ii] = xoff;
      HDRyoff[ii] = yoff;
      HDRtoff[ii] = 2 * toff;                                              //  images rotated +/- toff
   }
   
   HDRxoff[0] = HDRyoff[0] = HDRtoff[0] = 0;                               //  image 0 offsets = 0

   for (ii = 2; ii < HDRNF; ii++)                                          //  make all alignment offsets
   {                                                                       //    relative to image 0
      HDRxoff[ii] += HDRxoff[ii-1];
      HDRyoff[ii] += HDRyoff[ii-1];
      HDRtoff[ii] += HDRtoff[ii-1];
   }
   
   maxtoff = mintoff = 0;                                                  //  balance +/- theta offsets
   for (ii = 1; ii < HDRNF; ii++) {
      if (HDRtoff[ii] > maxtoff) maxtoff = HDRtoff[ii];
      if (HDRtoff[ii] < mintoff) mintoff = HDRtoff[ii];
   }
   midtoff = 0.5 * (maxtoff + mintoff);
   for (ii = 0; ii < HDRNF; ii++) HDRtoff[ii] -= midtoff;
   
   stat = HDR_brightness();                                                //  compute pixel brightness levels
   if (! stat) goto cancel;
   
   stat = HDR_dialog();                                                    //  combine images based on user inputs
   goto cleanup;

cancel:
   edit_cancel();

cleanup:
   if (flist) zfree(flist);                                                //  free memory
   for (ii = 0; ii < HDRNF; ii++) {
      if (HDRfile[ii]) zfree(HDRfile[ii]);
      if (HDRrgb[ii]) RGB_free(HDRrgb[ii]);
   }
   if (HDRbright) zfree(HDRbright);

   return;
}


//  HDR align two input images, output combined image to E3rgb16

void * HDR_align2_thread(void *)
{
   double      xfL, xfH, yfL, yfH, tfL, tfH;
   double      xystep, xylim, tstep, tlim;
   int         firstpass, lastpass;

   Radjust = Gadjust = Badjust = 1.0;                                      //  no manual color adjustments
   xshrink = yshrink = 0;                                                  //  no image shrinkage (pano)
   warpxu = warpyu = warpxl = warpyl = 0;                                  //  no warp factors (pano)
   warpxuB = warpyuB = warpxlB = warpylB = 0;

   Nalign = 1;                                                             //  alignment in progress
   aligntype = 1;                                                          //  HDR
   showRedpix = 1;
   pixsamp = 5000;                                                         //  pixel sample size
   Fzoom = 0;                                                              //  fit to window if big
   Fblowup = 1;                                                            //  scale up to window if small
   firstpass = 1;
   lastpass = 0;
   
   fullSize = Fww;                                                         //  full image size
   if (Fhh > Fww) fullSize = Fhh;                                          //  (largest dimension)

   alignSize = 140;                                                        //  initial alignment image size
   if (alignSize > fullSize) alignSize = fullSize;
   A1rgb16 = A2rgb16 = 0;

   xoff = yoff = toff = 0;                                                 //  initial offsets = 0

   while (true)                                                            //  next alignment stage / image size
   {   
      A1ww = Fww * alignSize / fullSize;                                   //  align width, height in same ratio
      A1hh = Fhh * alignSize / fullSize;
      A2ww = A1ww;
      A2hh = A1hh;
      
      if (! lastpass) 
      {
         RGB_free(A1rgb16);                                                //  align images = scaled input images
         RGB_free(A2rgb16);
         A1rgb16 = RGB_rescale(HDRrgb1,A1ww,A1hh);
         A2rgb16 = RGB_rescale(HDRrgb2,A2ww,A2hh);
         
         alignWidth = A1ww;                                                //  use full image for alignment
         alignHeight = A1hh;                                               //  v.8.0

         getAlignArea();                                                   //  get image overlap area
         getBrightRatios();                                                //  get color brightness ratios
         setColorfixFactors(1);                                            //  set color matching factors
         flagEdgePixels();                                                 //  flag high-contrast pixels

         mutex_lock(&pixmaps_lock);
         RGB_free(E3rgb16);                                                //  resize output image
         E3rgb16 = RGB_make(A1ww,A1hh,16);
         E3ww = A1ww;
         E3hh = A1hh;
         mutex_unlock(&pixmaps_lock);
      }

      xylim = 2;                                                           //  search range from prior stage:
      xystep = 1;                                                          //    -2 -1 0 +1 +2 pixels

      if (firstpass) xylim = 0.05 * alignSize;                             //  1st stage search range, huge

      if (lastpass) {
         xylim = 1;                                                        //  final stage search range:
         xystep = 0.5;                                                     //    -1.0 -0.5 0.0 +0.5 +1.0
      }

      tlim = xylim / alignSize / 2;                                        //  theta max offset, radians
      tstep = xystep / alignSize / 2;                                      //  theta step size

      xfL = xoff - xylim;
      xfH = xoff + xylim + xystep/2;
      yfL = yoff - xylim;
      yfH = yoff + xylim + xystep/2;
      tfL = toff - tlim;
      tfH = toff + tlim + tstep/2;

      xoffB = xoff;
      yoffB = yoff;
      toffB = toff;
      
      matchB = matchImages();                                              //  set base match level

      if (firstpass) HDR_combine2();                                       //  combine images >> E3rgb16

      for (xoff = xfL; xoff < xfH; xoff += xystep)                         //  test all offset dimensions
      for (yoff = yfL; yoff < yfH; yoff += xystep)                         //    in all combinations
      for (toff = tfL; toff < tfH; toff += tstep)
      {
         matchlev = matchImages();
         if (sigdiff(matchlev,matchB,0.00001) > 0) {
            matchB = matchlev;
            xoffB = xoff;
            yoffB = yoff;
            toffB = toff;
         }

         Nalign++;                                                         //  count alignment tests
      }

      xoff = xoffB;                                                        //  recover best offsets
      yoff = yoffB;
      toff = toffB;

      HDR_combine2();                                                      //  combine images >> E3rgb16
     
      firstpass = 0;
      if (lastpass) break;                                                 //  done

      if (alignSize == fullSize) {                                         //  full size image was aligned
         lastpass++;                                                       //  one more pass
         continue;                                                         //  no image size change
      }

      double R = alignSize;
      alignSize = 2 * alignSize;                                           //  next larger image size
      if (alignSize > 0.8 * fullSize) alignSize = fullSize;                //  if near goal, jump to it now
      R = alignSize / R;                                                   //  ratio of new / old image size
      xoff = R * xoff;                                                     //  adjust offsets for image size
      yoff = R * yoff;
   }

   RGB_free(A1rgb16);                                                      //  free alignment images
   RGB_free(A2rgb16);
   Nalign = 0;                                                             //  reset align counter
   showRedpix = 0;
   Fblowup = 0;                                                            //  reset forced image scaling
   exit_thread();
   return 0;                                                               //  not executed
}


//  Combine images A1rgb16 and A2rgb16 >> E3rgb16 (not reallocated).
//  Update window from E3rgb16.

int HDR_combine2()
{
   int         px3, py3, ii, vstat1, vstat2;
   double      px1, py1, px2, py2;
   double      sintf = sin(toff), costf = cos(toff);
   uint16      vpix1[3], vpix2[3], *pix3;
   
   for (py3 = 0; py3 < A1hh; py3++)                                        //  step through A1rgb16 pixels
   for (px3 = 0; px3 < A1ww; px3++)
   {
      px1 = costf * px3 - sintf * (py3 - yoff);                            //  A1rgb16 pixel, after offsets
      py1 = costf * py3 + sintf * (px3 - xoff);
      vstat1 = vpixel(A1rgb16,px1,py1,vpix1);

      px2 = costf * (px3 - xoff) + sintf * (py3 - yoff);                   //  corresponding A2rgb16 pixel
      py2 = costf * (py3 - yoff) - sintf * (px3 - xoff);
      vstat2 = vpixel(A2rgb16,px2,py2,vpix2);

      pix3 = bmpixel(E3rgb16,px3,py3);                                     //  output pixel

      if (! vstat1 || ! vstat2) {                                          //  if non-overlapping pixel,
         pix3[0] = pix3[1] = pix3[2] = 0;                                  //    set output pixel black
         continue;
      }
      
      pix3[0] = (vpix1[0] + vpix2[0]) / 2;                                 //  output pixel is simple average
      pix3[1] = (vpix1[1] + vpix2[1]) / 2;
      pix3[2] = (vpix1[2] + vpix2[2]) / 2;

      if (showRedpix) {
         ii = py3 * A1ww + px3;                                            //  highlight alignment pixels
         if (redpixels[ii]) {
            pix3[0] = 65535;
            pix3[1] = pix3[2] = 0;
         }
      }
   }

   mwpaint2();                                                             //  update window
   return 1;
}


//  Compute mean image pixel brightness levels.
//  (basis for setting image contributions per brightness level)

int HDR_brightness()
{
   int         px3, py3, ii, kk, vstat;
   double      px, py, red, green, blue;
   double      bright, maxbright, minbright;
   double      sintoff[10], costoff[10], sintf, costf;
   double      fnorm = 1.0 / 65536.0;
   uint16      vpix[3], *pix3;
   RGB         *rgb;
   
   for (ii = 0; ii < HDRNF; ii++)                                          //  pre-calculate trig functions
   {
      sintoff[ii] = sin(HDRtoff[ii]);
      costoff[ii] = cos(HDRtoff[ii]);
   }
   
   HDRbright = (float *) zmalloc(Fww*Fhh*sizeof(int),"hdr.brightmem");     //  get memory for brightness array
   
   minbright = 1.0;
   maxbright = 0.0;
   
   for (py3 = 0; py3 < Fhh; py3++)                                         //  step through all output pixels
   for (px3 = 0; px3 < Fww; px3++)
   {
      red = green = blue = 0;
      vstat = 0;

      for (ii = 0; ii < HDRNF; ii++)                                       //  step through all input images
      {
         rgb = HDRrgb[ii];

         xoff = HDRxoff[ii];
         yoff = HDRyoff[ii];
         sintf = sintoff[ii];
         costf = costoff[ii];

         px = costf * (px3 - xoff) + sintf * (py3 - yoff);                 //  image N pixel, after offsets
         py = costf * (py3 - yoff) - sintf * (px3 - xoff);
         vstat = vpixel(rgb,px,py,vpix);
         if (! vstat) break;
         
         red += fnorm * vpix[0];                                           //  sum input pixels
         green += fnorm * vpix[1];
         blue += fnorm * vpix[2];
      }
      
      if (! vstat) {                                                       //  pixel outside some image
         pix3 = bmpixel(E3rgb16,px3,py3);                                  //  output pixel = black
         pix3[0] = pix3[1] = pix3[2] = 0;
         kk = py3 * Fww + px3;
         HDRbright[kk] = 0;
         continue;
      }
      
      bright = (red + green + blue) / (3 * HDRNF);                         //  mean pixel brightness, 0.0 to 1.0
      kk = py3 * Fww + px3;
      HDRbright[kk] = bright;
      
      if (bright > maxbright) maxbright = bright;
      if (bright < minbright) minbright = bright;

      pix3 = bmpixel(E3rgb16,px3,py3);                                     //  output pixel
      pix3[0] = red * 65535 / HDRNF;
      pix3[1] = green * 65535 / HDRNF;
      pix3[2] = blue * 65535 / HDRNF;
   }
   
   double norm = 0.999 / (maxbright - minbright);   
   for (ii = 0; ii < Fww * Fhh; ii++)                                      //  normalize to range 0.0 to 0.999
      HDRbright[ii] = (HDRbright[ii] - minbright) * norm;

   mwpaint2();                                                             //  update window
   return 1;
}


//  Dialog for user to control the contributions of each input image
//  while watching the output image which is updated in real time.

int HDR_dialog()
{
   using namespace splinecurve;

   int    HDR_dialog_event(zdialog *zd, cchar *event);
   void   HDR_curvedit(int);

   int         ii;
   double      cww = 1.0 / (HDRNF-1);
   
   zdedit = zdialog_new(ZTX("Adjust Image Contributions"),mWin,Bdone,Bcancel,null);
   zdialog_add_widget(zdedit,"frame","brframe","dialog",0,"expand|space=2");
   zdialog_add_widget(zdedit,"hbox","hb1","dialog",0);
   zdialog_add_widget(zdedit,"label","lab11","hb1",ZTX("dark pixels"),"space=3");
   zdialog_add_widget(zdedit,"label","lab12","hb1",0,"expand");
   zdialog_add_widget(zdedit,"label","lab13","hb1",ZTX("light pixels"),"space=3");
   zdialog_add_widget(zdedit,"hbox","hb2","dialog",0,"space=3");
   zdialog_add_widget(zdedit,"label","labf1","hb2",ZTX("file:"),"space=3");
   zdialog_add_widget(zdedit,"label","labf2","hb2","*");

   GtkWidget *brframe = zdialog_widget(zdedit,"brframe");                  //  set up curve edit
   curve_init(brframe,HDR_curvedit);

   Nspc = HDRNF;                                                           //  no. curves = no. files
   
   for (ii = 0; ii < HDRNF; ii++)                                          //  set up initial response curve
   {                                                                       //    anchor points
      vert[ii] = 0;
      nap[ii] = 2;
      apx[ii][0] = 0.01;                                                   //  flatter curves, v.9.3
      apx[ii][1] = 0.99;
      apy[ii][0] = 0.9 - ii * 0.8 * cww;
      apy[ii][1] = 0.1 + ii * 0.8 * cww;
      curve_generate(ii);
   }
   
   zdialog_resize(zdedit,400,300);                                         //  run dialog
   zdialog_run(zdedit,HDR_dialog_event);

   start_thread(HDR_combine_thread,0);                                     //  start working thread
   signal_thread();                                                        //  bugfix  v.10.0
   wrapup_thread(0);                                                       //  wait for completion
   return 1;
}


//  dialog event and completion callback function

int HDR_dialog_event(zdialog *zd, cchar *event)
{
   if (zd->zstat)                                                          //  dialog complete
   {
      if (zd->zstat == 1) edit_done();
      else edit_cancel();
      return 0;
   }

   if (strEqu(event,"F1"))
      showz_userguide("HDR");                                              //  F1 help

   return 1;
}


//  this function is called when a curve is edited

void HDR_curvedit(int spc)
{
   cchar  *pp; 
   
   pp = strrchr(HDRfile[spc],'/');
   zdialog_stuff(zdedit,"labf2",pp+1);
   signal_thread();
   return;
}


//  Combine all input images >> E3rgb16 based on image response curves.

void * HDR_combine_thread(void *)
{
   using namespace splinecurve;

   double      respfac[10][1000];                                          //  contribution / image / pixel brightness
   double      xlo, xhi, xval, yval, sumrf;
   int         px3, py3, ii, jj, kk, vstat;
   double      px, py, red, green, blue, bright, factor;
   double      sintoff[10], costoff[10], sintf, costf;
   uint16      vpix[3], *pix3;
   RGB         *rgb;

   for (ii = 0; ii < HDRNF; ii++)                                          //  pre-calculate trig functions
   {
      sintoff[ii] = sin(HDRtoff[ii]);
      costoff[ii] = cos(HDRtoff[ii]);
   }

   while (true)
   {
      thread_idle_loop();                                                  //  wait for work or exit request
      
      for (ii = 0; ii < HDRNF; ii++)                                       //  loop input images
      {
         jj = nap[ii];                                                     //  get low and high anchor points
         xlo = apx[ii][0];                                                 //    for image response curve
         xhi = apx[ii][jj-1];
         if (xlo < 0.02) xlo = 0;                                          //  snap-to scale end points
         if (xhi > 0.98) xhi = 1;

         for (jj = 0; jj < 1000; jj++)                                     //  loop all brightness levels
         {
            respfac[ii][jj] = 0;
            xval = 0.001 * jj;
            if (xval < xlo || xval > xhi) continue;                        //  no influence for brightness level
            yval = curve_yval(ii,xval);                                    //  response curve value for brightness
            respfac[ii][jj] = yval;                                        //    = contribution of this input image
         }
      }
      
      for (jj = 0; jj < 1000; jj++)                                        //  normalize the factors so that
      {                                                                    //    they sum to 1.0
         sumrf = 0;
         for (ii = 0; ii < HDRNF; ii++)
            sumrf += respfac[ii][jj];
         if (! sumrf) continue;
         for (ii = 0; ii < HDRNF; ii++)
            respfac[ii][jj] = respfac[ii][jj] / sumrf;
      }
      
      for (py3 = 0; py3 < Fhh; py3++)                                      //  step through all output pixels
      for (px3 = 0; px3 < Fww; px3++)
      {
         kk = py3 * Fww + px3;
         bright = HDRbright[kk];                                           //  mean brightness, 0.0 to 1.0
         jj = 1000 * bright;
         
         red = green = blue = 0;
         
         for (ii = 0; ii < HDRNF; ii++)                                    //  loop input images
         {
            factor = respfac[ii][jj];                                      //  image contribution to this pixel
            if (! factor) continue;                                        //  none
            
            rgb = HDRrgb[ii];                                              //  alignment offsets
            xoff = HDRxoff[ii];
            yoff = HDRyoff[ii];
            sintf = sintoff[ii];
            costf = costoff[ii];

            px = costf * (px3 - xoff) + sintf * (py3 - yoff);              //  input virtual pixel mapping to
            py = costf * (py3 - yoff) - sintf * (px3 - xoff);              //    this output pixel

            vstat = vpixel(rgb,px,py,vpix);                                //  get input pixel

            red += factor * vpix[0];                                       //  accumulate brightness contribution
            green += factor * vpix[1];
            blue += factor * vpix[2];
         }
            
         pix3 = bmpixel(E3rgb16,px3,py3);                                  //  output pixel

         pix3[0] = red;                                                    //  = sum of input pixel contributions
         pix3[1] = green;
         pix3[2] = blue;
      }
   
      Fmodified = 1;                                                       //  image is modified
      mwpaint2();                                                          //  update window
   }
   
   return 0;                                                               //  not executed
}


/**************************************************************************
   Make an HDF (high depth of field) image from two images of the same           v.8.0
   subject with different focus settings, near and far. One image has
   the nearer parts of the subject in sharp focus, the other image has
   the farther parts in focus. The output image is constructed from the
   sharpest pixels in each of the two input images. Minor differences 
   in image center, rotation and size are automatically compensated.
**************************************************************************/

void * HDF_align_thread(void *);
void   HDF_combine(int Fcolor);
void   HDF_distort();
void   HDF_mousefunc();
int    HDF_dialog_event(zdialog *zd, cchar *event);

RGB      *A2rgb16cache = 0;
int      HDF_align_stat = 0;
double   HDF_zoffx[4], HDF_zoffxB[4];
double   HDF_zoffy[4], HDF_zoffyB[4];
int      HDF_image;
int      HDF_brush;
int      HDF_suspend;


void m_HDF(GtkWidget *, cchar *)
{
   char     *file2 = 0;

   if (! edit_setup("HDF",0,0)) return;                                    //  setup edit: no preview
   
   zfuncs::F1_help_topic = "HDF";                                          //  v.9.0

   file2 = zgetfile(ZTX("Select image to combine"),image_file,"open");     //  get 2nd HDF image
   if (! file2) {
      edit_cancel();
      return;
   }

   Grgb16 = f_load(file2,16);                                              //  load and validate image
   if (! Grgb16) {
      edit_cancel();
      return;
   }

   Gww = Grgb16->ww;
   Ghh = Grgb16->hh;

   start_thread(HDF_align_thread,0);                                       //  start thread to align images
   wrapup_thread(0);                                                       //  wait for thread exit

   if (HDF_align_stat != 1) {
      edit_cancel();                                                       //  failure
      return;
   }   
   
   HDF_combine(1);                                                         //  combine with color comp.
   Fmodified = 1;

   zdedit = zdialog_new(ZTX("Retouch Image"),mWin,Bdone,Bcancel,null);
   zdialog_add_widget(zdedit,"hbox","hb1","dialog",0,"space=5");
   zdialog_add_widget(zdedit,"radio","radio1","hb1","image 1");
   zdialog_add_widget(zdedit,"radio","radio2","hb1","image 2","space=10");
   zdialog_add_widget(zdedit,"hbox","hb2","dialog",0,"space=3");
   zdialog_add_widget(zdedit,"label","labr","hb2","brush","space=5");
   zdialog_add_widget(zdedit,"spin","radius","hb2","1|199|1|20");
   zdialog_add_widget(zdedit,"button","susp-resm","hb2",Bsuspend,"space=10");
   
   zdialog_stuff(zdedit,"radio1",1);
   HDF_image = 1;
   HDF_brush = 10;
   HDF_suspend = 0;

   mouseCBfunc = HDF_mousefunc;                                            //  connect mouse function
   Mcapture = 1;                                                           //  capture mouse clicks

   zdialog_run(zdedit,HDF_dialog_event);                                   //  run dialog, parallel
   return;
}


//  dialog event and completion callback function

int HDF_dialog_event(zdialog *zd, cchar *event)
{
   int      ii;
   
   if (zd->zstat) goto complete;

   zdialog_fetch(zd,"radio1",ii);
   if (ii) HDF_image = 1;
   else HDF_image = 2;
   
   if (strEqu(event,"radius"))
      zdialog_fetch(zd,"radius",HDF_brush);

   if (strEqu(event,"susp-resm"))                                          //  toggle suspend / resume
   {
      if (HDF_suspend) {
         HDF_suspend = 0;
         paint_toparc(3);                                                  //  start brush outline       v.8.3
         mouseCBfunc = HDF_mousefunc;                                      //  connect mouse function
         Mcapture++;
         zdialog_stuff(zd,"susp-resm",Bsuspend);
      }
      else  {
         HDF_suspend = 1;
         paint_toparc(2);                                                  //  stop brush outline        v.8.3
         mouseCBfunc = 0;                                                  //  disconnect mouse function
         Mcapture = 0;
         zdialog_stuff(zd,"susp-resm",Bresume);
      }
   }

   if (strEqu(event,"F1")) showz_userguide("HDF");                         //  F1 help       v.9.0

   return 1;

complete:

   if (zd->zstat != 1) edit_cancel();                                      //  user cancel
   else edit_done();

   RGB_free(A1rgb16);                                                      //  free memory
   RGB_free(A2rgb16);
   paint_toparc(2);                                                        //  stop brush outline        v.8.3
   Mcapture = 0;                                                           //  disconnect mouse
   mouseCBfunc = 0;
   return 0;
}


//  dialog mouse function

void HDF_mousefunc()
{
   uint16      vpixI[3], *ppix3;
   int         radius, radius2, vstat;
   int         mx, my, dx, dy, px3, py3;
   int         red, green, blue, max;
   double      pxI, pyI, f1;
   double      sintf = sin(toff), costf = cos(toff);

   radius = HDF_brush;
   radius2 = radius * radius;

   toparcx = Mxposn - radius;                                              //  paint brush outline circle
   toparcy = Myposn - radius;                                              //  v.8.3
   toparcw = toparch = 2 * radius;
   Ftoparc = 1;
   paint_toparc(3);

   if (LMclick) {                                                          //  mouse click
      mx = Mxclick;
      my = Myclick;
   }

   else if (Mxdrag || Mydrag) {                                            //  drag in progress
      mx = Mxdrag;
      my = Mydrag;
   }
   
   else return;

   LMclick = RMclick = 0;

   if (mx < 0 || mx > E3ww-1 || my < 0 || my > E3hh-1)                     //  outside image area
      return;

   for (dy = -radius; dy <= radius; dy++)                                  //  loop surrounding block of pixels
   for (dx = -radius; dx <= radius; dx++)
   {
      px3 = mx + dx;
      py3 = my + dy;
      
      if (px3 < 0 || px3 > E3ww-1) continue;                               //  outside image
      if (py3 < 0 || py3 > E3hh-1) continue;
      if (dx*dx + dy*dy > radius2) continue;                               //  outside radius

      if (HDF_image == 1) {
         pxI = costf * px3 - sintf * (py3 - yoff);                         //  image1 virtual pixel
         pyI = costf * py3 + sintf * (px3 - xoff);
         vstat = vpixel(A1rgb16,pxI,pyI,vpixI);
         if (! vstat) continue;
         red = int(R12match[vpixI[0]]);                                    //  compensate color
         green = int(G12match[vpixI[1]]);
         blue = int(B12match[vpixI[2]]);
      }

      else {
         pxI = costf * (px3 - xoff) + sintf * (py3 - yoff);                //  image2 virtual pixel
         pyI = costf * (py3 - yoff) - sintf * (px3 - xoff);
         vstat = vpixel(A2rgb16,pxI,pyI,vpixI);
         if (! vstat) continue;
         red = int(R21match[vpixI[0]]);
         green = int(G21match[vpixI[1]]);
         blue = int(B21match[vpixI[2]]);
      }

      if (red > 65535 || green > 65535 || blue > 65535) {                  //  fix overflow
         max = red;
         if (green > max) max = green;
         if (blue > max) max = blue;
         f1 = 65535.0 / max;
         red = int(red * f1);
         green = int(green * f1);
         blue = int(blue * f1);
      }
      
      ppix3 = bmpixel(E3rgb16,px3,py3);                                    //  image3 real pixel
      ppix3[0] = red;
      ppix3[1] = green;
      ppix3[2] = blue;
   }
   
   mwpaint2();
   return;
}


//  image align thread, combine Frgb16 + Grgb16 >> E3rgb16

void * HDF_align_thread(void *)
{
   double      xfL, xfH, yfL, yfH, xystep, xylim;
   double      tfL, tfH, tstep, tlim;
   double      zlim, zoffx0, zoffy0;
   double      xyrange;
   double      eighth = 0.7854;                                            //  1/8 of circle in radians
   int         ii, jj, lastpass;

   HDF_align_stat = 0;                                                     //  no status yet
   Radjust = Gadjust = Badjust = 1.0;                                      //  no manual color adjustments
   Nalign = 1;                                                             //  alignment in progress
   aligntype = 2;                                                          //  HDF
   pixsamp = 10000;                                                        //  pixel sample size
   showRedpix = 1;                                                         //  highlight alignment pixels
   Fzoom = 0;                                                              //  fit to window if big
   Fblowup = 1;                                                            //  scale up to window if small
   xoff = yoff = toff = 0;                                                 //  initial offsets = 0
   for (ii = 0; ii < 4; ii++)                                              //  initial distortions = 0
      HDF_zoffx[ii] = HDF_zoffy[ii] = 0;

   A1rgb16 = A2rgb16 = A2rgb16cache = 0;

   fullSize = Fww;                                                         //  full image size
   if (Fhh > Fww) fullSize = Fhh;                                          //  (largest dimension)

   alignSize = 200;                                                        //  initial alignment image size
   if (alignSize > fullSize) alignSize = fullSize;

   lastpass = 0;
   xyrange = 0.05 * alignSize;                                             //  first pass, huge search range
   xshrink = yshrink = xyrange;                                            //  image shrink from distortion  v.8.1
   warpxu = warpyu = warpxl = warpyl = 0;                                  //  no warp factors (pano)
   warpxuB = warpyuB = warpxlB = warpylB = 0;

   while (true)                                                            //  next alignment stage / image size
   {   
      A1ww = Fww * alignSize / fullSize;                                   //  align width, height in same ratio
      A1hh = Fhh * alignSize / fullSize;
      A2ww = A1ww;
      A2hh = A1hh;
      
      RGB_free(A1rgb16);
      RGB_free(A2rgb16cache);

      if (alignSize < fullSize) {
         A1rgb16 = RGB_rescale(Frgb16,A1ww,A1hh);                          //  alignment images are
         A2rgb16cache = RGB_rescale(Grgb16,A2ww,A2hh);                     //    down-scaled input images
      }
      else {
         A1rgb16 = RGB_copy(Frgb16);                                       //  full size, copy input images
         A2rgb16cache = RGB_copy(Grgb16);
      }

      RGB_free(A2rgb16);                                                   //  distort image2 using current
      HDF_distort();                                                       //    zoffx/y settings

      mutex_lock(&pixmaps_lock);
      RGB_free(E3rgb16);                                                   //  prepare new output RGB pixmap
      E3rgb16 = RGB_make(A1ww,A1hh,16);
      E3ww = A1ww;
      E3hh = A1hh;
      mutex_unlock(&pixmaps_lock);

      alignWidth = A1ww;
      alignHeight = A1hh;
      getAlignArea();                                                      //  get image align area
      getBrightRatios();                                                   //  get image brightness ratios
      setColorfixFactors(1);                                               //  compute color matching factors
      flagEdgePixels();                                                    //  flag high-contrast pixels

      HDF_combine(0);                                                      //  combine and update window

      xylim = xyrange;                                                     //  xy search range, pixels
      xystep = 0.5 * xyrange;                                              //    -1.0 -0.5 0.0 +0.5 +1.0
      tlim = xylim / alignSize;                                            //  theta search range, radians
      tstep = xystep / alignSize;                                          //  step size
      zlim = xylim;                                                        //  distortion search range

//  find best alignment based on xoff, yoff, toff

      matchB = 0;      

      xfL = xoff - xylim;                                                  //  set x, y, theta search ranges
      xfH = xoff + xylim + xystep/2;
      yfL = yoff - xylim;
      yfH = yoff + xylim + xystep/2;
      tfL = toff - tlim;
      tfH = toff + tlim + tstep/2;

      for (xoff = xfL; xoff < xfH; xoff += xystep)                         //  test all offset dimensions
      for (yoff = yfL; yoff < yfH; yoff += xystep)                         //    in all combinations
      for (toff = tfL; toff < tfH; toff += tstep)
      {
         matchlev = matchImages();
         if (sigdiff(matchlev,matchB,0.00001) > 0) {                       //  remember best match
            matchB = matchlev;
            xoffB = xoff;
            yoffB = yoff;
            toffB = toff;
            HDF_combine(0);                                                //  combine and update window
         }

         Nalign++;                                                         //  count alignment tests
      }

      xoff = xoffB;                                                        //  recover best offsets
      yoff = yoffB;
      toff = toffB;

//  find best distortion settings at the four corners

      for (int mpass = 1; mpass <= 2; mpass++)
      {
         for (ii = 0; ii < 4; ii++)                                        //  corner NW NE SE SW
         {
            HDF_zoffxB[ii] = HDF_zoffx[ii];                                //  save baseline match level
            HDF_zoffyB[ii] = HDF_zoffy[ii];
         
            zoffx0 = HDF_zoffx[ii];                                        //  current setting
            zoffy0 = HDF_zoffy[ii];

            for (jj = 0; jj < 8; jj++)                                     //  8 positions around current
            {                                                              //     distortion setting
               HDF_zoffx[ii] = zoffx0 + zlim * cos(eighth * jj);
               HDF_zoffy[ii] = zoffy0 + zlim * sin(eighth * jj);

               RGB_free(A2rgb16);                                          //  distort image2
               HDF_distort();

               matchlev = matchImages();
               if (sigdiff(matchlev,matchB,0.00001) > 0) {                 //  remember best match
                  matchB = matchlev;
                  HDF_zoffxB[ii] = HDF_zoffx[ii];
                  HDF_zoffyB[ii] = HDF_zoffy[ii];
                  HDF_combine(0);                                          //  combine and update window
               }
               
               Nalign++;                                                   //  count alignment tests
            }

            HDF_zoffx[ii] = HDF_zoffxB[ii];                                //  recover best offset
            HDF_zoffy[ii] = HDF_zoffyB[ii];
         }
      }

// set up for next pass

      if (lastpass) break;                                                 //  done

      if (alignSize == fullSize) {                                         //  full size image was aligned
         lastpass++;                                                       //  one more pass
         xyrange = 0.5;
         continue;
      }

      double R = alignSize;
      alignSize = 1.3 * alignSize;                                         //  next larger image size
      if (alignSize > 0.8 * fullSize) alignSize = fullSize;                //  if near goal, jump to it now
      R = alignSize / R;                                                   //  ratio of new / old image size
      xoff = R * xoff;                                                     //  adjust offsets for image size
      yoff = R * yoff;
      for (ii = 0; ii < 4; ii++) {
         HDF_zoffx[ii] = R * HDF_zoffx[ii];
         HDF_zoffy[ii] = R * HDF_zoffy[ii];
      }

      xyrange = 0.7 * xyrange;                                             //  reduce search range
      if (xyrange < 1) xyrange = 1;
      xshrink = yshrink = xyrange;                                         //  image shrink from distortion  v.8.1
   }
   
   RGB_free(A2rgb16cache);                                                 //  free memory
   RGB_free(Grgb16);                                                       //  A1/A2rgb16 still needed

   Fmodified = 1;                                                          //  image is modified
   showRedpix = 0;                                                         //  stop red pixel highlights
   Fblowup = 0;                                                            //  reset forced image scaling
   Nalign = 0;                                                             //  reset align counter
   HDF_align_stat = 1;                                                     //  signal success
   exit_thread();
   return 0;                                                               //  not executed, stop g++ warning
}


//  Combine images A1rgb16 and A2rgb16 using brightness adjustments.
//  Output is to E3rgb16 (not reallocated). Update main window.

void HDF_combine(int Fcolor)
{
   int         px3, py3, ii, vstat1, vstat2;
   int         red1, green1, blue1, red2, green2, blue2, max;
   double      px1, py1, px2, py2, f1;
   double      sintf = sin(toff), costf = cos(toff);
   uint16      vpix1[3], vpix2[3], *pix3;
   
   Radjust = Gadjust = Badjust = 1.0;

   for (py3 = 1; py3 < E3hh-1; py3++)                                      //  step through output pixels
   for (px3 = 1; px3 < E3ww-1; px3++)
   {
      px1 = costf * px3 - sintf * (py3 - yoff);                            //  A1rgb16 pixel, after offsets
      py1 = costf * py3 + sintf * (px3 - xoff);
      vstat1 = vpixel(A1rgb16,px1,py1,vpix1);

      px2 = costf * (px3 - xoff) + sintf * (py3 - yoff);                   //  corresponding A2rgb16 pixel
      py2 = costf * (py3 - yoff) - sintf * (px3 - xoff);
      vstat2 = vpixel(A2rgb16,px2,py2,vpix2);

      pix3 = bmpixel(E3rgb16,px3,py3);                                     //  output pixel

      if (! vstat1 || ! vstat2) {                                          //  no overlap
         pix3[0] = pix3[1] = pix3[2] = 0;                                  //  output pixel is black
         continue;
      }

      if (showRedpix) {                                                    //  show alignment pixels in red
         ii = py3 * A1ww + px3;
         if (redpixels[ii]) {
            pix3[0] = 65535;
            pix3[1] = pix3[2] = 0;
            continue;
         }
      }

      red1 = vpix1[0];                                                     //  image1 and image2 pixels
      green1 = vpix1[1];
      blue1 = vpix1[2];

      red2 = vpix2[0];
      green2 = vpix2[1];
      blue2 = vpix2[2];

      if (Fcolor)
      {
         red1 = int(R12match[red1]);                                       //  compensate color
         green1 = int(G12match[green1]);
         blue1 = int(B12match[blue1]);

         red2 = int(R21match[red2]);
         green2 = int(G21match[green2]);
         blue2 = int(B21match[blue2]);

         if (red1 > 65535 || green1 > 65535 || blue1 > 65535) {            //  fix overflow
            max = red1;
            if (green1 > max) max = green1;
            if (blue1 > max) max = blue1;
            f1 = 65535.0 / max;
            red1 = int(red1 * f1);
            green1 = int(green1 * f1);
            blue1 = int(blue1 * f1);
         }

         if (red2 > 65535 || green2 > 65535 || blue2 > 65535) {
            max = red2;
            if (green2 > max) max = green2;
            if (blue2 > max) max = blue2;
            f1 = 65535.0 / max;
            red2 = int(red2 * f1);
            green2 = int(green2 * f1);
            blue2 = int(blue2 * f1);
         }
      }
      
      pix3[0] = (red1 + red2) / 2;                                         //  output = combined inputs
      pix3[1] = (green1 + green2) / 2;
      pix3[2] = (blue1 + blue2) / 2;
   }

   mwpaint2();                                                             //  update window
   return;
}


//  Distort A2rgb16cache, returning A2rgb16
//  4 corners move HDF_zoffx[ii], HDF_zoffy[ii] pixels
//  and center does not move, 

void HDF_distort()
{
   void * HDF_distort_wthread(void *arg);

   RGB         *rgbin, *rgbout;
   int         ii, ww, hh;

   rgbin = A2rgb16cache;
   ww = rgbin->ww;                                                         //  create output RGB pixmap
   hh = rgbin->hh;
   rgbout = RGB_make(ww,hh,16);
   A2rgb16 = rgbout;
   
   for (ii = 0; ii < Nwt; ii++)                                            //  start worker threads
      start_wt(HDF_distort_wthread,&wtnx[ii]);
   wait_wts();                                                             //  wait for completion

   return;
}

void * HDF_distort_wthread(void *arg)                                      //  worker thread function
{
   int         index = *((int *) arg);
   int         pxm, pym, ww, hh, vstat;
   double      diag, px, py, dispx, dispy, dispx0, dispy0;
   double      disp0, disp1, disp2, disp3;
   double      disp0x, disp1x, disp2x, disp3x;
   double      disp0y, disp1y, disp2y, disp3y;
   uint16      vpix[3], *pixm;
   RGB         *rgbin, *rgbout;
   
   rgbin = A2rgb16cache;
   rgbout = A2rgb16;

   ww = rgbin->ww;                                                         //  create output RGB pixmap
   hh = rgbin->hh;
   diag = sqrt(ww*ww + hh*hh);
   
   pxm = ww/2;                                                             //  center pixel
   pym = hh/2;
   
   disp0 = (1 - pxm/diag) * (1 - pym/diag);
   disp0 = disp0 * disp0;
   disp0x = disp0 * HDF_zoffx[0];
   disp0y = disp0 * HDF_zoffy[0];
   
   disp1 = (1 - (ww-pxm)/diag) * (1 - pym/diag);
   disp1 = disp1 * disp1;
   disp1x = disp1 * HDF_zoffx[0];
   disp1y = disp1 * HDF_zoffy[0];

   disp2 = (1 - (ww-pxm)/diag) * (1 - (hh-pym)/diag);
   disp2 = disp2 * disp2;
   disp2x = disp2 * HDF_zoffx[0];
   disp2y = disp2 * HDF_zoffy[0];
   
   disp3 = (1 - pxm/diag) * (1 - (hh-pym)/diag);
   disp3 = disp3 * disp3;
   disp3x = disp3 * HDF_zoffx[0];
   disp3y = disp3 * HDF_zoffy[0];
   
   dispx = +disp0x - disp1x - disp2x + disp3x;                             //  center pixel displacement
   dispy = +disp0y + disp1y - disp2y - disp3y;
   
   dispx0 = -dispx;                                                        //  anti-displacement
   dispy0 = -dispy;

   for (pym = index; pym < hh; pym += Nwt)                                 //  loop all pixels
   for (pxm = 0; pxm < ww; pxm++)
   {
      disp0 = (1 - pxm/diag) * (1 - pym/diag);
      disp0 = disp0 * disp0;
      disp0x = disp0 * HDF_zoffx[0];
      disp0y = disp0 * HDF_zoffy[0];
      
      disp1 = (1 - (ww-pxm)/diag) * (1 - pym/diag);
      disp1 = disp1 * disp1;
      disp1x = disp1 * HDF_zoffx[0];
      disp1y = disp1 * HDF_zoffy[0];

      disp2 = (1 - (ww-pxm)/diag) * (1 - (hh-pym)/diag);
      disp2 = disp2 * disp2;
      disp2x = disp2 * HDF_zoffx[0];
      disp2y = disp2 * HDF_zoffy[0];
      
      disp3 = (1 - pxm/diag) * (1 - (hh-pym)/diag);
      disp3 = disp3 * disp3;
      disp3x = disp3 * HDF_zoffx[0];
      disp3y = disp3 * HDF_zoffy[0];
      
      dispx = +disp0x - disp1x - disp2x + disp3x;                          //  (pxm,pym) displacement
      dispy = +disp0y + disp1y - disp2y - disp3y;
      
      dispx += dispx0;                                                     //  relative to center pixel
      dispy += dispy0;

      px = pxm + dispx;                                                    //  source pixel location
      py = pym + dispy;

      vstat = vpixel(rgbin,px,py,vpix);                                    //  input virtual pixel
      pixm = bmpixel(rgbout,pxm,pym);                                      //  output real pixel

      if (vstat) {  
         pixm[0] = vpix[0];
         pixm[1] = vpix[1];
         pixm[2] = vpix[2];
      }
      else pixm[0] = pixm[1] = pixm[2] = 0;
   }
   
   exit_wt();
   return 0;                                                               //  not executed, avoid gcc warning
}


/**************************************************************************/

//  panorama function - combine left and right images into a wide image

void   pano_prealign();                                                    //  manual pre-align
void   pano_autolens();                                                    //  auto optimize lens parameters
void * pano_align_thread(void *);                                          //  auto align images
void   pano_final_adjust();                                                //  manual final adjustment
void   pano_get_align_images(int newf, int strf);                          //  scale and curve images for align
void   pano_combine(int fcolor);                                           //  combine images

int      pano_stat;                                                        //  dialog and thread status
int      pano_automatch;                                                   //  auto color matching on/off
RGB      *pano_A1cache, *pano_A2cache;                                     //  cached alignment images
double   pano_curve, pano_bow;                                             //  converted lens parameters


void m_pano(GtkWidget *, cchar *)
{
   char        *file2 = 0;

   Grgb16 = A1rgb16 = A2rgb16 = pano_A1cache = pano_A2cache = 0;

   if (! edit_setup("pano",0,0)) return;                                   //  setup edit: no preview

   zfuncs::F1_help_topic = "panorama";                                     //  v.9.0
   
   file2 = zgetfile(ZTX("Select image to combine"),image_file,"open");     //  2nd or next pano file
   if (! file2) goto pano_cancel;

   Grgb16 = f_load(file2,16);                                              //  load and validate image
   if (! Grgb16) goto pano_cancel;
   Gww = Grgb16->ww;
   Ghh = Grgb16->hh;
   
   xoff = yoff = xoffB = yoffB = toff = toffB = 0;                         //  initial offsets
   warpxu = warpyu = warpxl = warpyl = 0;                                  //  initial warp factors
   warpxuB = warpyuB = warpxlB = warpylB = 0;

   pano_prealign();                                                        //  do manual pre-align
   if (pano_stat != 1) goto pano_cancel;

   if (overlapixs < alignSize * 20) {                                      //  need > 20 pixel overlap
      zmessageACK(GTK_WINDOW(mWin),ZTX("Too little overlap, cannot align"));
      goto pano_cancel;
   }

   start_thread(pano_align_thread,0);                                      //  start thread to align images
   wrapup_thread(0);                                                       //  wait for thread exit
   if (pano_stat != 1) goto pano_cancel;
   
   pano_final_adjust();                                                    //  do manual final adjustments
   if (pano_stat != 1) goto pano_cancel;
   Fmodified = 1;
   edit_done();
   goto pano_cleanup;
   
pano_cancel:
   edit_cancel();

pano_cleanup:
   if (file2) zfree(file2);
   RGB_free(Grgb16);
   RGB_free(A1rgb16);
   RGB_free(A2rgb16);
   RGB_free(pano_A1cache);
   RGB_free(pano_A2cache);
   return;
}


//  perform manual pre-align of image2 to image1
//  return offsets: xoff, yoff, toff
//  lens_mm and lens_bow may also be altered

void pano_prealign()   
{  
   int    pano_prealign_event(zdialog *zd, cchar *event);                  //  dialog event function
   void * pano_prealign_thread(void *);                                    //  working thread

   cchar  *align_mess = ZTX("Drag right image into rough alignment with left \n"
                            " to rotate, drag right edge up or down");
   cchar  *proceed_mess = ZTX("Merge the images together");
   cchar  *search_mess = ZTX("Auto-search lens mm and bow");

   zdedit = zdialog_new(ZTX("Pre-align Images"),mWin,Bcancel,null);
   zdialog_add_widget(zdedit,"label","lab1","dialog",align_mess,"space=5");
   zdialog_add_widget(zdedit,"hbox","hb1","dialog",0,"space=5");
   zdialog_add_widget(zdedit,"spin","spmm","hb1","22|200|0.1|35","space=5");
   zdialog_add_widget(zdedit,"label","labmm","hb1",ZTX("lens mm"));        //  fix translation   v.8.4.1
   zdialog_add_widget(zdedit,"hbox","hb2","dialog",0,"space=5");
   zdialog_add_widget(zdedit,"spin","spbow","hb2","-9|9|0.01|0","space=5");
   zdialog_add_widget(zdedit,"label","labbow","hb2",ZTX("lens bow"));
   zdialog_add_widget(zdedit,"hbox","hb3","dialog",0,"space=5");
   zdialog_add_widget(zdedit,"button","proceed","hb3",Bproceed,"space=5");
   zdialog_add_widget(zdedit,"label","labproceed","hb3",proceed_mess);
   zdialog_add_widget(zdedit,"hbox","hb4","dialog",0,"space=5");
   zdialog_add_widget(zdedit,"button","search","hb4",Bsearch,"space=5");
   zdialog_add_widget(zdedit,"label","labsearch","hb4",search_mess);
   
   lens_mm = lens4_mm[curr_lens];                                          //  initz. curr. lens parameters
   lens_bow = lens4_bow[curr_lens];
   zdialog_stuff(zdedit,"spmm",lens_mm);
   zdialog_stuff(zdedit,"spbow",lens_bow);

   zdialog_run(zdedit,pano_prealign_event);                                //  run dialog, parallel

   set_cursor(dragcursor);                                                 //  set drag cursor  v.10.1
   
   Fzoom = 0;                                                              //  fit image to window       v.10.1.1
   pano_stat = -1;
   start_thread(pano_prealign_thread,0);                                   //  start working thread
   wrapup_thread(0);                                                       //  wait for completion

   set_cursor(0);                                                          //  restore normal cursor
   return;
}


//  dialog event and completion callback function

int pano_prealign_event(zdialog *zd, cchar *event)                         //  dialog event function
{
   if (zd->zstat) goto complete;                                           //  dialog complete

   if (strstr("spmm spbow",event)) {                                       //  revised lens data   v.7.8
      zdialog_fetch(zd,"spmm",lens_mm);
      zdialog_fetch(zd,"spbow",lens_bow);
   }

   if (strEqu(event,"search")) Fautolens = 1;                              //  trigger auto-lens function

   if (strEqu(event,"proceed")) {                                          //  proceed with pano
      pano_stat = 1;                                                       //  signal align success
      wrapup_thread(0);                                                    //  wait for thread exit
      zdialog_free(zdedit);                                                //  kill dialog
      zdedit = null;
   }

   if (strEqu(event,"F1")) showz_userguide("panorama");                    //  F1 help       v.9.0
   return 0;

complete:
   pano_stat = 0;                                                          //  signal cancel
   wrapup_thread(0);                                                       //  wait for thread exit
   zdialog_free(zdedit);                                                   //  kill dialog
   zdedit = null;
   return 0;
}


void * pano_prealign_thread(void *)                                        //  prealign working thread
{
   int         mx0, my0, mx, my;                                           //  mouse drag origin, position
   double      lens_mm0, lens_bow0;
   double      mlever = 0.5;
   double      dtoff;
   int         majupdate = 1;

   Radjust = Gadjust = Badjust = 1.0;                                      //  no manual color adjustments
   Nalign = 1;                                                             //  alignment in progress
   aligntype = 3;                                                          //  pano
   pixsamp = 5000;                                                         //  pixel sample size
   showRedpix = 0;                                                         //  no pixel highlight (yet)
   Fblowup = 1;                                                            //  scale-up small image to window

   fullSize = Ghh;                                                         //  full size = image2 height
   alignSize = int(pano_prealign_size);                                    //  prealign image size
   pano_get_align_images(1,0);                                             //  get prealign images and curve them

   xoff = xoffB = 0.8 * A1ww;                                              //  initial x offset (20% overlap)

   alignWidth = int(A1ww - xoff);                                          //  initial alignment on full overlap
   alignHeight = A1hh;
   getAlignArea();                                                         //  get image overlap area
   pano_combine(0);                                                        //  combine images

   lens_mm0 = lens_mm;                                                     //  to detect changes
   lens_bow0 = lens_bow;

   mx0 = my0 = 0;                                                          //  no drag in progress
   Mcapture = KBcapture = 1;                                               //  capture mouse drag and KB keys

   while (pano_stat == -1)                                                 //  loop and align until done
   {
      zsleep(0.05);                                                        //  logic simplified   v.7.5
      
      if (Fautolens) {
         pano_autolens();                                                  //  get lens parameters
         zdialog_stuff(zdedit,"spmm",lens_mm);                             //  update dialog
         zdialog_stuff(zdedit,"spbow",lens_bow);
         majupdate++;
      }
         
      if (lens_mm != lens_mm0 || lens_bow != lens_bow0) {                  //  change in lens parameters
         lens_mm0 = lens_mm;
         lens_bow0 = lens_bow;
         pano_get_align_images(0,0);
         majupdate++;
      }
      
      if (KBkey) {                                                         //  KB input
         if (KBkey == GDK_Left)  xoff -= 0.2;                              //  tweak alignment offsets
         if (KBkey == GDK_Right) xoff += 0.2;
         if (KBkey == GDK_Up)    yoff -= 0.2;
         if (KBkey == GDK_Down)  yoff += 0.2;
         if (KBkey == GDK_r)     toff += 0.0002;
         if (KBkey == GDK_l)     toff -= 0.0002;
         KBkey = 0;

         alignWidth = int(A1ww - xoff);                                    //  entire overlap
         alignHeight = A1hh;                                               //  v.8.0
         getAlignArea();
         pano_combine(0);                                                  //  show combined images
         majupdate++;
         continue;
      }
      
      if (! Mxdrag && ! Mydrag) mx0 = my0 = 0;                             //  no drag in progress

      if (Mxdrag || Mydrag)                                                //  mouse drag underway
      {
         mx = Mxdrag;                                                      //  mouse position in image
         my = Mydrag;

         if (mx < xoff || mx > xoff + A2ww || my > E3hh) {                 //  if mouse not in image2 area,
            mx0 = my0 = Mxdrag = Mydrag = 0;                               //    no drag in progress
            continue;
         }
         
         if (! mx0 && ! my0) {                                             //  new drag, set drag origin
            mx0 = mx;
            my0 = my;
         }
         
         if (mx != mx0 || my != my0)                                       //  drag is progressing
         {
            if (mx > xoff + 0.8 * A2ww) {                                  //  near right edge, theta drag 
               dtoff = mlever * (my - my0) / A2ww;                         //  delta theta, radians
               toff += dtoff;
               xoff += dtoff * (A1hh + yoff);                              //  change center of rotation
               yoff -= dtoff * (A1ww - xoff);                              //    to middle of overlap area
            }
            else  {                                                        //  x/y drag
               xoff += mlever * (mx - mx0);                                //  image2 offsets / mouse leverage
               yoff += mlever * (my - my0);
            }

            mx0 = mx;                                                      //  next drag origin = current mouse
            my0 = my;

            if (xoff > A1ww) xoff = A1ww;                                  //  limit nonsense
            if (xoff < 0.3 * A1ww) xoff = 0.3 * A1ww;
            if (yoff < -0.5 * A2hh) yoff = -0.5 * A2hh;
            if (yoff > 0.5 * A1hh) yoff = 0.5 * A1hh;
            if (toff < -0.20) toff = -0.20;
            if (toff > 0.20) toff = 0.20;

            alignWidth = int(A1ww - xoff);                                 //  entire overlap
            getAlignArea();
            pano_combine(0);                                               //  show combined images
            majupdate++;
            continue;
         }
      }

      if (majupdate) {                                                     //  do major update
         majupdate = 0;
         alignWidth = int(A1ww - xoff);                                    //  entire overlap
         getAlignArea();                                                   //  get image overlap area
         getBrightRatios();                                                //  get color brightness ratios
         setColorfixFactors(1);                                            //  set color matching factors
         flagEdgePixels();                                                 //  flag edge pixels in overlap 
         matchB = matchImages();                                           //  match images
         xoffB = xoff;                                                     //  new base alignment data
         yoffB = yoff;
         toffB = toff;
         pano_combine(0);                                                  //  show combined images
      }
   }
   
   if (Fzoom) {
      Fzoom = 0;                                                           //  bugfix, fit window   v.10.3.1
      mwpaint2();
   }

   Nalign = 0;
   KBcapture = Mcapture = 0;
   exit_thread();
   return 0;                                                               //  never executed, stop g++ warning
}


//  optimize lens parameters
//  assumes a good starting point since search ranges are limited
//  inputs and outputs: lens_mm, lens_bow, xoff, yoff, toff

void pano_autolens()
{
   double   mm_range, bow_range, xoff_range, yoff_range, toff_range;
   double   squeeze, xoff_rfinal, rnum;
   double   lens_mmB, lens_bowB;
   int      counter = 0;
   
   mm_range = 0.2 * lens_mm;                                               //  set initial search ranges
   bow_range = 0.5 * lens_bow;
   if (bow_range < 1) bow_range = 1;
   xoff_range = 7;
   yoff_range = 7;
   toff_range = 0.01;

   xoff_rfinal = 0.2;                                                      //  final xoff range - when to quit
   
   Nalign = 1;
   aligntype = 3;
   showRedpix = 1;
   pano_get_align_images(0,0);
   alignWidth = int(A1ww - xoff);
   if (alignWidth > A2ww/3) alignWidth = A2ww/3;
   alignHeight = A1hh - abs(yoff);
   alignWidth = 0.9 * alignWidth;                                          //  do not change align pixels
   alignHeight = 0.9 * alignHeight;                                        //   when align parms change  v.8.1
   getAlignArea();
   getBrightRatios();
   setColorfixFactors(1);
   flagEdgePixels();
   pano_combine(0);

   lens_mmB = lens_mm;                                                     //  initial best fit = current data
   lens_bowB = lens_bow;
   xoffB = xoff;
   yoffB = yoff;
   toffB = toff;
   matchB = matchImages();

   while (true)
   {
      srand48(time(0) + counter++);
      lens_mm = lens_mmB + mm_range * (drand48() - 0.5);                   //  new random lens factors
      lens_bow = lens_bowB + bow_range * (drand48() - 0.5);                //     within search range
      pano_get_align_images(0,0);                                          //  curve images
      getAlignArea();                                                      //  synch align data   v.8.5
      getBrightRatios();
      setColorfixFactors(1);
      flagEdgePixels();
      squeeze = 0.95;                                                      //  search range reduction
         
      for (int ii = 0; ii < 500; ii++)                                     //  loop random alignments  v.8.5
      {                                                                    
         rnum = drand48();
         if (rnum < 0.33)                                                  //  random change some alignment offset 
            xoff = xoffB + xoff_range * (drand48() - 0.5);                 //    within search range
         else if (rnum < 0.67)
            yoff = yoffB + yoff_range * (drand48() - 0.5);
         else
            toff = toffB + toff_range * (drand48() - 0.5);
      
         matchlev = matchImages();                                         //  test quality of image alignment
         if (sigdiff(matchlev,matchB,0.0001) > 0) {
            lens_mmB = lens_mm;                                            //  better
            lens_bowB = lens_bow;
            xoffB = xoff;                                                  //  (no refresh align pixels v.8.1)
            yoffB = yoff;
            toffB = toff;
            matchB = matchlev;                                             //  save new best fit
            pano_combine(0);
            squeeze = 1;                                                   //  keep same search range as long
            break;                                                         //    as improvements are found
         }

         Nalign++;
         if (pano_stat != -1) goto done;
      }
      
      if (xoff_range < xoff_rfinal) goto done;

      mm_range = squeeze * mm_range;                                       //  reduce search range if no 
      if (mm_range < 0.02 * lens_mmB) mm_range = 0.02 * lens_mmB;          //    improvements were found
      bow_range = squeeze * bow_range;
      if (bow_range < 0.1 * lens_bowB) bow_range = 0.1 * lens_bowB;
      if (bow_range < 0.2) bow_range = 0.2;
      xoff_range = squeeze * xoff_range;
      yoff_range = squeeze * yoff_range;
      toff_range = squeeze * toff_range;
   }

done:
   lens_mm = lens_mmB;                                                     //  set best alignment found
   lens_bow = lens_bowB;
   xoff = xoffB;
   yoff = yoffB;
   toff = toffB;
   Fautolens = 0;
   showRedpix = 0;
   pano_combine(0);
   Nalign = 0;
   return;
}


//  Thread function for combining A1 + A2 >> E3

void * pano_align_thread(void *)
{
   int         firstpass, lastpass;
   double      xystep, xylim, tstep, tlim;
   double      xfL, xfH, yfL, yfH, tfL, tfH;
   double      wxL, wxH, wyL, wyH;
   double      alignR;
   
   Nalign = 1;                                                             //  alignment in progress
   aligntype = 3;                                                          //  pano
   pano_stat = 0;
   Fzoom = 0;                                                              //  fit to window if big
   Fblowup = 1;                                                            //  scale up to window if small
   showRedpix = 1;                                                         //  highlight alignment pixels
   firstpass = 1;
   lastpass = 0;

   while (true)
   {
      alignR = alignSize;                                                  //  from pre-align or prior pass
      if (firstpass) alignSize = 140;                                      //  set next align size
      else if (lastpass) alignSize = fullSize;
      else  alignSize = int(pano_image_increase * alignSize);              //  next larger image size
      if (alignSize > 0.8 * fullSize) alignSize = fullSize;                //  if near goal, jump to it now
      alignR = alignSize / alignR;                                         //  ratio of new / old image size

      xoff = alignR * xoff;                                                //  adjust offsets for new image size
      yoff = alignR * yoff;
      toff = toff;

      warpxu = alignR * warpxu;                                            //  adjust warp values
      warpyu = alignR * warpyu;
      warpxl = alignR * warpxl;
      warpyl = alignR * warpyl;
      
      if (! lastpass) pano_get_align_images(1,0);                          //  get new alignment images

      if (firstpass) {
         alignWidth = int(A1ww - xoff);                                    //  set new alignment area
         if (alignWidth > A1ww/3) alignWidth = A1ww/3;
      }
      else  {
         alignWidth = int(alignR * alignWidth * pano_blend_decrease);
         if (alignWidth < pano_min_alignwidth * alignSize)                 //  keep within range
            alignWidth = int(pano_min_alignwidth * alignSize);
         if (alignWidth > pano_max_alignwidth * alignSize) 
            alignWidth = int(pano_max_alignwidth * alignSize);
      }

      alignHeight = A1hh;

      getAlignArea();                                                      //  get image overlap area
      getBrightRatios();                                                   //  get color brightness ratios
      setColorfixFactors(1);                                               //  set color matching factors
      flagEdgePixels();                                                    //  flag high-contrast pixels in blend

      xylim = 2;                                                           //  +/- search range, centered on
      xystep = 0.571;                                                      //    results from prior stage
      
      if (firstpass) {
         xylim = alignSize * 0.03;                                         //  3% error tolerance in pre-alignment
         xystep = 0.5; 
      }

      if (lastpass) {
         xylim = 1;                                                        //  final stage pixel search steps
         xystep = 0.5;                                                     //   -1.0 -0.5 0.0 +0.5 +1.0
      }

      tlim = xylim / alignSize / 2;                                        //  theta max offset, radians
      tstep = xystep / alignSize / 2;                                      //  theta step size

      xfL = xoff - xylim;                                                  //  set x/y/t search ranges, step sizes
      xfH = xoff + xylim + xystep/2;
      yfL = yoff - xylim;
      yfH = yoff + xylim + xystep/2;
      tfL = toff - tlim;
      tfH = toff + tlim + tstep/2;

      xoffB = xoff;                                                        //  initial offsets = best so far
      yoffB = yoff;
      toffB = toff;
      
      warpxuB = warpxu;                                                    //  initial warp values
      warpyuB = warpyu;
      warpxlB = warpxl;
      warpylB = warpyl;

      matchB = matchImages();                                              //  set base match level
      pano_combine(0);                                                     //  v.8.4

      for (xoff = xfL; xoff < xfH; xoff += xystep)                         //  test x, y, theta offsets
      for (yoff = yfL; yoff < yfH; yoff += xystep)                         //    in all possible combinations
      for (toff = tfL; toff < tfH; toff += tstep)
      {
         matchlev = matchImages();
         if (sigdiff(matchlev,matchB,0.00001) > 0) {                       //  remember best alignment and offsets
            matchB = matchlev;
            xoffB = xoff;
            yoffB = yoff;
            toffB = toff;
         }

         Nalign++;
      }
      
      xoff = xoffB;                                                        //  recover best offsets
      yoff = yoffB;
      toff = toffB;
      
      if (! firstpass)
      {
         wxL = warpxuB - xylim;                                            //  warp image2 corners    v.8.5
         wxH = warpxuB + xylim + xystep/2;                                 //  double range           v.8.6.1
         wyL = warpyuB - xylim;
         wyH = warpyuB + xylim + xystep/2;
         
         for (warpxu = wxL; warpxu < wxH; warpxu += xystep)                //  search upper warp
         for (warpyu = wyL; warpyu < wyH; warpyu += xystep)
         {
            pano_get_align_images(0,1);                                    //  curve and warp
            matchlev = matchImages();
            if (sigdiff(matchlev,matchB,0.00001) > 0) {                    //  remember best warp
               matchB = matchlev;
               warpxuB = warpxu;
               warpyuB = warpyu;
            }

            Nalign++;
         }

         warpxu = warpxuB;                                                 //  restore best warp
         warpyu = warpyuB;
         pano_get_align_images(0,0);

         wxL = warpxlB - xylim;
         wxH = warpxlB + xylim + xystep/2;
         wyL = warpylB - xylim;
         wyH = warpylB + xylim + xystep/2;
         
         for (warpxl = wxL; warpxl < wxH; warpxl += xystep)                //  search lower warp
         for (warpyl = wyL; warpyl < wyH; warpyl += xystep)
         {
            pano_get_align_images(0,2);
            matchlev = matchImages();
            if (sigdiff(matchlev,matchB,0.00001) > 0) {
               matchB = matchlev;
               warpxlB = warpxl;
               warpylB = warpyl;
            }

            Nalign++;
         }
         
         warpxl = warpxlB;
         warpyl = warpylB;
         pano_get_align_images(0,0);
      }

      pano_combine(0);                                                     //  combine images and update window

      firstpass = 0;
      if (lastpass) break;
      if (alignSize == fullSize) lastpass = 1;                             //  one more pass, reduced step size
   }

   pano_stat = 1;                                                          //  signal success
   showRedpix = 0;                                                         //  pixel highlights off
   Fzoom = Fblowup = 0;                                                    //  reset image scaling
   Nalign = 0;
   exit_thread();
   return 0;                                                               //  never executed, stop g++ warning
}


//  do manual adjustment of brightness, color, blend width

void pano_final_adjust()
{
   int    pano_adjust_event(zdialog *zd, cchar *event);                    //  dialog event function
   void * pano_adjust_thread(void *);
   
   cchar    *adjmessage = ZTX("\n Match Brightness and Color");
   
   pano_automatch = 1;                                                     //  init. auto color match on
   alignWidth = 1;
   
   getAlignArea();
   getBrightRatios();                                                      //  get color brightness ratios
   setColorfixFactors(pano_automatch);                                     //  set color matching factors
   pano_combine(1);                                                        //  show final results

   zdedit = zdialog_new(ZTX("Match Images"),mWin,Bdone,Bcancel,null);

   zdialog_add_widget(zdedit,"label","lab0","dialog",adjmessage,"space=10");
   zdialog_add_widget(zdedit,"hbox","hb1","dialog",0,"space=10");          //  match brightness and color
   zdialog_add_widget(zdedit,"vbox","vb1","hb1",0,"homog");
   zdialog_add_widget(zdedit,"vbox","vb2","hb1",0,"homog");                //  red           [ 100 ]
   zdialog_add_widget(zdedit,"label","lab1","vb1",Bred,"space=7");         //  green         [ 100 ]
   zdialog_add_widget(zdedit,"label","lab2","vb1",Bgreen,"space=7");       //  blue          [ 100 ]
   zdialog_add_widget(zdedit,"label","lab3","vb1",Bblue,"space=7");        //  brightness    [ 100 ]
   zdialog_add_widget(zdedit,"label","lab4","vb1",Bbrightness,"space=7");  //  blend width   [  0  ]
   zdialog_add_widget(zdedit,"label","lab5","vb1",Bblendwidth,"space=7");  //
   zdialog_add_widget(zdedit,"spin","spred","vb2","50|200|0.1|100");       //  [ apply ]  [ auto ]  off
   zdialog_add_widget(zdedit,"spin","spgreen","vb2","50|200|0.1|100");
   zdialog_add_widget(zdedit,"spin","spblue","vb2","50|200|0.1|100");
   zdialog_add_widget(zdedit,"spin","spbright","vb2","50|200|0.1|100");
   zdialog_add_widget(zdedit,"spin","spblend","vb2","1|200|1|1");
   zdialog_add_widget(zdedit,"hbox","hb2","dialog",0,"space=5");
   zdialog_add_widget(zdedit,"button","apply","hb2",Bapply,"space=5");
   zdialog_add_widget(zdedit,"button","auto","hb2",ZTX("Auto"));
   zdialog_add_widget(zdedit,"label","labauto","hb2","on");
   
   zdialog_run(zdedit,pano_adjust_event);                                  //  run dialog, parallel
   
   start_thread(pano_adjust_thread,0);                                     //  start thread
   wrapup_thread(0);                                                       //  wait for completion
   return;
}


//  thread function - stall return from final_adjust until dialog done

void * pano_adjust_thread(void *)
{
   while (true) thread_idle_loop();                                        //  wait for work or exit request
   return 0;                                                               //  not executed, stop g++ warning
}


//  dialog event and completion callback function
//  A1 + A2 >> E3 under control of spin buttons

int pano_adjust_event(zdialog *zd, cchar *event)
{
   double      red, green, blue, bright, bright2;
   
   if (zd->zstat) goto complete;
   
   if (strEqu(event,"F1")) showz_userguide("panorama");                    //  F1 help       v.9.0

   if (strEqu(event,"auto")) 
   {
      pano_automatch = 1 - pano_automatch;                                 //  toggle color automatch state
      setColorfixFactors(pano_automatch);                                  //  corresp. color matching factors
      pano_combine(1);                                                     //  combine images and update window
      if (pano_automatch) zdialog_stuff(zd,"labauto","on");
      else zdialog_stuff(zd,"labauto","off");
      return 1;
   }

   if (strNeq(event,"apply")) return 0;                                    //  wait for apply button

   zdialog_fetch(zd,"spred",red);                                          //  get color adjustments
   zdialog_fetch(zd,"spgreen",green);
   zdialog_fetch(zd,"spblue",blue);
   zdialog_fetch(zd,"spbright",bright);                                    //  brightness adjustment
   zdialog_fetch(zd,"spblend",alignWidth);                                 //  align and blend width

   bright2 = (red + green + blue) / 3;                                     //  RGB brightness
   bright = bright / bright2;                                              //  bright setpoint / RGB brightness
   red = red * bright;                                                     //  adjust RGB brightness
   green = green * bright;
   blue = blue * bright;
   
   bright = (red + green + blue) / 3;
   zdialog_stuff(zd,"spred",red);                                          //  force back into consistency
   zdialog_stuff(zd,"spgreen",green);
   zdialog_stuff(zd,"spblue",blue);
   zdialog_stuff(zd,"spbright",bright);

   Radjust = red / 100;                                                    //  normalize 0.5 ... 2.0
   Gadjust = green / 100;
   Badjust = blue / 100;

   getAlignArea();
   getBrightRatios();                                                      //  get color brightness ratios
   setColorfixFactors(pano_automatch);                                     //  set color matching factors
   pano_combine(1);                                                        //  combine and update window

   return 1;

complete:
   zdialog_free(zdedit);                                                   //  kill dialog
   zdedit = null;
   if (zd->zstat == 1) pano_stat = 1;
   else pano_stat = 0;
   wrapup_thread(8);                                                       //  kill thread
   return 0;
}


//  create scaled and curved alignment images in A1rgb16, A2rgb16
//  newf:  create new alignment images from scratch
//  warp:  0: curve both images, warp both halves of image2
//         1: curve image2 only, warp upper half only
//         2: curve image2 only, warp lower half only
//  global variables warpxu/yu and warpxl/yl determine the amount of warp

void pano_get_align_images(int newf, int warp)
{
   void  pano_curve_image(RGB *rgbin, RGB *rgbout, int curve, int warp);

   double      lens_curve, R;

   double   mm[12] =    { 20, 22, 25,   26,  28,  32,   40,   48,   60,  80,   100, 200 };
   double   curve[12] = { 35, 19, 10.1, 8.7, 6.8, 5.26, 3.42, 2.32, 1.7, 1.35, 1.2, 1.1 };

   if (newf) 
   {
      A1ww = Fww * alignSize / fullSize;                                   //  size of alignment images
      A1hh = Fhh * alignSize / fullSize;
      A2ww = Gww * alignSize / fullSize;
      A2hh = Ghh * alignSize / fullSize;
      
      RGB_free(pano_A1cache);                                              //  align images = scaled input images
      RGB_free(pano_A2cache);
      pano_A1cache = RGB_rescale(Frgb16,A1ww,A1hh);                        //  new scaled align images
      pano_A2cache = RGB_rescale(Grgb16,A2ww,A2hh);

      RGB_free(A1rgb16);                                                   //  A1/A2 will be curved
      RGB_free(A2rgb16);
      A1rgb16 = A2rgb16 = 0;

      spline1(12,mm,curve);                                                //  always initialize   bugfix v.6.2
   }
   
   lens_curve = spline2(lens_mm);
   pano_curve = lens_curve * 0.01 * A2ww;                                  //  curve % to pixels
   pano_bow = 0.01 * A2ww * lens_bow;                                      //  lens_bow % to pixels
   xshrink = 0.5 * pano_curve;                                             //  image shrinkage from curving
   yshrink = xshrink * pano_ycurveF * A2hh / A2ww;

   R = 1.0 * A2hh / A2ww;
   if (R > 1) pano_curve = pano_curve / R / R;                             //  adjust for vertical format
   
   if (warp == 0) {                                                        //  curve image1
      if (A2ww <= 0.8 * A1ww) {
         RGB_free(A1rgb16);                                                //  already curved via prior pano
         A1rgb16 = RGB_copy(pano_A1cache);                                 //  make a copy
      }
      else {
         if (! A1rgb16) A1rgb16 = RGB_make(A1ww,A1hh,16);                  //  curve image1, both halves, no warp
         pano_curve_image(pano_A1cache,A1rgb16,2,0);
      }
   }

   if (! A2rgb16) A2rgb16 = RGB_make(A2ww,A2hh,16);

   if (warp == 0) pano_curve_image(pano_A2cache,A2rgb16,2,3);              //  curve and warp all image2
   if (warp == 1) pano_curve_image(pano_A2cache,A2rgb16,1,1);              //    ""  left-upper quadrant
   if (warp == 2) pano_curve_image(pano_A2cache,A2rgb16,1,2);              //    ""  left-lower quadrant

   return;
}


//  curve and warp alignment image
//    curve:   1: left half only   2: both halves
//    warp:    0: none  1: upper half  2: lower half  3: both
//
//  global variables:
//    warpxu/yu:    upper left corner warp displacement
//    warpxl/yl:    lower left corner warp displacement

void  pano_curve_image(RGB *rgbin, RGB *rgbout, int curve, int warp)       //  overhauled  v.8.5
{
   int         pxc, pyc, ww, hh, vstat;
   int         pycL, pycH, pxcL, pxcH;
   double      px, py, xdisp, ydisp, xpull, ypull;
   uint16      vpix[3], *pixc;
   
   ww = rgbout->ww;
   hh = rgbout->hh;
   
   pxcL = 0;
   pxcH = ww;

   if (curve == 1) {                                                       //  curve left half only
      pxcL = pxmL - xoff;                                                  //  from blend stripe to middle
      pxcH = pxmH - xoff;                                                  //  v.7.7
   }

   pycL = 0;
   pycH = hh;
   if (warp == 1) pycH = hh / 2;                                           //  upper half only
   if (warp == 2) pycL = hh / 2;                                           //  lower half only

   for (pyc = pycL; pyc < pycH; pyc++)
   for (pxc = pxcL; pxc < pxcH; pxc++)
   {
      xdisp = (pxc - ww/2.0) / (ww/2.0);                                   //  -1 ... 0 ... +1
      ydisp = (pyc - hh/2.0) / (ww/2.0);
      xpull = xdisp * xdisp * xdisp;
      ypull = pano_ycurveF * ydisp * xdisp * xdisp;

      px = pxc + pano_curve * xpull;                                       //  apply lens curve factor
      py = pyc + pano_curve * ypull;
      px -= pano_bow * xdisp * ydisp * ydisp;                              //  apply lens bow factor

      if (warp && xdisp < 0) {                                             //  bugfix                    v.8.6.1
         if (pyc < hh / 2) {                                               //  warp upper-left quadrant
            px += warpxu * xdisp;
            py += warpyu * ydisp * xdisp;                                  //  bugfix                    v.8.6.1
         }
         else {
            px += warpxl * xdisp;                                          //  warp lower-left
            py += warpyl * ydisp * xdisp;
         }
      }

      vstat = vpixel(rgbin,px,py,vpix);                                    //  input virtual pixel
      pixc = bmpixel(rgbout,pxc,pyc);                                      //  output real pixel
      if (vstat) {  
         pixc[0] = vpix[0];
         pixc[1] = vpix[1];
         pixc[2] = vpix[2];
      }
      else pixc[0] = pixc[1] = pixc[2] = 0;
   }
   
   return;
}


//  combine and images: A1rgb16 + A2rgb16  >>  E3rgb16
//  update window showing current progress

void pano_combine(int fcolor)
{
   int            px3, py3, ii, max, vstat1, vstat2;
   int            red1, green1, blue1;
   int            red2, green2, blue2;
   int            red3, green3, blue3;
   uint16         vpix1[3], vpix2[3], *pix3;
   double         ww, px1, py1, px2, py2, f1, f2;
   double         costf = cos(toff), sintf = sin(toff);

   mutex_lock(&pixmaps_lock);
   
   ww = xoff + A2ww;                                                       //  combined width
   if (toff < 0) ww -= A2hh * toff;                                        //  adjust for theta
   E3ww = int(ww+1);
   E3hh = A1rgb16->hh;
   
   RGB_free(E3rgb16);                                                      //  allocate output pixmap
   E3rgb16 = RGB_make(E3ww,E3hh,16);
   
   overlapixs = 0;                                                         //  counts overlapping pixels
   red1 = green1 = blue1 = 0;                                              //  suppress compiler warnings
   red2 = green2 = blue2 = 0;
   
   for (py3 = 0; py3 < E3hh; py3++)                                        //  step through E3 rows
   for (px3 = 0; px3 < E3ww; px3++)                                        //  step through E3 pixels in row
   {
      vstat1 = vstat2 = 0;
      red3 = green3 = blue3 = 0;

      if (px3 < pxmH) {
         px1 = costf * px3 - sintf * (py3 - yoff);                         //  A1 pixel, after offsets  
         py1 = costf * py3 + sintf * (px3 - xoff);
         vstat1 = vpixel(A1rgb16,px1,py1,vpix1);
      }

      if (px3 >= pxmL) {
         px2 = costf * (px3 - xoff) + sintf * (py3 - yoff);                //  A2 pixel, after offsets
         py2 = costf * (py3 - yoff) - sintf * (px3 - xoff);
         vstat2 = vpixel(A2rgb16,px2,py2,vpix2);
      }

      if (vstat1) {
         red1 = vpix1[0];
         green1 = vpix1[1];
         blue1 = vpix1[2];
         if (!red1 && !green1 && !blue1) vstat1 = 0;                       //  ignore black pixels
      }

      if (vstat2) {
         red2 = vpix2[0];
         green2 = vpix2[1];
         blue2 = vpix2[2];
         if (!red2 && !green2 && !blue2) vstat2 = 0;
      }

      if (fcolor) {                                                        //  brightness compensation  
         if (vstat1) {                                                     //  (auto + manual adjustments)
            red1 = int(R12match[red1]);
            green1 = int(G12match[green1]);
            blue1 = int(B12match[blue1]);
            if (red1 > 65535 || green1 > 65535 || blue1 > 65535) {
               max = red1;
               if (green1 > max) max = green1;
               if (blue1 > max) max = blue1;
               f1 = 65535.0 / max;
               red1 = int(red1 * f1);
               green1 = int(green1 * f1);
               blue1 = int(blue1 * f1);
            }
         }

         if (vstat2) {                                                     //  adjust both images
            red2 = int(R21match[red2]);                                    //    in opposite directions
            green2 = int(G21match[green2]);
            blue2 = int(B21match[blue2]);
            if (red2 > 65535 || green2 > 65535 || blue2 > 65535) {
               max = red2;
               if (green2 > max) max = green2;
               if (blue2 > max) max = blue2;
               f1 = 65535.0 / max;
               red2 = int(red2 * f1);
               green2 = int(green2 * f1);
               blue2 = int(blue2 * f1);
            }
         }
      }

      if (vstat1) {
         if (! vstat2) {
            red3 = red1;                                                   //  use image1 pixel
            green3 = green1;
            blue3 = blue1; 
         }
         else {
            overlapixs++;                                                  //  count overlapped pixels
            if (fcolor) {
               if (alignWidth == 0) f1 = 1.0;
               else f1 = 1.0 * (pxmH - px3) / alignWidth;                  //  use progressive blend
               f2 = 1.0 - f1;
               red3 = int(f1 * red1 + f2 * red2);
               green3 = int(f1 * green1 + f2 * green2);
               blue3 = int(f1 * blue1 + f2 * blue2);
            }
            else {                                                         //  use 50/50 mix
               red3 = (red1 + red2) / 2;
               green3 = (green1 + green2) / 2;
               blue3 = (blue1 + blue2) / 2;
            }
         }
      }

      else if (vstat2) {
         red3 = red2;                                                      //  use image2 pixel
         green3 = green2;
         blue3 = blue2; 
      }

      pix3 = bmpixel(E3rgb16,px3,py3);                                     //  output pixel
      pix3[0] = red3;
      pix3[1] = green3;
      pix3[2] = blue3;
      
      if (showRedpix && vstat1 && vstat2) {                                //  highlight alignment pixels
         ii = py3 * A1ww + px3;
         if (redpixels[ii]) {
            pix3[0] = 65535;
            pix3[1] = pix3[2] = 0;
         }
      }
   }

   mutex_unlock(&pixmaps_lock);
   mwpaint2();                                                             //  update window
   return;
}


/**************************************************************************
    HDR and pano shared functions (image matching, alignment, overlay)
***************************************************************************/

//  compare two doubles for significant difference
//  return:  0  difference not significant
//          +1  d1 > d2
//          -1  d1 < d2

int sigdiff(double d1, double d2, double signf)
{
   double diff = fabs(d1-d2);
   if (diff == 0.0) return 0;
   diff = diff / (fabs(d1) + fabs(d2));
   if (diff < signf) return 0;
   if (d1 > d2) return 1;
   else return -1;
}


/**************************************************************************/

//  Get the rectangle containing the overlap region of two images
//  outputs: pxL pxH pyL pyH        total image overlap rectangle
//           pxmL pxmH pymL pymH    reduced to overlap align area

void getAlignArea()
{
   int         pxL2, pyL2;

   pxL = 0;
   if (xoff > 0) pxL = int(xoff);

   pxH = A1ww;
   if (pxH > xoff + A2ww) pxH = int(xoff + A2ww);
   
   pyL = 0;
   if (yoff > 0) pyL = int(yoff);
   
   pyH = A1hh;
   if (pyH > yoff + A2hh) pyH = int(yoff + A2hh);

   if (toff > 0) {
      pyL2 = int(yoff + toff * (pxH - pxL));
      if (pyL2 > pyL) pyL = pyL2;
   }

   if (toff < 0) {   
      pxL2 = int(xoff - toff * (pyH - pyL));
      if (pxL2 > pxL) pxL = pxL2;
   }
   
   if (xshrink > 0) {
      pxL = pxL + xshrink;                                                 //  reduce overlap area by amount
      pxH = pxH - xshrink;                                                 //   of image shrink (pano, HDF)
      pyL = pyL + yshrink;
      pyH = pyH - yshrink;
   }

   pxM = (pxL + pxH) / 2;                                                  //  midpoint of overlap
   pyM = (pyL + pyH) / 2;

   if (alignWidth < (pxH - pxL)) {
      pxmL = pxM - alignWidth/2;                                           //  overlap area width for
      pxmH = pxM + alignWidth/2;                                           //    image matching & blending
      if (pxmL < pxL) pxmL = pxL;
      if (pxmH > pxH) pxmH = pxH;
   }
   else {                                                                  //  use whole range
      pxmL = pxL;
      pxmH = pxH;
   }

   if (alignHeight < (pyH - pyL)) {
      pymL = pyM - alignHeight/2;
      pymH = pyM + alignHeight/2;
      if (pymL < pyL) pymL = pyL;
      if (pymH > pyH) pymH = pyH;
   }
   else {
      pymL = pyL;
      pymH = pyH;
   }

   return;
}


/**************************************************************************/

//  Compute brightness ratio by color for overlapping image areas.
//    (image2 is overlayed on image1, offset by xoff, yoff, toff)
//
//  Outputs: 
//    Bratios1[rgb][ii] = image2/image1 brightness ratio for color rgb
//                        and image1 brightness ii
//    Bratios2[rgb][ii] = image1/image2 brightness ratio for color rgb
//                        and image2 brightness ii

void getBrightRatios()
{
   uint16      vpix1[3], vpix2[3];
   int         vstat1, vstat2;
   int         px, py, pxinc, pyinc, ii, jj, rgb;
   int         npix, npix1, npix2, npix3;
   int         brdist1[3][256], brdist2[3][256];
   double      px1, py1, px2, py2;
   double      brlev1[3][256], brlev2[3][256];
   double      costf = cos(toff), sintf = sin(toff);
   double      a1, a2, b1, b2, bratio = 1;
   double      s8 = 1.0 / 256.0;
   
   for (rgb = 0; rgb < 3; rgb++)                                           //  clear distributions
   for (ii = 0; ii < 256; ii++)
      brdist1[rgb][ii] = brdist2[rgb][ii] = 0;

   pxinc = pyinc = 1;                                                      //  bugfix  v.7.7.1
   npix = (pxH - pxL) * (pyH - pyL);
   if (npix > 500000) pxinc = 2;                                           //  reduce excessive sample  v.7.7
   if (npix > 1000000) pyinc = 2;

   npix = 0;
   
   for (py = pyL; py < pyH; py += pyinc)                                   //  scan image1/image2 pixels parallel
   for (px = pxL; px < pxH; px += pxinc)                                   //  use entire overlap area
   {
      px1 = costf * px - sintf * (py - yoff);                              //  image1 pixel, after offsets
      py1 = costf * py + sintf * (px - xoff);
      vstat1 = vpixel(A1rgb16,px1,py1,vpix1);
      if (! vstat1) continue;                                              //  does not exist
      if (!vpix1[0] && !vpix1[1] && !vpix1[2]) continue;                   //  ignore black pixels

      px2 = costf * (px - xoff) + sintf * (py - yoff);                     //  corresponding image2 pixel
      py2 = costf * (py - yoff) - sintf * (px - xoff);
      vstat2 = vpixel(A2rgb16,px2,py2,vpix2);
      if (! vstat2) continue;                                              //  does not exist
      if (!vpix2[0] && !vpix2[1] && !vpix2[2]) continue;                   //  ignore black pixels

      ++npix;                                                              //  count overlapping pixels
      
      for (rgb = 0; rgb < 3; rgb++)                                        //  accumulate distributions
      {                                                                    //    by color in 256 bins
         ++brdist1[rgb][int(s8*vpix1[rgb])];
         ++brdist2[rgb][int(s8*vpix2[rgb])];
      }
   }
   
   npix1 = npix / 256;                                                     //  1/256th of total pixels
   
   for (rgb = 0; rgb < 3; rgb++)                                           //  get brlev1[rgb][N] = mean bright
   for (ii = jj = 0; jj < 256; jj++)                                       //    for Nth group of image1 pixels
   {                                                                       //      for color rgb
      brlev1[rgb][jj] = 0;
      npix2 = npix1;                                                       //  1/256th of total pixels

      while (npix2 > 0 && ii < 256)                                        //  next 1/256th group from distr,
      {
         npix3 = brdist1[rgb][ii];
         if (npix3 == 0) { ++ii; continue; }
         if (npix3 > npix2) npix3 = npix2;
         brlev1[rgb][jj] += ii * npix3;                                    //  brightness * (pixels with)
         brdist1[rgb][ii] -= npix3;
         npix2 -= npix3;
      }

      brlev1[rgb][jj] = brlev1[rgb][jj] / npix1;                           //  mean brightness for group, 0-255
   }

   for (rgb = 0; rgb < 3; rgb++)                                           //  do same for image2
   for (ii = jj = 0; jj < 256; jj++)
   {
      brlev2[rgb][jj] = 0;
      npix2 = npix1;

      while (npix2 > 0 && ii < 256)
      {
         npix3 = brdist2[rgb][ii];
         if (npix3 == 0) { ++ii; continue; }
         if (npix3 > npix2) npix3 = npix2;
         brlev2[rgb][jj] += ii * npix3;
         brdist2[rgb][ii] -= npix3;
         npix2 -= npix3;
      }

      brlev2[rgb][jj] = brlev2[rgb][jj] / npix1;
   }

   for (rgb = 0; rgb < 3; rgb++)                                           //  color
   for (ii = jj = 0; ii < 256; ii++)                                       //  brlev1 brightness, 0 to 255
   {                                                                       //  bugfix  v.6.4
      if (ii == 0) bratio = 1;
      while (ii > brlev2[rgb][jj] && jj < 256) ++jj;                       //  find matching brlev2 brightness
      a2 = brlev2[rgb][jj];                                                //  next higher value
      b2 = brlev1[rgb][jj];
      if (a2 > 0 && b2 > 0) {
         if (jj > 0) {
            a1 = brlev2[rgb][jj-1];                                        //  next lower value
            b1 = brlev1[rgb][jj-1];
         }
         else   a1 = b1 = 0;
         if (ii == 0)  bratio = b2 / a2;
         else   bratio = (b1 + (ii-a1)/(a2-a1) * (b2-b1)) / ii;            //  interpolate
      }

      if (bratio < 0.2) bratio = 0.2;                                      //  contain outliers
      if (bratio > 5) bratio = 5;
      Bratios2[rgb][ii] = bratio;
   }

   for (rgb = 0; rgb < 3; rgb++)                                           //  color
   for (ii = jj = 0; ii < 256; ii++)                                       //  brlev2 brightness, 0 to 255
   {                                                                       //  bugfix  v.6.4
      if (ii == 0) bratio = 1;
      while (ii > brlev1[rgb][jj] && jj < 256) ++jj;                       //  find matching brlev1 brightness
      a2 = brlev1[rgb][jj];                                                //  next higher value
      b2 = brlev2[rgb][jj];
      if (a2 > 0 && b2 > 0) {
         if (jj > 0) {
            a1 = brlev1[rgb][jj-1];                                        //  next lower value
            b1 = brlev2[rgb][jj-1];
         }
         else   a1 = b1 = 0;
         if (ii == 0)  bratio = b2 / a2;
         else   bratio = (b1 + (ii-a1)/(a2-a1) * (b2-b1)) / ii;            //  interpolate
      }

      if (bratio < 0.2) bratio = 0.2;                                      //  contain outliers
      if (bratio > 5) bratio = 5;
      Bratios1[rgb][ii] = bratio;
   }

   return;
}


/**************************************************************************/

//  Set color matching factors
//     on:   Bratios are used for color matching the two images
//     off:  Bratios are not used - use 1.0 instead
//  In both cases, manual settings Radjust/Gadjust/Badjust are used

void setColorfixFactors(int state)
{
   unsigned      ii, jj;

   if (state) 
   {
      for (ii = 0; ii < 65536; ii++)
      {
         jj = ii >> 8;

         R12match[ii] = sqrt(Bratios1[0][jj]) / Radjust * ii;              //  use sqrt(ratio) so that adjustment
         G12match[ii] = sqrt(Bratios1[1][jj]) / Gadjust * ii;              //    can be applied to both images
         B12match[ii] = sqrt(Bratios1[2][jj]) / Badjust * ii;              //      in opposite directions

         R21match[ii] = sqrt(Bratios2[0][jj]) * Radjust * ii;
         G21match[ii] = sqrt(Bratios2[1][jj]) * Gadjust * ii;
         B21match[ii] = sqrt(Bratios2[2][jj]) * Badjust * ii;
      }
   }

   else 
   {
      for (ii = 0; ii < 65536; ii++)
      {
         R12match[ii] = 1.0 / Radjust * ii;
         G12match[ii] = 1.0 / Gadjust * ii;
         B12match[ii] = 1.0 / Badjust * ii;

         R21match[ii] = Radjust * ii;
         G21match[ii] = Gadjust * ii;
         B21match[ii] = Badjust * ii;
      }
   }
   
   return;
}


/**************************************************************************/

//  find pixels of greatest contrast within overlap area
//  flag high-contrast pixels to use in each image compare region

void flagEdgePixels()
{
   void  flagEdgePixels2(int pxL, int pxH, int pyL, int pyH, int samp);

   int      samp = pixsamp / 9;
   int      pxm1, pxm2, pym1, pym2;
   
   if (redpixels) zfree(redpixels);                                        //  clear flags for alignment pixels
   redpixels = zmalloc(A1ww*A1hh,"redpixels");
   memset(redpixels,0,A1ww*A1hh);
   
   pxm1 = pxmL + 0.333 * (pxmH - pxmL);
   pxm2 = pxmL + 0.667 * (pxmH - pxmL);
   pym1 = pymL + 0.333 * (pymH - pymL);
   pym2 = pymL + 0.667 * (pymH - pymL);
   
   flagEdgePixels2(pxmL+8, pxm1,    pymL+8, pym1,    samp);                //  9 zones   v.8.0
   flagEdgePixels2(pxmL+8, pxm1,    pym1,   pym2,    samp);
   flagEdgePixels2(pxmL+8, pxm1,    pym2,   pymH-10, samp);
   flagEdgePixels2(pxm1,   pxm2,    pymL+8, pym1,    samp);
   flagEdgePixels2(pxm1,   pxm2,    pym1,   pym2,    samp);
   flagEdgePixels2(pxm1,   pxm2,    pym2,   pymH-10, samp);
   flagEdgePixels2(pxm2,   pxmH-10, pymL+8, pym1,    samp);
   flagEdgePixels2(pxm2,   pxmH-10, pym1,   pym2,    samp);
   flagEdgePixels2(pxm2,   pxmH-10, pym2,   pymH-10, samp);

   return;
}


//  Find the highest contrast pixels meeting sample size
//  within the specified sub-region of image overlap.

void flagEdgePixels2(int pxL, int pxH, int pyL, int pyH, int samp)
{
   int         px, py, ii, jj, npix, vstat1, vstat2, vstat3;
   int         red1, green1, blue1, red2, green2, blue2, tcon;
   int         Hdist[256], Vdist[256], Hmin, Vmin;
   double      costf = cos(toff), sintf = sin(toff);
   double      px1, py1, px2, py2, s8 = 1.0 / 769.0;
   uchar       *Hcon, *Vcon;
   uint16      vpix1[3], vpix2[3], vpix3[3];
   
   npix = (pxH - pxL) * (pyH - pyL);                                       //  overlapping pixels
   if (npix < 100) return;                                                 //  insignificant
   if (samp > npix / 4) samp = npix / 4;                                   //  use max. 1/4 of pixels

   Hcon = (uchar *) zmalloc(npix,"redpix.hcon");                           //  horizontal pixel contrast 0-255
   Vcon = (uchar *) zmalloc(npix,"redpix.vcon");                           //  vertical pixel contrast 0-255

   for (py = pyL; py < pyH; py++)                                          //  scan image pixels in sub-region
   for (px = pxL; px < pxH; px++)
   {
      ii = (py-pyL) * (pxH-pxL) + (px-pxL);
      Hcon[ii] = Vcon[ii] = 0;                                             //  horiz. = vert. contrast = 0

      px1 = costf * px - sintf * (py - yoff);                              //  image1 pixel
      py1 = costf * py + sintf * (px - xoff);
      vstat1 = vpixel(A1rgb16,px1,py1,vpix1);
      if (! vstat1) continue;                                              //  does not exist
      if (!vpix1[0] && !vpix1[1] && !vpix1[2]) continue;                   //  ignore black pixels

      px2 = costf * (px - xoff) + sintf * (py - yoff);                     //  corresponding image2 pixel
      py2 = costf * (py - yoff) - sintf * (px - xoff);                     //  v.7.5
      vstat2 = vpixel(A2rgb16,px2,py2,vpix2);
      if (! vstat2) continue;
      if (!vpix2[0] && !vpix2[1] && !vpix2[2]) continue;

      vstat3 = vpixel(A1rgb16,px1+4,py1,vpix3);                            //  4 pixels to right
      if (! vstat3) continue;                                              //  reject if off edge
      if (! vpix3[0] && ! vpix3[1] && ! vpix3[2]) continue;

      vstat3 = vpixel(A1rgb16,px1,py1+4,vpix3);                            //  4 pixels below
      if (! vstat3) continue;
      if (! vpix3[0] && ! vpix3[1] && ! vpix3[2]) continue;

      vstat3 = vpixel(A1rgb16,px1+4,py1+4,vpix3);                          //  verify overlap of +4 pixels
      if (! vstat3) continue;                                              //    in all directions   v.7.5
      if (!vpix3[0] && !vpix3[1] && !vpix3[2]) continue;
      
      vstat3 = vpixel(A2rgb16,px2+4,py2+4,vpix3);
      if (! vstat3) continue;
      if (!vpix3[0] && !vpix3[1] && !vpix3[2]) continue;

      vstat3 = vpixel(A1rgb16,px1-4,py1-4,vpix3);
      if (! vstat3) continue;
      if (!vpix3[0] && !vpix3[1] && !vpix3[2]) continue;
      
      vstat3 = vpixel(A2rgb16,px2-4,py2-4,vpix3);
      if (! vstat3) continue;
      if (!vpix3[0] && !vpix3[1] && !vpix3[2]) continue;

      red1 = vpix1[0];
      green1 = vpix1[1];
      blue1 = vpix1[2];

      vstat3 = vpixel(A1rgb16,px1+2,py1,vpix3);                            //  2 pixels to right
      red2 = vpix3[0];
      green2 = vpix3[1];
      blue2 = vpix3[2];
      tcon = abs(red1-red2) + abs(green1-green2) + abs(blue1-blue2);       //  horizontal contrast
      Hcon[ii] = int(tcon * s8);                                           //    0 - 255

      vstat3 = vpixel(A1rgb16,px1,py1+2,vpix3);                            //  2 pixels below
      red2 = vpix3[0];
      green2 = vpix3[1];
      blue2 = vpix3[2];
      tcon = abs(red1-red2) + abs(green1-green2) + abs(blue1-blue2);       //  vertical contrast
      Vcon[ii] = int(tcon * s8);
   }

   for (ii = 0; ii < 256; ii++) Hdist[ii] = Vdist[ii] = 0;                 //  clear contrast distributions

   for (py = pyL; py < pyH; py++)                                          //  scan image pixels      v.7.5
   for (px = pxL; px < pxH; px++)
   {                                                                       //  build contrast distributions
      ii = (py-pyL) * (pxH-pxL) + (px-pxL);
      ++Hdist[Hcon[ii]];
      ++Vdist[Vcon[ii]];
   }
   
   for (npix = 0, ii = 255; ii > 0; ii--)                                  //  find minimum contrast needed to get
   {                                                                       //    enough pixels for sample size
      npix += Hdist[ii];                                                   //      (horizontal contrast pixels)
      if (npix > samp) break; 
   }
   Hmin = ii; 

   for (npix = 0, ii = 255; ii > 0; ii--)                                  //  (verticle contrast pixels)
   {
      npix += Vdist[ii];
      if (npix > samp) break;
   }
   Vmin = ii;
   
   for (py = pyL; py < pyH; py++)                                          //  scan image pixels   v.7.5
   for (px = pxL; px < pxH; px++)
   {
      ii = (py-pyL) * (pxH-pxL) + (px-pxL);
      jj = py * A1ww + px;

      if (Hcon[ii] > Hmin) {
         redpixels[jj] = 1;                                                //  flag horizontal group of 3
         redpixels[jj+1] = 1;
         redpixels[jj+2] = 1;
      }

      if (Vcon[ii] > Vmin) {
         redpixels[jj] = 1;                                                //  flag verticle group of 3
         redpixels[jj+A1ww] = 1;
         redpixels[jj+2*A1ww] = 1;
      }
   }
   
   zfree(Hcon);
   zfree(Vcon);
   return;
}


/**************************************************************************/

//  Compare two images in overlapping areas.
//  (image2 is overlayed on image1, offset by xoff, yoff, toff).
//  Use pixels with contrast > minimum needed to reach sample size.
//  return: 1 = perfect match, 0 = total mismatch (black/white)

double matchImages()                                                       //  weighting removed   v.8.5
{
   uint16      vpix1[3], vpix2[3];
   int         px, py, ii, vstat1, vstat2;
   double      px1, py1, px2, py2;
   double      costf = cos(toff), sintf = sin(toff);
   double      match, cmatch, maxcmatch;
   
   if (pxM > A1ww || pyM > A1hh) return 0;                                 //  overlap runs off image, no match

   cmatch = maxcmatch = 0;

   for (py = pymL; py < pymH; py++)                                        //  step through image1 pixels, rows
   for (px = pxmL; px < pxmH; px++)                                        //  step through image1 pixels, cols
   {
      ii = py * A1ww + px;                                                 //  skip low-contrast pixels
      if (! redpixels[ii]) continue;

      px1 = costf * px - sintf * (py - yoff);                              //  image1 pixel
      py1 = costf * py + sintf * (px - xoff);
      vstat1 = vpixel(A1rgb16,px1,py1,vpix1);
      if (! vstat1) continue;                                              //  does not exist
      if (!vpix1[0] && !vpix1[1] && !vpix1[2]) continue;                   //  ignore black pixels

      px2 = costf * (px - xoff) + sintf * (py - yoff);                     //  corresponding image2 pixel
      py2 = costf * (py - yoff) - sintf * (px - xoff);
      vstat2 = vpixel(A2rgb16,px2,py2,vpix2);
      if (! vstat2) continue;
      if (!vpix2[0] && !vpix2[1] && !vpix2[2]) continue;
      
      match = matchPixels(vpix1,vpix2);                                    //  compare brightness adjusted
      cmatch += match;                                                     //  accumulate total match
      maxcmatch += 1.0;
   }

   return cmatch / maxcmatch;
}


/**************************************************************************/

//  Compare 2 pixels using precalculated brightness ratios
//  1.0 = perfect match   0 = total mismatch (black/white)

double matchPixels(uint16 *pix1, uint16 *pix2)
{
   double      red1, green1, blue1, red2, green2, blue2;
   double      reddiff, greendiff, bluediff, match;
   double      ff = 1.0 / 65536.0;

   red1 = R12match[pix1[0]];
   green1 = G12match[pix1[1]];
   blue1 = B12match[pix1[2]];

   red2 = R21match[pix2[0]];
   green2 = G21match[pix2[1]];
   blue2 = B21match[pix2[2]];

   reddiff = ff * fabs(red1-red2);                                         //  0 = perfect match
   greendiff = ff * fabs(green1-green2);                                   //  1 = total mismatch
   bluediff = ff * fabs(blue1-blue2);
   
   match = (1.0 - reddiff) * (1.0 - greendiff) * (1.0 - bluediff);         //  1 = perfect match
   return match;
}


/**************************************************************************/

//  Get a virtual pixel at location (px,py) (real) in an RGB-16 pixmap.
//  Get the overlapping real pixels and build a composite.

int vpixel(RGB *rgb, double px, double py, uint16 *vpix)                   //  overhauled   v.7.7
{
   int            ww, hh, px0, py0;
   uint16         *ppix, *pix0, *pix1, *pix2, *pix3;
   double         f0, f1, f2, f3;
   double         red, green, blue;
   
   ww = rgb->ww;
   hh = rgb->hh;
   ppix = (uint16 *) rgb->bmp;

   px0 = int(px);                                                          //  pixel containing (px,py)
   py0 = int(py);

   if (px0 < 1 || py0 < 1) return 0;
   if (px0 > ww-3 || py0 > hh-3) return 0;
   
   pix0 = ppix + (py0 * ww + px0) * 3;                                     //  4 pixels based at (px0,py0)
   pix1 = pix0 + ww * 3;
   pix2 = pix0 + 3;
   pix3 = pix0 + ww * 3 + 3;

   f0 = (px0+1 - px) * (py0+1 - py);                                       //  overlap of (px,py)
   f1 = (px0+1 - px) * (py - py0);                                         //   in each of the 4 pixels
   f2 = (px - px0) * (py0+1 - py);
   f3 = (px - px0) * (py - py0);
   
   red =   f0 * pix0[0] + f1 * pix1[0] + f2 * pix2[0] + f3 * pix3[0];      //  sum the weighted inputs
   green = f0 * pix0[1] + f1 * pix1[1] + f2 * pix2[1] + f3 * pix3[1];
   blue =  f0 * pix0[2] + f1 * pix1[2] + f2 * pix2[2] + f3 * pix3[2];
   
   vpix[0] = int(red);
   vpix[1] = int(green);
   vpix[2] = int(blue);

   return 1;
}


/**************************************************************************

   edit transaction and thread support functions         overhauled  v.6.2

   edit transaction management
      edit_setup()                     start new edit - copy E3 > E1
      edit_cancel()                    cancel edit - E1 > E3, delete E1
      edit_done()                      commit edit - add to undo stack
      edit_undo()                      undo edit - E1 > E3
      edit_redo()                      redo edit - run thread again
      edit_fullsize()                  convert preview to full-size pixmaps

   main level thread management
      start_thread(func,arg)           start thread running
      signal_thread()                  signal thread that work is pending
      wait_thread_idle()               wait for pending work complete
      wrapup_thread(command)           wait for exit or command thread exit
      thread_working()                 return idle/working status

   thread function
      thread_idle_loop()               wait for pending work, exit if commanded
      exit_thread()                    exit thread unconditionally
      
   thread_status (thread ownership
      0     no thread is running
      1     thread is running and idle (no work)
      2     thread is working
      0     thread has exited

   thread_command (main program ownership)
      0     idle, no work pending
      8     exit when pending work is done
      9     exit now, unconditionally

   thread_pend       work requested counter
   thread_done       work done counter
   thread_hiwater    high water mark
   edit_action       done/cancel/undo/redo in progress

***************************************************************************/

int      thread_command = 0, thread_status = 0;
int      thread_pend = 0, thread_done = 0, thread_hiwater = 0;
int      edit_action = 0;

/**************************************************************************

  Setup for a new edit transaction
  Create E1 (edit input) and E3 (edit output) pixmaps from
  previous edit output (Frgb16) or image file (new Frgb16).

  uprev      0     edit full-size image
             1     edit preview image unless select area exists

  uarea      0     select_area is invalid and will be deleted (e.g. rotate)
             1     select_area not used but remains valid (e.g. red-eye)
             2     select_area is used and remains valid (e.g. flatten)

***************************************************************************/

int edit_setup(cchar *func, int uprev, int uarea)
{
   int      yn;

   if (! image_file) return 0;                                             //  no image file
   if (! menulock(1)) return 0;                                            //  lock menu
   
   if (! Fexiftool && ! Fexifwarned) {
      zmessageACK(GTK_WINDOW(mWin),
                  ZTX("exiftool is not installed \n"                       //  warn if starting to edit
                      "edited images will lose EXIF data"));               //    and exiftool is missing  v.9.9
      Fexifwarned = 1;
   }

   if (Pundo > maxedits-2) {                                               //  v.10.2
      zmessageACK(GTK_WINDOW(mWin),ZTX("Too many edits, please save image"));
      menulock(0);
      return 0;
   }
   
   if (Fimageturned) {
      turn_image(-Fimageturned);                                           //  un-turn if needed
      uarea = 0;                                                           //  (select area did already)
   }

   if (uarea == 0 && sa_stat) {                                            //  select area will be lost, warn user
      yn = zmessageYN(GTK_WINDOW(mWin),ZTX("Select area cannot be kept.\n"
                                           "Continue?"));
      if (! yn) {
         menulock(0);
         return 0;
      }
      select_delete();
      zdialog_free(zdsela);
      zdsela = 0;
   }
   
   if (uprev && uarea == 2 && sa_stat && ! sa_active) {                    //  select area exists and can be used,
      yn = zmessageYN(GTK_WINDOW(mWin),ZTX("Select area not active.\n"     //    but not active, ask user    v.10.1
                                           "Continue?"));
      if (! yn) {
         menulock(0);
         return 0;
      }
   }
   
   Fpreview = 0;                                                           //  use preview image if supported
   if (uprev && ! (uarea == 2 && sa_active)) {                             //    and select area will not be used
      Fpreview = 1;
      if (Fzoom) {
         Fzoom = 0;                                                        //  bugfix, mpaint() req.    v.10.3.1
         mwpaint();
      }
   }

   if (Fpreview && Fshowarea) {                                            //  hide select area during preview mode edit
      select_show(0);                                                      //    and bring it back when done   v.9.7
      Fshowarea = 1;
   }
   
   mutex_lock(&pixmaps_lock);                                              //  lock pixmaps

   if (! Frgb16) Frgb16 = f_load(image_file,16);                           //  create Frgb16 if not already
   if (! Frgb16) {
      mutex_unlock(&pixmaps_lock);                                         //  should not happen
      menulock(0);
      return 0;
   }
   
   RGB_free(E1rgb16);                                                      //  free prior edit pixmaps
   RGB_free(E3rgb16);

   if (Fpreview)                                                           //  edit pixmaps are window-size
      E1rgb16 = RGB_rescale(Frgb16,dww,dhh);                               //  E1rgb16 = Frgb16 scaled to window
   else E1rgb16 = RGB_copy(Frgb16);                                        //  edit pixmaps are full-size
   E3rgb16 = RGB_copy(E1rgb16);                                            //  E1 >> E3

   E1ww = E3ww = E1rgb16->ww;
   E1hh = E3hh = E1rgb16->hh;
   
   if (Pundo == 0) {
      edit_function = "initial";                                           //  initial image >> undo stack    v.10.2
      save_undo();
   }

   edit_function = func;                                                   //  set current edit function    v.10.2
   Fmodified = 0;                                                          //  image not modified yet
   thread_command = thread_status = Fkillfunc = 0;                         //  no thread running
   thread_pend = thread_done = thread_hiwater = 0;                         //  no work pending or done

   mutex_unlock(&pixmaps_lock);
   mwpaint2();
   return 1;
}


/**************************************************************************/

//  process edit cancel

void edit_cancel()
{
   if (edit_action) return;                                                //  v.6.8
   edit_action++;

   Fkillfunc = 1;                                                          //  tell thread to quit now  v.9.1
   wrapup_thread(9);                                                       //  wait for exit
   Fkillfunc = 0;

   if (zdedit) {                                                           //  v.6.6
      zdialog_free(zdedit);                                                //  kill dialog
      zdedit = null;
   }

   mutex_lock(&pixmaps_lock);
   RGB_free(E1rgb16);                                                      //  free edit pixmaps E1, E3
   RGB_free(E3rgb16);
   RGB_free(ERrgb16);                                                      //  free redo copy    v.10.3
   E1rgb16 = E3rgb16 = ERrgb16 = 0;
   E1ww = E3ww = ERww = 0;

   Fmodified = Fpreview = 0;                                               //  reset flags
   Ntoplines = Nptoplines;                                                 //  no overlay lines
   paint_toparc(2);                                                        //  no brush outline
   mutex_unlock(&pixmaps_lock);
   menulock(0);                                                            //  unlock menu
   mwpaint2();                                                             //  refresh window
   edit_action = 0;
   return;
}   


/**************************************************************************/

//  process edit dialog [done]  
//  E3rgb16 >> Frgb16 >> Frgb8

void edit_done()
{
   if (edit_action) return;                                                //  v.6.8
   edit_action++;

   wait_thread_idle();                                                     //  bugfix           v.10.2

   if (Fpreview && Fmodified) {
      Fzoom = 0;                                                           //  v.8.3
      edit_fullsize();                                                     //  update full image
   }

   wrapup_thread(8);                                                       //  tell thread to exit

   if (zdedit) {                                                           //  v.6.6
      zdialog_free(zdedit);                                                //  kill dialog
      zdedit = null;
   }
   
   mutex_lock(&pixmaps_lock);

   if (Fmodified) {
      RGB_free(Frgb16);                                                    //  memory leak   v.6.8
      Frgb16 = RGB_copy(E3rgb16);                                          //  E3 >> Frgb16
      RGB_free(Frgb8);
      Frgb8 = RGB_convbpc(Frgb16);                                         //  Frgb16 >> Frgb8
      Fww = Frgb8->ww;
      Fhh = Frgb8->hh;
      Pundo++;
      Pumax = Pundo;
      save_undo();                                                         //  save next undo state
   }

   RGB_free(E1rgb16);                                                      //  free edit pixmaps
   RGB_free(E3rgb16);
   RGB_free(ERrgb16);                                                      //  free redo copy    v.10.3
   E1rgb16 = E3rgb16 = ERrgb16 = 0;
   E1ww = E3ww = ERww = 0;
   
   Fmodified = Fpreview = 0;                                               //  reset flags
   Ntoplines = Nptoplines;                                                 //  no overlay lines
   paint_toparc(2);                                                        //  no brush outline
   mutex_unlock(&pixmaps_lock);
   menulock(0);                                                            //  unlock menu
   mwpaint2();                                                             //  update window
   edit_action = 0;
   return;
}


/**************************************************************************/

//  edit undo, redo, reset functions

void edit_undo()
{
   if (! Fmodified) return;                                                //  v.6.3
   if (thread_status == 2) return;                                         //  bugfix, thread busy      v.10.2
   if (edit_action) return;                                                //  v.6.8
   edit_action++;
   
   mutex_lock(&pixmaps_lock);
   RGB_free(ERrgb16);                                                      //  make redo copy    v.10.3
   ERrgb16 = E3rgb16;
   ERww = E3ww;
   ERhh = E3hh;
   E3rgb16 = RGB_copy(E1rgb16);                                            //  restore initial image status
   E3ww = E1ww;
   E3hh = E1hh;
   Fmodified = 0;                                                          //  reset image modified status
   mutex_unlock(&pixmaps_lock);

   mwpaint2();                                                             //  refresh window      v.9.8
   edit_action = 0;
   return;
}


void edit_redo()
{
   if (! ERrgb16) return;
   if (edit_action) return;                                                //  v.6.8
   edit_action++;
   RGB_free(E3rgb16);                                                      //  edit image = redo copy    v.10.3
   E3rgb16 = ERrgb16;
   E3ww = ERww;
   E3hh = ERhh;
   ERrgb16 = 0;
   Fmodified = 1;
   mwpaint2();
   edit_action = 0;
   return;
}


void edit_reset()                                                          //  new  v.10.3
{
   if (! Fmodified) return;
   if (thread_status == 2) return;
   if (edit_action) return;
   edit_action++;

   mutex_lock(&pixmaps_lock);
   RGB_free(ERrgb16);                                                      //  no redo copy
   ERrgb16 = 0;
   RGB_free(E3rgb16);
   E3rgb16 = RGB_copy(E1rgb16);                                            //  restore initial image
   E3ww = E1ww;
   E3hh = E1hh;
   Fmodified = 0;                                                          //  reset image modified status
   mutex_unlock(&pixmaps_lock);

   mwpaint2();                                                             //  refresh window      v.9.8
   edit_action = 0;
   return;
}


void edit_zapredo()
{
   RGB_free(ERrgb16);                                                      //  no redo copy
   ERrgb16 = 0;
   return;
}


/**************************************************************************/

//  Convert from preview mode (window-size pixmaps) to full-size pixmaps.

void edit_fullsize()                                                       //  v.6.2
{
   if (! Fpreview) return;
   Fpreview = 0;

   mutex_lock(&pixmaps_lock);
   RGB_free(E1rgb16);                                                      //  free preview pixmaps
   RGB_free(E3rgb16);
   E1rgb16 = RGB_copy(Frgb16);                                             //  make full-size pixmaps
   E3rgb16 = RGB_copy(Frgb16);
   E1ww = E3ww = Fww;
   E1hh = E3hh = Fhh;
   mutex_unlock(&pixmaps_lock);

   signal_thread();                                                        //  signal thread to do edit
   wait_thread_idle();
   return;
}


/**************************************************************************/

//  start thread that does the edit work

void start_thread(threadfunc func, void *arg)
{
   thread_status = 1;                                                      //  thread is running
   thread_command = thread_pend = thread_done = thread_hiwater = 0;        //  nothing pending
   start_detached_thread(func,arg);
   return;
}


//  signal thread that work is pending

void signal_thread()
{
   edit_zapredo();                                                         //  reset redo copy
   if (thread_status > 0) thread_pend++;                                   //  v.6.2
   return;
}


//  wait for edit thread to complete pending work and become idle

void wait_thread_idle()
{
   while (thread_status && thread_pend > thread_done)
   {
      zmainloop();
      zsleep(0.01);
   }
   
   return;
}


//  wait for thread exit or command thread exit
//  command = 0    wait for normal completion
//            8    finish pending work and exit
//            9    quit, exit now

void wrapup_thread(int command)
{
   thread_command = command;                                               //  tell thread to quit or finish

   while (thread_status > 0)                                               //  wait for thread to finish
   {                                                                       //    pending work and exit
      zmainloop();
      zsleep(0.01);
   }

   return;
}


//  return thread idle/working status

int thread_working()                                                       //  v.8.4
{
   if (thread_status == 2) return 1;
   return 0;
}


//  called only from edit threads
//  idle loop - wait for work request or exit command

void thread_idle_loop()
{
   thread_status = 1;                                                      //  status = idle
   thread_done = thread_hiwater;                                           //  work done = high-water mark

   while (true)
   {
      if (thread_command == 9) exit_thread();                              //  quit now command
      if (thread_command == 8)                                             //  finish work and exit
         if (thread_pend <= thread_done) exit_thread();
      if (thread_pend > thread_done) break;                                //  wait for work request
      zsleep(0.01);
   }
   
   thread_hiwater = thread_pend;                                           //  set high-water mark
   thread_status = 2;                                                      //  thread is working
   return;                                                                 //  loop to thread
}


//  called only from edit threads
//  exit thread unconditionally

void exit_thread()
{
   thread_pend = thread_done = thread_hiwater = 0;
   thread_status = 0;
   pthread_exit(0);                                                        //  "return" cannot be used here
}


/**************************************************************************/

//  edit support functions for working threads (one per processor core)

void start_wt(threadfunc func, void *arg)
{
   zadd_locked(wthreads_busy,+1);
   start_detached_thread(func,arg);
   return;
}


void exit_wt()
{
   zadd_locked(wthreads_busy,-1);
   pthread_exit(0);                                                        //  "return" cannot be used here  v.9.4
}


void wait_wts()
{
   while (wthreads_busy) 
   {
      zmainloop();
      zsleep(0.01);
   }
   
   return;
}


/**************************************************************************
      undo / redo toolbar buttons
***************************************************************************/


//  [undo] menu function - reinstate previous edit in undo/redo stack

void m_undo(GtkWidget *, cchar *)
{
   if (zdedit) {                                                           //  v.10.2
      zdialog_send_event(zdedit,"undo");
      return;
   }
   if (Pundo == 0) return;
   if (! menulock(1)) return;
   Pundo--;
   load_undo();
   menulock(0);
   return;
}


//  [redo] menu function - reinstate next edit in undo/redo stack

void m_redo(GtkWidget *, cchar *)
{
   if (zdedit) {                                                           //  v.10.2
      zdialog_send_event(zdedit,"redo");
      return;
   }
   if (Pundo == Pumax) return;
   if (! menulock(1)) return;
   Pundo++;
   load_undo();
   menulock(0);
   return;
}


//  Save Frgb16 to undo/redo file stack
//  stack position = Pundo

void save_undo()
{
   char     *pp, buff[24];
   int      fid, cc, cc2;
   
   pp = strstr(undo_files,"_undo_");
   if (! pp) zappcrash("undo/redo stack corrupted 1");
   snprintf(pp+6,3,"%02d",Pundo);
   
   fid = open(undo_files,O_WRONLY|O_CREAT|O_TRUNC,0640);
   if (! fid) zappcrash("undo/redo stack corrupted 2");

   snprintf(buff,24," %05d %05d fotoxx ",Fww,Fhh);
   cc = write(fid,buff,20);
   if (cc != 20) zappcrash("undo/redo stack corrupted 3");
   
   cc = Fww * Fhh * 6;
   cc2 = write(fid,Frgb16->bmp,cc);
   if (cc2 != cc) zappcrash("undo/redo stack corrupted 4");

   close(fid);

   pvlist_replace(editlog,Pundo,edit_function);                            //  save log of edits done    v.10.2
   return;
}


//  Load Frgb16 from undo/redo file stack
//  stack position = Pundo

void load_undo()
{
   char     *pp, buff[24], fotoxx[8];
   int      fid, ww, hh, cc, cc2;

   pp = strstr(undo_files,"_undo_");
   if (! pp) zappcrash("undo/redo stack corrupted 1");
   snprintf(pp+6,3,"%02d",Pundo);
   
   fid = open(undo_files,O_RDONLY);
   if (! fid) zappcrash("undo/redo stack corrupted 2");
   
   *fotoxx = 0;
   cc = read(fid,buff,20);
   sscanf(buff," %d %d %8s ",&ww, &hh, fotoxx);
   if (! strEqu(fotoxx,"fotoxx")) zappcrash("undo/redo stack corrupted 4");

   mutex_lock(&pixmaps_lock);                                              //  v.6.3

   RGB_free(Frgb16);
   Frgb16 = RGB_make(ww,hh,16);
   cc = ww * hh * 6;
   cc2 = read(fid,Frgb16->bmp,cc);
   if (cc2 != cc) zappcrash("undo/redo stack corrupted 5");
   close(fid);
   
   RGB_free(Frgb8);
   Frgb8 = RGB_convbpc(Frgb16);
   Fww = ww;
   Fhh = hh;
   
   edit_function = pvlist_get(editlog,Pundo);                              //  last edit func not un-done   v.10.2

   mutex_unlock(&pixmaps_lock);
   mwpaint2();
   return;
}


/**************************************************************************
      other support functions
***************************************************************************/


//  help menu function

void m_help(GtkWidget *, cchar *menu)
{
   if (strEqu(menu,ZTX("About"))) 
      zmessageACK(GTK_WINDOW(mWin)," %s \n %s \n %s \n %s \n\n %s \n\n %s",
                   fversion,flicense,fhomepage,fcredits,ftranslators,fcontact);
      
   if (strEqu(menu,ZTX("User Guide"))) 
      showz_userguide();

   if (strEqu(menu,"README"))
      showz_readme();

   if (strEqu(menu,ZTX("Change Log")))
      showz_changelog();

   if (strEqu(menu,ZTX("Translate")))
      showz_translations();
      
   if (strEqu(menu,ZTX("Home Page")))
      showz_html(fhomepage);

   return;
}


/**************************************************************************/

//  restore state data from prior session

int load_fotoxx_state()
{
   int            ww, hh, ii, np;
   FILE           *fid;
   char           buff[maxfcc],text[100], *pp;
   float          parms[2];
   
   lens4_name[0] = strdupz("lens_1",lens_cc);                              //  set some defaults
   lens4_name[1] = strdupz("lens_2",lens_cc);
   lens4_name[2] = strdupz("lens_3",lens_cc);
   lens4_name[3] = strdupz("lens_4",lens_cc);
   lens4_mm[0] = 30;
   lens4_mm[1] = 40;
   lens4_mm[2] = 50;
   lens4_mm[3] = 60;
   lens4_bow[0] = 0;
   lens4_bow[1] = 0;
   lens4_bow[2] = 0;
   lens4_bow[3] = 0;
   curr_lens = 1;
   
   snprintf(buff,maxfcc,"%s/saved_state",get_zuserdir());                  //  open saved state file
   fid = fopen(buff,"r");
   if (! fid) return 0;
   
   pp = fgets_trim(buff,maxfcc,fid,1);                                     //  read last image file

   if (pp && *pp == '/') {
      if (image_file) zfree(image_file);
      image_file = strdupz(pp,0,"imagefile");
   }

   pp = fgets_trim(buff,maxfcc,fid,1);                                     //  top directory
   if (pp && *pp == '/') topdirk = strdupz(pp,0,"topdirk");

   pp = fgets(buff,maxfcc,fid);                                            //  main window size
   if (pp) {
      ww = hh = 0;
      sscanf(buff," %d %d ",&ww,&hh);
      if (ww > 200 && ww < 3000) Dww = ww;
      if (hh > 200 && hh < 2000) Dhh = hh;
   }

   pp = fgets(buff,maxfcc,fid);                                            //  image gallery window size
   if (pp) {
      ww = hh = 0;
      sscanf(buff," %d %d ",&ww,&hh);
      if (ww > 200 && ww < 3000) image_navi::xwinW = ww;
      if (hh > 200 && hh < 2000) image_navi::xwinH = hh;
   }

   pp = fgets_trim(buff,maxfcc,fid,1);                                     //  thumbnail image size
   if (pp) {
      sscanf(buff," %d ",&ww);
      if (ww > 32 && ww < 256) image_navi::thumbsize = ww;
   }

   for (ii = 0; ii < 4; ii++)                                              //  4 sets of lens parameters
   {   
      pp = fgets_trim(buff,maxfcc,fid,1);
      if (! pp) break;
      np = sscanf(buff," %s %f %f ",text,&parms[0],&parms[1]);
      if (np != 3) break;
      strncpy0(lens4_name[ii],text,lens_cc);
      lens4_mm[ii] = parms[0];
      lens4_bow[ii] = parms[1];
   }
   
   pp = fgets_trim(buff,maxfcc,fid,1);                                     //  current lens
   if (pp) {
      sscanf(buff," %f ",&parms[0]);
      if (parms[0] >= 0 && parms[0] < 4) curr_lens = int(parms[0]);
   }

   pp = fgets_trim(buff,maxfcc,fid,1);                                     //  fotoxx prior version
   if (pp && strNeq(pp,fversion))                                          //  fix segfault     v.10.1.1
      printf("prior fotoxx version: %s \n",pp);

   fclose(fid);

   for (ii = 0; ii < Nrecentfiles; ii++)                                   //  recent image file list = empty
      recentfiles[ii] = 0;

   snprintf(buff,maxfcc,"%s/recent_files",get_zuserdir());                 //  open recent files file  v.8.2
   fid = fopen(buff,"r");
   if (! fid) return 0;
   
   for (ii = 0; ii < Nrecentfiles; ii++)                                   //  read list of recent files
   {
      pp = fgets_trim(buff,maxfcc,fid,1);
      if (! pp) break;
      if (*pp == '/') recentfiles[ii] = strdupz(buff,0,"recentfile");
   }

   fclose(fid);

   return 1;
}


/**************************************************************************/

//  save state data for next session

int save_fotoxx_state()
{
   FILE           *fid;
   char           buff[200];
   int            ww, hh, ii;

   snprintf(buff,199,"%s/saved_state",get_zuserdir());                     //  open output file
   fid = fopen(buff,"w");
   if (! fid) return 0;
   
   if (image_file && *image_file == '/')                                   //  current image file 
      fputs(image_file,fid);
   fputs("\n",fid);

   if (topdirk && *topdirk == '/')                                         //  top image directory 
      fputs(topdirk,fid);
   fputs("\n",fid);

   gtk_window_get_size(GTK_WINDOW(mWin),&ww,&hh);
   snprintf(buff,20," %d %d \n",ww,hh);                                    //  window size
   fputs(buff,fid);

   snprintf(buff,20," %d %d \n",image_navi::xwinW,image_navi::xwinH);      //  image gallery window size
   fputs(buff,fid);

   snprintf(buff,20," %d \n",image_navi::thumbsize);                       //  thumbnail size
   fputs(buff,fid);
   
   for (ii = 0; ii < 4; ii++)                                              //  4 sets of lens parameters
   {
      snprintf(buff,100," %s %.1f %.2f \n",lens4_name[ii],lens4_mm[ii],lens4_bow[ii]);
      fputs(buff,fid);
   }
   
   snprintf(buff,100," %d \n",curr_lens);                                  //  current lens
   fputs(buff,fid);

   fputs(fversion,fid);                                                    //  fotoxx version
   fputs("\n",fid);
   fputs("\n",fid);
   
   fclose(fid);
   
   snprintf(buff,199,"%s/recent_files",get_zuserdir());                    //  open output file
   fid = fopen(buff,"w");
   if (! fid) return 0;
   
   for (ii = 0; ii < Nrecentfiles; ii++)                                   //  save list of recent files  v.8.2
      if (recentfiles[ii])
         fprintf(fid,"%s \n",recentfiles[ii]);
   
   fclose(fid);

   return 1;
}


/**************************************************************************/

//  ask user if modified image should be kept or discarded

int mod_keep()
{
   if (Fmodified == 0 && Pundo == 0) return 0;                             //  no mods
   if (Fsaved == Pundo) return 0;                                          //  last mods were saved  v.8.3
   if (zmessageYN(GTK_WINDOW(mWin),ZTX("Discard modifications?")))         //  OK to discard
      return 0;
   return 1;
}


/**************************************************************************/

//  menu lock/unlock - some functions must not run concurrently
//  returns 1 if success, else 0

int menulock(int lock)
{
   static int     mlock = 0;
   
   if (! lock && ! mlock) zappcrash("menu lock error");
   if (lock && mlock) return 0;                                            //  no message       v.10.1.1
   if (lock) mlock++;
   else mlock--;
   return 1;
}


/**************************************************************************/

//  Upright a turned image - not like an edit.
//  Rotate Frgb8 without setting the Fmodified flag.

void  turn_image(int angle)
{
   while (angle >= 360) angle -= 360;                                      //  amount to turn now
   while (angle <= -360) angle += 360;
   Fimageturned += angle;                                                  //  total turn  v.8.3
   while (Fimageturned >= 360) Fimageturned -= 360;
   while (Fimageturned <= -360) Fimageturned += 360;
   if (angle == 0) return;
   
   mutex_lock(&pixmaps_lock);                                              //  lock pixmaps  v.8.5
   RGB * temp_bmp = RGB_rotate(Frgb8,angle);
   RGB_free(Frgb8);
   Frgb8 = temp_bmp;
   Fww = Frgb8->ww;
   Fhh = Frgb8->hh;
   Fzoom = 0;
   mutex_unlock(&pixmaps_lock);

   mwpaint();                                                              //  synch Dww etc. immed.    v.6.8
   return;
}


/**************************************************************************/

//  Compute the mean brightness of all pixel neighborhoods,                //  new v.9.6
//  using a Guassian or a flat distribution for the weightings. 
//  If a select area is active, only inside pixels are calculated.
//  The flat method is 10-100x faster.

int         brhood_radius;
double      brhood_kernel[200][200];                                       //  up to radius = 99
float       *brhood_brightness = 0;                                        //  neighborhood brightness per pixel
char        brhood_method;                                                 //  g = gaussian, f = flat distribution


void brhood_calc(int radius, char method)
{
   void * brhood_wthread(void *arg);

   int      rad, rad2, dx, dy, cc, ii;
   double   kern;
   
   brhood_radius = radius;
   brhood_method = method;

   if (brhood_method == 'g')
   {                                                                       //  compute Gaussian kernel
      rad = brhood_radius;
      rad2 = rad * rad;

      for (dy = -rad; dy <= rad; dy++)
      for (dx = -rad; dx <= rad; dx++)
      {
         if (dx*dx + dy*dy <= rad2)                                        //  cells within radius
            kern = exp( - (dx*dx + dy*dy) / rad2);
         else kern = 0;                                                    //  outside radius
         brhood_kernel[dy+rad][dx+rad] = kern;
      }
   }
   
   if (brhood_brightness) zfree(brhood_brightness);                        //  allocate memory for pixel map
   cc = E1ww * E1hh * sizeof(float);
   brhood_brightness = (float *) zmalloc(cc,"brhood");
   memset(brhood_brightness,0,cc);

   if (sa_active) progress_goal = sa_Npixel;                               //  set up progress tracking
   else progress_goal = E1ww * E1hh;
   progress_done = 0;

   for (ii = 0; ii < Nwt; ii++)                                            //  start worker threads
      start_wt(brhood_wthread,&wtnx[ii]);
   wait_wts();                                                             //  wait for completion

   progress_goal = progress_done = 0;
   return;
}
 
 
//  worker thread function

void * brhood_wthread(void *arg)
{
   int      index = *((int *) arg);
   int      rad = brhood_radius;
   int      ii, px, py, qx, qy, Fstart;
   double   kern, bsum, bsamp, bmean;
   uint16   *pixel;

   if (brhood_method == 'g')                                               //  use round gaussian distribution
   {
      for (py = index; py < E1hh; py += Nwt)
      for (px = 0; px < E1ww; px++)
      {
         if (sa_active) {                                                  //  select area active,
            ii = py * E1ww + px;                                           //    use only inside pixels
            if (! sa_pixisin[ii]) continue;
         }
         
         bsum = bsamp = 0;

         for (qy = py-rad; qy <= py+rad; qy++)                             //  computed weighted sum of brightness
         for (qx = px-rad; qx <= px+rad; qx++)                             //    for pixels in neighborhood
         {
            if (qy < 0 || qy > E1hh-1) continue;
            if (qx < 0 || qx > E1ww-1) continue;
            kern = brhood_kernel[qy+rad-py][qx+rad-px];
            pixel = bmpixel(E1rgb16,qx,qy);
            bsum += brightness(pixel) * kern;                              //  sum brightness * weight
            bsamp += kern;                                                 //  sum weights
         }

         bmean = bsum / bsamp;                                             //  mean brightness
         ii = py * E1ww + px;
         brhood_brightness[ii] = bmean;                                    //  pixel value

         progress_done++;
         if (drandz() < 0.0001) zsleep(0.001);                             //  trigger sorry kernel scheduler
      }
   }

   if (brhood_method == 'f')                                               //  use square flat distribution
   {
      Fstart = 1;
      bsum = bsamp = 0;

      for (py = index; py < E1hh; py += Nwt)
      for (px = 0; px < E1ww; px++)
      {
         if (sa_active) {                                                  //  select area active
            ii = py * E1ww + px;
            if (! sa_pixisin[ii]) {
               Fstart = 1;
               continue;
            }
         }
         
         if (px == 0) Fstart = 1;
         
         if (Fstart) 
         {
            Fstart = 0;
            bsum = bsamp = 0;

            for (qy = py-rad; qy <= py+rad; qy++)                          //  add up all columns
            for (qx = px-rad; qx <= px+rad; qx++)
            {
               if (qy < 0 || qy > E1hh-1) continue;
               if (qx < 0 || qx > E1ww-1) continue;
               pixel = bmpixel(E1rgb16,qx,qy);
               bsum += brightness(pixel);
               bsamp += 1;
            }
         }
         else
         {
            qx = px-rad-1;                                                 //  subtract first-1 column
            if (qx >= 0) {
               for (qy = py-rad; qy <= py+rad; qy++)
               {
                  if (qy < 0 || qy > E1hh-1) continue;
                  pixel = bmpixel(E1rgb16,qx,qy);
                  bsum -= brightness(pixel);
                  bsamp -= 1;
               }
            }
            qx = px+rad;                                                   //  add last column
            if (qx < E1ww) {
               for (qy = py-rad; qy <= py+rad; qy++)
               {
                  if (qy < 0 || qy > E1hh-1) continue;
                  pixel = bmpixel(E1rgb16,qx,qy);
                  bsum += brightness(pixel);
                  bsamp += 1;
               }
            }
         }

         bmean = bsum / bsamp;                                             //  mean brightness
         ii = py * E1ww + px;
         brhood_brightness[ii] = bmean;

         progress_done++;
         if (drandz() < 0.0001) zsleep(0.001);                             //  trigger sorry kernel scheduler
      }
   }
         
   exit_wt();
   return 0;                                                               //  not executed, avoid gcc warning
}


//  get the neighborhood brightness for given pixel

inline float get_brhood(int px, int py)
{
   int ii = py * E1ww + px;
   return brhood_brightness[ii];
}


/**************************************************************************/

//  free all resources associated with the current image file

void free_resources()
{
   char        *pp;
   int         ignore;

   mutex_lock(&pixmaps_lock);                                              //  lock pixmaps
   
   strcpy(command,"rm -f ");                                               //  delete all undo files
   strcat(command,undo_files);
   pp = strstr(command,"_undo_");                                          //  clone edit, bugfix  v.6.5
   strcpy(pp + 6,"*");
   ignore = system(command);

   Fmodified = Pundo = Pumax = Fsaved = 0;                                 //  reset undo/redo stack
   Ntoplines = Nptoplines;                                                 //  no image overlay lines
   paint_toparc(2);                                                        //  no brush outline
   
   if (Fshutdown) {                                                        //  stop here if shutdown mode
      mutex_unlock(&pixmaps_lock);
      return;
   }
   
   if (image_file) {
      select_delete();                                                     //  delete select area
      zdialog_free(zdsela);                                                //  kill dialogs if active   v.8.7
      zdialog_free(zdRGB);
      zdsela = zdRGB = null;
      Mcapture = 0;                                                        //  kill mouse function if active  v.8.7
      mouseCBfunc = 0;
      set_cursor(0);                                                       //  restore normal cursor
      update_filetags(image_file);                                         //  commit tag changes, if any
      zfree(image_file);                                                   //  free image file
      image_file = 0;
      zfuncs::F1_help_topic = 0;                                           //  v.9.0
      progress_goal = progress_done = 0;                                   //  v.9.2
      Nalign = 0;
   }
   
   if (redpixels) zfree(redpixels);                                        //  clear align pixel flags   v.9.0
   redpixels = 0;
   
   if (brhood_brightness) zfree(brhood_brightness);                        //  free brightness map   v.9.6
   brhood_brightness = 0;

   RGB_free(Frgb8);
   RGB_free(Drgb8);
   Frgb8 = Drgb8 = 0;
   RGB_free(Frgb16);
   RGB_free(E1rgb16);
   RGB_free(E3rgb16);
   RGB_free(ERrgb16);                                                      //  v.10.3
   Frgb16 = E1rgb16 = E3rgb16 = ERrgb16 = 0;

   Fww = E1ww = E3ww = Dww = 0;                                            //  make unusable (crash)

   mutex_unlock(&pixmaps_lock);
   return;
}


//  set cursor for main/drawing window and track installed cursor

void set_cursor(GdkCursor *cursor)                                         //  v.10.1
{
   gdk_window_set_cursor(drWin->window,cursor);
   currentcursor = cursor;
   zmainloop();
   return;
}


/**************************************************************************
   file read and write utilities
   RGB pixmap <--> file on disk
***************************************************************************/

//  Load an image file into an RGB pixmap of 8 or 16 bits/color
//  Also sets the following global variables:
//    file_MB = disk file megabytes
//    file_bpc = disk file bits per color (1, 8, 16)
//    file_type = "jpeg" "png" "tiff" or "other"

RGB * f_load(cchar *filespec, int bpc)                                     //  use pixbuf or tiff library   v.9.8
{
   int            err;
   cchar          *pext;
   RGB            *rgb, *rgb2;
   struct stat    fstat;

   if (bpc != 8 && bpc != 16)                                              //  requested bpc must be 8 or 16
      zappcrash("image_load bpc: %d",bpc);

   err = stat(filespec,&fstat);
   if (err) return 0;                                                      //  file not found
   if (! S_ISREG(fstat.st_mode)) return 0;                                 //  not a regular file
   file_MB = 1.0 * fstat.st_size / mega;                                   //  disk file megabytes
   
   pext = strrchr(filespec,'/');
   if (! pext) pext = filespec;

   if (pext > filespec+12 && strnEqu(pext-12,"/.thumbnails/",13)) {        //  refuse thumbnail files    v.10.0
      zmessageACK(GTK_WINDOW(mWin),ZTX("cannot open thumbnail file"));
      return 0;
   }

   pext = strrchr(pext,'.');
   if (! pext) pext = "";
   
   if (strstr(".jpg .jpeg .JPG .JPEG",pext)) strcpy(file_type,"jpeg");
   else if (strstr(".tiff .TIFF",pext)) strcpy(file_type,"tiff");
   else if (strstr(".png .PNG",pext)) strcpy(file_type,"png");
   else strcpy(file_type,"other");
   
   if (strEqu(file_type,"tiff"))                                           //  use tiff lib to read tiff file
      rgb = TIFFread(filespec);
   else  rgb = PXBread(filespec);                                          //  use pixbuf lib for others
   if (! rgb) return 0;

   if (rgb->bpc == bpc) return rgb;                                        //  bpc same as requested, done

   rgb2 = RGB_convbpc(rgb);                                                //  convert to requested bpc
   RGB_free(rgb);                                                          //    8 <--> 16
   return rgb2;
}


/**************************************************************************/

//  save current image to specified disk file.
//  update tags index and gallery list.
//  return 0 if OK, else +N

int f_save(cchar *outfile, cchar *format)                                  //  use pixbuf or tiff library   v.9.8
{
   RGB            *rgb16;
   cchar          *exifkey[4] = { exif_width_key, exif_height_key, 
                                  exif_orientation_key, exif_log_key };
   cchar          *exifdata[4];
   char           **ppv, funcslist[1000];
   char           wwchar[8], hhchar[8];
   char           *pp, *tempfile;
   int            nkeys, err = 1, cc1, cc2;
   struct stat    fstat;

   set_cursor(busycursor);                                                 //  set function busy cursor  v.8.4

   update_filetags(image_file);                                            //  commit poss. tag changes   v.8.3

   tempfile = strdupz(get_zuserdir(),24,"tempfile");                       //  use temp output file
   strcat(tempfile,"/temp_");                                              //    ~/.fotoxx/temp_ppppp.ext
   strcat(tempfile,PIDstring);
   strcat(tempfile,".");
   strncat(tempfile,format,4);

   if (strEqu(format,"jpeg")) {                                            //  save as JPEG file
      if (Frgb16) err = PXBwrite(Frgb16,tempfile);
      else  err = PXBwrite(Frgb8,tempfile);
   }

   if (strEqu(format,"png")) {                                             //  save as PNG file
      if (Frgb16) err = PXBwrite(Frgb16,tempfile);
      else  err = PXBwrite(Frgb8,tempfile);
   }

   if (strEqu(format,"tiff-8")) {                                          //  save as tiff-8 file
      if (Frgb16) err = TIFFwrite(Frgb16,tempfile);
      else err = TIFFwrite(Frgb8,tempfile);
   }
   
   if (strEqu(format,"tiff-16")) {                                         //  save as tiff-16 file
      if (Frgb16) err = TIFFwrite(Frgb16,tempfile);                        //  edit file exists, use it
      else {
         if (file_bpc == 16) rgb16 = TIFFread(image_file);                 //  use original 16 bpc file
         else  rgb16 = RGB_convbpc(Frgb8);                                 //  or convert 8 to 16 bpc
         if (! rgb16) err = 1;
         else {
            err = TIFFwrite(rgb16,tempfile);
            RGB_free(rgb16);
         }
      }
   }

   if (err) {
      snprintf(command,ccc,"rm \"%s\"",tempfile);                          //  failure, clean up
      err = system(command);
      zfree(tempfile);
      set_cursor(0);
      return 1;
   }

   snprintf(wwchar,6,"%d",Fww);
   snprintf(hhchar,6,"%d",Fhh);                                            //  copy EXIF data from source file
   exifdata[0] = wwchar;                                                   //    with revisions
   exifdata[1] = hhchar;
   exifdata[2] = "";                                                       //  assume saved upright     v.6.2
   nkeys = 3;

   if (Pundo > 0)                                                          //  edits were made to image, update
   {                                                                       //    list of edits done    v.10.2
      *funcslist = 0;
      ppv = exif_get(image_file,&exifkey[3],1);                            //  get existing list of edits
      if (ppv[0]) {
         strncpy0(funcslist,ppv[0],998);
         zfree(ppv[0]);
      }
      cc1 = strlen(funcslist);
      if (cc1 && funcslist[cc1-1] != ' ') {
         strcpy(funcslist+cc1," ");
         cc1++;
      }

      for (int ii = 1; ii <= Pundo; ii++)                                  //  append new list of edits
      {                                                                    //  (omit index 0 = initial)
         pp = pvlist_get(editlog,ii);
         cc2 = strlen(pp);
         if (cc1 + cc2 > 998) break;
         strcpy(funcslist+cc1,pp);
         strcpy(funcslist+cc1+cc2," ");
         cc1 += cc2 + 1;
      }
      
      exifdata[3] = funcslist;                                             //  4th EXIF key to revise
      nkeys = 4;
   }

   err = exif_copy(image_file,tempfile,exifkey,exifdata,nkeys);            //  copy tags to temp file
   if (err) zmessageACK(GTK_WINDOW(mWin),ZTX("Unable to copy EXIF data"));
   
   snprintf(command,ccc,"cp -f \"%s\" \"%s\" ",tempfile,outfile);          //  copy to final destination
   err = system(command);
   if (err) zmessageACK(GTK_WINDOW(mWin),ZTX("Unable to save image: %s"),wstrerror(err));

   snprintf(command,ccc,"rm \"%s\"",tempfile);                             //  delete temp file
   err = system(command);
   zfree(tempfile);
   
   load_filetags(outfile);                                                 //  update tags index
   update_asstags(outfile);                                                //    for new file
   
   Fmodified = Fimageturned = 0;                                           //  reset file modified status
   Fsaved = Pundo;                                                         //  note which mods are saved    v.8.3

   if (! Fsearchlist && samedirk(image_file,outfile)) {                    //  update image gallery file list
      image_gallery(image_file,"init");                                    //    if output in same directory
      image_gallery(0,"paint2");                                           //  refresh gallery window if active
      gtk_window_present(GTK_WINDOW(mWin));                                //  bring main to foreground
   }

   stat(outfile,&fstat);                                                   //  update file size          v.8.3
   file_MB = 1.0 * fstat.st_size / mega;
   SBupdate++;                                                             //  update status bar

   add_recent_file(outfile);                                               //  first in recent files list   v.8.4

   set_cursor(0);                                                          //  restore normal cursor
   return 0;
}


/**************************************************************************/

//  Read from TIFF file using TIFF library. 
//  Use native TIFF file bits/pixel.

RGB * TIFFread(cchar *filespec)                                            //  new v.9.8
{
   TIFF        *tiff;
   RGB         *rgb;
   char        *tiffbuff;
   int         supported, tiffstat = 0;
   uint8       *tiff8, *rgb8;
   uint16      *tiff16, *rgb16;
   uint16      bpc, nch; 
   uint        ww, hh, row, col, rowcc;
   uint        byte, bit;
   uint8       bitmask[8] = { 128, 64, 32, 16, 8, 4, 2, 1 };
   
   tiff = TIFFOpen(filespec,"r");
   if (! tiff) {
      zmessageACK(GTK_WINDOW(mWin),ZTX("TIFF open failure"));
      return 0;
   }

   TIFFGetField(tiff, TIFFTAG_IMAGEWIDTH, &ww);                            //  width
   TIFFGetField(tiff, TIFFTAG_IMAGELENGTH, &hh);                           //  height
   TIFFGetField(tiff, TIFFTAG_BITSPERSAMPLE, &bpc);                        //  bits per color, 1/8/16
   TIFFGetField(tiff, TIFFTAG_SAMPLESPERPIXEL, &nch);                      //  no. channels (colors), 1/3
   
   supported = 0;
   if (nch == 1 && bpc == 1) supported = 1;
   if (nch == 1 && bpc == 8) supported = 1;
   if (nch == 1 && bpc == 16) supported = 1;
   if (nch == 3 && bpc == 8) supported = 1;
   if (nch == 3 && bpc == 16) supported = 1;
   
   if (! supported) {
      zmessageACK(GTK_WINDOW(mWin),ZTX("TIFF colors=%d depth=%d not supported"),nch,bpc);
      TIFFClose(tiff);
      return 0;
   }

   rowcc = TIFFScanlineSize(tiff);                                         //  row size
   tiffbuff = (char*) zmalloc(rowcc,"tiffbuff");

   if (bpc == 16) rgb = RGB_make(ww,hh,16);
   else rgb = RGB_make(ww,hh,8);

   for (row = 0; row < hh; row++)
   {
      tiffstat = TIFFReadScanline(tiff,tiffbuff,row,0);
      if (tiffstat != 1) break;
      
      if (nch == 1 && bpc == 1)                                            //  bi-level
      {
         tiff8 = (uint8 *) tiffbuff;
         rgb8 = (uint8 *) rgb->bmp + row * ww * 3;

         for (col = 0; col < ww; col++)
         {
            byte = col / 8;
            bit = col - 8 * byte;
            bit = tiff8[byte] & bitmask[bit]; 
            if (bit) rgb8[3*col] = rgb8[3*col+1] = rgb8[3*col+2] = 0;
            else     rgb8[3*col] = rgb8[3*col+1] = rgb8[3*col+2] = 255;
         }
      }
      
      if (nch == 1 && bpc == 8)                                            //  8 bit grayscale
      {
         tiff8 = (uint8 *) tiffbuff;
         rgb8 = (uint8 *) rgb->bmp + row * ww * 3;

         for (col = 0; col < ww; col++)
         {
            rgb8[0] = rgb8[1] = rgb8[2] = tiff8[0];
            rgb8 += 3;
            tiff8++;
         }
      }
      
      if (nch == 1 && bpc == 16)                                           //  16 bit grayscale
      {
         tiff16 = (uint16 *) tiffbuff;
         rgb16 = (uint16 *) rgb->bmp + row * ww * 3;

         for (col = 0; col < ww; col++)
         {
            rgb16[0] = rgb16[1] = rgb16[2] = tiff16[0];
            rgb16 += 3;
            tiff16++;
         }
      }
      
      if (nch == 3 && bpc == 8)                                            //  8 bit RGB
      {
         tiff8 = (uint8 *) tiffbuff;
         rgb8 = (uint8 *) rgb->bmp + row * ww * 3;

         for (col = 0; col < ww; col++)
         {
            rgb8[0] = tiff8[0];
            rgb8[1] = tiff8[1];
            rgb8[2] = tiff8[2];
            rgb8 += 3;
            tiff8 += nch;
         }
      }

      if (nch == 3 && bpc == 16)                                           //  16 bit RGB
      {
         tiff16 = (uint16 *) tiffbuff;
         rgb16 = (uint16 *) rgb->bmp + row * ww * 3;

         for (col = 0; col < ww; col++)
         {
            rgb16[0] = tiff16[0];
            rgb16[1] = tiff16[1];
            rgb16[2] = tiff16[2];
            rgb16 += 3;
            tiff16 += nch;
         }
      }
   }

   TIFFClose(tiff);
   zfree(tiffbuff);
   
   if (tiffstat == 1) {
      file_bpc = bpc;                                                      //  set file bits per color
      return rgb;
   }

   zmessageACK(GTK_WINDOW(mWin),ZTX("TIFF read failure"));
   RGB_free(rgb);
   return 0;
}


//  Write to TIFF file using TIFF library. File bpc taken from RGB.
//  returns 0 if OK, +N if error.

int TIFFwrite(RGB *rgb, cchar *filespec)                                   //  new v.9.8
{
   TIFF        *tiff;
   uint8       *tiff8, *rgb8;
   uint16      *tiff16, *rgb16;
   int         tiffstat = 0;
   uint        ww, hh, row, col, rowcc;
   uint16      bpc, nch, pm = 2, pc = 1, comp = 5; 
   char        *tiffbuff;
   
   tiff = TIFFOpen(filespec,"w");
   if (! tiff) {
      zmessageACK(GTK_WINDOW(mWin),ZTX("TIFF open failure"));
      return 1;
   }
   
   ww = rgb->ww;
   hh = rgb->hh;
   bpc = rgb->bpc;
   nch = 3;                                                                //  alpha channel removed

   TIFFSetField(tiff, TIFFTAG_IMAGEWIDTH, ww);
   TIFFSetField(tiff, TIFFTAG_BITSPERSAMPLE, bpc);
   TIFFSetField(tiff, TIFFTAG_SAMPLESPERPIXEL, nch);
   TIFFSetField(tiff, TIFFTAG_PHOTOMETRIC, pm);                            //  RGB
   TIFFSetField(tiff, TIFFTAG_PLANARCONFIG, pc);
   TIFFSetField(tiff, TIFFTAG_COMPRESSION, comp);                          //  LZW
   
   rowcc = TIFFScanlineSize(tiff);
   tiffbuff = (char*) zmalloc(rowcc,"tiffbuff");

   for (row = 0; row < hh; row++)
   {
      if (bpc == 8)
      {
         tiff8 = (uint8 *) tiffbuff;
         rgb8 = (uint8 *) rgb->bmp + row * ww * 3;

         for (col = 0; col < ww; col++)
         {
            tiff8[0] = rgb8[0];
            tiff8[1] = rgb8[1];
            tiff8[2] = rgb8[2];
            rgb8 += 3;
            tiff8 += nch;
         }
      }

      if (bpc == 16) 
      {
         tiff16 = (uint16 *) tiffbuff;
         rgb16 = (uint16 *) rgb->bmp + row * ww * 3;

         for (col = 0; col < ww; col++)
         {
            tiff16[0] = rgb16[0];
            tiff16[1] = rgb16[1];
            tiff16[2] = rgb16[2];
            rgb16 += 3;
            tiff16 += nch;
         }
      }

      tiffstat = TIFFWriteScanline(tiff,tiffbuff,row,0);
      if (tiffstat != 1) break;
   }

   TIFFClose(tiff);
   zfree(tiffbuff);
   
   if (tiffstat == 1) return 0;
   zmessageACK(GTK_WINDOW(mWin),ZTX("TIFF write failure"));
   return 2;
}


//  intercept TIFF warning messages (many)                                 //  new v.9.8

void tiffwarninghandler(cchar *module, cchar *format, va_list ap)
{
   return;                                                                 //  turned off
   char  message[200];
   vsnprintf(message,199,format,ap);
   printf("TIFF warning: %s %s \n",module,message);
   return;
}


//  Read from jpeg/png file using pixbuf library. bpc = 8.

RGB * PXBread(cchar *filespec)                                             //  new v.9.8
{
   GError      *gerror = 0;
   PXB         *pxb;
   RGB         *rgb;
   int         ww, hh, px, py, nch, rowst;
   uint8       *bmp1, *bmp2, *pix1, *pix2;
   
   pxb = gdk_pixbuf_new_from_file(filespec,&gerror);
   if (! pxb) {
      zmessageACK(GTK_WINDOW(mWin),ZTX("file type not supported"));
      return 0;
   }

   ww = gdk_pixbuf_get_width(pxb);
   hh = gdk_pixbuf_get_height(pxb);
   nch = gdk_pixbuf_get_n_channels(pxb);
   rowst = gdk_pixbuf_get_rowstride(pxb);
   bmp1 = gdk_pixbuf_get_pixels(pxb);
   
   rgb = RGB_make(ww,hh,8);
   bmp2 = (uint8 *) rgb->bmp;

   for (py = 0; py < hh; py++)
   {
      pix1 = bmp1 + rowst * py;
      pix2 = bmp2 + py * ww * 3;

      for (px = 0; px < ww; px++)
      {
         pix2[0] = pix1[0];
         pix2[1] = pix1[1];
         pix2[2] = pix1[2];
         pix1 += nch;
         pix2 += 3;
      }
   }
   
   file_bpc = 8;                                                           //  set file bits per color
   g_object_unref(pxb);
   return rgb;
}


//  Write to jpeg/png file using pixbuf library. bpc = 8.
//  returns 0 if OK, +N if error.

int PXBwrite(RGB *rgb, cchar *filespec)                                    //  new v.9.8
{
   int         ww, hh, bpc, px, py, rowst;
   uint8       *bmp8, *pix8, *bmp2, *pix2;
   uint16      *bmp16, *pix16;
   PXB         *pxb;
   cchar       *pext;
   GError      *gerror = 0;
   int         pxbstat;

   ww = rgb->ww;
   hh = rgb->hh;
   bpc = rgb->bpc;
   
   pxb = gdk_pixbuf_new(colorspace,0,8,ww,hh);
   if (! pxb) zappcrash("pixbuf allocation failure");

   bmp8 = (uint8 *) rgb->bmp;
   bmp16 = (uint16 *) rgb->bmp;
   bmp2 = gdk_pixbuf_get_pixels(pxb);
   rowst = gdk_pixbuf_get_rowstride(pxb);

   if (bpc == 8)
   {
      for (py = 0; py < hh; py++)
      {
         pix8 = bmp8 + py * ww * 3;
         pix2 = bmp2 + rowst * py;

         for (px = 0; px < ww; px++)
         {
            pix2[0] = pix8[0];
            pix2[1] = pix8[1];
            pix2[2] = pix8[2];
            pix2 += 3;
            pix8 += 3;
         }
      }
   }
   
   if (bpc == 16)
   {
      for (py = 0; py < hh; py++)
      {
         pix16 = bmp16 + py * ww * 3;
         pix2 = bmp2 + rowst * py;

         for (px = 0; px < ww; px++)
         {
            pix2[0] = pix16[0] >> 8;
            pix2[1] = pix16[1] >> 8;
            pix2[2] = pix16[2] >> 8;
            pix2 += 3;
            pix16 += 3;
         }
      }
   }
   
   pext = strrchr(filespec,'/');
   if (! pext) pext = filespec;
   pext = strrchr(pext,'.');
   if (! pext) pext = "";
   
   if (strstr(".png .PNG",pext))
      pxbstat = gdk_pixbuf_save(pxb,filespec,"png",&gerror,null);
   else pxbstat = gdk_pixbuf_save(pxb,filespec,"jpeg",&gerror,"quality",jpeg_quality,null);

   g_object_unref(pxb);

   if (pxbstat) return 0;
   zmessageACK(GTK_WINDOW(mWin),ZTX("pixbuf write failure"));
   return 1;
}


/**************************************************************************
      RGB pixmap conversion and rescale functions      revamped v.6.5
***************************************************************************/

//  initialize an RGB pixmap - allocate memory

RGB * RGB_make(int ww, int hh, int bpc)
{
   if (ww < 1 || hh < 1 || (bpc != 8 && bpc != 16))
      zappcrash("RGB_make() %d %d %d",ww,hh,bpc);
   
   RGB *rgb = (RGB *) zmalloc(sizeof(RGB),"rgb");
   rgb->ww = ww;
   rgb->hh = hh;
   rgb->bpc = bpc;
   if (bpc == 8) rgb->bmp = zmalloc(ww*hh*3,"rgb.bmp");
   if (bpc == 16) rgb->bmp = zmalloc(ww*hh*6,"rgb.bmp");
   strcpy(rgb->wmi,"rgbrgb");
   return rgb;
}


//  free RGB pixmap

void RGB_free(RGB *rgb)
{
   if (! rgb) return;                                                      //  v.6.8
   if (! strEqu(rgb->wmi,"rgbrgb")) 
      zappcrash("RGB_free(), bad RGB");
   strcpy(rgb->wmi,"xxxxxx");
   zfree(rgb->bmp);
   zfree(rgb);
   return;
}


//  create a copy of an RGB pixmap

RGB * RGB_copy(RGB *rgb1)
{
   int      cc;
   RGB      *rgb2;

   rgb2 = RGB_make(rgb1->ww, rgb1->hh, rgb1->bpc);
   cc = rgb1->ww * rgb1->hh * (rgb1->bpc / 8 * 3);
   memcpy(rgb2->bmp,rgb1->bmp,cc);
   return rgb2;
}


//  create a copy of an RGB area

RGB * RGB_copy_area(RGB *rgb1, int orgx, int orgy, int ww2, int hh2)
{
   uint8          *bmp1, *pix1, *bmp2, *pix2;
   uint16         *bmp3, *pix3, *bmp4, *pix4;
   RGB            *rgb2 = 0;
   int            ww1, hh1, bpc, px1, py1, px2, py2;

   ww1 = rgb1->ww;
   hh1 = rgb1->hh;
   bpc = rgb1->bpc;

   if (bpc == 8)
   {
      rgb2 = RGB_make(ww2,hh2,8);
      bmp1 = (uint8 *) rgb1->bmp;
      bmp2 = (uint8 *) rgb2->bmp;
     
      for (py1 = orgy, py2 = 0; py2 < hh2; py1++, py2++) 
      {
         for (px1 = orgx, px2 = 0; px2 < ww2; px1++, px2++)
         {
            pix1 = bmp1 + (py1 * ww1 + px1) * 3;
            pix2 = bmp2 + (py2 * ww2 + px2) * 3;

            pix2[0] = pix1[0];
            pix2[1] = pix1[1];
            pix2[2] = pix1[2];
            pix1 += 3;
            pix2 += 3;
         }
      }
   }

   if (bpc == 16)
   {
      rgb2 = RGB_make(ww2,hh2,16);
      bmp3 = (uint16 *) rgb1->bmp;
      bmp4 = (uint16 *) rgb2->bmp;
     
      for (py1 = orgy, py2 = 0; py2 < hh2; py1++, py2++) 
      {
         for (px1 = orgx, px2 = 0; px2 < ww2; px1++, px2++)
         {
            pix3 = bmp3 + (py1 * ww1 + px1) * 3;
            pix4 = bmp4 + (py2 * ww2 + px2) * 3;

            pix4[0] = pix3[0];
            pix4[1] = pix3[1];
            pix4[2] = pix3[2];
            pix3 += 3;
            pix4 += 3;
         }
      }
   }
   
   return rgb2;
}


//   convert RGB pixmap from 8/16 to 16/8 bits per color

RGB * RGB_convbpc(RGB *rgb1)
{
   uint8          *bmp8, *pix8;
   uint16         *bmp16, *pix16;
   RGB            *rgb2 = 0;
   int            ww, hh, bpc, px, py;

   ww = rgb1->ww;
   hh = rgb1->hh;
   bpc = rgb1->bpc;

   if (bpc == 8)                                                           //  8 > 16
   {
      rgb2 = RGB_make(ww,hh,16);
      bmp8 = (uint8 *) rgb1->bmp;
      bmp16 = (uint16 *) rgb2->bmp;
     
      for (py = 0; py < hh; py++) 
      {
         pix8 = bmp8 + py * ww * 3;
         pix16 = bmp16 + py * ww * 3;

         for (px = 0; px < ww; px++)
         {
            pix16[0] = pix8[0] << 8;
            pix16[1] = pix8[1] << 8;
            pix16[2] = pix8[2] << 8;
            pix8 += 3;
            pix16 += 3;
         }
      }
   }

   if (bpc == 16)                                                          //  16 > 8
   {
      rgb2 = RGB_make(ww,hh,8);
      bmp8 = (uint8 *) rgb2->bmp;
      bmp16 = (uint16 *) rgb1->bmp;
     
      for (py = 0; py < hh; py++) 
      {
         pix8 = bmp8 + py * ww * 3;
         pix16 = bmp16 + py * ww * 3;

         for (px = 0; px < ww; px++)
         {
            pix8[0] = pix16[0] >> 8;
            pix8[1] = pix16[1] >> 8;
            pix8[2] = pix16[2] >> 8;
            pix8 += 3;
            pix16 += 3;
         }
      }
   }

   return rgb2;
}


//  rescale RGB pixmap to size ww2 x hh2

RGB * RGB_rescale(RGB *rgb1, int ww2, int hh2)
{
   void RGB_rescale8(uint8*, uint8*, int, int, int, int);
   void RGB_rescale16(uint16*, uint16*, int, int, int, int);

   RGB      *rgb2;
   int      ww1, hh1, bpc;
   uint8    *bmp1, *bmp2;
   uint16   *bmp3, *bmp4;

   ww1 = rgb1->ww;
   hh1 = rgb1->hh;
   bpc = rgb1->bpc;
   
   rgb2 = RGB_make(ww2,hh2,bpc);
   
   if (bpc == 8) {
      bmp1 = (uint8 *) rgb1->bmp;
      bmp2 = (uint8 *) rgb2->bmp;
      RGB_rescale8(bmp1,bmp2,ww1,hh1,ww2,hh2);
   }

   if (bpc == 16) {
      bmp3 = (uint16 *) rgb1->bmp;
      bmp4 = (uint16 *) rgb2->bmp;
      RGB_rescale16(bmp3,bmp4,ww1,hh1,ww2,hh2);
   }
   
   return rgb2;
}


//  rotate RGB pixmap through given angle in degrees (+ = clockwise)

RGB * RGB_rotate(RGB *rgb1, double angle)
{
   RGB * RGB_rotate8(RGB *, double);
   RGB * RGB_rotate16(RGB *, double);

   RGB      *rgb2 = 0;
   int      bpc;
   
   bpc = rgb1->bpc;
   if (bpc == 8) rgb2 = RGB_rotate8(rgb1,angle);
   if (bpc == 16) rgb2 = RGB_rotate16(rgb1,angle);
   return rgb2;
}


/**************************************************************************

   Rescale 8 bpc image (3 x 8 bits per color) to new width and height.
   The scale ratios may be different for width and height.

   Method: 
   The input and output images are overlayed, stretching or shrinking the
   output pixels as needed. The contribution of each input pixel overlapping
   an output pixel is proportional to the area of the output pixel covered by
   the input pixel. The contributions of all overlaping input pixels are added.
   The work is spread among Nwt threads to reduce the elapsed time on modern 
   computers having multiple SMP processors.

   Example: if the output image is 40% of the input image, then:
     outpix[0,0] = 0.16 * inpix[0,0] + 0.16 * inpix[1,0] + 0.08 * inpix[2,0]
                 + 0.16 * inpix[0,1] + 0.16 * inpix[1,1] + 0.08 * inpix[2,1]
                 + 0.08 * inpix[0,2] + 0.08 * inpix[1,2] + 0.04 * inpix[2,2]

*********/

namespace rgbrescale8 {                                                    //  data for threads
   uint8    *pixmap1;
   uint8    *pixmap2;
   int      ww1;
   int      hh1;
   int      ww2;
   int      hh2;
   int      *px1L;
   int      *py1L;
   float    *pxmap;
   float    *pymap;
   int      maxmapx;
   int      maxmapy;
   int      busy[max_threads];
}


void RGB_rescale8(uint8 *pixmap1x, uint8 *pixmap2x, int ww1x, int hh1x, int ww2x, int hh2x)
{
   using namespace rgbrescale8;

   void * RGB_rescale8_thread(void *arg);

   int         px1, py1, px2, py2;
   int         pxl, pyl, pxm, pym, ii;
   float       scalex, scaley;
   float       px1a, py1a, px1b, py1b;
   float       fx, fy;
   
   pixmap1 = pixmap1x;
   pixmap2 = pixmap2x;
   ww1 = ww1x;
   hh1 = hh1x;
   ww2 = ww2x;
   hh2 = hh2x;

   memset(pixmap2, 0, ww2 * hh2 * 3 * sizeof(uint8));                      //  clear output pixmap

   scalex = 1.0 * ww1 / ww2;                                               //  compute x and y scales
   scaley = 1.0 * hh1 / hh2;
   
   if (scalex <= 1) maxmapx = 2;                                           //  compute max input pixels
   else maxmapx = int(scalex + 2);                                         //    mapping into output pixels
   maxmapx += 1;                                                           //      for both dimensions
   if (scaley <= 1) maxmapy = 2;                                           //  (pixels may not be square)
   else maxmapy = int(scaley + 2);
   maxmapy += 1;
   
   pymap = (float *) zmalloc(hh2 * maxmapy * sizeof(float));               //  maps overlap of < maxmap input
   pxmap = (float *) zmalloc(ww2 * maxmapx * sizeof(float));               //    pixels per output pixel

   py1L = (int *) zmalloc(hh2 * sizeof(int));                              //  maps first (lowest) input pixel
   px1L = (int *) zmalloc(ww2 * sizeof(int));                              //    per output pixel

   for (py2 = 0; py2 < hh2; py2++)                                         //  loop output y-pixels
   {
      py1a = py2 * scaley;                                                 //  corresponding input y-pixels
      py1b = py1a + scaley;
      if (py1b >= hh1) py1b = hh1 - 0.001;                                 //  fix precision limitation
      pyl = int(py1a);
      py1L[py2] = pyl;                                                     //  1st overlapping input pixel

      for (py1 = pyl, pym = 0; py1 < py1b; py1++, pym++)                   //  loop overlapping input pixels
      {
         if (py1 < py1a) {                                                 //  compute amount of overlap
            if (py1+1 < py1b) fy = py1+1 - py1a;                           //    0.0 to 1.0 
            else fy = scaley;
         }
         else if (py1+1 > py1b) fy = py1b - py1;
         else fy = 1;

         ii = py2 * maxmapy + pym;                                         //  save it
         pymap[ii] = 0.9999 * fy / scaley;
      }
      ii = py2 * maxmapy + pym;                                            //  set an end marker after
      pymap[ii] = -1;                                                      //    last overlapping pixel
   }
   
   for (px2 = 0; px2 < ww2; px2++)                                         //  do same for x-pixels
   {
      px1a = px2 * scalex;
      px1b = px1a + scalex;
      if (px1b >= ww1) px1b = ww1 - 0.001;
      pxl = int(px1a);
      px1L[px2] = pxl;

      for (px1 = pxl, pxm = 0; px1 < px1b; px1++, pxm++)
      {
         if (px1 < px1a) {
            if (px1+1 < px1b) fx = px1+1 - px1a;
            else fx = scalex;
         }
         else if (px1+1 > px1b) fx = px1b - px1;
         else fx = 1;

         ii = px2 * maxmapx + pxm;
         pxmap[ii] = 0.9999 * fx / scalex;
      }
      ii = px2 * maxmapx + pxm;
      pxmap[ii] = -1;
   }

   for (ii = 0; ii < Nwt; ii++) {                                          //  start working threads
      busy[ii] = 1;
      start_detached_thread(RGB_rescale8_thread,&wtnx[ii]);
   }
   
   for (ii = 0; ii < Nwt; ii++)                                            //  wait for all done
      while (busy[ii]) zsleep(0.004);

   zfree(px1L);
   zfree(py1L);
   zfree(pxmap);
   zfree(pymap);
   return;
}


void * RGB_rescale8_thread(void *arg)                                      //  worker thread function
{
   using namespace rgbrescale8;

   int         index = *((int *) arg);
   int         px1, py1, px2, py2;
   int         pxl, pyl, pxm, pym, ii;
   uint8       *pixel1, *pixel2;
   float       fx, fy, ftot;
   float       red, green, blue;

   for (py2 = index; py2 < hh2; py2 += Nwt)                                //  loop output y-pixels
   {
      pyl = py1L[py2];                                                     //  corresp. 1st input y-pixel

      for (px2 = 0; px2 < ww2; px2++)                                      //  loop output x-pixels
      {
         pxl = px1L[px2];                                                  //  corresp. 1st input x-pixel

         red = green = blue = 0;                                           //  initz. output pixel

         for (py1 = pyl, pym = 0; ; py1++, pym++)                          //  loop overlapping input y-pixels
         {
            ii = py2 * maxmapy + pym;                                      //  get y-overlap
            fy = pymap[ii];
            if (fy < 0) break;                                             //  no more pixels

            for (px1 = pxl, pxm = 0; ; px1++, pxm++)                       //  loop overlapping input x-pixels
            {
               ii = px2 * maxmapx + pxm;                                   //  get x-overlap
               fx = pxmap[ii];
               if (fx < 0) break;                                          //  no more pixels

               ftot = fx * fy;                                             //  area overlap = x * y overlap
               pixel1 = pixmap1 + (py1 * ww1 + px1) * 3;
               red += pixel1[0] * ftot;                                    //  add input pixel * overlap
               green += pixel1[1] * ftot;
               blue += pixel1[2] * ftot;
            }

            pixel2 = pixmap2 + (py2 * ww2 + px2) * 3;                      //  save output pixel
            pixel2[0] = int(red);
            pixel2[1] = int(green);
            pixel2[2] = int(blue);
         }
      }
   }

   busy[index] = 0;
   return 0;
}


/**************************************************************************

   Rescale 16 bpc image (3 x 16 bits per color) to new width and height.
   Identical to RGB_rescale8 except for the following: 
      uint8 >> uint16
      xxx8 >> xxx16

*******/

namespace rgbrescale16 {                                                   //  data for threads
   uint16   *pixmap1;
   uint16   *pixmap2;
   int      ww1;
   int      hh1;
   int      ww2;
   int      hh2;
   int      *px1L;
   int      *py1L;
   float    *pxmap;
   float    *pymap;
   int      maxmapx;
   int      maxmapy;
   int      busy[max_threads];
}


void RGB_rescale16(uint16 *pixmap1x, uint16 *pixmap2x, int ww1x, int hh1x, int ww2x, int hh2x)
{
   using namespace rgbrescale16;

   void * RGB_rescale16_thread(void *arg);

   int         px1, py1, px2, py2;
   int         pxl, pyl, pxm, pym, ii;
   float       scalex, scaley;
   float       px1a, py1a, px1b, py1b;
   float       fx, fy;
   
   pixmap1 = pixmap1x;
   pixmap2 = pixmap2x;
   ww1 = ww1x;
   hh1 = hh1x;
   ww2 = ww2x;
   hh2 = hh2x;

   memset(pixmap2, 0, ww2 * hh2 * 3 * sizeof(uint16));                     //  clear output pixmap

   scalex = 1.0 * ww1 / ww2;                                               //  compute x and y scales
   scaley = 1.0 * hh1 / hh2;
   
   if (scalex <= 1) maxmapx = 2;                                           //  compute max input pixels
   else maxmapx = int(scalex + 2);                                         //    mapping into output pixels
   maxmapx += 1;                                                           //      for both dimensions
   if (scaley <= 1) maxmapy = 2;                                           //  (pixels may not be square)
   else maxmapy = int(scaley + 2);
   maxmapy += 1;
   
   pymap = (float *) zmalloc(hh2 * maxmapy * sizeof(float));               //  maps overlap of < maxmap input
   pxmap = (float *) zmalloc(ww2 * maxmapx * sizeof(float));               //    pixels per output pixel

   py1L = (int *) zmalloc(hh2 * sizeof(int));                              //  maps first (lowest) input pixel
   px1L = (int *) zmalloc(ww2 * sizeof(int));                              //    per output pixel

   for (py2 = 0; py2 < hh2; py2++)                                         //  loop output y-pixels
   {
      py1a = py2 * scaley;                                                 //  corresponding input y-pixels
      py1b = py1a + scaley;
      if (py1b >= hh1) py1b = hh1 - 0.001;                                 //  fix precision limitation
      pyl = int(py1a);
      py1L[py2] = pyl;                                                     //  1st overlapping input pixel

      for (py1 = pyl, pym = 0; py1 < py1b; py1++, pym++)                   //  loop overlapping input pixels
      {
         if (py1 < py1a) {                                                 //  compute amount of overlap
            if (py1+1 < py1b) fy = py1+1 - py1a;                           //    0.0 to 1.0 
            else fy = scaley;
         }
         else if (py1+1 > py1b) fy = py1b - py1;
         else fy = 1;

         ii = py2 * maxmapy + pym;                                         //  save it
         pymap[ii] = 0.9999 * fy / scaley;
      }
      ii = py2 * maxmapy + pym;                                            //  set an end marker after
      pymap[ii] = -1;                                                      //    last overlapping pixel
   }
   
   for (px2 = 0; px2 < ww2; px2++)                                         //  do same for x-pixels
   {
      px1a = px2 * scalex;
      px1b = px1a + scalex;
      if (px1b >= ww1) px1b = ww1 - 0.001;
      pxl = int(px1a);
      px1L[px2] = pxl;

      for (px1 = pxl, pxm = 0; px1 < px1b; px1++, pxm++)
      {
         if (px1 < px1a) {
            if (px1+1 < px1b) fx = px1+1 - px1a;
            else fx = scalex;
         }
         else if (px1+1 > px1b) fx = px1b - px1;
         else fx = 1;

         ii = px2 * maxmapx + pxm;
         pxmap[ii] = 0.9999 * fx / scalex;
      }
      ii = px2 * maxmapx + pxm;
      pxmap[ii] = -1;
   }

   for (ii = 0; ii < Nwt; ii++) {                                          //  start working threads
      busy[ii] = 1;
      start_detached_thread(RGB_rescale16_thread,&wtnx[ii]);
   }
   
   for (ii = 0; ii < Nwt; ii++)                                            //  wait for all done
      while (busy[ii]) zsleep(0.004);

   zfree(px1L);
   zfree(py1L);
   zfree(pxmap);
   zfree(pymap);
   return;
}


void * RGB_rescale16_thread(void *arg)                                     //  worker thread function
{
   using namespace rgbrescale16;

   int         index = *((int *) arg);
   int         px1, py1, px2, py2;
   int         pxl, pyl, pxm, pym, ii;
   uint16      *pixel1, *pixel2;
   float       fx, fy, ftot;
   float       red, green, blue;

   for (py2 = index; py2 < hh2; py2 += Nwt)                                //  loop output y-pixels
   {
      pyl = py1L[py2];                                                     //  corresp. 1st input y-pixel

      for (px2 = 0; px2 < ww2; px2++)                                      //  loop output x-pixels
      {
         pxl = px1L[px2];                                                  //  corresp. 1st input x-pixel

         red = green = blue = 0;                                           //  initz. output pixel

         for (py1 = pyl, pym = 0; ; py1++, pym++)                          //  loop overlapping input y-pixels
         {
            ii = py2 * maxmapy + pym;                                      //  get y-overlap
            fy = pymap[ii];
            if (fy < 0) break;                                             //  no more pixels

            for (px1 = pxl, pxm = 0; ; px1++, pxm++)                       //  loop overlapping input x-pixels
            {
               ii = px2 * maxmapx + pxm;                                   //  get x-overlap
               fx = pxmap[ii];
               if (fx < 0) break;                                          //  no more pixels

               ftot = fx * fy;                                             //  area overlap = x * y overlap
               pixel1 = pixmap1 + (py1 * ww1 + px1) * 3;
               red += pixel1[0] * ftot;                                    //  add input pixel * overlap
               green += pixel1[1] * ftot;
               blue += pixel1[2] * ftot;
            }

            pixel2 = pixmap2 + (py2 * ww2 + px2) * 3;                      //  save output pixel
            pixel2[0] = int(red);
            pixel2[1] = int(green);
            pixel2[2] = int(blue);
         }
      }
   }

   busy[index] = 0;
   return 0;
}


/**************************************************************************

      RGB *rgb2 = RGB_rotate8(RGB *rgb1, double angle)

      Rotate RGB-8 pixmap through an arbitrary angle (degrees).

      The returned image has the same size as the original, but the
      pixmap size is increased to accomodate the rotated image.
      (e.g. a 100x100 image rotated 45 deg. needs a 142x142 pixmap).
      The parameters ww and hh are the dimensions of the input
      pixmap, and are updated to the dimensions of the output pixmap.

      The space added around the rotated image is black (RGB 0,0,0).
      Angle is in degrees. Positive direction is clockwise.
      Speed is about 3 million pixels/sec/thread for a 2.4 GHz CPU.
      Loss of resolution is less than 1 pixel.
      
      Work is divided among Nwt threads to gain speed.
      
      v.9.3: affine transform instead of trig functions, for speed

***************************************************************************/

namespace rotrgb8 {
   int      busy = 0;
   uint8    *pixmap1;
   uint8    *pixmap2;
   int      ww1;
   int      hh1;
   int      ww2;
   int      hh2;
   double   angle;
}


RGB * RGB_rotate8(RGB *rgb1, double anglex)
{
   using namespace rotrgb8;

   void     *RGB_rotate8_thread(void *);

   int      cc, ii;
   RGB      *rgb2;
   
   ww1 = rgb1->ww;
   hh1 = rgb1->hh;
   pixmap1 = (uint8 *) rgb1->bmp;
   angle = anglex;

   while (angle < -180) angle += 360;                                      //  normalize, -180 to +180
   while (angle > 180) angle -= 360;
   angle = angle * pi / 180;                                               //  radians, -pi to +pi
   
   if (fabs(angle) < 0.001) {                                              //  angle = 0 within my precision
      rgb2 = RGB_make(ww1,hh1,8);                                          //  return a copy of the input
      pixmap2 = (uint8 *) rgb2->bmp;
      cc = ww1 * hh1 * 3 * sizeof(uint8);
      memcpy(pixmap2,pixmap1,cc);
      return rgb2;
   }

   ww2 = int(ww1*fabs(cos(angle)) + hh1*fabs(sin(angle)));                 //  rectangle containing rotated image
   hh2 = int(ww1*fabs(sin(angle)) + hh1*fabs(cos(angle)));
   
   rgb2 = RGB_make(ww2,hh2,8);
   pixmap2 = (uint8 *) rgb2->bmp;

   for (ii = 0; ii < Nwt; ii++)                                            //  start worker threads
      start_detached_thread(RGB_rotate8_thread,&wtnx[ii]);
   zadd_locked(busy,+Nwt);

   while (busy) zsleep(0.004);                                             //  wait for completion
   return rgb2;
}


void * RGB_rotate8_thread(void *arg)
{
   using namespace rotrgb8;

   int      index = *((int *) (arg));
   int      px2, py2, px0, py0;
   uint8    *pix0, *pix1, *pix2, *pix3;
   double   px1, py1;
   double   f0, f1, f2, f3, red, green, blue;
   double   a, b, d, e, ww15, hh15, ww25, hh25;

   ww15 = 0.5 * ww1;
   hh15 = 0.5 * hh1;
   ww25 = 0.5 * ww2;
   hh25 = 0.5 * hh2;

   a = cos(angle);
   b = sin(angle);
   d = - sin(angle);
   e = cos(angle);
   
   for (py2 = index; py2 < hh2; py2 += Nwt)                                //  loop through output pixels
   for (px2 = 0; px2 < ww2; px2++)                                         //  outer loop y
   {
      px1 = a * (px2 - ww25) + b * (py2 - hh25) + ww15;                    //  (px1,py1) = corresponding    v.9.3
      py1 = d * (px2 - ww25) + e * (py2 - hh25) + hh15;                    //    point within input pixels

      px0 = int(px1);                                                      //  pixel containing (px1,py1)
      py0 = int(py1);
      
      if (px1 < 0 || px0 >= ww1-1 || py1 < 0 || py0 >= hh1-1) {            //  if outside input pixel array
         pix2 = pixmap2 + (py2 * ww2 + px2) * 3;                           //    output is black
         pix2[0] = pix2[1] = pix2[2] = 0;
         continue;
      }

      pix0 = pixmap1 + (py0 * ww1 + px0) * 3;                              //  4 input pixels based at (px0,py0)
      pix1 = pix0 + ww1 * 3;
      pix2 = pix0 + 3;
      pix3 = pix1 + 3;

      f0 = (px0+1 - px1) * (py0+1 - py1);                                  //  overlap of (px1,py1)
      f1 = (px0+1 - px1) * (py1 - py0);                                    //    in each of the 4 pixels
      f2 = (px1 - px0) * (py0+1 - py1);
      f3 = (px1 - px0) * (py1 - py0);
   
      red =   f0 * pix0[0] + f1 * pix1[0] + f2 * pix2[0] + f3 * pix3[0];   //  sum the weighted inputs
      green = f0 * pix0[1] + f1 * pix1[1] + f2 * pix2[1] + f3 * pix3[1];
      blue =  f0 * pix0[2] + f1 * pix1[2] + f2 * pix2[2] + f3 * pix3[2];
      
      pix2 = pixmap2 + (py2 * ww2 + px2) * 3;                              //  output pixel
      pix2[0] = int(red);
      pix2[1] = int(green);
      pix2[2] = int(blue);
   }
   
   zadd_locked(busy,-1);
   return 0;
}


/**************************************************************************

   RGB *rgb2 = RGB_rotate16(RGB *rgb1, double angle)
   Rotate RGB-16 pixmap through an arbitrary angle (degrees).
   Identical to RGB_rotate8() except for:
      uint8 >> uint16
      rotrgb8 >> rotrgb16
      8 >> 16   

**********/

namespace rotrgb16 {
   int      busy = 0;
   uint16   *pixmap1;
   uint16   *pixmap2;
   int      ww1;
   int      hh1;
   int      ww2;
   int      hh2;
   double   angle;
}


RGB * RGB_rotate16(RGB *rgb1, double anglex)
{
   using namespace rotrgb16;

   void     *RGB_rotate16_thread(void *);

   int      cc, ii;
   RGB      *rgb2;
   
   ww1 = rgb1->ww;
   hh1 = rgb1->hh;
   pixmap1 = (uint16 *) rgb1->bmp;
   angle = anglex;

   while (angle < -180) angle += 360;                                      //  normalize, -180 to +180
   while (angle > 180) angle -= 360;
   angle = angle * pi / 180;                                               //  radians, -pi to +pi
   
   if (fabs(angle) < 0.001) {                                              //  angle = 0 within my precision
      rgb2 = RGB_make(ww1,hh1,16);                                         //  return a copy of the input
      pixmap2 = (uint16 *) rgb2->bmp;
      cc = ww1 * hh1 * 3 * sizeof(uint16);
      memcpy(pixmap2,pixmap1,cc);
      return rgb2;
   }

   ww2 = int(ww1*fabs(cos(angle)) + hh1*fabs(sin(angle)));                 //  rectangle containing rotated image
   hh2 = int(ww1*fabs(sin(angle)) + hh1*fabs(cos(angle)));
   
   rgb2 = RGB_make(ww2,hh2,16);
   pixmap2 = (uint16 *) rgb2->bmp;

   for (ii = 0; ii < Nwt; ii++)                                            //  start worker threads
      start_detached_thread(RGB_rotate16_thread,&wtnx[ii]);
   zadd_locked(busy,+Nwt);

   while (busy) zsleep(0.004);                                             //  wait for completion
   return rgb2;
}


void * RGB_rotate16_thread(void *arg)
{
   using namespace rotrgb16;

   int      index = *((int *) (arg));
   int      px2, py2, px0, py0;
   uint16   *pix0, *pix1, *pix2, *pix3;
   double   px1, py1;
   double   f0, f1, f2, f3, red, green, blue;
   double   a, b, d, e, ww15, hh15, ww25, hh25;

   ww15 = 0.5 * ww1;
   hh15 = 0.5 * hh1;
   ww25 = 0.5 * ww2;
   hh25 = 0.5 * hh2;

   a = cos(angle);
   b = sin(angle);
   d = - sin(angle);
   e = cos(angle);
   
   for (py2 = index; py2 < hh2; py2 += Nwt)                                //  loop through output pixels
   for (px2 = 0; px2 < ww2; px2++)                                         //  outer loop y
   {
      px1 = a * (px2 - ww25) + b * (py2 - hh25) + ww15;                    //  (px1,py1) = corresponding    v.9.3
      py1 = d * (px2 - ww25) + e * (py2 - hh25) + hh15;                    //    point within input pixels

      px0 = int(px1);                                                      //  pixel containing (px1,py1)
      py0 = int(py1);
      
      if (px1 < 0 || px0 >= ww1-1 || py1 < 0 || py0 >= hh1-1) {            //  if outside input pixel array
         pix2 = pixmap2 + (py2 * ww2 + px2) * 3;                           //    output is black
         pix2[0] = pix2[1] = pix2[2] = 0;
         continue;
      }

      pix0 = pixmap1 + (py0 * ww1 + px0) * 3;                              //  4 input pixels based at (px0,py0)
      pix1 = pix0 + ww1 * 3;
      pix2 = pix0 + 3;
      pix3 = pix1 + 3;

      f0 = (px0+1 - px1) * (py0+1 - py1);                                  //  overlap of (px1,py1)
      f1 = (px0+1 - px1) * (py1 - py0);                                    //    in each of the 4 pixels
      f2 = (px1 - px0) * (py0+1 - py1);
      f3 = (px1 - px0) * (py1 - py0);
   
      red =   f0 * pix0[0] + f1 * pix1[0] + f2 * pix2[0] + f3 * pix3[0];   //  sum the weighted inputs
      green = f0 * pix0[1] + f1 * pix1[1] + f2 * pix2[1] + f3 * pix3[1];
      blue =  f0 * pix0[2] + f1 * pix1[2] + f2 * pix2[2] + f3 * pix3[2];
      
      pix2 = pixmap2 + (py2 * ww2 + px2) * 3;                              //  output pixel
      pix2[0] = int(red);
      pix2[1] = int(green);
      pix2[2] = int(blue);
   }
   
   zadd_locked(busy,-1);
   return 0;
}



