/**************************************************************************

   Fotoxx      edit photos and manage collections

   Copyright 2007-2014 Michael Cornelison
   Source URL: http://kornelix.com/fotoxx
   Contact: kornelix@posteo.de
   
   This program is free software: you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program. If not, see http://www.gnu.org/licenses/.

***************************************************************************/

#define EX extern                                                          //  enable extern declarations
#include "fotoxx.h"                                                        //  (variables in fotoxx.h are refs)

/**************************************************************************
   Fotoxx image edit - Color menu functions
***************************************************************************/


//  Shift Colors menu function
//  Gradually shift selected RGB colors into other colors.

editfunc    EFshiftcolr;                                                   //  edit function data
float       shiftred = 0.5, shiftgreen = 0.5, shiftblue = 0.5;

void m_shift_colors(GtkWidget *, const char *)                             //  v.13.12  (formerly m_false_colors())
{
   int    shiftcolr_dialog_event(zdialog* zd, const char *event);
   void * shiftcolr_thread(void *);

   F1_help_topic = "shift_colors";
   
   EFshiftcolr.menufunc = m_shift_colors;
   EFshiftcolr.funcname = "shift_colors";                                  //  function name
   EFshiftcolr.FprevReq = 1;                                               //  use preview
   EFshiftcolr.Farea = 2;                                                  //  select area 0/1/2 = delete/ignore/usable
   EFshiftcolr.threadfunc = shiftcolr_thread;                              //  thread function

   if (! edit_setup(EFshiftcolr)) return;                                  //  setup edit

/***
                  Shift Colors
        
      red:    green =======[]======= blue
      green:   blue ==========[]==== red
      blue:     red ======[]======== green
      all:      all ======[]======== all

                          [done] [cancel]
***/

   zdialog *zd = zdialog_new(ZTX("Shift Colors"),Mwin,Bdone,Bcancel,null);
   EFshiftcolr.zd = zd;

   zdialog_add_widget(zd,"hbox","hb1","dialog",0,"space=3");
   zdialog_add_widget(zd,"vbox","vb1","hb1",0,"homog|space=3");
   zdialog_add_widget(zd,"vbox","vb2","hb1",0,"homog|space=3");
   zdialog_add_widget(zd,"vbox","vb3","hb1",0,"homog|space=3|expand");
   zdialog_add_widget(zd,"vbox","vb4","hb1",0,"homog|space=3");

   zdialog_add_widget(zd,"label","labr","vb1",Bred);
   zdialog_add_widget(zd,"label","labr","vb1",Bgreen);
   zdialog_add_widget(zd,"label","labr","vb1",Bblue);
   zdialog_add_widget(zd,"label","labr","vb1",Ball);

   zdialog_add_widget(zd,"label","labg","vb2",Bgreen);
   zdialog_add_widget(zd,"label","labb","vb2",Bblue);
   zdialog_add_widget(zd,"label","labr","vb2",Bred);
   zdialog_add_widget(zd,"label","laba","vb2",Ball);

   zdialog_add_widget(zd,"hscale","red","vb3","0|1|0.01|0.5");
   zdialog_add_widget(zd,"hscale","green","vb3","0|1|0.01|0.5");
   zdialog_add_widget(zd,"hscale","blue","vb3","0|1|0.01|0.5");
   zdialog_add_widget(zd,"hscale","all","vb3","0|1|0.01|0.5");

   zdialog_add_widget(zd,"label","labb","vb4",Bblue);
   zdialog_add_widget(zd,"label","labr","vb4",Bred);
   zdialog_add_widget(zd,"label","labg","vb4",Bgreen);
   zdialog_add_widget(zd,"label","laba","vb4",Ball);
   
   zdialog_stuff(zd,"red",shiftred);                                       //  start with prior values
   zdialog_stuff(zd,"green",shiftgreen);
   zdialog_stuff(zd,"blue",shiftblue);

   zdialog_resize(zd,300,0);
   zdialog_run(zd,shiftcolr_dialog_event,"save");                          //  run dialog - parallel
   return;
}


//  shift color dialog event and completion function

int shiftcolr_dialog_event(zdialog *zd, const char *event)
{
   float    shiftall;

   if (strEqu(event,"done")) zd->zstat = 1;                                //  apply and quit               v.13.12

   if (zd->zstat)
   {
      if (zd->zstat == 1) edit_done(0);                                    //  done
      else edit_cancel(0);                                                 //  cancel or destroy
      return 1;
   }

   if (strEqu(event,"red")) 
      zdialog_fetch(zd,"red",shiftred);

   if (strEqu(event,"green")) 
      zdialog_fetch(zd,"green",shiftgreen);

   if (strEqu(event,"blue")) 
      zdialog_fetch(zd,"blue",shiftblue);
   
   if (strEqu(event,"all")) {                                              //  v.13.11
      zdialog_fetch(zd,"all",shiftall);
      shiftred = shiftgreen = shiftblue = shiftall;
      zdialog_stuff(zd,"red",shiftred);
      zdialog_stuff(zd,"green",shiftgreen);
      zdialog_stuff(zd,"blue",shiftblue);
   }

   signal_thread();                                                        //  trigger update thread
   return 1;
}


//  thread function - launch multiple working threads to update image

void * shiftcolr_thread(void *)
{
   void  * shiftcolr_wthread(void *arg);                                   //  worker thread

   while (true)
   {
      thread_idle_loop();                                                  //  wait for work or exit request

      for (int ii = 0; ii < Nwt; ii++)                                     //  start worker threads
         start_wthread(shiftcolr_wthread,&Nval[ii]);
      wait_wthreads();                                                     //  wait for completion

      CEF->Fmods++;                                                        //  image modified
      CEF->Fsaved = 0;
      Fpaint2();                                                           //  update window
   }

   return 0;                                                               //  not executed, stop warning
}


void * shiftcolr_wthread(void *arg)                                        //  worker thread function
{
   int         index = *((int *) (arg));
   int         px, py, ii, dist = 0;
   float       *pix1, *pix3;
   float       red1, green1, blue1, red3, green3, blue3;
   float       lossR, lossG, lossB, gainR, gainG, gainB;
   float       shift, move, maxrgb, f1, f2;

   for (py = index; py < E3pxm->hh; py += Nwt)                             //  loop all image pixels
   for (px = 0; px < E3pxm->ww; px++)
   {
      if (sa_stat == 3) {                                                  //  select area active
         ii = py * E3pxm->ww + px;
         dist = sa_pixmap[ii];                                             //  distance from edge
         if (! dist) continue;                                             //  outside pixel
      }
      
      pix1 = PXMpix(E1pxm,px,py);                                          //  input pixel

      red1 = pix1[0];
      green1 = pix1[1];
      blue1 = pix1[2];
      
      lossR = lossG = lossB = 0;
      gainR = gainG = gainB = 0;
      
      shift = shiftred;                                                    //  red shift: green <-- red --> blue
      if (shift < 0.5) {                                                   //  0.0 ... 0.5
         move = red1 * 2.0 * (0.5 - shift);
         lossR += move;
         gainG += move;
      }
      if (shift > 0.5) {                                                   //  0.5 ... 1.0
         move = red1 * 2.0 * (shift - 0.5);
         lossR += move;
         gainB += move;
      }

      shift = shiftgreen;                                                  //  green shift: blue <-- green --> red
      if (shift < 0.5) {                                                   //  0.0 ... 0.5
         move = green1 * 2.0 * (0.5 - shift);
         lossG += move;
         gainB += move;
      }
      if (shift > 0.5) {                                                   //  0.5 ... 1.0
         move = green1 * 2.0 * (shift - 0.5);
         lossG += move;
         gainR += move;
      }

      shift = shiftblue;                                                   //  blue shift: red <-- blue --> green
      if (shift < 0.5) {                                                   //  0.0 ... 0.5
         move = blue1 * 2.0 * (0.5 - shift);
         lossB += move;
         gainR += move;
      }
      if (shift > 0.5) {                                                   //  0.5 ... 1.0
         move = blue1 * 2.0 * (shift - 0.5);
         lossB += move;
         gainG += move;
      }

      pix3 = PXMpix(E3pxm,px,py);                                          //  output pixel
      
      red3 = red1 - lossR + gainR;
      green3 = green1 - lossG + gainG;
      blue3 = blue1 - lossB + gainB;
      
      maxrgb = red3;                                                       //  find max. new RGB color
      if (green3 > maxrgb) maxrgb = green3;
      if (blue3 > maxrgb) maxrgb = blue3;
      
      if (maxrgb > 255.9) {                                                //  if too big, rescale all colors
         red3 = red3 * 255.9 / maxrgb;
         green3 = green3 * 255.9 / maxrgb;
         blue3 = blue3 * 255.9 / maxrgb;
      }
      
      if (sa_stat == 3 && dist < sa_blend) {                               //  select area is active,
         f1 = 1.0 * dist / sa_blend;                                       //    blend changes over sa_blend
         f2 = 1.0 - f1;
         red3 = f1 * red3 + f2 * red1;
         green3 = f1 * green3 + f2 * green1;
         blue3 = f1 * blue3 + f2 * blue1;
      }

      pix3[0] = red3;
      pix3[1] = green3;
      pix3[2] = blue3;
   }

   exit_wthread();
   return 0;                                                               //  not executed, avoid gcc warning
}


/**************************************************************************/

//  make a black & white or color positive or negative, or sepia image

editfunc    EFcolormode;
int         color_mode1, color_mode2;

void m_colormode(GtkWidget *, cchar *)
{
   int colormode_dialog_event(zdialog *zd, cchar *event);

   F1_help_topic = "color_mode";
   
   EFcolormode.menufunc = m_colormode;
   EFcolormode.funcname = "color_mode";
   EFcolormode.Farea = 2;                                                  //  select area usable     v.12.08
   EFcolormode.Frestart = 1;                                               //  allow restart          v.13.04

   if (! edit_setup(EFcolormode)) return;                                  //  setup edit: no preview

   zdialog *zd = zdialog_new(ZTX("Color Mode"),Mwin,Bdone,Bcancel,null);
   EFcolormode.zd = zd;

   zdialog_add_widget(zd,"radio","b&wpos","dialog",ZTX("black/white positive"));
   zdialog_add_widget(zd,"radio","b&wneg","dialog",ZTX("black/white negative"));
   zdialog_add_widget(zd,"radio","colpos","dialog",ZTX("color positive"));
   zdialog_add_widget(zd,"radio","colneg","dialog",ZTX("color negative"));
   zdialog_add_widget(zd,"radio","sepia","dialog",ZTX("sepia"));

   zdialog_stuff(zd,"colpos",1);
   color_mode1 = color_mode2 = 3;

   zdialog_resize(zd,200,0);
   zdialog_run(zd,colormode_dialog_event,"save");                          //  run dialog - parallel

   return;
}


//  dialog event and completion callback function

int colormode_dialog_event(zdialog *zd, cchar *event)
{
   int         ii, dist, px, py;
   float       red1, green1, blue1;
   float       red3, green3, blue3;
   float       *pix1, *pix3;
   float       dnew, dold;
   
   dist = 0;                                                               //  stop compiler warnings
   red3 = green3 = blue3 = 0;
   
   if (strEqu(event,"done")) zd->zstat = 1;                                //  apply and quit               v.13.12

   if (zd->zstat)                                                          //  dialog complete
   {
      if (zd->zstat == 1) edit_done(0);
      else edit_cancel(0);
      return 1;
   }
   
   if (strEqu(event,"b&wpos")) color_mode2 = 1;
   if (strEqu(event,"b&wneg")) color_mode2 = 2;
   if (strEqu(event,"colpos")) color_mode2 = 3;
   if (strEqu(event,"colneg")) color_mode2 = 4;
   if (strEqu(event,"sepia")) color_mode2 = 5;

   if (color_mode2 == color_mode1) return 1;                               //  no changes
   
   for (py = 0; py < E3pxm->hh; py++)
   for (px = 0; px < E3pxm->ww; px++)
   {
      if (sa_stat == 3) {                                                  //  select area active        v.12.08
         ii = py * E1pxm->ww + px;
         dist = sa_pixmap[ii];                                             //  distance from edge
         if (! dist) continue;                                             //  pixel outside area
      }

      pix1 = PXMpix(E1pxm,px,py);

      red1 = pix1[0];
      green1 = pix1[1];
      blue1 = pix1[2];

      switch (color_mode2) 
      {
         case 1: {                                                         //  black and white positive
            red3 = green3 = blue3 = (red1 + green1 + blue1) / 3.0;
            CEF->Fmods++; 
            CEF->Fsaved = 0;
            break;
         }
         
         case 2: {                                                         //  black and white negative
            red3 = green3 = blue3 = 255.0 - (red1 + green1 + blue1) / 3.0;
            CEF->Fmods++; 
            CEF->Fsaved = 0;
            break;
         }
         
         case 3: {                                                         //  color positive (no change)
            red3 = red1;
            green3 = green1;
            blue3 = blue1;
            CEF->Fmods = 0;
            CEF->Fsaved = 0;
            break;
         }
         
         case 4: {                                                         //  color negative
            red3 = 255.0 - red1;
            green3 = 255.0 - green1;
            blue3 = 255.0 - blue1; 
            CEF->Fmods++; 
            CEF->Fsaved = 0;
            break;
         }
         
         case 5: {                                                         //  sepia                           v.12.08
            red3 = green3 = blue3 = (red1 + green1 + blue1) / 3.0;
            green3 = 0.59 * red3;
            blue3 = 0.18 * red3;
            CEF->Fmods++; 
            CEF->Fsaved = 0;
            break;
         }
      }

      pix3 = PXMpix(E3pxm,px,py);
      
      if (sa_stat == 3 && dist < sa_blend) {                               //  blend changes over blendwidth   v.12.08
         dnew = 1.0 * dist / sa_blend;
         dold = 1.0 - dnew;
         red3 = dnew * red3 + dold * red1;
         green3 = dnew * green3 + dold * green1;
         blue3 = dnew * blue3 + dold * blue1;
      }

      pix3[0] = red3;
      pix3[1] = green3;
      pix3[2] = blue3;
   }
   
   color_mode1 = color_mode2;
   Fpaint2();
   return 1;
}


/**************************************************************************/

//  convert color profile of current image

editfunc    EFcolorprof;
char        colorprof1[200] = "/usr/share/color/icc/sRGB.icc";
char        colorprof2[200] = "/usr/share/color/icc/compatibleWithAdobeRGB1998.icc";

void m_colorprof(GtkWidget *, cchar *menu)                                 //  v.13.02
{
   int colorprof_dialog_event(zdialog *zd, cchar *event);
   
   zdialog     *zd;
   
   F1_help_topic = "color_profile";
   
   EFcolorprof.menufunc = m_colorprof;
   EFcolorprof.funcname = "color_profile";
   EFcolorprof.Frestart = 1;                                               //  allow restart             v.13.04
   if (! edit_setup(EFcolorprof)) return;                                  //  setup edit

   zd = zdialog_new(ZTX("Change Color Profile"),Mwin,Bapply,Bdone,Bcancel,null);
   EFcolorprof.zd = zd;

   zdialog_add_widget(zd,"hbox","hb1","dialog",0,"space=3");
   zdialog_add_widget(zd,"label","lab1","hb1",ZTX("input profile"),"space=5");
   zdialog_add_widget(zd,"entry","prof1","hb1",0,"expand|scc=50");
   zdialog_add_widget(zd,"button","butt1","hb1",Bbrowse,"space=5");
   zdialog_add_widget(zd,"hbox","hb2","dialog",0,"space=5");
   zdialog_add_widget(zd,"label","lab2","hb2",ZTX("output profile"),"space=5");
   zdialog_add_widget(zd,"entry","prof2","hb2",0,"expand|scc=50");
   zdialog_add_widget(zd,"button","butt2","hb2",Bbrowse,"space=5");
   
   zdialog_stuff(zd,"prof1",colorprof1);
   zdialog_stuff(zd,"prof2",colorprof2);

   zdialog_run(zd,colorprof_dialog_event,"save");                          //  run dialog, parallel
   return;
}


//  dialog event and completion callback function

int colorprof_dialog_event(zdialog *zd, cchar *event)
{
   cchar    *title = ZTX("color profile");
   char     *file;
   float    *fpix1, *fpix2;
   float    f256 = 1.0 / 256.0;
   uint     Npix, nn;

   cmsHTRANSFORM  cmsxform;
   cmsHPROFILE    cmsprof1, cmsprof2;

   if (strEqu(event,"butt1")) {
      zdialog_fetch(zd,"prof1",colorprof1,200);                            //  select input profile
      file = zgetfile(title,"file",colorprof1);
      if (! file) return 1;
      zdialog_stuff(zd,"prof1",file);
      zfree(file);
   }

   if (strEqu(event,"butt2")) {
      zdialog_fetch(zd,"prof2",colorprof2,200);                            //  select output profile
      file = zgetfile(title,"file",colorprof2);
      if (! file) return 1;
      zdialog_stuff(zd,"prof2",file);
      zfree(file);
   }

   if (strEqu(event,"done")) zd->zstat = 2;                                //  apply and quit               v.13.12

   if (! zd->zstat) return 1;                                              //  wait for user completion
   
   if (zd->zstat) {
      if (zd->zstat == 1) zd->zstat = 0;                                   //  apply, keep dialog open
      else if (zd->zstat == 2) edit_done(0);                               //  done
      else edit_cancel(0);                                                 //  cancel
      if (zd->zstat) return 1;
   }
   
   zdialog_fetch(zd,"prof1",colorprof1,200);                               //  apply, get final profiles
   zdialog_fetch(zd,"prof2",colorprof2,200);

   cmsprof1 = cmsOpenProfileFromFile(colorprof1,"r");
   if (! cmsprof1) {
      zmessageACK(Mwin,0,ZTX("unknown cms profile %s"),colorprof1);
      return 1;
   }

   cmsprof2 = cmsOpenProfileFromFile(colorprof2,"r");
   if (! cmsprof2) {
      zmessageACK(Mwin,0,ZTX("unknown cms profile %s"),colorprof2);
      return 1;
   }
   
   //  calculate the color space transformation table

   cmsxform = cmsCreateTransform(cmsprof1,TYPE_RGB_FLT,cmsprof2,TYPE_RGB_FLT,INTENT_PERCEPTUAL,0);
   if (! cmsxform) {
      zmessageACK(Mwin,0,"cmsCreateTransform() failed");
      return 1;
   }

   fpix1 = E0pxm->pixels;                                                  //  input and output pixels
   fpix2 = E3pxm->pixels;
   Npix = E0pxm->ww * E0pxm->hh;
   
   Ffuncbusy++;  
   zmainloop();

   for (uint ii = 0; ii < 3 * Npix; ii++)                                  //  rescale to range 0 - 0.9999
      fpix2[ii] = f256 * fpix1[ii];

   while (Npix)
   {
      nn = Npix;
      if (nn > 100000) nn = 100000;
      cmsDoTransform(cmsxform,fpix2,fpix2,nn);                             //  speed: 3 megapixels/sec for 3.3 GHz CPU
      fpix2 += nn * 3;
      Npix -= nn;
      zmainloop();
   }

   fpix2 = E3pxm->pixels;
   Npix = E0pxm->ww * E0pxm->hh;
   for (uint ii = 0; ii < 3 * Npix; ii++) {                                //  rescale back to 0 - 255.99
      fpix2[ii] = fpix2[ii] * 256.0;
      if (fpix2[ii] >= 256.0) fpix2[ii] = 255.99;
   }

   Ffuncbusy--;  

   cmsDeleteTransform(cmsxform);                                           //  free resources
   cmsCloseProfile(cmsprof1);
   cmsCloseProfile(cmsprof2);

   CEF->Fmods++;                                                           //  image is modified
   CEF->Fsaved = 0;
   Fpaint2();                                                              //  update window image
   
   return 1;
}


/**************************************************************************/

//  match_color edit function
//  Adjust colors of image 2 to match the colors of image 1
//  using small selected areas in each image as the match standard.

void * match_color_thread(void *);
void   match_color_mousefunc();

float    match_color_RGB1[3];                                              //  image 1 base colors to match
float    match_color_RGB2[3];                                              //  image 2 target colors to match
int      match_color_radius = 10;                                          //  mouse radius
int      match_color_mode = 0;

editfunc    EFmatchcolor;


void m_match_color(GtkWidget *, const char *) 
{
   int    match_color_dialog_event(zdialog* zd, const char *event);
   
   cchar    *title = ZTX("Color Match Images");

   F1_help_topic = "match_colors";
   
   if (checkpend("all")) return;                                           //  v.13.12
   
/*
               Color Match Images
         1  [ 10 ]   mouse radius for color sample
         2  [Open]   image for source color
         3  click on image to get source color
         4  [Open]   image for target color
         5  click on image to set target color
                             [done] [cancel]
*/

   zdialog *zd = zdialog_new(title,Mwin,Bdone,Bcancel,null);               //  match_color dialog
   zdialog_add_widget(zd,"hbox","hb1","dialog",0,"space=2");
   zdialog_add_widget(zd,"vbox","vb1","hb1",0,"homog|space=3");
   zdialog_add_widget(zd,"vbox","vb2","hb1",0,"homog|space=3");
   zdialog_add_widget(zd,"label","labn1","vb1","1");
   zdialog_add_widget(zd,"label","labn2","vb1","2");
   zdialog_add_widget(zd,"label","labn3","vb1","3");
   zdialog_add_widget(zd,"label","labn4","vb1","4");
   zdialog_add_widget(zd,"label","labn5","vb1","5");
   zdialog_add_widget(zd,"hbox","hbrad","vb2");
   zdialog_add_widget(zd,"spin","radius","hbrad","1|20|1|10","space=5");
   zdialog_add_widget(zd,"label","labrad","hbrad",ZTX("mouse radius for color sample"));
   zdialog_add_widget(zd,"hbox","hbop1","vb2");
   zdialog_add_widget(zd,"button","open1","hbop1",ZTX("Open"),"space=5");
   zdialog_add_widget(zd,"label","labop1","hbop1",ZTX("image for source color"));
   zdialog_add_widget(zd,"hbox","hbclik1","vb2");
   zdialog_add_widget(zd,"label","labclik1","hbclik1",ZTX("click on image to get source color"));
   zdialog_add_widget(zd,"hbox","hbop2","vb2");
   zdialog_add_widget(zd,"button","open2","hbop2",ZTX("Open"),"space=5");
   zdialog_add_widget(zd,"label","labop2","hbop2",ZTX("image to set matching color"));
   zdialog_add_widget(zd,"hbox","hbclik2","vb2");
   zdialog_add_widget(zd,"label","labclik2","hbclik2",ZTX("click on image to set matching color"));

   zdialog_stuff(zd,"radius",match_color_radius);                          //  remember last radius

   EFmatchcolor.funcname = "match-color";
   EFmatchcolor.Farea = 1;                                                 //  select area ignored
   EFmatchcolor.zd = zd;                                                   //  v.13.05
   EFmatchcolor.threadfunc = match_color_thread;
   EFmatchcolor.mousefunc = match_color_mousefunc;
   
   match_color_mode = 0;
   if (curr_file) {
      match_color_mode = 1;                                                //  image 1 ready to click
      takeMouse(match_color_mousefunc,0);                                  //  connect mouse function
   }

   zdialog_run(zd,match_color_dialog_event);                               //  run dialog - parallel
   return;
}


//  match_color dialog event and completion function

int match_color_dialog_event(zdialog *zd, const char *event)               //  match_color dialog event function
{
   int      err;
   
   if (strEqu(event,"done")) zd->zstat = 1;                                //  apply and quit               v.13.12

   if (zd->zstat)                                                          //  end dialog
   {
      if (match_color_mode == 4) {                                         //  edit was started
         if (zd->zstat == 1) edit_done(0);
         else edit_cancel(0);
         match_color_mode = 0;
         return 1;
      }

      freeMouse();                                                         //  abandoned
      zdialog_free(zd);
      match_color_mode = 0;
      return 1;
   }
   
   if (strEqu(event,"radius"))                                             //  set new mouse radius
      zdialog_fetch(zd,"radius",match_color_radius);
   
   if (strEqu(event,"open1"))                                              //  get image 1 for color source
   {
      match_color_mode = 0;
      err = f_open(null);
      if (! err) match_color_mode = 1;                                     //  image 1 ready to click
   }

   if (strEqu(event,"open2"))                                              //  get image 2 to set matching color
   {
      if (match_color_mode < 2) {
         zmessageACK(Mwin,0,ZTX("select source image color first"));       //  check that RGB1 has been set
         return 1;
      }
      match_color_mode = 2;
      err = f_open(null);
      if (err) return 1;
      match_color_mode = 3;                                                //  image 2 ready to click
   }

   takeMouse(match_color_mousefunc,0);                                     //  reconnect mouse function
   return 1;
}


//  mouse function - click on image and get colors to match

void match_color_mousefunc()
{
   void  match_color_getRGB(int px, int py, float rgb[3]);

   int      px, py, mrect;

   if (match_color_mode < 1) return;                                       //  no image available yet
   
   mrect = 2 * match_color_radius;
   draw_mousearc(Mxposn,Myposn,mrect,mrect,0);
   
   if (LMclick)
   {
      LMclick = 0;
      px = Mxclick;
      py = Myclick;
   
      if (match_color_mode == 1 || match_color_mode == 2)                  //  image 1 ready to click
      {
         match_color_getRGB(px,py,match_color_RGB1);                       //  get RGB1 color
         match_color_mode = 2;
         return;
      }
      
      if (match_color_mode == 3 || match_color_mode == 4)                  //  image 2 ready to click
      {
         if (match_color_mode == 4) edit_reset();                          //  v.13.05
         else {
            if (! edit_setup(EFmatchcolor)) return;                        //  setup edit - thread will launch
            match_color_mode = 4;                                          //  edit waiting for cancel or done
         }

         match_color_getRGB(px,py,match_color_RGB2);                       //  get RGB2 color
         signal_thread();                                                  //  update the target image
         return;
      }
   }

   return;
}


//  get the RGB averages for pixels within mouse radius

void match_color_getRGB(int px, int py, float rgb[3])
{
   int      rad1 = match_color_radius;
   int      rad2 = rad1 * rad1;
   int      rad, npix, qx, qy;
   float    red, green, blue;
   float    *pix1;
   PXM      *pxm;

   pxm = PXM_load(curr_file,1);                                            //  popup ACK if error     v.12.03
   if (! pxm) return;

   npix = 0;   
   red = green = blue = 0;

   for (qy = py-rad1; qy <= py+rad1; qy++)
   for (qx = px-rad1; qx <= px+rad1; qx++)
   {
      if (qx < 0 || qx > pxm->ww-1) continue;
      if (qy < 0 || qy > pxm->hh-1) continue;
      rad = (qx-px) * (qx-px) + (qy-py) * (qy-py);
      if (rad > rad2) continue;
      pix1 = PXMpix(pxm,qx,qy);
      red += pix1[0];
      green += pix1[1];
      blue += pix1[2];
      npix++;
   }
   
   rgb[0] = red / npix;
   rgb[1] = green / npix;
   rgb[2] = blue / npix;
   
   PXM_free(pxm);
   return;
}


//  thread function - start multiple working threads

void * match_color_thread(void *)
{
   void  * match_color_wthread(void *arg);                                 //  worker thread
   
   while (true)
   {
      thread_idle_loop();                                                  //  wait for work or exit request
      
      for (int ii = 0; ii < Nwt; ii++)                                     //  start worker threads
         start_wthread(match_color_wthread,&Nval[ii]);
      wait_wthreads();                                                     //  wait for completion

      CEF->Fmods++; 
      CEF->Fsaved = 0;
      Fpaint2();                                                           //  update window
   }

   return 0;                                                               //  not executed, avoid warning
}


void * match_color_wthread(void *arg)                                      //  worker thread function
{
   int         index = *((int *) (arg));
   int         px, py;
   float       *pix3;
   float       Rred, Rgreen, Rblue;
   float       red, green, blue, cmax;
   
   Rred = match_color_RGB1[0] / match_color_RGB2[0];                       //  color adjustment ratios
   Rgreen = match_color_RGB1[1] / match_color_RGB2[1];
   Rblue = match_color_RGB1[2] / match_color_RGB2[2];

   for (py = index; py < E3pxm->hh; py += Nwt)                             //  loop all image pixels
   for (px = 0; px < E3pxm->ww; px++)
   {
      pix3 = PXMpix(E3pxm,px,py);
      
      red = pix3[0] * Rred;                                                //  adjust colors
      green = pix3[1] * Rgreen;
      blue = pix3[2] * Rblue;
      
      cmax = red;                                                          //  check for overflow
      if (green > cmax) cmax = green;
      if (blue > cmax) cmax = blue;

      if (cmax > 255.9) {                                                  //  fix overflow
         red = red * 255.9 / cmax;
         green = green * 255.9 / cmax;
         blue = blue * 255.9 / cmax;
      }      

      pix3[0] = red;
      pix3[1] = green;
      pix3[2] = blue;
   }

   exit_wthread();
   return 0;                                                               //  not executed, avoid gcc warning
}


/**************************************************************************/

//  Show RGB values for 1-9 pixels selected with mouse-clicks.
//  Revise RGB values for the selected points and then revise all other 
//  pixels to match, using a distance-weighted average of the selected pixels.

void  RGBR_dialog();
void  RGBR_mousefunc();
void  RGBR_stuff();
void * RGBR_thread(void *);

int      RGBRpixel[9][2];                                                  //  last 9 pixel locations clicked
float    RGBRval1[9][3];                                                   //  original RGB values, 0 to 255.9
float    RGBRval3[9][3];                                                   //  revised RGB values, 0 to 255.9
int      RGBRmetric = 1;                                                   //  1/2/3 = /RGB/EV/OD
int      RGBRdelta = 0;                                                    //  flag, delta mode
int      RGBRnpix;                                                         //  no. of selected pixels
double   RGBRtime;                                                         //  last click time
int      RGBRblend;                                                        //  blend factor, 1 to 100

editfunc EFrgbrevise;                                                      //  edit function parameters

#define RGB2EV(rgb) (log2(rgb)-7)                                          //  RGB 0.1 to 255.9 >> EV -10 to +1
#define EV2RGB(ev) (pow(2,ev+7))                                           //  inverse
#define RGB2OD(rgb) (2-log10(100*rgb/256))                                 //  RGB 0.1 to 255.9 >> OD 3.4 to 0.000017
#define OD2RGB(od) (pow(10,2.40824-od))                                    //  inverse


void m_revise_RGB(GtkWidget *, cchar *menu)
{
   int   RGBR_event(zdialog *zd, cchar *event);

   GtkWidget   *widget;
   cchar       *limits;
   zdialog     *zd;
   cchar       *mess = ZTX("Click image to select pixels.");

   PangoFontDescription    *widgetfont;
   widgetfont = pango_font_description_from_string("Monospace 8");         //  small monospace font for widgets

   F1_help_topic = "revise_RGB";
   
   EFrgbrevise.menufunc = m_revise_RGB;
   EFrgbrevise.funcname = "revise_RGB";                                    //  setup edit function
   EFrgbrevise.threadfunc = RGBR_thread;
   EFrgbrevise.mousefunc = RGBR_mousefunc;
   EFrgbrevise.Farea = 1;
   if (! edit_setup(EFrgbrevise)) return;

/***
   
      Click image to select pixels.
      [x] delta
      Metric: (o) RGB  (o) EV  (o) OD
      Pixel          Red      Green    Blue
      A xxxx xxxx   [xxxx|v] [xxxx|v] [xxxx|v]                             //  spin buttons for RGB/EV/OD values
      B xxxx xxxx   [xxxx|v] [xxxx|v] [xxxx|v]
      C xxxx xxxx   [xxxx|v] [xxxx|v] [xxxx|v] 
      D xxxx xxxx   [xxxx|v] [xxxx|v] [xxxx|v] 
      E xxxx xxxx   [xxxx|v] [xxxx|v] [xxxx|v] 
      F xxxx xxxx   [xxxx|v] [xxxx|v] [xxxx|v] 
      G xxxx xxxx   [xxxx|v] [xxxx|v] [xxxx|v] 
      H xxxx xxxx   [xxxx|v] [xxxx|v] [xxxx|v] 
      I xxxx xxxx   [xxxx|v] [xxxx|v] [xxxx|v] 
      Blend  ==============[]================

                     [reset] [done] [cancel]
***/
   
   zd = zdialog_new(ZTX("Revise RGB"),Mwin,Breset,Bdone,Bcancel,null);
   EFrgbrevise.zd = zd;

   zdialog_add_widget(zd,"hbox","hbmess","dialog",0,"space=3");
   zdialog_add_widget(zd,"label","labmess","hbmess",mess,"space=5");

   zdialog_add_widget(zd,"hbox","hbmym","dialog");
   zdialog_add_widget(zd,"check","delta","hbmym","delta","space=5");
   zdialog_stuff(zd,"delta",RGBRdelta);

   zdialog_add_widget(zd,"hbox","hbmetr","dialog",0,"space=3");
   zdialog_add_widget(zd,"label","labmetr","hbmetr",ZTX("Metric:"),"space=5");
   zdialog_add_widget(zd,"radio","radRGB","hbmetr","RGB","space=3");
   zdialog_add_widget(zd,"radio","radEV","hbmetr","EV","space=3");
   zdialog_add_widget(zd,"radio","radOD","hbmetr","OD","space=3");
   
   if (RGBRmetric == 1) zdialog_stuff(zd,"radRGB",1);                      //  set current metric
   if (RGBRmetric == 2) zdialog_stuff(zd,"radEV",1);
   if (RGBRmetric == 3) zdialog_stuff(zd,"radOD",1);

   zdialog_add_widget(zd,"hbox","hbdata","dialog");
   zdialog_add_widget(zd,"vbox","vbpix","hbdata",0,"space=3|homog");       //  vbox for pixel locations
   zdialog_add_widget(zd,"hbox","hbpix","vbpix");
   zdialog_add_widget(zd,"label","labpix","hbpix","Pixel");                //  header
   
   char hbpixx[8] = "hbpixx", pixx[8] = "pixx";

   for (int ii = 1; ii < 10; ii++) {                                       //  add labels for pixel locations
      hbpixx[5] = '0' + ii;
      pixx[3] = '0' + ii;
      zdialog_add_widget(zd,"hbox",hbpixx,"vbpix");                        //  add hbox "hbpix1" to "hbpix9"
      zdialog_add_widget(zd,"label",pixx,hbpixx);                          //  add label "pix1" to "pix9"
      widget = zdialog_widget(zd,pixx);
      gtk_widget_override_font(widget,widgetfont);                         //  use monofont
   }

   if (RGBRmetric == 1) limits = "-255.9|255.9|0.1|0.0";                   //  metric = RGB
   else if (RGBRmetric == 2) limits = "-8|8|0.01|0.0";                     //           EV
   else limits = "-3|3|0.002|0.0";                                         //           OD

   zdialog_add_widget(zd,"vbox","vbdat","hbdata",0,"space=3|homog");       //  vbox for pixel RGB values
   zdialog_add_widget(zd,"hbox","hbrgb","vbdat",0,"homog");
   zdialog_add_widget(zd,"label","labr","hbrgb",Bred);
   zdialog_add_widget(zd,"label","labg","hbrgb",Bgreen);
   zdialog_add_widget(zd,"label","labb","hbrgb",Bblue);

   char hbdatx[8] = "hbdatx", redx[8] = "redx", greenx[8] = "greenx", bluex[8] = "bluex";

   for (int ii = 1; ii < 10; ii++) {
      hbdatx[5] = '0' + ii;
      redx[3] = '0' + ii;
      greenx[5] = '0' + ii;
      bluex[4] = '0' + ii;
      zdialog_add_widget(zd,"hbox",hbdatx,"vbdat");                        //  add hbox "hbdat1" to "hbdat9"
      zdialog_add_widget(zd,"spin",redx,hbdatx,limits,"space=3");          //  add spin buttons for "red1" to "red9" etc.
      zdialog_add_widget(zd,"spin",greenx,hbdatx,limits,"space=3");
      zdialog_add_widget(zd,"spin",bluex,hbdatx,limits,"space=3");
      widget = zdialog_widget(zd,redx);
      gtk_widget_override_font(widget,widgetfont);                         //  use monofont
      widget = zdialog_widget(zd,greenx);
      gtk_widget_override_font(widget,widgetfont);
      widget = zdialog_widget(zd,bluex);
      gtk_widget_override_font(widget,widgetfont);
   }
   
   zdialog_add_widget(zd,"hbox","hbsoft","dialog","space=5");
   zdialog_add_widget(zd,"label","labsoft","hbsoft",ZTX("Blend"),"space=3");
   zdialog_add_widget(zd,"hscale","blend","hbsoft","0|100|1|20","space=5|expand");
   zdialog_add_widget(zd,"label","softval","hbsoft","20");

   if (strEqu(menu,"restart"))                                             //  stuff current pixels if restart
      RGBR_stuff();
   else {
      RGBRblend = 20;                                                      //  default slider value
      RGBRnpix = 0;                                                        //  no pixels selected yet
   }
   
   zdialog_run(zd,RGBR_event,"save");                                      //  run dialog

   takeMouse(RGBR_mousefunc,dragcursor);                                   //  connect mouse function
   return;
}


//  dialog event function

int RGBR_event(zdialog *zd, cchar *event)
{
   int         button;
   int         ii, jj;
   char        text[8];
   float       val1, val3;

   if (strEqu(event,"done")) zd->zstat = 2;                                //  apply and quit               v.13.12

   if (zd->zstat) 
   {
      if (zd->zstat == 1) {                                                //  Reset
         zd->zstat = 0;                                                    //  keep dialog active
         RGBRnpix = 0;                                                     //  no pixels selected yet
         RGBR_stuff();                                                     //  clear dialog
         edit_reset();                                                     //  reset edits 
         return 1;
      }
      else if (zd->zstat == 2) edit_done(0);
      else edit_cancel(0);                                                 //  Cancel or [x]
      EFrgbrevise.zd = 0;
      erase_toptext(101);                                                  //  erase pixel labels from image
      return 1;
   }
   
   if (strEqu(event,"focus"))                                              //  toggle mouse capture
      takeMouse(RGBR_mousefunc,dragcursor);                                //  connect mouse function
   
   if (strEqu(event,"blend")) {                                            //  new blend factor
      zdialog_fetch(zd,"blend",RGBRblend);
      sprintf(text,"%d",RGBRblend);                                        //  numeric feedback
      zdialog_stuff(zd,"softval",text);
      signal_thread();
      return 1;
   }

   if (strnEqu(event,"rad",3))                                             //  metric was changed
   {
      if (strEqu(event,"radRGB")) {                                        //  RGB
         zdialog_fetch(zd,event,button);
         if (button) RGBRmetric = 1;
      }
         
      if (strEqu(event,"radEV")) {                                         //  EV
         zdialog_fetch(zd,event,button);
         if (button) RGBRmetric = 2;
      }
         
      if (strEqu(event,"radOD")) {                                         //  OD
         zdialog_fetch(zd,event,button);
         if (button) RGBRmetric = 3;
      }

      if (button) {                                                        //  restart with new limits
         edit_cancel(0);
         m_revise_RGB(0,"restart");
      }

      return 1;
   }

   if (strEqu(event,"delta")) {                                            //  change absolute/delta mode
      zdialog_fetch(zd,"delta",RGBRdelta);
      edit_cancel(0);
      m_revise_RGB(0,"restart");
      return 1;
   }
   
   ii = -1;                                                                //  no RGB change yet

   if (strnEqu(event,"red",3)) {                                           //  red1 - red9 was changed
      ii = event[3] - '1';                                                 //  pixel index 0-8
      jj = 0;                                                              //  color = red
   }

   if (strnEqu(event,"green",5)) {                                         //  green1 - green9
      ii = event[5] - '1';
      jj = 1;
   }

   if (strnEqu(event,"blue",4)) {                                          //  blue1 - blue9
      ii = event[4] - '1';
      jj = 2;
   }
   
   if (ii >= 0 && ii < 9)                                                  //  RGB value was revised
   {
      val1 = RGBRval1[ii][jj];                                             //  original pixel RGB value
      if (RGBRmetric == 2) val1 = RGB2EV(val1);                            //  convert to EV or OD units
      if (RGBRmetric == 3) val1 = RGB2OD(val1);

      zdialog_fetch(zd,event,val3);                                        //  revised RGB/EV/OD value from dialog
      if (RGBRdelta) val3 += val1;                                         //  if delta mode, make absolute

      if (RGBRmetric == 2) val3 = EV2RGB(val3);                            //  convert EV/OD to RGB value
      if (RGBRmetric == 3) val3 = OD2RGB(val3);

      if (fabsf(RGBRval3[ii][jj] - val3) < 0.001) return 1;                //  ignore re-entry after change

      if (val3 < 0.1) val3 = 0.1;                                          //  limit RGB within 0.1 to 255.9
      if (val3 > 255.9) val3 = 255.9;

      RGBRval3[ii][jj] = val3;                                             //  new RGB value for pixel
      
      if (RGBRmetric == 2) val3 = RGB2EV(val3);                            //  convert RGB to EV/OD units
      if (RGBRmetric == 3) val3 = RGB2OD(val3);

      if (RGBRdelta) val3 -= val1;                                         //  absolute back to relative
      zdialog_stuff(zd,event,val3);                                        //  limited value back to dialog
      
      signal_thread();                                                     //  signal thread to update image
   }

   return 1;
}


//  mouse function

void RGBR_mousefunc()                                                      //  mouse function
{
   int         ii;
   float       *pix3;

   if (! LMclick) return;
   LMclick = 0;

   RGBRtime = get_seconds();                                               //  mark time of pixel click
   
   if (RGBRnpix == 9) {                                                    //  if table is full (9 entries)
      for (ii = 1; ii < 9; ii++) {                                         //    move positions 1-8 up 
         RGBRpixel[ii-1][0] = RGBRpixel[ii][0];                            //      to fill positions 0-7
         RGBRpixel[ii-1][1] = RGBRpixel[ii][1];
      }
      RGBRnpix = 8;                                                        //  count is now 8 entries
   }
   
   ii = RGBRnpix;                                                          //  next table position to fill
   RGBRpixel[ii][0] = Mxclick;                                             //  newest pixel
   RGBRpixel[ii][1] = Myclick;
   
   pix3 = PXMpix(E3pxm,Mxclick,Myclick);                                   //  get initial RGB values from
   RGBRval1[ii][0] = RGBRval3[ii][0] = pix3[0];                            //    modified image E3
   RGBRval1[ii][1] = RGBRval3[ii][1] = pix3[1];
   RGBRval1[ii][2] = RGBRval3[ii][2] = pix3[2];

   RGBRnpix++;                                                             //  up pixel count
   RGBR_stuff();                                                           //  stuff pixels and values into dialog
   return;
}


//  stuff dialog with current pixels and their RGB/EV/OD values

void RGBR_stuff()
{
   static char    lab1[9][4] = { "A", "B", "C", "D", "E", "F", "G", "H", "I" };
   static char    lab2[9][4] = { " A ", " B ", " C ", " D ", " E ", " F ", " G ", " H ", " I " };

   int         px, py;
   float       red1, green1, blue1, red3, green3, blue3;
   char        text[100], pixx[8] = "pixx";
   char        redx[8] = "redx", greenx[8] = "greenx", bluex[8] = "bluex";
   zdialog     *zd = EFrgbrevise.zd;
   
   erase_toptext(101);                                                     //  erase prior labels from image
   
   for (int ii = 0; ii < 9; ii++)                                          //  loop slots 0-8
   {
      pixx[3] = '1' + ii;                                                  //  widget names "pix1" ... "pix9"
      redx[3] = '1' + ii;                                                  //  widget names "red1" ... "red9"
      greenx[5] = '1' + ii;
      bluex[4] = '1' + ii;

      px = RGBRpixel[ii][0];                                               //  next pixel to report
      py = RGBRpixel[ii][1];
      if (ii >= RGBRnpix) {                                                //  > last pixel selected
         zdialog_stuff(zd,pixx,"");
         zdialog_stuff(zd,redx,0);                                         //  blank pixel and zero values
         zdialog_stuff(zd,greenx,0);
         zdialog_stuff(zd,bluex,0);
         continue;
      }

      sprintf(text,"%s %4d %4d",lab1[ii],px,py);                           //  format pixel "A nnnn nnnn"
      zdialog_stuff(zd,pixx,text);                                         //  pixel >> widget
      
      add_toptext(101,px,py,lab2[ii],"Sans 8");                            //  paint label on image at pixel
      
      red1 = RGBRval1[ii][0];                                              //  original RGB values for pixel
      green1 = RGBRval1[ii][1];
      blue1 = RGBRval1[ii][2];
      red3 = RGBRval3[ii][0];                                              //  revised RGB values
      green3 = RGBRval3[ii][1];
      blue3 = RGBRval3[ii][2];
      
      if (RGBRmetric == 2) {                                               //  convert to EV units if needed
         red1 = RGB2EV(red1);
         green1 = RGB2EV(green1);
         blue1 = RGB2EV(blue1);
         red3 = RGB2EV(red3);
         green3 = RGB2EV(green3);
         blue3 = RGB2EV(blue3);
      }
      
      if (RGBRmetric == 3) {                                               //  or OD units
         red1 = RGB2OD(red1);
         green1 = RGB2OD(green1);
         blue1 = RGB2OD(blue1);
         red3 = RGB2OD(red3);
         green3 = RGB2OD(green3);
         blue3 = RGB2OD(blue3);
      }
      
      if (RGBRdelta) {                                                     //  dialog is delta mode
         red3 -= red1;
         green3 -= green1;
         blue3 -= blue1;
      }
      
      zdialog_stuff(zd,redx,red3);
      zdialog_stuff(zd,greenx,green3);
      zdialog_stuff(zd,bluex,blue3);
   }
   
   zdialog_stuff(zd,"blend",RGBRblend);

   signal_thread();
   return;
}


//  thread function - multiple working threads to update image

void * RGBR_thread(void *)
{
   void  * RGBR_wthread(void *arg);                                        //  worker thread

   while (true)
   {
      thread_idle_loop();                                                  //  wait for work or exit request
      zsleep(0.3);                                                         //  more time for dialog controls
      
      if (RGBRnpix < 1) continue;                                          //  must have 1+ pixels in table
      
      for (int ii = 0; ii < Nwt; ii++)                                     //  start worker threads
         start_wthread(RGBR_wthread,&Nval[ii]);
      wait_wthreads();                                                     //  wait for completion

      CEF->Fmods++;                                                        //  image modified
      CEF->Fsaved = 0;
      Fpaint2();                                                           //  update window
   }

   return 0;                                                               //  not executed, stop warning
}


void * RGBR_wthread(void *arg)                                             //  worker thread function
{
   int         index = *((int *) (arg));
   int         px1, py1, px2, py2, ii;
   float       *pix1, *pix3;
   float       dist[9], weight[9], sumdist;
   float       blend, delta, red, green, blue, max;
   
   blend = E1pxm->ww;
   if (E1pxm->hh > E1pxm->ww) blend = E1pxm->hh;
   blend = blend * blend;
   blend = 0.0002 * RGBRblend * RGBRblend * blend + 100;                   //  100 to 2 * E1pxm->ww**2
   
   for (py1 = index; py1 < E3pxm->hh; py1 += Nwt)                          //  loop all image pixels
   for (px1 = 0; px1 < E3pxm->ww; px1++)
   {
      for (sumdist = ii = 0; ii < RGBRnpix; ii++)                          //  compute weight of each revision point
      {
         px2 = RGBRpixel[ii][0];
         py2 = RGBRpixel[ii][1];
         dist[ii] = (px1-px2)*(px1-px2) + (py1-py2)*(py1-py2);             //  distance (px1,py1) to (px2,py2)
         dist[ii] = 1.0 / (dist[ii] + blend);                              //  blend reduces peaks at revision points
         sumdist += dist[ii];                                              //  sum inverse distances
      }
      
      for (ii = 0; ii < RGBRnpix; ii++)                                    //  weight of each point
         weight[ii] = dist[ii] / sumdist;

      pix1 = PXMpix(E1pxm,px1,py1);                                        //  input pixel
      pix3 = PXMpix(E3pxm,px1,py1);                                        //  output pixel
      
      red = pix1[0];
      green = pix1[1];
      blue = pix1[2];
      
      for (ii = 0; ii < RGBRnpix; ii++) {                                  //  apply weighted color changes
         delta = RGBRval3[ii][0] - RGBRval1[ii][0];                        //    to each color
         red += weight[ii] * delta;
         delta = RGBRval3[ii][1] - RGBRval1[ii][1];
         green += weight[ii] * delta;
         delta = RGBRval3[ii][2] - RGBRval1[ii][2];
         blue += weight[ii] * delta;
      }
      
      max = red;
      if (green > max) max = green;
      if (blue > max) max = blue;

      if (max > 255.9) {                                                   //  stop overflow/underflow
         red = red * 255.9 / max;
         green = green * 255.9 / max;
         blue = blue * 255.9 / max;
      }
      
      if (red < 0) red = 0;
      if (green < 0) green = 0;
      if (blue < 0) blue = 0;
      
      pix3[0] = red;
      pix3[1] = green;
      pix3[2] = blue;
   }

   exit_wthread();
   return 0;                                                               //  not executed, avoid gcc warning
}


/**************************************************************************/

//  CMYK menu function
//  Adjust Brightness, contrast, and color levels using RGB or CMY colors

editfunc    EFcmyk;                                                        //  edit function data
float       CMYKinputs[8];

void m_CMYK(GtkWidget *, const char *)                                     //  UI simplified                v.13.10
{
   int    CMYK_dialog_event(zdialog *zd, cchar *event);
   void * CMYK_thread(void *);

   F1_help_topic = "CMYK";
   
   EFcmyk.menufunc = m_CMYK;
   EFcmyk.funcname = "CMYK";                                               //  function name
   EFcmyk.FprevReq = 1;                                                    //  use preview
   EFcmyk.Farea = 2;                                                       //  select area usable
   EFcmyk.Frestart = 1;                                                    //  allow restart                v.13.04
   EFcmyk.threadfunc = CMYK_thread;                                        //  thread function
   if (! edit_setup(EFcmyk)) return;                                       //  setup edit
   
/***
    _______________________________________________________
   |                                                       |
   |  +Brightness      =====[]=====  Contrast =====[]===== |
   |  +Red -Cyan       =====[]=====     Red   =====[]===== |
   |  +Green -Magenta  =====[]=====   Green   =====[]===== |
   |  +Blue -Yellow    =====[]=====    Blue   =====[]===== |
   |                                                       |
   |                              [reset] [done] [cancel]  |
   |_______________________________________________________|
   
***/

   zdialog *zd = zdialog_new("CMYK",Mwin,Breset,Bdone,Bcancel,null);       //  new dialog
   EFcmyk.zd = zd;                                                         //  add dialog to edit function

   zdialog_add_widget(zd,"hbox","hb2","dialog");
   zdialog_add_widget(zd,"vbox","vb1","hb2",0,"homog|space=5");
   zdialog_add_widget(zd,"vbox","vb2","hb2",0,"homog|expand|space=5");
   zdialog_add_widget(zd,"label","space","hb2",0,"space=8");
   zdialog_add_widget(zd,"vbox","vb3","hb2",0,"homog|space=5");
   zdialog_add_widget(zd,"vbox","vb4","hb2",0,"homog|expand|space=5");
   zdialog_add_widget(zd,"label","labBriteDens","vb1",ZTX("+Brightness"));
   zdialog_add_widget(zd,"label","labRedDens","vb1",ZTX("+Red -Cyan"));
   zdialog_add_widget(zd,"label","labGreenDens","vb1",ZTX("+Green -Magenta"));
   zdialog_add_widget(zd,"label","labBlueDens","vb1",ZTX("+Blue -Yellow"));
   zdialog_add_widget(zd,"hscale","BriteDens","vb2","-1|+1|0.001|0","expand");
   zdialog_add_widget(zd,"hscale","RedDens","vb2","-1|+1|0.001|0","expand");
   zdialog_add_widget(zd,"hscale","GreenDens","vb2","-1|+1|0.001|0","expand");
   zdialog_add_widget(zd,"hscale","BlueDens","vb2","-1|+1|0.001|0","expand");
   zdialog_add_widget(zd,"label","labContrast","vb3",Bcontrast);
   zdialog_add_widget(zd,"label","labRedCon","vb3",ZTX("Red"));
   zdialog_add_widget(zd,"label","labGreenCon","vb3",ZTX("Green"));
   zdialog_add_widget(zd,"label","labBlueCon","vb3",ZTX("Blue"));
   zdialog_add_widget(zd,"hscale","Contrast","vb4","-1|+1|0.001|0","expand");
   zdialog_add_widget(zd,"hscale","RedCon","vb4","-1|+1|0.001|0","expand");
   zdialog_add_widget(zd,"hscale","GreenCon","vb4","-1|+1|0.001|0","expand");
   zdialog_add_widget(zd,"hscale","BlueCon","vb4","-1|+1|0.001|0","expand");

   zdialog_resize(zd,500,0);
   zdialog_restore_inputs(zd);                                             //  restore prior inputs         v.13.12
   zdialog_run(zd,CMYK_dialog_event,"save");                               //  run dialog - parallel

   return;
}


//  CMYK dialog event and completion function

int CMYK_dialog_event(zdialog *zd, cchar *event)                           //  CMYK dialog event function
{
   int      mod = 0;

   if (strEqu(event,"done")) zd->zstat = 2;                                //  apply and quit               v.13.12

   if (zd->zstat)
   {
      if (zd->zstat == 1) {                                                //  reset
         zd->zstat = 0;                                                    //  keep dialog active
         CMYKinputs[0] = 0;
         CMYKinputs[1] = 0;
         CMYKinputs[2] = 0;
         CMYKinputs[3] = 0;
         CMYKinputs[4] = 0;
         CMYKinputs[5] = 0;
         CMYKinputs[6] = 0;
         CMYKinputs[7] = 0;
         zdialog_stuff(zd,"BriteDens",0);
         zdialog_stuff(zd,"RedDens",0);
         zdialog_stuff(zd,"GreenDens",0);
         zdialog_stuff(zd,"BlueDens",0);
         zdialog_stuff(zd,"Contrast",0);
         zdialog_stuff(zd,"RedCon",0);
         zdialog_stuff(zd,"GreenCon",0);
         zdialog_stuff(zd,"BlueCon",0);
         mod++;
      }
      else if (zd->zstat == 2) edit_done(0);                               //  done
      else edit_cancel(0);                                                 //  cancel or destroy
      if (zd->zstat) return 1;
   }
   
   if (strstr("focus",event)) return 1;

   zdialog_fetch(zd,"BriteDens",CMYKinputs[0]);                            //  get all inputs
   zdialog_fetch(zd,"RedDens",CMYKinputs[1]);
   zdialog_fetch(zd,"GreenDens",CMYKinputs[2]);
   zdialog_fetch(zd,"BlueDens",CMYKinputs[3]);
   zdialog_fetch(zd,"Contrast",CMYKinputs[4]);
   zdialog_fetch(zd,"RedCon",CMYKinputs[5]);
   zdialog_fetch(zd,"GreenCon",CMYKinputs[6]);
   zdialog_fetch(zd,"BlueCon",CMYKinputs[7]);
   
   if (CMYKinputs[0]) mod++;
   if (CMYKinputs[1]) mod++;
   if (CMYKinputs[2]) mod++;
   if (CMYKinputs[3]) mod++;
   if (CMYKinputs[4]) mod++;
   if (CMYKinputs[5]) mod++;
   if (CMYKinputs[6]) mod++;
   if (CMYKinputs[7]) mod++;
   
   if (mod) signal_thread();                                               //  trigger update thread

   CEF->Fmods = mod;
   CEF->Fsaved = 0;
   return 1;
}


//  thread function - multiple working threads to update image

void * CMYK_thread(void *)
{
   void  * CMYK_wthread(void *arg);                                        //  worker thread

   while (true)
   {
      thread_idle_loop();                                                  //  wait for work or exit request
      
      if (sa_stat == 3) Fbusy_goal = sa_Npixel;                            //  set up progress monitor
      else  Fbusy_goal = E3pxm->ww * E3pxm->hh;
      Fbusy_done = 0;

      for (int ii = 0; ii < Nwt; ii++)                                     //  start worker threads
         start_wthread(CMYK_wthread,&Nval[ii]);
      wait_wthreads();                                                     //  wait for completion

      Fbusy_goal = Fbusy_done = 0;
      CEF->Fmods++;                                                        //  image modified
      CEF->Fsaved = 0;
      Fpaint2();                                                           //  update window
   }

   return 0;                                                               //  not executed, stop warning
}


//  worker thread function

void * CMYK_wthread(void *arg)                                             //  overhauled                   v.13.12
{
   float    R1, G1, B1, R3, G3, B3;
   float    briA, briR, briG, briB;
   float    conA, conR, conG, conB;
   float    R, G, B;

   int      index = *((int *) (arg));
   int      px, py, ii, dist = 0;
   float    *pix1, *pix3;
   float    cmax, F, f1, f2;
   
   briA = CMYKinputs[0];                                                   //  color brightness inputs, -1.0 to +1.0
   briR = CMYKinputs[1];
   briG = CMYKinputs[2];
   briB = CMYKinputs[3];
   
   R = briR - 0.5 * briG - 0.5 * briB + briA;                              //  red = red - green - blue + all
   G = briG - 0.5 * briR - 0.5 * briB + briA;                              //  etc.
   B = briB - 0.5 * briR - 0.5 * briG + briA;
   
   if (R < 0) R = 0.5 * R + 1;                                             //  -1 ... 0  >>  0.5 ... 1.0
   else R = R + 1;                                                         //   0 ... 1  >>  1.0 ... 2.0
   if (G < 0) G = 0.5 * G + 1;
   else G = G + 1;
   if (B < 0) B = 0.5 * B + 1;
   else B = B + 1;

   briR = R;                                                               //  final color brightness factors
   briG = G;
   briB = B;
   
   conA = CMYKinputs[4];                                                   //  contrast inputs, -1.0 to +1.0
   conR = CMYKinputs[5];
   conG = CMYKinputs[6];
   conB = CMYKinputs[7];

   if (conA < 0) conA = 0.5 * conA + 1;                                    //  -1 ... 0  >>  0.5 ... 1.0
   else conA = conA + 1;                                                   //   0 ... 1  >>  1.0 ... 2.0
   if (conR < 0) conR = 0.5 * conR + 1;
   else conR = conR + 1;
   if (conG < 0) conG = 0.5 * conG + 1;
   else conG = conG + 1;
   if (conB < 0) conB = 0.5 * conB + 1;
   else conB = conB + 1;

   conR = conR * conA;                                                     //  apply overall contrast 
   conG = conG * conA;
   conB = conB * conA;

   for (py = index; py < E3pxm->hh; py += Nwt)                             //  loop all image pixels
   for (px = 0; px < E3pxm->ww; px++)
   {
      if (sa_stat == 3) {                                                  //  select area active
         ii = py * E3pxm->ww + px;
         dist = sa_pixmap[ii];                                             //  distance from edge
         if (! dist) continue;                                             //  outside area
      }
      
      pix1 = PXMpix(E1pxm,px,py);                                          //  input pixel
      pix3 = PXMpix(E3pxm,px,py);                                          //  output pixel
      
      R1 = pix1[0];                                                        //  input RGB values, 0-255
      G1 = pix1[1];
      B1 = pix1[2];
      
      R3 = R1 * briR;                                                      //  apply color brightness factors
      G3 = G1 * briG;
      B3 = B1 * briB;
      
      R3 = conR * (R3 - 128) + 128;                                        //  apply contrast factors
      G3 = conG * (G3 - 128) + 128;
      B3 = conB * (B3 - 128) + 128;
      
      if (R3 < 0) R3 = 0;                                                  //  stop underflow
      if (G3 < 0) G3 = 0;
      if (B3 < 0) B3 = 0;

      if (R3 > 255.9 || G3 > 255.9 || B3 > 255.9) {                        //  stop overflow
         cmax = R3;
         if (G3 > cmax) cmax = G3;
         if (B3 > cmax) cmax = B3;
         F = 255.9 / cmax;
         R3 *= F;
         G3 *= F;
         B3 *= F;
      }

      if (sa_stat == 3 && dist < sa_blend) {                               //  select area is active,
         f1 = 1.0 * dist / sa_blend;                                       //    blend changes over sa_blend
         f2 = 1.0 - f1;
         R3 = f1 * R3 + f2 * R1;
         G3 = f1 * G3 + f2 * G1;
         B3 = f1 * B3 + f2 * B1;
      }
      
      pix3[0] = R3;                                                        //  output RGB values
      pix3[1] = G3;
      pix3[2] = B3;
   }

   exit_wthread();
   return 0;                                                               //  not executed, avoid gcc warning
}



