/*
 * Copyright (C) 2002  Bogdan Surdu (tim@rdsnet.ro)
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 * $Revision: 1.9 $
 *
 * $Log: main.c,v $
 * Revision 1.9  2003/06/19 17:11:20  tim
 * new memory management
 * removed the fragment list
 *
 * Revision 1.8  2003/03/20 20:27:44  tim
 * modified a bit the memory allocation
 *
 * Revision 1.7  2003/03/07 18:57:31  tim
 * Applied patch from SUBREDU Manuel <diablo@iasi.roedu.net>
 *
 * Revision 1.6  2003/03/07 18:53:14  tim
 * *** empty log message ***
 *
 * Revision 1.5  2002/12/13 19:43:35  tim
 * - added support for IP fragments
 *
 * Revision 1.4  2002/12/03 23:28:36  tim
 * added support for snmp interface id
 *
 * Revision 1.3  2002/11/28 22:53:03  tim
 * command line arguments
 *
 * Revision 1.2  2002/11/26 09:28:14  tim
 * row level mutex locking
 * cleanup in flow_scan
 *
 * Revision 1.1  2002/11/24 23:13:38  tim
 * Initial revision
 *
 *
 * TODO
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <pcap.h>
#include <errno.h>

#include <pthread.h>

#include <net/ethernet.h>
#include <netinet/ether.h>
#include <netinet/ip.h>

#include <sys/types.h>
#include <unistd.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <signal.h>

#include "fprobe.h"
#include "flow_storage.h"
#include "flow_exporter.h"
#include "list.h"

extern int optind, opterr, optopt, errno;
extern char *optarg;

/* table which holds the flows */
HASH_TABLE h;

int scan_interval = 5;    /* how often to scan the flow table */
int export_active = 60;    /* if a flow is active for more than X seconds, export it */
int export_inactive = 30;  /* export a flow X seconds after last packet */

int promisc = 1;           /* put the interface in promisc mode */

pthread_mutex_t	mutex_scan[HASH_SLOTS], memory_mutex;
struct iif_node *iif_list = NULL;

/* default log file */
char * default_logfile_name = NULL;

/* Log file fd */
int logfd = -1;

/* Daemon mode */
int daemon_mode = 0;

void pcap_callback(u_char *useless, const struct pcap_pkthdr *pkthdr, const u_char *packet) {
  struct ether_header *eptr;
  struct iphdr *ip;
  int ether_type ;
  unsigned short *data;
  struct iif_node *src_iif, *dst_iif;
  unsigned short src_iif_id, dst_iif_id;
  unsigned long now;

  eptr = (struct ether_header *) packet;
  ether_type = ntohs(eptr->ether_type);

  if (ether_type == ETHERTYPE_IP) {
    /* IP header inside ethernet frame */
    ip = (struct iphdr *)(packet + sizeof(struct ether_header));
  } else {
    return;
#ifdef	DEBUG
    printf("!IP=%d\n", ether_type);
#endif
  }

  if (ip->version != 4) return;
  if (ip->ihl < 5) return;

  data = (unsigned short *)(packet + sizeof(struct ether_header) + sizeof(struct iphdr));

  /*
   * len = ntohs(ip->tot_len);
   * printf("ETH: %s -> ", ether_ntoa((struct ether_addr*)eptr->ether_shost));
   * printf("%s  ", ether_ntoa((struct ether_addr*)eptr->ether_dhost));
   */

  /* try to match a inteface id using the ethernet address */
  src_iif = (struct iif_node *)lookup_iif_mac(iif_list, (struct ether_addr*)eptr->ether_shost);
  dst_iif = (struct iif_node *)lookup_iif_mac(iif_list, (struct ether_addr*)eptr->ether_dhost);

  src_iif_id = src_iif ? src_iif->iif_id : 255;
  dst_iif_id = dst_iif ? dst_iif->iif_id : 255;

  now = timestamp(&pkthdr->ts);

#ifdef	DDOS
  if (src_iif_id == 4) {
    dos_packet_loop(ip, pkthdr->ts);
  }
#endif
  
  flow_add_packet(h, ip, data, src_iif_id, dst_iif_id, now);

  fflush(stdout);
}

void *thread_scan(void *argument) {
  struct timespec tv_req, tv_rem;

  printf("[ThreadScan:%d] Started\n", getpid());

  tv_req.tv_sec = scan_interval;
  tv_req.tv_nsec = 0;

  while(1) {
    nanosleep(&tv_req, &tv_rem);

    flow_scan(h);
  }
}

void *thread_export() {
  struct timespec tv_req, tv_rem;

  printf("[ThreadExport:%d] Started\n", getpid());
  tv_req.tv_sec = 1;
  tv_req.tv_nsec = 0;
  while(1) {
    nanosleep(&tv_req, &tv_rem);
    flow_export_xmit();
  }
}

void load_iif(char *filename) {
  FILE *f;
  char line[128], *tok;
  int iif_id;

  f = fopen(filename,"r");
  if (!f) {
    fprintf(stderr,"Can't open %s\n", filename);
    exit(1);
  }
  iif_list = NULL;
  while(!feof(f)) {
    if (fgets(line,128,f)) {
      if (line[0] == '#') continue;
      line[127] = '\0';
      tok = strtok(line,"=");
      if (!tok) { printf("Invalid line %s\n", line); exit(1); }
      iif_id = atoi(tok);
      if (!iif_id) { printf("Invalid IIF on line %s\n", line); exit(1); }
      tok = strtok(NULL,"\n");
      iif_list = (struct iif_node *)add_iif(iif_list, iif_id, tok);
    }
  }
  fclose(f);
#ifdef	DEBUG
  dump_list(iif_list);
#endif
}

void openLogFile()
{
   if ( (logfd = open(default_logfile_name, O_WRONLY | O_CREAT | O_APPEND, S_IRUSR | S_IWUSR)) == -1 )
   {
      fprintf(stderr, "Could not create %s\n", default_logfile_name);
      return;
   }   

   /* redirecting output */
   if ( dup2(logfd, 1) == -1 )
   {
      fprintf(stderr,"Could not redirect output\n");
   }

  /* redirect stderr */
  if (dup2(logfd, 2) == -1) {
    fprintf(stderr,"Could not redirect stderr\n");
  }
}

void signal_usr1() {

  flow_stats(h);
}

void usage(char *program) {
  printf("Usage: %s -t IP:PORT [ -i interface ] [ -s scan ] [ expression ]\n", program);
  printf("  -t IP:PORT    NetFlow collector address\n");
  printf("  -i interface  interface to listen for traffic (default eth0)\n");
  printf("  -s scan       interval in seconds between two flow tables scans (Default: %d)\n", scan_interval);
  printf("  -c file       file with MAC definitions\n");
  printf("  -p            don't put the interface in promisc mode\n");
  printf("  -b            go in background (daemon mode)\n");
  printf("  -l file       log file name\n");
  printf("  expression    a bpf expresion to filter traffic (See libpcap/tcpdump)\n");
  exit(1);
}

int main(int argc, char **argv) {
  int opt, i;
  char *interface;
  char errbuf[PCAP_ERRBUF_SIZE];
  pcap_t *descr;
  bpf_u_int32 maskp;
  bpf_u_int32 netp;
  struct bpf_program fp; /* compiled filter */
  pthread_t scan_pthread, export_pthread;
  char *collector = NULL, *collector_port = NULL;
  char *filter = NULL;
  int send_port;

  interface = "eth0";

#ifdef	MEM_STATS
  init_mem_stats();
#endif
 
  /* parse arguments */
  while((opt = getopt(argc,argv,"t:i:s:c:l:pvb")) != EOF) {
    switch (opt) {
       case 't': collector = (char *)strtok(optarg,":");
                 if (!collector) usage(argv[0]);
                 collector_port = (char *)strtok(NULL,":");
                 if (!collector_port) usage(argv[0]);

                 send_port = atoi(collector_port);
                 if (!send_port) {
                   printf("Invalid collector port\n");
                   usage(argv[0]);
                 }
                 break;
       case 'i': printf("Using interface %s\n", optarg);
                 interface = optarg;
                 break;
       case 's': printf("Scan interval %s\n", optarg);
                 scan_interval = atoi(optarg);
                 break;
       case 'c': load_iif(optarg);
                 break;
       case 'p': promisc = 0;
                 break;
       case 'b':
                 daemon_mode = 1;
                 break;
       case 'l':
                 default_logfile_name = optarg;
                 break;
       default:
                 usage(argv[0]);
                 break;
    } 
  }
  if (!collector || !collector_port) usage(argv[0]);

  if ( getuid() || geteuid() )
  {
     printf("You must be root to use this\n");
     return 1;
  }

  if (default_logfile_name) 
    openLogFile();
  
  if ( daemon_mode ) {
     switch (fork()) {
        /* child */
        case 0:
           setsid();
           break;

        /* error */
        case -1:
           fprintf(stderr,"Could not fork()\n");
           return errno;

        /* parent */
        default:
           exit(0);
     }
  }
  
  printf("Will send flows to %s on UDP port %d\n", collector, send_port);

  argc-=optind;

  if (argc) {
    filter = (char *)copy_argv(&argv[optind]);
    printf("Using filter: %s\n", filter);

    pcap_lookupnet(interface, &netp, &maskp, errbuf);
  }

  descr = pcap_open_live(interface, CAPTURE_LEN, promisc, 0, errbuf);
  if (descr == NULL) {
    fprintf(stderr,"pcap_open_live(): %s\n", errbuf);
    exit(1);
  }

  if (filter) {
    if (pcap_compile(descr, &fp, filter, 0, netp) == -1) { 
      fprintf(stderr, "Error compiling filter\n");
      exit(1);
    }
    if (pcap_setfilter(descr, &fp) == -1) {
      fprintf(stderr, "Error setting filter\n");
      exit(1);
    }
  }

  /* initialize our flow-table */
  flow_create_table(h);

  /* initialize flow collector */
  flow_export_init(collector,send_port);

#ifdef	DDOS
  /* initializa DoS detection */
  dos_init();
#endif

  signal(SIGUSR1, signal_usr1);

  /* our mutex */
  for(i=0;i<HASH_SLOTS;i++) 
    pthread_mutex_init(&mutex_scan[i],NULL);

  pthread_mutex_init(&memory_mutex, NULL);

  pthread_create(&export_pthread, NULL, &thread_export, NULL); 
  pthread_create(&scan_pthread, NULL, &thread_scan, NULL); 
   
  /* loop forever */ 
  pcap_loop(descr, 0, pcap_callback, NULL);  

  /* flow_dump(h); */

  /* time to kill everything */
  flow_delete_table(h);
  pcap_close(descr);

  if (filter)
    pcap_freecode(&fp);


  close(logfd);
  
  return 0;
}
