/********************************************************************** 
 Freeciv - Copyright (C) 1996 - A Kjeldberg, L Gregersen, P Unold
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
***********************************************************************/
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#ifdef HAVE_SYS_TYPES_H
/* Under Mac OS X sys/types.h must be included before dirent.h */
#include <sys/types.h>
#endif
#include <assert.h>
#include <dirent.h>
#include <errno.h>
#include <limits.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>

#ifdef HAVE_ICONV
#include <iconv.h>
#ifdef HAVE_LIBCHARSET
#include <libcharset.h>
#else
#ifdef HAVE_LANGINFO_CODESET
#include <langinfo.h>
#endif
#endif
#endif
#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif
#ifdef HAVE_PWD_H
#include <pwd.h>
#endif

#ifdef WIN32_NATIVE
#include <windows.h>
#endif

#include "astring.h"
#include "fcintl.h"
#include "log.h"
#include "mem.h"
#include "support.h"

#include "shared.h"

#ifndef PATH_SEPARATOR
#if defined _WIN32 || defined __WIN32__ || defined __EMX__ || defined __DJGPP__
  /* Win32, OS/2, DOS */
# define PATH_SEPARATOR ";"
#else
  /* Unix */
# define PATH_SEPARATOR ":"
#endif
#endif

/* If no default data path is defined use the default default one */
#ifndef DEFAULT_DATA_PATH
#define DEFAULT_DATA_PATH "." PATH_SEPARATOR "data" PATH_SEPARATOR \
                          "~/.freeciv"
#endif

/* Cached locale numeric formatting information.
   Defaults are as appropriate for the US. */
static char *grouping = "\3";
static char *grouping_sep = ",";
static size_t grouping_sep_len = 1;

/***************************************************************
  Take a string containing multiple lines and create a copy where
  each line is padded to the length of the longest line and centered.
  We do not cope with tabs etc.  Note that we're assuming that the
  last line does _not_ end with a newline.  The caller should
  free() the result.

  FIXME: This is only used in the Xaw client, and so probably does
  not belong in common.
***************************************************************/
char *create_centered_string(const char *s)
{
  /* Points to the part of the source that we're looking at. */
  const char *cp;

  /* Points to the beginning of the line in the source that we're
   * looking at. */
  const char *cp0;

  /* Points to the result. */
  char *r;

  /* Points to the part of the result that we're filling in right
     now. */
  char *rn;

  int i;

  int maxlen = 0;
  int curlen = 0;
  int nlines = 1;

  for(cp=s; *cp != '\0'; cp++) {
    if(*cp!='\n')
      curlen++;
    else {
      if(maxlen<curlen)
	maxlen=curlen;
      curlen=0;
      nlines++;
    }
  }
  if(maxlen<curlen)
    maxlen=curlen;
  
  r=rn=fc_malloc(nlines*(maxlen+1));
  
  curlen=0;
  for(cp0=cp=s; *cp != '\0'; cp++) {
    if(*cp!='\n')
      curlen++;
    else {
      for(i=0; i<(maxlen-curlen)/2; i++)
	*rn++=' ';
      memcpy(rn, cp0, curlen);
      rn+=curlen;
      *rn++='\n';
      curlen=0;
      cp0=cp+1;
    }
  }
  for(i=0; i<(maxlen-curlen)/2; i++)
    *rn++=' ';
  strcpy(rn, cp0);

  return r;
}

/**************************************************************************
  return a char * to the parameter of the option or NULL.
  *i can be increased to get next string in the array argv[].
  It is an error for the option to exist but be an empty string.
  This doesn't use freelog() because it is used before logging is set up.
**************************************************************************/
char *get_option(const char *option_name, char **argv, int *i, int argc)
{
  int len = strlen(option_name);

  if (strcmp(option_name, argv[*i]) == 0 ||
      (strncmp(option_name, argv[*i], len) == 0 && argv[*i][len] == '=') ||
      strncmp(option_name + 1, argv[*i], 2) == 0) {
    char *opt = argv[*i] + (argv[*i][1] != '-' ? 0 : len);

    if (*opt == '=') {
      opt++;
    } else {
      if (*i < argc - 1) {
	(*i)++;
	opt = argv[*i];
	if (strlen(opt)==0) {
	  fprintf(stderr, _("Empty argument for \"%s\".\n"), option_name);
	  exit(EXIT_FAILURE);
	}
      }	else {
	fprintf(stderr, _("Missing argument for \"%s\".\n"), option_name);
	exit(EXIT_FAILURE);
      }
    }

    return opt;
  }

  return NULL;
}

/***************************************************************
...
***************************************************************/
bool is_option(const char *option_name,char *option)
{
  return (strcmp(option_name, option) == 0 ||
	  strncmp(option_name + 1, option, 2) == 0);
}

/***************************************************************
  Like strcspn but also handles quotes, i.e. *reject chars are 
  ignored if they are inside single or double quotes.
***************************************************************/
static size_t my_strcspn(const char *s, const char *reject)
{
  bool in_single_quotes = FALSE, in_double_quotes = FALSE;
  size_t i, len = strlen(s);

  for (i = 0; i < len; i++) {
    if (s[i] == '"' && !in_single_quotes) {
      in_double_quotes = !in_double_quotes;
    } else if (s[i] == '\'' && !in_double_quotes) {
      in_single_quotes = !in_single_quotes;
    }

    if (in_single_quotes || in_double_quotes) {
      continue;
    }

    if (strchr(reject, s[i])) {
      break;
    }
  }

  return i;
}

/***************************************************************
 Splits the string into tokens. The individual tokens are
 returned. The delimiterset can freely be choosen.

 i.e. "34 abc 54 87" with a delimiterset of " " will yield 
      tokens={"34", "abc", "54", "87"}
 
 Part of the input string can be quoted (single or double) to embedded
 delimiter into tokens. For example,
   command 'a name' hard "1,2,3,4,5" 99
   create 'Mack "The Knife"'
 will yield 5 and 2 tokens respectively using the delimiterset " ,".

 Tokens which aren't used aren't modified (and memory is not
 allocated). If the string would yield more tokens only the first
 num_tokens are extracted.

 The user has the responsiblity to free the memory allocated by
 **tokens.
***************************************************************/
int get_tokens(const char *str, char **tokens, size_t num_tokens,
	       const char *delimiterset)
{
  int token = 0;

  assert(str != NULL);

  for(;;) {
    size_t len, padlength = 0;

    /* skip leading delimiters */
    str += strspn(str, delimiterset);

    if (*str == '\0') {
      break;
    }

    len = my_strcspn(str, delimiterset);

    if (token >= num_tokens) {
      break;
    }

    /* strip start/end quotes if they exist */
    if ((str[0] == '"' && str[len - 1] == '"')
	|| (str[0] == '\'' && str[len - 1] == '\'')) {
      len -= 2;
      padlength = 1;		/* to set the string past the end quote */
      str++;
    }
  
    tokens[token] = fc_malloc(len + 1);
    (void) mystrlcpy(tokens[token], str, len + 1);	/* adds the '\0' */

    token++;

    str += len + padlength;
  }

  return token;
}

/***************************************************************
  Returns a statically allocated string containing a nicely-formatted
  version of the given number according to the user's locale.  (Only
  works for numbers >= zero.) The actually number used for the
  formatting is: nr*10^decade_exponent
***************************************************************/
const char *general_int_to_text(int nr, int decade_exponent)
{
  static char buf[64]; /* Note that we'll be filling this in right to left. */
  char *grp = grouping;
  char *ptr;
  int cnt;

  assert(nr >= 0);
  assert(decade_exponent >= 0);

  if (nr == 0) {
    return "0";
  }

  ptr = &buf[sizeof(buf)];
  *(--ptr) = '\0';

  cnt = 0;
  while (nr != 0 && decade_exponent >= 0) {
    int dig;

    assert(ptr > buf);

    if (decade_exponent > 0) {
      dig = 0;
      decade_exponent--;
    } else {
      dig = nr % 10;
      nr /= 10;
    }

    *(--ptr) = '0' + dig;

    cnt++;
    if (nr != 0 && cnt == *grp) {
      /* Reached count of digits in group: insert separator and reset count. */
      cnt = 0;
      if (*grp == CHAR_MAX) {
	/* This test is unlikely to be necessary since we would need at
	   least 421-bit ints to break the 127 digit barrier, but why not. */
	break;
      }
      ptr -= grouping_sep_len;
      assert(ptr >= buf);
      memcpy(ptr, grouping_sep, grouping_sep_len);
      if (*(grp + 1) != 0) {
	/* Zero means to repeat the present group-size indefinitely. */
        grp++;
      }
    }
  }

  return ptr;
}


/***************************************************************
...
***************************************************************/
const char *int_to_text(int nr)
{
  return general_int_to_text(nr, 0);
}

/***************************************************************
...
***************************************************************/
const char *population_to_text(int thousand_citizen)
{
  return general_int_to_text(thousand_citizen, 3);
}

/***************************************************************
  Check whether or not the given char is a valid,
  printable ISO 8859-1 character.
***************************************************************/
static bool is_iso_latin1(char ch)
{
   int i=ch;
   
  /* this works with both signed and unsignd char */
  if (i>=0) {
    if (ch < ' ')  return FALSE;
    if (ch <= '~')  return TRUE;
  }
  if (ch < '')  return FALSE; /* FIXME: Is it really a good idea to
				 use 8 bit characters in source code? */
  return TRUE;
}

/***************************************************************
  This is used in sundry places to make sure that names of cities,
  players etc. do not contain yucky characters of various sorts.
  Returns the input argument if it points to an acceptable name,
  otherwise returns NULL.
  FIXME:  Not internationalised.
***************************************************************/
const char *get_sane_name(const char *name)
{
  const char *cp;

  /* must not be NULL or empty */
  if (!name || *name == '\0') {
    return NULL; 
  }

  /* must begin and end with some non-space character */
  if ((*name == ' ') || (*(strchr(name, '\0') - 1) == ' ')) {
    return NULL; 
  }

  /* must be composed entirely of printable ISO 8859-1 characters,
   * and no illegal characters which can break ranking scripts */
  for (cp = name; is_iso_latin1(*cp) && *cp != '|' && *cp != '%' 
       && *cp != '"' && *cp != ','; cp++) {
    /* nothing */
  }
  if (*cp != '\0') {
    return NULL; 
  }

  /* otherwise, it's okay... */
  return name;
}

/***************************************************************
  Produce a statically allocated textual representation of the given
  year.
***************************************************************/
const char *textyear(int year)
{
  static char y[32];
  if (year<0) 
    my_snprintf(y, sizeof(y), _("%d BC"), -year);
  else
    my_snprintf(y, sizeof(y), _("%d AD"), year);
  return y;
}

/**************************************************************************
  Compares two strings, in the collating order of the current locale,
  given pointers to the two strings (i.e., given "char *"s).
  Case-sensitive.  Designed to be called from qsort().
**************************************************************************/
int compare_strings(const void *first, const void *second)
{
#if defined(ENABLE_NLS) && defined(HAVE_STRCOLL)
  return strcoll((const char *)first, (const char *)second);
#else
  return strcmp((const char *)first, (const char *)second);
#endif
}

/**************************************************************************
  Compares two strings, in the collating order of the current locale,
  given pointers to the two string pointers (i.e., given "char **"s).
  Case-sensitive.  Designed to be called from qsort().
**************************************************************************/
int compare_strings_ptrs(const void *first, const void *second)
{
#if defined(ENABLE_NLS) && defined(HAVE_STRCOLL)
  return strcoll(*((const char **)first), *((const char **)second));
#else
  return strcmp(*((const char **)first), *((const char **)second));
#endif
}

/***************************************************************************
  Returns 's' incremented to first non-space character.
***************************************************************************/
char *skip_leading_spaces(char *s)
{
  assert(s!=NULL);
  while(*s != '\0' && my_isspace(*s)) {
    s++;
  }
  return s;
}

/***************************************************************************
  Removes leading spaces in string pointed to by 's'.
  Note 's' must point to writeable memory!
***************************************************************************/
void remove_leading_spaces(char *s)
{
  char *t;
  
  assert(s!=NULL);
  t = skip_leading_spaces(s);
  if (t != s) {
    while (*t != '\0') {
      *s++ = *t++;
    }
    *s = '\0';
  }
}

/***************************************************************************
  Terminates string pointed to by 's' to remove traling spaces;
  Note 's' must point to writeable memory!
***************************************************************************/
void remove_trailing_spaces(char *s)
{
  char *t;
  size_t len;
  
  assert(s!=NULL);
  len = strlen(s);
  if (len > 0) {
    t = s + len -1;
    while(my_isspace(*t)) {
      *t = '\0';
      if (t == s) {
	break;
      }
      t--;
    }
  }
}

/***************************************************************************
  Removes leading and trailing spaces in string pointed to by 's'.
  Note 's' must point to writeable memory!
***************************************************************************/
void remove_leading_trailing_spaces(char *s)
{
  remove_leading_spaces(s);
  remove_trailing_spaces(s);
}

/***************************************************************************
  As remove_trailing_spaces(), for specified char.
***************************************************************************/
void remove_trailing_char(char *s, char trailing)
{
  char *t;
  
  assert(s!=NULL);
  t = s + strlen(s) -1;
  while(t>=s && (*t) == trailing) {
    *t = '\0';
    t--;
  }
}

/***************************************************************************
  Change spaces in s into newlines, so as to keep lines length len
  or shorter.  That is, modifies s.
  Returns number of lines in modified s.
***************************************************************************/
int wordwrap_string(char *s, int len)
{
  int num_lines = 0;
  int slen = strlen(s);
  
  /* At top of this loop, s points to the rest of string,
   * either at start or after inserted newline: */
 top:
  if (s && *s != '\0' && slen > len) {
    char *c;

    num_lines++;
    
    /* check if there is already a newline: */
    for(c=s; c<s+len; c++) {
      if (*c == '\n') {
	slen -= c+1 - s;
	s = c+1;
	goto top;
      }
    }
    
    /* find space and break: */
    for(c=s+len; c>s; c--) {
      if (my_isspace(*c)) {
	*c = '\n';
	slen -= c+1 - s;
	s = c+1;
	goto top;
      }
    }

    /* couldn't find a good break; settle for a bad one... */
    for (c = s + len + 1; *c != '\0'; c++) {
      if (my_isspace(*c)) {
	*c = '\n';
	slen -= c+1 - s;
	s = c+1;
	goto top;
      }
    }
  }
  return num_lines;
}

/***************************************************************************
  Returns pointer to '\0' at end of string 'str', and decrements
  *nleft by the length of 'str'.  This is intended to be useful to
  allow strcat-ing without traversing the whole string each time,
  while still keeping track of the buffer length.
  Eg:
     char buf[128];
     int n = sizeof(buf);
     char *p = buf;

     my_snprintf(p, n, "foo%p", p);
     p = end_of_strn(p, &n);
     mystrlcpy(p, "yyy", n);
***************************************************************************/
char *end_of_strn(char *str, int *nleft)
{
  int len = strlen(str);
  *nleft -= len;
  assert((*nleft)>0);		/* space for the terminating nul */
  return str + len;
}

/********************************************************************** 
  Check the length of the given string.  If the string is too long,
  log errmsg, which should be a string in printf-format taking up to
  two arguments: the string and the length.
**********************************************************************/ 
bool check_strlen(const char *str, size_t len, const char *errmsg)
{
  if (strlen(str) >= len) {
    freelog(LOG_ERROR, errmsg, str, len);
    return TRUE;
  }
  return FALSE;
}

/********************************************************************** 
  Call check_strlen() on str and then strlcpy() it into buffer.
**********************************************************************/
size_t loud_strlcpy(char *buffer, const char *str, size_t len,
		   const char *errmsg)
{
  (void) check_strlen(str, len, errmsg);
  return mystrlcpy(buffer, str, len);
}

/********************************************************************** 
 cat_snprintf is like a combination of my_snprintf and mystrlcat;
 it does snprintf to the end of an existing string.
 
 Like mystrlcat, n is the total length available for str, including
 existing contents and trailing nul.  If there is no extra room
 available in str, does not change the string. 

 Also like mystrlcat, returns the final length that str would have
 had without truncation.  Ie, if return is >= n, truncation occured.
**********************************************************************/ 
int cat_snprintf(char *str, size_t n, const char *format, ...)
{
  size_t len;
  int ret;
  va_list ap;

  assert(format != NULL);
  assert(str != NULL);
  assert(n>0);
  
  len = strlen(str);
  assert(len < n);
  
  va_start(ap, format);
  ret = my_vsnprintf(str+len, n-len, format, ap);
  va_end(ap);
  return (int) (ret + len);
}

/***************************************************************************
  Returns string which gives users home dir, as specified by $HOME.
  Gets value once, and then caches result.
  If $HOME is not set, give a log message and returns NULL.
  Note the caller should not mess with the returned string.
***************************************************************************/
char *user_home_dir(void)
{
  static bool init = FALSE;
  static char *home_dir = NULL;
  
  if (!init) {
    char *env = getenv("HOME");
    if (env) {
      home_dir = mystrdup(env);	        /* never free()d */
      freelog(LOG_VERBOSE, "HOME is %s", home_dir);
    } else {
#ifdef WIN32_NATIVE
      home_dir=fc_malloc(PATH_MAX);
      if (!getcwd(home_dir,PATH_MAX)) {
	free(home_dir);
	home_dir=NULL;
	freelog(LOG_ERROR, "Could not find home directory (HOME is not set)");
      }
#else
      freelog(LOG_ERROR, "Could not find home directory (HOME is not set)");
      home_dir = NULL;
#endif
    }
    init = TRUE;
  }
  return home_dir;
}

/***************************************************************************
  Returns string which gives user's username, as specified by $USER or
  as given in password file for this user's uid, or a made up name if
  we can't get either of the above. 
  Gets value once, and then caches result.
  Note the caller should not mess with returned string.
***************************************************************************/
char *user_username(void)
{
  static char *username = NULL;	  /* allocated once, never free()d */

  if (username)
    return username;

  {
    char *env = getenv("USER");
    if (env) {
      username = mystrdup(env);
      freelog(LOG_VERBOSE, "USER username is %s", username);
      return username;
    }
  }
#ifdef HAVE_GETPWUID
  {
    struct passwd *pwent = getpwuid(getuid());
    if (pwent) {
      username = mystrdup(pwent->pw_name);
      freelog(LOG_VERBOSE, "getpwuid username is %s", username);
      return username;
    }
  }
#endif
  username = fc_malloc(MAX_LEN_NAME);

#ifdef ALWAYS_ROOT
  my_snprintf(username, MAX_LEN_NAME, "name");
#else
  my_snprintf(username, MAX_LEN_NAME, "name%d", (int)getuid());
#endif
  freelog(LOG_VERBOSE, "fake username is %s", username);
  return username;
}

/***************************************************************************
  Returns a list of data directory paths, in the order in which they should
  be searched.  These paths are specified internally or may be set as the
  environment variable $FREECIV_PATH (a separated list of directories,
  where the separator itself is specified internally, platform-dependent).
  '~' at the start of a component (provided followed by '/' or '\0') is
  expanded as $HOME.

  The returned value is a static NULL-terminated list of strings.

  num_dirs, if not NULL, will be set to the number of entries in the list.
***************************************************************************/
static const char **get_data_dirs(int *num_dirs)
{
  char *path, *path2, *tok;
  static int num = 0;
  static const char **dirs = NULL;

  /* The first time this function is called it will search and
   * allocate the directory listing.  Subsequently we will already
   * know the list and can just return it. */
  if (dirs) {
    if (num_dirs) {
      *num_dirs = num;
    }
    return dirs;
  }

  path = getenv("FREECIV_PATH");
  if (!path) {
    path = DEFAULT_DATA_PATH;
  } else if (*path == '\0') {
    freelog(LOG_ERROR, _("FREECIV_PATH is set but empty; "
			 "using default path instead."));
    path = DEFAULT_DATA_PATH;
  }
  assert(path != NULL);
  
  path2 = mystrdup(path);	/* something we can strtok */
    
  tok = strtok(path2, PATH_SEPARATOR);
  do {
    int i;			/* strlen(tok), or -1 as flag */

    tok = skip_leading_spaces(tok);
    remove_trailing_spaces(tok);
    if (strcmp(tok, "/") != 0) {
      remove_trailing_char(tok, '/');
    }
      
    i = strlen(tok);
    if (tok[0] == '~') {
      if (i > 1 && tok[1] != '/') {
	freelog(LOG_ERROR, "For \"%s\" in data path cannot expand '~'"
		" except as '~/'; ignoring", tok);
	i = 0;   /* skip this one */
      } else {
	char *home = user_home_dir();

	if (!home) {
	  freelog(LOG_VERBOSE,
		  "No HOME, skipping data path component %s", tok);
	  i = 0;
	} else {
	  int len = strlen(home) + i;	   /* +1 -1 */
	  char *tmp = fc_malloc(len);

	  my_snprintf(tmp, len, "%s%s", home, tok + 1);
	  tok = tmp;
	  i = -1;		/* flag to free tok below */
	}
      }
    }

    if (i != 0) {
      /* We could check whether the directory exists and
       * is readable etc?  Don't currently. */
      num++;
      dirs = fc_realloc(dirs, num * sizeof(char*));
      dirs[num - 1] = mystrdup(tok);
      freelog(LOG_VERBOSE, "Data path component (%d): %s", num - 1, tok);
      if (i == -1) {
	free(tok);
	tok = NULL;
      }
    }

    tok = strtok(NULL, PATH_SEPARATOR);
  } while(tok);

  /* NULL-terminate the list. */
  dirs = fc_realloc(dirs, (num + 1) * sizeof(char*));
  dirs[num] = NULL;

  free(path2);
  
  if (num_dirs) {
    *num_dirs = num;
  }
  return dirs;
}

/***************************************************************************
  Returns a NULL-terminated list of filenames in the data directories
  matching the given suffix.

  The list is allocated when the function is called; it should either
  be stored permanently or de-allocated (by free'ing each element and
  the whole list).

  The suffixes are removed from the filenames before the list is
  returned.
***************************************************************************/
const char **datafilelist(const char* suffix)
{
  const char **dirs = get_data_dirs(NULL);
  char **file_list = NULL;
  int num_matches = 0;
  int list_size = 0;
  int dir_num, i, j;
  size_t suffix_len = strlen(suffix);

  assert(!strchr(suffix, '/'));

  /* First assemble a full list of names. */
  for (dir_num = 0; dirs[dir_num]; dir_num++) {
    DIR* dir;
    struct dirent *entry;

    /* Open the directory for reading. */
    dir = opendir(dirs[dir_num]);
    if (!dir) {
      if (errno == ENOENT) {
	freelog(LOG_VERBOSE, "Skipping non-existing data directory %s.",
		dirs[dir_num]);
      } else {
	freelog(LOG_ERROR, _("Could not read data directory %s: %s."),
		dirs[dir_num], strerror(errno));
      }
      continue;
    }

    /* Scan all entries in the directory. */
    while ((entry = readdir(dir))) {
      size_t len = strlen(entry->d_name);

      /* Make sure the file name matches. */
      if (len > suffix_len
	  && strcmp(suffix, entry->d_name + len - suffix_len) == 0) {
	/* Strdup the entry so we can safely write to it. */
	char *match = mystrdup(entry->d_name);

	/* Make sure the list is big enough; grow exponentially to keep
	   constant ammortized overhead. */
	if (num_matches >= list_size) {
	  list_size = list_size > 0 ? list_size * 2 : 10;
	  file_list = fc_realloc(file_list, list_size * sizeof(*file_list));
	}

	/* Clip the suffix. */
	match[len - suffix_len] = '\0';

	file_list[num_matches++] = mystrdup(match);

	free(match);
      }
    }

    closedir(dir);
  }

  /* Sort the list. */
  qsort(file_list, num_matches, sizeof(*file_list), compare_strings_ptrs);

  /* Remove duplicates (easy since it's sorted). */
  i = j = 0;
  while (j < num_matches) {
    char *this = file_list[j];

    for (j++; j < num_matches && strcmp(this, file_list[j]) == 0; j++) {
      free(file_list[j]);
    }

    file_list[i] = this;

    i++;
  }
  num_matches = i;

  /* NULL-terminate the whole thing. */
  file_list = fc_realloc(file_list, (num_matches + 1) * sizeof(*file_list));
  file_list[num_matches] = NULL;

  return (const char **)file_list;
}

/***************************************************************************
  Returns a filename to access the specified file from a data
  directory by searching all data directories (as specified by
  get_data_dirs) for the file.

  If the specified 'filename' is NULL, the returned string contains
  the effective data path.  (But this should probably only be used for
  debug output.)
  
  Returns NULL if the specified filename cannot be found in any of the
  data directories.  (A file is considered "found" if it can be
  read-opened.)  The returned pointer points to static memory, so this
  function can only supply one filename at a time.
***************************************************************************/
char *datafilename(const char *filename)
{
  int num_dirs, i;
  const char **dirs = get_data_dirs(&num_dirs);
  static struct astring realfile = ASTRING_INIT;

  if (!filename) {
    size_t len = 1;		/* in case num_dirs==0 */
    size_t seplen = strlen(PATH_SEPARATOR);

    for (i = 0; i < num_dirs; i++) {
      len += strlen(dirs[i]) + MAX(1, seplen);	/* separator or '\0' */
    }
    astr_minsize(&realfile, len);
    realfile.str[0] = '\0';

    for (i = 0; i < num_dirs; i++) {
      (void) mystrlcat(realfile.str, dirs[i], len);
      if (i < num_dirs) {
	(void) mystrlcat(realfile.str, PATH_SEPARATOR, len);
      }
    }
    return realfile.str;
  }
  
  for (i = 0; i < num_dirs; i++) {
    struct stat buf;		/* see if we can open the file or directory */
    size_t len = strlen(dirs[i]) + strlen(filename) + 2;
    
    astr_minsize(&realfile, len);
    my_snprintf(realfile.str, len, "%s/%s", dirs[i], filename);
    if (stat(realfile.str, &buf) == 0) {
      return realfile.str;
    }
  }

  freelog(LOG_VERBOSE, "Could not find readable file \"%s\" in data path.",
	  filename);

  return NULL;
}

/***************************************************************************
  As datafilename(), above, except die with an appropriate log
  message if we can't find the file in the datapath.
***************************************************************************/
char *datafilename_required(const char *filename)
{
  char *dname;
  
  assert(filename!=NULL);
  dname = datafilename(filename);

  if (dname) {
    return dname;
  } else {
    freelog(LOG_ERROR, _("The data path may be set via"
			 " the environment variable FREECIV_PATH."));
    freelog(LOG_ERROR, _("Current data path is: \"%s\""), datafilename(NULL));
    freelog(LOG_FATAL,
		 _("The \"%s\" file is required ... aborting!"), filename);
    exit(EXIT_FAILURE);
  }
}

/***************************************************************************
  Setup for Native Language Support, if configured to use it.
  (Call this only once, or it may leak memory.)
***************************************************************************/
void init_nls(void)
{
#ifdef ENABLE_NLS
#ifdef WIN32_NATIVE
  /* set LANG by hand if it is not set */
  if (!getenv("LANG")) {
    char *langname = NULL;

    switch (PRIMARYLANGID(LANGIDFROMLCID(GetUserDefaultLCID()))) {
    case LANG_SPANISH:
      langname = "es";
      break;
    case LANG_GERMAN:
      langname = "de";
      break;
    case LANG_ENGLISH:
      langname = "en";
      break;
    case LANG_FRENCH:
      langname = "fr";
      break;
    case LANG_DUTCH:
      langname = "nl";
      break;
    case LANG_POLISH:
      langname = "pl";
      break;
    case LANG_HUNGARIAN:
      langname = "hu";
      break;
    case LANG_NORWEGIAN:
      langname = "no";
      break;
    case LANG_JAPANESE:
      langname = "ja";
      break;
    case LANG_PORTUGUESE:
      langname = "pt";
      break;
    case LANG_ROMANIAN:
      langname = "ro";
      break;
    case LANG_RUSSIAN:
      langname = "ru";
      break;
    }
    if (langname) {
      static char envstr[40];

      my_snprintf(envstr, sizeof(envstr), "LANG=%s", langname);
      putenv(envstr);
    }
  }
#endif

  (void) setlocale(LC_ALL, "");
  (void) bindtextdomain(PACKAGE, LOCALEDIR);
  (void) textdomain(PACKAGE);

  /* Don't touch the defaults when LC_NUMERIC == "C".
     This is intended to cater to the common case where:
       1) The user is from North America. ;-)
       2) The user has not set the proper environment variables.
	  (Most applications are (unfortunately) US-centric
	  by default, so why bother?)
     This would result in the "C" locale being used, with grouping ""
     and thousands_sep "", where we really want "\3" and ",". */

  if (strcmp(setlocale(LC_NUMERIC, NULL), "C") != 0) {
    struct lconv *lc = localeconv();

    if (lc->grouping[0] == '\0') {
      /* This actually indicates no grouping at all. */
      static char m = CHAR_MAX;
      grouping = &m;
    } else {
      size_t len;
      for (len = 0;
	   lc->grouping[len] != '\0' && lc->grouping[len] != CHAR_MAX; len++) {
	/* nothing */
      }
      len++;
      grouping = fc_malloc(len);
      memcpy(grouping, lc->grouping, len);
    }

    grouping_sep = mystrdup(lc->thousands_sep);
    grouping_sep_len = strlen(grouping_sep);
  }
#endif
}

/***************************************************************************
  If we have root privileges, die with an error.
  (Eg, for security reasons.)
  Param argv0 should be argv[0] or similar; fallback is
  used instead if argv0 is NULL.
  But don't die on systems where the user is always root...
  (a general test for this would be better).
  Doesn't use freelog() because gets called before logging is setup.
***************************************************************************/
void dont_run_as_root(const char *argv0, const char *fallback)
{
#if (defined(ALWAYS_ROOT) || defined(__EMX__) || defined(__BEOS__))
  return;
#else
  if (getuid()==0 || geteuid()==0) {
    fprintf(stderr,
	    _("%s: Fatal error: you're trying to run me as superuser!\n"),
	    (argv0 ? argv0 : fallback ? fallback : "freeciv"));
    fprintf(stderr, _("Use a non-privileged account instead.\n"));
    exit(EXIT_FAILURE);
  }
#endif
}

/***************************************************************************
  Return a description string of the result.
  In English, form of description is suitable to substitute in, eg:
     prefix is <description>
  (N.B.: The description is always in English, but they have all been marked
   for translation.  If you want a localized version, use _() on the return.)
***************************************************************************/
const char *m_pre_description(enum m_pre_result result)
{
  static const char * const descriptions[] = {
    N_("exact match"),
    N_("only match"),
    N_("ambiguous"),
    N_("empty"),
    N_("too long"),
    N_("non-match")
  };
  assert(result >= 0 && result < ARRAY_SIZE(descriptions));
  return descriptions[result];
}

/***************************************************************************
  Given n names, with maximum length max_len_name, accessed by
  accessor_fn(0) to accessor_fn(n-1), look for matching prefix
  according to given comparison function.
  Returns type of match or fail, and for return <= M_PRE_AMBIGUOUS
  sets *ind_result with matching index (or for ambiguous, first match).
  If max_len_name==0, treat as no maximum.
***************************************************************************/
enum m_pre_result match_prefix(m_pre_accessor_fn_t accessor_fn,
			       size_t n_names,
			       size_t max_len_name,
			       m_pre_strncmp_fn_t cmp_fn,
			       const char *prefix,
			       int *ind_result)
{
  int i, len, nmatches;

  len = strlen(prefix);
  if (len == 0) {
    return M_PRE_EMPTY;
  }
  if (len > max_len_name && max_len_name > 0) {
    return M_PRE_LONG;
  }

  nmatches = 0;
  for(i=0; i<n_names; i++) {
    const char *name = accessor_fn(i);
    if (cmp_fn(name, prefix, len)==0) {
      if (strlen(name) == len) {
	*ind_result = i;
	return M_PRE_EXACT;
      }
      if (nmatches==0) {
	*ind_result = i;	/* first match */
      }
      nmatches++;
    }
  }

  if (nmatches == 1) {
    return M_PRE_ONLY;
  } else if (nmatches > 1) {
    return M_PRE_AMBIGUOUS;
  } else {
    return M_PRE_FAIL;
  }
}

/***************************************************************************
  Return the Freeciv motto.
  (The motto is common code:
   only one instance of the string in the source;
   only one time gettext needs to translate it. --jjm)
***************************************************************************/
const char *freeciv_motto(void)
{
  return _("'Cause civilization should be free!");
}

/***************************************************************************
 Return whether two vectors: vec1 and vec2 have common
 bits. I.e. (vec1 & vec2) != 0.

 Don't call this function directly, use BV_CHECK_MASK macro
 instead. Don't call this function with two different bitvectors.
***************************************************************************/
bool bv_check_mask(unsigned char *vec1, unsigned char *vec2, size_t size1,
		   size_t size2)
{
  size_t i;
  assert(size1 == size2);

  for (i = 0; i < size1; i++) {
    if ((vec1[0] & vec2[0]) != 0) {
      return TRUE;
    }
    vec1++;
    vec2++;
  }
  return FALSE;
}

#ifdef HAVE_ICONV
/***************************************************************************
  Convert the text.  This assumes 'from' is an 8-bit charset.
***************************************************************************/
static char *convert_string_malloc(const char *text,
				   const char *from, const char *to)
{
  iconv_t cd;
  size_t from_len = strlen(text) + 1, to_len = from_len;
  char *result;

  cd = iconv_open(to, from);
  if (cd == (iconv_t) (-1)) {
    freelog(LOG_ERROR,
	    _("String conversion from %s not possible.  You may\n"
	      "want to set your local encoding manually by setting\n"
	      "the environment variable $FREECIV_LOCAL_ENCODING."
	      "Proceeding anyway..."),
	    from);
    return mystrdup(text); /* The best we can do? */
  }

  do {
    size_t flen = from_len, tlen = to_len, res;
    const char *mytext = text;
    char *myresult;

    if (to_len > 1000000) {
      /* Avoid unterminated loop. */
      /* NOTE: same as error message above. */
      freelog(LOG_ERROR,
	      _("String conversion from %s not possible.  You may\n"
		"want to set your local encoding manually by setting\n"
		"the environment variable $FREECIV_LOCAL_ENCODING."
		"Proceeding anyway..."),
	      from);
      iconv_close(cd);
      return mystrdup(text);
    }

    result = fc_malloc(to_len);

    myresult = result;

    /* Since we may do multiple translations, we may need to reset iconv
     * in between. */
    iconv(cd, NULL, NULL, NULL, NULL);

    res = iconv(cd, (ICONV_CONST char **)&mytext, &flen, &myresult, &tlen);
    if (res == (size_t) (-1)) {
      if (errno != E2BIG) {
	/* Invalid input. */
	freelog(LOG_ERROR,
		_("The string '%s' is not valid: %s. Ruleset files must\n"
		  "be encoded as %s; you can change this by setting\n"
		  "$FREECIV_DATA_ENCODING."),
		text, strerror(errno), from);
	free(result);
	iconv_close(cd);
	return mystrdup(text); /* The best we can do? */
      }
    } else {
      /* Success. */
      iconv_close(cd);

      /* There may be wasted space here.  But we don't want to call
       * mystrdup on result since it might not be in an 8-bit charset. */
      return result;
    }

    /* Not enough space; try again. */
    free(result);
    to_len *= 2;
  } while (TRUE);
}
#endif

/***************************************************************************
  We convert from the charset used by the rulesets into the local encoding.
***************************************************************************/
char *convert_data_string_malloc(const char *text)
{
#ifdef HAVE_ICONV
  char *local_encoding, *data_encoding;
  char target[128];

  data_encoding = getenv("FREECIV_DATA_ENCODING");
  if (!data_encoding) {
    /* Currently the rulesets are in latin1 (ISO-8859-1). */
    data_encoding = "ISO-8859-1";
  }

  local_encoding = get_local_charset();
  my_snprintf(target, sizeof(target), "%s//TRANSLIT", local_encoding);
  local_encoding = target;

  return convert_string_malloc(text, data_encoding, local_encoding);
#else
/* Don't expect that win32 users install iconv on their own, especially
 * not from http://gnu.org/ in source form... */
#ifndef WIN32_NATIVE
  static bool only_give_this_damn_warning_once = FALSE;

  if (!only_give_this_damn_warning_once) {
    only_give_this_damn_warning_once = TRUE;

   freelog(LOG_ERROR,
           _("You are running Freeciv without using iconv.  Unless\n"
           "you are using the latin1 character set, some characters\n"
           "may not be displayed properly.  You can download iconv\n"
           "at http://gnu.org/."));
   }
#endif
  return mystrdup(text);
#endif
}

/***************************************************************************
  Return the charset encoding of the local charset.
***************************************************************************/
char *get_local_charset(void)
{
#ifdef HAVE_ICONV
  char *local_encoding;

  local_encoding = getenv("FREECIV_LOCAL_ENCODING");
  if (!local_encoding) {
#ifdef HAVE_LIBCHARSET
    local_encoding = locale_charset();
#else
#ifdef HAVE_LANGINFO_CODESET
    local_encoding = nl_langinfo(CODESET);
#else
    local_encoding = "";
#endif
#endif
    if (mystrcasecmp(local_encoding, "ANSI_X3.4-1968") == 0
	|| mystrcasecmp(local_encoding, "ASCII") == 0) {
      /* HACK: Use latin1 instead of ascii. */
      local_encoding = "ISO-8859-1";
    }
  }

  return local_encoding;
#else
  /* Assume we're using latin1. */
  return "ISO-8859-1";
#endif
}
