//  This file is part of ff3d - http://www.freefem.org/ff3d
//  Copyright (C) 2001, 2002, 2003 Stphane Del Pino

//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2, or (at your option)
//  any later version.

//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.

//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software Foundation,
//  Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  

//  $Id: Torus.hpp,v 1.3 2004/02/08 18:44:19 delpinux Exp $

#ifndef TORUS_HPP
#define TORUS_HPP

class Torus
  : public Shape
{
private:
  real_t __radius1;
  real_t __radius2;

  //! returns true if the point \a p is inside the shape.
  inline bool inShape(const TinyVector<3, real_t>& p) const
  {
    const real_t d2 = p[0]*p[0]+p[2]*p[2];
    if (d2 != 0) {
      const real_t d = std::sqrt(d2);
      const real_t r=(1.-__radius1/d);
      TinyVector<3> x (r*p[0],
		       p[1],
		       r*p[2]);
      return (Norm(x) < __radius2);
    } else {
      return (Norm(p) < __radius2);
    }
  }

  //! Prints the Torus informations using the stream \a s.
  std::ostream& put(std::ostream& os) const
  {
    os << "torus (" << __radius1 << ',' << __radius2 << ')';
    return os;
  }

public:

  //! Copies the Torus \a C.
  const Torus& operator = (const Torus& T)
  {
    __radius1 = T.__radius1;
    __radius2 = T.__radius2;
    return *this;
  }

  //! Builds a Cube based on vertices \a a and \a b
  Torus(const real_t r1, const real_t r2)
    : Shape(Shape::torus),
      __radius1(r1),
      __radius2(r2)
  {
    ;
  }

  //! Default constructor.
  Torus()
    : Shape(Shape::torus)
  {
    ;
  }

  //! Copy constructor.
  Torus(const Torus& T)
    : Shape(T),
      __radius1(T.__radius1),
      __radius2(T.__radius2)
  {
    ;
  }

  //! Destructor.
  ~Torus()
  {
    ;
  }

};

#endif // TORUS_HPP

