//  This file is part of ff3d - http://www.freefem.org/ff3d
//  Copyright (C) 2001, 2002, 2003 Stphane Del Pino

//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2, or (at your option)
//  any later version.

//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.

//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software Foundation,
//  Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  

//  $Id: VariationalOperatorExpression.hpp,v 1.4 2004/04/18 14:34:42 delpinux Exp $

#ifndef VARIATIONAL_OPERATOR_EXPRESSION_HPP
#define VARIATIONAL_OPERATOR_EXPRESSION_HPP

#include <Expression.hpp>

#include <Variable.hpp>

#include <ReferenceCounting.hpp>
#include <FunctionExpression.hpp>

#include <BoundaryExpression.hpp>

/**
 * @file   VariationalOperatorExpression.hpp
 * @class  VariationalBilinearOperator
 * @author Stephane Del Pino
 * @date   Mon May 27 17:51:28 2002
 * 
 * @brief  Variational operator description
 * 
 * This file contain "Variational operators" description. It means
 *that all PDE operator's associated weak forms can be found here. 
 */
class VariationalBilinearOperatorExpression
  : public Expression
{
public:
  enum OperatorType {
    mugradUgradV,		/**< \f$ \int \mu \nabla u \nabla v \f$ */
    alphaDxUDxV,		/**< \f$ \int \nabla u A \nabla v \f$ */
    nuDxUV,			/**< \f$ \int \nu \partial_{x_i} u v \f$ */
    nuUdxV,			/**< \f$ \int \nu u \partial_{x_i} v \f$ */
    alphaUV,			/**< \f$ \int \alpha u v \f$ */
  };

private:
  OperatorType __operatorType;	/**< operator type */

  ReferenceCounting<BoundaryExpression> __border; /**< border */

  virtual void executeRemaining() = 0;

protected:
  ReferenceCounting<FunctionVariable>
  __unknown;			/**< unknown variable */

  ReferenceCounting<TestFunctionVariable>
  __testFunction;		/**< test function */

public:

  /** 
   * Returns the border Expression
   * 
   * @return __border
   */
  ReferenceCounting<BoundaryExpression> border()
  {
    return __border;
  }

  /** 
   * Expression::execute() overloading
   * 
   */
  void execute()
  {
    if(__border != 0) {
      (*__border).execute();
    }
    executeRemaining();
  }

  /** 
   * 
   * Access to the operator type
   * 
   * @return __operatorType
   */  OperatorType operatorType()
  {
    return __operatorType;
  }

  /** 
   * 
   * Access to the unknown variable
   * 
   * @return __unknown
   */
  ReferenceCounting<FunctionVariable> unknown()
  {
    return __unknown;
  }

  /** 
   * 
   * Access to the test function
   * 
   * @return __testFunction
   */
  ReferenceCounting<TestFunctionVariable> testFunction()
  {
    return __testFunction;
  }

  /** 
   * Copy Constructor
   * 
   * @param V Variational Bilinear Operator
   */
  VariationalBilinearOperatorExpression(const VariationalBilinearOperatorExpression& V)
    : Expression(V),
      __operatorType(V.__operatorType),
      __border(V.__border),
      __unknown(V.__unknown),
      __testFunction(V.__testFunction)
  {
    ;
  }

  /** 
   * Constructor
   * 
   * @param type type of operator 
   * @param border the border where to compute this term
   * @param u unknown function
   * @param t test function
   */
  VariationalBilinearOperatorExpression(VariationalBilinearOperatorExpression::OperatorType type,
					ReferenceCounting<BoundaryExpression> border,
					ReferenceCounting<FunctionVariable> u,
					ReferenceCounting<TestFunctionVariable> t)
    : Expression(Expression::variationalBilinearOperator),
      __operatorType(type),
      __border(border),
      __unknown(u),
      __testFunction(t)
  {
    ;
  }

  /** 
   * Destructor
   * 
   */
  virtual ~VariationalBilinearOperatorExpression()
  {
    ;
  }
};

/**
 * @file   VariationalOperatorExpression.hpp
 * @class  VariationalLinearOperatorExpression
 * @author Stephane Del Pino
 * @date   Wed May 29 11:13:20 2002
 * 
 * @brief  Describes linear operators
 * 
 * This is used to describe linear operators going to the right hand
 * side of a variational formulation.
 */

class VariationalLinearOperatorExpression
  : public Expression
{
public:
  enum OperatorType {
    FV,				/**< \f$\int fv\f$ */
    FdxGV			/**< \f$ \int f\partial_{x_i} g v \f$ */
  };

private:
  OperatorType __operatorType;	/**< operator type */

  ReferenceCounting<BoundaryExpression> __border; /**< border */

  virtual void executeRemaining() = 0;

protected:
  ReferenceCounting<TestFunctionVariable>
  __testFunction;		/**< test function */

public:
  /** 
   * Returns the border Expression
   * 
   * @return __border
   */
  ReferenceCounting<BoundaryExpression> border()
  {
    return __border;
  }

  /** 
   * Expression::execute() overloading
   * 
   */
  void execute()
  {
    if(__border != 0) {
      (*__border).execute();
    }
    executeRemaining();
  }

  /** 
   * 
   * Access to the operator type
   * 
   * @return __operatorType
   */  OperatorType operatorType()
  {
    return __operatorType;
  }

  /** 
   * 
   * Access to the test function
   * 
   * @return __testFunction
   */
  ReferenceCounting<TestFunctionVariable> testFunction()
  {
    return __testFunction;
  }

  /** 
   * Copy Constructor
   * 
   * @param V Variational Linear Operator
   */
  VariationalLinearOperatorExpression(const VariationalLinearOperatorExpression& V)
    : Expression(V),
      __operatorType(V.__operatorType),
      __border(V.__border),
      __testFunction(V.__testFunction)
  {
    ;
  }

  /** 
   * Constructor
   * 
   * @param type type of operator 
   * @param t test function
   * @param border the border where the variational linear operator is defined
   */
  VariationalLinearOperatorExpression(VariationalLinearOperatorExpression::OperatorType type,
				      ReferenceCounting<BoundaryExpression> border,
				      ReferenceCounting<TestFunctionVariable> t)
    : Expression(Expression::variationalBilinearOperator),
      __operatorType(type),
      __border(border),
      __testFunction(t)
  {
    ;
  }

  /** 
   * Destructor
   * 
   */
  virtual ~VariationalLinearOperatorExpression()
  {
    ;
  }
};

/**********************
 * BILINEAR OPERATORS *
 **********************/

/**
 * @file   VariationalOperatorExpression.hpp
 * @class  VariationalMuGradUGradV
 * @author Stephane Del Pino
 * @date   Tue May 28 15:43:26 2002
 * 
 * @brief  \f$\mu\nabla u\nabla v\f$ operator
 * 
 * \f$\mu\nabla u\nabla v\f$ operator
 */
class VariationalMuGradUGradVExpression
  : public VariationalBilinearOperatorExpression
{
private:
  ReferenceCounting<FunctionExpression> __mu; /**< viscosity \f$\mu\f$*/

public:

  /** 
   * Access to the viscosity \f$\mu\f$
   * 
   * 
   * @return __mu
   */
  ReferenceCounting<FunctionExpression> mu()
  {
    return __mu;
  }

  /** 
   * VariationalBilinearOperatorExpression::executeRemaining() overloading
   * 
   */
  void executeRemaining()
  {
    (*__mu).execute();
  }

  /** 
   * put function overloading
   * 
   * @param os 
   * 
   * @return os
   */
  std::ostream& put(std::ostream& os) const
  {
    os << *__mu
       << "*grad(" << (*__unknown).name()
       << ")*grad(" << (*__testFunction).name() << ')';
    return os;
  }

  /** 
   * Constructor
   * 
   * @param mu \f$\mu\f$, the viscosity
   * @param u  unknown function
   * @param t  test function
   * @param border the border where to compute the integral
   * 
   */
  VariationalMuGradUGradVExpression(ReferenceCounting<FunctionExpression> mu,
				    ReferenceCounting<FunctionVariable> u,
				    ReferenceCounting<TestFunctionVariable> t,
				    ReferenceCounting<BoundaryExpression> border = 0)
    : VariationalBilinearOperatorExpression(VariationalBilinearOperatorExpression::
					    mugradUgradV,
					    border,
					    u, t),
      __mu(mu)
  {
    ;
  }

  /** 
   * Copy Constructor
   * 
   * @param V 
   *
   */
  VariationalMuGradUGradVExpression(const VariationalMuGradUGradVExpression& V)
    : VariationalBilinearOperatorExpression(V),
      __mu(V.__mu)
  {
    ;
  }


  /** 
   * 
   * Destructor
   * 
   */
  ~VariationalMuGradUGradVExpression()
  {
    ;
  }
};


/**
 * @file   VariationalOperatorExpression.hpp
 * @class  VariationalAlphadxUdxV
 * @author Stephane Del Pino
 * @date   Tue May 28 15:43:26 2002
 * 
 * @brief  \f$\alpha \partial{x_j} u\partial_{x_i} v\f$ operator
 * 
 * \f$\alpha \partial_{x_j} u\partial{x_i} v\f$
 */
class VariationalAlphaDxUDxVExpression
  : public VariationalBilinearOperatorExpression
{
private:
  ReferenceCounting<FunctionExpression> __alpha; /**< \f$\alpha\f$*/
  size_t __i;			/**< the \f$ i\f$ in \f$\partial_{x_i}\f$ */
  size_t __j;			/**< the \f$ j\f$ in \f$\partial_{x_j}\f$  */

  /** 
   * put function overloading
   * 
   * @param os 
   * 
   * @return os
   */
  std::ostream& put(std::ostream& os) const
  {
    os << *__alpha
       << "*dx[" << __j << ']' << (*__unknown).name()
       << "*dx[" << __i << "](" << (*__testFunction).name() << ')';
    return os;
  }

public:
  /** 
   * Access to \f$\alpha\f$
   * 
   * 
   * @return __alpha
   */
  ReferenceCounting<FunctionExpression> alpha()
  {
    return __alpha;
  }

  /** 
   * Access to __i
   * 
   * 
   * @return __i
   */
  const size_t& i() const
  {
    return __i;
  }

  /** 
   * Access to __j
   * 
   * 
   * @return __j
   */
  const size_t& j() const
  {
    return __j;
  }

  /** 
   * VariationalBilinearOperatorExpression::executeRemaining() overloading
   * 
   */
  void executeRemaining()
  {
    (*__alpha).execute();
  }

  /** 
   * Constructor of the type \f$ \alpha \partial_{x_i} u \partial_{x_j} v\f$
   * 
   * @param alpha \f$ \alpha \f$
   * @param i \f$ \partial_{x_i} \f$
   * @param j \f$ \partial_{x_j} \f$
   * @param u unknown function
   * @param v test function
   * @param border the border where to compute this term if required.
   * 
   */
  VariationalAlphaDxUDxVExpression(ReferenceCounting<FunctionExpression> alpha,
				   const size_t i,
				   const size_t j,
				   ReferenceCounting<FunctionVariable> u,
				   ReferenceCounting<TestFunctionVariable> v,
				   ReferenceCounting<BoundaryExpression> border = 0)
    : VariationalBilinearOperatorExpression(VariationalBilinearOperatorExpression::
					    alphaDxUDxV,
					    border,
					    u, v),
      __alpha(alpha),
      __i(i),
      __j(j)
  {
    ;
  }

  /** 
   * Copy Constructor
   * 
   * @param V 
   *
   */
  VariationalAlphaDxUDxVExpression(const VariationalAlphaDxUDxVExpression& V)
    : VariationalBilinearOperatorExpression(V),
      __alpha(V.__alpha),
      __i(V.__i),
      __j(V.__j)
  {
    ;
  }

  /** 
   * 
   * Destructor
   * 
   */
  ~VariationalAlphaDxUDxVExpression()
  {
    ;
  }
};



/**
 * @file   VariationalOperatorExpression.hpp
 * @class  VariationalNuUdxV
 * @author Stephane Del Pino
 * @date   Tue May 28 15:43:26 2002
 * 
 * @brief  \f$\nu u\partial_{x_i} v\f$ operator
 * 
 * \f$\nu u\partial_{x_i} v\f$
 */
class VariationalNuUdxVExpression
  : public VariationalBilinearOperatorExpression
{
private:
  ReferenceCounting<FunctionExpression> __nu; /**< \f$\nu\f$*/
  size_t __number;		/**< the \f$ i\f$ in \f$\partial_{x_i}\f$ */

  /** 
   * put function overloading
   * 
   * @param os 
   * 
   * @return os
   */
  std::ostream& put(std::ostream& os) const
  {
    os << *__nu
       << '*' << (*__unknown).name()
       << "*dx[" << __number << "](" << (*__testFunction).name() << ')';
    return os;
  }

public:
  /** 
   * Access to \f$\nu\f$
   * 
   * 
   * @return __nu
   */
  ReferenceCounting<FunctionExpression> nu()
  {
    return __nu;
  }

  /** 
   * Access to __number
   * 
   * 
   * @return __number
   */
  const size_t& number() const
  {
    return __number;
  }

  /** 
   * VariationalBilinearOperatorExpression::executeRemaining() overloading
   * 
   */
  void executeRemaining()
  {
    (*__nu).execute();
  }

  /** 
   * Constructor
   * 
   * @param nu \f$\nu\f$
   * @param u  unknown function
   * @param t  test function
   * @param border the border where to compute the integral
   * 
   */
  VariationalNuUdxVExpression(ReferenceCounting<FunctionExpression> nu,
			      const size_t number,
			      ReferenceCounting<FunctionVariable> u,
			      ReferenceCounting<TestFunctionVariable> t,
			      ReferenceCounting<BoundaryExpression> border = 0)
    : VariationalBilinearOperatorExpression(VariationalBilinearOperatorExpression::
					    nuUdxV,
					    border,
					    u, t),
      __nu(nu),
      __number(number)
  {
    ;
  }

  /** 
   * Copy Constructor
   * 
   * @param V 
   *
   */
  VariationalNuUdxVExpression(const VariationalNuUdxVExpression& V)
    : VariationalBilinearOperatorExpression(V),
      __nu(V.__nu),
      __number(V.__number)
  {
    ;
  }

  /** 
   * 
   * Destructor
   * 
   */
  ~VariationalNuUdxVExpression()
  {
    ;
  }
};

/**
 * @file   VariationalOperatorExpression.hpp
 * @class  VariationalNuDxUV
 * @author Stephane Del Pino
 * @date   Tue May 28 15:43:26 2002
 * 
 * @brief  \f$\nu u\partial_{x_i} v\f$ operator
 * 
 * \f$\nu u\partial_{x_i} v\f$
 */
class VariationalNuDxUVExpression
  : public VariationalBilinearOperatorExpression
{
private:
  ReferenceCounting<FunctionExpression> __nu; /**< \f$\nu\f$*/
  size_t __number;		/**< the \f$ i\f$ in \f$\partial_{x_i}\f$ */

  /** 
   * put function overloading
   * 
   * @param os 
   * 
   * @return os
   */
  std::ostream& put(std::ostream& os) const
  {
    os << *__nu
       << '*' << (*__unknown).name()
       << "*dx[" << __number << "](" << (*__testFunction).name() << ')';
    return os;
  }

public:
  /** 
   * Access to \f$\nu\f$
   * 
   * 
   * @return __nu
   */
  ReferenceCounting<FunctionExpression> nu()
  {
    return __nu;
  }

  /** 
   * Access to __number
   * 
   * 
   * @return __number
   */
  const size_t& number() const
  {
    return __number;
  }

  /** 
   * VariationalBilinearOperatorExpression::executeRemaining() overloading
   * 
   */
  void executeRemaining()
  {
    (*__nu).execute();
  }

  /** 
   * Constructor
   * 
   * @param nu \f$\nu\f$
   * @param u  unknown function
   * @param t  test function
   * @param border the border where to compute the integral
   * 
   */
  VariationalNuDxUVExpression(ReferenceCounting<FunctionExpression> nu,
			      const size_t number,
			      ReferenceCounting<FunctionVariable> u,
			      ReferenceCounting<TestFunctionVariable> t,
			      ReferenceCounting<BoundaryExpression> border = 0)
    : VariationalBilinearOperatorExpression(VariationalBilinearOperatorExpression::
					    nuDxUV,
					    border,
					    u, t),
      __nu(nu),
      __number(number)
  {
    ;
  }

  /** 
   * Copy Constructor
   * 
   * @param V 
   *
   */
  VariationalNuDxUVExpression(const VariationalNuDxUVExpression& V)
    : VariationalBilinearOperatorExpression(V),
      __nu(V.__nu),
      __number(V.__number)
  {
    ;
  }

  /** 
   * 
   * Destructor
   * 
   */
  ~VariationalNuDxUVExpression()
  {
    ;
  }
};


/**
 * @file   VariationalOperatorExpression.hpp
 * @class  VariationalAlphaUVExpression
 * @author Stephane Del Pino
 * @date   Tue May 28 16:36:45 2002
 * 
 * @brief   \f$\alpha u v\f$ operator
 * 
 * \f$\alpha u v\f$ operator
 */
class VariationalAlphaUVExpression
  : public VariationalBilinearOperatorExpression
{
private:
  ReferenceCounting<FunctionExpression> __alpha; /**< mass term \f$\alpha\f$*/

public:

  ReferenceCounting<FunctionExpression> alpha()
  {
    return __alpha;
  }

  /** 
   * VariationalBilinearOperatorExpression::executeRemaining() overloading
   * 
   */
  void executeRemaining()
  {
    (*__alpha).execute();
  }

  /** 
   * put function overloading
   * 
   * @param os 
   * 
   * @return os
   */
  std::ostream& put(std::ostream& os) const
  {
    os << *__alpha
       << '*' << (*__unknown).name()
       << '*' << (*__testFunction).name();
    return os;
  }

  /** 
   * Constructor
   * 
   * @param alpha \f$\alpha\f$, the mass coeffiecient
   * @param u unknown function
   * @param v test function
   * @param border the border where to compute the integral if required.
   *
   */
  VariationalAlphaUVExpression(ReferenceCounting<FunctionExpression> alpha,
			       ReferenceCounting<FunctionVariable> u,
			       ReferenceCounting<TestFunctionVariable> v,
			       ReferenceCounting<BoundaryExpression> border = 0)
    : VariationalBilinearOperatorExpression(VariationalBilinearOperatorExpression::alphaUV,
					    border,
					    u, v),
      __alpha(alpha)
  {
    ;
  }

  /** 
   * Copy Constructor
   * 
   * @param V 
   *
   */
  VariationalAlphaUVExpression(const VariationalAlphaUVExpression& V)
    : VariationalBilinearOperatorExpression(V),
      __alpha(V.__alpha)
  {
    ;
  }

  /** 
   * 
   * Destructor
   * 
   */
  ~VariationalAlphaUVExpression()
  {
    ;
  }
};


/********************
 * LINEAR OPERATORS *
 ********************/

/**
 * @file   VariationalOperatorExpression.hpp
 * @class  VariationalFV
 * @author Stephane Del Pino
 * @date   Wed May 29 11:19:01 2002
 * 
 * @brief  describes \f$\int fv\f$
 * 
 * used to describe \f$\int fv\f$
 */
class VariationalFVExpression
  : public VariationalLinearOperatorExpression
{
private:
  ReferenceCounting<FunctionExpression> __f; /**< \f$ f\f$*/

public:

  /** 
   * 
   * Returns the \f$ f\f$ function
   * 
   * @return __f
   */
  ReferenceCounting<FunctionExpression> f()
  {
    return __f;
  }

  /** 
   * VariationalLinearOperatorExpression::executeRemaining() overloading
   * 
   */
  void executeRemaining()
  {
    (*__f).execute();
  }

  /** 
   * put function overloading
   * 
   * @param os the input (modified) stream
   * 
   * @return os
   */
  std::ostream& put(std::ostream& os) const
  {
    os << *__f
       << '*' << (*__testFunction).name();
    return os;
  }

  /** 
   * Constructor
   * 
   * @param f \f$ f\f$, the second member
   * @param t  test function
   * 
   */
  VariationalFVExpression(ReferenceCounting<FunctionExpression> f,
			  ReferenceCounting<TestFunctionVariable> t,
			  ReferenceCounting<BoundaryExpression> border = 0)
    : VariationalLinearOperatorExpression(VariationalLinearOperatorExpression::FV,
					  border, t),
      __f(f)
  {
    ;
  }

  /** 
   * Copy Constructor
   * 
   * @param V 
   *
   */
  VariationalFVExpression(const VariationalFVExpression& V)
    : VariationalLinearOperatorExpression(V),
      __f(V.__f)
  {
    ;
  }


  /** 
   * 
   * Destructor
   * 
   */
  ~VariationalFVExpression()
  {
    ;
  }
};

/**
 * @file   VariationalOperatorExpression.hpp
 * @author Stephane Del Pino
 * @date   Sun Jun 23 19:51:48 2002
 * 
 * @brief  \f$ b(v) = \int f\partial_{x_i} g v \f$
 * 
 */

class VariationalFdxGVExpression
  : public VariationalLinearOperatorExpression
{
private:
  ReferenceCounting<FunctionExpression> __f; /**< \f$ f\f$*/
  ReferenceCounting<FunctionExpression> __g; /**< \f$ g\f$*/
  const size_t __number;	/**< \f$ i \f$ in \f$\partial_{x_i}\f$ */
public:
  /** 
   * Access to __number
   * 
   * @return __number
   */
  const size_t& number() const
  {
    return __number;
  }

  /** 
   * 
   * Returns the \f$ f\f$ function
   * 
   * @return __f
   */
  ReferenceCounting<FunctionExpression> f()
  {
    return __f;
  }

  /** 
   * 
   * Returns the \f$ g\f$ function
   * 
   * @return __g
   */
  ReferenceCounting<FunctionExpression> g()
  {
    return __g;
  }

  /** 
   * VariationalLinearOperatorExpression::executeRemaining() overloading
   * 
   */
  void executeRemaining()
  {
    (*__f).execute();
    (*__g).execute();
  }

  /** 
   * put function overloading
   * 
   * @param os the input (modified) stream
   * 
   * @return os
   */
  std::ostream& put(std::ostream& os) const
  {
    os << *__f << "*dx["<< *__g
       << "]*" << (*__testFunction).name();
    return os;
  }

  /** 
   * Constructor
   * 
   * @param g \f$ g \f$
   * @param f \f$ f\f$
   * @param t  test function
   * 
   */
  VariationalFdxGVExpression(ReferenceCounting<FunctionExpression> f,
			     ReferenceCounting<FunctionExpression> g,
			     ReferenceCounting<TestFunctionVariable> t,
			     const size_t& n,
			     ReferenceCounting<BoundaryExpression> border = 0)
    : VariationalLinearOperatorExpression(VariationalLinearOperatorExpression::FdxGV,
					  border, t),
      __f(f),
      __g(g),
      __number(n)
  {
    ;
  }

  /** 
   * Copy Constructor
   * 
   * @param V 
   *
   */
  VariationalFdxGVExpression(const VariationalFdxGVExpression& V)
    : VariationalLinearOperatorExpression(V),
      __f(V.__f),
      __g(V.__g),
      __number(V.__number)
  {
    ;
  }


  /** 
   * 
   * Destructor
   * 
   */
  ~VariationalFdxGVExpression()
  {
    ;
  }
};

#endif // VARIATIONAL_OPERATOR_EXPRESSION_HPP

