//  This file is part of ff3d - http://www.freefem.org/ff3d
//  Copyright (C) 2001, 2002, 2003 Stphane Del Pino

//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2, or (at your option)
//  any later version.

//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.

//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software Foundation,
//  Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  

//  $Id: MassOperator.hpp,v 1.3 2003/09/20 21:31:09 delpinux Exp $


#ifndef _MASSOPERATOR_
#define _MASSOPERATOR_

#include <PDEOperator.hpp>
/*!
  \class MassOperator

  The class MassOperator describes partial differencial operators of order
  0: \f$ \alpha\cdot \f$ where \f$ \alpha \f$ where 
  \f$
  \begin{array}{ll}
  \alpha: &\Omega \to R   \\
       &x \mapsto \mu(x)
  \end{array}
  \f$.
  
  \par example:
  if \f$ \alpha = 1 \f$, the operator is the identity.

  \author Stphane Del Pino
 */
class MassOperator
  : public PDEOperator
{
private:
  //! The function \f$ \alpha \$f.
  ReferenceCounting<UserFunction> __alpha;

public:

  const UserFunction& coefficient(const size_t i)
  {
    assert (i<1);
    return (*__alpha);
  }

  std::string TypeName() const
  {
    return std::string("MassOperator");
  }

  //! Access to the function \f$ \alpha \f$.
  ReferenceCounting<UserFunction> alpha()
  {
    return __alpha;
  }

  //! Read-only access to the function \f$ \alpha \f$.
  const UserFunction& alpha() const
  {
    return *__alpha;
  }

  //! Returns a pointer on the opposed MassOperator operator.
  ReferenceCounting<PDEOperator> operator * (ConstReferenceCounting<UserFunction> u) const
  {
    ReferenceCounting<UserFunction> alpha
      = new MulUserFunction(__alpha, u);
    return (new MassOperator(alpha));
  }

  //! Returns a pointer on the opposed MassOperator operator.
  ReferenceCounting<PDEOperator> operator - () const
  {
    ReferenceCounting<UserFunction> alpha
      = new UnaryMinusUserFunction(__alpha);
    return (new MassOperator(alpha));
  }

  // Constructs the operator using a \a Mesh and a function \a initAlpha.
  MassOperator(ReferenceCounting<UserFunction> initAlpha)
    : PDEOperator(PDEOperator::massop,
		  1),
      __alpha(initAlpha)
  {
    ;
  }
  
  //! Copy constructor.
  MassOperator(const MassOperator& M)
    : PDEOperator(M),
      __alpha(M.__alpha)
  {
    ;
  }

  //! Destrutor.
  ~MassOperator()
  {
    ;
  }
};

#endif // _MASSOPERATOR_

