//  This file is part of ff3d - http://www.freefem.org/ff3d
//  Copyright (C) 2001, 2002, 2003 Stphane Del Pino

//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2, or (at your option)
//  any later version.

//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.

//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software Foundation,
//  Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  

//  $Id: MeshOfTetrahedra.hpp,v 1.5 2004/04/15 15:21:08 delpinux Exp $

#ifndef MESH_OF_TETRA_HPP
#define MESH_OF_TETRA_HPP

#include <Mesh.hpp>
#include <Tetrahedron.hpp>
#include <Connectivity.hpp>
#include <Octree.hpp>

#include <list>
#include <map>

#include<SurfaceMeshOfTriangles.hpp>

/**
 * @file   MeshOfTetrahedra.hpp
 * @author Stephane Del Pino
 * @date   Tue Dec 23 18:39:59 2003
 * 
 * @brief  managementof tetrahedra meshes
 * 
 */

class MeshOfTetrahedra
  : public Mesh
{
public:
  typedef Tetrahedron ElementGeometry;
  typedef MeshOfTetrahedra Transformed;

  typedef SurfaceMeshOfTriangles BorderMeshType;

  enum {
    family = Mesh::volume
  };
private:

  ReferenceCounting<Vector<Tetrahedron> > __cells; /**< Cells list */

  ReferenceCounting<Connectivity<Tetrahedron> > __connectivity; /**< Connectivity */

  ReferenceCounting<Octree<size_t> > __octree; /**< Octree refering to the cells */

  TinyVector<3,real_t> __a;	/**< bounding box lower  corner */
  TinyVector<3,real_t> __b;	/**< bounding box higher corner */

  ReferenceCounting<BorderMeshType> __surfaceMesh; /**< The various border lists*/
  /** 
   * Forbids copy contructor
   * 
   * @param m another mesh
   * 
   */

  MeshOfTetrahedra(const MeshOfTetrahedra& m)
    : Mesh(Mesh::tetrahedraMesh,0)
  {
    ;
  }

public:
  inline bool hasSurfaceMesh() const
  {
    return (__surfaceMesh != 0);
  }

  inline BorderMeshType& surfaceMesh()
  {
    return (*__surfaceMesh);
  }

  inline const BorderMeshType& surfaceMesh() const
  {
    return (*__surfaceMesh);
  }

  //! Read-only access to the number of cells.
  inline const size_t& numberOfCells() const
  {
    return (*__cells).size();
  }

  const size_t cellNumber(const Tetrahedron& h) const
  {
    return (*__cells).number(h);
  }

  typedef Mesh::T_iterator<MeshOfTetrahedra, Tetrahedron> iterator;
  typedef Mesh::T_iterator<const MeshOfTetrahedra, const Tetrahedron> const_iterator;

  void buildLocalizationTools();

  /** 
   * Find the element which contains a point \f$ P \f$
   * 
   * @param x \f$ P_1\f$
   * @param y \f$ P_2\f$ 
   * @param z  \f$ P_3\f$
   * 
   * @return an iterator on the element if found else returns the end
   * iterator
   * 
   * @bug Criteria uses an epsilon that should be relative to the
   * element size
   */
  MeshOfTetrahedra::const_iterator find(const double& x,
					const double& y,
					const double& z) const;

  //! Writes the mesh to the OpenDX Mesh format
  void plotOpenDX(std::ostream& fmesh,
		  const std::string& CR) const
  {
    fferr(0) << __FILE__ << ':' << __LINE__ << ": Not implemented\n";
    std::exit(1);
  }

  /** 
   * Access to the cell \a i of the mesh.
   * 
   * @param i the cell number
   * 
   * @return the \a i th tetrahedron
   */
  inline Tetrahedron& cell(const size_t& i)
  {
    return (*__cells)[i];
  }

  /** 
   * Read only access to the cell \a i of the mesh.
   * 
   * @param i the cell number
   * 
   * @return the \a i th tetrahedron
   */
  inline const Tetrahedron& cell(const size_t& i) const
  {
    return (*__cells)[i];
  }

  /** 
   * Read only access to the mesh connectivity
   * 
   * @return the connectivity
   */
  const Connectivity<Tetrahedron>& connectivity() const
  {
    return *__connectivity;
  }

  MeshOfTetrahedra(const size_t numberOfVertices,
		   const size_t numberOfCells)
    : Mesh(Mesh::tetrahedraMesh,
	   numberOfVertices), 
      __cells(new Vector<Tetrahedron>(numberOfCells))
  {
    ;
  }

  /** 
   * Constructs a Mesh of tetrahedra using a set of vertices, a set of
   * tetrahedra and a given border mesh. Connectivity is computed.
   * 
   * @param vertices given vertices
   * @param tetrahedra given tetrahedra
   * @param triangles a surface mesh
   */
  MeshOfTetrahedra(ReferenceCounting<VerticesSet> vertices,
		   ReferenceCounting<Vector<Tetrahedron> > tetrahedra,
		   ReferenceCounting<BorderMeshType> triangles = 0)
    : Mesh(Mesh::tetrahedraMesh,
	   vertices),
      __cells(tetrahedra),
      __surfaceMesh(triangles)
  {
    this->buildLocalizationTools();
  }

  ~MeshOfTetrahedra()
  {
    ;
  }
};

#endif // MESH_OF_TETRA_HPP
