//  This file is part of ff3d - http://www.freefem.org/ff3d
//  Copyright (C) 2001, 2002, 2003 Stphane Del Pino

//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2, or (at your option)
//  any later version.

//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.

//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software Foundation,
//  Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  

//  $Id: SurfElem.hpp,v 1.2 2003/04/26 18:45:15 delpinux Exp $

// This is the base class used to deal with Surfacic Elements

#ifndef _SURF_ELEM_
#define _SURF_ELEM_

#include <Vertex.hpp>
#include <QuadratureVertex.hpp>
class Domain;

enum ElemType {
  triangle,
  quadrangle
};

/*!
  \class SurfElem
  This class is the base class for Surfacic Elements.

  \author Stphane Del Pino
  \todo
  \li Checks implementation: \p qVertex(\p i) method should not be used.
  \li Make class more general (others quadrature vertices, ...).
  \li Move ElemType
  \li LOTS OF CLEAN-UP
*/
class SurfElem
{
protected:
  //! The geometry of elements is defined using pointers on vertices.
  Vertex** vp;

  //! The quadrature vertices of the element.
  QuadratureVertex **qp;

  //! Number of vertices defining the element
  size_t nbv;

  //! Number of quadrature vertices.
  size_t nbqv;

  //! The reference of the SurfElem.
  int ref;

  //! The type of the element.
  ElemType type;

public:

  //! Access to the pointer on the Vertex \a i.
  inline Vertex* pVertices(const size_t& i)
  { 
    assert ((i>=0)&&(i<nbv));
    return vp[i];
  }

  //! Access to the Vertex \a i.
  inline Vertex& Vertices(const size_t& i)
  {
    assert ((i>=0)&&(i<nbv));
    return *(vp[i]);
  }

  //! Read-only access to the Vertex \a i.
  inline const Vertex& Vertices(const size_t& i) const
  {
    assert ((i>=0)&&(i<nbv));
    return *(vp[i]);
  }

  //! Read-only access to the number of vertices.
  inline const size_t NbVertices() const
  {
    return nbv;
  }

//   //! Access to 
//   inline QuadratureVertex& QuadVertices(const size_t& i)
//   {
//     fferr(2) << __FILE__ << ':' << __LINE__ << ": Not implemented\n";
//     std::exit(1);
//     assert ((i>=0)&&(i<nbv));
//     return *(qp[i]);
//   }

//   //! Read-only to  quadrature vertices.
//   inline const QuadratureVertex& QuadVertices(const size_t& i) const
//   {
//     assert ((i>=0)&&(i<nbv));
//     return *(qp[i]);
//   }

//   //! Read-only access to the number of quadrature vertices.
//   inline const size_t& NbQuadVertices() const
//   {
//     return nbqv;
//   }

  //! access to the reference of the SurfElem.
  inline int& Ref()
  {
    return ref;
  }

  //! Read-only access to the type of the SurfElem.
  inline const ElemType& Type() const
  {
    return type;
  }

  //! Outputs the SurfElem \a se in the \p std::ostream \a os.
  inline friend std::ostream& operator << (std::ostream& os,
					   const SurfElem& se);

  // Virtual functions

  //! Returns to the area of the face.
  virtual const real_t Area() const = 0;

  //! Returns the normal to the face.
  virtual const TinyVector<3> Normal() const = 0;

  /*! Computes and returns the number of quadrature vertices that are inside
    \a Omega.
   */
  virtual const size_t& SetQuadratureVertices(const Domain& Omega) = 0;

  //! Virtual destructor
  virtual ~SurfElem()
  {
    if(nbv>0) {
      delete [] vp;
    }

    for (size_t i=0; i<nbqv; ++i)
      delete qp[i];
    if (nbqv>0)
      delete[] qp;
  }
};

//! Outputs the SurfElem \a se in the \p std::ostream \a os.
inline std::ostream& operator << (std::ostream& s,
				  const SurfElem& se)
{
  for (size_t i=0; i<se.nbv; i++)
    s << se.vp[i]->X() << ' '
      << se.vp[i]->Y() << ' '
      << se.vp[i]->Z() << '\n';

  s << se.vp[0]->X() << ' '
    << se.vp[0]->Y() << ' '
    << se.vp[0]->Z() << "\n\n\n"; 
  return s;
}

#endif // _SURF_ELEM_

