//  This file is part of ff3d - http://www.freefem.org/ff3d
//  Copyright (C) 2003 Pascal Hav
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2, or (at your option)
//  any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software Foundation,
//  Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
//  $Id: Thread.hpp,v 1.1 2003/10/07 19:39:04 opkod Exp $

/**
 * @file   Thread.hpp
 * @author Pascal Have
 * @date   Tue Oct  7 21:35:03 2003
 * 
 * @brief  Thread, Semaphore and Mutex classes
 * 
 * 
 */

#ifndef THREAD_HPP
#define THREAD_HPP

#include <cassert>
#include <pthread.h>
#include <semaphore.h>

//! Virtual Runnable Function
/*! Used as template independent virtual class */
class Runnable {
public:
  //! Destructor 
  virtual ~Runnable() { }
  
  //! Run function
  virtual void run() = 0;

  //! Exit function
  virtual void clean() { }
};

extern "C" {  
  //! C function called when cleaning a thread
  inline void clean_function(void * t) {
    reinterpret_cast<Runnable *>(t)->clean();
  }

  //! C function called for starting a thread class
  inline void * thread_function(void * t) {
    Runnable *f = reinterpret_cast<Runnable *>(t);
    assert(f != NULL);
    pthread_cleanup_push(clean_function,f);
    f->run(); // Appel par virtual
    pthread_cleanup_pop(0);
    delete f; // destruction de l'argument forcment dynamique
    pthread_exit(NULL);
  }
}

//! Thread Proxy class
template<typename T>
class ThreadFunction : public Runnable {
protected:
  T & _t;
public:
  ThreadFunction(T * const t) : _t(*t) { }
  virtual ~ThreadFunction() { }
  inline void run() { _t.run(); }
  inline void clean() { _t.clean(); }
};

//! Main Thread class
class Thread : public Runnable {
protected:
  pthread_t thread;
public:
  Thread() { }
  virtual ~Thread() { join(); }

  void start() {
#ifndef NDEBUG
    const int ret = 
#endif /* NDEBUG */
      pthread_create(&thread,NULL,
		     thread_function,
		     new ThreadFunction<Thread>(this));
    assert(ret == 0);
  }

  void cancel() const {
    pthread_cancel(thread);
  }

  static inline void cancellable(const bool b) {
    pthread_setcancelstate((b)?PTHREAD_CANCEL_ENABLE:PTHREAD_CANCEL_DISABLE
			   ,NULL);
  }
  
  static inline void cancellable() {
    pthread_testcancel();
  }

  inline void join() const {
    void * ret;
    pthread_join(thread,&ret);
  }

  inline unsigned getId() const {
    return thread;
  }
};

class Mutex {
protected:
  pthread_mutex_t mutex;
public:
  Mutex() { 
    pthread_mutex_init(&mutex,NULL);
  }

  ~Mutex() {
    pthread_mutex_destroy(&mutex);
  }
  inline void lock() { 
    pthread_mutex_lock(&mutex); 
  }
  inline void unlock() {
    pthread_mutex_unlock(&mutex);
  }
  inline int trylock() {
    return pthread_mutex_trylock(&mutex);
  }
};

class Semaphore {
protected:
  sem_t semaphore;
public:
  Semaphore() { 
    sem_init(&semaphore,
	     0, // Interprocessus?
	     1  // Number of threads which can access simultaneously to the protected zone
	     );
  }
  ~Semaphore() {
    sem_destroy(&semaphore);
  }
  inline void wait() {
    sem_wait(&semaphore);
  }
  inline void post() {
    sem_post(&semaphore);
  }
  inline int trywait() {
    return sem_trywait(&semaphore);
  }
};

#endif /* THREAD_HPP */
