#ifndef __FWEELIN_BROWSER_H
#define __FWEELIN_BROWSER_H

#include "fweelin_event.h"
#include "fweelin_config.h"

enum BrowserItemType {
  B_Loop,
  B_Scene,
#if USE_FLUIDSYNTH
  B_FluidPatch,
#endif
  B_Division,
  B_Last
};

class BrowserItem {
 public:
  // Initialize a browser item with name n
  BrowserItem(char *n = 0) : next(0), prev(0) {
    if (n == 0)
      name = 0;
    else {
      name = new char[strlen(n)+1];
      strcpy(name,n);
    }
  };
  virtual ~BrowserItem() {
    if (name != 0)
      delete[] name;
  };

  virtual BrowserItemType GetType() = 0;

  // Compare this item to a second item and return whether this item is
  // greater than (>0), less than (<0), or equal to (==0) second
  // Used for sorting browser items
  virtual int Compare(BrowserItem *second) { return 0; };

  char *name;
  BrowserItem *next, *prev;
};

class BrowserDivision : public BrowserItem {
 public:

  virtual BrowserItemType GetType() { return B_Division; };
};

class BrowserCallback {
 public:
  virtual void ItemBrowsed(BrowserItem *i) = 0;
  virtual void ItemSelected(BrowserItem *i) = 0;
};

class Browser : public FloDisplay, public EventListener {
 public:
  Browser(BrowserItemType btype) : 
    app(0), btype(btype), callback(0), first(0), cur(0) {
    if (btype >= B_Last || (int) btype < 0) {
      printf("BROWSER: Invalid browsetype %d!\n",btype);
      btype = (BrowserItemType) 0;
    }
  };
  virtual ~Browser();
  
  // Call after construction when we have app & callback ready
  void Setup(Fweelin *a, BrowserCallback *c);

  void MoveTo(int adjust, int jumpadjust);
  void Select();

  inline BrowserItemType GetType() { return btype; };
  virtual FloDisplayType GetFloDisplayType() { return FD_Browser; };

  inline static char *GetTypeName(BrowserItemType b) {
    switch (b) {
    case B_Loop :
      return "Loop";

    case B_Scene :
      return "Scene";

    case B_Division :
      return "Division";
      
#if USE_FLUIDSYNTH
    case B_FluidPatch :
      return "FluidSynth";
#endif
      
    default :
      return "**UNKNOWN**";  
    }
  };

  // Receive events
  virtual void ReceiveEvent(Event *ev, EventProducer *from);

  // Draw to screen
  virtual void Draw(SDL_Surface *screen);

  void ClearAllItems() {
    BrowserItem *cur = first;
    first = 0;
    while (cur != 0) {
      BrowserItem *tmp = cur->next;
      delete cur;
      cur = tmp;
    };
  };

  // Move to the beginning of the browser list
  inline void MoveToBeginning() { cur = first; };
  
  // Add divisions between browser items using the Compare function-
  // whenever two neighbouring items have difference greater than maxdelta,
  // a division is inserted. In browsing files, for example, this allows us
  // to group files that where created close to one another in time
  void AddDivisions(int maxdelta);

  // Add an item to this browser
  // (Doubley linked list add with sort)
  // Nonzero if we should sort (according to the BrowserItem::Compare() method)
  inline void AddItem(BrowserItem *nw, char sort = 0) {
    if (first == 0) {
      first = nw;
      cur = first;
      // printf("BROWSER: (id: %d) First item now: %s\n",id,first->name);
    } else {
      BrowserItem *cur = first;
      if (sort && nw->Compare(cur) < 0) {
	// Insert before first if that's where she goes
	nw->next = first;
	nw->prev = 0;
	first->prev = nw;
	first = nw;	
      } else {
	// Insert at end or in the right place if sorting
	if (sort) {
	  while (cur->next != 0 && nw->Compare(cur->next) >= 0)
	    cur = cur->next;
	} else {
	  while (cur->next != 0) 
	    cur = cur->next;
	}

	// Insert after cur
	nw->next = cur->next;
	if (cur->next != 0)
	  cur->next->prev = nw;
	nw->prev = cur;
	cur->next = nw;
      }
    }
  };
 
  Fweelin *app; 

  BrowserItemType btype;     // Browser type
  BrowserCallback *callback; // Callback to notify of browser events
  BrowserItem *first,        // First item
    *cur;                    // Current item
};

#endif
