/*  FSLView - 2D/3D Interactive Image Viewer

    Authors:    Rama Aravind Vorray
		James Saunders
		David Flitney 
		Mark Jenkinson
		Stephen Smith

    FMRIB Image Analysis Group

    Copyright (C) 2002-2005 University of Oxford  */

/*  CCOPYRIGHT */

#include "preferences.h"
#include <stdlib.h>

using namespace std;

struct Preferences::Implementation
{
  Implementation() : m_assistantpath(""), m_atlaspath(""), 
		     m_fsldir(""), m_mni("")
  {
  }
  
  string m_assistantpath;
  string m_atlaspath;
  string m_fsldir;
  string m_mni;
};

Preferences::Handle Preferences::m_instance;

Preferences::Preferences(): m_impl(new Preferences::Implementation())
{
  setPath("fmrib.ox.ac.uk", "fslview", User);
}

//! @brief Access Preferences singleton
//!
//! @return a handle to be used as a pointer to a Preferences object
Preferences::Handle Preferences::getInstance()
{
  if(!m_instance)
    m_instance = Preferences::Handle(new Preferences());

  return m_instance;
}

string Preferences::inqAtlasPath() const
{
  bool ok;
  if(m_impl->m_atlaspath == "") {
    QStringList l = readListEntry("/fsl/atlaspath", &ok);
    if(ok)
      m_impl->m_atlaspath = l.join(":").ascii();
  }

  if(m_impl->m_atlaspath == "")
    m_impl->m_atlaspath = string(getenv("FSLATLASPATH") ? 
				 getenv("FSLATLASPATH") :
				 inqFSLDir() + "/data/atlases");

  return m_impl->m_atlaspath;
}

//! @brief Prefered value of FSLATLASPATH
//!
//! Returns vector of the prefered values of FSLATLASPATH or FSLDIR/lib/atlases
//!
//! @return The locations to look for atlas data sets
vector<string> Preferences::inqAtlasPathElements() const
{
  vector<string> result;
  string delimiters(":");

  string str(inqAtlasPath());
  string::size_type lastPos = str.find_first_not_of(delimiters, 0);
  string::size_type pos = str.find_first_of(delimiters, lastPos);

  while (string::npos != pos || string::npos != lastPos)
    {
      result.push_back( str.substr(lastPos, pos - lastPos) );
      lastPos = str.find_first_not_of(delimiters, pos);
      pos = str.find_first_of(delimiters, lastPos);
    }
  
  return result;
}
   
//! @brief Prefered value of FSLDIR
//!
//! Returns the prefered value of FSLDIR
//!
//! @return The users prefered value of FSLDIR
string Preferences::inqFSLDir() const 
{
  if(m_impl->m_fsldir == "")
    m_impl->m_fsldir = readEntry("/fsl/fsldir", "").ascii();
  if(m_impl->m_fsldir == "")
    m_impl->m_fsldir = string(getenv("FSLDIR") ? getenv("FSLDIR") : "/usr/local/fsl");

  return m_impl->m_fsldir; 
}

//! @brief Prefered location of MNI152 T1 brain
//!
//! Returns the prefered location where we can find MNI152 T1 brain image
//!
//! @return The users prefered location for the MNI152 T1 brain image
string Preferences::inqMni152() const
{
  if(m_impl->m_mni == "")
    m_impl->m_mni = readEntry("/fsl/mni","").ascii();
  if(m_impl->m_mni == "")
    m_impl->m_mni = inqFSLDir() + "/data/standard/MNI152_T1_2mm_brain.nii.gz";

  return m_impl->m_mni;
}

string Preferences::inqAssistantPath() const
{
  if(m_impl->m_assistantpath == "")
    m_impl->m_assistantpath = readEntry("/qt/assistantpath","").ascii();
  if(m_impl->m_assistantpath == "")
    m_impl->m_assistantpath = string(getenv("FSLQTASSISTANTPATH") ? 
				     getenv("FSLQTASSISTANTPATH") : "");
  if(m_impl->m_assistantpath == "")
    m_impl->m_assistantpath = string(getenv("QTDIR") ? 
				     string(getenv("QTDIR")) + "/bin" : 
				     inqFSLDir() + "/bin");
  
  return m_impl->m_assistantpath;
}

void Preferences::setFSLDir(const std::string& dir)
{
  m_impl->m_fsldir = dir;
  writeEntry("/fsl/fsldir", dir);
}

void Preferences::setMni152(const std::string& filename)
{
  m_impl->m_mni = filename;
  writeEntry("/fsl/mni", filename);
}

void Preferences::setAssistantPath(const std::string& path)
{
  m_impl->m_assistantpath = path;
  writeEntry("/qt/assistantpath", path);
}

void Preferences::setAtlasPath(const std::string& path)
{
  m_impl->m_atlaspath = path;
  writeEntry("/fsl/atlaspath", QStringList::split(":", path));
}
