/* 

                          Firewall Builder

                 Copyright (C) 2000 NetCitadel, LLC

  Author:  Vadim Kurland     vadim@vk.crocodile.org

  $Id: ExecBgr-2.cc,v 1.10 2002/09/04 02:31:14 vkurland Exp $


  This program is free software which we release under the GNU General Public
  License. You may redistribute and/or modify this program under the terms
  of that license as published by the Free Software Foundation; either
  version 2 of the License, or (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
 
  To get a copy of the GNU General Public License, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/


#include <stdlib.h>
#include <stdio.h>
#include <pthread.h>
#include <unistd.h>
#include <fcntl.h>
#include <sys/types.h>
#ifndef __MINGW32__
#include <sys/wait.h>
#else
#include <process.h>
#endif
#include <errno.h>
#include <signal.h>
#include <iostream>



#include "ExecBgr.hh"


using namespace libfwbuilder;


/************************************************************************
 *
 *    popen taken directly from Richard Stevens  Advanced Pogramming
 *    In the Unix Environment
 *
 *    Standard popen is inconvenient because there is no way to find out
 *    pid of the child process
 */

/* ptr to array allocated at run-time */
static int	maxfd;	/* from our open_max(), {Prog openmax} */
static pid_t   *childpid = NULL;

#ifdef __MINGW32__

static unsigned pipeSerialNumber;

BOOL APIENTRY _createPipeEx(
    OUT LPHANDLE lpReadPipe,
    OUT LPHANDLE lpWritePipe,
    IN LPSECURITY_ATTRIBUTES lpPipeAttributes,
    IN DWORD nSize,
    DWORD dwReadMode,
    DWORD dwWriteMode
    )
{
    HANDLE ReadPipeHandle, WritePipeHandle;
    DWORD dwError;
    CHAR PipeNameBuffer[ MAX_PATH ];

    //
    // Only one valid OpenMode flag - FILE_FLAG_OVERLAPPED
    //

    if ((dwReadMode | dwWriteMode) & (~FILE_FLAG_OVERLAPPED)) {
        SetLastError(ERROR_INVALID_PARAMETER);
        return FALSE;
    }

    //
    //  Set the default timeout to 120 seconds
    //

    if (nSize == 0) {
        nSize = 4096;
        }

    sprintf( PipeNameBuffer,
             "\\\\.\\Pipe\\FwbPipeAnon.%08x.%08x",
             GetCurrentProcessId(),
             pipeSerialNumber++
           );

    ReadPipeHandle = CreateNamedPipeA(
                         PipeNameBuffer,
                         PIPE_ACCESS_INBOUND | dwReadMode,
                         PIPE_TYPE_BYTE | PIPE_WAIT,
                         1,             // Number of pipes
                         nSize,         // Out buffer size
                         nSize,         // In buffer size
                         120 * 1000,    // Timeout in ms
                         lpPipeAttributes
                         );

    if (! ReadPipeHandle) {
        return FALSE;
    }

    WritePipeHandle = CreateFileA(
                        PipeNameBuffer,
                        GENERIC_WRITE,
                        0,                         // No sharing
                        lpPipeAttributes,
                        OPEN_EXISTING,
                        FILE_ATTRIBUTE_NORMAL | dwWriteMode,
                        NULL                       // Template file
                      );

    if (INVALID_HANDLE_VALUE == WritePipeHandle) {
        dwError = GetLastError();
        CloseHandle( ReadPipeHandle );
        SetLastError(dwError);
        return FALSE;
    }

    *lpReadPipe = ReadPipeHandle;
    *lpWritePipe = WritePipeHandle;
    return( TRUE );
}

HANDLE
ForkChildProcess(           // Creates a new process
    char *cmd,              // Redirects its stdin,stdout
    PHANDLE inH,            // and stderr - returns the
    PHANDLE outH            // corresponding pipe ends.
    )

{
    SECURITY_ATTRIBUTES lsa;
    STARTUPINFO         si;
    PROCESS_INFORMATION pi;
    HANDLE ChildIn;
    HANDLE ChildOut, ChildOutDup;
    HANDLE hWriteChild;
    HANDLE hReadChild;
    BOOL Success;

    lsa.nLength=sizeof(SECURITY_ATTRIBUTES);
    lsa.lpSecurityDescriptor=NULL;
    lsa.bInheritHandle=TRUE;

    Success = CreatePipe(&ChildIn,&hWriteChild,&lsa,0) &&

              DuplicateHandle(
                  GetCurrentProcess(),
                  hWriteChild,
                  GetCurrentProcess(),
                  inH,
                  0,                       // ignored b/c SAME_ACCESS
                  FALSE,                   // not inheritable
                  DUPLICATE_SAME_ACCESS
                  );

    if (!Success) {

        throw FWException(_("Could Not Create Parent-->Child Pipe"));


    }

    //
    //Create ChildStdOut/stderr to Parent_Read pipe
    //

    Success = CreatePipe(&hReadChild,&ChildOut,&lsa,0) &&

              DuplicateHandle(
                  GetCurrentProcess(),
                  hReadChild,
                  GetCurrentProcess(),
                  outH,
                  0,                       // ignored b/c SAME_ACCESS
                  FALSE,                   // not inheritable
                  DUPLICATE_SAME_ACCESS 
                  ) &&

              DuplicateHandle(
                  GetCurrentProcess(),
                  ChildOut,
                  GetCurrentProcess(),
                  &ChildOutDup,
                  0,                       // ignored b/c SAME_ACCESS
                  TRUE,                    // inheritable
                  DUPLICATE_SAME_ACCESS
                  );

    if (!Success) {

        throw FWException(_("Could Not Create Child-->Parent Pipe"));


    }

    ZeroMemory(&si, sizeof(si));
    si.cb            = sizeof(STARTUPINFO);
    si.dwFlags       = STARTF_USESHOWWINDOW | STARTF_USESTDHANDLES;
    si.wShowWindow   = SW_HIDE;
    si.hStdInput     = ChildIn;
    si.hStdOutput    = ChildOut;
    si.hStdError     = ChildOutDup;

    //
    // Create Child Process
    //
    if ( ! CreateProcess(
               NULL,
               cmd,
               NULL,
               NULL,
               TRUE,
	       GetPriorityClass( GetCurrentProcess() ),
               NULL,
               NULL,
               &si,
               &pi)) {

        if (GetLastError()) {
		char errmsg[2048];
		strcpy(errmsg,_("Exec error ("));
		strcat(errmsg,cmd);
		strcat(errmsg,") ");
		strcat(errmsg,strerror(errno));
		strcat(errmsg,"\n");

	        fprintf(stderr,errmsg);
        } 

        throw FWException(_("Could Not Create Child Process"));


    }

    //
    // Close unneccesary Handles
    //

    CloseHandle(ChildIn);
    CloseHandle(ChildOut);
    CloseHandle(ChildOutDup);
    CloseHandle(hReadChild);
    CloseHandle(hWriteChild);
//    CloseHandle(pi.hThread);

    //pidChild = pi.dwProcessId;

    return(pi.hProcess);
}

HANDLE ExecBgr::popen(const char *cmd, 
		     const char* argv[], 
		     const char *type)  throw(FWException)
{
    int		i, pfd[2];
    pid_t	pid;
    HANDLE	fHandle;

    /* only allow "r" or "w" */
    if ((type[0] != 'r' && type[0] != 'w') || type[1] != 0) {
	errno = EINVAL;		/* required by POSIX.2 */

	throw FWException(string(_("popen error: Invalid type '"))+type[0]+"'");


	return(NULL);
    }

    if (childpid == NULL) {		/* first time through */
				/* allocate zeroed out array for child pids */
	maxfd = open_max();
	if ( (childpid = (pid_t*)calloc(maxfd, sizeof(pid_t))) == NULL) {

	    throw FWException(_("Memory allocation failure"));


	    return(NULL);
	}
    }

    string cmdline;

    if (argv)
	for(i=0;argv[i];i++)
	   cmdline += argv[i] + string(" ");

    HANDLE prHandle = ForkChildProcess((char*)cmdline.c_str(),&pIn,&pOut);
    if ( prHandle == INVALID_HANDLE_VALUE ) {

	throw FWException(string(_("Could not fork: "))+strerror(errno));


    }
    
    pgm_pid=(pid_t)prHandle;     

    /* parent */
    if (*type == 'r') {
	CloseHandle(pIn);
	fHandle = pOut;
    } else {
	CloseHandle(pOut);
	fHandle = pIn;
    }
    
    childpid[(int)fHandle] = pgm_pid;	/* remember child pid for this fd */
    return(fHandle);
}

int  ExecBgr::pclose(HANDLE fHandle)
{
  HANDLE prHandle;
  if (childpid == NULL)
    return(-1);		/* popen() has never been called */

  if ( (prHandle = (HANDLE)childpid[(int)fHandle]) == INVALID_HANDLE_VALUE)
    return(-1);		/* fHandle wasn't opened by popen() */

  childpid[(int)fHandle] = 0;
  if (!CloseHandle(fHandle))
    return(-1);

  WaitForSingleObject(prHandle,INFINITE);

  DWORD stat;
  if (!GetExitCodeProcess(prHandle,&stat) || !CloseHandle(prHandle) )
    return(-1);

  return(stat);	/* return child's termination status */
}

#else

FILE* ExecBgr::popen(const char *cmd, 
		     const char* argv[], 
		     const char *type)  throw(FWException)
{
    int		i, pfd[2];
    pid_t	pid;
    FILE	*fp;

    /* only allow "r" or "w" */
    if ((type[0] != 'r' && type[0] != 'w') || type[1] != 0) {
	errno = EINVAL;		/* required by POSIX.2 */

	throw FWException(string(_("popen error: Invalid type '"))+type[0]+"'");


	return(NULL);
    }

    if (childpid == NULL) {		/* first time through */
				/* allocate zeroed out array for child pids */
	maxfd = open_max();
	if ( (childpid = (pid_t*)calloc(maxfd, sizeof(pid_t))) == NULL) {

	    throw FWException(_("Memory allocation failure"));


	    return(NULL);
	}
    }

    if (pipe(pfd) < 0) {

	throw FWException(string(_("pipe failed: "))+strerror(errno));


	return(NULL);	/* errno set by pipe() */
    }

    if ( (pid = fork()) < 0) {

	throw FWException(string(_("Could not fork: "))+strerror(errno));


	return(NULL);	/* errno set by fork() */
    }
    else if (pid == 0) {	/* child */
	if (*type == 'r') {
	    close(pfd[0]);
	    if (pfd[1] != STDOUT_FILENO) {
		dup2(pfd[1], STDOUT_FILENO);
		close(pfd[1]);
	    }
	} else {
	    close(pfd[1]);
	    if (pfd[0] != STDIN_FILENO) {
		dup2(pfd[0], STDIN_FILENO);
		close(pfd[0]);
	    }
	}
	/* close all descriptors in childpid[] */
	for (i = 0; i < maxfd; i++)
	    if (childpid[i] > 0)	close(i);


//	cerr << _("Running '") << cmd << _("' with arguments: ") << endl;


//	for (i=0; argv[i]!=NULL; i++)
//	    cerr << i << ": '" << argv[i] << "'" << endl;

	dup2(STDOUT_FILENO, STDERR_FILENO);

	execvp(cmd, (char* const*)argv);

// if we got here there was an error

	char errmsg[2048];


	strcpy(errmsg,_("Exec error ("));


	strcat(errmsg,cmd);
	strcat(errmsg,") ");
	strcat(errmsg,strerror(errno));
	strcat(errmsg,"\n");

	write(STDOUT_FILENO, errmsg, strlen(errmsg) );

	_exit(1);
    }
    
    pgm_pid=pid;     

    /* parent */
    if (*type == 'r') {
	close(pfd[1]);
	if ( (fp = fdopen(pfd[0], type)) == NULL)      return(NULL);
    } else {
	close(pfd[0]);
	if ( (fp = fdopen(pfd[1], type)) == NULL)      return(NULL);
    }
    childpid[fileno(fp)] = pid;	/* remember child pid for this fd */
    return(fp);
}

int  ExecBgr::pclose(FILE *fp)
{
  int		fd, stat;
  pid_t	pid;

  if (childpid == NULL)
    return(-1);		/* popen() has never been called */

  fd = fileno(fp);
  if ( (pid = childpid[fd]) == 0)
    return(-1);		/* fp wasn't opened by popen() */

  childpid[fd] = 0;
  if (fclose(fp) == EOF)
    return(-1);

  while (waitpid(pid, &stat, 0) < 0)
    if (errno != EINTR)
      return(-1);	/* error other than EINTR from waitpid() */

  return(stat);	/* return child's termination status */
}

#endif

#ifdef	OPEN_MAX
static int	openmax = OPEN_MAX;
#else
static int	openmax = 0;
#endif

#ifndef __MINGW32__
#define	OPEN_MAX_GUESS	256		/* if OPEN_MAX is indeterminate */
#else
#define	OPEN_MAX_GUESS	65535  		/* if OPEN_MAX is indeterminate */
#endif
/* we're not guaranteed this is adequate */

int ExecBgr::open_max(void)
{
#ifndef __MINGW32__
  if (openmax == 0) {		/* first time through */
    errno = 0;
    if ( (openmax = sysconf(_SC_OPEN_MAX)) < 0) {
      if (errno == 0)
	openmax = OPEN_MAX_GUESS;	/* it's indeterminate */
      else

	cerr << _("sysconf error for _SC_OPEN_MAX");


    }
  }
#else
	openmax = OPEN_MAX_GUESS;	/* it's indeterminate */
#endif
  return(openmax);
}

/*******************************************************************/

