/* 

                          Firewall Builder

                 Copyright (C) 2001 NetCitadel, LLC

  Author:  Vadim Kurland     vadim@vk.crocodile.org

  $Id: InterfaceListWidget.cc,v 1.12 2003/01/19 08:43:03 vkurland Exp $

  This program is free software which we release under the GNU General Public
  License. You may redistribute and/or modify this program under the terms
  of that license as published by the Free Software Foundation; either
  version 2 of the License, or (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
 
  To get a copy of the GNU General Public License, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/


#include "InterfaceListWidget.hh"

#define COL_NAME     0
#define COL_LABEL    1
#define COL_ADDR     2
#define COL_NETMASK  3
#define COL_EXT      4
#define COL_DYN      5
#define COL_MAC      6
#define COL_NETZONE  7

using namespace libfwbuilder;
using namespace std;

InterfaceListWidget::InterfaceListWidget() : CList(8)
{
    show_addr= show_name= show_netmask= show_ext= show_dyn= show_mac= show_label=true;

    show_security_level=false;

    set_shadow_type(GTK_SHADOW_NONE);

    column(COL_NAME    ).set_title(_("Name"));
    column(COL_LABEL   ).set_title(_("Label"));
    column(COL_ADDR    ).set_title(_("Address"));
    column(COL_NETMASK ).set_title(_("Netmask"));
    column(COL_EXT     ).set_title(_("Ext"));
    column(COL_DYN     ).set_title(_("Dyn"));
    column(COL_MAC     ).set_title(_("MAC"));
    column(COL_NETZONE ).set_title("");


    column(COL_NAME    ).set_justification(GTK_JUSTIFY_LEFT);
    column(COL_LABEL   ).set_justification(GTK_JUSTIFY_LEFT);
    column(COL_ADDR    ).set_justification(GTK_JUSTIFY_LEFT);
    column(COL_NETMASK ).set_justification(GTK_JUSTIFY_LEFT);
    column(COL_EXT     ).set_justification(GTK_JUSTIFY_LEFT);
    column(COL_DYN     ).set_justification(GTK_JUSTIFY_LEFT);
    column(COL_MAC     ).set_justification(GTK_JUSTIFY_LEFT);
    column(COL_NETZONE ).set_justification(GTK_JUSTIFY_LEFT);


    column(COL_NAME    ).set_min_width(80 );
    column(COL_LABEL   ).set_min_width(80 );
    column(COL_ADDR    ).set_min_width(60 );
    column(COL_NETMASK ).set_min_width(60 );
    column(COL_EXT     ).set_min_width(20 );
    column(COL_DYN     ).set_min_width(20 );
    column(COL_MAC     ).set_min_width(60 );
    column(COL_NETZONE ).set_min_width(0  );  // invisible column
    column(COL_NETZONE ).set_max_width(2  );  // invisible column


    column(COL_NAME    ).set_auto_resize(true);
    column(COL_LABEL   ).set_auto_resize(true);
    column(COL_ADDR    ).set_auto_resize(true);
    column(COL_NETMASK ).set_auto_resize(true);
    column(COL_EXT     ).set_auto_resize(true);
    column(COL_DYN     ).set_auto_resize(true);
    column(COL_MAC     ).set_auto_resize(true);
    column(COL_NETZONE ).set_auto_resize(false);
    column(COL_NETZONE ).set_visiblity(false);  // invisible column

/*
    set_column_width(COL_NAME   , 80  );
    set_column_width(COL_ADDR   , 100 );
    set_column_width(COL_NETMASK, 100 );
    set_column_width(COL_EXT    , 20  );
    set_column_width(COL_DYN    , 20  );
    set_column_width(COL_MAC    , 100 );
    set_column_width(COL_LABEL  , 80  );
*/


    set_selection_mode(GTK_SELECTION_BROWSE);
    column_titles_show();


}

/*
 *   in gtkmm 1.2.5 it is "set_visiblity", not "set_visibility"
 */    
void InterfaceListWidget::set_show_name(bool onoff)    { show_name=onoff;    column(COL_NAME    ).set_visiblity(onoff); }
void InterfaceListWidget::set_show_label(bool onoff)   { show_label=onoff;   column(COL_LABEL   ).set_visiblity(onoff); }
void InterfaceListWidget::set_show_address(bool onoff) { show_addr=onoff;    column(COL_ADDR    ).set_visiblity(onoff); }
void InterfaceListWidget::set_show_netmask(bool onoff) { show_netmask=onoff; column(COL_NETMASK ).set_visiblity(onoff); }
void InterfaceListWidget::set_show_ext(bool onoff)     { show_ext=onoff;     column(COL_EXT     ).set_visiblity(onoff); }
void InterfaceListWidget::set_show_dyn(bool onoff)     { show_dyn=onoff;     column(COL_DYN     ).set_visiblity(onoff); }
void InterfaceListWidget::set_show_mac(bool onoff)     { show_mac=onoff;     column(COL_MAC     ).set_visiblity(onoff); }

void InterfaceListWidget::set_show_security_level(bool onoff)
{
    show_security_level=onoff;
    if (show_security_level)	column(COL_EXT     ).set_title("SecurityLvl");
    else                        column(COL_EXT     ).set_title(_("Ext"));
}


static void _iface_list_destroy_string(gpointer data)
{
    std::string *sp= (std::string*)data;
    delete sp;
}

void InterfaceListWidget::select(unsigned idx)
{
    if ( idx<rows().size())      rows()[idx].select();
}

void InterfaceListWidget::addInterface(const InterfaceData &idata)
{
    char *crow[8]={"","","","","","","",""};
            
    rows().push_back( (const char**)crow );

    gint row=rows().size()-1;

    updateInterface(row,idata);

    std::string *tmp_id=new string(idata.id);
    rows()[row].set_data( (void*)tmp_id , &_iface_list_destroy_string);
    rows()[row].select();
}

void InterfaceListWidget::updateInterface(unsigned             row,
                                          const InterfaceData &idata)
{
    if ( row<rows().size())
    {
        cell(row,COL_NAME   ).set_spacing(6);    
        cell(row,COL_NAME   ).set_text( idata.name );

        cell(row,COL_LABEL  ).set_spacing(6);
        cell(row,COL_LABEL  ).set_text( idata.label );

        cell(row,COL_ADDR   ).set_spacing(6);    
        cell(row,COL_ADDR   ).set_text( idata.address );

        cell(row,COL_NETMASK).set_spacing(6);    
        cell(row,COL_NETMASK).set_text( idata.netmask );

        cell(row,COL_EXT    ).set_spacing(6);
        if (show_security_level) {
            char str[80];
            sprintf(str,"%d",idata.securityLevel );
            cell(row,COL_EXT    ).set_text( str );
        } else
            cell(row,COL_EXT    ).set_text( (idata.securityLevel==0)?"ext":"");

        cell(row,COL_NETZONE).set_text( idata.networkZone );

        string s;
        if (idata.isDyn)        s=DYN_CODE;
        if (idata.isUnnumbered) s=UNNUM_CODE;

        cell(row,COL_DYN    ).set_spacing(6);
        cell(row,COL_DYN    ).set_text(s);

        cell(row,COL_MAC    ).set_spacing(6);
        cell(row,COL_MAC    ).set_text( idata.physicalAddress );
    }
}

void InterfaceListWidget::updateInterface(const InterfaceData &idata)
{
    updateInterface( getRowById(idata.id) , idata);
}


InterfaceData InterfaceListWidget::getInterface(const std::string &id)
{
    return getInterface( getRowById(id) );
}

InterfaceData InterfaceListWidget::getInterface(unsigned row)
{
    InterfaceData idata;
    string        s;
            
    if (row>=rows().size()) return idata;

    idata.id             =getIdByRow(row);

    idata.name           =cell(row,COL_NAME   ).get_text();
    idata.label          =cell(row,COL_LABEL  ).get_text();
    idata.address        =cell(row,COL_ADDR   ).get_text();
    idata.netmask        =cell(row,COL_NETMASK).get_text();

    if (show_security_level) {
        s                    =cell(row,COL_EXT   ).get_text();
        idata.securityLevel  =atoi(s.c_str());
    } else {
        s                    =cell(row,COL_EXT   ).get_text();
        idata.securityLevel  =(s==_("ext"))?0:100;
    }

    s=cell(row,COL_DYN   ).get_text();
    idata.isDyn       =(s==DYN_CODE);
    idata.isUnnumbered=(s==UNNUM_CODE);

    idata.physicalAddress=cell(row,COL_MAC    ).get_text();
    idata.networkZone    =cell(row,COL_NETZONE).get_text();

    return idata;
}

list<InterfaceData> InterfaceListWidget::getInterfaces()
{
    list<InterfaceData> res;
    for (int i=0; i<getSize(); i++)   res.push_back( getInterface(i) );
    return res;
}

void InterfaceListWidget::setInterfaces(list<InterfaceData> &ifaces)
{
    clear();
    for (list<InterfaceData>::iterator i=ifaces.begin(); i!=ifaces.end(); i++)
        addInterface(*i);
}

int InterfaceListWidget::getSize()
{
    return rows().size();
}

void InterfaceListWidget::removeInterface(const std::string &id)
{
    for( RowList::iterator i=rows().begin(); i!=rows().end(); ++i) {
        const string* row_id = (const string*)((*i).get_data());
        if (id == *row_id) {
            rows().remove( (*i) );
            return;
        }
    }
}

void   InterfaceListWidget::clear()
{
    rows().clear();
}

bool   InterfaceListWidget::haveInterface(const std::string &id)
{
    return (getRowById(id)!=-1);
}

int    InterfaceListWidget::getRowById(const std::string &id)
{
    int row=0;
    for( RowList::iterator i=rows().begin(); i!=rows().end(); ++row,++i) {
        const string* row_id = (const string*)((*i).get_data());
        if (id == *row_id) return row;
    }
    return -1;
}

string InterfaceListWidget::getIdByRow(unsigned row)
{
    if ( row<rows().size())
    {
        const string* id = (const string*)(get_row_data(row));
        return (id!=NULL) ? *id : "";
    }
    return "";
}


string InterfaceListWidget::getSelectedInterfaceId()
{
    if (rows().empty()) return "";

    Gtk::CList_Helpers::SelectionList sl = selection();
    if (sl.empty()) return NULL;

    Gtk::CList_Helpers::Row row = sl.front();
    int   rn = row.get_row_num();

    return getIdByRow(rn);
}

