/* 

                          Firewall Builder

                 Copyright (C) 2000 NetCitadel, LLC

  Author:  Vadim Kurland     vadim@vk.crocodile.org

  $Id: OptionsDlg.cc,v 1.59 2002/12/01 00:03:44 vkurland Exp $


  This program is free software which we release under the GNU General Public
  License. You may redistribute and/or modify this program under the terms
  of that license as published by the Free Software Foundation; either
  version 2 of the License, or (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
 
  To get a copy of the GNU General Public License, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/

#include "fwbuilder/libfwbuilder-config.h"

#include <gtk--/main.h>
#include <gdk-pixbuf/gdk-pixbuf.h>

#include "OptionsDlg.hh"

#include "helpers.hh"
#include "MessageDialog.hh"
#include "FirewallDialog.hh"
#include "FileSel.hh"
#include "SimplePasswordDialog.hh"
#include "SimpleTextDialog.hh"

#include "PixmapButton.hh"

#include "CertificateDruid.hh"

#include <assert.h>

using namespace libfwbuilder;
using namespace Gtk::CTree_Helpers;

RowList::iterator  OptionsDlg::addTreeRow(RowList &rowlist,
					  const string &lbl)
{
    RowList::iterator j;

    vector<const gchar*>      item;
    item.push_back( lbl.c_str() );
    rowlist.push_back(Element(item));
    j= --rowlist.end();

    return j;
}


RowList::iterator  OptionsDlg::addTreeRow(RowList &rowlist,
					  const string &lbl, 
					  unsigned notebook_page_number)
{
    RowList::iterator j;

    vector<const gchar*>      item;
    item.push_back( lbl.c_str() );
    rowlist.push_back(Element(item));
    j= --rowlist.end();
    /*
     *  store page name in the tree widget element
     */
    char *id_storage=cxx_strdup( lbl.c_str() );
    j->set_data( id_storage );

    notebook_pages[lbl]=notebook_page_number;

    return j;
}

/*
  Tree labels and Notebook pages
  
  Blank       0
  Paths       1
  Network     2
  GUI         3
  Tooltips    4
  Behavior    5
  Tree View   6
  
*/

OptionsDlg::OptionsDlg()
{

    commit_changes=false;


    PixmapButton::addPixmapAndText( *button12 , "Ok"     , _("OK")     );
    PixmapButton::addPixmapAndText( *button10 , "Cancel" , _("Cancel") );



    button10->set_flags(GTK_CAN_FOCUS|GTK_CAN_DEFAULT);
    button12->set_flags(GTK_CAN_FOCUS|GTK_CAN_DEFAULT);
    button12->grab_default();
    button12->grab_focus();


    label422->set_line_wrap(true);       label422->show();
    label423->set_line_wrap(true);       label423->show();
    label425->set_line_wrap(true);       label425->show();
    label426->set_line_wrap(true);       label426->show();
    label427->set_line_wrap(true);       label427->show();
    label431->set_line_wrap(true);       label431->show();

    label511->set_line_wrap(true);       label431->show();

    ctree->set_expander_style( GTK_CTREE_EXPANDER_CIRCULAR );
    ctree->set_line_style( GTK_CTREE_LINES_SOLID  );
    ctree->column(0).set_auto_resize(true);

/*
 *  Now build the tree
 */
    RowList  top_level = ctree->rows();

    RowList  subtree = (addTreeRow(top_level, _("General"), 0 ))->subtree();

    addTreeRow(subtree,_("Paths")          , 1);
    addTreeRow(top_level,_("Network")      , 2);
    subtree = (addTreeRow(top_level,_("GUI") , 3))->subtree();
    addTreeRow(subtree,_("Object Tooltips"), 4);
    addTreeRow(subtree,_("Behavior")       , 5);
    addTreeRow(subtree,_("Tree View")      , 6);

#ifdef HAVE_LIBSSL
    addTreeRow(top_level,_("Certificates")   , 7);
#endif

    notebook->set_show_tabs(false);
    notebook->set_show_border(false);

    delete_event.connect( SigC::slot(this,&OptionsDlg::on_delete_event));

    /*
     *  Paths  tab
     */
    
    wdir->set_text(
        Preferences::global_prefs->getOptStr(
                                 "/FWBuilderPreferences/Paths/Wdir") );
    wdir->grab_focus();

    /*
     *   User Interface  tab
     */

    ((Gtk::Label*)(remember_window->get_child()))->set_alignment(0.0,0.5);
    
    remember_window->set_active(
        Preferences::global_prefs->getOptBool(
            "/FWBuilderPreferences/UI/RememberWindowPositionAndSize") );

    ((Gtk::Label*)(hide_navbar->get_child()))->set_alignment(0.0,0.5);

    hide_navbar->set_active(
        Preferences::global_prefs->getOptBool(
            "/FWBuilderPreferences/UI/HideNavigationBar") );
 
    ((Gtk::Label*)(quick_view_off->get_child()))->set_alignment(0.0,0.5);
    ((Gtk::Label*)(quick_view_popup->get_child()))->set_alignment(0.0,0.5);
    ((Gtk::Label*)(quick_view_window->get_child()))->set_alignment(0.0,0.5);

    string qv=Preferences::global_prefs->getOptStr(
        "/FWBuilderPreferences/UI/ObjectQuickView");

    if ( qv=="off" )     quick_view_off->set_active(true);
    if ( qv=="popup" )   quick_view_popup->set_active(true);
    if ( qv=="window" )  quick_view_window->set_active(true);

    quick_view_timeout->set_value(
        Preferences::global_prefs->getOptInt(
            "/FWBuilderPreferences/UI/ObjectQuickViewTimeout") );

    ((Gtk::Label*)(autosave->get_child()))->set_alignment(0.0,0.5);
    ((Gtk::Label*)(autosave->get_child()))->set_line_wrap(true);

    autosave->set_active(
        Preferences::global_prefs->getOptBool(
            "/FWBuilderPreferences/UI/Autosave")  );

    ((Gtk::Label*)(obj_tree_split->get_child()))->set_alignment(0.0,0.5);
    ((Gtk::Label*)(obj_tree_combined->get_child()))->set_alignment(0.0,0.5);

    string ot=Preferences::global_prefs->getOptStr(
        "/FWBuilderPreferences/UI/ObjectTreeMode");

    if (ot=="Split")    obj_tree_split->set_active(true);
    if (ot=="Combined") obj_tree_combined->set_active(true);

    ((Gtk::Label*)(show_obj_props_in_tree->get_child()))->set_alignment(0.0,0.5);
    show_obj_props_in_tree->set_active(
        Preferences::global_prefs->getOptBool(
            "/FWBuilderPreferences/UI/ShowObjectPropertiesInTree") );

    show_icons_in_tree->set_active(
        Preferences::global_prefs->getOptBool(
            "/FWBuilderPreferences/UI/ShowIconsInTree") );

/*************** Network tab ******************/

    snmp_timeout->set_value(
        Preferences::global_prefs->getOptInt(
            "/FWBuilderPreferences/Network/SNMPTimeout") );

    snmp_retries->set_value(
        Preferences::global_prefs->getOptInt(
            "/FWBuilderPreferences/Network/SNMPRetries") );

    dns_timeout->set_value(
        Preferences::global_prefs->getOptInt(
            "/FWBuilderPreferences/Network/DNSTimeout") );

    dns_retries->set_value(
        Preferences::global_prefs->getOptInt(
            "/FWBuilderPreferences/Network/DNSRetries") );

#ifdef HAVE_LIBSSL
/************** Certificates ******************/
    fillListOfCertificates();
#endif

    show_all();

#ifdef __MINGW32__
    __fwb_win32_loadIcon("FWBuilderWin32Ico",this);
#endif
}

void _cert_list_destroy_string(gpointer data)
{
    string *sp= (string*)data;
    delete sp;
}

/*
 *  this method should fill CList widget 'cert_list' with info about
 *  installed certificates.
 *
 *  call this method after new certificate has been generated or imported
 */
void OptionsDlg::fillListOfCertificates()
{
    map<string, pair<string, string> > certs=
    Preferences::global_prefs->getCerificates();

    map<string, pair<string, string> >::iterator i;

    string icn = Resources::global_res->getIconPath("Key");

    GdkPixbuf *pb;
    GdkPixmap *pm;
    GdkBitmap *bm;

    pb=gdk_pixbuf_new_from_file( (char*)(icn.c_str() ) );
    assert (pb!=NULL);
    gdk_pixbuf_render_pixmap_and_mask(pb, &pm, &bm, 127);

    Gdk_Pixmap key_icon=Gdk_Pixmap(pm);
    Gdk_Bitmap key_mask=Gdk_Bitmap(bm);

    cert_list->clear();
    cert_list->freeze();

    for(i=certs.begin(); i!=certs.end(); i++) 
    {
	const char *crow[2]={"",""};
        string dsc = (*i).second.first;
        cert_list->append(crow);
        gint row = cert_list->rows().size()-1;
	cert_list->cell(row,0 ).set_pixmap(key_icon,key_mask);
        cert_list->cell(row,0 ).set_text( dsc );
	cert_list->set_row_data_full( row,
				      new string( (*i).first ) , 
				      &_cert_list_destroy_string );
    }

    gdk_pixbuf_unref(pb);

    cert_list->column(0).set_justification(GTK_JUSTIFY_LEFT);

    cert_list->thaw();
    cert_list->set_selection_mode(GTK_SELECTION_SINGLE);
}

void OptionsDlg::on_generate_cert_clicked()
{
    string icn=Resources::global_res->getIconPath("CertDruidLogo");

    CertificateDruid *cd=new CertificateDruid(_("Generating New Certificate"),icn);

    cd->run();

    fillListOfCertificates();
}

void OptionsDlg::on_import_cert_clicked()
{
    FileSel *fs=new FileSel(_("Import certificate from PKCS#12 file"),
			    Preferences::global_prefs->getWdir(),
			    "");
    string df=fs->run();
    delete fs;
    if(df.empty()) return; // cancel

    string descr;
    do
    {
        SimpleTextDialog *td = new SimpleTextDialog(_("Please enter certificate name (it will be shown in the list)"));
        if(!td->run())
        {
            // cancel
            delete td;
            return;
        }
        descr = td->getText();
        delete td;
    } while(descr.empty());
    
    try
    {
        Preferences::global_prefs->importPKCS12(df, descr);
    } catch(const FWException &ex)
    {
        MessageDialog::Error(string(_("Error importing certificate and key in PKCS#12 format : "))+ex.toString(),this);
    }

    fillListOfCertificates();
}

void OptionsDlg::on_export_cert_clicked()
{
    Gtk::CList_Helpers::SelectionList sl=cert_list->selection();
    if (sl.empty()) return;
    gpointer p=  sl.front().get_data();
    assert (p!=NULL);
    string *id= (string*)p;
    

    FileSel *fs=new FileSel(_("Export certificate to PKCS#12 file"),


			    Preferences::global_prefs->getWdir(),
			    "");
    string df=fs->run();
    delete fs;
    if(df.empty()) return; // cancel

    try
    {
        Preferences::global_prefs->exportPKCS12(*id, df);
    } catch(const FWException &ex)
    {
        MessageDialog::Error(string(_("Error exporting certificate and key in PKCS#12 format : "))+ex.toString(),this);
    }
}

void OptionsDlg::on_remove_cert_clicked()
{
    Gtk::CList_Helpers::SelectionList sl=cert_list->selection();
    if (sl.empty()) return;
    gpointer p=  sl.front().get_data();
    assert (p!=NULL);
    string *id= (string*)p;
    string  desc=sl.front()[0].get_text();

    MessageDialog::DlgReturnValue action;

    action=MessageDialog::Question(_("This will permanently remove certificate '")+
				   desc+_("'\nAre you sure?"),this);

    if(action== MessageDialog::NO || action== MessageDialog::CANCEL)  return;

    try
    {
        Preferences::global_prefs->removeKeyAndCert(*id);
        Preferences::global_prefs->savePrefs();
        //TODO: remove item from list
        // move selection to next (if any).
    } catch(const FWException &ex)
    {
        MessageDialog::Error(string(_("Error removing certificate and key : "))+ex.toString(),this);
    }

    fillListOfCertificates();

}

void OptionsDlg::on_export_public_key_clicked()
{
    Gtk::CList_Helpers::SelectionList sl=cert_list->selection();
    if (sl.empty()) return;
    gpointer p=  sl.front().get_data();
    assert (p!=NULL);
    string *id= (string*)p;
    

    FileSel *fs=new FileSel(_("Export public key to file"),
			    Preferences::global_prefs->getWdir(),
			    "");


    string df=fs->run();
    delete fs;
    if(df.empty())
        return; // cancel 
    

    SimplePasswordDialog *pd = new SimplePasswordDialog(_("Please enter passprhase to access certificate: "));


    
    if(!pd->run())
    {
        // cancel
        delete pd;
        return;
    }
    string pass = pd->getPwd();
    delete pd;

    try
    {
        Preferences::global_prefs->exportPublicKey(*id, pass, df);
    } catch(const FWException &ex)
    {
        MessageDialog::Error(string(_("Error removing exporting public key : "))+ex.toString(),this);
    }
}

gint OptionsDlg::on_delete_event(GdkEventAny* ev)
{
    commit_changes=false;
    Gtk::Main::quit();
    return true;
}

OptionsDlg::~OptionsDlg() 
{
    RowList::iterator i;

    for (i=ctree->rows().begin(); i!=ctree->rows().end(); ++i) {
	char *cptr= (char*)( (*i).get_data() );
	if (cptr) delete cptr;
    }
}

void OptionsDlg::on_ctree_select_row(gint row, gint column, GdkEvent *event)
{
    RowList          rl=ctree->rows();
    if (rl.empty()) return;  

    char      *lbl=(char*)(rl[row].get_data());
    if (lbl!=NULL) {
	unsigned  page=notebook_pages[ string(lbl) ];
	notebook->set_page(page);
    }
}

void OptionsDlg::on_path_browse_clicked()
{
    FileSel *fs=new FileSel(_("Working directory"),
			    Preferences::global_prefs->getWdir(),
                            "");
    string wd=fs->run();
    delete fs;
    if(wd.empty()) return; // cancel

    wdir->set_text(wd);
}

bool OptionsDlg::run()
{
    commit_changes=false;

    Gtk::Main::grab_add(*this);
    Gtk::Main::run();
    Gtk::Main::grab_remove(*this);

    if (commit_changes) {


/*
 *  Paths  tab
 */
	Preferences::global_prefs->setOpt(
	    "/FWBuilderPreferences/Paths/Wdir",
	    wdir->get_text().c_str() );

/*
 *   User Interface  tab
 */
	Preferences::global_prefs->setOptBool(
	    "/FWBuilderPreferences/UI/RememberWindowPositionAndSize",
	    remember_window->get_active() );

	Preferences::global_prefs->setOptBool(
	    "/FWBuilderPreferences/UI/HideNavigationBar",
	    hide_navbar->get_active() );



	string qv;

	if (quick_view_off->get_active())    qv="off";
	if (quick_view_popup->get_active())  qv="popup";
	if (quick_view_window->get_active()) qv="window";

	Preferences::global_prefs->setOpt("/FWBuilderPreferences/UI/ObjectQuickView", qv );

	Preferences::global_prefs->setOptInt(
	    "/FWBuilderPreferences/UI/ObjectQuickViewTimeout",
	    quick_view_timeout->get_value_as_int() );

	Preferences::global_prefs->setOptBool(
	    "/FWBuilderPreferences/UI/Autosave",
	    autosave->get_active() );

	string ot;

	if (obj_tree_split->get_active())     ot="Split";
	if (obj_tree_combined->get_active())  ot="Combined";

	Preferences::global_prefs->setOpt("/FWBuilderPreferences/UI/ObjectTreeMode", ot );

	Preferences::global_prefs->setOptBool(
	    "/FWBuilderPreferences/UI/ShowObjectPropertiesInTree",
	    show_obj_props_in_tree->get_active() );

	Preferences::global_prefs->setOptBool(
	    "/FWBuilderPreferences/UI/ShowIconsInTree",
	    show_icons_in_tree->get_active() );


/* Network */

	Preferences::global_prefs->setOptInt(
	    "/FWBuilderPreferences/Network/SNMPTimeout",
	    snmp_timeout->get_value_as_int() );

	Preferences::global_prefs->setOptInt(
	    "/FWBuilderPreferences/Network/SNMPRetries",
	    snmp_retries->get_value_as_int() );

	Preferences::global_prefs->setOptInt(
	    "/FWBuilderPreferences/Network/DNSTimeout",
	    dns_timeout->get_value_as_int() );

	Preferences::global_prefs->setOptInt(
	    "/FWBuilderPreferences/Network/DNSRetries",
	    dns_retries->get_value_as_int() );


//	Preferences::global_prefs->setOpt("/FWBuilderPreferences/UI/ShowHidden",
//			     show_hidden->get_active()?"1":"0");

        try
        {
            Preferences::global_prefs->savePrefs();
        } catch(FWException &ex)
        {
	    MessageDialog::Error("Error saving preferences.",ex.toString(),this);
        }

    }
    return commit_changes;
}

void OptionsDlg::on_ok_clicked()
{   
    commit_changes=true;
    Gtk::Main::quit();
}

void OptionsDlg::on_cancel_clicked()
{   
    commit_changes=false;
    Gtk::Main::quit();
}








