/* 

                          Firewall Builder

                 Copyright (C) 2000 NetCitadel, LLC

  Author:  Vadim Kurland     vadim@vk.crocodile.org

  $Id: main_window2.cc,v 1.75 2003/05/02 06:10:55 vkurland Exp $


  This program is free software which we release under the GNU General Public
  License. You may redistribute and/or modify this program under the terms
  of that license as published by the Free Software Foundation; either
  version 2 of the License, or (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
 
  To get a copy of the GNU General Public License, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/


#include "fwbuilder/libfwbuilder-config.h"

#include <iostream>
#include <gtk--/base.h>
#include <gtk--/main.h>

#include <assert.h>

#include "config.h"

#include "glademm_support.hh"

#include "helpers.hh"

#include "main_window.hh"

#include "fwbuilder/FWObjectDatabase.hh"
#include "FWObjectDatabaseGUI.hh"

#include "fwbuilder/Group.hh"
#include "fwbuilder/ObjectGroup.hh"
#include "fwbuilder/ServiceGroup.hh"
#include "fwbuilder/IntervalGroup.hh"
#include "fwbuilder/Interface.hh"
#include "fwbuilder/IPv4.hh"
#include "fwbuilder/physAddress.hh"
#include "fwbuilder/FWOptions.hh"
#include "fwbuilder/InterfacePolicy.hh"
#include "fwbuilder/Policy.hh"
#include "fwbuilder/NAT.hh"
#include "fwbuilder/Firewall.hh"
#include "fwbuilder/RuleElement.hh"
#include "fwbuilder/snmp.hh"
#include "fwbuilder/Management.hh"
#include "fwbuilder/physAddress.hh"

#include "GroupDialog.hh"
#include "BuiltinDialog.hh"
#include "DialogPlugin.hh"
#include "Preferences.hh"
#include "GenericBackgroundOpDialog.hh"
#include "NewHostDruid.hh"
#include "NewFirewallDruid.hh"
#include "ObjectTree.hh"

#include "FindDialog.hh"
#include "WhereUsedDialog.hh"

#include "MessageDialog.hh"

using namespace libfwbuilder;


bool main_window::safe_to_close_right_pane()
{
    BuiltinDialog *dlg=NULL;
    Gtk::Widget *w=right_pane_port->get_child();
    if (w) dlg = dynamic_cast<BuiltinDialog*>(w);

    
    if ( dlg!=NULL &&  dlg->IsDataChanged()) {
	      
	if ( Preferences::global_prefs->getOptBool(
		 "/FWBuilderPreferences/UI/Autosave") ) {
	    dlg->SaveData();
	    return true;
	
	} else {

	    MessageDialog::DlgReturnValue action=MessageDialog::Question(
_("Data in the current dialog has been modified.\n\
Do you want to save it before switching to \n\
another object?"), this);
    
	    switch (action)   {
	    case MessageDialog::OK_YES: 
		dlg->SaveData(); 
		return true;

	    case MessageDialog::NO:      // ignoring changes
		dlg->UndoChanges();
		return true;

	    case MessageDialog::CANCEL:                  
		return false ;
	    }

	    dlg->UndoChanges();
	    return true;

	    return false;
	}
    }
    return true;
}

void main_window::schedule_open_object(const string &id,const string &lib)
{
    oo_obj=id;
    oo_lib=lib;
    open_obj_conn.disconnect();
    open_obj_conn=Gtk::Main::idle.connect(slot(this,
				   &main_window::open_object_when_idle));
}

gint main_window::open_object_when_idle()
{
//    open_obj_conn.disconnect();

    if (oo_obj=="") return false;
    OpenObject(oo_obj,oo_lib);
    oo_obj="";
    oo_lib="";
    return false;
}

void main_window::ClearRightPane(bool logo)
{
    if(right_pane_port)
    {
        Gtk::Widget *w=right_pane_port->get_child();
        if (w!=0) {
            w->hide_all();
            right_pane_port->remove(); 

	    if ( ! Gtk::Label::isA(w))  delete w;

	    current_right_pane_dialog=NULL;
        }
    }      

    if (logo) 
    {
        /*
         * TODO:  show logo in the right pane if there is going to be no dialog
         */
    }

    Gtk::Widget *itm;

    itm=find_widget("duplicate",main_menubar1);
    if (itm) itm->set_sensitive(false);

    itm=find_widget("copy",main_menubar1);
    if (itm) itm->set_sensitive(false);

    itm=find_widget("cut",main_menubar1);
    if (itm) itm->set_sensitive(false);

    itm=find_widget("paste",main_menubar1);
    if (itm) itm->set_sensitive(false);
}

/*
 *  NB: OpenObject does not check if there is unsaved data in the current
 *  dialog anymore! Caller should check it before calling OpenObject.
 * 
 *  --vk 02/09/02
 */
bool main_window::OpenObject (const string &id,const string &lib)
{
    ClearRightPane(false);

//    if (lib=="")  obook->showObject(id );
//    else          obook->showObject(id , lib );

    obook->showObject(id , lib );

/* child of right_pane_port is deleted in ClearRightPane */
    BuiltinDialog *dptr = new BuiltinDialog(this,id);
    assert(dptr!=NULL);

    //  right_pane_port->remove(); 
    right_pane_port->add( *dptr );
        
    dptr->LoadData();

    dptr->show();
    right_pane_port->show();

    current_right_pane_dialog=dptr;
    
    dptr->getDialog()->updateMainMenu();

    if (navbar!=NULL &&
        ! Preferences::global_prefs->getOptBool("/FWBuilderPreferences/UI/HideNavigationBar") ) 
        navbar->update(id);

//    FWObject *o=nowShowingObject();

    return true;
}

void  main_window::on_new_fw()
{
    if ( safe_to_close_right_pane() ) 
    {
        NewFirewallDruid *dr=new NewFirewallDruid(this,"New Firewall Object",
                                                  Resources::global_res->getIconPath("RulesDruidLogo") );
        dr->show();
    }
}

void  main_window::on_new_host()
{
    if ( safe_to_close_right_pane() ) 
    {
        NewHostDruid *dr=new NewHostDruid(this,"New Host Object",
                                          Resources::global_res->getIconPath("NewHostDruidLogo") );
        dr->show();
    }
}

void  main_window::on_new_interface()
{
    try 
    {
        if ( safe_to_close_right_pane() ) 
        {

            string target_id=obook->getCurrentSelection();

            FWObject *o=FWObjectDatabaseGUI::newInterface( target_id );
            obook->showPage(0);  // User-defined objects
            insertInAllObjectBooks(o);
            obook->showObject(o);
        }
    } catch(FWException &ex)  {
        cerr << ex.toString();
    }
}

void  main_window::on_new_interface_via_snmp()
{
    try 
    {
        if ( safe_to_close_right_pane() ) 
        {
            string target_id=obook->getCurrentSelection();

            list<FWObject*> interfaces=FWObjectDatabaseGUI::newInterfacesViaSNMP( target_id );

            obook->showPage(0);  // User-defined objects
            for (list<FWObject*>::iterator i=interfaces.begin(); i!=interfaces.end(); ++i)
            {
                insertInAllObjectBooks( (*i) );
                obook->showObject(*i);
            }
        }
    } catch(FWException &ex)  {
        cerr << ex.toString();
    }

}

void  main_window::on_new_ipv4()
{
    try 
    {
        if ( safe_to_close_right_pane() ) 
        {
            string target_id=obook->getCurrentSelection();

            FWObject *o=FWObjectDatabaseGUI::newIPv4( target_id );
            obook->showPage(0);  // User-defined objects
            main_window::insertInAllObjectBooks(o);
            obook->showObject(o);
        }
    } catch(FWException &ex)  {
        cerr << ex.toString();
    }
}

void  main_window::on_new_physaddress()
{
    try 
    {
        if ( safe_to_close_right_pane() ) 
        {
            string target_id=obook->getCurrentSelection();

            FWObject *obj= FWObjectDatabase::db->getById( target_id, true);
            assert (Interface::cast(obj)!=NULL);

            physAddress *pa=Interface::cast(obj)->getPhysicalAddress();
            if ( pa==NULL )
            {
                FWObject *o=FWObjectDatabaseGUI::newPhysAddress( target_id );
                obook->showPage(0);  // User-defined objects
                main_window::insertInAllObjectBooks(o);
                obook->showObject(o);
            } else
            {
                schedule_open_object( pa->getId() , pa->getLibrary() );
            }
        }
    } catch(FWException &ex)  {
        cerr << ex.toString();
    }
}

void  main_window::on_new_net()
{
    if ( safe_to_close_right_pane() ) 
    {
	FWObject *o=FWObjectDatabaseGUI::newNetwork();
	obook->showPage(0);  // User-defined objects
        main_window::insertInAllObjectBooks(o);
        obook->showObject(o);
    }
}

void  main_window::on_new_addr_range()
{
    if ( safe_to_close_right_pane() ) 
    {
	FWObject *o=FWObjectDatabaseGUI::newAddressRange();
	obook->showPage(0);  // User-defined objects
        main_window::insertInAllObjectBooks(o);
//	obook->insertObject( o );
        obook->showObject(o);
    }
}

void  main_window::on_new_obj_group()
{
    if ( safe_to_close_right_pane() ) 
    {
	FWObject *o=FWObjectDatabaseGUI::newObjectGroup();
	obook->showPage(0);  // User-defined objects
        main_window::insertInAllObjectBooks(o);
//	obook->insertObject( o );
        obook->showObject(o);
    }
}

void  main_window::on_new_ip()
{
    if ( safe_to_close_right_pane() ) 
    {
	FWObject *o=FWObjectDatabaseGUI::newIPService();
	obook->showPage(0);  // User-defined objects
        main_window::insertInAllObjectBooks(o);
//	obook->insertObject( o );
        obook->showObject(o);
    }
}

void  main_window::on_new_icmp()
{
    if ( safe_to_close_right_pane() ) 
    {
	FWObject *o=FWObjectDatabaseGUI::newICMPService();
	obook->showPage(0);  // User-defined objects
        main_window::insertInAllObjectBooks(o);
//	obook->insertObject( o );
        obook->showObject(o);
    }
}

void  main_window::on_new_tcp()
{
    if ( safe_to_close_right_pane() ) 
    {
	FWObject *o=FWObjectDatabaseGUI::newTCPService();
	obook->showPage(0);  // User-defined objects
        main_window::insertInAllObjectBooks(o);
//	obook->insertObject( o );
        obook->showObject(o);
    }
}

void  main_window::on_new_udp()
{
    if ( safe_to_close_right_pane() ) 
    {
	FWObject *o=FWObjectDatabaseGUI::newUDPService();
	obook->showPage(0);  // User-defined objects
        main_window::insertInAllObjectBooks(o);
//	obook->insertObject( o );
        obook->showObject(o);
    }
}

void  main_window::on_new_srv_group()
{
    if ( safe_to_close_right_pane() ) 
    {
	FWObject *o=FWObjectDatabaseGUI::newServiceGroup();
	obook->showPage(0);  // User-defined objects
        main_window::insertInAllObjectBooks(o);
//	obook->insertObject( o );
        obook->showObject(o);
    }
}

void  main_window::on_new_custom()
{
    if ( safe_to_close_right_pane() ) 
    {
	FWObject *o=FWObjectDatabaseGUI::newCustomService();
	obook->showPage(0);  // User-defined objects
        main_window::insertInAllObjectBooks(o);
//	obook->insertObject( o );
        obook->showObject(o);
    }
}

void  main_window::on_new_time()
{
    if ( safe_to_close_right_pane() ) 
    {
	FWObject *o=FWObjectDatabaseGUI::newInterval();
	obook->showPage(0);  // User-defined objects
        main_window::insertInAllObjectBooks(o);
//	obook->insertObject( o );
        obook->showObject(o);
    }
}




void main_window::on_cutobj()
{   
    string id=obook->getCurrentSelection();
    FWObject *obj=FWObjectDatabase::db->getById( id, true);
    if ( ! Resources::isSystem(obj) && 
	 ! InterfacePolicy::isA(obj)  && 
	 ! Policy::isA(obj)  && 
	 ! NAT::isA(obj) ) 
    {
	FWObjectClipboard::obj_clipboard->putObject(obj);
	on_delobj();
    }
}

/*
 * this method checks if object with given id is referenced by some group
 * or firewall policy and shows a warning
 * If object is not referenced, it stills shows "yes/no" dialog to ask user
 * whether they are sure object is to be deleted
 *
 * This method should be called to confirm non-reversible deletion of
 * real objects, not references.
 */
bool main_window::confirm_remove_object(const string &id)
{
    return confirm_remove_object( FWObjectDatabase::db->getById( id, true) );
}

bool main_window::confirm_remove_object(FWObject *obj)
{
    if ( Resources::isSystem(obj)  ||
	 InterfacePolicy::isA(obj) ||
	 Policy::isA(obj)          ||
	 NAT::isA(obj) ) return false;


//	cerr << _("start scanning tree: ") << time(NULL) << endl;


    set<FWReference*> refs=FWObjectDatabase::db->findAllReferences(obj);

//	cerr << _("done scanning tree: ") << time(NULL) << endl;



    if (refs.size()!=0) 
    {
/* 
 * This object is referenced from some other groups or policies. Check
 * if only its own children objects reference it though.
 */
        bool only_self_references=true;
        for (set<FWReference*>::iterator r=refs.begin(); r!=refs.end(); r++)
        {
            FWObject *o=(*r);
            only_self_references &= o->isChildOf(obj);
        }
        if ( ! only_self_references )
        {
            char errstr[4096];
            sprintf(errstr,
"%s %s is used in some other group or policy. \n\
The operation you have chosen will remove it and \n\
all the references to it from the database.\n\
Are you sure you want to delete it ? ", obj->getTypeName().c_str(), obj->getName().c_str());

            MessageDialog::DlgReturnValue a = MessageDialog::YesNo(errstr,this);
            return (a==MessageDialog::OK_YES);
        }
    }

    for (FWObjectTypedChildIterator j=obj->findByType(Interface::TYPENAME); j!=j.end(); ++j ) 
    {
        FWObject *o=(*j);
        set<FWReference*> refs=FWObjectDatabase::db->findAllReferences(o);
        if (refs.size()!=0) 
        {
/*One of the interfaces of this object is referenced from some other groups or policies. */
            bool only_self_references=true;
            for (set<FWReference*>::iterator r=refs.begin(); r!=refs.end(); r++)
            {
                FWObject *o=(*r);
                only_self_references &= o->isChildOf(obj);
            }
            if ( ! only_self_references )
            {
                char errstr[4096];
                sprintf(errstr,
"Interface '%s' of this %s is used in some other \n\
group or policy. The operation you have chosen will\n\
remove it and all the references to it from the database. \n\
Are you sure you want to delete it ? ", o->getName().c_str(), obj->getTypeName().c_str());

                MessageDialog::DlgReturnValue a = MessageDialog::YesNo(errstr, this);;
                return (a==MessageDialog::OK_YES);
            }
        }
        for (FWObjectTypedChildIterator k=o->findByType(IPv4::TYPENAME); k!=k.end(); ++k ) 
        {
            FWObject *o1=(*k);
            set<FWReference*> refs=FWObjectDatabase::db->findAllReferences(o1);
            if (refs.size()!=0) 
            {
/*One of the addresses of this interface is referenced from some other groups or policies. */
                bool only_self_references=true;
                for (set<FWReference*>::iterator r=refs.begin(); r!=refs.end(); r++)
                {
                    FWObject *o=(*r);
                    only_self_references &= o->isChildOf(obj);
                }
                if ( ! only_self_references )
                {
                    char errstr[4096];
                    sprintf(errstr,
"IP address of the interface '%s' of this %s is used \n\
in some other group or policy. The operation you have \n\
chosen will remove it and all the references to it from the database.\n\
Are you sure you want to delete it ? ", o->getName().c_str(), obj->getTypeName().c_str());

                    MessageDialog::DlgReturnValue a = MessageDialog::YesNo(errstr, this);;
                    return (a==MessageDialog::OK_YES);
                }
            }
        }

        for (FWObjectTypedChildIterator k=o->findByType(physAddress::TYPENAME); k!=k.end(); ++k ) 
        {
            FWObject *o1=(*k);
            set<FWReference*> refs=FWObjectDatabase::db->findAllReferences(o1);
            if (refs.size()!=0) 
            {
/*One of the addresses of this interface is referenced from some other groups or policies. */
                bool only_self_references=true;
                for (set<FWReference*>::iterator r=refs.begin(); r!=refs.end(); r++)
                {
                    FWObject *o=(*r);
                    only_self_references &= o->isChildOf(obj);
                }
                if ( ! only_self_references )
                {
                    char errstr[4096];
                    sprintf(errstr,
"A physical address of the interface '%s' of this %s \n\
is used in some other group or policy. The operation you \n\
have chosen will remove it and all the references to it from the database.\n\
Are you sure you want to delete it ? ", o->getName().c_str(), obj->getTypeName().c_str());

                    MessageDialog::DlgReturnValue a = MessageDialog::YesNo(errstr, this);;
                    return (a==MessageDialog::OK_YES);
                }
            }
        }
    }
    char errstr[4096];
    sprintf(errstr,
"This operation will remove object %s from the database.\n\
Are you sure you want to delete it ?",obj->getName().c_str());

    MessageDialog::DlgReturnValue a = MessageDialog::YesNo(errstr,this);

    return (a==MessageDialog::OK_YES);
}

void main_window::on_delobj()
{   
    string id=obook->getCurrentSelection();
    string lib=obook->getCurrentLibrary();
    FWObject *obj=FWObjectDatabase::db->getById( id, true);
    if (confirm_remove_object(obj)) 
    {
        std::map<main_window*,FWObject*> wl;
        std::list<main_window*>          cwl;

        for (list<main_window*>::iterator i=windows.begin(); i!=windows.end(); ++i) 
        {
/*
 * check if this window shows either object that we are about to
 * delete or one of its children objects
 */
            FWObject *wo=(*i)->nowShowingObject();
            FWObject *wp=wo;
            while (wp!=NULL && wp->getId()!=obj->getId() )
                wp=wp->getParent();

            if (wp!=NULL)
            {
                wl[(*i)]=wp->getParent();
            }
/*
 * check if we should refresh dialog. This should be done if the
 * object we are about to delete is shown in the right panel of
 * "other" window. At this stage we just remember windows that need to
 * be refreshed, but postpone actual refresh until object has been
 * deleted from the database.
 */
            bool f=false;
            BuiltinDialog *bd=(*i)->getRightPaneDialog();
            if (bd!=NULL) f=bd->getDialog()->isPartOfDialog(obj);
            if (f) cwl.push_back( (*i) );
        }

//	ClearRightPane(true);
//	FWObject *p=obj->getParent();

        main_window::removeFromAllObjectBooks(obj);

        list<FWObject*> il=obj->getByType(Interface::TYPENAME);
        for (list<FWObject*>::iterator j=il.begin(); j!=il.end(); ++j ) 
        {
            FWObject *o=(*j);

            list<FWObject*>   cl=o->getByType(IPv4::TYPENAME);
            for (list<FWObject*>::iterator k=cl.begin(); k!=cl.end(); ++k ) 
            {
                FWObject *o1=(*k);
                FWObjectDatabase::db->removeAllInstances(o1);
            }

            cl=o->getByType(physAddress::TYPENAME);
            for (list<FWObject*>::iterator k=cl.begin(); k!=cl.end(); ++k ) 
            {
                FWObject *o1=(*k);
                FWObjectDatabase::db->removeAllInstances(o1);
            }
            FWObjectDatabase::db->removeAllInstances(o);
        }
	FWObjectDatabase::db->removeAllInstances(obj);


        for (std::map<main_window*,FWObject*>::iterator i=wl.begin(); i!=wl.end(); ++i) 
        {
            main_window *w=(*i).first;
            FWObject    *p=(*i).second;
            w->ClearRightPane(false);
            w->OpenObject( p->getId() , lib );
        }

        for (std::list<main_window*>::iterator i=cwl.begin(); i!=cwl.end(); ++i) 
        {
            BuiltinDialog *bd=(*i)->getRightPaneDialog();
            if (bd!=NULL) bd->LoadData();
        }

//	OpenObject( p->getId() );
    }
}

void main_window::on_where_used()
{   
    string id=obook->getCurrentSelection();
    FWObject *obj=FWObjectDatabase::db->getById( id, true);

    set<FWReference*> refs=FWObjectDatabase::db->findAllReferences(obj);
    set<FWObject*>    objects;

    FWObject *o;
    set<FWReference*>::iterator m;
    for (m=refs.begin(); m!=refs.end(); ++m) {
	if ( (o=(*m))!=NULL ) {
	    o=o->getParent();
	    if ( RuleElement::cast(o)!=NULL) {
		while ( o!=NULL && Firewall::cast(o)==NULL )
		    o=o->getParent();
	    }

	    if (o) objects.insert(o);
	}
    }
    WhereUsedDialog *wud=manage(new WhereUsedDialog(this, obj , objects ));
    wud->run();
}

void main_window::on_duplicate()
{   
    string id=obook->getCurrentSelection();
    if ( safe_to_close_right_pane() ) {
	FWObject *obj=FWObjectDatabase::db->getById( id , true);
	if (obj && ! Resources::isSystem(obj) ) {
	    FWObject *parent=obj->getParent();
	    assert (parent!=NULL);
            if ( Resources::isSystem(parent) ||
                 Interface::isA(obj) ||
                 IPv4::isA(obj) )   insertCopy(parent,obj);
	}
    }
}

void main_window::on_copyobj()
{   
    string id=obook->getCurrentSelection();
    if ( safe_to_close_right_pane() ) {
	FWObject *obj=FWObjectDatabase::db->getById( id , true);
	if (obj && ! Resources::isSystem(obj) ) 
	    FWObjectClipboard::obj_clipboard->putObject( obj );
    }
}

void  main_window::on_pasteobj()
{
    if ( safe_to_close_right_pane() ) {

	FWObject *obj=FWObjectClipboard::obj_clipboard->getObject();
	if (obj==NULL) return;

	FWObject *cursel=
	    FWObjectDatabase::db->getById( obook->getCurrentSelection() ,true);
	if (cursel==NULL) return;

        if ( Resources::isSystem(cursel) ||
             Interface::isA(obj) ||
             IPv4::isA(obj) )
        {
/*
 * clipboard holds a copy of the object, which it deletes when
 * another object is placed into it. We have to make a new copy here.
 */
            FWObject *nobj= FWObjectDatabaseGUI::db->create(obj->getTypeName(),true);
            assert (nobj!=NULL);
            nobj->ref();
            *nobj = *obj;

            insertCopy(cursel,nobj);
        } else {

            BuiltinDialog *b=getRightPaneDialog();
            if (b) {
                GroupDialog   *gd=dynamic_cast<GroupDialog*>(b->getDialog());
                if (gd) {
                    gd->addObject(obj);
                }            
            }
        }
    }
}


void main_window::insertCopy(FWObject *target, FWObject *obj)
{
    if ( FWObjectDatabaseGUI::validateObjectForPositionInTree(target, obj) )
    {
/*
 *  We need to check if such object already exists in the tree at this
 *  position and, if not, then add it, but if it does exist, then
 *  create and add its copy
 */
        FWObject *oo=FWObjectDatabaseGUI::db->getById( obj->getId() , true);
        FWObject *no=obj;

        if (oo!=NULL) {
            no=FWObjectDatabaseGUI::db->create(obj->getTypeName());
            assert(no!=NULL);

            no->duplicate(obj);
            no->setName(_("Copy of ")+obj->getName());
            FWObjectDatabaseGUI::newObject( target->getId() , no , false );
        } else
            FWObjectDatabaseGUI::newObject( target->getId() , no , true );

        main_window::insertInAllObjectBooks(no);
    }
}


class sort1_order_func_adaptor 
{
    private:    
    FWObject *o;
    public:
    explicit sort1_order_func_adaptor(FWObject *p) { o=p; }
    bool operator()(const FWObject *a, const FWObject *b) 
    { 
        return a->getName() < b->getName();
    }
};

class sort2_order_func_adaptor 
{
    private:    
    FWObject *o;
    public:
    explicit sort2_order_func_adaptor(FWObject *p) { o=p; }
    bool operator()(const FWObject *a, const FWObject *b) 
    { 
        return ObjectTree::get_properties(a) < ObjectTree::get_properties(b);
    }
};


void main_window::on_sort_by_name()
{
    FWObject *o;
    string id=obook->getCurrentSelection();
    string lib=obook->getCurrentLibrary();
    if (id.empty() || id==FWObjectDatabase::db->getRootId()) return;

    o=FWObjectDatabase::db->getById( id , true ) ;
    assert(o!=NULL);

    o->sort(sort1_order_func_adaptor(o));

//    o->sortChildren();

    sortSubtreeInAllObjectBooks(id,0);

    OpenObject(id,lib);
}

void main_window::on_sort_by_prop()
{
    FWObject *o;
    string id =obook->getCurrentSelection();
    string lib=obook->getCurrentLibrary();
    if (id.empty() || id==FWObjectDatabase::db->getRootId()) return;

    o=FWObjectDatabase::db->getById( id , true ) ;
    assert(o!=NULL);

    o->sort(sort2_order_func_adaptor(o));

//    o->sortChildren();

    sortSubtreeInAllObjectBooks(id,1);

    OpenObject(id,lib);
}

void main_window::on_find()
{
    if (glademm_get_Widget("FindDialog")==NULL)
	manage(new FindDialog(this));
}

/*
 *  the following methods emulate user pressing Ctrl-C, Ctrl-X, Ctrl-V
 *  for copy/cut/paste operations. It is not quite right way to
 *  implement clipboard support.  The better way is to scan widgets
 *  tree starting from the main window looking for the widget which
 *  holds focus and is a descendant of the class Editable. If such
 *  widget could be found, we would then use functions
 *  gtk_editable_copy_clipboard etc on it.  We can't use this approach
 *  yet because we heavily use widget Gtk::Table for which children
 *  lists interface is totally broken in gtk-- 1.2 
 *
 *                               --vk 12/15/2001
 *
 *   even worse, this hack caused loop if user pressed "Ctrl-C" when
 *   any object was highlighted in the tree and no editable widget was
 *   active at the moment. This happened because key_press event was
 *   processed by accelerator, associated with menu item "Copy", which
 *   called this method, which in turn generated new key_press event
 *   with key "Ctrl-C". To avoid this I had to create menu items without
 *   accelerators.
 *                               -- vk 01/13/2002
 */

static char ctrl_c='\003';
static char ctrl_x='\030';
static char ctrl_v='\026';

void  main_window::on_copy()
{
    Gtk::Main::idle.connect(slot(this,&main_window::emit_copy));
}

void  main_window::on_cut()
{
    Gtk::Main::idle.connect(slot(this,&main_window::emit_cut));
}

void  main_window::on_paste()
{
    Gtk::Main::idle.connect(slot(this,&main_window::emit_paste));
}

gint  main_window::emit_copy()
{
    GdkEvent ev;
    ev.key.window=current_right_pane_dialog->get_window();

    ev.key.type=GDK_KEY_PRESS;
    ev.key.send_event=0;
    ev.key.state=4;
    ev.key.length=1;
    ev.key.string= &ctrl_c;
    ev.key.keyval=GDK_c;            
    ev.key.time=time(NULL);
    Gdk::event_put(ev);
    return false;
}

gint  main_window::emit_cut()
{
    GdkEvent ev;
    ev.key.window=current_right_pane_dialog->get_window();

    ev.key.type=GDK_KEY_PRESS;
    ev.key.send_event=0;
    ev.key.state=4;
    ev.key.length=1;
    ev.key.string= &ctrl_x;
    ev.key.keyval=GDK_x;            
    ev.key.time=time(NULL);
    Gdk::event_put(ev);
    return false;
}

gint  main_window::emit_paste()
{
    GdkEvent ev;
    ev.key.window=current_right_pane_dialog->get_window();

    ev.key.type=GDK_KEY_PRESS;
    ev.key.send_event=0;
    ev.key.state=4;
    ev.key.length=1;
    ev.key.string= &ctrl_v;
    ev.key.keyval=GDK_v;            
    ev.key.time=time(NULL);
    Gdk::event_put(ev);
    return false;
}
