/* 

                          Firewall Builder

                 Copyright (C) 2002 NetCitadel, LLC

  Author:  Vadim Kurland     vadim@vk.crocodile.org

  $Id: PolicyCompiler_ipt_writers.cc,v 1.64 2003/12/19 06:48:43 vkurland Exp $

  This program is free software which we release under the GNU General Public
  License. You may redistribute and/or modify this program under the terms
  of that license as published by the Free Software Foundation; either
  version 2 of the License, or (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
 
  To get a copy of the GNU General Public License, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/

#include "PolicyCompiler_ipt.hh"

#include "fwbuilder/RuleElement.hh"
#include "fwbuilder/IPService.hh"
#include "fwbuilder/ICMPService.hh"
#include "fwbuilder/TCPService.hh"
#include "fwbuilder/UDPService.hh"
#include "fwbuilder/CustomService.hh"
#include "fwbuilder/Policy.hh"
#include "fwbuilder/Network.hh"

#include "fwbuilder/FWObjectDatabase.hh"
#include "fwbuilder/RuleElement.hh"
#include "fwbuilder/Policy.hh"
#include "fwbuilder/Interface.hh"
#include "fwbuilder/IPv4.hh"
#include "fwbuilder/Firewall.hh"

#include "combinedAddress.hh"


#include <iostream>
#if __GNUC__ > 3 || \
    (__GNUC__ == 3 && (__GNUC_MINOR__ > 2 || (__GNUC_MINOR__ == 2 ) ) )
#  include <streambuf>
#else
#  include <streambuf.h>
#endif
#include <iomanip>
#include <fstream>
#include <sstream>

#include <assert.h>

using namespace libfwbuilder;
using namespace fwcompiler;
using namespace std;



/**
 *-----------------------------------------------------------------------
 *                    Methods for printing
 */
string PolicyCompiler_ipt::PrintRule::_printChain(PolicyRule *rule)
{
    string s=rule->getStr("ipt_chain");
    if (s.empty()) s="UNKNOWN";
    s= s + " ";
    return s;
}

string PolicyCompiler_ipt::PrintRule::_printTarget(PolicyRule *rule)
{
    std::ostringstream ostr;

    string target=rule->getStr("ipt_target");
    if (target.empty()) target="UNKNOWN";
    int lim;

    if ( compiler->getCachedFwOpt()->getBool("use_ULOG") &&
         target=="LOG") target="ULOG";
    
/*
 * Here is what do we do with limits:
 *
 * Limit set globally in 'Firewall' tab of the firewall dialog 
 * applies only to logging
 *
 * Limit set in the rule options dialog applies only to this 
 * rule's target.
 * 
 *    this is so as of 1.0.11 ( 28/06/03 )  --vk
 */
    FWOptions *ruleopt =rule->getOptionsObject();

    if (target=="LOG" || target=="ULOG")    
    {
        FWOptions *compopt=compiler->getCachedFwOpt();
        if ((lim=compopt->getInt("limit_value"))>0)
        {
            ostr << "  -m limit --limit " << lim;
        
            string ls=compopt->getStr("limit_suffix");
            if (!ls.empty()) ostr << ls;

            string lb=compopt->getStr("limit_burst");
            if (!lb.empty()) ostr << " --limit-burst " << lb;
        }
    } else {
        if (ruleopt!=NULL && (lim=ruleopt->getInt("limit_value"))>0)
        {
            ostr << "  -m limit --limit " << lim;

            string ls=ruleopt->getStr("limit_suffix");
            if (!ls.empty()) ostr << ls;

            string lb=ruleopt->getStr("limit_burst");
            if (!lb.empty()) ostr << " --limit-burst " << lb;
        }
    }

    ostr << " -j " << target << " ";

    if (target=="REJECT")
      ostr << _printActionOnReject(rule);

    if (target=="LOG" || target=="ULOG")    
        ostr << _printLogParameters(rule);
    
    return ostr.str();
}

string PolicyCompiler_ipt::PrintRule::_printMultiport(PolicyRule *rule)
{
    string s;
    if( rule->getBool("ipt_multiport"))
	s= " -m multiport ";

    return s;
}

string PolicyCompiler_ipt::PrintRule::_printDirectionAndInterface(PolicyRule *rule)
{
    std::ostringstream ostr;

    string       iface_name = rule->getInterfaceStr();
    if (iface_name.empty() || iface_name=="nil" )   return "";

/* if interface name ends with '*', this is a wildcard
 * interface. Iptables supports wildcard interfaces but uses '+' as a
 * wildcard symbol */

    string::size_type n;
    if ( (n=iface_name.find("*"))!=string::npos)    iface_name[n]='+';

//    Interface *rule_iface =  compiler->getCachedFwInterface(iface_id);

    if (rule->getDirection()==PolicyRule::Inbound)   
        ostr << " -i "  << iface_name; 

    if (rule->getDirection()==PolicyRule::Outbound)  
        ostr << " -o "  << iface_name; 

//    if (rule->getDirection()==PolicyRule::Both)      
//        compiler->output << "-i "  << rule_iface->getName() 
//                         << " -o " << rule_iface->getName(); 
    ostr << " ";

    return ostr.str();
}

string PolicyCompiler_ipt::PrintRule::_printActionOnReject(libfwbuilder::PolicyRule *rule)
{
    std::ostringstream str;

    PolicyCompiler_ipt *ipt_comp=dynamic_cast<PolicyCompiler_ipt*>(compiler);

    RuleElementSrv *srvrel=rule->getSrv();
    Service        *srv   =compiler->getFirstSrv(rule);  assert(srv);

    string version=compiler->fw->getStr("version");
    string s=ipt_comp->getActionOnReject(rule);
    if (!s.empty()) 
    {
        if (ipt_comp->isActionOnRejectTCPRST(rule)) str << " --reject-with tcp-reset";

	if (s.find("ICMP")!=string::npos) 
        {
	    if (s.find("unreachable")!=string::npos) 
            {
		if (s.find("net")!=string::npos)   str << " --reject-with icmp-net-unreachable";
		if (s.find("host")!=string::npos)  str << " --reject-with icmp-host-unreachable";
		if (s.find("port")!=string::npos)  str << " --reject-with icmp-port-unreachable";
		if (s.find("proto")!=string::npos) str << " --reject-with icmp-proto-unreachable";
	    }
	    if (s.find("prohibited")!=string::npos) 
            {
		if (s.find("net")!=string::npos)   str << " --reject-with icmp-net-prohibited";
		if (s.find("host")!=string::npos)  str << " --reject-with icmp-host-prohibited";
                if (version=="1.2.9" && 
                    s.find("admin")!=string::npos)  str << " --reject-with icmp-admin-prohibited";
	    }
	}
    }
    str << " ";
    return str.str();
}

string PolicyCompiler_ipt::PrintRule::_printGlobalLogParameters()
{
    return _printLogParameters(NULL);
}

string PolicyCompiler_ipt::PrintRule::_printLogPrefix(PolicyRule *rule,
                                                      const string &prefix)
{
    string s=prefix;

/* deal with our logging macros:
 * %N - rule number
 * %A - action
 * %I - interface name
 * %C - chain name
 */
    string::size_type n;
    if (rule && (n=s.find("%N"))!=string::npos ) 
    {
        std::ostringstream s1;
        s1 << rule->getPosition();
        s.replace(n,2,s1.str());
    }
    if (rule && (n=s.find("%A"))!=string::npos ) 
    {
        std::ostringstream s1;

        char str[64];
        strncpy(str,rule->getStr("stored_action").c_str(),sizeof(str));
        for (char *cptr=str; *cptr; cptr++)  *cptr=toupper(*cptr);

        s.replace(n,2,str);
    }
    if (rule && (n=s.find("%I"))!=string::npos ) 
    {
        std::ostringstream s1;
        string rule_iface =  rule->getInterfaceStr();
        if (rule_iface!="") {
            s1 << rule_iface;
            s.replace(n,2,s1.str());
        } else
            s.replace(n,2,"global");
    }
    if (rule && (n=s.find("%C"))!=string::npos ) 
    {
        s.replace(n,2,rule->getStr("ipt_chain"));
    }

    if (s.length()>29)
    {
        compiler->warning(_("Log prefix has been truncated to 29 characters in rule ")+rule->getLabel());
        s=s.substr(0,29);
    }

    return "\"" + s + "\" ";
}

string PolicyCompiler_ipt::PrintRule::_printLogParameters(libfwbuilder::PolicyRule *rule)
{
    std::ostringstream str;
    string s;
//    int    l;
    FWOptions *ruleopt =(rule!=NULL)?rule->getOptionsObject():compiler->getCachedFwOpt();

    bool   use_ulog=compiler->getCachedFwOpt()->getBool("use_ULOG");

    if (use_ulog)
    {
        s=ruleopt->getStr("ulog_nlgroup");
        if (s.empty())  s=compiler->getCachedFwOpt()->getStr("ulog_nlgroup");
        if (!s.empty()) 
            str << " --ulog-nlgroup " << s;

        s=ruleopt->getStr("log_prefix");
        if (s.empty())  s=compiler->getCachedFwOpt()->getStr("log_prefix");
        if (!s.empty()) 
            str << " --ulog-prefix " << _printLogPrefix(rule,s);

        int r=compiler->getCachedFwOpt()->getInt("ulog_cprange");
        if (r!=0)  str << " --ulog-cprange " << r << " ";
        r=compiler->getCachedFwOpt()->getInt("ulog_qthreshold");
        if (r!=0)  str << " --ulog-qthreshold " << r << " ";
    } else
    {
        bool   numeric_levels;
        numeric_levels=compiler->getCachedFwOpt()->getBool("use_numeric_log_levels");
        s=ruleopt->getStr("log_level");
        if (s.empty())  s=compiler->getCachedFwOpt()->getStr("log_level");
        if (!s.empty()) 
        {
            if ( numeric_levels )
            {
                if (s=="alert")   s="1";
                if (s=="crit")    s="2";
                if (s=="error")   s="3";
                if (s=="warning") s="4";
                if (s=="notice")  s="5";
                if (s=="info")    s="6";
                if (s=="debug")   s="7";
            }
            str << " --log-level " << s;
        }

        s=ruleopt->getStr("log_prefix");
        if (s.empty())  s=compiler->getCachedFwOpt()->getStr("log_prefix");
        if (!s.empty())
            str << " --log-prefix " << _printLogPrefix(rule,s);

        if (ruleopt->getBool("log_tcp_seq") || compiler->getCachedFwOpt()->getBool("log_tcp_seq"))  
            str << " --log-tcp-sequence ";
        if (ruleopt->getBool("log_tcp_opt") || compiler->getCachedFwOpt()->getBool("log_tcp_opt"))  
            str << " --log-tcp-options ";
        if (ruleopt->getBool("log_ip_opt") || compiler->getCachedFwOpt()->getBool("log_ip_opt"))   
            str << " --log-ip-options ";
    }

    return str.str();
}

string PolicyCompiler_ipt::PrintRule::_printLimit(libfwbuilder::PolicyRule *rule)
{
    std::ostringstream str;
    string s;
    int    l;
    FWOptions *ruleopt =rule->getOptionsObject();
    FWOptions *compopt =compiler->getCachedFwOpt();

    if ( (ruleopt!=NULL && (l=ruleopt->getInt("limit_value"))>0) || 
         (l=compopt->getInt("limit_value"))>0 ) 
    {
	str << "  -m limit --limit " << l;

        if (ruleopt!=NULL) s=ruleopt->getStr("limit_suffix");
	if (s.empty()) 	   s=compopt->getStr("limit_suffix");
	if (!s.empty()) str << s;

	if (ruleopt!=NULL) s=ruleopt->getStr("limit_burst");
	if (s.empty())     s=compopt->getStr("limit_burst");
	if (!s.empty()) str << " --limit-burst " << s;
    }

    return str.str();
}

string PolicyCompiler_ipt::PrintRule::_printProtocol(libfwbuilder::Service *srv)
{
    string s;
    if ( ! srv->isAny() && ! CustomService::isA(srv) ) 
    {
        string pn=srv->getProtocolName();
        if (pn=="ip") pn="all";
        s= "-p " + pn + " ";
    }
    return s;
}

string PolicyCompiler_ipt::PrintRule::_printPorts(int rs,int re)
{
    std::ostringstream  str;

    compiler->normalizePortRange(rs,re);

    if (rs>0 || re>0) {
	if (rs==re)  str << rs;
	else
	    if (rs==0 && re!=0)      str << ":" << re;
	    else
                str << rs << ":" << re;
    }
    return str.str();
}

string PolicyCompiler_ipt::PrintRule::_printSrcPorts(Service *srv)
{
    std::ostringstream  str;
    if (TCPService::isA(srv) || UDPService::isA(srv)) 
    {
	int rs=srv->getInt("src_range_start");
	int re=srv->getInt("src_range_end");
	str << _printPorts(rs,re);
    }
    return str.str();
}

string PolicyCompiler_ipt::PrintRule::_printDstPorts(Service *srv)
{
    std::ostringstream  str;
    if (TCPService::isA(srv) || UDPService::isA(srv)) 
    {
	int rs=srv->getInt("dst_range_start");
	int re=srv->getInt("dst_range_end");
	str << _printPorts(rs,re);
    }
    return str.str();
}

string PolicyCompiler_ipt::PrintRule::_printICMP(ICMPService *srv)
{
    std::ostringstream  str;
    if (ICMPService::isA(srv) && srv->getInt("type")!=-1) {
	str << srv->getStr("type");
	if (srv->getInt("code")!=-1) 
	    str << "/" << srv->getStr("code") << " ";
    }
    return str.str();
}

string PolicyCompiler_ipt::PrintRule::_printIP(IPService *srv)
{
    std::ostringstream  str;
    if (IPService::isA(srv) ) {
	if (srv->getBool("fragm") || srv->getBool("short_fragm"))
	    str << " -f ";

        if  (srv->getBool("lsrr") ||
             srv->getBool("ssrr") ||
             srv->getBool("rr") ||
             srv->getBool("ts") ) str << " -m ipv4options ";

        if  (srv->getBool("lsrr")) str << " --lsrr";
        if  (srv->getBool("ssrr")) str << " --ssrr";
        if  (srv->getBool("rr"))   str << " --rr";
        if  (srv->getBool("ts"))   str << " --ts";
    }
    return str.str();
}

string PolicyCompiler_ipt::PrintRule::_printTCPFlags(libfwbuilder::TCPService *srv)
{
    string str;
    if (srv->inspectFlags())
    {
        TCPService::TCPFlag f1[2]={ TCPService::SYN };
        TCPService::TCPFlag f2[7]={ TCPService::URG,
                                    TCPService::ACK,
                                    TCPService::PSH,
                                    TCPService::RST,
                                    TCPService::SYN,
                                    TCPService::FIN };

        std::set<TCPService::TCPFlag> none;
        std::set<TCPService::TCPFlag> syn( f1, f1+1 );
        std::set<TCPService::TCPFlag> all_masks( f2 , f2+6 );

        if (srv->getAllTCPFlags()==syn  && srv->getAllTCPFlagMasks()==all_masks)
            str=" --syn ";
        else
        {
            str=" --tcp-flags ";
            bool first=true;

            if (srv->getAllTCPFlagMasks()==all_masks) str+="ALL";
            else
            {
                if (srv->getTCPFlagMask(TCPService::URG)) { if (!first) str+=","; str+="URG"; first=false; }
                if (srv->getTCPFlagMask(TCPService::ACK)) { if (!first) str+=","; str+="ACK"; first=false; }
                if (srv->getTCPFlagMask(TCPService::PSH)) { if (!first) str+=","; str+="PSH"; first=false; }
                if (srv->getTCPFlagMask(TCPService::RST)) { if (!first) str+=","; str+="RST"; first=false; }
                if (srv->getTCPFlagMask(TCPService::SYN)) { if (!first) str+=","; str+="SYN"; first=false; }
                if (srv->getTCPFlagMask(TCPService::FIN)) { if (!first) str+=","; str+="FIN"; first=false; }
            }

            str+=" ";

            if (srv->getAllTCPFlags()==none) str+="NONE";
            else
            {
                first=true;
                if (srv->getTCPFlag(TCPService::URG)) { if (!first) str+=","; str+="URG"; first=false; }
                if (srv->getTCPFlag(TCPService::ACK)) { if (!first) str+=","; str+="ACK"; first=false; }
                if (srv->getTCPFlag(TCPService::PSH)) { if (!first) str+=","; str+="PSH"; first=false; }
                if (srv->getTCPFlag(TCPService::RST)) { if (!first) str+=","; str+="RST"; first=false; }
                if (srv->getTCPFlag(TCPService::SYN)) { if (!first) str+=","; str+="SYN"; first=false; }
                if (srv->getTCPFlag(TCPService::FIN)) { if (!first) str+=","; str+="FIN"; first=false; }
            }
        }
    }
    return str;
}

/*
 * we made sure that all services in rel  represent the same protocol
 */
string PolicyCompiler_ipt::PrintRule::_printSrcService(RuleElementSrv  *rel)
{
    std::ostringstream  ostr;
/* I do not want to use rel->getFirst because it traverses the tree to
 * find the object. I'd rather use a cached copy in the compiler
 */
    FWObject *o=rel->front();
    if (o && FWReference::cast(o)!=NULL)
        o=compiler->getCachedObj( FWReference::cast(o)->getPointerId() );

    Service *srv= Service::cast(o);


    if (rel->size()==1) {
	if (UDPService::isA(srv) || TCPService::isA(srv)) {
	    string str=_printSrcPorts( srv );
	    if (! str.empty() ) 
            {
                ostr << " --source-port ";
                ostr  << _printSingleObjectNegation(rel) << str << " ";
            }
	}
    } else {
/* use multiport */

	string str;
	bool  first=true;
	for (FWObject::iterator i=rel->begin(); i!=rel->end(); i++) {
	    FWObject *o= *i;
//	    if (FWReference::cast(o)!=NULL) o=FWReference::cast(o)->getPointer();
	    if (FWReference::cast(o)!=NULL) o=compiler->getCachedObj(o->getStr("ref"));

	    Service *s=Service::cast( o );
	    assert(s);
	    if (UDPService::isA(srv) || TCPService::isA(srv)) {
		if (!first) str+=",";
		str+= _printSrcPorts( s );
		if (!str.empty()) first=false;
	    }
	}
	if ( !str.empty() ) 
        {
            string v=compiler->fw->getStr("version");
            if (v=="ge_1.2.6" || v=="1.2.9")
                ostr << " --source-ports ";
            else
                ostr << " --source-port ";

	    ostr << str << " ";
	}
    }
    return ostr.str();
}

string PolicyCompiler_ipt::PrintRule::_printDstService(RuleElementSrv  *rel)
{
    std::ostringstream  ostr;
    FWObject *o=rel->front();
    if (o && FWReference::cast(o)!=NULL)
        o=compiler->getCachedObj( FWReference::cast(o)->getPointerId() );

    Service *srv= Service::cast(o);

    if (rel->size()==1) 
    {
	if (UDPService::isA(srv) || TCPService::isA(srv)) 
        {
	    string str=_printDstPorts( srv );
	    if (! str.empty() )
            {
                ostr << " --destination-port ";
                ostr << _printSingleObjectNegation(rel) << str << " ";
            }
	}
	if (TCPService::isA(srv)) 
        {
	    string str=_printTCPFlags(TCPService::cast(srv));
	    if (!str.empty()) 
            {
                ostr  << _printSingleObjectNegation(rel)
                      << str << " ";
            }
	}
	if (ICMPService::isA(srv)) 
        {
	    string str=_printICMP( ICMPService::cast(srv) );
	    if (! str.empty() ) 
            {
                ostr << " --icmp-type " 
                     << _printSingleObjectNegation(rel)
                     << str << " ";
            }
	}
	if (IPService::isA(srv)) 
        {
	    string str=_printIP( IPService::cast(srv) );
	    if (! str.empty() ) 
            {
                ostr  << _printSingleObjectNegation(rel)
                      << str << " ";
            }
	}
	if (CustomService::isA(srv)) 
        {
	    ostr << _printSingleObjectNegation(rel)
                 << CustomService::cast(srv)->getCodeForPlatform( compiler->myPlatformName() ) << " ";
	}
    } else 
    {
/* use multiport */

	string str;
	for (FWObject::iterator i=rel->begin(); i!=rel->end(); i++) 
        {
	    FWObject *o= *i;
	    if (FWReference::cast(o)!=NULL) o=compiler->getCachedObj(o->getStr("ref"));

	    Service *s=Service::cast( o );
	    assert(s);
	    if (UDPService::isA(srv) || TCPService::isA(srv)) 
            {
		string str1 = _printDstPorts( s );
                if (str!="" && str1!="") str+=",";
                str+=str1;
	    }
	}
	if ( !str.empty() ) 
        {
            string v=compiler->fw->getStr("version");
            if (v=="ge_1.2.6" || v=="1.2.9")
                ostr << " --destination-ports ";
            else
                ostr << " --destination-port ";
	    ostr << str << " ";
	}
    }
    return ostr.str();
}

string PolicyCompiler_ipt::PrintRule::_printAddr(Address  *o)
{
    std::ostringstream  ostr;

    if (Interface::cast(o)!=NULL)
    {
        Interface *iface=Interface::cast(o);
        if (iface->isDyn())
            ostr << "$interface_" << iface->getName() << " ";
        return ostr.str();
    }

    IPAddress addr;
    Netmask   mask;
    try {
        addr=o->getAddress();

        if (Interface::cast(o)!=NULL || IPv4::cast(o)!=NULL) mask=Netmask("255.255.255.255");
        else            mask=o->getNetmask();
    }
    catch (FWException ex)  
    {
        FWObject *obj=o;
/*
 * check if this is object of class Address. since we want to
 * distinguish between Host, Interface and Address, and both Host and
 * Interface are inherited from Address, we can't use cast. Use isA
 * instead
 */
        while (obj!=NULL && 
               !Host::isA(obj) && 
               !Firewall::isA(obj)  && 
               !Network::isA(obj))  obj=obj->getParent();

        compiler->error(_("Problem with address or netmask in the object or one of its interfaces: '")+obj->getName()+"'");
        throw;
    }


    if (addr.toString()=="0.0.0.0" && mask.toString()=="0.0.0.0") 
    {
        ostr << "0/0 ";
    } else 
    {
        ostr << addr.toString();
        if (mask.toString()!="255.255.255.255") 
        {
            ostr << "/" << mask.getLength();
        }
        ostr << " ";
    }
    return ostr.str();
}


string PolicyCompiler_ipt::PrintRule::_printSingleObjectNegation(RuleElement *rel)
{
    if (rel->getBool("single_object_negation"))   return "! ";
    else return "";
}

string PolicyCompiler_ipt::PrintRule::_printTimeInterval(PolicyRule *r)
{
    std::ostringstream  ostr;

    RuleElementInterval* ri=r->getWhen();
    if (ri==NULL || ri->isAny()) return "";

    std::map<int,std::string>  daysofweek;

    daysofweek[0]="Sun";
    daysofweek[1]="Mon";
    daysofweek[2]="Tue";
    daysofweek[3]="Wed";
    daysofweek[4]="Thu";
    daysofweek[5]="Fri";
    daysofweek[6]="Sat";

    bool first;
    int smin, shour, sday, smonth, syear, sdayofweek;
    int emin, ehour, eday, emonth, eyear, edayofweek;

    Interval *interval=compiler->getFirstWhen(r);
    assert(interval!=NULL);

    interval->getStartTime( &smin, &shour, &sday, &smonth, &syear, &sdayofweek);
    interval->getEndTime(   &emin, &ehour, &eday, &emonth, &eyear, &edayofweek);

    ostr << "-m time ";

    if (shour<0) shour=0;
    if (smin<0)  smin=0;

    if (ehour<0) ehour=23;
    if (emin<0)  emin=59;

    ostr << " --timestart "
         << setw(2) << setfill('0') << shour << ":"
         << setw(2) << setfill('0') << smin  << " ";
    ostr << " --timestop "
         << setw(2) << setfill('0') << ehour << ":"
         << setw(2) << setfill('0') << emin  << " ";

    if (sdayofweek<0) sdayofweek=0;
    if (edayofweek<0) edayofweek=6;

    ostr << " --days ";
    first=true;
    for (int i=sdayofweek; i<=edayofweek; ++i) {
        if (!first) ostr << ",";
        first=false;
        ostr << daysofweek[i];
    }

    return ostr.str();
}

PolicyCompiler_ipt::PrintRule::PrintRule(const std::string &name) : PolicyRuleProcessor(name) 
{ 
    init=true; 
    print_once_on_top=true;

    chains["INPUT"]  =true;
    chains["OUTPUT"] =true;
    chains["FORWARD"]=true;
    chains["RETURN"] =true;
    chains["LOG"]    =true;
    chains["ACCEPT"] =true;
    chains["DROP"]   =true;
    chains["REJECT"] =true;

}


bool  PolicyCompiler_ipt::PrintRule::processNext()
{
    PolicyRule         *rule    =getNext(); 
    if (rule==NULL) return false;

    tmp_queue.push_back(rule);


    if (print_once_on_top)
    {
        if ( compiler->getCachedFwOpt()->getBool("log_all_dropped") ) 
        {
            compiler->output << "$IPTABLES -t drop -A DROPPING";

            if ( compiler->getCachedFwOpt()->getBool("use_ULOG") )
                compiler->output << " -j ULOG ";
            else
                compiler->output << " -j LOG ";

            compiler->output << _printGlobalLogParameters() << endl;
            compiler->output << endl;
        }

        if ( compiler->getCachedFwOpt()->getBool("clamp_mss_to_mtu") ) 
        {
            compiler->output << "$IPTABLES -A FORWARD -p tcp --tcp-flags SYN,RST SYN -j TCPMSS --clamp-mss-to-pmtu" 
                             << endl;
            compiler->output << endl;
        }

        if ( compiler->getCachedFwOpt()->getBool("accept_established") ) 
        {
            compiler->output << "$IPTABLES -A INPUT   -m state --state ESTABLISHED,RELATED -j ACCEPT"
                             << endl;
            compiler->output << "$IPTABLES -A OUTPUT  -m state --state ESTABLISHED,RELATED -j ACCEPT"
                             << endl;
            compiler->output << "$IPTABLES -A FORWARD -m state --state ESTABLISHED,RELATED -j ACCEPT"
                             << endl;
            compiler->output << endl;
        }

        if ( ! compiler->getCachedFwOpt()->getBool("accept_new_tcp_with_no_syn") ) 
        {
            compiler->output << "# dropping TCP sessions opened prior firewall restart" 
                             << endl;
            compiler->output << "#" 
                             << endl;
            compiler->output << "$IPTABLES -A INPUT   -p tcp ! --syn -m state --state NEW -j DROP" 
                             << endl;
            compiler->output << "$IPTABLES -A OUTPUT  -p tcp ! --syn -m state --state NEW -j DROP" 
                             << endl;
            compiler->output << "$IPTABLES -A FORWARD -p tcp ! --syn -m state --state NEW -j DROP" 
                             << endl;
            compiler->output << endl;
        }


        print_once_on_top=false;
    }




    string rl=rule->getLabel();
    if (rl!=current_rule_label) {
        compiler->output << "# " << endl;
        compiler->output << "# Rule " << rl << endl;
        compiler->output << "# " << endl;

        string    comm=rule->getComment();
        string::size_type c1,c2;
        c1=0;
        while ( (c2=comm.find('\n',c1))!=string::npos ) {
            compiler->output << "# " << comm.substr(c1,c2-c1) << endl;
            c1=c2+1;
        }
        compiler->output << "# " << comm.substr(c1) << endl;
        compiler->output << "# " << endl;

        current_rule_label=rl;
    }

/*
 *  check and create new chain if needed
 */
    string s;

    s=rule->getStr("ipt_chain");
    if ( ! chains[s] ) {
	compiler->output << "$IPTABLES -N " << s << endl;
	chains[s]=true;
    }

    s=rule->getStr("ipt_target");
    if ( ! chains[s] ) {
	compiler->output << "$IPTABLES -N " << s << endl;
	chains[s]=true;
    }

    string  command_line = PolicyRuleToString(rule);

/* if anywhere in command_line we used variable holding an address of
 * dynamic interface (named $interface_something) then we need to add
 * this command with a check for the value of this variable. We execute
 * iptables command only if the value is a non-empty string.
 */

    string::size_type p1=command_line.find("$interface_");
    string  iface_name;
    string  iface_var;
    if ( p1!=string::npos )
    {
        string::size_type p2=command_line.find(" ",p1);
        string::size_type p3=command_line.find("_",p1) +1;
        iface_name=command_line.substr(p3,p2-p3);
        iface_var= command_line.substr(p1,p2-p1);

/* if interface name ends with '*', this is a wildcard interface. */
        string::size_type p4;
        if ((p4=iface_name.find("*"))!=string::npos)
        {
            string cmdline=command_line;
            string iface_family_name=iface_name.substr(0,p4);
            compiler->output << "getinterfaces " << iface_family_name << " | while read I; do" << endl;
            compiler->output << "  getaddr $I \"interface_$I\"" << endl;
            compiler->output << "  cmd=\"$\"\"interface_\"$I"   << endl;
            compiler->output << "  eval \"addr=$cmd\""          << endl;
            cmdline.replace(p1,p2-p1,"$addr");
            compiler->output << "  test -n \"$addr\" && " << cmdline;
            compiler->output << "done" << endl;

            return true;
        } else
            compiler->output << "test -n \"" << iface_var << "\" && ";
    }

    compiler->output << command_line;

    return true;
}

string PolicyCompiler_ipt::PrintRule::PolicyRuleToString(PolicyRule *rule)
{

    FWOptions *ruleopt =rule->getOptionsObject();
    FWObject    *ref;

    RuleElementSrc *srcrel=rule->getSrc();
    ref=srcrel->front();
    Address        *src=Address::cast(compiler->getCachedObj(ref->getStr("ref")));
    if(src==NULL)
        throw FWException(_("Broken SRC in ")+rule->getLabel());

    RuleElementDst *dstrel=rule->getDst();
    ref=dstrel->front();
    Address        *dst=Address::cast(compiler->getCachedObj(ref->getStr("ref")));
    if(dst==NULL)
        throw FWException(_("Broken DST in ")+rule->getLabel());

    RuleElementSrv *srvrel=rule->getSrv();
    ref=srvrel->front();
    Service        *srv=Service::cast(compiler->getCachedObj(ref->getStr("ref")));
    if(srv==NULL)
        throw FWException(_("Broken SRV in ")+rule->getLabel());


    std::ostringstream  command_line;


    command_line << "$IPTABLES -A ";

    command_line << _printChain(rule);
    command_line << _printDirectionAndInterface(rule);
    command_line << _printProtocol(srv);
    command_line << _printMultiport(rule);

    if (!src->isAny()) 
    {
        string physaddress="";

        if (physAddress::isA(src))
            physaddress= physAddress::cast(src)->getPhysAddress();

        if (combinedAddress::isA(src))
            physaddress= combinedAddress::cast(src)->getPhysAddress();

        if ( ! physaddress.empty())
        {
	    command_line << " -m mac --mac-source " << _printSingleObjectNegation(srcrel);
            command_line << physaddress;
        }
/*
 * fool-proof: this is last resort check for situation when user created IPv4 object
 * for the interface but left it with empty address ( 0.0.0.0 ). 
 */
        if ( ! physaddress.empty() && src->getAddress()==IPAddress("0.0.0.0"))
        {
            ;
        } else
        {
            command_line << " -s " << _printSingleObjectNegation(srcrel);
            command_line << _printAddr(src);
        }
    }
    command_line << _printSrcService(srvrel);

    if (!dst->isAny()) 
    {
	command_line << " -d " << _printSingleObjectNegation(dstrel);
	command_line << _printAddr(dst);
    }
    command_line << _printDstService(srvrel);

/* keeping state does not apply to deny/reject 
   however some rules need state check even if action is Deny

    if ( ! ruleopt->getBool("stateless") && rule->getAction()==PolicyRule::Accept) 
    {
	command_line << " -m state --state NEW ";
    }

    if (rule->getBool("force_state_check"))
	command_line << " -m state --state NEW ";
*/

    if ((!ruleopt->getBool("stateless") && rule->getStr("stored_action")=="Accept") ||
         rule->getBool("force_state_check") )
    {
	command_line << " -m state --state NEW ";
    }

    command_line << _printTimeInterval(rule);
    command_line << _printTarget(rule);

    command_line << endl;

    return command_line.str();
}


