/* 

                          Firewall Builder

                 Copyright (C) 2005, 2006 NetCitadel, LLC

  Author:  Illiya Yalovoy <yalovoy@gmail.com>

  $Id: DiscoveryDruid.cpp,v 1.27 2006/09/18 06:59:52 vkurland Exp $

  This program is free software which we release under the GNU General Public
  License. You may redistribute and/or modify this program under the terms
  of that license as published by the Free Software Foundation; either
  version 2 of the License, or (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
 
  To get a copy of the GNU General Public License, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/

#include "config.h"
#include "global.h"
#include "utils.h"

#include <qradiobutton.h>
#include <qlineedit.h>
#include <qcheckbox.h>
#include <qcombobox.h>
#include <qspinbox.h>
#include <qlistbox.h>
#include <qtimer.h>
#include <qregexp.h>
#include <qlabel.h>
#include <qtextedit.h>
#include <qpushbutton.h>
#include <qfile.h>
#include <qfiledialog.h>
#include <qapplication.h>
#include <qprogressdialog.h>
#include <qlistview.h>
#include <qtextstream.h>
#include <qiodevice.h>
#include <qhostaddress.h>
#include <qdns.h>
#include <qbuttongroup.h>
#include <qmessagebox.h> 

#include "DiscoveryDruid.h"

#include <iostream>
#include <map>
#include <vector>
#include <set>

#include "fwbuilder/HostsFile.h"
#include "fwbuilder/IPv4.h"
#include "fwbuilder/Host.h"
#include "fwbuilder/Network.h"
#include "fwbuilder/IPAddress.h"
#include "fwbuilder/Firewall.h"

#include "fwbuilder/dns.h"
#include "fwbuilder/snmp.h"

#include "FWBSettings.h"
#include "ObjectManipulator.h"
#include "FWWindow.h"


using namespace std;
using namespace libfwbuilder;

DiscoveryDruid::DiscoveryDruid(QWidget *parent) : DiscoveryDruid_q(parent)
{
    thread=NULL;
    
    timer=new QTimer(this);
    prg_timer=new QTimer(this);
    unBar=NULL;
    unProg=0;
    
    connect(prg_timer,SIGNAL(timeout()),this,SLOT(updatePrg()));
    
    setDiscoveryMethod_file();
    
    flt_obj     = new Filter();
    flt_obj_d   = new FilterDialog(this);
    flt_obj_d->setFilter(flt_obj);
     
    flt_last    = new Filter();
    flt_last_d  = new FilterDialog(this);
    flt_last_d->setFilter(flt_last);
    
    flt_net     = new Filter();
    flt_net_d   = new FilterDialog(this);
    flt_net_d->setFilter(flt_net);

    helpButton()->hide();

    fillLibraries(libs,mw->db());
    libs->setCurrentText(om->getCurrentLib()->getName().c_str());

    typeChangingList->setColumnAlignment (1,Qt::AlignHCenter);
    typeChangingList->setColumnWidthMode(0, QListView::Maximum);

    DNSprogress->hide();
    DNSprogress_2->hide();

#ifndef HAVE_GOODLIBRESOLV
    dm_importdns->hide();
    snmpdnsparameters->hide();
#endif
    
#ifndef HAVE_LIBSNMP
    dm_usesnmp->setEnabled(false);
#endif

    dns=NULL;
    
    restore();
    prg_timer->start(100);
}
DiscoveryDruid::~DiscoveryDruid()
{
    delete flt_obj;
    delete flt_last;
    delete flt_net;
    delete flt_obj_d;
    delete flt_last_d;
    delete flt_net_d;
    if (dns!=NULL) delete dns;

    save();
}
const char * DISCOVERY_DRUID_PREFIX="DiscoveryDruid/";

const char * DISCOVERY_DRUID_DISCOVERYMETHOD="DiscoveryMethod";
const char * DISCOVERY_DRUID_FILENAME       ="Filename";
const char * DISCOVERY_DRUID_DOMAINNAME     ="Domainname";
const char * DISCOVERY_DRUID_USELONGNAME    ="UseLongName";
const char * DISCOVERY_DRUID_NAMESERVER     ="NameServer";
const char * DISCOVERY_DRUID_DNSTIMEOUT     ="DNSTimeout";
const char * DISCOVERY_DRUID_DNSRETRIES     ="DNSRetries";
const char * DISCOVERY_DRUID_SEEDHOST       ="SeedHost";
const char * DISCOVERY_DRUID_SNMPINADDR     ="SNMPInAddr";
const char * DISCOVERY_DRUID_SNMPINMASK     ="SNMPInMask";
const char * DISCOVERY_DRUID_SNMPRECURSIVE  ="SNMPRecursive";
const char * DISCOVERY_DRUID_SNMPFOLLOWP2P  ="SNMPFollowP2P";
const char * DISCOVERY_DRUID_SNMPINCLUDEVIRT="SNMPIncludeVirt";
const char * DISCOVERY_DRUID_SNMPDODNS      ="SNMPDoDNS";
const char * DISCOVERY_DRUID_SNMPCOMMUNITY  ="SNMPCommunity";
const char * DISCOVERY_DRUID_SNMPRETRIES    ="SNMPRetries";
const char * DISCOVERY_DRUID_SNMPTIMEOUT    ="SNMPTimeout";
const char * DISCOVERY_DRUID_SNMPDNSRETRIES ="DNSRetries";
const char * DISCOVERY_DRUID_SNMPDNSTIMEOUT ="DNSTimeout";
const char * DISCOVERY_DRUID_SNMPDNSTHREADS ="SNMPDnsThreads";


void DiscoveryDruid::restore()
{
    int i;
    QString s;
    //Restore from settings
    dm_buttonGroup->setButton(st->getInt(
                QString(DISCOVERY_DRUID_PREFIX) + DISCOVERY_DRUID_DISCOVERYMETHOD));
    changedDiscoveryMethod(dm_buttonGroup->id(dm_buttonGroup->selected()));
    //filename->setText(st->getStr(
    //            QString(DISCOVERY_DRUID_PREFIX) + DISCOVERY_DRUID_FILENAME));
    //domainname->setText(st->getStr(
    //            QString(DISCOVERY_DRUID_PREFIX) + DISCOVERY_DRUID_DOMAINNAME));
    uselongname->setChecked(st->getBool(
                QString(DISCOVERY_DRUID_PREFIX) + DISCOVERY_DRUID_USELONGNAME));
    //nameserverline->setText(st->getStr(
    //            QString(DISCOVERY_DRUID_PREFIX) + DISCOVERY_DRUID_NAMESERVER));
    i=st->getInt(QString(DISCOVERY_DRUID_PREFIX) + DISCOVERY_DRUID_DNSTIMEOUT);
    dnstimeout->setValue((i)?i:2);
    i=st->getInt(QString(DISCOVERY_DRUID_PREFIX) + DISCOVERY_DRUID_DNSRETRIES);
    dnsretries->setValue((i)?i:1);
    //seedhostname->setText(st->getStr(
    //            QString(DISCOVERY_DRUID_PREFIX) + DISCOVERY_DRUID_SEEDHOST));
    snmpinaddr->setText(st->getStr(
                QString(DISCOVERY_DRUID_PREFIX) + DISCOVERY_DRUID_SNMPINADDR));
    snmpinmask->setText(st->getStr(
                QString(DISCOVERY_DRUID_PREFIX) + DISCOVERY_DRUID_SNMPINMASK));
    snmprecursive->setChecked(st->getBool(
                QString(DISCOVERY_DRUID_PREFIX) + DISCOVERY_DRUID_SNMPRECURSIVE));
    snmpfollowp2p->setChecked(st->getBool(
                QString(DISCOVERY_DRUID_PREFIX) + DISCOVERY_DRUID_SNMPFOLLOWP2P));
    snmpincludevirt->setChecked(st->getBool(
                QString(DISCOVERY_DRUID_PREFIX) + DISCOVERY_DRUID_SNMPINCLUDEVIRT));
    snmpdodns->setChecked(st->getBool(
                QString(DISCOVERY_DRUID_PREFIX) + DISCOVERY_DRUID_SNMPDODNS));
    s=st->getStr(QString(DISCOVERY_DRUID_PREFIX) + DISCOVERY_DRUID_SNMPCOMMUNITY);
    snmpcommunity->setText((s.isEmpty())?"public":s);
    i=st->getInt(QString(DISCOVERY_DRUID_PREFIX) + DISCOVERY_DRUID_SNMPRETRIES);
    snmpretries->setValue((i)?i:1);
    i=st->getInt(QString(DISCOVERY_DRUID_PREFIX) + DISCOVERY_DRUID_SNMPTIMEOUT);
    snmptimeout->setValue((i)?i:2);
    i=st->getInt(QString(DISCOVERY_DRUID_PREFIX) + DISCOVERY_DRUID_SNMPDNSRETRIES);
    snmpdnsretries->setValue((i)?i:1);
    i=st->getInt(QString(DISCOVERY_DRUID_PREFIX) + DISCOVERY_DRUID_SNMPDNSTIMEOUT);
    snmpdnstimeout->setValue((i)?i:2);
    i=st->getInt(QString(DISCOVERY_DRUID_PREFIX) + DISCOVERY_DRUID_SNMPDNSTHREADS);
    snmpdnsthreads->setValue((i)?i:5);
}

void DiscoveryDruid::save()
{
    // Save to settings
    st->setInt(
            QString(DISCOVERY_DRUID_PREFIX) + DISCOVERY_DRUID_DISCOVERYMETHOD,
            dm_buttonGroup->id(dm_buttonGroup->selected()));
    st->setBool(
            QString(DISCOVERY_DRUID_PREFIX) + DISCOVERY_DRUID_USELONGNAME,
            uselongname->isChecked());
    if (Task==BT_DNS)
    {
        st->setInt(
                QString(DISCOVERY_DRUID_PREFIX) + DISCOVERY_DRUID_DNSTIMEOUT,
                dnstimeout->value());
        st->setInt(
                QString(DISCOVERY_DRUID_PREFIX) + DISCOVERY_DRUID_DNSRETRIES,
                dnsretries->value());
    }
    else
    {
        st->setInt(
                QString(DISCOVERY_DRUID_PREFIX) + DISCOVERY_DRUID_SNMPDNSRETRIES,
                snmpdnsretries->value());
        st->setInt(
                QString(DISCOVERY_DRUID_PREFIX) + DISCOVERY_DRUID_SNMPDNSTIMEOUT,
                snmpdnstimeout->value());
    }
    //st->setStr(
    //        QString(DISCOVERY_DRUID_PREFIX) + DISCOVERY_DRUID_SEEDHOST,
    //        seedhostname->text());
    st->setStr(
            QString(DISCOVERY_DRUID_PREFIX) + DISCOVERY_DRUID_SNMPINADDR,
            snmpinaddr->text());
    st->setStr(
            QString(DISCOVERY_DRUID_PREFIX) + DISCOVERY_DRUID_SNMPINMASK,
            snmpinmask->text());
    st->setBool(
            QString(DISCOVERY_DRUID_PREFIX) + DISCOVERY_DRUID_SNMPRECURSIVE,
            snmprecursive->isChecked());
    st->setBool(
            QString(DISCOVERY_DRUID_PREFIX) + DISCOVERY_DRUID_SNMPFOLLOWP2P,
            snmpfollowp2p->isChecked());
    st->setBool(
            QString(DISCOVERY_DRUID_PREFIX) + DISCOVERY_DRUID_SNMPINCLUDEVIRT,
            snmpincludevirt->isChecked());
    st->setBool(
            QString(DISCOVERY_DRUID_PREFIX) + DISCOVERY_DRUID_SNMPDODNS,
            snmpdodns->isChecked());
    st->setStr(
            QString(DISCOVERY_DRUID_PREFIX) + DISCOVERY_DRUID_SNMPCOMMUNITY,
            snmpcommunity->text());
    st->setInt(
            QString(DISCOVERY_DRUID_PREFIX) + DISCOVERY_DRUID_SNMPRETRIES,
            snmpretries->value());
    st->setInt(
            QString(DISCOVERY_DRUID_PREFIX) + DISCOVERY_DRUID_SNMPTIMEOUT,
            snmptimeout->value());
    st->setInt(
            QString(DISCOVERY_DRUID_PREFIX) + DISCOVERY_DRUID_SNMPDNSTHREADS,
            snmpdnsthreads->value());
}

void DiscoveryDruid::dnsFinish()
{
    QValueList<QHostAddress> list = dns->addresses();
    
    delete dns;
    dns=NULL;
    unBar->hide();
    
    if (userIsTyping)
    {
        //abandon the test result
        if (Task==BT_DNS)
        {
            changedNameServer();
        }
        else
        {
            changedSeedHost();
        }
    }
    else
    {
        //get the test result
        if (list.isEmpty())
        {
            errMessage->setPaletteForegroundColor(Qt::darkRed);
            errMessage->setText( "host name not found");
            isSeedHostOK=false;
        }
        else
        {
            errMessage->setPaletteForegroundColor(Qt::darkGreen);
            errMessage->setText( "host name verified");
            isSeedHostOK=true;
            
        }
        nextButton()->setEnabled(isSNMPInclNetOK && isSeedHostOK); 
    }
    
}

void DiscoveryDruid::changedSelected( const QString &s )
{
    QWidget *w=currentPage();
    int page=indexOf(w);
    switch (page)
    {
    
    case 1: // Reading file in hosts format 
        {
            setNextEnabled(w,false);
            changedHostsFileName();
            filename->setFocus();
            
            break;
        }
    case 2: // Import DNS zone
        {
            changedDomainName();
            domainname->setFocus();
            //setNextEnabled(w,false);
            break;
        }
    case 3: // Name server
        {
            if (page>FromPage)
                getNameServers();
            disconnect(timer,SIGNAL(timeout()),0,0);
            connect(timer,SIGNAL(timeout()),this,SLOT(checkHostName()));
            changedNameServer();
            nameserverline->setFocus();

            //setNextEnabled(w,false);
            break;
        }
    case 4: // Network discovery using SNMP
        {
            disconnect(timer,SIGNAL(timeout()),0,0);
            connect(timer,SIGNAL(timeout()),this,SLOT(checkHostName()));

            isSeedHostOK=false;
            isSNMPInclNetOK=false;
            
            changedSeedHost();
            changedInclNet();
            seedhostname->setFocus();
            break;
        }
    case 5: // Network scan options
        {
            snmprecursive->setFocus();
            //setNextEnabled(w,false);
            break;
        }
    case 6: // SNMP and DNS reverse lookup queries parameters
        {
            checkSNMPCommunity();
            snmpcommunity->setFocus();
            break;
        }
    case 7: // Background process
        {
            discoveryprogress->setProgress(-1);
            discoverylog->clear();
            discoveryStopButton->setEnabled(true);
            logSaveButton->setEnabled(false);
            setNextEnabled(w,false);
            cancelButton()->hide();
            setBackEnabled(w,false);
            disconnect(timer,SIGNAL(timeout()),0,0);
            connect(timer,SIGNAL(timeout()),this,SLOT(updateLog()));
            timer->start(1000,false);
            
            startBackgroundProcess();
            
            break;
        }
    case 8: // Networks
        {
            fillListOfNetworks();
            fillNetworks();
            backButton()->setEnabled(false);
            nextButton()->setEnabled(networklist->count ()>0 || Objects.size()>0);
            break;
        }
    case 9: // Objects
        {
            if (Networks.size()==0)
                setBackEnabled(w,false);
            
            fillListOfObjects();
            fillObjects();
            nextButton()->setEnabled(objectlist->count ()>0 || networklist->count()>0);
            break;
        }
    case 10: // Adjust Object type
        {
            setBackEnabled(w,true);
            fillTypeChangingList();
            break;
        }
    case 11: // Target library
        {
            break;
        }
    case 12: // Objects creation ...
        {
            setBackEnabled(w,false);
            cancelButton()->hide();
            createRealObjects();
            setFinishEnabled(w,true);
            finishButton()->setFocus();
            break;
        }
        
    default : {}
    
    }
    FromPage=page;
}

void DiscoveryDruid::startBackgroundProcess()
{
    switch (Task)
    {
        case BT_HOSTS:
        {
            discoveryprogress->setTotalSteps(100);
            discoveryprogress->setProgress(0);
            discoveryStopButton->setEnabled(false);
            startHostsScan();
            break;
        }
        case BT_DNS:
        {
            discoveryprogress->setTotalSteps(0);
            discoveryprogress->setProgress(-1);
            startDNSScan();
            break;
        }
        case BT_SNMP:
        {
            discoveryprogress->setTotalSteps(0);
            discoveryprogress->setProgress(-1);
            startSNMPScan();
            break;
        }
        default:
        {}
    }
}

void DiscoveryDruid::browseHostsFile()
{
    QString dir;
    dir=st->getWDir();
    if (dir.isEmpty())  dir=st->getOpenFileDir();
    if (dir.isEmpty())  dir="~";
    
    QString s = QFileDialog::getOpenFileName(
                    dir,
                    "All files (*.*)",
                    this,
                    "open file dialog",
                    "Choose a file" );
    
    if (!s.isEmpty())
    {
        filename->setText(s);
    }
    
}

void DiscoveryDruid::updatePrg()
{
    if (unBar!=NULL)
    {
        unBar->setProgress(unProg++);
    }
    
}

void DiscoveryDruid::setDiscoveryMethod_file()
{
    Task=BT_HOSTS;
    processname->setText(tr("Hosts file parsing ..."));
    for (int i=0;i<WIZARD_PAGES;i++)
    {
        
        setAppropriate(page(i),WIZARD_FILE_PAGES[i]);
    }
}

void DiscoveryDruid::getNameServers()
{
    multimap<string,libfwbuilder::IPAddress> ns_records;

    string domain_name=domainname->text().latin1();
    DNS_getNS_query *dns=new DNS_getNS_query(domain_name);
    int              n;
    try 
    {
        NullLogger nl;
        SyncFlag   stop_program(false);
        ns_records=dns->getNS(domain_name, &nl, &stop_program);
        dnsfromlist->setChecked(true);

    } catch (FWException &ex) 
    {
        //string(_("Could not find name servers for the domain: '"))+
        //domain_name+"'  ", ex.toString(), this);
        delete dns;
        nameserverlist->setEnabled(false);
        dnsfromlist->setEnabled(false);
        dnscustom->setChecked(true);
        return ;
    }
    multimap<string,IPAddress>::iterator i;
    nameserverlist->clear();
    NameServers.clear();
    
    for (n=0,i=ns_records.begin(); i!=ns_records.end(); ++n,++i) 
    {

        string s = (*i).first + "  (" + ((*i).second).toString() + ")";
        QString qs = s.c_str();
        nameserverlist->insertItem(qs);
    
        IPAddress *na=new IPAddress( (*i).second );
        NameServers[qs] = *na;
    }
}

void DiscoveryDruid::setDiscoveryMethod_DNS()
{
    processname->setText(tr("DNS zone transfer ..."));
    Task=BT_DNS;
    for (int i=0;i<WIZARD_PAGES;i++)
    {
        
        setAppropriate(page(i),WIZARD_DNS_PAGES[i]);
    }
}

void DiscoveryDruid::setDiscoveryMethod_SNTP()
{
    processname->setText(tr("Network discovery using SNMP ..."));
    Task=BT_SNMP;
    for (int i=0;i<WIZARD_PAGES;i++)
    {
        
        setAppropriate(page(i),WIZARD_SNMP_PAGES[i]);
    }    
}

void DiscoveryDruid::changedDiscoveryMethod(int c)
{
   switch (c)
   {
       case 0: 
           {
               setDiscoveryMethod_file();
               break;
           }
       case 1: 
           {
               setDiscoveryMethod_DNS();
               break;
           }
       case 2: 
           {
               setDiscoveryMethod_SNTP();
               break;
           }
       default : {}
   }
}

void DiscoveryDruid::saveScanLog()
{
    QString dir;
    dir=st->getWDir();
    if (dir.isEmpty())  dir=st->getOpenFileDir();
    if (dir.isEmpty())  dir="~";
    
    QString s = QFileDialog::getSaveFileName(
                    dir,
                    "Text file (*.txt)",
                    this,
                    "save file dialog",
                    "Choose a file" );
    
    
    if (!s.isEmpty())
    {
        if (s.findRev(".txt",-1,false)==-1)
        {
            s+=".txt";
        }
        QFile f(s);
        if (f.open(IO_WriteOnly))
        {
            if (fwbdebug)
            {
                qDebug("Saving crawler log to file: %d chars, %d lines, %d paragraphs",
                       discoverylog->length(),
                       discoverylog->lines(),
                       discoverylog->paragraphs());
                qDebug("--------------------------------");
            }
            QTextStream strm(&f);
            for (int p=0; p<discoverylog->paragraphs(); ++p)
            {
                QString txt = discoverylog->text(p);
                strm << txt << endl;
                if (fwbdebug) qDebug("%s",txt.ascii());
            }
            if (fwbdebug)
                qDebug("--------------------------------");
            f.close();
        }
    }    
}
    
void DiscoveryDruid::startHostsScan()
{
    if (thread!=NULL)
    {
        delete thread;
    }
    
    thread=new WorkerThread(BT_HOSTS);
    thread->setFileName( filename->text() );
    thread->setTargetWidget(this);
        
    thread->start();
    
}

IPAddress DiscoveryDruid::getNS()
{
    string ns;
    if (dnscustom->isChecked()) 
    {
        ns=nameserverline->text().latin1();  

        try 
        {
            return IPAddress(ns);
        } catch (FWException &ex) 
        {
        /* perhaps not address but host name */
            list<IPAddress> addr;
            try 
            {
                addr=DNS::getHostByName(ns);
            } catch (FWException &ex) 
            {
                return IPAddress();
            }

            return addr.front();
        }
    }

    return NameServers[nameserverlist->currentText()];
}

void DiscoveryDruid::startDNSScan()
{
    IPAddress ns=getNS(); 
    string domain_name=domainname->text().latin1();
    
    DNS_findA_query *q=new DNS_findA_query();
    q->init(
            domain_name, ns,
            dnsretries->value(),
            dnstimeout->value()
            );
    bop=q;
    
    discoveryprogress->setTotalSteps(0);
    unBar=discoveryprogress;
    try
    {
        logger=bop->start_operation();
    
        discoverylog->append("Reading DNS zone ...\n");

    } catch(const FWException &ex)
    {
        delete q;
        q=NULL;
        qDebug(ex.toString().c_str());
    }
}

IPAddress DiscoveryDruid::getSeedHostAddress()
{
    libfwbuilder::IPAddress   seed_host_addr;
    if (!seedhostname->text().isEmpty())
    {
        try
        {
            seed_host_addr=IPAddress(seedhostname->text().latin1());
            return seed_host_addr;
        } catch(const FWException &ex) 
        {
        }        
        
        try 
        {
            QString a = getAddrByName( seedhostname->text() );
            return IPAddress( a.latin1() );
#if 0
            list<IPAddress> v=DNS::getHostByName( seedhostname->text().latin1() );
            seed_host_addr = v.front();
            return seed_host_addr;
#endif
        } catch(const FWException &ex) 
        {
        }        
    }
    return seed_host_addr;
}

void DiscoveryDruid::startSNMPScan()
{
#ifdef HAVE_LIBSNMP   


    bool use_incl=!snmpinaddr->text().isEmpty() && !snmpinmask->text().isEmpty();
    if (use_incl)
    {
        try
        {
            IPNetwork in(
                 IPAddress(snmpinaddr->text().latin1()),
                 Netmask(snmpinmask->text().latin1()) 
                 );
            include_networks.push_back(in);
        }
        catch (const FWException &ex)
        {
            //TODO: to do something usefull
        }
    }
    libfwbuilder::SNMPCrawler *q=new SNMPCrawler();
    q->init(getSeedHostAddress(),
            snmpcommunity->text().latin1(),
            snmprecursive->isChecked(),
            ! snmpincludevirt->isChecked(),
            false,
            snmpfollowp2p->isChecked(),
            0,
            snmpretries->value(),
            1000000L*snmptimeout->value(),
            0,
            0,
            (use_incl) ? &include_networks : NULL);
    
    discoveryprogress->setTotalSteps(0);
    unBar=discoveryprogress;
    
    bop=q;
    try
    {
        logger=bop->start_operation();
        discoverylog->append("Collecting data ...\n");
        
    } catch(const FWException &ex)
    {
        delete q;
        q=NULL;
    }
            
    
#endif
}
    
void DiscoveryDruid::changedDomainName()
{
    if (domainname->text().isEmpty())
    {
        nextButton()->setEnabled(false);
    }
    else
    {
        nextButton()->setEnabled(true);
    }
}

void DiscoveryDruid::changedNameServer()
{
    userIsTyping=true;
    isSNMPInclNetOK=true;

    if(dnscustom->isChecked())
    {
        nextButton()->setEnabled(false);
        QString s=nameserverline->text();
        HostName=s;
        
        if (s.isEmpty())
        {
            timer->stop();
            DNSprogress_2->hide();
            nameserver_error->setPaletteForegroundColor(Qt::darkRed);
            nameserver_error->setText("Enter valid host name or address.");
            nextButton()->setEnabled(false);
            return;
        }
        
        if(isIPAddress(s))
        {
            timer->stop();
            DNSprogress_2->hide();
            
            QString rs=testIPAddress(s);
            if (rs.isEmpty())
            {
                nameserver_error->setText(" ");
                nextButton()->setEnabled(true);
            }
            else
            {
                nameserver_error->setPaletteForegroundColor(Qt::darkRed);
                nameserver_error->setText(rs);
                nextButton()->setEnabled(false);
            }
        }
        else
        {
            unBar=DNSprogress_2;
            unBar->show();
            timer->start(1000,true);
            errMessage=nameserver_error;
            userIsTyping=false;
            errMessage->setPaletteForegroundColor(Qt::black);
            errMessage->setText("DNS resolution in progress...");
        }
    }
    else
    {
        timer->stop();
        DNSprogress_2->hide();
        nameserver_error->setText(" ");
        nextButton()->setEnabled(true);
    }
}

void DiscoveryDruid::typedCustomNS()
{
    if(!dnscustom->isChecked())
    {
        dnscustom->setChecked(true);
    }
}

bool DiscoveryDruid::isIPAddress(const QString s)
{
    QRegExp r=QRegExp("^(\\d|\\.)+$",false,false);
    return r.exactMatch(s);
}

QString DiscoveryDruid::testIPAddress(const QString s)
{
    QString res;
    QRegExp r=QRegExp("^\\d{1,3}\\.\\d{1,3}\\.\\d{1,3}\\.\\d{1,3}$",false,false);
    if (r.exactMatch(s))
    {
        try
        {
            IPAddress(s.latin1());
        } catch(const FWException &ex) 
        {
            res=ex.toString().c_str();
        }
    }
    else
    {
        res="Wrong IPv4 format";
    }
    return res;
}

void DiscoveryDruid::changedHostsFileName()
{
    QFile f;
    f.setName(filename->text());
    if (f.exists())
    {
        setNextEnabled(currentPage(),true);
    }
    else
    {
        setNextEnabled(currentPage(),false);
    }
}
    
void DiscoveryDruid::changedSNMPOptions()
{

}
    
void DiscoveryDruid::stopBackgroundProcess()
{
    if (fwbdebug)
        qDebug("stopBackgroundProcess bop=%p  isRunning=%d",
               bop,(bop!=NULL)?bop->isRunning():-1);

    if (bop!=NULL && bop->isRunning())
    {
        discoverylog->append("Terminating task. Please wait...\n");
        
        bop->stop_operation();
        discoveryStopButton->setEnabled(false);
    }
}
    
void DiscoveryDruid::addNetwork()
{
    QListBoxText* item=(QListBoxText*)networkresultlist->firstItem();
   
    int count = networkresultlist->count();
    int upd_max=(count > 10)?count/10:1;
    int updc=upd_max;
    int t=0;
    QProgressDialog pd(tr("Adding objects ..."), tr("Cancel"), count,this,0,true);
    
    while (item!=0)
    {
        
        if (item->isSelected())
        {
            QString k=item->text();
            if (!Networks[k].isSelected) 
            {
                Networks[k].isSelected=true;
                networklist->insertItem(item->text());
            }
        }
        item=(QListBoxText*)item->next();
        
        if (updc--<=0)
        {
            pd.setProgress(t);
            qApp->processEvents();

            if (pd.wasCancelled())
            {
                break;   
            }
            updc=upd_max;
        }
        t++;
    }
    nextButton()->setEnabled(networklist->count ()>0 || Objects.size()>0);

}
    
void DiscoveryDruid::removeNetwork()
{
    QListBoxItem* item1=networklist->firstItem();
    QListBoxItem* item2;
    
    while (item1!=0)
    {
        item2=item1->next();
        if (item1->isSelected())
        {
            Networks[item1->text()].isSelected=false;
            delete item1;
        }
        item1=item2;
    }
    nextButton()->setEnabled(networklist->count ()>0 || Objects.size()>0);
}
    
void DiscoveryDruid::setNetworkFilter()
{
    flt_net_d->exec(); 
    fillListOfNetworks();
}
    
void DiscoveryDruid::removeNetworkFilter()
{
    flt_net->clear(); 
    fillListOfNetworks();
}
    
void DiscoveryDruid::addObject()
{
    
    QListBoxText* item=(QListBoxText*)objectresultlist->firstItem();
   
    int count = objectresultlist->count();
    int upd_max=(count > 10)?count/10:1;
    int updc=upd_max;
    int t=0;
    QProgressDialog pd(tr("Adding objects ..."),
                       tr("Cancel"), count,this,0,true);
    
    while (item!=0)
    {
        
        if (item->isSelected())
        {
            QString k=item->text();
            if (!Objects[k].isSelected)
            {
                Objects[k].isSelected=true;
                objectlist->insertItem(item->text());
            }
        }
        item=(QListBoxText*)item->next();
        
        if (updc--<=0)
        {
            pd.setProgress(t);
            qApp->processEvents();

            if (pd.wasCancelled())
            {
                break;   
            }
            updc=upd_max;
        }
        t++;
    }
    nextButton()->setEnabled(objectlist->count ()>0 || networklist->count()>0);
}
    
void DiscoveryDruid::removeObject()
{
    QListBoxItem* item1=objectlist->firstItem();
    QListBoxItem* item2;
    
    while (item1!=0)
    {
        item2=item1->next();
        if (item1->isSelected())
        {
            Objects[item1->text()].isSelected=false;
            delete item1;
        }
        item1=item2;
    }
    nextButton()->setEnabled(objectlist->count ()>0 || networklist->count()>0);
}
    
void DiscoveryDruid::setLastFilter()
{
    flt_last_d->exec(); 
    fillTypeChangingList();
}

void DiscoveryDruid::setObjectFilter()
{
    flt_obj_d->exec(); 
    fillListOfObjects();
}
    
void DiscoveryDruid::removeLastFilter()
{
    flt_last->clear();
    fillTypeChangingList();
}

void DiscoveryDruid::removeObjectFilter()
{
    flt_obj->clear(); 
    fillListOfObjects();
}

void DiscoveryDruid::selectAllResNets()
{
    networkresultlist->selectAll(true);
}

void DiscoveryDruid::selectAllNets()
{
    networklist->selectAll(true);
}

void DiscoveryDruid::selectAllResObjs()
{
    objectresultlist->selectAll(true);
}

void DiscoveryDruid::selectAllObjs()
{
    objectlist->selectAll(true);
}

void DiscoveryDruid::fillNetworks()
{
    ObjectDescriptor buf;

    networklist->clear();
    bool f=false;
    QMap<QString,ObjectDescriptor >::iterator i;
    for(i=Networks.begin();
        i!=Networks.end();
        ++i)
    {
        buf=i.data();
        if (buf.isSelected)
        {
            networklist->insertItem(new QListBoxText(i.key()));
            f=true;
        }
    }
    nextButton()->setEnabled(f);
}

void DiscoveryDruid::fillObjects()
{
    ObjectDescriptor buf;

    objectlist->clear();
    bool f=false; 
    QMap<QString,ObjectDescriptor >::iterator i;
    for(i=Objects.begin(); i!=Objects.end(); ++i)
    {
        buf=i.data();
        if (buf.isSelected)
        {
            objectlist->insertItem(new QListBoxText(i.key()));
            f=true;
        }
    }
    nextButton()->setEnabled(f);
}

void DiscoveryDruid::fillTypeChangingList()
{
    
    ObjectDescriptor buf;

    typeChangingList->clear();
    
    QMap<QString,ObjectDescriptor >::iterator i;
    for(i=Objects.begin(); i!=Objects.end(); ++i)
    {
        buf=i.data();
        if (buf.isSelected)
        {
            QString ins;
            if ( flt_last->test(buf) )
            {
                ins=(buf.interfaces.size())?
                    QString("%1").arg(buf.interfaces.size()):"";
                new QListViewItem( typeChangingList,buf.toString().c_str(),ins,buf.type.c_str() );
            }
        }
    }
}

void DiscoveryDruid::loadDataFromDNS()
{
    DNS_findA_query *q=(DNS_findA_query*)bop;
    Objects.clear();

    map<string,set<IPAddress> > t = q->getResult();
    
    for(map<string,set<IPAddress> >::iterator j = t.begin(); j!=t.end(); ++j)
    {
        ObjectDescriptor od;
        od.addr     = *((*j).second.begin());
        od.sysname  = (*j).first;
        if (!uselongname->isChecked())
        {
            string::size_type p=od.sysname.rfind(domainname->text().latin1());
            if (p!=string::npos)
            {
                od.sysname=od.sysname.substr(0,p-1);
            }
        }
        od.type =IPv4::TYPENAME;
        od.isSelected=false;
        
        if (od.sysname.empty())
        {
            od.sysname=string("h-") + od.addr.toString();
        }
        
        Objects[od.toString().c_str()]=od;
    }
}

void DiscoveryDruid::loadDataFromFile()
{
    objectresultlist->clear();
    int t=0;
    int count = thread->hosts.size();
    if (count > 0)
    {
        int upd_max=(count > 10)?count/10:1;
        
        int updc=upd_max;
     
        QProgressDialog pd(tr("Prepare objects ..."), tr("Cancel"), count,this,0,true);
        
        vector<ObjectDescriptor>::iterator i;
        for(i=thread->hosts.begin();
            i!=thread->hosts.end();
            ++i)
        {
            if (i->type.empty())
            {
                i->type=IPv4::TYPENAME;
            }
            i->isSelected=false;
                
            Objects[i->toString().c_str()] = *i;
            if (updc--<=0)
            {
                pd.setProgress(t);
                qApp->processEvents();

                if (pd.wasCancelled())
                {
                    break;   
                }
                updc=upd_max;
            }
            t++;
        }
    }
}

void DiscoveryDruid::loadDataFromCrawler()
{
#ifdef HAVE_LIBSNMP
    SNMPCrawler *q=(SNMPCrawler*)bop;
    Objects.clear();
    Networks.clear();
    
    set<IPNetwork>::iterator m;
    set<IPNetwork> s = q->getNetworks();

    if (fwbdebug)
        qDebug(QString("got %1 networks").arg(s.size()));

    for (m=s.begin(); m!=s.end(); ++m)
    {
        ObjectDescriptor od;
        
        od.sysname=(string)*m;
        od.addr=m->getAddress();
        od.netmask=m->getNetmask();
        od.type=Network::TYPENAME;
        od.isSelected=false;
        
        Networks[od.sysname.c_str()]= od ;
    }

    map<IPAddress, CrawlerFind>  t = q->getAllIPs();

    if (fwbdebug)
        qDebug(QString("got %1 addresses").arg(t.size()));

    discoveryprogress->setTotalSteps( t.size() );
    discoveryprogress->setProgress(0);

    int cntr = 0;
    map<IPAddress, CrawlerFind>::iterator j;
    for(j = t.begin(); j!=t.end(); ++j,++cntr) 
    {
        discoveryprogress->setProgress( cntr );

        ObjectDescriptor od( (*j).second );
        od.addr     = (*j).first;
        od.type=(od.interfaces.size()>1)?
            (Host::TYPENAME):(IPv4::TYPENAME);

        od.isSelected=false;

        if (od.sysname.empty())
        {
            od.sysname = string("h-") + od.addr.toString();
            if (snmpdodns->isChecked())
            {
                QString hostName = getNameByAddr( od.addr.toString().c_str() );
                if (!hostName.isEmpty())
                    od.sysname = hostName.latin1();
            }

            QString buf;

            buf = QString(od.addr.toString().c_str()) + " : " + od.sysname.c_str();
            discoverylog->append(buf);

        }

        Objects[od.toString().c_str()]=od;
        
        set<string>::iterator si;
        for(si=od.dns_info.aliases.begin();
                si!=od.dns_info.aliases.end();
                ++si)
        {
            od.sysname=(*si);
            Objects[od.toString().c_str()]=od;
        }
    }
#endif
/*
    (arg==0) ? 
    _("Network scan completed, click 'Next' to continue") : 
    _("There has been an error running the network scan. You can continue but data gathered by the scanner may be incomplete") 
*/
}



void DiscoveryDruid::fillListOfNetworks()
{
    networkresultlist->clear();
    int t=0;
    int count = Networks.size();
    if (count > 0)
    {
        int upd_max=(count > 10)?count/10:1;
        
        int updc=upd_max;
     
        QProgressDialog pd(tr("Copying results ..."), tr("Cancel"), count,this,0,true);
        
        QMap<QString, ObjectDescriptor>::iterator i;
        for(i=Networks.begin();
            i!=Networks.end();
            ++i)
        {
            
            if ( flt_net->test(i.data()) )
            {
            
                networkresultlist->insertItem(new QListBoxText(i.key()));
                if (updc--<=0)
                {
                    pd.setProgress(t);
                    qApp->processEvents();
    
                    if (pd.wasCancelled())
                    {
                        break;   
                    }
                    updc=upd_max;
                }
            }
            t++;
        }
    }
}

void DiscoveryDruid::fillListOfObjects()
{
    
    objectresultlist->clear();
    int t=0;
    int count = Objects.size();
    if (count > 0)
    {
        int upd_max=(count > 10)?count/10:1;
        
        int updc=upd_max;
     
        QProgressDialog pd(tr("Copying results ..."),
                           tr("Cancel"), count,this,0,true);
        
        QMap<QString,ObjectDescriptor >::iterator i;
        for(i=Objects.begin(); i!=Objects.end(); ++i)
        {
            if ( flt_obj->test(i.data()) )
            {
            
                objectresultlist->insertItem(new QListBoxText(i.key()));
                if (updc--<=0)
                {
                    pd.setProgress(t);
                    qApp->processEvents();
    
                    if (pd.wasCancelled())
                    {
                        break;   
                    }
                    updc=upd_max;
                }
            }
            t++;
        }
    }
}
    
void DiscoveryDruid::customEvent(QCustomEvent *event)
{
    int evtype=(int)event->type();
    if (evtype == ProgressEv)
    {
        ProgressEvent *e = (ProgressEvent*)event;
        discoveryprogress->setProgress(e->value);
    } else if (evtype == DoneEv)
    {
        
        cancelButton()->show();
        
        timer->stop();
        disconnect(timer,SIGNAL(timeout()),0,0);
        
        updateLog();
        logSaveButton->setEnabled(true);
        loadDataFromFile();
        
        thread->wait();
        QString er=thread->getError();
        delete thread;
        thread=NULL;
        
        if (!er.isEmpty())
        {
            QMessageBox::critical(this,tr("Discovery error"), er);
        }
        if (Objects.size()>0)
        {
            nextButton()->setDefault(true);
            nextButton()->setFocus();
            nextButton()->setEnabled(true);
            backButton()->setEnabled(false);
        }
        else
        {
            backButton()->setEnabled(true);
            nextButton()->setEnabled(false); 
        }
    }
}

void DiscoveryDruid::updateLog()
{
    if (Task==BT_HOSTS)
    {
        QString buf;
        if (thread!=NULL)
        {
            while(thread->Log->ready())
            {
                buf=thread->Log->getLine().c_str();
                discoverylog->append(buf+"\n");
            }
        }
    }
    else if (Task==BT_SNMP)
    {
       if (monitorOperation() > 0)
       {
           
           //discoveryprogress->setProgress(prg++);
       }
       else
       {
            timer->stop();
            disconnect(timer,SIGNAL(timeout()),0,0);

            if (fwbdebug) qDebug("Crawler finished");

            loadDataFromCrawler();
            
            cancelButton()->show();
            
            FWException * ex=bop->get_latest_error();
            if (ex!=NULL)
            {
                QMessageBox::critical(this,tr("Discovery error"), ex->toString().c_str());
                //discoverylog->append(QString("\nLast exception: ")+ex->toString().c_str()+"\n");    
            }
            if (Objects.size()>0 || Networks.size()>0)
            {
                if (Networks.size()==0) 
                    setAppropriate(page(8),0);
                nextButton()->setEnabled(true);
                nextButton()->setDefault(true);
                nextButton()->setFocus();
                backButton()->setEnabled(false);
            }
            else
            {
                nextButton()->setEnabled(false);
                backButton()->setEnabled(true);
            }
            
            logSaveButton->setEnabled(true);

            delete bop;
            bop=NULL;
            unBar=NULL;
            discoveryprogress->setTotalSteps(100);
            discoveryprogress->setProgress(100);
            discoveryStopButton->setEnabled(false);
       }
    }
    else if (Task==BT_DNS)
    {
       if (monitorOperation() > 0)
       {
           //discoveryprogress->setTotalSteps(0);
           //discoveryprogress->setProgress(
           //        discoveryprogress->progress()+1);
       }
       else
       {
            timer->stop();
            disconnect(timer,SIGNAL(timeout()),0,0);
           
            loadDataFromDNS();
            
            cancelButton()->show();
            FWException * ex=bop->get_latest_error();
            if (ex!=NULL)
            {    
                QMessageBox::critical(this,tr("Discovery error"), ex->toString().c_str());
                //discoverylog->append(QString("\nLast exception: ")+ex->toString().c_str()+"\n");    
            }
            if (Objects.size()>0)
            {
                nextButton()->setEnabled(true);
                nextButton()->setDefault(true);
                nextButton()->setFocus();
                backButton()->setEnabled(false);
            }
            else
            {
                nextButton()->setEnabled(false);
                backButton()->setEnabled(true);
            }
            logSaveButton->setEnabled(true);
            delete bop;
            bop=NULL;
            unBar=NULL;
            discoveryprogress->setTotalSteps(100);
            discoveryprogress->setProgress(100);
            discoveryStopButton->setEnabled(false);
       }
    }
}

void DiscoveryDruid::changedSeedHost()
{
    seedhosterror_message->setText(" ");
    userIsTyping=true;
    errMessage=seedhosterror_message;
    HostName=seedhostname->text();
    
    if (HostName.isEmpty())
    {
        timer->stop();
        DNSprogress->hide();
        seedhosterror_message->setPaletteForegroundColor(Qt::darkRed);
        seedhosterror_message->setText("Enter a valid host name or address.");
        isSeedHostOK=false;
    }
    else
    {
        if(isIPAddress(HostName))
        { // seems to be an IP Address
            DNSprogress->hide();
            timer->stop();
            QRegExp r=QRegExp("^\\d{1,3}\\.\\d{1,3}\\.\\d{1,3}\\.\\d{1,3}$",false,false);
            if (r.exactMatch(HostName))
            {
                try
                {
                    IPAddress(HostName.latin1());
                    
                    seedhosterror_message->setPaletteForegroundColor(Qt::darkGreen);
                    seedhosterror_message->setText("Address verified");
                    isSeedHostOK=true;
                } catch(const FWException &ex) 
                {
                    seedhosterror_message->setPaletteForegroundColor(Qt::darkRed);
                    seedhosterror_message->setText(ex.toString().c_str());
                    // need to return focus to the input field in case of error
                    //seedhostname->setFocus();
                    isSeedHostOK=false;
                }
            }
            else
            {
                seedhosterror_message->setPaletteForegroundColor(Qt::darkRed);
                seedhosterror_message->setText("Wrong IPv4 format");
                isSeedHostOK=false;
                
            }
        }
        else
        {// it looks like a DNS name
            isSeedHostOK=false;
            seedhosterror_message->setPaletteForegroundColor(Qt::black);
            seedhosterror_message->setText("DNS resolution in progress...");
            unBar=DNSprogress;
            errMessage=seedhosterror_message;
            DNSprogress->show();
            timer->start(1000,true);
        }
    }
    nextButton()->setEnabled(isSNMPInclNetOK && isSeedHostOK); 
}

void DiscoveryDruid::changedInclNet()
{
    setNextEnabled(currentPage(),false);
    confineerror_message->setText(" ");
    bool use_incl=!snmpinaddr->text().isEmpty() && !snmpinmask->text().isEmpty();
    if (use_incl)
    {
        try
        {
            
            IPAddress a(snmpinaddr->text().latin1());
            Netmask n(snmpinmask->text().latin1());
            IPNetwork(a,n);

            confineerror_message->setText(" ");
            isSNMPInclNetOK=true;
        } catch (const FWException &ex)
        {
            isSNMPInclNetOK=false;
            confineerror_message->setText(ex.toString().c_str());
        }
         
    }
    else
    {
        if (!snmpinaddr->text().isEmpty() || !snmpinmask->text().isEmpty())
        {
            isSNMPInclNetOK=false;
            confineerror_message->setText(tr("Incomlete network specification."));
        }
        else
        {
            confineerror_message->setText(" ");
            isSNMPInclNetOK=true;            
        }
    }
    nextButton()->setEnabled(isSNMPInclNetOK && isSeedHostOK); 
}

int DiscoveryDruid::monitorOperation()
{
    QString buf;
    bool fl;
    
    if (fwbdebug) qDebug("monitorOperation  bop=%p  isRunning=%d",
                         bop,(bop!=NULL)?bop->isRunning():-1);

    
    fl=false;
    while( logger->ready() ) 
    {
        buf= logger->getLine().c_str();
        discoverylog->append(buf);
        fl=true;
    }
    if (fl)
    {
        return 1;
    }
    if (bop==NULL)
    {

        return 0; // BackgroundOp has been disconnected
    }

    if (bop->isRunning()) 
    {
        return 1;
    }
    // send signal "completed", argument is 0 if ok and -1 if error

    
    FWException *ex=bop->get_latest_error();
    if (ex) 
    {
        buf= ex->toString().c_str();
        discoverylog->append(buf+"\n");
      //  completed(-1);   // this sends signal to another widget
    } else
    {
      //  completed(0);   // this sends signal to another widget
    }
    return 0;

}

void DiscoveryDruid::checkHostName()
{
    if (!HostName.isEmpty())
    {
        if (dns!=NULL)
        {
            delete dns;
            dns=NULL;
        }
        userIsTyping=false;
        dns=new QDns(HostName);
        connect(dns,SIGNAL(resultsReady()),this,SLOT(dnsFinish()));
    }
}

void DiscoveryDruid::checkSNMPCommunity()
{
    if (snmpcommunity->text().isEmpty())
    {
        snmpcommunity_message->setText(tr("Empty community string"));
        setNextEnabled(currentPage(),false);
    }
    else
    {
        snmpcommunity_message->setText("");
        setNextEnabled(currentPage(),true);
    }
}

void DiscoveryDruid::changeTargetObject(const QString &buf)
{
    
    QListViewItem* item=typeChangingList->firstChild();

    while (item!=0)
    {
        if (item->isSelected())
        {
            Objects[item->text(0)].type=buf.latin1();
            item->setText(2,buf);
        }
        item=item->itemBelow () ;
    }
}

void DiscoveryDruid::selectAllLast()
{
    typeChangingList->selectAll(true);
}

void DiscoveryDruid::unselectAllLast()
{
    typeChangingList->selectAll(false);
}

void DiscoveryDruid::typeAddress()
{
    changeTargetObject(IPv4::TYPENAME);
}

void DiscoveryDruid::typeHost()
{
    changeTargetObject(Host::TYPENAME);
}

void DiscoveryDruid::typeFirewall()
{
    changeTargetObject(Firewall::TYPENAME);
}

void DiscoveryDruid::createRealObjects()
{
    
    ObjectDescriptor od;
    string type,name,a;
    
    int t=0;
    lastprogress->setProgress(0);
    lastprogress->setTotalSteps( Objects.size());
    
    QMap<QString,ObjectDescriptor >::iterator i;
    for(i=Networks.begin();
        i!=Networks.end();
        ++i)
    {
        od=i.data();
        if (od.isSelected)
        {
            type = od.type;
            name=od.sysname;
            a = od.addr.toString().c_str();
            
            Network *net=dynamic_cast<Network*>(
                om->createObject(type.c_str(),name.c_str())
            );
            assert(net!=NULL);
            net->setName(name);
            net->setAddress(IPAddress(a));
            net->setNetmask(Netmask(IPAddress(a)));
            om->moveObject(libs->currentText(), net);
        }
    }
    
    for(i=Objects.begin();
        i!=Objects.end();
        ++i)
    {
        od=i.data();
        type=od.type;
        
        name=od.sysname;
        a=od.addr.toString();

        if(od.isSelected)
        {
            if (type==Host::TYPENAME || type==Firewall::TYPENAME) 
            {
                FWObject *o=NULL;

                o=om->createObject(type.c_str(),name.c_str());
                o->setName(name);

                if (od.interfaces.size()==0) 
                {
                    Interface *itf= Interface::cast(
                        om->createObject(o,Interface::TYPENAME,"nic1")
                    );
                    IPv4 *ipv4= IPv4::cast(
                        om->createObject(itf,IPv4::TYPENAME,a.c_str())
                    );
                    
                    
                    ipv4->setAddress(a);
                    ipv4->setNetmask("255.255.255.255");
                } else
                {
                    map<int,Interface>::const_iterator i;
                    for (i=od.interfaces.begin(); i!=od.interfaces.end(); ++i)
                    {
                        Interface in=i->second;
                        Interface *itf=
                            Interface::cast(om->createObject(
                                                o,
                                                Interface::TYPENAME,
                                                (i->second).getName().c_str(),
                                                &in));
                        om->autorename(itf,IPv4::TYPENAME,"ip");
                        om->autorename(itf,physAddress::TYPENAME,"mac");
                    }
                }
                if (!od.descr.empty())
                {
                    FWOptions* opt=(dynamic_cast<Host*>(o))->getOptionsObject();
                    opt->setStr("snmp_description",od.descr);
                    opt->setStr("snmp_location",   od.location);
                    opt->setStr("snmp_contact",    od.contact);
                }
                om->moveObject(libs->currentText(), o);
                if (type==Firewall::TYPENAME)
                {
                    map<string,string> platforms = Resources::getPlatforms();
                    map<string,string>::iterator i;
                    for (i=platforms.begin(); i!=platforms.end(); i++)
                        Resources::setDefaultTargetOptions( i->first,
                                                            Firewall::cast(o) );

                    map<string,string> OSs = Resources::getOS();
                    for (i=OSs.begin(); i!=OSs.end(); i++)
                        Resources::setDefaultTargetOptions( i->first,
                                                            Firewall::cast(o) );

                    mw->addFirewallToList(o);
                }
            }else if (type==Network::TYPENAME)
            {
                Network *net=dynamic_cast<Network*>(
                    om->createObject(type.c_str(),name.c_str())
                );
                assert(net!=NULL);
                net->setName(name);
                net->setAddress(IPAddress(a));
                net->setNetmask(Netmask(IPAddress(a)));
                om->moveObject(libs->currentText(), net);
            }else if (type==IPv4::TYPENAME)
            {
                IPv4 *obj=dynamic_cast<IPv4*>(
                    om->createObject(type.c_str(),name.c_str())
                );
                assert(obj!=NULL);
                obj->setName(name);
                obj->setAddress(IPAddress(a));
                obj->setNetmask("255.255.255.255");
                om->moveObject(libs->currentText(), obj);
            }
        }
        lastprogress->setProgress(t++);
        qApp->processEvents();
    }
    lastprogress->setProgress(Objects.size());
}

void DiscoveryDruid::autorename(FWObject *obj,
                                   const string &objtype,
                                   const string &namesuffix)
{
    FWObject      *hst = obj->getParent();
    list<FWObject*> ol = obj->getByType(objtype);
    int           sfxn = 1;

    for (list<FWObject*>::iterator j=ol.begin(); j!=ol.end(); ++j,sfxn++)
    {
        QString sfx;
        if (ol.size()==1) sfx="";
        else              sfx.setNum(sfxn);
        QString nn = QString("%1:%2:%3%4")
            .arg(QString::fromUtf8(hst->getName().c_str()))
            .arg(QString::fromUtf8(obj->getName().c_str()))
            .arg(namesuffix.c_str())
            .arg(sfx);

        (*j)->setName(string(nn.utf8()));
    }
    ol.clear();
}

//----------------------------------------------------------------------
ObjectDescriptor::ObjectDescriptor() {}

ObjectDescriptor::ObjectDescriptor(const ObjectDescriptor& od) {
    have_snmpd       = od.have_snmpd;
    descr            = od.descr;
    contact          = od.contact;
    location         = od.location;
    sysname          = od.sysname;
    interfaces       = od.interfaces;
    MAC_addr         = od.MAC_addr;
    dns_info.name    = od.dns_info.name;
    dns_info.aliases = od.dns_info.aliases;
    addr             = od.addr;
    type             = od.type;
    isSelected       = od.isSelected;
    netmask          = od.netmask;

}

#ifdef HAVE_LIBSNMP
ObjectDescriptor::ObjectDescriptor(const libfwbuilder::CrawlerFind& cf) {
    have_snmpd       = cf.have_snmpd;
    descr            = cf.descr;
    contact          = cf.contact;
    location         = cf.location;
    sysname          = cf.sysname;
    interfaces       = cf.interfaces;
    MAC_addr         = cf.found_phys_addr;
    dns_info.name    = cf.name;
    dns_info.aliases = cf.aliases;
}
#endif

ObjectDescriptor::~ObjectDescriptor()   {};

ObjectDescriptor& ObjectDescriptor::operator=(const ObjectDescriptor& od) {
    have_snmpd       = od.have_snmpd;
    descr            = od.descr;
    contact          = od.contact;
    location         = od.location;
    sysname          = od.sysname;
    interfaces       = od.interfaces;
    MAC_addr         = od.MAC_addr;
    dns_info.name    = od.dns_info.name;
    dns_info.aliases = od.dns_info.aliases;
    addr             = od.addr;
    type             = od.type;
    isSelected       = od.isSelected;
    netmask          = od.netmask;
    
    return *this;
}

WorkerThread::WorkerThread ( BackgroundTask t) : QThread()
{
    Task=t;
    Log=new QueueLogger();
}

WorkerThread::~WorkerThread ()
{
    delete Log;
}

void WorkerThread::setFileName(const QString &s)
{
    FileName=s;
}



void WorkerThread::setProgress(int p)
{
   ProgressEvent *event=new ProgressEvent();
   event->value=p;
   
   QApplication::postEvent(Widget,event);
}

void WorkerThread::done()
{
   DoneEvent *event=new DoneEvent();
   
   QApplication::postEvent(Widget,event);
}

QString WorkerThread::getError()
{
    return Error;
}
void WorkerThread::processHosts()
{
    *Log << "Discovery method:"
         << "Read file in hosts format.";
    
    map<IPAddress, vector<string> > reverse_hosts;
    HostsFile *hf;
/*
 *    read hosts file here 
 */
    hf=new HostsFile();
    Error="";
    setProgress(10);
    
    *Log << "Parsing file: " << FileName.latin1();
    if (!FileName.isEmpty())
    {
        try 
        {
            hf->parse( FileName.ascii() );
        } catch ( FWException &ex ) 
        {
            Error=ex.toString().c_str();
            *Log << "Exception: " << Error.ascii();
            
            delete hf;
            return;
        }
        reverse_hosts=hf->getAll();
        delete hf;
    
        setProgress(50);
        *Log << "Loading list ...";
    /*
    *    convert map format
    */
        hosts.clear();
    
        map<IPAddress,vector<string> >::iterator i;
        int count=reverse_hosts.size();
        int t=0;
        for (i=reverse_hosts.begin(); i!=reverse_hosts.end(); ++i) 
        {
    
            ObjectDescriptor od;
            od.addr    = (*i).first;
            od.sysname = ((*i).second).front();
            
            hosts.push_back( od );
    
            setProgress(50+(t++)*50/count);
        }
    }
    *Log << "end.";
    setProgress(100);            
            
    
}
void WorkerThread::processDNS()
{
    *Log << "Discovery method:"
         << "Import DNS zone";
    
    *Log << "end.";
    setProgress(100);    
}
void WorkerThread::processSNMP()
{
    *Log << "Discovery method:"
         << "Perform network discovery using SNMP";
    
    *Log << "end.";
    setProgress(100);
}


void WorkerThread::run()
{
    switch (Task)
    {
        case BT_NONE: { break;}
        case BT_HOSTS: 
        {
            processHosts();
            break;
        }
            
        case BT_DNS: { break;}
        case BT_SNMP: { break;}
                      
    }
    done();
}
