namespace Galago
{
	using System;
	using System.Collections;
	using System.ComponentModel;
	using System.Runtime.InteropServices;

	public class Core
	{
		private static Hashtable _Signals;
		private static EventHandlerList _EventHandlers;

		[DllImport("libgalago.so")]
		static extern bool galago_init(string name, bool feed);

		public static bool Init(string name, bool feed)
		{
			return galago_init(name, feed);
		}

		public static bool Init(string name)
		{
			return Init(name, false);
		}

		public static bool InitFeed(string name)
		{
			return Init(name, true);
		}

		/* TODO: InitWithMainloop */

		[DllImport("libgalago.so")]
		static extern void galago_uninit();

		public static void Uninit()
		{
			galago_uninit();
		}

		[DllImport("libgalago.so")]
		static extern bool galago_is_initted();

		public static bool Initted
		{
			get { return galago_is_initted(); }
		}

		[DllImport("libgalago.so")]
		static extern bool galago_is_connected();

		public static bool Connected
		{
			get { return galago_is_connected(); }
		}

		[DllImport("libgalago.so")]
		static extern bool galago_is_registered();

		public static bool Registered
		{
			get { return galago_is_registered(); }
		}

		[DllImport("libgalago.so")]
		static extern bool galago_is_daemon_active();

		public static bool DaemonActive
		{
			get { return galago_is_daemon_active(); }
		}

		[DllImport("libgalago.so")]
		static extern IntPtr galago_core_get_uid();

		public static string Uid
		{
			get
			{
				IntPtr raw_ret = galago_core_get_uid();

				return Marshal.PtrToStringAnsi(raw_ret);
			}
		}

		[DllImport("libgalago.so")]
		static extern IntPtr galago_core_get_obj_path();

		public static string ObjPath
		{
			get
			{
				IntPtr raw_ret = galago_core_get_obj_path();

				return Marshal.PtrToStringAnsi(raw_ret);
			}
		}

		[DllImport("libgalago.so")]
		static extern bool galago_core_is_feed();

		public static bool Feed
		{
			get { return galago_core_is_feed(); }
		}

		[DllImport("libgalago.so")]
		static extern void galago_core_set_exit_with_daemon(bool exit);

		[DllImport("libgalago.so")]
		static extern bool galago_core_get_exit_with_daemon();

		public static bool ExitWithDaemon
		{
			get { return galago_core_get_exit_with_daemon(); }
			set { galago_core_set_exit_with_daemon(value); }
		}

		[DllImport("libgalago.so")]
		static extern void galago_core_add_service(IntPtr service);

		public static void AddService(Service service)
		{
			galago_core_add_service(service.Handle);
		}

		[DllImport("libgalago.so")]
		static extern void galago_core_remove_service(IntPtr service);

		public static void RemoveService(Service service)
		{
			galago_core_remove_service(service.Handle);
		}

		[DllImport("libgalago.so")]
		static extern IntPtr galago_core_get_service(string id, bool native,
													 bool query);

		public static Service GetService(string id, bool native, bool query)
		{
			IntPtr raw_ret = galago_core_get_service(id, native, query);

			return Object.Wrap(raw_ret) as Service;
		}

		public static Service GetService(string id, bool native)
		{
			return GetService(id, native, true);
		}

		public static Service GetService(string id)
		{
			return GetService(id, false);
		}

		public static Account GetAccount(string service_id, string username,
										 bool native, bool query)
		{
			Service service = GetService(service_id, native, query);

			if (service == null)
				return null;

			return service.GetAccount(username, query);
		}

		public static Account GetAccount(string service_id, string username,
										 bool native)
		{
			return GetAccount(service_id, username, native, true);
		}

		public static Account GetAccount(string service_id, string username)
		{
			return GetAccount(service_id, username, false);
		}

		[DllImport("libgalago.so")]
		static extern IntPtr galago_core_get_services(bool native, bool query);

		public static List GetServices(bool native, bool query)
		{
			IntPtr raw_ret = galago_core_get_services(native, query);

			return new List(raw_ret);
		}

		public static List GetServices(bool native)
		{
			return GetServices(native, true);
		}

		public static List GetServices()
		{
			return GetServices(false);
		}

		public static List GetAccounts(string service_id, bool native,
									   bool query)
		{
			Service service = GetService(service_id, native, query);

			if (service == null)
				return null;

			return service.GetAccounts(query);
		}

		public static List GetAccounts(string service_id, bool native)
		{
			return GetAccounts(service_id, native, true);
		}

		public static List GetAccounts(string service_id)
		{
			return GetAccounts(service_id, false);
		}

		[DllImport("libgalago.so")]
		static extern void galago_core_add_person(IntPtr person);

		public static void AddPerson(Person person)
		{
			galago_core_add_person(person.Handle);
		}

		[DllImport("libgalago.so")]
		static extern void galago_core_remove_person(IntPtr person);

		public static void RemovePerson(Person person)
		{
			galago_core_remove_person(person.Handle);
		}

		[DllImport("libgalago.so")]
		static extern IntPtr galago_core_get_person(string id, bool native,
													bool query);

		public static Person GetPerson(string id, bool native, bool query)
		{
			IntPtr raw_ret = galago_core_get_person(id, native, query);

			return Object.Wrap(raw_ret) as Person;
		}

		public static Person GetPerson(string id, bool native)
		{
			return GetPerson(id, native, true);
		}

		public static Person GetPerson(string id)
		{
			return GetPerson(id, false);
		}

		[DllImport("libgalago.so")]
		static extern IntPtr galago_core_get_people(bool native, bool query);

		public static List GetPeople(bool native, bool query)
		{
			IntPtr raw_ret = galago_core_get_people(native, query);

			return new List(raw_ret);
		}

		public static List GetPeople(bool native)
		{
			return GetPeople(native, true);
		}

		public static List GetPeople()
		{
			return GetPeople(true);
		}

		[DllImport("libgalago.so")]
		static extern IntPtr galago_core_get_me(bool native, bool query);

		public static Person GetMe(bool native, bool query)
		{
			IntPtr raw_ret = galago_core_get_me(native, query);

			return Object.Wrap(raw_ret) as Person;
		}

		public static Person GetMe(bool native)
		{
			return GetMe(native, true);
		}

		public static Person GetMe()
		{
			return GetMe(false);
		}

		/*
		 * Signal framework
		 */
		internal static Hashtable Signals
		{
			get
			{
				if (_Signals == null)
					_Signals = new Hashtable();

				return _Signals;
			}
		}

		internal static EventHandlerList EventHandlers
		{
			get
			{
				if (_EventHandlers == null)
					_EventHandlers = new EventHandlerList();

				return _EventHandlers;
			}
		}

		[DllImport("libgalago.so")]
		static extern IntPtr galago_get_core();

		internal static void AddSignalHandler(string signal, Type type,
											  Type argsType, Delegate del)
		{
			if (EventHandlers[signal] == null)
			{
				object[] args = new object[4];
				args[0] = galago_get_core();
				args[1] = signal;
				args[2] = del;
				args[3] = argsType;

				Signals[signal] = Activator.CreateInstance(type, args);
			}
			else
				((Galago.SignalCallback)Signals[signal]).AddDelegate(del);

			EventHandlers.AddHandler(signal, del);
		}

		internal static void RemoveSignalHandler(string signal, Delegate del)
		{
			SignalCallback cb = Signals[signal] as SignalCallback;
			EventHandlers.RemoveHandler(signal, del);

			if (cb == null)
				return;

			cb.RemoveDelegate(del);

			if (EventHandlers[signal] == null) {
				Signals.Remove(signal);
				cb.Dispose();
			}
		}

		/* Signals */
		[Galago.Signal("registered")]
		public static event EventHandler OnRegistered
		{
			add
			{
				AddSignalHandler("registered",
								 typeof(SignalMarshals.VoidVoidSignal),
								 typeof(System.EventArgs),
								 value);
			}

			remove
			{
				RemoveSignalHandler("registered", value);
			}
		}

		[Galago.Signal("unregistered")]
		public static event EventHandler OnUnregistered
		{
			add
			{
				AddSignalHandler("unregistered",
								 typeof(SignalMarshals.VoidVoidSignal),
								 typeof(System.EventArgs),
								 value);
			}

			remove
			{
				RemoveSignalHandler("unregistered", value);
			}
		}

		[Galago.Signal("service-added")]
		public static event AddedHandler OnServiceAdded
		{
			add
			{
				AddSignalHandler("service-added",
								 typeof(SignalMarshals.VoidBoxedSignal),
								 typeof(AddedArgs),
								 value);
			}

			remove
			{
				RemoveSignalHandler("service-added", value);
			}
		}

		[Galago.Signal("service-removed")]
		public static event RemovedHandler OnServiceRemoved
		{
			add
			{
				AddSignalHandler("service-removed",
								 typeof(SignalMarshals.VoidBoxedSignal),
								 typeof(RemovedArgs),
								 value);
			}

			remove
			{
				RemoveSignalHandler("service-removed", value);
			}
		}

		[Galago.Signal("person-added")]
		public static event AddedHandler OnPersonAdded
		{
			add
			{
				AddSignalHandler("person-added",
								 typeof(SignalMarshals.VoidBoxedSignal),
								 typeof(AddedArgs),
								 value);
			}

			remove
			{
				RemoveSignalHandler("person-added", value);
			}
		}

		[Galago.Signal("person-removed")]
		public static event RemovedHandler OnPersonRemoved
		{
			add
			{
				AddSignalHandler("person-removed",
								 typeof(SignalMarshals.VoidBoxedSignal),
								 typeof(RemovedArgs),
								 value);
			}

			remove
			{
				RemoveSignalHandler("person-removed", value);
			}
		}

		[Galago.Signal("updated")]
		public static event EventHandler OnUpdated
		{
			add
			{
				AddSignalHandler("updated",
								 typeof(SignalMarshals.VoidVoidSignal),
								 typeof(EventArgs),
								 value);
			}

			remove
			{
				RemoveSignalHandler("updated", value);
			}
		}
	}
}
